/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  AbsoluteTime,
  CounterResultByEventName,
  EventReporting_TOPS_queries,
  EventReporting_VQF_queries,
  fetchTopsInfoFromServer,
  fetchVqfInfoFromServer,
  OfficerAccount,
  OperationOk,
  opFixedSuccess,
  TalerExchangeResultByMethod2,
  TalerHttpError
} from "@gnu-taler/taler-util";
// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import { useExchangeApiContext } from "@gnu-taler/web-util/browser";
import { endOfYear, setYear, startOfYear } from "date-fns";
import _useSWR, { SWRHook } from "swr";
import { useOfficer } from "./officer.js";
const useSWR = _useSWR as unknown as SWRHook;

export function useServerMeasures() {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;

  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  async function fetcher([officer]: [OfficerAccount]) {
    return await api.getAmlMeasures(officer);
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod2<"getAmlMeasures">,
    TalerHttpError
  >(!session ? undefined : [session], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
    errorRetryCount: 0,
    errorRetryInterval: 1,
    shouldRetryOnError: false,
    keepPreviousData: true,
  });

  if (data) return data;
  if (error) return error;
  return undefined;
}

export function useTopsServerStatistics() {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;

  const {
    unthrottledApi: { exchange: api },
  } = useExchangeApiContext();

  async function fetcher([officer]: [OfficerAccount]) {
    const final = await fetchTopsInfoFromServer(api, officer);

    return opFixedSuccess(final);
  }

  const { data, error } = useSWR<
    OperationOk<CounterResultByEventName<typeof EventReporting_TOPS_queries>>,
    TalerHttpError
  >(!session ? undefined : [session], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
    errorRetryCount: 0,
    errorRetryInterval: 1,
    shouldRetryOnError: false,
    keepPreviousData: true,
  });

  if (data) return data;
  if (error) return error;
  return undefined;
}

export function useVqfServerStatistics(year: number) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;

  const {
    unthrottledApi: { exchange: api },
  } = useExchangeApiContext();

  async function fetcher([officer, year]: [OfficerAccount, number]) {
    const date = setYear(new Date(), year);
    const jan_1st = AbsoluteTime.fromMilliseconds(startOfYear(date).getTime());
    const dec_31st = AbsoluteTime.fromMilliseconds(endOfYear(date).getTime());

    const final = await fetchVqfInfoFromServer(api, officer, jan_1st, dec_31st);

    return opFixedSuccess(final);
  }

  const { data, error } = useSWR<
    OperationOk<
      CounterResultByEventName<ReturnType<typeof EventReporting_VQF_queries>>
    >,
    TalerHttpError
  >(!session ? undefined : [session, year], fetcher, {
    refreshInterval: 0,
    refreshWhenHidden: false,
    revalidateOnFocus: false,
    revalidateOnReconnect: false,
    refreshWhenOffline: false,
    errorRetryCount: 0,
    errorRetryInterval: 1,
    shouldRetryOnError: false,
    keepPreviousData: true,
  });

  if (data) return data;
  if (error) return error;
  return undefined;
}
