/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: edit.c
 *    This file is part of mino.
 *
 *    mino is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    mino is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with mino.  If not, see <http://www.gnu.org/licenses/>.
 */    

#include "defs.h"
#include "edit.h"
#include "kbd.h"
#include "options.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#define tostr(x)    #x

struct undostruct *firstUndo, *lastUndo, *firstRedo;
char undoBuf[4096];
int  undoBufIndex    = 0;
int  undoBufChars    = 0;
char undoBufRep[4096];
int  undoBufRepIndex = 0;
int  undoBufRepChars = 0;
int  clipboardSize   = 0;

point find_result_pos[1024];
int   total_find_results;


void deleteUndoList(struct undostruct *first)
{
    while(first)
    {
        struct undostruct *undo = first->prev;
        if(first->text) free(first->text);
        free(first);
        first = undo;
    }
}

void deleteRedoList(struct undostruct *first)
{
    while(first)
    {
        struct undostruct *undo = first->next;
        if(first->text) free(first->text);
        free(first);
        first = undo;
    }
}

void initEdit()
{
    firstUndo = NULL;
    deleteUndoList(lastUndo);
    deleteRedoList(firstRedo);
    undoBufChars = 0;
    undoBufIndex = 0;
    undoBufRepChars = 0;
    undoBufRepIndex = 0;
    if(clipboardSize == 0)
    {
        clipboardSize = 1024;
        clipboard = (char *)malloc(clipboardSize);
    }
}

struct undostruct *allocNewUndo()
{
    if(lastUndo && lastUndo->type == UNDO_ACTION_NONE) return lastUndo;
    struct undostruct *undo = (struct undostruct *)malloc(sizeof(struct undostruct));
    if(!undo)
    {
        msgBox("Insufficient memory", OK, ERROR);
        return NULL;
    }
    undo->lineCount  = 0;
    undo->byteCount  = 0;
    undo->rbyteCount = 0;
    undo->rlineCount = 0;
    undo->lineStart  = -1;
    undo->charStart  = -1;
    undo->text  = NULL;
    undo->rtext = NULL;
    undo->type  = UNDO_ACTION_NONE;
    undo->next  = NULL;
    if(lastUndo)
    {
        undo->prev = lastUndo;
        lastUndo->next = undo;
        lastUndo = undo;
    }
    else
    {
        undo->prev = NULL;
        lastUndo = undo;
        firstUndo = undo;
    }
    deleteRedoList(firstRedo);
    return undo;
}

struct undostruct *getLastUndo(int allocIfNull)
{
    if(lastUndo) return lastUndo;
    if(!allocIfNull) return NULL;
    return allocNewUndo();
}

void initUndoAction(struct undostruct *undo, undoActionType utype, int lwhere, int cwhere)
{
    undo->type = utype;
    undo->lineStart = lwhere;
    undo->charStart = cwhere;
}

int copyFromBuf(char **dest, char *buf, int bcount)
{
    if(bcount == 0) return 1;
    *dest = (char *)malloc(bcount+1);
    if(!*dest) return 0;
    memcpy(*dest, buf, bcount);
    (*dest)[bcount] = '\0';
    return 1;
}

void flushUndoBuffer(struct undostruct *undo)
{
    if(undoBufIndex == 0) return;
    if(!copyFromBuf(&undo->text, undoBuf, undoBufIndex)) goto memerr;
    undo->byteCount  = undoBufIndex;
    if(!copyFromBuf(&undo->rtext, undoBufRep, undoBufRepIndex)) goto memerr;
    undo->rbyteCount = undoBufRepIndex;
    undoBufIndex = 0;
    undoBufChars = 0;
    undoBufRepChars = 0;
    undoBufRepIndex = 0;
    return;
    
memerr:
    msgBox("Insufficient memory", OK, ERROR);
}

// check if the new character to be added/deleted is continuous
// with the rest of the undo action.
int isContinuous(struct undostruct *undo, int l2, int c2)
{
    int l1 = undo->lineStart;
    if(l1 != l2) return 0;
    int c1 = undo->charStart;
    if(c1 == c2 || c2 == c1-1) return 1;
    if(c2 == c1+undoBufChars) return 1;
    return 0;
}

/*
 * lwhere : line where undo action occurred
 * cwhere : char where undo action occurred
 * what   : char to be added to the undo action
 * rwhat  : replacement char (only if utype == UNDO_ACTION_REPLACE)
 */
void undoAddChar(undoActionType utype, int lwhere, int cwhere, char what)
{
    struct undostruct *undo = getLastUndo(1);
    if(undo == NULL) goto memerr;
    if(undo->type != utype || !isContinuous(undo, lwhere, cwhere))
    {
        flushUndoBuffer(undo);
        if(undo->type != UNDO_ACTION_NONE)
        {
            undo = allocNewUndo();
            if(undo == NULL) goto memerr;
        }
        initUndoAction(undo, utype, lwhere, cwhere);
    }
    if(cwhere == undo->charStart)
    {
        int i;
        for(i = undoBufIndex; i > 0; i--)
            undoBuf[undoBufIndex] = undoBuf[undoBufIndex-1];
        undoBuf[0] = what;
        //initUndoAction(undo, utype, lwhere, cwhere);
    }
    else undoBuf[undoBufIndex] = what;
    undoBufIndex++;
    if((what & 0xc0) != 0x80) undoBufChars++;
    if(what == '\n')
    {
        flushUndoBuffer(undo);
        allocNewUndo();
    }
    return;
    
memerr:
    msgBox("Insufficient memory", OK, ERROR);
    return;
}

// adds one multibyte UTF-8 char to the undo action.
// returns the number of bytes added to the undo action.
int undoAddUtfChar(undoActionType utype, int lwhere, int cwhere, char *rwhat)
{
    int save = cwhere;
    char *what;
    if(utype == UNDO_ACTION_INSERT) what = rwhat;
    else what = lines[lwhere]->text+cwhere;
    /*
    if(utype == UNDO_ACTION_REPLACE)
    {
        struct undostruct *undo = getLastUndo(1);
        if(undo->type != UNDO_ACTION_NONE)
        {
            flushUndoBuffer(undo);
            allocNewUndo();
        }
    }
    */
    undoAddChar(utype, lwhere, cwhere, *what);
    // check for UTF-8 continuation sequence
    while(++cwhere, (*(++what) & 0xc0) == 0x80)
        undoAddChar(utype, lwhere, cwhere, *what);
    if(utype == UNDO_ACTION_REPLACE)
    {
        char c;
        //while((c = *rwhat++)) undoBuf[undoBufIndex++] = c;
        while((c = *rwhat++)) undoBufRep[undoBufRepIndex++] = c;
    }
    return cwhere-save;
}

int extendClipboardSize(int newSize)
{
    if(newSize < clipboardSize) return 1;
    clipboardSize = newSize+1;
    if(clipboard) free(clipboard);
    clipboard = (char *)malloc(clipboardSize);
    if(!clipboard)
    {
        msgBox("Insufficient memory", OK, ERROR);
    }
}


void swap_lines() 
{
  int tmp;
  tmp = sel_range_end.nline;
  sel_range_end.nline = sel_range_start.nline;
  sel_range_start.nline = tmp;
  tmp = sel_range_end.nchar;
  sel_range_end.nchar = sel_range_start.nchar;
  sel_range_start.nchar = tmp;
}

void swap_chars() 
{
  int tmp;
  tmp = sel_range_end.nchar;
  sel_range_end.nchar = sel_range_start.nchar;
  sel_range_start.nchar = tmp;
}

/*****************************************
 * Clears the selected range. This is
 * helpful when exiting from the select
 * mode (like using arrow keys after
 * releasing SHIFT).
 * ***************************************/
void clear_selected_range() 
{
  
}

void editMenu_DeleteLine() 
{
  deleteLine();
}

/*****************************************
 * Toggles select mode ON/OFF. Select mode
 * is useful when running under X, to
 * emulate SHIFT-selection.
 * ***************************************/
void editMenu_ToggleSelectMode() 
{
  if(X_IS_RUNNING) 
  {
    SELECTING = !SELECTING;
    refreshBottomView();
    if(SELECTING) 
    {
      sel_range_start.nline = firstVisLine+selectedLine;
      sel_range_start.nchar = selectedChar;
      sel_range_end.nline = firstVisLine+selectedLine;
      sel_range_end.nchar = selectedChar;
    }
  } 
  else 
  {
    msgBox("The select mode is only available under X.\nUse SHIFT to select text.", OK, INFO);
    refreshView();
  }
}

/*****************************************
 * Copies selection to clipboard.
 * ***************************************/
void editMenu_Copy() 
{
    if(!SELECTING && !SELECTED) return;
    int i, j, k, l;
    if(!clipboard) goto memerr;
    clipboard[0] = '\0';

    //swap the select range boundaries if needed
    int swap = 0;
    if(sel_range_start.nline > sel_range_end.nline) { swap = 1; swap_lines(); }
    else if(sel_range_start.nline == sel_range_end.nline &&
            sel_range_start.nchar > sel_range_end.nchar) 
    { swap = 2; swap_chars(); }

    total_lines_in_clipboard = sel_range_end.nline-sel_range_start.nline;
    if(total_lines_in_clipboard == 0)
    {
        j = charsToBytes(sel_range_start.nline, sel_range_start.nchar);
        k = charsToBytes(sel_range_end.nline  , sel_range_end.nchar  );
        i = k-j;
        if(!extendClipboardSize(i)) goto memerr;
        memcpy(clipboard, lines[sel_range_start.nline]->text+j, i);
        clipboard[i] = '\0';
    }
    else
    {
        l = 0;
        j = charsToBytes(sel_range_start.nline, sel_range_start.nchar);
        l += (lines[sel_range_start.nline]->byteCount-j);
        k = charsToBytes(sel_range_end.nline  , sel_range_end.nchar  );
        l += k;
        for(i = sel_range_start.nline+1; i < sel_range_end.nline; i++)
            l += lines[i]->byteCount;
        if(!extendClipboardSize(l)) goto memerr;
        strcpy(clipboard, lines[sel_range_start.nline]->text+j);
        for(i = sel_range_start.nline+1; i < sel_range_end.nline; i++)
            strcat(clipboard, lines[i]->text);
        strncat(clipboard, lines[sel_range_end.nline]->text, k);
        clipboard[l] = '\0';
    }
    CLIPBOARD_IS_EMPTY = 0;
  
    if(swap == 1) swap_lines();//return them back to normal
    if(swap == 2) swap_chars();//return them back to normal
    return;
memerr:
    msgBox("Insufficient memory", OK, ERROR);
    return;
}

/*********************************************
 * Copies selection to clipboard then cuts it.
 * *******************************************/
void editMenu_Cut() 
{
    if(SELECTING || SELECTED) 
    {
        editMenu_Copy();
        remove_selected_text(1);
    }
}

/*****************************************
 * Removes the selected range of text.
 * ***************************************/
void remove_selected_text(int recordAction)
{
    int swap = 0;
    if(sel_range_start.nline > sel_range_end.nline) { swap = 1; swap_lines(); }
    else if(sel_range_start.nline == sel_range_end.nline &&
            sel_range_start.nchar > sel_range_end.nchar)
    { swap = 2; swap_chars(); }

    int i, j, k, l;
    int refreshAll = 0;
    int first = sel_range_start.nline;
    int last  = sel_range_end.nline  ;
    int diff  = last-first;
    struct undostruct *undo = getLastUndo(1);
    if(recordAction)
    {
        flushUndoBuffer(undo);
        undo = allocNewUndo();
        undo->type = UNDO_ACTION_DELETE;
        undo->lineStart = sel_range_start.nline;
        undo->charStart = sel_range_start.nchar;
        undo->lineCount = diff;
    }
    
    if(diff == 0)
    {
        j = charsToBytes(first, sel_range_start.nchar);
        k = charsToBytes(last , sel_range_end.nchar  );
        i = k-j;
        if(recordAction)
        {
            undo->text = (char *)malloc(i+1);
            if(!undo->text) goto memerr;
            memcpy(undo->text, lines[first]->text+j, i);
            undo->text[i] = '\0';
            //undo->charCount = (lines[first]->charCount-sel_range_start.nchar)+sel_range_end.nchar;
            undo->byteCount = i;
        }
        if(lines[sel_range_start.nline]->linkedToNext) refreshAll = 1;
        copyInLine(sel_range_start.nline, j, k, 0);
        checkLineBounds(sel_range_start.nline);
    }
    else
    {
        l = 0;
        j = charsToBytes(first, sel_range_start.nchar);
        l += (lines[first]->byteCount-j);
        k = charsToBytes(last , sel_range_end.nchar  );
        l += k;
        if(recordAction)
        {
            for(i = first+1; i < last; i++)
                l += lines[i]->byteCount;
            undo->text = (char *)malloc(l+1);
            if(!undo->text) goto memerr;
            strcpy(undo->text, lines[first]->text+j);
            for(i = first+1; i < last; i++)
                strcat(undo->text, lines[i]->text);
            strncat(undo->text, lines[last]->text, k);
            undo->text[l] = '\0';
            /*
            undo->charCount = (lines[first]->charCount-sel_range_start.nchar)+sel_range_end.nchar;
            for(i = first+1; i < last; i++)
                undo->charCount += lines[i]->charCount;
            */
            undo->byteCount = l;
        }
        // prepare the first line to amalgamate the last line to.
        l = j+k;
        if(l >= lines[first]->byteCount)
        {
            char *s = (char *)realloc(lines[first]->text, l+1);
            if(!s) goto memerr;
            lines[first]->text = s;
        }
        strncat(lines[first]->text+j, lines[last]->text, k);
        lines[first]->text[j+k] = '\0';
        calcTotalCharsInLine(first);
        lines[first]->linkedToNext = lines[last]->linkedToNext;
        // shift lines up by the difference between first and last lines
        move_lines_upd(first+1, last-first);
        // then we can check our line's length
        checkLineBounds(first);
        refreshAll = 1;
    }

    SELECTING = 0; SELECTED = 0;
    FILE_STATE = MODIFIED;
    selectedChar = sel_range_start.nchar;
    refreshAll = fixViewPostUndo(first);

    if(refreshAll) refreshView();
    else refreshSelectedLine();

    if(swap == 1) swap_lines();//return them back to normal
    if(swap == 2) swap_chars();//return them back to normal
    return;
    
memerr:
    msgBox("Insufficient memory", OK, ERROR);
    return;
}

void _do_paste(char *src, int srcLineCount, int recordAction)
{
    int i = 0;
    int j = firstVisLine+selectedLine;
    int k = selectedChar;
    int l;
    if((srcLineCount+totalLines) >= MAX_LINES)
    {
        msgBox("Unable to paste text. Line count will exceed the maximum of "
                tostr(MAX_LINES) ".", OK, ERROR);
        return;
    }
    else l = srcLineCount;

    if(recordAction)
    {
        struct undostruct *undo = getLastUndo(1);
        flushUndoBuffer(undo);
        undo = allocNewUndo();
        undo->type = UNDO_ACTION_INSERT;
        undo->lineStart = j;
        undo->charStart = k;
        undo->lineCount = l;
        undo->text = (char *)malloc(strlen(src)+1);
        if(!undo->text)
        {
            msgBox("Insufficient memory", OK, ERROR);
            return;
        }
        strcpy(undo->text, src);
    }

    //if pasting in the middle of a line, save the rest of the line
    int tmpLen = lines[j]->byteCount-charsToBytes(j, k);
    char tmp[tmpLen+1];
    if(tmpLen) strcpy(tmp, lines[j]->text+k);
    else      tmp[0] = '\0';
  
    int n = l+1;
    for(i = totalLines+l; i > j+l; i--) copyLineStruct(i, i-l);
    for( ; i > j; i--) lines[i] = allocLineStructB(maxLen);
    i = 0;
    l += j+1;
    
    char *line = lines[j]->text;
    char *clip = src;
    while(j <= l)
    {
        if(k >= MAX_CHARS_PER_LINE) 
        {
            move_lines_down(totalLines, j+1);
            n++;
            lines[j]->linkedToNext = 1;
            *line = '\0';
            k = 0;
            calcTotalCharsInLine(j);
            line = lines[++j]->text;
            selectedLine++;
        }
        if(*clip == '\0')
        {
            *line = '\0';
            calcTotalCharsInLine(j);
            selectedLine++;
            break;
        }
        else if(*clip == '\n')
        {
            lines[j]->linkedToNext = 0;
            *line++ = '\n';
            *line = '\0';
            k = 0;
            calcTotalCharsInLine(j);
            line = lines[++j]->text;
            clip++;
            selectedLine++;
        }
        else
        {
            if(*clip == '\t')
            {
                i = TABSPACES(k+1);
                k += i;
            }
            else k++;
            *line++ = *clip++;
        }
    }
    totalLines += n;
  
    //if there is text in tmp, append it to the last pasted line
    if(tmpLen)
    {
        i = lines[j]->byteCount+tmpLen;
        if(i >= maxLen)
        {
            if(!extendLineText(j, i))
            {
                msgBox("Insufficient memory", OK, ERROR);
                return;
            }
        }
        strcat(lines[j]->text, tmp);
        checkLineBounds(j);
    }
  
    //adjust the view
    if(selectedLine >= totalVisLines)
    {
        int diff = selectedLine-totalVisLines+1;
        firstVisLine += diff;
        selectedLine -= diff;
    }
    if(totalLines <= totalVisLines) 
    {
        firstVisLine = 0;
        selectedLine = totalLines-1;
    } 
    else if((totalLines-j) < totalVisLines) 
    {
        firstVisLine = totalLines-totalVisLines;
        selectedLine = totalVisLines-(totalLines-j)-1;
    }
    selectedChar = k;
}

/*****************************************
 * Pastes whatever in the clipboard into
 * the current position.
 * ***************************************/
void editMenu_Paste() 
{
    if(CLIPBOARD_IS_EMPTY) return;
    _do_paste(clipboard, total_lines_in_clipboard, 1);
    SELECTED = 0;
    SELECTING = 0;
    FILE_STATE = MODIFIED;
    refreshView();
}

/*****************************************
 * 
 * ***************************************/
void editMenu_SelectAll() 
{
    SELECTING = 1;
    sel_range_start.nline = 0;
    sel_range_end.nline   = totalLines-1;
    sel_range_start.nchar = 0;
    sel_range_end.nchar   = lines[totalLines-1]->charCount;
    if(sel_range_end.nchar < 0) sel_range_end.nchar = 0;
    if(totalLines <= totalVisLines) 
    {
        firstVisLine = 0;
        selectedLine = totalLines-1;
        selectedChar = lines[selectedLine]->charCount;
    } 
    else 
    {
        firstVisLine = totalLines-totalVisLines;
        selectedLine = totalVisLines-1;
        selectedChar = lines[firstVisLine+selectedLine]->charCount;
    }
    calcCharCarry(firstVisLine+selectedLine);
    SELECTING = 0;
    SELECTED = 1;
    refreshView();
    //refreshBottomView();
}

int fixViewPostUndo(int first)
{
    int refreshAll = 0;
    if(first < firstVisLine)
    {
        selectedLine = 0;
        firstVisLine = first;
        refreshAll = 1;
    }
    else
    {
        selectedLine = first-firstVisLine;
        if(selectedLine >= totalVisLines)
        {
            firstVisLine += (selectedLine-totalVisLines+1);
            selectedLine = totalVisLines-1;
        }
    }
    // fix the view if needed
    if(firstVisLine+totalVisLines > totalLines)
    {
        int i = firstVisLine;
        firstVisLine = totalLines-totalVisLines;
        selectedLine += (firstVisLine-i);
        refreshAll = 1;
    }
    return refreshAll;

    /*
    if(firstVisLine+selectedLine != undo->lineStart)
    {
        if(totalLines <= totalVisLines)
        {
            firstVisLine = 0;
            selectedLine = undo->lineStart;
        }
        else
        {
            firstVisLine = undo->lineStart;
            selectedLine = 0;
            int i = firstVisLine+totalVisLines;
            if(i > totalLines)
            {
                i -= totalLines;
                firstVisLine += i;
            }
        }
    }
    */
}

/************************************************
 * Undo the last action done. Last action can be:
 * UNDO_ACTION_REPLACE: The user replaced a
 * 			character with INSERT on
 * UNDO_ACTION_INSERT: The user typed regularly
 * UNDO_ACTION_DELETE: The user deleted using
 * 			DEL or BACKSPACE
 * **********************************************/
void editMenu_Undo() 
{
    struct undostruct *undo = getLastUndo(0);
    if(undo == NULL) return;
    flushUndoBuffer(undo);

    int first = undo->lineStart;
    int i = undo->byteCount;
    int j = undo->rbyteCount;
    int k, l;
    
    if(undo->type == UNDO_ACTION_REPLACE)
    {
        k = charsToBytes(first, undo->charStart);
        if(undo->lineCount == 0)
        {
            if(i > j)
            {
                l = i-j;
                if(!extendLineText(first, lines[first]->byteCount+l)) goto memerr;
                copyInLine(first, k+i, k+j, 0);
                memcpy(lines[first]->text+k, undo->text, i);
            }
            else if(i < j)
            {
                memcpy(lines[first]->text+k, undo->text, i);
                copyInLine(first, k+i, k+j, 0);
            }
            else
            {
                memcpy(lines[first]->text+k, undo->text, i);
            }
            selectedChar = undo->charStart;
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            calcCharCarry(first);
        }
        else
        {
            // amalgamate first line (after first char) to 
            // last line (before last char).
            char *s = strrchr(undo->rtext, '\n');
            if(!s) return;
            k += strlen(s);
            if(!extendLineText(first, k+1)) goto memerr;
            strcat(lines[first]->text, s+1);
            lines[first]->linkedToNext = lines[first+undo->lineCount]->linkedToNext;
            move_lines_upd(first+1, undo->lineCount);
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            
            // then paste our text
            firstVisLine = undo->lineStart;
            selectedLine = 0;
            selectedChar = undo->charStart;
            _do_paste(undo->text, undo->lineCount, 0);
        }
    } else if(undo->type == UNDO_ACTION_INSERT) {
        sel_range_start.nline = undo->lineStart;
        sel_range_end.nline   = undo->lineStart+undo->lineCount;
        sel_range_start.nchar = undo->charStart;
        char *s = strrchr(undo->text, '\n');
        if(!s) s = undo->text-1;
        i = utfstrlen(s+1);
        sel_range_end.nchar = i;
        if(sel_range_start.nline == sel_range_end.nline)
            sel_range_end.nchar += undo->charStart;
        remove_selected_text(0);
    } else if(undo->type == UNDO_ACTION_DELETE) {
        // deletion was in one line
        if(undo->lineCount == 0)
        {
            k = lines[first]->byteCount+i;
            if(!extendLineText(first, k)) goto memerr;
            copyInLine(first, undo->charStart+i, undo->charStart, 0);
            memcpy(lines[first]->text+undo->charStart, undo->text, i);
            selectedChar = undo->charStart;
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            calcCharCarry(first);
        }
        else
        {
            firstVisLine = undo->lineStart;
            selectedLine = 0;
            selectedChar = undo->charStart;
            _do_paste(undo->text, undo->lineCount, 0);
        }
    }
    
    fixViewPostUndo(undo->lineStart);
    firstRedo = undo;
    lastUndo = undo->prev;
    calcCharCarry(undo->lineStart);
    refreshView();
    return;
memerr:
    msgBox("Insufficient memory", OK, ERROR);
}

/************************************************
 * Redo the last action done. Last action can be:
 * UNDO_ACTION_REPLACE: The user replaced a
 * 			character with INSERT on
 * UNDO_ACTION_INSERT: The user typed regularly
 * UNDO_ACTION_DELETE: The user deleted using
 * 			DEL or BACKSPACE
 * **********************************************/
void editMenu_Redo() 
{
    struct undostruct *undo = firstRedo;
    if(undo == NULL) return;

    int first = undo->lineStart;
    int i = undo->rbyteCount;
    int j = undo->byteCount;
    int k, l;
    
    if(undo->type == UNDO_ACTION_REPLACE)
    {
        k = charsToBytes(first, undo->charStart);
        if(undo->lineCount == 0)
        {
            if(i > j)
            {
                l = i-j;
                if(!extendLineText(first, lines[first]->byteCount+l)) goto memerr;
                copyInLine(first, k+i, k+j, 0);
                memcpy(lines[first]->text+k, undo->rtext, i);
            }
            else if(i < j)
            {
                memcpy(lines[first]->text+k, undo->rtext, i);
                copyInLine(first, k+i, k+j, 0);
            }
            else
            {
                memcpy(lines[first]->text+k, undo->rtext, i);
            }
            selectedChar = undo->charStart;
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            calcCharCarry(first);
        }
        else
        {
            // amalgamate first line (after first char) to 
            // last line (before last char).
            char *s = strrchr(undo->text, '\n');
            if(!s) return;
            k += strlen(s);
            if(!extendLineText(first, k+1)) goto memerr;
            strcat(lines[first]->text, s+1);
            lines[first]->linkedToNext = lines[first+undo->lineCount]->linkedToNext;
            move_lines_upd(first+1, undo->lineCount);
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            
            // then paste our text
            firstVisLine = undo->lineStart;
            selectedLine = 0;
            selectedChar = undo->charStart;
            _do_paste(undo->rtext, undo->lineCount, 0);
        }
    } else if(undo->type == UNDO_ACTION_INSERT) {
        // insertion was in one line
        if(undo->lineCount == 0)
        {
            k = lines[first]->byteCount+j;
            if(!extendLineText(first, k)) goto memerr;
            i = charsToBytes(first, undo->charStart);
            copyInLine(first, undo->charStart+j, undo->charStart, 0);
            memcpy(lines[first]->text+i, undo->text, j);
            selectedChar = undo->charStart;
            //calcTotalCharsInLine(first);
            checkLineBounds(first);
            calcCharCarry(first);
        }
        else
        {
            firstVisLine = undo->lineStart;
            selectedLine = 0;
            selectedChar = undo->charStart;
            _do_paste(undo->rtext, undo->lineCount, 0);
        }
    } else if(undo->type == UNDO_ACTION_DELETE) {
        sel_range_start.nline = undo->lineStart;
        sel_range_end.nline   = undo->lineStart+undo->lineCount;
        sel_range_start.nchar = undo->charStart;
        char *s = strrchr(undo->text, '\n');
        if(!s) s = undo->text-1;
        i = utfstrlen(s+1);
        sel_range_end.nchar = i;
        if(sel_range_start.nline == sel_range_end.nline)
            sel_range_end.nchar += undo->charStart;
        remove_selected_text(0);
    }
    
    fixViewPostUndo(undo->lineStart);
    firstRedo = undo->next;
    lastUndo = undo;
    calcCharCarry(undo->lineStart);
    refreshView();
    return;
memerr:
    msgBox("Insufficient memory", OK, ERROR);
}


/*****************************************
 * Finds a string text in the file.
 * ***************************************/
void editMenu_Find()
{
    char *f = getUserInput("Enter text to find:  ", " Find ");
    if(!f)
    {
        refreshView();
        return;
    }
  
    int i;
    _find(f);
  
    if(!total_find_results) 
    {
        free(f);
        msgBox("No matches were found.", OK, INFO);
        refreshView();
        return;
    }
  
    //infinite loop to get user input
    i = 0;
    char *c;
    while(1) 
    {
        int x = find_result_pos[i].nline-firstVisLine;
        if(x < 0)
        {
            firstVisLine += x;
            selectedLine = 0;
        }
        else
        {
            if(x >= totalVisLines)
            {
                x -= totalVisLines;
                firstVisLine += (x+1);
                selectedLine = totalVisLines-1;
            }
            else selectedLine = x;
        }
        selectedChar = find_result_pos[i].nchar;
        calcCharCarry(find_result_pos[i].nline);
        refreshView();
        setScreenColorsI(COLOR_STATUS_BAR);
        if(GNU_DOS_LEVEL > 2)
            fprintf(stdout, "\e[%d;1HFind(%d/%d): [C-p] Prev [C-n] Next [C-g] Cancel",
                    SCREEN_H, i+1, total_find_results);
        else if(GNU_DOS_LEVEL > 1)
            fprintf(stdout, "\e[%d;1HFind(%d/%d): [C-p] Prev [C-n] Next [ESC] Cancel",
                    SCREEN_H, i+1, total_find_results);
        else
            fprintf(stdout, "\e[%d;1HFind(%d/%d): [Up] Prev [Down] Next [ESC] Cancel",
                    SCREEN_H, i+1, total_find_results);
        fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
        fflush(stdout);
get_key:
        c = getKey();
        switch(c[0]) 
        {
            case('p'):
                if(GNU_DOS_LEVEL < 2) break;
                if(!CTRL) break;
                goto do_up;
            case(UP_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_up:
                if(i <= 0) i = total_find_results-1;
                else i--;
                break;
            case('n'):
                if(GNU_DOS_LEVEL < 2) break;
                if(!CTRL) break;
                goto do_down;
            case(DOWN_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_down:
            case(ENTER_KEY):
            case(SPACE_KEY):
                if(i >= total_find_results-1) i = 0;
                else i++;
                break;
            case('g'):
                if(GNU_DOS_LEVEL < 3) break;
                if(!CTRL) break;
                goto do_esc;
            case(ESC_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_esc:
                refreshView();
                free(f);
                return;
            default:
                goto get_key;
                break;
        }
    }
    return;
    
//memerr:
//    msgBox("Insufficient memory!", OK, ERROR);
}


/*****************************************
 * Replaces Find text with Replace text.
 * ***************************************/
void editMenu_Replace() 
{
    char *f = getUserInput("Enter text to find:  ", " Find ");
    if(!f)
    {
        refreshView();
        return;
    }
  
    char *r = getUserInput("Enter replacement text: ", " Replace ");
    if(!r)
    {
        free(f);
        refreshView();
        return;
    }
  
    int i;
    _find(f);
  
    if(!total_find_results) 
    {
        free(f);
        free(r);
        msgBox("No matches were found.", OK, INFO);
        refreshView();
        return;
    }
  
    //infinite loop to get user input
    i = 0;
    char *c;
    while(1)
    {
        int x = find_result_pos[i].nline-firstVisLine;
        if(x < 0)
        {
            firstVisLine += x;
            selectedLine = 0;
        }
        else
        {
            if(x >= totalVisLines)
            {
                x -= totalVisLines;
                firstVisLine += (x+1);
                selectedLine = totalVisLines-1;
            }
            else selectedLine = x;
        }
        selectedChar = find_result_pos[i].nchar;
        calcCharCarry(find_result_pos[i].nline);
        refreshView();
        setScreenColorsI(COLOR_STATUS_BAR);
        if(GNU_DOS_LEVEL > 2)
            fprintf(stdout, "\e[%d;1HFind(%d/%d): [ENTER] Replace [A] Replace All [C-g] Cancel",
                    SCREEN_H, i+1, total_find_results);
        else
            fprintf(stdout, "\e[%d;1HFind(%d/%d): [ENTER] Replace [A] Replace All [ESC] Cancel",
                    SCREEN_H, i+1, total_find_results);
        fprintf(stdout, "\e[%d;%dH", selectedLine+3, selectedChar+2+selectedCharCarry);
        fflush(stdout);
get_key:
        c = getKey();
        switch(c[0]) 
        {
            case('p'):
                if(GNU_DOS_LEVEL < 2) break;
                if(!CTRL) break;
                goto do_up;
            case(UP_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_up:
                if(i <= 0) i = total_find_results-1;
                else i--;
                break;
            case('n'):
                if(GNU_DOS_LEVEL < 2) break;
                if(!CTRL) break;
                goto do_down;
            case(DOWN_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_down:
                if(i >= total_find_results-1) i = 0;
                else i++;
                break;
            case('g'):
                if(GNU_DOS_LEVEL < 3) break;
                if(!CTRL) break;
                goto do_esc;
            case(ESC_KEY):
                if(GNU_DOS_LEVEL > 2) break;
do_esc:
                goto finish;
                break;
            case(SPACE_KEY):
            case(ENTER_KEY):
                _replace(i, f, r);
                if(i >= total_find_results) i--;
                if(total_find_results <= 0) goto finish;
                break;
            case('a'):
                _replace(-1, f, r);
                total_find_results = 0;
                goto finish;
                break;
            default:
                goto get_key;
                break;
        }
    }
    
finish:
    free(f);
    free(r);
    refreshView();
    return;
    
//memerr:
//    msgBox("Insufficient memory!", OK, ERROR);
}

/*******************************************
 * Function is called by minoMenu_Replace()
 * to replace find result number 'pos' with
 * the replacement text.
 * *****************************************/
void _replace(int pos, char *f, char *r)
{
    //pass position to replace in find_result_pos[] array,
    //or -1 to replace all find results.
    int i = strlen(f);
    int j = strlen(r);
  
    int old_firstVisLine = firstVisLine;
    int old_selectedLine = selectedLine;
    int old_selectedChar = selectedChar;
    int old_selectedCharCarry = selectedCharCarry;

    if(pos >= 0) 
    {
        _do_replace(pos, f, r);
        FILE_STATE = MODIFIED;
    }
    else
    {	//-1 means replace all
        if(i == j)
        {
            for(pos = 0; pos < total_find_results; pos++)
            {
                _do_replace(pos, f, r);
            }
        }
        else
        {
            while(total_find_results)
            {
                _do_replace(0, f, r);
            }
        }
    }
    firstVisLine = old_firstVisLine;
    selectedLine = old_selectedLine;
    selectedChar = old_selectedChar;
    selectedCharCarry = old_selectedCharCarry;
}

void _find(char *f)
{
    int i, k = 0;
    char *j;
    int flen = strlen(f);
    total_find_results = 0;
    for(i = 0; i < totalLines; i++)
    {
        char *line = lines[i]->text;
        while((j = strcasestr(line, f)))
        {
            find_result_pos[k].nline = i; 
            find_result_pos[k++].nchar = (j-lines[i]->text);
            total_find_results++;
            line = j+flen;
        }
    }
}

void _do_replace(int pos, char *f, char *r)
{
    int i = strlen(f);
    int j = strlen(r);
    int k, l, n, m;
    k = find_result_pos[pos].nchar;
    l = find_result_pos[pos].nline;
    n = 0;
    firstVisLine = l;
    selectedLine = 0;
    selectedChar = k;
    k = charsToBytes(l, selectedChar);
    // add the original text to the undo buffer
    char *f2 = f;
    while(*f2)
    {
        undoAddChar(UNDO_ACTION_REPLACE, l, k, *f2);
        // check for UTF-8 continuation sequence
        while(++k, (*(++f2) & 0xc0) == 0x80)
            undoAddChar(UNDO_ACTION_REPLACE, l, k, *f2);
    }
    k = charsToBytes(l, selectedChar);
    // then add the replacement text
    f2 = r;
    char c;
    while((c = *f2++)) undoBufRep[undoBufRepIndex++] = c;
    
    if(i == j)
    {
        /*
         * find & replace of same length
         */
        memcpy(lines[l]->text+k, r, j);
    }
    else if(i > j)
    {
        /*
         * find is longer than replace
         */
        memcpy(lines[l]->text+k, r, j);
        copyInLine(l, k+j, k+i, 0);
    }
    else
    {
        /*
         * replace is longer than find
         */
        if(!extendLineText(l, lines[l]->byteCount+j-i+1))
        {
            msgBox("Insufficient memory!", OK, ERROR);
            return;
        }
        copyInLine(l, k+j, k+i, 0);
        memcpy(lines[l]->text+k, r, j);
    }
    FILE_STATE = MODIFIED;
    checkLineBounds(l);
    /* we will need to update search results */
    _find(f);
}

void calcTotalCharsInLineC(int pos, int *carry)
{
    int totalCharsInLine = 0;
    *carry = 0;
    int k;
    char *c = lines[pos]->text;
    while(*c)
    {
        //char c = lines[pos]->text[j];
        // check it is not a UTF-8 continuation sequence
        if((*c & 0xc0) == 0x80) continue;
        if(*c == '\r' || *c == '\n' || *c == '\0') break;
        if(*c == '\t')
        {
            k = TABSPACES(totalCharsInLine+(*carry)+1);
            //totalCharsInLine += k;
            (*carry) += k;
        }
        //else totalCharsInLine++;
        totalCharsInLine++;
        c++;
    }
    lines[pos]->charCount = totalCharsInLine;
    lines[pos]->byteCount = c-lines[pos]->text;
}

void calcTotalCharsInLine(int pos)
{
    int carry = 0;
    calcTotalCharsInLineC(pos, &carry);
}

int utfstrlen(char *str)
{
    char c;
    int count = 0;
    while((c = *str++))
    {
        if((c & 0xc0) != 0x80) count++;
    }
    return count;
}
