// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccscript.
// 
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <cc++/config.h>
#include <cc++/misc.h>
#include <cc++/slog.h>
#include <cc++/process.h>
#include <cc++/strchar.h>
#include <cstdlib>
#include <sys/wait.h>
#include <iostream>
#include "script.h"
#include "config.h"

#ifndef	WEXITSTATUS
#define	WEXITSTATUS(x)	(x)
#endif

#ifdef	CCXX_NAMESPACES
using namespace ost;
#endif

static char *sigtype[2] = {NULL, NULL};
static bool exited = false;
static int oddeven = 0;

class shScript : public ScriptCommand
{
public:
	inline int symsize(void)
		{return 64;};

	inline int pagesize(void)
		{return 256;};

	shScript();
} _image;

class shInterp : public ScriptInterp
{
private:
	friend class shScript;

	bool scrEcho(void);
	bool scrSystem(void);
	bool scrAlarm(void);
	bool scrToken(void);

	void exit(void)
		{std::cout << "exiting..." << std::endl; std::exit(0);};

public:
	void signal(const char *sigid)
		{ScriptInterp::signal(sigid);};
	shInterp();
} interp;

class shImage : public ScriptImage
{
private:
	Name *getScript(const char *name);

public:
	shImage();
} shImage;

shImage::shImage() :
ScriptImage(&_image, "/ccscript/define")
{
	commit();
};

Script::Name *shImage::getScript(const char *name)
{
	char fname[128];
	Name *line = ScriptImage::getScript(name);
	if(!line)
	{
		strcpy(fname, name);
		strcat(fname, ".scr");
		compile(fname);
		line = ScriptImage::getScript(name);
	}
	return line;
}

shInterp::shInterp() :
ScriptInterp(&_image, _image.symsize(), _image.pagesize())
{
}

bool shInterp::scrEcho(void)
{
	char *val;
	while(NULL != (val = getValue(NULL)))
		std::cout << val;

	std::cout << std::endl;
	advance();
	return true;
}

bool shInterp::scrAlarm(void)
{
	alarm(atoi(getValue("0")));
	advance();
	return true;
}

bool shInterp::scrToken(void)
{
	int argc = 0;
	Line *line = getScript();

	while(argc < line->argc)
		std::cout << "[-" << line->args[argc++] << "-] ";

	std::cout << std::endl;
	advance();
	return true;
}

bool shInterp::scrSystem(void)
{
	char *argv[33];
	int argc = 0;
	char *val;
	int pid, status;

	while(argc < 32 && (NULL != (val = getValue(NULL))))
		argv[argc++] = val;

	argv[argc] = NULL;

	pid = vfork();
	if(!pid)
	{
		execvp(*argv, argv);
		std::exit(-1);
	}
#ifdef	__FreeBSD__
	wait4(pid, &status, 0, NULL);
#else
	waitpid(pid, &status, 0);
#endif
	status = WEXITSTATUS(status);
	if(status)
		error("sys-error");
	else
		advance();
	return false;
}

shScript::shScript() :
ScriptCommand()
{
	static Script::Define interp[] = {
		{"echo", (Method)&shInterp::scrEcho, 
			&ScriptCommand::chkHasArgs},
		{"sys", (Method)&shInterp::scrSystem, 
			&ScriptCommand::chkHasArgs},
		{"alarm", (Method)&shInterp::scrAlarm, 
			&ScriptCommand::chkHasArgs},
		{"token", (Method)&shInterp::scrToken,
			&ScriptCommand::chkIgnore},
		{NULL, NULL, NULL}};

	trap("alarm");
	trap("hangup");

	load(interp);
}

static RETSIGTYPE handler(int signo)
{
	switch(signo)
	{
	case SIGINT:
		if(exited)
			return;
		alarm(0);
		sigtype[oddeven] = "exit";
		break;
	case SIGHUP:
		if(sigtype[oddeven] && stricmp(sigtype[oddeven], "alarm"))
			return;
		alarm(0);
		sigtype[oddeven] = "hangup";
		break;
	case SIGALRM:
		if(!sigtype[oddeven])
			sigtype[oddeven] = "alarm";
		alarm(0);
		break;
	}
}

int main(int argc, char **argv)
{
	char *ext;
	char *v1, *v2, *v3;
	char *r1, *r2, *r3;
	char *ss;
	int stepper;
	char argname[10];
	int argcount = 0;

	if(argc < 2)
		exit(-1);

        if(!stricmp(argv[1], "--version") || !stricmp(argv[1], "-v"))
        {
                std::cout << VERSION << std::endl;
                exit(0);
        }

	if(!stricmp(argv[1], "--requires"))
	{
		v1 = VERSION;
		v2 = strchr(VERSION, '.') + 1;
		v3 = strchr(v2, '.') + 1;
		r1 = argv[2];
		if(!r1)
			exit(-1);
		r2 = strchr(r1, '.');
		if(!r2)
			r2 = "0";
		else
			++r2;
		r3 = strchr(r2, '.');
		if(!r3)
			r3 = "0";
		else
			++r3;
		if(atoi(v1) < atoi(r1))
			exit(-1);
		if(atoi(v1) > atoi(r1))
			exit(0);
		if(atoi(v2) < atoi(r2))
			exit(-1);
		if(atoi(v2) > atoi(r2))
			exit(0);
		if(atoi(v3) < atoi(r3))
			exit(-1);
		exit(0);
	}

	if(argc != 2)
	{
		std::cerr << "use: ccscript scriptname" << std::endl;
		exit(-1);
	}

	ext = strrchr(argv[1], '.');
	if(ext)
		if(!stricmp(ext, ".scr"))
			*ext = 0;

	interp.attach(argv[1]);
	interp.autoloop(false);

	while(argcount < argc)
	{
		snprintf(argname, sizeof(argname), "argv.%d", argcount);
		interp.setConst(argname, argv[argcount++]);
	}
	snprintf(argname, sizeof(argname), "%d", argcount);
	interp.setConst("argv.count", argname);		

	Process::setPosixSignal(SIGINT, &handler);
	Process::setPosixSignal(SIGHUP, &handler);
	Process::setPosixSignal(SIGALRM, &handler);

	for(;;)
	{
		stepper = oddeven;
		if(oddeven)
			oddeven = 0;
		else
			++oddeven;
		if(sigtype[stepper])
		{
			interp.signal(sigtype[stepper]);
			sigtype[stepper] = NULL;
		}
		interp.step();
	}

	exit(-1);
}

