#!/usr/bin/env python

##############################################################################
##
# This file is part of Sardana
##
# http://www.sardana-controls.org/
##
# Copyright 2011 CELLS / ALBA Synchrotron, Bellaterra, Spain
##
# Sardana is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
##
# Sardana is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
##
# You should have received a copy of the GNU Lesser General Public License
# along with Sardana.  If not, see <http://www.gnu.org/licenses/>.
##
##############################################################################

"""Tests for list macros"""

import pytest
from sardana.macroserver.macros.test import (MacroTester, SarDemoEnv)


@pytest.mark.parametrize("macro_name,macro_params,wait_timeout,elem_type",
    [
        ("lsm", [], 1, "moveable"),
        ("lsm", ["l.*"], 1, "moveable"),
        ("lspm", [], 1, "pseudomotor"),
        ("lsctrl", [], 1, "controller"),
        ("lsct", [], 1, "ctexpchannel"),
        ("ls0d", [], 1, "zerodexpchannel"),
        ("ls1d", [], 1, "onedexpchannel"),
        ("ls2d", [], 1, "twodexpchannel"),
    ]
)
def test_list_output(create_sar_demo, macro_executor, macro_name, macro_params,
                     wait_timeout, elem_type):
    """Test macros used to list elements.
        
    Use the lists of elem_type generated by :class:`.SarDemoEnv` as
    reference for compare with the output of the tested ls macro.
    """
    tester = MacroTester(macro_executor)
    tester.macro_runs(
        macro_name=macro_name,
        macro_params=macro_params,
        wait_timeout=wait_timeout,
    )
    log_output = tester.macro_executor.getLog("output")
    msg = "generic ls macro does not contain elements"
    assert len(log_output) > 0, msg

    list_sardemo = SarDemoEnv().getElements(elem_type)

    # parsing log output to get all elements
    header_rows = 2
    names_column_index = 0
    macro_output = []
    for row, in log_output[header_rows:]:
        macro_output.append(row.split()[names_column_index])
    
    def check_elements(list1, list2):
        """ A helper method to evaluate if all elements of list1 are in list2.
           :params list1: (seq<str>) List of elements to evaluate.
           :params list2: (seq<str>) List of elements for validate.
        """
        for elem in list1:
            msg = "{0} does not contain {1}".format(macro_name, elem)
            assert elem in list2, msg

    # Evaluate if element of A are in B
    if len(macro_output) >= len(list_sardemo):
        check_elements(list_sardemo, macro_output)
    else:
        check_elements(macro_output, list_sardemo)
