semver 0.40.1
=============

Synopsis
--------

```pgsql
=% CREATE EXTENSION semver;
CREATE EXTENSION

=% SELECT '1.2.1'::semver;
 semver
--------
 1.2.1

=% SELECT '1.2.0'::semver > '1.2.0-b1'::semver;
 ?column?
----------
 t
```

Description
-----------

This library contains a single PostgreSQL extension, a semantic version data
type called `semver`. It's an implementation of the version number format
specified by the [Semantic Versioning 2.0.0 Specification].

The salient points of [the spec], for the purposes of a data type and
comparison operators, are:

1. A normal version number MUST take the form X.Y.Z where X, Y, and Z are
   non-negative integers, and MUST NOT contain leading zeroes. X is the major
   version, Y is the minor version, and Z is the patch version. Each element
   MUST increase numerically. For instance: `1.9.0 < 1.10.0 < 1.11.0`.

2. A pre-release version MAY be denoted by appending a hyphen and a series
   of dot separated identifiers immediately following the patch version.
   Identifiers MUST comprise only ASCII alphanumerics and hyphen `[0-9A-Za-z-]`.
   Identifiers MUST NOT be empty. Numeric identifiers MUST NOT include leading
   zeroes. Pre-release versions have a lower precedence than the associated
   normal version. A pre-release version indicates that the version is
   unstable and might not satisfy the intended compatibility requirements
   as denoted by its associated normal version. Examples:
   `1.0.0-alpha, 1.0.0-alpha.1, 1.0.0-0.3.7, 1.0.0-x.7.z.92`.

3. Build metadata MAY be denoted by appending a plus sign and a series of
   dot separated identifiers immediately following the patch or pre-release
   version. Identifiers MUST comprise only ASCII alphanumerics and hyphen
   `[0-9A-Za-z-]`. Identifiers MUST NOT be empty. Build metadata SHOULD be
   ignored when determining version precedence. Thus two versions that differ
   only in the build metadata, have the same precedence. Examples:
   `1.0.0-alpha+001, 1.0.0+20130313144700, 1.0.0-beta+exp.sha.5114f85`.

4. Precedence refers to how versions are compared to each other when ordered.
   Precedence MUST be calculated by separating the version into major, minor,
   patch and pre-release identifiers in that order (Build metadata does not
   figure into precedence). Precedence is determined by the first difference
   when comparing each of these identifiers from left to right as follows:
   Major, minor, and patch versions are always compared numerically. Example:
   `1.0.0 < 2.0.0 < 2.1.0 < 2.1.1`. When major, minor, and patch are equal, a
   pre-release version has lower precedence than a normal version. Example:
   `1.0.0-alpha < 1.0.0`. Precedence for two pre-release versions with the same
   major, minor, and patch version MUST be determined by comparing each dot
   separated identifier from left to right until a difference is found as
   follows: identifiers consisting of only digits are compared numerically and
   identifiers with letters or hyphens are compared lexically in ASCII sort
   order. Numeric identifiers always have lower precedence than non-numeric
   identifiers. A larger set of pre-release fields has a higher precedence than
   a smaller set, if all of the preceding identifiers are equal. Example:
   `1.0.0-alpha < 1.0.0-alpha.1 < 1.0.0-alpha.beta < 1.0.0-beta < 1.0.0-beta.2 <
   1.0.0-beta.11 < 1.0.0-rc.1 < 1.0.0`.

🚨 v0.40.0 Upgrade Compatibility Warning 🚨
-------------------------------------------

Prior to v0.40.0, the semver extension incorrectly allowed some invalid
prerelease values. *BEFORE YOU UPGRADE*, we strongly recommend that you check
for and repair semvers. Use this query to find semvers with invalid
pre-release values:

```sql
SELECT name, version FROM packages
 WHERE get_semver_prerelease(version) ~ '^0\[0-9]+($|\+)';
```

For any results returned, update them by removing the leading `0` from the
pre-release, or appending an alphabetic value.

🚨 v0.30.0 Upgrade Compatibility Warning 🚨
-------------------------------------------

Prior to v0.30.0, the semver extension incorrectly allowed some invalid
prerelease and build metadata values. Details below, but *BEFORE YOU UPGRADE*
from an earlier version, we strongly recommend that you check for and repair
any invalid semvers. You can find them using the official [SemVer regular
expression] like so (replace `name`, `version`, and `packages` as appropriate
for your database):

```sql
SELECT name, version FROM packages
WHERE version::text !~ '^(0|[1-9]\d*)\.(0|[1-9]\d*)\.(0|[1-9]\d*)(?:-((?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\.(?:0|[1-9]\d*|\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\+([0-9a-zA-Z-]+(?:\.[0-9a-zA-Z-]+)*))?$';
```

If no rows are returned, you should be good to go. If there are results, here are
Examples of invalid semantic versions and how they should be repaired.

```text
 Invalid             Valid SemVer
-----------         ----------------------------
1.0.0-02799    ->   1.0.0-2799
1.0.0-0.02     ->   1.0.0-0.2
1.0.0-.20      ->   1.0.0-0.20
1.0.0+0+20     ->   1.0.0+0-20  or  1.0.0+0.20
1.0.0+.af      ->   1.0.0+0.af  or  1.0.0+af
```

Usage
-----

Add the extension to a database:

```sql
CREATE EXTENSION semver;
```

Now, use it like any other data type. Here's an example in a table:

```sql
CREATE TABLE extensions (
    name        TEXT,
    version     SEMVER,
    description TEXT,
    PRIMARY KEY (name, version)
);
```

The type can be in indexed using btree or hash indexes:

```sql
CREATE INDEX idx_extension_version ON extensions(version);
CREATE INDEX hdx_extension_version ON extensions USING hash (version);
```

Hash indexes aren't worth much, but the functionality is there to support hash
aggregates in query optimizations.

And some sample usage:

```pgsql
=% INSERT INTO extensions
-% VALUES ('pgtap', '0.35.0',    'PostgreSQL unit testing'),
          ('pgtap', '0.35.0-b1', 'PostgreSQL unit testing.'),
          ('pair',  '0.1.0',     'Key/value pair data type'),
          ('PostGIS', '1.5.0',   'Geolocation data types');

=% SELECT * FROM extensions WHERE VERSION = '1.5.0';
  name   │ version │      description
---------+---------+------------------------
 PostGIS │ 1.5.0   │ Geolocation data types

=% SELECT * FROM extensions WHERE VERSION < '0.35.0';
 name  │ version   │       description
-------+-----------+--------------------------
 pgtap │ 0.35.0-b1 │ PostgreSQL unit testing.
 pair  │ 0.1.0     │ Key/value pair data type
```

Note that "0.35.0-b1" is less than "0.35.0", as required by the specification.
Use `ORDER BY` to get more of a feel for semantic version ordering rules:

```pgsql
=% SELECT version FROM extensions ORDER BY version;
 version
-----------
 0.1.0
 0.35.0-b1
 0.35.0
 1.5.0

=% SELECT version FROM extensions ORDER BY version DESC;
 version
-----------
 1.5.0
 0.35.0
 0.35.0-b1
 0.1.0
```

Interface
---------

### Operators ###

 Operator | Description                               | Example                             | Result
----------|-------------------------------------------|-------------------------------------|--------
 `=`      | Are semvers equivalent                    | '1.2.0'semver = '1.2.00'::semver    | `t`
 `<>`     | Are semvers different                     | '1.2.0'semver <> '1.2.00'::semver   | `f`
 `<`      | Is semver less than right semver          | '3.4.0-b1'semver < '3.4.0'::semver  | `t`
 `<=`     | Is semver less than or equal to semver    | '3.4.0-b1'semver <= '3.4.0'::semver | `t`
 `>`      | Is semver greater than right semver       | '3.4.0-b1'semver > '3.4.0'::semver  | `f`
 `>=`     | Is semver greater than or equal to semver | '3.4.0-b1'semver >= '3.4.0'::semver | `f`

### Functions ###

 Function                        | Description                     | Example                                   | Result
---------------------------------|---------------------------------|-------------------------------------------|----------
 `to_semver(text)`               | Parse semver from text          | `to_semver('1.02')`                       | `1.2.0`
 `is_semver(text)`               | Test semver text                | `is_semver('1.2.0')`                       | true
 `semver(text)`                  | Cast text to semver             | `semver('1.2.1')`                         | `1.2.1`
 `semver(numeric)`               | Cast numeric to semver          | `semver(1.2)`                             | `1.2.0`
 `semver(real)`                  | Cast real to semver             | `semver(12.0::real)`                      | `12.0.0`
 `semver(double precision)`      | Cast double precision to semver | `semver(9.2::double precision)`           | `9.2.0`
 `semver(integer)`               | Cast integer to semver          | `semver(42::integer)`                     | `42.0.0`
 `semver(bigint)`                | Cast bigint to semver           | `semver(19::bigint)`                      | `19.0.0`
 `semver(smallint)`              | Cast smallint to semver         | `semver(2::smallint)`                     | `2.0.0`
 `text(semver)`                  | Cast semver to text             | `text('1.2.54'::semver)`                  | `1.2.54`
 `get_semver_major(semver)`      | Get major version part          | `get_semver_major('4.2.0')`               | `4`
 `get_semver_minor(semver)`      | Get minor version part          | `get_semver_minor('4.2.0')`               | `2`
 `get_semver_patch(semver)`      | Get patch version part          | `get_semver_patch('4.2.0')`               | `0`
 `get_semver_prerelease(semver)` | Get prerelease version part     | `get_semver_prerelease('2.1.0-b2+bfb13')` | `b2`

Numeric casts simply extract an integer from the decimal portion, so that `1.20`
and `1.02` would both be parsed as `1.2.0` (but their string equivalents would
not).

The difference between `semver()` and `to_semver()` is that the former
requires a valid semver format, while the latter is a bit more permissive,
doing its best to convert other version number formats (including the older
[semver 1.0.0-beta] prerelease format) to semantic versions:

```pgsql
=% select to_semver('1.0');
 to_semver
-----------
 1.0.0

=% select to_semver('1.0beta1');
 to_semver
-----------
 1.0.0-beta1
```

As for `is_semver()`, it returns true for a valid semver format, and false for
anything else, including formats that `semver()` would convert to valid
semvers. In other words, its interpretation of validity is strict.

And finally, the `get_semver_*` functions all return integers except for
`get_semver_prerelease()`, which returns text.

### Aggregate Functions ###

The examples assume the values inserted into the `extensions` table in the above
examples.

 Function      | Return Type | Description               | Example                                | Result
---------------|-------------|---------------------------|----------------------------------------|--------
 `MIN(semver)` |  `semver`   | Return the lowest semver  | `SELECT MIN(version) FROM extensions;` | `0.1.0`
 `MAX(semver)` |  `semver`   | Return the highest semver | `SELECT MAX(version) FROM extensions;` | `1.5.0`

### Casts ###

 From             | To     | Example                         | Result
------------------|--------|---------------------------------|---------
 text             | semver | `'1.2.1'::semver`               | `1.2.1`
 numeric          | semver | `1.2::semver`                   | `1.2.0`
 real             | semver | `12.0::real::semver`            | `12.0.0`
 double precision | semver | `9.2::double precision::semver` | `9.2.0`
 integer          | semver | `42::integer::semver`           | `42.0.0`
 bigint           | semver | `19::bigint::semver`            | `19.0.0`
 smallint         | semver | `2::smallint::semver`           | `2.0.0`
 semver           | text   | `'1.2.54'::semver::text`        | `1.2.54`

Note that numeric casts simply extract an integer from the decimal portion, so
that `1.20` and `1.02` would both be parsed as `1.2.0`  (but their string
equivalents would not).

### Range Type ###

As of v0.20.0, the semver extension includes the `semverrange` type, which
simply builds on the [range type] support on PostgreSQL 9.2 and higher. This
allows for easy specification of ranges of semantic versions. Some examples:

 Range                 | Description
-----------------------|-----------------------------------
 `['1.0.0', '2.0.0']`  | 1.0.0 inclusive - 2.0.0 inclusive
 `['1.0.0', '2.0.0')`  | 1.0.0 inclusive - 2.0.0 exclusive
 `('1.0.0', '2.0.0')`  | 1.0.0 exclusive - 2.0.0 exclusive
 `['1.0.0',]`.         | 1.0.0 inclusive - infinity

The cool thing is that you can use any of the [range operators], including the
"contains" operators: For example, to see if `1.0.5` falls falls within the
range `1.0.0` - `2.0.0` exclusive, run a query like this:

```pgsql
=% SELECT '1.0.5'::semver <@ '[1.0.0, 2.0.0)'::semverrange;
 ?column?
----------
 t
```

The `semverrange` constructor will build the same range,

```pgsql
=% SELECT semverrange('1.0.0', '2.0.0') @> '2.0.0'::semver;
 ?column?
----------
 f
=% SELECT semverrange('1.0.0', '2.0.0') @> '1.9999.9999'::semver;
 ?column?
----------
 t
```

Pass the optional third argument to determine the bounds inclusiveness:

```pgsql
=% SELECT semverrange('1.0.0', '2.0.0', '[]') @> '2.0.0'::semver;
 ?column?
----------
 t
```

It works for unlimited bound, as well. For example, this query ensure that
a semver is greater than or equal `1.0.0`:

```pgsql
=% SELECT '1000.0.0'::semver <@ '[1.0.0,]'::semverrange;
 ?column?
----------
 t
```

If you need to omit some values, you can use an array of semverrange values.
For example, say you want to require a version greater than `1.0.0` and less
than `2.0.0`, but versions `1.2.3` and `1.4.5` have such serious bugs that you
don't want to include them. We create three ranges that use exclusive bounds
to omit those versions, like so:

```sql
SELECT '{"(1.0.0,1.2.3)", "(1.2.3,1.4.5)", "(1.4.5,2.0.0)"}'::semverrange[];
```

Here's an sample how to query such an array of semverranges.

```pgsql
=% SELECT version, version <@ ANY(
-%     '{"(1.0.0,1.2.3)", "(1.2.3,1.4.5)", "(1.4.5,2.0.0)"}'::semverrange[]
-% ) AS valid FROM (VALUES
-%     ('1.0.0'::semver), ('1.0.1'), ('1.2.3'), ('1.2.4'), ('1.4.4'), ('1.4.5'),
-%     ('1.7.0'), ('2.0.0')
-% ) AS v(version)
 version | valid
---------+-------
 1.0.0   | f
 1.0.1   | t
 1.2.3   | f
 1.2.4   | t
 1.4.4   | t
 1.4.5   | f
 1.7.0   | t
 2.0.0   | f
```

Support
-------

The source code for pgTAP is available on [GitHub]. Please feel free to fork
and contribute! Please file bug reports via [GitHub Issues].

Authors
-------

* [David E. Wheeler](https://justatheory.com/)
* [Sam Vilain](http://sam.vilain.net/)
* [Tom Davis](https://github.com/tdavis)
* [Xavier Caron](https://github.com/maspalio)

Copyright and License
---------------------

Copyright (c) 2010-2025 The pg-semver Maintainers: David E. Wheeler, Sam
Vilain, Tom Davis, and Xavier Caron.

This module is free software; you can redistribute it and/or modify it under
the [PostgreSQL License].

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose, without fee, and without a written agreement is
hereby granted, provided that the above copyright notice and this paragraph
and the following two paragraphs appear in all copies.

In no event shall The pg-semver Maintainers be liable to any party for direct,
indirect, special, incidental, or consequential damages, including lost
profits, arising out of the use of this software and its documentation, even
if The pg-semver Maintainers have been advised of the possibility of such
damage.

The pg-semver Maintainers specifically disclaim any warranties, including, but
not limited to, the implied warranties of merchantability and fitness for a
particular purpose. The software provided hereunder is on an "as is" basis,
and The pg-semver Maintainers no obligations to provide maintenance, support,
updates, enhancements, or modifications.

  [Semantic Versioning 2.0.0 Specification]: https://semver.org/spec/v2.0.0.html
  [the spec]: https://semver.org/
  [SemVer regular expression]: https://regex101.com/r/vkijKf/1/
  [semver 1.0.0-beta]: https://semver.org/spec/v1.0.0-beta.html
  [range type]: https://www.postgresql.org/docs/current/static/rangetypes.html
  [range operators]: https://www.postgresql.org/docs/current/static/functions-range.html
  [GitHub]: https://github.com/theory/pgtap/
  [GitHub Issues]: https://github.com/theory/pg-semver/issues/
  [PostgreSQL License]: https://www.opensource.org/licenses/postgresql
