/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Compc      eqc             Equal constant
      Compc      nec             Not equal constant
      Compc      lec             Less equal constant
      Compc      ltc             Less then constant
      Compc      gec             Greater equal constant
      Compc      gtc             Greater then constant
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_vlist.h"
#include "param_conversion.h"

void *
Compc(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  const auto EQC = cdoOperatorAdd("eqc", 0, 0, nullptr);
  const auto NEC = cdoOperatorAdd("nec", 0, 0, nullptr);
  const auto LEC = cdoOperatorAdd("lec", 0, 0, nullptr);
  const auto LTC = cdoOperatorAdd("ltc", 0, 0, nullptr);
  const auto GEC = cdoOperatorAdd("gec", 0, 0, nullptr);
  const auto GTC = cdoOperatorAdd("gtc", 0, 0, nullptr);

  const auto operatorID = cdoOperatorID();

  operatorInputArg("constant value");
  const auto rc = parameter2double(cdoOperatorArgv(0));

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  nospec(vlistID1);

  VarList varList;
  varListInit(varList, vlistID1);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);
  Varray<double> array1(gridsizemax), array2(gridsizemax);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, array1.data(), &nmiss);

          const auto missval = varList[varID].missval;
          const auto gridsize = varList[varID].gridsize;
          const auto datatype = varList[varID].datatype;
          const double rcv = (datatype == CDI_DATATYPE_FLT32) ? (float) rc : rc;

          const bool rc_is_missval = DBL_IS_EQUAL(rc, missval);

          if (nmiss > 0) cdo_check_missval(missval, varList[varID].name);

          if (operatorID == EQC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : IS_EQUAL(array1[i], rcv);
            }
          else if (operatorID == NEC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : IS_NOT_EQUAL(array1[i], rcv);
            }
          else if (operatorID == LEC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : array1[i] <= rcv;
            }
          else if (operatorID == LTC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : array1[i] < rcv;
            }
          else if (operatorID == GEC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : array1[i] >= rcv;
            }
          else if (operatorID == GTC)
            {
              for (size_t i = 0; i < gridsize; i++)
                array2[i] = DBL_IS_EQUAL(array1[i], missval) || rc_is_missval ? missval : array1[i] > rcv;
            }
          else
            {
              cdoAbort("Operator not implemented!");
            }

          nmiss = varrayNumMV(gridsize, array2, missval);
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
