# DeviceDetector

[![CI](https://github.com/podigee/device_detector/workflows/CI/badge.svg)](https://github.com/podigee/device_detector/actions)

DeviceDetector is a precise and fast user agent parser and device detector written in Ruby, backed by the largest and most up-to-date user agent database.

DeviceDetector will parse any user agent and detect the browser, operating system, device used (desktop, tablet, mobile, tv, cars, console, etc.), brand and model. DeviceDetector detects thousands of user agent strings, even from rare and obscure browsers and devices.

The DeviceDetector is optimized for speed of detection, by providing optimized code and in-memory caching.

DeviceDetector is funded by the owners of [Podigee the podcast hosting, analytics & monetization SaaS for podcasters big and small.](https://www.podigee.com) and actively maintained by the development team.

This project originated as a Ruby port of the Universal Device Detection library written and maintained by Matomo Analytics.
You can find the original code here: [https://github.com/piwik/device-detector](https://github.com/matomo-org/device-detector).

## Disclaimer

This port does not aspire to be a one-to-one copy from the original code, but rather an adaptation for the Ruby language.

Still, our goal is to use the original, unchanged regex yaml files for user agent detection provided by the upstream version, in order to mutually benefit from updates and pull request to both the original and the ported versions.

## Installation

Add this line to your application's Gemfile:

```ruby
gem 'device_detector'
```

And then execute:

    $ bundle

Or install it yourself as:

    $ gem install device_detector

## Usage

```ruby
user_agent = 'Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.17 Safari/537.36'
client = DeviceDetector.new(user_agent)

client.name # => 'Chrome'
client.full_version # => '30.0.1599.69'

client.os_name # => 'Windows'
client.os_full_version # => '8'

# For many devices, you can also query the device name (usually the model name)
client.device_name # => 'iPhone 5'
# Device types can be one of the following: desktop, smartphone, tablet,
# feature phone, console, tv, car browser, smart display, camera,
# portable media player, phablet, smart speaker, wearable, peripheral
client.device_type # => 'smartphone'
```

`DeviceDetector` will return `nil` on all attributes, if the `user_agent` is unknown.
You can make a check to ensure the client has been detected:

```ruby
client.known? # => will return false if user_agent is unknown
```

### Using Client hint

Optionally `DeviceDetector` is using the content of `Sec-CH-UA` stored in the headers to improve the accuracy of the detection :

```ruby
user_agent = 'Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.17 Safari/537.36'
headers = {"Sec-CH-UA"=>'"Chromium";v="106", "Brave";v="106", "Not;A=Brand";v="99"'}
client = DeviceDetector.new(user_agent, headers)

client.name # => 'Brave'
```

Same goes with `http-x-requested-with`/`x-requested-with` :

```ruby
user_agent = 'Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.17 Safari/537.36'
headers = {"http-x-requested-with"=>"org.mozilla.focus"}
client = DeviceDetector.new(user_agent, headers)

client.name # => 'Firefox Focus'
```

### Memory cache

`DeviceDetector` will cache up 5,000 user agent strings to boost parsing performance.
You can tune the amount of keys that will get saved in the cache. You have to call this code **before** you initialize the Detector.

```ruby
DeviceDetector.configure do |config|
  config.max_cache_keys = 5_000 # increment this if you have enough RAM, proceed with care
end
```

If you have a Rails application, you can create an initializer, for example `config/initializers/device_detector.rb`.

## Benchmarks

We have measured the parsing speed of almost 200,000 non-unique user agent strings and compared the speed of DeviceDetector with the two most popular user agent parsers in the Ruby community, Browser and UserAgent.

### Testing machine specs

- MacBook Pro 15", Late 2013
- 2.6 GHz Intel Core i7
- 16 GB 1600 MHz DDR3

### Gem versions

- DeviceDetector - 0.5.1
- Browser - 0.8.0
- UserAgent - 0.13.1

### Code

```ruby
require 'device_detector'
require 'browser'
require 'user_agent'
require 'benchmark'

user_agent_strings = File.read('./tmp/user_agent_strings.txt').split("\n")

## Benchmarks

Benchmark.bm(15) do |x|
  x.report('device_detector') {
    user_agent_strings.each { |uas| DeviceDetector.new(uas).name }
  }
  x.report('browser') {
    user_agent_strings.each { |uas| Browser.new(ua: uas).name }
  }
  x.report('useragent') {
    user_agent_strings.each { |uas| UserAgent.parse(uas).browser }
  }
end
```

### Results

```
                      user     system      total        real
device_detector   0.925603   0.020928   0.946531 (  0.948750)
browser           3.555590   0.014877   3.570467 (  3.575951)
useragent         3.264011   0.032327   3.296338 (  3.301772)

                      user     system      total        real
device_detector   0.915578   0.021295   0.936873 (  0.938698)
browser           3.600674   0.022312   3.622986 (  3.635185)
useragent         3.425424   0.054298   3.479722 (  3.500929)

                      user     system      total        real
device_detector   0.916608   0.022739   0.939347 (  0.940993)
browser           3.624404   0.027489   3.651893 (  3.667863)
useragent         3.442139   0.054376   3.496515 (  3.519747)
```

## Detectable clients, bots and devices

Updated on 2024-06-27

### Bots

2GDPR, 2ip, 360 Monitoring, 360JK, 360Spider, Abonti, Aboundexbot, AccompanyBot, Acoon, AdAuth, Adbeat, AddThis.com, ADMantX, ADmantX Service Fetcher, Adsbot, Adscanner, AdsTxtCrawler, AdsTxtCrawlerTP, adstxtlab.com, Aegis, aHrefs Bot, AhrefsSiteAudit, aiHitBot, Alexa Crawler, Alexa Site Audit, Allloadin Favicon Bot, AlltheWeb, Amazon AdBot, Amazon Bot, Amazon ELB, Amazon Route53 Health Check, Amorank Spider, Analytics SEO Crawler, Ant, Anthropic AI, ApacheBench, Applebot, AppSignalBot, Arachni, archive.org bot, ArchiveBot, ArchiveBox, Arquivo.pt, ARSNova Filter System, Asana, Ask Jeeves, AspiegelBot, Automattic Analytics, Awario, Backlink-Check.de, BacklinkCrawler, BackupLand, Baidu Spider, Barkrowler, Barracuda Sentinel, BazQux Reader, BBC Forge URL Monitor, BBC Page Monitor, BDCbot, BDFetch, Better Uptime Bot, BingBot, Birdcrawlerbot, BitlyBot, BitSight, Blekkobot, BLEXBot Crawler, Bloglovin, Blogtrottr, BoardReader, BoardReader Blog Indexer, Botify, Bountii Bot, BrandVerity, BrightBot, BrightEdge, Browsershots, BUbiNG, Buck, BuiltWith, Butterfly Robot, Bytespider, CareerBot, Castopod, Castro 2, Catchpoint, CATExplorador, ccBot crawler, CensysInspect, Charlotte, Chartable, ChatGPT, CheckHost, CheckMark Network, Choosito, Chrome Privacy Preserving Prefetch Proxy, Cincraw, CISPA Web Analyzer, ClaudeBot, Clickagy, Cliqzbot, CloudFlare Always Online, CloudFlare AMP Fetcher, Cloudflare Custom Hostname Verification, Cloudflare Diagnostics, Cloudflare Health Checks, Cloudflare Observatory, Cloudflare Security Insights, Cloudflare Smart Transit, Cloudflare SSL Detector, Cloudflare Traffic Manager, Cocolyzebot, Cohere AI, Collectd, colly, CommaFeed, COMODO DCV, Comscore, ContentKing, Cookiebot, Crawldad, Crawlson, Crawly Project, CriteoBot, CrowdTangle, CSS Certificate Spider, CSSCheck, Cyberscan, Cypress, Cốc Cốc Bot, DaspeedBot, Datadog Agent, DataForSeoBot, datagnionbot, Datanyze, Dataprovider, DataXu, Daum, Dazoobot, Deep SEARCH 9, Deepfield Genome, deepnoc, DepSpid, Detectify, Diffbot, Discobot, Discord Bot, Disqus, DNSResearchBot, Domain Re-Animator Bot, Domain Research Project, DomainAppender, DomainCrawler, Domains Project, DomainStatsBot, DomCop Bot, DotBot, Dotcom Monitor, Dubbotbot, DuckDuckGo Bot, ducks.party, DynatraceSynthetic, Easou Spider, eCairn-Grabber, EFF Do Not Track Verifier, Elastic Synthetics, EMail Exractor, EmailWolf, Embedly, Entfer, evc-batch, Everyfeed, ExaBot, ExactSeek Crawler, Example3, Exchange check, Expanse, EyeMonit, eZ Publish Link Validator, Ezgif, Ezooms, Facebook Crawler, FacebookBot, Faveeo, Feed Wrangler, Feedbin, FeedBurner, Feedly, Feedspot, Femtosearch, Fever, Findxbot, Flipboard, FontRadar, fragFINN, FreeWebMonitoring, FreshRSS, Gaisbot, GDNP, GeedoBot, GeedoProductSearch, Generic Bot, Genieo Web filter, Ghost Inspector, Gigablast, Gigabot, GitCrawlerBot, GitHubCopilotChat, Gluten Free Crawler, Gmail Image Proxy, Gobuster, Goo, Google Apps Script, Google Area 120 Privacy Policy Fetcher, Google Cloud Scheduler, Google Favicon, Google PageSpeed Insights, Google Partner Monitoring, Google Search Console, Google Stackdriver Monitoring, Google StoreBot, Google Structured Data Testing Tool, Google Transparency Report, Googlebot, Googlebot News, Gowikibot, Gozle, GPTBot, Grammarly, Grapeshot, Gregarius, GTmetrix, GumGum Verity, hackermention, Hatena Bookmark, Hatena Favicon, Headline, Heart Rails Capture, Heritrix, Heureka Feed, htmlyse, HTTPMon, httpx, HuaweiWebCatBot, HubPages, HubSpot, ICC-Crawler, ichiro, IDG/IT, Iframely, IIS Site Analysis, ImageSift, Inetdex Bot, Infegy, InfoTigerBot, Inktomi Slurp, inoreader, InsytfulBot, Intelligence X, Interactsh, InternetMeasurement, IONOS Crawler, IP-Guide Crawler, IPIP, IPS Agent, IsItWP, iTMS, JobboerseBot, JungleKeyThumbnail, K6, KadoBot, Kaspersky, KeyCDN Tools, Keys.so, Kiwi TCMS GitOps, KlarnaBot, KomodiaBot, Konturbot, Kouio, Kozmonavt, l9explore, l9tcpid, LAC IA Harvester, Larbin web crawler, LastMod Bot, LCC, LeakIX, Let's Encrypt Validation, LetSearch, Lighthouse, Linespider, Linkdex Bot, LinkedIn Bot, LinkpadBot, LinkPreview, LinkWalker, LiveJournal, LTX71, Lumar, LumtelBot, Lycos, MaCoCu, MADBbot, Magpie-Crawler, MagpieRSS, Mail.Ru Bot, MariaDB/MySQL Knowledge Base, masscan, masscan-ng, Mastodon Bot, Matomo, Meanpath Bot, Mediatoolkit Bot, MegaIndex, MeltwaterNews, MetaInspector, MetaJobBot, MicroAdBot, Microsoft Preview, Mixnode, Mixrank Bot, MJ12 Bot, Mnogosearch, MojeekBot, Monitor Backlinks, Monitor.Us, Monsidobot, Montastic Monitor, MoodleBot Linkchecker, Morningscore Bot, MTRobot, Munin, MuscatFerret, Nagios check_http, NalezenCzBot, nbertaupete95, Neevabot, Netcraft Survey Bot, netEstate, NetLyzer FastProbe, Netpeak Checker, NetResearchServer, NetSystemsResearch, NetTrack, Netvibes, NETZZAPPEN, NewsBlur, NewsGator, Newslitbot, NiceCrawler, Nimbostratus Bot, NLCrawler, Nmap, NodePing, Notify Ninja, Nutch-based Bot, Nuzzel, oBot, Octopus, Odin, Odnoklassniki Bot, Omgili bot, OmtrBot, Onalytica, OnlineOrNot Bot, Openindex Spider, OpenLinkProfiler, OpenWebSpider, Orange Bot, OSZKbot, Outbrain, Overcast Podcast Sync, OWLer, Page Modified Pinger, Pageburst, PagePeeker, PageThing, Panscient, PaperLiBot, Paqlebot, parse.ly, PATHspider, PayPal IPN, PDR Labs, PerplexityBot, Petal Bot, Phantomas, PHP Server Monitor, phpMyAdmin, Picsearch bot, PingAdmin.Ru, Pingdom Bot, Pinterest, PiplBot, Plesk Screenshot Service, Plukkie, Pocket, Podroll Analyzer, PodUptime, Pompos, PritTorrent, Probely, Project Patchwatch, Project Resonance, Prometheus, PRTG Network Monitor, Punk Map, Quantcast, QuerySeekerSpider, Quora Bot, Quora Link Preview, Qwantify, Rainmeter, RamblerMail Image Proxy, RavenCrawler, Reddit Bot, RenovateBot, Repo Lookout, ReqBin, Research Scan, Riddler, Robozilla, RocketMonitorBot, Rogerbot, ROI Hunter, RSSRadio Bot, RuxitSynthetic, Ryowl, SabsimBot, SafeDNSBot, Sandoba//Crawler, Scamadviser External Hit, Scooter, ScoutJet, Scraping Robot, Scrapy, Screaming Frog SEO Spider, ScreenerBot, Sectigo DCV, security.txt scanserver, Seekport, Sellers.Guide, Semantic Scholar Bot, Semrush Reputation Management, SemrushBot, Sensika Bot, Sentry Bot, Senuto, Seobility, SEOENGBot, SEOkicks, SEOkicks-Robot, seolyt, Seolyt Bot, Seoscanners.net, Serendeputy Bot, Serenety, serpstatbot, Server Density, Seznam Bot, Seznam Email Proxy, Seznam Zbozi.cz, sfFeedReader, ShopAlike, Shopify Partner, ShopWiki, SilverReader, SimplePie, Sirdata, SISTRIX Crawler, SISTRIX Optimizer, Site24x7 Defacement Monitor, Site24x7 Website Monitoring, SiteAuditBot, Sitebulb, Siteimprove, SitemapParser-VIPnytt, SiteOne Crawler, SiteScore, SiteSucker, Sixy.ch, Skype URI Preview, Slackbot, SMTBot, Snap URL Preview Service, Snapchat Ads, Snapchat Proxy, Sogou Spider, Soso Spider, Sparkler, Spawning AI, SpazioDati, Speedy, Spinn3r, SplitSignalBot, Spotify, Sprinklr, Sputnik Bot, Sputnik Favicon Bot, Sputnik Image Bot, sqlmap, SSL Labs, start.me, Startpagina Linkchecker, Statista, StatOnline.ru, StatusCake, Steve Bot, Stract, Sublinq, Substack Content Fetch, Superfeedr Bot, SurdotlyBot, Survey Bot, Swiftbot, Synapse, t3versions, Taboolabot, TactiScout, Tag Inspector, Tarmot Gezgin, tchelebi, TelegramBot, Tenable.asm, TestCrawler, The Knowledge AI, The Trade Desk Content, theoldreader, ThinkChaos, ThousandEyes, TigerBot, TinEye, TinEye Crawler, Tiny Tiny RSS, TLSProbe, TraceMyFile, Trendiction Bot, Trendsmap, Turnitin, TurnitinBot, TweetedTimes Bot, Tweetmeme Bot, Twingly Recon, Twitterbot, Twurly, UCSB Network Measurement, UkrNet Mail Proxy, uMBot, UniversalFeedParser, Uptime-Kuma, Uptimebot, UptimeRobot, Uptimia, URLAppendBot, URLinspector, URLSuMaBot, Vagabondo, Velen Public Web Crawler, Vercel Bot, VeryHip, Viber Url Downloader, VirusTotal Cloud, Visual Site Mapper Crawler, VK Robot, VK Share Button, Vuhuv Bot, W3C CSS Validator, W3C I18N Checker, W3C Link Checker, W3C Markup Validation Service, W3C MobileOK Checker, W3C P3P Validator, W3C Unified Validator, Wappalyzer, WDG HTML Validator, WebbCrawler, WebCEO, WebDataStats, WebMon, Weborama, WebPageTest, WebPros, Website-info, WebSitePulse, WebThumbnail, webtru, Webwiki, WellKnownBot, WeSEE:Search, WeViKaBot, WhatCMS, WhatsMyIP.org, WhereGoes, Who.is Bot, Wibybot, WikiDo, Willow Internet Crawler, WireReaderBot, WooRank, WordPress, Wotbox, XenForo, XoviBot, YaCy, Yahoo Gemini, Yahoo! Cache System, Yahoo! Japan, Yahoo! Japan ASR, Yahoo! Japan BRW, Yahoo! Japan WSC, Yahoo! Link Preview, Yahoo! Mail Proxy, Yahoo! Slurp, YaK, Yandex Bot, Yeti/Naverbot, Yottaa Site Monitor, YouBot, Youdao Bot, Yourls, Yunyun Bot, Zaldamo, Zao, Ze List, Zeno, zgrab, Zookabot, ZoomBot, ZoominfoBot, ZumBot

### Clients

'sodes, +Simple, 115 Browser, 18+ Privacy Browser, 1Password, 2345 Browser, 2tch, 360 Phone Browser, 360 Secure Browser, 7654 Browser, 7Star, ABrowse, Acoo Browser, ActionExtension, AdBlock Browser, Adobe Creative Cloud, Adobe IPM, Adobe NGL, Adobe Synchronizer, Adori, Agora, Aha Radio 2, AIDA64, aiohttp, Airfind Secure Browser, Airmail, Airr, Airsonic, Akka HTTP, Akregator, Alexa, AliExpress, Alipay, All You Can Books, AllHitMusicRadio, Aloha Browser, Aloha Browser Lite, ALVA, Amarok, Amaya, Amazon Fire, Amazon Music, Amazon Shopping, Ameba, Amiga Aweb, Amiga Voyager, Amigo, AN WhatsApp, Anchor, AnchorFM, Android Browser, Android License Verification Library, AndroidDownloadManager, Anghami, ANT Fresco, AntennaPod, ANTGalio, AntiBrowserSpy, Anybox, AnyEvent HTTP, Anytime Podcast Player, AOL Desktop, AOL Explorer, AOL Shield, AOL Shield Pro, Apache HTTP Client, APK Downloader, Aplix, APN Browser, Apollo, Apple iMessage, Apple Mail, Apple News, Apple Podcasts, Apple PubSub, Apple Reminders, Apple TV, Arachne, Arc, Arctic Fox, Aria2, Arora, Artifactory, Arvin, Arvocast, Ask.com, ASUS Updater, Atom, Atomic Web Browser, Audacious, Audacy, Audials, Audible, Audio, Audio Now, Audiobooks, Autoplius.lt, Avant Browser, Avast Secure Browser, AVG Secure Browser, Avid Link, Avira Secure Browser, Awasu, AwoX, Axios, Azure Data Factory, B-Line, Babashka HTTP Client, Background Intelligent Transfer Service, Baidu Box App, Baidu Browser, Baidu Input, Baidu Spark, Ballz, Bang, Bangla Browser, Bank Millenium, Banshee, Barca, Basecamp, BashPodder, Basilisk, BathyScaphe, Battle.net, BB2C, BBC News, Be Focused, Beaker Browser, Beamrise, Bear, Beonex, BetBull, BeyondPod, Bible, Bible KJV, Binance, Bitcoin Core, Bitsboard, Bitwarden, Biyubi, BizBrowser, BlackBerry Browser, Blackboard, BlackHawk, Blitz, Bloket, Blue Browser, Blue Proxy, Bluefy, BlueMail, BlueStacks, Bolt, BonPrix, Bonsai, Bookmobile, Bookshelf, Boom, Boomplay, Borealis Navigator, Bose Music, Bose SoundTouch, Boto3, Boxee, bPod, Brave, Breaker, Breez, Bridge, BriskBard, Broadcast, Broadway Podcast Network, Browlser, BrowseHere, Browser app, Browser-Anonymizer, BrowserPlus, BrowseX, Browzar, Buildah, BuildKit, Bullhorn, Bunjalloo, BuzzVideo, Byffox, C++ REST SDK, CakePHP, Camino, CamScanner, Capital, capsule.fm, CarrierWave, Castamatic, Castaway, CastBox, castero, castget, Castro, Castro 2, Catalyst, CCleaner, Centaury, CGN, ChanjetCloud, Charon, Chedot, Cheetah Browser, Cheshire, Chim Lac, ChMate, Chrome, Chrome Frame, Chrome Mobile, Chrome Mobile iOS, Chrome Update, Chrome Webview, ChromePlus, Chromium, Chromium GOST, Ciisaa, Citrix Workspace, Classic FM, Classilla, Clementine, Client, Clipbox+, Cliqz, Clovia, CM Browser, COAF SMART Citizen, Coast, Coc Coc, Coinbase, Colibri, Colom Browser, Columbus Browser, CometBird, Comodo Dragon, Conkeror, Containerd, containers, CoolBrowser, Cooler, CoolNovo, Copied, Cornowser, Cortana, COS Browser, Cosmicast, Covenant Eyes, cPanel HTTP Client, CPod, cpp-httplib, CPU-Z, Craving Explorer, Crazy Browser, cri-o, CrosswalkApp, Crusta, Crypto.com DeFi Wallet, CSDN, Cunaguaro, curl, Cyberfox, CyBrowser, Damus, Dart, Daum, DAVdroid, dbrowser, De Standaard, De Telegraaf, Decentr, Deepnet Explorer, Deezer, deg-degan, Deledao, Delta Browser, Deno, DeskBrowse, DevCasts, DeviantArt, DIGA, Diigo Browser, Dillo, DingTalk, DIRECTV, Discord, DManager, docker, DoCoMo, Dogecoin Core, DoggCatcher, Dolphin, Don't Waste My Time!, Dooble, Dorado, Dot Browser, douban App, DoubleTwist CloudPlayer, Doughnut, Down, Downcast, Downie, Downloader, Dr. Watson, DStream Air, DuckDuckGo Privacy Browser, East Browser, Ecosia, Edge Update, Edge WebView, Electron Fetch, Element Browser, Elements Browser, Elinks, eM Client, Emacs, EMAudioPlayer, Embarcadero URI Client, Emby Theater, Eolie, Epic, Epic Games Launcher, ESET Remote Administrator, ESP32 HTTP Client, Espial TV Browser, eToro, EudoraWeb, EUI Browser, Evernote, Evolve Podcast, Expedia, Expo, eZ Browser, F-Secure SAFE, Facebook, Facebook Audience Network, Facebook Groups, Facebook Lite, Facebook Messenger, Facebook Messenger Lite, faidr, Falkon, Faraday, fasthttp, Fathom, Faux Browser, FeedDemon, Feeddler RSS Reader, FeedR, FeedStation, Fennec, ffmpeg, FFUF, fGet, FileDownloader, Files, Firebird, Firefox, Firefox Focus, Firefox Mobile, Firefox Mobile iOS, Firefox Reality, Firefox Rocket, Fireweb, Fireweb Navigator, Flash Browser, Flast, Flipboard App, Flipp, Flock, Floorp, Flow, Flow Browser, Fluid, FlyCast, FM WhatsApp, Focus Keeper, Focus Matrix, foobar2000, Foobar2000, Fountain, Foxmail, Franz, Free Download Manager, Freespoke, FreeU, Gaana, Galeon, Garmin fenix 5X, Garmin Forerunner, GBWhatsApp, GeoIP Update, GetPodcast, Ghostery Privacy Browser, GinxDroid Browser, Git, git-annex, GitHub Desktop, Glass Browser, GlobalProtect, Gmail, GNOME Web, go-container registry, Go-http-client, GoBrowser, GOG Galaxy, GoKu, Gold, GoldenPod, GoLoud, GoNative, Goodpods, GoodReader, Google Assistant, Google Drive, Google Earth, Google Earth Pro, Google Fiber TV, Google Go, Google HTTP Java Client, Google Lens, Google Nest Hub, Google Photos, Google Play Newsstand, Google Plus, Google Podcasts, Google Search App, Google Tag Manager, got, gPodder, GreenBrowser, GRequests, GroupMe, gRPC-Java, GStreamer, Guacamole, Guzzle (PHP HTTP Client), gvfs, hackney, Hago, Hammel, HandBrake, Harbor registry client, HardCast, Hark Audio, Harman Browser, HasBrowser, Hawk Quick Browser, Hawk Turbo Browser, Headless Chrome, Heart, Helio, Helm, HermesPod, HeyTapBrowser, Hi Browser, HiCast, HideX, Hik-Connect, Himalaya, HiSearch, HisThumbnail, HistoryHound, hola! Browser, HotBrowser, Hotels.com, HotJava, HP Smart, HTC Streaming Player, HTML Parser, HTTP request maker, http.rb, HTTP:Tiny, HTTP_Request2, HTTPie, httplib2, httprs, HTTPX, Huawei Browser, Huawei Browser Mobile, Huawei Mobile Services, Hubhopper, HyperCatcher, IBrowse, iBrowser, iBrowser Mini, iCab, iCab Mobile, ICAP Client, iCatcher, IceCat, IceDragon, Iceweasel, IE Mobile, iHeartRadio, IMO HD Video Calls & Chat, IMO International Calls & Chat, Impervious Browser, Insomnia REST Client, Inspect Browser, Instabridge, Instacast, Instagram, Instapaper, Internet Explorer, Intune Managed Browser, iOS Application, IPinfo, iPlayTV, IPTV, IPTV Pro, Iridium, Iron, Iron Mobile, Isivioo, iTunes, iVoox, Jakarta Commons HttpClient, Jam, JaneStyle, JaneView, Japan Browser, Jasmine, Java, Java HTTP Client, JavaFX, JetBrains Omea Reader, JHelioviewer, Jig Browser, Jig Browser Plus, JioSaavn, JioSphere, Jitsi Meet, JJ2GO, JRiver Media Center, jsdom, Juice, Jungle Disk, Just Audio, Just Listen, JUZI Browser, K-meleon, K-Ninja, K.Browser, KaiOS Downloader, Kajabi, KakaoTalk, Kapiko, Kasts, Kazehakase, Keeper Password Manager, Keepsafe Browser, Kids Listen, KidsPod, Kik, Kindle Browser, Kinza, Kitt, Kiwi, Kiwi TCMS, Kiwi TCMS API, KKBOX, Klarna, Kode Browser, Kodi, Konqueror, KPN Veilig, KUN, Kylo, L.A. Times, Ladybird, Lagatos Browser, Landis+Gyr AIM Browser, Lark, Laughable, Lazada, LBC, Legan Browser, Lenovo Browser, Lexi Browser, LG Browser, LG Player, libdnf, libpod, libsoup, Libsyn, LieBaoFast, Liferea, Light, Lilo, Line, LinkedIn, Links, Liri Browser, Listen, LiSTNR, Liulo, Live5ch, Logi Options+, Lolifox, LoseIt!, Lotus, Lotus Notes, Lovense Browser, LT Browser, LUA OpenResty NGINX, LuaKit, Lulumi, Luminary, Lunascape, Lunascape Lite, Lynx, Macrium Reflect, Maelstrom, Mail Master, mailapp, MailBar, Mailbird, Mailspring, Mandarin, Mandrill PHP, MAUI WAP Browser, MaxBrowser, Maxthon, MBolsa, mCent, Mechanize, MediaMonkey, Megaphone, Meizu Browser, MEmpresas, Menucast, Mercantile Bank of Michigan, Mercury, MessengerX, Meta Business Suite, Metacast, MetaMask, MetaTrader, MicroB, Microsoft Bing, Microsoft Copilot, Microsoft Edge, Microsoft Lync, Microsoft Office, Microsoft Office $1, Microsoft Office Mobile, Microsoft OneDrive, Microsoft Outlook, Microsoft Power Query, Microsoft Start, Microsoft Store, Midori, Mikrotik Fetch, Mimir, Minimo, Mint Browser, Miro, MIUI Browser, MixerBox, MixerBox AI, Mobicip, Mobile Safari, Mobile Silk, mobile.de, MobileSMS, Mogok Browser, MOMO, Monument Browser, MoonFM, mowPod, Moya, MPlayer, mpv, Msray-Plus, Music Player Daemon, MusicBee, MX Player, MxNitro, My Bentley, My Watch Party, My World, Mypal, MyTuner, Naenara Browser, nate, Naver, NAVER Dictionary, NAVER Mail, Navigateur Web, NCSA Mosaic, NET.mede, Netflix, NetFront, NetFront Life, NetNewsWire, NetPositive, Netscape, NetSurf, NewsArticle App, Newsbeuter, NewsBlur, NewsBlur Mobile App, Newsboat, Newsly, NexPlayer, Nextcloud, NFS Browser, Nightingale, Ninetails, Node Fetch, Nokia Browser, Nokia OSS Browser, Nokia Ovi Browser, NOOK Browser, Norton Private Browser, Notion, Nox Browser, NPR, NRC, NRC Audio, NTENT Browser, NTV Mobil, Nuanti Meta, NuMuKi Browser, Nuviu, Obigo, OBS Studio, OceanHero, Oculus Browser, Odd Browser, Odin, Odnoklassniki, Odyssey Web Browser, Off By One, OfferUp, OhHai Browser, OKDownload Library, OkHttp, OmniWeb, OnBrowser Lite, ONE Browser, Onion Browser, Opal Travel, Open Build Service, OpenFin, Openwave Mobile Browser, Opera, Opera Devices, Opera GX, Opera Mini, Opera Mini iOS, Opera Mobile, Opera Neon, Opera News, Opera Next, Opera Touch, Opera Updater, Oppo Browser, Orange Radio, Orca, Ordissimo, Oregano, Origin In-Game Overlay, Origyn Web Browser, Otter Browser, Outcast, Outlook Express, Overcast, Overhaul FM, Pa11y, Paint by Number, Palco MP3, Pale Moon, Palm Blazer, Palm Pre, Palm WebPro, Palmscape, Pandora, Papers, PeaCast, Peeps dBrowser, Perfect Browser, Perl, Perl REST::Client, Petal Search, Phantom Browser, Phoenix, Phoenix Browser, PHP cURL Class, Pi Browser, Pic Collage, PICO Browser, Pinterest, PirateBrowser, Playapod, Player FM, PlayFree Browser, PLAYit, Plex Media Server, Pocket Casts, PocketBook Browser, Podbay, Podbean, Podcast & Radio Addict, Podcast App, Podcast Guru, Podcast Player, Podcast Republic, Podcastly, Podcat, Podcatcher Deluxe, Podchaser, Podclipper, PodCruncher, Podeo, Podfriend, Podgrab, Podhero, Podimo, PodKast, Podkicker Pro, Podkicker$1, PodLP, PodMe, PodMN, PodNL, Podopolo, Podplay, PodPuppy, Pods, PodTrapper, podU, Podurama, Podverse, Podvine, Podyssey, Polaris, Polarity, PolyBrowser, Polypane, Postbox, Postman Desktop, PowerShell, PRDownloader, PritTorrent, PrivacyWall, Procast, Proxy Browser, PSI Secure Browser, Puffin, PugPig Bolt, Pulp, Python Requests, Python urllib, Q-municate, Qazweb, QbHttp, qBittorrent, Qiyu, QQ Browser, QQ Browser Lite, QQ Browser Mini, QQMusic, QtWebEngine, Quark, quic-go, Quick Search TV, QuickCast, Quicksilver, QuickTime, QuiteRSS, Quora, QupZilla, Qutebrowser, Qwant Mobile, R, r-curl, Radio Downloader, Radio Italiane, Radio Next, radio.$1, RadioApp, Radioline, RadioPublic, Raindrop.io, Rakuten Browser, Rakuten Web Search, Rambox Pro, Raspbian Chromium, Rave Social, Razer Synapse, RDDocuments, ReactorNetty, ReadKit, Realme Browser, Reddit, Redditor, Reeder, Rekonq, rekordbox, Repod, Report Runner, req, request, Requests, reqwest, Reqwireless WebViewer, REST Client for Ruby, RestSharp, Resty, resty-requests, Rhythmbox, RNPS Action Cards, Roblox, RoboForm, Rocket Chat, RockMelt, Roku Browser, RSS Bandit, RSS Junkie, RSSDemon, RSSOwl, RSSRadio, ruby, Rutube, SachNoi, Safari, Safari Search Helper, Safari Technology Preview, Safari View Service, Safe Exam Browser, SafeIP, Sailfish Browser, SalamWeb, Samsung Browser, Samsung Magician, Samsung Podcasts, SberBrowser, ScalaJ HTTP, SeaMonkey, SearchCraft, Secure Browser, Secure Private Browser, SecureX, Seewo Browser, SEMC-Browser, Seraphic Sraf, ServeStream, Seznam Browser, SFive, Shadow, Shadowrocket, Shiira, Shopee, ShowMe, Signal, SimpleBrowser, Sina Weibo, Siri, SiriusXM, SiteKiosk, Sizzy, Skopeo, Skye, Skyeng, Skyeng Teachers, Skyfire, SkyLeap, Skype, Skype for Business, Slack, Sleipnir, Slim Framework, SlimBoat, SlimerJS, Slimjet, Smart Lenovo Browser, Smooz, Snapchat, Snipd, Snowshoe, SoFi, Sogou Explorer, Sogou Mobile Browser, SogouSearch App, SohuNews, Soldier, Songbird, Sonnet, SONOS, Sony Media Go, Sony PlayStation 5, Soul Browser, SoundOn, SoundWaves, SP Browser, Spectre Browser, Spicebird, Splash, SPORT1, Spotify, Spreaker, Sputnik Browser, Stagefright, Stampy Browser, Stargon, START Internet Browser, Startsiden, Stealer, Steam In-Game Overlay, Stitcher, StoryShots, Stream Master, Streamlabs OBS, Streamy, Strimio, Stringer, StudioDisplay, SubStream, Sunrise, Super Fast Browser, superagent, SuperBird, surf, Surf Browser, Surfshark, Surfy Browser, Sushi Browser, Swiftfox, Swinsian, Swoot, Symfony, T+Browser, T-Browser, t-online.de Browser, Tao Browser, Taobao, tararia, Teams, TenFourFox, Tenta Browser, Tesla Browser, The Bat!, The New York Times, The Wall Street Journal, Theyub, Threads, Thunder, Thunderbird, tieba, TikTok, TiviMate, Tizen Browser, ToGate, TopBuzz, Trade Me, TradingView, Treble.fm, TrueLocation Browser, TuneIn Radio, TuneIn Radio Pro, Tungsten, Turtlecast, Tuya Smart Life, TV Bro, TVirl, TweakStyle, twinkle, Twitch Studio, Twitter, Twitterrific, Typhoeus, U Browser, U-Cursos, Ubook Player, UBrowser, UC Browser, UC Browser HD, UC Browser Mini, UC Browser Turbo, UCast, uclient-fetch, Uconnect LIVE, Uforia, Ultimate Sitemap Parser, undici, Unibox, Unirest for Java, UnityPlayer, UR Browser, urlgrabber (yum), uTorrent, Uzbl, Vast Browser, VD Browser, Venus Browser, Vertex Surf, Viasat Browser, VibeMate, Viber, Victor Reader Stream 3, Victor Reader Stream New Generation, Victor Reader Stream Trek, Virgin Radio, Vision Mobile Browser, Visual Studio Code, Vivaldi, vivo Browser, VLC, VMware AirWatch, Vodacast, Vonkeror, Vuhuv, Vuze, w3m, waipu.tv, Waterfox, Wattpad, Wayback Machine, Wear Internet Browser, Web Explorer, WebDAV, WebDiscover, Webian Shell, WebPositive, WeChat, WeChat Share Extension, Weltweitimnetz Browser, WeTab Browser, Wexond, Wget, WH Questions, Whale Browser, Whatplay, WhatsApp, WhatsApp Business, WhatsApp+2, Whisper, Winamp, Windows Antivirus, Windows CryptoAPI, Windows Delivery Optimization, Windows HTTP, Windows Mail, Windows Media Player, Windows Push Notification Services, Windows Update Agent, WinHttp WinHttpRequest, Wireshark, Wirtschafts Woche, WNYC, Wolvic, Word Cookies!, wOSBrowser, WPS Office, WWW-Mechanize, Wynk Music, Wyzo, XBMC, Xiao Yu Zhou, Xiino, XING, XML-RPC, XSplit Broadcaster, xStand, Xtream Player, Xvast, Y8 Browser, Yaani Browser, YAGI, Yahoo Mail, Yahoo OneSearch, Yahoo! Japan, Yahoo! Japan Browser, Yahoo! Mail, YakYak, Yandex, Yandex Browser, Yandex Browser Lite, Yandex Music, Yapa, Yelp Mobile, Yo WhatsApp, Yolo Browser, YouBrowser, YouCare, YouTube, Youtube Music, YouView, Zalo, Zen, ZEPETO, Zetakey, Zoho Chat, Zune, Zvu

### Devices

10moons, 2E, 360, 3GNET, 3GO, 3Q, 4Good, 4ife, 5IVE, 7 Mobile, 8848, A&K, A1, A95X, AAUW, Accent, Accesstyle, Ace, Aceline, Acer, Acteck, actiMirror, Adronix, Advan, Advance, Advantage Air, AEEZO, AFFIX, AfriOne, AG Mobile, AGM, AIDATA, Ainol, Airis, Airness, AIRON, Airpha, Airtel, Airties, AirTouch, AIS, Aiuto, Aiwa, Ajib, Akai, AKIRA, Alba, Alcatel, Alcor, ALDI NORD, ALDI SÜD, Alfawise, Alienware, Aligator, AllCall, AllDocube, ALLINmobile, Allview, Allwinner, Alps, Altech UEC, Altice, Altimo, altron, Altus, AMA, Amazon, Amazon Basics, AMCV, AMGOO, Amigoo, Amino, Amoi, ANBERNIC, Andowl, Angelcare, AngelTech, Anker, Anry, ANS, ANXONIT, AOC, Aocos, Aocwei, AOpen, Aoro, Aoson, AOYODKG, Apple, Aquarius, Archos, Arian Space, Arival, Ark, ArmPhone, Arnova, ARRIS, Artel, Artizlee, ArtLine, Arçelik, Asano, Asanzo, Ask, Aspera, ASSE, Assistant, Astro, Asus, AT&T, Athesi, Atlantic Electrics, Atmaca Elektronik, ATMAN, ATOL, Atom, Attila, Atvio, Audiovox, AUPO, AURIS, Autan, AUX, Avaya, Avenzo, AVH, Avvio, Awow, AWOX, AXEN, Axioo, AXXA, Axxion, AYA, AYYA, Azeyou, Azumi Mobile, b2m, Backcell, BAFF, BangOlufsen, Barnes & Noble, BARTEC, BASE, BAUHN, BB Mobile, BBK, BDF, BDQ, BDsharing, Beafon, Becker, Beeline, Beelink, Beetel, Beista, Beko, Bellphone, Benco, Benesse, BenQ, BenQ-Siemens, BenWee, Benzo, Beyond, Bezkam, BGH, Bigben, BIHEE, BilimLand, Billion, Billow, BioRugged, Bird, Bitel, Bitmore, Bittium, Bkav, Black Bear, Black Box, Black Fox, Blackpcs, Blackview, Blaupunkt, Bleck, BLISS, Blloc, Blow, Blu, Bluboo, Bluebird, Bluedot, Bluegood, BlueSky, Bluewave, BluSlate, BMAX, Bmobile, BMW, BMXC, Bobarry, bogo, Bolva, Bookeen, Boost, Botech, Boway, bq, BrandCode, Brandt, BRAVE, Bravis, BrightSign, Brigmton, Brondi, BROR, BS Mobile, Bubblegum, Bundy, Bush, BuzzTV, C Idea, C5 Mobile, CADENA, CAGI, Camfone, Canal Digital, Canal+, Canguro, Capitel, Captiva, Carbon Mobile, Carrefour, Casio, Casper, Cat, Cavion, CCIT, Cecotec, Ceibal, Celcus, Celkon, Cell-C, Cellacom, CellAllure, Cellution, CENTEK, Centric, CEPTER, CG Mobile, CGV, Chainway, Changhong, Cherry Mobile, Chico Mobile, ChiliGreen, China Mobile, China Telecom, Chuwi, CipherLab, Citycall, CKK Mobile, Claresta, Clarmin, CLAYTON, ClearPHONE, Clementoni, Cloud, Cloudfone, Cloudpad, Clout, Clovertek, CnM, Cobalt, Coby Kyros, Colors, Comio, Compal, Compaq, COMPUMAX, ComTrade Tesla, Conceptum, Concord, ConCorde, Condor, Connectce, Connex, Conquest, Continental Edison, Contixo, COOD-E, Coolpad, Coopers, CORN, Cosmote, Covia, Cowon, COYOTE, CreNova, Crescent, Cricket, Crius Mea, Crony, Crosscall, Crown, Ctroniq, Cube, CUBOT, Cuiud, CVTE, Cwowdefu, Cyrus, D-Link, D-Tech, Daewoo, Danew, DangcapHD, Dany, Daria, DASS, Datalogic, Datamini, Datang, Datawind, Datsun, Dazen, DbPhone, Dbtel, Dcode, DEALDIG, Dell, Denali, Denver, Desay, DeWalt, DEXP, DEYI, DF, DGTEC, DIALN, Dialog, Dicam, Digi, Digicel, DIGICOM, Digidragon, DIGIFORS, Digihome, Digiland, Digit4G, Digma, DIJITSU, DIKOM, DIMO, Dinalink, Dinax, DING DING, DIORA, DISH, Disney, Ditecma, Diva, DiverMax, Divisat, DIXON, DL, DMM, DNS, DoCoMo, Doffler, Dolamee, Dom.ru, Doogee, Doopro, Doov, Dopod, Doppio, DORLAND, Doro, DPA, DRAGON, Dragon Touch, Dreamgate, DreamStar, DreamTab, Droidlogic, Droxio, DSDevices, DSIC, Dtac, Dune HD, DUNNS Mobile, Durabook, Duubee, Dyon, E-Boda, E-Ceros, E-tel, Eagle, EagleSoar, EAS Electric, Easypix, EBEN, EBEST, Echo Mobiles, ecom, ECON, ECOO, ECS, Edenwood, EE, EFT, EGL, Einstein, EKINOX, EKO, Eks Mobility, EKT, ELARI, Elecson, Electroneum, ELECTRONIA, Elekta, Elektroland, Element, Elenberg, Elephone, Elevate, Elong Mobile, Eltex, Ematic, Emporia, ENACOM, Energizer, Energy Sistem, Engel, ENIE, Enot, eNOVA, Entity, Envizen, Ephone, Epic, Epik One, Epson, Equator, Ergo, Ericsson, Ericy, Erisson, Essential, Essentielb, eSTAR, ETOE, Eton, eTouch, Etuline, Eurocase, Eurostar, Evercoss, Everest, Everex, Evertek, Evolio, Evolveo, Evoo, EVPAD, EvroMedia, EWIS, EXCEED, Exmart, ExMobile, EXO, Explay, Express LUCK, ExtraLink, Extrem, Eyemoo, EYU, Ezio, Ezze, F&U, F+, F150, F2 Mobile, Facebook, Facetel, Facime, Fairphone, Famoco, Famous, Fantec, FaRao Pro, Farassoo, FarEasTone, Fengxiang, Fenoti, FEONAL, Fero, FFF SmartLife, Figgers, FiGi, FiGO, FiiO, Filimo, FILIX, FinePower, Finlux, FireFly Mobile, FISE, FITCO, Fluo, Fly, FLYCAT, FLYCOAY, FMT, FNB, FNF, Fobem, Fondi, Fonos, FOODO, FORME, Formuler, Forstar, Fortis, FOSSiBOT, Four Mobile, Fourel, Foxconn, FoxxD, FPT, free, Freetel, FreeYond, Frunsi, Fuego, Fujitsu, Funai, Fusion5, Future Mobile Technology, Fxtec, G-TiDE, G-Touch, Galactic, Galaxy Innovations, Gamma, Garmin-Asus, Gateway, Gazer, Geanee, Geant, Gear Mobile, Gemini, General Mobile, Genesis, GEOFOX, Geotel, Geotex, GEOZON, Getnord, GFive, Gfone, Ghia, Ghong, Ghost, Gigabyte, Gigaset, Gini, Ginzzu, Gionee, GIRASOLE, Globex, Globmall, Glofiish, GLONYX, GLX, GOCLEVER, Gocomma, GoGEN, Gol Mobile, GOLDBERG, GoldMaster, GoldStar, Goly, Gome, GoMobile, GOODTEL, Google, Goophone, Gooweel, Gplus, Gradiente, Graetz, Grape, Great Asia, Gree, Green Lion, Green Orange, Greentel, Gresso, Gretel, GroBerwert, Grundig, Gtel, GTMEDIA, GTX, Guophone, H133, H96, Hafury, Haier, Haipai, Hamlet, Hammer, Handheld, HannSpree, Hanseatic, HAOQIN, HAOVM, Hardkernel, Harper, Hartens, Hasee, Hathway, HDC, HeadWolf, HEC, Heimat, Helio, HERO, HexaByte, Hezire, Hi, Hi Nova, Hi-Level, Hiberg, HiBy, High Q, Highscreen, HiGrace, HiHi, HiKing, HiMax, HIPER, Hipstreet, Hiremco, Hisense, Hitachi, Hitech, HKC, HKPro, HLLO, HMD, HOFER, Hoffmann, HOLLEBERG, Homatics, Hometech, Homtom, Honeywell, Hoozo, Horizon, Horizont, Hosin, Hot Pepper, Hotel, HOTREALS, Hotwav, How, HP, HTC, Huadoo, Huagan, Huavi, Huawei, Hugerock, Humax, Hurricane, Huskee, Hykker, Hytera, Hyundai, Hyve, i-Cherry, I-INN, i-Joy, i-mate, i-mobile, I-Plus, iBall, iBerry, ibowin, iBrit, IconBIT, iData, iDino, iDroid, iGet, iHunt, Ikea, IKI Mobile, iKoMo, iKon, iKonia, IKU Mobile, iLA, iLepo, iLife, iMan, Imaq, iMars, iMI, IMO Mobile, Imose, Impression, iMuz, iNavi, INCAR, Inch, Inco, iNew, Infiniton, Infinix, InFocus, InfoKit, Infomir, InFone, Inhon, Inka, Inkti, InnJoo, Innos, Innostream, iNo Mobile, Inoi, iNOVA, inovo, INQ, Insignia, INSYS, Intek, Intel, Intex, Invens, Inverto, Invin, iOcean, IOTWE, iOutdoor, iPEGTOP, iPro, iQ&T, IQM, IRA, Irbis, iReplace, Iris, iRobot, iRola, iRulu, iSafe Mobile, iStar, iSWAG, IT, iTel, iTruck, IUNI, iVA, iView, iVooMi, ivvi, iWaylink, iXTech, iYou, iZotron, JAY-Tech, Jedi, Jeep, Jeka, Jesy, JFone, Jiake, Jiayu, Jin Tu, Jinga, Jio, Jivi, JKL, Jolla, Joy, JoySurf, JPay, JREN, Jumper, Juniper Systems, Just5, JVC, JXD, K-Lite, K-Touch, Kaan, Kaiomy, Kalley, Kanji, Kapsys, Karbonn, Kata, KATV1, Kazam, Kazuna, KDDI, Kempler & Strauss, Kenbo, Kendo, Keneksi, KENSHI, Kenxinda, Khadas, Kiano, Kingbox, Kingstar, Kingsun, KINGZONE, Kinstone, Kiowa, Kivi, Klipad, KN Mobile, Kocaso, Kodak, Kogan, Komu, Konka, Konrow, Koobee, Koolnee, Kooper, KOPO, Koridy, Koslam, Kraft, KREZ, KRIP, KRONO, Krüger&Matz, KT-Tech, KUBO, KuGou, Kuliao, Kult, Kumai, Kurio, KVADRA, Kvant, Kydos, Kyocera, Kyowon, Kzen, KZG, L-Max, LAIQ, Land Rover, Landvo, Lanin, Lanix, Lark, Laurus, Lava, LCT, Le Pan, Leader Phone, Leagoo, Leben, LeBest, Lectrus, Ledstar, LeEco, Leelbox, Leff, Legend, Leke, Lemco, LEMFO, Lemhoov, Lenco, Lenovo, Leotec, Lephone, Lesia, Lexand, Lexibook, LG, Liberton, Lifemaxx, Lime, Lingbo, Lingwin, Linnex, Linsar, Linsay, Listo, LNMBBS, Loewe, Logic, Logic Instrument, Logicom, Logik, LOKMAT, Loview, Lovme, LPX-G, LT Mobile, Lumigon, Lumitel, Lumus, Luna, Luxor, Lville, LYF, M-Horse, M-Tech, M.T.T., M3 Mobile, M4tel, MAC AUDIO, Macoox, Mafe, MAG, MAGCH, Magicsee, Magnus, Majestic, Malata, Mango, Manhattan, Mann, Manta Multimedia, Mantra, Mara, Marshal, Mascom, Massgo, Masstel, Master-G, Mastertech, Matco Tools, Matrix, Maunfeld, Maxcom, Maxfone, Maximus, Maxtron, MAXVI, Maxwell, Maxwest, MAXX, Maze, Maze Speed, MBI, MBK, MBOX, MDC Store, MDTV, meanIT, Mecer, Mecool, Mediacom, MediaTek, Medion, MEEG, MEGA VISION, MegaFon, Meitu, Meizu, Melrose, MeMobile, Memup, MEO, Meta, Metz, MEU, MicroMax, Microsoft, Microtech, Minix, Mint, Mintt, Mio, Mione, mipo, Miray, Mitchell & Brown, Mito, Mitsubishi, Mitsui, MIVO, MIWANG, MIXC, MiXzo, MLAB, MLLED, MLS, MMI, Mobell, Mobicel, MobiIoT, Mobiistar, Mobile Kingdom, Mobiola, Mobistel, MobiWire, Mobo, Mobvoi, Modecom, Mofut, Mosimosi, Motiv, Motorola, Movic, MOVISUN, Movitel, Moxee, mPhone, Mpman, MSI, MStar, MTC, MTN, Multilaser, MultiPOS, MwalimuPlus, MYFON, MyGica, MygPad, Mymaga, MyMobile, MyPhone, Myria, Myros, Mystery, MyTab, MyWigo, N-one, Nabi, NABO, Nanho, Naomi Phone, NASCO, National, Navcity, Navitech, Navitel, Navon, NavRoad, NEC, Necnot, Nedaphone, Neffos, NEKO, Neo, neoCore, Neolix, Neomi, Neon IQ, Neoregent, NetBox, Netgear, Netmak, NeuImage, NeuTab, NEVIR, New Balance, New Bridge, Newgen, Newland, Newman, Newsday, NewsMy, Nexa, NEXBOX, Nexian, NEXON, NEXT, Next & NextStar, Nextbit, NextBook, NextTab, NG Optics, NGM, NGpon, Nikon, NINETEC, Nintendo, nJoy, NOA, Noain, Nobby, Noblex, NOBUX, noDROPOUT, NOGA, Nokia, Nomi, Nomu, Noontec, Nordmende, NORMANDE, NorthTech, Nos, Nothing Phone, Nous, Novacom, Novex, Novey, NoviSea, NOVO, NTT West, NuAns, Nubia, NUU Mobile, NuVision, Nuvo, Nvidia, NYX Mobile, O+, O2, Oale, Oangcc, OASYS, Obabox, Ober, Obi, OCEANIC, Odotpad, Odys, OINOM, Ok, Okapia, Oking, OKSI, OKWU, Olax, Olkya, Ollee, OLTO, Olympia, OMIX, Onda, OneClick, OneLern, OnePlus, Onida, Onix, Onkyo, ONN, ONVO, ONYX BOOX, Ookee, Ooredoo, OpelMobile, Openbox, Ophone, OPPO, Opsson, Optoma, Orange, Orava, Orbic, Orbita, Orbsmart, Ordissimo, Orion, OSCAL, OTTO, OUJIA, Ouki, Oukitel, OUYA, Overmax, Ovvi, Owwo, OYSIN, Oysters, Oyyu, OzoneHD, P-UP, Pacific Research Alliance, Packard Bell, Padpro, PAGRAER, Paladin, Palm, Panacom, Panasonic, Panavox, Pano, Panodic, Panoramic, Pantech, PAPYRE, Parrot Mobile, Partner Mobile, PC Smart, PCBOX, PCD, PCD Argentina, PEAQ, Pelitt, Pendoo, Penta, Pentagram, Perfeo, Phicomm, Philco, Philips, Phonemax, phoneOne, Pico, PINE, Pioneer, Pioneer Computers, PiPO, PIRANHA, Pixela, Pixelphone, Pixus, Planet Computers, Platoon, Play Now, Ployer, Plum, PlusStyle, Pluzz, PocketBook, POCO, Point Mobile, Point of View, Polar, PolarLine, Polaroid, Polestar, PolyPad, Polytron, Pomp, Poppox, POPTEL, Porsche, Positivo, Positivo BGH, PPTV, Premier, Premio, Prestigio, PRIME, Primepad, Primux, Pritom, Prixton, PROFiLO, Proline, Prology, ProScan, PROSONIC, Protruly, ProVision, PULID, Punos, Purism, Q-Box, Q-Touch, Q.Bell, QFX, Qilive, QIN, QLink, QMobile, Qnet Mobile, QTECH, Qtek, Quantum, Quatro, Qubo, Quechua, Quest, Quipus, Qumo, Qware, QWATT, R-TV, Rakuten, Ramos, Raspberry, Ravoz, Raylandz, Razer, RCA Tablets, Reach, Readboy, Realme, RED, Redbean, Redfox, RedLine, Redway, Reeder, REGAL, RelNAT, Remdun, Renova, Retroid Pocket, Revo, Revomovil, Ricoh, Rikomagic, RIM, Rinno, Ritmix, Ritzviva, Riviera, Rivo, Rizzen, ROADMAX, Roadrover, Roam Cat, Rocket, ROiK, Rokit, Roku, Rombica, Ross&Moor, Rover, RoverPad, Royole, RoyQueen, RT Project, RugGear, RuggeTech, Ruggex, Ruio, Runbo, Rupa, Ryte, S-Color, S-TELL, S2Tel, Saba, Safaricom, Sagem, Sagemcom, Saiet, SAILF, Salora, Samsung, Samtech, Samtron, Sanei, Sankey, Sansui, Santin, SANY, Sanyo, Savio, Sber, SCBC, Schneider, Schok, Scosmos, Seatel, SEBBE, Seeken, SEEWO, SEG, Sega, SEHMAX, Selecline, Selenga, Selevision, Selfix, SEMP TCL, Sencor, Sendo, Senkatel, Senseit, Senwa, Seuic, Sewoo, SFR, SGIN, Shanling, Sharp, Shift Phones, Shivaki, Shtrikh-M, Shuttle, Sico, Siemens, Sigma, Silelis, Silent Circle, Silva Schneider, Simbans, simfer, Simply, Singtech, Siragon, Sirin Labs, SK Broadband, SKG, SKK Mobile, Sky, Skyline, SkyStream, Skytech, Skyworth, Smadl, Smailo, Smart, Smart Electronic, Smart Kassel, Smartab, SmartBook, SMARTEC, Smartex, Smartfren, Smartisan, Smarty, Smooth Mobile, Smotreshka, SMUX, SNAMI, SobieTech, Soda, Softbank, Soho Style, Solas, SOLE, SOLO, Solone, Sonim, SONOS, Sony, SOSH, SoulLink, Soundmax, Soyes, Spark, Sparx, SPC, Spectralink, Spectrum, Spice, Sprint, SPURT, SQOOL, SSKY, Star, Starlight, Starmobile, Starway, Starwind, STF Mobile, STG Telecom, STK, Stonex, Storex, StrawBerry, Stream, STRONG, Stylo, Subor, Sugar, Sumvision, Sunmax, Sunmi, Sunny, Sunstech, SunVan, Sunvell, SUNWIND, Super General, SuperBOX, SuperSonic, SuperTab, Supra, Supraim, Surfans, Surge, Suzuki, Sveon, Swipe, SWISSMOBILITY, Swisstone, Switel, Syco, SYH, Sylvania, Symphony, Syrox, System76, T-Mobile, T96, TADAAM, TAG Tech, Taiga System, Takara, Talius, Tambo, Tanix, TAUBE, TB Touch, TCL, TD Systems, TD Tech, TeachTouch, Technicolor, Technika, TechniSat, Technopc, TECHNOSAT, TechnoTrend, TechPad, TechSmart, Techstorm, Techwood, Teclast, Tecno Mobile, TecToy, TEENO, Teknosa, Tele2, Telefunken, Telego, Telenor, Telia, Telit, Telkom, Telly, Telma, TeloSystems, Telpo, TENPLUS, Teracube, Tesco, Tesla, TETC, Tetratab, teXet, ThL, Thomson, Thuraya, TIANYU, Tibuta, Tigers, Time2, Timovi, TIMvision, Tinai, Tinmo, TiPhone, TiVo, TJC, TOKYO, Tolino, Tone, TOOGO, Tooky, Top House, TopDevice, TOPDON, Topelotek, Toplux, TOPSHOWS, Topsion, Topway, Torex, Torque, TOSCIDO, Toshiba, Touch Plus, Touchmate, TOX, TPS, Transpeed, TrekStor, Trevi, TriaPlay, Trident, Trifone, Trio, Tronsmart, True, True Slim, Tsinghua Tongfang, TTEC, TTfone, TTK-TV, TuCEL, Tunisie Telecom, Turbo, Turbo-X, TurboKids, TurboPad, Turkcell, Tuvio, TVC, TwinMOS, TWM, Twoe, TWZ, TYD, Tymes, Türk Telekom, U-Magic, U.S. Cellular, UD, UE, UGINE, Ugoos, Uhans, Uhappy, Ulefone, Umax, UMIDIGI, Unblock Tech, Uniden, Unihertz, Unimax, Uniqcell, Uniscope, Unistrong, Unitech, UNIWA, Unknown, Unnecto, Unnion Technologies, UNNO, Unonu, Unowhy, UOOGOU, Urovo, UTime, UTOK, UTStarcom, UZ Mobile, V-Gen, V-HOME, V-HOPE, v-mobile, VAIO, VALE, VALEM, VALTECH, VANGUARD, Vankyo, VANWIN, Vargo, Vastking, VAVA, VC, VDVD, Vega, Vekta, Venso, Venstar, Venturer, VEON, Verico, Verizon, Vernee, Verssed, Versus, Vertex, Vertu, Verykool, Vesta, Vestel, VETAS, Vexia, VGO TEL, ViBox, Victurio, VIDA, Videocon, Videoweb, ViewSonic, VIIPOO, VILLAON, Vinabox, Vinga, Vinsoc, Vios, Viper, Vipro, Virzo, Vision Touch, Visual Land, Vitelcom, Vityaz, Viumee, Vivax, VIVIMAGE, Vivo, VIWA, Vizio, Vizmo, VK Mobile, VKworld, VNPT Technology, VOCAL, Vodacom, Vodafone, VOGA, VOLIA, VOLKANO, Volla, Volt, Vonino, Vontar, Vorago, Vorcom, Vorke, Vormor, Vortex, Voto, VOX, Voxtel, Voyo, Vsmart, Vsun, VUCATIMES, Vue Micro, Vulcan, VVETIME, Völfen, W&O, WAF, Wainyok, Walker, Walton, Waltter, Wanmukang, WANSA, WE, We. by Loewe., Webfleet, WeChip, Wecool, Weelikeit, Weimei, WellcoM, WELLINGTON, Western Digital, Westpoint, Wexler, White Mobile, Whoop, Wieppo, Wigor, Wiko, Wileyfox, Winds, Wink, Winmax, Winnovo, Winstar, Wintouch, Wiseasy, WIWA, WizarPos, Wizz, Wolder, Wolfgang, Wolki, WONDER, Wonu, Woo, Wortmann, Woxter, WS, X-AGE, X-BO, X-Mobile, X-TIGI, X-View, X.Vision, X88, X96, X96Q, Xcell, XCOM, Xcruiser, XElectron, XGEM, XGIMI, Xgody, Xiaodu, Xiaolajiao, Xiaomi, Xion, Xolo, Xoro, Xshitou, Xsmart, Xtouch, Xtratech, Xwave, XY Auto, Yandex, Yarvik, YASIN, YELLYOUTH, YEPEN, Yes, Yestel, Yezz, Yoka TV, Yooz, Yota, YOTOPT, Youin, Youwei, Ytone, Yu, YU Fly, Yuandao, YUHO, YUMKEM, YUNDOO, Yuno, YunSong, Yusun, Yxtel, Z-Kai, Zaith, Zamolxe, Zatec, Zealot, Zeblaze, Zebra, Zeeker, Zeemi, Zen, Zenek, Zentality, Zfiner, ZH&K, Zidoo, ZIFRO, Zigo, ZIK, Zinox, Ziox, Zonda, Zonko, Zoom, ZoomSmart, Zopo, ZTE, Zuum, Zync, ZYQ, Zyrex, öwn

## Maintainers

- Mateusz Sojka: https://github.com/matisojka
- Ben Zimmer: https://github.com/benzimmer

## Contributors

Thanks a lot to the following contributors:

- Peter Gao: https://github.com/peteygao
- Stefan Kaes: https://github.com/skaes
- Dennis Wu: https://github.com/dnswus
- Steve Robinson: https://github.com/steverob
- Mark Dodwell: https://github.com/mkdynamic
- Sean Dilda: https://github.com/seandilda
- Stephan Leibelt: https://github.com/sliminas
- Rafael Porras Lucena: https://github.com/rporrasluc
- Anton Rieder: https://github.com/aried3r
- Bruno Arueira: https://github.com/brunoarueira
- Nicolas Rodriguez: https://github.com/n-rodriguez
- Igor Drozdov: https://github.com/igor-drozdov
- Axeleander: https://github.com/Axeleander
- Igor Pstyga: https://github.com/opti

## Contributing

1. Open an issue and explain your feature request or bug before writing any code (this can save a lot of time, both the contributor and the maintainers!)
2. Fork the project (https://github.com/podigee/device_detector/fork)
3. Create your feature branch (`git checkout -b my-new-feature`)
4. Commit your changes (`git commit -am 'Add some feature'`)
5. Push to the branch (`git push origin my-new-feature`)
6. Create a new Pull Request (compare with develop)
7. When adding new data to the yaml files, please make sure to open a PR in the original project, as well (https://github.com/piwik/device-detector)
