% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/s2-constructors-formatters.R
\name{s2_geog_point}
\alias{s2_geog_point}
\alias{s2_make_line}
\alias{s2_make_polygon}
\alias{s2_geog_from_text}
\alias{s2_geog_from_wkb}
\alias{s2_as_text}
\alias{s2_as_binary}
\alias{s2_tessellate_tol_default}
\title{Create and Format Geography Vectors}
\usage{
s2_geog_point(longitude, latitude)

s2_make_line(longitude, latitude, feature_id = 1L)

s2_make_polygon(
  longitude,
  latitude,
  feature_id = 1L,
  ring_id = 1L,
  oriented = FALSE,
  check = TRUE
)

s2_geog_from_text(
  wkt_string,
  oriented = FALSE,
  check = TRUE,
  planar = FALSE,
  tessellate_tol_m = s2_tessellate_tol_default()
)

s2_geog_from_wkb(
  wkb_bytes,
  oriented = FALSE,
  check = TRUE,
  planar = FALSE,
  tessellate_tol_m = s2_tessellate_tol_default()
)

s2_as_text(
  x,
  precision = 16,
  trim = TRUE,
  planar = FALSE,
  tessellate_tol_m = s2_tessellate_tol_default()
)

s2_as_binary(
  x,
  endian = wk::wk_platform_endian(),
  planar = FALSE,
  tessellate_tol_m = s2_tessellate_tol_default()
)

s2_tessellate_tol_default()
}
\arguments{
\item{longitude, latitude}{Vectors of latitude and longitude}

\item{feature_id, ring_id}{Vectors for which a change in
sequential values indicates a new feature or ring. Use \code{\link[=factor]{factor()}}
to convert from a character vector.}

\item{oriented}{TRUE if polygon ring directions are known to be correct
(i.e., exterior rings are defined counter clockwise and interior
rings are defined clockwise).}

\item{check}{Use \code{check = FALSE} to skip error on invalid geometries}

\item{wkt_string}{Well-known text}

\item{planar}{Use \code{TRUE} to force planar edges in import or export.}

\item{tessellate_tol_m}{The maximum number of meters to that a point must
be moved to satisfy the planar edge constraint.}

\item{wkb_bytes}{A \code{list()} of \code{raw()}}

\item{x}{An object that can be converted to an s2_geography vector}

\item{precision}{The number of significant digits to export when
writing well-known text. If \code{trim = FALSE}, the number of
digits after the decimal place.}

\item{trim}{Should trailing zeroes be included after the decimal place?}

\item{endian}{The endian-ness of the well-known binary. See \code{\link[wk:deprecated]{wk::wkb_translate_wkb()}}.}
}
\description{
These functions create and export \link[=as_s2_geography]{geography vectors}.
Unlike the BigQuery geography constructors, these functions do not sanitize
invalid or redundant input using \code{\link[=s2_union]{s2_union()}}. Note that when creating polygons
using \code{\link[=s2_make_polygon]{s2_make_polygon()}}, rings can be open or closed.
}
\examples{
# create point geographies using coordinate values:
s2_geog_point(-64, 45)

# create line geographies using coordinate values:
s2_make_line(c(-64, 8), c(45, 71))

# optionally, separate features using feature_id:
s2_make_line(
  c(-64, 8, -27, -27), c(45, 71, 0, 45),
  feature_id = c(1, 1, 2, 2)
)

# create polygon geographies using coordinate values:
# (rings can be open or closed)
s2_make_polygon(c(-45, 8, 0), c(64, 71, 90))

# optionally, separate rings and/or features using
# ring_id and/or feature_id
s2_make_polygon(
  c(20, 10, 10, 30, 45, 30, 20, 20, 40, 20, 45),
  c(35, 30, 10, 5, 20, 20, 15, 25, 40, 45, 30),
  feature_id = c(rep(1, 8), rep(2, 3)),
  ring_id = c(1, 1, 1, 1, 1, 2, 2, 2, 1, 1, 1)
)

# import and export well-known text
(geog <- s2_geog_from_text("POINT (-64 45)"))
s2_as_text(geog)

# import and export well-known binary
(geog <- s2_geog_from_wkb(wk::as_wkb("POINT (-64 45)")))
s2_as_binary(geog)

# import geometry from planar space
s2_geog_from_text(
   "POLYGON ((0 0, 1 0, 0 1, 0 0))",
   planar = TRUE,
   tessellate_tol_m = 1
)

# export geographies into planar space
geog <- s2_make_polygon(c(179, -179, 179), c(10, 10, 11))
s2_as_text(geog, planar = TRUE)

# polygons containing a pole need an extra step
geog <- s2_data_countries("Antarctica")
geom <- s2_as_text(
  s2_intersection(geog, s2_world_plate_carree()),
  planar = TRUE
)

}
\seealso{
See \code{\link[=as_s2_geography]{as_s2_geography()}} for other ways to construct geography vectors.

BigQuery's geography function reference:
\itemize{
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_geogpoint}{ST_GEOGPOINT}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_makeline}{ST_MAKELINE}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_makepolygon}{ST_MAKEPOLYGON}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_geogfromtext}{ST_GEOGFROMTEXT}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_geogfromwkb}{ST_GEOGFROMWKB}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_astext}{ST_ASTEXT}
\item \href{https://cloud.google.com/bigquery/docs/reference/standard-sql/geography_functions#st_asbinary}{ST_ASBINARY}
}
}
