% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fixedPCA.R
\name{fixedPCA}
\alias{fixedPCA}
\title{PCA with a fixed number of components}
\usage{
fixedPCA(
  x,
  rank = 50,
  value = c("pca", "lowrank"),
  subset.row,
  preserve.shape = TRUE,
  assay.type = "logcounts",
  name = NULL,
  BSPARAM = bsparam(),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{x}{A \linkS4class{SingleCellExperiment} object containing a log-expression amtrix.}

\item{rank}{Integer scalar specifying the number of components.}

\item{value}{String specifying the type of value to return.
\code{"pca"} will return the PCs, \code{"n"} will return the number of retained components, 
and \code{"lowrank"} will return a low-rank approximation.}

\item{subset.row}{A logical, character or integer vector specifying the rows of \code{x} to use in the PCA.
Defaults to \code{NULL} (i.e., all rows used) with a warning.}

\item{preserve.shape}{Logical scalar indicating whether or not the output SingleCellExperiment should be subsetted to \code{subset.row}.
Only used if \code{subset.row} is not \code{NULL}.}

\item{assay.type}{A string specifying which assay values to use.}

\item{name}{String containing the name which which to store the results.
Defaults to \code{"PCA"} in the \code{\link{reducedDimNames}} for \code{value="pca"} and \code{"lowrank"} in the \code{\link{assays}} for \code{value="lowrank"}.}

\item{BSPARAM}{A \linkS4class{BiocSingularParam} object specifying the algorithm to use for PCA.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object to use for parallel processing.}
}
\value{
A modified \code{x} with:
\itemize{
\item the PC results stored in the \code{\link{reducedDims}} as a \code{"PCA"} entry, if \code{type="pca"}.
The attributes contain the rotation matrix, the variance explained and the percentage of variance explained.
(Note that the last may not sum to 100\% if \code{max.rank} is smaller than the total number of PCs.)
\item a low-rank approximation stored as a new \code{"lowrank"} assay, if \code{type="lowrank"}.
This is represented as a \linkS4class{LowRankMatrix}.
}
}
\description{
Perform a PCA where the desired number of components is known ahead of time.
}
\details{
In theory, there is an optimal number of components for any given application,
but in practice, the criterion for the optimum is difficult to define.
As a result, it is often satisfactory to take an \emph{a priori}-defined \dQuote{reasonable} number of PCs for downstream analyses.
A good rule of thumb is to set this to the upper bound on the expected number of subpopulations in the dataset
(see the reasoning in \code{\link{getClusteredPCs}}.

We can use \code{subset.row} to perform the PCA on a subset of genes.
This is typically used to subset to HVGs to reduce computational time and increase the signal-to-noise ratio of downstream analyses.
If \code{preserve.shape=TRUE}, the rotation matrix is extrapolated to include loadings for \dQuote{unselected} genes, i.e., not in \code{subset.row}.
This is done by projecting their expression profiles into the low-dimensional space defined by the SVD on the selected genes.
By doing so, we ensure that the output always has the same number of rows as \code{x} such that any \code{value="lowrank"} can fit into the assays.

Otherwise, if \code{preserve.shape=FALSE}, the output is subsetted by any non-\code{NULL} value of \code{subset.row}.
This is equivalent to the return value after calling the function on \code{x[subset.row,]}.
}
\examples{
library(scuttle)
sce <- mockSCE()
sce <- logNormCounts(sce)

# Modelling the variance:
var.stats <- modelGeneVar(sce)
hvgs <- getTopHVGs(var.stats, n=1000)

# Defaults to pulling out the top 50 PCs.
set.seed(1000)
sce <- fixedPCA(sce, subset.row=hvgs)
reducedDimNames(sce)

# Get the percentage of variance explained. 
attr(reducedDim(sce), "percentVar")

}
\seealso{
\code{\link{denoisePCA}}, where the number of PCs is automatically chosen.

\code{\link{getClusteredPCs}}, another method to choose the number of PCs.
}
\author{
Aaron Lun
}
