"""Helper methods to handle the time in Home Assistant."""

from __future__ import annotations

import asyncio
from collections.abc import Awaitable, Callable, Mapping
import datetime as dt
from logging import Logger
import random
import ssl
from typing import Any, TypeVar

import ciso8601
import jwt

CALLABLE_T = TypeVar("CALLABLE_T", bound=Callable)  # pylint: disable=invalid-name
UTC = dt.UTC


def utcnow() -> dt.datetime:
    """Get now in UTC time."""
    return dt.datetime.now(UTC)


def utc_from_timestamp(timestamp: float) -> dt.datetime:
    """Return a UTC time from a timestamp."""
    return dt.datetime.fromtimestamp(timestamp, UTC)


def parse_date(dt_str: str) -> dt.date | None:
    """Convert a date string to a date object."""
    try:
        return ciso8601.parse_datetime(dt_str).date()
    except ValueError:  # If dt_str did not match our format
        return None


def seconds_as_dhms(seconds: float) -> str:
    """Convert seconds to a DDd:HHh:MMm:SSs string."""
    days, seconds = divmod(int(seconds), 86400)
    hours, seconds = divmod(seconds, 3600)
    minutes, seconds = divmod(seconds, 60)

    parts = []
    if days > 0:
        parts.append(f"{days}d")
    if hours > 0 or (parts and (minutes > 0 or seconds > 0)):
        parts.append(f"{hours}h")
    if minutes > 0 or (parts and seconds > 0):
        parts.append(f"{minutes}m")
    if seconds > 0 or not parts:
        parts.append(f"{seconds}s")
    return ":".join(parts)


def expiration_from_token(token: str | None) -> int | None:
    """Return the expiration time from a token."""
    if not token:
        return None

    try:
        decoded_token: Mapping[str, Any] = jwt.decode(
            token,
            options={"verify_signature": False},
        )
        return int(decoded_token["exp"])
    except (jwt.DecodeError, KeyError):
        return None


def server_context_modern() -> ssl.SSLContext:
    """
    Return an SSL context following the Mozilla recommendations.

    TLS configuration follows the best-practice guidelines specified here:
    https://wiki.mozilla.org/Security/Server_Side_TLS
    Modern guidelines are followed.
    """
    context = ssl.SSLContext(ssl.PROTOCOL_TLS)  # pylint: disable=no-member

    context.options |= (
        ssl.OP_NO_SSLv2
        | ssl.OP_NO_SSLv3
        | ssl.OP_NO_TLSv1
        | ssl.OP_NO_TLSv1_1
        | ssl.OP_CIPHER_SERVER_PREFERENCE
    )
    if hasattr(ssl, "OP_NO_COMPRESSION"):
        context.options |= ssl.OP_NO_COMPRESSION

    context.set_ciphers(
        "ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:"
        "ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:"
        "ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:"
        "ECDHE-ECDSA-AES256-SHA384:ECDHE-RSA-AES256-SHA384:"
        "ECDHE-ECDSA-AES128-SHA256:ECDHE-RSA-AES128-SHA256",
    )

    return context


def next_midnight() -> float:
    """Return the seconds till next local midnight."""
    midnight = dt.datetime.now().replace(
        hour=0,
        minute=0,
        second=0,
        microsecond=0,
    ) + dt.timedelta(days=1)
    return (midnight - dt.datetime.now()).total_seconds()


def jitter(minimum: float, maximum: float) -> float:
    """Return a random float between minimum and maximum for backoff jitter."""
    return random.uniform(minimum, maximum)


async def gather_callbacks(
    logger: Logger,
    name: str,
    callbacks: list[Callable[[], Awaitable[None]]],
) -> None:
    """Gather callbacks and log exceptions."""
    results = await asyncio.gather(*[cb() for cb in callbacks], return_exceptions=True)
    for result, callback in zip(results, callbacks, strict=False):
        if not isinstance(result, Exception):
            continue
        logger.error(
            "Unexpected error in %s callback %s", name, callback, exc_info=result
        )


class Registry(dict):
    """Registry of items."""

    def register(self, name: str) -> Callable[[CALLABLE_T], CALLABLE_T]:
        """Return decorator to register item with a specific name."""

        def decorator(func: CALLABLE_T) -> CALLABLE_T:
            """Register decorated function."""
            self[name] = func
            return func

        return decorator
