/*
  protocols.h - declarations extracted from Garmin device interface
  specification dated 2004-09-16
*/

typedef unsigned char uint8;
#define bool uint8
#define sint16 int16_t
#define sint32 int32_t
#define uint16 u_int16_t
#define uint32 u_int32_t
#define float32 float
#define float64 double
#define time_type uint32

typedef struct
{
  sint32 lat;			/* latitude in semicircles */
  sint32 lon;			/* longitude in semicircles */
} position_type;

typedef uint16 symbol_type;

enum
{
  /* packet IDs for protocol L001, used for the majority of devices */
  Pid_Command_Data =        10,	/* 0a */
  Pid_Xfer_Cmplt =          12,	/* 0c */
  Pid_Date_Time_Data =      14,	/* 0e */
  Pid_Position_Data =       17,	/* 11 */
  Pid_Prx_Wpt_Data =        19,	/* 13 */
  Pid_Records =             27,	/* 1b */
  Pid_Rte_Hdr =             29,	/* 1d */
  Pid_Rte_Wpt_Data =        30,	/* 1e */
  Pid_Almanac_Data =        31,	/* 1f */
  Pid_Trk_Data =            34,	/* 22 */
  Pid_Wpt_Data =            35,	/* 23 */
  Pid_Pvt_Data =            51,	/* 33 */
  Pid_Rte_Link_Data =       98,	/* 62 */
  Pid_Trk_Hdr =             99,	/* 63 */
  Pid_FlightBook_Record =  134,	/* packet with FlightBook data */
  Pid_Lap =                149,	/* part of Forerunner data */
  Pid_Wpt_Cat =            152,

  /* packet IDs for protocol L002, used mainly for panel-mounted
     aviation devices */
  Pid_Almanac_Data_L002 =    4,
  Pid_Command_Data_L002 =   11,
  Pid_Xfer_Cmplt_L002 =     12,
  Pid_Date_Time_Data_L002 = 20,
  Pid_Position_Data_L002 =  24,
  Pid_Prx_Wpt_Data_L002 =   27,
  Pid_Records_L002 =        35,
  Pid_Rte_Hdr_L002 =        37,
  Pid_Rte_Wpt_Data_L002 =   39,
  Pid_Wpt_Data_L002 =       43,

  /* following are "basic packet IDs", for protocol L000, used in
     addition to one of the above sets */
  Pid_Ext_Product_Data =   248,
  Pid_Protocol_Array =     253,
  Pid_Product_Rqst =       254,
  Pid_Product_Data =       255
};


/* The enumerated values for symbol_type are shown below. Note that
most devices that use this type are limited to a much smaller subset
of these symbols, and no attempt is made in this document to indicate
which subsets are valid for each of these devices. However, the device
will ignore any disallowed symbol values that are received and instead
substitute the value for a generic dot symbol. Therefore, there is no
harm in attempting to use any value shown in the table below except
that the device may not accept the requested value. */

typedef enum
{
/*---------------------------------------------------------------
  Marine symbols
---------------------------------------------------------------*/
  sym_anchor =            0,	/* white anchor symbol */
  sym_bell =              1,	/* white bell symbol */
  sym_diamond_grn =       2,	/* green diamond symbol */
  sym_diamond_red =       3,	/* red diamond symbol */
  sym_dive1 =             4,	/* diver down flag 1 */
  sym_dive2 =             5,	/* diver down flag 2 */
  sym_dollar =            6,	/* white dollar symbol */
  sym_fish =              7,	/* white fish symbol */
  sym_fuel =              8,	/* white fuel symbol */
  sym_horn =              9,	/* white horn symbol */
  sym_house =            10,	/* white house symbol */
  sym_knife =            11,	/* white knife & fork symbol */
  sym_light =            12,	/* white light symbol */
  sym_mug =              13,	/* white mug symbol */
  sym_skull =            14,   /* white skull and crossbones symbol */
  sym_square_grn =       15,	/* green square symbol */
  sym_square_red =       16,	/* red square symbol */
  sym_wbuoy =            17,	/* white buoy waypoint symbol */
  sym_wpt_dot =          18,	/* waypoint dot */
  sym_wreck =            19,	/* white wreck symbol */
  sym_null =             20,	/* null symbol (transparent) */
  sym_mob =              21,	/* man overboard symbol */
  sym_buoy_ambr =        22,	/* amber map buoy symbol */
  sym_buoy_blck =        23,	/* black map buoy symbol */
  sym_buoy_blue =        24,	/* blue map buoy symbol */
  sym_buoy_grn =         25,	/* green map buoy symbol */
  sym_buoy_grn_red =     26,	/* green/red map buoy symbol */
  sym_buoy_grn_wht =     27,	/* green/white map buoy symbol */
  sym_buoy_orng =        28,	/* orange map buoy symbol */
  sym_buoy_red =         29,	/* red map buoy symbol */
  sym_buoy_red_grn =     30,	/* red/green map buoy symbol */
  sym_buoy_red_wht =     31,	/* red/white map buoy symbol */
  sym_buoy_violet =      32,	/* violet map buoy symbol */
  sym_buoy_wht =         33,	/* white map buoy symbol */
  sym_buoy_wht_grn =     34,	/* white/green map buoy symbol */
  sym_buoy_wht_red =     35,	/* white/red map buoy symbol */
  sym_dot =              36,	/* white dot symbol */
  sym_rbcn =             37,	/* radio beacon symbol */
  sym_boat_ramp =       150,	/* boat ramp symbol */
  sym_camp =            151,	/* campground symbol */
  sym_restrooms =       152,	/* restrooms symbol */
  sym_showers =         153,	/* shower symbol */
  sym_drinking_wtr =    154,	/* drinking water symbol */
  sym_phone =           155,	/* telephone symbol */
  sym_1st_aid =         156,	/* first aid symbol */
  sym_info =            157,	/* information symbol */
  sym_parking =         158,	/* parking symbol */
  sym_park =            159,	/* park symbol */
  sym_picnic =          160,	/* picnic symbol */
  sym_scenic =          161,	/* scenic area symbol */
  sym_skiing =          162,	/* skiing symbol */
  sym_swimming =        163,	/* swimming symbol */
  sym_dam =             164,	/* dam symbol */
  sym_controlled =      165,	/* controlled area symbol */
  sym_danger =          166,	/* danger symbol */
  sym_restricted =      167,	/* restricted area symbol */
  sym_null_2 =          168,	/* null symbol */
  sym_ball =            169,	/* ball symbol */
  sym_car =             170,	/* car symbol */
  sym_deer =            171,	/* deer symbol */
  sym_shpng_cart =      172,	/* shopping cart symbol */
  sym_lodging =         173,	/* lodging symbol */
  sym_mine =            174,	/* mine symbol */
  sym_trail_head =      175,	/* trail head symbol */
  sym_truck_stop =      176,	/* truck stop symbol */
  sym_user_exit =       177,	/* user exit symbol */
  sym_flag =            178,	/* flag symbol */
  sym_circle_x =        179,	/* circle with x in the center */
  sym_open_24hr =       180,	/* open 24 hours symbol */
  sym_fhs_facility =    181,	/* U Fishing Hot SpotsTM Facility */
  sym_bot_cond =        182,	/* Bottom Conditions */
  sym_tide_pred_stn =   183,	/* Tide/Current Prediction Station */
  sym_anchor_prohib =   184,	/* U anchor prohibited symbol */
  sym_beacon =          185,	/* U beacon symbol */
  sym_coast_guard =     186,	/* U coast guard symbol */
  sym_reef =            187,	/* U reef symbol */
  sym_weedbed =         188,	/* U weedbed symbol */
  sym_dropoff =         189,	/* U dropoff symbol */
  sym_dock =            190,	/* U dock symbol */
  sym_marina =          191,	/* U marina symbol */
  sym_bait_tackle =     192,	/* U bait and tackle symbol */
  sym_stump =           193,	/* U stump symbol */
/*---------------------------------------------------------------
  User customizable symbols	
  The values from sym_begin_custom to sym_end_custom inclusive are
  reserved for the identification of user customizable symbols. 
---------------------------------------------------------------*/ 
  sym_begin_custom =   7680,	/* first user customizable symbol */
  sym_end_custom =     8191,	/* last user customizable symbol */
/*---------------------------------------------------------------
  Land symbols			
---------------------------------------------------------------*/ 
  sym_is_hwy =         8192,	/* interstate hwy symbol */
  sym_us_hwy =         8193,	/* us hwy symbol */
  sym_st_hwy =         8194,	/* state hwy symbol */
  sym_mi_mrkr =        8195,	/* mile marker symbol */
  sym_trcbck =         8196,	/* TracBack (feet) symbol */
  sym_golf =           8197,	/* golf symbol */
  sym_sml_cty =        8198,	/* small city symbol */
  sym_med_cty =        8199,	/* medium city symbol */
  sym_lrg_cty =        8200,	/* large city symbol */
  sym_freeway =        8201,	/* intl freeway hwy symbol */
  sym_ntl_hwy =        8202,	/* intl national hwy symbol */
  sym_cap_cty =        8203,	/* capitol city symbol (star) */
  sym_amuse_pk =       8204,	/* amusement park symbol */
  sym_bowling =        8205,	/* bowling symbol */
  sym_car_rental =     8206,	/* car rental symbol */
  sym_car_repair =     8207,	/* car repair symbol */
  sym_fastfood =       8208,	/* fast food symbol */
  sym_fitness =        8209,	/* fitness symbol */
  sym_movie =          8210,	/* movie symbol */
  sym_museum =         8211,	/* museum symbol */
  sym_pharmacy =       8212,	/* pharmacy symbol */
  sym_pizza =          8213,	/* pizza symbol */
  sym_post_ofc =       8214,	/* post office symbol */
  sym_rv_park =        8215,	/* RV park symbol */
  sym_school =         8216,	/* school symbol */
  sym_stadium =        8217,	/* stadium symbol */
  sym_store =          8218,	/* dept. store symbol */
  sym_zoo =            8219,	/* zoo symbol */
  sym_gas_plus =       8220,	/* convenience store symbol */
  sym_faces =          8221,	/* live theater symbol */
  sym_ramp_int =       8222,	/* ramp intersection symbol */
  sym_st_int =         8223,	/* street intersection symbol */
  sym_weigh_sttn =     8226,	/* inspection/weigh station symbol */
  sym_toll_booth =     8227,	/* toll booth symbol */
  sym_elev_pt =        8228,	/* elevation point symbol */
  sym_ex_no_srvc =     8229,	/* exit without services symbol */
  sym_geo_place_mm =   8230,	/* Geographic place name, man-made */
  sym_geo_place_wtr =  8231,	/* Geographic place name, water */
  sym_geo_place_lnd =  8232,	/* Geographic place name, land */
  sym_bridge =         8233,	/* bridge symbol */
  sym_building =       8234,	/* building symbol */
  sym_cemetery =       8235,	/* cemetery symbol */
  sym_church =         8236,	/* church symbol */
  sym_civil =          8237,	/* civil location symbol */
  sym_crossing =       8238,	/* crossing symbol */
  sym_hist_town =      8239,	/* historical town symbol */
  sym_levee =          8240,	/* levee symbol */
  sym_military =       8241,	/* military location symbol */
  sym_oil_field =      8242,	/* oil field symbol */
  sym_tunnel =         8243,	/* tunnel symbol */
  sym_beach =          8244,	/* beach symbol */
  sym_forest =         8245,	/* forest symbol */
  sym_summit =         8246,	/* summit symbol */
  sym_lrg_ramp_int =   8247,	/* large ramp intersection symbol */
  sym_lrg_ex_no_srvc = 8248,	/* large exit without services smbl */
  sym_badge =          8249,	/* police/official badge symbol */
  sym_cards =          8250,	/* gambling/casino symbol */
  sym_snowski =        8251,	/* snow skiing symbol */
  sym_iceskate =       8252,	/* ice skating symbol */
  sym_wrecker =        8253,	/* tow truck (wrecker) symbol */
  sym_border =         8254,	/* border crossing (port of entry) */
  sym_geocache =       8255,	/* geocache location */
  sym_geocache_fnd =   8256,	/* found geocache */
  sym_cntct_smiley =   8257,	/* Rino contact symbol, "smiley" */
  sym_cntct_ball_cap = 8258,	/* Rino contact symbol, "ball cap" */
  sym_cntct_big_ears = 8259,	/* Rino contact symbol, "big ear" */
  sym_cntct_spike =    8260,	/* Rino contact symbol, "spike" */
  sym_cntct_goatee =   8261,	/* Rino contact symbol, "goatee" */
  sym_cntct_afro =     8262,	/* Rino contact symbol, "afro" */
  sym_cntct_dreads =   8263,	/* Rino contact symbol, "dreads" */
  sym_cntct_female1 =  8264,	/* Rino contact symbol, "female 1" */
  sym_cntct_female2 =  8265,	/* Rino contact symbol, "female 2" */
  sym_cntct_female3 =  8266,	/* Rino contact symbol, "female 3" */
  sym_cntct_ranger =   8267,	/* Rino contact symbol, "ranger" */
  sym_cntct_kung_fu =  8268,	/* Rino contact symbol, "kung fu" */
  sym_cntct_sumo =     8269,	/* Rino contact symbol, "sumo" */
  sym_cntct_pirate =   8270,	/* Rino contact symbol, "pirate" */
  sym_cntct_biker =    8271,	/* Rino contact symbol, "biker" */
  sym_cntct_alien =    8272,	/* Rino contact symbol, "alien" */
  sym_cntct_bug =      8273,	/* Rino contact symbol, "bug" */
  sym_cntct_cat =      8274,	/* Rino contact symbol, "cat" */
  sym_cntct_dog =      8275,	/* Rino contact symbol, "dog" */
  sym_cntct_pig =      8276,	/* Rino contact symbol, "pig" */
  sym_hydrant =        8282,	/* water hydrant symbol */
  sym_flag_blue =      8284,	/* blue flag symbol */
  sym_flag_green =     8285,	/* green flag symbol */
  sym_flag_red =       8286,	/* red flag symbol */
  sym_pin_blue =       8287,	/* blue pin symbol */
  sym_pin_green =      8288,	/* green pin symbol */
  sym_pin_red =        8289,	/* red pin symbol */
  sym_block_blue =     8290,	/* blue block symbol */
  sym_block_green =    8291,	/* green block symbol */
  sym_block_red =      8292,	/* red block symbol */
  sym_bike_trail =     8293,	/* bike trail symbol */
  sym_circle_red =     8294,	/* red circle symbol */
  sym_circle_green =   8295,	/* green circle symbol */
  sym_circle_blue =    8296,	/* blue circle symbol */
  sym_diamond_blue =   8299,	/* blue diamond symbol */
  sym_oval_red =       8300,	/* red oval symbol */
  sym_oval_green =     8301,	/* green oval symbol */
  sym_oval_blue =      8302,	/* blue oval symbol */
  sym_rect_red =       8303,	/* red rectangle symbol */
  sym_rect_green =     8304,	/* green rectangle symbol */
  sym_rect_blue =      8305,	/* blue rectangle symbol */
  sym_square_blue =    8308,	/* blue square symbol */
  sym_letter_a_red =   8309,	/* red letter 'A' symbol */
  sym_letter_b_red =   8310,	/* red letter 'B' symbol */
  sym_letter_c_red =   8311,	/* red letter 'C' symbol */
  sym_letter_d_red =   8312,	/* red letter 'D' symbol */
  sym_letter_a_green = 8313,	/* green letter 'A' symbol */
  sym_letter_c_green = 8314,	/* green letter 'C' symbol */
  sym_letter_b_green = 8315,	/* green letter 'B' symbol */
  sym_letter_d_green = 8316,	/* green letter 'D' symbol */
  sym_letter_a_blue =  8317,	/* blue letter 'A' symbol */
  sym_letter_b_blue =  8318,	/* blue letter 'B' symbol */
  sym_letter_c_blue =  8319,	/* blue letter 'C' symbol */
  sym_letter_d_blue =  8320,	/* blue letter 'D' symbol */
  sym_number_0_red =   8321,	/* red number '0' symbol */
  sym_number_1_red =   8322,	/* red number '1' symbol */
  sym_number_2_red =   8323,	/* red number '2' symbol */
  sym_number_3_red =   8324,	/* red number '3' symbol */
  sym_number_4_red =   8325,	/* red number '4' symbol */
  sym_number_5_red =   8326,	/* red number '5' symbol */
  sym_number_6_red =   8327,	/* red number '6' symbol */
  sym_number_7_red =   8328,	/* red number '7' symbol */
  sym_number_8_red =   8329,	/* red number '8' symbol */
  sym_number_9_red =   8330,	/* red number '9' symbol */
  sym_number_0_green = 8331,	/* green number '0' symbol */
  sym_number_1_green = 8332,	/* green number '1' symbol */
  sym_number_2_green = 8333,	/* green number '2' symbol */
  sym_number_3_green = 8334,	/* green number '3' symbol */
  sym_number_4_green = 8335,	/* green number '4' symbol */
  sym_number_5_green = 8336,	/* green number '5' symbol */
  sym_number_6_green = 8337,	/* green number '6' symbol */
  sym_number_7_green = 8338,	/* green number '7' symbol */
  sym_number_8_green = 8339,	/* green number '8' symbol */
  sym_number_9_green = 8340,	/* green number '9' symbol */
  sym_number_0_blue =  8341,	/* blue number '0' symbol */
  sym_number_1_blue =  8342,	/* blue number '1' symbol */
  sym_number_2_blue =  8343,	/* blue number '2' symbol */
  sym_number_3_blue =  8344,	/* blue number '3' symbol */
  sym_number_4_blue =  8345,	/* blue number '4' symbol */
  sym_number_5_blue =  8346,	/* blue number '5' symbol */
  sym_number_6_blue =  8347,	/* blue number '6' symbol */
  sym_number_7_blue =  8348,	/* blue number '7' symbol */
  sym_number_8_blue =  8349,	/* blue number '8' symbol */
  sym_number_9_blue =  8350,	/* blue number '9' symbol */
  sym_triangle_blue =  8351,	/* blue triangle symbol */
  sym_triangle_green = 8352,	/* green triangle symbol */
  sym_triangle_red =   8353,	/* red triangle symbol */
/*---------------------------------------------------------------
  Aviation symbols		
---------------------------------------------------------------*/ 
  sym_airport =       16384,	/* airport symbol */
  sym_int =           16385,	/* intersection symbol */
  sym_ndb =           16386,	/* non-directional beacon symbol */
  sym_vor =           16387,	/* VHF omni-range symbol */
  sym_heliport =      16388,	/* heliport symbol */
  sym_private =       16389,	/* private field symbol */
  sym_soft_fld =      16390,	/* soft field symbol */
  sym_tall_tower =    16391,	/* tall tower symbol */
  sym_short_tower =   16392,	/* short tower symbol */
  sym_glider =        16393,	/* glider symbol */
  sym_ultralight =    16394,	/* ultralight symbol */
  sym_parachute =     16395,	/* parachute symbol */
  sym_vortac =        16396,	/* VOR/TACAN symbol */
  sym_vordme =        16397,	/* VOR-DME symbol */
  sym_faf =           16398,	/* first approach fix */
  sym_lom =           16399,	/* localizer outer marker */
  sym_map =           16400,	/* missed approach point */
  sym_tacan =         16401,	/* TACAN symbol */
  sym_seaplane =      16402	/* Seaplane Base */
} symbol_enum_type;
typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
} D100_Wpt_Type;

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  uint8 smbl          __attribute__ ((packed)); /* symbol id */
} D101_Wpt_Type;
/* The enumerated values for the "smbl" member of the D101_Wpt_Type
are the same as those for symbol_type (see section 7.3.15 on page
21). However, since the "smbl" member of the D101_Wpt_Type is only
8-bits (instead of 16- bits), all symbol_type values whose upper byte
is non-zero are disallowed in the D101_Wpt_Type.  The "dst" member is
valid only during the Proximity Waypoint Transfer Protocol. */

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  symbol_type smbl    __attribute__ ((packed)); /* symbol id */
} D102_Wpt_Type;
/*
The "dst" member is valid only during the Proximity Waypoint Transfer
Protocol.
*/

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  uint8 smbl          __attribute__ ((packed)); /* symbol id */
  uint8 dspl          __attribute__ ((packed)); /* display option */
} D103_Wpt_Type;
/*    The enumerated values for the "smbl" member of the D103_Wpt_Type
   are shown below: */
typedef enum
{
  smbl_dot_D103 =         0,		/* dot symbol */
  smbl_house_D103 =       1,		/* house symbol */
  smbl_gas_D103 =         2,		/* gas symbol */
  smbl_car_D103 =         3,		/* car symbol */
  smbl_fish_D103 =        4,		/* fish symbol */
  smbl_boat_D103 =        5,		/* boat symbol */
  smbl_anchor_D103 =      6,		/* anchor symbol */
  smbl_wreck_D103 =       7,		/* wreck symbol */
  smbl_exit_D103 =        8,		/* exit symbol */
  smbl_skull_D103 =       9,		/* skull symbol */
  smbl_flag_D103 =       10,		/* flag symbol */
  smbl_camp_D103 =       11,		/* camp symbol */
  smbl_circle_x_D103 =   12,		/* circle with x symbol */
  smbl_deer_D103 =       13,		/* deer symbol */
  smbl_1st_aid_D103 =    14,		/* first aid symbol */
  smbl_back_track_D103 = 15		/* back track symbol */
} D103_smbl_type;
/* The enumerated values for the "dspl" member of the D103_Wpt_Type
   are shown below: */
typedef enum
{
  dspl_name_D103 = 0,		/* Display symbol with waypoint name */
  dspl_none_D103 = 1,		/* Display symbol by itself */
  dspl_cmnt_D103 = 2			/* Display symbol with comment */
} D103_dspl_type;

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  symbol_type smbl    __attribute__ ((packed)); /* symbol id */
  uint8 dspl          __attribute__ ((packed)); /* display option */
} D104_Wpt_Type;
/* The enumerated values for the "dspl" member of the D104_Wpt_Type
   are shown below: */

typedef enum
{
  dspl_smbl_none_D104 = 0,	/* Display symbol by itself */
  dspl_smbl_only_D104 = 1,	/* Display symbol by itself */
  dspl_smbl_name_D104 = 3,     /* Display symbol with waypoint name */
  dspl_smbl_cmnt_D104 = 5	/* Display symbol with comment */
} D104_dspl_type;
/* The "dst" member is valid only during the Proximity Waypoint
   Transfer Protocol. */

typedef struct
{
  position_type posn;		/* position */
  symbol_type smbl    __attribute__ ((packed)); /* symbol id */
  char ident[1]       __attribute__ ((packed)); /* null-terminated string */
} D105_Wpt_Type;

typedef struct
{
  uint8 wpt_class;		/* class */
  uint8 subclass[13]  __attribute__ ((packed)); /* subclass */
  position_type posn  __attribute__ ((packed)); /* position */
  symbol_type smbl    __attribute__ ((packed)); /* symbol id */
  char ident[1]       __attribute__ ((packed)); /* null-terminated string */
/* char lnk_ident[]; null-terminated string */
} D106_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D106_Wpt_Type are as follows:

Zero: indicates a user waypoint ("subclass" is ignored).

Non-zero: indicates a non-user waypoint ("subclass" must be valid).

For non-user waypoints (such as a city in the device map database),
the device will provide a non-zero value in the "wpt_class" member,
and the "subclass" member will contain valid data to further identify
the non-user waypoint. If the host wishes to transfer this waypoint
back to the device (as part of a route), the host must leave the
"wpt_class" and "subclass" members unmodified. For user waypoints, the
host must ensure that the "wpt_class" member is zero, but the
"subclass" member will be ignored and should be set to zero.

The "lnk_ident" member provides a string that indicates the name of
the path from the previous waypoint in the route to this one. For
example, "HIGHWAY 101" might be placed in "lnk_ident" to show that the
path from the previous waypoint to this waypoint is along Highway
101. The "lnk_ident" string may be empty (i.e., no characters other
than the null terminator), which indicates that no particular path is
specified.
*/
typedef struct
{
  char ident[6];		/* identifier */
  position_type posn        __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  uint8 smbl          __attribute__ ((packed)); /* symbol id */
  uint8 dspl          __attribute__ ((packed)); /* display option */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  uint8 color         __attribute__ ((packed));	/* waypoint color */
} D107_Wpt_Type;
/* 
The enumerated values for the "smbl" member of the D107_Wpt_Type are
the same as the "smbl" member of the D103_Wpt_Type.

The enumerated values for the "dspl" member of the D107_Wpt_Type are
the same as the "dspl" member of the D103_Wpt_Type.

The enumerated values for the "color" member of the D107_Wpt_Type are
shown below: */
typedef enum
{
  clr_default_D107 = 0,		// Default waypoint color 
  clr_red_D107 =     1,		// Red 
  clr_green_D107 =   2,		// Green 
  clr_blue_D107 =    3		// Blue 
} D107_color_type;

typedef struct
{
  uint8 wpt_class;		/* class (see below) */
  uint8 color        __attribute__ ((packed));	/* color (see below) */
  uint8 dspl         __attribute__ ((packed)); /* display options (see below) */
  uint8 attr         __attribute__ ((packed)); /* attributes (see below) */
  symbol_type smbl   __attribute__ ((packed)); /* waypoint symbol */
  uint8 subclass[18] __attribute__ ((packed)); /* subclass */
  position_type posn __attribute__ ((packed)); /* position */
  float32 alt        __attribute__ ((packed));	/* altitude in meters */
  float32 dpth       __attribute__ ((packed)); /* depth in meters */
  float32 dist       __attribute__ ((packed)); /* proximity distance in meters */
  char state[2]      __attribute__ ((packed)); /* state */
  char cc[2]         __attribute__ ((packed)); /* country code */
  char ident[1]      __attribute__ ((packed)); /* variable length string*/
/* char comment[]; waypoint user comment */
/* char facility[]; facility name */
/* char city[]; city name */
/* char addr[]; address number */
/* char cross_road[]; intersecting road label */
} D108_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D108_Wpt_Type are defined as follows: */
typedef enum
{
  user_wpt_D108 =      0x00,	// user waypoint 
  avtn_apt_wpt_D108 =  0x40,	// aviation airport waypoint 
  avtn_int_wpt_D108 =  0x41,	// aviation intersection waypoint 
  avtn_ndb_wpt_D108 =  0x42,	// aviation NDB waypoint 
  avtn_vor_wpt_D108 =  0x43,	// aviation VOR waypoint 
  avtn_arwy_wpt_D108 = 0x44,	// aviation airport runway waypoint 
  avtn_aint_wpt_D108 = 0x45,	// aviation airport intersection 
  avtn_andb_wpt_D108 = 0x46,	// aviation airport ndb waypoint 
  map_pnt_wpt_D108 =   0x80,	// map point waypoint 
  map_area_wpt_D108 =  0x81,	// map area waypoint 
  map_int_wpt_D108 =   0x82,	// map intersection waypoint 
  map_adrs_wpt_D108 =  0x83,	// map address waypoint 
  map_line_wpt_D108 =  0x84	// map line waypoint 
} D108_class_type;

/* The "color" member can be one of the following values: */
typedef enum
{
  clr_black_D108 =            0,
  clr_dark_red_D108 =         1,
  clr_dark_green_D108 =       2,
  clr_dark_yellow_D108 =      3,
  clr_dark_blue_D108 =        4,
  clr_dark_magenta_D108 =     5,
  clr_dark_cyan_D108 =        6,
  clr_light_gray_D108 =       7,
  clr_dark_gray_D108 =        8,
  clr_red_D108 =              9,
  clr_green_D108 =           10,
  clr_yellow_D108 =          11,
  clr_blue_D108 =            12,
  clr_magenta_D108 =         13,
  clr_cyan_D108 =            14,
  clr_white_D108 =           15,
  clr_default_color_D108 =  255, /* default for D108 */
  clr_default_color_D109 = 0x1f /* default for D109 */
} D108_color_type;

/* The enumerated values for the "dspl" member of the D108_Wpt_Type
are the same as the "dspl" member of the D103_Wpt_Type.

The "attr" member should be set to a value of 0x60.

The "subclass" member of the D108_Wpt_Type is used for map waypoints
only, and should be set to 0x0000 0x00000000 0xFFFFFFFF 0xFFFFFFFF
0xFFFFFFFF for other classes of waypoints.

The "alt" and "dpth" members may or may not be supported on a given
device. A value of 1.0e25 in either of these fields indicates that
this parameter is not supported or is unknown for this waypoint.

The "dist" member is used during the Proximity Waypoint Transfer
Protocol only, and should be set to 1.0e25 for other cases.

The "comment" member of the D108_Wpt_Type is used for user waypoints
only, and should be an empty string for other waypoint classes.

The "facility" and "city" members are used only for aviation
waypoints, and should be empty strings for other waypoint classes.

The "addr" member is only valid for MAP_ADRS_WPT class waypoints and
will be an empty string otherwise.

The "cross_road" member is valid only for MAP_INT_WPT class waypoints,
and will be an empty string otherwise.
*/

typedef struct
{
  uint8 dtyp;			/* data packet type (0x01 for D109) */
  uint8 wpt_class    __attribute__ ((packed)); /* class */
  uint8 dspl_color   __attribute__ ((packed)); /* display & color (see below) */
  uint8 attr         __attribute__ ((packed)); /* attributes (0x70 for D109) */
  symbol_type smbl   __attribute__ ((packed)); /* waypoint symbol */
  uint8 subclass[18] __attribute__ ((packed)); /* subclass */
  position_type posn __attribute__ ((packed)); /* position */
  float32 alt        __attribute__ ((packed));	/* altitude in meters */
  float32 dpth       __attribute__ ((packed)); /* depth in meters */
  float32 dist       __attribute__ ((packed)); /* proximity distance in meters */
  char state[2]      __attribute__ ((packed)); /* state */
  char cc[2]         __attribute__ ((packed)); /* country code */
  uint32 ete         __attribute__ ((packed)); /* outbound link ete in seconds */
  char ident[1]      __attribute__ ((packed)); /* variable length string */
/* char comment[]; waypoint user comment */
/* char facility[]; facility name */
/* char city[]; city name */
/* char addr[]; address number */
/* char cross_road[]; intersecting road label */
} D109_Wpt_Type;
/* All fields are defined the same as D108_Wpt_Type except as noted
   below.

dtyp - Data packet type, must be 0x01 for D109_Wpt_Type.

dspl_color - The 'dspl_color' member contains three fields; bits 0-4
specify the color, bits 5-6 specify the waypoint display attribute and
bit 7 is unused and must be 0. Color values are as specified for
D108_Wpt_Type except that the default value is 0x1f. Display attribute
values are as specified for D108_Wpt_Type.

attr - Attribute. Must be 0x70 for D109_Wpt_Type.

ete - Estimated time en route in seconds to next waypoint. Default
value is 0xFFFFFFFF.
*/

typedef struct
{
  uint8 dtyp;			/* data packet type (0x01 for D110) */
  uint8 wpt_class    __attribute__ ((packed)); /* class */
  uint8 dspl_color   __attribute__ ((packed)); /* display & color (see below) */
  uint8 attr         __attribute__ ((packed)); /* attributes (0x80 for D110) */
  symbol_type smbl   __attribute__ ((packed)); /* waypoint symbol */
  uint8 subclass[18] __attribute__ ((packed)); /* subclass */
  position_type posn __attribute__ ((packed)); /* position */
  float32 alt        __attribute__ ((packed)); /* altitude in meters */
  float32 dpth       __attribute__ ((packed)); /* depth in meters */
  float32 dist       __attribute__ ((packed)); /* proximity distance in meters */
  char state[2]      __attribute__ ((packed)); /* state */
  char cc[2]         __attribute__ ((packed)); /* country code */
  uint32 ete         __attribute__ ((packed)); /* outbound link ete in seconds */
  float32 temp       __attribute__ ((packed)); /* temperature */
  time_type time     __attribute__ ((packed)); /* timestamp */
  uint16 wpt_cat     __attribute__ ((packed)); /* category membership */
  char ident[1]      __attribute__ ((packed)); /* variable length string */
/* char comment[]; waypoint user comment */
/* char facility[]; facility name */
/* char city[]; city name */
/* char addr[]; address number */
/* char cross_road[]; intersecting road label */
} D110_Wpt_Type;
/* All fields are defined the same as D109_Wpt_Type except as noted
   below.

The valid values for the "wpt_class" member of the D110_Wpt_Type are
defined as follows. If an invalid value is received, the value shall
be user_wpt. */
typedef enum
{
  user_wpt_D110 =      0x00,	// user waypoint 
  avtn_apt_wpt_D110 =  0x40,	// aviation airport waypoint 
  avtn_int_wpt_D110 =  0x41,	// aviation intersection waypoint 
  avtn_ndb_wpt_D110 =  0x42,	// aviation NDB waypoint 
  avtn_vor_wpt_D110 =  0x43,	// aviation VOR waypoint 
  avtn_arwy_wpt_D110 = 0x44,	// aviation airport runway waypoint 
  avtn_aint_wpt_D110 = 0x45,	// aviation airport intersection 
  avtn_andb_wpt_D110 = 0x46,	// aviation airport ndb waypoint 
  map_pnt_wpt_D110 =   0x80,	// map point waypoint 
  map_area_wpt_D110 =  0x81,	// map area waypoint 
  map_int_wpt_D110 =   0x82,	// map intersection waypoint 
  map_adrs_wpt_D110 =  0x83,	// map address waypoint 
  map_line_wpt_D110 =  0x84	// map line waypoint 
} D110_type_type;


/* wpt_cat - Waypoint Category. May not be supported by all
devices. Default value is 0x0000. This is a bit field that provides
category membership information for the waypoint. The waypoint may be
a member of up to 16 categories. If a bit is set then the waypoint is
a member of the corresponding category. For example, if bits 0 and 4
are set then the waypoint is a member of categories 1 and 5. For more
information see section 6.5 on page 12.

temp - Temperature. May not be supported by all devices. A value of
1.0e25 in this field indicates that this parameter is not supported or
is unknown for this waypoint.

time - Time. May not be supported by all devices. A value of
0xFFFFFFFF in this field indicates that this parameter is not
supported or is unknown for this waypoint.

attr - Attribute. Must be 0x80 for D110_Wpt_Type.

dspl_color - The 'dspl_color' member contains three fields; bits 0-4
specify the color, bits 5-6 specify the waypoint display attribute and
bit 7 is unused and must be 0. Valid color values are specified
below. If an invalid color value is received, the value shall be
Black. Valid display attribute values are as shown below. If an
invalid display attribute value is received, the value shall be Name. */

typedef enum
{
  clr_Black_D110 =        0,
  clr_Dark_Red_D110 =     1,
  clr_Dark_Green_D110 =   2,
  clr_Dark_Yellow_D110 =  3,
  clr_Dark_Blue_D110 =    4,
  clr_Dark_Magenta_D110 = 5,
  clr_Dark_Cyan_D110 =    6,
  clr_Light_Gray_D110 =   7,
  clr_Dark_Gray_D110 =    8,
  clr_Red_D110 =          9,
  clr_Green_D110 =       10,
  clr_Yellow_D110 =      11,
  clr_Blue_D110 =        12,
  clr_Magenta_D110 =     13,
  clr_Cyan_D110 =        14,
  clr_White_D110 =       15,
  clr_Transparent_D110 = 16
} D110_color_type;

typedef enum
{
  dspl_Smbl_Name_D110 =    0,	// Display symbol with waypoint name 
  dspl_Smbl_Only_D110 =    1,	// Display symbol by itself 
  dspl_Smbl_Comment_D110 = 2	// Display symbol with comment 
} D110_attribute_type;

/* posn - Position. If a D110 waypoint is received that contains a
value in the lat field of the posn field that is greater than 2^30 or
less than -2^30, then that waypoint shall be rejected.
*/

typedef struct
{
  char name[17];		/* category name */
} D120_Wpt_Cat_Type;
/* The name field contains a null-terminated string with a maximum
length of 16 consecutive non-null characters. If a D120 waypoint
category is received that contains a string with more than 16
consecutive non-null characters then that name should be truncated to
the first 16 characters and then null terminated. If a D120 waypoint
category is received with a null in the first character of the name
field then that packet should not be processed.
*/

typedef struct
{
  char ident[6];		/* identifier */
  char cc[2]          __attribute__ ((packed)); /* country code */
  uint8 wpt_class     __attribute__ ((packed)); /* class */
  position_type posn  __attribute__ ((packed)); /* position */
  sint16 alt          __attribute__ ((packed)); /* altitude (meters) */
  char city[24]       __attribute__ ((packed)); /* city */
  char state[2]       __attribute__ ((packed)); /* state */
  char name[30]       __attribute__ ((packed)); /* facility name */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
} D150_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D150_Wpt_Type are shown below: */
typedef enum
{
  apt_wpt_class_D150 =    0,	// airport waypoint class 
  int_wpt_class_D150 =    1,	// intersection waypoint class 
  ndb_wpt_class_D150 =    2,	// NDB waypoint class 
  vor_wpt_class_D150 =    3,	// VOR waypoint class 
  usr_wpt_class_D150 =    4,	// user defined waypoint class 
  rwy_wpt_class_D150 =    5, // airport runway threshold waypoint class 
  aint_wpt_class_D150 =   6,   // airport intersection waypoint class 
  locked_wpt_class_D150 = 7	// locked waypoint class 
} D150_class_type;


/* The "locked_wpt_class" code indicates that a route within a device
contains an aviation database waypoint that the device could not find
in its aviation database (presumably because the aviation database was
updated to a newer version). The host should never send the
"locked_wpt_class" code to the device.

The "city," "state," "name," and "cc" members are invalid when the
"wpt_class" member is equal to usr_wpt_class.

The "alt" member is valid only when the "wpt_class" member is equal to
apt_wpt_class.
*/
typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  char name[30]       __attribute__ ((packed)); /* facility name */
  char city[24]       __attribute__ ((packed)); /* city */
  char state[2]       __attribute__ ((packed)); /* state */
  sint16 alt          __attribute__ ((packed)); /* altitude (meters) */
  char cc[2]          __attribute__ ((packed)); /* country code */
  char unused2        __attribute__ ((packed)); /* should be set to zero */
  uint8 wpt_class     __attribute__ ((packed)); /* class */
} D151_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D151_Wpt_Type are shown below: */
typedef enum
{
  apt_wpt_class_D151 =    0,	// airport waypoint class 
  vor_wpt_class_D151 =    1,	// VOR waypoint class 
  usr_wpt_class_D151 =    2,	// user defined waypoint class 
  locked_wpt_class_D151 = 3	// locked waypoint class 
} D151_class_type;

/* The "locked_wpt_class" code indicates that a route within a device
contains an aviation database waypoint that the device could not find
in its aviation database (presumably because the aviation database was
updated to a newer version). The host should never send the
"locked_wpt_class" code to the device.

The "dst" member is valid only during the Proximity Waypoint Transfer
Protocol.

The "city," "state," "name," and "cc" members are invalid when the
"wpt_class" member is equal to usr_wpt_class.

The "alt" member is valid only when the "wpt_class" member is equal to
apt_wpt_class.
*/

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn  __attribute__ ((packed)); /* position */
  uint32 unused       __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]       __attribute__ ((packed)); /* comment */
  float32 dst         __attribute__ ((packed));	/* proximity distance (meters) */
  char name[30]       __attribute__ ((packed)); /* facility name */
  char city[24]       __attribute__ ((packed)); /* city */
  char state[2]       __attribute__ ((packed)); /* state */
  sint16 alt          __attribute__ ((packed)); /* altitude (meters) */
  char cc[2]          __attribute__ ((packed)); /* country code */
  uint8 unused2       __attribute__ ((packed)); /* should be set to zero */
  uint8 wpt_class     __attribute__ ((packed)); /* class */
} D152_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D152_Wpt_Type are shown below: */
typedef enum
{
  apt_wpt_class_D152 =    0,	// airport waypoint class 
  int_wpt_class_D152 =    1,	// intersection waypoint class 
  ndb_wpt_class_D152 =    2,	// NDB waypoint class 
  vor_wpt_class_D152 =    3,	// VOR waypoint class 
  usr_wpt_class_D152 =    4,	// user defined waypoint class 
  locked_wpt_class_D152 = 5	// locked waypoint class 
} D152_class_type;

/* The "locked_wpt_class" code indicates that a route within a device
contains an aviation database waypoint that the device could not find
in its aviation database (presumably because the aviation database was
updated to a newer version). The host should never send the
"locked_wpt_class" code to the device.

The "dst" member is valid only during the Proximity Waypoint Transfer
Protocol.

The "city," "state," "name," and "cc" members are invalid when the
"wpt_class" member is equal to usr_wpt_class.

The "alt" member is valid only when the "wpt_class" member is equal to
apt_wpt_class. */

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn __attribute__ ((packed)); /* position */
  uint32 unused     __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]     __attribute__ ((packed)); /* comment */
  float32 dst       __attribute__ ((packed));	/* proximity distance (meters) */
  char name[30]     __attribute__ ((packed)); /* facility name */
  char city[24]     __attribute__ ((packed)); /* city */
  char state[2]     __attribute__ ((packed)); /* state */
  sint16 alt        __attribute__ ((packed)); /* altitude (meters) */
  char cc[2]        __attribute__ ((packed)); /* country code */
  uint8 unused2     __attribute__ ((packed)); /* should be set to zero */
  uint8 wpt_class   __attribute__ ((packed)); /* class */
  symbol_type smbl  __attribute__ ((packed)); /* symbol id */
} D154_Wpt_Type;
/* The enumerated values for the "wpt_class" member of the
   D154_Wpt_Type are shown below: */
typedef enum
{
  apt_wpt_class_D154 =    0,	// airport waypoint class 
  int_wpt_class_D154 =    1,	// intersection waypoint class 
  ndb_wpt_class_D154 =    2,	// NDB waypoint class 
  vor_wpt_class_D154 =    3,	// VOR waypoint class 
  usr_wpt_class_D154 =    4,	// user defined waypoint class 
  rwy_wpt_class_D154 =    5, // airport runway threshold waypoint class 
  aint_wpt_class_D154 =   6,   // airport intersection waypoint class 
  andb_wpt_class_D154 =   7,	// airport NDB waypoint class 
  sym_wpt_class_D154 =    8, // user defined symbol-only waypoint class 
  locked_wpt_class_D154 = 9	// locked waypoint class 
} D154_class_type;

/* The "locked_wpt_class" code indicates that a route within a device
contains an aviation database waypoint that the device could not find
in its aviation database (presumably because the aviation database was
updated to a newer version). The host should never send the
"locked_wpt_class" code to the device.

The "dst" member is valid only during the Proximity Waypoint Transfer
Protocol.

The "city," "state," "name," and "cc" members are invalid when the
"wpt_class" member is equal to usr_wpt_class or sym_wpt_class. The
"alt" member is valid only when the "wpt_class" member is equal to
apt_wpt_class. */

typedef struct
{
  char ident[6];		/* identifier */
  position_type posn __attribute__ ((packed)); /* position */
  uint32 unused     __attribute__ ((packed)); /* should be set to zero */
  char cmnt[40]     __attribute__ ((packed)); /* comment */
  float32 dst       __attribute__ ((packed));	/* proximity distance (meters) */
  char name[30]     __attribute__ ((packed)); /* facility name */
  char city[24]     __attribute__ ((packed)); /* city */
  char state[2]     __attribute__ ((packed)); /* state */
  sint16 alt        __attribute__ ((packed)); /* altitude (meters) */
  char cc[2]        __attribute__ ((packed)); /* country code */
  uint8 unused2     __attribute__ ((packed)); /* should be set to zero */
  uint8 wpt_class   __attribute__ ((packed)); /* class */
  symbol_type smbl  __attribute__ ((packed)); /* symbol id */
  uint8 dspl        __attribute__ ((packed)); /* display option */
} D155_Wpt_Type;
/* The enumerated values for the "dspl" member of the D155_Wpt_Type
   are shown below: */
typedef enum
{
  dspl_smbl_only_D155 = 1,	// Display symbol by itself 
  dspl_smbl_name_D155 = 3,	// Display symbol with waypoint name 
  dspl_smbl_cmnt_D155 = 5	// Display symbol with comment 
} D155_dspl_type;

/* The enumerated values for the "wpt_class" member of the
   D155_Wpt_Type are shown below: */
typedef enum
{
  apt_wpt_class_D155 =    0,	/* airport waypoint class */
  int_wpt_class_D155 =    1,	/* intersection waypoint class */
  ndb_wpt_class_D155 =    2,	/* NDB waypoint class */
  vor_wpt_class_D155 =    3,	/* VOR waypoint class */
  usr_wpt_class_D155 =    4,	/* user defined waypoint class */
  locked_wpt_class_D155 = 5	/* locked waypoint class */
} D155_class_type;
/* The "locked_wpt_class" code indicates that a route within a device
contains an aviation database waypoint that the device could not find
in its aviation database (presumably because the aviation database was
updated to a newer version). The host should never send the
"locked_wpt_class" code to the device.

The "dst" member is valid only during the Proximity Waypoint Transfer
Protocol.

The "city," "state," "name," and "cc" members are invalid when the
"wpt_class" member is equal to usr_wpt_class.

The "alt" member is valid only when the "wpt_class" member is equal to
apt_wpt_class. */

typedef uint8 D200_Rte_Hdr_Type;	/* route number */
/* The route number contained in the D200_Rte_Hdr_Type must be unique
   for each route. */

typedef struct
{
  uint8 nmbr;			/* route number */
  char cmnt[20]     __attribute__ ((packed)); /* comment */
} D201_Rte_Hdr_Type;
/* The "nmbr" member must be unique for each route. Some devices
require a unique "cmnt" for each route, and other devices do
not. There is no mechanism available for the host to determine whether
a device requires a unique "cmnt", and the host must be prepared to
receive unique or non-unique "cmnt" from the device. */

typedef struct
{
  char rte_ident[1];		/* variable length string */
} D202_Rte_Hdr_Type;

typedef struct
{
  uint16 class;			/* link class; see below */
  uint8 subclass[18]__attribute__ ((packed)); /* subclass */
/* char ident[]; variable length string */
} D210_Rte_Link_Type;
/* The "class" member can be one of the following values: */
typedef enum
{
  line_D202 =      0,
  link_D202 =      1,
  net_D202 =       2,
  direct_D202 =    3,
  snap_D202 =   0xFF
} D202_class_type;

/* The "ident" member has a maximum length of 51 characters, including
the terminating NULL.  If "class" is set to "direct" or "snap",
subclass should be set to its default value of 0x0000 0x00000000
0xFFFFFFFF 0xFFFFFFFF 0xFFFFFFFF. */

typedef struct
{
  position_type posn;		/* position */
  time_type time    __attribute__ ((packed)); /* time */
  bool new_trk      __attribute__ ((packed)); /* new track segment? */
} D300_Trk_Point_Type;
/* The "time" member indicates the time at which the track log point
   was recorded.

   When true, the "new_trk" member indicates that the track log point
   marks the beginning of a new track log segment. */

typedef struct
{
  position_type posn;		/* position */
  time_type time    __attribute__ ((packed)); /* time */
  float32 alt       __attribute__ ((packed)); /* altitude in meters */
  float32 dpth      __attribute__ ((packed)); /* depth in meters */
  bool new_trk      __attribute__ ((packed)); /* new track segment? */
} D301_Trk_Point_Type;
/* 
The "time" member indicates the time at which the track log point was recorded.

The `alt' and `dpth' members may or may not be supported on a given
device. A value of 1.0e25 in either of these fields indicates that
this parameter is not supported or is unknown for this track point.

When true, the "new_trk" member indicates that the track log point
marks the beginning of a new track log segment. */

typedef struct
{
  position_type posn;		/* position */
  time_type time    __attribute__ ((packed)); /* time */
  float32 alt       __attribute__ ((packed)); /* altitude in meters */
  float32 dpth      __attribute__ ((packed)); /* depth in meters */
  float32 temp      __attribute__ ((packed)); /* temp in degrees C */
  bool new_trk      __attribute__ ((packed)); /* new track segment? */
} D302_Trk_Point_Type;
/* All fields are defined the same as D301_Trk_Point_Type except as
noted below.  temp - Temperature. May not be supported by all
devices. A value of 1.0e25 in this field indicates that this parameter
is not supported or is unknown for this track point. */

typedef struct
{
  bool dspl;			/* display on the map? */
  uint8 color       __attribute__ ((packed)); /* color (same as D108) */
  char trk_ident[1] __attribute__ ((packed)); /* null-terminated string */
} D310_Trk_Hdr_Type;
/* The `trk_ident' member has a maximum length of 51 characters
   including the terminating NULL. */

typedef struct
{
  uint16 index;			/* unique among all tracks received from device */
} D311_Trk_Hdr_Type;

typedef struct
{
  bool dspl;			/* display on the map? */
  uint8 color       __attribute__ ((packed)); /* color (same as D110) */
  char trk_ident[1] __attribute__ ((packed)); /* null-terminated string */
} D312_Trk_Hdr_Type;
/* The 'trk_ident' member has a maximum length of 51 characters
   including the terminating NULL. */

typedef struct
{
  D100_Wpt_Type wpt;		/* waypoint */
  float32 dst       __attribute__ ((packed));	/* proximity distance (meters) */
} D400_Prx_Wpt_Type;
/* The "dst" member is valid only during the Proximity Waypoint
   Transfer Protocol. */

typedef struct
{
  D103_Wpt_Type wpt;		/* waypoint */
  float32 dst       __attribute__ ((packed));	/* proximity distance (meters) */
} D403_Prx_Wpt_Type;
/* The "dst" member is valid only during the Proximity Waypoint
   Transfer Protocol. */

typedef struct
{
  int idx;			/* proximity index */
  D150_Wpt_Type wpt __attribute__ ((packed)); /* waypoint */
  float32 dst       __attribute__ ((packed));	/* proximity distance (meters) */
} D450_Prx_Wpt_Type;
/* The "dst" member is valid only during the Proximity Waypoint
   Transfer Protocol. */

typedef struct
{
  uint16 wn;			/* week number (weeks) */
  float32 toa   __attribute__ ((packed)); /* almanac data reference time (s) */
  float32 af0   __attribute__ ((packed)); /* clock correction coefficient (s) */
  float32 af1   __attribute__ ((packed)); /* clock correction coefficient (s/s) */
  float32 e     __attribute__ ((packed)); /* eccentricity (-) */
  float32 sqrta __attribute__ ((packed)); /* square root of semi-major axis (a)(m**1/2) */
  float32 m0    __attribute__ ((packed)); /* mean anomaly at reference time (r) */
  float32 w     __attribute__ ((packed)); /* argument of perigee (r) */
  float32 omg0  __attribute__ ((packed)); /* right ascension (r) */
  float32 odot  __attribute__ ((packed)); /* rate of right ascension (r/s) */
  float32 i     __attribute__ ((packed)); /* inclination angle (r) */
} D500_Almanac_Type;

typedef struct
{
  uint16 wn;			/* week number (weeks) */
  float32 toa   __attribute__ ((packed)); /* almanac data reference time (s) */
  float32 af0   __attribute__ ((packed)); /* clock correction coefficient (s) */
  float32 af1   __attribute__ ((packed)); /* clock correction coefficient (s/s) */
  float32 e     __attribute__ ((packed)); /* eccentricity (-) */
  float32 sqrta __attribute__ ((packed)); /* square root of semi-major axis (a)(m**1/2) */
  float32 m0    __attribute__ ((packed)); /* mean anomaly at reference time (r) */
  float32 w     __attribute__ ((packed)); /* argument of perigee (r) */
  float32 omg0  __attribute__ ((packed)); /* right ascension (r) */
  float32 odot  __attribute__ ((packed)); /* rate of right ascension (r/s) */
  float32 i     __attribute__ ((packed)); /* inclination angle (r) */
  uint8 hlth    __attribute__ ((packed)); /* almanac health */
} D501_Almanac_Type;

typedef struct
{
  uint8 svid;			/* satellite id */
  uint16 wn     __attribute__ ((packed)); /* week number (weeks) */
  float32 toa   __attribute__ ((packed)); /* almanac data reference time (s) */
  float32 af0   __attribute__ ((packed)); /* clock correction coefficient (s) */
  float32 af1   __attribute__ ((packed)); /* clock correction coefficient (s/s) */
  float32 e     __attribute__ ((packed)); /* eccentricity (-) */
  float32 sqrta __attribute__ ((packed)); /* square root of semi-major axis (a)(m**1/2) */
  float32 m0    __attribute__ ((packed)); /* mean anomaly at reference time (r) */
  float32 w     __attribute__ ((packed)); /* argument of perigee (r) */
  float32 omg0  __attribute__ ((packed)); /* right ascension (r) */
  float32 odot  __attribute__ ((packed)); /* rate of right ascension (r/s) */
  float32 i     __attribute__ ((packed)); /* inclination angle (r) */
} D550_Almanac_Type;
/* The "svid" member identifies a satellite in the GPS constellation
   as follows: PRN-01 through PRN-32 are indicated by "svid" equal to
   0 through 31, respectively. */

typedef struct
{
  uint8 svid;			/* satellite id */
  uint16 wn     __attribute__ ((packed)); /* week number (weeks) */
  float32 toa   __attribute__ ((packed)); /* almanac data reference time (s) */
  float32 af0   __attribute__ ((packed)); /* clock correction coefficient (s) */
  float32 af1   __attribute__ ((packed)); /* clock correction coefficient (s/s) */
  float32 e     __attribute__ ((packed)); /* eccentricity (-) */
  float32 sqrta __attribute__ ((packed)); /* square root of semi-major axis (a)(m**1/2) */
  float32 m0    __attribute__ ((packed)); /* mean anomaly at reference time (r) */
  float32 w     __attribute__ ((packed)); /* argument of perigee (r) */
  float32 omg0  __attribute__ ((packed)); /* right ascension (r) */
  float32 odot  __attribute__ ((packed)); /* rate of right ascension (r/s) */
  float32 i     __attribute__ ((packed)); /* inclination angle (r) */
  uint8 hlth    __attribute__ ((packed)); /* almanac health bits 17:24 (coded) */
} D551_Almanac_Type;
/* The "svid" member identifies a satellite in the GPS constellation
   as follows: PRN-01 through PRN-32 are indicated by "svid" equal to
   0 through 31, respectively. */

typedef struct
{
  uint8 month;			/* month (1-12) */
  uint8 day     __attribute__ ((packed)); /* day (1-31) */
  uint16 year   __attribute__ ((packed)); /* year (1990 means 1990) */
  uint16 hour   __attribute__ ((packed)); /* hour (0-23) */
  uint8 minute  __attribute__ ((packed)); /* minute (0-59) */
  uint8 second  __attribute__ ((packed)); /* second (0-59) */
} D600_Date_Time_Type;
/* The D600_Date_Time_Type contains the UTC date and UTC time. */

typedef struct
{
  time_type takeoff_time;	/* Time flight started */
  time_type landing_time     __attribute__ ((packed)); /* Time flight ended */
  position_type takeoff_posn __attribute__ ((packed)); /* Takeoff lat/lon */
  position_type landing_posn __attribute__ ((packed)); /* Takeoff lat/lon */
  uint32 night_time          __attribute__ ((packed)); /* Seconds flown in night time conditions */
  uint32 num_landings        __attribute__ ((packed)); /* Number of landings during the flight */
  float32 max_speed          __attribute__ ((packed)); /* Max velocity during flight (meters/sec) */
  float32 max_alt            __attribute__ ((packed)); /* Max altitude above WGS84 ellipsoid (meters) */
  float32 distance           __attribute__ ((packed)); /* Distance of flight (meters) */
  bool cross_country_flag    __attribute__ ((packed)); /* Flight met cross country criteria */
/* char departure_name[]; Name of airport <= 31 bytes */
/* char departure_ident[]; ID of airport <= 11 bytes */
/* char arrival_name[]; Name of airport <= 31 bytes */
/* char arrival_ident[];ID of airport <= 11 bytes */
/* char ac_id[]; N Number of airplane <= 11 bytes */
} D650_Flight_Book_Record_Type;

typedef struct
{
  float64 lat;
  float64 lon        __attribute__ ((packed));	/*  */
} radian_position_type;

typedef radian_position_type D700_Position_Type;

typedef struct
{
  float32 alt;			/* altitude above WGS 84 ellipsoid (meters) */
  float32 epe               __attribute__ ((packed)); /* estimated position error, 2 sigma (meters) */
  float32 eph               __attribute__ ((packed)); /* epe, but horizontal only (meters) */
  float32 epv               __attribute__ ((packed)); /* epe, but vertical only (meters) */
  uint16 fix                __attribute__ ((packed)); /* type of position fix */
  float64 tow               __attribute__ ((packed)); /* time of week (seconds) */
  radian_position_type posn __attribute__ ((packed)); /* latitude and longitude (radians) */
  float32 east              __attribute__ ((packed)); /* velocity east (meters/second) */
  float32 north             __attribute__ ((packed)); /* velocity north (meters/second) */
  float32 up                __attribute__ ((packed)); /* velocity up (meters/second) */
  float32 msl_hght          __attribute__ ((packed)); /* height of WGS84 ellipsoid above MSL(meters) */
  sint16 leap_scnds         __attribute__ ((packed)); /* difference between GPS and UTC (seconds) */
  uint32 wn_days            __attribute__ ((packed)); /* week number days */
} D800_Pvt_Data_Type;
/* The "alt" parameter provides the altitude above the WGS 84
ellipsoid. To find the altitude above mean sea level, add "msl_hght"
to "alt" ("msl_hght" gives the height of the WGS 84 ellipsoid above
mean sea level at the current position).  

The "tow" parameter provides the number of seconds (excluding leap
seconds) since the beginning of the current week, which begins on
Sunday at 12:00 AM (i.e., midnight Saturday night-Sunday morning). The
"tow" parameter is based on Universal Coordinated Time (UTC), except
UTC is periodically corrected for leap seconds while "tow" is not
corrected for leap seconds. To find UTC, subtract "leap_scnds" from
"tow." Since this may cause a negative result for the first few
seconds of the week (i.e., when "tow" is less than "leap_scnds"), care
must be taken to properly translate this negative result to a positive
time value in the previous day. Also, since "tow" is a floating point
number and may contain fractional seconds, care must be taken to
properly round off when using "tow" in integer conversions and
calculations.  

The "wn_days" parameter provides the number of days that have occurred
from UTC December 31st, 1989 to the beginning of the current week
(thus, "wn_days" always represents a Sunday). To find the total number
of days that have occurred from UTC December 31st, 1989 to the current
day, add "wn_days" to the number of days that have occurred in the
current week (as calculated from the "tow" parameter).  

The default enumerated values for the "fix" member of the
D800_Pvt_Data_Type are shown below. It is important for the host to
inspect this value to ensure that other data members in the
D800_Pvt_Data_Type are valid. No indication is given as to whether the
device is in simulator mode versus having an actual position fix. */
typedef enum
{
  fix_unusable = 0,		// failed integrity check 
  fix_invalid =  1,		// invalid or unavailable 
  fix_2D =       2,		// two dimensional 
  fix_3D =       3,		// three dimensional 
  fix_2D_diff =  4,		// two dimensional differential 
  fix_3D_diff =  5		// three dimensional differential 
} D800_fix_type;

