/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.metastore.util;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.io.Closer;
import com.google.common.io.Files;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.gobblin.metastore.nameParser.GuidDatasetUrnStateStoreNameParser;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StateStoreCleaner
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(StateStoreCleaner.class);
    private static final String STATE_STORE_CLEANER_RETENTION_KEY = "state.store.retention";
    private static final String DEFAULT_STATE_STORE_CLEANER_RETENTION = "7";
    private static final String STATE_STORE_CLEANER_RETENTION_TIMEUNIT_KEY = "state.store.retention.timeunit";
    private static final String DEFAULT_STATE_STORE_CLEANER_RETENTION_TIMEUNIT = TimeUnit.DAYS.toString();
    private static final String STATE_STORE_CLEANER_EXECUTOR_THREADS_KEY = "state.store.cleaner.executor.threads";
    private static final String DEFAULT_STATE_STORE_CLEANER_EXECUTOR_THREADS = "50";
    private final Path stateStoreRootDir;
    private final long retention;
    private final TimeUnit retentionTimeUnit;
    private final ExecutorService cleanerRunnerExecutor;
    private final FileSystem fs;

    public StateStoreCleaner(Properties properties) throws IOException {
        Preconditions.checkArgument((boolean)properties.containsKey("state.store.dir"), (Object)"Missing configuration property for the state store root directory: state.store.dir");
        this.stateStoreRootDir = new Path(properties.getProperty("state.store.dir"));
        this.retention = Long.parseLong(properties.getProperty(STATE_STORE_CLEANER_RETENTION_KEY, DEFAULT_STATE_STORE_CLEANER_RETENTION));
        this.retentionTimeUnit = TimeUnit.valueOf(properties.getProperty(STATE_STORE_CLEANER_RETENTION_TIMEUNIT_KEY, DEFAULT_STATE_STORE_CLEANER_RETENTION_TIMEUNIT).toUpperCase());
        this.cleanerRunnerExecutor = Executors.newFixedThreadPool(Integer.parseInt(properties.getProperty(STATE_STORE_CLEANER_EXECUTOR_THREADS_KEY, DEFAULT_STATE_STORE_CLEANER_EXECUTOR_THREADS)), ExecutorsUtils.newThreadFactory((Optional)Optional.of((Object)LOGGER), (Optional)Optional.of((Object)"StateStoreCleaner")));
        URI fsUri = URI.create(properties.getProperty("state.store.fs.uri", "file:///"));
        this.fs = FileSystem.get((URI)fsUri, (Configuration)new Configuration());
    }

    public void run() throws IOException, ExecutionException {
        FileStatus[] stateStoreDirs = this.fs.listStatus(this.stateStoreRootDir);
        if (stateStoreDirs == null || stateStoreDirs.length == 0) {
            LOGGER.warn("The state store root directory does not exist or is empty");
            return;
        }
        ArrayList futures = Lists.newArrayList();
        for (FileStatus stateStoreDir : stateStoreDirs) {
            futures.add(this.cleanerRunnerExecutor.submit(new CleanerRunner(this.fs, stateStoreDir.getPath(), this.retention, this.retentionTimeUnit)));
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException e) {
                throw new ExecutionException("Thread interrupted", e);
            }
        }
        ExecutorsUtils.shutdownExecutorService((ExecutorService)this.cleanerRunnerExecutor, (Optional)Optional.of((Object)LOGGER), (long)60L, (TimeUnit)TimeUnit.SECONDS);
    }

    @Override
    public void close() throws IOException {
        this.cleanerRunnerExecutor.shutdown();
    }

    public static void main(String[] args) throws IOException {
        if (args.length != 1) {
            System.err.println("Usage: " + StateStoreCleaner.class.getSimpleName() + " <configuration file>");
            System.exit(1);
        }
        try (Closer closer = Closer.create();){
            Properties properties = new Properties();
            properties.load((InputStream)closer.register((Closeable)new FileInputStream(args[0])));
            ((StateStoreCleaner)closer.register((Closeable)new StateStoreCleaner(properties))).run();
        }
    }

    private static class CleanerRunner
    implements Runnable {
        private final FileSystem fs;
        private final Path stateStoreDir;
        private final long retention;
        private final TimeUnit retentionTimeUnit;

        CleanerRunner(FileSystem fs, Path stateStoreDir, long retention, TimeUnit retentionTimeUnit) {
            this.fs = fs;
            this.stateStoreDir = stateStoreDir;
            this.retention = retention;
            this.retentionTimeUnit = retentionTimeUnit;
        }

        @Override
        public void run() {
            try {
                FileStatus[] stateStoreFiles = this.fs.listStatus(this.stateStoreDir, (PathFilter)new StateStoreFileFilter());
                if (stateStoreFiles == null || stateStoreFiles.length == 0) {
                    LOGGER.warn("No state store files found in directory: " + this.stateStoreDir);
                    return;
                }
                LOGGER.info("Cleaning up state store directory: " + this.stateStoreDir);
                for (FileStatus file : stateStoreFiles) {
                    if (!this.shouldCleanUp(file) || this.fs.delete(file.getPath(), false)) continue;
                    LOGGER.error("Failed to delete state store file: " + file.getPath());
                }
            }
            catch (IOException ioe) {
                LOGGER.error("Failed to run state store cleaner for directory: " + this.stateStoreDir, (Throwable)ioe);
            }
        }

        private boolean shouldCleanUp(FileStatus file) {
            DateTime now = new DateTime();
            DateTime modificationDateTime = new DateTime(file.getModificationTime());
            long retentionInMills = this.retentionTimeUnit.toMillis(this.retention);
            return modificationDateTime.plus(retentionInMills).isBefore((ReadableInstant)now);
        }
    }

    private static class StateStoreFileFilter
    implements PathFilter {
        private StateStoreFileFilter() {
        }

        public boolean accept(Path path) {
            String fileName = path.getName();
            String extension = Files.getFileExtension((String)fileName);
            return this.isStateMetaFile(fileName) || extension.equalsIgnoreCase("jst") || extension.equalsIgnoreCase("tst");
        }

        boolean isStateMetaFile(String fileName) {
            return fileName.startsWith(GuidDatasetUrnStateStoreNameParser.StateStoreNameVersion.V1.getDatasetUrnNameMapFile()) && !fileName.equals(GuidDatasetUrnStateStoreNameParser.StateStoreNameVersion.V1.getDatasetUrnNameMapFile());
        }
    }
}

