/*****************************************************************************
 * ps.c: PS and PES stream reader
 *****************************************************************************
 * Copyright (C) 1999, 2000 VideoLAN
 *
 * Authors:
 * Samuel Hocevar,     VIA, ECP, France <sam@via.ecp.fr>,    17/12/99
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*
 *  ps.c - PS stream reader
 *  this program is GPLed. Yes.
 */

#include "vlms.h"
#include "ps.h"

static void CalculateCRC( unsigned char *p_begin, unsigned char *p_end );
 
/******************************************************************************
 * ps_thread
 ******************************************************************************
 * We use threading to allow cool non-blocking read from the disk. This
 * implicit thread is the disk (producer) thread, it reads packets from
 * the PS file on the disk, and stores them in a FIFO.
 ******************************************************************************/

void ps_thread(options_t *options)
{
    unsigned int left;
    int i;
    struct s_netthread netthread;
    pthread_t network_tid;
    struct s_ps ps;

    /* 4000 Go should be enough (FIXME: dunno how to do it nicer) */
    netthread.left = left = 0xffffffff;

    /* connect to the network to emit */
    netthread.out = open_socket( options, &netthread.remote_addr );
    /* Initialize the structures */
    own_pcr.start = own_pcr.end = 0; /* empty FIFO */
    pthread_mutex_init(&own_pcr.lock, NULL);
    in_data.start = in_data.end = 0; /* empty FIFO */
    pthread_mutex_init(&in_data.lock, NULL);
    pthread_cond_init(&in_data.notfull, NULL);
    pthread_cond_init(&in_data.notempty, NULL);
    in_data.b_die = 0;


    /* Arguments that were passed
     * in the command line */
    ps.audio_type = options->audio_type;
    ps.audio_channel = options->audio_channel;
    ps.subtitles_channel = options->subtitles_channel;

    /* Main structure initialization */
    ps.pes_type = NO_PES;
    ps.pes_id = 0;
    ps.private_id = 0;
    ps.pes_size = 0;
    ps.to_skip = 0;
    ps.pmt_counter = 0;
    ps.pat_counter = 0;
    for( i=0; i<256; i++ ) ps.association_table[i] = 0;
    ps.offset = 0;
    ps.scr = 0;
    ps.found_scr = 0;
    ps.found_streams = 0;
    ps.pcr_pid = options->pcr_pid;

    ps.ps_buffer = malloc( PS_BUFFER_SIZE );
    /* those 2 addresses are initialized so that a new packet is read */
    ps.ps_data = ps.ps_buffer + PS_BUFFER_SIZE - 1;
    ps.ps_end = ps.ps_buffer + PS_BUFFER_SIZE;

    /* set the first byte to zero because we know it was PS */
    ps.ps_data[0] = 0x00;

    /* The network (consumer) thread must be able to keep on emitting UDP
     * packets even when another process uses the hard disk. So we must fill
     * this buffer before.
     */
    ps_fill( options, &left, 0, &ps );

    /* Start the network thread */
    pthread_create( &network_tid, NULL, network_thread, &netthread );
    ps_fill( options, &left, 1, &ps );

    in_data.b_die = 1;
    /* Wait for the network thread to terminate */
    pthread_join( network_tid, NULL );
}

/******************************************************************************
 * ps_fill : Fill the data buffer with TS created from a PS file
 ******************************************************************************/

void ps_fill( options_t *options, unsigned int *left, int wait, ps_t *ps )
{
    int i, how_many;
    int pcr_flag;
    file_ts_packet *ts;

    /* How many TS packet for the next UDP packet */
    how_many = TS_IN_UDP;

    pcr_flag = 0;
    /* for every single TS packet */
    while( *left )
    {
        if( *left < TS_IN_UDP )
            how_many = *left;

        /* wait until we have one free item to store the UDP packet read */
        pthread_mutex_lock( &in_data.lock );
        while( (in_data.end+TS_BUFFER_SIZE+1-in_data.start)%(TS_BUFFER_SIZE+1) == TS_BUFFER_SIZE )
        {
            /* The buffer is full */
            if( wait )
            {
                pthread_cond_wait(&in_data.notfull, &in_data.lock);
            }
            else
            {
                pthread_mutex_unlock( &in_data.lock );
                if( !pcr_flag )
                {
                    printf( "Bad PCR PID\n" );
                    exit( 1 );
                }
                return;
            }
        }
        pthread_mutex_unlock(&in_data.lock);

        /* read a whole UDP packet from the file */
        ps->ts_to_write = how_many;
        if( ps_read( options, ps, ts = (file_ts_packet *)(in_data.buf + in_data.end) ) != how_many )
        {
            return;
        }

        /* Scan to mark TS packets containing a PCR */
        for( i=0; i<how_many; i++, ts++ )
        {
            pcr_flag |= keep_pcr( ps->pcr_pid, ts );
        }

        pthread_mutex_lock( &in_data.lock );
        in_data.end++;
        in_data.end %= TS_BUFFER_SIZE+1;
        pthread_cond_signal( &in_data.notempty );
        pthread_mutex_unlock( &in_data.lock );
        *left -= how_many;
    }
}

/******************************************************************************
 * ps_read : ps reading method
 ******************************************************************************/

ssize_t ps_read( options_t *options, ps_t *ps, void *ts )
{
    int i, pid, readbytes = 0;
    int datasize;
    ps->ts_written = 0;

    while( ps->ts_to_write )
    {

        /* Check that there is enough data to send,
         * otherwise read a few more bytes in the buffer */

        if( (datasize = ps->ps_end - ps->ps_data) <= TS_PACKET_SIZE )
        {
            /* copy the remaining bits at the beginning of the PS buffer */
            memmove ( ps->ps_buffer, ps->ps_data, datasize );

            /* read some bytes */
            readbytes = safe_read( options, ps->ps_buffer + datasize,
                                   PS_BUFFER_SIZE - datasize );

            ps->ps_data = ps->ps_buffer;
            ps->ps_end = ps->ps_data + datasize + readbytes;

            if(readbytes == 0 && ps->ps_end - ps->ps_data < TS_PACKET_SIZE)
            {
                fprintf ( stderr, "end of PS file\n" );
                return -1;
            }
        }

        /* If we are at the beginning of a new MPEG sequence, check
         * that it says 00 00 01 xx, then read its PTS if it is a PES */

        if( ps->to_skip == 0 && ps->offset == ps->pes_size )
        {
            while( ps->ps_data[0] || ps->ps_data[1] || (ps->ps_data[2] != 0x01)  || (ps->ps_data[3] < 0xb9) )
            {
                fprintf ( stderr, "Error: not a startcode: %.2x %.2x %.2x "
                          "instead of 00 00 01\n", ps->ps_data[0],
                          ps->ps_data[1], ps->ps_data[2] );
                ps->ps_data++;
                if( (datasize = ps->ps_end - ps->ps_data) <= TS_PACKET_SIZE )
                {
                    memmove( ps->ps_buffer, ps->ps_data, datasize );
                    readbytes = safe_read( options, ps->ps_buffer + datasize,
                                           PS_BUFFER_SIZE - datasize );
                    if(readbytes == 0)
                    {
                        fprintf ( stderr, "end of PS file 2\n" );
                        return -1;
                    }
                    ps->ps_data = ps->ps_buffer;
                    ps->ps_end = ps->ps_data + datasize + readbytes;
                }
            }

            if( ps->ps_data[3] == 0xba )
            {
                /* Pack header : Read the SCR */
                if( (ps->ps_data[4] & 0xC0) == 0x40 )
                {
                    /* MPEG-2 */
                    ps->scr = ((u64)(ps->ps_data[4] & 0x38) << 27) |
                              ((u64)(ps->ps_data[4] & 0x3) << 28) |
                              ((u64)(ps->ps_data[5] & 0xff) << 20) |
                              ((u64)(ps->ps_data[6] & 0xf8) << 12) |
                              ((u64)(ps->ps_data[6] & 0x3) << 13) |
                              ((u64)(ps->ps_data[7] & 0xff) << 5) |
                              ((u64)(ps->ps_data[8] & 0xf8) >> 3);
                }
                else
                {
                    /* MPEG-1 */
                    ps->scr = ((u64)(ps->ps_data[4] & 0x0E) << 29) |
                         ((u64)ps->ps_data[5] << 22) |
                         (((u64)ps->ps_data[6] & 0xFE) << 14) |
                         ((u64)ps->ps_data[7] << 7) |
                         ((u64)ps->ps_data[8] >> 1);
                }
            }

            ps->pes_type = NO_PES;
            ps->offset = 0;
            ps->pes_size = (ps->ps_data[4] << 8) + ps->ps_data[5] + 6;
        }

        /* if the actual data we have in pes_data is not a PES, then
         * we read the next one. */
        if( (ps->pes_type == NO_PES) && !ps->to_skip )
        {
            ps->pes_id = ps->ps_data[3];

            if (ps->pes_id == 0xbd)
            {
                ps->private_id = ps->ps_data[ 9 + ps->ps_data[8] ];
                if ((ps->private_id & 0xf0) == 0x80)
                {
                    /* ac3 audio stream (0x80 - 0x8f) */
                    ps->pes_type = AC3_PES;
                }
                else if ((ps->private_id & 0xe0) == 0x20)
                {
                    /* subtitles (0x20 - 0x3f) */
                    ps->pes_type = SUBTITLE_PES;
                }
                else if ((ps->private_id & 0xf0) == 0xa0)
                {
                    /* lpcm audio stream (0xa0 - 0xaf) */
                    ps->pes_type = LPCM_PES;
                }
                else
                {
                    /* unknown private data */
                    ps->pes_type = PRIVATE_PES;
                    ps->to_skip = ps->pes_size;
                }
            }
            else if ((ps->pes_id & 0xe0) == 0xc0)
            {
                /* audio stream */
                ps->pes_type = AUDIO_PES;
            }
            else if ((ps->pes_id & 0xf0) == 0xe0)
            {
                /* video stream */
                ps->pes_type = VIDEO_PES;
            }
            else if (ps->pes_id == 0xba)
            {
                /* pack header */
                int i = 4;
                ps->pes_type = NO_PES;

                if( (ps->ps_data[i] & 0xC0) == 0x40 )
                {
                    i += 10 + ( ps->ps_data[i+9] & 0x07 );
                }
                else
                {
                    i += 8;
                }
                ps->pes_size = ps->to_skip = i;
            }
            else if (ps->pes_id == 0xb9)
            {
                /* sequence end stream */
                ps->pes_size = ps->to_skip = 4;
            }
            else
            {
                ps->pes_type = UNKNOWN_PES;
                ps->to_skip = ps->pes_size;
            }
        }

        if( ps->to_skip )
        {
            if( ps->to_skip < TS_PACKET_SIZE )
            {
                ps->ps_data += ps->to_skip;
                ps->offset  += ps->to_skip;
                ps->to_skip  = 0;
            }
            else
            {
                ps->ps_data += TS_PACKET_SIZE;
                ps->offset  += TS_PACKET_SIZE;
                ps->to_skip -= TS_PACKET_SIZE;
            }
        }

        /* now that we know what we have, we can either
         * write this packet's data in the buffer, skip it,
         * or write a PMT or PAT table and wait for the next
         * turn before writing the packet. */
        switch (ps->sent_ts & 0xff)
        {
            case 0x01:
                write_pmt(ps,ts);
                ps->pmt_counter++;
                ps->ts_to_write--; ps->ts_written++; ts+=188;
                break;

            case 0x00:
                write_pat(ps,ts);
                ps->pat_counter++;
                ps->ts_to_write--; ps->ts_written++; ts+=188;
                break;
        }

        /* if there's still no found PCR_PID, and no PTS in this PES,
         * then we just trash it */
        if (!ps->found_scr)
        {
            if (ps->scr)
            {
                fprintf( stderr, "Found a PTS - entering main loop\n" );
                ps->found_scr = 1;
            }
            else
            {
                ps->pes_type = NO_PES;
            }
        }

        if (ps->ts_to_write)
        {
            switch(ps->pes_type)
            {
                case VIDEO_PES:
                case AUDIO_PES:
                case AC3_PES:
                case SUBTITLE_PES:
                case LPCM_PES:
                    pid = get_pid (ps);
                    write_media_ts(ps, ts, pid);
                    ps->ts_to_write--; ps->ts_written++; ts+=188;
                    ps->media_counter[pid]++;
                    break;
                case UNKNOWN_PES:
                default:
                    //fprintf(stderr, "setting PES type to NO_PES\n");
                    ps->pes_type = NO_PES;
                    break;
            }
        }
    }

    //ps->ps_data += TS_PACKET_SIZE;

    return ps->ts_written;
}

/******************************************************************************
 * get_pid : gets a pid from a PES type
 ******************************************************************************/

int get_pid (ps_t *ps)
{
    int i, tofind, delta;
    char* type;

    switch(ps->pes_type)
    {
        case VIDEO_PES:
            delta = 0x20; /* 0x20 - 0x2f */
            type = "MPEG video";
            tofind = ps->pes_id;
            break;
        case AUDIO_PES:
            delta = 0x40; /* 0x40 - 0x5f */
            type = "MPEG audio";
            tofind = ps->pes_id;
            break;
        /* XXX: 0x64 is for the PMT, so don't take it !!! */
        case AC3_PES:
            delta = 0x80; /* 0x80 - 0x8f */
            type = "MPEG private (AC3 audio)";
            tofind = ps->private_id;
            break;
        case LPCM_PES:
            delta = 0x90; /* 0x90 - 0x9f */
            type = "MPEG private (LPCM audio)";
            tofind = ps->private_id;
            break;
        case SUBTITLE_PES:
            delta = 0xa0; /* 0xa0 - 0xbf */
            type = "MPEG private (DVD subtitle)";
            tofind = ps->private_id;
            break;
        default:
            return(-1);
    }

    i = delta + (tofind & 0x1f);

    if( ps->association_table[i] == 0)
    {
        fprintf( stderr, "Found %s stream at 0x%.2x, allocating PID 0x%.2x\n",
                 type, tofind, i );
        ps->association_table[i] = 1;
    }

    return ( i );

}

/******************************************************************************
 * write_media_ts : writes a ts packet from a ps stream
 ******************************************************************************/

void write_media_ts(ps_t *ps, unsigned char *ts, unsigned int pid)
{
    int i,j;
    long int extclock;

    /* if offset == 0, it means we haven't examined the PS yet */
    if (ps->offset == 0)
    {
        if (ps->pes_size < 184) {

            ts[0] = 0x47;    /* sync_byte */
            ts[1] = 0x40;    /* payload_unit_start_indicator si dbut de PES */
            ts[2] = pid;

            ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
            ts[4] = 184 - ps->pes_size - 1;
            ts[5] = 0x00;
            /* needed stuffing bytes */
            for (i=6 ; i < 188 - ps->pes_size ; i++) ts[i]=0xff;
            memcpy(ts + 188 - ps->pes_size, ps->ps_data, ps->pes_size);

            /* this PS is finished, next time we'll pick a new one */
            ps->pes_type = NO_PES;
            ps->ps_data += ps->pes_size;
            ps->offset += ps->pes_size;
            return;

        }
    }

    /* now we still can have offset == 0, but size is initialized */

    /* sync_byte */
    ts[0] = 0x47;
    /* payload_unit_start_indicator si dbut de PES */
    ts[1] = (ps->offset == 0) ? 0x40 : 0x00;
    /* the PID */
    ts[2] = pid;

    if ( (ps->offset == 0) && ps->scr && (ps->pcr_pid == pid) )
    {

        ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
        ts[4] = 0x07; /* taille de l'adaptation field */
        ts[5] = 0x50; /* rtfm */

        //fprintf( stderr, "writing PTS %.16llx\n", ps->scr );
        extclock = 0x000;

        /* ---111111110000000000000000000000000 */
        ts[6] = (ps->scr & 0x1fe000000) >> 25;
        /* ---000000001111111100000000000000000 */
        ts[7] = (ps->scr & 0x001fe0000) >> 17;
        /* ---000000000000000011111111000000000 */
        ts[8] = (ps->scr & 0x00001fe00) >>  9;
        /* ---000000000000000000000000111111110 */
        ts[9] = (ps->scr & 0x0000001fe) >>  1;

        ts[10] = 0x7e + ((ps->scr & 0x01) << 7) + ((extclock & 0x100) >> 8);
        ts[11] = extclock & 0xff;

        ps->scr = 0;

        memcpy( ts + 4 + 8, ps->ps_data, 184 - 8 );

        ts[15] = 0xe0; /* FIXME : we don't know how to choose program yet */

        ps->offset += 184 - 8;
        ps->ps_data += 184 - 8;
    }
    else if (ps->offset <= ps->pes_size - 184)
    {

        ts[3] = 0x10 + (ps->media_counter[pid] & 0x0f);
        memcpy(ts + 4, ps->ps_data, 184);

        ps->offset += 184;
        ps->ps_data += 184;

    }
    else
    {
        j = ps->pes_size - ps->offset;
        ts[3] = 0x30 + (ps->media_counter[pid] & 0x0f);
        ts[4] = 184 - j - 1;
        ts[5] = 0x00;
        for (i=6 ; i < 188 - j ; i++) ts[i]=0xff; /* needed stuffing bytes */
        memcpy(ts + 4 + 184 - j, ps->ps_data, j);
        ps->offset += j; /* offset = size */
        ps->ps_data += j; /* offset = size */

        /* the PES is finished */
        ps->pes_type = NO_PES;
        ps->sent_ts++;
    }

    //fprintf(stderr, "[PES size: %i]\n", ps->pes_size);
}

/******************************************************************************
 * write_pat : writes a program association table
 ******************************************************************************/

void write_pat(ps_t *ps, unsigned char *ts)
{
    int i;

    ts[0x00] = 0x47; /* sync_byte */
    ts[0x01] = 0x40;
    ts[0x02] = 0x00; /* PID = 0x0000 */
    ts[0x03] = 0x10 | (ps->pat_counter & 0x0f);
    ts[0x04] = 0x00; /* CRC calculation begins here */
    
    ts[0x05] = 0x00; /* 0x00: Program association section */

    ts[0x06] = 0xb0; /* */
    ts[0x07] = 0x11; /* section_length = 0x011 */

    ts[0x08] = 0x00;
    ts[0x09] = 0xbb; /* TS id = 0x00b0 (what the vlc calls "Stream ID") */

    ts[0x0a] = 0xc1;
    /* section # and last section # */
    ts[0x0b] = ts[0x0c] = 0x00;

    /* Network PID (useless) */
    ts[0x0d] = ts[0x0e] = 0x00; ts[0x0f] = 0xe0; ts[0x10] = 0x10;

    /* Program Map PID */
    ts[0x11] = 0x03; ts[0x12] = 0xe8; ts[0x13] = 0xe0; ts[0x14] = 0x64;

    /* Put CRC in ts[0x15...0x18] */
    CalculateCRC( ts + 0x05, ts + 0x15 );

    /* needed stuffing bytes */
    for (i=0x19 ; i < 188 ; i++) ts[i]=0xff;

    ps->sent_ts++;
}

/******************************************************************************
 * write_pmt : writes a program map table
 ******************************************************************************/

void write_pmt(ps_t *ps, unsigned char *ts)
{
    int i;

    ts[0x00] = 0x47; /* sync_byte */
    ts[0x01] = 0x40;
    ts[0x02] = 0x0064; /* PID = 0x0064 */
    ts[0x03] = 0x10 | (ps->pmt_counter & 0x0f);
    ts[0x04] = 0x00; /* CRC calculation begins here */

    ts[0x05] = 0x02; /* 0x02: Program map section */

    ts[0x06] = 0xb0; /* */
    ts[0x07] = 0x25; /* section_length = 0x025 */

    ts[0x08] = 0x03;
    ts[0x09] = 0xe8; /* prog number */

    ts[0x0a] = 0xc1;
    /* section # and last section # */
    ts[0x0b] = ts[0x0c] = 0x00;

    /* PCR PID */
    ts[0x0d] = 0xe0;
    ts[0x0e] = 0x20;

    /* program_info_length == 0 */
    ts[0x0f] = 0xf0; ts[0x10] = 0x00;

    /* Program Map / Video PID */
    ts[0x11] = 0x02; /* stream type = video */
    ts[0x12] = 0xe0; ts[0x13] = 0x20;
    ts[0x14] = 0xf0; ts[0x15] = 0x09; /* es info length */
    /* useless info */
    ts[0x16] = 0x07; ts[0x17] = 0x04; ts[0x18] = 0x08; ts[0x19] = 0x80;
    ts[0x1a] = 0x24; ts[0x1b] = 0x02; ts[0x1c] = 0x11; ts[0x1d] = 0x01;
    ts[0x1e] = 0xfe;

    switch ( ps->audio_type )
    {
    case REQUESTED_AC3 :
        /* ac3 */
        ts[0x1f] = 0x91; /* stream type = audio */
        ts[0x20] = 0xe0; ts[0x21] = 0x80 + ps->audio_channel;
        ts[0x22] = 0xf0; ts[0x23] = 0x00; /* es info length */
        break;

    case REQUESTED_MPEG :
        /* mpeg */
        ts[0x1f] = 0x04; /* stream type = audio */
        ts[0x20] = 0xe0; ts[0x21] = 0x40 + ps->audio_channel;
        ts[0x22] = 0xf0; ts[0x23] = 0x00; /* es info length */
        break;

    case REQUESTED_LPCM :
        /* LPCM audio */
        ts[0x1f] = 0x93;
        ts[0x20] = 0xe0; ts[0x21] = 0xa0 + ps->audio_channel;
        ts[0x22] = 0xf0; ts[0x23] = 0x00; /* es info  length */
        break;

    default :
        /* No audio */
        ts[0x1f] = 0x00;
        ts[0x23] = 0x00; /* es info  length */
    }

    /* Subtitles */
    if( ps->subtitles_channel == NO_SUBTITLES )
    {
        ts[0x24] = 0x00;
        ts[0x25] = 0x00; ts[0x26] = 0x00;
        ts[0x27] = 0xf0; ts[0x28] = 0x00; /* es info length */
    }
    else
    {
        ts[0x24] = 0x92;
        ts[0x25] = 0xe0; ts[0x26] = 0xa0 + ( ps->subtitles_channel );
        ts[0x27] = 0xf0; ts[0x28] = 0x00; /* es info length */
        //fprintf(stderr, "subtitle %.2x\n", 0xa0 + ( ps->subtitles_channel ) );
    }

    /* Put CRC in ts[0x29...0x2c] */
    CalculateCRC( ts + 0x05, ts + 0x29 );

    /* needed stuffing bytes */
    for (i=0x2d ; i < 188 ; i++) ts[i]=0xff;

    ps->sent_ts++;
}

static void CalculateCRC( unsigned char *p_begin, unsigned char *p_end )
{
    static unsigned int CRC32[256] =
    {
        0x00000000, 0x04c11db7, 0x09823b6e, 0x0d4326d9,
        0x130476dc, 0x17c56b6b, 0x1a864db2, 0x1e475005,
        0x2608edb8, 0x22c9f00f, 0x2f8ad6d6, 0x2b4bcb61,
        0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd,
        0x4c11db70, 0x48d0c6c7, 0x4593e01e, 0x4152fda9,
        0x5f15adac, 0x5bd4b01b, 0x569796c2, 0x52568b75,
        0x6a1936c8, 0x6ed82b7f, 0x639b0da6, 0x675a1011,
        0x791d4014, 0x7ddc5da3, 0x709f7b7a, 0x745e66cd,
        0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
        0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5,
        0xbe2b5b58, 0xbaea46ef, 0xb7a96036, 0xb3687d81,
        0xad2f2d84, 0xa9ee3033, 0xa4ad16ea, 0xa06c0b5d,
        0xd4326d90, 0xd0f37027, 0xddb056fe, 0xd9714b49,
        0xc7361b4c, 0xc3f706fb, 0xceb42022, 0xca753d95,
        0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1,
        0xe13ef6f4, 0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d,
        0x34867077, 0x30476dc0, 0x3d044b19, 0x39c556ae,
        0x278206ab, 0x23431b1c, 0x2e003dc5, 0x2ac12072,
        0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16,
        0x018aeb13, 0x054bf6a4, 0x0808d07d, 0x0cc9cdca,
        0x7897ab07, 0x7c56b6b0, 0x71159069, 0x75d48dde,
        0x6b93dddb, 0x6f52c06c, 0x6211e6b5, 0x66d0fb02,
        0x5e9f46bf, 0x5a5e5b08, 0x571d7dd1, 0x53dc6066,
        0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
        0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e,
        0xbfa1b04b, 0xbb60adfc, 0xb6238b25, 0xb2e29692,
        0x8aad2b2f, 0x8e6c3698, 0x832f1041, 0x87ee0df6,
        0x99a95df3, 0x9d684044, 0x902b669d, 0x94ea7b2a,
        0xe0b41de7, 0xe4750050, 0xe9362689, 0xedf73b3e,
        0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2,
        0xc6bcf05f, 0xc27dede8, 0xcf3ecb31, 0xcbffd686,
        0xd5b88683, 0xd1799b34, 0xdc3abded, 0xd8fba05a,
        0x690ce0ee, 0x6dcdfd59, 0x608edb80, 0x644fc637,
        0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb,
        0x4f040d56, 0x4bc510e1, 0x46863638, 0x42472b8f,
        0x5c007b8a, 0x58c1663d, 0x558240e4, 0x51435d53,
        0x251d3b9e, 0x21dc2629, 0x2c9f00f0, 0x285e1d47,
        0x36194d42, 0x32d850f5, 0x3f9b762c, 0x3b5a6b9b,
        0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
        0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623,
        0xf12f560e, 0xf5ee4bb9, 0xf8ad6d60, 0xfc6c70d7,
        0xe22b20d2, 0xe6ea3d65, 0xeba91bbc, 0xef68060b,
        0xd727bbb6, 0xd3e6a601, 0xdea580d8, 0xda649d6f,
        0xc423cd6a, 0xc0e2d0dd, 0xcda1f604, 0xc960ebb3,
        0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7,
        0xae3afba2, 0xaafbe615, 0xa7b8c0cc, 0xa379dd7b,
        0x9b3660c6, 0x9ff77d71, 0x92b45ba8, 0x9675461f,
        0x8832161a, 0x8cf30bad, 0x81b02d74, 0x857130c3,
        0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640,
        0x4e8ee645, 0x4a4ffbf2, 0x470cdd2b, 0x43cdc09c,
        0x7b827d21, 0x7f436096, 0x7200464f, 0x76c15bf8,
        0x68860bfd, 0x6c47164a, 0x61043093, 0x65c52d24,
        0x119b4be9, 0x155a565e, 0x18197087, 0x1cd86d30,
        0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
        0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088,
        0x2497d08d, 0x2056cd3a, 0x2d15ebe3, 0x29d4f654,
        0xc5a92679, 0xc1683bce, 0xcc2b1d17, 0xc8ea00a0,
        0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb, 0xdbee767c,
        0xe3a1cbc1, 0xe760d676, 0xea23f0af, 0xeee2ed18,
        0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4,
        0x89b8fd09, 0x8d79e0be, 0x803ac667, 0x84fbdbd0,
        0x9abc8bd5, 0x9e7d9662, 0x933eb0bb, 0x97ffad0c,
        0xafb010b1, 0xab710d06, 0xa6322bdf, 0xa2f33668,
        0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4
    };

    unsigned int i_crc = 0xffffffff;

    /* Calculate the CRC */
    while( p_begin < p_end )
    {
        i_crc = (i_crc<<8) ^ CRC32[ (i_crc>>24) ^ ((unsigned int)*p_begin) ];
        p_begin++;
    }

    /* Store it after the data */
    p_end[0] = (i_crc >> 24) & 0xff;
    p_end[1] = (i_crc >> 16) & 0xff;
    p_end[2] = (i_crc >>  8) & 0xff;
    p_end[3] = (i_crc >>  0) & 0xff;
}

