/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isnan = require( './../../../../base/assert/is-nan' );
var abs = require( './../../../../base/special/abs' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var betaincinv = require( './../lib' );


// FIXTURES //

var fixtures = require( './fixtures/cpp/output.json' );
var lowerRegularized = fixtures.lower_regularized;
var upperRegularized = fixtures.upper_regularized;
var x = fixtures.x;
var a = fixtures.a;
var b = fixtures.b;


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof betaincinv, 'function', 'main export is a function' );
	t.end();
});

tape( 'if provided `NaN` for any parameter, the function returns `NaN`', function test( t ) {
	var y = betaincinv( NaN, 1.0, 1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.2, NaN, 1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.2, 1.0, NaN );
	t.equal( isnan( y ), true, 'returns NaN' );
	t.end();
});

tape( 'the function returns `NaN` if `p` is outside the interval `[0,1]`', function test( t ) {
	var y = betaincinv( 1.5, 1.0, 1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( -0.5, 1.0, 1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );
	t.end();
});

tape( 'if provided a nonpositive `a`, the function returns `NaN`', function test( t ) {
	var y;

	y = betaincinv( 0.5, 0.0, 2.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, -1.0, 2.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, -1.0, 2.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NINF, 1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NINF, PINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NINF, NINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NINF, NaN );
	t.equal( isnan( y ), true, 'returns NaN' );

	t.end();
});

tape( 'if provided a nonpositive `b`, the function returns `NaN`', function test( t ) {
	var y;

	y = betaincinv( 0.5, 2.0, 0.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, 2.0, -1.0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, 2.0, -1/0 );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, 1.0, NINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, PINF, NINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NINF, NINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	y = betaincinv( 0.5, NaN, NINF );
	t.equal( isnan( y ), true, 'returns NaN' );

	t.end();
});

tape( 'the function evaluates the inverse of the lower regularized incomplete beta function', function test( t ) {
	var expected;
	var delta;
	var tol;
	var i;
	var y;

	expected = lowerRegularized;
	for ( i = 0; i < x.length; i++ ) {
		y = betaincinv( x[i], a[i], b[i] );
		if ( y === expected[i] ) {
			t.equal( y, expected[i], 'x: '+x[i]+', y: '+y+'. a: '+a[i]+'. b: '+b[i]+', expected: '+expected[i] );
		} else {
			delta = abs( y - expected[ i ] );
			tol = 15.0 * EPS * abs( expected[ i ] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. a: '+a[i]+'. b: '+b[i]+'. y: '+y+'. E: '+expected[i]+'. Δ: '+delta+'. tol: '+tol );
		}
	}
	t.end();
});

tape( 'the function evaluates the inverse of the upper regularized incomplete beta function', function test( t ) {
	var expected;
	var delta;
	var tol;
	var i;
	var y;

	expected = upperRegularized;
	for ( i = 0; i < x.length; i++ ) {
		y = betaincinv( x[i], a[i], b[i], true );
		if ( y === expected[i] ) {
			t.equal( y, expected[i], 'x: '+x[i]+', y: '+y+'. a: '+a[i]+'. b: '+b[i]+', expected: '+expected[i] );
		} else {
			delta = abs( y - expected[ i ] );
			tol = 15.0 * EPS * abs( expected[ i ] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. a: '+a[i]+'. b: '+b[i]+'. y: '+y+'. E: '+expected[i]+'. Δ: '+delta+'. tol: '+tol );
		}
	}
	t.end();
});
