/*
 * BSD 2-Clause License
 *
 * Copyright (c) 2020, Agnieszka Cicha-Cisek
 * Copyright (c) 2020, Patryk Cisek
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef GETTOTPCODERESPONSE_H
#define GETTOTPCODERESPONSE_H

#include <QString>

/**
 * @brief Represents TOTP Code response from the USB key.
 */
struct GetTOTPCodeResponse
{
    /**
     * @brief Constructs instance of the response.
     * @param totpCode TOTP code generated by the key.
     * @param slotNumber Key slot number for which the code's been
     * generated.
     * @param unixTimestamp UNIX timestamp (number of seconds since
     * Jan 1st, 1970) for which the code has been generated.
     */
    GetTOTPCodeResponse(const QString &totpCode,
                        const std::uint8_t slotNumber,
                        const unsigned int unixTimestamp)
        : TOTP_CODE(totpCode),
          SLOT_NUMBER(slotNumber),
          UNIX_TIMESTAMP(unixTimestamp) {}

    /**
     * @brief Default constructor -- we don't use it in code, but
     * it's required by Qt's MetaType mechanism (allows emitting
     * signals with this response struct as parameter).
     */
    GetTOTPCodeResponse()
        : SLOT_NUMBER(-1),
          UNIX_TIMESTAMP(0) {}

    /**
     * @brief Default copy-constructor. Also required by Qt's MetaType
     * mechanise (see description on default constructor).
     */
    GetTOTPCodeResponse(const GetTOTPCodeResponse &rhs)
        : TOTP_CODE(rhs.TOTP_CODE),
          SLOT_NUMBER(rhs.SLOT_NUMBER),
          UNIX_TIMESTAMP(rhs.UNIX_TIMESTAMP) {}

    /**
     * @brief Default destructor.
     */
    ~GetTOTPCodeResponse() = default;

    /**
     * @brief TOTP code generated by the USB key.
     */
    const QString TOTP_CODE;

    /**
     * @brief Slot number on the USB key for which the code
     * has been generated.
     */
    const std::uint8_t SLOT_NUMBER;

    /**
     * @brief Timestamp for which the code has been generated (seconds
     * since Jan 1st, 1970).
     */
    const unsigned int UNIX_TIMESTAMP;
};

#endif // GETTOTPCODERESPONSE_H
