package image

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"io"
	"strings"
	"testing"

	"github.com/moby/go-archive"
	"github.com/moby/moby/api/types/jsonstream"
	"github.com/moby/moby/client"
	"github.com/moby/moby/v2/internal/testutil/specialimage"
	"gotest.tools/v3/assert"
)

func Load(ctx context.Context, t *testing.T, apiClient client.APIClient, imageFunc specialimage.SpecialImageFunc) string {
	tempDir := t.TempDir()

	_, err := imageFunc(tempDir)
	assert.NilError(t, err)

	rc, err := archive.TarWithOptions(tempDir, &archive.TarOptions{})
	assert.NilError(t, err)

	defer rc.Close()

	resp, err := apiClient.ImageLoad(ctx, rc, client.ImageLoadWithQuiet(true))
	assert.NilError(t, err, "Failed to load dangling image")

	defer func() { _ = resp.Close() }()

	if !assert.Check(t, err) {
		respBody, err := io.ReadAll(resp)
		if err != nil {
			t.Fatalf("Failed to read response body: %v", err)
			return ""
		}
		t.Fatalf("Failed load: %s", string(respBody))
	}

	all, err := io.ReadAll(resp)
	assert.NilError(t, err)

	decoder := json.NewDecoder(bytes.NewReader(all))
	for {
		var msg jsonstream.Message
		err := decoder.Decode(&msg)
		if errors.Is(err, io.EOF) {
			break
		}
		assert.NilError(t, err)

		msg.Stream = strings.TrimSpace(msg.Stream)

		if _, imageID, hasID := strings.Cut(msg.Stream, "Loaded image ID: "); hasID {
			return imageID
		}
		if _, imageRef, hasRef := strings.Cut(msg.Stream, "Loaded image: "); hasRef {
			return imageRef
		}
	}

	t.Fatalf("failed to read image ID\n%s", string(all))
	return ""
}
