/*
 * UFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * wb_presets.c - White balance preset values for various cameras
 * Copyright 2004-2013 by Udi Fuchs
 *
 * Thanks goes for all the people who sent in the preset values
 * for their cameras.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>
#include <glib/gi18n.h>

typedef struct
{
  const char *make;
  const char *model;
  const char *name;
  int tuning;
  double channel[4];
}
wb_data;

/* Column 1 - "make" of the camera.
 * Column 2 - "model" (use the "make" and "model" as provided by DCRaw).
 * Column 3 - WB name.
 * Column 4 - Fine tuning. MUST be in increasing order. 0 for no fine tuning.
 *	      It is enough to give only the extreme values, the other values
 *	      will be interpolated.
 * Column 5 - Channel multipliers.
 *
 * Minolta's ALPHA and MAXXUM models are treated as the Dynax model.
 *
 * WB name is standardized to one of the following: */
// "Sunlight" and other variation should be switched to this:
static const char Daylight[] = N_("daylight");
// Probably same as above:
static const char DirectSunlight[] = N_("direct sunlight");
static const char Cloudy[] = N_("cloudy");
// "Shadows" should be switched to this:
static const char Shade[] = N_("shade");
static const char Incandescent[] = N_("incandescent");
static const char IncandescentWarm[] = N_("incandescent warm");
// Same as "Incandescent":
static const char Tungsten[] = N_("tungsten");
static const char Fluorescent[] = N_("fluorescent");
// In Canon cameras and some newer Nikon cameras:
static const char FluorescentHigh[] = N_("fluorescent high");
static const char CoolWhiteFluorescent[] = N_("cool white fluorescent");
static const char WarmWhiteFluorescent[] = N_("warm white fluorescent");
static const char DaylightFluorescent[] = N_("daylight fluorescent");
static const char NeutralFluorescent[] = N_("neutral fluorescent");
static const char WhiteFluorescent[] = N_("white fluorescent");
// In some newer Nikon cameras:
static const char SodiumVaporFluorescent[] = N_("sodium-vapor fluorescent");
static const char DayWhiteFluorescent[] = N_("day white fluorescent");
static const char HighTempMercuryVaporFluorescent[] = N_("high temp. mercury-vapor fluorescent");
// Found in Nikon Coolpix P1000
static const char HTMercury[] = N_("high temp. mercury-vapor");
// On Some Panasonic
static const char D55[] = N_("D55");

static const char Flash[] = N_("flash");
// For Olympus with no real "Flash" preset:
static const char FlashAuto[] = N_("flash (auto mode)");
static const char EveningSun[] = N_("evening sun");
static const char Underwater[] = N_("underwater");
static const char BlackNWhite[] = N_("black & white");

const char uf_spot_wb[] = N_("spot WB");
const char uf_manual_wb[] = N_("manual WB");
const char uf_camera_wb[] = N_("camera WB");
const char uf_auto_wb[] = N_("auto WB");

const wb_data wb_preset[] =
{


  { "Canon", "PowerShot A630", Daylight, 0,	{ 1.831422, 1, 1.245671, 0 } },
  { "Canon", "PowerShot A630", Cloudy, 0,	{ 1.669924, 1, 1.326299, 0 } },
  { "Canon", "PowerShot A630", Tungsten, 0,	{ 1.696768, 1, 1.268658, 0 } },
  { "Canon", "PowerShot A630", Fluorescent, 0,	{ 1.869859, 1, 1.209110, 0 } },
  { "Canon", "PowerShot A630", FluorescentHigh, 0, { 1.855491, 1, 1.206855, 0 } },

  { "Canon", "PowerShot A710 IS", Daylight, 0,	{ 1.683007, 1, 1.893246, 0 } },
  { "Canon", "PowerShot A710 IS", Cloudy, 0,	{ 1.871320, 1, 1.718648, 0 } },
  { "Canon", "PowerShot A710 IS", Tungsten, 0,	{ 1.268692, 1, 2.707944, 0 } },
  { "Canon", "PowerShot A710 IS", Fluorescent, 0, { 1.589857, 1, 2.051819, 0 } },
  { "Canon", "PowerShot A710 IS", DaylightFluorescent, 0, { 1.820287, 1, 1.820287, 0 } },
  { "Canon", "PowerShot A710 IS", Underwater, 0, { 2.926108, 1, 1.376847, 0 } },

  /* These presets were extracted from CHDK-generated DNG files. */
  { "Canon", "PowerShot A720 IS", Daylight, 0,	{ 2.059371, 1, 1.975553, 0 } },
  { "Canon", "PowerShot A720 IS", Cloudy, 0,	{ 2.262722, 1, 1.818935, 0 } },
  { "Canon", "PowerShot A720 IS", Tungsten, 0,	{ 1.632258, 1, 2.974194, 0 } },
  { "Canon", "PowerShot A720 IS", Fluorescent, 0, { 1.902728, 1, 2.365362, 0 } },
  { "Canon", "PowerShot A720 IS", FluorescentHigh, 0, { 2.243961, 1, 1.935990, 0 } },
  { "Canon", "PowerShot A720 IS", Underwater, 0, { 2.303465, 1, 1.882915, 0 } },
  { "Canon", "PowerShot A720 IS", Flash, 0,	{ 2.151586, 1, 1.864865, 0 } },

  { "Canon", "PowerShot G1 X", Daylight, 0,    { 1.95087336244541, 1, 1.72052401746725, 0 } },
  { "Canon", "PowerShot G1 X", Shade, 0,       { 2.12305986696231, 1, 1.57871396895787, 0 } },
  { "Canon", "PowerShot G1 X", Cloudy, 0,      { 2.12305986696231, 1, 1.57871396895787, 0 } },
  { "Canon", "PowerShot G1 X", Tungsten, 0,    { 1.24484848484848, 1, 3.17575757575758, 0 } },
  { "Canon", "PowerShot G1 X", Fluorescent, 0, { 1.67480577136515, 1, 2.21642619311876, 0 } },
  { "Canon", "PowerShot G1 X", Flash, 0,       { 2.46899661781285, 1, 1.29988726042841, 0 } },

  { "Canon", "PowerShot G1 X Mark II", Daylight, 0, { 1.9815217391304347, 1, 1.5793478260869565, 0 } },
  { "Canon", "PowerShot G1 X Mark II", Shade, 0, { 2.258771929824561, 1, 1.3508771929824561, 0 } },
  { "Canon", "PowerShot G1 X Mark II", Cloudy, 0, { 2.1383442265795205, 1, 1.4357298474945535, 0 } },
  { "Canon", "PowerShot G1 X Mark II", Tungsten, 0, { 1.3710359408033828, 1, 2.464059196617336, 0 } },
  { "Canon", "PowerShot G1 X Mark II", Fluorescent, 0, { 1.8006795016987542, 1, 2.060022650056625, 0 } },
  { "Canon", "PowerShot G1 X Mark II", Flash, 0, { 2.44296788482835, 1, 1.221483942414175, 0 } },

  { "Canon", "PowerShot G2", Daylight, 0,	{ 2.011483, 1, 1.299522, 0 } },
  { "Canon", "PowerShot G2", Cloudy, 0,		{ 2.032505, 1, 1.285851, 0 } },
  { "Canon", "PowerShot G2", Tungsten, 0,	{ 1.976008, 1, 1.332054, 0 } },
  { "Canon", "PowerShot G2", Fluorescent, 0,	{ 2.022010, 1, 1.295694, 0 } },
  { "Canon", "PowerShot G2", FluorescentHigh, 0, { 2.029637, 1, 1.286807, 0 } },
  { "Canon", "PowerShot G2", Flash, 0,		{ 2.153576, 1, 1.140680, 0 } },

  { "Canon", "PowerShot G3", Daylight, 0,	{ 1.858513, 1, 1.387290, 0 } },
  { "Canon", "PowerShot G3", Cloudy, 0,		{ 1.951132, 1, 1.305125, 0 } },
  { "Canon", "PowerShot G3", Tungsten, 0,	{ 1.128386, 1, 2.313310, 0 } },
  { "Canon", "PowerShot G3", Fluorescent, 0,	{ 1.715573, 1, 2.194337, 0 } },
  { "Canon", "PowerShot G3", FluorescentHigh, 0, { 2.580563, 1, 1.496164, 0 } },
  { "Canon", "PowerShot G3", Flash, 0,		{ 2.293173, 1, 1.187416, 0 } },

  { "Canon", "PowerShot G3 X", Daylight, 0,    { 2.09049255441008, 1, 1.7709049255441, 0 } },
  { "Canon", "PowerShot G3 X", Shade, 0,       { 2.37126436781609, 1, 1.51149425287356, 0 } },
  { "Canon", "PowerShot G3 X", Cloudy, 0,      { 2.24228571428571, 1, 1.608, 0 } },
  { "Canon", "PowerShot G3 X", Tungsten, 0,    { 1.46367521367521, 1, 2.69871794871795, 0 } },
  { "Canon", "PowerShot G3 X", Fluorescent, 0, { 2.11970979443773, 1, 2.44377267230955, 0 } },
  { "Canon", "PowerShot G3 X", Flash, 0,       { 2.55542590431739, 1, 1.41773628938156, 0 } },

  { "Canon", "PowerShot G5", Daylight, 0,	{ 1.639521, 1, 1.528144, 0 } },
  { "Canon", "PowerShot G5", Cloudy, 0,		{ 1.702153, 1, 1.462919, 0 } },
  { "Canon", "PowerShot G5", Tungsten, 0,	{ 1.135071, 1, 2.374408, 0 } },
  { "Canon", "PowerShot G5", Fluorescent, 0,	{ 1.660281, 1, 2.186462, 0 } },
  { "Canon", "PowerShot G5", FluorescentHigh, 0, { 1.463297, 1, 1.764140, 0 } },
  { "Canon", "PowerShot G5", Flash, 0,		{ 1.603593, 1, 1.562874, 0 } },

  { "Canon", "PowerShot G5 X", Daylight, 0,{ 2.21040462427746, 1, 1.70520231213873, 0 } },
  { "Canon", "PowerShot G5 X", Shade, 0, { 2.54460093896714, 1, 1.46244131455399, 0 } },
  { "Canon", "PowerShot G5 X", Cloudy, 0,{ 2.38927738927739, 1, 1.57459207459207, 0 } },
  { "Canon", "PowerShot G5 X", Tungsten, 0,{ 1.49786780383795, 1, 2.71535181236674, 0 } },
  { "Canon", "PowerShot G5 X", Fluorescent, 0,{ 2.18576598311218, 1, 2.39686369119421, 0 } },
  { "Canon", "PowerShot G5 X", Flash, 0, { 2.63764705882353, 1, 1.38470588235294, 0 } },

  { "Canon", "PowerShot G6", Daylight, 0,	{ 1.769704, 1, 1.637931, 0 } },
  { "Canon", "PowerShot G6", Cloudy, 0,		{ 2.062731, 1, 1.442804, 0 } },
  { "Canon", "PowerShot G6", Tungsten, 0,	{ 1.077106, 1, 2.721234, 0 } },
  { "Canon", "PowerShot G6", Fluorescent, 0,	{ 1.914922, 1, 2.142670, 0 } },
  { "Canon", "PowerShot G6", FluorescentHigh, 0, { 2.543677, 1, 1.650587, 0 } },
  { "Canon", "PowerShot G6", Flash, 0,		{ 2.285322, 1, 1.333333, 0 } },

  /* Canon PowerShot G7 X Firmware Version 1.0.0 */
  { "Canon", "PowerShot G7 X", Daylight, -9,   { 1.78316032295271, 1, 2.1199538638985, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -8,   { 1.80826636050517, 1, 2.07233065442021, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -7,   { 1.83619702176403, 1, 2.02863688430699, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -6,   { 1.86514285714286, 1, 1.98628571428571, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -5,   { 1.90193842645382, 1, 1.93044469783352, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -4,   { 1.93977272727273, 1, 1.87840909090909, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -3,   { 1.98409090909091, 1, 1.83181818181818, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -2,   { 2.0375, 1, 1.78068181818182, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, -1,   { 2.0876993166287, 1, 1.73917995444191, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 0,    { 2.125284738041, 1, 1.70387243735763, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 1,    { 2.17009132420091, 1, 1.67009132420091, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 2,    { 2.21322690992018, 1, 1.6248574686431, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 3,    { 2.26285714285714, 1, 1.584, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 4,    { 2.30628571428571, 1, 1.544, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 5,    { 2.375, 1, 1.49426605504587, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 6,    { 2.45444059976932, 1, 1.44521337946943, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 7,    { 2.53863636363636, 1, 1.39659090909091, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 8,    { 2.64444444444444, 1, 1.34035087719298, 0 } },
  { "Canon", "PowerShot G7 X", Daylight, 9,    { 2.78884934756821, 1, 1.28232502965599, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -9,      { 1.96481271282633, 1, 1.84790011350738, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -8,      { 2.01021566401816, 1, 1.80022701475596, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -7,      { 2.06242905788876, 1, 1.74801362088536, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -6,      { 2.10933940774487, 1, 1.71640091116173, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -5,      { 2.14464692482916, 1, 1.68564920273349, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -4,      { 2.19156214367161, 1, 1.6442417331813, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -3,      { 2.2380410022779, 1, 1.6002277904328, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -2,      { 2.28424657534247, 1, 1.55936073059361, 0 } },
  { "Canon", "PowerShot G7 X", Shade, -1,      { 2.34249713631157, 1, 1.51775486827033, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 0,       { 2.41331802525832, 1, 1.4638346727899, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 1,       { 2.49422632794457, 1, 1.41454965357968, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 2,       { 2.58837209302326, 1, 1.36395348837209, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 3,       { 2.72, 1, 1.30235294117647, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 4,       { 2.8744019138756, 1, 1.24760765550239, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 5,       { 3.06479217603912, 1, 1.18948655256724, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 6,       { 3.33712121212121, 1, 1.12626262626263, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 7,       { 3.66842105263158, 1, 1.06842105263158, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 8,       { 4.07202216066482, 1, 1.01523545706371, 0 } },
  { "Canon", "PowerShot G7 X", Shade, 9,       { 4.59675036927622, 1, 0.964549483013294, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -9,     { 1.87287173666288, 1, 1.93757094211124, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -8,     { 1.91827468785471, 1, 1.89216799091941, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -7,     { 1.95804988662132, 1, 1.84353741496599, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -6,     { 2.00226500566251, 1, 1.79275198187995, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -5,     { 2.05895691609977, 1, 1.74603174603175, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -4,     { 2.09761634506243, 1, 1.7094211123723, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -3,     { 2.13393870601589, 1, 1.67536889897843, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -2,     { 2.18678815489749, 1, 1.64009111617312, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, -1,     { 2.23006833712984, 1, 1.59567198177677, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 0,      { 2.27822120866591, 1, 1.55302166476625, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 1,      { 2.327251995439, 1, 1.50855188141391, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 2,      { 2.40320733104238, 1, 1.46048109965636, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 3,      { 2.48502304147465, 1, 1.41129032258065, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 4,      { 2.5754060324826, 1, 1.35962877030162, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 5,      { 2.70774647887324, 1, 1.29929577464789, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 6,      { 2.86038186157518, 1, 1.24463007159905, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 7,      { 3.0462850182704, 1, 1.18757612667479, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 8,      { 3.31698113207547, 1, 1.12201257861635, 0 } },
  { "Canon", "PowerShot G7 X", Cloudy, 9,      { 3.64482306684142, 1, 1.06422018348624, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -9,   { 1.25761421319797, 1, 3.23857868020305, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -8,   { 1.27777777777778, 1, 3.18434343434343, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -7,   { 1.30188679245283, 1, 3.13836477987421, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -6,   { 1.3237139272271, 1, 3.09410288582183, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -5,   { 1.34456928838951, 1, 3.04744069912609, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -4,   { 1.36848635235732, 1, 2.98014888337469, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -3,   { 1.38916256157635, 1, 2.91379310344828, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -2,   { 1.41544117647059, 1, 2.8578431372549, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, -1,   { 1.43742405832321, 1, 2.78007290400972, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 0,    { 1.46610169491525, 1, 2.72760290556901, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 1,    { 1.49278846153846, 1, 2.65865384615385, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 2,    { 1.52335329341317, 1, 2.6059880239521, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 3,    { 1.54653937947494, 1, 2.56205250596659, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 4,    { 1.57074910820452, 1, 2.51605231866825, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 5,    { 1.60142348754448, 1, 2.46975088967972, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 6,    { 1.62958579881657, 1, 2.42721893491124, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 7,    { 1.65919811320755, 1, 2.37853773584906, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 8,    { 1.67602339181287, 1, 2.31111111111111, 0 } },
  { "Canon", "PowerShot G7 X", Tungsten, 9,    { 1.70979020979021, 1, 2.25174825174825, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -9,{ 1.78421052631579, 1, 2.95, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -8,{ 1.8178243774574, 1, 2.89121887287025, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -7,{ 1.85117493472585, 1, 2.8289817232376, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -6,{ 1.88311688311688, 1, 2.76233766233766, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -5,{ 1.91612903225806, 1, 2.68645161290323, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -4,{ 1.94871794871795, 1, 2.61153846153846, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -3,{ 1.98471337579618, 1, 2.53630573248408, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -2,{ 2.01776649746193, 1, 2.4758883248731, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, -1,{ 2.04924242424242, 1, 2.41161616161616, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 0, { 2.08301886792453, 1, 2.3559748427673, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 1, { 2.11668757841907, 1, 2.30363864491844, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 2, { 2.15875, 1, 2.24125, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 3, { 2.21223470661673, 1, 2.17977528089888, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 4, { 2.26591760299625, 1, 2.12234706616729, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 5, { 2.32459425717853, 1, 2.06367041198502, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 6, { 2.38403990024938, 1, 1.99750623441397, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 7, { 2.43445692883895, 1, 1.95755305867665, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 8, { 2.48872180451128, 1, 1.92105263157895, 0 } },
  { "Canon", "PowerShot G7 X", Fluorescent, 9, { 2.54705144291092, 1, 1.87076537013802, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -9, { 2.13784461152882, 1, 2.2719298245614, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -8, { 2.18601747815231, 1, 2.20349563046192, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -7, { 2.23815461346633, 1, 2.14214463840399, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -6, { 2.28891656288917, 1, 2.08094645080946, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -5, { 2.34803921568627, 1, 2.01593137254902, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -4, { 2.4139650872818, 1, 1.96758104738155, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -3, { 2.46067415730337, 1, 1.93008739076155, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -2, { 2.51375, 1, 1.885, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, -1, { 2.57644110275689, 1, 1.83583959899749, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 0,  { 2.63739021329987, 1, 1.78419071518193, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 1,  { 2.70062893081761, 1, 1.73836477987421, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 2,  { 2.79013906447535, 1, 1.67762326169406, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 3,  { 2.89185750636132, 1, 1.61832061068702, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 4,  { 3.00128205128205, 1, 1.56153846153846, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 5,  { 3.15714285714286, 1, 1.49480519480519, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 6,  { 3.3523178807947, 1, 1.42649006622517, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 7,  { 3.57666214382632, 1, 1.36635006784261, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 8,  { 3.91139240506329, 1, 1.28832630098453, 0 } },
  { "Canon", "PowerShot G7 X", DaylightFluorescent, 9,  { 4.35851851851852, 1, 1.2237037037037, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -9,      { 2.06356413166856, 1, 1.74574347332577, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -8,      { 2.09761634506243, 1, 1.7116912599319, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -7,      { 2.1375, 1, 1.67840909090909, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -6,      { 2.18181818181818, 1, 1.6375, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -5,      { 2.23348519362187, 1, 1.59681093394077, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -4,      { 2.28164196123147, 1, 1.557582668187, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -3,      { 2.33714285714286, 1, 1.51428571428571, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -2,      { 2.40596330275229, 1, 1.46215596330275, 0 } },
  { "Canon", "PowerShot G7 X", Flash, -1,      { 2.48904267589389, 1, 1.41407151095732, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 0,       { 2.58536585365854, 1, 1.36120789779326, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 1,       { 2.71092831962397, 1, 1.30199764982374, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 2,       { 2.86977299880526, 1, 1.24492234169654, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 3,       { 3.05487804878049, 1, 1.18536585365854, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 4,       { 3.31823899371069, 1, 1.12201257861635, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 5,       { 3.6496062992126, 1, 1.06692913385827, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 6,       { 4.06639004149378, 1, 1.01383125864454, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 7,       { 4.59379615952733, 1, 0.964549483013294, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 8,       { 5.25878594249201, 1, 0.9185303514377, 0 } },
  { "Canon", "PowerShot G7 X", Flash, 9,       { 6.22953736654804, 1, 0.875444839857651, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -9, { 1.82677165354331, 1, 2.08548931383577, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -8, { 1.85888888888889, 1, 2.03666666666667, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -7, { 1.89025755879059, 1, 1.99552071668533, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -6, { 1.93798449612403, 1, 1.93798449612403, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -5, { 1.97991071428571, 1, 1.88839285714286, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -4, { 2.03096330275229, 1, 1.83715596330275, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -3, { 2.07936507936508, 1, 1.78344671201814, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -2, { 2.11431742508324, 1, 1.74472807991121, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, -1, { 2.15102481121899, 1, 1.71413160733549, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 0,  { 2.2021978021978, 1, 1.67582417582418, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 1,  { 2.26152073732719, 1, 1.63248847926267, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 2,  { 2.30837004405286, 1, 1.59140969162996, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 3,  { 2.36139169472503, 1, 1.55106621773288, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 4,  { 2.43735498839907, 1, 1.49767981438515, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 5,  { 2.50886917960089, 1, 1.44456762749446, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 6,  { 2.60385925085131, 1, 1.39841089670829, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 7,  { 2.73462414578588, 1, 1.3371298405467, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 8,  { 2.88201603665521, 1, 1.27835051546392, 0 } },
  { "Canon", "PowerShot G7 X", Underwater, 9,  { 3.02536231884058, 1, 1.22584541062802, 0 } },

  { "Canon", "PowerShot G7 X Mark II", Daylight, 0,		{ 2.111328125, 1, 1.7919921875, 0 } },
  { "Canon", "PowerShot G7 X Mark II", Shade, 0,		{ 2.419921875, 1, 1.4853515625, 0 } },
  { "Canon", "PowerShot G7 X Mark II", Cloudy, 0,		{ 2.2744140625, 1, 1.6318359375, 0 } },
  { "Canon", "PowerShot G7 X Mark II", Tungsten, 0,		{ 1.4404296875, 1, 2.7822265625, 0 } },
  { "Canon", "PowerShot G7 X Mark II", CoolWhiteFluorescent, 0,	{ 2.1494140625, 1, 2.474609375, 0 } },
  { "Canon", "PowerShot G7 X Mark II", WhiteFluorescent, 0,	{ 2.419921875, 1, 1.4853515625, 0 } },
  { "Canon", "PowerShot G7 X Mark II", Underwater, 0,		{ 2.3271484375, 1, 1.642578125, 0 } },
  { "Canon", "PowerShot G7 X Mark II", Flash, 0,		{ 2.576171875, 1, 1.3623046875, 0 } },

  { "Canon", "PowerShot G9", Daylight, 0,	{ 2.089552, 1, 1.786452, 0 } },
  { "Canon", "PowerShot G9", Cloudy, 0,		{ 2.208716, 1, 1.660550, 0 } },
  { "Canon", "PowerShot G9", Tungsten, 0,	{ 1.533493, 1, 2.586124, 0 } },
  { "Canon", "PowerShot G9", Fluorescent, 0,	{ 2.065668, 1, 1.829493, 0 } },
  { "Canon", "PowerShot G9", FluorescentHigh, 0, { 2.237601, 1, 1.668974, 0 } },
  { "Canon", "PowerShot G9", Flash, 0,		{ 2.461538, 1, 1.498834, 0 } },
  { "Canon", "PowerShot G9", Underwater, 0,	{ 2.237327, 1, 1.661290, 0 } },

  /* Does the Canon PowerShot G10 support native WB presets? Please test. */
  { "Canon", "PowerShot G10", Daylight, 0,	{ 1.598980, 1, 1.830612, 0 } },
  { "Canon", "PowerShot G10", Cloudy, 0,	{ 1.738120, 1, 1.722281, 0 } },
  { "Canon", "PowerShot G10", Tungsten, 0,	{ 1, 1.035550, 3.569954, 0 } },
  { "Canon", "PowerShot G10", Fluorescent, 0,	{ 1.341633, 1, 2.434263, 0 } },
  { "Canon", "PowerShot G10", FluorescentHigh, 0, { 1.749171, 1, 1.907182, 0 } },
  { "Canon", "PowerShot G10", Flash, 0,		{ 1.926829, 1, 1.501591, 0 } },
  { "Canon", "PowerShot G10", Underwater, 0,	{ 1.822314, 1, 1.841942, 0 } },

  { "Canon", "PowerShot G11", Daylight, 0,	{ 1.721591, 1, 2.097727, 0 } },
  { "Canon", "PowerShot G11", Cloudy, 0,	{ 1.910936, 1, 1.856821, 0 } },
  { "Canon", "PowerShot G11", Tungsten, 0,	{ 1.380435, 1, 3.576087, 0 } },
  { "Canon", "PowerShot G11", Fluorescent, 0,	{ 1.649143, 1, 2.693714, 0 } },
  { "Canon", "PowerShot G11", FluorescentHigh, 0, { 2.008168, 1, 1.961494, 0 } },
  { "Canon", "PowerShot G11", Flash, 0,		{ 1.985556, 1, 1.703333, 0 } },
  { "Canon", "PowerShot G11", Underwater, 0,	{ 2.225624, 1, 1.577098, 0 } },

  { "Canon", "PowerShot G12", Daylight, 0,	{ 1.785877, 1, 2.042141, 0 } },
  { "Canon", "PowerShot G12", Cloudy, 0,	{ 1.804323, 1, 2.021615, 0 } },
  { "Canon", "PowerShot G12", Tungsten, 0,	{ 1.310127, 1, 3.170886, 0 } },
  { "Canon", "PowerShot G12", Fluorescent, 0,	{ 1.771139, 1, 2.064262, 0 } },
  { "Canon", "PowerShot G12", FluorescentHigh, 0, { 1.806122, 1, 2.032880, 0 } },
  { "Canon", "PowerShot G12", Flash, 0,		{ 2.102157, 1, 1.706016, 0 } },
  { "Canon", "PowerShot G12", Underwater, 0,	{ 1.807650, 1, 2.112568, 0 } },

  { "Canon", "PowerShot G15", Daylight, 0,    { 2.32114882506527, 1, 2.35770234986945, 0 } },
  { "Canon", "PowerShot G15", Shade, 0,       { 2.73360107095047, 1, 2.11646586345382, 0 } },
  { "Canon", "PowerShot G15", Cloudy, 0,      { 2.73360107095047, 1, 2.11646586345382, 0 } },
  { "Canon", "PowerShot G15", Tungsten, 0,    { 1.5990639625585, 1, 4.38689547581903, 0 } },
  { "Canon", "PowerShot G15", Fluorescent, 0, { 2.23393316195373, 1, 2.72879177377892, 0 } },
  { "Canon", "PowerShot G15", Flash, 0,       { 2.91902834008097, 1, 1.98785425101215, 0 } },

  { "Canon", "PowerShot G16", Daylight, 0,    { 2.04274465691789, 1, 1.78627671541057, 0 } },
  { "Canon", "PowerShot G16", Shade, 0,       { 2.21257142857143, 1, 1.65714285714286, 0 } },
  { "Canon", "PowerShot G16", Cloudy, 0,      { 2.21257142857143, 1, 1.65714285714286, 0 } },
  { "Canon", "PowerShot G16", Tungsten, 0,    { 1.67857142857143, 1, 2.6256157635468, 0 } },
  { "Canon", "PowerShot G16", Fluorescent, 0, { 2.07547169811321, 1, 1.95754716981132, 0 } },
  { "Canon", "PowerShot G16", Flash, 0,       { 2.63488372093023, 1, 1.31744186046512, 0 } },

  /* Canon PowerShot S3 IS does not support native WB presets. These are made
     as custom WB presets. */
  { "Canon", "PowerShot S3 IS", Daylight, 0,	{ 1.627271, 1, 1.823491, 0 } },
  { "Canon", "PowerShot S3 IS", Cloudy, 0,	{ 1.794382, 1, 1.618412, 0 } },
  { "Canon", "PowerShot S3 IS", Tungsten, 0,	{ 1, 1.192243, 4.546950, 0 } },
  { "Canon", "PowerShot S3 IS", Flash, 0,	{ 1.884691, 1, 1.553869, 0 } },

  { "Canon", "PowerShot S30", Daylight, 0,	{ 1.741088, 1, 1.318949, 0 } },
  { "Canon", "PowerShot S30", Cloudy, 0,	{ 1.766635, 1, 1.298969, 0 } },
  { "Canon", "PowerShot S30", Tungsten, 0,	{ 1.498106, 1, 1.576705, 0 } },
  { "Canon", "PowerShot S30", Fluorescent, 0,	{ 1.660075, 1, 1.394539, 0 } },
  { "Canon", "PowerShot S30", FluorescentHigh, 0, { 1.753515, 1, 1.306467, 0 } },
  { "Canon", "PowerShot S30", Flash, 0,		{ 2.141705, 1, 1.097926, 0 } },

  { "Canon", "PowerShot S45", Daylight, 0,	{ 2.325175, 1, 1.080420, 0 } },
  { "Canon", "PowerShot S45", Cloudy, 0,	{ 2.145047, 1, 1.173349, 0 } },
  { "Canon", "PowerShot S45", Tungsten, 0,	{ 1.213018, 1, 2.087574, 0 } },
  { "Canon", "PowerShot S45", Fluorescent, 0,	{ 1.888183, 1, 1.822109, 0 } },
  { "Canon", "PowerShot S45", FluorescentHigh, 0, { 2.964422, 1, 1.354511, 0 } },
  { "Canon", "PowerShot S45", Flash, 0,		{ 2.534884, 1, 1.065663, 0 } },

  { "Canon", "PowerShot S50", Daylight, 0,	{ 1.772506, 1, 1.536496, 0 } },
  { "Canon", "PowerShot S50", Cloudy, 0,	{ 1.831311, 1, 1.484223, 0 } },
  { "Canon", "PowerShot S50", Tungsten, 0,	{ 1.185542, 1, 2.480723, 0 } },
  { "Canon", "PowerShot S50", Fluorescent, 0,	{ 1.706410, 1, 2.160256, 0 } },
  { "Canon", "PowerShot S50", FluorescentHigh, 0, { 1.562500, 1, 1.817402, 0 } },
  { "Canon", "PowerShot S50", Flash, 0,		{ 1.776156, 1, 1.531630, 0 } },

  { "Canon", "PowerShot S60", Daylight, 0,	{ 1.759169, 1, 1.590465, 0 } },
  { "Canon", "PowerShot S60", Cloudy, 0,	{ 1.903659, 1, 1.467073, 0 } },
  { "Canon", "PowerShot S60", Tungsten, 0,	{ 1.138554, 1, 2.704819, 0 } },
  { "Canon", "PowerShot S60", Fluorescent, 0,	{ 1.720721, 1, 2.185328, 0 } },
  { "Canon", "PowerShot S60", FluorescentHigh, 0, { 2.877095, 1, 2.216480, 0 } },
  { "Canon", "PowerShot S60", Flash, 0,		{ 2.182540, 1, 1.236773, 0 } },
  { "Canon", "PowerShot S60", Underwater, 0,	{ 2.725369, 1, 1.240148, 0 } },

  { "Canon", "PowerShot S70", Daylight, 0,	{ 1.943834, 1, 1.456654, 0 } },
  { "Canon", "PowerShot S70", Cloudy, 0,	{ 2.049939, 1, 1.382460, 0 } },
  { "Canon", "PowerShot S70", Tungsten, 0,	{ 1.169492, 1, 2.654964, 0 } },
  { "Canon", "PowerShot S70", Fluorescent, 0,	{ 1.993456, 1, 2.056283, 0 } },
  { "Canon", "PowerShot S70", FluorescentHigh, 0, { 2.645914, 1, 1.565499, 0 } },
  { "Canon", "PowerShot S70", Flash, 0,		{ 2.389189, 1, 1.147297, 0 } },
  { "Canon", "PowerShot S70", Underwater, 0,	{ 3.110565, 1, 1.162162, 0 } },

  { "Canon", "PowerShot S90", Daylight, 0,	{ 1.955056, 1, 1.797753, 0 } },
  { "Canon", "PowerShot S90", Cloudy, 0,	{ 1.945067, 1, 1.795964, 0 } },
  { "Canon", "PowerShot S90", Tungsten, 0,	{ 2.000000, 1, 1.828018, 0 } },
  { "Canon", "PowerShot S90", Fluorescent, 0,	{ 2.019473, 1, 1.841924, 0 } },
  { "Canon", "PowerShot S90", FluorescentHigh, 0, { 2.009143, 1, 1.840000, 0 } },
  { "Canon", "PowerShot S90", Flash, 0,		{ 2.045784, 1, 1.671692, 0 } },
  { "Canon", "PowerShot S90", Underwater, 0,	{ 2.022297, 1, 1.830546, 0 } },

  { "Canon", "PowerShot S95", Daylight, 0,	{ 1.6182, 1, 2.1430, 0 } },
  { "Canon", "PowerShot S95", Cloudy, 0,	{ 1.6920, 1, 2.0179, 0 } },
  { "Canon", "PowerShot S95", Tungsten, 0,	{ 1.2314, 1, 3.1015, 0 } },
  { "Canon", "PowerShot S95", Fluorescent, 0,	{ 1.6593, 1, 2.0940, 0 } },
  { "Canon", "PowerShot S95", FluorescentHigh, 0, { 1.7272, 1, 1.9811, 0 } },
  { "Canon", "PowerShot S95", Flash, 0,		{ 1.9955, 1, 1.7768, 0 } },
  { "Canon", "PowerShot S95", Underwater, 0,	{ 1.7607, 1, 2.1224, 0 } },

  { "Canon", "PowerShot S100", Daylight, 0,	{ 2.077707, 1, 2.551592, 0 } },
  { "Canon", "PowerShot S100", Cloudy, 0,	{ 2.276402, 1, 2.393742, 0 } },
  { "Canon", "PowerShot S100", Tungsten, 0,	{ 1.267936, 1, 4.224012, 0 } },
  { "Canon", "PowerShot S100", Fluorescent, 0,	{ 1.815115, 1, 3.093117, 0 } },
  { "Canon", "PowerShot S100", FluorescentHigh, 0, { 2.398148, 1, 2.374339, 0 } },
  { "Canon", "PowerShot S100", Flash, 0,	{ 2.615783, 1, 2.001294, 0 } },
  { "Canon", "PowerShot S100", Underwater, 0,	{ 2.248391, 1, 2.338481, 0 } },

  { "Canon", "PowerShot S110", Daylight, 0, { 2.111842105263158, 1, 2.6223684210526317, 0 } },
  { "Canon", "PowerShot S110", Shade, 0, { 2.3174603174603177, 1, 2.4497354497354498, 0 } },
  { "Canon", "PowerShot S110", Cloudy, 0, { 2.3174603174603177, 1, 2.4497354497354498, 0 } },
  { "Canon", "PowerShot S110", Tungsten, 0, { 1.5206611570247934, 1, 4.009641873278237, 0 } },
  { "Canon", "PowerShot S110", Fluorescent, 0, { 1.8797814207650274, 1, 3.1147540983606556, 0 } },
  { "Canon", "PowerShot S110", Flash, 0, { 2.627968337730871, 1, 2.1213720316622693, 0 } },

  { "Canon", "PowerShot S120", Daylight, 0,   { 2.33333333333333, 1, 1.46938775510204, 0 } },
  { "Canon", "PowerShot S120", Cloudy, 0,     { 2.5, 1, 1.3704128440367, 0 } },
  { "Canon", "PowerShot S120", Flash, 0,      { 2.63519813519814, 1, 1.32983682983683, 0 } },
  { "Canon", "PowerShot S120", Fluorescent, 0,{ 2.52364475201845, 1, 1.37831603229527, 0 } },
  { "Canon", "PowerShot S120", Shade, 0,      { 2.5, 1, 1.3704128440367, 0 } },
  { "Canon", "PowerShot S120", Tungsten, 0,   { 1.7545045045045, 1, 2.00675675675676, 0 } },

  { "Canon", "PowerShot Pro1", Daylight, 0,	{ 1.829238, 1, 1.571253, 0 } },
  { "Canon", "PowerShot Pro1", Cloudy, 0,	{ 1.194139, 1, 2.755800, 0 } },
  { "Canon", "PowerShot Pro1", Tungsten, 0,	{ 1.701416, 1, 2.218790, 0 } },
  { "Canon", "PowerShot Pro1", Fluorescent, 0,	{ 2.014066, 1, 1.776215, 0 } },
  { "Canon", "PowerShot Pro1", FluorescentHigh, 0, { 2.248663, 1, 1.227273, 0 } },
  { "Canon", "PowerShot Pro1", Flash, 0,	{ 2.130081, 1, 1.422764, 0 } },

  { "Canon", "PowerShot SX1 IS", Daylight, 0,	{ 1.574586, 1, 2.114917, 0 } },
  { "Canon", "PowerShot SX1 IS", Cloudy, 0,	{ 1.682628, 1, 2.015590, 0 } },
  { "Canon", "PowerShot SX1 IS", Tungsten, 0,	{ 1.088836, 1, 3.056423, 0 } },
  { "Canon", "PowerShot SX1 IS", Fluorescent, 0, { 1.398259, 1, 2.414581, 0 } },
  { "Canon", "PowerShot SX1 IS", FluorescentHigh, 0, { 1.687500, 1, 2.025670, 0 } },
  { "Canon", "PowerShot SX1 IS", Flash, 0,	{ 1.909699, 1, 1.795987, 0 } },

  { "Canon", "Canon PowerShot SX160 IS", Daylight, 0,		{ 1.725252, 1, 1.444444, 1 } },
  { "Canon", "Canon PowerShot SX160 IS", Cloudy, 0,		{ 1.843781, 1, 1.359204, 1 } },
  { "Canon", "Canon PowerShot SX160 IS", Tungsten, 0,		{ 1.103726, 1, 2.322256, 1 } },
  { "Canon", "Canon PowerShot SX160 IS", Fluorescent, 0,	{ 1.704705, 1, 1.463463, 1 } },
  { "Canon", "Canon PowerShot SX160 IS", FluorescentHigh, 0,	{ 1.848423, 1, 1.355036, 1 } },

  /* -9/+9 fine tuning is B9/A9 in blue-amber and zero in magenta-green */
  { "Canon", "PowerShot SX50 HS", Daylight, -9,	{ 1.260171, 1, 2.286938, 0 } },
  { "Canon", "PowerShot SX50 HS", Daylight, 0,	{ 1.502585, 1, 1.871768, 0 } },
  { "Canon", "PowerShot SX50 HS", Daylight, 9,	{ 1.868853, 1, 1.460041, 0 } },
  { "Canon", "PowerShot SX50 HS", Cloudy, -9,	{ 1.413865, 1, 1.962185, 0 } },
  { "Canon", "PowerShot SX50 HS", Cloudy, 0,	{ 1.767708, 1, 1.563542, 0 } },
  { "Canon", "PowerShot SX50 HS", Cloudy, 9,	{ 2.782759, 1, 1.100000, 0 } },
  { "Canon", "PowerShot SX50 HS", Tungsten, -9,	{ 1, 1.092348, 3.565963, 0 } },
  { "Canon", "PowerShot SX50 HS", Tungsten, 0,	{ 1.056582, 1, 2.853349, 0 } },
  { "Canon", "PowerShot SX50 HS", Tungsten, 9,	{ 1.251378, 1, 2.389195, 0 } },
  { "Canon", "PowerShot SX50 HS", Fluorescent, -9, { 1.197637, 1, 2.493018, 0 } },
  { "Canon", "PowerShot SX50 HS", Fluorescent, 0, { 1.383910, 1, 2.011202, 0 } },
  { "Canon", "PowerShot SX50 HS", Fluorescent, 9, { 1.677323, 1, 1.619381, 0 } },
  { "Canon", "PowerShot SX50 HS", FluorescentHigh, -9, { 1.468421, 1, 1.918947, 0 } },
  { "Canon", "PowerShot SX50 HS", FluorescentHigh, 0, { 1.805208, 1, 1.527083, 0 } },
  { "Canon", "PowerShot SX50 HS", FluorescentHigh, 9, { 3.464467, 1, 1.029188, 0 } },
  { "Canon", "PowerShot SX50 HS", Flash, -9,	{ 1.488683, 1, 1.777778, 0 } },
  { "Canon", "PowerShot SX50 HS", Flash, 0,	{ 1.876289, 1, 1.398969, 0 } },
  { "Canon", "PowerShot SX50 HS", Flash, 9,	{ 4.536036, 1.073574, 1, 0 } },

  { "Canon", "PowerShot SX60 HS", Daylight, 0,{ 1.60520833333333, 1, 1.72604166666667, 0 } },
  { "Canon", "PowerShot SX60 HS", Shade, 0, { 1.9936974789916, 1, 1.37920168067227, 0 } },
  { "Canon", "PowerShot SX60 HS", Cloudy, 0,{ 1.87604166666667, 1, 1.45625, 0 } },
  { "Canon", "PowerShot SX60 HS", Tungsten, 0,{ 1.15801354401806, 1, 2.61625282167043, 0 } },
  { "Canon", "PowerShot SX60 HS", Fluorescent, 0,{ 1.67549668874172, 1, 2.25496688741722, 0 } },
  { "Canon", "PowerShot SX60 HS", FluorescentHigh, 0,{ 2.09885057471264, 1, 1.78505747126437, 0 } },
  { "Canon", "PowerShot SX60 HS", Flash, 0, { 1.96638655462185, 1, 1.40861344537815, 0 } },

  { "Canon", "EOS D60", Daylight, 0,		{ 2.472594, 1, 1.225335, 0 } },
  { "Canon", "EOS D60", Cloudy, 0,		{ 2.723926, 1, 1.137423, 0 } },
  { "Canon", "EOS D60", Tungsten, 0,		{ 1.543054, 1, 1.907003, 0 } },
  { "Canon", "EOS D60", Fluorescent, 0,		{ 1.957346, 1, 1.662322, 0 } },
  { "Canon", "EOS D60", Flash, 0,		{ 2.829840, 1, 1.108508, 0 } },

  { "Canon", "EOS 5D", Flash, 0,		{ 2.211914, 1, 1.260742, 0 } }, /*6550K*/
  { "Canon", "EOS 5D", Fluorescent, 0,		{ 1.726054, 1, 2.088123, 0 } }, /*3850K*/
  { "Canon", "EOS 5D", Tungsten, 0,		{ 1.373285, 1, 2.301006, 0 } }, /*3250K*/
  { "Canon", "EOS 5D", Cloudy, 0,		{ 2.151367, 1, 1.321289, 0 } }, /*6100K*/
  { "Canon", "EOS 5D", Shade, 0,		{ 2.300781, 1, 1.208008, 0 } }, /*7200K*/
  { "Canon", "EOS 5D", Daylight, 0,		{ 1.988281, 1, 1.457031, 0 } }, /*5250K*/

  /* Canon EOS 5D Mark II firmware 2.0.7 */
  { "Canon", "EOS 5D Mark II", Daylight, -9,	{ 1.954102, 1, 1.984375, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -8,	{ 1.976563, 1, 1.954102, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -7,	{ 2.003906, 1, 1.917969, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -6,	{ 2.032227, 1, 1.885742, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -5,	{ 2.060547, 1, 1.851563, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -4,	{ 2.093750, 1, 1.818359, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -3,	{ 2.120117, 1, 1.787109, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -2,	{ 2.151367, 1, 1.756836, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, -1,	{ 2.183594, 1, 1.723633, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 0,	{ 2.216797, 1, 1.689453, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 1,	{ 2.250977, 1, 1.659180, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 2,	{ 2.291016, 1, 1.627930, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 3,	{ 2.327148, 1, 1.594727, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 4,	{ 2.359375, 1, 1.563477, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 5,	{ 2.392578, 1, 1.528320, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 6,	{ 2.420898, 1, 1.503906, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 7,	{ 2.450195, 1, 1.477539, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 8,	{ 2.485352, 1, 1.450195, 0 } },
  { "Canon", "EOS 5D Mark II", Daylight, 9,	{ 2.528320, 1, 1.421875, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -9,	{ 2.250977, 1, 1.662109, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -8,	{ 2.286133, 1, 1.630859, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -7,	{ 2.322266, 1, 1.599609, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -6,	{ 2.354492, 1, 1.565430, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -5,	{ 2.386719, 1, 1.533203, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -4,	{ 2.415039, 1, 1.505859, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -3,	{ 2.450195, 1, 1.479492, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -2,	{ 2.485352, 1, 1.452148, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, -1,	{ 2.522461, 1, 1.425781, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 0,	{ 2.559570, 1, 1.397461, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 1,	{ 2.585938, 1, 1.383789, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 2,	{ 2.605469, 1, 1.371094, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 3,	{ 2.632813, 1, 1.354492, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 4,	{ 2.660156, 1, 1.338867, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 5,	{ 2.687500, 1, 1.316406, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 6,	{ 2.723633, 1, 1.290039, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 7,	{ 2.767578, 1, 1.260742, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 8,	{ 2.821289, 1, 1.232422, 0 } },
  { "Canon", "EOS 5D Mark II", Shade, 9,	{ 2.868164, 1, 1.203125, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -9,	{ 2.098633, 1, 1.812500, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -8,	{ 2.124023, 1, 1.784180, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -7,	{ 2.156250, 1, 1.750000, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -6,	{ 2.188477, 1, 1.717773, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -5,	{ 2.221680, 1, 1.686523, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -4,	{ 2.255859, 1, 1.657227, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -3,	{ 2.295898, 1, 1.623047, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -2,	{ 2.333008, 1, 1.592773, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, -1,	{ 2.359375, 1, 1.558594, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 0,	{ 2.392578, 1, 1.523438, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 1,	{ 2.426758, 1, 1.499023, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 2,	{ 2.456055, 1, 1.473633, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 3,	{ 2.491211, 1, 1.446289, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 4,	{ 2.528320, 1, 1.419922, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 5,	{ 2.566406, 1, 1.395508, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 6,	{ 2.585938, 1, 1.379883, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 7,	{ 2.612305, 1, 1.367188, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 8,	{ 2.639648, 1, 1.350586, 0 } },
  { "Canon", "EOS 5D Mark II", Cloudy, 9,	{ 2.660156, 1, 1.334961, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -9,	{ 1.383128, 1, 3.084359, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -8,	{ 1.399119, 1, 3.026432, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -7,	{ 1.419098, 1, 2.984085, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -6,	{ 1.439219, 1, 2.935226, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -5,	{ 1.460374, 1, 2.886910, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -4,	{ 1.484361, 1, 2.840036, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -3,	{ 1.503139, 1, 2.787444, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -2,	{ 1.527027, 1, 2.740541, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, -1,	{ 1.550226, 1, 2.695023, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 0,	{ 1.573115, 1, 2.646685, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 1,	{ 1.594891, 1, 2.596715, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 2,	{ 1.615949, 1, 2.549038, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 3,	{ 1.638710, 1, 2.494009, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 4,	{ 1.661111, 1, 2.443519, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 5,	{ 1.689013, 1, 2.391993, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 6,	{ 1.712816, 1, 2.349860, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 7,	{ 1.736595, 1, 2.305738, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 8,	{ 1.760417, 1, 2.266098, 0 } },
  { "Canon", "EOS 5D Mark II", Tungsten, 9,	{ 1.787417, 1, 2.222116, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -9,	{ 1.669691, 1, 2.900181, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -8,	{ 1.695533, 1, 2.849590, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -7,	{ 1.717033, 1, 2.806777, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -6,	{ 1.740809, 1, 2.761029, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -5,	{ 1.765683, 1, 2.716790, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -4,	{ 1.794063, 1, 2.667904, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -3,	{ 1.820298, 1, 2.619181, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -2,	{ 1.841760, 1, 2.567416, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, -1,	{ 1.870056, 1, 2.516949, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 0,	{ 1.895833, 1, 2.461174, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 1,	{ 1.928503, 1, 2.416587, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 2,	{ 1.956772, 1, 2.365994, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 3,	{ 1.978744, 1, 2.320773, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 4,	{ 2.009718, 1, 2.281827, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 5,	{ 2.036133, 1, 2.240234, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 6,	{ 2.068359, 1, 2.197266, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 7,	{ 2.098633, 1, 2.160156, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 8,	{ 2.128906, 1, 2.120117, 0 } },
  { "Canon", "EOS 5D Mark II", Fluorescent, 9,	{ 2.156250, 1, 2.081055, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -9,	{ 2.111328, 1, 1.792969, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -8,	{ 2.142578, 1, 1.762695, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -7,	{ 2.173828, 1, 1.732422, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -6,	{ 2.207031, 1, 1.698242, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -5,	{ 2.240234, 1, 1.667969, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -4,	{ 2.280273, 1, 1.635742, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -3,	{ 2.316406, 1, 1.602539, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -2,	{ 2.348633, 1, 1.570313, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, -1,	{ 2.381836, 1, 1.537109, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 0,	{ 2.409180, 1, 1.507813, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 1,	{ 2.444336, 1, 1.484375, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 2,	{ 2.479492, 1, 1.457031, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 3,	{ 2.515625, 1, 1.429688, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 4,	{ 2.553711, 1, 1.402344, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 5,	{ 2.579102, 1, 1.385742, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 6,	{ 2.598633, 1, 1.373047, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 7,	{ 2.625977, 1, 1.356445, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 8,	{ 2.653320, 1, 1.339844, 0 } },
  { "Canon", "EOS 5D Mark II", Flash, 9,	{ 2.680664, 1, 1.321289, 0 } },

  /* Canon EOS 5D Mark III Firmware Version 1.1.3 */
  /* Fine-tuning is the camera's Amber-Blue bracketing. */
  { "Canon", "EOS 5D Mark III", Daylight, -9,	{ 1.784180, 1, 1.907227, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -8,	{ 1.805664, 1, 1.878906, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -7,	{ 1.828125, 1, 1.848633, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -6,	{ 1.855469, 1, 1.818359, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -5,	{ 1.882813, 1, 1.790039, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -4,	{ 1.910156, 1, 1.759766, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -3,	{ 1.931641, 1, 1.726563, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -2,	{ 1.958008, 1, 1.695313, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, -1,	{ 1.980469, 1, 1.665039, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 0,	{ 2.007813, 1, 1.630859, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 1,	{ 2.036133, 1, 1.607422, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 2,	{ 2.064453, 1, 1.583008, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 3,	{ 2.093750, 1, 1.556641, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 4,	{ 2.124023, 1, 1.533203, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 5,	{ 2.160156, 1, 1.507813, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 6,	{ 2.183594, 1, 1.486328, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 7,	{ 2.211914, 1, 1.462891, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 8,	{ 2.240234, 1, 1.440430, 0 } },
  { "Canon", "EOS 5D Mark III", Daylight, 9,	{ 2.275391, 1, 1.416016, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -9,	{ 2.032227, 1, 1.610352, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -8,	{ 2.060547, 1, 1.584961, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -7,	{ 2.089844, 1, 1.558594, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -6,	{ 2.120117, 1, 1.535156, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -5,	{ 2.156250, 1, 1.510742, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -4,	{ 2.183594, 1, 1.488281, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -3,	{ 2.211914, 1, 1.464844, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -2,	{ 2.240234, 1, 1.442383, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, -1,	{ 2.270508, 1, 1.417969, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 0,	{ 2.306641, 1, 1.393555, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 1,	{ 2.337891, 1, 1.374023, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 2,	{ 2.370117, 1, 1.352539, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 3,	{ 2.403320, 1, 1.332031, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 4,	{ 2.444336, 1, 1.307617, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 5,	{ 2.479492, 1, 1.286133, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 6,	{ 2.509766, 1, 1.267578, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 7,	{ 2.541016, 1, 1.246094, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 8,	{ 2.579102, 1, 1.224609, 0 } },
  { "Canon", "EOS 5D Mark III", Shade, 9,	{ 2.612305, 1, 1.203125, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -9,	{ 1.914063, 1, 1.753906, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -8,	{ 1.935547, 1, 1.723633, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -7,	{ 1.958008, 1, 1.692383, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -6,	{ 1.984375, 1, 1.659180, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -5,	{ 2.011719, 1, 1.627930, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -4,	{ 2.040039, 1, 1.605469, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -3,	{ 2.068359, 1, 1.578125, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -2,	{ 2.098633, 1, 1.553711, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, -1,	{ 2.128906, 1, 1.528320, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 0,	{ 2.160156, 1, 1.503906, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 1,	{ 2.188477, 1, 1.484375, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 2,	{ 2.216797, 1, 1.460938, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 3,	{ 2.246094, 1, 1.436523, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 4,	{ 2.280273, 1, 1.412109, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 5,	{ 2.316406, 1, 1.389648, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 6,	{ 2.342773, 1, 1.369141, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 7,	{ 2.375977, 1, 1.347656, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 8,	{ 2.415039, 1, 1.326172, 0 } },
  { "Canon", "EOS 5D Mark III", Cloudy, 9,	{ 2.456055, 1, 1.302734, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -9,	{ 1.283203, 1, 2.782227, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -8,	{ 1.297852, 1, 2.752930, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -7,	{ 1.314453, 1, 2.723633, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -6,	{ 1.333008, 1, 2.694336, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -5,	{ 1.350586, 1, 2.666992, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -4,	{ 1.371094, 1, 2.632813, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -3,	{ 1.387695, 1, 2.579102, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -2,	{ 1.404297, 1, 2.528320, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, -1,	{ 1.423828, 1, 2.479492, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 0,	{ 1.442383, 1, 2.426758, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 1,	{ 1.460938, 1, 2.392578, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 2,	{ 1.479492, 1, 2.354492, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 3,	{ 1.501953, 1, 2.316406, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 4,	{ 1.523438, 1, 2.280273, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 5,	{ 1.546875, 1, 2.240234, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 6,	{ 1.568359, 1, 2.207031, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 7,	{ 1.587891, 1, 2.169922, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 8,	{ 1.610352, 1, 2.133789, 0 } },
  { "Canon", "EOS 5D Mark III", Tungsten, 9,	{ 1.632813, 1, 2.098633, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -9, { 1.551758, 1, 2.702148, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -8, { 1.573242, 1, 2.673828, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -7, { 1.592773, 1, 2.645508, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -6, { 1.615234, 1, 2.592773, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -5, { 1.638672, 1, 2.541016, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -4, { 1.662109, 1, 2.491211, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -3, { 1.684570, 1, 2.438477, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -2, { 1.709961, 1, 2.398438, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, -1, { 1.732422, 1, 2.365234, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 0, { 1.759766, 1, 2.322266, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 1, { 1.787109, 1, 2.286133, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 2, { 1.809570, 1, 2.250977, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 3, { 1.832031, 1, 2.211914, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 4, { 1.858398, 1, 2.178711, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 5, { 1.885742, 1, 2.142578, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 6, { 1.914063, 1, 2.107422, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 7, { 1.935547, 1, 2.068359, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 8, { 1.961914, 1, 2.040039, 0 } },
  { "Canon", "EOS 5D Mark III", WhiteFluorescent, 9, { 1.988281, 1, 2.007813, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -9,	{ 1.976563, 1, 1.717773, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -8,	{ 2.003906, 1, 1.686523, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -7,	{ 2.032227, 1, 1.654297, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -6,	{ 2.060547, 1, 1.623047, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -5,	{ 2.089844, 1, 1.599609, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -4,	{ 2.120117, 1, 1.573242, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -3,	{ 2.151367, 1, 1.548828, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -2,	{ 2.178711, 1, 1.523438, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, -1,	{ 2.207031, 1, 1.501953, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 0,	{ 2.235352, 1, 1.477539, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 1,	{ 2.270508, 1, 1.457031, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 2,	{ 2.306641, 1, 1.432617, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 3,	{ 2.337891, 1, 1.408203, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 4,	{ 2.370117, 1, 1.385742, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 5,	{ 2.403320, 1, 1.365234, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 6,	{ 2.444336, 1, 1.343750, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 7,	{ 2.479492, 1, 1.321289, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 8,	{ 2.509766, 1, 1.297852, 0 } },
  { "Canon", "EOS 5D Mark III", Flash, 9,	{ 2.541016, 1, 1.278320, 0 } },

  { "Canon", "EOS 5D Mark IV", Daylight, 0, { 1.9765625, 1, 1.4345703125, 0 } },
  { "Canon", "EOS 5D Mark IV", Shade, 0, { 2.2607421875, 1, 1.2431640625, 0 } },
  { "Canon", "EOS 5D Mark IV", Cloudy, 0, { 2.115234375, 1, 1.3330078125, 0 } },
  { "Canon", "EOS 5D Mark IV", Tungsten, 0, { 1.416015625, 1, 2.01953125, 0 } },
  { "Canon", "EOS 5D Mark IV", WhiteFluorescent, 0, { 1.6728515625, 1, 1.9462890625, 0 } },
  { "Canon", "EOS 5D Mark IV", Flash, 0, { 2.1650390625, 1, 1.330078125, 0 } },

  { "Canon", "EOS 5DS", Daylight            , 0, { 2.255859, 1, 1.717773, 0 } },
  { "Canon", "EOS 5DS", Shade               , 0, { 2.605469, 1, 1.484375, 0 } },
  { "Canon", "EOS 5DS", Cloudy              , 0, { 2.438477, 1, 1.589844, 0 } },
  { "Canon", "EOS 5DS", Tungsten            , 0, { 1.605469, 1, 2.467773, 0 } },
  { "Canon", "EOS 5DS", CoolWhiteFluorescent, 0, { 1.931641, 1, 2.354492, 0 } },
  { "Canon", "EOS 5DS", Flash               , 0, { 2.528320, 1, 1.583008, 0 } },

  { "Canon", "EOS 5DS R", Daylight            , 0, { 2.192383, 1, 1.738281, 0 } },
  { "Canon", "EOS 5DS R", Shade               , 0, { 2.541016, 1, 1.497070, 0 } },
  { "Canon", "EOS 5DS R", Cloudy              , 0, { 2.370117, 1, 1.607422, 0 } },
  { "Canon", "EOS 5DS R", Tungsten            , 0, { 1.546875, 1, 2.503906, 0 } },
  { "Canon", "EOS 5DS R", CoolWhiteFluorescent, 0, { 1.865234, 1, 2.386719, 0 } },
  { "Canon", "EOS 5DS R", Flash               , 0, { 2.461914, 1, 1.599609, 0 } },

  /* Canon EOS 6D Firmware Version 1.0.9 */
  { "Canon", "EOS 6D", Daylight, 0,		{ 1.976562, 1, 1.635742, 0 } },
  { "Canon", "EOS 6D", Shade, 0,		{ 2.265625, 1, 1.371094, 0 } },
  { "Canon", "EOS 6D", Cloudy, 0,		{ 2.128906, 1, 1.497070, 0 } },
  { "Canon", "EOS 6D", Tungsten, 0,		{ 1.412109, 1, 2.473633, 0 } },
  { "Canon", "EOS 6D", Fluorescent, 0,		{ 1.726562, 1, 2.337891, 0 } },
  { "Canon", "EOS 6D", Flash, 0,		{ 2.192383, 1, 1.458984, 0 } },

  /* Canon EOS 6D Mark II Firmware Version 1.0.2 */
  { "Canon", "EOS 6D Mark II", Daylight, 0,	{ 2.0810546875, 1, 1.6435546875, 0 } },
  { "Canon", "EOS 6D Mark II", Shade, 0,	{ 2.4208984375, 1, 1.4023437500, 0 } },
  { "Canon", "EOS 6D Mark II", Cloudy, 0,	{ 2.2558593750, 1, 1.5146484375, 0 } },
  { "Canon", "EOS 6D Mark II", Tungsten, 0,	{ 1.4501953125, 1, 2.4150390625, 0 } },
  { "Canon", "EOS 6D Mark II", Fluorescent, 0,	{ 1.7294921875, 1, 2.2802734375, 0 } },
  { "Canon", "EOS 6D Mark II", Flash, 0,	{ 2.2802734375, 1, 1.4882812500, 0 } },

  /* Canon EOS 7D Firmware Version 2.0.3 */
  /* Fine-tuning for the 7D are the camera's Amber-Blue bracketing. */
  { "Canon", "EOS 7D", Daylight, -9,		{ 1.8281, 1, 1.8281, 0 } },
  { "Canon", "EOS 7D", Daylight, -8,		{ 1.8516, 1, 1.7969, 0 } },
  { "Canon", "EOS 7D", Daylight, -7,		{ 1.8750, 1, 1.7656, 0 } },
  { "Canon", "EOS 7D", Daylight, -6,		{ 1.9033, 1, 1.7354, 0 } },
  { "Canon", "EOS 7D", Daylight, -5,		{ 1.9316, 1, 1.7041, 0 } },
  { "Canon", "EOS 7D", Daylight, -4,		{ 1.9619, 1, 1.6729, 0 } },
  { "Canon", "EOS 7D", Daylight, -3,		{ 1.9844, 1, 1.6406, 0 } },
  { "Canon", "EOS 7D", Daylight, -2,		{ 2.0117, 1, 1.6123, 0 } },
  { "Canon", "EOS 7D", Daylight, -1,		{ 2.0361, 1, 1.5801, 0 } },
  { "Canon", "EOS 7D", Daylight, 0,		{ 2.0645, 1, 1.5488, 0 } },
  { "Canon", "EOS 7D", Daylight, 1,		{ 2.0938, 1, 1.5234, 0 } },
  { "Canon", "EOS 7D", Daylight, 2,		{ 2.1289, 1, 1.4990, 0 } },
  { "Canon", "EOS 7D", Daylight, 3,		{ 2.1602, 1, 1.4736, 0 } },
  { "Canon", "EOS 7D", Daylight, 4,		{ 2.1924, 1, 1.4482, 0 } },
  { "Canon", "EOS 7D", Daylight, 5,		{ 2.2266, 1, 1.4219, 0 } },
  { "Canon", "EOS 7D", Daylight, 6,		{ 2.2559, 1, 1.4004, 0 } },
  { "Canon", "EOS 7D", Daylight, 7,		{ 2.2910, 1, 1.3779, 0 } },
  { "Canon", "EOS 7D", Daylight, 8,		{ 2.3271, 1, 1.3545, 0 } },
  { "Canon", "EOS 7D", Daylight, 9,		{ 2.3652, 1, 1.3301, 0 } },
  { "Canon", "EOS 7D", Shade, -9,		{ 2.0938, 1, 1.5283, 0 } },
  { "Canon", "EOS 7D", Shade, -8,		{ 2.1240, 1, 1.5020, 0 } },
  { "Canon", "EOS 7D", Shade, -7,		{ 2.1562, 1, 1.4756, 0 } },
  { "Canon", "EOS 7D", Shade, -6,		{ 2.1885, 1, 1.4502, 0 } },
  { "Canon", "EOS 7D", Shade, -5,		{ 2.2217, 1, 1.4258, 0 } },
  { "Canon", "EOS 7D", Shade, -4,		{ 2.2510, 1, 1.4023, 0 } },
  { "Canon", "EOS 7D", Shade, -3,		{ 2.2861, 1, 1.3799, 0 } },
  { "Canon", "EOS 7D", Shade, -2,		{ 2.3223, 1, 1.3564, 0 } },
  { "Canon", "EOS 7D", Shade, -1,		{ 2.3594, 1, 1.3330, 0 } },
  { "Canon", "EOS 7D", Shade, 0,		{ 2.4033, 1, 1.3076, 0 } },
  { "Canon", "EOS 7D", Shade, 1,		{ 2.4326, 1, 1.2930, 0 } },
  { "Canon", "EOS 7D", Shade, 2,		{ 2.4678, 1, 1.2754, 0 } },
  { "Canon", "EOS 7D", Shade, 3,		{ 2.4980, 1, 1.2568, 0 } },
  { "Canon", "EOS 7D", Shade, 4,		{ 2.5342, 1, 1.2383, 0 } },
  { "Canon", "EOS 7D", Shade, 5,		{ 2.5664, 1, 1.2188, 0 } },
  { "Canon", "EOS 7D", Shade, 6,		{ 2.5928, 1, 1.2021, 0 } },
  { "Canon", "EOS 7D", Shade, 7,		{ 2.6260, 1, 1.1826, 0 } },
  { "Canon", "EOS 7D", Shade, 8,		{ 2.6602, 1, 1.1641, 0 } },
  { "Canon", "EOS 7D", Shade, 9,		{ 2.6943, 1, 1.1416, 0 } },
  { "Canon", "EOS 7D", Cloudy, -9,		{ 1.9658, 1, 1.6680, 0 } },
  { "Canon", "EOS 7D", Cloudy, -8,		{ 1.9883, 1, 1.6387, 0 } },
  { "Canon", "EOS 7D", Cloudy, -7,		{ 2.0117, 1, 1.6074, 0 } },
  { "Canon", "EOS 7D", Cloudy, -6,		{ 2.0400, 1, 1.5781, 0 } },
  { "Canon", "EOS 7D", Cloudy, -5,		{ 2.0684, 1, 1.5469, 0 } },
  { "Canon", "EOS 7D", Cloudy, -4,		{ 2.0986, 1, 1.5215, 0 } },
  { "Canon", "EOS 7D", Cloudy, -3,		{ 2.1338, 1, 1.4951, 0 } },
  { "Canon", "EOS 7D", Cloudy, -2,		{ 2.1650, 1, 1.4687, 0 } },
  { "Canon", "EOS 7D", Cloudy, -1,		{ 2.1924, 1, 1.4443, 0 } },
  { "Canon", "EOS 7D", Cloudy, 0,		{ 2.2266, 1, 1.4180, 0 } },
  { "Canon", "EOS 7D", Cloudy, 1,		{ 2.2607, 1, 1.3975, 0 } },
  { "Canon", "EOS 7D", Cloudy, 2,		{ 2.2959, 1, 1.3740, 0 } },
  { "Canon", "EOS 7D", Cloudy, 3,		{ 2.3330, 1, 1.3506, 0 } },
  { "Canon", "EOS 7D", Cloudy, 4,		{ 2.3701, 1, 1.3281, 0 } },
  { "Canon", "EOS 7D", Cloudy, 5,		{ 2.4150, 1, 1.3047, 0 } },
  { "Canon", "EOS 7D", Cloudy, 6,		{ 2.4443, 1, 1.2881, 0 } },
  { "Canon", "EOS 7D", Cloudy, 7,		{ 2.4736, 1, 1.2705, 0 } },
  { "Canon", "EOS 7D", Cloudy, 8,		{ 2.5098, 1, 1.2520, 0 } },
  { "Canon", "EOS 7D", Cloudy, 9,		{ 2.5469, 1, 1.2334, 0 } },
  { "Canon", "EOS 7D", Tungsten, -9,		{ 1.2686, 1, 2.7158, 0 } },
  { "Canon", "EOS 7D", Tungsten, -8,		{ 1.2861, 1, 2.6807, 0 } },
  { "Canon", "EOS 7D", Tungsten, -7,		{ 1.3047, 1, 2.6533, 0 } },
  { "Canon", "EOS 7D", Tungsten, -6,		{ 1.3232, 1, 2.6191, 0 } },
  { "Canon", "EOS 7D", Tungsten, -5,		{ 1.3418, 1, 2.5859, 0 } },
  { "Canon", "EOS 7D", Tungsten, -4,		{ 1.3633, 1, 2.5469, 0 } },
  { "Canon", "EOS 7D", Tungsten, -3,		{ 1.3838, 1, 2.4980, 0 } },
  { "Canon", "EOS 7D", Tungsten, -2,		{ 1.4023, 1, 2.4502, 0 } },
  { "Canon", "EOS 7D", Tungsten, -1,		{ 1.4258, 1, 2.3984, 0 } },
  { "Canon", "EOS 7D", Tungsten, 0,		{ 1.4482, 1, 2.3486, 0 } },
  { "Canon", "EOS 7D", Tungsten, 1,		{ 1.4687, 1, 2.3115, 0 } },
  { "Canon", "EOS 7D", Tungsten, 2,		{ 1.4902, 1, 2.2754, 0 } },
  { "Canon", "EOS 7D", Tungsten, 3,		{ 1.5127, 1, 2.2354, 0 } },
  { "Canon", "EOS 7D", Tungsten, 4,		{ 1.5352, 1, 2.1973, 0 } },
  { "Canon", "EOS 7D", Tungsten, 5,		{ 1.5605, 1, 2.1602, 0 } },
  { "Canon", "EOS 7D", Tungsten, 6,		{ 1.5850, 1, 2.1240, 0 } },
  { "Canon", "EOS 7D", Tungsten, 7,		{ 1.6104, 1, 2.0859, 0 } },
  { "Canon", "EOS 7D", Tungsten, 8,		{ 1.6328, 1, 2.0518, 0 } },
  { "Canon", "EOS 7D", Tungsten, 9,		{ 1.6592, 1, 2.0156, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -9,	{ 1.5850, 1, 2.5859, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -8,	{ 1.6104, 1, 2.5469, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -7,	{ 1.6328, 1, 2.4980, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -6,	{ 1.6621, 1, 2.4502, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -5,	{ 1.6895, 1, 2.3984, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -4,	{ 1.7119, 1, 2.3486, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -3,	{ 1.7383, 1, 2.3115, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -2,	{ 1.7656, 1, 2.2754, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, -1,	{ 1.7969, 1, 2.2354, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 0,	{ 1.8252, 1, 2.1924, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 1,	{ 1.8486, 1, 2.1562, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 2,	{ 1.8750, 1, 2.1201, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 3,	{ 1.9033, 1, 2.0859, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 4,	{ 1.9316, 1, 2.0518, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 5,	{ 1.9619, 1, 2.0156, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 6,	{ 1.9844, 1, 1.9805, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 7,	{ 2.0078, 1, 1.9502, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 8,	{ 2.0361, 1, 1.9209, 0 } },
  { "Canon", "EOS 7D", WhiteFluorescent, 9,	{ 2.0645, 1, 1.8896, 0 } },
  { "Canon", "EOS 7D", Flash, -9,		{ 2.0039, 1, 1.6572, 0 } },
  { "Canon", "EOS 7D", Flash, -8,		{ 2.0322, 1, 1.6279, 0 } },
  { "Canon", "EOS 7D", Flash, -7,		{ 2.0605, 1, 1.5977, 0 } },
  { "Canon", "EOS 7D", Flash, -6,		{ 2.0898, 1, 1.5654, 0 } },
  { "Canon", "EOS 7D", Flash, -5,		{ 2.1201, 1, 1.5371, 0 } },
  { "Canon", "EOS 7D", Flash, -4,		{ 2.1562, 1, 1.5127, 0 } },
  { "Canon", "EOS 7D", Flash, -3,		{ 2.1836, 1, 1.4863, 0 } },
  { "Canon", "EOS 7D", Flash, -2,		{ 2.2168, 1, 1.4609, 0 } },
  { "Canon", "EOS 7D", Flash, -1,		{ 2.2510, 1, 1.4365, 0 } },
  { "Canon", "EOS 7D", Flash, 0,		{ 2.2803, 1, 1.4102, 0 } },
  { "Canon", "EOS 7D", Flash, 1,		{ 2.3164, 1, 1.3896, 0 } },
  { "Canon", "EOS 7D", Flash, 2,		{ 2.3594, 1, 1.3672, 0 } },
  { "Canon", "EOS 7D", Flash, 3,		{ 2.4033, 1, 1.3438, 0 } },
  { "Canon", "EOS 7D", Flash, 4,		{ 2.4326, 1, 1.3184, 0 } },
  { "Canon", "EOS 7D", Flash, 5,		{ 2.4619, 1, 1.2998, 0 } },
  { "Canon", "EOS 7D", Flash, 6,		{ 2.4980, 1, 1.2812, 0 } },
  { "Canon", "EOS 7D", Flash, 7,		{ 2.5342, 1, 1.2646, 0 } },
  { "Canon", "EOS 7D", Flash, 8,		{ 2.5664, 1, 1.2461, 0 } },
  { "Canon", "EOS 7D", Flash, 9,		{ 2.5928, 1, 1.2266, 0 } },

  { "Canon", "EOS 7D Mark II", Daylight, 0,     { 1.9462890625, 1, 1.6386718750, 0 } },
  { "Canon", "EOS 7D Mark II", Shade, 0,        { 2.2402343750, 1, 1.4101562500, 0 } },
  { "Canon", "EOS 7D Mark II", Cloudy, 0,       { 2.0937500000, 1, 1.5195312500, 0 } },
  { "Canon", "EOS 7D Mark II", Tungsten, 0,     { 1.3730468750, 1, 2.3427734375, 0 } },
  { "Canon", "EOS 7D Mark II", Fluorescent, 0,  { 1.7177734375, 1, 2.2509765625, 0 } },
  { "Canon", "EOS 7D Mark II", Flash, 0,        { 2.1601562500, 1, 1.4882812500, 0 } },

  { "Canon", "EOS 10D", Daylight, 0,		{ 2.159856, 1, 1.218750, 0 } },
  { "Canon", "EOS 10D", Shade, 0,		{ 2.533654, 1, 1.036058, 0 } },
  { "Canon", "EOS 10D", Cloudy, 0,		{ 2.348558, 1, 1.116587, 0 } },
  { "Canon", "EOS 10D", Tungsten, 0,		{ 1.431544, 1, 1.851040, 0 } },
  { "Canon", "EOS 10D", Fluorescent, 0,		{ 1.891509, 1, 1.647406, 0 } },
  { "Canon", "EOS 10D", Flash, 0,		{ 2.385817, 1, 1.115385, 0 } },

  { "Canon", "EOS 20D", Daylight, 0,		{ 1.954680, 1, 1.478818, 0 } },
  { "Canon", "EOS 20D", Shade, 0,		{ 2.248276, 1, 1.227586, 0 } },
  { "Canon", "EOS 20D", Cloudy, 0,		{ 2.115271, 1, 1.336946, 0 } },
  { "Canon", "EOS 20D", Tungsten, 0,		{ 1.368087, 1, 2.417044, 0 } },
  { "Canon", "EOS 20D", Fluorescent, 0,		{ 1.752709, 1, 2.060098, 0 } },
  { "Canon", "EOS 20D", Flash, 0,		{ 2.145813, 1, 1.293596, 0 } },

  { "Canon", "EOS 30D", Daylight, 0,		{ 2.032227, 1, 1.537109, 0 } },
  { "Canon", "EOS 30D", Shade, 0,		{ 2.354492, 1, 1.264648, 0 } },
  { "Canon", "EOS 30D", Cloudy, 0,		{ 2.197266, 1, 1.389648, 0 } },
  { "Canon", "EOS 30D", Tungsten, 0,		{ 1.411084, 1, 2.447477, 0 } },
  { "Canon", "EOS 30D", Fluorescent, 0,		{ 1.761601, 1, 2.303913, 0 } },
  { "Canon", "EOS 30D", Flash, 0,		{ 2.226562, 1, 1.347656, 0 } },

  { "Canon", "EOS 40D", Daylight, 0,		{ 2.197266, 1, 1.438477, 0 } },
  { "Canon", "EOS 40D", Shade, 0,		{ 2.546875, 1, 1.185547, 0 } },
  { "Canon", "EOS 40D", Cloudy, 0,		{ 2.370117, 1, 1.290039, 0 } },
  { "Canon", "EOS 40D", Tungsten, 0,		{ 1.510563, 1, 2.235915, 0 } },
  { "Canon", "EOS 40D", Fluorescent, 0,		{ 2.019084, 1, 2.129771, 0 } },
  { "Canon", "EOS 40D", Flash, 0,		{ 2.409180, 1, 1.260742, 0 } },

  // Canon EOS 50D (firmware 1.0.3)
  { "Canon", "EOS 50D", Daylight, -9,		{ 1.865234, 1, 1.599609, 0 } },
  { "Canon", "EOS 50D", Daylight, -8,		{ 1.889648, 1, 1.580078, 0 } },
  { "Canon", "EOS 50D", Daylight, -7,		{ 1.910156, 1, 1.556641, 0 } },
  { "Canon", "EOS 50D", Daylight, -6,		{ 1.935547, 1, 1.535156, 0 } },
  { "Canon", "EOS 50D", Daylight, -5,		{ 1.965820, 1, 1.512695, 0 } },
  { "Canon", "EOS 50D", Daylight, -4,		{ 1.992188, 1, 1.490234, 0 } },
  { "Canon", "EOS 50D", Daylight, -3,		{ 2.015625, 1, 1.468750, 0 } },
  { "Canon", "EOS 50D", Daylight, -2,		{ 2.043945, 1, 1.448242, 0 } },
  { "Canon", "EOS 50D", Daylight, -1,		{ 2.068359, 1, 1.425781, 0 } },
  { "Canon", "EOS 50D", Daylight, 0,		{ 2.098633, 1, 1.402344, 0 } },
  { "Canon", "EOS 50D", Daylight, 1,		{ 2.124023, 1, 1.381836, 0 } },
  { "Canon", "EOS 50D", Daylight, 2,		{ 2.156250, 1, 1.358398, 0 } },
  { "Canon", "EOS 50D", Daylight, 3,		{ 2.183594, 1, 1.334961, 0 } },
  { "Canon", "EOS 50D", Daylight, 4,		{ 2.211914, 1, 1.312500, 0 } },
  { "Canon", "EOS 50D", Daylight, 5,		{ 2.240234, 1, 1.288086, 0 } },
  { "Canon", "EOS 50D", Daylight, 6,		{ 2.265625, 1, 1.270508, 0 } },
  { "Canon", "EOS 50D", Daylight, 7,		{ 2.291016, 1, 1.251953, 0 } },
  { "Canon", "EOS 50D", Daylight, 8,		{ 2.322266, 1, 1.233398, 0 } },
  { "Canon", "EOS 50D", Daylight, 9,		{ 2.359375, 1, 1.214844, 0 } },
  { "Canon", "EOS 50D", Shade, -9,		{ 2.124023, 1, 1.383789, 0 } },
  { "Canon", "EOS 50D", Shade, -8,		{ 2.151367, 1, 1.361328, 0 } },
  { "Canon", "EOS 50D", Shade, -7,		{ 2.178711, 1, 1.338867, 0 } },
  { "Canon", "EOS 50D", Shade, -6,		{ 2.207031, 1, 1.314453, 0 } },
  { "Canon", "EOS 50D", Shade, -5,		{ 2.235352, 1, 1.291016, 0 } },
  { "Canon", "EOS 50D", Shade, -4,		{ 2.260742, 1, 1.272461, 0 } },
  { "Canon", "EOS 50D", Shade, -3,		{ 2.291016, 1, 1.254883, 0 } },
  { "Canon", "EOS 50D", Shade, -2,		{ 2.322266, 1, 1.236328, 0 } },
  { "Canon", "EOS 50D", Shade, -1,		{ 2.354492, 1, 1.215820, 0 } },
  { "Canon", "EOS 50D", Shade, 0,		{ 2.386719, 1, 1.196289, 0 } },
  { "Canon", "EOS 50D", Shade, 1,		{ 2.403320, 1, 1.186523, 0 } },
  { "Canon", "EOS 50D", Shade, 2,		{ 2.420898, 1, 1.175781, 0 } },
  { "Canon", "EOS 50D", Shade, 3,		{ 2.438477, 1, 1.165039, 0 } },
  { "Canon", "EOS 50D", Shade, 4,		{ 2.461914, 1, 1.152344, 0 } },
  { "Canon", "EOS 50D", Shade, 5,		{ 2.485352, 1, 1.136719, 0 } },
  { "Canon", "EOS 50D", Shade, 6,		{ 2.522461, 1, 1.115234, 0 } },
  { "Canon", "EOS 50D", Shade, 7,		{ 2.559570, 1, 1.094727, 0 } },
  { "Canon", "EOS 50D", Shade, 8,		{ 2.598633, 1, 1.072266, 0 } },
  { "Canon", "EOS 50D", Shade, 9,		{ 2.645508, 1, 1.051758, 0 } },
  { "Canon", "EOS 50D", Cloudy, -9,		{ 1.996094, 1, 1.486328, 0 } },
  { "Canon", "EOS 50D", Cloudy, -8,		{ 2.019531, 1, 1.466797, 0 } },
  { "Canon", "EOS 50D", Cloudy, -7,		{ 2.043945, 1, 1.444336, 0 } },
  { "Canon", "EOS 50D", Cloudy, -6,		{ 2.073242, 1, 1.421875, 0 } },
  { "Canon", "EOS 50D", Cloudy, -5,		{ 2.102539, 1, 1.400391, 0 } },
  { "Canon", "EOS 50D", Cloudy, -4,		{ 2.128906, 1, 1.377930, 0 } },
  { "Canon", "EOS 50D", Cloudy, -3,		{ 2.156250, 1, 1.356445, 0 } },
  { "Canon", "EOS 50D", Cloudy, -2,		{ 2.188477, 1, 1.333008, 0 } },
  { "Canon", "EOS 50D", Cloudy, -1,		{ 2.211914, 1, 1.309570, 0 } },
  { "Canon", "EOS 50D", Cloudy, 0,		{ 2.240234, 1, 1.285156, 0 } },
  { "Canon", "EOS 50D", Cloudy, 1,		{ 2.270508, 1, 1.268555, 0 } },
  { "Canon", "EOS 50D", Cloudy, 2,		{ 2.295898, 1, 1.250000, 0 } },
  { "Canon", "EOS 50D", Cloudy, 3,		{ 2.327148, 1, 1.232422, 0 } },
  { "Canon", "EOS 50D", Cloudy, 4,		{ 2.359375, 1, 1.211914, 0 } },
  { "Canon", "EOS 50D", Cloudy, 5,		{ 2.392578, 1, 1.195313, 0 } },
  { "Canon", "EOS 50D", Cloudy, 6,		{ 2.409180, 1, 1.183594, 0 } },
  { "Canon", "EOS 50D", Cloudy, 7,		{ 2.426758, 1, 1.172852, 0 } },
  { "Canon", "EOS 50D", Cloudy, 8,		{ 2.444336, 1, 1.161133, 0 } },
  { "Canon", "EOS 50D", Cloudy, 9,		{ 2.467773, 1, 1.149414, 0 } },
  { "Canon", "EOS 50D", Tungsten, -9,		{ 1.379189, 1, 2.206349, 0 } },
  { "Canon", "EOS 50D", Tungsten, -8,		{ 1.394690, 1, 2.176991, 0 } },
  { "Canon", "EOS 50D", Tungsten, -7,		{ 1.412600, 1, 2.155280, 0 } },
  { "Canon", "EOS 50D", Tungsten, -6,		{ 1.428317, 1, 2.127337, 0 } },
  { "Canon", "EOS 50D", Tungsten, -5,		{ 1.448122, 1, 2.101073, 0 } },
  { "Canon", "EOS 50D", Tungsten, -4,		{ 1.467684, 1, 2.078097, 0 } },
  { "Canon", "EOS 50D", Tungsten, -3,		{ 1.484220, 1, 2.054103, 0 } },
  { "Canon", "EOS 50D", Tungsten, -2,		{ 1.501357, 1, 2.027149, 0 } },
  { "Canon", "EOS 50D", Tungsten, -1,		{ 1.521818, 1, 2.003636, 0 } },
  { "Canon", "EOS 50D", Tungsten, 0,		{ 1.542466, 1, 1.976256, 0 } },
  { "Canon", "EOS 50D", Tungsten, 1,		{ 1.561468, 1, 1.949541, 0 } },
  { "Canon", "EOS 50D", Tungsten, 2,		{ 1.581567, 1, 1.923502, 0 } },
  { "Canon", "EOS 50D", Tungsten, 3,		{ 1.602778, 1, 1.894444, 0 } },
  { "Canon", "EOS 50D", Tungsten, 4,		{ 1.624767, 1, 1.867784, 0 } },
  { "Canon", "EOS 50D", Tungsten, 5,		{ 1.647940, 1, 1.841760, 0 } },
  { "Canon", "EOS 50D", Tungsten, 6,		{ 1.669492, 1, 1.815443, 0 } },
  { "Canon", "EOS 50D", Tungsten, 7,		{ 1.686553, 1, 1.789773, 0 } },
  { "Canon", "EOS 50D", Tungsten, 8,		{ 1.708294, 1, 1.766444, 0 } },
  { "Canon", "EOS 50D", Tungsten, 9,		{ 1.729626, 1, 1.738255, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -9,	{ 1.683196, 1, 2.110193, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -8,	{ 1.704797, 1, 2.084871, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -7,	{ 1.727778, 1, 2.061111, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -6,	{ 1.747907, 1, 2.036279, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -5,	{ 1.767507, 1, 2.013072, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -4,	{ 1.791745, 1, 1.988743, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -3,	{ 1.812264, 1, 1.963208, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -2,	{ 1.834758, 1, 1.932574, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, -1,	{ 1.863419, 1, 1.907354, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 0,	{ 1.882805, 1, 1.876081, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 1,	{ 1.908124, 1, 1.852998, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 2,	{ 1.931774, 1, 1.822612, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 3,	{ 1.958008, 1, 1.799805, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 4,	{ 1.988281, 1, 1.771484, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 5,	{ 2.011719, 1, 1.747070, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 6,	{ 2.036133, 1, 1.720703, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 7,	{ 2.064453, 1, 1.698242, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 8,	{ 2.093750, 1, 1.678711, 0 } },
  { "Canon", "EOS 50D", WhiteFluorescent, 9,	{ 2.120117, 1, 1.654297, 0 } },
  { "Canon", "EOS 50D", Flash, -9,		{ 2.027344, 1, 1.466797, 0 } },
  { "Canon", "EOS 50D", Flash, -8,		{ 2.056641, 1, 1.446289, 0 } },
  { "Canon", "EOS 50D", Flash, -7,		{ 2.085938, 1, 1.423828, 0 } },
  { "Canon", "EOS 50D", Flash, -6,		{ 2.111328, 1, 1.402344, 0 } },
  { "Canon", "EOS 50D", Flash, -5,		{ 2.137695, 1, 1.379883, 0 } },
  { "Canon", "EOS 50D", Flash, -4,		{ 2.169922, 1, 1.358398, 0 } },
  { "Canon", "EOS 50D", Flash, -3,		{ 2.192383, 1, 1.334961, 0 } },
  { "Canon", "EOS 50D", Flash, -2,		{ 2.221680, 1, 1.311523, 0 } },
  { "Canon", "EOS 50D", Flash, -1,		{ 2.250977, 1, 1.288086, 0 } },
  { "Canon", "EOS 50D", Flash, 0,		{ 2.275391, 1, 1.268555, 0 } },
  { "Canon", "EOS 50D", Flash, 1,		{ 2.306641, 1, 1.251953, 0 } },
  { "Canon", "EOS 50D", Flash, 2,		{ 2.337891, 1, 1.233398, 0 } },
  { "Canon", "EOS 50D", Flash, 3,		{ 2.375977, 1, 1.212891, 0 } },
  { "Canon", "EOS 50D", Flash, 4,		{ 2.398438, 1, 1.195313, 0 } },
  { "Canon", "EOS 50D", Flash, 5,		{ 2.415039, 1, 1.185547, 0 } },
  { "Canon", "EOS 50D", Flash, 6,		{ 2.432617, 1, 1.173828, 0 } },
  { "Canon", "EOS 50D", Flash, 7,		{ 2.450195, 1, 1.162109, 0 } },
  { "Canon", "EOS 50D", Flash, 8,		{ 2.473633, 1, 1.150391, 0 } },
  { "Canon", "EOS 50D", Flash, 9,		{ 2.503906, 1, 1.132813, 0 } },
  { "Canon", "EOS 50D", "5000K", 0,		{ 2.056641, 1, 1.438477, 0 } },
  { "Canon", "EOS 50D", "6500K", 0,		{ 2.311523, 1, 1.239258, 0 } },

  /* Canon EOS 60D Firmware Version 1.1.1 */
  { "Canon", "EOS 60D", Daylight, -9,         { 1.765625, 1, 1.79296875, 0 } },
  { "Canon", "EOS 60D", Daylight, -8,         { 1.787109375, 1, 1.765625, 0 } },
  { "Canon", "EOS 60D", Daylight, -7,         { 1.8125, 1, 1.7353515625, 0 } },
  { "Canon", "EOS 60D", Daylight, -6,         { 1.8388671875, 1, 1.70703125, 0 } },
  { "Canon", "EOS 60D", Daylight, -5,         { 1.865234375, 1, 1.6787109375, 0 } },
  { "Canon", "EOS 60D", Daylight, -4,         { 1.896484375, 1, 1.6494140625, 0 } },
  { "Canon", "EOS 60D", Daylight, -3,         { 1.91796875, 1, 1.6181640625, 0 } },
  { "Canon", "EOS 60D", Daylight, -2,         { 1.943359375, 1, 1.587890625, 0 } },
  { "Canon", "EOS 60D", Daylight, -1,         { 1.9658203125, 1, 1.55859375, 0 } },
  { "Canon", "EOS 60D", Daylight, 0,          { 1.9921875, 1, 1.5263671875, 0 } },
  { "Canon", "EOS 60D", Daylight, 1,          { 2.01953125, 1, 1.50390625, 0 } },
  { "Canon", "EOS 60D", Daylight, 2,          { 2.0517578125, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 60D", Daylight, 3,          { 2.0810546875, 1, 1.4541015625, 0 } },
  { "Canon", "EOS 60D", Daylight, 4,          { 2.111328125, 1, 1.4296875, 0 } },
  { "Canon", "EOS 60D", Daylight, 5,          { 2.146484375, 1, 1.40625, 0 } },
  { "Canon", "EOS 60D", Daylight, 6,          { 2.173828125, 1, 1.3857421875, 0 } },
  { "Canon", "EOS 60D", Daylight, 7,          { 2.20703125, 1, 1.36328125, 0 } },
  { "Canon", "EOS 60D", Daylight, 8,          { 2.240234375, 1, 1.341796875, 0 } },
  { "Canon", "EOS 60D", Daylight, 9,          { 2.275390625, 1, 1.318359375, 0 } },
  { "Canon", "EOS 60D", Shade, -9,            { 2.01953125, 1, 1.505859375, 0 } },
  { "Canon", "EOS 60D", Shade, -8,            { 2.0478515625, 1, 1.4814453125, 0 } },
  { "Canon", "EOS 60D", Shade, -7,            { 2.0771484375, 1, 1.45703125, 0 } },
  { "Canon", "EOS 60D", Shade, -6,            { 2.107421875, 1, 1.4345703125, 0 } },
  { "Canon", "EOS 60D", Shade, -5,            { 2.142578125, 1, 1.41015625, 0 } },
  { "Canon", "EOS 60D", Shade, -4,            { 2.169921875, 1, 1.3876953125, 0 } },
  { "Canon", "EOS 60D", Shade, -3,            { 2.2021484375, 1, 1.365234375, 0 } },
  { "Canon", "EOS 60D", Shade, -2,            { 2.2353515625, 1, 1.34375, 0 } },
  { "Canon", "EOS 60D", Shade, -1,            { 2.2705078125, 1, 1.3212890625, 0 } },
  { "Canon", "EOS 60D", Shade, 0,             { 2.3115234375, 1, 1.2958984375, 0 } },
  { "Canon", "EOS 60D", Shade, 1,             { 2.337890625, 1, 1.28125, 0 } },
  { "Canon", "EOS 60D", Shade, 2,             { 2.365234375, 1, 1.2646484375, 0 } },
  { "Canon", "EOS 60D", Shade, 3,             { 2.3984375, 1, 1.2470703125, 0 } },
  { "Canon", "EOS 60D", Shade, 4,             { 2.4326171875, 1, 1.2294921875, 0 } },
  { "Canon", "EOS 60D", Shade, 5,             { 2.4619140625, 1, 1.2099609375, 0 } },
  { "Canon", "EOS 60D", Shade, 6,             { 2.4853515625, 1, 1.193359375, 0 } },
  { "Canon", "EOS 60D", Shade, 7,             { 2.515625, 1, 1.173828125, 0 } },
  { "Canon", "EOS 60D", Shade, 8,             { 2.546875, 1, 1.154296875, 0 } },
  { "Canon", "EOS 60D", Shade, 9,             { 2.5859375, 1, 1.1328125, 0 } },
  { "Canon", "EOS 60D", Cloudy, -9,           { 1.8994140625, 1, 1.6435546875, 0 } },
  { "Canon", "EOS 60D", Cloudy, -8,           { 1.9208984375, 1, 1.615234375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -7,           { 1.943359375, 1, 1.5849609375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -6,           { 1.96875, 1, 1.5537109375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -5,           { 1.99609375, 1, 1.5234375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -4,           { 2.0234375, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -3,           { 2.056640625, 1, 1.4755859375, 0 } },
  { "Canon", "EOS 60D", Cloudy, -2,           { 2.0859375, 1, 1.4501953125, 0 } },
  { "Canon", "EOS 60D", Cloudy, -1,           { 2.115234375, 1, 1.427734375, 0 } },
  { "Canon", "EOS 60D", Cloudy, 0,            { 2.146484375, 1, 1.40234375, 0 } },
  { "Canon", "EOS 60D", Cloudy, 1,            { 2.1787109375, 1, 1.3818359375, 0 } },
  { "Canon", "EOS 60D", Cloudy, 2,            { 2.2119140625, 1, 1.361328125, 0 } },
  { "Canon", "EOS 60D", Cloudy, 3,            { 2.24609375, 1, 1.3388671875, 0 } },
  { "Canon", "EOS 60D", Cloudy, 4,            { 2.2802734375, 1, 1.314453125, 0 } },
  { "Canon", "EOS 60D", Cloudy, 5,            { 2.322265625, 1, 1.29296875, 0 } },
  { "Canon", "EOS 60D", Cloudy, 6,            { 2.3427734375, 1, 1.2763671875, 0 } },
  { "Canon", "EOS 60D", Cloudy, 7,            { 2.3701171875, 1, 1.2607421875, 0 } },
  { "Canon", "EOS 60D", Cloudy, 8,            { 2.4033203125, 1, 1.2431640625, 0 } },
  { "Canon", "EOS 60D", Cloudy, 9,            { 2.4384765625, 1, 1.224609375, 0 } },
  { "Canon", "EOS 60D", Tungsten, -9,         { 1.2490234375, 1, 2.6455078125, 0 } },
  { "Canon", "EOS 60D", Tungsten, -8,         { 1.265625, 1, 2.619140625, 0 } },
  { "Canon", "EOS 60D", Tungsten, -7,         { 1.283203125, 1, 2.5927734375, 0 } },
  { "Canon", "EOS 60D", Tungsten, -6,         { 1.30078125, 1, 2.56640625, 0 } },
  { "Canon", "EOS 60D", Tungsten, -5,         { 1.3193359375, 1, 2.5341796875, 0 } },
  { "Canon", "EOS 60D", Tungsten, -4,         { 1.33984375, 1, 2.50390625, 0 } },
  { "Canon", "EOS 60D", Tungsten, -3,         { 1.3583984375, 1, 2.4560546875, 0 } },
  { "Canon", "EOS 60D", Tungsten, -2,         { 1.3759765625, 1, 2.4033203125, 0 } },
  { "Canon", "EOS 60D", Tungsten, -1,         { 1.3974609375, 1, 2.359375, 0 } },
  { "Canon", "EOS 60D", Tungsten, 0,          { 1.41796875, 1, 2.306640625, 0 } },
  { "Canon", "EOS 60D", Tungsten, 1,          { 1.4384765625, 1, 2.2705078125, 0 } },
  { "Canon", "EOS 60D", Tungsten, 2,          { 1.458984375, 1, 2.2353515625, 0 } },
  { "Canon", "EOS 60D", Tungsten, 3,          { 1.4794921875, 1, 2.197265625, 0 } },
  { "Canon", "EOS 60D", Tungsten, 4,          { 1.501953125, 1, 2.16015625, 0 } },
  { "Canon", "EOS 60D", Tungsten, 5,          { 1.5263671875, 1, 2.1240234375, 0 } },
  { "Canon", "EOS 60D", Tungsten, 6,          { 1.548828125, 1, 2.0859375, 0 } },
  { "Canon", "EOS 60D", Tungsten, 7,          { 1.5703125, 1, 2.0517578125, 0 } },
  { "Canon", "EOS 60D", Tungsten, 8,          { 1.5927734375, 1, 2.01953125, 0 } },
  { "Canon", "EOS 60D", Tungsten, 9,          { 1.6181640625, 1, 1.984375, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -9,      { 1.5732421875, 1, 2.5341796875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -8,      { 1.5947265625, 1, 2.498046875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -7,      { 1.6201171875, 1, 2.4501953125, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -6,      { 1.6435546875, 1, 2.3984375, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -5,      { 1.6650390625, 1, 2.3486328125, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -4,      { 1.6865234375, 1, 2.30078125, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -3,      { 1.7119140625, 1, 2.265625, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -2,      { 1.73828125, 1, 2.23046875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, -1,      { 1.765625, 1, 2.1923828125, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 0,       { 1.787109375, 1, 2.1513671875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 1,       { 1.8125, 1, 2.115234375, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 2,       { 1.8388671875, 1, 2.0810546875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 3,       { 1.8681640625, 1, 2.0478515625, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 4,       { 1.896484375, 1, 2.01171875, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 5,       { 1.91796875, 1, 1.9765625, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 6,       { 1.943359375, 1, 1.943359375, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 7,       { 1.96875, 1, 1.9140625, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 8,       { 1.99609375, 1, 1.8828125, 0 } },
  { "Canon", "EOS 60D", Fluorescent, 9,       { 2.0234375, 1, 1.8515625, 0 } },
  { "Canon", "EOS 60D", Flash, -9,            { 1.9501953125, 1, 1.6279296875, 0 } },
  { "Canon", "EOS 60D", Flash, -8,            { 1.9765625, 1, 1.59765625, 0 } },
  { "Canon", "EOS 60D", Flash, -7,            { 2.00390625, 1, 1.568359375, 0 } },
  { "Canon", "EOS 60D", Flash, -6,            { 2.0322265625, 1, 1.537109375, 0 } },
  { "Canon", "EOS 60D", Flash, -5,            { 2.064453125, 1, 1.5107421875, 0 } },
  { "Canon", "EOS 60D", Flash, -4,            { 2.09375, 1, 1.486328125, 0 } },
  { "Canon", "EOS 60D", Flash, -3,            { 2.1240234375, 1, 1.462890625, 0 } },
  { "Canon", "EOS 60D", Flash, -2,            { 2.15625, 1, 1.4384765625, 0 } },
  { "Canon", "EOS 60D", Flash, -1,            { 2.1884765625, 1, 1.4140625, 0 } },
  { "Canon", "EOS 60D", Flash, 0,             { 2.216796875, 1, 1.3916015625, 0 } },
  { "Canon", "EOS 60D", Flash, 1,             { 2.255859375, 1, 1.37109375, 0 } },
  { "Canon", "EOS 60D", Flash, 2,             { 2.2958984375, 1, 1.349609375, 0 } },
  { "Canon", "EOS 60D", Flash, 3,             { 2.3271484375, 1, 1.326171875, 0 } },
  { "Canon", "EOS 60D", Flash, 4,             { 2.3544921875, 1, 1.302734375, 0 } },
  { "Canon", "EOS 60D", Flash, 5,             { 2.3818359375, 1, 1.28515625, 0 } },
  { "Canon", "EOS 60D", Flash, 6,             { 2.4150390625, 1, 1.2685546875, 0 } },
  { "Canon", "EOS 60D", Flash, 7,             { 2.4501953125, 1, 1.251953125, 0 } },
  { "Canon", "EOS 60D", Flash, 8,             { 2.4736328125, 1, 1.232421875, 0 } },
  { "Canon", "EOS 60D", Flash, 9,             { 2.50390625, 1, 1.21484375, 0 } },

  /* Canon EOS 70D Firmware Version 1.1.1 */
  { "Canon", "EOS 70D", Daylight, 0,          { 2.068359375, 1, 1.6728515625, 0 } },
  { "Canon", "EOS 70D", Shade, 0,             { 2.4033203125, 1, 1.4462890625, 0 } },
  { "Canon", "EOS 70D", Cloudy, 0,            { 2.2265625, 1, 1.5517578125, 0 } },
  { "Canon", "EOS 70D", Tungsten, 0,          { 1.4482421875, 1, 2.4091796875, 0 } },
  { "Canon", "EOS 70D", Fluorescent, 0,       { 1.7900390625, 1, 2.2802734375, 0 } },
  { "Canon", "EOS 70D", Flash, 0,             { 2.2861328125, 1, 1.548828125, 0 } },

  /* Canon EOS 77D Firmware Version 1.0.2 */
  { "Canon", "EOS 77D", Daylight        , -9, { 1.556640625, 1, 1.9248046875, 0 } },
  { "Canon", "EOS 77D", Daylight        , -8, { 1.5751953125, 1, 1.8994140625, 0 } },
  { "Canon", "EOS 77D", Daylight        , -7, { 1.599609375, 1, 1.8720703125, 0 } },
  { "Canon", "EOS 77D", Daylight        , -6, { 1.623046875, 1, 1.841796875, 0 } },
  { "Canon", "EOS 77D", Daylight        , -5, { 1.6494140625, 1, 1.8154296875, 0 } },
  { "Canon", "EOS 77D", Daylight        , -4, { 1.67578125, 1, 1.787109375, 0 } },
  { "Canon", "EOS 77D", Daylight        , -3, { 1.6982421875, 1, 1.759765625, 0 } },
  { "Canon", "EOS 77D", Daylight        , -2, { 1.7236328125, 1, 1.7294921875, 0 } },
  { "Canon", "EOS 77D", Daylight        , -1, { 1.7470703125, 1, 1.701171875, 0 } },
  { "Canon", "EOS 77D", Daylight        , 0, { 1.7744140625, 1, 1.6708984375, 0 } },
  { "Canon", "EOS 77D", Daylight        , 1, { 1.7998046875, 1, 1.6513671875, 0 } },
  { "Canon", "EOS 77D", Daylight        , 2, { 1.822265625, 1, 1.6279296875, 0 } },
  { "Canon", "EOS 77D", Daylight        , 3, { 1.8486328125, 1, 1.60546875, 0 } },
  { "Canon", "EOS 77D", Daylight        , 4, { 1.875, 1, 1.5849609375, 0 } },
  { "Canon", "EOS 77D", Daylight        , 5, { 1.8994140625, 1, 1.5634765625, 0 } },
  { "Canon", "EOS 77D", Daylight        , 6, { 1.9287109375, 1, 1.5419921875, 0 } },
  { "Canon", "EOS 77D", Daylight        , 7, { 1.9541015625, 1, 1.51953125, 0 } },
  { "Canon", "EOS 77D", Daylight        , 8, { 1.98828125, 1, 1.4970703125, 0 } },
  { "Canon", "EOS 77D", Daylight        , 9, { 2.01953125, 1, 1.4755859375, 0 } },
  { "Canon", "EOS 77D", Shade           , -9, { 1.796875, 1, 1.6513671875, 0 } },
  { "Canon", "EOS 77D", Shade           , -8, { 1.822265625, 1, 1.630859375, 0 } },
  { "Canon", "EOS 77D", Shade           , -7, { 1.8447265625, 1, 1.607421875, 0 } },
  { "Canon", "EOS 77D", Shade           , -6, { 1.8720703125, 1, 1.587890625, 0 } },
  { "Canon", "EOS 77D", Shade           , -5, { 1.8994140625, 1, 1.5654296875, 0 } },
  { "Canon", "EOS 77D", Shade           , -4, { 1.9248046875, 1, 1.544921875, 0 } },
  { "Canon", "EOS 77D", Shade           , -3, { 1.9541015625, 1, 1.521484375, 0 } },
  { "Canon", "EOS 77D", Shade           , -2, { 1.984375, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 77D", Shade           , -1, { 2.015625, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 77D", Shade           , 0, { 2.0517578125, 1, 1.4521484375, 0 } },
  { "Canon", "EOS 77D", Shade           , 1, { 2.0771484375, 1, 1.4365234375, 0 } },
  { "Canon", "EOS 77D", Shade           , 2, { 2.1025390625, 1, 1.416015625, 0 } },
  { "Canon", "EOS 77D", Shade           , 3, { 2.1337890625, 1, 1.3974609375, 0 } },
  { "Canon", "EOS 77D", Shade           , 4, { 2.1650390625, 1, 1.3759765625, 0 } },
  { "Canon", "EOS 77D", Shade           , 5, { 2.1923828125, 1, 1.3583984375, 0 } },
  { "Canon", "EOS 77D", Shade           , 6, { 2.2216796875, 1, 1.3388671875, 0 } },
  { "Canon", "EOS 77D", Shade           , 7, { 2.2509765625, 1, 1.3193359375, 0 } },
  { "Canon", "EOS 77D", Shade           , 8, { 2.2861328125, 1, 1.2998046875, 0 } },
  { "Canon", "EOS 77D", Shade           , 9, { 2.322265625, 1, 1.2783203125, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -9, { 1.6787109375, 1, 1.7841796875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -8, { 1.701171875, 1, 1.75390625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -7, { 1.7265625, 1, 1.7265625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -6, { 1.75, 1, 1.6982421875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -5, { 1.77734375, 1, 1.6708984375, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -4, { 1.802734375, 1, 1.646484375, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -3, { 1.8251953125, 1, 1.625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -2, { 1.8515625, 1, 1.6025390625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , -1, { 1.87890625, 1, 1.5830078125, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 0, { 1.9033203125, 1, 1.55859375, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 1, { 1.931640625, 1, 1.5400390625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 2, { 1.9580078125, 1, 1.5166015625, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 3, { 1.9921875, 1, 1.4951171875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 4, { 2.0234375, 1, 1.4716796875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 5, { 2.060546875, 1, 1.4482421875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 6, { 2.0859375, 1, 1.4326171875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 7, { 2.111328125, 1, 1.412109375, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 8, { 2.1376953125, 1, 1.3935546875, 0 } },
  { "Canon", "EOS 77D", Cloudy          , 9, { 2.173828125, 1, 1.37109375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -9, { 1.099609375, 1, 2.673828125, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -8, { 1.1142578125, 1, 2.6455078125, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -7, { 1.12890625, 1, 2.619140625, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -6, { 1.1455078125, 1, 2.5927734375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -5, { 1.1611328125, 1, 2.5595703125, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -4, { 1.1787109375, 1, 2.5341796875, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -3, { 1.1953125, 1, 2.498046875, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -2, { 1.2119140625, 1, 2.4560546875, 0 } },
  { "Canon", "EOS 77D", Tungsten        , -1, { 1.2294921875, 1, 2.4208984375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 0, { 1.2490234375, 1, 2.3818359375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 1, { 1.265625, 1, 2.3486328125, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 2, { 1.28515625, 1, 2.31640625, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 3, { 1.302734375, 1, 2.2861328125, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 4, { 1.3232421875, 1, 2.2509765625, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 5, { 1.345703125, 1, 2.216796875, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 6, { 1.365234375, 1, 2.18359375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 7, { 1.3857421875, 1, 2.15625, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 8, { 1.40625, 1, 2.1240234375, 0 } },
  { "Canon", "EOS 77D", Tungsten        , 9, { 1.427734375, 1, 2.09375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -9, { 1.29296875, 1, 2.5537109375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -8, { 1.3125, 1, 2.5224609375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -7, { 1.3330078125, 1, 2.4853515625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -6, { 1.3544921875, 1, 2.4443359375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -5, { 1.3740234375, 1, 2.4091796875, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -4, { 1.3955078125, 1, 2.3759765625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -3, { 1.416015625, 1, 2.337890625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -2, { 1.4404296875, 1, 2.306640625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , -1, { 1.4609375, 1, 2.275390625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 0, { 1.4775390625, 1, 2.2353515625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 1, { 1.4990234375, 1, 2.20703125, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 2, { 1.51953125, 1, 2.173828125, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 3, { 1.5419921875, 1, 2.142578125, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 4, { 1.5634765625, 1, 2.115234375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 5, { 1.5849609375, 1, 2.0810546875, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 6, { 1.6103515625, 1, 2.056640625, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 7, { 1.6328125, 1, 2.02734375, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 8, { 1.6591796875, 1, 2.0, 0 } },
  { "Canon", "EOS 77D", Fluorescent     , 9, { 1.6865234375, 1, 1.97265625, 0 } },
  { "Canon", "EOS 77D", Flash           , -9, { 1.7265625, 1, 1.75390625, 0 } },
  { "Canon", "EOS 77D", Flash           , -8, { 1.75, 1, 1.7265625, 0 } },
  { "Canon", "EOS 77D", Flash           , -7, { 1.77734375, 1, 1.6982421875, 0 } },
  { "Canon", "EOS 77D", Flash           , -6, { 1.802734375, 1, 1.66796875, 0 } },
  { "Canon", "EOS 77D", Flash           , -5, { 1.828125, 1, 1.646484375, 0 } },
  { "Canon", "EOS 77D", Flash           , -4, { 1.8515625, 1, 1.625, 0 } },
  { "Canon", "EOS 77D", Flash           , -3, { 1.87890625, 1, 1.6025390625, 0 } },
  { "Canon", "EOS 77D", Flash           , -2, { 1.9033203125, 1, 1.5830078125, 0 } },
  { "Canon", "EOS 77D", Flash           , -1, { 1.931640625, 1, 1.55859375, 0 } },
  { "Canon", "EOS 77D", Flash           , 0, { 1.9580078125, 1, 1.537109375, 0 } },
  { "Canon", "EOS 77D", Flash           , -1, { 1.9921875, 1, 1.5166015625, 0 } },
  { "Canon", "EOS 77D", Flash           , -2, { 2.0234375, 1, 1.4951171875, 0 } },
  { "Canon", "EOS 77D", Flash           , -3, { 2.060546875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS 77D", Flash           , -4, { 2.0859375, 1, 1.4482421875, 0 } },
  { "Canon", "EOS 77D", Flash           , -5, { 2.111328125, 1, 1.4296875, 0 } },
  { "Canon", "EOS 77D", Flash           , -6, { 2.1376953125, 1, 1.412109375, 0 } },
  { "Canon", "EOS 77D", Flash           , -7, { 2.173828125, 1, 1.3916015625, 0 } },
  { "Canon", "EOS 77D", Flash           , -8, { 2.2021484375, 1, 1.37109375, 0 } },
  { "Canon", "EOS 77D", Flash           , -9, { 2.23046875, 1, 1.3525390625, 0 } },

  /* Canon EOS 80D Firmware Version Ver.1.0.1 */
  /* Fine-tuning is the camera's Amber-Blue bracketing. */
  { "Canon", "EOS 80D", Daylight, -9,         { 1.5517578125, 1, 1.892578125, 0 } },
  { "Canon", "EOS 80D", Daylight, -8,         { 1.5703125, 1, 1.865234375, 0 } },
  { "Canon", "EOS 80D", Daylight, -7,         { 1.5947265625, 1, 1.8388671875, 0 } },
  { "Canon", "EOS 80D", Daylight, -6,         { 1.6181640625, 1, 1.8095703125, 0 } },
  { "Canon", "EOS 80D", Daylight, -5,         { 1.6435546875, 1, 1.78125, 0 } },
  { "Canon", "EOS 80D", Daylight, -4,         { 1.6708984375, 1, 1.75390625, 0 } },
  { "Canon", "EOS 80D", Daylight, -3,         { 1.6923828125, 1, 1.7236328125, 0 } },
  { "Canon", "EOS 80D", Daylight, -2,         { 1.7177734375, 1, 1.6953125, 0 } },
  { "Canon", "EOS 80D", Daylight, -1,         { 1.7412109375, 1, 1.6650390625, 0 } },
  { "Canon", "EOS 80D", Daylight, 0,          { 1.7685546875, 1, 1.6328125, 0 } },
  { "Canon", "EOS 80D", Daylight, 1,          { 1.7900390625, 1, 1.6123046875, 0 } },
  { "Canon", "EOS 80D", Daylight, 2,          { 1.8154296875, 1, 1.58984375, 0 } },
  { "Canon", "EOS 80D", Daylight, 3,          { 1.8388671875, 1, 1.5654296875, 0 } },
  { "Canon", "EOS 80D", Daylight, 4,          { 1.8623046875, 1, 1.544921875, 0 } },
  { "Canon", "EOS 80D", Daylight, 5,          { 1.8896484375, 1, 1.5234375, 0 } },
  { "Canon", "EOS 80D", Daylight, 6,          { 1.9140625, 1, 1.501953125, 0 } },
  { "Canon", "EOS 80D", Daylight, 7,          { 1.939453125, 1, 1.4794921875, 0 } },
  { "Canon", "EOS 80D", Daylight, 8,          { 1.97265625, 1, 1.458984375, 0 } },
  { "Canon", "EOS 80D", Daylight, 9,          { 2.00390625, 1, 1.4345703125, 0 } },
  { "Canon", "EOS 80D", Shade, -9,            { 1.7900390625, 1, 1.615234375, 0 } },
  { "Canon", "EOS 80D", Shade, -8,            { 1.8125, 1, 1.5927734375, 0 } },
  { "Canon", "EOS 80D", Shade, -7,            { 1.8349609375, 1, 1.568359375, 0 } },
  { "Canon", "EOS 80D", Shade, -6,            { 1.8583984375, 1, 1.546875, 0 } },
  { "Canon", "EOS 80D", Shade, -5,            { 1.8857421875, 1, 1.5263671875, 0 } },
  { "Canon", "EOS 80D", Shade, -4,            { 1.91015625, 1, 1.50390625, 0 } },
  { "Canon", "EOS 80D", Shade, -3,            { 1.939453125, 1, 1.4814453125, 0 } },
  { "Canon", "EOS 80D", Shade, -2,            { 1.96875, 1, 1.4609375, 0 } },
  { "Canon", "EOS 80D", Shade, -1,            { 2, 1, 1.4384765625, 0 } },
  { "Canon", "EOS 80D", Shade, 0,             { 2.0361328125, 1, 1.412109375, 0 } },
  { "Canon", "EOS 80D", Shade, 1,             { 2.060546875, 1, 1.3955078125, 0 } },
  { "Canon", "EOS 80D", Shade, 2,             { 2.0859375, 1, 1.3759765625, 0 } },
  { "Canon", "EOS 80D", Shade, 3,             { 2.111328125, 1, 1.3564453125, 0 } },
  { "Canon", "EOS 80D", Shade, 4,             { 2.142578125, 1, 1.3349609375, 0 } },
  { "Canon", "EOS 80D", Shade, 5,             { 2.169921875, 1, 1.31640625, 0 } },
  { "Canon", "EOS 80D", Shade, 6,             { 2.197265625, 1, 1.2978515625, 0 } },
  { "Canon", "EOS 80D", Shade, 7,             { 2.2265625, 1, 1.2783203125, 0 } },
  { "Canon", "EOS 80D", Shade, 8,             { 2.2607421875, 1, 1.259765625, 0 } },
  { "Canon", "EOS 80D", Shade, 9,             { 2.2958984375, 1, 1.23828125, 0 } },
  { "Canon", "EOS 80D", Cloudy, -9,           { 1.6728515625, 1, 1.7470703125, 0 } },
  { "Canon", "EOS 80D", Cloudy, -8,           { 1.6953125, 1, 1.720703125, 0 } },
  { "Canon", "EOS 80D", Cloudy, -7,           { 1.720703125, 1, 1.689453125, 0 } },
  { "Canon", "EOS 80D", Cloudy, -6,           { 1.744140625, 1, 1.6591796875, 0 } },
  { "Canon", "EOS 80D", Cloudy, -5,           { 1.771484375, 1, 1.630859375, 0 } },
  { "Canon", "EOS 80D", Cloudy, -4,           { 1.79296875, 1, 1.6103515625, 0 } },
  { "Canon", "EOS 80D", Cloudy, -3,           { 1.818359375, 1, 1.587890625, 0 } },
  { "Canon", "EOS 80D", Cloudy, -2,           { 1.841796875, 1, 1.5634765625, 0 } },
  { "Canon", "EOS 80D", Cloudy, -1,           { 1.865234375, 1, 1.5419921875, 0 } },
  { "Canon", "EOS 80D", Cloudy, 0,            { 1.8896484375, 1, 1.51953125, 0 } },
  { "Canon", "EOS 80D", Cloudy, 1,            { 1.91796875, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 80D", Cloudy, 2,            { 1.943359375, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 80D", Cloudy, 3,            { 1.9765625, 1, 1.4541015625, 0 } },
  { "Canon", "EOS 80D", Cloudy, 4,            { 2.0078125, 1, 1.4326171875, 0 } },
  { "Canon", "EOS 80D", Cloudy, 5,            { 2.0439453125, 1, 1.408203125, 0 } },
  { "Canon", "EOS 80D", Cloudy, 6,            { 2.064453125, 1, 1.3916015625, 0 } },
  { "Canon", "EOS 80D", Cloudy, 7,            { 2.08984375, 1, 1.373046875, 0 } },
  { "Canon", "EOS 80D", Cloudy, 8,            { 2.1201171875, 1, 1.3505859375, 0 } },
  { "Canon", "EOS 80D", Cloudy, 9,            { 2.1513671875, 1, 1.330078125, 0 } },
  { "Canon", "EOS 80D", Tungsten, -9,         { 1.09765625, 1, 2.7021484375, 0 } },
  { "Canon", "EOS 80D", Tungsten, -8,         { 1.1123046875, 1, 2.673828125, 0 } },
  { "Canon", "EOS 80D", Tungsten, -7,         { 1.1279296875, 1, 2.6396484375, 0 } },
  { "Canon", "EOS 80D", Tungsten, -6,         { 1.142578125, 1, 2.60546875, 0 } },
  { "Canon", "EOS 80D", Tungsten, -5,         { 1.16015625, 1, 2.5791015625, 0 } },
  { "Canon", "EOS 80D", Tungsten, -4,         { 1.1767578125, 1, 2.541015625, 0 } },
  { "Canon", "EOS 80D", Tungsten, -3,         { 1.193359375, 1, 2.50390625, 0 } },
  { "Canon", "EOS 80D", Tungsten, -2,         { 1.2099609375, 1, 2.4619140625, 0 } },
  { "Canon", "EOS 80D", Tungsten, -1,         { 1.2275390625, 1, 2.4208984375, 0 } },
  { "Canon", "EOS 80D", Tungsten, 0,          { 1.2470703125, 1, 2.3759765625, 0 } },
  { "Canon", "EOS 80D", Tungsten, 1,          { 1.2646484375, 1, 2.3427734375, 0 } },
  { "Canon", "EOS 80D", Tungsten, 2,          { 1.283203125, 1, 2.3115234375, 0 } },
  { "Canon", "EOS 80D", Tungsten, 3,          { 1.30078125, 1, 2.275390625, 0 } },
  { "Canon", "EOS 80D", Tungsten, 4,          { 1.3212890625, 1, 2.240234375, 0 } },
  { "Canon", "EOS 80D", Tungsten, 5,          { 1.34375, 1, 2.2021484375, 0 } },
  { "Canon", "EOS 80D", Tungsten, 6,          { 1.36328125, 1, 2.169921875, 0 } },
  { "Canon", "EOS 80D", Tungsten, 7,          { 1.3837890625, 1, 2.1376953125, 0 } },
  { "Canon", "EOS 80D", Tungsten, 8,          { 1.404296875, 1, 2.1025390625, 0 } },
  { "Canon", "EOS 80D", Tungsten, 9,          { 1.42578125, 1, 2.0732421875, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -9,      { 1.291015625, 1, 2.5595703125, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -8,      { 1.3115234375, 1, 2.5224609375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -7,      { 1.33203125, 1, 2.4794921875, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -6,      { 1.3544921875, 1, 2.4384765625, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -5,      { 1.373046875, 1, 2.3984375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -4,      { 1.3935546875, 1, 2.359375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -3,      { 1.4140625, 1, 2.3271484375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -2,      { 1.4384765625, 1, 2.291015625, 0 } },
  { "Canon", "EOS 80D", Fluorescent, -1,      { 1.458984375, 1, 2.255859375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 0,       { 1.4755859375, 1, 2.216796875, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 1,       { 1.4951171875, 1, 2.18359375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 2,       { 1.5166015625, 1, 2.1513671875, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 3,       { 1.5400390625, 1, 2.1201171875, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 4,       { 1.560546875, 1, 2.0859375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 5,       { 1.5830078125, 1, 2.0517578125, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 6,       { 1.60546875, 1, 2.0234375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 7,       { 1.630859375, 1, 1.99609375, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 8,       { 1.6572265625, 1, 1.9658203125, 0 } },
  { "Canon", "EOS 80D", Fluorescent, 9,       { 1.681640625, 1, 1.935546875, 0 } },
  { "Canon", "EOS 80D", Flash, -9,            { 1.720703125, 1, 1.720703125, 0 } },
  { "Canon", "EOS 80D", Flash, -8,            { 1.744140625, 1, 1.6923828125, 0 } },
  { "Canon", "EOS 80D", Flash, -7,            { 1.771484375, 1, 1.662109375, 0 } },
  { "Canon", "EOS 80D", Flash, -6,            { 1.79296875, 1, 1.6328125, 0 } },
  { "Canon", "EOS 80D", Flash, -5,            { 1.818359375, 1, 1.6103515625, 0 } },
  { "Canon", "EOS 80D", Flash, -4,            { 1.841796875, 1, 1.587890625, 0 } },
  { "Canon", "EOS 80D", Flash, -3,            { 1.865234375, 1, 1.5654296875, 0 } },
  { "Canon", "EOS 80D", Flash, -2,            { 1.8896484375, 1, 1.544921875, 0 } },
  { "Canon", "EOS 80D", Flash, -1,            { 1.91796875, 1, 1.521484375, 0 } },
  { "Canon", "EOS 80D", Flash, 0,             { 1.943359375, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 80D", Flash, 1,             { 1.9765625, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 80D", Flash, 2,             { 2.0078125, 1, 1.45703125, 0 } },
  { "Canon", "EOS 80D", Flash, 3,             { 2.0439453125, 1, 1.4326171875, 0 } },
  { "Canon", "EOS 80D", Flash, 4,             { 2.064453125, 1, 1.41015625, 0 } },
  { "Canon", "EOS 80D", Flash, 5,             { 2.08984375, 1, 1.3916015625, 0 } },
  { "Canon", "EOS 80D", Flash, 6,             { 2.1201171875, 1, 1.373046875, 0 } },
  { "Canon", "EOS 80D", Flash, 7,             { 2.1513671875, 1, 1.3525390625, 0 } },
  { "Canon", "EOS 80D", Flash, 8,             { 2.1787109375, 1, 1.33203125, 0 } },
  { "Canon", "EOS 80D", Flash, 9,             { 2.20703125, 1, 1.3125, 0 } },

  // Copied from Canon EOS 650D
  { "Canon", "EOS 100D", Daylight, -3,		{ 1.9502, 1, 1.7539, 0 } },
  { "Canon", "EOS 100D", Daylight, 0,		{ 2.0322, 1, 1.6572, 0 } },
  { "Canon", "EOS 100D", Daylight, 3,		{ 2.1152, 1, 1.5850, 0 } },
  { "Canon", "EOS 100D", Shade, -3,		{ 2.2305, 1, 1.4951, 0 } },
  { "Canon", "EOS 100D", Shade, 0,		{ 2.3379, 1, 1.4238, 0 } },
  { "Canon", "EOS 100D", Shade, 3,		{ 2.4443, 1, 1.3604, 0 } },
  { "Canon", "EOS 100D", Cloudy, -3,		{ 2.0898, 1, 1.6074, 0 } },
  { "Canon", "EOS 100D", Cloudy, 0,		{ 2.1787, 1, 1.5332, 0 } },
  { "Canon", "EOS 100D", Cloudy, 3,		{ 2.2705, 1, 1.4668, 0 } },
  { "Canon", "EOS 100D", Tungsten, -3,		{ 1.3975, 1, 2.5928, 0 } },
  { "Canon", "EOS 100D", Tungsten, 0,		{ 1.4541, 1, 2.4561, 0 } },
  { "Canon", "EOS 100D", Tungsten, 3,		{ 1.5127, 1, 2.3428, 0 } },
  { "Canon", "EOS 100D", Fluorescent, -3,	{ 1.6982, 1, 2.4268, 0 } },
  { "Canon", "EOS 100D", Fluorescent, 0,	{ 1.7715, 1, 2.3066, 0 } },
  { "Canon", "EOS 100D", Fluorescent, 3,	{ 1.8486, 1, 2.1973, 0 } },
  { "Canon", "EOS 100D", Flash, -3,		{ 2.1699, 1, 1.5879, 0 } },
  { "Canon", "EOS 100D", Flash, 0,		{ 2.2607, 1, 1.5166, 0 } },
  { "Canon", "EOS 100D", Flash, 3,		{ 2.3701, 1, 1.4502, 0 } },

  { "Canon", "EOS 300D", Daylight, 0,	{ 2.072115, 1, 1.217548, 0 } },
  { "Canon", "EOS 300D", Shade, 0,	{ 2.455529, 1, 1.026442, 0 } },
  { "Canon", "EOS 300D", Cloudy, 0,	{ 2.254808, 1, 1.108173, 0 } },
  { "Canon", "EOS 300D", Tungsten, 0,	{ 1.349057, 1, 1.896226, 0 } },
  { "Canon", "EOS 300D", Fluorescent, 0, { 1.794664, 1, 1.711137, 0 } },
  { "Canon", "EOS 300D", Flash, 0,	{ 2.326923, 1, 1.098558, 0 } },

  // taken from EOS Rebel SL2, matches 200D and Kiss X9
  { "Canon", "EOS 200D", Daylight                , -9, { 1.5537109375, 1, 1.9033203125, 0 } },
  { "Canon", "EOS 200D", Daylight                , -8, { 1.5732421875, 1, 1.87890625, 0 } },
  { "Canon", "EOS 200D", Daylight                , -7, { 1.5947265625, 1, 1.8486328125, 0 } },
  { "Canon", "EOS 200D", Daylight                , -6, { 1.6201171875, 1, 1.822265625, 0 } },
  { "Canon", "EOS 200D", Daylight                , -5, { 1.6435546875, 1, 1.79296875, 0 } },
  { "Canon", "EOS 200D", Daylight                , -4, { 1.6708984375, 1, 1.765625, 0 } },
  { "Canon", "EOS 200D", Daylight                , -3, { 1.6923828125, 1, 1.7353515625, 0 } },
  { "Canon", "EOS 200D", Daylight                , -2, { 1.7177734375, 1, 1.7099609375, 0 } },
  { "Canon", "EOS 200D", Daylight                , -1, { 1.7412109375, 1, 1.6787109375, 0 } },
  { "Canon", "EOS 200D", Daylight                , 0, { 1.7685546875, 1, 1.6494140625, 0 } },
  { "Canon", "EOS 200D", Daylight                , 1, { 1.7900390625, 1, 1.6279296875, 0 } },
  { "Canon", "EOS 200D", Daylight                , 2, { 1.8154296875, 1, 1.60546875, 0 } },
  { "Canon", "EOS 200D", Daylight                , 3, { 1.8388671875, 1, 1.5830078125, 0 } },
  { "Canon", "EOS 200D", Daylight                , 4, { 1.865234375, 1, 1.5634765625, 0 } },
  { "Canon", "EOS 200D", Daylight                , 5, { 1.8896484375, 1, 1.5419921875, 0 } },
  { "Canon", "EOS 200D", Daylight                , 6, { 1.91796875, 1, 1.521484375, 0 } },
  { "Canon", "EOS 200D", Daylight                , 7, { 1.943359375, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 200D", Daylight                , 8, { 1.97265625, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 200D", Daylight                , 9, { 2.00390625, 1, 1.4541015625, 0 } },
  { "Canon", "EOS 200D", Shade                   , -9, { 1.7900390625, 1, 1.630859375, 0 } },
  { "Canon", "EOS 200D", Shade                   , -8, { 1.8125, 1, 1.607421875, 0 } },
  { "Canon", "EOS 200D", Shade                   , -7, { 1.8349609375, 1, 1.5849609375, 0 } },
  { "Canon", "EOS 200D", Shade                   , -6, { 1.8623046875, 1, 1.5654296875, 0 } },
  { "Canon", "EOS 200D", Shade                   , -5, { 1.8896484375, 1, 1.544921875, 0 } },
  { "Canon", "EOS 200D", Shade                   , -4, { 1.9140625, 1, 1.5234375, 0 } },
  { "Canon", "EOS 200D", Shade                   , -3, { 1.939453125, 1, 1.501953125, 0 } },
  { "Canon", "EOS 200D", Shade                   , -2, { 1.96875, 1, 1.4794921875, 0 } },
  { "Canon", "EOS 200D", Shade                   , -1, { 2.0, 1, 1.45703125, 0 } },
  { "Canon", "EOS 200D", Shade                   , 0, { 2.0361328125, 1, 1.4326171875, 0 } },
  { "Canon", "EOS 200D", Shade                   , 1, { 2.060546875, 1, 1.416015625, 0 } },
  { "Canon", "EOS 200D", Shade                   , 2, { 2.0859375, 1, 1.3974609375, 0 } },
  { "Canon", "EOS 200D", Shade                   , 3, { 2.115234375, 1, 1.3779296875, 0 } },
  { "Canon", "EOS 200D", Shade                   , 4, { 2.146484375, 1, 1.3583984375, 0 } },
  { "Canon", "EOS 200D", Shade                   , 5, { 2.173828125, 1, 1.3388671875, 0 } },
  { "Canon", "EOS 200D", Shade                   , 6, { 2.2021484375, 1, 1.3212890625, 0 } },
  { "Canon", "EOS 200D", Shade                   , 7, { 2.23046875, 1, 1.302734375, 0 } },
  { "Canon", "EOS 200D", Shade                   , 8, { 2.2607421875, 1, 1.28125, 0 } },
  { "Canon", "EOS 200D", Shade                   , 9, { 2.2958984375, 1, 1.2607421875, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -9, { 1.6728515625, 1, 1.7626953125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -8, { 1.6953125, 1, 1.732421875, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -7, { 1.720703125, 1, 1.7041015625, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -6, { 1.744140625, 1, 1.67578125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -5, { 1.771484375, 1, 1.646484375, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -4, { 1.79296875, 1, 1.625, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -3, { 1.818359375, 1, 1.6025390625, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -2, { 1.841796875, 1, 1.580078125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , -1, { 1.8681640625, 1, 1.560546875, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 0, { 1.892578125, 1, 1.537109375, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 1, { 1.9208984375, 1, 1.5166015625, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 2, { 1.9462890625, 1, 1.4970703125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 3, { 1.9765625, 1, 1.4736328125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 4, { 2.01171875, 1, 1.4501953125, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 5, { 2.0439453125, 1, 1.427734375, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 6, { 2.068359375, 1, 1.412109375, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 7, { 2.09375, 1, 1.3935546875, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 8, { 2.1201171875, 1, 1.3740234375, 0 } },
  { "Canon", "EOS 200D", Cloudy                  , 9, { 2.15625, 1, 1.3525390625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -9, { 1.0986328125, 1, 2.66015625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -8, { 1.11328125, 1, 2.6328125, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -7, { 1.12890625, 1, 2.5986328125, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -6, { 1.14453125, 1, 2.5732421875, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -5, { 1.1611328125, 1, 2.541015625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -4, { 1.1787109375, 1, 2.515625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -3, { 1.1953125, 1, 2.4736328125, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -2, { 1.2119140625, 1, 2.4384765625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , -1, { 1.2294921875, 1, 2.3984375, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 0, { 1.2490234375, 1, 2.359375, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 1, { 1.265625, 1, 2.3271484375, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 2, { 1.283203125, 1, 2.2958984375, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 3, { 1.302734375, 1, 2.2607421875, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 4, { 1.3232421875, 1, 2.2265625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 5, { 1.34375, 1, 2.1923828125, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 6, { 1.36328125, 1, 2.16015625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 7, { 1.3837890625, 1, 2.1337890625, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 8, { 1.404296875, 1, 2.0986328125, 0 } },
  { "Canon", "EOS 200D", Tungsten                , 9, { 1.42578125, 1, 2.068359375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -9, { 1.29296875, 1, 2.5341796875, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -8, { 1.3115234375, 1, 2.498046875, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -7, { 1.33203125, 1, 2.4619140625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -6, { 1.3525390625, 1, 2.4208984375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -5, { 1.373046875, 1, 2.38671875, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -4, { 1.3935546875, 1, 2.3486328125, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -3, { 1.4140625, 1, 2.31640625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -2, { 1.4384765625, 1, 2.2802734375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , -1, { 1.458984375, 1, 2.2509765625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 0, { 1.4755859375, 1, 2.2119140625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 1, { 1.4970703125, 1, 2.1787109375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 2, { 1.5166015625, 1, 2.1513671875, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 3, { 1.5400390625, 1, 2.1201171875, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 4, { 1.560546875, 1, 2.08984375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 5, { 1.5830078125, 1, 2.056640625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 6, { 1.60546875, 1, 2.02734375, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 7, { 1.630859375, 1, 2.00390625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 8, { 1.6572265625, 1, 1.9765625, 0 } },
  { "Canon", "EOS 200D", CoolWhiteFluorescent    , 9, { 1.681640625, 1, 1.9501953125, 0 } },
  { "Canon", "EOS 200D", Flash                   , -9, { 1.720703125, 1, 1.732421875, 0 } },
  { "Canon", "EOS 200D", Flash                   , -8, { 1.7470703125, 1, 1.7041015625, 0 } },
  { "Canon", "EOS 200D", Flash                   , -7, { 1.7744140625, 1, 1.67578125, 0 } },
  { "Canon", "EOS 200D", Flash                   , -6, { 1.796875, 1, 1.6494140625, 0 } },
  { "Canon", "EOS 200D", Flash                   , -5, { 1.818359375, 1, 1.625, 0 } },
  { "Canon", "EOS 200D", Flash                   , -4, { 1.8447265625, 1, 1.6025390625, 0 } },
  { "Canon", "EOS 200D", Flash                   , -3, { 1.8681640625, 1, 1.580078125, 0 } },
  { "Canon", "EOS 200D", Flash                   , -2, { 1.896484375, 1, 1.560546875, 0 } },
  { "Canon", "EOS 200D", Flash                   , -1, { 1.9208984375, 1, 1.5400390625, 0 } },
  { "Canon", "EOS 200D", Flash                   , 0, { 1.9462890625, 1, 1.5166015625, 0 } },
  { "Canon", "EOS 200D", Flash                   , 1, { 1.9765625, 1, 1.4970703125, 0 } },
  { "Canon", "EOS 200D", Flash                   , 2, { 2.01171875, 1, 1.4755859375, 0 } },
  { "Canon", "EOS 200D", Flash                   , 3, { 2.0439453125, 1, 1.4521484375, 0 } },
  { "Canon", "EOS 200D", Flash                   , 4, { 2.068359375, 1, 1.4296875, 0 } },
  { "Canon", "EOS 200D", Flash                   , 5, { 2.09375, 1, 1.412109375, 0 } },
  { "Canon", "EOS 200D", Flash                   , 6, { 2.1240234375, 1, 1.3935546875, 0 } },
  { "Canon", "EOS 200D", Flash                   , 7, { 2.15625, 1, 1.3740234375, 0 } },
  { "Canon", "EOS 200D", Flash                   , 8, { 2.18359375, 1, 1.3525390625, 0 } },
  { "Canon", "EOS 200D", Flash                   , 9, { 2.20703125, 1, 1.3349609375, 0 } },

  // Firmware version 1.0.3. Fine tuning is from A9 to B9 on amber-blue.
  { "Canon", "EOS 350D", Daylight, -9,	{ 2.7436, 1, 1.2240, 0 } },
  { "Canon", "EOS 350D", Daylight, 0,	{ 2.3605, 1, 1.4450, 0 } },
  { "Canon", "EOS 350D", Daylight, 9,	{ 2.0138, 1, 1.7151, 0 } },
  { "Canon", "EOS 350D", Shade, -9,	{ 3.1857, 1, 1.0285, 0 } },
  { "Canon", "EOS 350D", Shade, 0,	{ 2.7888, 1, 1.2024, 0 } },
  { "Canon", "EOS 350D", Shade, 9,	{ 2.3988, 1, 1.4214, 0 } },
  { "Canon", "EOS 350D", Cloudy, -9,	{ 2.9912, 1, 1.1169, 0 } },
  { "Canon", "EOS 350D", Cloudy, 0,	{ 2.5727, 1, 1.3075, 0 } },
  { "Canon", "EOS 350D", Cloudy, 9,	{ 2.2033, 1, 1.5589, 0 } },
  { "Canon", "EOS 350D", Tungsten, -9,	{ 1.5589, 1, 1.9205, 0 } },
  { "Canon", "EOS 350D", Tungsten, 0,	{ 1.5343, 1, 2.2880, 0 } },
  { "Canon", "EOS 350D", Tungsten, 9,	{ 1.3145, 1, 2.6873, 0 } },
  { "Canon", "EOS 350D", Fluorescent, -9, { 2.3124, 1, 1.6356, 0 } },
  { "Canon", "EOS 350D", Fluorescent, 0, { 1.9754, 1, 1.9303, 0 } },
  { "Canon", "EOS 350D", Fluorescent, 9, { 1.6657, 1, 2.3034, 0 } },
  { "Canon", "EOS 350D", Flash, -9,	{ 3.0904, 1, 1.0756, 0 } },
  { "Canon", "EOS 350D", Flash, 0,	{ 2.6729, 1, 1.2613, 0 } },
  { "Canon", "EOS 350D", Flash, 9,	{ 2.3026, 1, 1.4961, 0 } },

  // Canon EOS 400D (firmware 1.1.1) white balance presets, 5 mireds per step
  { "Canon", "EOS 400D", Daylight, -9,	{ 1.972656, 1, 1.735352, 0 } },
  { "Canon", "EOS 400D", Daylight, -8,	{ 2.003906, 1, 1.707031, 0 } },
  { "Canon", "EOS 400D", Daylight, -7,	{ 2.036133, 1, 1.675781, 0 } },
  { "Canon", "EOS 400D", Daylight, -6,	{ 2.073242, 1, 1.646484, 0 } },
  { "Canon", "EOS 400D", Daylight, -5,	{ 2.111328, 1, 1.615234, 0 } },
  { "Canon", "EOS 400D", Daylight, -4,	{ 2.151367, 1, 1.583008, 0 } },
  { "Canon", "EOS 400D", Daylight, -3,	{ 2.183594, 1, 1.553711, 0 } },
  { "Canon", "EOS 400D", Daylight, -2,	{ 2.221680, 1, 1.523438, 0 } },
  { "Canon", "EOS 400D", Daylight, -1,	{ 2.260742, 1, 1.495117, 0 } },
  { "Canon", "EOS 400D", Daylight, 0,	{ 2.300781, 1, 1.462891, 0 } },
  { "Canon", "EOS 400D", Daylight, 1,	{ 2.337891, 1, 1.436523, 0 } },
  { "Canon", "EOS 400D", Daylight, 2,	{ 2.375977, 1, 1.408203, 0 } },
  { "Canon", "EOS 400D", Daylight, 3,	{ 2.415039, 1, 1.379883, 0 } },
  { "Canon", "EOS 400D", Daylight, 4,	{ 2.461914, 1, 1.354492, 0 } },
  { "Canon", "EOS 400D", Daylight, 5,	{ 2.503906, 1, 1.328125, 0 } },
  { "Canon", "EOS 400D", Daylight, 6,	{ 2.541016, 1, 1.304688, 0 } },
  { "Canon", "EOS 400D", Daylight, 7,	{ 2.579102, 1, 1.280273, 0 } },
  { "Canon", "EOS 400D", Daylight, 8,	{ 2.619141, 1, 1.256836, 0 } },
  { "Canon", "EOS 400D", Daylight, 9,	{ 2.666992, 1, 1.232422, 0 } },
  { "Canon", "EOS 400D", Shade, -9,	{ 2.333008, 1, 1.440430, 0 } },
  { "Canon", "EOS 400D", Shade, -8,	{ 2.370117, 1, 1.410156, 0 } },
  { "Canon", "EOS 400D", Shade, -7,	{ 2.409180, 1, 1.383789, 0 } },
  { "Canon", "EOS 400D", Shade, -6,	{ 2.456055, 1, 1.356445, 0 } },
  { "Canon", "EOS 400D", Shade, -5,	{ 2.503906, 1, 1.330078, 0 } },
  { "Canon", "EOS 400D", Shade, -4,	{ 2.541016, 1, 1.305664, 0 } },
  { "Canon", "EOS 400D", Shade, -3,	{ 2.579102, 1, 1.283203, 0 } },
  { "Canon", "EOS 400D", Shade, -2,	{ 2.619141, 1, 1.259766, 0 } },
  { "Canon", "EOS 400D", Shade, -1,	{ 2.660156, 1, 1.235352, 0 } },
  { "Canon", "EOS 400D", Shade, 0,	{ 2.708984, 1, 1.208984, 0 } },
  { "Canon", "EOS 400D", Shade, 1,	{ 2.745117, 1, 1.189453, 0 } },
  { "Canon", "EOS 400D", Shade, 2,	{ 2.782227, 1, 1.168945, 0 } },
  { "Canon", "EOS 400D", Shade, 3,	{ 2.829102, 1, 1.148438, 0 } },
  { "Canon", "EOS 400D", Shade, 4,	{ 2.875977, 1, 1.125000, 0 } },
  { "Canon", "EOS 400D", Shade, 5,	{ 2.916992, 1, 1.105469, 0 } },
  { "Canon", "EOS 400D", Shade, 6,	{ 2.951172, 1, 1.087891, 0 } },
  { "Canon", "EOS 400D", Shade, 7,	{ 2.994141, 1, 1.069336, 0 } },
  { "Canon", "EOS 400D", Shade, 8,	{ 3.039063, 1, 1.048828, 0 } },
  { "Canon", "EOS 400D", Shade, 9,	{ 3.083984, 1, 1.030273, 0 } },
  { "Canon", "EOS 400D", Cloudy, -9,	{ 2.156250, 1, 1.580078, 0 } },
  { "Canon", "EOS 400D", Cloudy, -8,	{ 2.188477, 1, 1.551758, 0 } },
  { "Canon", "EOS 400D", Cloudy, -7,	{ 2.226563, 1, 1.521484, 0 } },
  { "Canon", "EOS 400D", Cloudy, -6,	{ 2.265625, 1, 1.490234, 0 } },
  { "Canon", "EOS 400D", Cloudy, -5,	{ 2.306641, 1, 1.460938, 0 } },
  { "Canon", "EOS 400D", Cloudy, -4,	{ 2.342773, 1, 1.432617, 0 } },
  { "Canon", "EOS 400D", Cloudy, -3,	{ 2.381836, 1, 1.404297, 0 } },
  { "Canon", "EOS 400D", Cloudy, -2,	{ 2.420898, 1, 1.375977, 0 } },
  { "Canon", "EOS 400D", Cloudy, -1,	{ 2.467773, 1, 1.350586, 0 } },
  { "Canon", "EOS 400D", Cloudy, 0,	{ 2.509766, 1, 1.323242, 0 } },
  { "Canon", "EOS 400D", Cloudy, 1,	{ 2.546875, 1, 1.300781, 0 } },
  { "Canon", "EOS 400D", Cloudy, 2,	{ 2.585938, 1, 1.278320, 0 } },
  { "Canon", "EOS 400D", Cloudy, 3,	{ 2.625977, 1, 1.252930, 0 } },
  { "Canon", "EOS 400D", Cloudy, 4,	{ 2.673828, 1, 1.229492, 0 } },
  { "Canon", "EOS 400D", Cloudy, 5,	{ 2.723633, 1, 1.205078, 0 } },
  { "Canon", "EOS 400D", Cloudy, 6,	{ 2.752930, 1, 1.185547, 0 } },
  { "Canon", "EOS 400D", Cloudy, 7,	{ 2.790039, 1, 1.165039, 0 } },
  { "Canon", "EOS 400D", Cloudy, 8,	{ 2.836914, 1, 1.142578, 0 } },
  { "Canon", "EOS 400D", Cloudy, 9,	{ 2.884766, 1, 1.120117, 0 } },
  { "Canon", "EOS 400D", Tungsten, -9,	{ 1.320106, 1, 2.752205, 0 } },
  { "Canon", "EOS 400D", Tungsten, -8,	{ 1.340708, 1, 2.703540, 0 } },
  { "Canon", "EOS 400D", Tungsten, -7,	{ 1.359680, 1, 2.655417, 0 } },
  { "Canon", "EOS 400D", Tungsten, -6,	{ 1.381802, 1, 2.606601, 0 } },
  { "Canon", "EOS 400D", Tungsten, -5,	{ 1.406446, 1, 2.555953, 0 } },
  { "Canon", "EOS 400D", Tungsten, -4,	{ 1.428957, 1, 2.504496, 0 } },
  { "Canon", "EOS 400D", Tungsten, -3,	{ 1.452575, 1, 2.459801, 0 } },
  { "Canon", "EOS 400D", Tungsten, -2,	{ 1.475931, 1, 2.419619, 0 } },
  { "Canon", "EOS 400D", Tungsten, -1,	{ 1.501825, 1, 2.377737, 0 } },
  { "Canon", "EOS 400D", Tungsten, 0,	{ 1.526123, 1, 2.330889, 0 } },
  { "Canon", "EOS 400D", Tungsten, 1,	{ 1.548893, 1, 2.286900, 0 } },
  { "Canon", "EOS 400D", Tungsten, 2,	{ 1.572753, 1, 2.238184, 0 } },
  { "Canon", "EOS 400D", Tungsten, 3,	{ 1.599254, 1, 2.198509, 0 } },
  { "Canon", "EOS 400D", Tungsten, 4,	{ 1.624765, 1, 2.149156, 0 } },
  { "Canon", "EOS 400D", Tungsten, 5,	{ 1.653774, 1, 2.102830, 0 } },
  { "Canon", "EOS 400D", Tungsten, 6,	{ 1.681861, 1, 2.064577, 0 } },
  { "Canon", "EOS 400D", Tungsten, 7,	{ 1.709369, 1, 2.022945, 0 } },
  { "Canon", "EOS 400D", Tungsten, 8,	{ 1.737247, 1, 1.982676, 0 } },
  { "Canon", "EOS 400D", Tungsten, 9,	{ 1.770349, 1, 1.946705, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -9, { 1.638122, 1, 2.485267, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -8, { 1.667900, 1, 2.445883, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -7, { 1.695814, 1, 2.404651, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -6, { 1.723364, 1, 2.361682, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -5, { 1.752820, 1, 2.317669, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -4, { 1.788079, 1, 2.263009, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -3, { 1.815414, 1, 2.221694, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -2, { 1.844828, 1, 2.175287, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, -1, { 1.880309, 1, 2.127413, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 0, { 1.910506, 1, 2.080739, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 1, { 1.950195, 1, 2.043945, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 2, { 1.984375, 1, 2.007813, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 3, { 2.015625, 1, 1.968750, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 4, { 2.047852, 1, 1.928711, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 5, { 2.085938, 1, 1.892578, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 6, { 2.124023, 1, 1.858398, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 7, { 2.165039, 1, 1.825195, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 8, { 2.197266, 1, 1.790039, 0 } },
  { "Canon", "EOS 400D", WhiteFluorescent, 9, { 2.235352, 1, 1.756836, 0 } },
  { "Canon", "EOS 400D", Flash, -9,	{ 2.398438, 1, 1.432617, 0 } },
  { "Canon", "EOS 400D", Flash, -8,	{ 2.438477, 1, 1.402344, 0 } },
  { "Canon", "EOS 400D", Flash, -7,	{ 2.485352, 1, 1.375977, 0 } },
  { "Canon", "EOS 400D", Flash, -6,	{ 2.528320, 1, 1.349609, 0 } },
  { "Canon", "EOS 400D", Flash, -5,	{ 2.566406, 1, 1.323242, 0 } },
  { "Canon", "EOS 400D", Flash, -4,	{ 2.605469, 1, 1.299805, 0 } },
  { "Canon", "EOS 400D", Flash, -3,	{ 2.645508, 1, 1.276367, 0 } },
  { "Canon", "EOS 400D", Flash, -2,	{ 2.694336, 1, 1.251953, 0 } },
  { "Canon", "EOS 400D", Flash, -1,	{ 2.738281, 1, 1.227539, 0 } },
  { "Canon", "EOS 400D", Flash, 0,	{ 2.767578, 1, 1.203125, 0 } },
  { "Canon", "EOS 400D", Flash, 1,	{ 2.813477, 1, 1.183594, 0 } },
  { "Canon", "EOS 400D", Flash, 2,	{ 2.860352, 1, 1.164063, 0 } },
  { "Canon", "EOS 400D", Flash, 3,	{ 2.900391, 1, 1.141602, 0 } },
  { "Canon", "EOS 400D", Flash, 4,	{ 2.942383, 1, 1.118164, 0 } },
  { "Canon", "EOS 400D", Flash, 5,	{ 2.976563, 1, 1.101563, 0 } },
  { "Canon", "EOS 400D", Flash, 6,	{ 3.020508, 1, 1.082031, 0 } },
  { "Canon", "EOS 400D", Flash, 7,	{ 3.065430, 1, 1.063477, 0 } },
  { "Canon", "EOS 400D", Flash, 8,	{ 3.122070, 1, 1.041992, 0 } },
  { "Canon", "EOS 400D", Flash, 9,	{ 3.169922, 1, 1.024414, 0 } },

  { "Canon", "EOS 450D", Daylight, 0,		{ 2.216797, 1, 1.471680, 0 } },
  { "Canon", "EOS 450D", Shade, 0,		{ 2.566406, 1, 1.241211, 0 } },
  { "Canon", "EOS 450D", Cloudy, 0,		{ 2.386719, 1, 1.345703, 0 } },
  { "Canon", "EOS 450D", Tungsten, 0,		{ 1.559034, 1, 2.170841, 0 } },
  { "Canon", "EOS 450D", Fluorescent, 0,	{ 1.922857, 1, 1.996190, 0 } },
  { "Canon", "EOS 450D", Flash, 0,		{ 2.456055, 1, 1.318359, 0 } },

  { "Canon", "EOS 500D", Daylight, 0,		{ 2.023438, 1, 1.417969, 0 } },
  { "Canon", "EOS 500D", Shade, 0,		{ 2.291016, 1, 1.217773, 0 } },
  { "Canon", "EOS 500D", Cloudy, 0,		{ 2.156250, 1, 1.304687, 0 } },
  { "Canon", "EOS 500D", Tungsten, 0,		{ 1.481347, 1, 1.976342, 0 } },
  { "Canon", "EOS 500D", Fluorescent, 0,	{ 1.799224, 1, 1.824442, 0 } },
  { "Canon", "EOS 500D", Flash, 0,		{ 2.207031, 1, 1.295898, 0 } },

  /* Canon EOS 550D Firmware Version 1.0.9 */
  /* Fine-tuning is the camera's Amber-Blue bracketing. */
  { "Canon", "EOS 550D", Daylight, -9,		{ 1.903320, 1, 1.784180, 0 } },
  { "Canon", "EOS 550D", Daylight, -8,		{ 1.924805, 1, 1.756836, 0 } },
  { "Canon", "EOS 550D", Daylight, -7,		{ 1.950195, 1, 1.729492, 0 } },
  { "Canon", "EOS 550D", Daylight, -6,		{ 1.980469, 1, 1.701172, 0 } },
  { "Canon", "EOS 550D", Daylight, -5,		{ 2.007813, 1, 1.672852, 0 } },
  { "Canon", "EOS 550D", Daylight, -4,		{ 2.040039, 1, 1.646484, 0 } },
  { "Canon", "EOS 550D", Daylight, -3,		{ 2.064453, 1, 1.615234, 0 } },
  { "Canon", "EOS 550D", Daylight, -2,		{ 2.089844, 1, 1.587891, 0 } },
  { "Canon", "EOS 550D", Daylight, -1,		{ 2.120117, 1, 1.556641, 0 } },
  { "Canon", "EOS 550D", Daylight, 0,		{ 2.146484, 1, 1.526367, 0 } },
  { "Canon", "EOS 550D", Daylight, 1,		{ 2.178711, 1, 1.503906, 0 } },
  { "Canon", "EOS 550D", Daylight, 2,		{ 2.211914, 1, 1.481445, 0 } },
  { "Canon", "EOS 550D", Daylight, 3,		{ 2.246094, 1, 1.458984, 0 } },
  { "Canon", "EOS 550D", Daylight, 4,		{ 2.280273, 1, 1.436523, 0 } },
  { "Canon", "EOS 550D", Daylight, 5,		{ 2.316406, 1, 1.412109, 0 } },
  { "Canon", "EOS 550D", Daylight, 6,		{ 2.342773, 1, 1.391602, 0 } },
  { "Canon", "EOS 550D", Daylight, 7,		{ 2.375977, 1, 1.373047, 0 } },
  { "Canon", "EOS 550D", Daylight, 8,		{ 2.409180, 1, 1.350586, 0 } },
  { "Canon", "EOS 550D", Daylight, 9,		{ 2.444336, 1, 1.328125, 0 } },
  { "Canon", "EOS 550D", Shade, -9,		{ 2.173828, 1, 1.507813, 0 } },
  { "Canon", "EOS 550D", Shade, -8,		{ 2.207031, 1, 1.484375, 0 } },
  { "Canon", "EOS 550D", Shade, -7,		{ 2.240234, 1, 1.460938, 0 } },
  { "Canon", "EOS 550D", Shade, -6,		{ 2.275391, 1, 1.438477, 0 } },
  { "Canon", "EOS 550D", Shade, -5,		{ 2.311523, 1, 1.414063, 0 } },
  { "Canon", "EOS 550D", Shade, -4,		{ 2.342773, 1, 1.395508, 0 } },
  { "Canon", "EOS 550D", Shade, -3,		{ 2.370117, 1, 1.374023, 0 } },
  { "Canon", "EOS 550D", Shade, -2,		{ 2.403320, 1, 1.352539, 0 } },
  { "Canon", "EOS 550D", Shade, -1,		{ 2.444336, 1, 1.332031, 0 } },
  { "Canon", "EOS 550D", Shade, 0,		{ 2.479492, 1, 1.307617, 0 } },
  { "Canon", "EOS 550D", Shade, 1,		{ 2.509766, 1, 1.292969, 0 } },
  { "Canon", "EOS 550D", Shade, 2,		{ 2.541016, 1, 1.276367, 0 } },
  { "Canon", "EOS 550D", Shade, 3,		{ 2.573242, 1, 1.259766, 0 } },
  { "Canon", "EOS 550D", Shade, 4,		{ 2.612305, 1, 1.241211, 0 } },
  { "Canon", "EOS 550D", Shade, 5,		{ 2.645508, 1, 1.223633, 0 } },
  { "Canon", "EOS 550D", Shade, 6,		{ 2.673828, 1, 1.206055, 0 } },
  { "Canon", "EOS 550D", Shade, 7,		{ 2.702148, 1, 1.187500, 0 } },
  { "Canon", "EOS 550D", Shade, 8,		{ 2.738281, 1, 1.168945, 0 } },
  { "Canon", "EOS 550D", Shade, 9,		{ 2.782227, 1, 1.148438, 0 } },
  { "Canon", "EOS 550D", Cloudy, -9,		{ 2.043945, 1, 1.640625, 0 } },
  { "Canon", "EOS 550D", Cloudy, -8,		{ 2.068359, 1, 1.612305, 0 } },
  { "Canon", "EOS 550D", Cloudy, -7,		{ 2.093750, 1, 1.583008, 0 } },
  { "Canon", "EOS 550D", Cloudy, -6,		{ 2.120117, 1, 1.553711, 0 } },
  { "Canon", "EOS 550D", Cloudy, -5,		{ 2.151367, 1, 1.523438, 0 } },
  { "Canon", "EOS 550D", Cloudy, -4,		{ 2.183594, 1, 1.501953, 0 } },
  { "Canon", "EOS 550D", Cloudy, -3,		{ 2.216797, 1, 1.477539, 0 } },
  { "Canon", "EOS 550D", Cloudy, -2,		{ 2.250977, 1, 1.454102, 0 } },
  { "Canon", "EOS 550D", Cloudy, -1,		{ 2.280273, 1, 1.432617, 0 } },
  { "Canon", "EOS 550D", Cloudy, 0,		{ 2.316406, 1, 1.408203, 0 } },
  { "Canon", "EOS 550D", Cloudy, 1,		{ 2.348633, 1, 1.389648, 0 } },
  { "Canon", "EOS 550D", Cloudy, 2,		{ 2.381836, 1, 1.369141, 0 } },
  { "Canon", "EOS 550D", Cloudy, 3,		{ 2.415039, 1, 1.347656, 0 } },
  { "Canon", "EOS 550D", Cloudy, 4,		{ 2.450195, 1, 1.326172, 0 } },
  { "Canon", "EOS 550D", Cloudy, 5,		{ 2.491211, 1, 1.304688, 0 } },
  { "Canon", "EOS 550D", Cloudy, 6,		{ 2.515625, 1, 1.290039, 0 } },
  { "Canon", "EOS 550D", Cloudy, 7,		{ 2.546875, 1, 1.272461, 0 } },
  { "Canon", "EOS 550D", Cloudy, 8,		{ 2.579102, 1, 1.254883, 0 } },
  { "Canon", "EOS 550D", Cloudy, 9,		{ 2.619141, 1, 1.236328, 0 } },
  { "Canon", "EOS 550D", Tungsten, -9,		{ 1.345703, 1, 2.605469, 0 } },
  { "Canon", "EOS 550D", Tungsten, -8,		{ 1.361328, 1, 2.579102, 0 } },
  { "Canon", "EOS 550D", Tungsten, -7,		{ 1.379883, 1, 2.546875, 0 } },
  { "Canon", "EOS 550D", Tungsten, -6,		{ 1.398438, 1, 2.515625, 0 } },
  { "Canon", "EOS 550D", Tungsten, -5,		{ 1.417969, 1, 2.491211, 0 } },
  { "Canon", "EOS 550D", Tungsten, -4,		{ 1.440430, 1, 2.456055, 0 } },
  { "Canon", "EOS 550D", Tungsten, -3,		{ 1.460938, 1, 2.409180, 0 } },
  { "Canon", "EOS 550D", Tungsten, -2,		{ 1.479492, 1, 2.365234, 0 } },
  { "Canon", "EOS 550D", Tungsten, -1,		{ 1.503906, 1, 2.322266, 0 } },
  { "Canon", "EOS 550D", Tungsten, 0,		{ 1.526367, 1, 2.275391, 0 } },
  { "Canon", "EOS 550D", Tungsten, 1,		{ 1.546875, 1, 2.240234, 0 } },
  { "Canon", "EOS 550D", Tungsten, 2,		{ 1.568359, 1, 2.207031, 0 } },
  { "Canon", "EOS 550D", Tungsten, 3,		{ 1.589844, 1, 2.169922, 0 } },
  { "Canon", "EOS 550D", Tungsten, 4,		{ 1.612305, 1, 2.137695, 0 } },
  { "Canon", "EOS 550D", Tungsten, 5,		{ 1.638672, 1, 2.102539, 0 } },
  { "Canon", "EOS 550D", Tungsten, 6,		{ 1.662109, 1, 2.068359, 0 } },
  { "Canon", "EOS 550D", Tungsten, 7,		{ 1.684570, 1, 2.032227, 0 } },
  { "Canon", "EOS 550D", Tungsten, 8,		{ 1.707031, 1, 2.000000, 0 } },
  { "Canon", "EOS 550D", Tungsten, 9,		{ 1.732422, 1, 1.965820, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -9,	{ 1.662109, 1, 2.473633, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -8,	{ 1.684570, 1, 2.432617, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -7,	{ 1.709961, 1, 2.392578, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -6,	{ 1.735352, 1, 2.342773, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -5,	{ 1.762695, 1, 2.300781, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -4,	{ 1.787109, 1, 2.260742, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -3,	{ 1.812500, 1, 2.226563, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -2,	{ 1.841797, 1, 2.188477, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, -1,	{ 1.872070, 1, 2.156250, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 0,	{ 1.899414, 1, 2.115234, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 1,	{ 1.924805, 1, 2.081055, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 2,	{ 1.950195, 1, 2.051758, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 3,	{ 1.976563, 1, 2.015625, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 4,	{ 2.007813, 1, 1.984375, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 5,	{ 2.040039, 1, 1.950195, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 6,	{ 2.064453, 1, 1.920898, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 7,	{ 2.089844, 1, 1.889648, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 8,	{ 2.120117, 1, 1.862305, 0 } },
  { "Canon", "EOS 550D", WhiteFluorescent, 9,	{ 2.146484, 1, 1.832031, 0 } },
  { "Canon", "EOS 550D", Flash, -9,		{ 2.098633, 1, 1.625000, 0 } },
  { "Canon", "EOS 550D", Flash, -8,		{ 2.124023, 1, 1.594727, 0 } },
  { "Canon", "EOS 550D", Flash, -7,		{ 2.156250, 1, 1.565430, 0 } },
  { "Canon", "EOS 550D", Flash, -6,		{ 2.188477, 1, 1.535156, 0 } },
  { "Canon", "EOS 550D", Flash, -5,		{ 2.221680, 1, 1.510742, 0 } },
  { "Canon", "EOS 550D", Flash, -4,		{ 2.255859, 1, 1.488281, 0 } },
  { "Canon", "EOS 550D", Flash, -3,		{ 2.286133, 1, 1.464844, 0 } },
  { "Canon", "EOS 550D", Flash, -2,		{ 2.322266, 1, 1.442383, 0 } },
  { "Canon", "EOS 550D", Flash, -1,		{ 2.354492, 1, 1.419922, 0 } },
  { "Canon", "EOS 550D", Flash, 0,		{ 2.381836, 1, 1.397461, 0 } },
  { "Canon", "EOS 550D", Flash, 1,		{ 2.420898, 1, 1.377930, 0 } },
  { "Canon", "EOS 550D", Flash, 2,		{ 2.456055, 1, 1.356445, 0 } },
  { "Canon", "EOS 550D", Flash, 3,		{ 2.491211, 1, 1.334961, 0 } },
  { "Canon", "EOS 550D", Flash, 4,		{ 2.522461, 1, 1.312500, 0 } },
  { "Canon", "EOS 550D", Flash, 5,		{ 2.553711, 1, 1.295898, 0 } },
  { "Canon", "EOS 550D", Flash, 6,		{ 2.585938, 1, 1.280273, 0 } },
  { "Canon", "EOS 550D", Flash, 7,		{ 2.625977, 1, 1.262695, 0 } },
  { "Canon", "EOS 550D", Flash, 8,		{ 2.653320, 1, 1.244141, 0 } },
  { "Canon", "EOS 550D", Flash, 9,		{ 2.680664, 1, 1.226563, 0 } },

  // Canon EOS 600D firmware version 1.0.2
  { "Canon", "EOS 600D", Daylight, 0,		{ 2.235352, 1, 1.612305, 0 } },
  { "Canon", "EOS 600D", Shade, 0,		{ 2.592773, 1, 1.377930, 0 } },
  { "Canon", "EOS 600D", Cloudy, 0,		{ 2.409180, 1, 1.486328, 0 } },
  { "Canon", "EOS 600D", Tungsten, 0,		{ 1.597656, 1, 2.409180, 0 } },
  { "Canon", "EOS 600D", Fluorescent, 0,	{ 1.958008, 1, 2.260742, 0 } },
  { "Canon", "EOS 600D", Flash, 0,		{ 2.509766, 1, 1.464844, 0 } },

  /* Fine-tuning is the camera's Amber-Blue bracketing. */
  { "Canon", "EOS 650D", Daylight, -3,		{ 1.9502, 1, 1.7539, 0 } },
  { "Canon", "EOS 650D", Daylight, 0,		{ 2.0322, 1, 1.6572, 0 } },
  { "Canon", "EOS 650D", Daylight, 3,		{ 2.1152, 1, 1.5850, 0 } },
  { "Canon", "EOS 650D", Shade, -3,		{ 2.2305, 1, 1.4951, 0 } },
  { "Canon", "EOS 650D", Shade, 0,		{ 2.3379, 1, 1.4238, 0 } },
  { "Canon", "EOS 650D", Shade, 3,		{ 2.4443, 1, 1.3604, 0 } },
  { "Canon", "EOS 650D", Cloudy, -3,		{ 2.0898, 1, 1.6074, 0 } },
  { "Canon", "EOS 650D", Cloudy, 0,		{ 2.1787, 1, 1.5332, 0 } },
  { "Canon", "EOS 650D", Cloudy, 3,		{ 2.2705, 1, 1.4668, 0 } },
  { "Canon", "EOS 650D", Tungsten, -3,		{ 1.3975, 1, 2.5928, 0 } },
  { "Canon", "EOS 650D", Tungsten, 0,		{ 1.4541, 1, 2.4561, 0 } },
  { "Canon", "EOS 650D", Tungsten, 3,		{ 1.5127, 1, 2.3428, 0 } },
  { "Canon", "EOS 650D", Fluorescent, -3,	{ 1.6982, 1, 2.4268, 0 } },
  { "Canon", "EOS 650D", Fluorescent, 0,	{ 1.7715, 1, 2.3066, 0 } },
  { "Canon", "EOS 650D", Fluorescent, 3,	{ 1.8486, 1, 2.1973, 0 } },
  { "Canon", "EOS 650D", Flash, -3,		{ 2.1699, 1, 1.5879, 0 } },
  { "Canon", "EOS 650D", Flash, 0,		{ 2.2607, 1, 1.5166, 0 } },
  { "Canon", "EOS 650D", Flash, 3,		{ 2.3701, 1, 1.4502, 0 } },

  // Canon EOS 700D firmware Version 1.1.4
  { "Canon", "EOS 700D", Daylight, 0,		{ 2.08984375, 1, 1.5400390625, 0 } },
  { "Canon", "EOS 700D", Shade, 0,		{ 2.4033203125, 1, 1.302734375, 0 } },
  { "Canon", "EOS 700D", Cloudy, 0,		{ 2.240234375, 1, 1.412109375, 0 } },
  { "Canon", "EOS 700D", Tungsten, 0,		{ 1.501953125, 1, 2.365234375, 0 } },
  { "Canon", "EOS 700D", Fluorescent, 0,	{ 1.83203125, 1, 2.2509765625, 0 } },
  { "Canon", "EOS 700D", Flash, 0,		{ 2.3271484375, 1, 1.3984375, 0 } },

  { "Canon", "EOS 750D", Daylight            , -9, { 1.91796875, 1, 1.9619140625, 0 } },
  { "Canon", "EOS 750D", Daylight            , -8, { 1.939453125, 1, 1.931640625, 0 } },
  { "Canon", "EOS 750D", Daylight            , -7, { 1.96875, 1, 1.9033203125, 0 } },
  { "Canon", "EOS 750D", Daylight            , -6, { 1.99609375, 1, 1.8720703125, 0 } },
  { "Canon", "EOS 750D", Daylight            , -5, { 2.02734375, 1, 1.8388671875, 0 } },
  { "Canon", "EOS 750D", Daylight            , -4, { 2.060546875, 1, 1.8095703125, 0 } },
  { "Canon", "EOS 750D", Daylight            , -3, { 2.08984375, 1, 1.78125, 0 } },
  { "Canon", "EOS 750D", Daylight            , -2, { 2.1201171875, 1, 1.75390625, 0 } },
  { "Canon", "EOS 750D", Daylight            , -1, { 2.1513671875, 1, 1.7265625, 0 } },
  { "Canon", "EOS 750D", Daylight            , 0, { 2.18359375, 1, 1.6953125, 0 } },
  { "Canon", "EOS 750D", Daylight            , 1, { 2.216796875, 1, 1.6708984375, 0 } },
  { "Canon", "EOS 750D", Daylight            , 2, { 2.24609375, 1, 1.6435546875, 0 } },
  { "Canon", "EOS 750D", Daylight            , 3, { 2.2802734375, 1, 1.6181640625, 0 } },
  { "Canon", "EOS 750D", Daylight            , 4, { 2.31640625, 1, 1.5927734375, 0 } },
  { "Canon", "EOS 750D", Daylight            , 5, { 2.3544921875, 1, 1.568359375, 0 } },
  { "Canon", "EOS 750D", Daylight            , 6, { 2.3818359375, 1, 1.544921875, 0 } },
  { "Canon", "EOS 750D", Daylight            , 7, { 2.4150390625, 1, 1.521484375, 0 } },
  { "Canon", "EOS 750D", Daylight            , 8, { 2.4501953125, 1, 1.4990234375, 0 } },
  { "Canon", "EOS 750D", Daylight            , 9, { 2.4912109375, 1, 1.4736328125, 0 } },
  { "Canon", "EOS 750D", Shade               , -9, { 2.2119140625, 1, 1.6728515625, 0 } },
  { "Canon", "EOS 750D", Shade               , -8, { 2.24609375, 1, 1.646484375, 0 } },
  { "Canon", "EOS 750D", Shade               , -7, { 2.275390625, 1, 1.6201171875, 0 } },
  { "Canon", "EOS 750D", Shade               , -6, { 2.3115234375, 1, 1.5947265625, 0 } },
  { "Canon", "EOS 750D", Shade               , -5, { 2.3486328125, 1, 1.5703125, 0 } },
  { "Canon", "EOS 750D", Shade               , -4, { 2.3818359375, 1, 1.546875, 0 } },
  { "Canon", "EOS 750D", Shade               , -3, { 2.4150390625, 1, 1.5234375, 0 } },
  { "Canon", "EOS 750D", Shade               , -2, { 2.4501953125, 1, 1.501953125, 0 } },
  { "Canon", "EOS 750D", Shade               , -1, { 2.4853515625, 1, 1.4775390625, 0 } },
  { "Canon", "EOS 750D", Shade               , 0, { 2.5283203125, 1, 1.4501953125, 0 } },
  { "Canon", "EOS 750D", Shade               , 1, { 2.5595703125, 1, 1.4326171875, 0 } },
  { "Canon", "EOS 750D", Shade               , 2, { 2.5927734375, 1, 1.412109375, 0 } },
  { "Canon", "EOS 750D", Shade               , 3, { 2.6259765625, 1, 1.3916015625, 0 } },
  { "Canon", "EOS 750D", Shade               , 4, { 2.6669921875, 1, 1.369140625, 0 } },
  { "Canon", "EOS 750D", Shade               , 5, { 2.7021484375, 1, 1.349609375, 0 } },
  { "Canon", "EOS 750D", Shade               , 6, { 2.73046875, 1, 1.33203125, 0 } },
  { "Canon", "EOS 750D", Shade               , 7, { 2.775390625, 1, 1.3115234375, 0 } },
  { "Canon", "EOS 750D", Shade               , 8, { 2.8134765625, 1, 1.2900390625, 0 } },
  { "Canon", "EOS 750D", Shade               , 9, { 2.8525390625, 1, 1.2705078125, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -9, { 2.064453125, 1, 1.8056640625, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -8, { 2.09375, 1, 1.77734375, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -7, { 2.1201171875, 1, 1.75, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -6, { 2.15625, 1, 1.720703125, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -5, { 2.1884765625, 1, 1.6923828125, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -4, { 2.2216796875, 1, 1.66796875, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -3, { 2.2509765625, 1, 1.640625, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -2, { 2.2861328125, 1, 1.615234375, 0 } },
  { "Canon", "EOS 750D", Cloudy              , -1, { 2.31640625, 1, 1.58984375, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 0, { 2.3544921875, 1, 1.5634765625, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 1, { 2.38671875, 1, 1.5419921875, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 2, { 2.4208984375, 1, 1.51953125, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 3, { 2.4560546875, 1, 1.4951171875, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 4, { 2.498046875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 5, { 2.541015625, 1, 1.4462890625, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 6, { 2.56640625, 1, 1.427734375, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 7, { 2.5986328125, 1, 1.408203125, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 8, { 2.6328125, 1, 1.3857421875, 0 } },
  { "Canon", "EOS 750D", Cloudy              , 9, { 2.673828125, 1, 1.36328125, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -9, { 1.3349609375, 1, 2.8369140625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -8, { 1.3544921875, 1, 2.7978515625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -7, { 1.3740234375, 1, 2.767578125, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -6, { 1.3955078125, 1, 2.73046875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -5, { 1.416015625, 1, 2.6943359375, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -4, { 1.4404296875, 1, 2.66015625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -3, { 1.4609375, 1, 2.6123046875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -2, { 1.4794921875, 1, 2.56640625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , -1, { 1.50390625, 1, 2.5283203125, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 0, { 1.5263671875, 1, 2.4794921875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 1, { 1.548828125, 1, 2.4443359375, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 2, { 1.5703125, 1, 2.4091796875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 3, { 1.5947265625, 1, 2.3701171875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 4, { 1.6201171875, 1, 2.3330078125, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 5, { 1.6494140625, 1, 2.291015625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 6, { 1.6728515625, 1, 2.255859375, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 7, { 1.6953125, 1, 2.2216796875, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 8, { 1.720703125, 1, 2.1884765625, 0 } },
  { "Canon", "EOS 750D", Tungsten            , 9, { 1.7470703125, 1, 2.1513671875, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -9, { 1.6201171875, 1, 2.6806640625, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -8, { 1.646484375, 1, 2.6396484375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -7, { 1.6728515625, 1, 2.5927734375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -6, { 1.6953125, 1, 2.5537109375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -5, { 1.720703125, 1, 2.509765625, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -4, { 1.7470703125, 1, 2.4677734375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -3, { 1.7744140625, 1, 2.4326171875, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -2, { 1.7998046875, 1, 2.392578125, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, -1, { 1.8251953125, 1, 2.3544921875, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 0, { 1.8515625, 1, 2.3115234375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 1, { 1.8828125, 1, 2.275390625, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 2, { 1.9140625, 1, 2.240234375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 3, { 1.939453125, 1, 2.20703125, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 4, { 1.9658203125, 1, 2.173828125, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 5, { 1.99609375, 1, 2.1376953125, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 6, { 2.0234375, 1, 2.107421875, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 7, { 2.060546875, 1, 2.0771484375, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 8, { 2.0859375, 1, 2.0439453125, 0 } },
  { "Canon", "EOS 750D", CoolWhiteFluorescent, 9, { 2.115234375, 1, 2.01171875, 0 } },
  { "Canon", "EOS 750D", Flash               , -9, { 2.12890625, 1, 1.77734375, 0 } },
  { "Canon", "EOS 750D", Flash               , -8, { 2.1650390625, 1, 1.75, 0 } },
  { "Canon", "EOS 750D", Flash               , -7, { 2.197265625, 1, 1.7236328125, 0 } },
  { "Canon", "EOS 750D", Flash               , -6, { 2.2265625, 1, 1.6953125, 0 } },
  { "Canon", "EOS 750D", Flash               , -5, { 2.2607421875, 1, 1.66796875, 0 } },
  { "Canon", "EOS 750D", Flash               , -4, { 2.291015625, 1, 1.640625, 0 } },
  { "Canon", "EOS 750D", Flash               , -3, { 2.3271484375, 1, 1.615234375, 0 } },
  { "Canon", "EOS 750D", Flash               , -2, { 2.365234375, 1, 1.5927734375, 0 } },
  { "Canon", "EOS 750D", Flash               , -1, { 2.392578125, 1, 1.5654296875, 0 } },
  { "Canon", "EOS 750D", Flash               , 0, { 2.4267578125, 1, 1.5419921875, 0 } },
  { "Canon", "EOS 750D", Flash               , 1, { 2.4677734375, 1, 1.51953125, 0 } },
  { "Canon", "EOS 750D", Flash               , 2, { 2.509765625, 1, 1.4970703125, 0 } },
  { "Canon", "EOS 750D", Flash               , 3, { 2.546875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS 750D", Flash               , 4, { 2.5732421875, 1, 1.4482421875, 0 } },
  { "Canon", "EOS 750D", Flash               , 5, { 2.60546875, 1, 1.427734375, 0 } },
  { "Canon", "EOS 750D", Flash               , 6, { 2.6455078125, 1, 1.408203125, 0 } },
  { "Canon", "EOS 750D", Flash               , 7, { 2.6875, 1, 1.3876953125, 0 } },
  { "Canon", "EOS 750D", Flash               , 8, { 2.7158203125, 1, 1.365234375, 0 } },
  { "Canon", "EOS 750D", Flash               , 9, { 2.7529296875, 1, 1.345703125, 0 } },

  // Firmware version 1.0.0
  { "Canon", "EOS 760D", Daylight            , 0, { 2.169922, 1, 1.709961, 0 } },
  { "Canon", "EOS 760D", Shade               , 0, { 2.528320, 1, 1.466797, 0 } },
  { "Canon", "EOS 760D", Cloudy              , 0, { 2.342773, 1, 1.578125, 0 } },
  { "Canon", "EOS 760D", Tungsten            , 0, { 1.510742, 1, 2.479492, 0 } },
  { "Canon", "EOS 760D", CoolWhiteFluorescent, 0, { 1.825195, 1, 2.311523, 0 } },
  { "Canon", "EOS 760D", Flash               , 0, { 2.420898, 1, 1.556641, 0 } },

  { "Canon", "EOS 1000D", Daylight, 0,		{ 2.183594, 1, 1.526367, 0 } },
  { "Canon", "EOS 1000D", Shade, 0,		{ 2.553711, 1, 1.262695, 0 } },
  { "Canon", "EOS 1000D", Cloudy, 0,		{ 2.365234, 1, 1.375977, 0 } },
  { "Canon", "EOS 1000D", Tungsten, 0,		{ 1.470328, 1, 2.402126, 0 } },
  { "Canon", "EOS 1000D", Fluorescent, 0,	{ 1.889648, 1, 2.133789, 0 } },
  { "Canon", "EOS 1000D", Flash, 0,		{ 2.541830, 1, 1.769099, 0 } },

  { "Canon", "EOS 1100D", Daylight, 0,		{ 2.2559, 1, 1.4844, 0 } },
  { "Canon", "EOS 1100D", Shade, 0,		{ 2.6455, 1, 1.2725, 0 } },
  { "Canon", "EOS 1100D", Cloudy, 0,		{ 2.4443, 1, 1.3652, 0 } },
  { "Canon", "EOS 1100D", Tungsten, 0,		{ 1.5654, 1, 2.2402, 0 } },
  { "Canon", "EOS 1100D", WhiteFluorescent, 0,	{ 1.9541, 1, 2.0732, 0 } },
  { "Canon", "EOS 1100D", Flash, 0,		{ 2.5283, 1, 1.3584, 0 } },

  { "Canon", "EOS 1200D", Daylight            , 0, { 2.007812, 1, 1.599609, 0 } },
  { "Canon", "EOS 1200D", Shade               , 0, { 2.322266, 1, 1.361328, 0 } },
  { "Canon", "EOS 1200D", Cloudy              , 0, { 2.160156, 1, 1.471680, 0 } },
  { "Canon", "EOS 1200D", Tungsten            , 0, { 1.444336, 1, 2.392578, 0 } },
  { "Canon", "EOS 1200D", CoolWhiteFluorescent, 0, { 1.759766, 1, 2.275391, 0 } },
  { "Canon", "EOS 1200D", Flash               , 0, { 2.246094, 1, 1.452148, 0 } },

  { "Canon", "EOS 1300D", Daylight,		0, { 1.988281, 1, 1.530273, 0 } },
  { "Canon", "EOS 1300D", Shade,		0, { 2.295898, 1, 1.300781, 0 } },
  { "Canon", "EOS 1300D", Cloudy,		0, { 2.137695, 1, 1.406250, 0 } },
  { "Canon", "EOS 1300D", Tungsten,		0, { 1.432617, 1, 2.300781, 0 } },
  { "Canon", "EOS 1300D", WhiteFluorescent,	0, { 1.747070, 1, 2.188477, 0 } },
  { "Canon", "EOS 1300D", Flash,		0, { 2.221680, 1, 1.387695, 0 } },

  { "Canon", "EOS-1Ds", Daylight, 0,		{ 1.6133, 1, 1.2500, 0 } }, /* 5302K */
  { "Canon", "EOS-1Ds", Shade, 0,		{ 1.8477, 1, 1.0684, 0 } }, /* 6685K */
  { "Canon", "EOS-1Ds", Cloudy, 0,		{ 1.7285, 1, 1.1504, 0 } }, /* 5940K */
  { "Canon", "EOS-1Ds", Tungsten, 0,		{ 1.1270, 1, 1.9199, 0 } }, /* 3334K */
  { "Canon", "EOS-1Ds", Fluorescent, 0,		{ 1.2012, 1, 1.7168, 0 } }, /* 3643K */
  { "Canon", "EOS-1Ds", Flash, 0,		{ 1.7793, 1, 1.1445, 0 } }, /* 6184K */

  // Firmware Version 1.1.6
  { "Canon", "EOS-1Ds Mark II", Daylight, 0,	{ 1.992188, 1, 1.503906, 0 } },
  { "Canon", "EOS-1Ds Mark II", Shade, 0,	{ 2.300781, 1, 1.243164, 0 } },
  { "Canon", "EOS-1Ds Mark II", Cloudy, 0,	{ 2.146484, 1, 1.363281, 0 } },
  { "Canon", "EOS-1Ds Mark II", Tungsten, 0,	{ 1.395018, 1, 2.370107, 0 } },
  { "Canon", "EOS-1Ds Mark II", Fluorescent, 0,	{ 1.854792, 1, 2.067764, 0 } },
  { "Canon", "EOS-1Ds Mark II", Flash, 0,	{ 2.235352, 1, 1.297852, 0 } },

  { "Canon", "EOS-1D Mark II", Cloudy, 0,	{ 2.093750, 1, 1.166016, 0 } },
  { "Canon", "EOS-1D Mark II", Daylight, 0,	{ 1.957031, 1, 1.295898, 0 } },
  { "Canon", "EOS-1D Mark II", Flash, 0,	{ 2.225586, 1, 1.172852, 0 } },
  { "Canon", "EOS-1D Mark II", Fluorescent, 0,	{ 1.785853, 1, 1.785853, 0 } },
  { "Canon", "EOS-1D Mark II", Shade, 0,	{ 2.220703, 1, 1.069336, 0 } },
  { "Canon", "EOS-1D Mark II", Tungsten, 0,	{ 1.415480, 1, 2.160142, 0 } },

  { "Canon", "EOS-1D Mark III", Daylight, 0,  { 1.98046875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS-1D Mark III", Shade, 0,     { 2.291015625, 1, 1.205078125, 0 } },
  { "Canon", "EOS-1D Mark III", Cloudy, 0,    { 2.142578125, 1, 1.3251953125, 0 } },
  { "Canon", "EOS-1D Mark III", Tungsten, 0,  { 1.3634703196347, 1, 2.38082191780822, 0 } },
  { "Canon", "EOS-1D Mark III", Fluorescent, 0,{ 1.74261201143947, 1, 2.22211630123928, 0 } },
  { "Canon", "EOS-1D Mark III", Flash, 0,     { 2.173828125, 1, 1.2880859375, 0 } },

  { "Canon", "EOS-1Ds Mark III", Daylight, 0,    { 2.09375, 1, 1.521484375, 0 } },
  { "Canon", "EOS-1Ds Mark III", Shade, 0,       { 2.4208984375, 1, 1.23828125, 0 } },
  { "Canon", "EOS-1Ds Mark III", Cloudy, 0,      { 2.2509765625, 1, 1.361328125, 0 } },
  { "Canon", "EOS-1Ds Mark III", Tungsten, 0,    { 1.45538178472861, 1, 2.40478380864765, 0 } },
  { "Canon", "EOS-1Ds Mark III", Fluorescent, 0, { 1.84271844660194, 1, 2.33009708737864, 0 } },
  { "Canon", "EOS-1Ds Mark III", Flash, 0,       { 2.3115234375, 1, 1.31640625, 0 } },

  { "Canon", "EOS-1D Mark II N", Cloudy, 0,	{ 2.183594, 1, 1.220703, 0 } },
  { "Canon", "EOS-1D Mark II N", Daylight, 0,	{ 2.019531, 1, 1.349609, 0 } },
  { "Canon", "EOS-1D Mark II N", Flash, 0,	{ 2.291016, 1, 1.149414, 0 } },
  { "Canon", "EOS-1D Mark II N", Fluorescent, 0, { 1.802899, 1, 1.990338, 0 } },
  { "Canon", "EOS-1D Mark II N", Shade, 0,	{ 2.337891, 1, 1.112305, 0 } },
  { "Canon", "EOS-1D Mark II N", Tungsten, 0,	{ 1.408514, 1, 2.147645, 0 } },

  // Firmware Version 1.0.6 (Temporary)
  { "Canon", "EOS-1D Mark IV", Daylight, 0,	{ 2.040039, 1, 1.558594, 0 } },
  { "Canon", "EOS-1D Mark IV", Shade, 0,	{ 2.342773, 1, 1.333008, 0 } },
  { "Canon", "EOS-1D Mark IV", Cloudy, 0,	{ 2.188477, 1, 1.440430, 0 } },
  { "Canon", "EOS-1D Mark IV", Tungsten, 0,	{ 1.458333, 1, 2.305254, 0 } },
  { "Canon", "EOS-1D Mark IV", Fluorescent, 0,	{ 1.767892, 1, 2.205029, 0 } },
  { "Canon", "EOS-1D Mark IV", Flash, 0,	{ 2.230469, 1, 1.423828, 0 } },

  /* Canon EOS-1D X Firmware Version 1.0.6 */
  { "Canon", "EOS-1D X", Daylight, 0,         { 1.943359375, 1, 1.3740234375, 0 } },
  { "Canon", "EOS-1D X", Shade, 0,            { 2.23046875, 1, 1.17578125, 0 } },
  { "Canon", "EOS-1D X", Cloudy, 0,           { 2.0859375, 1, 1.2685546875, 0 } },
  { "Canon", "EOS-1D X", Tungsten, 0,         { 1.3857421875, 1, 2.068359375, 0 } },
  { "Canon", "EOS-1D X", Fluorescent, 0,      { 1.7265625, 1, 1.9501953125, 0 } },
  { "Canon", "EOS-1D X", Flash, 0,            { 2.173828125, 1, 1.2392578125, 0 } },

  { "Canon", "EOS-1D X Mark II", Daylight, 0,		{ 1.822265625, 1, 1.5849609375, 0 } },
  { "Canon", "EOS-1D X Mark II", Shade, 0,		{ 2.0810546875, 1, 1.3818359375, 0 } },
  { "Canon", "EOS-1D X Mark II", Cloudy, 0,		{ 1.9541015625, 1, 1.4814453125, 0 } },
  { "Canon", "EOS-1D X Mark II", Tungsten, 0,		{ 1.30078125, 1, 2.275390625, 0 } },
  { "Canon", "EOS-1D X Mark II", WhiteFluorescent, 0,	{ 1.5400390625, 1, 2.169921875, 0 } },
  { "Canon", "EOS-1D X Mark II", Flash, 0,		{ 2.01171875, 1, 1.4541015625, 0 } },

  // Canon EOS M Firmware 1.0.6
  { "Canon", "EOS M", Cloudy, 0,		{ 2.156250, 1, 1.448242, 0 } },
  { "Canon", "EOS M", Daylight, 0,		{ 2.007813, 1, 1.575195, 0 } },
  { "Canon", "EOS M", Flash, 0,			{ 2.230469, 1, 1.429688, 0 } },
  { "Canon", "EOS M", Tungsten, 0,		{ 1.448242, 1, 2.386719, 0 } },
  { "Canon", "EOS M", Shade, 0,			{ 2.300781, 1, 1.334961, 0 } },
  { "Canon", "EOS M", Fluorescent, 0,		{ 1.784180, 1, 2.275391, 0 } },

  { "Canon", "EOS M2", Cloudy, 0,             { 2.0859375, 1, 1.4931640625, 0 } },
  { "Canon", "EOS M2", Daylight, 0,           { 1.943359375, 1, 1.615234375, 0 } },
  { "Canon", "EOS M2", Flash, 0,              { 2.1650390625, 1, 1.4755859375, 0 } },
  { "Canon", "EOS M2", Tungsten, 0,           { 1.3916015625, 1, 2.392578125, 0 } },
  { "Canon", "EOS M2", Shade, 0,              { 2.2265625, 1, 1.3818359375, 0 } },
  { "Canon", "EOS M2", Fluorescent, 0,        { 1.7353515625, 1, 2.2802734375, 0 } },

  { "Canon", "EOS M3", Cloudy, 0,{ 2.31603229527105, 1, 1.58592848904268, 0 } },
  { "Canon", "EOS M3", Daylight, 0,{ 2.14041095890411, 1, 1.70205479452055, 0 } },
  { "Canon", "EOS M3", Flash, 0, { 2.7875150060024, 1, 1.3577430972389, 0 } },
  { "Canon", "EOS M3", Tungsten, 0,{ 1.48864994026284, 1, 2.62485065710872, 0 } },
  { "Canon", "EOS M3", Shade, 0, { 2.47785547785548, 1, 1.48834498834499, 0 } },
  { "Canon", "EOS M3", Fluorescent, 0,{ 1.87046004842615, 1, 2.32566585956416, 0 } },

  { "Canon", "EOS M5", Daylight            , -9, { 1.3603515625, 1, 1.5302734375, 0 } },
  { "Canon", "EOS M5", Daylight            , -8, { 1.3798828125, 1, 1.5078125, 0 } },
  { "Canon", "EOS M5", Daylight            , -7, { 1.3984375, 1, 1.4814453125, 0 } },
  { "Canon", "EOS M5", Daylight            , -6, { 1.419921875, 1, 1.4580078125, 0 } },
  { "Canon", "EOS M5", Daylight            , -5, { 1.4404296875, 1, 1.4365234375, 0 } },
  { "Canon", "EOS M5", Daylight            , -4, { 1.462890625, 1, 1.412109375, 0 } },
  { "Canon", "EOS M5", Daylight            , -3, { 1.4853515625, 1, 1.3935546875, 0 } },
  { "Canon", "EOS M5", Daylight            , -2, { 1.509765625, 1, 1.3720703125, 0 } },
  { "Canon", "EOS M5", Daylight            , -1, { 1.5322265625, 1, 1.3486328125, 0 } },
  { "Canon", "EOS M5", Daylight            , 0, { 1.5556640625, 1, 1.326171875, 0 } },
  { "Canon", "EOS M5", Daylight            , 1, { 1.5791015625, 1, 1.3095703125, 0 } },
  { "Canon", "EOS M5", Daylight            , 2, { 1.59765625, 1, 1.2890625, 0 } },
  { "Canon", "EOS M5", Daylight            , 3, { 1.6201171875, 1, 1.267578125, 0 } },
  { "Canon", "EOS M5", Daylight            , 4, { 1.6376953125, 1, 1.2490234375, 0 } },
  { "Canon", "EOS M5", Daylight            , 5, { 1.6669921875, 1, 1.232421875, 0 } },
  { "Canon", "EOS M5", Daylight            , 6, { 1.69140625, 1, 1.216796875, 0 } },
  { "Canon", "EOS M5", Daylight            , 7, { 1.7138671875, 1, 1.1982421875, 0 } },
  { "Canon", "EOS M5", Daylight            , 8, { 1.7470703125, 1, 1.181640625, 0 } },
  { "Canon", "EOS M5", Daylight            , 9, { 1.7734375, 1, 1.1640625, 0 } },
  { "Canon", "EOS M5", Shade               , -9, { 1.5751953125, 1, 1.314453125, 0 } },
  { "Canon", "EOS M5", Shade               , -8, { 1.5966796875, 1, 1.2919921875, 0 } },
  { "Canon", "EOS M5", Shade               , -7, { 1.619140625, 1, 1.2734375, 0 } },
  { "Canon", "EOS M5", Shade               , -6, { 1.6376953125, 1, 1.25390625, 0 } },
  { "Canon", "EOS M5", Shade               , -5, { 1.666015625, 1, 1.2373046875, 0 } },
  { "Canon", "EOS M5", Shade               , -4, { 1.6865234375, 1, 1.2197265625, 0 } },
  { "Canon", "EOS M5", Shade               , -3, { 1.7158203125, 1, 1.203125, 0 } },
  { "Canon", "EOS M5", Shade               , -2, { 1.7412109375, 1, 1.1845703125, 0 } },
  { "Canon", "EOS M5", Shade               , -1, { 1.775390625, 1, 1.166015625, 0 } },
  { "Canon", "EOS M5", Shade               , 0, { 1.7919921875, 1, 1.1484375, 0 } },
  { "Canon", "EOS M5", Shade               , 1, { 1.8173828125, 1, 1.1337890625, 0 } },
  { "Canon", "EOS M5", Shade               , 2, { 1.8447265625, 1, 1.119140625, 0 } },
  { "Canon", "EOS M5", Shade               , 3, { 1.869140625, 1, 1.099609375, 0 } },
  { "Canon", "EOS M5", Shade               , 4, { 1.8916015625, 1, 1.0810546875, 0 } },
  { "Canon", "EOS M5", Shade               , 5, { 1.9140625, 1, 1.06640625, 0 } },
  { "Canon", "EOS M5", Shade               , 6, { 1.9326171875, 1, 1.052734375, 0 } },
  { "Canon", "EOS M5", Shade               , 7, { 1.9599609375, 1, 1.0380859375, 0 } },
  { "Canon", "EOS M5", Shade               , 8, { 1.990234375, 1, 1.0205078125, 0 } },
  { "Canon", "EOS M5", Shade               , 9, { 2.0234375, 1, 1.0048828125, 0 } },
  { "Canon", "EOS M5", Cloudy              , -9, { 1.46875, 1, 1.4091796875, 0 } },
  { "Canon", "EOS M5", Cloudy              , -8, { 1.4912109375, 1, 1.3896484375, 0 } },
  { "Canon", "EOS M5", Cloudy              , -7, { 1.5146484375, 1, 1.369140625, 0 } },
  { "Canon", "EOS M5", Cloudy              , -6, { 1.5419921875, 1, 1.3486328125, 0 } },
  { "Canon", "EOS M5", Cloudy              , -5, { 1.5625, 1, 1.3271484375, 0 } },
  { "Canon", "EOS M5", Cloudy              , -4, { 1.5849609375, 1, 1.306640625, 0 } },
  { "Canon", "EOS M5", Cloudy              , -3, { 1.60546875, 1, 1.2861328125, 0 } },
  { "Canon", "EOS M5", Cloudy              , -2, { 1.623046875, 1, 1.2666015625, 0 } },
  { "Canon", "EOS M5", Cloudy              , -1, { 1.646484375, 1, 1.248046875, 0 } },
  { "Canon", "EOS M5", Cloudy              , 0, { 1.6708984375, 1, 1.2294921875, 0 } },
  { "Canon", "EOS M5", Cloudy              , 1, { 1.7001953125, 1, 1.21484375, 0 } },
  { "Canon", "EOS M5", Cloudy              , 2, { 1.72265625, 1, 1.197265625, 0 } },
  { "Canon", "EOS M5", Cloudy              , 3, { 1.755859375, 1, 1.1796875, 0 } },
  { "Canon", "EOS M5", Cloudy              , 4, { 1.783203125, 1, 1.1630859375, 0 } },
  { "Canon", "EOS M5", Cloudy              , 5, { 1.8037109375, 1, 1.14453125, 0 } },
  { "Canon", "EOS M5", Cloudy              , 6, { 1.8291015625, 1, 1.12890625, 0 } },
  { "Canon", "EOS M5", Cloudy              , 7, { 1.8525390625, 1, 1.1123046875, 0 } },
  { "Canon", "EOS M5", Cloudy              , 8, { 1.8818359375, 1, 1.095703125, 0 } },
  { "Canon", "EOS M5", Cloudy              , 9, { 1.90234375, 1, 1.0751953125, 0 } },
  { "Canon", "EOS M5", Tungsten            , -9, { 0.9365234375, 1, 2.16796875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -8, { 0.9521484375, 1, 2.1435546875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -7, { 0.9638671875, 1, 2.123046875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -6, { 0.978515625, 1, 2.09375, 0 } },
  { "Canon", "EOS M5", Tungsten            , -5, { 0.9912109375, 1, 2.060546875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -4, { 1.0087890625, 1, 2.0341796875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -3, { 1.0224609375, 1, 2.0107421875, 0 } },
  { "Canon", "EOS M5", Tungsten            , -2, { 1.0380859375, 1, 1.9833984375, 0 } },
  { "Canon", "EOS M5", Tungsten            , -1, { 1.0517578125, 1, 1.955078125, 0 } },
  { "Canon", "EOS M5", Tungsten            , 0, { 1.06640625, 1, 1.9287109375, 0 } },
  { "Canon", "EOS M5", Tungsten            , 1, { 1.0810546875, 1, 1.90234375, 0 } },
  { "Canon", "EOS M5", Tungsten            , 2, { 1.095703125, 1, 1.8818359375, 0 } },
  { "Canon", "EOS M5", Tungsten            , 3, { 1.1162109375, 1, 1.85546875, 0 } },
  { "Canon", "EOS M5", Tungsten            , 4, { 1.1318359375, 1, 1.82421875, 0 } },
  { "Canon", "EOS M5", Tungsten            , 5, { 1.15234375, 1, 1.7998046875, 0 } },
  { "Canon", "EOS M5", Tungsten            , 6, { 1.169921875, 1, 1.7744140625, 0 } },
  { "Canon", "EOS M5", Tungsten            , 7, { 1.1875, 1, 1.748046875, 0 } },
  { "Canon", "EOS M5", Tungsten            , 8, { 1.2041015625, 1, 1.724609375, 0 } },
  { "Canon", "EOS M5", Tungsten            , 9, { 1.22265625, 1, 1.6962890625, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -9, { 1.1806640625, 1, 1.994140625, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -8, { 1.1943359375, 1, 1.96875, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -7, { 1.2138671875, 1, 1.943359375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -6, { 1.2314453125, 1, 1.9189453125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -5, { 1.2509765625, 1, 1.8935546875, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -4, { 1.26953125, 1, 1.8671875, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -3, { 1.2822265625, 1, 1.83984375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -2, { 1.3017578125, 1, 1.814453125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, -1, { 1.3212890625, 1, 1.787109375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 0, { 1.3427734375, 1, 1.7578125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 1, { 1.3564453125, 1, 1.734375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 2, { 1.3740234375, 1, 1.70703125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 3, { 1.39453125, 1, 1.68359375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 4, { 1.4140625, 1, 1.6611328125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 5, { 1.439453125, 1, 1.6376953125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 6, { 1.4609375, 1, 1.619140625, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 7, { 1.4833984375, 1, 1.6005859375, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 8, { 1.5078125, 1, 1.578125, 0 } },
  { "Canon", "EOS M5", CoolWhiteFluorescent, 9, { 1.533203125, 1, 1.556640625, 0 } },
  { "Canon", "EOS M5", WhiteFluorescent    , 0, { 1.7919921875, 1, 1.1484375, 0 } },
  { "Canon", "EOS M5", Flash               , -9, { 1.779296875, 1, 1.1865234375, 0 } },
  { "Canon", "EOS M5", Flash               , -8, { 1.8017578125, 1, 1.16796875, 0 } },
  { "Canon", "EOS M5", Flash               , -7, { 1.82421875, 1, 1.150390625, 0 } },
  { "Canon", "EOS M5", Flash               , -6, { 1.853515625, 1, 1.1357421875, 0 } },
  { "Canon", "EOS M5", Flash               , -5, { 1.87890625, 1, 1.119140625, 0 } },
  { "Canon", "EOS M5", Flash               , -4, { 1.8994140625, 1, 1.103515625, 0 } },
  { "Canon", "EOS M5", Flash               , -3, { 1.921875, 1, 1.0859375, 0 } },
  { "Canon", "EOS M5", Flash               , -2, { 1.939453125, 1, 1.06640625, 0 } },
  { "Canon", "EOS M5", Flash               , -1, { 1.9736328125, 1, 1.0537109375, 0 } },
  { "Canon", "EOS M5", Flash               , 0, { 1.99609375, 1, 1.0361328125, 0 } },
  { "Canon", "EOS M5", Flash               , 1, { 2.0302734375, 1, 1.0224609375, 0 } },
  { "Canon", "EOS M5", Flash               , 2, { 2.0703125, 1, 1.005859375, 0 } },
  { "Canon", "EOS M5", Flash               , 3, { 2.115234375, 1, 0.994140625, 0 } },
  { "Canon", "EOS M5", Flash               , 4, { 2.1748046875, 1, 0.9794921875, 0 } },
  { "Canon", "EOS M5", Flash               , 5, { 2.2373046875, 1, 0.962890625, 0 } },
  { "Canon", "EOS M5", Flash               , 6, { 2.3193359375, 1, 0.9443359375, 0 } },
  { "Canon", "EOS M5", Flash               , 7, { 2.4189453125, 1, 0.92578125, 0 } },
  { "Canon", "EOS M5", Flash               , 8, { 2.552734375, 1, 0.90625, 0 } },
  { "Canon", "EOS M5", Flash               , 9, { 2.7265625, 1, 0.8828125, 0 } },

  /* copied from M5 */
  { "Canon", "EOS M6", Daylight            , -9, { 1.3603515625, 1, 1.5302734375, 0 } },
  { "Canon", "EOS M6", Daylight            , -8, { 1.3798828125, 1, 1.5078125, 0 } },
  { "Canon", "EOS M6", Daylight            , -7, { 1.3984375, 1, 1.4814453125, 0 } },
  { "Canon", "EOS M6", Daylight            , -6, { 1.419921875, 1, 1.4580078125, 0 } },
  { "Canon", "EOS M6", Daylight            , -5, { 1.4404296875, 1, 1.4365234375, 0 } },
  { "Canon", "EOS M6", Daylight            , -4, { 1.462890625, 1, 1.412109375, 0 } },
  { "Canon", "EOS M6", Daylight            , -3, { 1.4853515625, 1, 1.3935546875, 0 } },
  { "Canon", "EOS M6", Daylight            , -2, { 1.509765625, 1, 1.3720703125, 0 } },
  { "Canon", "EOS M6", Daylight            , -1, { 1.5322265625, 1, 1.3486328125, 0 } },
  { "Canon", "EOS M6", Daylight            , 0, { 1.5556640625, 1, 1.326171875, 0 } },
  { "Canon", "EOS M6", Daylight            , 1, { 1.5791015625, 1, 1.3095703125, 0 } },
  { "Canon", "EOS M6", Daylight            , 2, { 1.59765625, 1, 1.2890625, 0 } },
  { "Canon", "EOS M6", Daylight            , 3, { 1.6201171875, 1, 1.267578125, 0 } },
  { "Canon", "EOS M6", Daylight            , 4, { 1.6376953125, 1, 1.2490234375, 0 } },
  { "Canon", "EOS M6", Daylight            , 5, { 1.6669921875, 1, 1.232421875, 0 } },
  { "Canon", "EOS M6", Daylight            , 6, { 1.69140625, 1, 1.216796875, 0 } },
  { "Canon", "EOS M6", Daylight            , 7, { 1.7138671875, 1, 1.1982421875, 0 } },
  { "Canon", "EOS M6", Daylight            , 8, { 1.7470703125, 1, 1.181640625, 0 } },
  { "Canon", "EOS M6", Daylight            , 9, { 1.7734375, 1, 1.1640625, 0 } },
  { "Canon", "EOS M6", Shade               , -9, { 1.5751953125, 1, 1.314453125, 0 } },
  { "Canon", "EOS M6", Shade               , -8, { 1.5966796875, 1, 1.2919921875, 0 } },
  { "Canon", "EOS M6", Shade               , -7, { 1.619140625, 1, 1.2734375, 0 } },
  { "Canon", "EOS M6", Shade               , -6, { 1.6376953125, 1, 1.25390625, 0 } },
  { "Canon", "EOS M6", Shade               , -5, { 1.666015625, 1, 1.2373046875, 0 } },
  { "Canon", "EOS M6", Shade               , -4, { 1.6865234375, 1, 1.2197265625, 0 } },
  { "Canon", "EOS M6", Shade               , -3, { 1.7158203125, 1, 1.203125, 0 } },
  { "Canon", "EOS M6", Shade               , -2, { 1.7412109375, 1, 1.1845703125, 0 } },
  { "Canon", "EOS M6", Shade               , -1, { 1.775390625, 1, 1.166015625, 0 } },
  { "Canon", "EOS M6", Shade               , 0, { 1.7919921875, 1, 1.1484375, 0 } },
  { "Canon", "EOS M6", Shade               , 1, { 1.8173828125, 1, 1.1337890625, 0 } },
  { "Canon", "EOS M6", Shade               , 2, { 1.8447265625, 1, 1.119140625, 0 } },
  { "Canon", "EOS M6", Shade               , 3, { 1.869140625, 1, 1.099609375, 0 } },
  { "Canon", "EOS M6", Shade               , 4, { 1.8916015625, 1, 1.0810546875, 0 } },
  { "Canon", "EOS M6", Shade               , 5, { 1.9140625, 1, 1.06640625, 0 } },
  { "Canon", "EOS M6", Shade               , 6, { 1.9326171875, 1, 1.052734375, 0 } },
  { "Canon", "EOS M6", Shade               , 7, { 1.9599609375, 1, 1.0380859375, 0 } },
  { "Canon", "EOS M6", Shade               , 8, { 1.990234375, 1, 1.0205078125, 0 } },
  { "Canon", "EOS M6", Shade               , 9, { 2.0234375, 1, 1.0048828125, 0 } },
  { "Canon", "EOS M6", Cloudy              , -9, { 1.46875, 1, 1.4091796875, 0 } },
  { "Canon", "EOS M6", Cloudy              , -8, { 1.4912109375, 1, 1.3896484375, 0 } },
  { "Canon", "EOS M6", Cloudy              , -7, { 1.5146484375, 1, 1.369140625, 0 } },
  { "Canon", "EOS M6", Cloudy              , -6, { 1.5419921875, 1, 1.3486328125, 0 } },
  { "Canon", "EOS M6", Cloudy              , -5, { 1.5625, 1, 1.3271484375, 0 } },
  { "Canon", "EOS M6", Cloudy              , -4, { 1.5849609375, 1, 1.306640625, 0 } },
  { "Canon", "EOS M6", Cloudy              , -3, { 1.60546875, 1, 1.2861328125, 0 } },
  { "Canon", "EOS M6", Cloudy              , -2, { 1.623046875, 1, 1.2666015625, 0 } },
  { "Canon", "EOS M6", Cloudy              , -1, { 1.646484375, 1, 1.248046875, 0 } },
  { "Canon", "EOS M6", Cloudy              , 0, { 1.6708984375, 1, 1.2294921875, 0 } },
  { "Canon", "EOS M6", Cloudy              , 1, { 1.7001953125, 1, 1.21484375, 0 } },
  { "Canon", "EOS M6", Cloudy              , 2, { 1.72265625, 1, 1.197265625, 0 } },
  { "Canon", "EOS M6", Cloudy              , 3, { 1.755859375, 1, 1.1796875, 0 } },
  { "Canon", "EOS M6", Cloudy              , 4, { 1.783203125, 1, 1.1630859375, 0 } },
  { "Canon", "EOS M6", Cloudy              , 5, { 1.8037109375, 1, 1.14453125, 0 } },
  { "Canon", "EOS M6", Cloudy              , 6, { 1.8291015625, 1, 1.12890625, 0 } },
  { "Canon", "EOS M6", Cloudy              , 7, { 1.8525390625, 1, 1.1123046875, 0 } },
  { "Canon", "EOS M6", Cloudy              , 8, { 1.8818359375, 1, 1.095703125, 0 } },
  { "Canon", "EOS M6", Cloudy              , 9, { 1.90234375, 1, 1.0751953125, 0 } },
  { "Canon", "EOS M6", Tungsten            , -9, { 0.9365234375, 1, 2.16796875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -8, { 0.9521484375, 1, 2.1435546875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -7, { 0.9638671875, 1, 2.123046875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -6, { 0.978515625, 1, 2.09375, 0 } },
  { "Canon", "EOS M6", Tungsten            , -5, { 0.9912109375, 1, 2.060546875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -4, { 1.0087890625, 1, 2.0341796875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -3, { 1.0224609375, 1, 2.0107421875, 0 } },
  { "Canon", "EOS M6", Tungsten            , -2, { 1.0380859375, 1, 1.9833984375, 0 } },
  { "Canon", "EOS M6", Tungsten            , -1, { 1.0517578125, 1, 1.955078125, 0 } },
  { "Canon", "EOS M6", Tungsten            , 0, { 1.06640625, 1, 1.9287109375, 0 } },
  { "Canon", "EOS M6", Tungsten            , 1, { 1.0810546875, 1, 1.90234375, 0 } },
  { "Canon", "EOS M6", Tungsten            , 2, { 1.095703125, 1, 1.8818359375, 0 } },
  { "Canon", "EOS M6", Tungsten            , 3, { 1.1162109375, 1, 1.85546875, 0 } },
  { "Canon", "EOS M6", Tungsten            , 4, { 1.1318359375, 1, 1.82421875, 0 } },
  { "Canon", "EOS M6", Tungsten            , 5, { 1.15234375, 1, 1.7998046875, 0 } },
  { "Canon", "EOS M6", Tungsten            , 6, { 1.169921875, 1, 1.7744140625, 0 } },
  { "Canon", "EOS M6", Tungsten            , 7, { 1.1875, 1, 1.748046875, 0 } },
  { "Canon", "EOS M6", Tungsten            , 8, { 1.2041015625, 1, 1.724609375, 0 } },
  { "Canon", "EOS M6", Tungsten            , 9, { 1.22265625, 1, 1.6962890625, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -9, { 1.1806640625, 1, 1.994140625, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -8, { 1.1943359375, 1, 1.96875, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -7, { 1.2138671875, 1, 1.943359375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -6, { 1.2314453125, 1, 1.9189453125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -5, { 1.2509765625, 1, 1.8935546875, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -4, { 1.26953125, 1, 1.8671875, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -3, { 1.2822265625, 1, 1.83984375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -2, { 1.3017578125, 1, 1.814453125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, -1, { 1.3212890625, 1, 1.787109375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 0, { 1.3427734375, 1, 1.7578125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 1, { 1.3564453125, 1, 1.734375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 2, { 1.3740234375, 1, 1.70703125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 3, { 1.39453125, 1, 1.68359375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 4, { 1.4140625, 1, 1.6611328125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 5, { 1.439453125, 1, 1.6376953125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 6, { 1.4609375, 1, 1.619140625, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 7, { 1.4833984375, 1, 1.6005859375, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 8, { 1.5078125, 1, 1.578125, 0 } },
  { "Canon", "EOS M6", CoolWhiteFluorescent, 9, { 1.533203125, 1, 1.556640625, 0 } },
  { "Canon", "EOS M6", WhiteFluorescent    , 0, { 1.7919921875, 1, 1.1484375, 0 } },
  { "Canon", "EOS M6", Flash               , -9, { 1.779296875, 1, 1.1865234375, 0 } },
  { "Canon", "EOS M6", Flash               , -8, { 1.8017578125, 1, 1.16796875, 0 } },
  { "Canon", "EOS M6", Flash               , -7, { 1.82421875, 1, 1.150390625, 0 } },
  { "Canon", "EOS M6", Flash               , -6, { 1.853515625, 1, 1.1357421875, 0 } },
  { "Canon", "EOS M6", Flash               , -5, { 1.87890625, 1, 1.119140625, 0 } },
  { "Canon", "EOS M6", Flash               , -4, { 1.8994140625, 1, 1.103515625, 0 } },
  { "Canon", "EOS M6", Flash               , -3, { 1.921875, 1, 1.0859375, 0 } },
  { "Canon", "EOS M6", Flash               , -2, { 1.939453125, 1, 1.06640625, 0 } },
  { "Canon", "EOS M6", Flash               , -1, { 1.9736328125, 1, 1.0537109375, 0 } },
  { "Canon", "EOS M6", Flash               , 0, { 1.99609375, 1, 1.0361328125, 0 } },
  { "Canon", "EOS M6", Flash               , 1, { 2.0302734375, 1, 1.0224609375, 0 } },
  { "Canon", "EOS M6", Flash               , 2, { 2.0703125, 1, 1.005859375, 0 } },
  { "Canon", "EOS M6", Flash               , 3, { 2.115234375, 1, 0.994140625, 0 } },
  { "Canon", "EOS M6", Flash               , 4, { 2.1748046875, 1, 0.9794921875, 0 } },
  { "Canon", "EOS M6", Flash               , 5, { 2.2373046875, 1, 0.962890625, 0 } },
  { "Canon", "EOS M6", Flash               , 6, { 2.3193359375, 1, 0.9443359375, 0 } },
  { "Canon", "EOS M6", Flash               , 7, { 2.4189453125, 1, 0.92578125, 0 } },
  { "Canon", "EOS M6", Flash               , 8, { 2.552734375, 1, 0.90625, 0 } },
  { "Canon", "EOS M6", Flash               , 9, { 2.7265625, 1, 0.8828125, 0 } },

  { "Canon", "EOS M10", Cloudy          , 0, { 2.171053, 1, 1.438596, 0 } },
  { "Canon", "EOS M10", Daylight        , 0, { 1.992408, 1, 1.556399, 0 } },
  { "Canon", "EOS M10", Flash           , 0, { 2.557562, 1, 1.220090, 0 } },
  { "Canon", "EOS M10", Tungsten        , 0, { 1.399530, 1, 2.615746, 0 } },
  { "Canon", "EOS M10", Shade           , 0, { 2.334441, 1, 1.330011, 0 } },
  { "Canon", "EOS M10", WhiteFluorescent, 0, { 1.809693, 1, 2.239953, 0 } },

  // Canon EOS M50 (KISS M) Firmware Version 1.0.3
  { "Canon", "EOS M50", Daylight            , -9, { 1.3388671875, 1, 1.4951171875, 0 } },
  { "Canon", "EOS M50", Daylight            , -8, { 1.3544921875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS M50", Daylight            , -7, { 1.37109375, 1, 1.4482421875, 0 } },
  { "Canon", "EOS M50", Daylight            , -6, { 1.3896484375, 1, 1.42578125, 0 } },
  { "Canon", "EOS M50", Daylight            , -5, { 1.41015625, 1, 1.40234375, 0 } },
  { "Canon", "EOS M50", Daylight            , -4, { 1.4296875, 1, 1.3779296875, 0 } },
  { "Canon", "EOS M50", Daylight            , -3, { 1.4482421875, 1, 1.3583984375, 0 } },
  { "Canon", "EOS M50", Daylight            , -2, { 1.46875, 1, 1.3369140625, 0 } },
  { "Canon", "EOS M50", Daylight            , -1, { 1.490234375, 1, 1.314453125, 0 } },
  { "Canon", "EOS M50", Daylight            , 0, { 1.5126953125, 1, 1.291015625, 0 } },
  { "Canon", "EOS M50", Daylight            , 1, { 1.53515625, 1, 1.275390625, 0 } },
  { "Canon", "EOS M50", Daylight            , 2, { 1.55859375, 1, 1.259765625, 0 } },
  { "Canon", "EOS M50", Daylight            , 3, { 1.5849609375, 1, 1.2431640625, 0 } },
  { "Canon", "EOS M50", Daylight            , 4, { 1.60546875, 1, 1.224609375, 0 } },
  { "Canon", "EOS M50", Daylight            , 5, { 1.6279296875, 1, 1.2080078125, 0 } },
  { "Canon", "EOS M50", Daylight            , 6, { 1.6513671875, 1, 1.1904296875, 0 } },
  { "Canon", "EOS M50", Daylight            , 7, { 1.6728515625, 1, 1.173828125, 0 } },
  { "Canon", "EOS M50", Daylight            , 8, { 1.6982421875, 1, 1.1572265625, 0 } },
  { "Canon", "EOS M50", Daylight            , 9, { 1.7265625, 1, 1.138671875, 0 } },
  { "Canon", "EOS M50", Shade               , -9, { 1.533203125, 1, 1.2783203125, 0 } },
  { "Canon", "EOS M50", Shade               , -8, { 1.556640625, 1, 1.2607421875, 0 } },
  { "Canon", "EOS M50", Shade               , -7, { 1.5830078125, 1, 1.244140625, 0 } },
  { "Canon", "EOS M50", Shade               , -6, { 1.6025390625, 1, 1.2275390625, 0 } },
  { "Canon", "EOS M50", Shade               , -5, { 1.625, 1, 1.2099609375, 0 } },
  { "Canon", "EOS M50", Shade               , -4, { 1.6494140625, 1, 1.193359375, 0 } },
  { "Canon", "EOS M50", Shade               , -3, { 1.6708984375, 1, 1.17578125, 0 } },
  { "Canon", "EOS M50", Shade               , -2, { 1.6953125, 1, 1.158203125, 0 } },
  { "Canon", "EOS M50", Shade               , -1, { 1.7236328125, 1, 1.140625, 0 } },
  { "Canon", "EOS M50", Shade               , 0, { 1.75390625, 1, 1.12109375, 0 } },
  { "Canon", "EOS M50", Shade               , 1, { 1.7744140625, 1, 1.107421875, 0 } },
  { "Canon", "EOS M50", Shade               , 2, { 1.796875, 1, 1.0908203125, 0 } },
  { "Canon", "EOS M50", Shade               , 3, { 1.822265625, 1, 1.0732421875, 0 } },
  { "Canon", "EOS M50", Shade               , 4, { 1.8486328125, 1, 1.056640625, 0 } },
  { "Canon", "EOS M50", Shade               , 5, { 1.875, 1, 1.041015625, 0 } },
  { "Canon", "EOS M50", Shade               , 6, { 1.896484375, 1, 1.0263671875, 0 } },
  { "Canon", "EOS M50", Shade               , 7, { 1.9208984375, 1, 1.009765625, 0 } },
  { "Canon", "EOS M50", Shade               , 8, { 1.9462890625, 1, 0.994140625, 0 } },
  { "Canon", "EOS M50", Shade               , 9, { 1.97265625, 1, 0.978515625, 0 } },
  { "Canon", "EOS M50", Cloudy              , -9, { 1.4326171875, 1, 1.3759765625, 0 } },
  { "Canon", "EOS M50", Cloudy              , -8, { 1.4521484375, 1, 1.3544921875, 0 } },
  { "Canon", "EOS M50", Cloudy              , -7, { 1.4716796875, 1, 1.3330078125, 0 } },
  { "Canon", "EOS M50", Cloudy              , -6, { 1.4931640625, 1, 1.3115234375, 0 } },
  { "Canon", "EOS M50", Cloudy              , -5, { 1.5166015625, 1, 1.2900390625, 0 } },
  { "Canon", "EOS M50", Cloudy              , -4, { 1.537109375, 1, 1.2734375, 0 } },
  { "Canon", "EOS M50", Cloudy              , -3, { 1.5634765625, 1, 1.2568359375, 0 } },
  { "Canon", "EOS M50", Cloudy              , -2, { 1.587890625, 1, 1.2392578125, 0 } },
  { "Canon", "EOS M50", Cloudy              , -1, { 1.607421875, 1, 1.2236328125, 0 } },
  { "Canon", "EOS M50", Cloudy              , 0, { 1.630859375, 1, 1.205078125, 0 } },
  { "Canon", "EOS M50", Cloudy              , 1, { 1.654296875, 1, 1.189453125, 0 } },
  { "Canon", "EOS M50", Cloudy              , 2, { 1.67578125, 1, 1.171875, 0 } },
  { "Canon", "EOS M50", Cloudy              , 3, { 1.7041015625, 1, 1.154296875, 0 } },
  { "Canon", "EOS M50", Cloudy              , 4, { 1.7294921875, 1, 1.13671875, 0 } },
  { "Canon", "EOS M50", Cloudy              , 5, { 1.759765625, 1, 1.119140625, 0 } },
  { "Canon", "EOS M50", Cloudy              , 6, { 1.78125, 1, 1.103515625, 0 } },
  { "Canon", "EOS M50", Cloudy              , 7, { 1.802734375, 1, 1.0869140625, 0 } },
  { "Canon", "EOS M50", Cloudy              , 8, { 1.828125, 1, 1.0703125, 0 } },
  { "Canon", "EOS M50", Cloudy              , 9, { 1.85546875, 1, 1.052734375, 0 } },
  { "Canon", "EOS M50", Tungsten            , -9, { 0.94921875, 1, 2.09375, 0 } },
  { "Canon", "EOS M50", Tungsten            , -8, { 0.9619140625, 1, 2.0732421875, 0 } },
  { "Canon", "EOS M50", Tungsten            , -7, { 0.974609375, 1, 2.0478515625, 0 } },
  { "Canon", "EOS M50", Tungsten            , -6, { 0.98828125, 1, 2.0234375, 0 } },
  { "Canon", "EOS M50", Tungsten            , -5, { 1.001953125, 1, 2.00390625, 0 } },
  { "Canon", "EOS M50", Tungsten            , -4, { 1.0166015625, 1, 1.9765625, 0 } },
  { "Canon", "EOS M50", Tungsten            , -3, { 1.03125, 1, 1.9501953125, 0 } },
  { "Canon", "EOS M50", Tungsten            , -2, { 1.044921875, 1, 1.9208984375, 0 } },
  { "Canon", "EOS M50", Tungsten            , -1, { 1.0595703125, 1, 1.892578125, 0 } },
  { "Canon", "EOS M50", Tungsten            , 0, { 1.0751953125, 1, 1.8623046875, 0 } },
  { "Canon", "EOS M50", Tungsten            , 1, { 1.0908203125, 1, 1.8388671875, 0 } },
  { "Canon", "EOS M50", Tungsten            , 2, { 1.1044921875, 1, 1.8125, 0 } },
  { "Canon", "EOS M50", Tungsten            , 3, { 1.12109375, 1, 1.787109375, 0 } },
  { "Canon", "EOS M50", Tungsten            , 4, { 1.1376953125, 1, 1.7626953125, 0 } },
  { "Canon", "EOS M50", Tungsten            , 5, { 1.1552734375, 1, 1.73828125, 0 } },
  { "Canon", "EOS M50", Tungsten            , 6, { 1.1728515625, 1, 1.7119140625, 0 } },
  { "Canon", "EOS M50", Tungsten            , 7, { 1.1904296875, 1, 1.689453125, 0 } },
  { "Canon", "EOS M50", Tungsten            , 8, { 1.208984375, 1, 1.6650390625, 0 } },
  { "Canon", "EOS M50", Tungsten            , 9, { 1.2275390625, 1, 1.640625, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -9, { 1.1123046875, 1, 1.99609375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -8, { 1.1279296875, 1, 1.97265625, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -7, { 1.1455078125, 1, 1.943359375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -6, { 1.1640625, 1, 1.9140625, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -5, { 1.1796875, 1, 1.8857421875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -4, { 1.197265625, 1, 1.8583984375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -3, { 1.2158203125, 1, 1.8349609375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -2, { 1.236328125, 1, 1.8095703125, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, -1, { 1.2548828125, 1, 1.7841796875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 0, { 1.2705078125, 1, 1.7568359375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 1, { 1.2880859375, 1, 1.732421875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 2, { 1.3056640625, 1, 1.7099609375, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 3, { 1.326171875, 1, 1.6845703125, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 4, { 1.34375, 1, 1.6591796875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 5, { 1.361328125, 1, 1.6357421875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 6, { 1.3779296875, 1, 1.6123046875, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 7, { 1.3974609375, 1, 1.587890625, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 8, { 1.41796875, 1, 1.5634765625, 0 } },
  { "Canon", "EOS M50", CoolWhiteFluorescent, 9, { 1.4384765625, 1, 1.5400390625, 0 } },
  { "Canon", "EOS M50", WhiteFluorescent    , 0, { 1.2705078125, 1, 1.7568359375, 0 } },
  { "Canon", "EOS M50", Flash               , -9, { 1.45703125, 1, 1.361328125, 0 } },
  { "Canon", "EOS M50", Flash               , -8, { 1.4755859375, 1, 1.33984375, 0 } },
  { "Canon", "EOS M50", Flash               , -7, { 1.4970703125, 1, 1.318359375, 0 } },
  { "Canon", "EOS M50", Flash               , -6, { 1.521484375, 1, 1.2958984375, 0 } },
  { "Canon", "EOS M50", Flash               , -5, { 1.544921875, 1, 1.2783203125, 0 } },
  { "Canon", "EOS M50", Flash               , -4, { 1.568359375, 1, 1.2626953125, 0 } },
  { "Canon", "EOS M50", Flash               , -3, { 1.58984375, 1, 1.24609375, 0 } },
  { "Canon", "EOS M50", Flash               , -2, { 1.6123046875, 1, 1.2275390625, 0 } },
  { "Canon", "EOS M50", Flash               , -1, { 1.6357421875, 1, 1.2099609375, 0 } },
  { "Canon", "EOS M50", Flash               , 0, { 1.6572265625, 1, 1.193359375, 0 } },
  { "Canon", "EOS M50", Flash               , 1, { 1.681640625, 1, 1.1767578125, 0 } },
  { "Canon", "EOS M50", Flash               , 2, { 1.7099609375, 1, 1.16015625, 0 } },
  { "Canon", "EOS M50", Flash               , 3, { 1.73828125, 1, 1.142578125, 0 } },
  { "Canon", "EOS M50", Flash               , 4, { 1.7626953125, 1, 1.1240234375, 0 } },
  { "Canon", "EOS M50", Flash               , 5, { 1.7841796875, 1, 1.1083984375, 0 } },
  { "Canon", "EOS M50", Flash               , 6, { 1.8095703125, 1, 1.091796875, 0 } },
  { "Canon", "EOS M50", Flash               , 7, { 1.8349609375, 1, 1.0751953125, 0 } },
  { "Canon", "EOS M50", Flash               , 8, { 1.8623046875, 1, 1.0576171875, 0 } },
  { "Canon", "EOS M50", Flash               , 9, { 1.8828125, 1, 1.0419921875, 0 } },

  // Canon EOS M100 Firmware Version 1.0.0
  { "Canon", "EOS M100", Cloudy          , 0, { 1.70703125, 1, 1.181640625, 0 } },
  { "Canon", "EOS M100", Daylight        , 0, { 1.5791015625, 1, 1.2734375, 0 } },
  { "Canon", "EOS M100", Flash           , 0, { 1.9755859375, 1, 1.0107421875, 0 } },
  { "Canon", "EOS M100", Tungsten        , 0, { 1.0810546875, 1, 1.8857421875, 0 } },
  { "Canon", "EOS M100", Shade           , 0, { 1.8232421875, 1, 1.1015625, 0 } },
  { "Canon", "EOS M100", Fluorescent     , 0, { 1.3544921875, 1, 1.7021484375, 0 } },

  //Canon EOS R Firmware Version 1.8.0
  { "Canon", "EOS R", Daylight            , -9, { 1.7900390625, 1, 1.640625, 0 } },
  { "Canon", "EOS R", Daylight            , -8, { 1.8125, 1, 1.6181640625, 0 } },
  { "Canon", "EOS R", Daylight            , -7, { 1.8349609375, 1, 1.5947265625, 0 } },
  { "Canon", "EOS R", Daylight            , -6, { 1.8623046875, 1, 1.5703125, 0 } },
  { "Canon", "EOS R", Daylight            , -5, { 1.892578125, 1, 1.546875, 0 } },
  { "Canon", "EOS R", Daylight            , -4, { 1.9208984375, 1, 1.5234375, 0 } },
  { "Canon", "EOS R", Daylight            , -3, { 1.943359375, 1, 1.4990234375, 0 } },
  { "Canon", "EOS R", Daylight            , -2, { 1.96875, 1, 1.4755859375, 0 } },
  { "Canon", "EOS R", Daylight            , -1, { 1.99609375, 1, 1.4501953125, 0 } },
  { "Canon", "EOS R", Daylight            , 0, { 2.0234375, 1, 1.423828125, 0 } },
  { "Canon", "EOS R", Daylight            , 1, { 2.0517578125, 1, 1.404296875, 0 } },
  { "Canon", "EOS R", Daylight            , 2, { 2.0771484375, 1, 1.3837890625, 0 } },
  { "Canon", "EOS R", Daylight            , 3, { 2.107421875, 1, 1.36328125, 0 } },
  { "Canon", "EOS R", Daylight            , 4, { 2.1337890625, 1, 1.34375, 0 } },
  { "Canon", "EOS R", Daylight            , 5, { 2.1650390625, 1, 1.3232421875, 0 } },
  { "Canon", "EOS R", Daylight            , 6, { 2.1884765625, 1, 1.3046875, 0 } },
  { "Canon", "EOS R", Daylight            , 7, { 2.216796875, 1, 1.2861328125, 0 } },
  { "Canon", "EOS R", Daylight            , 8, { 2.24609375, 1, 1.267578125, 0 } },
  { "Canon", "EOS R", Daylight            , 9, { 2.2802734375, 1, 1.2470703125, 0 } },
  { "Canon", "EOS R", Shade               , -9, { 2.0478515625, 1, 1.40625, 0 } },
  { "Canon", "EOS R", Shade               , -8, { 2.0771484375, 1, 1.3857421875, 0 } },
  { "Canon", "EOS R", Shade               , -7, { 2.1025390625, 1, 1.365234375, 0 } },
  { "Canon", "EOS R", Shade               , -6, { 2.12890625, 1, 1.345703125, 0 } },
  { "Canon", "EOS R", Shade               , -5, { 2.16015625, 1, 1.3251953125, 0 } },
  { "Canon", "EOS R", Shade               , -4, { 2.1884765625, 1, 1.3056640625, 0 } },
  { "Canon", "EOS R", Shade               , -3, { 2.216796875, 1, 1.2880859375, 0 } },
  { "Canon", "EOS R", Shade               , -2, { 2.24609375, 1, 1.2685546875, 0 } },
  { "Canon", "EOS R", Shade               , -1, { 2.275390625, 1, 1.2490234375, 0 } },
  { "Canon", "EOS R", Shade               , 0, { 2.3115234375, 1, 1.2275390625, 0 } },
  { "Canon", "EOS R", Shade               , 1, { 2.337890625, 1, 1.2119140625, 0 } },
  { "Canon", "EOS R", Shade               , 2, { 2.3701171875, 1, 1.1953125, 0 } },
  { "Canon", "EOS R", Shade               , 3, { 2.3984375, 1, 1.17578125, 0 } },
  { "Canon", "EOS R", Shade               , 4, { 2.4326171875, 1, 1.1572265625, 0 } },
  { "Canon", "EOS R", Shade               , 5, { 2.4677734375, 1, 1.138671875, 0 } },
  { "Canon", "EOS R", Shade               , 6, { 2.4912109375, 1, 1.1240234375, 0 } },
  { "Canon", "EOS R", Shade               , 7, { 2.515625, 1, 1.107421875, 0 } },
  { "Canon", "EOS R", Shade               , 8, { 2.546875, 1, 1.0908203125, 0 } },
  { "Canon", "EOS R", Shade               , 9, { 2.5791015625, 1, 1.0732421875, 0 } },
  { "Canon", "EOS R", Cloudy              , -9, { 1.9248046875, 1, 1.51953125, 0 } },
  { "Canon", "EOS R", Cloudy              , -8, { 1.9462890625, 1, 1.4970703125, 0 } },
  { "Canon", "EOS R", Cloudy              , -7, { 1.97265625, 1, 1.4716796875, 0 } },
  { "Canon", "EOS R", Cloudy              , -6, { 2.0, 1, 1.4462890625, 0 } },
  { "Canon", "EOS R", Cloudy              , -5, { 2.02734375, 1, 1.421875, 0 } },
  { "Canon", "EOS R", Cloudy              , -4, { 2.056640625, 1, 1.40234375, 0 } },
  { "Canon", "EOS R", Cloudy              , -3, { 2.0810546875, 1, 1.3818359375, 0 } },
  { "Canon", "EOS R", Cloudy              , -2, { 2.111328125, 1, 1.3603515625, 0 } },
  { "Canon", "EOS R", Cloudy              , -1, { 2.1376953125, 1, 1.33984375, 0 } },
  { "Canon", "EOS R", Cloudy              , 0, { 2.1650390625, 1, 1.3193359375, 0 } },
  { "Canon", "EOS R", Cloudy              , 1, { 2.1923828125, 1, 1.302734375, 0 } },
  { "Canon", "EOS R", Cloudy              , 2, { 2.2216796875, 1, 1.283203125, 0 } },
  { "Canon", "EOS R", Cloudy              , 3, { 2.2509765625, 1, 1.2646484375, 0 } },
  { "Canon", "EOS R", Cloudy              , 4, { 2.2861328125, 1, 1.244140625, 0 } },
  { "Canon", "EOS R", Cloudy              , 5, { 2.322265625, 1, 1.224609375, 0 } },
  { "Canon", "EOS R", Cloudy              , 6, { 2.3486328125, 1, 1.2080078125, 0 } },
  { "Canon", "EOS R", Cloudy              , 7, { 2.3759765625, 1, 1.1904296875, 0 } },
  { "Canon", "EOS R", Cloudy              , 8, { 2.4091796875, 1, 1.171875, 0 } },
  { "Canon", "EOS R", Cloudy              , 9, { 2.4443359375, 1, 1.15234375, 0 } },
  { "Canon", "EOS R", Tungsten            , -9, { 1.2763671875, 1, 2.2861328125, 0 } },
  { "Canon", "EOS R", Tungsten            , -8, { 1.294921875, 1, 2.2607421875, 0 } },
  { "Canon", "EOS R", Tungsten            , -7, { 1.3115234375, 1, 2.2353515625, 0 } },
  { "Canon", "EOS R", Tungsten            , -6, { 1.330078125, 1, 2.2119140625, 0 } },
  { "Canon", "EOS R", Tungsten            , -5, { 1.349609375, 1, 2.18359375, 0 } },
  { "Canon", "EOS R", Tungsten            , -4, { 1.369140625, 1, 2.15625, 0 } },
  { "Canon", "EOS R", Tungsten            , -3, { 1.3857421875, 1, 2.12890625, 0 } },
  { "Canon", "EOS R", Tungsten            , -2, { 1.404296875, 1, 2.09375, 0 } },
  { "Canon", "EOS R", Tungsten            , -1, { 1.423828125, 1, 2.064453125, 0 } },
  { "Canon", "EOS R", Tungsten            , 0, { 1.4443359375, 1, 2.0322265625, 0 } },
  { "Canon", "EOS R", Tungsten            , 1, { 1.462890625, 1, 2.0078125, 0 } },
  { "Canon", "EOS R", Tungsten            , 2, { 1.4814453125, 1, 1.98046875, 0 } },
  { "Canon", "EOS R", Tungsten            , 3, { 1.50390625, 1, 1.9501953125, 0 } },
  { "Canon", "EOS R", Tungsten            , 4, { 1.5263671875, 1, 1.9208984375, 0 } },
  { "Canon", "EOS R", Tungsten            , 5, { 1.548828125, 1, 1.892578125, 0 } },
  { "Canon", "EOS R", Tungsten            , 6, { 1.5703125, 1, 1.8681640625, 0 } },
  { "Canon", "EOS R", Tungsten            , 7, { 1.5927734375, 1, 1.841796875, 0 } },
  { "Canon", "EOS R", Tungsten            , 8, { 1.615234375, 1, 1.8154296875, 0 } },
  { "Canon", "EOS R", Tungsten            , 9, { 1.640625, 1, 1.7900390625, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -9, { 1.5078125, 1, 2.23046875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -8, { 1.5302734375, 1, 2.2021484375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -7, { 1.5537109375, 1, 2.1787109375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -6, { 1.5751953125, 1, 2.1513671875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -5, { 1.59765625, 1, 2.1201171875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -4, { 1.623046875, 1, 2.08984375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -3, { 1.646484375, 1, 2.056640625, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -2, { 1.6708984375, 1, 2.02734375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, -1, { 1.6923828125, 1, 2.0, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 0, { 1.71484375, 1, 1.96875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 1, { 1.7412109375, 1, 1.943359375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 2, { 1.7685546875, 1, 1.9140625, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 3, { 1.79296875, 1, 1.8896484375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 4, { 1.8154296875, 1, 1.8623046875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 5, { 1.841796875, 1, 1.8349609375, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 6, { 1.8681640625, 1, 1.8095703125, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 7, { 1.896484375, 1, 1.7841796875, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 8, { 1.9248046875, 1, 1.759765625, 0 } },
  { "Canon", "EOS R", CoolWhiteFluorescent, 9, { 1.9501953125, 1, 1.7353515625, 0 } },
  { "Canon", "EOS R", WhiteFluorescent    , 0, { 1.71484375, 1, 1.96875, 0 } },
  { "Canon", "EOS R", Flash               , -9, { 1.9658203125, 1, 1.5166015625, 0 } },
  { "Canon", "EOS R", Flash               , -8, { 1.9921875, 1, 1.4951171875, 0 } },
  { "Canon", "EOS R", Flash               , -7, { 2.0234375, 1, 1.46875, 0 } },
  { "Canon", "EOS R", Flash               , -6, { 2.0478515625, 1, 1.4443359375, 0 } },
  { "Canon", "EOS R", Flash               , -5, { 2.0771484375, 1, 1.419921875, 0 } },
  { "Canon", "EOS R", Flash               , -4, { 2.1025390625, 1, 1.400390625, 0 } },
  { "Canon", "EOS R", Flash               , -3, { 2.12890625, 1, 1.3798828125, 0 } },
  { "Canon", "EOS R", Flash               , -2, { 2.16015625, 1, 1.3583984375, 0 } },
  { "Canon", "EOS R", Flash               , -1, { 2.1884765625, 1, 1.3388671875, 0 } },
  { "Canon", "EOS R", Flash               , 0, { 2.2119140625, 1, 1.318359375, 0 } },
  { "Canon", "EOS R", Flash               , 1, { 2.24609375, 1, 1.30078125, 0 } },
  { "Canon", "EOS R", Flash               , 2, { 2.275390625, 1, 1.28125, 0 } },
  { "Canon", "EOS R", Flash               , 3, { 2.3115234375, 1, 1.2626953125, 0 } },
  { "Canon", "EOS R", Flash               , 4, { 2.337890625, 1, 1.2431640625, 0 } },
  { "Canon", "EOS R", Flash               , 5, { 2.3701171875, 1, 1.2236328125, 0 } },
  { "Canon", "EOS R", Flash               , 6, { 2.3984375, 1, 1.2060546875, 0 } },
  { "Canon", "EOS R", Flash               , 7, { 2.4326171875, 1, 1.1875, 0 } },
  { "Canon", "EOS R", Flash               , 8, { 2.4677734375, 1, 1.169921875, 0 } },
  { "Canon", "EOS R", Flash               , 9, { 2.4912109375, 1, 1.150390625, 0 } },

  //Canon EOS RP Firmware Version 1.6.0
  { "Canon", "EOS RP", Daylight            , -9, { 1.759765625, 1, 1.9248046875, 0 } },
  { "Canon", "EOS RP", Daylight            , -8, { 1.78125, 1, 1.892578125, 0 } },
  { "Canon", "EOS RP", Daylight            , -7, { 1.8056640625, 1, 1.8623046875, 0 } },
  { "Canon", "EOS RP", Daylight            , -6, { 1.83203125, 1, 1.828125, 0 } },
  { "Canon", "EOS RP", Daylight            , -5, { 1.8583984375, 1, 1.796875, 0 } },
  { "Canon", "EOS RP", Daylight            , -4, { 1.8896484375, 1, 1.765625, 0 } },
  { "Canon", "EOS RP", Daylight            , -3, { 1.9140625, 1, 1.73828125, 0 } },
  { "Canon", "EOS RP", Daylight            , -2, { 1.943359375, 1, 1.7119140625, 0 } },
  { "Canon", "EOS RP", Daylight            , -1, { 1.97265625, 1, 1.6845703125, 0 } },
  { "Canon", "EOS RP", Daylight            , 0, { 2.00390625, 1, 1.654296875, 0 } },
  { "Canon", "EOS RP", Daylight            , 1, { 2.0361328125, 1, 1.630859375, 0 } },
  { "Canon", "EOS RP", Daylight            , 2, { 2.064453125, 1, 1.60546875, 0 } },
  { "Canon", "EOS RP", Daylight            , 3, { 2.0986328125, 1, 1.580078125, 0 } },
  { "Canon", "EOS RP", Daylight            , 4, { 2.1337890625, 1, 1.556640625, 0 } },
  { "Canon", "EOS RP", Daylight            , 5, { 2.1650390625, 1, 1.5302734375, 0 } },
  { "Canon", "EOS RP", Daylight            , 6, { 2.197265625, 1, 1.5078125, 0 } },
  { "Canon", "EOS RP", Daylight            , 7, { 2.2265625, 1, 1.486328125, 0 } },
  { "Canon", "EOS RP", Daylight            , 8, { 2.2607421875, 1, 1.462890625, 0 } },
  { "Canon", "EOS RP", Daylight            , 9, { 2.2958984375, 1, 1.4384765625, 0 } },
  { "Canon", "EOS RP", Shade               , -9, { 2.0322265625, 1, 1.6328125, 0 } },
  { "Canon", "EOS RP", Shade               , -8, { 2.064453125, 1, 1.607421875, 0 } },
  { "Canon", "EOS RP", Shade               , -7, { 2.09375, 1, 1.5830078125, 0 } },
  { "Canon", "EOS RP", Shade               , -6, { 2.12890625, 1, 1.55859375, 0 } },
  { "Canon", "EOS RP", Shade               , -5, { 2.1650390625, 1, 1.533203125, 0 } },
  { "Canon", "EOS RP", Shade               , -4, { 2.1923828125, 1, 1.5107421875, 0 } },
  { "Canon", "EOS RP", Shade               , -3, { 2.2216796875, 1, 1.48828125, 0 } },
  { "Canon", "EOS RP", Shade               , -2, { 2.255859375, 1, 1.46484375, 0 } },
  { "Canon", "EOS RP", Shade               , -1, { 2.291015625, 1, 1.4404296875, 0 } },
  { "Canon", "EOS RP", Shade               , 0, { 2.3271484375, 1, 1.4140625, 0 } },
  { "Canon", "EOS RP", Shade               , 1, { 2.359375, 1, 1.3955078125, 0 } },
  { "Canon", "EOS RP", Shade               , 2, { 2.392578125, 1, 1.3740234375, 0 } },
  { "Canon", "EOS RP", Shade               , 3, { 2.4326171875, 1, 1.3544921875, 0 } },
  { "Canon", "EOS RP", Shade               , 4, { 2.4736328125, 1, 1.33203125, 0 } },
  { "Canon", "EOS RP", Shade               , 5, { 2.509765625, 1, 1.3115234375, 0 } },
  { "Canon", "EOS RP", Shade               , 6, { 2.541015625, 1, 1.29296875, 0 } },
  { "Canon", "EOS RP", Shade               , 7, { 2.5791015625, 1, 1.2734375, 0 } },
  { "Canon", "EOS RP", Shade               , 8, { 2.619140625, 1, 1.2548828125, 0 } },
  { "Canon", "EOS RP", Shade               , 9, { 2.66015625, 1, 1.2353515625, 0 } },
  { "Canon", "EOS RP", Cloudy              , -9, { 1.892578125, 1, 1.7626953125, 0 } },
  { "Canon", "EOS RP", Cloudy              , -8, { 1.91796875, 1, 1.7353515625, 0 } },
  { "Canon", "EOS RP", Cloudy              , -7, { 1.9462890625, 1, 1.70703125, 0 } },
  { "Canon", "EOS RP", Cloudy              , -6, { 1.9765625, 1, 1.681640625, 0 } },
  { "Canon", "EOS RP", Cloudy              , -5, { 2.0078125, 1, 1.6513671875, 0 } },
  { "Canon", "EOS RP", Cloudy              , -4, { 2.0400390625, 1, 1.6279296875, 0 } },
  { "Canon", "EOS RP", Cloudy              , -3, { 2.068359375, 1, 1.6025390625, 0 } },
  { "Canon", "EOS RP", Cloudy              , -2, { 2.1025390625, 1, 1.578125, 0 } },
  { "Canon", "EOS RP", Cloudy              , -1, { 2.1376953125, 1, 1.5517578125, 0 } },
  { "Canon", "EOS RP", Cloudy              , 0, { 2.169921875, 1, 1.5263671875, 0 } },
  { "Canon", "EOS RP", Cloudy              , 1, { 2.197265625, 1, 1.505859375, 0 } },
  { "Canon", "EOS RP", Cloudy              , 2, { 2.23046875, 1, 1.4814453125, 0 } },
  { "Canon", "EOS RP", Cloudy              , 3, { 2.2607421875, 1, 1.458984375, 0 } },
  { "Canon", "EOS RP", Cloudy              , 4, { 2.30078125, 1, 1.4345703125, 0 } },
  { "Canon", "EOS RP", Cloudy              , 5, { 2.337890625, 1, 1.41015625, 0 } },
  { "Canon", "EOS RP", Cloudy              , 6, { 2.3701171875, 1, 1.3916015625, 0 } },
  { "Canon", "EOS RP", Cloudy              , 7, { 2.4033203125, 1, 1.37109375, 0 } },
  { "Canon", "EOS RP", Cloudy              , 8, { 2.4384765625, 1, 1.349609375, 0 } },
  { "Canon", "EOS RP", Cloudy              , 9, { 2.4853515625, 1, 1.326171875, 0 } },
  { "Canon", "EOS RP", Tungsten            , -9, { 1.2353515625, 1, 2.775390625, 0 } },
  { "Canon", "EOS RP", Tungsten            , -8, { 1.251953125, 1, 2.73828125, 0 } },
  { "Canon", "EOS RP", Tungsten            , -7, { 1.2685546875, 1, 2.6943359375, 0 } },
  { "Canon", "EOS RP", Tungsten            , -6, { 1.28515625, 1, 2.66015625, 0 } },
  { "Canon", "EOS RP", Tungsten            , -5, { 1.3046875, 1, 2.619140625, 0 } },
  { "Canon", "EOS RP", Tungsten            , -4, { 1.3232421875, 1, 2.5859375, 0 } },
  { "Canon", "EOS RP", Tungsten            , -3, { 1.33984375, 1, 2.546875, 0 } },
  { "Canon", "EOS RP", Tungsten            , -2, { 1.3603515625, 1, 2.50390625, 0 } },
  { "Canon", "EOS RP", Tungsten            , -1, { 1.3798828125, 1, 2.4677734375, 0 } },
  { "Canon", "EOS RP", Tungsten            , 0, { 1.400390625, 1, 2.4267578125, 0 } },
  { "Canon", "EOS RP", Tungsten            , 1, { 1.419921875, 1, 2.38671875, 0 } },
  { "Canon", "EOS RP", Tungsten            , 2, { 1.4404296875, 1, 2.3486328125, 0 } },
  { "Canon", "EOS RP", Tungsten            , 3, { 1.462890625, 1, 2.306640625, 0 } },
  { "Canon", "EOS RP", Tungsten            , 4, { 1.484375, 1, 2.2705078125, 0 } },
  { "Canon", "EOS RP", Tungsten            , 5, { 1.5078125, 1, 2.23046875, 0 } },
  { "Canon", "EOS RP", Tungsten            , 6, { 1.5302734375, 1, 2.1923828125, 0 } },
  { "Canon", "EOS RP", Tungsten            , 7, { 1.5537109375, 1, 2.1650390625, 0 } },
  { "Canon", "EOS RP", Tungsten            , 8, { 1.5751953125, 1, 2.12890625, 0 } },
  { "Canon", "EOS RP", Tungsten            , 9, { 1.599609375, 1, 2.0986328125, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -9, { 1.4501953125, 1, 2.6328125, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -8, { 1.4716796875, 1, 2.5927734375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -7, { 1.4951171875, 1, 2.5537109375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -6, { 1.51953125, 1, 2.515625, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -5, { 1.5419921875, 1, 2.4794921875, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -4, { 1.5634765625, 1, 2.4384765625, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -3, { 1.587890625, 1, 2.3984375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -2, { 1.6123046875, 1, 2.359375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, -1, { 1.638671875, 1, 2.31640625, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 0, { 1.6591796875, 1, 2.275390625, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 1, { 1.6865234375, 1, 2.240234375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 2, { 1.7119140625, 1, 2.2021484375, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 3, { 1.7412109375, 1, 2.169921875, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 4, { 1.7685546875, 1, 2.1376953125, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 5, { 1.7900390625, 1, 2.107421875, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 6, { 1.8154296875, 1, 2.0732421875, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 7, { 1.8447265625, 1, 2.0478515625, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 8, { 1.8720703125, 1, 2.01953125, 0 } },
  { "Canon", "EOS RP", CoolWhiteFluorescent, 9, { 1.8994140625, 1, 1.9921875, 0 } },
  { "Canon", "EOS RP", WhiteFluorescent    , 0, { 1.6591796875, 1, 2.275390625, 0 } },
  { "Canon", "EOS RP", Flash               , -9, { 1.9140625, 1, 1.7294921875, 0 } },
  { "Canon", "EOS RP", Flash               , -8, { 1.943359375, 1, 1.701171875, 0 } },
  { "Canon", "EOS RP", Flash               , -7, { 1.97265625, 1, 1.6728515625, 0 } },
  { "Canon", "EOS RP", Flash               , -6, { 2.00390625, 1, 1.646484375, 0 } },
  { "Canon", "EOS RP", Flash               , -5, { 2.0361328125, 1, 1.623046875, 0 } },
  { "Canon", "EOS RP", Flash               , -4, { 2.064453125, 1, 1.59765625, 0 } },
  { "Canon", "EOS RP", Flash               , -3, { 2.0986328125, 1, 1.5732421875, 0 } },
  { "Canon", "EOS RP", Flash               , -2, { 2.12890625, 1, 1.546875, 0 } },
  { "Canon", "EOS RP", Flash               , -1, { 2.1650390625, 1, 1.521484375, 0 } },
  { "Canon", "EOS RP", Flash               , 0, { 2.1923828125, 1, 1.4990234375, 0 } },
  { "Canon", "EOS RP", Flash               , 1, { 2.2265625, 1, 1.4775390625, 0 } },
  { "Canon", "EOS RP", Flash               , 2, { 2.255859375, 1, 1.4541015625, 0 } },
  { "Canon", "EOS RP", Flash               , 3, { 2.2958984375, 1, 1.4296875, 0 } },
  { "Canon", "EOS RP", Flash               , 4, { 2.3330078125, 1, 1.40625, 0 } },
  { "Canon", "EOS RP", Flash               , 5, { 2.365234375, 1, 1.3876953125, 0 } },
  { "Canon", "EOS RP", Flash               , 6, { 2.3984375, 1, 1.365234375, 0 } },
  { "Canon", "EOS RP", Flash               , 7, { 2.4326171875, 1, 1.34375, 0 } },
  { "Canon", "EOS RP", Flash               , 8, { 2.4794921875, 1, 1.3212890625, 0 } },
  { "Canon", "EOS RP", Flash               , 9, { 2.509765625, 1, 1.302734375, 0 } },

  // Canon EOS R5 Firmware Version 1.3.1
  { "Canon", "EOS R5", Daylight            , -9, { 1.7041015625, 1, 2.1650390625, 0 } },
  { "Canon", "EOS R5", Daylight            , -8, { 1.7236328125, 1, 2.1337890625, 0 } },
  { "Canon", "EOS R5", Daylight            , -7, { 1.744140625, 1, 2.0986328125, 0 } },
  { "Canon", "EOS R5", Daylight            , -6, { 1.7685546875, 1, 2.060546875, 0 } },
  { "Canon", "EOS R5", Daylight            , -5, { 1.796875, 1, 2.02734375, 0 } },
  { "Canon", "EOS R5", Daylight            , -4, { 1.822265625, 1, 1.9921875, 0 } },
  { "Canon", "EOS R5", Daylight            , -3, { 1.8486328125, 1, 1.9501953125, 0 } },
  { "Canon", "EOS R5", Daylight            , -2, { 1.87890625, 1, 1.9140625, 0 } },
  { "Canon", "EOS R5", Daylight            , -1, { 1.91015625, 1, 1.8720703125, 0 } },
  { "Canon", "EOS R5", Daylight            , 0, { 1.943359375, 1, 1.83203125, 0 } },
  { "Canon", "EOS R5", Daylight            , 1, { 1.97265625, 1, 1.8095703125, 0 } },
  { "Canon", "EOS R5", Daylight            , 2, { 2.0, 1, 1.78125, 0 } },
  { "Canon", "EOS R5", Daylight            , 3, { 2.0322265625, 1, 1.7568359375, 0 } },
  { "Canon", "EOS R5", Daylight            , 4, { 2.064453125, 1, 1.7294921875, 0 } },
  { "Canon", "EOS R5", Daylight            , 5, { 2.0986328125, 1, 1.70703125, 0 } },
  { "Canon", "EOS R5", Daylight            , 6, { 2.12890625, 1, 1.6787109375, 0 } },
  { "Canon", "EOS R5", Daylight            , 7, { 2.1650390625, 1, 1.6513671875, 0 } },
  { "Canon", "EOS R5", Daylight            , 8, { 2.197265625, 1, 1.623046875, 0 } },
  { "Canon", "EOS R5", Daylight            , 9, { 2.240234375, 1, 1.5947265625, 0 } },
  { "Canon", "EOS R5", Shade               , -9, { 1.96875, 1, 1.8095703125, 0 } },
  { "Canon", "EOS R5", Shade               , -8, { 2.0, 1, 1.7841796875, 0 } },
  { "Canon", "EOS R5", Shade               , -7, { 2.0322265625, 1, 1.7568359375, 0 } },
  { "Canon", "EOS R5", Shade               , -6, { 2.060546875, 1, 1.732421875, 0 } },
  { "Canon", "EOS R5", Shade               , -5, { 2.09375, 1, 1.7099609375, 0 } },
  { "Canon", "EOS R5", Shade               , -4, { 2.12890625, 1, 1.681640625, 0 } },
  { "Canon", "EOS R5", Shade               , -3, { 2.16015625, 1, 1.654296875, 0 } },
  { "Canon", "EOS R5", Shade               , -2, { 2.197265625, 1, 1.625, 0 } },
  { "Canon", "EOS R5", Shade               , -1, { 2.2353515625, 1, 1.59765625, 0 } },
  { "Canon", "EOS R5", Shade               , 0, { 2.275390625, 1, 1.5654296875, 0 } },
  { "Canon", "EOS R5", Shade               , 1, { 2.306640625, 1, 1.5419921875, 0 } },
  { "Canon", "EOS R5", Shade               , 2, { 2.3330078125, 1, 1.5166015625, 0 } },
  { "Canon", "EOS R5", Shade               , 3, { 2.3701171875, 1, 1.490234375, 0 } },
  { "Canon", "EOS R5", Shade               , 4, { 2.4033203125, 1, 1.462890625, 0 } },
  { "Canon", "EOS R5", Shade               , 5, { 2.4384765625, 1, 1.4384765625, 0 } },
  { "Canon", "EOS R5", Shade               , 6, { 2.4677734375, 1, 1.416015625, 0 } },
  { "Canon", "EOS R5", Shade               , 7, { 2.50390625, 1, 1.3935546875, 0 } },
  { "Canon", "EOS R5", Shade               , 8, { 2.546875, 1, 1.369140625, 0 } },
  { "Canon", "EOS R5", Shade               , 9, { 2.5859375, 1, 1.345703125, 0 } },
  { "Canon", "EOS R5", Cloudy              , -9, { 1.8251953125, 1, 1.984375, 0 } },
  { "Canon", "EOS R5", Cloudy              , -8, { 1.8515625, 1, 1.9462890625, 0 } },
  { "Canon", "EOS R5", Cloudy              , -7, { 1.8828125, 1, 1.9072265625, 0 } },
  { "Canon", "EOS R5", Cloudy              , -6, { 1.9140625, 1, 1.8681640625, 0 } },
  { "Canon", "EOS R5", Cloudy              , -5, { 1.9462890625, 1, 1.83203125, 0 } },
  { "Canon", "EOS R5", Cloudy              , -4, { 1.9765625, 1, 1.802734375, 0 } },
  { "Canon", "EOS R5", Cloudy              , -3, { 2.00390625, 1, 1.77734375, 0 } },
  { "Canon", "EOS R5", Cloudy              , -2, { 2.0361328125, 1, 1.75390625, 0 } },
  { "Canon", "EOS R5", Cloudy              , -1, { 2.068359375, 1, 1.7265625, 0 } },
  { "Canon", "EOS R5", Cloudy              , 0, { 2.1025390625, 1, 1.701171875, 0 } },
  { "Canon", "EOS R5", Cloudy              , 1, { 2.1337890625, 1, 1.67578125, 0 } },
  { "Canon", "EOS R5", Cloudy              , 2, { 2.169921875, 1, 1.6494140625, 0 } },
  { "Canon", "EOS R5", Cloudy              , 3, { 2.2021484375, 1, 1.6201171875, 0 } },
  { "Canon", "EOS R5", Cloudy              , 4, { 2.24609375, 1, 1.58984375, 0 } },
  { "Canon", "EOS R5", Cloudy              , 5, { 2.2861328125, 1, 1.560546875, 0 } },
  { "Canon", "EOS R5", Cloudy              , 6, { 2.3115234375, 1, 1.537109375, 0 } },
  { "Canon", "EOS R5", Cloudy              , 7, { 2.3427734375, 1, 1.5107421875, 0 } },
  { "Canon", "EOS R5", Cloudy              , 8, { 2.3759765625, 1, 1.484375, 0 } },
  { "Canon", "EOS R5", Cloudy              , 9, { 2.4150390625, 1, 1.45703125, 0 } },
  { "Canon", "EOS R5", Tungsten            , -9, { 1.16796875, 1, 3.3134765625, 0 } },
  { "Canon", "EOS R5", Tungsten            , -8, { 1.18359375, 1, 3.2607421875, 0 } },
  { "Canon", "EOS R5", Tungsten            , -7, { 1.2021484375, 1, 3.2197265625, 0 } },
  { "Canon", "EOS R5", Tungsten            , -6, { 1.21875, 1, 3.169921875, 0 } },
  { "Canon", "EOS R5", Tungsten            , -5, { 1.23828125, 1, 3.1220703125, 0 } },
  { "Canon", "EOS R5", Tungsten            , -4, { 1.2578125, 1, 3.0751953125, 0 } },
  { "Canon", "EOS R5", Tungsten            , -3, { 1.2763671875, 1, 3.0205078125, 0 } },
  { "Canon", "EOS R5", Tungsten            , -2, { 1.294921875, 1, 2.9599609375, 0 } },
  { "Canon", "EOS R5", Tungsten            , -1, { 1.31640625, 1, 2.9091796875, 0 } },
  { "Canon", "EOS R5", Tungsten            , 0, { 1.3369140625, 1, 2.8525390625, 0 } },
  { "Canon", "EOS R5", Tungsten            , 1, { 1.3564453125, 1, 2.8056640625, 0 } },
  { "Canon", "EOS R5", Tungsten            , 2, { 1.3740234375, 1, 2.759765625, 0 } },
  { "Canon", "EOS R5", Tungsten            , 3, { 1.3974609375, 1, 2.708984375, 0 } },
  { "Canon", "EOS R5", Tungsten            , 4, { 1.41796875, 1, 2.66015625, 0 } },
  { "Canon", "EOS R5", Tungsten            , 5, { 1.4423828125, 1, 2.60546875, 0 } },
  { "Canon", "EOS R5", Tungsten            , 6, { 1.46484375, 1, 2.5595703125, 0 } },
  { "Canon", "EOS R5", Tungsten            , 7, { 1.48828125, 1, 2.515625, 0 } },
  { "Canon", "EOS R5", Tungsten            , 8, { 1.5126953125, 1, 2.4736328125, 0 } },
  { "Canon", "EOS R5", Tungsten            , 9, { 1.5400390625, 1, 2.4267578125, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -9, { 1.42578125, 1, 3.1904296875, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -8, { 1.4501953125, 1, 3.140625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -7, { 1.4736328125, 1, 3.09375, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -6, { 1.4970703125, 1, 3.0390625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -5, { 1.521484375, 1, 2.9853515625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -4, { 1.548828125, 1, 2.9345703125, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -3, { 1.5751953125, 1, 2.8759765625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -2, { 1.599609375, 1, 2.8291015625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, -1, { 1.625, 1, 2.775390625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 0, { 1.6513671875, 1, 2.7236328125, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 1, { 1.681640625, 1, 2.6806640625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 2, { 1.7099609375, 1, 2.6259765625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 3, { 1.7294921875, 1, 2.5791015625, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 4, { 1.75390625, 1, 2.5341796875, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 5, { 1.77734375, 1, 2.4912109375, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 6, { 1.802734375, 1, 2.4501953125, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 7, { 1.83203125, 1, 2.4033203125, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 8, { 1.8583984375, 1, 2.359375, 0 } },
  { "Canon", "EOS R5", CoolWhiteFluorescent, 9, { 1.8896484375, 1, 2.31640625, 0 } },
  { "Canon", "EOS R5", WhiteFluorescent    , 0, { 1.6513671875, 1, 2.7236328125, 0 } },
  { "Canon", "EOS R5", Flash               , -9, { 1.892578125, 1, 1.8896484375, 0 } },
  { "Canon", "EOS R5", Flash               , -8, { 1.9287109375, 1, 1.8486328125, 0 } },
  { "Canon", "EOS R5", Flash               , -7, { 1.9580078125, 1, 1.818359375, 0 } },
  { "Canon", "EOS R5", Flash               , -6, { 1.98828125, 1, 1.79296875, 0 } },
  { "Canon", "EOS R5", Flash               , -5, { 2.015625, 1, 1.765625, 0 } },
  { "Canon", "EOS R5", Flash               , -4, { 2.0478515625, 1, 1.7412109375, 0 } },
  { "Canon", "EOS R5", Flash               , -3, { 2.0810546875, 1, 1.71484375, 0 } },
  { "Canon", "EOS R5", Flash               , -2, { 2.115234375, 1, 1.689453125, 0 } },
  { "Canon", "EOS R5", Flash               , -1, { 2.146484375, 1, 1.662109375, 0 } },
  { "Canon", "EOS R5", Flash               , 0, { 2.1787109375, 1, 1.6328125, 0 } },
  { "Canon", "EOS R5", Flash               , 1, { 2.216796875, 1, 1.60546875, 0 } },
  { "Canon", "EOS R5", Flash               , 2, { 2.2607421875, 1, 1.5751953125, 0 } },
  { "Canon", "EOS R5", Flash               , 3, { 2.2958984375, 1, 1.548828125, 0 } },
  { "Canon", "EOS R5", Flash               , 4, { 2.322265625, 1, 1.5234375, 0 } },
  { "Canon", "EOS R5", Flash               , 5, { 2.3544921875, 1, 1.4990234375, 0 } },
  { "Canon", "EOS R5", Flash               , 6, { 2.38671875, 1, 1.4716796875, 0 } },
  { "Canon", "EOS R5", Flash               , 7, { 2.4267578125, 1, 1.4443359375, 0 } },
  { "Canon", "EOS R5", Flash               , 8, { 2.4560546875, 1, 1.421875, 0 } },
  { "Canon", "EOS R5", Flash               , 9, { 2.4912109375, 1, 1.400390625, 0 } },

  { "Canon", "EOS R6", Daylight                   , 0, { 1.8681640625, 1, 1.53515625, 0 } },
  { "Canon", "EOS R6", Shade                      , 0, { 2.16015625, 1, 1.3193359375, 0 } },
  { "Canon", "EOS R6", Cloudy                     , 0, { 2.015625, 1, 1.421875, 0 } },
  { "Canon", "EOS R6", Tungsten                   , 0, { 1.31640625, 1, 2.240234375, 0 } },
  { "Canon", "EOS R6", CoolWhiteFluorescent       , 0, { 1.5634765625, 1, 2.169921875, 0 } },
  { "Canon", "EOS R6", WhiteFluorescent           , 0, { 1.5634765625, 1, 2.169921875, 0 } },
  { "Canon", "EOS R6", Flash                      , 0, { 2.060546875, 1, 1.3974609375, 0 } },

  { "Fujifilm", "FinePix E900", Daylight, 0,		{ 1.571875, 1, 1.128125, 0 } },
  { "Fujifilm", "FinePix E900", Shade, 0,		{ 1.668750, 1, 1.006250, 0 } },
  { "Fujifilm", "FinePix E900", DaylightFluorescent, 0,	{ 1.907609, 1, 1.016304, 0 } },
  { "Fujifilm", "FinePix E900", WarmWhiteFluorescent, 0, { 1.654891, 1, 1.241848, 0 } },
  { "Fujifilm", "FinePix E900", CoolWhiteFluorescent, 0, { 1.554348, 1, 1.519022, 0 } },
  { "Fujifilm", "FinePix E900", Incandescent, 0,	{ 1.037611, 1, 1.842920, 0 } },

  { "Fujifilm", "FinePix F700", Daylight, 0,		{ 1.725000, 1, 1.500000, 0 } },
  { "Fujifilm", "FinePix F700", Shade, 0,		{ 1.950000, 1, 1.325000, 0 } },
  { "Fujifilm", "FinePix F700", DaylightFluorescent, 0,	{ 2.032609, 1, 1.336957, 0 } },
  { "Fujifilm", "FinePix F700", WarmWhiteFluorescent, 0, { 1.706522, 1, 1.663043, 0 } },
  { "Fujifilm", "FinePix F700", CoolWhiteFluorescent, 0, { 1.684783, 1, 2.152174, 0 } },
  { "Fujifilm", "FinePix F700", Incandescent, 0,	{ 1.168142, 1, 2.477876, 0 } },

  { "Fujifilm", "FinePix HS20EXR", Daylight, 0,		{ 1.4107, 1, 1.9702, 0 } },
  { "Fujifilm", "FinePix HS20EXR", Shade, 0,		{ 1.5804, 1, 1.7440, 0 } },
  { "Fujifilm", "FinePix HS20EXR", DaylightFluorescent, 0, { 1.7292, 1, 1.7470, 0 } },
  { "Fujifilm", "FinePix HS20EXR", WarmWhiteFluorescent, 0, { 1.4821, 1, 2.0476, 0 } },
  { "Fujifilm", "FinePix HS20EXR", CoolWhiteFluorescent, 0, { 1.5625, 1, 2.5714, 0 } },
  { "Fujifilm", "FinePix HS20EXR", Incandescent, 0,	{ 1, 1.0633, 2.9430, 0 } },

  { "Fujifilm", "FinePix HS50EXR", Daylight, 0,			{ 1.589286, 1, 1.892857, 0 } },
  { "Fujifilm", "FinePix HS50EXR", Cloudy, 0,			{ 1.705357, 1, 1.663690, 0 } },
  { "Fujifilm", "FinePix HS50EXR", DaylightFluorescent, 0,	{ 1.875000, 1, 1.648810, 0 } },
  { "Fujifilm", "FinePix HS50EXR", DayWhiteFluorescent, 0,	{ 1.619048, 1, 1.946429, 0 } },
  { "Fujifilm", "FinePix HS50EXR", WhiteFluorescent, 0,		{ 1.651786, 1, 2.464286, 0 } },
  { "Fujifilm", "FinePix HS50EXR", Incandescent, 0,		{ 1.041667, 1, 2.601190, 0 } },

  { "Fujifilm", "FinePix S100FS", Daylight, 0,		{ 1.702381, 1, 1.845238, 0 } },
  { "Fujifilm", "FinePix S100FS", Shade, 0,		{ 1.830357, 1, 1.601190, 0 } },
  { "Fujifilm", "FinePix S100FS", DaylightFluorescent, 0, { 1.895833, 1, 1.461309, 0 } },
  { "Fujifilm", "FinePix S100FS", WarmWhiteFluorescent, 0, { 1.574405, 1, 1.818452, 0 } },
  { "Fujifilm", "FinePix S100FS", CoolWhiteFluorescent, 0, { 1.663690, 1, 2.309524, 0 } },
  { "Fujifilm", "FinePix S100FS", Incandescent, 0,	{ 1.107143, 1, 2.815476, 0 } },

  { "Fujifilm", "FinePix S20Pro", Daylight, 0,		{ 1.712500, 1, 1.500000, 0 } },
  { "Fujifilm", "FinePix S20Pro", Cloudy, 0,		{ 1.887500, 1, 1.262500, 0 } },
  { "Fujifilm", "FinePix S20Pro", DaylightFluorescent, 0, { 2.097826, 1, 1.304348, 0 } },
  { "Fujifilm", "FinePix S20Pro", WarmWhiteFluorescent, 0, { 1.782609, 1, 1.619565, 0 } },
  { "Fujifilm", "FinePix S20Pro", CoolWhiteFluorescent, 0, { 1.670213, 1, 2.063830, 0 } },
  { "Fujifilm", "FinePix S20Pro", Incandescent, 0,	{ 1.069565, 1, 2.486957, 0 } },

  { "Fujifilm", "FinePix S2Pro", Daylight, 0,		{ 1.509804, 1, 1.401961, 0 } },
  { "Fujifilm", "FinePix S2Pro", Cloudy, 0,		{ 1.666667, 1, 1.166667, 0 } },
  { "Fujifilm", "FinePix S2Pro", Flash, 0,		{ 1, 1.014084, 2.542253, 0 } },
  { "Fujifilm", "FinePix S2Pro", DaylightFluorescent, 0, { 1.948718, 1, 1.230769, 0 } },
  { "Fujifilm", "FinePix S2Pro", WarmWhiteFluorescent, 0, { 1.675214, 1, 1.572650, 0 } },
  { "Fujifilm", "FinePix S2Pro", CoolWhiteFluorescent, 0, { 1.649573, 1, 2.094017, 0 } },

  { "Fujifilm", "FinePix S5000", Incandescent, 0,	{ 1.212081, 1, 2.672364, 0 } },
  { "Fujifilm", "FinePix S5000", Fluorescent, 0,	{ 1.772316, 1, 2.349902, 0 } },
  { "Fujifilm", "FinePix S5000", Daylight, 0,		{ 1.860403, 1, 1.515946, 0 } },
  { "Fujifilm", "FinePix S5000", Flash, 0,		{ 2.202181, 1, 1.423284, 0 } },
  { "Fujifilm", "FinePix S5000", Cloudy, 0,		{ 2.036578, 1, 1.382513, 0 } },
  { "Fujifilm", "FinePix S5000", Shade, 0,		{ 2.357215, 1, 1.212016, 0 } },

  { "Fujifilm", "FinePix S5200", Daylight, 0,		{ 1.587500, 1, 1.381250, 0 } },
  { "Fujifilm", "FinePix S5200", Shade, 0,		{ 1.946875, 1, 1.175000, 0 } },
  { "Fujifilm", "FinePix S5200", DaylightFluorescent, 0, { 1.948370, 1, 1.187500, 0 } },
  { "Fujifilm", "FinePix S5200", WarmWhiteFluorescent, 0, { 1.682065, 1, 1.437500, 0 } },
  { "Fujifilm", "FinePix S5200", CoolWhiteFluorescent, 0, { 1.595109, 1, 1.839674, 0 } },
  { "Fujifilm", "FinePix S5200", Incandescent, 0,	{ 1.077434, 1, 2.170354, 0 } },

  { "Fujifilm", "FinePix S5500", Daylight, 0,		{ 1.712500, 1, 1.550000, 0 } },
  { "Fujifilm", "FinePix S5500", Shade, 0,		{ 1.912500, 1, 1.375000, 0 } },
  { "Fujifilm", "FinePix S5500", DaylightFluorescent, 0, { 1.978261, 1, 1.380435, 0 } },
  { "Fujifilm", "FinePix S5500", WarmWhiteFluorescent, 0, { 1.673913, 1, 1.673913, 0 } },
  { "Fujifilm", "FinePix S5500", CoolWhiteFluorescent, 0, { 1.663043, 1, 2.163043, 0 } },
  { "Fujifilm", "FinePix S5500", Incandescent, 0,	{ 1.115044, 1, 2.566372, 0 } },

  { "Fujifilm", "FinePix S5600", Daylight, 0,		{ 1.587500, 1, 1.381250, 0 } },
  { "Fujifilm", "FinePix S5600", Shade, 0,		{ 1.946875, 1, 1.175000, 0 } },
  { "Fujifilm", "FinePix S5600", DaylightFluorescent, 0, { 1.948370, 1, 1.187500, 0 } },
  { "Fujifilm", "FinePix S5600", WarmWhiteFluorescent, 0, { 1.682065, 1, 1.437500, 0 } },
  { "Fujifilm", "FinePix S5600", CoolWhiteFluorescent, 0, { 1.595109, 1, 1.839674, 0 } },
  { "Fujifilm", "FinePix S5600", Incandescent, 0,	{ 1.077434, 1, 2.170354, 0 } },

  { "Fujifilm", "FinePix S6000fd", Daylight, 0,		{ 1.511905, 1, 1.431548, 0 } },
  { "Fujifilm", "FinePix S6000fd", Shade, 0,		{ 1.699405, 1, 1.232143, 0 } },
  { "Fujifilm", "FinePix S6000fd", DaylightFluorescent, 0, { 1.866071, 1, 1.309524, 0 } },
  { "Fujifilm", "FinePix S6000fd", WarmWhiteFluorescent, 0, { 1.568452, 1, 1.627976, 0 } },
  { "Fujifilm", "FinePix S6000fd", CoolWhiteFluorescent, 0, { 1.598214, 1, 2.038691, 0 } },
  { "Fujifilm", "FinePix S6000fd", Incandescent, 0,	{ 1, 1.024390, 2.466463, 0 } },

  { "Fujifilm", "FinePix S6500fd", Daylight, 0,		{ 1.398810, 1, 1.470238, 0 } },
  { "Fujifilm", "FinePix S6500fd", Shade, 0,		{ 1.580357, 1, 1.270833, 0 } },
  { "Fujifilm", "FinePix S6500fd", DaylightFluorescent, 0, { 1.735119, 1, 1.348214, 0 } },
  { "Fujifilm", "FinePix S6500fd", WarmWhiteFluorescent, 0, { 1.455357, 1, 1.672619, 0 } },
  { "Fujifilm", "FinePix S6500fd", CoolWhiteFluorescent, 0, { 1.482143, 1, 2.089286, 0 } },
  { "Fujifilm", "FinePix S6500fd", Incandescent, 0,	{ 1, 1.123746, 2.769231, 0 } },

  { "Fujifilm", "FinePix S7000", Daylight, 0,		{ 1.900000, 1, 1.525000, 0 } },
  { "Fujifilm", "FinePix S7000", Shade, 0,		{ 2.137500, 1, 1.350000, 0 } },
  { "Fujifilm", "FinePix S7000", DaylightFluorescent, 0, { 2.315217, 1, 1.347826, 0 } },
  { "Fujifilm", "FinePix S7000", WarmWhiteFluorescent, 0, { 1.902174, 1, 1.663043, 0 } },
  { "Fujifilm", "FinePix S7000", CoolWhiteFluorescent, 0, { 1.836957, 1, 2.130435, 0 } },
  { "Fujifilm", "FinePix S7000", Incandescent, 0,	{ 1.221239, 1, 2.548673, 0 } },

  { "Fujifilm", "FinePix S9100", Daylight, 0,		{ 1.506250, 1, 1.318750, 0 } },
  { "Fujifilm", "FinePix S9100", Cloudy, 0,		{ 1.587500, 1, 1.128125, 0 } },
  { "Fujifilm", "FinePix S9100", DaylightFluorescent, 0, { 1.777174, 1, 1.138587, 0 } },
  { "Fujifilm", "FinePix S9100", WarmWhiteFluorescent, 0, { 1.521739, 1, 1.380435, 0 } },
  { "Fujifilm", "FinePix S9100", CoolWhiteFluorescent, 0, { 1.437500, 1, 1.720109, 0 } },
  { "Fujifilm", "FinePix S9100", Incandescent, 0,	{ 1, 1.024943, 2.113379, 0 } },

  { "Fujifilm", "FinePix S9500", Daylight, 0,		{ 1.618750, 1, 1.231250, 0 } },
  { "Fujifilm", "FinePix S9500", Cloudy, 0,		{ 1.700000, 1, 1.046875, 0 } },
  { "Fujifilm", "FinePix S9500", DaylightFluorescent, 0, { 1.902174, 1, 1.057065, 0 } },
  { "Fujifilm", "FinePix S9500", WarmWhiteFluorescent, 0, { 1.633152, 1, 1.293478, 0 } },
  { "Fujifilm", "FinePix S9500", CoolWhiteFluorescent, 0, { 1.546196, 1, 1.622283, 0 } },
  { "Fujifilm", "FinePix S9500", Incandescent, 0,	{ 1.064159, 1, 1.960177, 0 } },

  { "Fujifilm", "FinePix S9600", Daylight, 0,		{ 1.534375, 1, 1.300000, 0 } },
  { "Fujifilm", "FinePix S9600", Shade, 0,		{ 1.615625, 1, 1.112500, 0 } },
  { "Fujifilm", "FinePix S9600", DaylightFluorescent, 0, { 1.809783, 1, 1.122283, 0 } },
  { "Fujifilm", "FinePix S9600", WarmWhiteFluorescent, 0, { 1.551630, 1, 1.361413, 0 } },
  { "Fujifilm", "FinePix S9600", CoolWhiteFluorescent, 0, { 1.467391, 1, 1.692935, 0 } },
  { "Fujifilm", "FinePix S9600", Incandescent, 0,	{ 1, 1.004444, 2.040000, 0 } },

  { "Fujifilm", "FinePix X100", Daylight, 0,		{ 1.4503, 1, 1.5033, 0 } },
  { "Fujifilm", "FinePix X100", Shade, 0,		{ 1.5861, 1, 1.2947, 0 } },
  { "Fujifilm", "FinePix X100", DaylightFluorescent, 0,	{ 1.8841, 1, 1.3179, 0 } },
  { "Fujifilm", "FinePix X100", WarmWhiteFluorescent, 0, { 1.6291, 1, 1.5927, 0 } },
  { "Fujifilm", "FinePix X100", CoolWhiteFluorescent, 0, { 1.5662, 1, 2.0265, 0 } },
  { "Fujifilm", "FinePix X100", Incandescent, 0,	{ 1, 1.0272, 2.4966, 0 } },
  { "Fujifilm", "FinePix X100", Underwater, 0,		{ 1.4603, 1, 1.5662, 0 } },
  { "Fujifilm", "FinePix X100", "2700K", 0,		{ 1, 1.3790, 3.8447, 0 } },
  { "Fujifilm", "FinePix X100", "3000K", 0,		{ 1, 1.1353, 2.7180, 0 } },
  { "Fujifilm", "FinePix X100", "3300K", 0,		{ 1.0066, 1, 2.1060, 0 } },
  { "Fujifilm", "FinePix X100", "5000K", 0,		{ 1.4536, 1, 1.3742, 0 } },

  /* Fujifilm GFX100S firmware 1.00 */
  { "Fujifilm", "GFX100S", Daylight           , 0, { 1.6920529801324504, 1, 1.8708609271523178, 0 } },
  { "Fujifilm", "GFX100S", Cloudy             , 0, { 1.8509933774834437, 1, 1.5927152317880795, 0 } },
  { "Fujifilm", "GFX100S", Incandescent       , 0, { 1.1258278145695364, 1, 2.9304635761589406, 0 } },
  { "Fujifilm", "GFX100S", DayWhiteFluorescent, 0, { 1.8576158940397351, 1, 1.9205298013245033, 0 } },
  { "Fujifilm", "GFX100S", DaylightFluorescent, 0, { 2.19205298013245, 1, 1.5827814569536425, 0 } },
  { "Fujifilm", "GFX100S", WhiteFluorescent   , 0, { 1.7748344370860927, 1, 2.4966887417218544, 0 } },
  { "Fujifilm", "GFX100S", Underwater         , 0, { 1.6887417218543046, 1, 1.880794701986755, 0 } },

  /* Fujifilm X100S firmware 1.20 */
  { "Fujifilm", "X100S", Daylight, 0,		{ 1.874172, 1, 1.490066, 0 } },
  { "Fujifilm", "X100S", Shade, 0,		{ 2.049669, 1, 1.271523, 0 } },
  { "Fujifilm", "X100S", DaylightFluorescent, 0, { 2.344371, 1, 1.264901, 0 } },
  { "Fujifilm", "X100S", WarmWhiteFluorescent, 0, { 2.000000, 1, 1.509934, 0 } },
  { "Fujifilm", "X100S", CoolWhiteFluorescent, 0, { 1.874172, 1, 1.913907, 0 } },
  { "Fujifilm", "X100S", Incandescent, 0,	{ 1.198675, 1, 2.341060, 0 } },
  { "Fujifilm", "X100S", Underwater, 0,		{ 1.874172, 1, 1.490066, 0 } },
  { "Fujifilm", "X100S", "2500K", 0,		{ 1, 1.074733, 3.597865, 0 } },
  { "Fujifilm", "X100S", "2700K", 0,		{ 1, 1.003322, 2.950166, 0 } },
  { "Fujifilm", "X100S", "3000K", 0,		{ 1.119205, 1, 2.509934, 0 } },
  { "Fujifilm", "X100S", "3300K", 0,		{ 1.241722, 1, 2.218543, 0 } },
  { "Fujifilm", "X100S", "5000K", 0,		{ 1.774835, 1, 1.529801, 0 } },
  { "Fujifilm", "X100S", "5600K", 0,		{ 1.900662, 1, 1.423841, 0 } },
  { "Fujifilm", "X100S", "6300K", 0,		{ 2.019868, 1, 1.334437, 0 } },
  { "Fujifilm", "X100S", "6700K", 0,		{ 2.076159, 1, 1.291391, 0 } },
  { "Fujifilm", "X100S", "10000K", 0,		{ 2.360927, 1, 1.079470, 0 } },

  { "Fujifilm", "X100T", Daylight           , 0, { 1.930464, 1, 1.539735, 0 } },
  { "Fujifilm", "X100T", Cloudy             , 0, { 2.109272, 1, 1.324503, 0 } },
  { "Fujifilm", "X100T", Incandescent       , 0, { 1.238411, 1, 2.307947, 0 } },
  { "Fujifilm", "X100T", DayWhiteFluorescent, 0, { 2.102649, 1, 1.589404, 0 } },
  { "Fujifilm", "X100T", DaylightFluorescent, 0, { 2.456954, 1, 1.334437, 0 } },
  { "Fujifilm", "X100T", WhiteFluorescent   , 0, { 2.013245, 1, 1.970199, 0 } },
  { "Fujifilm", "X100T", Underwater         , 0, { 1.927152, 1, 1.549669, 0 } },

  { "Fujifilm", "X100F", Daylight, 0,            { 1.9503311258278146, 1, 1.8543046357615893, 0 } },
  { "Fujifilm", "X100F", Cloudy, 0,              { 2.142384105960265, 1, 1.5927152317880795, 0 } },
  { "Fujifilm", "X100F", DaylightFluorescent, 0, { 2.486754966887417, 1, 1.6291390728476822, 0 } },
  { "Fujifilm", "X100F", DayWhiteFluorescent, 0, { 2.0827814569536423, 1, 1.9834437086092715, 0 } },
  { "Fujifilm", "X100F", WhiteFluorescent, 0,    { 1.9569536423841059, 1, 2.5596026490066226, 0 } },
  { "Fujifilm", "X100F", Incandescent, 0,        { 1.271523178807947, 1, 2.8311258278145695, 0 } },
  { "Fujifilm", "X100F", Underwater, 0,          { 1.9503311258278146, 1, 1.8543046357615893, 0 } },


  /* Fujifilm X100V firmware 1.10 */
  { "Fujifilm", "X100V", Daylight, 0,                 { 1.784768, 1, 1.768212, 0 } },
  { "Fujifilm", "X100V", Cloudy, 0,                   { 1.94702, 1, 1.516556, 0 } },
  { "Fujifilm", "X100V", DaylightFluorescent, 0,     { 2.327815, 1, 1.536424, 0 } },
  { "Fujifilm", "X100V", DayWhiteFluorescent, 0,    { 1.980132, 1, 1.907285, 0 } },
  { "Fujifilm", "X100V", WhiteFluorescent, 0,        { 1.887417, 1, 2.42053, 0 } },
  { "Fujifilm", "X100V", Incandescent, 0,             { 1.192053, 1, 2.701987, 0 } },
  { "Fujifilm", "X100V", Underwater, 0,               { 1.778146, 1, 1.751656, 0 } },


  { "Fujifilm", "X20", Daylight           , 0, { 1.688742, 1, 1.850993, 0 } },
  { "Fujifilm", "X20", Cloudy             , 0, { 1.827815, 1, 1.622517, 0 } },
  { "Fujifilm", "X20", Incandescent       , 0, { 1.066225, 1, 2.605960, 0 } },
  { "Fujifilm", "X20", DayWhiteFluorescent, 0, { 1.754967, 1, 1.864238, 0 } },
  { "Fujifilm", "X20", DaylightFluorescent, 0, { 2.052980, 1, 1.625828, 0 } },
  { "Fujifilm", "X20", WhiteFluorescent   , 0, { 1.754967, 1, 2.341060, 0 } },
  { "Fujifilm", "X20", Underwater         , 0, { 1.688742, 1, 1.850993, 0 } },

  { "Fujifilm", "X70", Daylight           , 0, { 2.009934, 1, 1.513245, 0 } },
  { "Fujifilm", "X70", Cloudy             , 0, { 2.211921, 1, 1.311258, 0 } },
  { "Fujifilm", "X70", Incandescent       , 0, { 1.314570, 1, 2.251656, 0 } },
  { "Fujifilm", "X70", DayWhiteFluorescent, 0, { 2.188742, 1, 1.566225, 0 } },
  { "Fujifilm", "X70", DaylightFluorescent, 0, { 2.619205, 1, 1.311258, 0 } },
  { "Fujifilm", "X70", WhiteFluorescent   , 0, { 2.056291, 1, 2.013245, 0 } },
  { "Fujifilm", "X70", Underwater         , 0, { 2.006623, 1, 1.533113, 0 } },

  { "Fujifilm", "X-M1", Daylight, 0,                  { 1.943709, 1, 1.824503, 0 } },
  { "Fujifilm", "X-M1", Cloudy, 0,                    { 2.139073, 1, 1.599338, 0 } },
  { "Fujifilm", "X-M1", DaylightFluorescent, 0,      { 2.440397, 1, 1.592715, 0 } },
  { "Fujifilm", "X-M1", DayWhiteFluorescent, 0,     { 2.066225, 1, 1.870861, 0 } },
  { "Fujifilm", "X-M1", WhiteFluorescent, 0,         { 1.937086, 1, 2.360927, 0 } },
  { "Fujifilm", "X-M1", Incandescent, 0,              { 1.225166, 1, 2.976821, 0 } },

  { "Fujifilm", "X-A2", Daylight, 0,                  { 1.824503, 1, 1.758278, 0 } },
  { "Fujifilm", "X-A2", Cloudy, 0,                    { 2.033113, 1, 1.496689, 0 } },
  { "Fujifilm", "X-A2", DaylightFluorescent, 0,      { 2.327815, 1, 1.5, 0 } },
  { "Fujifilm", "X-A2", DayWhiteFluorescent, 0,     { 1.986755, 1, 1.804636, 0 } },
  { "Fujifilm", "X-A2", WhiteFluorescent, 0,         { 1.900662, 1, 2.284768, 0 } },
  { "Fujifilm", "X-A2", Incandescent, 0,              { 1.162252, 1, 2.864238, 0 } },

  /* Fujifilm X-E1 Firmware Version 1.01 */
  // The manual calls it "Direct sunlight".
  { "Fujifilm", "X-E1", Daylight, 0,		{ 1.821192, 1, 1.804636, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-E1", Shade, 0,		{ 2.039735, 1, 1.572848, 0 } },
  { "Fujifilm", "X-E1", DaylightFluorescent, 0,	{ 2.321192, 1, 1.566225, 0 } },
  { "Fujifilm", "X-E1", WarmWhiteFluorescent, 0, { 1.966887, 1, 1.837748, 0 } },
  { "Fujifilm", "X-E1", CoolWhiteFluorescent, 0, { 1.847682, 1, 2.317881, 0 } },
  { "Fujifilm", "X-E1", Incandescent, 0,	{ 1.165563, 1, 2.920530, 0 } },
  { "Fujifilm", "X-E1", Underwater, 0,		{ 1.821192, 1, 1.804636, 0 } },
  { "Fujifilm", "X-E1", "5000K", 0,		{ 1.814570, 1, 1.731788, 0 } },

  /* Fujifilm X-E2 Firmware Version 4.10 */
  { "Fujifilm", "X-E2", Daylight, 0,		{ 1.8576158940397351, 1, 1.5496688741721854, 0 } },
  { "Fujifilm", "X-E2", Shade, 0,		{ 2.056291390728477, 1, 1.3410596026490067, 0 } },
  { "Fujifilm", "X-E2", DaylightFluorescent, 0,	{ 2.3874172185430464, 1, 1.3377483443708609, 0 } },
  { "Fujifilm", "X-E2", WarmWhiteFluorescent, 0,{ 2.0298013245033113, 1, 1.576158940397351, 0 } },
  { "Fujifilm", "X-E2", CoolWhiteFluorescent, 0,{ 1.903973509933775, 1, 1.9701986754966887, 0 } },
  { "Fujifilm", "X-E2", Incandescent, 0,	{ 1.1986754966887416, 1, 2.3642384105960264, 0 } },
  { "Fujifilm", "X-E2", Underwater, 0,		{ 1.8576158940397351, 1, 1.5496688741721854, 0 } },

  /* Fujifilm X-E3 Firmware Version 1.22 */
  { "Fujifilm", "X-E3", Daylight, 0, { 1.9503311258278146, 1, 1.794701986754967, 0 } },
  { "Fujifilm", "X-E3", Cloudy, 0, { 2.1589403973509933, 1, 1.5264900662251655, 0 } },
  { "Fujifilm", "X-E3", Incandescent, 0, { 1.2649006622516556, 1, 2.7913907284768213, 0 } },
  { "Fujifilm", "X-E3", DayWhiteFluorescent, 0, { 2.0860927152317883, 1, 1.8708609271523178, 0 } },
  { "Fujifilm", "X-E3", DaylightFluorescent, 0, { 2.4701986754966887, 1, 1.4933774834437086, 0 } },
  { "Fujifilm", "X-E3", WhiteFluorescent, 0, { 1.9834437086092715, 1, 2.4900662251655628, 0 } },
  { "Fujifilm", "X-E3", Underwater, 0, { 1.9503311258278146, 1, 1.8311258278145695, 0 } },

  { "Fujifilm", "X-E4", Daylight           , -9, { 1.9172185430463575, 1, 1.0596026490066226, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -8, { 1.9172185430463575, 1, 1.1986754966887416, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -7, { 1.9172185430463575, 1, 1.3609271523178808, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -6, { 1.9172185430463575, 1, 1.4437086092715232, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -5, { 1.9172185430463575, 1, 1.5364238410596027, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -4, { 1.9172185430463575, 1, 1.6324503311258278, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -3, { 1.9172185430463575, 1, 1.6821192052980132, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -2, { 1.9172185430463575, 1, 1.718543046357616, 0 } },
  { "Fujifilm", "X-E4", Daylight           , -1, { 1.9172185430463575, 1, 1.7847682119205297, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 0, { 1.9172185430463575, 1, 1.8476821192052981, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 1, { 1.9172185430463575, 1, 1.8774834437086092, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 2, { 1.9172185430463575, 1, 1.9205298013245033, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 3, { 1.9172185430463575, 1, 1.9768211920529801, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 4, { 1.9172185430463575, 1, 2.019867549668874, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 5, { 1.9172185430463575, 1, 2.109271523178808, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 6, { 1.9172185430463575, 1, 2.2119205298013247, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 7, { 1.9172185430463575, 1, 2.3046357615894038, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 8, { 1.9172185430463575, 1, 2.5231788079470197, 0 } },
  { "Fujifilm", "X-E4", Daylight           , 9, { 1.9172185430463575, 1, 2.7682119205298013, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -9, { 2.092814371257485, 1, 0.9041916167664671, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -8, { 2.0927152317880795, 1, 1.0231788079470199, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -7, { 2.0927152317880795, 1, 1.1622516556291391, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -6, { 2.0927152317880795, 1, 1.2317880794701987, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -5, { 2.0927152317880795, 1, 1.3112582781456954, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -4, { 2.0927152317880795, 1, 1.390728476821192, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -3, { 2.0927152317880795, 1, 1.4337748344370862, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -2, { 2.0927152317880795, 1, 1.466887417218543, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , -1, { 2.0927152317880795, 1, 1.5231788079470199, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 0, { 2.0927152317880795, 1, 1.576158940397351, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 1, { 2.0927152317880795, 1, 1.6026490066225165, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 2, { 2.0927152317880795, 1, 1.6390728476821192, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 3, { 2.0927152317880795, 1, 1.685430463576159, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 4, { 2.0927152317880795, 1, 1.7251655629139073, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 5, { 2.0927152317880795, 1, 1.7980132450331126, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 6, { 2.0927152317880795, 1, 1.8841059602649006, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 7, { 2.0927152317880795, 1, 1.966887417218543, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 8, { 2.0927152317880795, 1, 2.152317880794702, 0 } },
  { "Fujifilm", "X-E4", Cloudy             , 9, { 2.0927152317880795, 1, 2.3609271523178808, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -9, { 1.3013245033112584, 1, 1.6158940397350994, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -8, { 1.3013245033112584, 1, 1.8311258278145695, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -7, { 1.3013245033112584, 1, 2.076158940397351, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -6, { 1.3013245033112584, 1, 2.2019867549668874, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -5, { 1.3013245033112584, 1, 2.3443708609271523, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -4, { 1.3013245033112584, 1, 2.486754966887417, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -3, { 1.3013245033112584, 1, 2.566225165562914, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -2, { 1.3013245033112584, 1, 2.622516556291391, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , -1, { 1.3013245033112584, 1, 2.725165562913907, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 0, { 1.3013245033112584, 1, 2.8178807947019866, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 1, { 1.3013245033112584, 1, 2.8642384105960264, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 2, { 1.3013245033112584, 1, 2.9271523178807946, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 3, { 1.3013245033112584, 1, 3.013245033112583, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 4, { 1.3013245033112584, 1, 3.0827814569536423, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 5, { 1.3013245033112584, 1, 3.218543046357616, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 6, { 1.3013245033112584, 1, 3.370860927152318, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 7, { 1.3013245033112584, 1, 3.5165562913907285, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 8, { 1.3013245033112584, 1, 3.847682119205298, 0 } },
  { "Fujifilm", "X-E4", Incandescent       , 9, { 1.3013245033112584, 1, 4.2218543046357615, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -9, { 2.089403973509934, 1, 1.1225165562913908, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -8, { 2.089403973509934, 1, 1.271523178807947, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -7, { 2.089403973509934, 1, 1.4437086092715232, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -6, { 2.089403973509934, 1, 1.533112582781457, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -5, { 2.089403973509934, 1, 1.6324503311258278, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -4, { 2.089403973509934, 1, 1.7317880794701987, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -3, { 2.089403973509934, 1, 1.7847682119205297, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -2, { 2.089403973509934, 1, 1.8245033112582782, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, -1, { 2.089403973509934, 1, 1.8940397350993377, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 0, { 2.089403973509934, 1, 1.9602649006622517, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 1, { 2.089403973509934, 1, 1.9933774834437086, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 2, { 2.089403973509934, 1, 2.0364238410596025, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 3, { 2.089403973509934, 1, 2.096026490066225, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 4, { 2.089403973509934, 1, 2.1456953642384105, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 5, { 2.089403973509934, 1, 2.23841059602649, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 6, { 2.089403973509934, 1, 2.3443708609271523, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 7, { 2.089403973509934, 1, 2.447019867549669, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 8, { 2.089403973509934, 1, 2.6754966887417218, 0 } },
  { "Fujifilm", "X-E4", DayWhiteFluorescent, 9, { 2.089403973509934, 1, 2.937086092715232, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -9, { 2.4716417910447763, 1, 0.9014925373134328, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -8, { 2.4701986754966887, 1, 1.0198675496688743, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -7, { 2.4701986754966887, 1, 1.1589403973509933, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -6, { 2.4701986754966887, 1, 1.228476821192053, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -5, { 2.4701986754966887, 1, 1.3079470198675496, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -4, { 2.4701986754966887, 1, 1.3874172185430464, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -3, { 2.4701986754966887, 1, 1.4304635761589404, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -2, { 2.4701986754966887, 1, 1.4635761589403973, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, -1, { 2.4701986754966887, 1, 1.5198675496688743, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 0, { 2.4701986754966887, 1, 1.5728476821192052, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 1, { 2.4701986754966887, 1, 1.599337748344371, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 2, { 2.4701986754966887, 1, 1.6357615894039734, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 3, { 2.4701986754966887, 1, 1.6821192052980132, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 4, { 2.4701986754966887, 1, 1.7218543046357615, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 5, { 2.4701986754966887, 1, 1.794701986754967, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 6, { 2.4701986754966887, 1, 1.880794701986755, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 7, { 2.4701986754966887, 1, 1.9635761589403973, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 8, { 2.4701986754966887, 1, 2.1456953642384105, 0 } },
  { "Fujifilm", "X-E4", DaylightFluorescent, 9, { 2.4701986754966887, 1, 2.357615894039735, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -9, { 1.9834437086092715, 1, 1.4304635761589404, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -8, { 1.9834437086092715, 1, 1.6225165562913908, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -7, { 1.9834437086092715, 1, 1.8410596026490067, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -6, { 1.9834437086092715, 1, 1.9503311258278146, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -5, { 1.9834437086092715, 1, 2.076158940397351, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -4, { 1.9834437086092715, 1, 2.205298013245033, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -3, { 1.9834437086092715, 1, 2.271523178807947, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -2, { 1.9834437086092715, 1, 2.3245033112582782, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , -1, { 1.9834437086092715, 1, 2.4139072847682117, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 0, { 1.9834437086092715, 1, 2.4966887417218544, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 1, { 1.9834437086092715, 1, 2.5397350993377485, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 2, { 1.9834437086092715, 1, 2.5927152317880795, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 3, { 1.9834437086092715, 1, 2.6688741721854305, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 4, { 1.9834437086092715, 1, 2.7317880794701987, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 5, { 1.9834437086092715, 1, 2.8509933774834435, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 6, { 1.9834437086092715, 1, 2.986754966887417, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 7, { 1.9834437086092715, 1, 3.115894039735099, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 8, { 1.9834437086092715, 1, 3.4072847682119205, 0 } },
  { "Fujifilm", "X-E4", WhiteFluorescent   , 9, { 1.9834437086092715, 1, 3.7417218543046356, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -9, { 1.9172185430463575, 1, 1.0596026490066226, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -8, { 1.9172185430463575, 1, 1.1986754966887416, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -7, { 1.9172185430463575, 1, 1.3609271523178808, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -6, { 1.9172185430463575, 1, 1.4437086092715232, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -5, { 1.9172185430463575, 1, 1.5364238410596027, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -4, { 1.9172185430463575, 1, 1.6324503311258278, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -3, { 1.9172185430463575, 1, 1.6821192052980132, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -2, { 1.9172185430463575, 1, 1.718543046357616, 0 } },
  { "Fujifilm", "X-E4", Underwater         , -1, { 1.9172185430463575, 1, 1.7847682119205297, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 0, { 1.9172185430463575, 1, 1.8476821192052981, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 1, { 1.9172185430463575, 1, 1.8774834437086092, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 2, { 1.9172185430463575, 1, 1.9205298013245033, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 3, { 1.9172185430463575, 1, 1.9768211920529801, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 4, { 1.9172185430463575, 1, 2.019867549668874, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 5, { 1.9172185430463575, 1, 2.109271523178808, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 6, { 1.9172185430463575, 1, 2.2119205298013247, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 7, { 1.9172185430463575, 1, 2.3046357615894038, 0 } },
  { "Fujifilm", "X-E4", Underwater         , 8, { 1.9172185430463575, 1, 2.5231788079470197, 0 } },

  /* FUJIFILM X-S10 Firmware Version Digital Camera X-S10 Ver1.00 */
  { "Fujifilm", "X-S10", Daylight           , 0, { 1.8576158940397351, 1, 1.8940397350993377, 0 } },
  { "Fujifilm", "X-S10", Cloudy             , 0, { 2.0231788079470197, 1, 1.6258278145695364, 0 } },
  { "Fujifilm", "X-S10", Incandescent       , 0, { 1.2483443708609272, 1, 2.8675496688741724, 0 } },
  { "Fujifilm", "X-S10", DayWhiteFluorescent, 0, { 2.0264900662251657, 1, 2.0496688741721854, 0 } },
  { "Fujifilm", "X-S10", DaylightFluorescent, 0, { 2.3874172185430464, 1, 1.6556291390728477, 0 } },
  { "Fujifilm", "X-S10", WhiteFluorescent   , 0, { 1.9437086092715232, 1, 2.5794701986754967, 0 } },
  { "Fujifilm", "X-S10", Underwater         , 0, { 1.8509933774834437, 1, 1.900662251655629, 0 } },

  /* Fujifilm X-T1 Firmware Version 1.10 */
  { "Fujifilm", "X-T1", Daylight, 0,          { 1.89404, 1, 1.569536, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-T1", Shade, 0,            { 2.092715, 1, 1.347682, 0 } },
  { "Fujifilm", "X-T1", DaylightFluorescent, 0,{ 2.39404, 1, 1.377483, 0 } },
  { "Fujifilm", "X-T1", WarmWhiteFluorescent, 0,{ 2.023179, 1, 1.622517, 0 } },
  { "Fujifilm", "X-T1", CoolWhiteFluorescent, 0,{ 1.94702, 1, 2.066225, 0 } },
  { "Fujifilm", "X-T1", Incandescent, 0,      { 1.268212, 1, 2.284768, 0 } },
  { "Fujifilm", "X-T1", Underwater, 0,        { 1.89404, 1, 1.569536, 0 } },
  { "Fujifilm", "X-T1", "5000K", 0,             { 1.847682, 1, 1.546358, 0 } },

  { "Fujifilm", "X-T2", Daylight, 0, { 1.877483, 1, 1.940397, 0 } },
  { "Fujifilm", "X-T2", Shade, 0, { 2.086093, 1, 1.672185, 0 } },
  { "Fujifilm", "X-T2", DaylightFluorescent, 0, { 2.360927, 1, 1.668874, 0 } },
  { "Fujifilm", "X-T2", DayWhiteFluorescent, 0, { 1.976821, 1, 2.05298, 0 } },
  { "Fujifilm", "X-T2", WhiteFluorescent, 0, { 1.86755, 1, 2.639073, 0 } },
  { "Fujifilm", "X-T2", Incandescent, 0, { 1.248344, 1, 2.907285, 0 } },
  { "Fujifilm", "X-T2", Underwater, 0, { 1.877483, 1, 1.940397, 0 } },

  { "Fujifilm", "X-T3", Daylight           , 0, { 1.7549668874172186, 1, 1.8642384105960266, 0 } },
  { "Fujifilm", "X-T3", Cloudy             , 0, { 1.9105960264900663, 1, 1.6059602649006623, 0 } },
  { "Fujifilm", "X-T3", Incandescent       , 0, { 1.1821192052980132, 1, 2.801324503311258, 0 } },
  { "Fujifilm", "X-T3", DayWhiteFluorescent, 0, { 1.9105960264900663, 1, 2.0165562913907285, 0 } },
  { "Fujifilm", "X-T3", DaylightFluorescent, 0, { 2.251655629139073, 1, 1.6357615894039734, 0 } },
  { "Fujifilm", "X-T3", WhiteFluorescent   , 0, { 1.8311258278145695, 1, 2.5264900662251657, 0 } },
  { "Fujifilm", "X-T3", Underwater         , 0, { 1.7516556291390728, 1, 1.8774834437086092, 0 } },

  { "Fujifilm", "X-T4", Daylight           , 0, { 1.8377483443708609, 1, 1.8278145695364238, 0 } },
  { "Fujifilm", "X-T4", Cloudy             , 0, { 2.0033112582781456, 1, 1.576158940397351, 0 } },
  { "Fujifilm", "X-T4", Incandescent       , 0, { 1.2384105960264902, 1, 2.8046357615894038, 0 } },
  { "Fujifilm", "X-T4", DayWhiteFluorescent, 0, { 2.006622516556291, 1, 1.9966887417218544, 0 } },
  { "Fujifilm", "X-T4", DaylightFluorescent, 0, { 2.3609271523178808, 1, 1.6059602649006623, 0 } },
  { "Fujifilm", "X-T4", WhiteFluorescent   , 0, { 1.9205298013245033, 1, 2.519867549668874, 0 } },
  { "Fujifilm", "X-T4", Underwater         , 0, { 1.8377483443708609, 1, 1.8278145695364238, 0 } },

  /* Fujifilm X-T10 Firmware Version 1.21 */
  { "Fujifilm", "X-T10", Daylight, 0,		{ 1.884106, 1, 1.526490, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-T10", Shade, 0,		{ 2.072848, 1, 1.324503, 0 } },
  { "Fujifilm", "X-T10", DaylightFluorescent, 0, { 2.423841, 1, 1.334437, 0 } },
  { "Fujifilm", "X-T10", WarmWhiteFluorescent, 0, { 2.026490, 1, 1.599338, 0 } },
  { "Fujifilm", "X-T10", CoolWhiteFluorescent, 0, { 1.890728, 1, 2.046358, 0 } },
  { "Fujifilm", "X-T10", Incandescent, 0,	{ 1.211921, 1, 2.321192, 0 } },
  { "Fujifilm", "X-T10", Underwater, 0,		{ 1.990066, 1, 1.440397, 0 } },

  { "Fujifilm", "X-T20", Daylight           , 0, { 1.880794701986755, 1, 1.8741721854304636, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-T20", Shade              , 0, { 2.089403973509934, 1, 1.6026490066225165, 0 } },
  { "Fujifilm", "X-T20", Incandescent       , 0, { 1.2152317880794703, 1, 2.857615894039735, 0 } },
  { "Fujifilm", "X-T20", DayWhiteFluorescent, 0, { 1.9834437086092715, 1, 1.966887417218543, 0 } },
  { "Fujifilm", "X-T20", DaylightFluorescent, 0, { 2.3675496688741724, 1, 1.576158940397351, 0 } },
  { "Fujifilm", "X-T20", WhiteFluorescent   , 0, { 1.9072847682119205, 1, 2.5794701986754967, 0 } },
  { "Fujifilm", "X-T20", Underwater         , 0, { 1.880794701986755, 1, 1.8741721854304636, 0 } },

  /* Fujifilm X-T30 Firmware Version v1.20 */
  { "Fujifilm", "X-T30", Daylight           , 0, { 1.827815, 1, 1.741722, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-T30", Shade              , 0, { 1.986755, 1, 1.493377, 0 } },
  { "Fujifilm", "X-T30", Incandescent       , 0, { 1.231788, 1, 2.639073, 0 } },
  { "Fujifilm", "X-T30", DayWhiteFluorescent, 0, { 1.990066, 1, 1.887417, 0 } },
  { "Fujifilm", "X-T30", DaylightFluorescent, 0, { 2.34106, 1, 1.523179, 0 } },
  { "Fujifilm", "X-T30", WhiteFluorescent   , 0, { 1.907285, 1, 2.374172, 0 } },
  { "Fujifilm", "X-T30", Underwater         , 0, { 1.827815, 1, 1.741722, 0 } },

  /* Fujifilm X-Pro1 Firmware Version v3.30 */
  // The manual calls it "Fine".
  { "Fujifilm", "X-Pro1", Daylight, 0,		{ 1.860927, 1, 1.708609, 0 } },
  // The manual calls it "Shade", but exiftool shows it as "Cloudy".
  { "Fujifilm", "X-Pro1", Shade, 0,		{ 2.105960, 1, 1.460265, 0 } },
  { "Fujifilm", "X-Pro1", DaylightFluorescent, 0, { 2.390728, 1, 1.500000, 0 } },
  { "Fujifilm", "X-Pro1", WarmWhiteFluorescent, 0, { 2.059603, 1, 1.794702, 0 } },
  { "Fujifilm", "X-Pro1", CoolWhiteFluorescent, 0, { 1.966887, 1, 2.254967, 0 } },
  { "Fujifilm", "X-Pro1", Incandescent, 0,	{ 1.225166, 1, 2.708609, 0 } },
  { "Fujifilm", "X-Pro1", Underwater, 0,	{ 1.860927, 1, 1.708609, 0 } },
  { "Fujifilm", "X-Pro1", "5000K", 0,		{ 1.857616, 1, 1.701987, 0 } },

  { "Fujifilm", "X-Pro2", Daylight           , 0, { 1.947020, 1, 1.831126, 0 } },
  { "Fujifilm", "X-Pro2", Cloudy             , 0, { 2.145695, 1, 1.582781, 0 } },
  { "Fujifilm", "X-Pro2", Incandescent       , 0, { 1.264901, 1, 2.754967, 0 } },
  { "Fujifilm", "X-Pro2", DayWhiteFluorescent, 0, { 2.069536, 1, 1.950331, 0 } },
  { "Fujifilm", "X-Pro2", DaylightFluorescent, 0, { 2.463576, 1, 1.602649, 0 } },
  { "Fujifilm", "X-Pro2", WhiteFluorescent   , 0, { 1.973510, 1, 2.513245, 0 } },
  { "Fujifilm", "X-Pro2", Underwater         , 0, { 1.943709, 1, 1.827815, 0 } },

  { "Fujifilm", "X-Pro3", Daylight           , 0, { 1.8675496688741722, 1, 1.8079470198675496, 0 } },
  { "Fujifilm", "X-Pro3", Cloudy             , 0, { 2.013245033112583, 1, 1.5463576158940397, 0 } },
  { "Fujifilm", "X-Pro3", Incandescent       , 0, { 1.2516556291390728, 1, 2.771523178807947, 0 } },
  { "Fujifilm", "X-Pro3", DayWhiteFluorescent, 0, { 2.013245033112583, 1, 1.9701986754966887, 0 } },
  { "Fujifilm", "X-Pro3", DaylightFluorescent, 0, { 2.3642384105960264, 1, 1.576158940397351, 0 } },
  { "Fujifilm", "X-Pro3", WhiteFluorescent   , 0, { 1.9304635761589404, 1, 2.486754966887417, 0 } },
  { "Fujifilm", "X-Pro3", Underwater         , 0, { 1.8675496688741722, 1, 1.8079470198675496, 0 } },

  { "Kodak", "DCS Pro 14N", Daylight, 0,	{ 1, 1.055126, 1.353602, 0 } },
  { "Kodak", "DCS Pro 14N", Tungsten, 0,	{ 1, 1.337688, 2.218852, 0 } },
  { "Kodak", "DCS Pro 14N", Fluorescent, 0,	{ 1, 1.102854, 1.513673, 0 } },
  { "Kodak", "DCS Pro 14N", Flash, 0,		{ 1, 1.055670, 1.353602, 0 } },

  /* Median of 3 DCS Pro SLR/n cameras. */
  { "Kodak", "DCS Pro SLR/n", Daylight, 0,	{ 1, 0.816913, 1.267327, 0 } },
  { "Kodak", "DCS Pro SLR/n", Tungsten, 0,	{ 1, 1.045431, 1.948620, 0 } },
  { "Kodak", "DCS Pro SLR/n", Fluorescent, 0,	{ 1, 0.778411, 1.244228, 0 } },
  { "Kodak", "DCS Pro SLR/n", Flash, 0,		{ 1, 0.789819, 1.224880, 0 } },

  { "Kodak", "DCS560C", Daylight, 0,		{ 1.238959, 1.057305, 2.133333, 0 } },
  { "Kodak", "DCS560C", Tungsten, 0,		{ 1.061690, 1.043833, 2.612245, 0 } },
  { "Kodak", "DCS560C", Fluorescent, 0,		{ 1.463903, 0.994658, 2.081301, 0 } },
  { "Kodak", "DCS560C", Flash, 0,		{ 1.238956, 1.057305, 2.133333, 0 } },

  { "Kodak", "ProBack645", Daylight, 0,		{ 1.122807, 1.001467, 1.368984, 0 } },
  { "Kodak", "ProBack645", Tungsten, 0,		{ 1, 1.204706, 2.409412, 0 } },
  { "Kodak", "ProBack645", Fluorescent, 0,	{ 1.099893, 1, 1.699585, 0 } },
  { "Kodak", "ProBack645", Flash, 0,		{ 1.134626, 1.011858, 1.382849, 0 } },

  { "Kodak", "P850 ZOOM", Daylight, 0,		{ 1.859375, 1, 1.566406, 0 } },
  { "Kodak", "P850 ZOOM", Cloudy, 0,		{ 1.960938, 1, 1.570313, 0 } },
  { "Kodak", "P850 ZOOM", Shade, 0,		{ 2.027344, 1, 1.519531, 0 } },
  { "Kodak", "P850 ZOOM", EveningSun, 0,	{ 1.679688, 1, 1.812500, 0 } },
  { "Kodak", "P850 ZOOM", Tungsten, 0,		{ 1.140625, 1, 2.726563, 0 } },
  { "Kodak", "P850 ZOOM", Fluorescent, 0,	{ 1.113281, 1, 2.949219, 0 } },

  { "Kodak", "Z1015 IS", Daylight, 0,	{ 1.546875, 1, 2.082031, 0 } },
  { "Kodak", "Z1015 IS", Tungsten, 0,	{ 1, 1.024000, 3.384000, 0 } },
  { "Kodak", "Z1015 IS", Fluorescent, 0, { 1.562500, 1, 2.515625, 0 } },
  { "Kodak", "Z1015 IS", Shade, 0,	{ 1.820313, 1, 1.789062, 0 } },

  { "Leica", "M8", Cloudy, 0, { 2.136719, 1, 1.168213, 0 } },
  { "Leica", "M8", Daylight, 0, { 2.007996, 1, 1.268982, 0 } },
  { "Leica", "M8", Flash, 0, { 2.164490, 1, 1.177795, 0 } },
  { "Leica", "M8", Fluorescent, 0, { 1.655579, 1, 2.070374, 0 } },
  { "Leica", "M8", Shade, 0, { 2.197754, 1, 1.111084, 0 } },
  { "Leica", "M8", Tungsten, 0, { 1.160034, 1, 2.028381, 0 } },

  { "Leica", "M9 Digital Camera", Tungsten, 0,	{ 1.321288, 1, 2.077024, 0 } },
  { "Leica", "M9 Digital Camera", Fluorescent, 0, { 1.673827, 1, 1.855043, 0 } },
  { "Leica", "M9 Digital Camera", DaylightFluorescent, 0, { 2.224852, 1, 1.388000, 0 } },
  { "Leica", "M9 Digital Camera", Daylight, 0,	{ 2.013733, 1, 1.364869, 0 } },
  { "Leica", "M9 Digital Camera", Flash, 0,	{ 1.980652, 1, 1.331111, 0 } },
  { "Leica", "M9 Digital Camera", Cloudy, 0,	{ 2.146728, 1, 1.252197, 0 } },
  { "Leica", "M9 Digital Camera", Shade, 0,	{ 2.249268, 1, 1.179015, 0 } },

  { "Leica Camera AG", "R8 - Digital Back DMR", Incandescent, 0, { 1, 1.109985, 2.430664, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", Fluorescent, 0, { 1.234985, 1, 1.791138, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", Daylight, 0, { 1.459961, 1, 1.184937, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", Flash, 0, { 1.395020, 1, 1.144897, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", Cloudy, 0, { 1.541992, 1, 1.052856, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", Shade, 0, { 1.644897, 1.033936, 1, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", "2700K", 0, { 1, 1.172607, 2.747192, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", "3000K", 0, { 1, 1.049438, 2.156494, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", "3300K", 0, { 1.032349, 1, 1.841187, 0 } },
  { "Leica Camera AG", "R8 - Digital Back DMR", "5000K", 0, { 1.344360, 1, 1.233032, 0 } },

  { "Leica Camera AG", "R9 - Digital Back DMR", Incandescent, 0, { 1, 1.109985, 2.430664, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", Fluorescent, 0, { 1.234985, 1, 1.791138, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", Daylight, 0, { 1.459961, 1, 1.184937, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", Flash, 0, { 1.395020, 1, 1.144897, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", Cloudy, 0, { 1.541992, 1, 1.052856, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", Shade, 0, { 1.644897, 1.033936, 1, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", "2700K", 0, { 1, 1.172607, 2.747192, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", "3000K", 0, { 1, 1.049438, 2.156494, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", "3300K", 0, { 1.032349, 1, 1.841187, 0 } },
  { "Leica Camera AG", "R9 - Digital Back DMR", "5000K", 0, { 1.344360, 1, 1.233032, 0 } },

  { "Leica", "Digilux 2", Daylight, 0,		{ 1.628906, 1, 1.488281, 0 } },
  { "Leica", "Digilux 2", Cloudy, 0,		{ 1.835938, 1, 1.343750, 0 } },
  { "Leica", "Digilux 2", Incandescent, 0,	{ 1.078125, 1, 2.203125, 0 } },
  { "Leica", "Digilux 2", Flash, 0,		{ 2.074219, 1, 1.304688, 0 } },
  { "Leica", "Digilux 2", BlackNWhite, 0,	{ 1.632812, 1, 1.550781, 0 } },

  { "Leica", "Digilux 3", Daylight, 0,		{ 1.942966, 1, 1.399240, 0 } },
  { "Leica", "Digilux 3", Cloudy, 0,		{ 2.083650, 1, 1.247148, 0 } },
  { "Leica", "Digilux 3", Shade, 0,		{ 2.296578, 1, 1.110266, 0 } },
  { "Leica", "Digilux 3", Incandescent, 0,	{ 1.372624, 1, 2.079848, 0 } },
  /* Flash multipliers are variable */
  { "Leica", "Digilux 3", Flash, 0,		{ 2.095057, 1, 1.091255, 0 } },

  /* Digilux 3 Kelvin presets */
  { "Leica", "Digilux 3", "2700K", 0,		{ 1.250951, 1, 2.539924, 0 } },
  { "Leica", "Digilux 3", "3000K", 0,		{ 1.361217, 1, 2.212928, 0 } },
  { "Leica", "Digilux 3", "3300K", 0,		{ 1.471483, 1, 2.167300, 0 } },
  { "Leica", "Digilux 3", "5000K", 0,		{ 1.851711, 1, 1.490494, 0 } },
  { "Leica", "Digilux 3", "5500K", 0,		{ 1.923954, 1, 1.361217, 0 } },
  { "Leica", "Digilux 3", "6500K", 0,		{ 2.102662, 1, 1.193916, 0 } },

  { "Leica", "Q2", Daylight, 0,			{ 1 / 0.493256262, 1, 1 / 0.6627831715, 0 } },
  { "Leica", "Q2", Cloudy, 0,			{ 1 / 0.4673664993, 1, 1 / 0.7246992215, 0 } },
  { "Leica", "Q2", Shade, 0,			{ 1 / 0.4493198771, 1, 1 / 0.769924812, 0 } },
  { "Leica", "Q2", Incandescent, 0,		{ 1 / 0.7018505826, 1, 1 / 0.3823749066, 0 } },
  { "Leica", "Q2", Flash, 0,			{ 1 / 0.4949250846, 1, 1 / 0.7314285714, 0 } },

  { "Minolta", "DiMAGE 5", Daylight, 0,		{ 2.023438, 1, 1.371094, 0 } },
  { "Minolta", "DiMAGE 5", Incandescent, 0,	{ 1.113281, 1, 2.480469, 0 } },
  { "Minolta", "DiMAGE 5", Fluorescent, 0,	{ 1.957031, 1, 2.058594, 0 } },
  { "Minolta", "DiMAGE 5", Cloudy, 0,		{ 2.199219, 1, 1.300781, 0 } },

  { "Minolta", "DiMAGE 7", Cloudy, 0,		{ 2.082031, 1, 1.226562, 0 } },
  { "Minolta", "DiMAGE 7", Daylight, 0,		{ 1.914062, 1, 1.527344, 0 } },
  { "Minolta", "DiMAGE 7", Fluorescent, 0,	{ 1.917969, 1, 2.007812, 0 } },
  { "Minolta", "DiMAGE 7", Tungsten, 0,		{ 1.050781, 1, 2.437500, 0 } },

  { "Minolta", "DiMAGE 7i", Daylight, 0,	{ 1.441406, 1, 1.457031, 0 } },
  { "Minolta", "DiMAGE 7i", Tungsten, 0,	{ 1, 1.333333, 3.572917, 0 } },
  { "Minolta", "DiMAGE 7i", Fluorescent, 0,	{ 1.554688, 1, 2.230469, 0 } },
  { "Minolta", "DiMAGE 7i", Cloudy, 0,		{ 1.550781, 1, 1.402344, 0 } },

  { "Minolta", "DiMAGE 7Hi", Daylight, 0,	{ 1.609375, 1, 1.328125, 0 } }, /*5500K*/
  { "Minolta", "DiMAGE 7Hi", Tungsten, 0,	{ 1, 1.137778, 2.768889, 0 } }, /*2800K*/
  { "Minolta", "DiMAGE 7Hi", WhiteFluorescent, 0, { 1.664062, 1, 2.105469, 0 } }, /*4060K*/
  { "Minolta", "DiMAGE 7Hi", CoolWhiteFluorescent, 0, { 1.796875, 1, 1.734375, 0 } }, /*4938K*/
  { "Minolta", "DiMAGE 7Hi", Cloudy, 0,		{ 1.730469, 1, 1.269531, 0 } }, /*5823K*/

  { "Minolta", "DiMAGE A1", Daylight, 0,	{ 1.808594, 1, 1.304688, 0 } },
  { "Minolta", "DiMAGE A1", Tungsten, 0,	{ 1.062500, 1, 2.675781, 0 } },
  { "Minolta", "DiMAGE A1", Fluorescent, 0,	{ 1.707031, 1, 2.039063, 0 } },
  { "Minolta", "DiMAGE A1", Cloudy, 0,		{ 1.960938, 1, 1.339844, 0 } },
  { "Minolta", "DiMAGE A1", Shade, 0,		{ 2.253906, 1, 1.199219, 0 } },
  { "Minolta", "DiMAGE A1", Shade, 2,		{ 2.000000, 1, 1.183594, 0 } },
  { "Minolta", "DiMAGE A1", Flash, 0,		{ 1.972656, 1, 1.265625, 0 } },

  { "Minolta", "DiMAGE A2", Cloudy, -3,		{ 2.109375, 1, 1.578125, 0 } },
  { "Minolta", "DiMAGE A2", Cloudy, 0,		{ 2.203125, 1, 1.296875, 0 } },
  { "Minolta", "DiMAGE A2", Cloudy, 3,		{ 2.296875, 1, 1.015625, 0 } },
  { "Minolta", "DiMAGE A2", Daylight, -3,	{ 1.867188, 1, 1.683594, 0 } },
  { "Minolta", "DiMAGE A2", Daylight, 0,	{ 1.960938, 1, 1.402344, 0 } },
  { "Minolta", "DiMAGE A2", Daylight, 3,	{ 2.054688, 1, 1.121094, 0 } },
  { "Minolta", "DiMAGE A2", Flash, -3,		{ 1.945312, 1, 1.613281, 0 } },
  { "Minolta", "DiMAGE A2", Flash, 0,		{ 2.039062, 1, 1.332031, 0 } },
  { "Minolta", "DiMAGE A2", Flash, 3,		{ 2.132812, 1, 1.050781, 0 } },
  { "Minolta", "DiMAGE A2", Fluorescent, -2,	{ 1.136719, 1, 2.746094, 0 } },
  { "Minolta", "DiMAGE A2", Fluorescent, 0,	{ 1.722656, 1, 2.132812, 0 } },
  { "Minolta", "DiMAGE A2", Fluorescent, 4,	{ 2.347656, 1, 1.535156, 0 } },
  { "Minolta", "DiMAGE A2", Shade, -3,		{ 2.273438, 1, 1.546875, 0 } },
  { "Minolta", "DiMAGE A2", Shade, 0,		{ 2.367188, 1, 1.265625, 0 } },
  { "Minolta", "DiMAGE A2", Shade, 3,		{ 2.500000, 1.015873, 1, 0 } },
  { "Minolta", "DiMAGE A2", Tungsten, -3,	{ 1.003906, 1, 3.164062, 0 } },
  { "Minolta", "DiMAGE A2", Tungsten, 0,	{ 1.097656, 1, 2.882812, 0 } },
  { "Minolta", "DiMAGE A2", Tungsten, 3,	{ 1.191406, 1, 2.601562, 0 } },

  { "Minolta", "DiMAGE Z2", Daylight, 0,	{ 1.843749, 1, 1.664062, 0 } },
  { "Minolta", "DiMAGE Z2", Cloudy, 0,		{ 2.195312, 1, 1.449218, 0 } },
  { "Minolta", "DiMAGE Z2", Tungsten, 0,	{ 1.097656, 1, 3.050780, 0 } },
  { "Minolta", "DiMAGE Z2", Fluorescent, 0,	{ 1.796874, 1, 2.257810, 0 } },
  { "Minolta", "DiMAGE Z2", Flash, 0,		{ 2.117186, 1, 1.472656, 0 } },

  { "Minolta", "DiMAGE G500", Daylight, 0,	{ 1.496094, 1, 1.121094, 0 } },
  { "Minolta", "DiMAGE G500", Cloudy, 0,	{ 1.527344, 1, 1.105469, 0 } },
  { "Minolta", "DiMAGE G500", Fluorescent, 0,	{ 1.382813, 1, 1.347656, 0 } },
  { "Minolta", "DiMAGE G500", Tungsten, 0,	{ 1.042969, 1, 1.859375, 0 } },
  { "Minolta", "DiMAGE G500", Flash, 0,		{ 1.647078, 1, 1.218159, 0 } },

  { "Minolta", "Dynax 5D", Daylight, -3,	{ 1.593750, 1, 1.875000, 0 } },
  { "Minolta", "Dynax 5D", Daylight, -2,	{ 1.644531, 1, 1.792969, 0 } },
  { "Minolta", "Dynax 5D", Daylight, -1,	{ 1.699219, 1, 1.718750, 0 } },
  { "Minolta", "Dynax 5D", Daylight, 0,		{ 1.757812, 1, 1.636719, 0 } },
  { "Minolta", "Dynax 5D", Daylight, 1,		{ 1.804688, 1, 1.566406, 0 } },
  { "Minolta", "Dynax 5D", Daylight, 2,		{ 1.863281, 1, 1.500000, 0 } },
  { "Minolta", "Dynax 5D", Daylight, 3,		{ 1.925781, 1, 1.437500, 0 } },
  { "Minolta", "Dynax 5D", Shade, -3,		{ 1.835938, 1, 1.644531, 0 } },
  { "Minolta", "Dynax 5D", Shade, -2,		{ 1.894531, 1, 1.574219, 0 } },
  { "Minolta", "Dynax 5D", Shade, -1,		{ 1.957031, 1, 1.507812, 0 } },
  { "Minolta", "Dynax 5D", Shade, 0,		{ 2.011719, 1, 1.433594, 0 } },
  { "Minolta", "Dynax 5D", Shade, 1,		{ 2.078125, 1, 1.375000, 0 } },
  { "Minolta", "Dynax 5D", Shade, 2,		{ 2.148438, 1, 1.316406, 0 } },
  { "Minolta", "Dynax 5D", Shade, 3,		{ 2.218750, 1, 1.261719, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, -3,		{ 1.718750, 1, 1.738281, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, -2,		{ 1.773438, 1, 1.664062, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, -1,		{ 1.835938, 1, 1.593750, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, 0,		{ 1.886719, 1, 1.500000, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, 1,		{ 1.945312, 1, 1.460938, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, 2,		{ 2.007812, 1, 1.390625, 0 } },
  { "Minolta", "Dynax 5D", Cloudy, 3,		{ 2.078125, 1, 1.332031, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, -3,	{ 1, 1.066667, 4.262500, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, -2,	{ 1, 1.032258, 3.951613, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, -1,	{ 1, 1.000000, 3.671875, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, 0,		{ 1.023438, 1, 3.496094, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, 1,		{ 1.062500, 1, 3.367188, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, 2,		{ 1.097656, 1, 3.203125, 0 } },
  { "Minolta", "Dynax 5D", Tungsten, 3,		{ 1.132812, 1, 3.070312, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, -2,	{ 1.148438, 1, 3.429688, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, -1,	{ 1.285156, 1, 3.250000, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, 0,	{ 1.703125, 1, 2.582031, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, 1,	{ 1.761719, 1, 2.335938, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, 2,	{ 1.730469, 1, 1.878906, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, 3,	{ 1.996094, 1, 1.527344, 0 } },
  { "Minolta", "Dynax 5D", Fluorescent, 4,	{ 2.218750, 1, 1.714844, 0 } },
  { "Minolta", "Dynax 5D", Flash, -3,		{ 1.738281, 1, 1.683594, 0 } },
  { "Minolta", "Dynax 5D", Flash, -2,		{ 1.792969, 1, 1.609375, 0 } },
  { "Minolta", "Dynax 5D", Flash, -1,		{ 1.855469, 1, 1.542969, 0 } },
  { "Minolta", "Dynax 5D", Flash, 0,		{ 1.917969, 1, 1.457031, 0 } },
  { "Minolta", "Dynax 5D", Flash, 1,		{ 1.968750, 1, 1.406250, 0 } },
  { "Minolta", "Dynax 5D", Flash, 2,		{ 2.031250, 1, 1.347656, 0 } },
  { "Minolta", "Dynax 5D", Flash, 3,		{ 2.101562, 1, 1.289062, 0 } },

  { "Minolta", "Dynax 7D", Daylight, -3,	{ 1.476562, 1, 1.824219, 0 } },
  { "Minolta", "Dynax 7D", Daylight, 0,		{ 1.621094, 1, 1.601562, 0 } },
  { "Minolta", "Dynax 7D", Daylight, 3,		{ 1.785156, 1, 1.414062, 0 } },
  { "Minolta", "Dynax 7D", Shade, -3,		{ 1.683594, 1, 1.585938, 0 } },
  { "Minolta", "Dynax 7D", Shade, 0,		{ 1.855469, 1, 1.402344, 0 } },
  { "Minolta", "Dynax 7D", Shade, 3,		{ 2.031250, 1, 1.226562, 0 } },
  { "Minolta", "Dynax 7D", Cloudy, -3,		{ 1.593750, 1, 1.671875, 0 } },
  { "Minolta", "Dynax 7D", Cloudy, 0,		{ 1.738281, 1, 1.464844, 0 } },
  { "Minolta", "Dynax 7D", Cloudy, 3,		{ 1.925781, 1, 1.296875, 0 } },
  { "Minolta", "Dynax 7D", Tungsten, -3,	{ 0.867188, 1, 3.765625, 0 } },
  { "Minolta", "Dynax 7D", Tungsten, 0,		{ 0.945312, 1, 3.292969, 0 } },
  { "Minolta", "Dynax 7D", Tungsten, 3,		{ 1.050781, 1, 2.921875, 0 } },
  { "Minolta", "Dynax 7D", Fluorescent, -2,	{ 1.058594, 1, 3.230469, 0 } },
  { "Minolta", "Dynax 7D", Fluorescent, 0,	{ 1.570312, 1, 2.453125, 0 } },
  { "Minolta", "Dynax 7D", Fluorescent, 1,	{ 1.625000, 1, 2.226562, 0 } },
  { "Minolta", "Dynax 7D", Fluorescent, 4,	{ 2.046875, 1, 1.675781, 0 } },
  { "Minolta", "Dynax 7D", Flash, -3,		{ 1.738281, 1, 1.656250, 0 } },
  { "Minolta", "Dynax 7D", Flash, 0,		{ 1.890625, 1, 1.445312, 0 } },
  { "Minolta", "Dynax 7D", Flash, 3,		{ 2.101562, 1, 1.281250, 0 } },
  { "Minolta", "Dynax 7D", "2700K", 0,		{ 1, 1.089362, 3.829787, 0 } },
  { "Minolta", "Dynax 7D", "3000K", 0,		{ 1.031250, 1, 3.000000, 0 } },
  { "Minolta", "Dynax 7D", "3300K", 0,		{ 1.144531, 1, 2.648438, 0 } },
  { "Minolta", "Dynax 7D", "5000K", 0,		{ 1.609375, 1, 1.726562, 0 } },
  { "Minolta", "Dynax 7D", "5500K", 0,		{ 1.710938, 1, 1.601562, 0 } },
  { "Minolta", "Dynax 7D", "6500K", 0,		{ 1.875000, 1, 1.421875, 0 } },

  // Firmware  A: 1.11 / B:1.11 / L:1.00
  { "Nikon", "1 AW1", Incandescent, 0,		{ 1.484375, 1, 2.281250, 0 } },
  { "Nikon", "1 AW1", CoolWhiteFluorescent, 0,	{ 2.003906, 1, 2.132813, 0 } },
  { "Nikon", "1 AW1", DirectSunlight, 0,	{ 2.269531, 1, 1.601563, 0 } },
  { "Nikon", "1 AW1", Flash, 0,			{ 2.546875, 1, 1.433594, 0 } },
  { "Nikon", "1 AW1", Cloudy, 0,		{ 2.394531, 1, 1.500000, 0 } },
  { "Nikon", "1 AW1", Shade, 0,			{ 2.718750, 1, 1.367188, 0 } },
  { "Nikon", "1 AW1", Underwater, 0,		{ 2.152344, 1, 1.644531, 0 } },

  // copied from Nikon 1 V1
  { "Nikon", "1 J1", Incandescent, 0,		{ 1.210938, 1, 2.308594, 0 } },
  { "Nikon", "1 J1", CoolWhiteFluorescent, 0,	{ 1.687500, 1, 2.199219, 0 } },
  { "Nikon", "1 J1", DirectSunlight, 0,		{ 1.828125, 1, 1.550781, 0 } },
  { "Nikon", "1 J1", Flash, 0,			{ 2.101563, 1, 1.335938, 0 } },
  { "Nikon", "1 J1", Cloudy, 0,			{ 1.960938, 1, 1.406250, 0 } },
  { "Nikon", "1 J1", Shade, 0,			{ 2.257813, 1, 1.277344, 0 } },

  { "Nikon", "1 V1", Incandescent, 0,		{ 1.210938, 1, 2.308594, 0 } },
  { "Nikon", "1 V1", CoolWhiteFluorescent, 0,	{ 1.687500, 1, 2.199219, 0 } },
  { "Nikon", "1 V1", DirectSunlight, 0,		{ 1.828125, 1, 1.550781, 0 } },
  { "Nikon", "1 V1", Flash, 0,			{ 2.101563, 1, 1.335938, 0 } },
  { "Nikon", "1 V1", Cloudy, 0,			{ 1.960938, 1, 1.406250, 0 } },
  { "Nikon", "1 V1", Shade, 0,			{ 2.257813, 1, 1.277344, 0 } },

  { "Nikon", "1 V2", Incandescent, 0, 		{ 1.484375, 1, 2.30859375, 0 } },
  { "Nikon", "1 V2", CoolWhiteFluorescent, 0,	{ 2.00390625, 1, 2.16015625, 0 } },
  { "Nikon", "1 V2", DirectSunlight, 0,		{ 2.26953125, 1, 1.62109375, 0 } },
  { "Nikon", "1 V2", Flash, 0,			{ 2.6484375, 1, 1.390625, 0 } },
  { "Nikon", "1 V2", Cloudy, 0,			{ 2.39453125, 1, 1.51953125, 0 } },
  { "Nikon", "1 V2", Shade, 0,			{ 2.71875, 1, 1.3828125, 0 } },

  { "Nikon", "1 V3", DirectSunlight      , 0, { 2.722656, 1, 1.796875, 0 } },
  { "Nikon", "1 V3", Shade               , 0, { 3.359375, 1, 1.507812, 0 } },
  { "Nikon", "1 V3", Cloudy              , 0, { 2.910156, 1, 1.644531, 0 } },
  { "Nikon", "1 V3", Incandescent        , 0, { 1.738281, 1, 2.679688, 0 } },
  { "Nikon", "1 V3", CoolWhiteFluorescent, 0, { 2.429688, 1, 2.406250, 0 } },
  { "Nikon", "1 V3", Flash               , 0, { 3.152344, 1, 1.562500, 0 } },

  { "Nikon", "1 J3", Incandescent, 0,		{ 1.484375, 1, 2.308594, 0 } },
  { "Nikon", "1 J3", CoolWhiteFluorescent, 0,	{ 2.003906, 1, 2.160156, 0 } },
  { "Nikon", "1 J3", DirectSunlight, 0,		{ 2.269531, 1, 1.621094, 0 } },
  { "Nikon", "1 J3", Flash, 0,			{ 2.648438, 1, 1.390625, 0 } },
  { "Nikon", "1 J3", Cloudy, 0,			{ 2.394531, 1, 1.519531, 0 } },
  { "Nikon", "1 J3", Shade, 0,			{ 2.718750, 1, 1.382813, 0 } },
  { "Nikon", "1 J3", Underwater, 0,		{ 2.144531, 1, 1.660156, 0 } },

  { "Nikon", "1 J5", Incandescent, 0, { 1.546875, 1, 2.171875, 0 } },
  { "Nikon", "1 J5", CoolWhiteFluorescent, 0, { 2.15625, 1, 1.94921875, 0 } },
  { "Nikon", "1 J5", DirectSunlight, 0, { 2.31640625, 1, 1.37890625, 0 } },
  { "Nikon", "1 J5", Flash, 0, { 2.62109375, 1, 1.16796875, 0 } },
  { "Nikon", "1 J5", Cloudy, 0, { 2.4609375, 1, 1.2578125, 0 } },
  { "Nikon", "1 J5", Shade, 0, { 2.78515625, 1, 1.125, 0 } },

  { "Nikon", "Coolpix A", DirectSunlight      , 0, { 2.02734375, 1, 1.390625, 0 } },
  { "Nikon", "Coolpix A", Shade               , 0, { 2.4921875, 1, 1.13671875, 0 } },
  { "Nikon", "Coolpix A", Cloudy              , 0, { 2.16796875, 1, 1.265625, 0 } },
  { "Nikon", "Coolpix A", Incandescent        , 0, { 1.3046875, 1, 2.21484375, 0 } },
  { "Nikon", "Coolpix A", CoolWhiteFluorescent, 0, { 1.83203125, 1, 2.02734375, 0 } },
  { "Nikon", "Coolpix A", Flash               , 0, { 2.39453125, 1, 1.25, 0 } },

  // Nikon CoolPix P330 firmware 1.1
  { "Nikon", "Coolpix P330", DirectSunlight      , 0, { 1.789063, 1, 1.660156, 0 } },
  { "Nikon", "Coolpix P330", Cloudy              , 0, { 1.921875, 1, 1.484375, 0 } },
  { "Nikon", "Coolpix P330", Incandescent        , 0, { 1.164063, 1, 2.738281, 0 } },
  { "Nikon", "Coolpix P330", CoolWhiteFluorescent, 0, { 1.859375, 1, 2.429688, 0 } },
  { "Nikon", "Coolpix P330", DayWhiteFluorescent , 0, { 1.808594, 1, 1.750000, 0 } },
  { "Nikon", "Coolpix P330", DaylightFluorescent , 0, { 2.039063, 1, 1.398438, 0 } },
  { "Nikon", "Coolpix P330", Flash               , 0, { 2.031250, 1, 1.582031, 0 } },

  { "Nikon", "Coolpix P340", DirectSunlight, 0,		{ 1.835938, 1, 1.632813, 0 } },
  { "Nikon", "Coolpix P340", Incandescent, 0,		{ 1.199219, 1, 2.691406, 0 } },
  { "Nikon", "Coolpix P340", CoolWhiteFluorescent, 0,	{ 1.914063, 1, 2.386719, 0 } },
  { "Nikon", "Coolpix P340", DayWhiteFluorescent, 0,	{ 1.855469, 1, 1.718750, 0 } },
  { "Nikon", "Coolpix P340", DaylightFluorescent, 0,	{ 2.093750, 1, 1.371094, 0 } },
  { "Nikon", "Coolpix P340", Cloudy, 0,			{ 1.976563, 1, 1.457031, 0 } },
  { "Nikon", "Coolpix P340", Flash, 0,			{ 2.085938, 1, 1.554688, 0 } },

  // Nikon CoolPix P1000 firmware 1.2
  { "Nikon", "COOLPIX P1000", DirectSunlight      , -3, { 1.40625, 1, 2.05078125, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , -2, { 1.4375, 1, 1.99609375, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , -1, { 1.46875, 1, 1.94140625, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , 0, { 1.50390625, 1, 1.88671875, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , 1, { 1.5390625, 1, 1.8203125, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , 2, { 1.578125, 1, 1.7578125, 0 } },
  { "Nikon", "COOLPIX P1000", DirectSunlight      , 3, { 1.61328125, 1, 1.6953125, 0 } },
  { "Nikon", "COOLPIX P1000", Daylight            , 0, { 1.50390625, 1, 1.88671875, 0 } },
  { "Nikon", "COOLPIX P1000", Shade               , 0, { 1.87890625, 1, 1.4140625, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , -3, { 1.54296875, 1, 1.87109375, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , -2, { 1.578125, 1, 1.8203125, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , -1, { 1.61328125, 1, 1.765625, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , 0, { 1.65234375, 1, 1.71875, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , 1, { 1.69140625, 1, 1.66015625, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , 2, { 1.734375, 1, 1.6015625, 0 } },
  { "Nikon", "COOLPIX P1000", Cloudy              , 3, { 1.7734375, 1, 1.546875, 0 } },
  { "Nikon", "COOLPIX P1000", Tungsten            , 0, { 1.015625, 1, 2.8046875, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , -3, { 0.94921875, 1, 3.05078125, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , -2, { 0.96875, 1, 2.96875, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , -1, { 0.9921875, 1, 2.88671875, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , 0, { 1.015625, 1, 2.8046875, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , 1, { 1.0390625, 1, 2.7109375, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , 2, { 1.0625, 1, 2.6171875, 0 } },
  { "Nikon", "COOLPIX P1000", Incandescent        , 3, { 1.08984375, 1, 2.5234375, 0 } },
  { "Nikon", "COOLPIX P1000", CoolWhiteFluorescent, 0, { 1.5859375, 1, 2.5, 0 } },
  { "Nikon", "COOLPIX P1000", DayWhiteFluorescent , 0, { 1.59765625, 1, 1.83203125, 0 } },
  { "Nikon", "COOLPIX P1000", DaylightFluorescent , 0, { 1.796875, 1, 1.47265625, 0 } },
  { "Nikon", "COOLPIX P1000", NeutralFluorescent  , 0, { 1.59765625, 1, 1.83203125, 0 } },
  { "Nikon", "COOLPIX P1000", WhiteFluorescent    , 0, { 1.5859375, 1, 2.5, 0 } },
  { "Nikon", "COOLPIX P1000", HTMercury           , 0, { 1.76953125, 1, 1.61328125, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , -3, { 1.65234375, 1, 1.75390625, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , -2, { 1.69140625, 1, 1.70703125, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , -1, { 1.73046875, 1, 1.66015625, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , 0, { 1.76953125, 1, 1.61328125, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , 1, { 1.8125, 1, 1.55859375, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , 2, { 1.85546875, 1, 1.50390625, 0 } },
  { "Nikon", "COOLPIX P1000", Flash               , 3, { 1.8984375, 1, 1.44921875, 0 } },
  { "Nikon", "COOLPIX P1000", "2500K"             , 0, { 1.078125, 1, 2.98828125, 0 } },
  { "Nikon", "COOLPIX P1000", "2560K"             , 0, { 1.07421875, 1, 2.94140625, 0 } },
  { "Nikon", "COOLPIX P1000", "2630K"             , 0, { 1.06640625, 1, 2.890625, 0 } },
  { "Nikon", "COOLPIX P1000", "2700K"             , 0, { 1.05859375, 1, 2.83984375, 0 } },
  { "Nikon", "COOLPIX P1000", "2780K"             , 0, { 1.0625, 1, 2.78515625, 0 } },
  { "Nikon", "COOLPIX P1000", "2860K"             , 0, { 1.0625, 1, 2.73046875, 0 } },
  { "Nikon", "COOLPIX P1000", "2940K"             , 0, { 1.06640625, 1, 2.67578125, 0 } },
  { "Nikon", "COOLPIX P1000", "3030K"             , 0, { 1.0703125, 1, 2.62109375, 0 } },
  { "Nikon", "COOLPIX P1000", "3130K"             , 0, { 1.0703125, 1, 2.5625, 0 } },
  { "Nikon", "COOLPIX P1000", "3230K"             , 0, { 1.07421875, 1, 2.51171875, 0 } },
  { "Nikon", "COOLPIX P1000", "3330K"             , 0, { 1.11328125, 1, 2.45703125, 0 } },
  { "Nikon", "COOLPIX P1000", "3450K"             , 0, { 1.15234375, 1, 2.3984375, 0 } },
  { "Nikon", "COOLPIX P1000", "3570K"             , 0, { 1.19140625, 1, 2.34375, 0 } },
  { "Nikon", "COOLPIX P1000", "3700K"             , 0, { 1.23046875, 1, 2.28515625, 0 } },
  { "Nikon", "COOLPIX P1000", "3850K"             , 0, { 1.25390625, 1, 2.2109375, 0 } },
  { "Nikon", "COOLPIX P1000", "4000K"             , 0, { 1.2734375, 1, 2.13671875, 0 } },
  { "Nikon", "COOLPIX P1000", "4170K"             , 0, { 1.31640625, 1, 2.09375, 0 } },
  { "Nikon", "COOLPIX P1000", "4350K"             , 0, { 1.359375, 1, 2.05078125, 0 } },
  { "Nikon", "COOLPIX P1000", "4550K"             , 0, { 1.3984375, 1, 2.00390625, 0 } },
  { "Nikon", "COOLPIX P1000", "4760K"             , 0, { 1.44140625, 1, 1.9609375, 0 } },
  { "Nikon", "COOLPIX P1000", "5000K"             , 0, { 1.48046875, 1, 1.91796875, 0 } },
  { "Nikon", "COOLPIX P1000", "5260K"             , 0, { 1.5078125, 1, 1.85546875, 0 } },
  { "Nikon", "COOLPIX P1000", "5560K"             , 0, { 1.53515625, 1, 1.79296875, 0 } },
  { "Nikon", "COOLPIX P1000", "5880K"             , 0, { 1.5625, 1, 1.734375, 0 } },
  { "Nikon", "COOLPIX P1000", "6250K"             , 0, { 1.5859375, 1, 1.671875, 0 } },
  { "Nikon", "COOLPIX P1000", "6670K"             , 0, { 1.65625, 1, 1.625, 0 } },
  { "Nikon", "COOLPIX P1000", "7140K"             , 0, { 1.73046875, 1, 1.58203125, 0 } },
  { "Nikon", "COOLPIX P1000", "7690K"             , 0, { 1.80078125, 1, 1.5390625, 0 } },
  { "Nikon", "COOLPIX P1000", "8330K"             , 0, { 1.88671875, 1, 1.48828125, 0 } },
  { "Nikon", "COOLPIX P1000", "9090K"             , 0, { 1.97265625, 1, 1.4375, 0 } },
  { "Nikon", "COOLPIX P1000", "10000K"            , 0, { 2.0703125, 1, 1.375, 0 } },

  // Fine tuning is from A6 to B6 on amber-blue.
  { "Nikon", "Coolpix P7100", DirectSunlight, -6, { 1.7908, 1, 1.9042, 0 } },
  { "Nikon", "Coolpix P7100", DirectSunlight, 0, { 1.6500, 1, 2.1349, 0 } },
  { "Nikon", "Coolpix P7100", DirectSunlight, 6, { 1.5171, 1, 2.3891, 0 } },
  { "Nikon", "Coolpix P7100", Incandescent, -6,	{ 1.2708, 1, 3.0068, 0 } },
  { "Nikon", "Coolpix P7100", Incandescent, 0,	{ 1.1730, 1, 3.3705, 0 } },
  { "Nikon", "Coolpix P7100", Incandescent, 6,	{ 1.0753, 1, 3.7693, 0 } },
  { "Nikon", "Coolpix P7100", WhiteFluorescent, -6, { 2.0020, 1, 2.6158, 0 } },
  { "Nikon", "Coolpix P7100", WhiteFluorescent, 0, { 1.8456, 1, 2.9326, 0 } },
  { "Nikon", "Coolpix P7100", WhiteFluorescent, 6, { 1.6970, 1, 3.2805, 0 } },
  { "Nikon", "Coolpix P7100", DayWhiteFluorescent, -6, { 1.9707, 1, 1.9003, 0 } },
  { "Nikon", "Coolpix P7100", DayWhiteFluorescent, 0, { 1.8182, 1, 2.1310, 0 } },
  { "Nikon", "Coolpix P7100", DayWhiteFluorescent, 6, { 1.6696, 1, 2.3812, 0 } },
  { "Nikon", "Coolpix P7100", DaylightFluorescent, -6, { 2.3069, 1, 1.5601, 0 } },
  { "Nikon", "Coolpix P7100", DaylightFluorescent, 0, { 2.1271, 1, 1.7517, 0 } },
  { "Nikon", "Coolpix P7100", DaylightFluorescent, 6, { 1.9550, 1, 1.9589, 0 } },
  { "Nikon", "Coolpix P7100", Cloudy, -6,	{ 1.9707, 1, 1.6540, 0 } },
  { "Nikon", "Coolpix P7100", Cloudy, 0,	{ 1.8182, 1, 1.8534, 0 } },
  { "Nikon", "Coolpix P7100", Cloudy, 6,	{ 1.6696, 1, 2.0723, 0 } },
  { "Nikon", "Coolpix P7100", Flash, -6,	{ 2.3148, 1, 1.4780, 0 } },
  { "Nikon", "Coolpix P7100", Flash, 0,		{ 2.1349, 1, 1.6579, 0 } },
  { "Nikon", "Coolpix P7100", Flash, 6,		{ 1.9629, 1, 1.8534, 0 } },

  { "Nikon", "D1", Incandescent, -3,		{ 1, 1.439891, 2.125769, 0 } },
  { "Nikon", "D1", Incandescent, 0,		{ 1, 1.582583, 2.556096, 0 } },
  { "Nikon", "D1", Incandescent, 3,		{ 1, 1.745033, 3.044175, 0 } },
  { "Nikon", "D1", Fluorescent, -3,		{ 1, 1.013461, 1.489820, 0 } },
  { "Nikon", "D1", Fluorescent, 0,		{ 1, 1.077710, 1.672660, 0 } },
  { "Nikon", "D1", Fluorescent, 3,		{ 1, 1.143167, 1.875227, 0 } },
  { "Nikon", "D1", DirectSunlight, -3,		{ 1.084705, 1.039344, 1, 0 } },
  { "Nikon", "D1", DirectSunlight, 0,		{ 1.000000, 1.000000, 1, 0 } },
  { "Nikon", "D1", DirectSunlight, 3,		{ 1, 1.049801, 1.109411, 0 } },
  { "Nikon", "D1", Flash, -3,			{ 1.317409, 1.116197, 1, 0 } },
  { "Nikon", "D1", Flash, 0,			{ 1.235772, 1.078231, 1, 0 } },
  { "Nikon", "D1", Flash, 3,			{ 1.100855, 1.016026, 1, 0 } },
  { "Nikon", "D1", Cloudy, -3,			{ 1.241160, 1.116197, 1, 0 } },
  { "Nikon", "D1", Cloudy, 0,			{ 1.162116, 1.078231, 1, 0 } },
  { "Nikon", "D1", Cloudy, 3,			{ 1.063923, 1.032573, 1, 0 } },
  { "Nikon", "D1", Shade, -3,			{ 1.361330, 1.191729, 1, 0 } },
  { "Nikon", "D1", Shade, 0,			{ 1.284963, 1.136201, 1, 0 } },
  { "Nikon", "D1", Shade, 3,			{ 1.205117, 1.096886, 1, 0 } },

  { "Nikon", "D1H", Incandescent, -3,		{ 1.503906, 1, 1.832031, 0 } },
  { "Nikon", "D1H", Incandescent, 0,		{ 1.363281, 1, 1.996094, 0 } },
  { "Nikon", "D1H", Incandescent, 3,		{ 1.246094, 1, 2.148438, 0 } },
  { "Nikon", "D1H", Fluorescent, -3,		{ 2.546875, 1, 1.175781, 0 } },
  { "Nikon", "D1H", Fluorescent, 0,		{ 1.925781, 1, 2.054688, 0 } },
  { "Nikon", "D1H", Fluorescent, 3,		{ 1.234375, 1, 2.171875, 0 } },
  { "Nikon", "D1H", DirectSunlight, -3,		{ 2.230469, 1, 1.187500, 0 } },
  { "Nikon", "D1H", DirectSunlight, 0,		{ 2.148438, 1, 1.246094, 0 } },
  { "Nikon", "D1H", DirectSunlight, 3,		{ 2.066406, 1, 1.316406, 0 } },
  { "Nikon", "D1H", Flash, -3,			{ 2.453125, 1, 1.117188, 0 } },
  { "Nikon", "D1H", Flash, 0,			{ 2.347656, 1, 1.140625, 0 } },
  { "Nikon", "D1H", Flash, 3,			{ 2.242188, 1, 1.164062, 0 } },
  { "Nikon", "D1H", Cloudy, -3,			{ 2.441406, 1, 1.046875, 0 } },
  { "Nikon", "D1H", Cloudy, 0,			{ 2.300781, 1, 1.128906, 0 } },
  { "Nikon", "D1H", Cloudy, 3,			{ 2.207031, 1, 1.199219, 0 } },
  { "Nikon", "D1H", Shade, -3,			{ 2.839844, 1, 1.000000, 0 } },
  { "Nikon", "D1H", Shade, 0,			{ 2.628906, 1, 1.011719, 0 } },
  { "Nikon", "D1H", Shade, 3,			{ 2.441406, 1, 1.046875, 0 } },

  { "Nikon", "D1X", Incandescent, -3,		{ 1.503906, 1, 1.832031, 0 } }, /*3250K*/
  { "Nikon", "D1X", Incandescent, -2,		{ 1.445312, 1, 1.890625, 0 } }, /*3150K*/
  { "Nikon", "D1X", Incandescent, -1,		{ 1.410156, 1, 1.937500, 0 } }, /*3100K*/
  { "Nikon", "D1X", Incandescent, 0,		{ 1.363281, 1, 1.996094, 0 } }, /*3000K*/
  { "Nikon", "D1X", Incandescent, 1,		{ 1.316406, 1, 2.042969, 0 } }, /*2900K*/
  { "Nikon", "D1X", Incandescent, 2,		{ 1.281250, 1, 2.101562, 0 } }, /*2800K*/
  { "Nikon", "D1X", Incandescent, 3,		{ 1.246094, 1, 2.148438, 0 } }, /*2700K*/
  { "Nikon", "D1X", Fluorescent, -3,		{ 2.546875, 1, 1.175781, 0 } }, /*7200K*/
  { "Nikon", "D1X", Fluorescent, -2,		{ 2.464844, 1, 1.210938, 0 } }, /*6500K*/
  { "Nikon", "D1X", Fluorescent, -1,		{ 2.160156, 1, 1.386719, 0 } }, /*5000K*/
  { "Nikon", "D1X", Fluorescent, 0,		{ 1.925781, 1, 2.054688, 0 } }, /*4200K*/
  { "Nikon", "D1X", Fluorescent, 1,		{ 1.703125, 1, 2.277344, 0 } }, /*3700K*/
  { "Nikon", "D1X", Fluorescent, 2,		{ 1.328125, 1, 2.394531, 0 } }, /*3000K*/
  { "Nikon", "D1X", Fluorescent, 3,		{ 1.234375, 1, 2.171875, 0 } }, /*2700K*/
  { "Nikon", "D1X", DirectSunlight, -3,		{ 2.230469, 1, 1.187500, 0 } }, /*5600K*/
  { "Nikon", "D1X", DirectSunlight, -2,		{ 2.207031, 1, 1.210938, 0 } }, /*5400K*/
  { "Nikon", "D1X", DirectSunlight, -1,		{ 2.171875, 1, 1.222656, 0 } }, /*5300K*/
  { "Nikon", "D1X", DirectSunlight, 0,		{ 2.148438, 1, 1.246094, 0 } }, /*5200K*/
  { "Nikon", "D1X", DirectSunlight, 1,		{ 2.113281, 1, 1.269531, 0 } }, /*5000K*/
  { "Nikon", "D1X", DirectSunlight, 2,		{ 2.089844, 1, 1.292969, 0 } }, /*4900K*/
  { "Nikon", "D1X", DirectSunlight, 3,		{ 2.066406, 1, 1.316406, 0 } }, /*4800K*/
  { "Nikon", "D1X", Flash, -3,			{ 2.453125, 1, 1.117188, 0 } }, /*6000K*/
  { "Nikon", "D1X", Flash, -2,			{ 2.417969, 1, 1.128906, 0 } }, /*5800K*/
  { "Nikon", "D1X", Flash, -1,			{ 2.382812, 1, 1.128906, 0 } }, /*5600K*/
  { "Nikon", "D1X", Flash, 0,			{ 2.347656, 1, 1.140625, 0 } }, /*5400K*/
  { "Nikon", "D1X", Flash, 1,			{ 2.312500, 1, 1.152344, 0 } }, /*5200K*/
  { "Nikon", "D1X", Flash, 2,			{ 2.277344, 1, 1.164062, 0 } }, /*5000K*/
  { "Nikon", "D1X", Flash, 3,			{ 2.242188, 1, 1.164062, 0 } }, /*4800K*/
  { "Nikon", "D1X", Cloudy, -3,			{ 2.441406, 1, 1.046875, 0 } }, /*6600K*/
  { "Nikon", "D1X", Cloudy, -2,			{ 2.394531, 1, 1.082031, 0 } }, /*6400K*/
  { "Nikon", "D1X", Cloudy, -1,			{ 2.347656, 1, 1.105469, 0 } }, /*6200K*/
  { "Nikon", "D1X", Cloudy, 0,			{ 2.300781, 1, 1.128906, 0 } }, /*6000K*/
  { "Nikon", "D1X", Cloudy, 1,			{ 2.253906, 1, 1.164062, 0 } }, /*5800K*/
  { "Nikon", "D1X", Cloudy, 2,			{ 2.230469, 1, 1.187500, 0 } }, /*5600K*/
  { "Nikon", "D1X", Cloudy, 3,			{ 2.207031, 1, 1.199219, 0 } }, /*5400K*/
  { "Nikon", "D1X", Shade, -3,			{ 2.839844, 1, 1.000000, 0 } }, /*9200K*/
  { "Nikon", "D1X", Shade, -2,			{ 2.769531, 1, 1.000000, 0 } }, /*8800K*/
  { "Nikon", "D1X", Shade, -1,			{ 2.699219, 1, 1.000000, 0 } }, /*8400K*/
  { "Nikon", "D1X", Shade, 0,			{ 2.628906, 1, 1.011719, 0 } }, /*8000K*/
  { "Nikon", "D1X", Shade, 1,			{ 2.558594, 1, 1.023438, 0 } }, /*7500K*/
  { "Nikon", "D1X", Shade, 2,			{ 2.500000, 1, 1.035156, 0 } }, /*7100K*/
  { "Nikon", "D1X", Shade, 3,			{ 2.441406, 1, 1.046875, 0 } }, /*6700K*/

  /*
   * D2X with firmware A 1.01 and B 1.01
   */

  /* D2X basic + fine tune presets */
  { "Nikon", "D2X", Incandescent, -3,		{ 0.98462, 1, 2.61154, 0 } }, /*3300K*/
  { "Nikon", "D2X", Incandescent, -2,		{ 0.95880, 1, 2.71536, 0 } }, /*3200K*/
  { "Nikon", "D2X", Incandescent, -1,		{ 0.94465, 1, 2.77122, 0 } }, /*3100K*/
  { "Nikon", "D2X", Incandescent, 0,		{ 0.92086, 1, 2.89928, 0 } }, /*3000K*/
  { "Nikon", "D2X", Incandescent, 1,		{ 0.89510, 1, 3.03846, 0 } }, /*2900K*/
  { "Nikon", "D2X", Incandescent, 2,		{ 0.86486, 1, 3.17905, 0 } }, /*2800K*/
  { "Nikon", "D2X", Incandescent, 3,		{ 0.83388, 1, 3.34528, 0 } }, /*2700K*/
  { "Nikon", "D2X", Fluorescent, -3,		{ 2.01562, 1, 1.72266, 0 } }, /*7200K*/
  { "Nikon", "D2X", Fluorescent, -2,		{ 1.67969, 1, 1.42578, 0 } }, /*6500K*/
  { "Nikon", "D2X", Fluorescent, -1,		{ 1.42969, 1, 1.80078, 0 } }, /*5000K*/
  { "Nikon", "D2X", Fluorescent, 0,		{ 1.42969, 1, 2.62891, 0 } }, /*4200K*/
  { "Nikon", "D2X", Fluorescent, 1,		{ 1.13672, 1, 3.02734, 0 } }, /*3700K*/
  { "Nikon", "D2X", Fluorescent, 2,		{ 0.94118, 1, 2.68498, 0 } }, /*3000K*/
  { "Nikon", "D2X", Fluorescent, 3,		{ 0.83388, 1, 3.51140, 0 } }, /*2700K*/
  { "Nikon", "D2X", DirectSunlight, -3,		{ 1.61328, 1, 1.61328, 0 } }, /*5600K*/
  { "Nikon", "D2X", DirectSunlight, -2,		{ 1.57031, 1, 1.65234, 0 } }, /*5400K*/
  { "Nikon", "D2X", DirectSunlight, -1,		{ 1.55078, 1, 1.67578, 0 } }, /*5300K*/
  { "Nikon", "D2X", DirectSunlight, 0,		{ 1.52734, 1, 1.69531, 0 } }, /*5200K*/
  { "Nikon", "D2X", DirectSunlight, 1,		{ 1.48438, 1, 1.74609, 0 } }, /*5000K*/
  { "Nikon", "D2X", DirectSunlight, 2,		{ 1.45312, 1, 1.76953, 0 } }, /*4900K*/
  { "Nikon", "D2X", DirectSunlight, 3,		{ 1.42578, 1, 1.78906, 0 } }, /*4800K*/
  { "Nikon", "D2X", Flash, -3,			{ 1.71484, 1, 1.48438, 0 } }, /*6000K*/
  { "Nikon", "D2X", Flash, -2,			{ 1.67578, 1, 1.48438, 0 } }, /*5800K*/
  { "Nikon", "D2X", Flash, -1,			{ 1.66797, 1, 1.50781, 0 } }, /*5600K*/
  { "Nikon", "D2X", Flash, 0,			{ 1.66016, 1, 1.53125, 0 } }, /*5400K*/
  { "Nikon", "D2X", Flash, 1,			{ 1.64453, 1, 1.54297, 0 } }, /*5200K*/
  { "Nikon", "D2X", Flash, 2,			{ 1.62891, 1, 1.54297, 0 } }, /*5000K*/
  { "Nikon", "D2X", Flash, 3,			{ 1.57031, 1, 1.56641, 0 } }, /*4800K*/
  { "Nikon", "D2X", Cloudy, -3,			{ 1.79297, 1, 1.46875, 0 } }, /*6600K*/
  { "Nikon", "D2X", Cloudy, -2,			{ 1.76172, 1, 1.49219, 0 } }, /*6400K*/
  { "Nikon", "D2X", Cloudy, -1,			{ 1.72656, 1, 1.51953, 0 } }, /*6200K*/
  { "Nikon", "D2X", Cloudy, 0,			{ 1.69141, 1, 1.54688, 0 } }, /*6000K*/
  { "Nikon", "D2X", Cloudy, 1,			{ 1.65234, 1, 1.57812, 0 } }, /*5800K*/
  { "Nikon", "D2X", Cloudy, 2,			{ 1.61328, 1, 1.61328, 0 } }, /*5600K*/
  { "Nikon", "D2X", Cloudy, 3,			{ 1.57031, 1, 1.65234, 0 } }, /*5400K*/
  { "Nikon", "D2X", Shade, -3,			{ 2.10938, 1, 1.23828, 0 } }, /*9200K*/
  { "Nikon", "D2X", Shade, -2,			{ 2.07031, 1, 1.26562, 0 } }, /*8800K*/
  { "Nikon", "D2X", Shade, -1,			{ 2.02734, 1, 1.29688, 0 } }, /*8400K*/
  { "Nikon", "D2X", Shade, 0,			{ 1.98047, 1, 1.32812, 0 } }, /*8000K*/
  { "Nikon", "D2X", Shade, 1,			{ 1.92188, 1, 1.37109, 0 } }, /*7500K*/
  { "Nikon", "D2X", Shade, 2,			{ 1.86719, 1, 1.41406, 0 } }, /*7100K*/
  { "Nikon", "D2X", Shade, 3,			{ 1.80859, 1, 1.45703, 0 } }, /*6700K*/

  /* D2X Kelvin presets */
  { "Nikon", "D2X", "2700K", 0,			{ 0.83388, 1, 3.34528, 0 } },
  { "Nikon", "D2X", "3000K", 0,			{ 0.92086, 1, 2.89928, 0 } },
  { "Nikon", "D2X", "3300K", 0,			{ 0.99611, 1, 2.55642, 0 } },
  { "Nikon", "D2X", "5000K", 0,			{ 1.48438, 1, 1.74609, 0 } },

  { "Nikon", "D3", Daylight, 0,			{ 1.81640, 1, 1.35546, 0 } },
  { "Nikon", "D3", Flash, 0,			{ 2.03906, 1, 1.17187, 0 } },
  { "Nikon", "D3", Cloudy, 0,			{ 1.94921, 1, 1.22265, 0 } },
  { "Nikon", "D3", Shade, 0,			{ 2.24609, 1, 1.08593, 0 } },
  { "Nikon", "D3", Incandescent, 0,		{ 1.16796, 1, 2.31640, 0 } },
  { "Nikon", "D3", Fluorescent, 0,		{ 1.68750, 1, 2.10156, 0 } },
  { "Nikon", "D3", "2700K", 0,			{ 1.06640, 1, 2.67968, 0 } },
  { "Nikon", "D3", "3000K", 0,			{ 1.16796, 1, 2.31640, 0 } },
  { "Nikon", "D3", "5000K", 0,			{ 1.77734, 1, 1.40234, 0 } },

  { "Nikon", "D3S", Incandescent, 0,		{ 1.191406, 1, 2.242188, 0 } },
  { "Nikon", "D3S", SodiumVaporFluorescent, 0,	{ 1.132812, 1, 2.511719, 0 } },
  { "Nikon", "D3S", WarmWhiteFluorescent, 0,	{ 1.179688, 1, 1.996094, 0 } },
  { "Nikon", "D3S", WhiteFluorescent, 0,	{ 1.394531, 1, 2.402344, 0 } },
  { "Nikon", "D3S", CoolWhiteFluorescent, 0,	{ 1.703125, 1, 2.066406, 0 } },
  { "Nikon", "D3S", DayWhiteFluorescent, 0,	{ 1.710938, 1, 1.390625, 0 } },
  { "Nikon", "D3S", DaylightFluorescent, 0,	{ 1.941406, 1, 1.113281, 0 } },
  { "Nikon", "D3S", HighTempMercuryVaporFluorescent, 0, { 2.289062, 1, 1.355469, 0 } },
  { "Nikon", "D3S", DirectSunlight, 0,		{ 1.835938, 1, 1.359375, 0 } },
  { "Nikon", "D3S", Flash, 0,			{ 2.035156, 1, 1.183594, 0 } },
  { "Nikon", "D3S", Cloudy, 0,			{ 1.964844, 1, 1.226562, 0 } },
  { "Nikon", "D3S", Shade, 0,			{ 2.253906, 1, 1.089844, 0 } },
  { "Nikon", "D3S", "2700K", 0,			{ 1.093750, 1, 2.558594, 0 } },
  { "Nikon", "D3S", "5000K", 0,			{ 1.796875, 1, 1.402344, 0 } },

  /* D3X with firmware A 1.00 and B 1.01 */
  { "Nikon", "D3X", Incandescent, -4,		{ 1.441406, 1, 2.125000, 0 } },
  { "Nikon", "D3X", Incandescent, -3,		{ 1.421875, 1, 2.167969, 0 } },
  { "Nikon", "D3X", Incandescent, -2,		{ 1.402344, 1, 2.210938, 0 } },
  { "Nikon", "D3X", Incandescent, -1,		{ 1.382813, 1, 2.250000, 0 } },
  { "Nikon", "D3X", Incandescent, 0,		{ 1.367188, 1, 2.292969, 0 } },
  { "Nikon", "D3X", Incandescent, 1,		{ 1.351563, 1, 2.332031, 0 } },
  { "Nikon", "D3X", Incandescent, 2,		{ 1.332031, 1, 2.371093, 0 } },
  { "Nikon", "D3X", Incandescent, 3,		{ 1.316406, 1, 2.414063, 0 } },
  { "Nikon", "D3X", Incandescent, 4,		{ 1.300781, 1, 2.457031, 0 } },
  { "Nikon", "D3X", Fluorescent, -4,		{ 2.183594, 1, 1.980469, 0 } },
  { "Nikon", "D3X", Fluorescent, -3,		{ 2.136719, 1, 2.015625, 0 } },
  { "Nikon", "D3X", Fluorescent, -2,		{ 2.089844, 1, 2.054688, 0 } },
  { "Nikon", "D3X", Fluorescent, -1,		{ 2.039064, 1, 2.089844, 0 } },
  { "Nikon", "D3X", Fluorescent, 0,		{ 1.984375, 1, 2.128906, 0 } },
  { "Nikon", "D3X", Fluorescent, 1,		{ 1.929688, 1, 2.167969, 0 } },
  { "Nikon", "D3X", Fluorescent, 2,		{ 1.875000, 1, 2.207031, 0 } },
  { "Nikon", "D3X", Fluorescent, 3,		{ 1.816406, 1, 2.246094, 0 } },
  { "Nikon", "D3X", Fluorescent, 4,		{ 1.753906, 1, 2.292969, 0 } },
  { "Nikon", "D3X", DirectSunlight, -4,		{ 2.289063, 1, 1.308594, 0 } },
  { "Nikon", "D3X", DirectSunlight, -3,		{ 2.253906, 1, 1.335938, 0 } },
  { "Nikon", "D3X", DirectSunlight, -2,		{ 2.222656, 1, 1.359375, 0 } },
  { "Nikon", "D3X", DirectSunlight, -1,		{ 2.187500, 1, 1.386719, 0 } },
  { "Nikon", "D3X", DirectSunlight, 0,		{ 2.156250, 1, 1.417969, 0 } },
  { "Nikon", "D3X", DirectSunlight, 1,		{ 2.125000, 1, 1.445313, 0 } },
  { "Nikon", "D3X", DirectSunlight, 2,		{ 2.093750, 1, 1.472656, 0 } },
  { "Nikon", "D3X", DirectSunlight, 3,		{ 2.062500, 1, 1.496094, 0 } },
  { "Nikon", "D3X", DirectSunlight, 4,		{ 2.027344, 1, 1.519531, 0 } },
  { "Nikon", "D3X", Flash, -4,			{ 2.566406, 1, 1.183594, 0 } },
  { "Nikon", "D3X", Flash, -3,			{ 2.523438, 1, 1.199219, 0 } },
  { "Nikon", "D3X", Flash, -2,			{ 2.484375, 1, 1.214844, 0 } },
  { "Nikon", "D3X", Flash, -1,			{ 2.445313, 1, 1.226563, 0 } },
  { "Nikon", "D3X", Flash, 0,			{ 2.402344, 1, 1.242187, 0 } },
  { "Nikon", "D3X", Flash, 1,			{ 2.371094, 1, 1.257813, 0 } },
  { "Nikon", "D3X", Flash, 2,			{ 2.343750, 1, 1.273438, 0 } },
  { "Nikon", "D3X", Flash, 3,			{ 2.320313, 1, 1.292969, 0 } },
  { "Nikon", "D3X", Flash, 4,			{ 2.289063, 1, 1.308594, 0 } },
  { "Nikon", "D3X", Cloudy, -4,			{ 2.488281, 1, 1.214844, 0 } },
  { "Nikon", "D3X", Cloudy, -3,			{ 2.445313, 1, 1.230469, 0 } },
  { "Nikon", "D3X", Cloudy, -2,			{ 2.406250, 1, 1.250000, 0 } },
  { "Nikon", "D3X", Cloudy, -1,			{ 2.363281, 1, 1.265625, 0 } },
  { "Nikon", "D3X", Cloudy, 0,			{ 2.328125, 1, 1.289062, 0 } },
  { "Nikon", "D3X", Cloudy, 1,			{ 2.289063, 1, 1.308594, 0 } },
  { "Nikon", "D3X", Cloudy, 2,			{ 2.253906, 1, 1.335938, 0 } },
  { "Nikon", "D3X", Cloudy, 3,			{ 2.222656, 1, 1.359375, 0 } },
  { "Nikon", "D3X", Cloudy, 4,			{ 2.187500, 1, 1.386719, 0 } },
  { "Nikon", "D3X", Shade, -4,			{ 2.937500, 1, 1.089844, 0 } },
  { "Nikon", "D3X", Shade, -3,			{ 2.878906, 1, 1.113281, 0 } },
  { "Nikon", "D3X", Shade, -2,			{ 2.820313, 1, 1.128906, 0 } },
  { "Nikon", "D3X", Shade, -1,			{ 2.761719, 1, 1.144531, 0 } },
  { "Nikon", "D3X", Shade, 0,			{ 2.707031, 1, 1.160156, 0 } },
  { "Nikon", "D3X", Shade, 1,			{ 2.652344, 1, 1.171875, 0 } },
  { "Nikon", "D3X", Shade, 2,			{ 2.601563, 1, 1.183594, 0 } },
  { "Nikon", "D3X", Shade, 3,			{ 2.554688, 1, 1.199219, 0 } },
  { "Nikon", "D3X", Shade, 4,			{ 2.507813, 1, 1.210938, 0 } },

  /* D3X Kelvin presets */
  { "Nikon", "D3X", "2700K", 0,			{ 1.253906, 1, 2.605469, 0 } },
  { "Nikon", "D3X", "5000K", 0,			{ 2.105469, 1, 1.460938, 0 } },

  { "Nikon", "D4", Incandescent, 0,		{ 1.281250, 1, 2.371094, 0 } },
  { "Nikon", "D4", SodiumVaporFluorescent, 0,	{ 1.195313, 1, 2.589844, 0 } },
  { "Nikon", "D4", WarmWhiteFluorescent, 0,	{ 1.277344, 1, 2.109375, 0 } },
  { "Nikon", "D4", WhiteFluorescent, 0,		{ 1.488281, 1, 2.476563, 0 } },
  { "Nikon", "D4", CoolWhiteFluorescent, 0,	{ 1.855469, 1, 2.156250, 0 } },
  { "Nikon", "D4", DayWhiteFluorescent, 0,	{ 1.867188, 1, 1.476563, 0 } },
  { "Nikon", "D4", DaylightFluorescent, 0,	{ 2.132813, 1, 1.156250, 0 } },
  { "Nikon", "D4", HighTempMercuryVaporFluorescent, 0, { 2.546875, 1, 1.425781, 0 } },
  { "Nikon", "D4", DirectSunlight, 0,		{ 2.019531, 1, 1.437500, 0 } },
  { "Nikon", "D4", Flash, 0,			{ 2.300781, 1, 1.253906, 0 } },
  { "Nikon", "D4", Cloudy, 0,			{ 2.175781, 1, 1.300781, 0 } },
  { "Nikon", "D4", Shade, 0,			{ 2.511719, 1, 1.164063, 0 } },
  { "Nikon", "D4", "2700K", 0,			{ 1.167969, 1, 2.710938, 0 } },
  { "Nikon", "D4", "3000K", 0,			{ 1.281250, 1, 2.363281, 0 } },
  { "Nikon", "D4", "5000K", 0,			{ 1.972656, 1, 1.488281, 0 } },
  { "Nikon", "D4", "6500K", 0,			{ 2.261719, 1, 1.242188, 0 } },

  { "Nikon", "D100", Incandescent, -3,		{ 1.527344, 1, 2.539062, 0 } }, /*3300K*/
  { "Nikon", "D100", Incandescent, -2,		{ 1.476562, 1, 2.656250, 0 } }, /*3200K*/
  { "Nikon", "D100", Incandescent, -1,		{ 1.457031, 1, 2.707031, 0 } }, /*3100K*/
  { "Nikon", "D100", Incandescent, 0,		{ 1.406250, 1, 2.828125, 0 } }, /*3000K*/
  { "Nikon", "D100", Incandescent, 1,		{ 1.367188, 1, 2.937500, 0 } }, /*2900K*/
  { "Nikon", "D100", Incandescent, 2,		{ 1.316406, 1, 3.046875, 0 } }, /*2800K*/
  { "Nikon", "D100", Incandescent, 3,		{ 1.269531, 1, 3.167969, 0 } }, /*2700K*/
  { "Nikon", "D100", Fluorescent, -3,		{ 3.148438, 1, 1.847656, 0 } }, /*7200K*/
  { "Nikon", "D100", Fluorescent, -2,		{ 2.609375, 1, 1.617187, 0 } }, /*6500K*/
  { "Nikon", "D100", Fluorescent, -1,		{ 2.250000, 1, 2.039062, 0 } }, /*5000K*/
  { "Nikon", "D100", Fluorescent, 0,		{ 2.058594, 1, 2.617187, 0 } }, /*4200K*/
  { "Nikon", "D100", Fluorescent, 1,		{ 1.886719, 1, 2.726562, 0 } }, /*3700K*/
  { "Nikon", "D100", Fluorescent, 2,		{ 1.429688, 1, 3.359375, 0 } }, /*3000K*/
  { "Nikon", "D100", Fluorescent, 3,		{ 1.250000, 1, 2.699219, 0 } }, /*2700K*/
  { "Nikon", "D100", DirectSunlight, -3,	{ 2.386719, 1, 1.687500, 0 } }, /*5600K*/
  { "Nikon", "D100", DirectSunlight, -2,	{ 2.316406, 1, 1.726563, 0 } }, /*5400K*/
  { "Nikon", "D100", DirectSunlight, -1,	{ 2.296875, 1, 1.738281, 0 } }, /*5300K*/
  { "Nikon", "D100", DirectSunlight, 0,		{ 2.257812, 1, 1.757812, 0 } }, /*5200K*/
  { "Nikon", "D100", DirectSunlight, 1,		{ 2.187500, 1, 1.796875, 0 } }, /*5000K*/
  { "Nikon", "D100", DirectSunlight, 2,		{ 2.156250, 1, 1.816406, 0 } }, /*4900K*/
  { "Nikon", "D100", DirectSunlight, 3,		{ 2.117187, 1, 1.847656, 0 } }, /*4800K*/
  { "Nikon", "D100", Flash, -3,			{ 2.718750, 1, 1.519531, 0 } }, /*6000K*/
  { "Nikon", "D100", Flash, -2,			{ 2.656250, 1, 1.527344, 0 } }, /*5800K*/
  { "Nikon", "D100", Flash, -1,			{ 2.597656, 1, 1.527344, 0 } }, /*5600K*/
  { "Nikon", "D100", Flash, 0,			{ 2.539062, 1, 1.539062, 0 } }, /*5400K*/
  { "Nikon", "D100", Flash, 1,			{ 2.476562, 1, 1.539062, 0 } }, /*5200K*/
  { "Nikon", "D100", Flash, 2,			{ 2.437500, 1, 1.546875, 0 } }, /*5000K*/
  { "Nikon", "D100", Flash, 3,			{ 2.398438, 1, 1.546875, 0 } }, /*4800K*/
  { "Nikon", "D100", Cloudy, -3,		{ 2.648438, 1, 1.558594, 0 } }, /*6600K*/
  { "Nikon", "D100", Cloudy, -2,		{ 2.609375, 1, 1.578125, 0 } }, /*6400K*/
  { "Nikon", "D100", Cloudy, -1,		{ 2.558594, 1, 1.597656, 0 } }, /*6200K*/
  { "Nikon", "D100", Cloudy, 0,			{ 2.507813, 1, 1.628906, 0 } }, /*6000K*/
  { "Nikon", "D100", Cloudy, 1,			{ 2.449219, 1, 1.656250, 0 } }, /*5800K*/
  { "Nikon", "D100", Cloudy, 2,			{ 2.398438, 1, 1.687500, 0 } }, /*5600K*/
  { "Nikon", "D100", Cloudy, 3,			{ 2.316406, 1, 1.726563, 0 } }, /*5400K*/
  { "Nikon", "D100", Shade, -3,			{ 3.046875, 1, 1.386719, 0 } }, /*9200K*/
  { "Nikon", "D100", Shade, -2,			{ 3.000000, 1, 1.406250, 0 } }, /*8800K*/
  { "Nikon", "D100", Shade, -1,			{ 2.957031, 1, 1.417969, 0 } }, /*8400K*/
  { "Nikon", "D100", Shade, 0,			{ 2.906250, 1, 1.437500, 0 } }, /*8000K*/
  { "Nikon", "D100", Shade, 1,			{ 2.816406, 1, 1.476562, 0 } }, /*7500K*/
  { "Nikon", "D100", Shade, 2,			{ 2.750000, 1, 1.519531, 0 } }, /*7100K*/
  { "Nikon", "D100", Shade, 3,			{ 2.667969, 1, 1.546875, 0 } }, /*6700K*/

  /* D200 basic + fine tune WB presets */
  { "Nikon", "D200", Incandescent, -3,		{ 1.234375, 1, 2.136719, 0 } },
  { "Nikon", "D200", Incandescent, -2,		{ 1.199219, 1, 2.238281, 0 } },
  { "Nikon", "D200", Incandescent, -1,		{ 1.183594, 1, 2.289063, 0 } },
  { "Nikon", "D200", Incandescent, 0,		{ 1.148434, 1, 2.398438, 0 } },
  { "Nikon", "D200", Incandescent, 1,		{ 1.113281, 1, 2.519531, 0 } },
  { "Nikon", "D200", Incandescent, 2,		{ 1.074219, 1, 2.648438, 0 } },
  { "Nikon", "D200", Incandescent, 3,		{ 1.031250, 1, 2.804688, 0 } },
  { "Nikon", "D200", Fluorescent, -3,		{ 2.273438, 1, 1.410156, 0 } },
  { "Nikon", "D200", Fluorescent, -2,		{ 1.933594, 1, 1.152344, 0 } },
  { "Nikon", "D200", Fluorescent, -1,		{ 1.675781, 1, 1.453125, 0 } },
  { "Nikon", "D200", Fluorescent, 0,		{ 1.664063, 1, 2.148438, 0 } },
  { "Nikon", "D200", Fluorescent, 1,		{ 1.335938, 1, 2.453125, 0 } },
  { "Nikon", "D200", Fluorescent, 2,		{ 1.140625, 1, 2.214844, 0 } },
  { "Nikon", "D200", Fluorescent, 3,		{ 1.035156, 1, 2.410156, 0 } },
  { "Nikon", "D200", DirectSunlight, -3,	{ 1.863281, 1, 1.320313, 0 } },
  { "Nikon", "D200", DirectSunlight, -2,	{ 1.835938, 1, 1.355469, 0 } },
  { "Nikon", "D200", DirectSunlight, -1,	{ 1.820313, 1, 1.375000, 0 } },
  { "Nikon", "D200", DirectSunlight, 0,		{ 1.804688, 1, 1.398438, 0 } },
  { "Nikon", "D200", DirectSunlight, 1,		{ 1.746094, 1, 1.425781, 0 } },
  { "Nikon", "D200", DirectSunlight, 2,		{ 1.714844, 1, 1.437500, 0 } },
  { "Nikon", "D200", DirectSunlight, 3,		{ 1.687500, 1, 1.449219, 0 } },
  { "Nikon", "D200", Flash, -3,			{ 2.066406, 1, 1.183594, 0 } },
  { "Nikon", "D200", Flash, -2,			{ 2.046875, 1, 1.191406, 0 } },
  { "Nikon", "D200", Flash, -1,			{ 2.027344, 1, 1.199219, 0 } },
  { "Nikon", "D200", Flash, 0,			{ 2.007813, 1, 1.171875, 0 } },
  { "Nikon", "D200", Flash, 1,			{ 1.984375, 1, 1.207031, 0 } },
  { "Nikon", "D200", Flash, 2,			{ 1.964844, 1, 1.214844, 0 } },
  { "Nikon", "D200", Flash, 3,			{ 1.945313, 1, 1.222656, 0 } },
  { "Nikon", "D200", Cloudy, -3,		{ 2.027344, 1, 1.210938, 0 } },
  { "Nikon", "D200", Cloudy, -2,		{ 1.992188, 1, 1.226563, 0 } },
  { "Nikon", "D200", Cloudy, -1,		{ 1.953125, 1, 1.242188, 0 } },
  { "Nikon", "D200", Cloudy, 0,			{ 1.917969, 1, 1.261719, 0 } },
  { "Nikon", "D200", Cloudy, 1,			{ 1.890625, 1, 1.285156, 0 } },
  { "Nikon", "D200", Cloudy, 2,			{ 1.863281, 1, 1.320313, 0 } },
  { "Nikon", "D200", Cloudy, 3,			{ 1.835938, 1, 1.355469, 0 } },
  { "Nikon", "D200", Shade, -3,			{ 2.378906, 1, 1.066406, 0 } },
  { "Nikon", "D200", Shade, -2,			{ 2.332031, 1, 1.085938, 0 } },
  { "Nikon", "D200", Shade, -1,			{ 2.289063, 1, 1.105469, 0 } },
  { "Nikon", "D200", Shade, 0,			{ 2.234375, 1, 1.125000, 0 } },
  { "Nikon", "D200", Shade, 1,			{ 2.167969, 1, 1.152344, 0 } },
  { "Nikon", "D200", Shade, 2,			{ 2.105469, 1, 1.175781, 0 } },
  { "Nikon", "D200", Shade, 3,			{ 2.046875, 1, 1.199219, 0 } },

  /* D200 Kelvin presets */
  { "Nikon", "D200", "2700K", 0,		{ 1.031250, 1, 2.804688, 0 } },
  { "Nikon", "D200", "3000K", 0,		{ 1.148438, 1, 2.398438, 0 } },
  { "Nikon", "D200", "3300K", 0,		{ 1.250000, 1, 2.097656, 0 } },
  { "Nikon", "D200", "5000K", 0,		{ 1.746094, 1, 1.425781, 0 } },

  { "Nikon", "D300", Incandescent, -6,		{ 1.097656, 1, 1.898438, 0 } },
  { "Nikon", "D300", Incandescent, -5,		{ 1.085938, 1, 1.929688, 0 } },
  { "Nikon", "D300", Incandescent, -4,		{ 1.070313, 1, 1.964844, 0 } },
  { "Nikon", "D300", Incandescent, -3,		{ 1.058594, 1, 2.000000, 0 } },
  { "Nikon", "D300", Incandescent, -2,		{ 1.042969, 1, 2.035156, 0 } },
  { "Nikon", "D300", Incandescent, -1,		{ 1.031250, 1, 2.074219, 0 } },
  { "Nikon", "D300", Incandescent, 0,		{ 1.019531, 1, 2.109375, 0 } },
  { "Nikon", "D300", Incandescent, 1,		{ 1.007813, 1, 2.144531, 0 } },
  { "Nikon", "D300", Incandescent, 2,		{ 0.996094, 1, 2.183594, 0 } },
  { "Nikon", "D300", Incandescent, 3,		{ 0.984375, 1, 2.218750, 0 } },
  { "Nikon", "D300", Incandescent, 4,		{ 0.972656, 1, 2.257813, 0 } },
  { "Nikon", "D300", Incandescent, 5,		{ 0.964844, 1, 2.296875, 0 } },
  { "Nikon", "D300", Incandescent, 6,		{ 0.953125, 1, 2.335938, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -6, { 1.031250, 1, 2.101563, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -5, { 1.015625, 1, 2.136719, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -4, { 1.003906, 1, 2.167969, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -3, { 0.988281, 1, 2.207031, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -2, { 0.976563, 1, 2.242188, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, -1, { 0.960938, 1, 2.281250, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 0,	{ 0.949219, 1, 2.320313, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 1,	{ 0.937500, 1, 2.363281, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 2,	{ 0.925781, 1, 2.410156, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 3,	{ 0.914063, 1, 2.457031, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 4,	{ 0.902344, 1, 2.503906, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 5,	{ 0.890625, 1, 2.558594, 0 } },
  { "Nikon", "D300", SodiumVaporFluorescent, 6,	{ 0.878906, 1, 2.613281, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -6,	{ 1.128906, 1, 1.847656, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -5,	{ 1.113281, 1, 1.867188, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -4,	{ 1.097656, 1, 1.886719, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -3,	{ 1.085938, 1, 1.906250, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -2,	{ 1.070313, 1, 1.925781, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, -1,	{ 1.058594, 1, 1.945313, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 0,	{ 1.046875, 1, 1.960938, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 1,	{ 1.035156, 1, 1.980469, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 2,	{ 1.023438, 1, 1.996094, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 3,	{ 1.007813, 1, 2.015625, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 4,	{ 1.000000, 1, 2.031250, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 5,	{ 0.988281, 1, 2.046875, 0 } },
  { "Nikon", "D300", WarmWhiteFluorescent, 6,	{ 0.976563, 1, 2.062500, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -6,	{ 1.453125, 1, 2.050781, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -5,	{ 1.414063, 1, 2.093750, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -4,	{ 1.371094, 1, 2.132813, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -3,	{ 1.328125, 1, 2.175781, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -2,	{ 1.285156, 1, 2.218750, 0 } },
  { "Nikon", "D300", WhiteFluorescent, -1,	{ 1.238281, 1, 2.261719, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 0,	{ 1.191406, 1, 2.304688, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 1,	{ 1.140625, 1, 2.351563, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 2,	{ 1.089844, 1, 2.394531, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 3,	{ 1.039063, 1, 2.441406, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 4,	{ 0.984375, 1, 2.488281, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 5,	{ 0.925781, 1, 2.535156, 0 } },
  { "Nikon", "D300", WhiteFluorescent, 6,	{ 0.867188, 1, 2.582031, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -6,	{ 1.667969, 1, 1.800781, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -5,	{ 1.636719, 1, 1.835938, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -4,	{ 1.605469, 1, 1.875000, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -3,	{ 1.574219, 1, 1.914063, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -2,	{ 1.539063, 1, 1.953125, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, -1,	{ 1.503906, 1, 1.996094, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 0,	{ 1.468750, 1, 2.035156, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 1,	{ 1.429688, 1, 2.074219, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 2,	{ 1.386719, 1, 2.117188, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 3,	{ 1.347656, 1, 2.160156, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 4,	{ 1.304688, 1, 2.203125, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 5,	{ 1.257813, 1, 2.246094, 0 } },
  { "Nikon", "D300", CoolWhiteFluorescent, 6,	{ 1.210938, 1, 2.289063, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -6,	{ 1.625000, 1, 1.195313, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -5,	{ 1.601563, 1, 1.222656, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -4,	{ 1.582031, 1, 1.253906, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -3,	{ 1.558594, 1, 1.281250, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -2,	{ 1.535156, 1, 1.308594, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, -1,	{ 1.515625, 1, 1.335938, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 0,	{ 1.492188, 1, 1.363281, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 1,	{ 1.472656, 1, 1.390625, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 2,	{ 1.453125, 1, 1.417969, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 3,	{ 1.433594, 1, 1.441406, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 4,	{ 1.410156, 1, 1.468750, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 5,	{ 1.390625, 1, 1.492188, 0 } },
  { "Nikon", "D300", DayWhiteFluorescent, 6,	{ 1.375000, 1, 1.519531, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -6,	{ 1.851563, 1, 1.000000, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -5,	{ 1.824219, 1, 1.000000, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -4,	{ 1.796875, 1, 1.000000, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -3,	{ 1.773438, 1, 1.007813, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -2,	{ 1.750000, 1, 1.039063, 0 } },
  { "Nikon", "D300", DaylightFluorescent, -1,	{ 1.722656, 1, 1.070313, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 0,	{ 1.699219, 1, 1.101563, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 1,	{ 1.675781, 1, 1.128906, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 2,	{ 1.652344, 1, 1.160156, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 3,	{ 1.628906, 1, 1.187500, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 4,	{ 1.605469, 1, 1.218750, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 5,	{ 1.585938, 1, 1.246094, 0 } },
  { "Nikon", "D300", DaylightFluorescent, 6,	{ 1.562500, 1, 1.273438, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -6, { 2.039063, 1, 1.156250, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -5, { 2.027344, 1, 1.183594, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -4, { 2.015625, 1, 1.210938, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -3, { 2.003906, 1, 1.238281, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -2, { 1.992188, 1, 1.269531, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, -1, { 1.976563, 1, 1.300781, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 0, { 1.960938, 1, 1.328125, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 1, { 1.945313, 1, 1.359375, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 2, { 1.929688, 1, 1.390625, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 3, { 1.914063, 1, 1.421875, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 4, { 1.894531, 1, 1.457031, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 5, { 1.875000, 1, 1.488281, 0 } },
  { "Nikon", "D300", HighTempMercuryVaporFluorescent, 6, { 1.855469, 1, 1.523438, 0 } },
  { "Nikon", "D300", DirectSunlight, -6,	{ 1.687500, 1, 1.167969, 0 } },
  { "Nikon", "D300", DirectSunlight, -5,	{ 1.664063, 1, 1.187500, 0 } },
  { "Nikon", "D300", DirectSunlight, -4,	{ 1.644531, 1, 1.207031, 0 } },
  { "Nikon", "D300", DirectSunlight, -3,	{ 1.625000, 1, 1.230469, 0 } },
  { "Nikon", "D300", DirectSunlight, -2,	{ 1.601563, 1, 1.253906, 0 } },
  { "Nikon", "D300", DirectSunlight, -1,	{ 1.582031, 1, 1.281250, 0 } },
  { "Nikon", "D300", DirectSunlight, 0,		{ 1.562500, 1, 1.308594, 0 } },
  { "Nikon", "D300", DirectSunlight, 1,		{ 1.542969, 1, 1.335938, 0 } },
  { "Nikon", "D300", DirectSunlight, 2,		{ 1.523438, 1, 1.359375, 0 } },
  { "Nikon", "D300", DirectSunlight, 3,		{ 1.503906, 1, 1.386719, 0 } },
  { "Nikon", "D300", DirectSunlight, 4,		{ 1.480469, 1, 1.414063, 0 } },
  { "Nikon", "D300", DirectSunlight, 5,		{ 1.457031, 1, 1.437500, 0 } },
  { "Nikon", "D300", DirectSunlight, 6,		{ 1.429688, 1, 1.457031, 0 } },
  { "Nikon", "D300", Flash, -6,			{ 1.910156, 1, 1.058594, 0 } },
  { "Nikon", "D300", Flash, -5,			{ 1.863281, 1, 1.078125, 0 } },
  { "Nikon", "D300", Flash, -4,			{ 1.820313, 1, 1.093750, 0 } },
  { "Nikon", "D300", Flash, -3,			{ 1.781250, 1, 1.105469, 0 } },
  { "Nikon", "D300", Flash, -2,			{ 1.746094, 1, 1.121094, 0 } },
  { "Nikon", "D300", Flash, -1,			{ 1.714844, 1, 1.136719, 0 } },
  { "Nikon", "D300", Flash, 0,			{ 1.687500, 1, 1.152344, 0 } },
  { "Nikon", "D300", Flash, 1,			{ 1.660156, 1, 1.164063, 0 } },
  { "Nikon", "D300", Flash, 2,			{ 1.636719, 1, 1.179688, 0 } },
  { "Nikon", "D300", Flash, 3,			{ 1.613281, 1, 1.195313, 0 } },
  { "Nikon", "D300", Flash, 4,			{ 1.593750, 1, 1.210938, 0 } },
  { "Nikon", "D300", Flash, 5,			{ 1.574219, 1, 1.230469, 0 } },
  { "Nikon", "D300", Flash, 6,			{ 1.554688, 1, 1.246094, 0 } },
  { "Nikon", "D300", Cloudy, -6,		{ 1.820313, 1, 1.093750, 0 } },
  { "Nikon", "D300", Cloudy, -5,		{ 1.789063, 1, 1.105469, 0 } },
  { "Nikon", "D300", Cloudy, -4,		{ 1.761719, 1, 1.117188, 0 } },
  { "Nikon", "D300", Cloudy, -3,		{ 1.734375, 1, 1.132813, 0 } },
  { "Nikon", "D300", Cloudy, -2,		{ 1.710938, 1, 1.148438, 0 } },
  { "Nikon", "D300", Cloudy, -1,		{ 1.687500, 1, 1.167969, 0 } },
  { "Nikon", "D300", Cloudy, 0,			{ 1.664063, 1, 1.187500, 0 } },
  { "Nikon", "D300", Cloudy, 1,			{ 1.644531, 1, 1.207031, 0 } },
  { "Nikon", "D300", Cloudy, 2,			{ 1.625000, 1, 1.230469, 0 } },
  { "Nikon", "D300", Cloudy, 3,			{ 1.601563, 1, 1.253906, 0 } },
  { "Nikon", "D300", Cloudy, 4,			{ 1.582031, 1, 1.281250, 0 } },
  { "Nikon", "D300", Cloudy, 5,			{ 1.562500, 1, 1.308594, 0 } },
  { "Nikon", "D300", Cloudy, 6,			{ 1.542969, 1, 1.335938, 0 } },
  { "Nikon", "D300", Shade, -6,			{ 2.156250, 1, 1.000000, 0 } },
  { "Nikon", "D300", Shade, -5,			{ 2.109375, 1, 1.000000, 0 } },
  { "Nikon", "D300", Shade, -4,			{ 2.062500, 1, 1.011719, 0 } },
  { "Nikon", "D300", Shade, -3,			{ 2.019531, 1, 1.027344, 0 } },
  { "Nikon", "D300", Shade, -2,			{ 1.976563, 1, 1.042969, 0 } },
  { "Nikon", "D300", Shade, -1,			{ 1.937500, 1, 1.054688, 0 } },
  { "Nikon", "D300", Shade, 0,			{ 1.902344, 1, 1.066406, 0 } },
  { "Nikon", "D300", Shade, 1,			{ 1.867188, 1, 1.074219, 0 } },
  { "Nikon", "D300", Shade, 2,			{ 1.832031, 1, 1.085938, 0 } },
  { "Nikon", "D300", Shade, 3,			{ 1.804688, 1, 1.097656, 0 } },
  { "Nikon", "D300", Shade, 4,			{ 1.773438, 1, 1.113281, 0 } },
  { "Nikon", "D300", Shade, 5,			{ 1.746094, 1, 1.125000, 0 } },
  { "Nikon", "D300", Shade, 6,			{ 1.722656, 1, 1.140625, 0 } },
  { "Nikon", "D300", "2700K", 0,		{ 0.941406, 1, 2.390625, 0 } },
  { "Nikon", "D300", "5000K", 0,		{ 1.531250, 1, 1.351563, 0 } },

  { "Nikon", "D300S", DirectSunlight, -6,	{ 1.687, 1, 1.168, 0 } },
  { "Nikon", "D300S", DirectSunlight, 0,	{ 1.563, 1, 1.309, 0 } },
  { "Nikon", "D300S", DirectSunlight, 6,	{ 1.430, 1, 1.457, 0 } },
  { "Nikon", "D300S", Flash, -6,		{ 1.910, 1, 1.059, 0 } },
  { "Nikon", "D300S", Flash, 0,			{ 1.687, 1, 1.152, 0 } },
  { "Nikon", "D300S", Flash, 6,			{ 1.555, 1, 1.246, 0 } },
  { "Nikon", "D300S", Cloudy, -6,		{ 1.820, 1, 1.094, 0 } },
  { "Nikon", "D300S", Cloudy, 0,		{ 1.664, 1, 1.187, 0 } },
  { "Nikon", "D300S", Cloudy, 6,		{ 1.543, 1, 1.336, 0 } },
  { "Nikon", "D300S", Shade, -6,		{ 2.156, 1, 1.000, 0 } },
  { "Nikon", "D300S", Shade, 0,			{ 1.902, 1, 1.066, 0 } },
  { "Nikon", "D300S", Shade, 6,			{ 1.723, 1, 1.141, 0 } },
  { "Nikon", "D300S", Incandescent, -6,		{ 1.098, 1, 1.898, 0 } },
  { "Nikon", "D300S", Incandescent, 0,		{ 1.020, 1, 2.109, 0 } },
  { "Nikon", "D300S", Incandescent, 6,		{ 1, 1.049, 2.451, 0 } },
  { "Nikon", "D300S", SodiumVaporFluorescent, -6, { 1.031, 1, 2.102, 0 } },
  { "Nikon", "D300S", SodiumVaporFluorescent, 0, { 1, 1.053, 2.444, 0 } },
  { "Nikon", "D300S", SodiumVaporFluorescent, 6, { 1, 1.138, 2.973, 0 } },
  { "Nikon", "D300S", WarmWhiteFluorescent, -6,	{ 1.129, 1, 1.848, 0 } },
  { "Nikon", "D300S", WarmWhiteFluorescent, 0,	{ 1.047, 1, 1.961, 0 } },
  { "Nikon", "D300S", WarmWhiteFluorescent, 6,	{ 1, 1.024, 2.112, 0 } },
  { "Nikon", "D300S", WhiteFluorescent, -6,	{ 1.453, 1, 2.051, 0 } },
  { "Nikon", "D300S", WhiteFluorescent, 0,	{ 1.191, 1, 2.305, 0 } },
  { "Nikon", "D300S", WhiteFluorescent, 6,	{ 1, 1.153, 2.977, 0 } },
  { "Nikon", "D300S", CoolWhiteFluorescent, -6,	{ 1.668, 1, 1.801, 0 } },
  { "Nikon", "D300S", CoolWhiteFluorescent, 0,	{ 1.469, 1, 2.035, 0 } },
  { "Nikon", "D300S", CoolWhiteFluorescent, 6,	{ 1.211, 1, 2.289, 0 } },
  { "Nikon", "D300S", DayWhiteFluorescent, -6,	{ 1.625, 1, 1.195, 0 } },
  { "Nikon", "D300S", DayWhiteFluorescent, 0,	{ 1.492, 1, 1.363, 0 } },
  { "Nikon", "D300S", DayWhiteFluorescent, 6,	{ 1.375, 1, 1.520, 0 } },
  { "Nikon", "D300S", DaylightFluorescent, -6,	{ 1.852, 1, 1.000, 0 } },
  { "Nikon", "D300S", DaylightFluorescent, 0,	{ 1.699, 1, 1.102, 0 } },
  { "Nikon", "D300S", DaylightFluorescent, 6,	{ 1.563, 1, 1.273, 0 } },
  { "Nikon", "D300S", HighTempMercuryVaporFluorescent, -6, { 2.039, 1, 1.156, 0 } },
  { "Nikon", "D300S", HighTempMercuryVaporFluorescent, 0, { 1.961, 1, 1.328, 0 } },
  { "Nikon", "D300S", HighTempMercuryVaporFluorescent, 6, { 1.855, 1, 1.523, 0 } },

  /*NIKON D500 Software Ver.1.12*/
  { "Nikon", "D500", DirectSunlight      , 0, { 2.271484375, 1, 1.526855469, 0 } },
  { "Nikon", "D500", Shade               , 0, { 2.876464844, 1, 1.242675781, 0 } },
  { "Nikon", "D500", Cloudy              , 0, { 2.452636719, 1, 1.384765625, 0 } },
  { "Nikon", "D500", Incandescent        , 0, { 1.431640625, 1, 2.411621094, 0 } },
  { "Nikon", "D500", CoolWhiteFluorescent, 0, { 2.032714844, 1, 2.232910156, 0 } },
  { "Nikon", "D500", Flash               , 0, { 2.592773438, 1, 1.320800781, 0 } },
  { "Nikon", "D500", "5000K"             , 0, { 2.216308594, 1, 1.577148438, 0 } },

  { "Nikon", "D600", DirectSunlight, 0,		{ 1.906250, 1, 1.390625, 0 } },
  { "Nikon", "D600", Flash, 0,			{ 2.136719, 1, 1.214844, 0 } },
  { "Nikon", "D600", Cloudy, 0,			{ 2.039063, 1, 1.261719, 0 } },
  { "Nikon", "D600", Shade, 0,			{ 2.328125, 1, 1.136719, 0 } },
  { "Nikon", "D600", Incandescent, 0,		{ 1.226563, 1, 2.214844, 0 } },
  { "Nikon", "D600", SodiumVaporFluorescent, 0,	{ 1.148438, 1, 2.433594, 0 } },
  { "Nikon", "D600", WarmWhiteFluorescent, 0,	{ 1.230469, 1, 1.953125, 0 } },
  { "Nikon", "D600", WhiteFluorescent, 0,	{ 1.421875, 1, 2.281250, 0 } },
  { "Nikon", "D600", CoolWhiteFluorescent, 0,	{ 1.730469, 1, 2.003906, 0 } },
  { "Nikon", "D600", DayWhiteFluorescent, 0,	{ 1.777344, 1, 1.375000, 0 } },
  { "Nikon", "D600", DaylightFluorescent, 0,	{ 2.039063, 1, 1.117188, 0 } },
  { "Nikon", "D600", HighTempMercuryVaporFluorescent, 0, { 2.332031, 1, 1.355469, 0 } },

  /* D610 with firmware C1.00 L1.009 */
  { "Nikon", "D610", DirectSunlight, 0,         { 1.90625, 1, 1.390625, 0 } },
  { "Nikon", "D610", Flash, 0,                  { 2.136719, 1, 1.214844, 0 } },
  { "Nikon", "D610", Cloudy, 0,                 { 2.039063, 1, 1.261719, 0 } },
  { "Nikon", "D610", Shade, 0,                  { 2.328125, 1, 1.136719, 0 } },
  { "Nikon", "D610", Incandescent, 0,           { 1.226563, 1, 2.214844, 0 } },
  { "Nikon", "D610", SodiumVaporFluorescent, 0, { 1.148438, 1, 2.433594, 0 } },
  { "Nikon", "D610", WarmWhiteFluorescent, 0,   { 1.230469, 1, 1.953125, 0 } },
  { "Nikon", "D610", WhiteFluorescent, 0,       { 1.421875, 1, 2.28125, 0 } },
  { "Nikon", "D610", CoolWhiteFluorescent, 0,   { 1.730469, 1, 2.003906, 0 } },
  { "Nikon", "D610", DayWhiteFluorescent, 0,    { 1.777344, 1, 1.375, 0 } },
  { "Nikon", "D610", DaylightFluorescent, 0,    { 2.039063, 1, 1.117188, 0 } },
  { "Nikon", "D610", HighTempMercuryVaporFluorescent, 0, { 2.332031, 1, 1.355469, 0 } },

  { "Nikon", "D700", DirectSunlight, -6,	{ 1.980469, 1, 1.199219, 0 } },
  { "Nikon", "D700", DirectSunlight, 0,		{ 1.816406, 1, 1.355469, 0 } },
  { "Nikon", "D700", DirectSunlight, 6,		{ 1.652344, 1, 1.523437, 0 } },
  { "Nikon", "D700", Flash, -6,			{ 2.261719, 1, 1.082031, 0 } },
  { "Nikon", "D700", Flash, 0,			{ 2.039063, 1, 1.171875, 0 } },
  { "Nikon", "D700", Flash, 6,			{ 1.871094, 1, 1.281250, 0 } },
  { "Nikon", "D700", Cloudy, -6,		{ 2.148437, 1, 1.117187, 0 } },
  { "Nikon", "D700", Cloudy, 0,			{ 1.949219, 1, 1.222656, 0 } },
  { "Nikon", "D700", Cloudy, 6,			{ 1.792969, 1, 1.386719, 0 } },
  { "Nikon", "D700", Shade, -6,			{ 2.535156, 1, 1.000000, 0 } },
  { "Nikon", "D700", Shade, 0,			{ 2.246094, 1, 1.085937, 0 } },
  { "Nikon", "D700", Shade, 6,			{ 2.023438, 1, 1.171875, 0 } },
  { "Nikon", "D700", Incandescent , -6,		{ 1.265625, 1, 2.050781, 0 } },
  { "Nikon", "D700", Incandescent , 0,		{ 1.167969, 1, 2.316406, 0 } },
  { "Nikon", "D700", Incandescent , 6,		{ 1.085938, 1, 2.605469, 0 } },
  { "Nikon", "D700", SodiumVaporFluorescent, -6, { 1.175781, 1, 2.191406, 0 } },
  { "Nikon", "D700", SodiumVaporFluorescent, 0, { 1.062500, 1, 2.464844, 0 } },
  { "Nikon", "D700", SodiumVaporFluorescent, 6, { 1.000000, 1, 2.789062, 0 } },
  { "Nikon", "D700", WarmWhiteFluorescent, -6,	{ 1.269531, 1, 1.968750, 0 } },
  { "Nikon", "D700", WarmWhiteFluorescent, 0,	{ 1.167969, 1, 2.109375, 0 } },
  { "Nikon", "D700", WarmWhiteFluorescent, 6,	{ 1.078125, 1, 2.230469, 0 } },
  { "Nikon", "D700", WhiteFluorescent, -6,	{ 1.671875, 1, 2.121094, 0 } },
  { "Nikon", "D700", WhiteFluorescent, 0,	{ 1.363281, 1, 2.425781, 0 } },
  { "Nikon", "D700", WhiteFluorescent, 6,	{ 1, 1.015873, 2.813492, 0 } },
  { "Nikon", "D700", CoolWhiteFluorescent, -6,	{ 1.929687, 1, 1.835938, 0 } },
  { "Nikon", "D700", CoolWhiteFluorescent, 0,	{ 1.687500, 1, 2.101563, 0 } },
  { "Nikon", "D700", CoolWhiteFluorescent, 6,	{ 1.386719, 1, 2.406250, 0 } },
  { "Nikon", "D700", DayWhiteFluorescent, -6,	{ 1.867188, 1, 1.218750, 0 } },
  { "Nikon", "D700", DayWhiteFluorescent, 0,	{ 1.710938, 1, 1.410156, 0 } },
  { "Nikon", "D700", DayWhiteFluorescent, 6,	{ 1.570313, 1, 1.585938, 0 } },
  { "Nikon", "D700", DaylightFluorescent, -6,	{ 2.128906, 1, 1.000000, 0 } },
  { "Nikon", "D700", DaylightFluorescent, 0,	{ 1.953125, 1, 1.113281, 0 } },
  { "Nikon", "D700", DaylightFluorescent, 6,	{ 1.792969, 1, 1.308594, 0 } },
  { "Nikon", "D700", HighTempMercuryVaporFluorescent, -6, { 2.378906, 1, 1.218750, 0 } },
  { "Nikon", "D700", HighTempMercuryVaporFluorescent, 0, { 2.289063, 1, 1.363281, 0 } },
  { "Nikon", "D700", HighTempMercuryVaporFluorescent, 6, { 2.164063, 1, 1.542969, 0 } },
  { "Nikon", "D700", "2700K", 0,		{ 1.066406, 1, 2.679687, 0 } },
  { "Nikon", "D700", "5000K", 0,		{ 1.777344, 1, 1.402344, 0 } },

  { "Nikon", "D750", DirectSunlight                 , -12, { 1.77734375, 1, 1.4921875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -11, { 1.796875, 1, 1.48046875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -10, { 1.8125, 1, 1.47265625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -9, { 1.828125, 1, 1.4609375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -8, { 1.83984375, 1, 1.44921875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -7, { 1.85546875, 1, 1.4375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -6, { 1.87109375, 1, 1.421875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -5, { 1.8828125, 1, 1.41015625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -4, { 1.89453125, 1, 1.3984375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -3, { 1.91015625, 1, 1.3828125, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -2, { 1.921875, 1, 1.37109375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , -1, { 1.9375, 1, 1.35546875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 0, { 1.94921875, 1, 1.34375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 1, { 1.9609375, 1, 1.328125, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 2, { 1.97265625, 1, 1.31640625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 3, { 1.98828125, 1, 1.3046875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 4, { 2.0, 1, 1.2890625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 5, { 2.01171875, 1, 1.27734375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 6, { 2.02734375, 1, 1.265625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 7, { 2.0390625, 1, 1.25390625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 8, { 2.0546875, 1, 1.2421875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 9, { 2.06640625, 1, 1.23046875, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 10, { 2.08203125, 1, 1.22265625, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 11, { 2.09765625, 1, 1.2109375, 0 } },
  { "Nikon", "D750", DirectSunlight                 , 12, { 2.11328125, 1, 1.203125, 0 } },
  { "Nikon", "D750", Shade                          , -12, { 2.15625, 1, 1.1796875, 0 } },
  { "Nikon", "D750", Shade                          , -11, { 2.171875, 1, 1.171875, 0 } },
  { "Nikon", "D750", Shade                          , -10, { 2.19140625, 1, 1.1640625, 0 } },
  { "Nikon", "D750", Shade                          , -9, { 2.20703125, 1, 1.15625, 0 } },
  { "Nikon", "D750", Shade                          , -8, { 2.2265625, 1, 1.15234375, 0 } },
  { "Nikon", "D750", Shade                          , -7, { 2.24609375, 1, 1.14453125, 0 } },
  { "Nikon", "D750", Shade                          , -6, { 2.265625, 1, 1.13671875, 0 } },
  { "Nikon", "D750", Shade                          , -5, { 2.28515625, 1, 1.1328125, 0 } },
  { "Nikon", "D750", Shade                          , -4, { 2.3046875, 1, 1.12890625, 0 } },
  { "Nikon", "D750", Shade                          , -3, { 2.32421875, 1, 1.12109375, 0 } },
  { "Nikon", "D750", Shade                          , -2, { 2.34765625, 1, 1.1171875, 0 } },
  { "Nikon", "D750", Shade                          , -1, { 2.3671875, 1, 1.109375, 0 } },
  { "Nikon", "D750", Shade                          , 0, { 2.390625, 1, 1.10546875, 0 } },
  { "Nikon", "D750", Shade                          , 1, { 2.4140625, 1, 1.09765625, 0 } },
  { "Nikon", "D750", Shade                          , 2, { 2.4375, 1, 1.09375, 0 } },
  { "Nikon", "D750", Shade                          , 3, { 2.4609375, 1, 1.0859375, 0 } },
  { "Nikon", "D750", Shade                          , 4, { 2.484375, 1, 1.08203125, 0 } },
  { "Nikon", "D750", Shade                          , 5, { 2.5078125, 1, 1.07421875, 0 } },
  { "Nikon", "D750", Shade                          , 6, { 2.53515625, 1, 1.06640625, 0 } },
  { "Nikon", "D750", Shade                          , 7, { 2.55859375, 1, 1.0546875, 0 } },
  { "Nikon", "D750", Shade                          , 8, { 2.5859375, 1, 1.046875, 0 } },
  { "Nikon", "D750", Shade                          , 9, { 2.61328125, 1, 1.03515625, 0 } },
  { "Nikon", "D750", Shade                          , 10, { 2.640625, 1, 1.0234375, 0 } },
  { "Nikon", "D750", Shade                          , 11, { 2.66796875, 1, 1.01171875, 0 } },
  { "Nikon", "D750", Shade                          , 12, { 2.6953125, 1, 1.0, 0 } },
  { "Nikon", "D750", Cloudy                         , -12, { 1.921875, 1, 1.37109375, 0 } },
  { "Nikon", "D750", Cloudy                         , -11, { 1.9375, 1, 1.35546875, 0 } },
  { "Nikon", "D750", Cloudy                         , -10, { 1.94921875, 1, 1.34375, 0 } },
  { "Nikon", "D750", Cloudy                         , -9, { 1.9609375, 1, 1.328125, 0 } },
  { "Nikon", "D750", Cloudy                         , -8, { 1.97265625, 1, 1.31640625, 0 } },
  { "Nikon", "D750", Cloudy                         , -7, { 1.98828125, 1, 1.3046875, 0 } },
  { "Nikon", "D750", Cloudy                         , -6, { 2.0, 1, 1.2890625, 0 } },
  { "Nikon", "D750", Cloudy                         , -5, { 2.01171875, 1, 1.27734375, 0 } },
  { "Nikon", "D750", Cloudy                         , -4, { 2.02734375, 1, 1.265625, 0 } },
  { "Nikon", "D750", Cloudy                         , -3, { 2.0390625, 1, 1.25390625, 0 } },
  { "Nikon", "D750", Cloudy                         , -2, { 2.0546875, 1, 1.2421875, 0 } },
  { "Nikon", "D750", Cloudy                         , -1, { 2.06640625, 1, 1.23046875, 0 } },
  { "Nikon", "D750", Cloudy                         , 0, { 2.08203125, 1, 1.22265625, 0 } },
  { "Nikon", "D750", Cloudy                         , 1, { 2.09765625, 1, 1.2109375, 0 } },
  { "Nikon", "D750", Cloudy                         , 2, { 2.11328125, 1, 1.203125, 0 } },
  { "Nikon", "D750", Cloudy                         , 3, { 2.12890625, 1, 1.1953125, 0 } },
  { "Nikon", "D750", Cloudy                         , 4, { 2.14453125, 1, 1.18359375, 0 } },
  { "Nikon", "D750", Cloudy                         , 5, { 2.16015625, 1, 1.17578125, 0 } },
  { "Nikon", "D750", Cloudy                         , 6, { 2.17578125, 1, 1.16796875, 0 } },
  { "Nikon", "D750", Cloudy                         , 7, { 2.1953125, 1, 1.1640625, 0 } },
  { "Nikon", "D750", Cloudy                         , 8, { 2.2109375, 1, 1.15625, 0 } },
  { "Nikon", "D750", Cloudy                         , 9, { 2.23046875, 1, 1.1484375, 0 } },
  { "Nikon", "D750", Cloudy                         , 10, { 2.25, 1, 1.14453125, 0 } },
  { "Nikon", "D750", Cloudy                         , 11, { 2.26953125, 1, 1.13671875, 0 } },
  { "Nikon", "D750", Cloudy                         , 12, { 2.2890625, 1, 1.1328125, 0 } },
  { "Nikon", "D750", Incandescent                   , -12, { 1.171875, 1, 2.3046875, 0 } },
  { "Nikon", "D750", Incandescent                   , -11, { 1.1796875, 1, 2.2890625, 0 } },
  { "Nikon", "D750", Incandescent                   , -10, { 1.18359375, 1, 2.2734375, 0 } },
  { "Nikon", "D750", Incandescent                   , -9, { 1.19140625, 1, 2.25390625, 0 } },
  { "Nikon", "D750", Incandescent                   , -8, { 1.19921875, 1, 2.23828125, 0 } },
  { "Nikon", "D750", Incandescent                   , -7, { 1.203125, 1, 2.21875, 0 } },
  { "Nikon", "D750", Incandescent                   , -6, { 1.2109375, 1, 2.203125, 0 } },
  { "Nikon", "D750", Incandescent                   , -5, { 1.21875, 1, 2.18359375, 0 } },
  { "Nikon", "D750", Incandescent                   , -4, { 1.2265625, 1, 2.16796875, 0 } },
  { "Nikon", "D750", Incandescent                   , -3, { 1.234375, 1, 2.1484375, 0 } },
  { "Nikon", "D750", Incandescent                   , -2, { 1.2421875, 1, 2.1328125, 0 } },
  { "Nikon", "D750", Incandescent                   , -1, { 1.25, 1, 2.1171875, 0 } },
  { "Nikon", "D750", Incandescent                   , 0, { 1.2578125, 1, 2.09765625, 0 } },
  { "Nikon", "D750", Incandescent                   , 1, { 1.265625, 1, 2.08203125, 0 } },
  { "Nikon", "D750", Incandescent                   , 2, { 1.2734375, 1, 2.0625, 0 } },
  { "Nikon", "D750", Incandescent                   , 3, { 1.28125, 1, 2.046875, 0 } },
  { "Nikon", "D750", Incandescent                   , 4, { 1.2890625, 1, 2.03125, 0 } },
  { "Nikon", "D750", Incandescent                   , 5, { 1.30078125, 1, 2.01171875, 0 } },
  { "Nikon", "D750", Incandescent                   , 6, { 1.30859375, 1, 1.99609375, 0 } },
  { "Nikon", "D750", Incandescent                   , 7, { 1.31640625, 1, 1.98046875, 0 } },
  { "Nikon", "D750", Incandescent                   , 8, { 1.328125, 1, 1.9609375, 0 } },
  { "Nikon", "D750", Incandescent                   , 9, { 1.3359375, 1, 1.94921875, 0 } },
  { "Nikon", "D750", Incandescent                   , 10, { 1.34765625, 1, 1.9375, 0 } },
  { "Nikon", "D750", Incandescent                   , 11, { 1.359375, 1, 1.92578125, 0 } },
  { "Nikon", "D750", Incandescent                   , 12, { 1.3671875, 1, 1.91015625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -12, { 1.171875, 1, 1.96875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -11, { 1.1796875, 1, 1.9609375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -10, { 1.1875, 1, 1.94921875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -9, { 1.1953125, 1, 1.94140625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -8, { 1.203125, 1, 1.93359375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -7, { 1.2109375, 1, 1.92578125, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -6, { 1.21875, 1, 1.9140625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -5, { 1.2265625, 1, 1.90625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -4, { 1.234375, 1, 1.8984375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -3, { 1.2421875, 1, 1.890625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -2, { 1.25, 1, 1.87890625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , -1, { 1.2578125, 1, 1.87109375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 0, { 1.265625, 1, 1.86328125, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 1, { 1.27734375, 1, 1.85546875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 2, { 1.28515625, 1, 1.84765625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 3, { 1.29296875, 1, 1.83984375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 4, { 1.30078125, 1, 1.83203125, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 5, { 1.30859375, 1, 1.82421875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 6, { 1.3203125, 1, 1.81640625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 7, { 1.328125, 1, 1.80859375, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 8, { 1.3359375, 1, 1.8046875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 9, { 1.34765625, 1, 1.796875, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 10, { 1.35546875, 1, 1.7890625, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 11, { 1.36328125, 1, 1.78125, 0 } },
  { "Nikon", "D750", WarmWhiteFluorescent           , 12, { 1.375, 1, 1.7734375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -12, { 1.4921875, 1, 2.20703125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -11, { 1.51953125, 1, 2.18359375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -10, { 1.546875, 1, 2.16015625, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -9, { 1.57421875, 1, 2.140625, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -8, { 1.6015625, 1, 2.12109375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -7, { 1.625, 1, 2.09765625, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -6, { 1.6484375, 1, 2.078125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -5, { 1.671875, 1, 2.05859375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -4, { 1.6953125, 1, 2.0390625, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -3, { 1.71875, 1, 2.01953125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -2, { 1.7421875, 1, 2.0, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , -1, { 1.765625, 1, 1.98046875, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 0, { 1.7890625, 1, 1.96484375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 1, { 1.8125, 1, 1.9453125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 2, { 1.8359375, 1, 1.92578125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 3, { 1.859375, 1, 1.91015625, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 4, { 1.8828125, 1, 1.89453125, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 5, { 1.90625, 1, 1.875, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 6, { 1.9296875, 1, 1.859375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 7, { 1.953125, 1, 1.84375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 8, { 1.97265625, 1, 1.82421875, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 9, { 1.99609375, 1, 1.80859375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 10, { 2.015625, 1, 1.79296875, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 11, { 2.03515625, 1, 1.77734375, 0 } },
  { "Nikon", "D750", CoolWhiteFluorescent           , 12, { 2.0546875, 1, 1.7578125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -12, { 1.68359375, 1, 1.5, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -11, { 1.6953125, 1, 1.48828125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -10, { 1.70703125, 1, 1.4765625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -9, { 1.72265625, 1, 1.46484375, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -8, { 1.734375, 1, 1.453125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -7, { 1.74609375, 1, 1.44140625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -6, { 1.7578125, 1, 1.4296875, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -5, { 1.76953125, 1, 1.41796875, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -4, { 1.78125, 1, 1.40625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -3, { 1.796875, 1, 1.39453125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -2, { 1.80859375, 1, 1.3828125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , -1, { 1.8203125, 1, 1.37109375, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 0, { 1.8359375, 1, 1.359375, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 1, { 1.84765625, 1, 1.34765625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 2, { 1.859375, 1, 1.33203125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 3, { 1.875, 1, 1.3203125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 4, { 1.88671875, 1, 1.30859375, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 5, { 1.8984375, 1, 1.29296875, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 6, { 1.9140625, 1, 1.28125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 7, { 1.92578125, 1, 1.26953125, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 8, { 1.94140625, 1, 1.25390625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 9, { 1.953125, 1, 1.2421875, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 10, { 1.96875, 1, 1.2265625, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 11, { 1.98046875, 1, 1.21484375, 0 } },
  { "Nikon", "D750", DayWhiteFluorescent            , 12, { 1.99609375, 1, 1.19921875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -12, { 1.91796875, 1, 1.27734375, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -11, { 1.93359375, 1, 1.26171875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -10, { 1.9453125, 1, 1.25, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -9, { 1.9609375, 1, 1.234375, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -8, { 1.97265625, 1, 1.22265625, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -7, { 1.98828125, 1, 1.20703125, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -6, { 2.0, 1, 1.19140625, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -5, { 2.015625, 1, 1.1796875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -4, { 2.03125, 1, 1.1640625, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -3, { 2.04296875, 1, 1.1484375, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -2, { 2.05859375, 1, 1.1328125, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , -1, { 2.07421875, 1, 1.1171875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 0, { 2.08984375, 1, 1.10546875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 1, { 2.1015625, 1, 1.08984375, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 2, { 2.1171875, 1, 1.0703125, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 3, { 2.1328125, 1, 1.0546875, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 4, { 2.1484375, 1, 1.0390625, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 5, { 2.1640625, 1, 1.0234375, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 6, { 2.1796875, 1, 1.0078125, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 7, { 2.19140625, 1, 1.0, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 8, { 2.20703125, 1, 1.0, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 9, { 2.22265625, 1, 1.0, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 10, { 2.23828125, 1, 1.0, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 11, { 2.25390625, 1, 1.0, 0 } },
  { "Nikon", "D750", DaylightFluorescent            , 12, { 2.26953125, 1, 1.0, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -12, { 1.0625, 1, 2.53125, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -11, { 1.09765625, 1, 2.50390625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -10, { 1.1328125, 1, 2.4765625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -9, { 1.16796875, 1, 2.44921875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -8, { 1.203125, 1, 2.421875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -7, { 1.23828125, 1, 2.39453125, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -6, { 1.2734375, 1, 2.37109375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -5, { 1.30859375, 1, 2.34375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -4, { 1.34375, 1, 2.3203125, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -3, { 1.375, 1, 2.296875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -2, { 1.40625, 1, 2.2734375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , -1, { 1.4375, 1, 2.24609375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 0, { 1.46875, 1, 2.2265625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 1, { 1.49609375, 1, 2.203125, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 2, { 1.52734375, 1, 2.1796875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 3, { 1.5546875, 1, 2.15625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 4, { 1.578125, 1, 2.13671875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 5, { 1.60546875, 1, 2.11328125, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 6, { 1.62890625, 1, 2.09375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 7, { 1.65625, 1, 2.07421875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 8, { 1.6796875, 1, 2.0546875, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 9, { 1.703125, 1, 2.03515625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 10, { 1.72265625, 1, 2.015625, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 11, { 1.74609375, 1, 1.99609375, 0 } },
  { "Nikon", "D750", WhiteFluorescent               , 12, { 1.76953125, 1, 1.9765625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -12, { 2.2890625, 1, 1.4921875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -11, { 2.30078125, 1, 1.48046875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -10, { 2.3125, 1, 1.46484375, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -9, { 2.32421875, 1, 1.44921875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -8, { 2.3359375, 1, 1.43359375, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -7, { 2.34375, 1, 1.421875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -6, { 2.35546875, 1, 1.40625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -5, { 2.3671875, 1, 1.390625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -4, { 2.375, 1, 1.37890625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -3, { 2.38671875, 1, 1.36328125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -2, { 2.39453125, 1, 1.3515625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, -1, { 2.40234375, 1, 1.3359375, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 0, { 2.4140625, 1, 1.32421875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 1, { 2.421875, 1, 1.3125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 2, { 2.4296875, 1, 1.30078125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 3, { 2.4375, 1, 1.28515625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 4, { 2.4453125, 1, 1.2734375, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 5, { 2.453125, 1, 1.26171875, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 6, { 2.4609375, 1, 1.25, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 7, { 2.46875, 1, 1.23828125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 8, { 2.4765625, 1, 1.2265625, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 9, { 2.484375, 1, 1.21484375, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 10, { 2.4921875, 1, 1.203125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 11, { 2.49609375, 1, 1.1953125, 0 } },
  { "Nikon", "D750", HighTempMercuryVaporFluorescent, 12, { 2.50390625, 1, 1.18359375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -12, { 1.10546875, 1, 2.5546875, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -11, { 1.11328125, 1, 2.53125, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -10, { 1.1171875, 1, 2.50390625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -9, { 1.125, 1, 2.484375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -8, { 1.1328125, 1, 2.4609375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -7, { 1.13671875, 1, 2.44140625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -6, { 1.14453125, 1, 2.41796875, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -5, { 1.15234375, 1, 2.3984375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -4, { 1.16015625, 1, 2.37890625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -3, { 1.1640625, 1, 2.359375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -2, { 1.171875, 1, 2.33984375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , -1, { 1.1796875, 1, 2.32421875, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 0, { 1.1875, 1, 2.3046875, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 1, { 1.1953125, 1, 2.28515625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 2, { 1.20703125, 1, 2.26953125, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 3, { 1.21484375, 1, 2.25, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 4, { 1.22265625, 1, 2.234375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 5, { 1.234375, 1, 2.21484375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 6, { 1.2421875, 1, 2.1953125, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 7, { 1.25390625, 1, 2.1796875, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 8, { 1.265625, 1, 2.16015625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 9, { 1.27734375, 1, 2.140625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 10, { 1.2890625, 1, 2.12109375, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 11, { 1.30078125, 1, 2.1015625, 0 } },
  { "Nikon", "D750", SodiumVaporFluorescent         , 12, { 1.3125, 1, 2.08203125, 0 } },
  { "Nikon", "D750", Flash                          , -12, { 1.98046875, 1, 1.2734375, 0 } },
  { "Nikon", "D750", Flash                          , -11, { 2.0, 1, 1.265625, 0 } },
  { "Nikon", "D750", Flash                          , -10, { 2.015625, 1, 1.2578125, 0 } },
  { "Nikon", "D750", Flash                          , -9, { 2.03125, 1, 1.24609375, 0 } },
  { "Nikon", "D750", Flash                          , -8, { 2.046875, 1, 1.23828125, 0 } },
  { "Nikon", "D750", Flash                          , -7, { 2.05859375, 1, 1.23046875, 0 } },
  { "Nikon", "D750", Flash                          , -6, { 2.07421875, 1, 1.22265625, 0 } },
  { "Nikon", "D750", Flash                          , -5, { 2.0859375, 1, 1.21484375, 0 } },
  { "Nikon", "D750", Flash                          , -4, { 2.1015625, 1, 1.2109375, 0 } },
  { "Nikon", "D750", Flash                          , -3, { 2.11328125, 1, 1.203125, 0 } },
  { "Nikon", "D750", Flash                          , -2, { 2.12890625, 1, 1.1953125, 0 } },
  { "Nikon", "D750", Flash                          , -1, { 2.140625, 1, 1.19140625, 0 } },
  { "Nikon", "D750", Flash                          , 0, { 2.15625, 1, 1.18359375, 0 } },
  { "Nikon", "D750", Flash                          , 1, { 2.16796875, 1, 1.1796875, 0 } },
  { "Nikon", "D750", Flash                          , 2, { 2.18359375, 1, 1.171875, 0 } },
  { "Nikon", "D750", Flash                          , 3, { 2.19921875, 1, 1.16796875, 0 } },
  { "Nikon", "D750", Flash                          , 4, { 2.21484375, 1, 1.16015625, 0 } },
  { "Nikon", "D750", Flash                          , 5, { 2.234375, 1, 1.15625, 0 } },
  { "Nikon", "D750", Flash                          , 6, { 2.25, 1, 1.1484375, 0 } },
  { "Nikon", "D750", Flash                          , 7, { 2.26953125, 1, 1.140625, 0 } },
  { "Nikon", "D750", Flash                          , 8, { 2.2890625, 1, 1.13671875, 0 } },
  { "Nikon", "D750", Flash                          , 9, { 2.3125, 1, 1.12890625, 0 } },
  { "Nikon", "D750", Flash                          , 10, { 2.3359375, 1, 1.12109375, 0 } },
  { "Nikon", "D750", Flash                          , 11, { 2.359375, 1, 1.1171875, 0 } },
  { "Nikon", "D750", Flash                          , 12, { 2.38671875, 1, 1.109375, 0 } },
  { "Nikon", "D750", "2700K"                        ,  0, { 1.15625, 1, 2.359375, 0 } },
  { "Nikon", "D750", "3300K"                        ,  0, { 1.367188, 1, 1.914063, 0 } },
  { "Nikon", "D750", "5000K"                        ,  0, { 1.90625, 1, 1.386719, 0 } },
  { "Nikon", "D750", "6500K"                        ,  0, { 2.164063, 1, 1.175781, 0 } },

  { "Nikon", "D780", DirectSunlight                 , -12, { 1.58203125, 1, 1.48828125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -11, { 1.595703125, 1, 1.478515625, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -10, { 1.609375, 1, 1.466796875, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -9, { 1.623046875, 1, 1.455078125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -8, { 1.63671875, 1, 1.443359375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -7, { 1.6484375, 1, 1.4296875, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -6, { 1.66015625, 1, 1.41796875, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -5, { 1.671875, 1, 1.404296875, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -4, { 1.68359375, 1, 1.390625, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -3, { 1.6953125, 1, 1.376953125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -2, { 1.705078125, 1, 1.36328125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , -1, { 1.716796875, 1, 1.349609375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 0, { 1.7265625, 1, 1.3359375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 1, { 1.73828125, 1, 1.322265625, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 2, { 1.75, 1, 1.30859375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 3, { 1.759765625, 1, 1.294921875, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 4, { 1.771484375, 1, 1.283203125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 5, { 1.78125, 1, 1.26953125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 6, { 1.79296875, 1, 1.2578125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 7, { 1.8046875, 1, 1.24609375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 8, { 1.81640625, 1, 1.234375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 9, { 1.828125, 1, 1.224609375, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 10, { 1.83984375, 1, 1.212890625, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 11, { 1.8515625, 1, 1.203125, 0 } },
  { "Nikon", "D780", DirectSunlight                 , 12, { 1.865234375, 1, 1.193359375, 0 } },
  { "Nikon", "D780", Shade                          , -12, { 1.90234375, 1, 1.16796875, 0 } },
  { "Nikon", "D780", Shade                          , -11, { 1.916015625, 1, 1.16015625, 0 } },
  { "Nikon", "D780", Shade                          , -10, { 1.9296875, 1, 1.15234375, 0 } },
  { "Nikon", "D780", Shade                          , -9, { 1.9453125, 1, 1.146484375, 0 } },
  { "Nikon", "D780", Shade                          , -8, { 1.958984375, 1, 1.138671875, 0 } },
  { "Nikon", "D780", Shade                          , -7, { 1.974609375, 1, 1.1328125, 0 } },
  { "Nikon", "D780", Shade                          , -6, { 1.9921875, 1, 1.125, 0 } },
  { "Nikon", "D780", Shade                          , -5, { 2.0078125, 1, 1.119140625, 0 } },
  { "Nikon", "D780", Shade                          , -4, { 2.025390625, 1, 1.11328125, 0 } },
  { "Nikon", "D780", Shade                          , -3, { 2.04296875, 1, 1.107421875, 0 } },
  { "Nikon", "D780", Shade                          , -2, { 2.060546875, 1, 1.1015625, 0 } },
  { "Nikon", "D780", Shade                          , -1, { 2.080078125, 1, 1.095703125, 0 } },
  { "Nikon", "D780", Shade                          , 0, { 2.09765625, 1, 1.08984375, 0 } },
  { "Nikon", "D780", Shade                          , 1, { 2.1171875, 1, 1.083984375, 0 } },
  { "Nikon", "D780", Shade                          , 2, { 2.138671875, 1, 1.078125, 0 } },
  { "Nikon", "D780", Shade                          , 3, { 2.158203125, 1, 1.072265625, 0 } },
  { "Nikon", "D780", Shade                          , 4, { 2.1796875, 1, 1.064453125, 0 } },
  { "Nikon", "D780", Shade                          , 5, { 2.201171875, 1, 1.05859375, 0 } },
  { "Nikon", "D780", Shade                          , 6, { 2.22265625, 1, 1.05078125, 0 } },
  { "Nikon", "D780", Shade                          , 7, { 2.24609375, 1, 1.041015625, 0 } },
  { "Nikon", "D780", Shade                          , 8, { 2.26953125, 1, 1.033203125, 0 } },
  { "Nikon", "D780", Shade                          , 9, { 2.29296875, 1, 1.021484375, 0 } },
  { "Nikon", "D780", Shade                          , 10, { 2.31640625, 1, 1.01171875, 0 } },
  { "Nikon", "D780", Shade                          , 11, { 2.341796875, 1, 1.0, 0 } },
  { "Nikon", "D780", Shade                          , 12, { 2.3671875, 1, 1.0, 0 } },
  { "Nikon", "D780", Cloudy                         , -12, { 1.705078125, 1, 1.36328125, 0 } },
  { "Nikon", "D780", Cloudy                         , -11, { 1.716796875, 1, 1.349609375, 0 } },
  { "Nikon", "D780", Cloudy                         , -10, { 1.7265625, 1, 1.3359375, 0 } },
  { "Nikon", "D780", Cloudy                         , -9, { 1.73828125, 1, 1.322265625, 0 } },
  { "Nikon", "D780", Cloudy                         , -8, { 1.75, 1, 1.30859375, 0 } },
  { "Nikon", "D780", Cloudy                         , -7, { 1.759765625, 1, 1.294921875, 0 } },
  { "Nikon", "D780", Cloudy                         , -6, { 1.771484375, 1, 1.283203125, 0 } },
  { "Nikon", "D780", Cloudy                         , -5, { 1.78125, 1, 1.26953125, 0 } },
  { "Nikon", "D780", Cloudy                         , -4, { 1.79296875, 1, 1.2578125, 0 } },
  { "Nikon", "D780", Cloudy                         , -3, { 1.8046875, 1, 1.24609375, 0 } },
  { "Nikon", "D780", Cloudy                         , -2, { 1.81640625, 1, 1.234375, 0 } },
  { "Nikon", "D780", Cloudy                         , -1, { 1.828125, 1, 1.224609375, 0 } },
  { "Nikon", "D780", Cloudy                         , 0, { 1.83984375, 1, 1.212890625, 0 } },
  { "Nikon", "D780", Cloudy                         , 1, { 1.8515625, 1, 1.203125, 0 } },
  { "Nikon", "D780", Cloudy                         , 2, { 1.865234375, 1, 1.193359375, 0 } },
  { "Nikon", "D780", Cloudy                         , 3, { 1.876953125, 1, 1.18359375, 0 } },
  { "Nikon", "D780", Cloudy                         , 4, { 1.890625, 1, 1.17578125, 0 } },
  { "Nikon", "D780", Cloudy                         , 5, { 1.904296875, 1, 1.166015625, 0 } },
  { "Nikon", "D780", Cloudy                         , 6, { 1.91796875, 1, 1.158203125, 0 } },
  { "Nikon", "D780", Cloudy                         , 7, { 1.93359375, 1, 1.150390625, 0 } },
  { "Nikon", "D780", Cloudy                         , 8, { 1.947265625, 1, 1.14453125, 0 } },
  { "Nikon", "D780", Cloudy                         , 9, { 1.962890625, 1, 1.13671875, 0 } },
  { "Nikon", "D780", Cloudy                         , 10, { 1.978515625, 1, 1.130859375, 0 } },
  { "Nikon", "D780", Cloudy                         , 11, { 1.994140625, 1, 1.125, 0 } },
  { "Nikon", "D780", Cloudy                         , 12, { 2.01171875, 1, 1.119140625, 0 } },
  { "Nikon", "D780", Incandescent                   , -12, { 1.0625, 1, 2.302734375, 0 } },
  { "Nikon", "D780", Incandescent                   , -11, { 1.068359375, 1, 2.28515625, 0 } },
  { "Nikon", "D780", Incandescent                   , -10, { 1.07421875, 1, 2.265625, 0 } },
  { "Nikon", "D780", Incandescent                   , -9, { 1.080078125, 1, 2.248046875, 0 } },
  { "Nikon", "D780", Incandescent                   , -8, { 1.0859375, 1, 2.23046875, 0 } },
  { "Nikon", "D780", Incandescent                   , -7, { 1.091796875, 1, 2.212890625, 0 } },
  { "Nikon", "D780", Incandescent                   , -6, { 1.099609375, 1, 2.193359375, 0 } },
  { "Nikon", "D780", Incandescent                   , -5, { 1.10546875, 1, 2.17578125, 0 } },
  { "Nikon", "D780", Incandescent                   , -4, { 1.111328125, 1, 2.158203125, 0 } },
  { "Nikon", "D780", Incandescent                   , -3, { 1.1171875, 1, 2.138671875, 0 } },
  { "Nikon", "D780", Incandescent                   , -2, { 1.125, 1, 2.12109375, 0 } },
  { "Nikon", "D780", Incandescent                   , -1, { 1.130859375, 1, 2.103515625, 0 } },
  { "Nikon", "D780", Incandescent                   , 0, { 1.138671875, 1, 2.083984375, 0 } },
  { "Nikon", "D780", Incandescent                   , 1, { 1.146484375, 1, 2.06640625, 0 } },
  { "Nikon", "D780", Incandescent                   , 2, { 1.15234375, 1, 2.046875, 0 } },
  { "Nikon", "D780", Incandescent                   , 3, { 1.16015625, 1, 2.029296875, 0 } },
  { "Nikon", "D780", Incandescent                   , 4, { 1.16796875, 1, 2.009765625, 0 } },
  { "Nikon", "D780", Incandescent                   , 5, { 1.173828125, 1, 1.990234375, 0 } },
  { "Nikon", "D780", Incandescent                   , 6, { 1.181640625, 1, 1.97265625, 0 } },
  { "Nikon", "D780", Incandescent                   , 7, { 1.189453125, 1, 1.953125, 0 } },
  { "Nikon", "D780", Incandescent                   , 8, { 1.197265625, 1, 1.9453125, 0 } },
  { "Nikon", "D780", Incandescent                   , 9, { 1.205078125, 1, 1.939453125, 0 } },
  { "Nikon", "D780", Incandescent                   , 10, { 1.212890625, 1, 1.9375, 0 } },
  { "Nikon", "D780", Incandescent                   , 11, { 1.220703125, 1, 1.935546875, 0 } },
  { "Nikon", "D780", Incandescent                   , 12, { 1.228515625, 1, 1.93359375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -12, { 1.04296875, 1, 1.978515625, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -11, { 1.05078125, 1, 1.96875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -10, { 1.056640625, 1, 1.9609375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -9, { 1.064453125, 1, 1.951171875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -8, { 1.072265625, 1, 1.943359375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -7, { 1.078125, 1, 1.93359375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -6, { 1.0859375, 1, 1.92578125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -5, { 1.09375, 1, 1.91796875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -4, { 1.099609375, 1, 1.908203125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -3, { 1.107421875, 1, 1.900390625, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -2, { 1.115234375, 1, 1.892578125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , -1, { 1.123046875, 1, 1.8828125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 0, { 1.130859375, 1, 1.875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 1, { 1.138671875, 1, 1.8671875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 2, { 1.146484375, 1, 1.857421875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 3, { 1.154296875, 1, 1.849609375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 4, { 1.16015625, 1, 1.841796875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 5, { 1.169921875, 1, 1.83203125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 6, { 1.177734375, 1, 1.82421875, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 7, { 1.185546875, 1, 1.81640625, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 8, { 1.193359375, 1, 1.80859375, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 9, { 1.201171875, 1, 1.798828125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 10, { 1.208984375, 1, 1.791015625, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 11, { 1.216796875, 1, 1.783203125, 0 } },
  { "Nikon", "D780", WarmWhiteFluorescent           , 12, { 1.224609375, 1, 1.7734375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -12, { 1.35546875, 1, 2.205078125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -11, { 1.380859375, 1, 2.181640625, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -10, { 1.40625, 1, 2.158203125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -9, { 1.431640625, 1, 2.13671875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -8, { 1.455078125, 1, 2.115234375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -7, { 1.4765625, 1, 2.09375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -6, { 1.5, 1, 2.07421875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -5, { 1.521484375, 1, 2.052734375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -4, { 1.54296875, 1, 2.033203125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -3, { 1.5625, 1, 2.01171875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -2, { 1.583984375, 1, 1.9921875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , -1, { 1.603515625, 1, 1.97265625, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 0, { 1.623046875, 1, 1.953125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 1, { 1.64453125, 1, 1.93359375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 2, { 1.6640625, 1, 1.9140625, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 3, { 1.68359375, 1, 1.89453125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 4, { 1.703125, 1, 1.875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 5, { 1.72265625, 1, 1.85546875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 6, { 1.740234375, 1, 1.837890625, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 7, { 1.7578125, 1, 1.818359375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 8, { 1.775390625, 1, 1.80078125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 9, { 1.79296875, 1, 1.783203125, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 10, { 1.810546875, 1, 1.763671875, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 11, { 1.826171875, 1, 1.74609375, 0 } },
  { "Nikon", "D780", CoolWhiteFluorescent           , 12, { 1.84375, 1, 1.728515625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -12, { 1.494140625, 1, 1.48828125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -11, { 1.50390625, 1, 1.4765625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -10, { 1.513671875, 1, 1.46484375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -9, { 1.525390625, 1, 1.453125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -8, { 1.53515625, 1, 1.44140625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -7, { 1.544921875, 1, 1.427734375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -6, { 1.556640625, 1, 1.416015625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -5, { 1.56640625, 1, 1.404296875, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -4, { 1.578125, 1, 1.392578125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -3, { 1.587890625, 1, 1.380859375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -2, { 1.599609375, 1, 1.3671875, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , -1, { 1.609375, 1, 1.35546875, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 0, { 1.62109375, 1, 1.341796875, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 1, { 1.6328125, 1, 1.330078125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 2, { 1.642578125, 1, 1.31640625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 3, { 1.654296875, 1, 1.3046875, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 4, { 1.666015625, 1, 1.291015625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 5, { 1.677734375, 1, 1.27734375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 6, { 1.689453125, 1, 1.265625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 7, { 1.701171875, 1, 1.251953125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 8, { 1.712890625, 1, 1.23828125, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 9, { 1.724609375, 1, 1.224609375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 10, { 1.736328125, 1, 1.2109375, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 11, { 1.748046875, 1, 1.197265625, 0 } },
  { "Nikon", "D780", DayWhiteFluorescent            , 12, { 1.759765625, 1, 1.181640625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -12, { 1.693359375, 1, 1.259765625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -11, { 1.705078125, 1, 1.24609375, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -10, { 1.716796875, 1, 1.232421875, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -9, { 1.728515625, 1, 1.21875, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -8, { 1.740234375, 1, 1.205078125, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -7, { 1.751953125, 1, 1.19140625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -6, { 1.763671875, 1, 1.17578125, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -5, { 1.77734375, 1, 1.162109375, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -4, { 1.7890625, 1, 1.1484375, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -3, { 1.80078125, 1, 1.1328125, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -2, { 1.8125, 1, 1.119140625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , -1, { 1.826171875, 1, 1.103515625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 0, { 1.837890625, 1, 1.087890625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 1, { 1.8515625, 1, 1.072265625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 2, { 1.86328125, 1, 1.056640625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 3, { 1.876953125, 1, 1.041015625, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 4, { 1.888671875, 1, 1.0234375, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 5, { 1.90234375, 1, 1.0078125, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 6, { 1.916015625, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 7, { 1.9296875, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 8, { 1.94140625, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 9, { 1.955078125, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 10, { 1.96875, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 11, { 1.982421875, 1, 1.0, 0 } },
  { "Nikon", "D780", DaylightFluorescent            , 12, { 1.99609375, 1, 1.0, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -12, { 0.974609375, 1, 2.53515625, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -11, { 1.0, 1, 2.505859375, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -10, { 1.03125, 1, 2.478515625, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -9, { 1.064453125, 1, 2.453125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -8, { 1.095703125, 1, 2.42578125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -7, { 1.12890625, 1, 2.3984375, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -6, { 1.16015625, 1, 2.373046875, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -5, { 1.189453125, 1, 2.34765625, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -4, { 1.220703125, 1, 2.3203125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -3, { 1.25, 1, 2.296875, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -2, { 1.279296875, 1, 2.271484375, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , -1, { 1.306640625, 1, 2.24609375, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 0, { 1.333984375, 1, 2.22265625, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 1, { 1.361328125, 1, 2.19921875, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 2, { 1.38671875, 1, 2.17578125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 3, { 1.412109375, 1, 2.154296875, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 4, { 1.435546875, 1, 2.1328125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 5, { 1.458984375, 1, 2.111328125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 6, { 1.482421875, 1, 2.08984375, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 7, { 1.50390625, 1, 2.0703125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 8, { 1.525390625, 1, 2.048828125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 9, { 1.546875, 1, 2.029296875, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 10, { 1.56640625, 1, 2.0078125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 11, { 1.587890625, 1, 1.98828125, 0 } },
  { "Nikon", "D780", WhiteFluorescent               , 12, { 1.607421875, 1, 1.96875, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -12, { 2.041015625, 1, 1.474609375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -11, { 2.05078125, 1, 1.4609375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -10, { 2.060546875, 1, 1.4453125, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -9, { 2.0703125, 1, 1.431640625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -8, { 2.078125, 1, 1.416015625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -7, { 2.087890625, 1, 1.40234375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -6, { 2.095703125, 1, 1.388671875, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -5, { 2.103515625, 1, 1.375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -4, { 2.111328125, 1, 1.361328125, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -3, { 2.119140625, 1, 1.34765625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -2, { 2.126953125, 1, 1.3359375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, -1, { 2.134765625, 1, 1.322265625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 0, { 2.142578125, 1, 1.30859375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 1, { 2.1484375, 1, 1.296875, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 2, { 2.15625, 1, 1.28515625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 3, { 2.1640625, 1, 1.2734375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 4, { 2.171875, 1, 1.259765625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 5, { 2.177734375, 1, 1.248046875, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 6, { 2.185546875, 1, 1.236328125, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 7, { 2.193359375, 1, 1.2265625, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 8, { 2.19921875, 1, 1.21484375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 9, { 2.205078125, 1, 1.203125, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 10, { 2.2109375, 1, 1.193359375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 11, { 2.216796875, 1, 1.18359375, 0 } },
  { "Nikon", "D780", HighTempMercuryVaporFluorescent, 12, { 2.22265625, 1, 1.171875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -12, { 1.017578125, 1, 2.591796875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -11, { 1.0234375, 1, 2.56640625, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -10, { 1.02734375, 1, 2.54296875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -9, { 1.03125, 1, 2.51953125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -8, { 1.03515625, 1, 2.498046875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -7, { 1.041015625, 1, 2.4765625, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -6, { 1.044921875, 1, 2.455078125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -5, { 1.05078125, 1, 2.435546875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -4, { 1.0546875, 1, 2.416015625, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -3, { 1.060546875, 1, 2.396484375, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -2, { 1.064453125, 1, 2.376953125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , -1, { 1.0703125, 1, 2.359375, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 0, { 1.076171875, 1, 2.33984375, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 1, { 1.08203125, 1, 2.322265625, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 2, { 1.08984375, 1, 2.3046875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 3, { 1.095703125, 1, 2.287109375, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 4, { 1.103515625, 1, 2.26953125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 5, { 1.111328125, 1, 2.25, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 6, { 1.119140625, 1, 2.232421875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 7, { 1.126953125, 1, 2.21484375, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 8, { 1.13671875, 1, 2.1953125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 9, { 1.146484375, 1, 2.17578125, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 10, { 1.15625, 1, 2.15625, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 11, { 1.166015625, 1, 2.13671875, 0 } },
  { "Nikon", "D780", SodiumVaporFluorescent         , 12, { 1.177734375, 1, 2.115234375, 0 } },
  { "Nikon", "D780", Flash                          , -12, { 1.771484375, 1, 1.25390625, 0 } },
  { "Nikon", "D780", Flash                          , -11, { 1.78515625, 1, 1.24609375, 0 } },
  { "Nikon", "D780", Flash                          , -10, { 1.80078125, 1, 1.236328125, 0 } },
  { "Nikon", "D780", Flash                          , -9, { 1.814453125, 1, 1.2265625, 0 } },
  { "Nikon", "D780", Flash                          , -8, { 1.826171875, 1, 1.21875, 0 } },
  { "Nikon", "D780", Flash                          , -7, { 1.837890625, 1, 1.2109375, 0 } },
  { "Nikon", "D780", Flash                          , -6, { 1.849609375, 1, 1.203125, 0 } },
  { "Nikon", "D780", Flash                          , -5, { 1.861328125, 1, 1.197265625, 0 } },
  { "Nikon", "D780", Flash                          , -4, { 1.87109375, 1, 1.189453125, 0 } },
  { "Nikon", "D780", Flash                          , -3, { 1.8828125, 1, 1.181640625, 0 } },
  { "Nikon", "D780", Flash                          , -2, { 1.892578125, 1, 1.17578125, 0 } },
  { "Nikon", "D780", Flash                          , -1, { 1.904296875, 1, 1.169921875, 0 } },
  { "Nikon", "D780", Flash                          , 0, { 1.916015625, 1, 1.1640625, 0 } },
  { "Nikon", "D780", Flash                          , 1, { 1.92578125, 1, 1.15625, 0 } },
  { "Nikon", "D780", Flash                          , 2, { 1.9375, 1, 1.150390625, 0 } },
  { "Nikon", "D780", Flash                          , 3, { 1.951171875, 1, 1.14453125, 0 } },
  { "Nikon", "D780", Flash                          , 4, { 1.962890625, 1, 1.138671875, 0 } },
  { "Nikon", "D780", Flash                          , 5, { 1.9765625, 1, 1.1328125, 0 } },
  { "Nikon", "D780", Flash                          , 6, { 1.9921875, 1, 1.126953125, 0 } },
  { "Nikon", "D780", Flash                          , 7, { 2.0078125, 1, 1.12109375, 0 } },
  { "Nikon", "D780", Flash                          , 8, { 2.0234375, 1, 1.11328125, 0 } },
  { "Nikon", "D780", Flash                          , 9, { 2.041015625, 1, 1.107421875, 0 } },
  { "Nikon", "D780", Flash                          , 10, { 2.060546875, 1, 1.099609375, 0 } },
  { "Nikon", "D780", Flash                          , 11, { 2.08203125, 1, 1.09375, 0 } },
  { "Nikon", "D780", Flash                          , 12, { 2.103515625, 1, 1.0859375, 0 } },

  { "Nikon", "D800", Incandescent, -6,		{ 1.390625, 1, 1.984375, 0 } },
  { "Nikon", "D800", Incandescent, 0,		{ 1.289063, 1, 2.175781, 0 } },
  { "Nikon", "D800", Incandescent, 6,		{ 1.199219, 1, 2.429688, 0 } },
  { "Nikon", "D800", CoolWhiteFluorescent, -6,	{ 2.003906, 1, 1.769531, 0 } },
  { "Nikon", "D800", CoolWhiteFluorescent, 0,	{ 1.765625, 1, 2.003906, 0 } },
  { "Nikon", "D800", CoolWhiteFluorescent, 6,	{ 1.480469, 1, 2.257813, 0 } },
  { "Nikon", "D800", DirectSunlight, -6,	{ 2.128906, 1, 1.230469, 0 } },
  { "Nikon", "D800", DirectSunlight, 0,		{ 1.972656, 1, 1.375000, 0 } },
  { "Nikon", "D800", DirectSunlight, 6,		{ 1.804688, 1, 1.531250, 0 } },
  { "Nikon", "D800", Flash, -6,			{ 2.398438, 1, 1.125000, 0 } },
  { "Nikon", "D800", Flash, 0,			{ 2.187500, 1, 1.207031, 0 } },
  { "Nikon", "D800", Flash, 6,			{ 2.027344, 1, 1.300781, 0 } },
  { "Nikon", "D800", Cloudy, -6,		{ 2.292969, 1, 1.156250, 0 } },
  { "Nikon", "D800", Cloudy, 0,			{ 2.101563, 1, 1.253906, 0 } },
  { "Nikon", "D800", Cloudy, 6,			{ 1.945313, 1, 1.402344, 0 } },
  { "Nikon", "D800", Shade, -6,			{ 2.687500, 1, 1.015625, 0 } },
  { "Nikon", "D800", Shade, 0,			{ 2.378906, 1, 1.128906, 0 } },
  { "Nikon", "D800", Shade, 6,			{ 2.167969, 1, 1.207031, 0 } },
  { "Nikon", "D800", "2700K", 0,		{ 1.179688, 1, 2.488281, 0 } },
  { "Nikon", "D800", "5000K", 0,		{ 1.929688, 1, 1.417969, 0 } },

  // Copied from D800
  { "Nikon", "D800E", Incandescent, -6,		{ 1.390625, 1, 1.984375, 0 } },
  { "Nikon", "D800E", Incandescent, 0,		{ 1.289063, 1, 2.175781, 0 } },
  { "Nikon", "D800E", Incandescent, 6,		{ 1.199219, 1, 2.429688, 0 } },
  { "Nikon", "D800E", CoolWhiteFluorescent, -6,	{ 2.003906, 1, 1.769531, 0 } },
  { "Nikon", "D800E", CoolWhiteFluorescent, 0,	{ 1.765625, 1, 2.003906, 0 } },
  { "Nikon", "D800E", CoolWhiteFluorescent, 6,	{ 1.480469, 1, 2.257813, 0 } },
  { "Nikon", "D800E", DirectSunlight, -6,	{ 2.128906, 1, 1.230469, 0 } },
  { "Nikon", "D800E", DirectSunlight, 0,		{ 1.972656, 1, 1.375000, 0 } },
  { "Nikon", "D800E", DirectSunlight, 6,		{ 1.804688, 1, 1.531250, 0 } },
  { "Nikon", "D800E", Flash, -6,			{ 2.398438, 1, 1.125000, 0 } },
  { "Nikon", "D800E", Flash, 0,			{ 2.187500, 1, 1.207031, 0 } },
  { "Nikon", "D800E", Flash, 6,			{ 2.027344, 1, 1.300781, 0 } },
  { "Nikon", "D800E", Cloudy, -6,		{ 2.292969, 1, 1.156250, 0 } },
  { "Nikon", "D800E", Cloudy, 0,			{ 2.101563, 1, 1.253906, 0 } },
  { "Nikon", "D800E", Cloudy, 6,			{ 1.945313, 1, 1.402344, 0 } },
  { "Nikon", "D800E", Shade, -6,			{ 2.687500, 1, 1.015625, 0 } },
  { "Nikon", "D800E", Shade, 0,			{ 2.378906, 1, 1.128906, 0 } },
  { "Nikon", "D800E", Shade, 6,			{ 2.167969, 1, 1.207031, 0 } },
  { "Nikon", "D800E", "2700K", 0,		{ 1.179688, 1, 2.488281, 0 } },
  { "Nikon", "D800E", "5000K", 0,		{ 1.929688, 1, 1.417969, 0 } },

  /* D810 with firmware C: 1.10 L: 2.009 */
  { "Nikon", "D810", DirectSunlight                 , 0, { 1.953125, 1, 1.335938, 0 } },
  { "Nikon", "D810", Shade                          , 0, { 2.410156, 1, 1.089844, 0 } },
  { "Nikon", "D810", Cloudy                         , 0, { 2.089844, 1, 1.210938, 0 } },
  { "Nikon", "D810", Incandescent                   , 0, { 1.253906, 1, 2.136719, 0 } },
  { "Nikon", "D810", WarmWhiteFluorescent           , 0, { 1.253906, 1, 1.917969, 0 } },
  { "Nikon", "D810", CoolWhiteFluorescent           , 0, { 1.789062, 1, 1.968750, 0 } },
  { "Nikon", "D810", DayWhiteFluorescent            , 0, { 1.820312, 1, 1.355469, 0 } },
  { "Nikon", "D810", DaylightFluorescent            , 0, { 2.074219, 1, 1.093750, 0 } },
  { "Nikon", "D810", WhiteFluorescent               , 0, { 1.445312, 1, 2.210938, 0 } },
  { "Nikon", "D810", HighTempMercuryVaporFluorescent, 0, { 2.414062, 1, 1.320312, 0 } },
  { "Nikon", "D810", SodiumVaporFluorescent         , 0, { 1.164062, 1, 2.320312, 0 } },
  { "Nikon", "D810", Flash                          , 0, { 2.183594, 1, 1.171875, 0 } },

  /* Nikon D850 Firmware Version C: 1.03 LD: 2.017 */
  { "Nikon", "D850", DirectSunlight                 , -12, { 1.703125, 1, 1.533203125, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -11, { 1.719726562, 1, 1.522949219, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -10, { 1.736328125, 1, 1.51171875, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -9, { 1.752929688, 1, 1.499511719, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -8, { 1.76953125, 1, 1.486816406, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -7, { 1.786132812, 1, 1.474121094, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -6, { 1.802734375, 1, 1.460449219, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -5, { 1.819335938, 1, 1.446777344, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -4, { 1.8359375, 1, 1.432617188, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -3, { 1.852050781, 1, 1.418457031, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -2, { 1.868652344, 1, 1.404296875, 0 } },
  { "Nikon", "D850", DirectSunlight                 , -1, { 1.884765625, 1, 1.389648438, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 0, { 1.900390625, 1, 1.375488281, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 1, { 1.916015625, 1, 1.361328125, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 2, { 1.931640625, 1, 1.34765625, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 3, { 1.946777344, 1, 1.333496094, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 4, { 1.961425781, 1, 1.3203125, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 5, { 1.976074219, 1, 1.306640625, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 6, { 1.989746094, 1, 1.293945312, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 7, { 2.002929688, 1, 1.28125, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 8, { 2.016113281, 1, 1.269042969, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 9, { 2.031738281, 1, 1.257324219, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 10, { 2.049804688, 1, 1.244628906, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 11, { 2.067871094, 1, 1.233398438, 0 } },
  { "Nikon", "D850", DirectSunlight                 , 12, { 2.0859375, 1, 1.223144531, 0 } },
  { "Nikon", "D850", Shade                          , -12, { 2.13671875, 1, 1.197265625, 0 } },
  { "Nikon", "D850", Shade                          , -11, { 2.155761719, 1, 1.188964844, 0 } },
  { "Nikon", "D850", Shade                          , -10, { 2.174804688, 1, 1.181152344, 0 } },
  { "Nikon", "D850", Shade                          , -9, { 2.194335938, 1, 1.173828125, 0 } },
  { "Nikon", "D850", Shade                          , -8, { 2.214355469, 1, 1.166992188, 0 } },
  { "Nikon", "D850", Shade                          , -7, { 2.234863281, 1, 1.16015625, 0 } },
  { "Nikon", "D850", Shade                          , -6, { 2.255859375, 1, 1.153320312, 0 } },
  { "Nikon", "D850", Shade                          , -5, { 2.277832031, 1, 1.146972656, 0 } },
  { "Nikon", "D850", Shade                          , -4, { 2.300292969, 1, 1.140625, 0 } },
  { "Nikon", "D850", Shade                          , -3, { 2.323242188, 1, 1.133789062, 0 } },
  { "Nikon", "D850", Shade                          , -2, { 2.346679688, 1, 1.127441406, 0 } },
  { "Nikon", "D850", Shade                          , -1, { 2.37109375, 1, 1.12109375, 0 } },
  { "Nikon", "D850", Shade                          , 0, { 2.396484375, 1, 1.114257812, 0 } },
  { "Nikon", "D850", Shade                          , 1, { 2.422363281, 1, 1.107910156, 0 } },
  { "Nikon", "D850", Shade                          , 2, { 2.448730469, 1, 1.101074219, 0 } },
  { "Nikon", "D850", Shade                          , 3, { 2.4765625, 1, 1.09375, 0 } },
  { "Nikon", "D850", Shade                          , 4, { 2.504882812, 1, 1.086914062, 0 } },
  { "Nikon", "D850", Shade                          , 5, { 2.533691406, 1, 1.079589844, 0 } },
  { "Nikon", "D850", Shade                          , 6, { 2.563476562, 1, 1.071777344, 0 } },
  { "Nikon", "D850", Shade                          , 7, { 2.594238281, 1, 1.063964844, 0 } },
  { "Nikon", "D850", Shade                          , 8, { 2.625976562, 1, 1.056152344, 0 } },
  { "Nikon", "D850", Shade                          , 9, { 2.658203125, 1, 1.047851562, 0 } },
  { "Nikon", "D850", Shade                          , 10, { 2.690917969, 1, 1.039550781, 0 } },
  { "Nikon", "D850", Shade                          , 11, { 2.724609375, 1, 1.030761719, 0 } },
  { "Nikon", "D850", Shade                          , 12, { 2.759277344, 1, 1.021972656, 0 } },
  { "Nikon", "D850", Cloudy                         , -12, { 1.868652344, 1, 1.404296875, 0 } },
  { "Nikon", "D850", Cloudy                         , -11, { 1.884765625, 1, 1.389648438, 0 } },
  { "Nikon", "D850", Cloudy                         , -10, { 1.900390625, 1, 1.375488281, 0 } },
  { "Nikon", "D850", Cloudy                         , -9, { 1.916015625, 1, 1.361328125, 0 } },
  { "Nikon", "D850", Cloudy                         , -8, { 1.931640625, 1, 1.34765625, 0 } },
  { "Nikon", "D850", Cloudy                         , -7, { 1.946777344, 1, 1.333496094, 0 } },
  { "Nikon", "D850", Cloudy                         , -6, { 1.961425781, 1, 1.3203125, 0 } },
  { "Nikon", "D850", Cloudy                         , -5, { 1.976074219, 1, 1.306640625, 0 } },
  { "Nikon", "D850", Cloudy                         , -4, { 1.989746094, 1, 1.293945312, 0 } },
  { "Nikon", "D850", Cloudy                         , -3, { 2.002929688, 1, 1.28125, 0 } },
  { "Nikon", "D850", Cloudy                         , -2, { 2.016113281, 1, 1.269042969, 0 } },
  { "Nikon", "D850", Cloudy                         , -1, { 2.031738281, 1, 1.257324219, 0 } },
  { "Nikon", "D850", Cloudy                         , 0, { 2.049804688, 1, 1.244628906, 0 } },
  { "Nikon", "D850", Cloudy                         , 1, { 2.067871094, 1, 1.233398438, 0 } },
  { "Nikon", "D850", Cloudy                         , 2, { 2.0859375, 1, 1.223144531, 0 } },
  { "Nikon", "D850", Cloudy                         , 3, { 2.104003906, 1, 1.213378906, 0 } },
  { "Nikon", "D850", Cloudy                         , 4, { 2.122070312, 1, 1.204101562, 0 } },
  { "Nikon", "D850", Cloudy                         , 5, { 2.140625, 1, 1.195800781, 0 } },
  { "Nikon", "D850", Cloudy                         , 6, { 2.159667969, 1, 1.1875, 0 } },
  { "Nikon", "D850", Cloudy                         , 7, { 2.178710938, 1, 1.1796875, 0 } },
  { "Nikon", "D850", Cloudy                         , 8, { 2.198242188, 1, 1.172363281, 0 } },
  { "Nikon", "D850", Cloudy                         , 9, { 2.21875, 1, 1.165527344, 0 } },
  { "Nikon", "D850", Cloudy                         , 10, { 2.239257812, 1, 1.158691406, 0 } },
  { "Nikon", "D850", Cloudy                         , 11, { 2.260253906, 1, 1.151855469, 0 } },
  { "Nikon", "D850", Cloudy                         , 12, { 2.282226562, 1, 1.145507812, 0 } },
  { "Nikon", "D850", Incandescent                   , -12, { 1.11328125, 1, 2.446289062, 0 } },
  { "Nikon", "D850", Incandescent                   , -11, { 1.120117188, 1, 2.426269531, 0 } },
  { "Nikon", "D850", Incandescent                   , -10, { 1.127441406, 1, 2.40625, 0 } },
  { "Nikon", "D850", Incandescent                   , -9, { 1.134765625, 1, 2.386230469, 0 } },
  { "Nikon", "D850", Incandescent                   , -8, { 1.142578125, 1, 2.365722656, 0 } },
  { "Nikon", "D850", Incandescent                   , -7, { 1.149902344, 1, 2.345703125, 0 } },
  { "Nikon", "D850", Incandescent                   , -6, { 1.157714844, 1, 2.325683594, 0 } },
  { "Nikon", "D850", Incandescent                   , -5, { 1.165527344, 1, 2.305175781, 0 } },
  { "Nikon", "D850", Incandescent                   , -4, { 1.173339844, 1, 2.28515625, 0 } },
  { "Nikon", "D850", Incandescent                   , -3, { 1.181152344, 1, 2.265136719, 0 } },
  { "Nikon", "D850", Incandescent                   , -2, { 1.188964844, 1, 2.245605469, 0 } },
  { "Nikon", "D850", Incandescent                   , -1, { 1.196777344, 1, 2.226074219, 0 } },
  { "Nikon", "D850", Incandescent                   , 0, { 1.205078125, 1, 2.206542969, 0 } },
  { "Nikon", "D850", Incandescent                   , 1, { 1.212890625, 1, 2.1875, 0 } },
  { "Nikon", "D850", Incandescent                   , 2, { 1.220703125, 1, 2.168457031, 0 } },
  { "Nikon", "D850", Incandescent                   , 3, { 1.228027344, 1, 2.149902344, 0 } },
  { "Nikon", "D850", Incandescent                   , 4, { 1.235351562, 1, 2.131835938, 0 } },
  { "Nikon", "D850", Incandescent                   , 5, { 1.242675781, 1, 2.114257812, 0 } },
  { "Nikon", "D850", Incandescent                   , 6, { 1.250488281, 1, 2.096679688, 0 } },
  { "Nikon", "D850", Incandescent                   , 7, { 1.258300781, 1, 2.080078125, 0 } },
  { "Nikon", "D850", Incandescent                   , 8, { 1.266601562, 1, 2.063964844, 0 } },
  { "Nikon", "D850", Incandescent                   , 9, { 1.274902344, 1, 2.048828125, 0 } },
  { "Nikon", "D850", Incandescent                   , 10, { 1.282714844, 1, 2.033691406, 0 } },
  { "Nikon", "D850", Incandescent                   , 11, { 1.291503906, 1, 2.020019531, 0 } },
  { "Nikon", "D850", Incandescent                   , 12, { 1.299804688, 1, 2.006835938, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -12, { 1.116210938, 1, 2.083984375, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -11, { 1.123535156, 1, 2.075195312, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -10, { 1.130859375, 1, 2.06640625, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -9, { 1.138183594, 1, 2.057617188, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -8, { 1.145507812, 1, 2.049316406, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -7, { 1.152832031, 1, 2.040527344, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -6, { 1.160644531, 1, 2.031738281, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -5, { 1.168457031, 1, 2.022949219, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -4, { 1.176269531, 1, 2.014160156, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -3, { 1.184082031, 1, 2.005371094, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -2, { 1.191894531, 1, 1.996582031, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , -1, { 1.200195312, 1, 1.987792969, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 0, { 1.208496094, 1, 1.979003906, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 1, { 1.216796875, 1, 1.969726562, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 2, { 1.225097656, 1, 1.9609375, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 3, { 1.233398438, 1, 1.951660156, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 4, { 1.2421875, 1, 1.942871094, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 5, { 1.250488281, 1, 1.93359375, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 6, { 1.259277344, 1, 1.924804688, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 7, { 1.268554688, 1, 1.915527344, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 8, { 1.27734375, 1, 1.90625, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 9, { 1.286132812, 1, 1.896972656, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 10, { 1.295410156, 1, 1.887695312, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 11, { 1.3046875, 1, 1.877929688, 0 } },
  { "Nikon", "D850", WarmWhiteFluorescent           , 12, { 1.313964844, 1, 1.868652344, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -12, { 1.401367188, 1, 2.341308594, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -11, { 1.430175781, 1, 2.31640625, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -10, { 1.458496094, 1, 2.291992188, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -9, { 1.486328125, 1, 2.268066406, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -8, { 1.513671875, 1, 2.244140625, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -7, { 1.541015625, 1, 2.220703125, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -6, { 1.567871094, 1, 2.197753906, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -5, { 1.594726562, 1, 2.174804688, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -4, { 1.620605469, 1, 2.151855469, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -3, { 1.646484375, 1, 2.129882812, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -2, { 1.671875, 1, 2.107910156, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , -1, { 1.696777344, 1, 2.0859375, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 0, { 1.721191406, 1, 2.064453125, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 1, { 1.745605469, 1, 2.043457031, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 2, { 1.769042969, 1, 2.022460938, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 3, { 1.792480469, 1, 2.001464844, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 4, { 1.815917969, 1, 1.980957031, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 5, { 1.838378906, 1, 1.9609375, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 6, { 1.860351562, 1, 1.940917969, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 7, { 1.882324219, 1, 1.921386719, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 8, { 1.903808594, 1, 1.901367188, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 9, { 1.924804688, 1, 1.882324219, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 10, { 1.9453125, 1, 1.86328125, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 11, { 1.965820312, 1, 1.844238281, 0 } },
  { "Nikon", "D850", CoolWhiteFluorescent           , 12, { 1.985351562, 1, 1.825195312, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -12, { 1.622558594, 1, 1.555175781, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -11, { 1.634765625, 1, 1.54296875, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -10, { 1.646484375, 1, 1.530761719, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -9, { 1.658203125, 1, 1.518066406, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -8, { 1.670410156, 1, 1.505371094, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -7, { 1.682617188, 1, 1.4921875, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -6, { 1.694824219, 1, 1.479492188, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -5, { 1.70703125, 1, 1.466308594, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -4, { 1.719238281, 1, 1.453125, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -3, { 1.731933594, 1, 1.439453125, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -2, { 1.744628906, 1, 1.42578125, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , -1, { 1.756835938, 1, 1.412109375, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 0, { 1.76953125, 1, 1.3984375, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 1, { 1.782714844, 1, 1.384277344, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 2, { 1.795410156, 1, 1.370117188, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 3, { 1.808105469, 1, 1.35546875, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 4, { 1.821289062, 1, 1.341308594, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 5, { 1.834472656, 1, 1.326171875, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 6, { 1.84765625, 1, 1.311523438, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 7, { 1.860839844, 1, 1.296386719, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 8, { 1.874023438, 1, 1.28125, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 9, { 1.887695312, 1, 1.266113281, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 10, { 1.901367188, 1, 1.250488281, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 11, { 1.914550781, 1, 1.234863281, 0 } },
  { "Nikon", "D850", DayWhiteFluorescent            , 12, { 1.928222656, 1, 1.21875, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -12, { 1.853027344, 1, 1.305664062, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -11, { 1.866210938, 1, 1.290527344, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -10, { 1.879394531, 1, 1.275390625, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -9, { 1.893066406, 1, 1.259765625, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -8, { 1.906738281, 1, 1.244140625, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -7, { 1.920410156, 1, 1.228515625, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -6, { 1.934082031, 1, 1.212402344, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -5, { 1.947753906, 1, 1.196289062, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -4, { 1.961425781, 1, 1.1796875, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -3, { 1.975585938, 1, 1.163085938, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -2, { 1.989746094, 1, 1.146484375, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , -1, { 2.00390625, 1, 1.129394531, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 0, { 2.018066406, 1, 1.112304688, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 1, { 2.032226562, 1, 1.095214844, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 2, { 2.046875, 1, 1.077636719, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 3, { 2.061035156, 1, 1.060058594, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 4, { 2.075683594, 1, 1.041992188, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 5, { 2.090332031, 1, 1.023925781, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 6, { 2.104980469, 1, 1.005371094, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 7, { 2.120117188, 1, 1.0, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 8, { 2.134765625, 1, 1.0, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 9, { 2.149902344, 1, 1.0, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 10, { 2.165039062, 1, 1.0, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 11, { 2.180175781, 1, 1.0, 0 } },
  { "Nikon", "D850", DaylightFluorescent            , 12, { 2.1953125, 1, 1.0, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -12, { 0.9990234375, 1, 2.685058594, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -11, { 1.034179688, 1, 2.65625, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -10, { 1.068359375, 1, 2.627929688, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -9, { 1.102050781, 1, 2.600097656, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -8, { 1.134765625, 1, 2.572265625, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -7, { 1.166015625, 1, 2.544433594, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -6, { 1.197265625, 1, 2.517578125, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -5, { 1.228515625, 1, 2.490722656, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -4, { 1.258789062, 1, 2.463867188, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -3, { 1.2890625, 1, 2.437988281, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -2, { 1.319335938, 1, 2.412109375, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , -1, { 1.348632812, 1, 2.386230469, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 0, { 1.377929688, 1, 2.361328125, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 1, { 1.407226562, 1, 2.336425781, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 2, { 1.435546875, 1, 2.311523438, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 3, { 1.463867188, 1, 2.287109375, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 4, { 1.491699219, 1, 2.263183594, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 5, { 1.51953125, 1, 2.239257812, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 6, { 1.546386719, 1, 2.215820312, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 7, { 1.573242188, 1, 2.192871094, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 8, { 1.599609375, 1, 2.169921875, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 9, { 1.625976562, 1, 2.147460938, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 10, { 1.651367188, 1, 2.125488281, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 11, { 1.676757812, 1, 2.103515625, 0 } },
  { "Nikon", "D850", WhiteFluorescent               , 12, { 1.701660156, 1, 2.081542969, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -12, { 2.243164062, 1, 1.549316406, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -11, { 2.256835938, 1, 1.532714844, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -10, { 2.26953125, 1, 1.516601562, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -9, { 2.282226562, 1, 1.500976562, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -8, { 2.294433594, 1, 1.484863281, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -7, { 2.306152344, 1, 1.469726562, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -6, { 2.317871094, 1, 1.454101562, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -5, { 2.329101562, 1, 1.438964844, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -4, { 2.33984375, 1, 1.424316406, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -3, { 2.350097656, 1, 1.409667969, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -2, { 2.360351562, 1, 1.395507812, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, -1, { 2.370117188, 1, 1.381347656, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 0, { 2.379882812, 1, 1.3671875, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 1, { 2.388671875, 1, 1.354003906, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 2, { 2.397949219, 1, 1.340332031, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 3, { 2.40625, 1, 1.327636719, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 4, { 2.414550781, 1, 1.314453125, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 5, { 2.422363281, 1, 1.302246094, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 6, { 2.430175781, 1, 1.290039062, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 7, { 2.4375, 1, 1.277832031, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 8, { 2.444824219, 1, 1.266113281, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 9, { 2.451660156, 1, 1.254882812, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 10, { 2.458007812, 1, 1.243652344, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 11, { 2.464355469, 1, 1.232910156, 0 } },
  { "Nikon", "D850", HighTempMercuryVaporFluorescent, 12, { 2.470703125, 1, 1.222167969, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -12, { 1.053710938, 1, 2.75, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -11, { 1.059082031, 1, 2.72265625, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -10, { 1.063964844, 1, 2.696777344, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -9, { 1.069335938, 1, 2.671386719, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -8, { 1.07421875, 1, 2.646972656, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -7, { 1.079589844, 1, 2.623535156, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -6, { 1.084960938, 1, 2.601074219, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -5, { 1.090820312, 1, 2.579101562, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -4, { 1.096191406, 1, 2.557617188, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -3, { 1.102050781, 1, 2.537109375, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -2, { 1.108398438, 1, 2.516601562, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , -1, { 1.114746094, 1, 2.496582031, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 0, { 1.121582031, 1, 2.477050781, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 1, { 1.128417969, 1, 2.457519531, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 2, { 1.135742188, 1, 2.438476562, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 3, { 1.143554688, 1, 2.418945312, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 4, { 1.151855469, 1, 2.399902344, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 5, { 1.16015625, 1, 2.380371094, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 6, { 1.169433594, 1, 2.360839844, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 7, { 1.179199219, 1, 2.340820312, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 8, { 1.189453125, 1, 2.3203125, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 9, { 1.199707031, 1, 2.299316406, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 10, { 1.2109375, 1, 2.277832031, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 11, { 1.223144531, 1, 2.255859375, 0 } },
  { "Nikon", "D850", SodiumVaporFluorescent         , 12, { 1.235351562, 1, 2.232421875, 0 } },
  { "Nikon", "D850", Flash                          , -12, { 1.961914062, 1, 1.300292969, 0 } },
  { "Nikon", "D850", Flash                          , -11, { 1.981933594, 1, 1.290527344, 0 } },
  { "Nikon", "D850", Flash                          , -10, { 2.000976562, 1, 1.280761719, 0 } },
  { "Nikon", "D850", Flash                          , -9, { 2.01953125, 1, 1.271972656, 0 } },
  { "Nikon", "D850", Flash                          , -8, { 2.036621094, 1, 1.263183594, 0 } },
  { "Nikon", "D850", Flash                          , -7, { 2.053710938, 1, 1.254882812, 0 } },
  { "Nikon", "D850", Flash                          , -6, { 2.069824219, 1, 1.246582031, 0 } },
  { "Nikon", "D850", Flash                          , -5, { 2.0859375, 1, 1.239257812, 0 } },
  { "Nikon", "D850", Flash                          , -4, { 2.1015625, 1, 1.231933594, 0 } },
  { "Nikon", "D850", Flash                          , -3, { 2.1171875, 1, 1.224609375, 0 } },
  { "Nikon", "D850", Flash                          , -2, { 2.132324219, 1, 1.217773438, 0 } },
  { "Nikon", "D850", Flash                          , -1, { 2.147949219, 1, 1.2109375, 0 } },
  { "Nikon", "D850", Flash                          , 0, { 2.1640625, 1, 1.204101562, 0 } },
  { "Nikon", "D850", Flash                          , 1, { 2.180175781, 1, 1.196777344, 0 } },
  { "Nikon", "D850", Flash                          , 2, { 2.197265625, 1, 1.189941406, 0 } },
  { "Nikon", "D850", Flash                          , 3, { 2.214355469, 1, 1.182617188, 0 } },
  { "Nikon", "D850", Flash                          , 4, { 2.232910156, 1, 1.17578125, 0 } },
  { "Nikon", "D850", Flash                          , 5, { 2.251953125, 1, 1.168457031, 0 } },
  { "Nikon", "D850", Flash                          , 6, { 2.272460938, 1, 1.161132812, 0 } },
  { "Nikon", "D850", Flash                          , 7, { 2.293945312, 1, 1.154296875, 0 } },
  { "Nikon", "D850", Flash                          , 8, { 2.31640625, 1, 1.146484375, 0 } },
  { "Nikon", "D850", Flash                          , 9, { 2.340820312, 1, 1.139160156, 0 } },
  { "Nikon", "D850", Flash                          , 10, { 2.366699219, 1, 1.131347656, 0 } },
  { "Nikon", "D850", Flash                          , 11, { 2.39453125, 1, 1.123535156, 0 } },
  { "Nikon", "D850", Flash                          , 12, { 2.424316406, 1, 1.115234375, 0 } },

  { "Nikon", "D40", Incandescent, -3,		{ 1.492188, 1, 2.164063, 0 } },
  { "Nikon", "D40", Incandescent, -2,		{ 1.437500, 1, 2.367188, 0 } },
  { "Nikon", "D40", Incandescent, -1,		{ 1.417969, 1, 2.414062, 0 } },
  { "Nikon", "D40", Incandescent, 0,		{ 1.375000, 1, 2.511719, 0 } },
  { "Nikon", "D40", Incandescent, 1,		{ 1.324219, 1, 2.628906, 0 } },
  { "Nikon", "D40", Incandescent, 2,		{ 1.277344, 1, 2.753906, 0 } },
  { "Nikon", "D40", Incandescent, 3,		{ 1.222656, 1, 2.914063, 0 } },
  { "Nikon", "D40", Fluorescent, -3,		{ 2.738281, 1, 1.492188, 0 } },
  { "Nikon", "D40", Fluorescent, -2,		{ 2.417969, 1, 1.246094, 0 } },
  { "Nikon", "D40", Fluorescent, -1,		{ 2.093750, 1, 1.570312, 0 } },
  { "Nikon", "D40", Fluorescent, 0,		{ 2.007813, 1, 2.269531, 0 } },
  { "Nikon", "D40", Fluorescent, 1,		{ 1.613281, 1, 2.593750, 0 } },
  { "Nikon", "D40", Fluorescent, 2,		{ 1.394531, 1, 2.343750, 0 } },
  { "Nikon", "D40", Fluorescent, 3,		{ 1.210938, 1, 2.621094, 0 } },
  { "Nikon", "D40", DirectSunlight, -3,		{ 2.328125, 1, 1.371094, 0 } },
  { "Nikon", "D40", DirectSunlight, -2,		{ 2.269531, 1, 1.394531, 0 } },
  { "Nikon", "D40", DirectSunlight, -1,		{ 2.230469, 1, 1.410156, 0 } },
  { "Nikon", "D40", DirectSunlight, 0,		{ 2.195313, 1, 1.421875, 0 } },
  { "Nikon", "D40", DirectSunlight, 1,		{ 2.113281, 1, 1.445312, 0 } },
  { "Nikon", "D40", DirectSunlight, 2,		{ 2.070312, 1, 1.453125, 0 } },
  { "Nikon", "D40", DirectSunlight, 3,		{ 2.039063, 1, 1.457031, 0 } },
  { "Nikon", "D40", Flash, -3,			{ 2.667969, 1, 1.214844, 0 } },
  { "Nikon", "D40", Flash, -2,			{ 2.605469, 1, 1.234375, 0 } },
  { "Nikon", "D40", Flash, -1,			{ 2.539062, 1, 1.257812, 0 } },
  { "Nikon", "D40", Flash, 0,			{ 2.464844, 1, 1.281250, 0 } },
  { "Nikon", "D40", Flash, 1,			{ 2.390625, 1, 1.312500, 0 } },
  { "Nikon", "D40", Flash, 2,			{ 2.308594, 1, 1.343750, 0 } },
  { "Nikon", "D40", Flash, 3,			{ 2.222656, 1, 1.386719, 0 } },
  { "Nikon", "D40", Cloudy, -3,			{ 2.570313, 1, 1.246094, 0 } },
  { "Nikon", "D40", Cloudy, -2,			{ 2.523438, 1, 1.269531, 0 } },
  { "Nikon", "D40", Cloudy, -1,			{ 2.476562, 1, 1.296875, 0 } },
  { "Nikon", "D40", Cloudy, 0,			{ 2.429688, 1, 1.320313, 0 } },
  { "Nikon", "D40", Cloudy, 1,			{ 2.382812, 1, 1.343750, 0 } },
  { "Nikon", "D40", Cloudy, 2,			{ 2.328125, 1, 1.371094, 0 } },
  { "Nikon", "D40", Cloudy, 3,			{ 2.269531, 1, 1.394531, 0 } },
  { "Nikon", "D40", Shade, -3,			{ 2.957031, 1, 1.054688, 0 } },
  { "Nikon", "D40", Shade, -2,			{ 2.921875, 1, 1.074219, 0 } },
  { "Nikon", "D40", Shade, -1,			{ 2.878906, 1, 1.097656, 0 } },
  { "Nikon", "D40", Shade, 0,			{ 2.820313, 1, 1.125000, 0 } },
  { "Nikon", "D40", Shade, 1,			{ 2.746094, 1, 1.160156, 0 } },
  { "Nikon", "D40", Shade, 2,			{ 2.671875, 1, 1.195312, 0 } },
  { "Nikon", "D40", Shade, 3,			{ 2.597656, 1, 1.234375, 0 } },

  { "Nikon", "D40X", Incandescent, -3,		{ 1.234375, 1, 2.140625, 0 } },
  { "Nikon", "D40X", Incandescent, 0,		{ 1.148438, 1, 2.386719, 0 } },
  { "Nikon", "D40X", Incandescent, 3,		{ 1.039062, 1, 2.734375, 0 } },
  { "Nikon", "D40X", Fluorescent, -3,		{ 2.296875, 1, 1.398438, 0 } },
  { "Nikon", "D40X", Fluorescent, 0,		{ 1.683594, 1, 2.117188, 0 } },
  { "Nikon", "D40X", Fluorescent, 3,		{ 1.000000, 1, 2.527344, 0 } },
  { "Nikon", "D40X", DirectSunlight, -3,	{ 1.882812, 1, 1.300781, 0 } },
  { "Nikon", "D40X", DirectSunlight, 0,		{ 1.792969, 1, 1.371094, 0 } },
  { "Nikon", "D40X", DirectSunlight, 3,		{ 1.695312, 1, 1.437500, 0 } },
  { "Nikon", "D40X", Flash, -3,			{ 2.089844, 1, 1.132812, 0 } },
  { "Nikon", "D40X", Flash, 0,			{ 1.949219, 1, 1.187500, 0 } },
  { "Nikon", "D40X", Flash, 3,			{ 1.769531, 1, 1.269531, 0 } },
  { "Nikon", "D40X", Cloudy, -3,		{ 2.070312, 1, 1.191406, 0 } },
  { "Nikon", "D40X", Cloudy, 0,			{ 1.960938, 1, 1.253906, 0 } },
  { "Nikon", "D40X", Cloudy, 3,			{ 1.835938, 1, 1.332031, 0 } },
  { "Nikon", "D40X", Shade, -3,			{ 2.414062, 1, 1.042969, 0 } },
  { "Nikon", "D40X", Shade, 0,			{ 2.277344, 1, 1.089844, 0 } },
  { "Nikon", "D40X", Shade, 3,			{ 2.085938, 1, 1.183594, 0 } },

  { "Nikon", "D50", Incandescent, 0,		{ 1.328125, 1, 2.500000, 0 } },
  { "Nikon", "D50", Fluorescent, 0,		{ 1.945312, 1, 2.191406, 0 } },
  { "Nikon", "D50", DirectSunlight, 0,		{ 2.140625, 1, 1.398438, 0 } },
  { "Nikon", "D50", Flash, 0,			{ 2.398438, 1, 1.339844, 0 } },
  { "Nikon", "D50", Cloudy, 0,			{ 2.360269, 1, 1.282828, 0 } },
  { "Nikon", "D50", Shade, 0,			{ 2.746094, 1, 1.156250, 0 } },

  { "Nikon", "D60", DirectSunlight, 0,		{ 1.792969, 1, 1.371094, 0 } },
  { "Nikon", "D60", Flash, 0,			{ 2.007813, 1, 1.187500, 0 } },
  { "Nikon", "D60", Cloudy, 0,			{ 1.960937, 1, 1.253906, 0 } },
  { "Nikon", "D60", Shade, 0,			{ 2.277344, 1, 1.089844, 0 } },
  { "Nikon", "D60", Incandescent, 0,		{ 1.148437, 1, 2.382812, 0 } },
  { "Nikon", "D60", SodiumVaporFluorescent, 0,	{ 1.035156, 1, 2.468750, 0 } },
  { "Nikon", "D60", WarmWhiteFluorescent, 0,	{ 1.136719, 1, 2.167969, 0 } },
  { "Nikon", "D60", WhiteFluorescent, 0,	{ 1.343750, 1, 2.480469, 0 } },
  { "Nikon", "D60", CoolWhiteFluorescent, 0,	{ 1.683594, 1, 2.117187, 0 } },
  { "Nikon", "D60", DayWhiteFluorescent, 0,	{ 1.679688, 1, 1.414063, 0 } },
  { "Nikon", "D60", DaylightFluorescent, 0,	{ 1.953125, 1, 1.121094, 0 } },
  { "Nikon", "D60", HighTempMercuryVaporFluorescent, 0, { 2.296875, 1, 1.398438, 0 } },

  { "Nikon", "D70", Incandescent, -3,		{ 1.429688, 1, 2.539063, 0 } }, /*3300K*/
  { "Nikon", "D70", Incandescent, -2,		{ 1.398438, 1, 2.632813, 0 } }, /*3200K*/
  { "Nikon", "D70", Incandescent, -1,		{ 1.378906, 1, 2.687500, 0 } }, /*3100K*/
  { "Nikon", "D70", Incandescent, 0,		{ 1.343750, 1, 2.816406, 0 } }, /*3000K*/
  { "Nikon", "D70", Incandescent, 1,		{ 1.312500, 1, 2.937500, 0 } }, /*2900K*/
  { "Nikon", "D70", Incandescent, 2,		{ 1.281250, 1, 3.089844, 0 } }, /*2800K*/
  { "Nikon", "D70", Incandescent, 3,		{ 1.253906, 1, 3.250000, 0 } }, /*2700K*/
  { "Nikon", "D70", Fluorescent, -3,		{ 2.734375, 1, 1.621094, 0 } }, /*7200K*/
  { "Nikon", "D70", Fluorescent, -2,		{ 2.417969, 1, 1.343750, 0 } }, /*6500K*/
  { "Nikon", "D70", Fluorescent, -1,		{ 2.078125, 1, 1.691406, 0 } }, /*5000K*/
  { "Nikon", "D70", Fluorescent, 0,		{ 1.964844, 1, 2.476563, 0 } }, /*4200K*/
  { "Nikon", "D70", Fluorescent, 1,		{ 1.566406, 1, 2.753906, 0 } }, /*3700K*/
  { "Nikon", "D70", Fluorescent, 2,		{ 1.406250, 1, 2.550781, 0 } }, /*3000K*/
  { "Nikon", "D70", Fluorescent, 3,		{ 1.312500, 1, 2.562500, 0 } }, /*2700K*/
  { "Nikon", "D70", DirectSunlight, -3,		{ 2.156250, 1, 1.523438, 0 } }, /*5600K*/
  { "Nikon", "D70", DirectSunlight, -2,		{ 2.109375, 1, 1.562500, 0 } }, /*5400K*/
  { "Nikon", "D70", DirectSunlight, -1,		{ 2.089844, 1, 1.574219, 0 } }, /*5300K*/
  { "Nikon", "D70", DirectSunlight, 0,		{ 2.062500, 1, 1.597656, 0 } }, /*5200K*/
  { "Nikon", "D70", DirectSunlight, 1,		{ 2.007813, 1, 1.648438, 0 } }, /*5000K*/
  { "Nikon", "D70", DirectSunlight, 2,		{ 1.980469, 1, 1.671875, 0 } }, /*4900K*/
  { "Nikon", "D70", DirectSunlight, 3,		{ 1.953125, 1, 1.695313, 0 } }, /*4800K*/
  { "Nikon", "D70", Flash, -3,			{ 2.578125, 1, 1.476563, 0 } }, /*6000K*/
  { "Nikon", "D70", Flash, -2,			{ 2.535156, 1, 1.484375, 0 } }, /*5800K*/
  { "Nikon", "D70", Flash, -1,			{ 2.488281, 1, 1.492188, 0 } }, /*5600K*/
  { "Nikon", "D70", Flash, 0,			{ 2.441406, 1, 1.500000, 0 } }, /*5400K*/
  { "Nikon", "D70", Flash, 1,			{ 2.421875, 1, 1.507813, 0 } }, /*5200K*/
  { "Nikon", "D70", Flash, 2,			{ 2.398438, 1, 1.515625, 0 } }, /*5000K*/
  { "Nikon", "D70", Flash, 3,			{ 2.378906, 1, 1.523438, 0 } }, /*4800K*/
  { "Nikon", "D70", Cloudy, -3,			{ 2.375000, 1, 1.386719, 0 } }, /*6600K*/
  { "Nikon", "D70", Cloudy, -2,			{ 2.343750, 1, 1.406250, 0 } }, /*6400K*/
  { "Nikon", "D70", Cloudy, -1,			{ 2.300781, 1, 1.429688, 0 } }, /*6200K*/
  { "Nikon", "D70", Cloudy, 0,			{ 2.257813, 1, 1.457031, 0 } }, /*6000K*/
  { "Nikon", "D70", Cloudy, 1,			{ 2.207031, 1, 1.488281, 0 } }, /*5800K*/
  { "Nikon", "D70", Cloudy, 2,			{ 2.156250, 1, 1.523438, 0 } }, /*5600K*/
  { "Nikon", "D70", Cloudy, 3,			{ 2.109375, 1, 1.562500, 0 } }, /*5400K*/
  { "Nikon", "D70", Shade, -3,			{ 2.757813, 1, 1.226563, 0 } }, /*9200K*/
  { "Nikon", "D70", Shade, -2,			{ 2.710938, 1, 1.242188, 0 } }, /*8800K*/
  { "Nikon", "D70", Shade, -1,			{ 2.660156, 1, 1.257813, 0 } }, /*8400K*/
  { "Nikon", "D70", Shade, 0,			{ 2.613281, 1, 1.277344, 0 } }, /*8000K*/
  { "Nikon", "D70", Shade, 1,			{ 2.531250, 1, 1.308594, 0 } }, /*7500K*/
  { "Nikon", "D70", Shade, 2,			{ 2.472656, 1, 1.335938, 0 } }, /*7100K*/
  { "Nikon", "D70", Shade, 3,			{ 2.394531, 1, 1.375000, 0 } }, /*6700K*/

  { "Nikon", "D70s", Incandescent, -3,		{ 1.429688, 1, 2.539063, 0 } }, /*3300K*/
  { "Nikon", "D70s", Incandescent, -2,		{ 1.398438, 1, 2.632813, 0 } }, /*3200K*/
  { "Nikon", "D70s", Incandescent, -1,		{ 1.378906, 1, 2.687500, 0 } }, /*3100K*/
  { "Nikon", "D70s", Incandescent, 0,		{ 1.343750, 1, 2.816406, 0 } }, /*3000K*/
  { "Nikon", "D70s", Incandescent, 1,		{ 1.312500, 1, 2.937500, 0 } }, /*2900K*/
  { "Nikon", "D70s", Incandescent, 2,		{ 1.281250, 1, 3.089844, 0 } }, /*2800K*/
  { "Nikon", "D70s", Incandescent, 3,		{ 1.253906, 1, 3.250000, 0 } }, /*2700K*/
  { "Nikon", "D70s", Fluorescent, -3,		{ 2.734375, 1, 1.621094, 0 } }, /*7200K*/
  { "Nikon", "D70s", Fluorescent, -2,		{ 2.417969, 1, 1.343750, 0 } }, /*6500K*/
  { "Nikon", "D70s", Fluorescent, -1,		{ 2.078125, 1, 1.691406, 0 } }, /*5000K*/
  { "Nikon", "D70s", Fluorescent, 0,		{ 1.964844, 1, 2.476563, 0 } }, /*4200K*/
  { "Nikon", "D70s", Fluorescent, 1,		{ 1.566406, 1, 2.753906, 0 } }, /*3700K*/
  { "Nikon", "D70s", Fluorescent, 2,		{ 1.406250, 1, 2.550781, 0 } }, /*3000K*/
  { "Nikon", "D70s", Fluorescent, 3,		{ 1.312500, 1, 2.562500, 0 } }, /*2700K*/
  { "Nikon", "D70s", DirectSunlight, -3,	{ 2.156250, 1, 1.523438, 0 } }, /*5600K*/
  { "Nikon", "D70s", DirectSunlight, -2,	{ 2.109375, 1, 1.562500, 0 } }, /*5400K*/
  { "Nikon", "D70s", DirectSunlight, -1,	{ 2.089844, 1, 1.574219, 0 } }, /*5300K*/
  { "Nikon", "D70s", DirectSunlight, 0,		{ 2.062500, 1, 1.597656, 0 } }, /*5200K*/
  { "Nikon", "D70s", DirectSunlight, 1,		{ 2.007813, 1, 1.648438, 0 } }, /*5000K*/
  { "Nikon", "D70s", DirectSunlight, 2,		{ 1.980469, 1, 1.671875, 0 } }, /*4900K*/
  { "Nikon", "D70s", DirectSunlight, 3,		{ 1.953125, 1, 1.695313, 0 } }, /*4800K*/
  { "Nikon", "D70s", Flash, -3,			{ 2.578125, 1, 1.476563, 0 } }, /*6000K*/
  { "Nikon", "D70s", Flash, -2,			{ 2.535156, 1, 1.484375, 0 } }, /*5800K*/
  { "Nikon", "D70s", Flash, -1,			{ 2.488281, 1, 1.492188, 0 } }, /*5600K*/
  { "Nikon", "D70s", Flash, 0,			{ 2.441406, 1, 1.500000, 0 } }, /*5400K*/
  { "Nikon", "D70s", Flash, 1,			{ 2.421875, 1, 1.507813, 0 } }, /*5200K*/
  { "Nikon", "D70s", Flash, 2,			{ 2.398438, 1, 1.515625, 0 } }, /*5000K*/
  { "Nikon", "D70s", Flash, 3,			{ 2.378906, 1, 1.523438, 0 } }, /*4800K*/
  { "Nikon", "D70s", Cloudy, -3,		{ 2.375000, 1, 1.386719, 0 } }, /*6600K*/
  { "Nikon", "D70s", Cloudy, -2,		{ 2.343750, 1, 1.406250, 0 } }, /*6400K*/
  { "Nikon", "D70s", Cloudy, -1,		{ 2.300781, 1, 1.429688, 0 } }, /*6200K*/
  { "Nikon", "D70s", Cloudy, 0,			{ 2.257813, 1, 1.457031, 0 } }, /*6000K*/
  { "Nikon", "D70s", Cloudy, 1,			{ 2.207031, 1, 1.488281, 0 } }, /*5800K*/
  { "Nikon", "D70s", Cloudy, 2,			{ 2.156250, 1, 1.523438, 0 } }, /*5600K*/
  { "Nikon", "D70s", Cloudy, 3,			{ 2.109375, 1, 1.562500, 0 } }, /*5400K*/
  { "Nikon", "D70s", Shade, -3,			{ 2.757813, 1, 1.226563, 0 } }, /*9200K*/
  { "Nikon", "D70s", Shade, -2,			{ 2.710938, 1, 1.242188, 0 } }, /*8800K*/
  { "Nikon", "D70s", Shade, -1,			{ 2.660156, 1, 1.257813, 0 } }, /*8400K*/
  { "Nikon", "D70s", Shade, 0,			{ 2.613281, 1, 1.277344, 0 } }, /*8000K*/
  { "Nikon", "D70s", Shade, 1,			{ 2.531250, 1, 1.308594, 0 } }, /*7500K*/
  { "Nikon", "D70s", Shade, 2,			{ 2.472656, 1, 1.335938, 0 } }, /*7100K*/
  { "Nikon", "D70s", Shade, 3,			{ 2.394531, 1, 1.375000, 0 } }, /*6700K*/

  { "Nikon", "D80", Incandescent, -3,		{ 1.234375, 1, 2.140625, 0 } },
  { "Nikon", "D80", Incandescent, 0,		{ 1.148438, 1, 2.386719, 0 } },
  { "Nikon", "D80", Incandescent, 3,		{ 1.039062, 1, 2.734375, 0 } },
  { "Nikon", "D80", Fluorescent, -3,		{ 2.296875, 1, 1.398438, 0 } },
  { "Nikon", "D80", Fluorescent, 0,		{ 1.683594, 1, 2.117188, 0 } },
  { "Nikon", "D80", Fluorescent, 3,		{ 1.000000, 1, 2.527344, 0 } },
  { "Nikon", "D80", Daylight, -3,		{ 1.882812, 1, 1.300781, 0 } },
  { "Nikon", "D80", Daylight, 0,		{ 1.792969, 1, 1.371094, 0 } },
  { "Nikon", "D80", Daylight, 3,		{ 1.695312, 1, 1.437500, 0 } },
  { "Nikon", "D80", Flash, -3,			{ 2.070312, 1, 1.144531, 0 } },
  { "Nikon", "D80", Flash, 0,			{ 2.007812, 1, 1.242188, 0 } },
  { "Nikon", "D80", Flash, 3,			{ 1.972656, 1, 1.156250, 0 } },
  { "Nikon", "D80", Cloudy, -3,			{ 2.070312, 1, 1.191406, 0 } },
  { "Nikon", "D80", Cloudy, 0,			{ 1.960938, 1, 1.253906, 0 } },
  { "Nikon", "D80", Cloudy, 3,			{ 1.835938, 1, 1.332031, 0 } },
  { "Nikon", "D80", Shade, -3,			{ 2.414062, 1, 1.042969, 0 } },
  { "Nikon", "D80", Shade, 0,			{ 2.277344, 1, 1.089844, 0 } },
  { "Nikon", "D80", Shade, 3,			{ 2.085938, 1, 1.183594, 0 } },
  { "Nikon", "D80", "5000K", 0,			{ 1.746094, 1, 1.410156, 0 } },

  { "Nikon", "D90", Incandescent, -6,		{ 1.273438, 1, 1.906250, 0 } },
  { "Nikon", "D90", Incandescent, 0,		{ 1.179688, 1, 2.097656, 0 } },
  { "Nikon", "D90", Incandescent, 6,		{ 1.113281, 1, 2.320313, 0 } },
  { "Nikon", "D90", SodiumVaporFluorescent, -6,	{ 1.164063, 1, 2.058594, 0 } },
  { "Nikon", "D90", SodiumVaporFluorescent, 0,	{ 1.062500, 1, 2.289063, 0 } },
  { "Nikon", "D90", SodiumVaporFluorescent, 6,	{ 1.000000, 1, 2.554688, 0 } },
  { "Nikon", "D90", WarmWhiteFluorescent, -6,	{ 1.285156, 1, 1.761719, 0 } },
  { "Nikon", "D90", WarmWhiteFluorescent, 0,	{ 1.191406, 1, 1.871094, 0 } },
  { "Nikon", "D90", WarmWhiteFluorescent, 6,	{ 1.105469, 1, 1.968750, 0 } },
  { "Nikon", "D90", WhiteFluorescent, -6,	{ 1.628906, 1, 1.953125, 0 } },
  { "Nikon", "D90", WhiteFluorescent, 0,	{ 1.343750, 1, 2.183594, 0 } },
  { "Nikon", "D90", WhiteFluorescent, 6,	{ 1.000000, 1, 2.429688, 0 } },
  { "Nikon", "D90", CoolWhiteFluorescent, -6,	{ 1.867188, 1, 1.722656, 0 } },
  { "Nikon", "D90", CoolWhiteFluorescent, 0,	{ 1.644531, 1, 1.937500, 0 } },
  { "Nikon", "D90", CoolWhiteFluorescent, 6,	{ 1.363281, 1, 2.167969, 0 } },
  { "Nikon", "D90", DayWhiteFluorescent, -6,	{ 1.843750, 1, 1.160156, 0 } },
  { "Nikon", "D90", DayWhiteFluorescent, 0,	{ 1.695313, 1, 1.312500, 0 } },
  { "Nikon", "D90", DayWhiteFluorescent, 6,	{ 1.562500, 1, 1.457031, 0 } },
  { "Nikon", "D90", DaylightFluorescent, -6,	{ 2.089844, 1, 1.000000, 0 } },
  { "Nikon", "D90", DaylightFluorescent, 0,	{ 1.925781, 1, 1.074219, 0 } },
  { "Nikon", "D90", DaylightFluorescent, 6,	{ 1.773438, 1, 1.234375, 0 } },
  { "Nikon", "D90", HighTempMercuryVaporFluorescent, -6, { 2.308594, 1, 1.132813, 0 } },
  { "Nikon", "D90", HighTempMercuryVaporFluorescent, 0, { 2.207031, 1, 1.292969, 0 } },
  { "Nikon", "D90", HighTempMercuryVaporFluorescent, 6, { 2.085938, 1, 1.468750, 0 } },
  { "Nikon", "D90", DirectSunlight, -6,		{ 1.949219, 1, 1.171875, 0 } },
  { "Nikon", "D90", DirectSunlight, 0,		{ 1.800781, 1, 1.308594, 0 } },
  { "Nikon", "D90", DirectSunlight, 6,		{ 1.640625, 1, 1.457031, 0 } },
  { "Nikon", "D90", Flash, -6,			{ 2.218750, 1, 1.062500, 0 } },
  { "Nikon", "D90", Flash, 0,			{ 1.976563, 1, 1.152344, 0 } },
  { "Nikon", "D90", Flash, 6,			{ 1.789063, 1, 1.253906, 0 } },
  { "Nikon", "D90", Cloudy, -6,			{ 2.093750, 1, 1.093750, 0 } },
  { "Nikon", "D90", Cloudy, 0,			{ 1.917969, 1, 1.187500, 0 } },
  { "Nikon", "D90", Cloudy, 6,			{ 1.765625, 1, 1.332031, 0 } },
  { "Nikon", "D90", Shade, -6,			{ 2.453125, 1, 1.000000, 0 } },
  { "Nikon", "D90", Shade, 0,			{ 2.183594, 1, 1.062500, 0 } },
  { "Nikon", "D90", Shade, 6,			{ 1.984375, 1, 1.140625, 0 } },
  { "Nikon", "D90", "2700K", 0,			{ 1.093750, 1, 2.378906, 0 } },
  { "Nikon", "D90", "5000K", 0,			{ 1.761719, 1, 1.347656, 0 } },

  { "Nikon", "D3000", DirectSunlight, 0,	{ 1.851563, 1, 1.347656, 0 } },
  { "Nikon", "D3000", Flash, 0,			{ 2.113281, 1, 1.164062, 0 } },
  { "Nikon", "D3000", Cloudy, 0,		{ 2.019531, 1, 1.214844, 0 } },
  { "Nikon", "D3000", Shade, 0,			{ 2.355469, 1, 1.082031, 0 } },
  { "Nikon", "D3000", Incandescent, 0,		{ 1.171875, 1, 2.316406, 0 } },
  { "Nikon", "D3000", SodiumVaporFluorescent, 0, { 1.023438, 1, 2.371094, 0 } },
  { "Nikon", "D3000", WarmWhiteFluorescent, 0,	{ 1.179688, 1, 2.074219, 0 } },
  { "Nikon", "D3000", WhiteFluorescent, 0,	{ 1.355469, 1, 2.328125, 0 } },
  { "Nikon", "D3000", CoolWhiteFluorescent, 0,	{ 1.703125, 1, 2.019531, 0 } },
  { "Nikon", "D3000", DayWhiteFluorescent, 0,	{ 1.750000, 1, 1.386719, 0 } },
  { "Nikon", "D3000", DaylightFluorescent, 0,	{ 1.960937, 1, 1.105469, 0 } },
  { "Nikon", "D3000", HighTempMercuryVaporFluorescent, 0, { 2.351563, 1, 1.328125, 0 } },

  { "Nikon", "D3100", DirectSunlight, 0,	{ 2.109375, 1, 1.257813, 0 } },
  { "Nikon", "D3100", Flash, 0,			{ 2.386719, 1, 1.097656, 0 } },
  { "Nikon", "D3100", Cloudy, 0,		{ 2.257812, 1, 1.140625, 0 } },
  { "Nikon", "D3100", Shade, 0,			{ 2.609375, 1, 1.015625, 0 } },
  { "Nikon", "D3100", Incandescent, 0,		{ 1.320313, 1, 2.039063, 0 } },
  { "Nikon", "D3100", SodiumVaporFluorescent, 0, { 1.222656, 1, 2.238281, 0 } },
  { "Nikon", "D3100", WarmWhiteFluorescent, 0,	{ 1.312500, 1, 1.847656, 0 } },
  { "Nikon", "D3100", WhiteFluorescent, 0,	{ 1.531250, 1, 2.152344, 0 } },
  { "Nikon", "D3100", CoolWhiteFluorescent, 0,	{ 1.894531, 1, 1.875000, 0 } },
  { "Nikon", "D3100", DayWhiteFluorescent, 0,	{ 1.941406, 1, 1.281250, 0 } },
  { "Nikon", "D3100", DaylightFluorescent, 0,	{ 2.214844, 1, 1.015625, 0 } },
  { "Nikon", "D3100", HighTempMercuryVaporFluorescent, 0, { 2.589844, 1, 1.250000, 0 } },

  /* Firmware version 1.01 */
  /* -6/+6 fine tuning is A6/B6 in amber-blue and zero in green-magenta */
  { "Nikon", "D3200", DirectSunlight, -6,	{ 2.199219, 1, 1.140625, 0 } },
  { "Nikon", "D3200", DirectSunlight, 0,	{ 2.050781, 1, 1.273438, 0 } },
  { "Nikon", "D3200", DirectSunlight, 6,	{ 1.882813, 1, 1.406250, 0 } },
  { "Nikon", "D3200", Flash, -6,		{ 2.515625, 1, 1.062500, 0 } },
  { "Nikon", "D3200", Flash, 0,			{ 2.320313, 1, 1.148438, 0 } },
  { "Nikon", "D3200", Flash, 6,			{ 2.121094, 1, 1.218750, 0 } },
  { "Nikon", "D3200", Cloudy, -6,		{ 2.355469, 1, 1.078125, 0 } },
  { "Nikon", "D3200", Cloudy, 0,		{ 2.171875, 1, 1.164063, 0 } },
  { "Nikon", "D3200", Cloudy, 6,		{ 2.031250, 1, 1.300781, 0 } },
  { "Nikon", "D3200", Shade, -6,		{ 2.750000, 1, 1.000000, 0 } },
  { "Nikon", "D3200", Shade, 0,			{ 2.449219, 1, 1.050781, 0 } },
  { "Nikon", "D3200", Shade, 6,			{ 2.242188, 1, 1.121094, 0 } },
  { "Nikon", "D3200", Incandescent, -6,		{ 1.468750, 1, 1.796875, 0 } },
  { "Nikon", "D3200", Incandescent, 0,		{ 1.359375, 1, 1.964844, 0 } },
  { "Nikon", "D3200", Incandescent, 6,		{ 1.269531, 1, 2.144531, 0 } },
  { "Nikon", "D3200", SodiumVaporFluorescent, -6, { 1.390625, 1, 1.917969, 0 } },
  { "Nikon", "D3200", SodiumVaporFluorescent, 0, { 1.277344, 1, 2.093750, 0 } },
  { "Nikon", "D3200", SodiumVaporFluorescent, 6, { 1.203125, 1, 2.281250, 0 } },
  { "Nikon", "D3200", WarmWhiteFluorescent, -6,	{ 1.453125, 1, 1.679688, 0 } },
  { "Nikon", "D3200", WarmWhiteFluorescent, 0,	{ 1.351563, 1, 1.773438, 0 } },
  { "Nikon", "D3200", WarmWhiteFluorescent, 6,	{ 1.246094, 1, 1.878906, 0 } },
  { "Nikon", "D3200", WhiteFluorescent, -6,	{ 1.800781, 1, 1.812500, 0 } },
  { "Nikon", "D3200", WhiteFluorescent, 0,	{ 1.503906, 1, 1.972656, 0 } },
  { "Nikon", "D3200", WhiteFluorescent, 6,	{ 1.085938, 1, 2.253906, 0 } },
  { "Nikon", "D3200", CoolWhiteFluorescent, -6,	{ 2.085938, 1, 1.628906, 0 } },
  { "Nikon", "D3200", CoolWhiteFluorescent, 0,	{ 1.820313, 1, 1.796875, 0 } },
  { "Nikon", "D3200", CoolWhiteFluorescent, 6,	{ 1.531250, 1, 1.957031, 0 } },
  { "Nikon", "D3200", DayWhiteFluorescent, -6,	{ 2.027344, 1, 1.117188, 0 } },
  { "Nikon", "D3200", DayWhiteFluorescent, 0,	{ 1.878906, 1, 1.261719, 0 } },
  { "Nikon", "D3200", DayWhiteFluorescent, 6,	{ 1.742188, 1, 1.398438, 0 } },
  { "Nikon", "D3200", DaylightFluorescent, -6,	{ 2.296875, 1, 1.000000, 0 } },
  { "Nikon", "D3200", DaylightFluorescent, 0,	{ 2.109375, 1, 1.031250, 0 } },
  { "Nikon", "D3200", DaylightFluorescent, 6,	{ 1.957031, 1, 1.183594, 0 } },
  { "Nikon", "D3200", HighTempMercuryVaporFluorescent, -6, { 2.542969, 1, 1.117188, 0 } },
  { "Nikon", "D3200", HighTempMercuryVaporFluorescent, 0, { 2.445313, 1, 1.250000, 0 } },
  { "Nikon", "D3200", HighTempMercuryVaporFluorescent, 6, { 2.324219, 1, 1.394531, 0 } },

  { "Nikon", "D3300", DirectSunlight, 0,	{ 2.054688, 1, 1.394531, 0 } },
  { "Nikon", "D3300", Flash, 0,			{ 2.289063, 1, 1.230469, 0 } },
  { "Nikon", "D3300", Cloudy, 0,		{ 2.199219, 1, 1.277344, 0 } },
  { "Nikon", "D3300", Shade, 0,			{ 2.527344, 1, 1.152344, 0 } },
  { "Nikon", "D3300", Incandescent, 0,		{ 1.312500, 1, 2.214844, 0 } },
  { "Nikon", "D3300", CoolWhiteFluorescent, 0,	{ 1.855469, 1, 2.023438, 0 } },

  { "Nikon", "D3400", DirectSunlight                 , -12, { 2.13671875, 1, 1.7890625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -11, { 2.16015625, 1, 1.77734375, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -10, { 2.18359375, 1, 1.765625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -9, { 2.203125, 1, 1.75390625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -8, { 2.2265625, 1, 1.73828125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -7, { 2.24609375, 1, 1.72265625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -6, { 2.26953125, 1, 1.70703125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -5, { 2.2890625, 1, 1.69140625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -4, { 2.30859375, 1, 1.671875, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -3, { 2.33203125, 1, 1.65234375, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -2, { 2.3515625, 1, 1.6328125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , -1, { 2.375, 1, 1.6171875, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 0, { 2.39453125, 1, 1.59765625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 1, { 2.41796875, 1, 1.578125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 2, { 2.4375, 1, 1.5625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 3, { 2.45703125, 1, 1.55078125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 4, { 2.48046875, 1, 1.53515625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 5, { 2.50390625, 1, 1.51953125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 6, { 2.5234375, 1, 1.50390625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 7, { 2.546875, 1, 1.48828125, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 8, { 2.5703125, 1, 1.4765625, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 9, { 2.59375, 1, 1.4609375, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 10, { 2.6171875, 1, 1.44921875, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 11, { 2.64453125, 1, 1.4375, 0 } },
  { "Nikon", "D3400", DirectSunlight                 , 12, { 2.671875, 1, 1.42578125, 0 } },
  { "Nikon", "D3400", Shade                          , -12, { 2.75, 1, 1.39453125, 0 } },
  { "Nikon", "D3400", Shade                          , -11, { 2.77734375, 1, 1.38671875, 0 } },
  { "Nikon", "D3400", Shade                          , -10, { 2.80859375, 1, 1.375, 0 } },
  { "Nikon", "D3400", Shade                          , -9, { 2.83984375, 1, 1.3671875, 0 } },
  { "Nikon", "D3400", Shade                          , -8, { 2.87109375, 1, 1.359375, 0 } },
  { "Nikon", "D3400", Shade                          , -7, { 2.90234375, 1, 1.34765625, 0 } },
  { "Nikon", "D3400", Shade                          , -6, { 2.93359375, 1, 1.33984375, 0 } },
  { "Nikon", "D3400", Shade                          , -5, { 2.96875, 1, 1.33203125, 0 } },
  { "Nikon", "D3400", Shade                          , -4, { 3.00390625, 1, 1.3203125, 0 } },
  { "Nikon", "D3400", Shade                          , -3, { 3.0390625, 1, 1.3125, 0 } },
  { "Nikon", "D3400", Shade                          , -2, { 3.07421875, 1, 1.3046875, 0 } },
  { "Nikon", "D3400", Shade                          , -1, { 3.109375, 1, 1.296875, 0 } },
  { "Nikon", "D3400", Shade                          , 0, { 3.1484375, 1, 1.2890625, 0 } },
  { "Nikon", "D3400", Shade                          , 1, { 3.1875, 1, 1.28125, 0 } },
  { "Nikon", "D3400", Shade                          , 2, { 3.2265625, 1, 1.2734375, 0 } },
  { "Nikon", "D3400", Shade                          , 3, { 3.265625, 1, 1.26171875, 0 } },
  { "Nikon", "D3400", Shade                          , 4, { 3.3046875, 1, 1.25390625, 0 } },
  { "Nikon", "D3400", Shade                          , 5, { 3.34765625, 1, 1.24609375, 0 } },
  { "Nikon", "D3400", Shade                          , 6, { 3.390625, 1, 1.23828125, 0 } },
  { "Nikon", "D3400", Shade                          , 7, { 3.43359375, 1, 1.23046875, 0 } },
  { "Nikon", "D3400", Shade                          , 8, { 3.48046875, 1, 1.22265625, 0 } },
  { "Nikon", "D3400", Shade                          , 9, { 3.5234375, 1, 1.21484375, 0 } },
  { "Nikon", "D3400", Shade                          , 10, { 3.5703125, 1, 1.20703125, 0 } },
  { "Nikon", "D3400", Shade                          , 11, { 3.6171875, 1, 1.19921875, 0 } },
  { "Nikon", "D3400", Shade                          , 12, { 3.6640625, 1, 1.19140625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -12, { 2.3515625, 1, 1.6328125, 0 } },
  { "Nikon", "D3400", Cloudy                         , -11, { 2.375, 1, 1.6171875, 0 } },
  { "Nikon", "D3400", Cloudy                         , -10, { 2.39453125, 1, 1.59765625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -9, { 2.41796875, 1, 1.578125, 0 } },
  { "Nikon", "D3400", Cloudy                         , -8, { 2.4375, 1, 1.5625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -7, { 2.45703125, 1, 1.55078125, 0 } },
  { "Nikon", "D3400", Cloudy                         , -6, { 2.48046875, 1, 1.53515625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -5, { 2.50390625, 1, 1.51953125, 0 } },
  { "Nikon", "D3400", Cloudy                         , -4, { 2.5234375, 1, 1.50390625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -3, { 2.546875, 1, 1.48828125, 0 } },
  { "Nikon", "D3400", Cloudy                         , -2, { 2.5703125, 1, 1.4765625, 0 } },
  { "Nikon", "D3400", Cloudy                         , -1, { 2.59375, 1, 1.4609375, 0 } },
  { "Nikon", "D3400", Cloudy                         , 0, { 2.6171875, 1, 1.44921875, 0 } },
  { "Nikon", "D3400", Cloudy                         , 1, { 2.64453125, 1, 1.4375, 0 } },
  { "Nikon", "D3400", Cloudy                         , 2, { 2.671875, 1, 1.42578125, 0 } },
  { "Nikon", "D3400", Cloudy                         , 3, { 2.69921875, 1, 1.4140625, 0 } },
  { "Nikon", "D3400", Cloudy                         , 4, { 2.7265625, 1, 1.40625, 0 } },
  { "Nikon", "D3400", Cloudy                         , 5, { 2.75390625, 1, 1.39453125, 0 } },
  { "Nikon", "D3400", Cloudy                         , 6, { 2.78515625, 1, 1.3828125, 0 } },
  { "Nikon", "D3400", Cloudy                         , 7, { 2.8125, 1, 1.375, 0 } },
  { "Nikon", "D3400", Cloudy                         , 8, { 2.84375, 1, 1.36328125, 0 } },
  { "Nikon", "D3400", Cloudy                         , 9, { 2.875, 1, 1.35546875, 0 } },
  { "Nikon", "D3400", Cloudy                         , 10, { 2.90625, 1, 1.34765625, 0 } },
  { "Nikon", "D3400", Incandescent                   , -12, { 1.33203125, 1, 2.98046875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -11, { 1.34375, 1, 2.95703125, 0 } },
  { "Nikon", "D3400", Incandescent                   , -10, { 1.3515625, 1, 2.9296875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -9, { 1.359375, 1, 2.90234375, 0 } },
  { "Nikon", "D3400", Incandescent                   , -8, { 1.3671875, 1, 2.875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -7, { 1.37890625, 1, 2.8515625, 0 } },
  { "Nikon", "D3400", Incandescent                   , -6, { 1.38671875, 1, 2.82421875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -5, { 1.39453125, 1, 2.796875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -4, { 1.40625, 1, 2.7734375, 0 } },
  { "Nikon", "D3400", Incandescent                   , -3, { 1.4140625, 1, 2.74609375, 0 } },
  { "Nikon", "D3400", Incandescent                   , -2, { 1.42578125, 1, 2.71875, 0 } },
  { "Nikon", "D3400", Incandescent                   , -1, { 1.43359375, 1, 2.6953125, 0 } },
  { "Nikon", "D3400", Incandescent                   , 0, { 1.4453125, 1, 2.66796875, 0 } },
  { "Nikon", "D3400", Incandescent                   , 1, { 1.45703125, 1, 2.64453125, 0 } },
  { "Nikon", "D3400", Incandescent                   , 2, { 1.46484375, 1, 2.6171875, 0 } },
  { "Nikon", "D3400", Incandescent                   , 3, { 1.4765625, 1, 2.59375, 0 } },
  { "Nikon", "D3400", Incandescent                   , 4, { 1.48828125, 1, 2.5703125, 0 } },
  { "Nikon", "D3400", Incandescent                   , 5, { 1.5, 1, 2.54296875, 0 } },
  { "Nikon", "D3400", Incandescent                   , 6, { 1.51171875, 1, 2.51953125, 0 } },
  { "Nikon", "D3400", Incandescent                   , 7, { 1.5234375, 1, 2.49609375, 0 } },
  { "Nikon", "D3400", Incandescent                   , 8, { 1.53515625, 1, 2.46875, 0 } },
  { "Nikon", "D3400", Incandescent                   , 9, { 1.546875, 1, 2.4453125, 0 } },
  { "Nikon", "D3400", Incandescent                   , 10, { 1.55859375, 1, 2.421875, 0 } },
  { "Nikon", "D3400", Incandescent                   , 11, { 1.5703125, 1, 2.3984375, 0 } },
  { "Nikon", "D3400", Incandescent                   , 12, { 1.58203125, 1, 2.375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -12, { 1.36328125, 1, 2.57421875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -11, { 1.37109375, 1, 2.5625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -10, { 1.37890625, 1, 2.55078125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -9, { 1.390625, 1, 2.5390625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -8, { 1.3984375, 1, 2.52734375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -7, { 1.40625, 1, 2.515625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -6, { 1.41796875, 1, 2.50390625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -5, { 1.42578125, 1, 2.4921875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -4, { 1.4375, 1, 2.48046875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -3, { 1.4453125, 1, 2.46875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -2, { 1.45703125, 1, 2.45703125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , -1, { 1.46875, 1, 2.4453125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 0, { 1.4765625, 1, 2.43359375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 1, { 1.48828125, 1, 2.421875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 2, { 1.5, 1, 2.41015625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 3, { 1.51171875, 1, 2.39453125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 4, { 1.51953125, 1, 2.3828125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 5, { 1.53125, 1, 2.37109375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 6, { 1.54296875, 1, 2.359375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 7, { 1.5546875, 1, 2.34765625, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 8, { 1.56640625, 1, 2.33203125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 9, { 1.578125, 1, 2.3203125, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 10, { 1.58984375, 1, 2.30859375, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 11, { 1.6015625, 1, 2.29296875, 0 } },
  { "Nikon", "D3400", WarmWhiteFluorescent           , 12, { 1.61328125, 1, 2.28125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -12, { 1.77734375, 1, 2.7578125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -11, { 1.81640625, 1, 2.73046875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -10, { 1.85546875, 1, 2.69921875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -9, { 1.89453125, 1, 2.671875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -8, { 1.93359375, 1, 2.64453125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -7, { 1.96875, 1, 2.6171875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -6, { 2.0078125, 1, 2.5859375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -5, { 2.04296875, 1, 2.55859375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -4, { 2.078125, 1, 2.53125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -3, { 2.1171875, 1, 2.50390625, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -2, { 2.15234375, 1, 2.48046875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , -1, { 2.1875, 1, 2.453125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 0, { 2.22265625, 1, 2.42578125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 1, { 2.25390625, 1, 2.3984375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 2, { 2.2890625, 1, 2.375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 3, { 2.3203125, 1, 2.34765625, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 4, { 2.3515625, 1, 2.32421875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 5, { 2.3828125, 1, 2.296875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 6, { 2.4140625, 1, 2.2734375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 7, { 2.4453125, 1, 2.25, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 8, { 2.4765625, 1, 2.22265625, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 9, { 2.5078125, 1, 2.19921875, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 10, { 2.53515625, 1, 2.17578125, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 11, { 2.5625, 1, 2.15234375, 0 } },
  { "Nikon", "D3400", CoolWhiteFluorescent           , 12, { 2.59375, 1, 2.12890625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -12, { 2.03515625, 1, 1.85546875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -11, { 2.05078125, 1, 1.83984375, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -10, { 2.0703125, 1, 1.82421875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -9, { 2.0859375, 1, 1.80859375, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -8, { 2.10546875, 1, 1.7890625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -7, { 2.12109375, 1, 1.7734375, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -6, { 2.13671875, 1, 1.7578125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -5, { 2.15625, 1, 1.73828125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -4, { 2.17578125, 1, 1.72265625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -3, { 2.19140625, 1, 1.70703125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -2, { 2.2109375, 1, 1.6875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , -1, { 2.2265625, 1, 1.671875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 0, { 2.24609375, 1, 1.65234375, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 1, { 2.265625, 1, 1.63671875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 2, { 2.28125, 1, 1.6171875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 3, { 2.30078125, 1, 1.6015625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 4, { 2.3203125, 1, 1.58203125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 5, { 2.3359375, 1, 1.56640625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 6, { 2.35546875, 1, 1.546875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 7, { 2.375, 1, 1.52734375, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 8, { 2.39453125, 1, 1.5078125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 9, { 2.4140625, 1, 1.4921875, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 10, { 2.4296875, 1, 1.47265625, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 11, { 2.44921875, 1, 1.453125, 0 } },
  { "Nikon", "D3400", DayWhiteFluorescent            , 12, { 2.46875, 1, 1.43359375, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -12, { 2.36328125, 1, 1.5390625, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -11, { 2.3828125, 1, 1.51953125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -10, { 2.40234375, 1, 1.50390625, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -9, { 2.421875, 1, 1.484375, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -8, { 2.4375, 1, 1.46484375, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -7, { 2.45703125, 1, 1.4453125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -6, { 2.4765625, 1, 1.42578125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -5, { 2.49609375, 1, 1.40625, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -4, { 2.515625, 1, 1.38671875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -3, { 2.53515625, 1, 1.3671875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -2, { 2.5546875, 1, 1.34765625, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , -1, { 2.57421875, 1, 1.328125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 0, { 2.59375, 1, 1.30859375, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 1, { 2.61328125, 1, 1.2890625, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 2, { 2.63671875, 1, 1.26953125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 3, { 2.65625, 1, 1.25, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 4, { 2.67578125, 1, 1.23046875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 5, { 2.6953125, 1, 1.20703125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 6, { 2.71484375, 1, 1.1875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 7, { 2.734375, 1, 1.16796875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 8, { 2.75390625, 1, 1.14453125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 9, { 2.77734375, 1, 1.125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 10, { 2.796875, 1, 1.10546875, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 11, { 2.81640625, 1, 1.08203125, 0 } },
  { "Nikon", "D3400", DaylightFluorescent            , 12, { 2.8359375, 1, 1.0625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -12, { 1.25390625, 1, 3.16015625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -11, { 1.296875, 1, 3.12890625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -10, { 1.33984375, 1, 3.09375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -9, { 1.37890625, 1, 3.0625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -8, { 1.421875, 1, 3.03125, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -7, { 1.46484375, 1, 2.99609375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -6, { 1.50390625, 1, 2.96484375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -5, { 1.546875, 1, 2.93359375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -4, { 1.5859375, 1, 2.90234375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -3, { 1.62890625, 1, 2.875, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -2, { 1.66796875, 1, 2.84375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , -1, { 1.70703125, 1, 2.8125, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 0, { 1.74609375, 1, 2.78125, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 1, { 1.78515625, 1, 2.75390625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 2, { 1.82421875, 1, 2.72265625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 3, { 1.86328125, 1, 2.6953125, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 4, { 1.90234375, 1, 2.66796875, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 5, { 1.94140625, 1, 2.63671875, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 6, { 1.9765625, 1, 2.609375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 7, { 2.015625, 1, 2.58203125, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 8, { 2.05078125, 1, 2.5546875, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 9, { 2.0859375, 1, 2.52734375, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 10, { 2.12109375, 1, 2.5, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 11, { 2.16015625, 1, 2.47265625, 0 } },
  { "Nikon", "D3400", WhiteFluorescent               , 12, { 2.19140625, 1, 2.4453125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -12, { 2.96875, 1, 1.80078125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -11, { 2.98828125, 1, 1.78125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -10, { 3.0078125, 1, 1.76171875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -9, { 3.02734375, 1, 1.7421875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -8, { 3.04296875, 1, 1.7265625, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -7, { 3.0625, 1, 1.70703125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -6, { 3.078125, 1, 1.69140625, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -5, { 3.09765625, 1, 1.671875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -4, { 3.11328125, 1, 1.65625, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -3, { 3.12890625, 1, 1.63671875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -2, { 3.14453125, 1, 1.62109375, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, -1, { 3.15625, 1, 1.60546875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 0, { 3.171875, 1, 1.58984375, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 1, { 3.1875, 1, 1.57421875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 2, { 3.19921875, 1, 1.55859375, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 3, { 3.2109375, 1, 1.54296875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 4, { 3.2265625, 1, 1.52734375, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 5, { 3.23828125, 1, 1.51171875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 6, { 3.25, 1, 1.5, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 7, { 3.26171875, 1, 1.484375, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 8, { 3.26953125, 1, 1.46875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 9, { 3.28125, 1, 1.45703125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 10, { 3.29296875, 1, 1.4453125, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 11, { 3.30078125, 1, 1.4296875, 0 } },
  { "Nikon", "D3400", HighTempMercuryVaporFluorescent, 12, { 3.30859375, 1, 1.41796875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -12, { 1.2265625, 1, 3.25390625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -11, { 1.234375, 1, 3.21875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -10, { 1.2421875, 1, 3.1875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -9, { 1.25, 1, 3.15625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -8, { 1.2578125, 1, 3.125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -7, { 1.265625, 1, 3.09375, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -6, { 1.2734375, 1, 3.06640625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -5, { 1.28125, 1, 3.03515625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -4, { 1.2890625, 1, 3.0078125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -3, { 1.296875, 1, 2.984375, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -2, { 1.3046875, 1, 2.95703125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , -1, { 1.31640625, 1, 2.9296875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 0, { 1.32421875, 1, 2.90625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 1, { 1.3359375, 1, 2.87890625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 2, { 1.34765625, 1, 2.85546875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 3, { 1.35546875, 1, 2.83203125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 4, { 1.3671875, 1, 2.8046875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 5, { 1.3828125, 1, 2.78125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 6, { 1.39453125, 1, 2.75390625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 7, { 1.40625, 1, 2.73046875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 8, { 1.421875, 1, 2.70703125, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 9, { 1.43359375, 1, 2.6796875, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 10, { 1.44921875, 1, 2.65234375, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 11, { 1.46484375, 1, 2.625, 0 } },
  { "Nikon", "D3400", SodiumVaporFluorescent         , 12, { 1.48046875, 1, 2.59765625, 0 } },
  { "Nikon", "D3400", Flash                          , -12, { 2.47265625, 1, 1.5, 0 } },
  { "Nikon", "D3400", Flash                          , -11, { 2.5, 1, 1.48828125, 0 } },
  { "Nikon", "D3400", Flash                          , -10, { 2.5234375, 1, 1.47265625, 0 } },
  { "Nikon", "D3400", Flash                          , -9, { 2.546875, 1, 1.46484375, 0 } },
  { "Nikon", "D3400", Flash                          , -8, { 2.5703125, 1, 1.453125, 0 } },
  { "Nikon", "D3400", Flash                          , -7, { 2.58984375, 1, 1.44140625, 0 } },
  { "Nikon", "D3400", Flash                          , -6, { 2.61328125, 1, 1.4296875, 0 } },
  { "Nikon", "D3400", Flash                          , -5, { 2.6328125, 1, 1.421875, 0 } },
  { "Nikon", "D3400", Flash                          , -4, { 2.65625, 1, 1.41015625, 0 } },
  { "Nikon", "D3400", Flash                          , -3, { 2.67578125, 1, 1.40234375, 0 } },
  { "Nikon", "D3400", Flash                          , -2, { 2.6953125, 1, 1.39453125, 0 } },
  { "Nikon", "D3400", Flash                          , -1, { 2.71875, 1, 1.3828125, 0 } },
  { "Nikon", "D3400", Flash                          , 0, { 2.73828125, 1, 1.375, 0 } },
  { "Nikon", "D3400", Flash                          , 1, { 2.76171875, 1, 1.3671875, 0 } },
  { "Nikon", "D3400", Flash                          , 2, { 2.78515625, 1, 1.359375, 0 } },
  { "Nikon", "D3400", Flash                          , 3, { 2.80859375, 1, 1.3515625, 0 } },
  { "Nikon", "D3400", Flash                          , 4, { 2.8359375, 1, 1.33984375, 0 } },
  { "Nikon", "D3400", Flash                          , 5, { 2.86328125, 1, 1.33203125, 0 } },
  { "Nikon", "D3400", Flash                          , 6, { 2.890625, 1, 1.32421875, 0 } },
  { "Nikon", "D3400", Flash                          , 7, { 2.921875, 1, 1.31640625, 0 } },
  { "Nikon", "D3400", Flash                          , 8, { 2.95703125, 1, 1.30859375, 0 } },
  { "Nikon", "D3400", Flash                          , 9, { 2.9921875, 1, 1.30078125, 0 } },
  { "Nikon", "D3400", Flash                          , 10, { 3.02734375, 1, 1.29296875, 0 } },
  { "Nikon", "D3400", Flash                          , 11, { 3.06640625, 1, 1.28515625, 0 } },
  { "Nikon", "D3400", Flash                          , 12, { 3.11328125, 1, 1.27734375, 0 } },

  { "Nikon", "D5000", DirectSunlight, 0,	{ 1.800781, 1, 1.308594, 0 } },
  { "Nikon", "D5000", Flash, 0,			{ 1.976562, 1, 1.152344, 0 } },
  { "Nikon", "D5000", Cloudy, 0,		{ 1.917969, 1, 1.187500, 0 } },
  { "Nikon", "D5000", Shade, 0,			{ 2.183594, 1, 1.062500, 0 } },
  { "Nikon", "D5000", Incandescent, 0,		{ 1.179687, 1, 2.097656, 0 } },
  { "Nikon", "D5000", SodiumVaporFluorescent, 0, { 1.062500, 1, 2.289063, 0 } },
  { "Nikon", "D5000", WarmWhiteFluorescent, 0,	{ 1.191406, 1, 1.871094, 0 } },
  { "Nikon", "D5000", WhiteFluorescent, 0,	{ 1.343750, 1, 2.183594, 0 } },
  { "Nikon", "D5000", CoolWhiteFluorescent, 0,	{ 1.644531, 1, 1.937500, 0 } },
  { "Nikon", "D5000", DayWhiteFluorescent, 0,	{ 1.695313, 1, 1.312500, 0 } },
  { "Nikon", "D5000", DaylightFluorescent, 0,	{ 1.925781, 1, 1.074219, 0 } },
  { "Nikon", "D5000", HighTempMercuryVaporFluorescent, 0, { 2.207031, 1, 1.292969, 0 } },

  { "Nikon", "D5100", DirectSunlight, -6,     { 2.199219, 1, 1.242188, 0 } },
  { "Nikon", "D5100", DirectSunlight, -5,     { 2.167969, 1, 1.265625, 0 } },
  { "Nikon", "D5100", DirectSunlight, -4,     { 2.140625, 1, 1.285156, 0 } },
  { "Nikon", "D5100", DirectSunlight, -3,     { 2.117188, 1, 1.308594, 0 } },
  { "Nikon", "D5100", DirectSunlight, -2,     { 2.085938, 1, 1.335938, 0 } },
  { "Nikon", "D5100", DirectSunlight, -1,     { 2.058594, 1, 1.363281, 0 } },
  { "Nikon", "D5100", DirectSunlight, 0,      { 2.027344, 1, 1.390625, 0 } },
  { "Nikon", "D5100", DirectSunlight, 1,      { 2.000000, 1, 1.421875, 0 } },
  { "Nikon", "D5100", DirectSunlight, 2,      { 1.972656, 1, 1.449219, 0 } },
  { "Nikon", "D5100", DirectSunlight, 3,      { 1.941406, 1, 1.472656, 0 } },
  { "Nikon", "D5100", DirectSunlight, 4,      { 1.910156, 1, 1.500000, 0 } },
  { "Nikon", "D5100", DirectSunlight, 5,      { 1.882813, 1, 1.523438, 0 } },
  { "Nikon", "D5100", DirectSunlight, 6,      { 1.847656, 1, 1.542969, 0 } },
  { "Nikon", "D5100", Flash, -6,              { 2.511719, 1, 1.132813, 0 } },
  { "Nikon", "D5100", Flash, -5,              { 2.460938, 1, 1.144531, 0 } },
  { "Nikon", "D5100", Flash, -4,              { 2.414063, 1, 1.160156, 0 } },
  { "Nikon", "D5100", Flash, -3,              { 2.375000, 1, 1.171875, 0 } },
  { "Nikon", "D5100", Flash, -2,              { 2.339844, 1, 1.183594, 0 } },
  { "Nikon", "D5100", Flash, -1,              { 2.308594, 1, 1.199219, 0 } },
  { "Nikon", "D5100", Flash, 0,               { 2.281250, 1, 1.210938, 0 } },
  { "Nikon", "D5100", Flash, 1,               { 2.250000, 1, 1.226563, 0 } },
  { "Nikon", "D5100", Flash, 2,               { 2.222656, 1, 1.238281, 0 } },
  { "Nikon", "D5100", Flash, 3,               { 2.191406, 1, 1.253906, 0 } },
  { "Nikon", "D5100", Flash, 4,               { 2.160156, 1, 1.273438, 0 } },
  { "Nikon", "D5100", Flash, 5,               { 2.125000, 1, 1.289063, 0 } },
  { "Nikon", "D5100", Flash, 6,               { 2.089844, 1, 1.308594, 0 } },
  { "Nikon", "D5100", Cloudy, -6,             { 2.386719, 1, 1.164063, 0 } },
  { "Nikon", "D5100", Cloudy, -5,             { 2.343750, 1, 1.175781, 0 } },
  { "Nikon", "D5100", Cloudy, -4,             { 2.300781, 1, 1.195313, 0 } },
  { "Nikon", "D5100", Cloudy, -3,             { 2.261719, 1, 1.210938, 0 } },
  { "Nikon", "D5100", Cloudy, -2,             { 2.230469, 1, 1.226563, 0 } },
  { "Nikon", "D5100", Cloudy, -1,             { 2.199219, 1, 1.242188, 0 } },
  { "Nikon", "D5100", Cloudy, 0,              { 2.167969, 1, 1.265625, 0 } },
  { "Nikon", "D5100", Cloudy, 1,              { 2.140625, 1, 1.285156, 0 } },
  { "Nikon", "D5100", Cloudy, 2,              { 2.117188, 1, 1.308594, 0 } },
  { "Nikon", "D5100", Cloudy, 3,              { 2.085938, 1, 1.335938, 0 } },
  { "Nikon", "D5100", Cloudy, 4,              { 2.058594, 1, 1.363281, 0 } },
  { "Nikon", "D5100", Cloudy, 5,              { 2.027344, 1, 1.390625, 0 } },
  { "Nikon", "D5100", Cloudy, 6,              { 2.000000, 1, 1.421875, 0 } },
  { "Nikon", "D5100", Shade, -6,              { 2.828125, 1, 1.023438, 0 } },
  { "Nikon", "D5100", Shade, -5,              { 2.769531, 1, 1.050781, 0 } },
  { "Nikon", "D5100", Shade, -4,              { 2.703125, 1, 1.074219, 0 } },
  { "Nikon", "D5100", Shade, -3,              { 2.644531, 1, 1.093750, 0 } },
  { "Nikon", "D5100", Shade, -2,              { 2.585938, 1, 1.109375, 0 } },
  { "Nikon", "D5100", Shade, -1,              { 2.539063, 1, 1.121094, 0 } },
  { "Nikon", "D5100", Shade, 0,               { 2.492188, 1, 1.136719, 0 } },
  { "Nikon", "D5100", Shade, 1,               { 2.445313, 1, 1.148438, 0 } },
  { "Nikon", "D5100", Shade, 2,               { 2.406250, 1, 1.156250, 0 } },
  { "Nikon", "D5100", Shade, 3,               { 2.359375, 1, 1.171875, 0 } },
  { "Nikon", "D5100", Shade, 4,               { 2.316406, 1, 1.187500, 0 } },
  { "Nikon", "D5100", Shade, 5,               { 2.277344, 1, 1.203125, 0 } },
  { "Nikon", "D5100", Shade, 6,               { 2.246094, 1, 1.218750, 0 } },
  { "Nikon", "D5100", Incandescent, -6,       { 1.417969, 1, 2.003906, 0 } },
  { "Nikon", "D5100", Incandescent, -5,       { 1.394531, 1, 2.035156, 0 } },
  { "Nikon", "D5100", Incandescent, -4,       { 1.375000, 1, 2.066406, 0 } },
  { "Nikon", "D5100", Incandescent, -3,       { 1.355469, 1, 2.101563, 0 } },
  { "Nikon", "D5100", Incandescent, -2,       { 1.339844, 1, 2.140625, 0 } },
  { "Nikon", "D5100", Incandescent, -1,       { 1.324219, 1, 2.179688, 0 } },
  { "Nikon", "D5100", Incandescent, 0,        { 1.304688, 1, 2.214844, 0 } },
  { "Nikon", "D5100", Incandescent, 1,        { 1.289063, 1, 2.257813, 0 } },
  { "Nikon", "D5100", Incandescent, 2,        { 1.273438, 1, 2.296875, 0 } },
  { "Nikon", "D5100", Incandescent, 3,        { 1.261719, 1, 2.335938, 0 } },
  { "Nikon", "D5100", Incandescent, 4,        { 1.246094, 1, 2.378906, 0 } },
  { "Nikon", "D5100", Incandescent, 5,        { 1.230469, 1, 2.421875, 0 } },
  { "Nikon", "D5100", Incandescent, 6,        { 1.218750, 1, 2.464844, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -6,       { 1.324219, 1, 2.187500, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -5,       { 1.296875, 1, 2.230469, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -4,       { 1.273438, 1, 2.277344, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -3,       { 1.253906, 1, 2.316406, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -2,       { 1.238281, 1, 2.355469, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, -1,       { 1.218750, 1, 2.398438, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 0,        { 1.203125, 1, 2.437500, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 1,        { 1.187500, 1, 2.480469, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 2,        { 1.175781, 1, 2.523438, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 3,        { 1.164063, 1, 2.566406, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 4,        { 1.152344, 1, 2.609375, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 5,        { 1.140625, 1, 2.664063, 0 } },
  { "Nikon", "D5100", SodiumVaporFluorescent, 6,        { 1.128906, 1, 2.718750, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -6,         { 1.414063, 1, 1.859375, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -5,         { 1.394531, 1, 1.878906, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -4,         { 1.375000, 1, 1.894531, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -3,         { 1.355469, 1, 1.914063, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -2,         { 1.335938, 1, 1.929688, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, -1,         { 1.316406, 1, 1.949219, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 0,          { 1.300781, 1, 1.964844, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 1,          { 1.281250, 1, 1.984375, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 2,          { 1.261719, 1, 2.003906, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 3,          { 1.246094, 1, 2.023438, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 4,          { 1.230469, 1, 2.042969, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 5,          { 1.214844, 1, 2.062500, 0 } },
  { "Nikon", "D5100", WarmWhiteFluorescent, 6,          { 1.199219, 1, 2.082031, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -6,             { 1.812500, 1, 2.042969, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -5,             { 1.765625, 1, 2.085938, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -4,             { 1.714844, 1, 2.128906, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -3,             { 1.664063, 1, 2.171875, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -2,             { 1.609375, 1, 2.214844, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, -1,             { 1.550781, 1, 2.261719, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 0,              { 1.496094, 1, 2.304688, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 1,              { 1.429688, 1, 2.355469, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 2,              { 1.363281, 1, 2.406250, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 3,              { 1.296875, 1, 2.460938, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 4,              { 1.226563, 1, 2.515625, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 5,              { 1.156250, 1, 2.570313, 0 } },
  { "Nikon", "D5100", WhiteFluorescent, 6,              { 1.078125, 1, 2.628906, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -6,         { 2.105469, 1, 1.808594, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -5,         { 2.062500, 1, 1.843750, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -4,         { 2.019531, 1, 1.878906, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -3,         { 1.976563, 1, 1.917969, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -2,         { 1.933594, 1, 1.953125, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, -1,         { 1.882813, 1, 1.988281, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 0,          { 1.832031, 1, 2.027344, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 1,          { 1.789063, 1, 2.070313, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 2,          { 1.738281, 1, 2.109375, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 3,          { 1.687500, 1, 2.152344, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 4,          { 1.632813, 1, 2.195313, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 5,          { 1.574219, 1, 2.238281, 0 } },
  { "Nikon", "D5100", CoolWhiteFluorescent, 6,          { 1.519531, 1, 2.281250, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -6,          { 2.070313, 1, 1.214844, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -5,          { 2.042969, 1, 1.246094, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -4,          { 2.011719, 1, 1.277344, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -3,          { 1.984375, 1, 1.304688, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -2,          { 1.953125, 1, 1.332031, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, -1,          { 1.925781, 1, 1.363281, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 0,           { 1.898438, 1, 1.390625, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 1,           { 1.871094, 1, 1.417969, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 2,           { 1.843750, 1, 1.441406, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 3,           { 1.816406, 1, 1.472656, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 4,           { 1.792969, 1, 1.496094, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 5,           { 1.765625, 1, 1.523438, 0 } },
  { "Nikon", "D5100", DayWhiteFluorescent, 6,           { 1.742188, 1, 1.546875, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -6,          { 2.359375, 1, 1.000000, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -5,          { 2.328125, 1, 1.000000, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -4,          { 2.296875, 1, 1.000000, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -3,          { 2.261719, 1, 1.015625, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -2,          { 2.230469, 1, 1.050781, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, -1,          { 2.199219, 1, 1.082031, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 0,           { 2.167969, 1, 1.117188, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 1,           { 2.136719, 1, 1.148438, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 2,           { 2.109375, 1, 1.179688, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 3,           { 2.078125, 1, 1.210938, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 4,           { 2.046875, 1, 1.242188, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 5,           { 2.019531, 1, 1.269531, 0 } },
  { "Nikon", "D5100", DaylightFluorescent, 6,           { 1.988281, 1, 1.300781, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -6, { 2.578125, 1, 1.222656, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -5, { 2.566406, 1, 1.246094, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -4, { 2.550781, 1, 1.265625, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -3, { 2.535156, 1, 1.292969, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -2, { 2.519531, 1, 1.316406, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, -1, { 2.503906, 1, 1.343750, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 0,  { 2.484375, 1, 1.367188, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 1,  { 2.464844, 1, 1.394531, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 2,  { 2.441406, 1, 1.421875, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 3,  { 2.425781, 1, 1.453125, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 4,  { 2.402344, 1, 1.476563, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 5,  { 2.375000, 1, 1.511719, 0 } },
  { "Nikon", "D5100", HighTempMercuryVaporFluorescent, 6,  { 2.375000, 1, 1.511719, 0 } },

  /* Nikon D5200 firmware version 1.01 */
  { "Nikon", "D5200", Incandescent, 0,		{ 1.445313, 1, 2.667969, 0 } },
  { "Nikon", "D5200", CoolWhiteFluorescent, 0,	{ 2.222656, 1, 2.417969, 0 } },
  { "Nikon", "D5200", DirectSunlight, 0,	{ 2.386719, 1, 1.597656, 0 } },
  { "Nikon", "D5200", Flash, 0,			{ 2.753906, 1, 1.371094, 0 } },
  { "Nikon", "D5200", Cloudy, 0,		{ 2.656250, 1, 1.453125, 0 } },
  { "Nikon", "D5200", Shade, 0,			{ 3.183594, 1, 1.292969, 0 } },

  { "Nikon", "D5300", Incandescent, 0,		{ 1.312500, 1, 2.214844, 0 } },
  { "Nikon", "D5300", CoolWhiteFluorescent, 0,	{ 1.855469, 1, 2.023438, 0 } },
  { "Nikon", "D5300", DirectSunlight, 0,	{ 2.054688, 1, 1.394531, 0 } },
  { "Nikon", "D5300", Flash, 0,			{ 2.289063, 1, 1.230469, 0 } },
  { "Nikon", "D5300", Cloudy, 0,		{ 2.199219, 1, 1.277344, 0 } },
  { "Nikon", "D5300", Shade, 0,			{ 2.527344, 1, 1.152344, 0 } },

  { "Nikon", "D5500", DirectSunlight                 , 0, { 2.050781, 1, 1.402344, 0 } },
  { "Nikon", "D5500", Shade                          , 0, { 2.527344, 1, 1.144531, 0 } },
  { "Nikon", "D5500", Cloudy                         , 0, { 2.191406, 1, 1.281250, 0 } },
  { "Nikon", "D5500", Incandescent                   , 0, { 1.308594, 1, 2.214844, 0 } },
  { "Nikon", "D5500", WarmWhiteFluorescent           , 0, { 1.289062, 1, 1.984375, 0 } },
  { "Nikon", "D5500", CoolWhiteFluorescent           , 0, { 1.863281, 1, 2.046875, 0 } },
  { "Nikon", "D5500", DayWhiteFluorescent            , 0, { 1.890625, 1, 1.406250, 0 } },
  { "Nikon", "D5500", DaylightFluorescent            , 0, { 2.148438, 1, 1.140625, 0 } },
  { "Nikon", "D5500", WhiteFluorescent               , 0, { 1.511719, 1, 2.332031, 0 } },
  { "Nikon", "D5500", HighTempMercuryVaporFluorescent, 0, { 2.519531, 1, 1.378906, 0 } },
  { "Nikon", "D5500", SodiumVaporFluorescent         , 0, { 1.210938, 1, 2.437500, 0 } },
  { "Nikon", "D5500", Flash                          , 0, { 2.285156, 1, 1.226562, 0 } },

  /* Nikon D5600 Firmware Version 1.10 */
  { "Nikon", "D5600", DirectSunlight                 , -12, { 1.8671875, 1, 1.5546875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -11, { 1.8828125, 1, 1.54296875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -10, { 1.90234375, 1, 1.53125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -9, { 1.91796875, 1, 1.51953125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -8, { 1.93359375, 1, 1.5078125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -7, { 1.94921875, 1, 1.4921875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -6, { 1.96484375, 1, 1.48046875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -5, { 1.98046875, 1, 1.46875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -4, { 1.9921875, 1, 1.45703125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -3, { 2.0078125, 1, 1.44140625, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -2, { 2.0234375, 1, 1.4296875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , -1, { 2.03515625, 1, 1.41796875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 0, { 2.05078125, 1, 1.40234375, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 1, { 2.0625, 1, 1.390625, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 2, { 2.078125, 1, 1.37890625, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 3, { 2.08984375, 1, 1.36328125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 4, { 2.10546875, 1, 1.3515625, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 5, { 2.1171875, 1, 1.33984375, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 6, { 2.1328125, 1, 1.328125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 7, { 2.1484375, 1, 1.31640625, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 8, { 2.16015625, 1, 1.3046875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 9, { 2.17578125, 1, 1.29296875, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 10, { 2.19140625, 1, 1.28125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 11, { 2.20703125, 1, 1.26953125, 0 } },
  { "Nikon", "D5600", DirectSunlight                 , 12, { 2.22265625, 1, 1.2578125, 0 } },
  { "Nikon", "D5600", Shade                          , -12, { 2.2734375, 1, 1.23046875, 0 } },
  { "Nikon", "D5600", Shade                          , -11, { 2.2890625, 1, 1.22265625, 0 } },
  { "Nikon", "D5600", Shade                          , -10, { 2.30859375, 1, 1.2109375, 0 } },
  { "Nikon", "D5600", Shade                          , -9, { 2.328125, 1, 1.203125, 0 } },
  { "Nikon", "D5600", Shade                          , -8, { 2.34765625, 1, 1.1953125, 0 } },
  { "Nikon", "D5600", Shade                          , -7, { 2.3671875, 1, 1.1875, 0 } },
  { "Nikon", "D5600", Shade                          , -6, { 2.390625, 1, 1.1796875, 0 } },
  { "Nikon", "D5600", Shade                          , -5, { 2.41015625, 1, 1.171875, 0 } },
  { "Nikon", "D5600", Shade                          , -4, { 2.43359375, 1, 1.16796875, 0 } },
  { "Nikon", "D5600", Shade                          , -3, { 2.45703125, 1, 1.16015625, 0 } },
  { "Nikon", "D5600", Shade                          , -2, { 2.48046875, 1, 1.15234375, 0 } },
  { "Nikon", "D5600", Shade                          , -1, { 2.50390625, 1, 1.1484375, 0 } },
  { "Nikon", "D5600", Shade                          , 0, { 2.52734375, 1, 1.14453125, 0 } },
  { "Nikon", "D5600", Shade                          , 1, { 2.55078125, 1, 1.140625, 0 } },
  { "Nikon", "D5600", Shade                          , 2, { 2.578125, 1, 1.1328125, 0 } },
  { "Nikon", "D5600", Shade                          , 3, { 2.60546875, 1, 1.12890625, 0 } },
  { "Nikon", "D5600", Shade                          , 4, { 2.62890625, 1, 1.125, 0 } },
  { "Nikon", "D5600", Shade                          , 5, { 2.65625, 1, 1.125, 0 } },
  { "Nikon", "D5600", Shade                          , 6, { 2.6875, 1, 1.12109375, 0 } },
  { "Nikon", "D5600", Shade                          , 7, { 2.71484375, 1, 1.1171875, 0 } },
  { "Nikon", "D5600", Shade                          , 8, { 2.7421875, 1, 1.1171875, 0 } },
  { "Nikon", "D5600", Shade                          , 9, { 2.7734375, 1, 1.11328125, 0 } },
  { "Nikon", "D5600", Shade                          , 10, { 2.8046875, 1, 1.11328125, 0 } },
  { "Nikon", "D5600", Shade                          , 11, { 2.83203125, 1, 1.109375, 0 } },
  { "Nikon", "D5600", Shade                          , 12, { 2.86328125, 1, 1.109375, 0 } },
  { "Nikon", "D5600", Cloudy                         , -12, { 2.0234375, 1, 1.4296875, 0 } },
  { "Nikon", "D5600", Cloudy                         , -11, { 2.03515625, 1, 1.41796875, 0 } },
  { "Nikon", "D5600", Cloudy                         , -10, { 2.05078125, 1, 1.40234375, 0 } },
  { "Nikon", "D5600", Cloudy                         , -9, { 2.0625, 1, 1.390625, 0 } },
  { "Nikon", "D5600", Cloudy                         , -8, { 2.078125, 1, 1.37890625, 0 } },
  { "Nikon", "D5600", Cloudy                         , -7, { 2.08984375, 1, 1.36328125, 0 } },
  { "Nikon", "D5600", Cloudy                         , -6, { 2.10546875, 1, 1.3515625, 0 } },
  { "Nikon", "D5600", Cloudy                         , -5, { 2.1171875, 1, 1.33984375, 0 } },
  { "Nikon", "D5600", Cloudy                         , -4, { 2.1328125, 1, 1.328125, 0 } },
  { "Nikon", "D5600", Cloudy                         , -3, { 2.1484375, 1, 1.31640625, 0 } },
  { "Nikon", "D5600", Cloudy                         , -2, { 2.16015625, 1, 1.3046875, 0 } },
  { "Nikon", "D5600", Cloudy                         , -1, { 2.17578125, 1, 1.29296875, 0 } },
  { "Nikon", "D5600", Cloudy                         , 0, { 2.19140625, 1, 1.28125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 1, { 2.20703125, 1, 1.26953125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 2, { 2.22265625, 1, 1.2578125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 3, { 2.2421875, 1, 1.25, 0 } },
  { "Nikon", "D5600", Cloudy                         , 4, { 2.2578125, 1, 1.23828125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 5, { 2.27734375, 1, 1.23046875, 0 } },
  { "Nikon", "D5600", Cloudy                         , 6, { 2.29296875, 1, 1.21875, 0 } },
  { "Nikon", "D5600", Cloudy                         , 7, { 2.3125, 1, 1.2109375, 0 } },
  { "Nikon", "D5600", Cloudy                         , 8, { 2.33203125, 1, 1.203125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 9, { 2.3515625, 1, 1.1953125, 0 } },
  { "Nikon", "D5600", Cloudy                         , 10, { 2.375, 1, 1.1875, 0 } },
  { "Nikon", "D5600", Cloudy                         , 11, { 2.39453125, 1, 1.1796875, 0 } },
  { "Nikon", "D5600", Cloudy                         , 12, { 2.4140625, 1, 1.171875, 0 } },
  { "Nikon", "D5600", Incandescent                   , -12, { 1.21875, 1, 2.453125, 0 } },
  { "Nikon", "D5600", Incandescent                   , -11, { 1.2265625, 1, 2.43359375, 0 } },
  { "Nikon", "D5600", Incandescent                   , -10, { 1.234375, 1, 2.4140625, 0 } },
  { "Nikon", "D5600", Incandescent                   , -9, { 1.23828125, 1, 2.39453125, 0 } },
  { "Nikon", "D5600", Incandescent                   , -8, { 1.24609375, 1, 2.375, 0 } },
  { "Nikon", "D5600", Incandescent                   , -7, { 1.25390625, 1, 2.3515625, 0 } },
  { "Nikon", "D5600", Incandescent                   , -6, { 1.26171875, 1, 2.33203125, 0 } },
  { "Nikon", "D5600", Incandescent                   , -5, { 1.26953125, 1, 2.3125, 0 } },
  { "Nikon", "D5600", Incandescent                   , -4, { 1.27734375, 1, 2.29296875, 0 } },
  { "Nikon", "D5600", Incandescent                   , -3, { 1.28515625, 1, 2.2734375, 0 } },
  { "Nikon", "D5600", Incandescent                   , -2, { 1.29296875, 1, 2.25390625, 0 } },
  { "Nikon", "D5600", Incandescent                   , -1, { 1.30078125, 1, 2.234375, 0 } },
  { "Nikon", "D5600", Incandescent                   , 0, { 1.30859375, 1, 2.21484375, 0 } },
  { "Nikon", "D5600", Incandescent                   , 1, { 1.31640625, 1, 2.19921875, 0 } },
  { "Nikon", "D5600", Incandescent                   , 2, { 1.32421875, 1, 2.1796875, 0 } },
  { "Nikon", "D5600", Incandescent                   , 3, { 1.33203125, 1, 2.16015625, 0 } },
  { "Nikon", "D5600", Incandescent                   , 4, { 1.34375, 1, 2.14453125, 0 } },
  { "Nikon", "D5600", Incandescent                   , 5, { 1.3515625, 1, 2.12890625, 0 } },
  { "Nikon", "D5600", Incandescent                   , 6, { 1.359375, 1, 2.109375, 0 } },
  { "Nikon", "D5600", Incandescent                   , 7, { 1.37109375, 1, 2.09375, 0 } },
  { "Nikon", "D5600", Incandescent                   , 8, { 1.3828125, 1, 2.078125, 0 } },
  { "Nikon", "D5600", Incandescent                   , 9, { 1.390625, 1, 2.06640625, 0 } },
  { "Nikon", "D5600", Incandescent                   , 10, { 1.40234375, 1, 2.05078125, 0 } },
  { "Nikon", "D5600", Incandescent                   , 11, { 1.4140625, 1, 2.0390625, 0 } },
  { "Nikon", "D5600", Incandescent                   , 12, { 1.42578125, 1, 2.02734375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -12, { 1.19140625, 1, 2.08984375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -11, { 1.19921875, 1, 2.08203125, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -10, { 1.20703125, 1, 2.07421875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -9, { 1.21484375, 1, 2.06640625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -8, { 1.22265625, 1, 2.05859375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -7, { 1.23046875, 1, 2.046875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -6, { 1.23828125, 1, 2.0390625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -5, { 1.24609375, 1, 2.03125, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -4, { 1.25390625, 1, 2.01953125, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -3, { 1.26171875, 1, 2.01171875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -2, { 1.2734375, 1, 2.00390625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , -1, { 1.28125, 1, 1.9921875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 0, { 1.2890625, 1, 1.984375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 1, { 1.296875, 1, 1.9765625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 2, { 1.30859375, 1, 1.96484375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 3, { 1.31640625, 1, 1.95703125, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 4, { 1.32421875, 1, 1.94921875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 5, { 1.3359375, 1, 1.9375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 6, { 1.34375, 1, 1.9296875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 7, { 1.35546875, 1, 1.921875, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 8, { 1.36328125, 1, 1.91015625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 9, { 1.37109375, 1, 1.90234375, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 10, { 1.3828125, 1, 1.890625, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 11, { 1.390625, 1, 1.8828125, 0 } },
  { "Nikon", "D5600", WarmWhiteFluorescent           , 12, { 1.40234375, 1, 1.87109375, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -12, { 1.53515625, 1, 2.3125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -11, { 1.56640625, 1, 2.2890625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -10, { 1.59375, 1, 2.265625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -9, { 1.625, 1, 2.23828125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -8, { 1.65234375, 1, 2.21484375, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -7, { 1.6796875, 1, 2.19140625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -6, { 1.70703125, 1, 2.16796875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -5, { 1.734375, 1, 2.14453125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -4, { 1.76171875, 1, 2.12109375, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -3, { 1.7890625, 1, 2.09765625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -2, { 1.8125, 1, 2.08203125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , -1, { 1.83984375, 1, 2.0625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 0, { 1.86328125, 1, 2.046875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 1, { 1.88671875, 1, 2.03125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 2, { 1.91015625, 1, 2.01171875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 3, { 1.93359375, 1, 1.9921875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 4, { 1.95703125, 1, 1.9765625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 5, { 1.98046875, 1, 1.95703125, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 6, { 2.0, 1, 1.94140625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 7, { 2.0234375, 1, 1.921875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 8, { 2.04296875, 1, 1.90234375, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 9, { 2.0625, 1, 1.88671875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 10, { 2.08203125, 1, 1.8671875, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 11, { 2.10546875, 1, 1.84765625, 0 } },
  { "Nikon", "D5600", CoolWhiteFluorescent           , 12, { 2.12109375, 1, 1.83203125, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -12, { 1.73046875, 1, 1.55859375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -11, { 1.7421875, 1, 1.546875, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -10, { 1.7578125, 1, 1.53515625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -9, { 1.76953125, 1, 1.5234375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -8, { 1.78125, 1, 1.5078125, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -7, { 1.79296875, 1, 1.49609375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -6, { 1.80859375, 1, 1.484375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -5, { 1.8203125, 1, 1.47265625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -4, { 1.8359375, 1, 1.45703125, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -3, { 1.84765625, 1, 1.4453125, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -2, { 1.86328125, 1, 1.43359375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , -1, { 1.875, 1, 1.41796875, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 0, { 1.890625, 1, 1.40625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 1, { 1.90234375, 1, 1.390625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 2, { 1.91796875, 1, 1.37890625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 3, { 1.9296875, 1, 1.3671875, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 4, { 1.94140625, 1, 1.3515625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 5, { 1.95703125, 1, 1.3359375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 6, { 1.97265625, 1, 1.32421875, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 7, { 1.984375, 1, 1.30859375, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 8, { 2.0, 1, 1.296875, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 9, { 2.01171875, 1, 1.28125, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 10, { 2.02734375, 1, 1.265625, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 11, { 2.04296875, 1, 1.25, 0 } },
  { "Nikon", "D5600", DayWhiteFluorescent            , 12, { 2.0546875, 1, 1.23828125, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -12, { 1.9765625, 1, 1.31640625, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -11, { 1.9921875, 1, 1.3046875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -10, { 2.00390625, 1, 1.2890625, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -9, { 2.01953125, 1, 1.2734375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -8, { 2.03125, 1, 1.26171875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -7, { 2.046875, 1, 1.24609375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -6, { 2.0625, 1, 1.23046875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -5, { 2.07421875, 1, 1.21484375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -4, { 2.08984375, 1, 1.19921875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -3, { 2.10546875, 1, 1.1875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -2, { 2.12109375, 1, 1.171875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , -1, { 2.13671875, 1, 1.15625, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 0, { 2.1484375, 1, 1.140625, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 1, { 2.1640625, 1, 1.12109375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 2, { 2.1796875, 1, 1.10546875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 3, { 2.1953125, 1, 1.08984375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 4, { 2.2109375, 1, 1.07421875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 5, { 2.2265625, 1, 1.05859375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 6, { 2.2421875, 1, 1.04296875, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 7, { 2.2578125, 1, 1.0234375, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 8, { 2.2734375, 1, 1.0078125, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 9, { 2.2890625, 1, 1.0, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 10, { 2.3046875, 1, 1.0, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 11, { 2.3203125, 1, 1.0, 0 } },
  { "Nikon", "D5600", DaylightFluorescent            , 12, { 2.3359375, 1, 1.0, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -12, { 1.09765625, 1, 2.65625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -11, { 1.1328125, 1, 2.625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -10, { 1.171875, 1, 2.59765625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -9, { 1.20703125, 1, 2.5703125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -8, { 1.2421875, 1, 2.54296875, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -7, { 1.28125, 1, 2.515625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -6, { 1.3125, 1, 2.48828125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -5, { 1.34765625, 1, 2.46484375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -4, { 1.3828125, 1, 2.4375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -3, { 1.4140625, 1, 2.41015625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -2, { 1.4453125, 1, 2.3828125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , -1, { 1.48046875, 1, 2.359375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 0, { 1.51171875, 1, 2.33203125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 1, { 1.5390625, 1, 2.30859375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 2, { 1.5703125, 1, 2.28515625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 3, { 1.6015625, 1, 2.2578125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 4, { 1.62890625, 1, 2.234375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 5, { 1.65625, 1, 2.2109375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 6, { 1.6875, 1, 2.1875, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 7, { 1.71484375, 1, 2.1640625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 8, { 1.73828125, 1, 2.140625, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 9, { 1.765625, 1, 2.1171875, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 10, { 1.79296875, 1, 2.09375, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 11, { 1.81640625, 1, 2.078125, 0 } },
  { "Nikon", "D5600", WhiteFluorescent               , 12, { 1.84375, 1, 2.0625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -12, { 2.37890625, 1, 1.5625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -11, { 2.390625, 1, 1.546875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -10, { 2.40234375, 1, 1.53125, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -9, { 2.41796875, 1, 1.51171875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -8, { 2.4296875, 1, 1.49609375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -7, { 2.44140625, 1, 1.484375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -6, { 2.453125, 1, 1.46875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -5, { 2.46484375, 1, 1.453125, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -4, { 2.4765625, 1, 1.4375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -3, { 2.48828125, 1, 1.421875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -2, { 2.49609375, 1, 1.41015625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, -1, { 2.5078125, 1, 1.39453125, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 0, { 2.51953125, 1, 1.37890625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 1, { 2.52734375, 1, 1.3671875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 2, { 2.5390625, 1, 1.35546875, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 3, { 2.546875, 1, 1.33984375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 4, { 2.5546875, 1, 1.328125, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 5, { 2.5625, 1, 1.31640625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 6, { 2.57421875, 1, 1.30078125, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 7, { 2.58203125, 1, 1.2890625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 8, { 2.58984375, 1, 1.27734375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 9, { 2.59765625, 1, 1.265625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 10, { 2.60546875, 1, 1.25390625, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 11, { 2.61328125, 1, 1.24609375, 0 } },
  { "Nikon", "D5600", HighTempMercuryVaporFluorescent, 12, { 2.6171875, 1, 1.234375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -12, { 1.140625, 1, 2.703125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -11, { 1.14453125, 1, 2.67578125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -10, { 1.15234375, 1, 2.65234375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -9, { 1.15625, 1, 2.62890625, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -8, { 1.16015625, 1, 2.60546875, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -7, { 1.16796875, 1, 2.58203125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -6, { 1.171875, 1, 2.55859375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -5, { 1.17578125, 1, 2.5390625, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -4, { 1.18359375, 1, 2.51953125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -3, { 1.19140625, 1, 2.49609375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -2, { 1.1953125, 1, 2.4765625, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , -1, { 1.203125, 1, 2.45703125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 0, { 1.2109375, 1, 2.4375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 1, { 1.21875, 1, 2.41796875, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 2, { 1.2265625, 1, 2.3984375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 3, { 1.234375, 1, 2.37890625, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 4, { 1.24609375, 1, 2.359375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 5, { 1.25390625, 1, 2.33984375, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 6, { 1.265625, 1, 2.3203125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 7, { 1.27734375, 1, 2.30078125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 8, { 1.28515625, 1, 2.28125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 9, { 1.296875, 1, 2.26171875, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 10, { 1.3125, 1, 2.23828125, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 11, { 1.32421875, 1, 2.21875, 0 } },
  { "Nikon", "D5600", SodiumVaporFluorescent         , 12, { 1.3359375, 1, 2.1953125, 0 } },
  { "Nikon", "D5600", Flash                          , -12, { 2.08984375, 1, 1.32421875, 0 } },
  { "Nikon", "D5600", Flash                          , -11, { 2.109375, 1, 1.3125, 0 } },
  { "Nikon", "D5600", Flash                          , -10, { 2.12890625, 1, 1.3046875, 0 } },
  { "Nikon", "D5600", Flash                          , -9, { 2.14453125, 1, 1.29296875, 0 } },
  { "Nikon", "D5600", Flash                          , -8, { 2.1640625, 1, 1.28515625, 0 } },
  { "Nikon", "D5600", Flash                          , -7, { 2.1796875, 1, 1.27734375, 0 } },
  { "Nikon", "D5600", Flash                          , -6, { 2.1953125, 1, 1.26953125, 0 } },
  { "Nikon", "D5600", Flash                          , -5, { 2.2109375, 1, 1.26171875, 0 } },
  { "Nikon", "D5600", Flash                          , -4, { 2.22265625, 1, 1.25390625, 0 } },
  { "Nikon", "D5600", Flash                          , -3, { 2.23828125, 1, 1.24609375, 0 } },
  { "Nikon", "D5600", Flash                          , -2, { 2.25390625, 1, 1.2421875, 0 } },
  { "Nikon", "D5600", Flash                          , -1, { 2.26953125, 1, 1.234375, 0 } },
  { "Nikon", "D5600", Flash                          , 0, { 2.28515625, 1, 1.2265625, 0 } },
  { "Nikon", "D5600", Flash                          , 1, { 2.30078125, 1, 1.22265625, 0 } },
  { "Nikon", "D5600", Flash                          , 2, { 2.31640625, 1, 1.21484375, 0 } },
  { "Nikon", "D5600", Flash                          , 3, { 2.33203125, 1, 1.20703125, 0 } },
  { "Nikon", "D5600", Flash                          , 4, { 2.3515625, 1, 1.203125, 0 } },
  { "Nikon", "D5600", Flash                          , 5, { 2.3671875, 1, 1.1953125, 0 } },
  { "Nikon", "D5600", Flash                          , 6, { 2.38671875, 1, 1.1875, 0 } },
  { "Nikon", "D5600", Flash                          , 7, { 2.41015625, 1, 1.1796875, 0 } },
  { "Nikon", "D5600", Flash                          , 8, { 2.4296875, 1, 1.171875, 0 } },
  { "Nikon", "D5600", Flash                          , 9, { 2.453125, 1, 1.1640625, 0 } },
  { "Nikon", "D5600", Flash                          , 10, { 2.48046875, 1, 1.16015625, 0 } },
  { "Nikon", "D5600", Flash                          , 11, { 2.5078125, 1, 1.15234375, 0 } },
  { "Nikon", "D5600", Flash                          , 12, { 2.53515625, 1, 1.14453125, 0 } },

  { "Nikon", "D7000", DirectSunlight                 , -6, { 1.84765625, 1, 1.54296875, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , -5, { 1.8828125, 1, 1.5234375, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , -4, { 1.91015625, 1, 1.5, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , -3, { 1.94140625, 1, 1.47265625, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , -2, { 1.97265625, 1, 1.44921875, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , -1, { 2.0, 1, 1.421875, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 0, { 2.02734375, 1, 1.390625, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 1, { 2.05859375, 1, 1.36328125, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 2, { 2.0859375, 1, 1.3359375, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 3, { 2.1171875, 1, 1.30859375, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 4, { 2.140625, 1, 1.28515625, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 5, { 2.16796875, 1, 1.265625, 0 } },
  { "Nikon", "D7000", DirectSunlight                 , 6, { 2.19921875, 1, 1.2421875, 0 } },
  { "Nikon", "D7000", Shade                          , -6, { 2.24609375, 1, 1.21875, 0 } },
  { "Nikon", "D7000", Shade                          , -5, { 2.27734375, 1, 1.203125, 0 } },
  { "Nikon", "D7000", Shade                          , -4, { 2.31640625, 1, 1.1875, 0 } },
  { "Nikon", "D7000", Shade                          , -3, { 2.359375, 1, 1.171875, 0 } },
  { "Nikon", "D7000", Shade                          , -2, { 2.40625, 1, 1.15625, 0 } },
  { "Nikon", "D7000", Shade                          , -1, { 2.4453125, 1, 1.1484375, 0 } },
  { "Nikon", "D7000", Shade                          , 0, { 2.4921875, 1, 1.13671875, 0 } },
  { "Nikon", "D7000", Shade                          , 1, { 2.5390625, 1, 1.12109375, 0 } },
  { "Nikon", "D7000", Shade                          , 2, { 2.5859375, 1, 1.109375, 0 } },
  { "Nikon", "D7000", Shade                          , 3, { 2.64453125, 1, 1.09375, 0 } },
  { "Nikon", "D7000", Shade                          , 4, { 2.703125, 1, 1.07421875, 0 } },
  { "Nikon", "D7000", Shade                          , 5, { 2.76953125, 1, 1.05078125, 0 } },
  { "Nikon", "D7000", Shade                          , 6, { 2.828125, 1, 1.0234375, 0 } },
  { "Nikon", "D7000", Cloudy                         , -6, { 2.0, 1, 1.421875, 0 } },
  { "Nikon", "D7000", Cloudy                         , -5, { 2.02734375, 1, 1.390625, 0 } },
  { "Nikon", "D7000", Cloudy                         , -4, { 2.05859375, 1, 1.36328125, 0 } },
  { "Nikon", "D7000", Cloudy                         , -3, { 2.0859375, 1, 1.3359375, 0 } },
  { "Nikon", "D7000", Cloudy                         , -2, { 2.1171875, 1, 1.30859375, 0 } },
  { "Nikon", "D7000", Cloudy                         , -1, { 2.140625, 1, 1.28515625, 0 } },
  { "Nikon", "D7000", Cloudy                         , 0, { 2.16796875, 1, 1.265625, 0 } },
  { "Nikon", "D7000", Cloudy                         , 1, { 2.19921875, 1, 1.2421875, 0 } },
  { "Nikon", "D7000", Cloudy                         , 2, { 2.23046875, 1, 1.2265625, 0 } },
  { "Nikon", "D7000", Cloudy                         , 3, { 2.26171875, 1, 1.2109375, 0 } },
  { "Nikon", "D7000", Cloudy                         , 4, { 2.30078125, 1, 1.1953125, 0 } },
  { "Nikon", "D7000", Cloudy                         , 5, { 2.34375, 1, 1.17578125, 0 } },
  { "Nikon", "D7000", Cloudy                         , 6, { 2.38671875, 1, 1.1640625, 0 } },
  { "Nikon", "D7000", Incandescent                   , -6, { 1.21875, 1, 2.46484375, 0 } },
  { "Nikon", "D7000", Incandescent                   , -5, { 1.23046875, 1, 2.421875, 0 } },
  { "Nikon", "D7000", Incandescent                   , -4, { 1.24609375, 1, 2.37890625, 0 } },
  { "Nikon", "D7000", Incandescent                   , -3, { 1.26171875, 1, 2.3359375, 0 } },
  { "Nikon", "D7000", Incandescent                   , -2, { 1.2734375, 1, 2.296875, 0 } },
  { "Nikon", "D7000", Incandescent                   , -1, { 1.2890625, 1, 2.2578125, 0 } },
  { "Nikon", "D7000", Incandescent                   , 0, { 1.3046875, 1, 2.21484375, 0 } },
  { "Nikon", "D7000", Incandescent                   , 1, { 1.32421875, 1, 2.1796875, 0 } },
  { "Nikon", "D7000", Incandescent                   , 2, { 1.33984375, 1, 2.140625, 0 } },
  { "Nikon", "D7000", Incandescent                   , 3, { 1.35546875, 1, 2.1015625, 0 } },
  { "Nikon", "D7000", Incandescent                   , 4, { 1.375, 1, 2.06640625, 0 } },
  { "Nikon", "D7000", Incandescent                   , 5, { 1.39453125, 1, 2.03515625, 0 } },
  { "Nikon", "D7000", Incandescent                   , 6, { 1.41796875, 1, 2.00390625, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -6, { 1.19921875, 1, 2.08203125, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -5, { 1.21484375, 1, 2.0625, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -4, { 1.23046875, 1, 2.04296875, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -3, { 1.24609375, 1, 2.0234375, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -2, { 1.26171875, 1, 2.00390625, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , -1, { 1.28125, 1, 1.984375, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 0, { 1.30078125, 1, 1.96484375, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 1, { 1.31640625, 1, 1.94921875, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 2, { 1.3359375, 1, 1.9296875, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 3, { 1.35546875, 1, 1.9140625, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 4, { 1.375, 1, 1.89453125, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 5, { 1.39453125, 1, 1.87890625, 0 } },
  { "Nikon", "D7000", WarmWhiteFluorescent           , 6, { 1.4140625, 1, 1.859375, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -6, { 1.51953125, 1, 2.28125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -5, { 1.57421875, 1, 2.23828125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -4, { 1.6328125, 1, 2.1953125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -3, { 1.6875, 1, 2.15234375, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -2, { 1.73828125, 1, 2.109375, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , -1, { 1.7890625, 1, 2.0703125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 0, { 1.83203125, 1, 2.02734375, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 1, { 1.8828125, 1, 1.98828125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 2, { 1.93359375, 1, 1.953125, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 3, { 1.9765625, 1, 1.91796875, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 4, { 2.01953125, 1, 1.87890625, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 5, { 2.0625, 1, 1.84375, 0 } },
  { "Nikon", "D7000", CoolWhiteFluorescent           , 6, { 2.10546875, 1, 1.80859375, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -6, { 1.7421875, 1, 1.546875, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -5, { 1.765625, 1, 1.5234375, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -4, { 1.79296875, 1, 1.49609375, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -3, { 1.81640625, 1, 1.47265625, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -2, { 1.84375, 1, 1.44140625, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , -1, { 1.87109375, 1, 1.41796875, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 0, { 1.8984375, 1, 1.390625, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 1, { 1.92578125, 1, 1.36328125, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 2, { 1.953125, 1, 1.33203125, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 3, { 1.984375, 1, 1.3046875, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 4, { 2.01171875, 1, 1.27734375, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 5, { 2.04296875, 1, 1.24609375, 0 } },
  { "Nikon", "D7000", DayWhiteFluorescent            , 6, { 2.0703125, 1, 1.21484375, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -6, { 1.98828125, 1, 1.30078125, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -5, { 2.01953125, 1, 1.26953125, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -4, { 2.046875, 1, 1.2421875, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -3, { 2.078125, 1, 1.2109375, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -2, { 2.109375, 1, 1.1796875, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , -1, { 2.13671875, 1, 1.1484375, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 0, { 2.16796875, 1, 1.1171875, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 1, { 2.19921875, 1, 1.08203125, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 2, { 2.23046875, 1, 1.05078125, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 3, { 2.26171875, 1, 1.015625, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 4, { 2.296875, 1, 1.0, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 5, { 2.328125, 1, 1.0, 0 } },
  { "Nikon", "D7000", DaylightFluorescent            , 6, { 2.359375, 1, 1.0, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -6, { 1.078125, 1, 2.62890625, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -5, { 1.15625, 1, 2.5703125, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -4, { 1.2265625, 1, 2.515625, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -3, { 1.296875, 1, 2.4609375, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -2, { 1.36328125, 1, 2.40625, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , -1, { 1.4296875, 1, 2.35546875, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 0, { 1.49609375, 1, 2.3046875, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 1, { 1.55078125, 1, 2.26171875, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 2, { 1.609375, 1, 2.21484375, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 3, { 1.6640625, 1, 2.171875, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 4, { 1.71484375, 1, 2.12890625, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 5, { 1.765625, 1, 2.0859375, 0 } },
  { "Nikon", "D7000", WhiteFluorescent               , 6, { 1.8125, 1, 2.04296875, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -6, { 2.3515625, 1, 1.5390625, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -5, { 2.375, 1, 1.51171875, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -4, { 2.40234375, 1, 1.4765625, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -3, { 2.42578125, 1, 1.453125, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -2, { 2.44140625, 1, 1.421875, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, -1, { 2.46484375, 1, 1.39453125, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 0, { 2.484375, 1, 1.3671875, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 1, { 2.50390625, 1, 1.34375, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 2, { 2.51953125, 1, 1.31640625, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 3, { 2.53515625, 1, 1.29296875, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 4, { 2.55078125, 1, 1.265625, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 5, { 2.56640625, 1, 1.24609375, 0 } },
  { "Nikon", "D7000", HighTempMercuryVaporFluorescent, 6, { 2.578125, 1, 1.22265625, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -6, { 1.12890625, 1, 2.71875, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -5, { 1.140625, 1, 2.6640625, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -4, { 1.15234375, 1, 2.609375, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -3, { 1.1640625, 1, 2.56640625, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -2, { 1.17578125, 1, 2.5234375, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , -1, { 1.1875, 1, 2.48046875, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 0, { 1.203125, 1, 2.4375, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 1, { 1.21875, 1, 2.3984375, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 2, { 1.23828125, 1, 2.35546875, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 3, { 1.25390625, 1, 2.31640625, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 4, { 1.2734375, 1, 2.27734375, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 5, { 1.296875, 1, 2.23046875, 0 } },
  { "Nikon", "D7000", SodiumVaporFluorescent         , 6, { 1.32421875, 1, 2.1875, 0 } },
  { "Nikon", "D7000", Flash                          , -6, { 2.08984375, 1, 1.30859375, 0 } },
  { "Nikon", "D7000", Flash                          , -5, { 2.125, 1, 1.2890625, 0 } },
  { "Nikon", "D7000", Flash                          , -4, { 2.16015625, 1, 1.2734375, 0 } },
  { "Nikon", "D7000", Flash                          , -3, { 2.19140625, 1, 1.25390625, 0 } },
  { "Nikon", "D7000", Flash                          , -2, { 2.22265625, 1, 1.23828125, 0 } },
  { "Nikon", "D7000", Flash                          , -1, { 2.25, 1, 1.2265625, 0 } },
  { "Nikon", "D7000", Flash                          , 0, { 2.28125, 1, 1.2109375, 0 } },
  { "Nikon", "D7000", Flash                          , 1, { 2.30859375, 1, 1.19921875, 0 } },
  { "Nikon", "D7000", Flash                          , 2, { 2.33984375, 1, 1.18359375, 0 } },
  { "Nikon", "D7000", Flash                          , 3, { 2.375, 1, 1.171875, 0 } },
  { "Nikon", "D7000", Flash                          , 4, { 2.4140625, 1, 1.16015625, 0 } },
  { "Nikon", "D7000", Flash                          , 5, { 2.4609375, 1, 1.14453125, 0 } },
  { "Nikon", "D7000", Flash                          , 6, { 2.51171875, 1, 1.1328125, 0 } },


  /* Nikon D7100 / Firmware C 1.03 / L 2.008 */
  { "Nikon", "D7100", Incandescent, -6,        { 1.589844, 1, 2.371094, 0 } },
  { "Nikon", "D7100", Incandescent, -5,        { 1.5625, 1, 2.417969, 0 } },
  { "Nikon", "D7100", Incandescent, -4,        { 1.542969, 1, 2.46875, 0 } },
  { "Nikon", "D7100", Incandescent, -3,        { 1.519531, 1, 2.519531, 0 } },
  { "Nikon", "D7100", Incandescent, -2,        { 1.492188, 1, 2.566406, 0 } },
  { "Nikon", "D7100", Incandescent, -1,        { 1.46875, 1, 2.617188, 0 } },
  { "Nikon", "D7100", Incandescent,  0,        { 1.445313, 1, 2.667969, 0 } },
  { "Nikon", "D7100", Incandescent,  1,        { 1.425781, 1, 2.71875, 0 } },
  { "Nikon", "D7100", Incandescent,  2,        { 1.410156, 1, 2.769531, 0 } },
  { "Nikon", "D7100", Incandescent,  3,        { 1.390625, 1, 2.820313, 0 } },
  { "Nikon", "D7100", Incandescent,  4,        { 1.371094, 1, 2.875, 0 } },
  { "Nikon", "D7100", Incandescent,  5,        { 1.355469, 1, 2.925781, 0 } },
  { "Nikon", "D7100", Incandescent,  6,        { 1.335938, 1, 2.980469, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -6,         { 1.488281, 1, 2.589844, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -5,         { 1.457031, 1, 2.644531, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -4,         { 1.425781, 1, 2.695313, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -3,         { 1.398438, 1, 2.746094, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -2,         { 1.375, 1, 2.792969, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent, -1,         { 1.355469, 1, 2.847656, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  0,         { 1.332031, 1, 2.894531, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  1,         { 1.3125, 1, 2.949219, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  2,         { 1.296875, 1, 3, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  3,         { 1.28125, 1, 3.058594, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  4,         { 1.261719, 1, 3.113281, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  5,         { 1.25, 1, 3.179688, 0 } },
  { "Nikon", "D7100", SodiumVaporFluorescent,  6,         { 1.234375, 1, 3.246094, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -6,         { 1.621094, 1, 2.28125, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -5,         { 1.597656, 1, 2.304688, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -4,         { 1.574219, 1, 2.328125, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -3,         { 1.546875, 1, 2.351563, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -2,         { 1.527344, 1, 2.378906, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent, -1,         { 1.5, 1, 2.398438, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  0,         { 1.480469, 1, 2.425781, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  1,         { 1.460938, 1, 2.449219, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  2,         { 1.441406, 1, 2.476563, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  3,         { 1.421875, 1, 2.5, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  4,         { 1.402344, 1, 2.523438, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  5,         { 1.382813, 1, 2.546875, 0 } },
  { "Nikon", "D7100", WarmWhiteFluorescent,  6,         { 1.363281, 1, 2.570313, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -6,            { 2.199219, 1, 2.4375, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -5,            { 2.128906, 1, 2.492188, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -4,            { 2.054688, 1, 2.546875, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -3,            { 1.980469, 1, 2.605469, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -2,            { 1.902344, 1, 2.660156, 0 } },
  { "Nikon", "D7100", WhiteFluorescent, -1,            { 1.824219, 1, 2.71875, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  0,            { 1.746094, 1, 2.777344, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  1,            { 1.671875, 1, 2.839844, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  2,            { 1.59375, 1, 2.902344, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  3,            { 1.515625, 1, 2.96875, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  4,            { 1.433594, 1, 3.03125, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  5,            { 1.347656, 1, 3.101563, 0 } },
  { "Nikon", "D7100", WhiteFluorescent,  6,            { 1.261719, 1, 3.167969, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -6,         { 2.601563, 1, 2.136719, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -5,         { 2.542969, 1, 2.179688, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -4,         { 2.484375, 1, 2.226563, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -3,         { 2.421875, 1, 2.273438, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -2,         { 2.359375, 1, 2.320313, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent, -1,         { 2.289063, 1, 2.367188, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  0,         { 2.222656, 1, 2.417969, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  1,         { 2.15625, 1, 2.472656, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  2,         { 2.082031, 1, 2.523438, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  3,         { 2.007813, 1, 2.582031, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  4,         { 1.933594, 1, 2.636719, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  5,         { 1.851563, 1, 2.695313, 0 } },
  { "Nikon", "D7100", CoolWhiteFluorescent,  6,         { 1.773438, 1, 2.75, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -6,        { 2.488281, 1, 1.433594, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -5,        { 2.445313, 1, 1.472656, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -4,        { 2.402344, 1, 1.507813, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -3,        { 2.363281, 1, 1.546875, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -2,        { 2.324219, 1, 1.582031, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent, -1,        { 2.285156, 1, 1.617188, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  0,        { 2.242188, 1, 1.65625, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  1,        { 2.207031, 1, 1.691406, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  2,        { 2.175781, 1, 1.722656, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  3,        { 2.140625, 1, 1.757813, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  4,        { 2.105469, 1, 1.792969, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  5,        { 2.074219, 1, 1.824219, 0 } },
  { "Nikon", "D7100", DayWhiteFluorescent,  6,        { 2.039063, 1, 1.855469, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -6,         { 2.859375, 1, 1.0625, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -5,         { 2.820313, 1, 1.105469, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -4,         { 2.78125, 1, 1.144531, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -3,         { 2.742188, 1, 1.1875, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -2,         { 2.703125, 1, 1.226563, 0 } },
  { "Nikon", "D7100", DaylightFluorescent, -1,         { 2.664063, 1, 1.269531, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  0,         { 2.628906, 1, 1.308594, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  1,         { 2.582031, 1, 1.347656, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  2,         { 2.539063, 1, 1.386719, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  3,         { 2.496094, 1, 1.425781, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  4,         { 2.453125, 1, 1.464844, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  5,         { 2.410156, 1, 1.5, 0 } },
  { "Nikon", "D7100", DaylightFluorescent,  6,         { 2.367188, 1, 1.539063, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -6,        { 3.3125, 1, 1.417969, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -5,        { 3.296875, 1, 1.441406, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -4,        { 3.277344, 1, 1.46875, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -3,        { 3.257813, 1, 1.496094, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -2,        { 3.234375, 1, 1.527344, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent, -1,        { 3.214844, 1, 1.554688, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  0,        { 3.1875, 1, 1.585938, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  1,        { 3.15625, 1, 1.617188, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  2,        { 3.121094, 1, 1.652344, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  3,        { 3.085938, 1, 1.6875, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  4,        { 3.046875, 1, 1.722656, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  5,        { 3.007813, 1, 1.761719, 0 } },
  { "Nikon", "D7100", HighTempMercuryVaporFluorescent,  6,        { 2.96875, 1, 1.796875, 0 } },
  { "Nikon", "D7100", DirectSunlight, -6,     { 2.675781, 1, 1.425781, 0 } },
  { "Nikon", "D7100", DirectSunlight, -5,     { 2.628906, 1, 1.449219, 0 } },
  { "Nikon", "D7100", DirectSunlight, -4,     { 2.582031, 1, 1.476563, 0 } },
  { "Nikon", "D7100", DirectSunlight, -3,     { 2.535156, 1, 1.503906, 0 } },
  { "Nikon", "D7100", DirectSunlight, -2,     { 2.492188, 1, 1.535156, 0 } },
  { "Nikon", "D7100", DirectSunlight, -1,     { 2.441406, 1, 1.566406, 0 } },
  { "Nikon", "D7100", DirectSunlight,  0,     { 2.386719, 1, 1.597656, 0 } },
  { "Nikon", "D7100", DirectSunlight,  1,     { 2.351563, 1, 1.632813, 0 } },
  { "Nikon", "D7100", DirectSunlight,  2,     { 2.3125, 1, 1.671875, 0 } },
  { "Nikon", "D7100", DirectSunlight,  3,     { 2.273438, 1, 1.703125, 0 } },
  { "Nikon", "D7100", DirectSunlight,  4,     { 2.230469, 1, 1.734375, 0 } },
  { "Nikon", "D7100", DirectSunlight,  5,     { 2.1875, 1, 1.761719, 0 } },
  { "Nikon", "D7100", DirectSunlight,  6,     { 2.144531, 1, 1.789063, 0 } },
  { "Nikon", "D7100", Flash, -6,               { 3.132813, 1, 1.28125, 0 } },
  { "Nikon", "D7100", Flash, -5,               { 3.050781, 1, 1.296875, 0 } },
  { "Nikon", "D7100", Flash, -4,               { 2.976563, 1, 1.3125, 0 } },
  { "Nikon", "D7100", Flash, -3,               { 2.914063, 1, 1.328125, 0 } },
  { "Nikon", "D7100", Flash, -2,               { 2.855469, 1, 1.34375, 0 } },
  { "Nikon", "D7100", Flash, -1,               { 2.800781, 1, 1.355469, 0 } },
  { "Nikon", "D7100", Flash,  0,               { 2.753906, 1, 1.371094, 0 } },
  { "Nikon", "D7100", Flash,  1,               { 2.710938, 1, 1.386719, 0 } },
  { "Nikon", "D7100", Flash,  2,               { 2.671875, 1, 1.40625, 0 } },
  { "Nikon", "D7100", Flash,  3,               { 2.628906, 1, 1.425781, 0 } },
  { "Nikon", "D7100", Flash,  4,               { 2.585938, 1, 1.449219, 0 } },
  { "Nikon", "D7100", Flash,  5,               { 2.539063, 1, 1.472656, 0 } },
  { "Nikon", "D7100", Flash,  6,               { 2.488281, 1, 1.5, 0 } },
  { "Nikon", "D7100", Cloudy, -6,              { 2.972656, 1, 1.328125, 0 } },
  { "Nikon", "D7100", Cloudy, -5,              { 2.902344, 1, 1.347656, 0 } },
  { "Nikon", "D7100", Cloudy, -4,              { 2.839844, 1, 1.363281, 0 } },
  { "Nikon", "D7100", Cloudy, -3,              { 2.78125, 1, 1.382813, 0 } },
  { "Nikon", "D7100", Cloudy, -2,              { 2.734375, 1, 1.402344, 0 } },
  { "Nikon", "D7100", Cloudy, -1,              { 2.695313, 1, 1.425781, 0 } },
  { "Nikon", "D7100", Cloudy,  0,              { 2.65625, 1, 1.453125, 0 } },
  { "Nikon", "D7100", Cloudy,  1,              { 2.597656, 1, 1.476563, 0 } },
  { "Nikon", "D7100", Cloudy,  2,              { 2.542969, 1, 1.507813, 0 } },
  { "Nikon", "D7100", Cloudy,  3,              { 2.492188, 1, 1.535156, 0 } },
  { "Nikon", "D7100", Cloudy,  4,              { 2.445313, 1, 1.566406, 0 } },
  { "Nikon", "D7100", Cloudy,  5,              { 2.398438, 1, 1.601563, 0 } },
  { "Nikon", "D7100", Cloudy,  6,              { 2.351563, 1, 1.632813, 0 } },
  { "Nikon", "D7100", Shade, -6,               { 3.644531, 1, 1.148438, 0 } },
  { "Nikon", "D7100", Shade, -5,               { 3.554688, 1, 1.183594, 0 } },
  { "Nikon", "D7100", Shade, -4,               { 3.472656, 1, 1.214844, 0 } },
  { "Nikon", "D7100", Shade, -3,               { 3.386719, 1, 1.234375, 0 } },
  { "Nikon", "D7100", Shade, -2,               { 3.308594, 1, 1.257813, 0 } },
  { "Nikon", "D7100", Shade, -1,               { 3.242188, 1, 1.277344, 0 } },
  { "Nikon", "D7100", Shade,  0,               { 3.183594, 1, 1.292969, 0 } },
  { "Nikon", "D7100", Shade,  1,               { 3.085938, 1, 1.308594, 0 } },
  { "Nikon", "D7100", Shade,  2,               { 2.992188, 1, 1.324219, 0 } },
  { "Nikon", "D7100", Shade,  3,               { 2.925781, 1, 1.339844, 0 } },
  { "Nikon", "D7100", Shade,  4,               { 2.859375, 1, 1.355469, 0 } },
  { "Nikon", "D7100", Shade,  5,               { 2.796875, 1, 1.371094, 0 } },
  { "Nikon", "D7100", Shade,  6,               { 2.746094, 1, 1.394531, 0 } },
  { "Nikon", "D7100", "5000K", 0,               { 2.332031, 1, 1.652344, 0 } },

  // Firmware C 1.00 / L 2.008
  { "Nikon", "D7200", Incandescent, -6,		{ 1.582031, 1, 2.375, 0 } },
  { "Nikon", "D7200", Incandescent, -5,		{ 1.558594, 1, 2.421875, 0 } },
  { "Nikon", "D7200", Incandescent, -4,		{ 1.535156, 1, 2.46875, 0 } },
  { "Nikon", "D7200", Incandescent, -3,		{ 1.511719, 1, 2.519531, 0 } },
  { "Nikon", "D7200", Incandescent, -2,		{ 1.488281, 1, 2.570313, 0 } },
  { "Nikon", "D7200", Incandescent, -1,		{ 1.464844, 1, 2.617188, 0 } },
  { "Nikon", "D7200", Incandescent, 0,		{ 1.445313, 1, 2.667969, 0 } },
  { "Nikon", "D7200", Incandescent, 1,		{ 1.425781, 1, 2.71875, 0 } },
  { "Nikon", "D7200", Incandescent, 2,		{ 1.40625, 1, 2.773438, 0 } },
  { "Nikon", "D7200", Incandescent, 3,		{ 1.386719, 1, 2.824219, 0 } },
  { "Nikon", "D7200", Incandescent, 4,		{ 1.367188, 1, 2.875, 0 } },
  { "Nikon", "D7200", Incandescent, 5,		{ 1.351563, 1, 2.929688, 0 } },
  { "Nikon", "D7200", Incandescent, 6,		{ 1.332031, 1, 2.980469, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -6, { 1.480469, 1, 2.597656, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -5, { 1.449219, 1, 2.652344, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -4, { 1.421875, 1, 2.707031, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -3, { 1.394531, 1, 2.753906, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -2, { 1.367188, 1, 2.804688, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, -1, { 1.347656, 1, 2.855469, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 0,{ 1.324219, 1, 2.906250, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 1, { 1.304688, 1, 2.957031, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 2, { 1.289063, 1, 3.007813, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 3, { 1.273438, 1, 3.066406, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 4, { 1.257813, 1, 3.125000, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 5, { 1.242188, 1, 3.187500, 0 } },
  { "Nikon", "D7200", SodiumVaporFluorescent, 6,{ 1.226563, 1, 3.253906, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -6, { 1.613281, 1, 2.28125, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -5,	{ 1.589844, 1, 2.308594, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -4,	{ 1.566406, 1, 2.332031, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -3,	{ 1.542969, 1, 2.359375, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -2,	{ 1.519531, 1, 2.382813, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, -1,	{ 1.500000, 1, 2.410156, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 0,	{ 1.476563, 1, 2.433594, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 1,	{ 1.457031, 1, 2.457031, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 2,	{ 1.437500, 1, 2.480469, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 3,	{ 1.417969, 1, 2.503906, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 4,	{ 1.398438, 1, 2.527344, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 5,	{ 1.378906, 1, 2.550781, 0 } },
  { "Nikon", "D7200", WarmWhiteFluorescent, 6,	{ 1.363281, 1, 2.574219, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -6,	{ 2.191406, 1, 2.445313, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -5,	{ 2.121094, 1, 2.500000, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -4,	{ 2.050781, 1, 2.554688, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -3,	{ 1.976563, 1, 2.609375, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -2,	{ 1.902344, 1, 2.667969, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, -1,	{ 1.824219, 1, 2.722656, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 0,	{ 1.746094, 1, 2.781250, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 1,	{ 1.667969, 1, 2.843750, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 2,	{ 1.585938, 1, 2.902344, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 3,	{ 1.503906, 1, 2.964844, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 4,	{ 1.421875, 1, 3.031250, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 5,	{ 1.339844, 1, 3.093750, 0 } },
  { "Nikon", "D7200", WhiteFluorescent, 6,	{ 1.253906, 1, 3.160156, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -6, { 2.593750, 1, 2.128906, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -5,	{ 2.535156, 1, 2.175781, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -4,	{ 2.476563, 1, 2.222656, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -3,	{ 2.414063, 1, 2.273438, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -2,	{ 2.351563, 1, 2.324219, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, -1,	{ 2.289063, 1, 2.375000, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 0,	{ 2.222656, 1, 2.425781, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 1,	{ 2.152344, 1, 2.480469, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 2,	{ 2.078125, 1, 2.531250, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 3,	{ 2.007813, 1, 2.585938, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 4,	{ 1.933594, 1, 2.644531, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 5,	{ 1.855469, 1, 2.699219, 0 } },
  { "Nikon", "D7200", CoolWhiteFluorescent, 6,	{ 1.777344, 1, 2.757813, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -6,	{ 2.468750, 1, 1.433594, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -5,	{ 2.429688, 1, 1.472656, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -4,	{ 2.394531, 1, 1.507813, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -3,	{ 2.355469, 1, 1.546875, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -2,	{ 2.320313, 1, 1.582031, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, -1,	{ 2.281250, 1, 1.617188, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 0,	{ 2.246094, 1, 1.652344, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 1,	{ 2.210938, 1, 1.687500, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 2,	{ 2.175781, 1, 1.722656, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 3,	{ 2.136719, 1, 1.757813, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 4,	{ 2.105469, 1, 1.789063, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 5,	{ 2.070313, 1, 1.824219, 0 } },
  { "Nikon", "D7200", DayWhiteFluorescent, 6,	{ 2.035156, 1, 1.855469, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -6,	{ 2.835938, 1, 1.062500, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -5,	{ 2.796875, 1, 1.105469, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -4,	{ 2.753906, 1, 1.144531, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -3,	{ 2.714844, 1, 1.187500, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -2,	{ 2.675781, 1, 1.230469, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, -1,	{ 2.636719, 1, 1.269531, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 0,	{ 2.593750, 1, 1.308594, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 1,	{ 2.554688, 1, 1.347656, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 2,	{ 2.515625, 1, 1.386719, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 3,	{ 2.476563, 1, 1.425781, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 4,	{ 2.437500, 1, 1.464844, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 5,	{ 2.402344, 1, 1.503906, 0 } },
  { "Nikon", "D7200", DaylightFluorescent, 6,	{ 2.363281, 1, 1.539063, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -6, { 3.308594, 1, 1.417969, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -5, { 3.292969, 1, 1.445313, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -4, { 3.269531, 1, 1.468750, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -3, { 3.250000, 1, 1.500000, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -2, { 3.226563, 1, 1.527344, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, -1, { 3.199219, 1, 1.558594, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 0, { 3.171875, 1, 1.589844, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 1, { 3.144531, 1, 1.621094, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 2, { 3.113281, 1, 1.656250, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 3, { 3.078125, 1, 1.691406, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 4, { 3.042969, 1, 1.726563, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 5, { 3.007813, 1, 1.761719, 0 } },
  { "Nikon", "D7200", HighTempMercuryVaporFluorescent, 6, { 2.968750, 1, 1.800781, 0 } },
  { "Nikon", "D7200", DirectSunlight, -6,	{ 2.671875, 1, 1.425781, 0 } },
  { "Nikon", "D7200", DirectSunlight, -5,	{ 2.617188, 1, 1.449219, 0 } },
  { "Nikon", "D7200", DirectSunlight, -4,	{ 2.570313, 1, 1.476563, 0 } },
  { "Nikon", "D7200", DirectSunlight, -3,	{ 2.523438, 1, 1.503906, 0 } },
  { "Nikon", "D7200", DirectSunlight, -2,	{ 2.480469, 1, 1.535156, 0 } },
  { "Nikon", "D7200", DirectSunlight, -1,	{ 2.437500, 1, 1.562500, 0 } },
  { "Nikon", "D7200", DirectSunlight, 0,	{ 2.394531, 1, 1.597656, 0 } },
  { "Nikon", "D7200", DirectSunlight, 1,	{ 2.351563, 1, 1.632813, 0 } },
  { "Nikon", "D7200", DirectSunlight, 2,	{ 2.308594, 1, 1.671875, 0 } },
  { "Nikon", "D7200", DirectSunlight, 3,	{ 2.269531, 1, 1.707031, 0 } },
  { "Nikon", "D7200", DirectSunlight, 4,	{ 2.226563, 1, 1.738281, 0 } },
  { "Nikon", "D7200", DirectSunlight, 5,	{ 2.183594, 1, 1.765625, 0 } },
  { "Nikon", "D7200", DirectSunlight, 6,	{ 2.136719, 1, 1.789063, 0 } },
  { "Nikon", "D7200", Flash, -6,		{ 3.113281, 1, 1.277344, 0 } },
  { "Nikon", "D7200", Flash, -5,		{ 3.027344, 1, 1.292969, 0 } },
  { "Nikon", "D7200", Flash, -4,		{ 2.957031, 1, 1.308594, 0 } },
  { "Nikon", "D7200", Flash, -3,		{ 2.890625, 1, 1.324219, 0 } },
  { "Nikon", "D7200", Flash, -2,		{ 2.835938, 1, 1.339844, 0 } },
  { "Nikon", "D7200", Flash, -1,		{ 2.785156, 1, 1.359375, 0 } },
  { "Nikon", "D7200", Flash, 0,			{ 2.738281, 1, 1.375000, 0 } },
  { "Nikon", "D7200", Flash, 1,			{ 2.695313, 1, 1.394531, 0 } },
  { "Nikon", "D7200", Flash, 2,			{ 2.656250, 1, 1.410156, 0 } },
  { "Nikon", "D7200", Flash, 3,			{ 2.613281, 1, 1.429688, 0 } },
  { "Nikon", "D7200", Flash, 4,			{ 2.570313, 1, 1.453125, 0 } },
  { "Nikon", "D7200", Flash, 5,			{ 2.523438, 1, 1.472656, 0 } },
  { "Nikon", "D7200", Flash, 6,			{ 2.472656, 1, 1.500000, 0 } },
  { "Nikon", "D7200", Cloudy, -6,		{ 2.972656, 1, 1.328125, 0 } },
  { "Nikon", "D7200", Cloudy, -5,		{ 2.906250, 1, 1.347656, 0 } },
  { "Nikon", "D7200", Cloudy, -4,		{ 2.843750, 1, 1.363281, 0 } },
  { "Nikon", "D7200", Cloudy, -3,		{ 2.785156, 1, 1.382813, 0 } },
  { "Nikon", "D7200", Cloudy, -2,		{ 2.726563, 1, 1.406250, 0 } },
  { "Nikon", "D7200", Cloudy, -1,		{ 2.671875, 1, 1.425781, 0 } },
  { "Nikon", "D7200", Cloudy, 0,		{ 2.617188, 1, 1.449219, 0 } },
  { "Nikon", "D7200", Cloudy, 1,		{ 2.570313, 1, 1.476563, 0 } },
  { "Nikon", "D7200", Cloudy, 2,		{ 2.523438, 1, 1.503906, 0 } },
  { "Nikon", "D7200", Cloudy, 3,		{ 2.480469, 1, 1.535156, 0 } },
  { "Nikon", "D7200", Cloudy, 4,		{ 2.437500, 1, 1.562500, 0 } },
  { "Nikon", "D7200", Cloudy, 5,		{ 2.394531, 1, 1.597656, 0 } },
  { "Nikon", "D7200", Cloudy, 6,		{ 2.351563, 1, 1.632813, 0 } },
  { "Nikon", "D7200", Shade, -6,		{ 3.664063, 1, 1.191406, 0 } },
  { "Nikon", "D7200", Shade, -5,		{ 3.570313, 1, 1.207031, 0 } },
  { "Nikon", "D7200", Shade, -4,		{ 3.480469, 1, 1.222656, 0 } },
  { "Nikon", "D7200", Shade, -3,		{ 3.390625, 1, 1.238281, 0 } },
  { "Nikon", "D7200", Shade, -2,		{ 3.304688, 1, 1.253906, 0 } },
  { "Nikon", "D7200", Shade, -1,		{ 3.226563, 1, 1.273438, 0 } },
  { "Nikon", "D7200", Shade, 0,			{ 3.148438, 1, 1.289063, 0 } },
  { "Nikon", "D7200", Shade, 1,			{ 3.074219, 1, 1.304688, 0 } },
  { "Nikon", "D7200", Shade, 2,			{ 3.003906, 1, 1.320313, 0 } },
  { "Nikon", "D7200", Shade, 3,			{ 2.933594, 1, 1.339844, 0 } },
  { "Nikon", "D7200", Shade, 4,			{ 2.871094, 1, 1.359375, 0 } },
  { "Nikon", "D7200", Shade, 5,			{ 2.808594, 1, 1.375000, 0 } },
  { "Nikon", "D7200", Shade, 6,			{ 2.750000, 1, 1.394531, 0 } },
  { "Nikon", "D7200", "2700K", 0,		{ 1.308594, 1, 3.062500, 0 } },
  { "Nikon", "D7200", "5000K", 0,		{ 2.328125, 1, 1.656250, 0 } },
  { "Nikon", "D7200", "5500K", 0,		{ 2.480469, 1, 1.531250, 0 } },
  { "Nikon", "D7200", "6500K", 0,		{ 2.761719, 1, 1.390625, 0 } },

  /* Nikon D7500 / Firmware  C 1.10 / LD 2.018 */
  { "Nikon", "D7500", DirectSunlight                 , -12, { 2.052246094, 1, 1.700683594, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -11, { 2.072753906, 1, 1.688964844, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -10, { 2.092285156, 1, 1.676757812, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -9, { 2.111816406, 1, 1.663574219, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -8, { 2.130859375, 1, 1.649414062, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -7, { 2.148925781, 1, 1.635253906, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -6, { 2.166992188, 1, 1.620117188, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -5, { 2.185058594, 1, 1.604980469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -4, { 2.202148438, 1, 1.589355469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -3, { 2.219726562, 1, 1.57421875, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -2, { 2.236816406, 1, 1.558105469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , -1, { 2.254394531, 1, 1.542480469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 0, { 2.271484375, 1, 1.526855469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 1, { 2.288574219, 1, 1.511230469, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 2, { 2.305664062, 1, 1.49609375, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 3, { 2.323242188, 1, 1.480957031, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 4, { 2.340332031, 1, 1.465820312, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 5, { 2.358398438, 1, 1.451171875, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 6, { 2.376464844, 1, 1.437011719, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 7, { 2.395019531, 1, 1.423339844, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 8, { 2.413574219, 1, 1.41015625, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 9, { 2.432617188, 1, 1.397460938, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 10, { 2.452636719, 1, 1.384765625, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 11, { 2.47265625, 1, 1.373046875, 0 } },
  { "Nikon", "D7500", DirectSunlight                 , 12, { 2.493652344, 1, 1.361816406, 0 } },
  { "Nikon", "D7500", Shade                          , -12, { 2.5546875, 1, 1.333007812, 0 } },
  { "Nikon", "D7500", Shade                          , -11, { 2.577636719, 1, 1.323730469, 0 } },
  { "Nikon", "D7500", Shade                          , -10, { 2.601074219, 1, 1.314453125, 0 } },
  { "Nikon", "D7500", Shade                          , -9, { 2.625488281, 1, 1.306152344, 0 } },
  { "Nikon", "D7500", Shade                          , -8, { 2.650390625, 1, 1.297851562, 0 } },
  { "Nikon", "D7500", Shade                          , -7, { 2.676269531, 1, 1.290527344, 0 } },
  { "Nikon", "D7500", Shade                          , -6, { 2.702636719, 1, 1.282714844, 0 } },
  { "Nikon", "D7500", Shade                          , -5, { 2.729980469, 1, 1.275878906, 0 } },
  { "Nikon", "D7500", Shade                          , -4, { 2.7578125, 1, 1.269042969, 0 } },
  { "Nikon", "D7500", Shade                          , -3, { 2.786132812, 1, 1.262207031, 0 } },
  { "Nikon", "D7500", Shade                          , -2, { 2.815429688, 1, 1.255371094, 0 } },
  { "Nikon", "D7500", Shade                          , -1, { 2.845703125, 1, 1.249023438, 0 } },
  { "Nikon", "D7500", Shade                          , 0, { 2.876464844, 1, 1.242675781, 0 } },
  { "Nikon", "D7500", Shade                          , 1, { 2.908203125, 1, 1.235839844, 0 } },
  { "Nikon", "D7500", Shade                          , 2, { 2.940917969, 1, 1.229003906, 0 } },
  { "Nikon", "D7500", Shade                          , 3, { 2.974121094, 1, 1.221679688, 0 } },
  { "Nikon", "D7500", Shade                          , 4, { 3.008300781, 1, 1.214355469, 0 } },
  { "Nikon", "D7500", Shade                          , 5, { 3.04296875, 1, 1.206054688, 0 } },
  { "Nikon", "D7500", Shade                          , 6, { 3.078613281, 1, 1.197753906, 0 } },
  { "Nikon", "D7500", Shade                          , 7, { 3.114746094, 1, 1.188476562, 0 } },
  { "Nikon", "D7500", Shade                          , 8, { 3.151855469, 1, 1.178710938, 0 } },
  { "Nikon", "D7500", Shade                          , 9, { 3.189453125, 1, 1.167480469, 0 } },
  { "Nikon", "D7500", Shade                          , 10, { 3.228027344, 1, 1.155761719, 0 } },
  { "Nikon", "D7500", Shade                          , 11, { 3.267089844, 1, 1.142578125, 0 } },
  { "Nikon", "D7500", Shade                          , 12, { 3.306640625, 1, 1.127929688, 0 } },
  { "Nikon", "D7500", Cloudy                         , -12, { 2.236816406, 1, 1.558105469, 0 } },
  { "Nikon", "D7500", Cloudy                         , -11, { 2.254394531, 1, 1.542480469, 0 } },
  { "Nikon", "D7500", Cloudy                         , -10, { 2.271484375, 1, 1.526855469, 0 } },
  { "Nikon", "D7500", Cloudy                         , -9, { 2.288574219, 1, 1.511230469, 0 } },
  { "Nikon", "D7500", Cloudy                         , -8, { 2.305664062, 1, 1.49609375, 0 } },
  { "Nikon", "D7500", Cloudy                         , -7, { 2.323242188, 1, 1.480957031, 0 } },
  { "Nikon", "D7500", Cloudy                         , -6, { 2.340332031, 1, 1.465820312, 0 } },
  { "Nikon", "D7500", Cloudy                         , -5, { 2.358398438, 1, 1.451171875, 0 } },
  { "Nikon", "D7500", Cloudy                         , -4, { 2.376464844, 1, 1.437011719, 0 } },
  { "Nikon", "D7500", Cloudy                         , -3, { 2.395019531, 1, 1.423339844, 0 } },
  { "Nikon", "D7500", Cloudy                         , -2, { 2.413574219, 1, 1.41015625, 0 } },
  { "Nikon", "D7500", Cloudy                         , -1, { 2.432617188, 1, 1.397460938, 0 } },
  { "Nikon", "D7500", Cloudy                         , 0, { 2.452636719, 1, 1.384765625, 0 } },
  { "Nikon", "D7500", Cloudy                         , 1, { 2.47265625, 1, 1.373046875, 0 } },
  { "Nikon", "D7500", Cloudy                         , 2, { 2.493652344, 1, 1.361816406, 0 } },
  { "Nikon", "D7500", Cloudy                         , 3, { 2.514648438, 1, 1.351074219, 0 } },
  { "Nikon", "D7500", Cloudy                         , 4, { 2.536621094, 1, 1.340820312, 0 } },
  { "Nikon", "D7500", Cloudy                         , 5, { 2.559082031, 1, 1.331054688, 0 } },
  { "Nikon", "D7500", Cloudy                         , 6, { 2.582519531, 1, 1.321777344, 0 } },
  { "Nikon", "D7500", Cloudy                         , 7, { 2.605957031, 1, 1.312988281, 0 } },
  { "Nikon", "D7500", Cloudy                         , 8, { 2.630371094, 1, 1.3046875, 0 } },
  { "Nikon", "D7500", Cloudy                         , 9, { 2.655761719, 1, 1.296386719, 0 } },
  { "Nikon", "D7500", Cloudy                         , 10, { 2.681640625, 1, 1.2890625, 0 } },
  { "Nikon", "D7500", Cloudy                         , 11, { 2.708007812, 1, 1.28125, 0 } },
  { "Nikon", "D7500", Cloudy                         , 12, { 2.735351562, 1, 1.274414062, 0 } },
  { "Nikon", "D7500", Incandescent                   , -12, { 1.331542969, 1, 2.669433594, 0 } },
  { "Nikon", "D7500", Incandescent                   , -11, { 1.338867188, 1, 2.647949219, 0 } },
  { "Nikon", "D7500", Incandescent                   , -10, { 1.346679688, 1, 2.626464844, 0 } },
  { "Nikon", "D7500", Incandescent                   , -9, { 1.354492188, 1, 2.604492188, 0 } },
  { "Nikon", "D7500", Incandescent                   , -8, { 1.362792969, 1, 2.582519531, 0 } },
  { "Nikon", "D7500", Incandescent                   , -7, { 1.370605469, 1, 2.561035156, 0 } },
  { "Nikon", "D7500", Incandescent                   , -6, { 1.37890625, 1, 2.539550781, 0 } },
  { "Nikon", "D7500", Incandescent                   , -5, { 1.387695312, 1, 2.517578125, 0 } },
  { "Nikon", "D7500", Incandescent                   , -4, { 1.395996094, 1, 2.49609375, 0 } },
  { "Nikon", "D7500", Incandescent                   , -3, { 1.404785156, 1, 2.474609375, 0 } },
  { "Nikon", "D7500", Incandescent                   , -2, { 1.413574219, 1, 2.453613281, 0 } },
  { "Nikon", "D7500", Incandescent                   , -1, { 1.422363281, 1, 2.432617188, 0 } },
  { "Nikon", "D7500", Incandescent                   , 0, { 1.431640625, 1, 2.411621094, 0 } },
  { "Nikon", "D7500", Incandescent                   , 1, { 1.440917969, 1, 2.391113281, 0 } },
  { "Nikon", "D7500", Incandescent                   , 2, { 1.450195312, 1, 2.37109375, 0 } },
  { "Nikon", "D7500", Incandescent                   , 3, { 1.459960938, 1, 2.3515625, 0 } },
  { "Nikon", "D7500", Incandescent                   , 4, { 1.469238281, 1, 2.33203125, 0 } },
  { "Nikon", "D7500", Incandescent                   , 5, { 1.479003906, 1, 2.312988281, 0 } },
  { "Nikon", "D7500", Incandescent                   , 6, { 1.489257812, 1, 2.294433594, 0 } },
  { "Nikon", "D7500", Incandescent                   , 7, { 1.499023438, 1, 2.276855469, 0 } },
  { "Nikon", "D7500", Incandescent                   , 8, { 1.509277344, 1, 2.259765625, 0 } },
  { "Nikon", "D7500", Incandescent                   , 9, { 1.520019531, 1, 2.243164062, 0 } },
  { "Nikon", "D7500", Incandescent                   , 10, { 1.530273438, 1, 2.227539062, 0 } },
  { "Nikon", "D7500", Incandescent                   , 11, { 1.541015625, 1, 2.212402344, 0 } },
  { "Nikon", "D7500", Incandescent                   , 12, { 1.551757812, 1, 2.198242188, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -12, { 1.319335938, 1, 2.265625, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -11, { 1.327636719, 1, 2.255859375, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -10, { 1.336914062, 1, 2.246582031, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -9, { 1.345703125, 1, 2.237304688, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -8, { 1.354492188, 1, 2.227539062, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -7, { 1.363769531, 1, 2.218261719, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -6, { 1.373046875, 1, 2.208496094, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -5, { 1.382324219, 1, 2.19921875, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -4, { 1.391601562, 1, 2.189453125, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -3, { 1.400878906, 1, 2.180175781, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -2, { 1.410644531, 1, 2.170410156, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , -1, { 1.420410156, 1, 2.160644531, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 0, { 1.430175781, 1, 2.150878906, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 1, { 1.439941406, 1, 2.141113281, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 2, { 1.449707031, 1, 2.131347656, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 3, { 1.459960938, 1, 2.121582031, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 4, { 1.470214844, 1, 2.111328125, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 5, { 1.48046875, 1, 2.1015625, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 6, { 1.490722656, 1, 2.091308594, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 7, { 1.500976562, 1, 2.081054688, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 8, { 1.51171875, 1, 2.071289062, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 9, { 1.522460938, 1, 2.061035156, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 10, { 1.533203125, 1, 2.05078125, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 11, { 1.543945312, 1, 2.040527344, 0 } },
  { "Nikon", "D7500", WarmWhiteFluorescent           , 12, { 1.5546875, 1, 2.029785156, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -12, { 1.665527344, 1, 2.53125, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -11, { 1.698730469, 1, 2.50390625, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -10, { 1.731445312, 1, 2.477050781, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -9, { 1.763671875, 1, 2.450683594, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -8, { 1.794921875, 1, 2.424804688, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -7, { 1.826171875, 1, 2.399414062, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -6, { 1.856933594, 1, 2.374511719, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -5, { 1.887207031, 1, 2.349609375, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -4, { 1.916992188, 1, 2.325195312, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -3, { 1.946777344, 1, 2.301757812, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -2, { 1.975585938, 1, 2.278320312, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , -1, { 2.004394531, 1, 2.255371094, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 0, { 2.032714844, 1, 2.232910156, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 1, { 2.061035156, 1, 2.2109375, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 2, { 2.088867188, 1, 2.189453125, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 3, { 2.116699219, 1, 2.168457031, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 4, { 2.144042969, 1, 2.147949219, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 5, { 2.171386719, 1, 2.127441406, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 6, { 2.198730469, 1, 2.107910156, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 7, { 2.226074219, 1, 2.089355469, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 8, { 2.252929688, 1, 2.070800781, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 9, { 2.280273438, 1, 2.052734375, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 10, { 2.307128906, 1, 2.035644531, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 11, { 2.33203125, 1, 2.015136719, 0 } },
  { "Nikon", "D7500", CoolWhiteFluorescent           , 12, { 2.355957031, 1, 1.994628906, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -12, { 1.919433594, 1, 1.687988281, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -11, { 1.93359375, 1, 1.674804688, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -10, { 1.947753906, 1, 1.661621094, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -9, { 1.961914062, 1, 1.647949219, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -8, { 1.976074219, 1, 1.634277344, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -7, { 1.990722656, 1, 1.620605469, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -6, { 2.005371094, 1, 1.606445312, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -5, { 2.020019531, 1, 1.592285156, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -4, { 2.034667969, 1, 1.578125, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -3, { 2.049316406, 1, 1.563964844, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -2, { 2.064453125, 1, 1.549316406, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , -1, { 2.079589844, 1, 1.53515625, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 0, { 2.094726562, 1, 1.520019531, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 1, { 2.109863281, 1, 1.505371094, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 2, { 2.125488281, 1, 1.490234375, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 3, { 2.140625, 1, 1.474609375, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 4, { 2.15625, 1, 1.459472656, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 5, { 2.171875, 1, 1.443847656, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 6, { 2.187988281, 1, 1.428222656, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 7, { 2.203613281, 1, 1.412597656, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 8, { 2.219726562, 1, 1.396484375, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 9, { 2.235839844, 1, 1.380371094, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 10, { 2.251953125, 1, 1.363769531, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 11, { 2.268066406, 1, 1.34765625, 0 } },
  { "Nikon", "D7500", DayWhiteFluorescent            , 12, { 2.284667969, 1, 1.331054688, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -12, { 2.194335938, 1, 1.421875, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -11, { 2.209960938, 1, 1.405761719, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -10, { 2.226074219, 1, 1.390136719, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -9, { 2.2421875, 1, 1.373535156, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -8, { 2.258300781, 1, 1.357421875, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -7, { 2.274902344, 1, 1.340820312, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -6, { 2.291015625, 1, 1.32421875, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -5, { 2.307617188, 1, 1.307128906, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -4, { 2.32421875, 1, 1.290527344, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -3, { 2.340820312, 1, 1.272949219, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -2, { 2.357910156, 1, 1.255859375, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , -1, { 2.375, 1, 1.23828125, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 0, { 2.392089844, 1, 1.220703125, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 1, { 2.409179688, 1, 1.202636719, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 2, { 2.426269531, 1, 1.184570312, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 3, { 2.443359375, 1, 1.166503906, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 4, { 2.4609375, 1, 1.1484375, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 5, { 2.478515625, 1, 1.129882812, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 6, { 2.49609375, 1, 1.110839844, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 7, { 2.514160156, 1, 1.092285156, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 8, { 2.531738281, 1, 1.073242188, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 9, { 2.549804688, 1, 1.053710938, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 10, { 2.567871094, 1, 1.034179688, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 11, { 2.5859375, 1, 1.014648438, 0 } },
  { "Nikon", "D7500", DaylightFluorescent            , 12, { 2.604003906, 1, 1.0, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -12, { 1.185058594, 1, 2.909179688, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -11, { 1.2265625, 1, 2.877929688, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -10, { 1.267578125, 1, 2.846679688, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -9, { 1.307617188, 1, 2.815917969, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -8, { 1.347167969, 1, 2.78515625, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -7, { 1.385742188, 1, 2.754882812, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -6, { 1.423828125, 1, 2.725097656, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -5, { 1.461425781, 1, 2.695800781, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -4, { 1.498046875, 1, 2.666503906, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -3, { 1.534179688, 1, 2.637695312, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -2, { 1.569335938, 1, 2.608886719, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , -1, { 1.604492188, 1, 2.581054688, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 0, { 1.638671875, 1, 2.553222656, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 1, { 1.672363281, 1, 2.525390625, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 2, { 1.705078125, 1, 2.498535156, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 3, { 1.737792969, 1, 2.471679688, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 4, { 1.770019531, 1, 2.445800781, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 5, { 1.801269531, 1, 2.419921875, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 6, { 1.832519531, 1, 2.39453125, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 7, { 1.86328125, 1, 2.369140625, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 8, { 1.893066406, 1, 2.344726562, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 9, { 1.922851562, 1, 2.320800781, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 10, { 1.952636719, 1, 2.296875, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 11, { 1.981445312, 1, 2.2734375, 0 } },
  { "Nikon", "D7500", WhiteFluorescent               , 12, { 2.010253906, 1, 2.250976562, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -12, { 2.668457031, 1, 1.693359375, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -11, { 2.685058594, 1, 1.67578125, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -10, { 2.701171875, 1, 1.658203125, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -9, { 2.716796875, 1, 1.641113281, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -8, { 2.732421875, 1, 1.624023438, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -7, { 2.747070312, 1, 1.607421875, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -6, { 2.76171875, 1, 1.590820312, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -5, { 2.775390625, 1, 1.574707031, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -4, { 2.7890625, 1, 1.559082031, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -3, { 2.802246094, 1, 1.543457031, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -2, { 2.815429688, 1, 1.527832031, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, -1, { 2.827636719, 1, 1.512695312, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 0, { 2.83984375, 1, 1.498046875, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 1, { 2.8515625, 1, 1.483398438, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 2, { 2.862792969, 1, 1.469238281, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 3, { 2.873535156, 1, 1.455078125, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 4, { 2.884277344, 1, 1.44140625, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 5, { 2.89453125, 1, 1.428222656, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 6, { 2.904296875, 1, 1.415039062, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 7, { 2.913574219, 1, 1.401855469, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 8, { 2.922851562, 1, 1.389648438, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 9, { 2.931640625, 1, 1.377441406, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 10, { 2.939941406, 1, 1.365234375, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 11, { 2.948242188, 1, 1.353515625, 0 } },
  { "Nikon", "D7500", HighTempMercuryVaporFluorescent, 12, { 2.956054688, 1, 1.341796875, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -12, { 1.279296875, 1, 3.14453125, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -11, { 1.284667969, 1, 3.112792969, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -10, { 1.290527344, 1, 3.08203125, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -9, { 1.296386719, 1, 3.052734375, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -8, { 1.301757812, 1, 3.024902344, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -7, { 1.307617188, 1, 2.997558594, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -6, { 1.313476562, 1, 2.970703125, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -5, { 1.319824219, 1, 2.9453125, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -4, { 1.326171875, 1, 2.920410156, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -3, { 1.332519531, 1, 2.895996094, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -2, { 1.339355469, 1, 2.872070312, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , -1, { 1.346191406, 1, 2.849121094, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 0, { 1.354003906, 1, 2.826171875, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 1, { 1.361816406, 1, 2.803710938, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 2, { 1.369628906, 1, 2.78125, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 3, { 1.378417969, 1, 2.758789062, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 4, { 1.387695312, 1, 2.736816406, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 5, { 1.397460938, 1, 2.71484375, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 6, { 1.407714844, 1, 2.692382812, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 7, { 1.418457031, 1, 2.669921875, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 8, { 1.4296875, 1, 2.646972656, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 9, { 1.441894531, 1, 2.624023438, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 10, { 1.454589844, 1, 2.600097656, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 11, { 1.468261719, 1, 2.576171875, 0 } },
  { "Nikon", "D7500", SodiumVaporFluorescent         , 12, { 1.482421875, 1, 2.551269531, 0 } },
  { "Nikon", "D7500", Flash                          , -12, { 2.351074219, 1, 1.426269531, 0 } },
  { "Nikon", "D7500", Flash                          , -11, { 2.375, 1, 1.415527344, 0 } },
  { "Nikon", "D7500", Flash                          , -10, { 2.397949219, 1, 1.404785156, 0 } },
  { "Nikon", "D7500", Flash                          , -9, { 2.419433594, 1, 1.395019531, 0 } },
  { "Nikon", "D7500", Flash                          , -8, { 2.440429688, 1, 1.385253906, 0 } },
  { "Nikon", "D7500", Flash                          , -7, { 2.460449219, 1, 1.376464844, 0 } },
  { "Nikon", "D7500", Flash                          , -6, { 2.479980469, 1, 1.367675781, 0 } },
  { "Nikon", "D7500", Flash                          , -5, { 2.499023438, 1, 1.359375, 0 } },
  { "Nikon", "D7500", Flash                          , -4, { 2.517578125, 1, 1.351074219, 0 } },
  { "Nikon", "D7500", Flash                          , -3, { 2.536132812, 1, 1.343261719, 0 } },
  { "Nikon", "D7500", Flash                          , -2, { 2.555175781, 1, 1.335449219, 0 } },
  { "Nikon", "D7500", Flash                          , -1, { 2.573730469, 1, 1.328125, 0 } },
  { "Nikon", "D7500", Flash                          , 0, { 2.592773438, 1, 1.320800781, 0 } },
  { "Nikon", "D7500", Flash                          , 1, { 2.612304688, 1, 1.313476562, 0 } },
  { "Nikon", "D7500", Flash                          , 2, { 2.6328125, 1, 1.306640625, 0 } },
  { "Nikon", "D7500", Flash                          , 3, { 2.653808594, 1, 1.299316406, 0 } },
  { "Nikon", "D7500", Flash                          , 4, { 2.676269531, 1, 1.292480469, 0 } },
  { "Nikon", "D7500", Flash                          , 5, { 2.699707031, 1, 1.28515625, 0 } },
  { "Nikon", "D7500", Flash                          , 6, { 2.724121094, 1, 1.277832031, 0 } },
  { "Nikon", "D7500", Flash                          , 7, { 2.750488281, 1, 1.270507812, 0 } },
  { "Nikon", "D7500", Flash                          , 8, { 2.778320312, 1, 1.263183594, 0 } },
  { "Nikon", "D7500", Flash                          , 9, { 2.808105469, 1, 1.255371094, 0 } },
  { "Nikon", "D7500", Flash                          , 10, { 2.840332031, 1, 1.247558594, 0 } },
  { "Nikon", "D7500", Flash                          , 11, { 2.874511719, 1, 1.239257812, 0 } },
  { "Nikon", "D7500", Flash                          , 12, { 2.910644531, 1, 1.23046875, 0 } },

  { "Nikon", "E5400", Daylight, -3,		{ 2.046875, 1, 1.449219, 0 } },
  { "Nikon", "E5400", Daylight, 0,		{ 1.800781, 1, 1.636719, 0 } },
  { "Nikon", "E5400", Daylight, 3,		{ 1.539062, 1, 1.820312, 0 } },
  { "Nikon", "E5400", Incandescent, -3,		{ 1.218750, 1, 2.656250, 0 } },
  { "Nikon", "E5400", Incandescent, 0,		{ 1.218750, 1, 2.656250, 0 } },
  { "Nikon", "E5400", Incandescent, 3,		{ 1.382812, 1, 2.351562, 0 } },
  { "Nikon", "E5400", Fluorescent, -3,		{ 1.703125, 1, 2.460938, 0 } },
  { "Nikon", "E5400", Fluorescent, 0,		{ 1.218750, 1, 2.656250, 0 } },
  { "Nikon", "E5400", Fluorescent, 3,		{ 1.953125, 1, 1.906250, 0 } },
  { "Nikon", "E5400", Cloudy, -3,		{ 1.703125, 1, 2.460938, 0 } },
  { "Nikon", "E5400", Cloudy, 0,		{ 1.996094, 1, 1.421875, 0 } },
  { "Nikon", "E5400", Cloudy, 3,		{ 2.265625, 1, 1.261719, 0 } },
  { "Nikon", "E5400", Flash, -3,		{ 2.792969, 1, 1.152344, 0 } },
  { "Nikon", "E5400", Flash, 0,			{ 2.328125, 1, 1.386719, 0 } },
  { "Nikon", "E5400", Flash, 3,			{ 2.328125, 1, 1.386719, 0 } },
  { "Nikon", "E5400", Shade, -3,		{ 2.722656, 1, 1.011719, 0 } },
  { "Nikon", "E5400", Shade, 0,			{ 2.269531, 1, 1.218750, 0 } },
  { "Nikon", "E5400", Shade, 3,			{ 2.269531, 1, 1.218750, 0 } },

  { "Nikon", "E8700", Daylight, 0,		{ 1.968750, 1, 1.582031, 0 } },
  { "Nikon", "E8700", Incandescent, 0,		{ 1.265625, 1, 2.765625, 0 } },
  { "Nikon", "E8700", Fluorescent, 0,		{ 1.863281, 1, 2.304688, 0 } },
  { "Nikon", "E8700", Cloudy, 0,		{ 2.218750, 1, 1.359375, 0 } },
  { "Nikon", "E8700", Flash, 0,			{ 2.535156, 1, 1.273438, 0 } },
  { "Nikon", "E8700", Shade, 0,			{ 2.527344, 1, 1.175781, 0 } },

  // Nikon Z6, firmware 3.0
  { "Nikon", "Z 6", DirectSunlight, -6,         { 1.582031, 1, 1.488281, 0 } },
  { "Nikon", "Z 6", DirectSunlight, -5,         { 1.609375, 1, 1.466797, 0 } },
  { "Nikon", "Z 6", DirectSunlight, -4,         { 1.636719, 1, 1.443359, 0 } },
  { "Nikon", "Z 6", DirectSunlight, -3,         { 1.660156, 1, 1.417969, 0 } },
  { "Nikon", "Z 6", DirectSunlight, -2,         { 1.683594, 1, 1.390625, 0 } },
  { "Nikon", "Z 6", DirectSunlight, -1,         { 1.705078, 1, 1.363281, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  0,         { 1.726563, 1, 1.335938, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  1,         { 1.75, 1, 1.308594, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  2,         { 1.771484, 1, 1.283203, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  3,         { 1.792969, 1, 1.257813, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  4,         { 1.816406, 1, 1.234375, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  5,         { 1.839844, 1, 1.212891, 0 } },
  { "Nikon", "Z 6", DirectSunlight,  6,         { 1.865234, 1, 1.193359, 0 } },
  { "Nikon", "Z 6", Cloudy, -6, { 1.705078, 1, 1.363281, 0 } },
  { "Nikon", "Z 6", Cloudy, -5, { 1.726563, 1, 1.335938, 0 } },
  { "Nikon", "Z 6", Cloudy, -4, { 1.75, 1, 1.308594, 0 } },
  { "Nikon", "Z 6", Cloudy, -3, { 1.771484, 1, 1.283203, 0 } },
  { "Nikon", "Z 6", Cloudy, -2, { 1.792969, 1, 1.257813, 0 } },
  { "Nikon", "Z 6", Cloudy, -1, { 1.816406, 1, 1.234375, 0 } },
  { "Nikon", "Z 6", Cloudy,  0, { 1.839844, 1, 1.212891, 0 } },
  { "Nikon", "Z 6", Cloudy,  1, { 1.865234, 1, 1.193359, 0 } },
  { "Nikon", "Z 6", Cloudy,  2, { 1.890625, 1, 1.175781, 0 } },
  { "Nikon", "Z 6", Cloudy,  3, { 1.917969, 1, 1.158203, 0 } },
  { "Nikon", "Z 6", Cloudy,  4, { 1.947266, 1, 1.144531, 0 } },
  { "Nikon", "Z 6", Cloudy,  5, { 1.978516, 1, 1.130859, 0 } },
  { "Nikon", "Z 6", Cloudy,  6, { 2.011719, 1, 1.119141, 0 } },
  { "Nikon", "Z 6", Shade, -6,  { 1.902344, 1, 1.167969, 0 } },
  { "Nikon", "Z 6", Shade, -5,  { 1.929688, 1, 1.152344, 0 } },
  { "Nikon", "Z 6", Shade, -4,  { 1.958984, 1, 1.138672, 0 } },
  { "Nikon", "Z 6", Shade, -3,  { 1.992188, 1, 1.125, 0 } },
  { "Nikon", "Z 6", Shade, -2,  { 2.025391, 1, 1.113281, 0 } },
  { "Nikon", "Z 6", Shade, -1,  { 2.060547, 1, 1.101563, 0 } },
  { "Nikon", "Z 6", Shade,  0,  { 2.097656, 1, 1.089844, 0 } },
  { "Nikon", "Z 6", Shade,  1,  { 2.138672, 1, 1.078125, 0 } },
  { "Nikon", "Z 6", Shade,  2,  { 2.179688, 1, 1.064453, 0 } },
  { "Nikon", "Z 6", Shade,  3,  { 2.222656, 1, 1.050781, 0 } },
  { "Nikon", "Z 6", Shade,  4,  { 2.269531, 1, 1.033203, 0 } },
  { "Nikon", "Z 6", Shade,  5,  { 2.316406, 1, 1.011719, 0 } },
  { "Nikon", "Z 6", Shade,  6,  { 2.367188, 1, 1, 0 } },
  { "Nikon", "Z 6", Incandescent, -6, { 1.0625, 1, 2.302734, 0 } },
  { "Nikon", "Z 6", Incandescent, -5, { 1.074219, 1, 2.265625, 0 } },
  { "Nikon", "Z 6", Incandescent, -4, { 1.085938, 1, 2.230469, 0 } },
  { "Nikon", "Z 6", Incandescent, -3, { 1.099609, 1, 2.193359, 0 } },
  { "Nikon", "Z 6", Incandescent, -2, { 1.111328, 1, 2.158203, 0 } },
  { "Nikon", "Z 6", Incandescent, -1, { 1.125, 1, 2.121094, 0 } },
  { "Nikon", "Z 6", Incandescent,  0, { 1.138672, 1, 2.083984, 0 } },
  { "Nikon", "Z 6", Incandescent,  1, { 1.152344, 1, 2.046875, 0 } },
  { "Nikon", "Z 6", Incandescent,  2, { 1.167969, 1, 2.009766, 0 } },
  { "Nikon", "Z 6", Incandescent,  3, { 1.181641, 1, 1.972656, 0 } },
  { "Nikon", "Z 6", Incandescent,  4, { 1.197266, 1, 1.945313, 0 } },
  { "Nikon", "Z 6", Incandescent,  5, { 1.212891, 1, 1.9375, 0 } },
  { "Nikon", "Z 6", Incandescent,  6, { 1.228516, 1, 1.933594, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -6, { 1.017578, 1, 2.591797, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -5, { 1.027344, 1, 2.542969, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -4, { 1.035156, 1, 2.498047, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -3, { 1.044922, 1, 2.455078, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -2, { 1.054688, 1, 2.416016, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent, -1, { 1.064453, 1, 2.376953, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  0, { 1.076172, 1, 2.339844, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  1, { 1.089844, 1, 2.304688, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  2, { 1.103516, 1, 2.269531, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  3, { 1.119141, 1, 2.232422, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  4, { 1.136719, 1, 2.195313, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  5, { 1.15625, 1, 2.15625, 0 } },
  { "Nikon", "Z 6", SodiumVaporFluorescent,  6, { 1.177734, 1, 2.115234, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -6, { 1.042969, 1, 1.978516, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -5, { 1.056641, 1, 1.960938, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -4, { 1.072266, 1, 1.943359, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -3, { 1.085938, 1, 1.925781, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -2, { 1.099609, 1, 1.908203, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent, -1, { 1.115234, 1, 1.892578, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  0, { 1.130859, 1, 1.875, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  1, { 1.146484, 1, 1.857422, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  2, { 1.160156, 1, 1.841797, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  3, { 1.177734, 1, 1.824219, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  4, { 1.193359, 1, 1.808594, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  5, { 1.208984, 1, 1.791016, 0 } },
  { "Nikon", "Z 6", WarmWhiteFluorescent,  6, { 1.224609, 1, 1.773438, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -6, { 0.974609, 1, 2.535156, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -5, { 1.03125, 1, 2.478516, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -4, { 1.095703, 1, 2.425781, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -3, { 1.160156, 1, 2.373047, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -2, { 1.220703, 1, 2.320313, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent, -1, { 1.279297, 1, 2.271484, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  0, { 1.333984, 1, 2.222656, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  1, { 1.386719, 1, 2.175781, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  2, { 1.435547, 1, 2.132813, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  3, { 1.482422, 1, 2.089844, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  4, { 1.525391, 1, 2.048828, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  5, { 1.566406, 1, 2.007813, 0 } },
  { "Nikon", "Z 6", WhiteFluorescent,  6, { 1.607422, 1, 1.96875, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -6, { 1.355469, 1, 2.205078, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -5, { 1.40625, 1, 2.158203, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -4, { 1.455078, 1, 2.115234, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -3, { 1.5, 1, 2.074219, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -2, { 1.542969, 1, 2.033203, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent, -1, { 1.583984, 1, 1.992188, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  0, { 1.623047, 1, 1.953125, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  1, { 1.664063, 1, 1.914063, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  2, { 1.703125, 1, 1.875, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  3, { 1.740234, 1, 1.837891, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  4, { 1.775391, 1, 1.800781, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  5, { 1.810547, 1, 1.763672, 0 } },
  { "Nikon", "Z 6", CoolWhiteFluorescent,  6, { 1.84375, 1, 1.728516, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -6, { 1.494141, 1, 1.488281, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -5, { 1.513672, 1, 1.464844, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -4, { 1.535156, 1, 1.441406, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -3, { 1.556641, 1, 1.416016, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -2, { 1.578125, 1, 1.392578, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent, -1, { 1.599609, 1, 1.367188, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  0, { 1.621094, 1, 1.341797, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  1, { 1.642578, 1, 1.316406, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  2, { 1.666016, 1, 1.291016, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  3, { 1.689453, 1, 1.265625, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  4, { 1.712891, 1, 1.238281, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  5, { 1.736328, 1, 1.210938, 0 } },
  { "Nikon", "Z 6", DayWhiteFluorescent,  6, { 1.759766, 1, 1.181641, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -6, { 1.693359, 1, 1.259766, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -5, { 1.716797, 1, 1.232422, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -4, { 1.740234, 1, 1.205078, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -3, { 1.763672, 1, 1.175781, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -2, { 1.789063, 1, 1.148438, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent, -1, { 1.8125, 1, 1.119141, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  0, { 1.837891, 1, 1.087891, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  1, { 1.863281, 1, 1.056641, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  2, { 1.888672, 1, 1.023438, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  3, { 1.916016, 1, 1, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  4, { 1.941406, 1, 1, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  5, { 1.96875, 1, 1, 0 } },
  { "Nikon", "Z 6", DaylightFluorescent,  6, { 1.996094, 1, 1, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -6, { 2.041016, 1, 1.474609, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -5, { 2.060547, 1, 1.445313, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -4, { 2.078125, 1, 1.416016, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -3, { 2.095703, 1, 1.388672, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -2, { 2.111328, 1, 1.361328, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent, -1, { 2.126953, 1, 1.335938, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  0, { 2.142578, 1, 1.308594, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  1, { 2.15625, 1, 1.285156, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  2, { 2.171875, 1, 1.259766, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  3, { 2.185547, 1, 1.236328, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  4, { 2.199219, 1, 1.214844, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  5, { 2.210938, 1, 1.193359, 0 } },
  { "Nikon", "Z 6", HighTempMercuryVaporFluorescent,  6, { 2.222656, 1, 1.171875, 0 } },
  { "Nikon", "Z 6", Flash, -6,                   { 1.771484, 1, 1.253906, 0 } },
  { "Nikon", "Z 6", Flash, -5,                   { 1.800781, 1, 1.236328, 0 } },
  { "Nikon", "Z 6", Flash, -4,                   { 1.826172, 1, 1.21875, 0 } },
  { "Nikon", "Z 6", Flash, -3,                   { 1.849609, 1, 1.203125, 0 } },
  { "Nikon", "Z 6", Flash, -2,                   { 1.871094, 1, 1.189453, 0 } },
  { "Nikon", "Z 6", Flash, -1,                   { 1.892578, 1, 1.175781, 0 } },
  { "Nikon", "Z 6", Flash, 0,                   { 1.916016, 1, 1.164063, 0 } },
  { "Nikon", "Z 6", Flash, 1,                   { 1.9375, 1, 1.150391, 0 } },
  { "Nikon", "Z 6", Flash, 2,                   { 1.962891, 1, 1.138672, 0 } },
  { "Nikon", "Z 6", Flash, 3,                   { 1.992188, 1, 1.126953, 0 } },
  { "Nikon", "Z 6", Flash, 4,                   { 2.023438, 1, 1.113281, 0 } },
  { "Nikon", "Z 6", Flash, 5,                   { 2.060547, 1, 1.099609, 0 } },
  { "Nikon", "Z 6", Flash, 6,                   { 2.103516, 1, 1.085938, 0 } },

  { "Nikon", "Z 7", DirectSunlight                 , 0, { 1.71484375, 1, 1.359375, 0 } },
  { "Nikon", "Z 7", Shade                          , 0, { 2.1171875, 1, 1.09765625, 0 } },
  { "Nikon", "Z 7", Cloudy                         , 0, { 1.837890625, 1, 1.228515625, 0 } },
  { "Nikon", "Z 7", WarmWhiteFluorescent           , 0, { 1.1015625, 1, 1.982421875, 0 } },
  { "Nikon", "Z 7", CoolWhiteFluorescent           , 0, { 1.599609375, 1, 2.0234375, 0 } },
  { "Nikon", "Z 7", DayWhiteFluorescent            , 0, { 1.6015625, 1, 1.37890625, 0 } },
  { "Nikon", "Z 7", DaylightFluorescent            , 0, { 1.828125, 1, 1.11328125, 0 } },
  { "Nikon", "Z 7", WhiteFluorescent               , 0, { 1.3046875, 1, 2.328125, 0 } },
  { "Nikon", "Z 7", HighTempMercuryVaporFluorescent, 0, { 2.1484375, 1, 1.349609375, 0 } },
  { "Nikon", "Z 7", SodiumVaporFluorescent         , 0, { 1.0390625, 1, 2.43359375, 0 } },
  { "Nikon", "Z 7", Flash                          , 0, { 1.904296875, 1, 1.185546875, 0 } },

  { "Nikon", "Z 7_2", DirectSunlight                 , -12, { 1.564453125, 1, 1.51953125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -11, { 1.578125, 1, 1.509765625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -10, { 1.591796875, 1, 1.498046875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -9, { 1.60546875, 1, 1.486328125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -8, { 1.619140625, 1, 1.47265625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -7, { 1.630859375, 1, 1.458984375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -6, { 1.64453125, 1, 1.4453125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -5, { 1.65625, 1, 1.431640625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -4, { 1.66796875, 1, 1.41796875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -3, { 1.6796875, 1, 1.404296875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -2, { 1.69140625, 1, 1.388671875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , -1, { 1.703125, 1, 1.375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 0, { 1.71484375, 1, 1.359375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 1, { 1.7265625, 1, 1.345703125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 2, { 1.73828125, 1, 1.33203125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 3, { 1.75, 1, 1.31640625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 4, { 1.76171875, 1, 1.302734375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 5, { 1.775390625, 1, 1.291015625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 6, { 1.787109375, 1, 1.27734375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 7, { 1.798828125, 1, 1.263671875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 8, { 1.8125, 1, 1.251953125, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 9, { 1.82421875, 1, 1.240234375, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 10, { 1.837890625, 1, 1.228515625, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 11, { 1.8515625, 1, 1.21875, 0 } },
  { "Nikon", "Z 7_2", DirectSunlight                 , 12, { 1.865234375, 1, 1.20703125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -12, { 1.90625, 1, 1.181640625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -11, { 1.921875, 1, 1.171875, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -10, { 1.9375, 1, 1.1640625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -9, { 1.953125, 1, 1.15625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -8, { 1.970703125, 1, 1.1484375, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -7, { 1.986328125, 1, 1.140625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -6, { 2.00390625, 1, 1.134765625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -5, { 2.021484375, 1, 1.12890625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -4, { 2.041015625, 1, 1.12109375, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -3, { 2.05859375, 1, 1.115234375, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -2, { 2.078125, 1, 1.109375, 0 } },
  { "Nikon", "Z 7_2", Shade                          , -1, { 2.09765625, 1, 1.103515625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 0, { 2.1171875, 1, 1.09765625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 1, { 2.138671875, 1, 1.091796875, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 2, { 2.158203125, 1, 1.0859375, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 3, { 2.1796875, 1, 1.078125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 4, { 2.203125, 1, 1.072265625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 5, { 2.224609375, 1, 1.064453125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 6, { 2.248046875, 1, 1.056640625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 7, { 2.271484375, 1, 1.048828125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 8, { 2.294921875, 1, 1.0390625, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 9, { 2.318359375, 1, 1.029296875, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 10, { 2.34375, 1, 1.01953125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 11, { 2.3671875, 1, 1.0078125, 0 } },
  { "Nikon", "Z 7_2", Shade                          , 12, { 2.392578125, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -12, { 1.69140625, 1, 1.388671875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -11, { 1.703125, 1, 1.375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -10, { 1.71484375, 1, 1.359375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -9, { 1.7265625, 1, 1.345703125, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -8, { 1.73828125, 1, 1.33203125, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -7, { 1.75, 1, 1.31640625, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -6, { 1.76171875, 1, 1.302734375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -5, { 1.775390625, 1, 1.291015625, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -4, { 1.787109375, 1, 1.27734375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -3, { 1.798828125, 1, 1.263671875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -2, { 1.8125, 1, 1.251953125, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , -1, { 1.82421875, 1, 1.240234375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 0, { 1.837890625, 1, 1.228515625, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 1, { 1.8515625, 1, 1.21875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 2, { 1.865234375, 1, 1.20703125, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 3, { 1.880859375, 1, 1.197265625, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 4, { 1.89453125, 1, 1.1875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 5, { 1.91015625, 1, 1.1796875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 6, { 1.92578125, 1, 1.169921875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 7, { 1.94140625, 1, 1.162109375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 8, { 1.95703125, 1, 1.154296875, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 9, { 1.97265625, 1, 1.146484375, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 10, { 1.990234375, 1, 1.140625, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 11, { 2.0078125, 1, 1.1328125, 0 } },
  { "Nikon", "Z 7_2", Cloudy                         , 12, { 2.025390625, 1, 1.126953125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -12, { 1.0390625, 1, 2.42578125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -11, { 1.044921875, 1, 2.404296875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -10, { 1.05078125, 1, 2.384765625, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -9, { 1.056640625, 1, 2.36328125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -8, { 1.0625, 1, 2.34375, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -7, { 1.068359375, 1, 2.32421875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -6, { 1.076171875, 1, 2.3046875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -5, { 1.08203125, 1, 2.28515625, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -4, { 1.087890625, 1, 2.265625, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -3, { 1.09375, 1, 2.248046875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -2, { 1.1015625, 1, 2.228515625, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , -1, { 1.107421875, 1, 2.205078125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 0, { 1.115234375, 1, 2.18359375, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 1, { 1.12109375, 1, 2.162109375, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 2, { 1.12890625, 1, 2.142578125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 3, { 1.134765625, 1, 2.125, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 4, { 1.142578125, 1, 2.107421875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 5, { 1.150390625, 1, 2.091796875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 6, { 1.158203125, 1, 2.07421875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 7, { 1.166015625, 1, 2.05859375, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 8, { 1.171875, 1, 2.04296875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 9, { 1.1796875, 1, 2.02734375, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 10, { 1.189453125, 1, 2.013671875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 11, { 1.197265625, 1, 1.998046875, 0 } },
  { "Nikon", "Z 7_2", Incandescent                   , 12, { 1.205078125, 1, 1.982421875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -12, { 1.015625, 1, 2.095703125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -11, { 1.021484375, 1, 2.0859375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -10, { 1.029296875, 1, 2.076171875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -9, { 1.03515625, 1, 2.068359375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -8, { 1.04296875, 1, 2.05859375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -7, { 1.048828125, 1, 2.048828125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -6, { 1.056640625, 1, 2.0390625, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -5, { 1.064453125, 1, 2.029296875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -4, { 1.0703125, 1, 2.01953125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -3, { 1.078125, 1, 2.01171875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -2, { 1.0859375, 1, 2.001953125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , -1, { 1.09375, 1, 1.9921875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 0, { 1.1015625, 1, 1.982421875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 1, { 1.107421875, 1, 1.97265625, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 2, { 1.115234375, 1, 1.962890625, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 3, { 1.123046875, 1, 1.953125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 4, { 1.130859375, 1, 1.943359375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 5, { 1.138671875, 1, 1.93359375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 6, { 1.146484375, 1, 1.923828125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 7, { 1.154296875, 1, 1.916015625, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 8, { 1.1640625, 1, 1.90625, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 9, { 1.171875, 1, 1.896484375, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 10, { 1.1796875, 1, 1.88671875, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 11, { 1.1875, 1, 1.876953125, 0 } },
  { "Nikon", "Z 7_2", WarmWhiteFluorescent           , 12, { 1.1953125, 1, 1.8671875, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -12, { 1.326171875, 1, 2.306640625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -11, { 1.3515625, 1, 2.28125, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -10, { 1.376953125, 1, 2.2578125, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -9, { 1.400390625, 1, 2.232421875, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -8, { 1.42578125, 1, 2.208984375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -7, { 1.44921875, 1, 2.18359375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -6, { 1.47265625, 1, 2.16015625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -5, { 1.494140625, 1, 2.134765625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -4, { 1.515625, 1, 2.111328125, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -3, { 1.537109375, 1, 2.087890625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -2, { 1.55859375, 1, 2.06640625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , -1, { 1.580078125, 1, 2.044921875, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 0, { 1.599609375, 1, 2.0234375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 1, { 1.62109375, 1, 2.001953125, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 2, { 1.640625, 1, 1.982421875, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 3, { 1.662109375, 1, 1.962890625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 4, { 1.681640625, 1, 1.943359375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 5, { 1.69921875, 1, 1.92578125, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 6, { 1.71875, 1, 1.90625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 7, { 1.736328125, 1, 1.888671875, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 8, { 1.755859375, 1, 1.87109375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 9, { 1.7734375, 1, 1.853515625, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 10, { 1.7890625, 1, 1.8359375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 11, { 1.806640625, 1, 1.818359375, 0 } },
  { "Nikon", "Z 7_2", CoolWhiteFluorescent           , 12, { 1.82421875, 1, 1.798828125, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -12, { 1.470703125, 1, 1.5390625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -11, { 1.48046875, 1, 1.525390625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -10, { 1.490234375, 1, 1.51171875, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -9, { 1.501953125, 1, 1.5, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -8, { 1.51171875, 1, 1.486328125, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -7, { 1.5234375, 1, 1.47265625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -6, { 1.533203125, 1, 1.458984375, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -5, { 1.544921875, 1, 1.447265625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -4, { 1.5546875, 1, 1.43359375, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -3, { 1.56640625, 1, 1.419921875, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -2, { 1.578125, 1, 1.40625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , -1, { 1.58984375, 1, 1.392578125, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 0, { 1.6015625, 1, 1.37890625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 1, { 1.61328125, 1, 1.365234375, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 2, { 1.623046875, 1, 1.3515625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 3, { 1.634765625, 1, 1.337890625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 4, { 1.6484375, 1, 1.32421875, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 5, { 1.66015625, 1, 1.310546875, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 6, { 1.671875, 1, 1.296875, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 7, { 1.68359375, 1, 1.283203125, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 8, { 1.6953125, 1, 1.267578125, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 9, { 1.708984375, 1, 1.25390625, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 10, { 1.720703125, 1, 1.240234375, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 11, { 1.732421875, 1, 1.224609375, 0 } },
  { "Nikon", "Z 7_2", DayWhiteFluorescent            , 12, { 1.74609375, 1, 1.2109375, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -12, { 1.67578125, 1, 1.291015625, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -11, { 1.689453125, 1, 1.27734375, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -10, { 1.701171875, 1, 1.26171875, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -9, { 1.712890625, 1, 1.248046875, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -8, { 1.7265625, 1, 1.234375, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -7, { 1.73828125, 1, 1.220703125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -6, { 1.75, 1, 1.205078125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -5, { 1.763671875, 1, 1.189453125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -4, { 1.775390625, 1, 1.17578125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -3, { 1.7890625, 1, 1.16015625, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -2, { 1.802734375, 1, 1.14453125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , -1, { 1.814453125, 1, 1.12890625, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 0, { 1.828125, 1, 1.11328125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 1, { 1.841796875, 1, 1.09765625, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 2, { 1.853515625, 1, 1.080078125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 3, { 1.8671875, 1, 1.064453125, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 4, { 1.880859375, 1, 1.046875, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 5, { 1.89453125, 1, 1.029296875, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 6, { 1.908203125, 1, 1.01171875, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 7, { 1.919921875, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 8, { 1.93359375, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 9, { 1.947265625, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 10, { 1.9609375, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 11, { 1.974609375, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", DaylightFluorescent            , 12, { 1.990234375, 1, 1.0, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -12, { 0.943359375, 1, 2.65625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -11, { 0.974609375, 1, 2.626953125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -10, { 1.0078125, 1, 2.599609375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -9, { 1.0390625, 1, 2.5703125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -8, { 1.0703125, 1, 2.54296875, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -7, { 1.1015625, 1, 2.513671875, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -6, { 1.1328125, 1, 2.486328125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -5, { 1.162109375, 1, 2.458984375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -4, { 1.193359375, 1, 2.43359375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -3, { 1.220703125, 1, 2.40625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -2, { 1.25, 1, 2.37890625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , -1, { 1.27734375, 1, 2.353515625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 0, { 1.3046875, 1, 2.328125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 1, { 1.330078125, 1, 2.302734375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 2, { 1.35546875, 1, 2.27734375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 3, { 1.380859375, 1, 2.251953125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 4, { 1.40625, 1, 2.228515625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 5, { 1.4296875, 1, 2.203125, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 6, { 1.453125, 1, 2.177734375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 7, { 1.4765625, 1, 2.154296875, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 8, { 1.498046875, 1, 2.130859375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 9, { 1.521484375, 1, 2.107421875, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 10, { 1.54296875, 1, 2.083984375, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 11, { 1.5625, 1, 2.0625, 0 } },
  { "Nikon", "Z 7_2", WhiteFluorescent               , 12, { 1.583984375, 1, 2.0390625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -12, { 2.03515625, 1, 1.5234375, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -11, { 2.044921875, 1, 1.5078125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -10, { 2.056640625, 1, 1.4921875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -9, { 2.06640625, 1, 1.478515625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -8, { 2.076171875, 1, 1.462890625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -7, { 2.0859375, 1, 1.447265625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -6, { 2.095703125, 1, 1.43359375, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -5, { 2.10546875, 1, 1.41796875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -4, { 2.115234375, 1, 1.404296875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -3, { 2.123046875, 1, 1.390625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -2, { 2.130859375, 1, 1.376953125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, -1, { 2.140625, 1, 1.36328125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 0, { 2.1484375, 1, 1.349609375, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 1, { 2.15625, 1, 1.337890625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 2, { 2.1640625, 1, 1.32421875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 3, { 2.171875, 1, 1.3125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 4, { 2.177734375, 1, 1.298828125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 5, { 2.185546875, 1, 1.287109375, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 6, { 2.193359375, 1, 1.275390625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 7, { 2.19921875, 1, 1.263671875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 8, { 2.20703125, 1, 1.251953125, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 9, { 2.212890625, 1, 1.240234375, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 10, { 2.21875, 1, 1.228515625, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 11, { 2.224609375, 1, 1.21875, 0 } },
  { "Nikon", "Z 7_2", HighTempMercuryVaporFluorescent, 12, { 2.23046875, 1, 1.208984375, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -12, { 0.978515625, 1, 2.72265625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -11, { 0.982421875, 1, 2.693359375, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -10, { 0.98828125, 1, 2.666015625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -9, { 0.9921875, 1, 2.640625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -8, { 0.99609375, 1, 2.61328125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -7, { 1.001953125, 1, 2.58984375, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -6, { 1.005859375, 1, 2.56640625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -5, { 1.01171875, 1, 2.54296875, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -4, { 1.015625, 1, 2.51953125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -3, { 1.021484375, 1, 2.498046875, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -2, { 1.02734375, 1, 2.4765625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , -1, { 1.033203125, 1, 2.455078125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 0, { 1.0390625, 1, 2.43359375, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 1, { 1.044921875, 1, 2.412109375, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 2, { 1.052734375, 1, 2.392578125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 3, { 1.060546875, 1, 2.373046875, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 4, { 1.068359375, 1, 2.3515625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 5, { 1.076171875, 1, 2.33203125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 6, { 1.083984375, 1, 2.310546875, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 7, { 1.09375, 1, 2.291015625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 8, { 1.1015625, 1, 2.26953125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 9, { 1.11328125, 1, 2.248046875, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 10, { 1.123046875, 1, 2.2265625, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 11, { 1.1328125, 1, 2.205078125, 0 } },
  { "Nikon", "Z 7_2", SodiumVaporFluorescent         , 12, { 1.14453125, 1, 2.18359375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -12, { 1.748046875, 1, 1.283203125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -11, { 1.763671875, 1, 1.271484375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -10, { 1.779296875, 1, 1.263671875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -9, { 1.79296875, 1, 1.25390625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -8, { 1.806640625, 1, 1.244140625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -7, { 1.8203125, 1, 1.236328125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -6, { 1.833984375, 1, 1.228515625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -5, { 1.845703125, 1, 1.220703125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -4, { 1.857421875, 1, 1.212890625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -3, { 1.869140625, 1, 1.205078125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -2, { 1.880859375, 1, 1.19921875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , -1, { 1.892578125, 1, 1.19140625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 0, { 1.904296875, 1, 1.185546875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 1, { 1.91796875, 1, 1.177734375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 2, { 1.9296875, 1, 1.171875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 3, { 1.943359375, 1, 1.166015625, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 4, { 1.95703125, 1, 1.158203125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 5, { 1.97265625, 1, 1.15234375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 6, { 1.98828125, 1, 1.14453125, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 7, { 2.00390625, 1, 1.138671875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 8, { 2.021484375, 1, 1.130859375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 9, { 2.041015625, 1, 1.123046875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 10, { 2.060546875, 1, 1.115234375, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 11, { 2.08203125, 1, 1.107421875, 0 } },
  { "Nikon", "Z 7_2", Flash                          , 12, { 2.10546875, 1, 1.099609375, 0 } },

  /* Nikon Z 50 / Firmware  C 2.00 / LF 1.00 */
  { "Nikon", "Z 50", DirectSunlight                 , -12, { 1.71875, 1, 1.591796875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -11, { 1.734375, 1, 1.58203125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -10, { 1.748046875, 1, 1.5703125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -9, { 1.763671875, 1, 1.55859375, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -8, { 1.77734375, 1, 1.544921875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -7, { 1.791015625, 1, 1.533203125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -6, { 1.8046875, 1, 1.51953125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -5, { 1.818359375, 1, 1.505859375, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -4, { 1.83203125, 1, 1.4921875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -3, { 1.845703125, 1, 1.478515625, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -2, { 1.857421875, 1, 1.46484375, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , -1, { 1.87109375, 1, 1.451171875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 0, { 1.8828125, 1, 1.435546875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 1, { 1.896484375, 1, 1.421875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 2, { 1.908203125, 1, 1.408203125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 3, { 1.921875, 1, 1.39453125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 4, { 1.935546875, 1, 1.380859375, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 5, { 1.947265625, 1, 1.3671875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 6, { 1.9609375, 1, 1.353515625, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 7, { 1.974609375, 1, 1.341796875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 8, { 1.98828125, 1, 1.328125, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 9, { 2.001953125, 1, 1.31640625, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 10, { 2.015625, 1, 1.3046875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 11, { 2.029296875, 1, 1.29296875, 0 } },
  { "Nikon", "Z 50", DirectSunlight                 , 12, { 2.044921875, 1, 1.28125, 0 } },
  { "Nikon", "Z 50", Shade                          , -12, { 2.087890625, 1, 1.25390625, 0 } },
  { "Nikon", "Z 50", Shade                          , -11, { 2.103515625, 1, 1.24609375, 0 } },
  { "Nikon", "Z 50", Shade                          , -10, { 2.12109375, 1, 1.23828125, 0 } },
  { "Nikon", "Z 50", Shade                          , -9, { 2.138671875, 1, 1.23046875, 0 } },
  { "Nikon", "Z 50", Shade                          , -8, { 2.15625, 1, 1.22265625, 0 } },
  { "Nikon", "Z 50", Shade                          , -7, { 2.173828125, 1, 1.216796875, 0 } },
  { "Nikon", "Z 50", Shade                          , -6, { 2.193359375, 1, 1.2109375, 0 } },
  { "Nikon", "Z 50", Shade                          , -5, { 2.212890625, 1, 1.20703125, 0 } },
  { "Nikon", "Z 50", Shade                          , -4, { 2.232421875, 1, 1.203125, 0 } },
  { "Nikon", "Z 50", Shade                          , -3, { 2.25390625, 1, 1.1953125, 0 } },
  { "Nikon", "Z 50", Shade                          , -2, { 2.2734375, 1, 1.185546875, 0 } },
  { "Nikon", "Z 50", Shade                          , -1, { 2.296875, 1, 1.177734375, 0 } },
  { "Nikon", "Z 50", Shade                          , 0, { 2.318359375, 1, 1.171875, 0 } },
  { "Nikon", "Z 50", Shade                          , 1, { 2.341796875, 1, 1.16796875, 0 } },
  { "Nikon", "Z 50", Shade                          , 2, { 2.369140625, 1, 1.162109375, 0 } },
  { "Nikon", "Z 50", Shade                          , 3, { 2.39453125, 1, 1.158203125, 0 } },
  { "Nikon", "Z 50", Shade                          , 4, { 2.419921875, 1, 1.15234375, 0 } },
  { "Nikon", "Z 50", Shade                          , 5, { 2.447265625, 1, 1.14453125, 0 } },
  { "Nikon", "Z 50", Shade                          , 6, { 2.47265625, 1, 1.13671875, 0 } },
  { "Nikon", "Z 50", Shade                          , 7, { 2.5, 1, 1.126953125, 0 } },
  { "Nikon", "Z 50", Shade                          , 8, { 2.52734375, 1, 1.1171875, 0 } },
  { "Nikon", "Z 50", Shade                          , 9, { 2.552734375, 1, 1.10546875, 0 } },
  { "Nikon", "Z 50", Shade                          , 10, { 2.580078125, 1, 1.091796875, 0 } },
  { "Nikon", "Z 50", Shade                          , 11, { 2.607421875, 1, 1.076171875, 0 } },
  { "Nikon", "Z 50", Shade                          , 12, { 2.634765625, 1, 1.05859375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -12, { 1.857421875, 1, 1.46484375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -11, { 1.87109375, 1, 1.451171875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -10, { 1.8828125, 1, 1.435546875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -9, { 1.896484375, 1, 1.421875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -8, { 1.908203125, 1, 1.408203125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -7, { 1.921875, 1, 1.39453125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -6, { 1.935546875, 1, 1.380859375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -5, { 1.947265625, 1, 1.3671875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -4, { 1.9609375, 1, 1.353515625, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -3, { 1.974609375, 1, 1.341796875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -2, { 1.98828125, 1, 1.328125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , -1, { 2.001953125, 1, 1.31640625, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 0, { 2.015625, 1, 1.3046875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 1, { 2.029296875, 1, 1.29296875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 2, { 2.044921875, 1, 1.28125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 3, { 2.060546875, 1, 1.271484375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 4, { 2.076171875, 1, 1.26171875, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 5, { 2.091796875, 1, 1.251953125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 6, { 2.107421875, 1, 1.244140625, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 7, { 2.125, 1, 1.236328125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 8, { 2.142578125, 1, 1.228515625, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 9, { 2.16015625, 1, 1.220703125, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 10, { 2.177734375, 1, 1.21484375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 11, { 2.197265625, 1, 1.2109375, 0 } },
  { "Nikon", "Z 50", Cloudy                         , 12, { 2.216796875, 1, 1.205078125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -12, { 1.146484375, 1, 2.498046875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -11, { 1.15234375, 1, 2.478515625, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -10, { 1.158203125, 1, 2.45703125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -9, { 1.166015625, 1, 2.4375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -8, { 1.171875, 1, 2.416015625, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -7, { 1.177734375, 1, 2.396484375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -6, { 1.18359375, 1, 2.376953125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -5, { 1.19140625, 1, 2.35546875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -4, { 1.197265625, 1, 2.3359375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -3, { 1.205078125, 1, 2.31640625, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -2, { 1.2109375, 1, 2.296875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , -1, { 1.21875, 1, 2.27734375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 0, { 1.2265625, 1, 2.2578125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 1, { 1.234375, 1, 2.23828125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 2, { 1.240234375, 1, 2.21875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 3, { 1.248046875, 1, 2.19921875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 4, { 1.255859375, 1, 2.1796875, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 5, { 1.265625, 1, 2.162109375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 6, { 1.2734375, 1, 2.142578125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 7, { 1.28125, 1, 2.126953125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 8, { 1.291015625, 1, 2.12109375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 9, { 1.298828125, 1, 2.111328125, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 10, { 1.30859375, 1, 2.1015625, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 11, { 1.318359375, 1, 2.08984375, 0 } },
  { "Nikon", "Z 50", Incandescent                   , 12, { 1.326171875, 1, 2.076171875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -12, { 1.12109375, 1, 2.1328125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -11, { 1.12890625, 1, 2.123046875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -10, { 1.13671875, 1, 2.11328125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -9, { 1.14453125, 1, 2.103515625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -8, { 1.15234375, 1, 2.095703125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -7, { 1.16015625, 1, 2.0859375, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -6, { 1.166015625, 1, 2.076171875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -5, { 1.173828125, 1, 2.06640625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -4, { 1.181640625, 1, 2.05859375, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -3, { 1.189453125, 1, 2.048828125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -2, { 1.197265625, 1, 2.0390625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , -1, { 1.20703125, 1, 2.029296875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 0, { 1.21484375, 1, 2.021484375, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 1, { 1.22265625, 1, 2.01171875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 2, { 1.23046875, 1, 2.001953125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 3, { 1.23828125, 1, 1.994140625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 4, { 1.24609375, 1, 1.984375, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 5, { 1.255859375, 1, 1.9765625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 6, { 1.263671875, 1, 1.966796875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 7, { 1.271484375, 1, 1.95703125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 8, { 1.28125, 1, 1.94921875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 9, { 1.2890625, 1, 1.939453125, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 10, { 1.298828125, 1, 1.931640625, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 11, { 1.306640625, 1, 1.921875, 0 } },
  { "Nikon", "Z 50", WarmWhiteFluorescent           , 12, { 1.31640625, 1, 1.912109375, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -12, { 1.455078125, 1, 2.404296875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -11, { 1.484375, 1, 2.37890625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -10, { 1.509765625, 1, 2.353515625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -9, { 1.537109375, 1, 2.328125, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -8, { 1.560546875, 1, 2.302734375, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -7, { 1.5859375, 1, 2.279296875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -6, { 1.609375, 1, 2.25390625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -5, { 1.6328125, 1, 2.23046875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -4, { 1.654296875, 1, 2.20703125, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -3, { 1.677734375, 1, 2.185546875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -2, { 1.69921875, 1, 2.162109375, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , -1, { 1.720703125, 1, 2.140625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 0, { 1.7421875, 1, 2.1171875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 1, { 1.765625, 1, 2.095703125, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 2, { 1.787109375, 1, 2.076171875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 3, { 1.810546875, 1, 2.0546875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 4, { 1.83203125, 1, 2.03515625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 5, { 1.853515625, 1, 2.013671875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 6, { 1.875, 1, 1.994140625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 7, { 1.896484375, 1, 1.9765625, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 8, { 1.91796875, 1, 1.95703125, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 9, { 1.9375, 1, 1.9375, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 10, { 1.95703125, 1, 1.91796875, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 11, { 1.9765625, 1, 1.8984375, 0 } },
  { "Nikon", "Z 50", CoolWhiteFluorescent           , 12, { 1.99609375, 1, 1.87890625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -12, { 1.611328125, 1, 1.599609375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -11, { 1.623046875, 1, 1.587890625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -10, { 1.634765625, 1, 1.57421875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -9, { 1.646484375, 1, 1.5625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -8, { 1.658203125, 1, 1.548828125, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -7, { 1.669921875, 1, 1.53515625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -6, { 1.681640625, 1, 1.5234375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -5, { 1.693359375, 1, 1.509765625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -4, { 1.70703125, 1, 1.49609375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -3, { 1.71875, 1, 1.482421875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -2, { 1.73046875, 1, 1.46875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , -1, { 1.7421875, 1, 1.455078125, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 0, { 1.755859375, 1, 1.443359375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 1, { 1.767578125, 1, 1.4296875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 2, { 1.78125, 1, 1.416015625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 3, { 1.79296875, 1, 1.40234375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 4, { 1.806640625, 1, 1.38671875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 5, { 1.818359375, 1, 1.373046875, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 6, { 1.83203125, 1, 1.359375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 7, { 1.845703125, 1, 1.345703125, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 8, { 1.857421875, 1, 1.33203125, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 9, { 1.87109375, 1, 1.31640625, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 10, { 1.884765625, 1, 1.302734375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 11, { 1.8984375, 1, 1.287109375, 0 } },
  { "Nikon", "Z 50", DayWhiteFluorescent            , 12, { 1.912109375, 1, 1.2734375, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -12, { 1.837890625, 1, 1.353515625, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -11, { 1.849609375, 1, 1.33984375, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -10, { 1.86328125, 1, 1.326171875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -9, { 1.876953125, 1, 1.310546875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -8, { 1.890625, 1, 1.296875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -7, { 1.904296875, 1, 1.28125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -6, { 1.91796875, 1, 1.267578125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -5, { 1.931640625, 1, 1.251953125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -4, { 1.9453125, 1, 1.236328125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -3, { 1.958984375, 1, 1.220703125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -2, { 1.97265625, 1, 1.205078125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , -1, { 1.986328125, 1, 1.189453125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 0, { 2.0, 1, 1.171875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 1, { 2.015625, 1, 1.15625, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 2, { 2.029296875, 1, 1.138671875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 3, { 2.04296875, 1, 1.12109375, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 4, { 2.05859375, 1, 1.103515625, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 5, { 2.072265625, 1, 1.0859375, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 6, { 2.087890625, 1, 1.06640625, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 7, { 2.1015625, 1, 1.046875, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 8, { 2.1171875, 1, 1.02734375, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 9, { 2.1328125, 1, 1.0078125, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 10, { 2.146484375, 1, 1.0, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 11, { 2.162109375, 1, 1.0, 0 } },
  { "Nikon", "Z 50", DaylightFluorescent            , 12, { 2.177734375, 1, 1.0, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -12, { 1.037109375, 1, 2.763671875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -11, { 1.072265625, 1, 2.734375, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -10, { 1.107421875, 1, 2.703125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -9, { 1.142578125, 1, 2.673828125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -8, { 1.177734375, 1, 2.646484375, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -7, { 1.212890625, 1, 2.6171875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -6, { 1.24609375, 1, 2.587890625, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -5, { 1.279296875, 1, 2.560546875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -4, { 1.310546875, 1, 2.533203125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -3, { 1.34375, 1, 2.505859375, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -2, { 1.373046875, 1, 2.478515625, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , -1, { 1.404296875, 1, 2.451171875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 0, { 1.43359375, 1, 2.42578125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 1, { 1.4609375, 1, 2.3984375, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 2, { 1.48828125, 1, 2.373046875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 3, { 1.515625, 1, 2.34765625, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 4, { 1.541015625, 1, 2.322265625, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 5, { 1.56640625, 1, 2.298828125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 6, { 1.58984375, 1, 2.2734375, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 7, { 1.61328125, 1, 2.25, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 8, { 1.63671875, 1, 2.2265625, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 9, { 1.66015625, 1, 2.203125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 10, { 1.681640625, 1, 2.1796875, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 11, { 1.703125, 1, 2.158203125, 0 } },
  { "Nikon", "Z 50", WhiteFluorescent               , 12, { 1.724609375, 1, 2.134765625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -12, { 2.2265625, 1, 1.6015625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -11, { 2.23828125, 1, 1.5859375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -10, { 2.248046875, 1, 1.568359375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -9, { 2.259765625, 1, 1.552734375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -8, { 2.271484375, 1, 1.537109375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -7, { 2.28125, 1, 1.521484375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -6, { 2.291015625, 1, 1.505859375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -5, { 2.30078125, 1, 1.4921875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -4, { 2.310546875, 1, 1.4765625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -3, { 2.318359375, 1, 1.462890625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -2, { 2.328125, 1, 1.447265625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, -1, { 2.3359375, 1, 1.43359375, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 0, { 2.345703125, 1, 1.419921875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 1, { 2.353515625, 1, 1.40625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 2, { 2.36328125, 1, 1.392578125, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 3, { 2.37109375, 1, 1.37890625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 4, { 2.37890625, 1, 1.3671875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 5, { 2.38671875, 1, 1.353515625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 6, { 2.39453125, 1, 1.341796875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 7, { 2.40234375, 1, 1.328125, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 8, { 2.408203125, 1, 1.31640625, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 9, { 2.416015625, 1, 1.3046875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 10, { 2.421875, 1, 1.29296875, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 11, { 2.4296875, 1, 1.28125, 0 } },
  { "Nikon", "Z 50", HighTempMercuryVaporFluorescent, 12, { 2.435546875, 1, 1.271484375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -12, { 1.091796875, 1, 2.853515625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -11, { 1.09765625, 1, 2.826171875, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -10, { 1.1015625, 1, 2.798828125, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -9, { 1.107421875, 1, 2.7734375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -8, { 1.11328125, 1, 2.75, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -7, { 1.119140625, 1, 2.7265625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -6, { 1.123046875, 1, 2.703125, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -5, { 1.12890625, 1, 2.681640625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -4, { 1.134765625, 1, 2.662109375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -3, { 1.140625, 1, 2.640625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -2, { 1.1484375, 1, 2.62109375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , -1, { 1.154296875, 1, 2.599609375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 0, { 1.16015625, 1, 2.580078125, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 1, { 1.16796875, 1, 2.55859375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 2, { 1.17578125, 1, 2.537109375, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 3, { 1.18359375, 1, 2.515625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 4, { 1.19140625, 1, 2.494140625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 5, { 1.19921875, 1, 2.47265625, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 6, { 1.208984375, 1, 2.451171875, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 7, { 1.216796875, 1, 2.4296875, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 8, { 1.2265625, 1, 2.408203125, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 9, { 1.23828125, 1, 2.38671875, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 10, { 1.248046875, 1, 2.36328125, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 11, { 1.259765625, 1, 2.341796875, 0 } },
  { "Nikon", "Z 50", SodiumVaporFluorescent         , 12, { 1.271484375, 1, 2.318359375, 0 } },
  { "Nikon", "Z 50", Flash                          , -12, { 1.94140625, 1, 1.349609375, 0 } },
  { "Nikon", "Z 50", Flash                          , -11, { 1.95703125, 1, 1.33984375, 0 } },
  { "Nikon", "Z 50", Flash                          , -10, { 1.974609375, 1, 1.330078125, 0 } },
  { "Nikon", "Z 50", Flash                          , -9, { 1.990234375, 1, 1.322265625, 0 } },
  { "Nikon", "Z 50", Flash                          , -8, { 2.00390625, 1, 1.3125, 0 } },
  { "Nikon", "Z 50", Flash                          , -7, { 2.01953125, 1, 1.3046875, 0 } },
  { "Nikon", "Z 50", Flash                          , -6, { 2.03125, 1, 1.296875, 0 } },
  { "Nikon", "Z 50", Flash                          , -5, { 2.044921875, 1, 1.291015625, 0 } },
  { "Nikon", "Z 50", Flash                          , -4, { 2.05859375, 1, 1.283203125, 0 } },
  { "Nikon", "Z 50", Flash                          , -3, { 2.072265625, 1, 1.27734375, 0 } },
  { "Nikon", "Z 50", Flash                          , -2, { 2.083984375, 1, 1.26953125, 0 } },
  { "Nikon", "Z 50", Flash                          , -1, { 2.09765625, 1, 1.263671875, 0 } },
  { "Nikon", "Z 50", Flash                          , 0, { 2.111328125, 1, 1.255859375, 0 } },
  { "Nikon", "Z 50", Flash                          , 1, { 2.125, 1, 1.25, 0 } },
  { "Nikon", "Z 50", Flash                          , 2, { 2.138671875, 1, 1.2421875, 0 } },
  { "Nikon", "Z 50", Flash                          , 3, { 2.154296875, 1, 1.236328125, 0 } },
  { "Nikon", "Z 50", Flash                          , 4, { 2.169921875, 1, 1.228515625, 0 } },
  { "Nikon", "Z 50", Flash                          , 5, { 2.185546875, 1, 1.220703125, 0 } },
  { "Nikon", "Z 50", Flash                          , 6, { 2.203125, 1, 1.212890625, 0 } },
  { "Nikon", "Z 50", Flash                          , 7, { 2.22265625, 1, 1.205078125, 0 } },
  { "Nikon", "Z 50", Flash                          , 8, { 2.2421875, 1, 1.197265625, 0 } },
  { "Nikon", "Z 50", Flash                          , 9, { 2.263671875, 1, 1.189453125, 0 } },
  { "Nikon", "Z 50", Flash                          , 10, { 2.287109375, 1, 1.181640625, 0 } },
  { "Nikon", "Z 50", Flash                          , 11, { 2.3125, 1, 1.173828125, 0 } },
  { "Nikon", "Z 50", Flash                          , 12, { 2.337890625, 1, 1.166015625, 0 } },

  { "Olympus", "C5050Z", Shade, -7,		{ 3.887324, 1.201878, 1, 0 } },
  { "Olympus", "C5050Z", Shade, 0,		{ 1.757812, 1, 1.437500, 0 } },
  { "Olympus", "C5050Z", Shade, 7,		{ 1.019531, 1, 2.140625, 0 } },
  { "Olympus", "C5050Z", Cloudy, -7,		{ 3.255507, 1.127753, 1, 0 } },
  { "Olympus", "C5050Z", Cloudy, 0,		{ 1.570312, 1, 1.531250, 0 } },
  { "Olympus", "C5050Z", Cloudy, 7,		{ 1, 1.098712, 2.506438, 0 } },
  { "Olympus", "C5050Z", Daylight, -7,		{ 2.892116, 1.062241, 1, 0 } },
  { "Olympus", "C5050Z", Daylight, 0,		{ 1.480469, 1, 1.628906, 0 } },
  { "Olympus", "C5050Z", Daylight, 7,		{ 1, 1.168950, 2.835616, 0 } },
  { "Olympus", "C5050Z", EveningSun, -7,	{ 3.072649, 1.094017, 1, 0 } },
  { "Olympus", "C5050Z", EveningSun, 0,		{ 1.527344, 1, 1.578125, 0 } },
  { "Olympus", "C5050Z", EveningSun, 7,		{ 1, 1.132743, 2.659292, 0 } },
  { "Olympus", "C5050Z", DaylightFluorescent, -7, { 3.321267, 1.158371, 1, 0 } }, /*6700K*/
  { "Olympus", "C5050Z", DaylightFluorescent, 0, { 1.558594, 1, 1.492188, 0 } }, /*6700K*/
  { "Olympus", "C5050Z", DaylightFluorescent, 7, { 1, 1.108225, 2.463203, 0 } }, /*6700K*/
  { "Olympus", "C5050Z", NeutralFluorescent, -7, { 2.606426, 1.028112, 1, 0 } }, /*5000K*/
  { "Olympus", "C5050Z", NeutralFluorescent, 0,	{ 1.378906, 1, 1.679688, 0 } }, /*5000K*/
  { "Olympus", "C5050Z", NeutralFluorescent, 7,	{ 1, 1.254902, 3.137255, 0 } }, /*5000K*/
  { "Olympus", "C5050Z", CoolWhiteFluorescent, -7, { 2.519531, 1, 1.281250, 0 } }, /*4200K*/
  { "Olympus", "C5050Z", CoolWhiteFluorescent, 0, { 1.371094, 1, 2.210938, 0 } }, /*4200K*/
  { "Olympus", "C5050Z", CoolWhiteFluorescent, 7, { 1, 1.261084, 4.152709, 0 } }, /*4200K*/
  { "Olympus", "C5050Z", WhiteFluorescent, -7,	{ 1.707031, 1, 1.699219, 0 } }, /*3500K*/
  { "Olympus", "C5050Z", WhiteFluorescent, 0,	{ 1, 1.075630, 3.151261, 0 } }, /*3500K*/
  { "Olympus", "C5050Z", WhiteFluorescent, 7,	{ 1, 1.855072, 8.094203, 0 } }, /*3500K*/
  { "Olympus", "C5050Z", Incandescent, -7,	{ 1.679688, 1, 1.652344, 0 } }, /*3000K*/
  { "Olympus", "C5050Z", Incandescent, 0,	{ 1, 1.094017, 3.123932, 0 } }, /*3000K*/
  { "Olympus", "C5050Z", Incandescent, 7,	{ 1, 1.896296, 8.066667, 0 } }, /*3000K*/

  { "Olympus", "C5060WZ", Shade, 0,		{ 1.949219, 1, 1.195312, 0 } },
  { "Olympus", "C5060WZ", Cloudy, 0,		{ 1.621094, 1, 1.410156, 0 } },
  { "Olympus", "C5060WZ", DirectSunlight, 0,	{ 1.511719, 1, 1.500000, 0 } },
  { "Olympus", "C5060WZ", EveningSun, 0,	{ 1.636719, 1, 1.496094, 0 } },
  { "Olympus", "C5060WZ", DaylightFluorescent, 0, { 1.734375, 1, 1.343750, 0 } },
  { "Olympus", "C5060WZ", NeutralFluorescent, 0, { 1.457031, 1, 1.691406, 0 } },
  { "Olympus", "C5060WZ", CoolWhiteFluorescent, 0, { 1.417969, 1, 2.230469, 0 } },
  { "Olympus", "C5060WZ", WhiteFluorescent, 0,	{ 1, 1.103448, 3.422414, 0 } },
  { "Olympus", "C5060WZ", Incandescent, 0,	{ 1, 1.153153, 3.662162, 0 } },
  { "Olympus", "C5060WZ", FlashAuto, 0,		{ 1.850000, 1, 1.308044, 0 } },

  // Olympus C8080WZ - firmware 757-78
  { "Olympus", "C8080WZ", Shade, -7,		{ 1.515625, 1, 1.773438, 0 } },
  { "Olympus", "C8080WZ", Shade, -6,		{ 1.671875, 1, 1.691406, 0 } },
  { "Olympus", "C8080WZ", Shade, -5,		{ 1.832031, 1, 1.605469, 0 } },
  { "Olympus", "C8080WZ", Shade, -4,		{ 1.988281, 1, 1.523438, 0 } },
  { "Olympus", "C8080WZ", Shade, -3,		{ 2.144531, 1, 1.441406, 0 } },
  { "Olympus", "C8080WZ", Shade, -2,		{ 2.300781, 1, 1.355469, 0 } },
  { "Olympus", "C8080WZ", Shade, -1,		{ 2.457031, 1, 1.273438, 0 } },
  { "Olympus", "C8080WZ", Shade, 0,		{ 2.617188, 1, 1.191406, 0 } },
  { "Olympus", "C8080WZ", Shade, 1,		{ 2.929688, 1, 1.117188, 0 } },
  { "Olympus", "C8080WZ", Shade, 2,		{ 3.242188, 1, 1.046875, 0 } },
  { "Olympus", "C8080WZ", Shade, 3,		{ 3.644000, 1.024000, 1, 0 } },
  { "Olympus", "C8080WZ", Shade, 4,		{ 4.290043, 1.108225, 1, 0 } },
  { "Olympus", "C8080WZ", Shade, 5,		{ 5.032864, 1.201878, 1, 0 } },
  { "Olympus", "C8080WZ", Shade, 6,		{ 5.907692, 1.312821, 1, 0 } },
  { "Olympus", "C8080WZ", Shade, 7,		{ 7.000000, 1.454545, 1, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -7,		{ 1.277344, 1, 2.164062, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -6,		{ 1.406250, 1, 2.062500, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -5,		{ 1.539062, 1, 1.960938, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -4,		{ 1.671875, 1, 1.859375, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -3,		{ 1.804688, 1, 1.757812, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -2,		{ 1.937500, 1, 1.656250, 0 } },
  { "Olympus", "C8080WZ", Cloudy, -1,		{ 2.070312, 1, 1.554688, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 0,		{ 2.203125, 1, 1.453125, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 1,		{ 2.464844, 1, 1.363281, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 2,		{ 2.730469, 1, 1.277344, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 3,		{ 2.996094, 1, 1.191406, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 4,		{ 3.257812, 1, 1.101562, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 5,		{ 3.523438, 1, 1.015625, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 6,		{ 4.075630, 1.075630, 1, 0 } },
  { "Olympus", "C8080WZ", Cloudy, 7,		{ 4.823256, 1.190698, 1, 0 } },
  { "Olympus", "C8080WZ", Daylight, -7,		{ 1.234375, 1, 2.343750, 0 } },
  { "Olympus", "C8080WZ", Daylight, -6,		{ 1.359375, 1, 2.234375, 0 } },
  { "Olympus", "C8080WZ", Daylight, -5,		{ 1.488281, 1, 2.125000, 0 } },
  { "Olympus", "C8080WZ", Daylight, -4,		{ 1.617188, 1, 2.011719, 0 } },
  { "Olympus", "C8080WZ", Daylight, -3,		{ 1.742188, 1, 1.902344, 0 } },
  { "Olympus", "C8080WZ", Daylight, -2,		{ 1.871094, 1, 1.792969, 0 } },
  { "Olympus", "C8080WZ", Daylight, -1,		{ 2.000000, 1, 1.683594, 0 } },
  { "Olympus", "C8080WZ", Daylight, 0,		{ 2.128906, 1, 1.574219, 0 } },
  { "Olympus", "C8080WZ", Daylight, 1,		{ 2.382812, 1, 1.476562, 0 } },
  { "Olympus", "C8080WZ", Daylight, 2,		{ 2.636719, 1, 1.382812, 0 } },
  { "Olympus", "C8080WZ", Daylight, 3,		{ 2.894531, 1, 1.289062, 0 } },
  { "Olympus", "C8080WZ", Daylight, 4,		{ 3.148438, 1, 1.195312, 0 } },
  { "Olympus", "C8080WZ", Daylight, 5,		{ 3.406250, 1, 1.101562, 0 } },
  { "Olympus", "C8080WZ", Daylight, 6,		{ 3.660156, 1, 1.003906, 0 } },
  { "Olympus", "C8080WZ", Daylight, 7,		{ 4.300429, 1.098712, 1, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -7,	{ 1.308594, 1, 2.199219, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -6,	{ 1.445312, 1, 2.093750, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -5,	{ 1.582031, 1, 1.992188, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -4,	{ 1.718750, 1, 1.886719, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -3,	{ 1.851562, 1, 1.785156, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -2,	{ 1.988281, 1, 1.679688, 0 } },
  { "Olympus", "C8080WZ", EveningSun, -1,	{ 2.125000, 1, 1.578125, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 0,	{ 2.261719, 1, 1.476562, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 1,	{ 2.531250, 1, 1.386719, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 2,	{ 2.800781, 1, 1.296875, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 3,	{ 3.074219, 1, 1.207031, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 4,	{ 3.343750, 1, 1.121094, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 5,	{ 3.617188, 1, 1.031250, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 6,	{ 4.128631, 1.062241, 1, 0 } },
  { "Olympus", "C8080WZ", EveningSun, 7,	{ 4.863014, 1.168950, 1, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -7,	{ 1.488281, 1, 2.214844, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -6,	{ 1.652344, 1, 2.105469, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -5,	{ 1.812500, 1, 1.992188, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -4,	{ 1.976562, 1, 1.882812, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -3,	{ 2.117188, 1, 1.773438, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -2,	{ 2.253906, 1, 1.675781, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, -1,	{ 2.425781, 1, 1.585938, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 0,		{ 2.570312, 1, 1.468750, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 1,		{ 2.890625, 1, 1.386719, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 2,		{ 3.199219, 1, 1.308594, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 3,		{ 3.500000, 1, 1.214844, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 4,		{ 3.820312, 1, 1.125000, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 5,		{ 4.128906, 1, 1.039062, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 6,		{ 4.711934, 1.053498, 1, 0 } },
  { "Olympus", "C8080WZ", FlashAuto, 7,		{ 5.450450, 1.153153, 1, 0 } },
  { "Olympus", "C8080WZ", DaylightFluorescent, -7, { 1.425781, 1, 2.097656, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -6, { 1.574219, 1, 2.000000, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -5, { 1.722656, 1, 1.902344, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -4, { 1.867188, 1, 1.804688, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -3, { 2.015625, 1, 1.703125, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -2, { 2.164062, 1, 1.605469, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, -1, { 2.312500, 1, 1.507812, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 0, { 2.460938, 1, 1.410156, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 1, { 2.753906, 1, 1.324219, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 2, { 3.050781, 1, 1.238281, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 3, { 3.343750, 1, 1.156250, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 4, { 3.640625, 1, 1.070312, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 5, { 4.000000, 1.015873, 1, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 6, { 4.688312, 1.108225, 1, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", DaylightFluorescent, 7, { 5.545455, 1.224880, 1, 0 } }, /*6700K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -7, { 1.195312, 1, 2.589844, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -6, { 1.316406, 1, 2.464844, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -5, { 1.441406, 1, 2.343750, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -4, { 1.566406, 1, 2.222656, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -3, { 1.687500, 1, 2.101562, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -2, { 1.812500, 1, 1.980469, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, -1, { 1.937500, 1, 1.859375, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 0, { 2.062500, 1, 1.738281, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 1, { 2.308594, 1, 1.632812, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 2, { 2.554688, 1, 1.527344, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 3, { 2.804688, 1, 1.421875, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 4, { 3.050781, 1, 1.320312, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 5, { 3.296875, 1, 1.214844, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 6, { 3.546875, 1, 1.109375, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", NeutralFluorescent, 7, { 3.792969, 1, 1.007812, 0 } }, /*5000K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -7, { 1.109375, 1, 3.257812, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -6, { 1.226562, 1, 3.105469, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -5, { 1.339844, 1, 2.953125, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -4, { 1.457031, 1, 2.796875, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -3, { 1.570312, 1, 2.644531, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -2, { 1.687500, 1, 2.492188, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, -1, { 1.800781, 1, 2.339844, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 0, { 1.917969, 1, 2.187500, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 1, { 2.144531, 1, 2.054688, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 2, { 2.375000, 1, 1.921875, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 3, { 2.605469, 1, 1.792969, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 4, { 2.835938, 1, 1.660156, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 5, { 3.066406, 1, 1.531250, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 6, { 3.296875, 1, 1.398438, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", CoolWhiteFluorescent, 7, { 3.527344, 1, 1.265625, 0 } }, /*4200K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -7,	{ 1, 1.347368, 5.963158, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -6,	{ 1, 1.224880, 5.167464, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -5,	{ 1, 1.117904, 4.484716, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -4,	{ 1, 1.028112, 3.911647, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -3,	{ 1.046875, 1, 3.593750, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -2,	{ 1.125000, 1, 3.386719, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, -1,	{ 1.203125, 1, 3.179688, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 0,	{ 1.281250, 1, 2.972656, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 1,	{ 1.433594, 1, 2.792969, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 2,	{ 1.585938, 1, 2.613281, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 3,	{ 1.742188, 1, 2.437500, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 4,	{ 1.894531, 1, 2.257812, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 5,	{ 2.046875, 1, 2.078125, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 6,	{ 2.203125, 1, 1.902344, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", WhiteFluorescent, 7,	{ 2.355469, 1, 1.722656, 0 } }, /*3500K*/
  { "Olympus", "C8080WZ", Tungsten, -7,		{ 1, 1.488372, 6.988372, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -6,		{ 1, 1.347368, 6.026316, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -5,		{ 1, 1.230769, 5.235577, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -4,		{ 1, 1.132743, 4.566372, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -3,		{ 1, 1.049180, 4.000000, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -2,		{ 1.023438, 1, 3.589844, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, -1,		{ 1.093750, 1, 3.371094, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 0,		{ 1.164062, 1, 3.152344, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 1,		{ 1.300781, 1, 2.960938, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 2,		{ 1.441406, 1, 2.773438, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 3,		{ 1.582031, 1, 2.582031, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 4,		{ 1.722656, 1, 2.394531, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 5,		{ 1.722656, 1, 2.394531, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 6,		{ 2.000000, 1, 2.015625, 0 } }, /*3000K*/
  { "Olympus", "C8080WZ", Tungsten, 7,		{ 2.140625, 1, 1.828125, 0 } }, /*3000K*/
// Fin ajout

  { "Olympus", "E-1", Incandescent, -7,		{ 1.195312, 1, 1.562500, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -6,		{ 1.187500, 1, 1.578125, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -5,		{ 1.187500, 1, 1.585938, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -4,		{ 1.179688, 1, 1.601562, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -3,		{ 1.171875, 1, 1.609375, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -2,		{ 1.164062, 1, 1.617188, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, -1,		{ 1.156250, 1, 1.632812, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 0,		{ 1.156250, 1, 1.640625, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 1,		{ 1.140625, 1, 1.648438, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 2,		{ 1.132812, 1, 1.664062, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 3,		{ 1.125000, 1, 1.671875, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 4,		{ 1.117188, 1, 1.679688, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 5,		{ 1.117188, 1, 1.695312, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 6,		{ 1.109375, 1, 1.703125, 0 } }, /*3600K*/
  { "Olympus", "E-1", Incandescent, 7,		{ 1.101562, 1, 1.718750, 0 } }, /*3600K*/
  { "Olympus", "E-1", IncandescentWarm, -7,	{ 1.015625, 1, 1.867188, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -6,	{ 1.007812, 1, 1.875000, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -5,	{ 1.000000, 1, 1.890625, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -4,	{ 1, 1.007874, 1.913386, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -3,	{ 1, 1.015873, 1.944444, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -2,	{ 1, 1.015873, 1.952381, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, -1,	{ 1, 1.024000, 1.984000, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 0,	{ 1, 1.024000, 1.992000, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 1,	{ 1, 1.032258, 2.008065, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 2,	{ 1, 1.040650, 2.040650, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 3,	{ 1, 1.040650, 2.048780, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 4,	{ 1, 1.049180, 2.081967, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 5,	{ 1, 1.057851, 2.107438, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 6,	{ 1, 1.066667, 2.141667, 0 } }, /*3000K*/
  { "Olympus", "E-1", IncandescentWarm, 7,	{ 1, 1.075630, 2.168067, 0 } }, /*3000K*/
  { "Olympus", "E-1", WhiteFluorescent, -7,	{ 2.296875, 1, 1.445312, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -6,	{ 2.273438, 1, 1.468750, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -5,	{ 2.242188, 1, 1.492188, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -4,	{ 2.210938, 1, 1.523438, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -3,	{ 2.171875, 1, 1.562500, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -2,	{ 2.132812, 1, 1.601562, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, -1,	{ 2.093750, 1, 1.640625, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 0,	{ 2.062500, 1, 1.679688, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 1,	{ 2.039062, 1, 1.703125, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 2,	{ 2.015625, 1, 1.734375, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 3,	{ 2.000000, 1, 1.757812, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 4,	{ 1.984375, 1, 1.789062, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 5,	{ 1.968750, 1, 1.812500, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 6,	{ 1.945312, 1, 1.835938, 0 } }, /*4000K*/
  { "Olympus", "E-1", WhiteFluorescent, 7,	{ 1.929688, 1, 1.867188, 0 } }, /*4000K*/
  { "Olympus", "E-1", NeutralFluorescent, -7,	{ 1.984375, 1, 1.203125, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -6,	{ 1.960938, 1, 1.218750, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -5,	{ 1.937500, 1, 1.234375, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -4,	{ 1.921875, 1, 1.257812, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -3,	{ 1.898438, 1, 1.273438, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -2,	{ 1.875000, 1, 1.289062, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, -1,	{ 1.851562, 1, 1.304688, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 0,	{ 1.835938, 1, 1.320312, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 1,	{ 1.804688, 1, 1.343750, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 2,	{ 1.773438, 1, 1.367188, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 3,	{ 1.750000, 1, 1.390625, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 4,	{ 1.718750, 1, 1.414062, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 5,	{ 1.695312, 1, 1.437500, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 6,	{ 1.656250, 1, 1.476562, 0 } }, /*4500K*/
  { "Olympus", "E-1", NeutralFluorescent, 7,	{ 1.617188, 1, 1.515625, 0 } }, /*4500K*/
  { "Olympus", "E-1", DaylightFluorescent, -7,	{ 2.819820, 1.153153, 1, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -6,	{ 2.669565, 1.113043, 1, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -5,	{ 2.521008, 1.075630, 1, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -4,	{ 2.390244, 1.040650, 1, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -3,	{ 2.259843, 1.007874, 1, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -2,	{ 2.195312, 1, 1.023438, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, -1,	{ 2.140625, 1, 1.054688, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 0,	{ 2.101562, 1, 1.085938, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 1,	{ 2.070312, 1, 1.101562, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 2,	{ 2.046875, 1, 1.117188, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 3,	{ 2.023438, 1, 1.132812, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 4,	{ 2.000000, 1, 1.156250, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 5,	{ 1.976562, 1, 1.171875, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 6,	{ 1.953125, 1, 1.187500, 0 } }, /*6600K*/
  { "Olympus", "E-1", DaylightFluorescent, 7,	{ 1.929688, 1, 1.203125, 0 } }, /*6600K*/
  { "Olympus", "E-1", Daylight, -7,		{ 1.726562, 1, 1.093750, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -6,		{ 1.710938, 1, 1.101562, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -5,		{ 1.703125, 1, 1.109375, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -4,		{ 1.695312, 1, 1.117188, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -3,		{ 1.687500, 1, 1.117188, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -2,		{ 1.671875, 1, 1.125000, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, -1,		{ 1.664062, 1, 1.132812, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 0,		{ 1.664062, 1, 1.140625, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 1,		{ 1.648438, 1, 1.148438, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 2,		{ 1.640625, 1, 1.156250, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 3,		{ 1.632812, 1, 1.164062, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 4,		{ 1.617188, 1, 1.164062, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 5,		{ 1.609375, 1, 1.171875, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 6,		{ 1.601562, 1, 1.179688, 0 } }, /*5300K*/
  { "Olympus", "E-1", Daylight, 7,		{ 1.593750, 1, 1.187500, 0 } }, /*5300K*/
  { "Olympus", "E-1", Cloudy, -7,		{ 2.008130, 1.040650, 1, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -6,		{ 1.967742, 1.032258, 1, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -5,		{ 1.920635, 1.015873, 1, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -4,		{ 1.867188, 1, 1.000000, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -3,		{ 1.851562, 1, 1.007812, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -2,		{ 1.828125, 1, 1.023438, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, -1,		{ 1.812500, 1, 1.031250, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 0,		{ 1.796875, 1, 1.046875, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 1,		{ 1.781250, 1, 1.054688, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 2,		{ 1.773438, 1, 1.062500, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 3,		{ 1.757812, 1, 1.070312, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 4,		{ 1.750000, 1, 1.070312, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 5,		{ 1.742188, 1, 1.078125, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 6,		{ 1.734375, 1, 1.085938, 0 } }, /*6000K*/
  { "Olympus", "E-1", Cloudy, 7,		{ 1.726562, 1, 1.093750, 0 } }, /*6000K*/
  { "Olympus", "E-1", Shade, -7,		{ 2.584906, 1.207547, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -6,		{ 2.532710, 1.196262, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -5,		{ 2.467890, 1.174312, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -4,		{ 2.396396, 1.153153, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -3,		{ 2.357143, 1.142857, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -2,		{ 2.289474, 1.122807, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, -1,		{ 2.252174, 1.113043, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 0,			{ 2.196581, 1.094017, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 1,			{ 2.126050, 1.075630, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 2,			{ 2.091667, 1.066667, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 3,			{ 2.032787, 1.049180, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 4,			{ 2.000000, 1.040650, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 5,			{ 1.944000, 1.024000, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 6,			{ 1.897638, 1.007874, 1, 0 } }, /*7500K*/
  { "Olympus", "E-1", Shade, 7,			{ 1.859375, 1, 1.000000, 0 } }, /*7500K*/
  { "Olympus", "E-1", "3300K", -7,		{ 1.109375, 1, 1.695312, 0 } },
  { "Olympus", "E-1", "3300K", -6,		{ 1.101562, 1, 1.710938, 0 } },
  { "Olympus", "E-1", "3300K", -5,		{ 1.093750, 1, 1.718750, 0 } },
  { "Olympus", "E-1", "3300K", -4,		{ 1.093750, 1, 1.734375, 0 } },
  { "Olympus", "E-1", "3300K", -3,		{ 1.085938, 1, 1.742188, 0 } },
  { "Olympus", "E-1", "3300K", -2,		{ 1.078125, 1, 1.750000, 0 } },
  { "Olympus", "E-1", "3300K", -1,		{ 1.070312, 1, 1.765625, 0 } },
  { "Olympus", "E-1", "3300K", 0,		{ 1.070312, 1, 1.773438, 0 } },
  { "Olympus", "E-1", "3300K", 1,		{ 1.054688, 1, 1.781250, 0 } },
  { "Olympus", "E-1", "3300K", 2,		{ 1.046875, 1, 1.796875, 0 } },
  { "Olympus", "E-1", "3300K", 3,		{ 1.046875, 1, 1.804688, 0 } },
  { "Olympus", "E-1", "3300K", 4,		{ 1.039062, 1, 1.820312, 0 } },
  { "Olympus", "E-1", "3300K", 5,		{ 1.031250, 1, 1.828125, 0 } },
  { "Olympus", "E-1", "3300K", 6,		{ 1.023438, 1, 1.843750, 0 } },
  { "Olympus", "E-1", "3300K", 7,		{ 1.015625, 1, 1.851562, 0 } },

  { "Olympus", "E-10", Incandescent, 0,		{ 1, 1.153153, 3.441442, 0 } }, /*3000K*/
  { "Olympus", "E-10", IncandescentWarm, 0,	{ 1.101562, 1, 2.351562, 0 } }, /*3700K*/
  { "Olympus", "E-10", WhiteFluorescent, 0,	{ 1.460938, 1, 2.546875, 0 } }, /*4000K*/
  { "Olympus", "E-10", DaylightFluorescent, 0,	{ 1.460938, 1, 1.843750, 0 } }, /*4500K*/
  { "Olympus", "E-10", Daylight, 0,		{ 1.523438, 1, 1.617188, 0 } }, /*5500K*/
  { "Olympus", "E-10", Cloudy, 0,		{ 1.687500, 1, 1.437500, 0 } }, /*6500K*/
  { "Olympus", "E-10", Shade, 0,		{ 1.812500, 1, 1.312500, 0 } }, /*7500K*/

  /* Firmware version 1.4 */
  { "Olympus", "E-3", Daylight, 0,		{ 1.7812, 1, 1.4688, 0 } },
  { "Olympus", "E-3", Shade, 0,			{ 2.1406, 1, 1.1484, 0 } },
  { "Olympus", "E-3", Cloudy, 0,		{ 1.9531, 1, 1.3359, 0 } },
  { "Olympus", "E-3", Incandescent, 0,		{ 1.0312, 1, 2.8125, 0 } },
  { "Olympus", "E-3", WhiteFluorescent, 0,	{ 1.6328, 1, 2.1953, 0 } },
  { "Olympus", "E-3", NeutralFluorescent, 0,	{ 1.6641, 1, 1.7734, 0 } },
  { "Olympus", "E-3", DaylightFluorescent, 0,	{ 1.9531, 1, 1.4844, 0 } },
  { "Olympus", "E-3", Flash, 0,			{ 1.9609, 1, 1.3359, 0 } },
  { "Olympus", "E-3", "2700K", 0,		{ 1, 1.0240, 3.0080, 0 } },
  { "Olympus", "E-3", "3000K", 0,		{ 1.1328, 1, 2.5156, 0 } },
  { "Olympus", "E-3", "3300K", 0,		{ 1.2734, 1, 2.1641, 0 } },
  { "Olympus", "E-3", "5000K", 0,		{ 1.7266, 1, 1.5156, 0 } },

  /* Firmware version 1.3 */
  { "Olympus", "E-5", Daylight, 0,		{ 1.7344, 1, 1.3203, 0 } },
  { "Olympus", "E-5", Shade, 0,			{ 2.1016, 1, 1.0313, 0 } },
  { "Olympus", "E-5", Cloudy, 0,		{ 1.9141, 1, 1.1953, 0 } },
  { "Olympus", "E-5", Incandescent, 0,		{ 1.0000, 1, 2.3906, 0 } },
  { "Olympus", "E-5", Fluorescent, 0,		{ 1.6484, 1, 1.9141, 0 } },
  { "Olympus", "E-5", Underwater, 0,		{ 1.7266, 1, 1.3828, 0 } },
  { "Olympus", "E-5", Flash, 0,			{ 1.9063, 1, 1.1797, 0 } },
  { "Olympus", "E-5", "2700K", 0,		{ 1, 1.0579, 2.6529, 0 } },
  { "Olympus", "E-5", "3000K", 0,		{ 1.1016, 1, 2.1641, 0 } },
  { "Olympus", "E-5", "3300K", 0,		{ 1.2344, 1, 1.8750, 0 } },
  { "Olympus", "E-5", "5000K", 0,		{ 1.6641, 1, 1.3828, 0 } },

  { "Olympus", "E-30", Daylight, -7,		{ 1.554688, 1, 1.515625, 0 } },
  { "Olympus", "E-30", Daylight, 0,		{ 1.812500, 1, 1.335937, 0 } },
  { "Olympus", "E-30", Daylight, 7,		{ 2.062500, 1, 1.148437, 0 } },
  { "Olympus", "E-30", Shade, -7,		{ 1.867188, 1, 1.171875, 0 } },
  { "Olympus", "E-30", Shade, 0,		{ 2.179688, 1, 1.031250, 0 } },
  { "Olympus", "E-30", Shade, 7,		{ 2.814159, 1.132743, 1, 0 } },
  { "Olympus", "E-30", Cloudy, -7,		{ 1.710938, 1, 1.359375, 0 } },
  { "Olympus", "E-30", Cloudy, 0,		{ 1.992187, 1, 1.195312, 0 } },
  { "Olympus", "E-30", Cloudy, 7,		{ 2.265625, 1, 1.023438, 0 } },
  { "Olympus", "E-30", Incandescent, -7,	{ 1, 1.103448, 3.137931, 0 } },
  { "Olympus", "E-30", Incandescent, 0,		{ 1.054687, 1, 2.500000, 0 } },
  { "Olympus", "E-30", Incandescent, 7,		{ 1.195313, 1, 2.148437, 0 } },
  { "Olympus", "E-30", WhiteFluorescent, -7,	{ 1.453125, 1, 2.187500, 0 } },
  { "Olympus", "E-30", WhiteFluorescent, 0,	{ 1.695313, 1, 1.921875, 0 } },
  { "Olympus", "E-30", WhiteFluorescent, 7,	{ 1.929687, 1, 1.648437, 0 } },
  { "Olympus", "E-30", NeutralFluorescent, -7,	{ 1.437500, 1, 1.929687, 0 } },
  { "Olympus", "E-30", NeutralFluorescent, 0,	{ 1.679687, 1, 1.695313, 0 } },
  { "Olympus", "E-30", NeutralFluorescent, 7,	{ 1.914063, 1, 1.453125, 0 } },
  { "Olympus", "E-30", DaylightFluorescent, -7,	{ 1.765625, 1, 1.500000, 0 } },
  { "Olympus", "E-30", DaylightFluorescent, 0,	{ 2.054688, 1, 1.320313, 0 } },
  { "Olympus", "E-30", DaylightFluorescent, 7,	{ 2.335938, 1, 1.132812, 0 } },
  { "Olympus", "E-30", Flash, -7,		{ 1.710938, 1, 1.359375, 0 } },
  { "Olympus", "E-30", Flash, 0,		{ 1.992187, 1, 1.195312, 0 } },
  { "Olympus", "E-30", Flash, 7,		{ 2.265625, 1, 1.023438, 0 } },
  { "Olympus", "E-30", "2700K",  0,		{ 1.000000, 1, 2.632813, 0 } },
  { "Olympus", "E-30", "3000K",  0,		{ 1.164062, 1, 2.250000, 0 } },
  { "Olympus", "E-30", "3300K",  0,		{ 1.304687, 1, 1.929687, 0 } },
  { "Olympus", "E-30", "5000K",  0,		{ 1.742187, 1, 1.406250, 0 } },

  { "Olympus", "E-300", Incandescent, -7,	{ 1.179688, 1, 2.125000, 0 } },
  { "Olympus", "E-300", Incandescent, 0,	{ 1.140625, 1, 2.203125, 0 } },
  { "Olympus", "E-300", Incandescent, 7,	{ 1.093750, 1, 2.273438, 0 } },
  { "Olympus", "E-300", IncandescentWarm, -7,	{ 1.382812, 1, 1.859375, 0 } },
  { "Olympus", "E-300", IncandescentWarm, 0,	{ 1.312500, 1, 1.906250, 0 } },
  { "Olympus", "E-300", IncandescentWarm, 7,	{ 1.257812, 1, 1.984375, 0 } },
  { "Olympus", "E-300", WhiteFluorescent, -7,	{ 2.109375, 1, 1.710938, 0 } },
  { "Olympus", "E-300", WhiteFluorescent, 0,	{ 1.976562, 1, 1.921875, 0 } },
  { "Olympus", "E-300", WhiteFluorescent, 7,	{ 1.804688, 1, 2.062500, 0 } },
  { "Olympus", "E-300", NeutralFluorescent, -7,	{ 1.945312, 1, 1.445312, 0 } },
  { "Olympus", "E-300", NeutralFluorescent, 0,	{ 1.820312, 1, 1.562500, 0 } },
  { "Olympus", "E-300", NeutralFluorescent, 7,	{ 1.585938, 1, 1.945312, 0 } },
  { "Olympus", "E-300", DaylightFluorescent, -7, { 2.203125, 1, 1.000000, 0 } },
  { "Olympus", "E-300", DaylightFluorescent, 0,	{ 2.031250, 1, 1.328125, 0 } },
  { "Olympus", "E-300", DaylightFluorescent, 7,	{ 1.765625, 1, 1.367188, 0 } },
  { "Olympus", "E-300", Daylight, -7,		{ 1.835938, 1, 1.304688, 0 } },
  { "Olympus", "E-300", Daylight, 0,		{ 1.789062, 1, 1.351562, 0 } },
  { "Olympus", "E-300", Daylight, 7,		{ 1.726562, 1, 1.398438, 0 } },
  { "Olympus", "E-300", Cloudy, -7,		{ 2.000000, 1, 1.156250, 0 } },
  { "Olympus", "E-300", Cloudy, 0,		{ 1.890625, 1, 1.257812, 0 } },
  { "Olympus", "E-300", Cloudy, 7,		{ 1.835938, 1, 1.304688, 0 } },
  { "Olympus", "E-300", Shade, -7,		{ 2.179688, 1, 1.007812, 0 } },
  { "Olympus", "E-300", Shade, 0,		{ 2.070312, 1, 1.109375, 0 } },
  { "Olympus", "E-300", Shade, 7,		{ 1.945312, 1, 1.210938, 0 } },

  { "Olympus", "E-330", Daylight, 0,		{ 1.812500, 1, 1.296875, 0 } }, /*5300K*/
  { "Olympus", "E-330", Cloudy, 0,		{ 1.953125, 1, 1.195312, 0 } }, /*6000K*/
  { "Olympus", "E-330", Shade, 0,		{ 2.187500, 1, 1.054688, 0 } }, /*7500K*/
  { "Olympus", "E-330", Incandescent, 0,	{ 1.039062, 1, 2.437500, 0 } }, /*3000K*/
  { "Olympus", "E-330", WhiteFluorescent, 0,	{ 1.710938, 1, 1.906250, 0 } }, /*4000K*/
  { "Olympus", "E-330", NeutralFluorescent, 0,	{ 1.750000, 1, 1.531250, 0 } }, /*4500K*/
  { "Olympus", "E-330", DaylightFluorescent, 0,	{ 2.062500, 1, 1.289062, 0 } }, /*6600K*/

  { "Olympus", "E-400", Daylight, -7,		{ 2.554687, 1, 1.390625, 0 } },
  { "Olympus", "E-400", Daylight, 0,		{ 2.312500, 1, 1.179687, 0 } },
  { "Olympus", "E-400", Daylight, 7,		{ 2.096774, 1.032258, 1, 0 } },
  { "Olympus", "E-400", Cloudy, -7,		{ 2.695312, 1, 1.289062, 0 } },
  { "Olympus", "E-400", Cloudy, 0,		{ 2.437500, 1, 1.093750, 0 } },
  { "Olympus", "E-400", Cloudy, 7,		{ 2.554545, 1.163636, 1, 0 } },
  { "Olympus", "E-400", Shade, -7,		{ 2.835937, 1, 1.187500, 0 } },
  { "Olympus", "E-400", Shade, 0,		{ 2.754098, 1.049180, 1, 0 } },
  { "Olympus", "E-400", Shade, 7,		{ 3.202128, 1.361702, 1, 0 } },
  { "Olympus", "E-400", Incandescent, -7,	{ 1.500000, 1, 2.710938, 0 } },
  { "Olympus", "E-400", Incandescent, 0,	{ 1.460937, 1, 2.171875, 0 } },
  { "Olympus", "E-400", Incandescent, 7,	{ 1.367187, 1, 1.679688, 0 } },
  { "Olympus", "E-400", WhiteFluorescent, -7,	{ 2.523438, 1, 2.250000, 0 } },
  { "Olympus", "E-400", WhiteFluorescent, 0,	{ 2.390625, 1, 1.796875, 0 } },
  { "Olympus", "E-400", WhiteFluorescent, 7,	{ 2.164063, 1, 1.429688, 0 } },
  { "Olympus", "E-400", NeutralFluorescent, -7,	{ 2.226562, 1, 1.828125, 0 } },
  { "Olympus", "E-400", NeutralFluorescent, 0,	{ 2.132812, 1, 1.468750, 0 } },
  { "Olympus", "E-400", NeutralFluorescent, 7,	{ 1.953125, 1, 1.156250, 0 } },
  { "Olympus", "E-400", DaylightFluorescent, -7, { 2.593750, 1, 1.359375, 0 } },
  { "Olympus", "E-400", DaylightFluorescent, 0,	{ 2.445313, 1, 1.195313, 0 } },
  { "Olympus", "E-400", DaylightFluorescent, 7,	{ 3.293478, 1.391304, 1, 0 } },

  { "Olympus", "E-410", Daylight, 0,		{ 1.914063, 1, 1.367188, 0 } }, /*5300K*/
  { "Olympus", "E-410", Cloudy, 0,		{ 2.054688, 1, 1.250000, 0 } }, /*6000K*/
  { "Olympus", "E-410", Shade, 0,		{ 2.304688, 1, 1.031250, 0 } }, /*7500K*/
  { "Olympus", "E-410", Incandescent, 0,	{ 1.062500, 1, 2.781250, 0 } }, /*3000K*/
  { "Olympus", "E-410", WhiteFluorescent, 0,	{ 1.726562, 1, 2.226562, 0 } }, /*4000K*/
  { "Olympus", "E-410", NeutralFluorescent, 0,	{ 1.703125, 1, 1.796875, 0 } }, /*4500K*/
  { "Olympus", "E-410", DaylightFluorescent, 0,	{ 2.039063, 1, 1.476562, 0 } }, /*6600K*/

  { "Olympus", "E-420", Daylight, 0,		{ 1.820313, 1, 1.437500, 0 } },
  { "Olympus", "E-420", Shade, 0,		{ 2.179688, 1, 1.140625, 0 } },
  { "Olympus", "E-420", Cloudy, 0,		{ 2.000000, 1, 1.289062, 0 } },
  { "Olympus", "E-420", Incandescent, 0,	{ 1.039062, 1, 2.726562, 0 } },
  { "Olympus", "E-420", WhiteFluorescent, 0,	{ 1.703125, 1, 2.109375, 0 } },
  { "Olympus", "E-420", NeutralFluorescent, 0,	{ 1.703125, 1, 1.757812, 0 } },
  { "Olympus", "E-420", Flash, 0,		{ 2.078125, 1, 1.375000, 0 } },

  { "Olympus", "E-500", Daylight, 0,		{ 1.898438, 1, 1.359375, 0 } }, /*5300K*/
  { "Olympus", "E-500", Cloudy, 0,		{ 1.992188, 1, 1.265625, 0 } }, /*6000K*/
  { "Olympus", "E-500", Shade, 0,		{ 2.148438, 1, 1.125000, 0 } }, /*7500K*/
  { "Olympus", "E-500", Incandescent, 0,	{ 1.265625, 1, 2.195312, 0 } }, /*3000K*/
  { "Olympus", "E-500", WhiteFluorescent, 0,	{ 1.976562, 1, 1.914062, 0 } }, /*4000K*/
  { "Olympus", "E-500", NeutralFluorescent, 0,	{ 1.828125, 1, 1.562500, 0 } }, /*4500K*/
  { "Olympus", "E-500", DaylightFluorescent, 0,	{ 2.046875, 1, 1.359375, 0 } }, /*6600K*/

  { "Olympus", "E-510", Daylight, -7,		{ 2.164063, 1, 1.546875, 0 } },
  { "Olympus", "E-510", Daylight, 0,		{ 1.968750, 1, 1.296875, 0 } },
  { "Olympus", "E-510", Daylight, 7,		{ 1.742187, 1, 1.062500, 0 } },
  { "Olympus", "E-510", Shade, -7,		{ 2.492188, 1, 1.273438, 0 } },
  { "Olympus", "E-510", Shade, 0,		{ 2.439024, 1.040650, 1, 0 } },
  { "Olympus", "E-510", Shade, 7,		{ 3.055556, 1.422222, 1, 0 } },
  { "Olympus", "E-510", Cloudy, -7,		{ 2.312500, 1, 1.414062, 0 } },
  { "Olympus", "E-510", Cloudy, 0,		{ 2.109375, 1, 1.187500, 0 } },
  { "Olympus", "E-510", Cloudy, 7,		{ 2.192982, 1.122807, 1, 0 } },
  { "Olympus", "E-510", Incandescent, -7,	{ 1.109375, 1, 3.351562, 0 } },
  { "Olympus", "E-510", Incandescent, 0,	{ 1.093750, 1, 2.671875, 0 } },
  { "Olympus", "E-510", Incandescent, 7,	{ 1.031250, 1, 2.054688, 0 } },
  { "Olympus", "E-510", WhiteFluorescent, -7,	{ 1.578125, 1, 2.250000, 0 } },
  { "Olympus", "E-510", WhiteFluorescent, 0,	{ 1.718750, 1, 2.109375, 0 } },
  { "Olympus", "E-510", WhiteFluorescent, 7,	{ 1.523437, 1, 1.265625, 0 } },
  { "Olympus", "E-510", NeutralFluorescent, -7,	{ 1.835938, 1, 1.828125, 0 } },
  { "Olympus", "E-510", NeutralFluorescent, 0,	{ 1.687500, 1, 1.710938, 0 } },
  { "Olympus", "E-510", NeutralFluorescent, 7,	{ 1.726562, 1, 1.078125, 0 } },
  { "Olympus", "E-510", DaylightFluorescent, -7, { 2.203125, 1, 1.500000, 0 } },
  { "Olympus", "E-510", DaylightFluorescent, 0,	{ 2.023438, 1, 1.398437, 0 } },
  { "Olympus", "E-510", DaylightFluorescent, 7,	{ 3.193182, 1.454545, 1, 0 } },

  { "Olympus", "E-520", Daylight, 0,		{ 1.859375, 1, 1.445312, 0 } },
  { "Olympus", "E-520", Shade, 0,		{ 2.234375, 1, 1.140625, 0 } },
  { "Olympus", "E-520", Cloudy, 0,		{ 2.046875, 1, 1.296875, 0 } },
  { "Olympus", "E-520", Tungsten, 0,		{ 1.062500, 1, 2.687500, 0 } },
  { "Olympus", "E-520", WhiteFluorescent, 0,	{ 1.703125, 1, 2.109375, 0 } },
  { "Olympus", "E-520", NeutralFluorescent, 0,	{ 1.718750, 1, 1.765625, 0 } },
  { "Olympus", "E-520", DaylightFluorescent, 0,	{ 2.101563, 1, 1.375000, 0 } },
  { "Olympus", "E-520", Flash, 0,		{ 2.039063, 1, 1.296875, 0 } },
  { "Olympus", "E-520", "2700K", 0,		{ 1.007812, 1, 2.828125, 0 } },
  { "Olympus", "E-520", "3000K", 0,		{ 1.171875, 1, 2.421875, 0 } },
  { "Olympus", "E-520", "3300K", 0,		{ 1.312500, 1, 2.078125, 0 } },
  { "Olympus", "E-520", "5000K", 0,		{ 1.789062, 1, 1.515625, 0 } },

  /* -7/+7 fine tuning is -7/+7 in both amber-blue and green-magenta */
  { "Olympus", "E-600", Daylight, -7,		{ 1.804688, 1, 1.671875, 0 } },
  { "Olympus", "E-600", Daylight, 0,		{ 1.851563, 1, 1.289063, 0 } },
  { "Olympus", "E-600", Daylight, 7,		{ 1.917355, 1.057851, 1, 0 } },
  { "Olympus", "E-600", Shade, -7,		{ 2.179688, 1, 1.281250, 0 } },
  { "Olympus", "E-600", Shade, 0,		{ 2.244094, 1.007874, 1, 0 } },
  { "Olympus", "E-600", Shade, 7,		{ 2.989247, 1.376344, 1, 0 } },
  { "Olympus", "E-600", Cloudy, -7,		{ 2.000000, 1, 1.500000, 0 } },
  { "Olympus", "E-600", Cloudy, 0,		{ 2.046875, 1, 1.164062, 0 } },
  { "Olympus", "E-600", Cloudy, 7,		{ 2.327273, 1.163636, 1, 0 } },
  { "Olympus", "E-600", Incandescent, -7,	{ 1.062500, 1, 3.156250, 0 } },
  { "Olympus", "E-600", Incandescent, 0,	{ 1.093750, 1, 2.437500, 0 } },
  { "Olympus", "E-600", Incandescent, 7,	{ 1.062500, 1, 1.796875, 0 } },
  { "Olympus", "E-600", WhiteFluorescent, -7,	{ 1.703125, 1, 2.398438, 0 } },
  { "Olympus", "E-600", WhiteFluorescent, 0,	{ 1.750000, 1, 1.851563, 0 } },
  { "Olympus", "E-600", WhiteFluorescent, 7,	{ 1.710938, 1, 1.359375, 0 } },
  { "Olympus", "E-600", NeutralFluorescent, -7,	{ 1.671875, 1, 2.109375, 0 } },
  { "Olympus", "E-600", NeutralFluorescent, 0,	{ 1.710938, 1, 1.625000, 0 } },
  { "Olympus", "E-600", NeutralFluorescent, 7,	{ 1.671875, 1, 1.195312, 0 } },
  { "Olympus", "E-600", DaylightFluorescent, -7, { 2.039063, 1, 1.632813, 0 } },
  { "Olympus", "E-600", DaylightFluorescent, 0,	{ 2.085937, 1, 1.265625, 0 } },
  { "Olympus", "E-600", DaylightFluorescent, 7,	{ 2.193277, 1.075630, 1, 0 } },
  { "Olympus", "E-600", Flash, -7,		{ 1.992187, 1, 1.492187, 0 } },
  { "Olympus", "E-600", Flash, 0,		{ 2.039063, 1, 1.156250, 0 } },
  { "Olympus", "E-600", Flash, 7,		{ 2.339450, 1.174312, 1, 0 } },

  /* -7/+7 fine tuning is -7/+7 in both amber-blue and green-magenta */
  { "Olympus", "E-620", Daylight, -7,		{ 1.804688, 1, 1.726563, 0 } },
  { "Olympus", "E-620", Daylight, 0,		{ 1.851563, 1, 1.335938, 0 } },
  { "Olympus", "E-620", Daylight, 7,		{ 1.841270, 1.015873, 1, 0 } },
  { "Olympus", "E-620", Shade, -7,		{ 2.171875, 1, 1.320312, 0 } },
  { "Olympus", "E-620", Shade, 0,		{ 2.218750, 1, 1.023438, 0 } },
  { "Olympus", "E-620", Shade, 7,		{ 2.885417, 1.333333, 1, 0 } },
  { "Olympus", "E-620", Cloudy, -7,		{ 1.992187, 1, 1.539062, 0 } },
  { "Olympus", "E-620", Cloudy, 0,		{ 2.039063, 1, 1.187500, 0 } },
  { "Olympus", "E-620", Cloudy, 7,		{ 2.297297, 1.153153, 1, 0 } },
  { "Olympus", "E-620", Incandescent, -7,	{ 1.070312, 1, 3.281250, 0 } },
  { "Olympus", "E-620", Incandescent, 0,	{ 1.101563, 1, 2.531250, 0 } },
  { "Olympus", "E-620", Incandescent, 7,	{ 1.070313, 1, 1.867188, 0 } },
  { "Olympus", "E-620", WhiteFluorescent, -7,	{ 1.679687, 1, 2.500000, 0 } },
  { "Olympus", "E-620", WhiteFluorescent, 0,	{ 1.718750, 1, 1.929687, 0 } },
  { "Olympus", "E-620", WhiteFluorescent, 7,	{ 1.679688, 1, 1.421875, 0 } },
  { "Olympus", "E-620", NeutralFluorescent, -7,	{ 1.632813, 1, 2.179688, 0 } },
  { "Olympus", "E-620", NeutralFluorescent, 0,	{ 1.671875, 1, 1.679688, 0 } },
  { "Olympus", "E-620", NeutralFluorescent, 7,	{ 1.625000, 1, 1.234375, 0 } },
  { "Olympus", "E-620", DaylightFluorescent, -7, { 2.000000, 1, 1.687500, 0 } },
  { "Olympus", "E-620", DaylightFluorescent, 0,	{ 2.046875, 1, 1.304687, 0 } },
  { "Olympus", "E-620", DaylightFluorescent, 7,	{ 2.098361, 1.049180, 1, 0 } },
  { "Olympus", "E-620", Flash, -7,		{ 1.992187, 1, 1.546875, 0 } },
  { "Olympus", "E-620", Flash, 0,		{ 2.039063, 1, 1.195313, 0 } },
  { "Olympus", "E-620", Flash, 7,		{ 2.276786, 1.142857, 1, 0 } },

  { "Olympus", "E-M1", Daylight            , -7, { 1.640625, 1, 1.890625, 0 } },
  { "Olympus", "E-M1", Daylight            , -6, { 1.6796875, 1, 1.859375, 0 } },
  { "Olympus", "E-M1", Daylight            , -5, { 1.71875, 1, 1.828125, 0 } },
  { "Olympus", "E-M1", Daylight            , -4, { 1.7578125, 1, 1.796875, 0 } },
  { "Olympus", "E-M1", Daylight            , -3, { 1.796875, 1, 1.7578125, 0 } },
  { "Olympus", "E-M1", Daylight            , -2, { 1.8359375, 1, 1.7265625, 0 } },
  { "Olympus", "E-M1", Daylight            , -1, { 1.875, 1, 1.6953125, 0 } },
  { "Olympus", "E-M1", Daylight            , 0, { 1.9140625, 1, 1.6640625, 0 } },
  { "Olympus", "E-M1", Daylight            , 1, { 1.9453125, 1, 1.625, 0 } },
  { "Olympus", "E-M1", Daylight            , 2, { 1.984375, 1, 1.59375, 0 } },
  { "Olympus", "E-M1", Daylight            , 3, { 2.0234375, 1, 1.5625, 0 } },
  { "Olympus", "E-M1", Daylight            , 4, { 2.0625, 1, 1.5234375, 0 } },
  { "Olympus", "E-M1", Daylight            , 5, { 2.1015625, 1, 1.4921875, 0 } },
  { "Olympus", "E-M1", Daylight            , 6, { 2.140625, 1, 1.4609375, 0 } },
  { "Olympus", "E-M1", Daylight            , 7, { 2.1796875, 1, 1.4296875, 0 } },
  { "Olympus", "E-M1", Shade               , -7, { 1.9140625, 1, 1.5, 0 } },
  { "Olympus", "E-M1", Shade               , -6, { 1.953125, 1, 1.4765625, 0 } },
  { "Olympus", "E-M1", Shade               , -5, { 2.0, 1, 1.4453125, 0 } },
  { "Olympus", "E-M1", Shade               , -4, { 2.046875, 1, 1.421875, 0 } },
  { "Olympus", "E-M1", Shade               , -3, { 2.0859375, 1, 1.3984375, 0 } },
  { "Olympus", "E-M1", Shade               , -2, { 2.1328125, 1, 1.3671875, 0 } },
  { "Olympus", "E-M1", Shade               , -1, { 2.1796875, 1, 1.34375, 0 } },
  { "Olympus", "E-M1", Shade               , 0, { 2.2265625, 1, 1.3203125, 0 } },
  { "Olympus", "E-M1", Shade               , 1, { 2.265625, 1, 1.2890625, 0 } },
  { "Olympus", "E-M1", Shade               , 2, { 2.3125, 1, 1.265625, 0 } },
  { "Olympus", "E-M1", Shade               , 3, { 2.359375, 1, 1.234375, 0 } },
  { "Olympus", "E-M1", Shade               , 4, { 2.3984375, 1, 1.2109375, 0 } },
  { "Olympus", "E-M1", Shade               , 5, { 2.4453125, 1, 1.1875, 0 } },
  { "Olympus", "E-M1", Shade               , 6, { 2.4921875, 1, 1.15625, 0 } },
  { "Olympus", "E-M1", Shade               , 7, { 2.53125, 1, 1.1328125, 0 } },
  { "Olympus", "E-M1", Cloudy              , -7, { 1.7734375, 1, 1.7421875, 0 } },
  { "Olympus", "E-M1", Cloudy              , -6, { 1.8125, 1, 1.7109375, 0 } },
  { "Olympus", "E-M1", Cloudy              , -5, { 1.8515625, 1, 1.6796875, 0 } },
  { "Olympus", "E-M1", Cloudy              , -4, { 1.890625, 1, 1.6484375, 0 } },
  { "Olympus", "E-M1", Cloudy              , -3, { 1.9375, 1, 1.6171875, 0 } },
  { "Olympus", "E-M1", Cloudy              , -2, { 1.9765625, 1, 1.5859375, 0 } },
  { "Olympus", "E-M1", Cloudy              , -1, { 2.015625, 1, 1.5546875, 0 } },
  { "Olympus", "E-M1", Cloudy              , 0, { 2.0625, 1, 1.53125, 0 } },
  { "Olympus", "E-M1", Cloudy              , 1, { 2.1015625, 1, 1.5, 0 } },
  { "Olympus", "E-M1", Cloudy              , 2, { 2.140625, 1, 1.46875, 0 } },
  { "Olympus", "E-M1", Cloudy              , 3, { 2.1796875, 1, 1.4375, 0 } },
  { "Olympus", "E-M1", Cloudy              , 4, { 2.2265625, 1, 1.40625, 0 } },
  { "Olympus", "E-M1", Cloudy              , 5, { 2.265625, 1, 1.375, 0 } },
  { "Olympus", "E-M1", Cloudy              , 6, { 2.3046875, 1, 1.34375, 0 } },
  { "Olympus", "E-M1", Cloudy              , 7, { 2.34375, 1, 1.3125, 0 } },
  { "Olympus", "E-M1", Tungsten            , -7, { 0.9609375, 1, 3.328125, 0 } },
  { "Olympus", "E-M1", Tungsten            , -6, { 0.984375, 1, 3.265625, 0 } },
  { "Olympus", "E-M1", Tungsten            , -5, { 1.0078125, 1, 3.2109375, 0 } },
  { "Olympus", "E-M1", Tungsten            , -4, { 1.03125, 1, 3.1484375, 0 } },
  { "Olympus", "E-M1", Tungsten            , -3, { 1.0546875, 1, 3.09375, 0 } },
  { "Olympus", "E-M1", Tungsten            , -2, { 1.078125, 1, 3.03125, 0 } },
  { "Olympus", "E-M1", Tungsten            , -1, { 1.1015625, 1, 2.9765625, 0 } },
  { "Olympus", "E-M1", Tungsten            , 0, { 1.125, 1, 2.921875, 0 } },
  { "Olympus", "E-M1", Tungsten            , 1, { 1.140625, 1, 2.859375, 0 } },
  { "Olympus", "E-M1", Tungsten            , 2, { 1.1640625, 1, 2.8046875, 0 } },
  { "Olympus", "E-M1", Tungsten            , 3, { 1.1875, 1, 2.7421875, 0 } },
  { "Olympus", "E-M1", Tungsten            , 4, { 1.2109375, 1, 2.6875, 0 } },
  { "Olympus", "E-M1", Tungsten            , 5, { 1.234375, 1, 2.625, 0 } },
  { "Olympus", "E-M1", Tungsten            , 6, { 1.2578125, 1, 2.5703125, 0 } },
  { "Olympus", "E-M1", Tungsten            , 7, { 1.28125, 1, 2.5078125, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -7, { 1.6015625, 1, 2.6484375, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -6, { 1.640625, 1, 2.6015625, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -5, { 1.6796875, 1, 2.5546875, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -4, { 1.7109375, 1, 2.5078125, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -3, { 1.75, 1, 2.4609375, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -2, { 1.7890625, 1, 2.4140625, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, -1, { 1.828125, 1, 2.3671875, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 0, { 1.8671875, 1, 2.328125, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 1, { 1.8984375, 1, 2.28125, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 2, { 1.9375, 1, 2.234375, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 3, { 1.9765625, 1, 2.1875, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 4, { 2.015625, 1, 2.140625, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 5, { 2.046875, 1, 2.09375, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 6, { 2.0859375, 1, 2.046875, 0 } },
  { "Olympus", "E-M1", CoolWhiteFluorescent, 7, { 2.125, 1, 2.0, 0 } },
  { "Olympus", "E-M1", Underwater          , -7, { 1.453125, 1, 2.203125, 0 } },
  { "Olympus", "E-M1", Underwater          , -6, { 1.484375, 1, 2.1484375, 0 } },
  { "Olympus", "E-M1", Underwater          , -5, { 1.5078125, 1, 2.125, 0 } },
  { "Olympus", "E-M1", Underwater          , -4, { 1.5546875, 1, 2.0703125, 0 } },
  { "Olympus", "E-M1", Underwater          , -3, { 1.5859375, 1, 2.046875, 0 } },
  { "Olympus", "E-M1", Underwater          , -2, { 1.625, 1, 2.0078125, 0 } },
  { "Olympus", "E-M1", Underwater          , -1, { 1.65625, 1, 1.953125, 0 } },
  { "Olympus", "E-M1", Underwater          , 0, { 1.6953125, 1, 1.9375, 0 } },
  { "Olympus", "E-M1", Underwater          , 1, { 1.7265625, 1, 1.8984375, 0 } },
  { "Olympus", "E-M1", Underwater          , 2, { 1.7578125, 1, 1.859375, 0 } },
  { "Olympus", "E-M1", Underwater          , 3, { 1.796875, 1, 1.8203125, 0 } },
  { "Olympus", "E-M1", Underwater          , 4, { 1.828125, 1, 1.78125, 0 } },
  { "Olympus", "E-M1", Underwater          , 5, { 1.859375, 1, 1.7421875, 0 } },
  { "Olympus", "E-M1", Underwater          , 6, { 1.8984375, 1, 1.703125, 0 } },
  { "Olympus", "E-M1", Underwater          , 7, { 1.9296875, 1, 1.6484375, 0 } },
  { "Olympus", "E-M1", Flash               , -7, { 1.9453125, 1, 1.671875, 0 } },
  { "Olympus", "E-M1", Flash               , -6, { 1.9921875, 1, 1.640625, 0 } },
  { "Olympus", "E-M1", Flash               , -5, { 2.0390625, 1, 1.609375, 0 } },
  { "Olympus", "E-M1", Flash               , -4, { 2.078125, 1, 1.5859375, 0 } },
  { "Olympus", "E-M1", Flash               , -3, { 2.125, 1, 1.5546875, 0 } },
  { "Olympus", "E-M1", Flash               , -2, { 2.171875, 1, 1.5234375, 0 } },
  { "Olympus", "E-M1", Flash               , -1, { 2.21875, 1, 1.4921875, 0 } },
  { "Olympus", "E-M1", Flash               , 0, { 2.265625, 1, 1.46875, 0 } },
  { "Olympus", "E-M1", Flash               , 1, { 2.3046875, 1, 1.4375, 0 } },
  { "Olympus", "E-M1", Flash               , 2, { 2.3515625, 1, 1.40625, 0 } },
  { "Olympus", "E-M1", Flash               , 3, { 2.3984375, 1, 1.375, 0 } },
  { "Olympus", "E-M1", Flash               , 4, { 2.4453125, 1, 1.34375, 0 } },
  { "Olympus", "E-M1", Flash               , 5, { 2.4921875, 1, 1.3203125, 0 } },
  { "Olympus", "E-M1", Flash               , 6, { 2.53125, 1, 1.2890625, 0 } },
  { "Olympus", "E-M1", Flash               , 7, { 2.578125, 1, 1.2578125, 0 } },
  { "Olympus", "E-M1", "3000K"             , 0, { 1.125, 1, 2.921875, 0 } },
  { "Olympus", "E-M1", "3300K"             , 0, { 1.3671875, 1, 2.3515625, 0 } },
  { "Olympus", "E-M1", "3600K"             , 0, { 1.4609375, 1, 2.1640625, 0 } },
  { "Olympus", "E-M1", "3900K"             , 0, { 1.5546875, 1, 2.0234375, 0 } },
  { "Olympus", "E-M1", "4000K"             , 0, { 1.8671875, 1, 2.328125, 0 } },
  { "Olympus", "E-M1", "4300K"             , 0, { 1.6640625, 1, 1.9296875, 0 } },
  { "Olympus", "E-M1", "4500K"             , 0, { 1.8203125, 1, 1.96875, 0 } },
  { "Olympus", "E-M1", "4800K"             , 0, { 1.796875, 1, 1.78125, 0 } },
  { "Olympus", "E-M1", "5300K"             , 0, { 1.9140625, 1, 1.6640625, 0 } },
  { "Olympus", "E-M1", "6000K"             , 0, { 2.0625, 1, 1.53125, 0 } },
  { "Olympus", "E-M1", "6600K"             , 0, { 2.1640625, 1, 1.640625, 0 } },
  { "Olympus", "E-M1", "7500K"             , 0, { 2.2265625, 1, 1.3203125, 0 } },


  // Firmware version 1.1
  { "Olympus", "E-M1MarkII", Daylight, 0,       { 1.8203125, 1, 1.9140625, 0 } },
  { "Olympus", "E-M1MarkII", Shade, 0,		{ 2.1328125, 1, 1.484375, 0 } },
  { "Olympus", "E-M1MarkII", Cloudy, 0,		{ 1.953125, 1, 1.7265625, 0 } },
  { "Olympus", "E-M1MarkII", Tungsten, 0,       { 1.1328125, 1, 3.546875, 0 } },
  { "Olympus", "E-M1MarkII", Fluorescent, 0,    { 1.75, 1, 2.734375, 0 } },
  { "Olympus", "E-M1MarkII", Underwater, 0,     { 1.828125, 1, 1.921875, 0 } },
  { "Olympus", "E-M1MarkII", Flash, 0,		{ 2.0703125, 1, 1.6328125, 0 } },

  // Firmware version 1.0
  { "Olympus", "E-M1MarkIII", Daylight, 0,      { 1.835938, 1, 2.0625, 0 } },
  { "Olympus", "E-M1MarkIII", Shade, 0,	        { 2.171875, 1, 1.617188, 0 } },
  { "Olympus", "E-M1MarkIII", Cloudy, 0,	{ 1.976563, 1, 1.875, 0 } },
  { "Olympus", "E-M1MarkIII", Tungsten, 0,      { 1.125, 1, 3.820313, 0 } },
  { "Olympus", "E-M1MarkIII", Fluorescent, 0,   { 1.757813, 1, 3.007813, 0 } },
  { "Olympus", "E-M1MarkIII", Underwater, 0,    { 1.523438, 1, 3.65625, 0 } },
  { "Olympus", "E-M1MarkIII", Flash, 0,		{ 2.09375, 1, 1.757813, 0 } },

  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-M5", Daylight, -7,		{ 1.8047, 1, 2.0547, 0 } },
  { "Olympus", "E-M5", Daylight, 0,		{ 2.1016, 1, 1.8047, 0 } },
  { "Olympus", "E-M5", Daylight, 7,		{ 2.3906, 1, 1.5469, 0 } },
  { "Olympus", "E-M5", Shade, -7,		{ 2.1484, 1, 1.6172, 0 } },
  { "Olympus", "E-M5", Shade, 0,		{ 2.5000, 1, 1.4219, 0 } },
  { "Olympus", "E-M5", Shade, 7,		{ 2.8437, 1, 1.2187, 0 } },
  { "Olympus", "E-M5", Cloudy, -7,		{ 1.9766, 1, 1.8516, 0 } },
  { "Olympus", "E-M5", Cloudy, 0,		{ 2.3047, 1, 1.6250, 0 } },
  { "Olympus", "E-M5", Cloudy, 7,		{ 2.6250, 1, 1.3906, 0 } },
  { "Olympus", "E-M5", Incandescent, -7,	{ 1.1250, 1, 3.6953, 0 } },
  { "Olympus", "E-M5", Incandescent, 0,		{ 1.3125, 1, 3.2422, 0 } },
  { "Olympus", "E-M5", Incandescent, 7,		{ 1.4922, 1, 2.7812, 0 } },
  { "Olympus", "E-M5", Fluorescent, -7,		{ 1.7344, 1, 2.9375, 0 } },
  { "Olympus", "E-M5", Fluorescent, 0,		{ 2.0234, 1, 2.5781, 0 } },
  { "Olympus", "E-M5", Fluorescent, 7,		{ 2.3047, 1, 2.2109, 0 } },
  { "Olympus", "E-M5", Underwater, -7,		{ 1.3906, 1, 2.8281, 0 } },
  { "Olympus", "E-M5", Underwater, 0,		{ 1.6250, 1, 2.4844, 0 } },
  { "Olympus", "E-M5", Underwater, 7,		{ 1.8516, 1, 2.1328, 0 } },
  { "Olympus", "E-M5", Flash, -7,		{ 2.0391, 1, 1.8203, 0 } },
  { "Olympus", "E-M5", Flash, 0,		{ 2.3750, 1, 1.6016, 0 } },
  { "Olympus", "E-M5", Flash, 7,		{ 2.7031, 1, 1.3750, 0 } },

  { "Olympus", "E-M5 Mark II", Daylight            , -7, { 1.6171875, 1, 2.0625, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -6, { 1.65625, 1, 2.0234375, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -5, { 1.6875, 1, 1.9921875, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -4, { 1.7265625, 1, 1.953125, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -3, { 1.765625, 1, 1.9140625, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -2, { 1.8046875, 1, 1.8828125, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , -1, { 1.84375, 1, 1.84375, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 0, { 1.8828125, 1, 1.8125, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 1, { 1.9140625, 1, 1.7734375, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 2, { 1.953125, 1, 1.734375, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 3, { 1.9921875, 1, 1.703125, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 4, { 2.03125, 1, 1.6640625, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 5, { 2.0703125, 1, 1.625, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 6, { 2.1015625, 1, 1.59375, 0 } },
  { "Olympus", "E-M5 Mark II", Daylight            , 7, { 2.140625, 1, 1.5546875, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -7, { 1.9140625, 1, 1.6015625, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -6, { 1.9609375, 1, 1.5703125, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -5, { 2.0078125, 1, 1.546875, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -4, { 2.0546875, 1, 1.515625, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -3, { 2.09375, 1, 1.484375, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -2, { 2.140625, 1, 1.4609375, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , -1, { 2.1875, 1, 1.4296875, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 0, { 2.234375, 1, 1.40625, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 1, { 2.2734375, 1, 1.375, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 2, { 2.3203125, 1, 1.34375, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 3, { 2.3671875, 1, 1.3203125, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 4, { 2.40625, 1, 1.2890625, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 5, { 2.453125, 1, 1.265625, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 6, { 2.5, 1, 1.234375, 0 } },
  { "Olympus", "E-M5 Mark II", Shade               , 7, { 2.546875, 1, 1.203125, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -7, { 1.7421875, 1, 1.8671875, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -6, { 1.78125, 1, 1.8359375, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -5, { 1.828125, 1, 1.8046875, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -4, { 1.8671875, 1, 1.765625, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -3, { 1.90625, 1, 1.734375, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -2, { 1.9453125, 1, 1.703125, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , -1, { 1.984375, 1, 1.671875, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 0, { 2.03125, 1, 1.640625, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 1, { 2.0703125, 1, 1.6015625, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 2, { 2.109375, 1, 1.5703125, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 3, { 2.1484375, 1, 1.5390625, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 4, { 2.1875, 1, 1.5078125, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 5, { 2.234375, 1, 1.4765625, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 6, { 2.2734375, 1, 1.4375, 0 } },
  { "Olympus", "E-M5 Mark II", Cloudy              , 7, { 2.3125, 1, 1.40625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -7, { 0.9765625, 1, 4.046875, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -6, { 1.0, 1, 3.9765625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -5, { 1.0234375, 1, 3.90625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -4, { 1.046875, 1, 3.8359375, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -3, { 1.0703125, 1, 3.765625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -2, { 1.09375, 1, 3.6953125, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , -1, { 1.1171875, 1, 3.625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 0, { 1.140625, 1, 3.5546875, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 1, { 1.15625, 1, 3.4765625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 2, { 1.1796875, 1, 3.40625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 3, { 1.203125, 1, 3.3359375, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 4, { 1.2265625, 1, 3.265625, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 5, { 1.25, 1, 3.1953125, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 6, { 1.2734375, 1, 3.125, 0 } },
  { "Olympus", "E-M5 Mark II", Tungsten            , 7, { 1.296875, 1, 3.0546875, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -7, { 1.6015625, 1, 3.0546875, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -6, { 1.640625, 1, 3.0, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -5, { 1.6796875, 1, 2.9453125, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -4, { 1.7109375, 1, 2.890625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -3, { 1.75, 1, 2.8359375, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -2, { 1.7890625, 1, 2.78125, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, -1, { 1.828125, 1, 2.7265625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 0, { 1.8671875, 1, 2.6796875, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 1, { 1.8984375, 1, 2.625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 2, { 1.9375, 1, 2.5703125, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 3, { 1.9765625, 1, 2.515625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 4, { 2.015625, 1, 2.4609375, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 5, { 2.046875, 1, 2.40625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 6, { 2.0859375, 1, 2.3515625, 0 } },
  { "Olympus", "E-M5 Mark II", CoolWhiteFluorescent, 7, { 2.125, 1, 2.296875, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -7, { 1.15625, 1, 3.0, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -6, { 1.1875, 1, 2.9765625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -5, { 1.21875, 1, 2.890625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -4, { 1.2578125, 1, 2.8359375, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -3, { 1.265625, 1, 2.7890625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -2, { 1.296875, 1, 2.765625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , -1, { 1.3203125, 1, 2.7109375, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 0, { 1.3515625, 1, 2.6640625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 1, { 1.375, 1, 2.609375, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 2, { 1.3984375, 1, 2.5546875, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 3, { 1.4375, 1, 2.46875, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 4, { 1.4765625, 1, 2.421875, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 5, { 1.484375, 1, 2.390625, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 6, { 1.5078125, 1, 2.3125, 0 } },
  { "Olympus", "E-M5 Mark II", Underwater          , 7, { 1.546875, 1, 2.2578125, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -7, { 1.875, 1, 1.8046875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -6, { 1.921875, 1, 1.7734375, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -5, { 1.96875, 1, 1.7421875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -4, { 2.0078125, 1, 1.7109375, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -3, { 2.0546875, 1, 1.6796875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -2, { 2.09375, 1, 1.6484375, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , -1, { 2.140625, 1, 1.6171875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 0, { 2.1875, 1, 1.5859375, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 1, { 2.2265625, 1, 1.546875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 2, { 2.2734375, 1, 1.515625, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 3, { 2.3125, 1, 1.484375, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 4, { 2.359375, 1, 1.453125, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 5, { 2.40625, 1, 1.421875, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 6, { 2.4453125, 1, 1.390625, 0 } },
  { "Olympus", "E-M5 Mark II", Flash               , 7, { 2.4921875, 1, 1.359375, 0 } },
  { "Olympus", "E-M5 Mark II", "3000K"             , 0, { 1.140625, 1, 3.5546875, 0 } },
  { "Olympus", "E-M5 Mark II", "3300K"             , 0, { 1.3515625, 1, 2.7734375, 0 } },
  { "Olympus", "E-M5 Mark II", "3600K"             , 0, { 1.4453125, 1, 2.5390625, 0 } },
  { "Olympus", "E-M5 Mark II", "3900K"             , 0, { 1.53125, 1, 2.3515625, 0 } },
  { "Olympus", "E-M5 Mark II", "4000K"             , 0, { 1.8671875, 1, 2.6796875, 0 } },
  { "Olympus", "E-M5 Mark II", "4300K"             , 0, { 1.6640625, 1, 2.1328125, 0 } },
  { "Olympus", "E-M5 Mark II", "4500K"             , 0, { 1.8125, 1, 2.109375, 0 } },
  { "Olympus", "E-M5 Mark II", "4800K"             , 0, { 1.78125, 1, 1.9453125, 0 } },
  { "Olympus", "E-M5 Mark II", "5300K"             , 0, { 1.8828125, 1, 1.8125, 0 } },
  { "Olympus", "E-M5 Mark II", "6000K"             , 0, { 2.03125, 1, 1.640625, 0 } },
  { "Olympus", "E-M5 Mark II", "6600K"             , 0, { 2.1171875, 1, 1.6796875, 0 } },
  { "Olympus", "E-M5 Mark II", "7500K"             , 0, { 2.234375, 1, 1.40625, 0 } },


  /* Firmware 1.0 */
  { "Olympus", "E-M5 Mark III", Daylight            , 0, { 1.8515625, 1, 2.03125, 0 } },
  { "Olympus", "E-M5 Mark III", Shade               , 0, { 2.1796875, 1, 1.6015625, 0 } },
  { "Olympus", "E-M5 Mark III", Cloudy              , 0, { 1.9921875, 1, 1.8515625, 0 } },
  { "Olympus", "E-M5 Mark III", Tungsten            , 0, { 1.1328125, 1, 3.765625, 0 } },
  { "Olympus", "E-M5 Mark III", CoolWhiteFluorescent, 0, { 1.765625, 1, 2.953125, 0 } },
  { "Olympus", "E-M5 Mark III", Underwater          , 0, { 2.03125, 1, 2.03125, 0 } },
  { "Olympus", "E-M5 Mark III", Flash               , 0, { 2.109375, 1, 1.734375, 0 } },

  /* Firmware version 1.2 */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-M10", Daylight, -7,		{ 1.671875, 1, 2.070312, 0 } },
  { "Olympus", "E-M10", Daylight, 0,		{ 1.945313, 1, 1.820313, 0 } },
  { "Olympus", "E-M10", Daylight, 7,		{ 2.210937, 1, 1.562500, 0 } },
  { "Olympus", "E-M10", Shade, -7,		{ 1.898438, 1, 1.609375, 0 } },
  { "Olympus", "E-M10", Shade, 0,		{ 2.210937, 1, 1.414062, 0 } },
  { "Olympus", "E-M10", Shade, 7,		{ 2.515625, 1, 1.210937, 0 } },
  { "Olympus", "E-M10", Cloudy, -7,		{ 1.781250, 1, 1.882812, 0 } },
  { "Olympus", "E-M10", Cloudy, 0,		{ 2.078125, 1, 1.656250, 0 } },
  { "Olympus", "E-M10", Cloudy, 7,		{ 2.367187, 1, 1.421875, 0 } },
  { "Olympus", "E-M10", Tungsten, -7,		{ 1.046875, 1, 3.820313, 0 } },
  { "Olympus", "E-M10", Tungsten, 0,		{ 1.218750, 1, 3.351562, 0 } },
  { "Olympus", "E-M10", Tungsten, 7,		{ 1.382812, 1, 2.875000, 0 } },
  { "Olympus", "E-M10", Fluorescent, -7,	{ 1.640625, 1, 2.945313, 0 } },
  { "Olympus", "E-M10", Fluorescent, 0,		{ 1.914062, 1, 2.585938, 0 } },
  { "Olympus", "E-M10", Fluorescent, 7,		{ 2.179687, 1, 2.218750, 0 } },
  { "Olympus", "E-M10", Underwater, -7,		{ 1.921875, 1, 2.015625, 0 } },
  { "Olympus", "E-M10", Underwater, 0,		{ 2.242187, 1, 1.773437, 0 } },
  { "Olympus", "E-M10", Underwater, 7,		{ 2.554687, 1, 1.523437, 0 } },
  { "Olympus", "E-M10", Flash, -7,		{ 1.960937, 1, 1.796875, 0 } },
  { "Olympus", "E-M10", Flash, 0,		{ 2.281250, 1, 1.578125, 0 } },
  { "Olympus", "E-M10", Flash, 7,		{ 2.593750, 1, 1.351562, 0 } },
  { "Olympus", "E-M10", "2700K", 0,		{ 1.164062, 1, 3.515625, 0 } },
  { "Olympus", "E-M10", "3000K", 0,		{ 1.320312, 1, 3.039063, 0 } },
  { "Olympus", "E-M10", "3300K", 0,		{ 1.453125, 1, 2.640625, 0 } },
  { "Olympus", "E-M10", "5000K", 0,		{ 1.882813, 1, 1.906250, 0 } },

  { "Olympus", "E-M10 Mark II", Daylight, 0, { 1.851563, 1, 1.835938, 0 } },
  { "Olympus", "E-M10 Mark II", Shade, 0, { 2.171875, 1, 1.429688, 0 } },
  { "Olympus", "E-M10 Mark II", Cloudy, 0, { 1.992188, 1, 1.664063, 0 } },
  { "Olympus", "E-M10 Mark II", Tungsten, 0, { 1.140625, 1, 3.429688, 0 } },
  { "Olympus", "E-M10 Mark II", Fluorescent, 0, { 1.875, 1, 2.695313, 0 } },
  { "Olympus", "E-M10 Mark II", Underwater, 0, { 1.890625, 1, 2.257813, 0 } },
  { "Olympus", "E-M10 Mark II", Flash, 0, { 2.148438, 1, 1.601563, 0 } },

  { "Olympus", "E-M10 Mark III", Daylight, 0, { 1.8828125, 1, 1.8125, 0 } },
  { "Olympus", "E-M10 Mark III", Shade, 0, { 2.1953125, 1, 1.421875, 0 } },
  { "Olympus", "E-M10 Mark III", Cloudy, 0, { 2.015625, 1, 1.640625, 0 } },
  { "Olympus", "E-M10 Mark III", Tungsten, 0, { 1.15625, 1, 3.4140625, 0 } },
  { "Olympus", "E-M10 Mark III", CoolWhiteFluorescent, 0, { 1.875, 1, 2.6328125, 0 } },
  { "Olympus", "E-M10 Mark III", Flash, 0, { 2.140625, 1, 1.546875, 0 } },

  { "Olympus", "E-M10 Mark IV", Daylight            , -7, { 1.75, 1, 2.0703125, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -6, { 1.7890625, 1, 2.03125, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -5, { 1.828125, 1, 2.0, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -4, { 1.875, 1, 1.9609375, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -3, { 1.9140625, 1, 1.921875, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -2, { 1.953125, 1, 1.890625, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , -1, { 1.9921875, 1, 1.8515625, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 0, { 2.0390625, 1, 1.8203125, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 1, { 2.078125, 1, 1.78125, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 2, { 2.1171875, 1, 1.7421875, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 3, { 2.15625, 1, 1.7109375, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 4, { 2.1953125, 1, 1.671875, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 5, { 2.2421875, 1, 1.6328125, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 6, { 2.28125, 1, 1.6015625, 0 } },
  { "Olympus", "E-M10 Mark IV", Daylight            , 7, { 2.3203125, 1, 1.5625, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -7, { 2.0390625, 1, 1.609375, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -6, { 2.0859375, 1, 1.578125, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -5, { 2.1328125, 1, 1.5546875, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -4, { 2.1796875, 1, 1.5234375, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -3, { 2.2265625, 1, 1.4921875, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -2, { 2.2734375, 1, 1.46875, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , -1, { 2.3203125, 1, 1.4375, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 0, { 2.375, 1, 1.4140625, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 1, { 2.421875, 1, 1.3828125, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 2, { 2.46875, 1, 1.3515625, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 3, { 2.515625, 1, 1.328125, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 4, { 2.5625, 1, 1.296875, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 5, { 2.609375, 1, 1.265625, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 6, { 2.65625, 1, 1.2421875, 0 } },
  { "Olympus", "E-M10 Mark IV", Shade               , 7, { 2.703125, 1, 1.2109375, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -7, { 1.875, 1, 1.859375, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -6, { 1.921875, 1, 1.828125, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -5, { 1.96875, 1, 1.7890625, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -4, { 2.0078125, 1, 1.7578125, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -3, { 2.0546875, 1, 1.7265625, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -2, { 2.09375, 1, 1.6953125, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , -1, { 2.140625, 1, 1.6640625, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 0, { 2.1875, 1, 1.6328125, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 1, { 2.2265625, 1, 1.59375, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 2, { 2.2734375, 1, 1.5625, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 3, { 2.3125, 1, 1.53125, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 4, { 2.359375, 1, 1.5, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 5, { 2.40625, 1, 1.46875, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 6, { 2.4453125, 1, 1.4296875, 0 } },
  { "Olympus", "E-M10 Mark IV", Cloudy              , 7, { 2.4921875, 1, 1.3984375, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -7, { 1.078125, 1, 3.890625, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -6, { 1.1015625, 1, 3.8203125, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -5, { 1.125, 1, 3.75, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -4, { 1.15625, 1, 3.6796875, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -3, { 1.1796875, 1, 3.6171875, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -2, { 1.203125, 1, 3.546875, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , -1, { 1.2265625, 1, 3.4765625, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 0, { 1.2578125, 1, 3.4140625, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 1, { 1.28125, 1, 3.34375, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 2, { 1.3046875, 1, 3.2734375, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 3, { 1.328125, 1, 3.203125, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 4, { 1.3515625, 1, 3.140625, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 5, { 1.3828125, 1, 3.0703125, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 6, { 1.40625, 1, 3.0, 0 } },
  { "Olympus", "E-M10 Mark IV", Tungsten            , 7, { 1.4296875, 1, 2.9296875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -7, { 1.671875, 1, 2.9765625, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -6, { 1.7109375, 1, 2.9296875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -5, { 1.75, 1, 2.875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -4, { 1.7890625, 1, 2.8203125, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -3, { 1.828125, 1, 2.7734375, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -2, { 1.8671875, 1, 2.71875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, -1, { 1.90625, 1, 2.6640625, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 0, { 1.9453125, 1, 2.6171875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 1, { 1.9765625, 1, 2.5625, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 2, { 2.015625, 1, 2.5078125, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 3, { 2.0546875, 1, 2.453125, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 4, { 2.09375, 1, 2.40625, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 5, { 2.1328125, 1, 2.3515625, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 6, { 2.171875, 1, 2.296875, 0 } },
  { "Olympus", "E-M10 Mark IV", CoolWhiteFluorescent, 7, { 2.2109375, 1, 2.25, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -7, { 1.9921875, 1, 1.765625, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -6, { 2.0390625, 1, 1.734375, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -5, { 2.0859375, 1, 1.703125, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -4, { 2.1328125, 1, 1.671875, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -3, { 2.1796875, 1, 1.640625, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -2, { 2.2265625, 1, 1.609375, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , -1, { 2.2734375, 1, 1.578125, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 0, { 2.3203125, 1, 1.5546875, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 1, { 2.359375, 1, 1.5234375, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 2, { 2.40625, 1, 1.4921875, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 3, { 2.453125, 1, 1.4609375, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 4, { 2.5, 1, 1.4296875, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 5, { 2.546875, 1, 1.3984375, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 6, { 2.59375, 1, 1.3671875, 0 } },
  { "Olympus", "E-M10 Mark IV", Flash               , 7, { 2.640625, 1, 1.3359375, 0 } },
  { "Olympus", "E-M10 Mark IV", "3000K"             , 0, { 1.2578125, 1, 3.4140625, 0 } },
  { "Olympus", "E-M10 Mark IV", "3300K"             , 0, { 1.5, 1, 2.625, 0 } },
  { "Olympus", "E-M10 Mark IV", "3600K"             , 0, { 1.609375, 1, 2.3984375, 0 } },
  { "Olympus", "E-M10 Mark IV", "3900K"             , 0, { 1.6953125, 1, 2.2265625, 0 } },
  { "Olympus", "E-M10 Mark IV", "4000K"             , 0, { 1.9453125, 1, 2.6171875, 0 } },
  { "Olympus", "E-M10 Mark IV", "4300K"             , 0, { 1.796875, 1, 2.1328125, 0 } },
  { "Olympus", "E-M10 Mark IV", "4500K"             , 0, { 1.9765625, 1, 2.203125, 0 } },
  { "Olympus", "E-M10 Mark IV", "4800K"             , 0, { 1.921875, 1, 1.9609375, 0 } },
  { "Olympus", "E-M10 Mark IV", "5300K"             , 0, { 2.0390625, 1, 1.8203125, 0 } },
  { "Olympus", "E-M10 Mark IV", "6000K"             , 0, { 2.1875, 1, 1.6328125, 0 } },
  { "Olympus", "E-M10 Mark IV", "6600K"             , 0, { 2.296875, 1, 1.8125, 0 } },

  { "Olympus", "E-P1", Daylight, 0,		{ 1.835938, 1, 1.351563, 0 } },
  { "Olympus", "E-P1", Shade, 0,		{ 2.195313, 1, 1.046875, 0 } },
  { "Olympus", "E-P1", Cloudy, 0,		{ 2.031250, 1, 1.203125, 0 } },
  { "Olympus", "E-P1", Incandescent, 0,		{ 1.078125, 1, 2.570312, 0 } },
  { "Olympus", "E-P1", WhiteFluorescent, 0,	{ 1.695313, 1, 1.937500, 0 } },
  { "Olympus", "E-P1", NeutralFluorescent, 0,	{ 1.687500, 1, 1.703125, 0 } },
  { "Olympus", "E-P1", DaylightFluorescent, 0,	{ 2.070312, 1, 1.312500, 0 } },

  { "Olympus", "E-P2", Daylight, -7,		{ 1.789063, 1, 1.789063, 0 } },
  { "Olympus", "E-P2", Daylight, -6,		{ 1.789063, 1, 1.726563, 0 } },
  { "Olympus", "E-P2", Daylight, -5,		{ 1.804688, 1, 1.664063, 0 } },
  { "Olympus", "E-P2", Daylight, -4,		{ 1.812500, 1, 1.609375, 0 } },
  { "Olympus", "E-P2", Daylight, -3,		{ 1.812500, 1, 1.546875, 0 } },
  { "Olympus", "E-P2", Daylight, -2,		{ 1.812500, 1, 1.492188, 0 } },
  { "Olympus", "E-P2", Daylight, -1,		{ 1.820313, 1, 1.429688, 0 } },
  { "Olympus", "E-P2", Daylight, 0,		{ 1.828125, 1, 1.382813, 0 } },
  { "Olympus", "E-P2", Daylight, 1,		{ 1.820313, 1, 1.320313, 0 } },
  { "Olympus", "E-P2", Daylight, 2,		{ 1.820313, 1, 1.265625, 0 } },
  { "Olympus", "E-P2", Daylight, 3,		{ 1.820313, 1, 1.218750, 0 } },
  { "Olympus", "E-P2", Daylight, 4,		{ 1.804688, 1, 1.164063, 0 } },
  { "Olympus", "E-P2", Daylight, 5,		{ 1.804688, 1, 1.117188, 0 } },
  { "Olympus", "E-P2", Daylight, 6,		{ 1.796875, 1, 1.062500, 0 } },
  { "Olympus", "E-P2", Daylight, 7,		{ 1.781250, 1, 1.015625, 0 } },
  { "Olympus", "E-P2", Shade, -7,		{ 2.125000, 1, 1.382813, 0 } },
  { "Olympus", "E-P2", Shade, -6,		{ 2.132813, 1, 1.335938, 0 } },
  { "Olympus", "E-P2", Shade, -5,		{ 2.148438, 1, 1.289063, 0 } },
  { "Olympus", "E-P2", Shade, -4,		{ 2.148438, 1, 1.234375, 0 } },
  { "Olympus", "E-P2", Shade, -3,		{ 2.156250, 1, 1.195313, 0 } },
  { "Olympus", "E-P2", Shade, -2,		{ 2.156250, 1, 1.148438, 0 } },
  { "Olympus", "E-P2", Shade, -1,		{ 2.164063, 1, 1.101563, 0 } },
  { "Olympus", "E-P2", Shade, 0,		{ 2.171875, 1, 1.070313, 0 } },
  { "Olympus", "E-P2", Shade, 1,		{ 2.164063, 1, 1.023438, 0 } },
  { "Olympus", "E-P2", Shade, 2,		{ 2.164063, 1, 0.976563, 0 } },
  { "Olympus", "E-P2", Shade, 3,		{ 2.156250, 1, 0.937500, 0 } },
  { "Olympus", "E-P2", Shade, 4,		{ 2.156250, 1, 0.898438, 0 } },
  { "Olympus", "E-P2", Shade, 5,		{ 2.140625, 1, 0.859375, 0 } },
  { "Olympus", "E-P2", Shade, 6,		{ 2.132813, 1, 0.820313, 0 } },
  { "Olympus", "E-P2", Shade, 7,		{ 2.117188, 1, 0.781250, 0 } },
  { "Olympus", "E-P2", Cloudy, -7,		{ 1.953125, 1, 1.617188, 0 } },
  { "Olympus", "E-P2", Cloudy, -6,		{ 1.968750, 1, 1.562500, 0 } },
  { "Olympus", "E-P2", Cloudy, -5,		{ 1.976563, 1, 1.507813, 0 } },
  { "Olympus", "E-P2", Cloudy, -4,		{ 1.976563, 1, 1.445313, 0 } },
  { "Olympus", "E-P2", Cloudy, -3,		{ 1.984375, 1, 1.398438, 0 } },
  { "Olympus", "E-P2", Cloudy, -2,		{ 1.984375, 1, 1.343750, 0 } },
  { "Olympus", "E-P2", Cloudy, -1,		{ 1.992188, 1, 1.296875, 0 } },
  { "Olympus", "E-P2", Cloudy, 0,		{ 2.000000, 1, 1.250000, 0 } },
  { "Olympus", "E-P2", Cloudy, 1,		{ 1.992188, 1, 1.187500, 0 } },
  { "Olympus", "E-P2", Cloudy, 2,		{ 1.992188, 1, 1.140625, 0 } },
  { "Olympus", "E-P2", Cloudy, 3,		{ 1.984375, 1, 1.101563, 0 } },
  { "Olympus", "E-P2", Cloudy, 4,		{ 1.976563, 1, 1.054688, 0 } },
  { "Olympus", "E-P2", Cloudy, 5,		{ 1.968750, 1, 1.007813, 0 } },
  { "Olympus", "E-P2", Cloudy, 6,		{ 1.960938, 1, 0.960938, 0 } },
  { "Olympus", "E-P2", Cloudy, 7,		{ 1.953125, 1, 0.914063, 0 } },
  { "Olympus", "E-P2", Incandescent, -7,	{ 1.039063, 1, 3.445313, 0 } },
  { "Olympus", "E-P2", Incandescent, -6,	{ 1.046875, 1, 3.320313, 0 } },
  { "Olympus", "E-P2", Incandescent, -5,	{ 1.054688, 1, 3.210938, 0 } },
  { "Olympus", "E-P2", Incandescent, -4,	{ 1.062500, 1, 3.093750, 0 } },
  { "Olympus", "E-P2", Incandescent, -3,	{ 1.054688, 1, 2.976563, 0 } },
  { "Olympus", "E-P2", Incandescent, -2,	{ 1.062500, 1, 2.867188, 0 } },
  { "Olympus", "E-P2", Incandescent, -1,	{ 1.062500, 1, 2.750000, 0 } },
  { "Olympus", "E-P2", Incandescent, 0,		{ 1.070313, 1, 2.656250, 0 } },
  { "Olympus", "E-P2", Incandescent, 1,		{ 1.062500, 1, 2.546875, 0 } },
  { "Olympus", "E-P2", Incandescent, 2,		{ 1.062500, 1, 2.437500, 0 } },
  { "Olympus", "E-P2", Incandescent, 3,		{ 1.062500, 1, 2.335938, 0 } },
  { "Olympus", "E-P2", Incandescent, 4,		{ 1.054688, 1, 2.242188, 0 } },
  { "Olympus", "E-P2", Incandescent, 5,		{ 1.054688, 1, 2.148438, 0 } },
  { "Olympus", "E-P2", Incandescent, 6,		{ 1.046875, 1, 2.054688, 0 } },
  { "Olympus", "E-P2", Incandescent, 7,		{ 1.046875, 1, 1.960938, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -7,	{ 1.742188, 1, 2.617188, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -6,	{ 1.750000, 1, 2.515625, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -5,	{ 1.757813, 1, 2.429688, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -4,	{ 1.757813, 1, 2.343750, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -3,	{ 1.765625, 1, 2.257813, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -2,	{ 1.765625, 1, 2.171875, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, -1,	{ 1.773438, 1, 2.093750, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 0,	{ 1.781250, 1, 2.015625, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 1,	{ 1.773438, 1, 1.921875, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 2,	{ 1.773438, 1, 1.851563, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 3,	{ 1.765625, 1, 1.773438, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 4,	{ 1.765625, 1, 1.703125, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 5,	{ 1.757813, 1, 1.625000, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 6,	{ 1.750000, 1, 1.554688, 0 } },
  { "Olympus", "E-P2", WhiteFluorescent, 7,	{ 1.734375, 1, 1.484375, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -7,	{ 1.695313, 1, 2.328125, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -6,	{ 1.710938, 1, 2.242188, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -5,	{ 1.718750, 1, 2.171875, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -4,	{ 1.726563, 1, 2.085938, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -3,	{ 1.726563, 1, 2.007813, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -2,	{ 1.734375, 1, 1.937500, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, -1,	{ 1.734375, 1, 1.859375, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 0,	{ 1.742188, 1, 1.796875, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 1,	{ 1.734375, 1, 1.718750, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 2,	{ 1.726563, 1, 1.648438, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 3,	{ 1.726563, 1, 1.585938, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 4,	{ 1.718750, 1, 1.515625, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 5,	{ 1.718750, 1, 1.453125, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 6,	{ 1.710938, 1, 1.382813, 0 } },
  { "Olympus", "E-P2", NeutralFluorescent, 7,	{ 1.703125, 1, 1.320313, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -7,	{ 2.078125, 1, 1.765625, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -6,	{ 2.093750, 1, 1.710938, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -5,	{ 2.101563, 1, 1.648438, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -4,	{ 2.117188, 1, 1.593750, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -3,	{ 2.117188, 1, 1.531250, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -2,	{ 2.125000, 1, 1.476563, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, -1,	{ 2.125000, 1, 1.414063, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 0,	{ 2.132813, 1, 1.367188, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 1,	{ 2.125000, 1, 1.304688, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 2,	{ 2.117188, 1, 1.257813, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 3,	{ 2.117188, 1, 1.203125, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 4,	{ 2.109375, 1, 1.156250, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 5,	{ 2.109375, 1, 1.101563, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 6,	{ 2.093750, 1, 1.054688, 0 } },
  { "Olympus", "E-P2", DaylightFluorescent, 7,	{ 2.085938, 1, 1.007813, 0 } },
  { "Olympus", "E-P2", Flash, -7,		{ 1.960938, 1, 1.609375, 0 } },
  { "Olympus", "E-P2", Flash, -6,		{ 1.976563, 1, 1.554688, 0 } },
  { "Olympus", "E-P2", Flash, -5,		{ 1.984375, 1, 1.492188, 0 } },
  { "Olympus", "E-P2", Flash, -4,		{ 1.984375, 1, 1.437500, 0 } },
  { "Olympus", "E-P2", Flash, -3,		{ 1.992188, 1, 1.390625, 0 } },
  { "Olympus", "E-P2", Flash, -2,		{ 1.992188, 1, 1.335938, 0 } },
  { "Olympus", "E-P2", Flash, -1,		{ 2.000000, 1, 1.289063, 0 } },
  { "Olympus", "E-P2", Flash, 0,		{ 2.007813, 1, 1.242188, 0 } },
  { "Olympus", "E-P2", Flash, 1,		{ 2.000000, 1, 1.179688, 0 } },
  { "Olympus", "E-P2", Flash, 2,		{ 2.000000, 1, 1.132813, 0 } },
  { "Olympus", "E-P2", Flash, 3,		{ 1.992188, 1, 1.093750, 0 } },
  { "Olympus", "E-P2", Flash, 4,		{ 1.984375, 1, 1.046875, 0 } },
  { "Olympus", "E-P2", Flash, 5,		{ 1.976563, 1, 1.000000, 0 } },
  { "Olympus", "E-P2", Flash, 6,		{ 1.968750, 1, 0.953125, 0 } },
  { "Olympus", "E-P2", Flash, 7,		{ 1.960938, 1, 0.906250, 0 } },

  { "Olympus", "E-P3", Daylight, 0,		{ 2.0469, 1, 1.4922, 0 } },
  { "Olympus", "E-P3", Shade, 0,		{ 2.4375, 1, 1.1875, 0 } },
  { "Olympus", "E-P3", Cloudy, 0,		{ 2.2188, 1, 1.3750, 0 } },
  { "Olympus", "E-P3", Incandescent, 0,		{ 1.2266, 1, 2.5312, 0 } },
  { "Olympus", "E-P3", Fluorescent, 0,		{ 1.9766, 1, 1.9766, 0 } },
  { "Olympus", "E-P3", Flash, 0,		{ 2.2109, 1, 1.3672, 0 } },

  /* Firmware version 1.5 */
  { "Olympus", "E-P5", Daylight, 0,		{ 1.945313, 1, 1.796875, 0 } },
  { "Olympus", "E-P5", Shade, 0,		{ 2.273438, 1, 1.40625, 0 } },
  { "Olympus", "E-P5", Cloudy, 0,		{ 2.09375, 1, 1.625, 0 } },
  { "Olympus", "E-P5", Incandescent, 0,	{ 1.203125, 1, 3.21875, 0 } },
  { "Olympus", "E-P5", Fluorescent, 0,		{ 1.890625, 1, 2.585938, 0 } },
  { "Olympus", "E-P5", Flash, 0,		{ 2.21875, 1, 1.53125, 0 } },

  /* Firmware version 1.2 */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-PL1", Daylight, -7,		{ 1.492188, 1, 1.531250, 0 } },
  { "Olympus", "E-PL1", Daylight, 0,		{ 1.726562, 1, 1.343750, 0 } },
  { "Olympus", "E-PL1", Daylight, 7,		{ 1.984375, 1, 1.148438, 0 } },
  { "Olympus", "E-PL1", Shade, -7,		{ 1.796875, 1, 1.171875, 0 } },
  { "Olympus", "E-PL1", Shade, 0,		{ 2.085937, 1, 1.039062, 0 } },
  { "Olympus", "E-PL1", Shade, 7,		{ 2.699115, 1.132743, 1, 0 } },
  { "Olympus", "E-PL1", Cloudy, -7,		{ 1.640625, 1, 1.367187, 0 } },
  { "Olympus", "E-PL1", Cloudy, 0,		{ 1.906250, 1, 1.210938, 0 } },
  { "Olympus", "E-PL1", Cloudy, 7,		{ 2.179688, 1, 1.031250, 0 } },
  { "Olympus", "E-PL1", Incandescent, -7,	{ 1, 1.174312, 3.467890, 0 } },
  { "Olympus", "E-PL1", Incandescent, 0,	{ 1, 1.015873, 2.619048, 0 } },
  { "Olympus", "E-PL1", Incandescent, 7,	{ 1.125000, 1, 2.226562, 0 } },
  { "Olympus", "E-PL1", WhiteFluorescent, -7,	{ 1.421875, 1, 2.234375, 0 } },
  { "Olympus", "E-PL1", WhiteFluorescent, 0,	{ 1.648437, 1, 1.960937, 0 } },
  { "Olympus", "E-PL1", WhiteFluorescent, 7,	{ 1.882812, 1, 1.679687, 0 } },
  { "Olympus", "E-PL1", NeutralFluorescent, -7,	{ 1.390625, 1, 1.945313, 0 } },
  { "Olympus", "E-PL1", NeutralFluorescent, 0,	{ 1.609375, 1, 1.710938, 0 } },
  { "Olympus", "E-PL1", NeutralFluorescent, 7,	{ 1.851563, 1, 1.468750, 0 } },
  { "Olympus", "E-PL1", DaylightFluorescent, -7, { 1.750000, 1, 1.507812, 0 } },
  { "Olympus", "E-PL1", DaylightFluorescent, 0,	{ 2.015625, 1, 1.328125, 0 } },
  { "Olympus", "E-PL1", DaylightFluorescent, 7,	{ 2.320313, 1.140625, 1, 0 } },
  { "Olympus", "E-PL1", Flash, -7,		{ 1.640625, 1, 1.367187, 0 } },
  { "Olympus", "E-PL1", Flash, 0,		{ 1.898438, 1, 1.203125, 0 } },
  { "Olympus", "E-PL1", Flash, 7,		{ 2.179688, 1, 1.031250, 0 } },
  { "Olympus", "E-PL1", "2700K", 0,		{ 1, 1.075630, 2.924370, 0 } },
  { "Olympus", "E-PL1", "3000K", 0,		{ 1.085937, 1, 2.312500, 0 } },
  { "Olympus", "E-PL1", "3300K", 0,		{ 1.218750, 1, 1.976562, 0 } },
  { "Olympus", "E-PL1", "5000K", 0,		{ 1.656250, 1, 1.414062, 0 } },

  { "Olympus", "E-PL2", Daylight, 0,		{ 1.4609, 1, 1.4219, 0 } },
  { "Olympus", "E-PL2", Shade, 0,		{ 1.7422, 1, 1.1094, 0 } },
  { "Olympus", "E-PL2", Cloudy, 0,		{ 1.6172, 1, 1.2891, 0 } },
  { "Olympus", "E-PL2", Tungsten, 0,		{ 1, 1.1327, 2.9115, 0 } },
  { "Olympus", "E-PL2", Fluorescent, 0,		{ 1.3828, 1, 2.0859, 0 } },
  { "Olympus", "E-PL2", Flash, 0,		{ 1.6016, 1, 1.2891, 0 } },
  { "Olympus", "E-PL2", "2700K", 0,		{ 1, 1.1852, 3.1852, 0 } },
  { "Olympus", "E-PL2", "3000K", 0,		{ 1, 1.0492, 2.4672, 0 } },
  { "Olympus", "E-PL2", "3300K", 0,		{ 1.0547, 1, 2.0625, 0 } },
  { "Olympus", "E-PL2", "5000K", 0,		{ 1.4062, 1, 1.4922, 0 } },

  /* Firmware version 1.3 */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-PL3", Daylight, -7,		{ 1.695313, 1, 1.656250, 0 } },
  { "Olympus", "E-PL3", Daylight, 0,		{ 1.976562, 1, 1.453125, 0 } },
  { "Olympus", "E-PL3", Daylight, 7,		{ 2.250000, 1, 1.242187, 0 } },
  { "Olympus", "E-PL3", Shade, -7,		{ 2.046875, 1, 1.312500, 0 } },
  { "Olympus", "E-PL3", Shade, 0,		{ 2.382812, 1, 1.156250, 0 } },
  { "Olympus", "E-PL3", Shade, 7,		{ 2.732283, 1.007874, 1, 0 } },
  { "Olympus", "E-PL3", Cloudy, -7,		{ 1.859375, 1, 1.515625, 0 } },
  { "Olympus", "E-PL3", Cloudy, 0,		{ 2.164063, 1, 1.335938, 0 } },
  { "Olympus", "E-PL3", Cloudy, 7,		{ 2.460937, 1, 1.148437, 0 } },
  { "Olympus", "E-PL3", Tungsten, -7,		{ 1.007812, 1, 2.812500, 0 } },
  { "Olympus", "E-PL3", Tungsten, 0,		{ 1.171875, 1, 2.468750, 0 } },
  { "Olympus", "E-PL3", Tungsten, 7,		{ 1.335937, 1, 2.117187, 0 } },
  { "Olympus", "E-PL3", Fluorescent, -7,	{ 1.679688, 1, 2.203125, 0 } },
  { "Olympus", "E-PL3", Fluorescent, 0,		{ 1.953125, 1, 1.937500, 0 } },
  { "Olympus", "E-PL3", Fluorescent, 7,		{ 2.226562, 1, 1.664062, 0 } },
  { "Olympus", "E-PL3", Underwater, -7,		{ 1.812500, 1, 1.789062, 0 } },
  { "Olympus", "E-PL3", Underwater, 0,		{ 2.390625, 1, 1.484375, 0 } },
  { "Olympus", "E-PL3", Underwater, 7,		{ 2.429688, 1, 1.335937, 0 } },
  { "Olympus", "E-PL3", Flash, -7,		{ 2.218750, 1, 1.421875, 0 } },
  { "Olympus", "E-PL3", Flash, 0,		{ 2.585938, 1, 1.250000, 0 } },
  { "Olympus", "E-PL3", Flash, 7,		{ 2.945313, 1, 1.070312, 0 } },
  { "Olympus", "E-PL3", "2700K", 0,		{ 1.117187, 1, 2.570312, 0 } },
  { "Olympus", "E-PL3", "3000K", 0,		{ 1.273437, 1, 2.273437, 0 } },
  { "Olympus", "E-PL3", "3300K", 0,		{ 1.414062, 1, 2.015625, 0 } },
  { "Olympus", "E-PL3", "5000K", 0,		{ 1.906250, 1, 1.515625, 0 } },

  /* Firmware version 1.2 */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-PL5", Daylight, -7,		{ 1.671875, 1, 2.164063, 0 } },
  { "Olympus", "E-PL5", Daylight, 0,		{ 1.945313, 1, 1.898438, 0 } },
  { "Olympus", "E-PL5", Daylight, 7,		{ 2.210938, 1, 1.625000, 0 } },
  { "Olympus", "E-PL5", Shade, -7,		{ 1.953125, 1, 1.687500, 0 } },
  { "Olympus", "E-PL5", Shade, 0,		{ 2.273438, 1, 1.484375, 0 } },
  { "Olympus", "E-PL5", Shade, 7,		{ 2.585938, 1, 1.273438, 0 } },
  { "Olympus", "E-PL5", Cloudy, -7,		{ 1.789063, 1, 1.945313, 0 } },
  { "Olympus", "E-PL5", Cloudy, 0,		{ 2.085938, 1, 1.710938, 0 } },
  { "Olympus", "E-PL5", Cloudy, 7,		{ 2.375000, 1, 1.468750, 0 } },
  { "Olympus", "E-PL5", Tungsten, -7,		{ 1.023438, 1, 3.851563, 0 } },
  { "Olympus", "E-PL5", Tungsten, 0,		{ 1.195313, 1, 3.382813, 0 } },
  { "Olympus", "E-PL5", Tungsten, 7,		{ 1.359375, 1, 2.906250, 0 } },
  { "Olympus", "E-PL5", Fluorescent, -7,	{ 1.625000, 1, 3.101563, 0 } },
  { "Olympus", "E-PL5", Fluorescent, 0,		{ 1.890625, 1, 2.726563, 0 } },
  { "Olympus", "E-PL5", Fluorescent, 7,		{ 2.148438, 1, 2.343750, 0 } },
  { "Olympus", "E-PL5", Flash, -7,		{ 1.906250, 1, 1.835938, 0 } },
  { "Olympus", "E-PL5", Flash, 0,		{ 2.218750, 1, 1.617188, 0 } },
  { "Olympus", "E-PL5", Flash, 7,		{ 2.523438, 1, 1.390625, 0 } },
  { "Olympus", "E-PL5", "2700K", 0,		{ 1.140625, 1, 3.554688, 0 } },
  { "Olympus", "E-PL5", "3000K", 0,		{ 1.296875, 1, 3.054688, 0 } },
  { "Olympus", "E-PL5", "3300K", 0,		{ 1.429688, 1, 2.640625, 0 } },
  { "Olympus", "E-PL5", "5000K", 0,		{ 1.882813, 1, 1.984375, 0 } },

  { "Olympus", "E-PL6", Daylight            , -7, { 1.65625, 1, 2.171875, 0 } },
  { "Olympus", "E-PL6", Daylight            , -6, { 1.6953125, 1, 2.1328125, 0 } },
  { "Olympus", "E-PL6", Daylight            , -5, { 1.734375, 1, 2.09375, 0 } },
  { "Olympus", "E-PL6", Daylight            , -4, { 1.7734375, 1, 2.0546875, 0 } },
  { "Olympus", "E-PL6", Daylight            , -3, { 1.8125, 1, 2.015625, 0 } },
  { "Olympus", "E-PL6", Daylight            , -2, { 1.8515625, 1, 1.9765625, 0 } },
  { "Olympus", "E-PL6", Daylight            , -1, { 1.890625, 1, 1.9375, 0 } },
  { "Olympus", "E-PL6", Daylight            , 0, { 1.9296875, 1, 1.90625, 0 } },
  { "Olympus", "E-PL6", Daylight            , 1, { 1.9609375, 1, 1.8671875, 0 } },
  { "Olympus", "E-PL6", Daylight            , 2, { 2.0, 1, 1.828125, 0 } },
  { "Olympus", "E-PL6", Daylight            , 3, { 2.0390625, 1, 1.7890625, 0 } },
  { "Olympus", "E-PL6", Daylight            , 4, { 2.078125, 1, 1.75, 0 } },
  { "Olympus", "E-PL6", Daylight            , 5, { 2.1171875, 1, 1.7109375, 0 } },
  { "Olympus", "E-PL6", Daylight            , 6, { 2.15625, 1, 1.671875, 0 } },
  { "Olympus", "E-PL6", Daylight            , 7, { 2.1953125, 1, 1.6328125, 0 } },
  { "Olympus", "E-PL6", Shade               , -7, { 1.9140625, 1, 1.6953125, 0 } },
  { "Olympus", "E-PL6", Shade               , -6, { 1.953125, 1, 1.6640625, 0 } },
  { "Olympus", "E-PL6", Shade               , -5, { 2.0, 1, 1.640625, 0 } },
  { "Olympus", "E-PL6", Shade               , -4, { 2.046875, 1, 1.609375, 0 } },
  { "Olympus", "E-PL6", Shade               , -3, { 2.0859375, 1, 1.578125, 0 } },
  { "Olympus", "E-PL6", Shade               , -2, { 2.1328125, 1, 1.546875, 0 } },
  { "Olympus", "E-PL6", Shade               , -1, { 2.1796875, 1, 1.515625, 0 } },
  { "Olympus", "E-PL6", Shade               , 0, { 2.2265625, 1, 1.4921875, 0 } },
  { "Olympus", "E-PL6", Shade               , 1, { 2.265625, 1, 1.4609375, 0 } },
  { "Olympus", "E-PL6", Shade               , 2, { 2.3125, 1, 1.4296875, 0 } },
  { "Olympus", "E-PL6", Shade               , 3, { 2.359375, 1, 1.3984375, 0 } },
  { "Olympus", "E-PL6", Shade               , 4, { 2.3984375, 1, 1.3671875, 0 } },
  { "Olympus", "E-PL6", Shade               , 5, { 2.4453125, 1, 1.3359375, 0 } },
  { "Olympus", "E-PL6", Shade               , 6, { 2.4921875, 1, 1.3125, 0 } },
  { "Olympus", "E-PL6", Shade               , 7, { 2.53125, 1, 1.28125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -7, { 1.765625, 1, 1.953125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -6, { 1.8046875, 1, 1.921875, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -5, { 1.84375, 1, 1.890625, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -4, { 1.8828125, 1, 1.8515625, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -3, { 1.9296875, 1, 1.8203125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -2, { 1.96875, 1, 1.78125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , -1, { 2.0078125, 1, 1.75, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 0, { 2.0546875, 1, 1.71875, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 1, { 2.09375, 1, 1.6796875, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 2, { 2.1328125, 1, 1.6484375, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 3, { 2.171875, 1, 1.609375, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 4, { 2.21875, 1, 1.578125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 5, { 2.2578125, 1, 1.546875, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 6, { 2.296875, 1, 1.5078125, 0 } },
  { "Olympus", "E-PL6", Cloudy              , 7, { 2.3359375, 1, 1.4765625, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -7, { 1.0546875, 1, 3.796875, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -6, { 1.078125, 1, 3.734375, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -5, { 1.1015625, 1, 3.6640625, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -4, { 1.125, 1, 3.6015625, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -3, { 1.1484375, 1, 3.53125, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -2, { 1.171875, 1, 3.46875, 0 } },
  { "Olympus", "E-PL6", Tungsten            , -1, { 1.1953125, 1, 3.3984375, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 0, { 1.2265625, 1, 3.3359375, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 1, { 1.25, 1, 3.265625, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 2, { 1.2734375, 1, 3.1953125, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 3, { 1.296875, 1, 3.1328125, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 4, { 1.3203125, 1, 3.0625, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 5, { 1.34375, 1, 3.0, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 6, { 1.3671875, 1, 2.9296875, 0 } },
  { "Olympus", "E-PL6", Tungsten            , 7, { 1.390625, 1, 2.8671875, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -7, { 1.6171875, 1, 3.1171875, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -6, { 1.65625, 1, 3.0625, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -5, { 1.6875, 1, 3.0078125, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -4, { 1.7265625, 1, 2.953125, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -3, { 1.765625, 1, 2.8984375, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -2, { 1.8046875, 1, 2.84375, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, -1, { 1.84375, 1, 2.7890625, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 0, { 1.8828125, 1, 2.734375, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 1, { 1.9140625, 1, 2.6796875, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 2, { 1.953125, 1, 2.625, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 3, { 1.9921875, 1, 2.5703125, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 4, { 2.03125, 1, 2.515625, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 5, { 2.0703125, 1, 2.4609375, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 6, { 2.1015625, 1, 2.40625, 0 } },
  { "Olympus", "E-PL6", CoolWhiteFluorescent, 7, { 2.140625, 1, 2.3515625, 0 } },
  { "Olympus", "E-PL6", Flash               , -7, { 1.8828125, 1, 1.8515625, 0 } },
  { "Olympus", "E-PL6", Flash               , -6, { 1.9296875, 1, 1.8125, 0 } },
  { "Olympus", "E-PL6", Flash               , -5, { 1.96875, 1, 1.78125, 0 } },
  { "Olympus", "E-PL6", Flash               , -4, { 2.015625, 1, 1.75, 0 } },
  { "Olympus", "E-PL6", Flash               , -3, { 2.0625, 1, 1.71875, 0 } },
  { "Olympus", "E-PL6", Flash               , -2, { 2.1015625, 1, 1.6875, 0 } },
  { "Olympus", "E-PL6", Flash               , -1, { 2.1484375, 1, 1.65625, 0 } },
  { "Olympus", "E-PL6", Flash               , 0, { 2.1953125, 1, 1.625, 0 } },
  { "Olympus", "E-PL6", Flash               , 1, { 2.234375, 1, 1.5859375, 0 } },
  { "Olympus", "E-PL6", Flash               , 2, { 2.28125, 1, 1.5546875, 0 } },
  { "Olympus", "E-PL6", Flash               , 3, { 2.3203125, 1, 1.5234375, 0 } },
  { "Olympus", "E-PL6", Flash               , 4, { 2.3671875, 1, 1.4921875, 0 } },
  { "Olympus", "E-PL6", Flash               , 5, { 2.4140625, 1, 1.4609375, 0 } },
  { "Olympus", "E-PL6", Flash               , 6, { 2.453125, 1, 1.4296875, 0 } },
  { "Olympus", "E-PL6", Flash               , 7, { 2.5, 1, 1.390625, 0 } },
  { "Olympus", "E-PL6", "3000K"             , 0, { 1.2265625, 1, 3.3359375, 0 } },
  { "Olympus", "E-PL6", "3300K"             , 0, { 1.453125, 1, 2.6328125, 0 } },
  { "Olympus", "E-PL6", "3600K"             , 0, { 1.5390625, 1, 2.4140625, 0 } },
  { "Olympus", "E-PL6", "3900K"             , 0, { 1.6171875, 1, 2.25, 0 } },
  { "Olympus", "E-PL6", "4000K"             , 0, { 1.8828125, 1, 2.734375, 0 } },
  { "Olympus", "E-PL6", "4300K"             , 0, { 1.71875, 1, 2.234375, 0 } },
  { "Olympus", "E-PL6", "4500K"             , 0, { 2.0, 1, 2.2734375, 0 } },
  { "Olympus", "E-PL6", "4800K"             , 0, { 1.828125, 1, 2.046875, 0 } },
  { "Olympus", "E-PL6", "5300K"             , 0, { 1.9296875, 1, 1.90625, 0 } },
  { "Olympus", "E-PL6", "6000K"             , 0, { 2.0546875, 1, 1.71875, 0 } },
  { "Olympus", "E-PL6", "6600K"             , 0, { 2.140625, 1, 1.859375, 0 } },
  { "Olympus", "E-PL6", "7500K"             , 0, { 2.2265625, 1, 1.4921875, 0 } },


  { "Olympus", "E-PL7", Daylight, 0,		{ 1.890625, 1, 1.773438, 0 } },
  { "Olympus", "E-PL7", Shade, 0,		{ 2.250000, 1, 1.359375, 0 } },
  { "Olympus", "E-PL7", Cloudy, 0,		{ 2.046875, 1, 1.601563, 0 } },
  { "Olympus", "E-PL7", Tungsten, 0,		{ 1.140625, 1, 3.585938, 0 } },
  { "Olympus", "E-PL7", Fluorescent, 0,		{ 1.882813, 1, 2.671875, 0 } },
  { "Olympus", "E-PL7", Underwater, 0,		{ 1.359375, 1, 2.632813, 0 } },
  { "Olympus", "E-PL7", Flash, 0,		{ 2.218750, 1, 1.531250, 0 } },
  { "Olympus", "E-PL7", "5000K", 0,		{ 1.828125, 1, 1.867188, 0 } },
  { "Olympus", "E-PL7", "2700K", 0,		{ 1.093750, 1, 3.781250, 0 } },

  /* Body Firmware Version : 1.0 */
  { "OLYMPUS", "E-PL8", "Sunny", 0,           { 1.859375, 1, 1.726563, 0 } }, /*5300K*/
  { "OLYMPUS", "E-PL8", "Shadow", 0,          { 2.1875, 1, 1.320313, 0 } },   /*7500K*/
  { "OLYMPUS", "E-PL8", "Cloudy", 0,          { 2, 1, 1.5625, 0 } },          /*6000K*/
  { "OLYMPUS", "E-PL8", "Incandescent", 0,    { 1.132813, 1, 3.5, 0 } },      /*3000K*/
  { "OLYMPUS", "E-PL8", "Fluorescent", 0,     { 1.835938, 1, 2.601563, 0 } }, /*4000K*/
  { "OLYMPUS", "E-PL8", "Underwater", 0,      { 1.429688, 1, 2.414063, 0 } },
  { "OLYMPUS", "E-PL8", "WB Flash", 0,        { 2.179688, 1, 1.492188, 0 } }, /*5500K*/

  /* Copied from Olympus E-PL3 presets */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-PM1", Daylight, -7,		{ 1.695313, 1, 1.656250, 0 } },
  { "Olympus", "E-PM1", Daylight, 0,		{ 1.976562, 1, 1.453125, 0 } },
  { "Olympus", "E-PM1", Daylight, 7,		{ 2.250000, 1, 1.242187, 0 } },
  { "Olympus", "E-PM1", Shade, -7,		{ 2.046875, 1, 1.312500, 0 } },
  { "Olympus", "E-PM1", Shade, 0,		{ 2.382812, 1, 1.156250, 0 } },
  { "Olympus", "E-PM1", Shade, 7,		{ 2.732283, 1.007874, 1, 0 } },
  { "Olympus", "E-PM1", Cloudy, -7,		{ 1.859375, 1, 1.515625, 0 } },
  { "Olympus", "E-PM1", Cloudy, 0,		{ 2.164063, 1, 1.335938, 0 } },
  { "Olympus", "E-PM1", Cloudy, 7,		{ 2.460937, 1, 1.148437, 0 } },
  { "Olympus", "E-PM1", Tungsten, -7,		{ 1.007812, 1, 2.812500, 0 } },
  { "Olympus", "E-PM1", Tungsten, 0,		{ 1.171875, 1, 2.468750, 0 } },
  { "Olympus", "E-PM1", Tungsten, 7,		{ 1.335937, 1, 2.117187, 0 } },
  { "Olympus", "E-PM1", Fluorescent, -7,	{ 1.679688, 1, 2.203125, 0 } },
  { "Olympus", "E-PM1", Fluorescent, 0,		{ 1.953125, 1, 1.937500, 0 } },
  { "Olympus", "E-PM1", Fluorescent, 7,		{ 2.226562, 1, 1.664062, 0 } },
  { "Olympus", "E-PM1", Underwater, -7,		{ 1.812500, 1, 1.789062, 0 } },
  { "Olympus", "E-PM1", Underwater, 0,		{ 2.390625, 1, 1.484375, 0 } },
  { "Olympus", "E-PM1", Underwater, 7,		{ 2.429688, 1, 1.335937, 0 } },
  { "Olympus", "E-PM1", Flash, -7,		{ 2.218750, 1, 1.421875, 0 } },
  { "Olympus", "E-PM1", Flash, 0,		{ 2.585938, 1, 1.250000, 0 } },
  { "Olympus", "E-PM1", Flash, 7,		{ 2.945313, 1, 1.070312, 0 } },
  { "Olympus", "E-PM1", "2700K", 0,		{ 1.117187, 1, 2.570312, 0 } },
  { "Olympus", "E-PM1", "3000K", 0,		{ 1.273437, 1, 2.273437, 0 } },
  { "Olympus", "E-PM1", "3300K", 0,		{ 1.414062, 1, 2.015625, 0 } },
  { "Olympus", "E-PM1", "5000K", 0,		{ 1.906250, 1, 1.515625, 0 } },

  /* Copied from Olympus E-PL5 presets */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "E-PM2", Daylight, -7,		{ 1.671875, 1, 2.164063, 0 } },
  { "Olympus", "E-PM2", Daylight, 0,		{ 1.945313, 1, 1.898438, 0 } },
  { "Olympus", "E-PM2", Daylight, 7,		{ 2.210938, 1, 1.625000, 0 } },
  { "Olympus", "E-PM2", Shade, -7,		{ 1.953125, 1, 1.687500, 0 } },
  { "Olympus", "E-PM2", Shade, 0,		{ 2.273438, 1, 1.484375, 0 } },
  { "Olympus", "E-PM2", Shade, 7,		{ 2.585938, 1, 1.273438, 0 } },
  { "Olympus", "E-PM2", Cloudy, -7,		{ 1.789063, 1, 1.945313, 0 } },
  { "Olympus", "E-PM2", Cloudy, 0,		{ 2.085938, 1, 1.710938, 0 } },
  { "Olympus", "E-PM2", Cloudy, 7,		{ 2.375000, 1, 1.468750, 0 } },
  { "Olympus", "E-PM2", Tungsten, -7,		{ 1.023438, 1, 3.851563, 0 } },
  { "Olympus", "E-PM2", Tungsten, 0,		{ 1.195313, 1, 3.382813, 0 } },
  { "Olympus", "E-PM2", Tungsten, 7,		{ 1.359375, 1, 2.906250, 0 } },
  { "Olympus", "E-PM2", Fluorescent, -7,	{ 1.625000, 1, 3.101563, 0 } },
  { "Olympus", "E-PM2", Fluorescent, 0,		{ 1.890625, 1, 2.726563, 0 } },
  { "Olympus", "E-PM2", Fluorescent, 7,		{ 2.148438, 1, 2.343750, 0 } },
  { "Olympus", "E-PM2", Flash, -7,		{ 1.906250, 1, 1.835938, 0 } },
  { "Olympus", "E-PM2", Flash, 0,		{ 2.218750, 1, 1.617188, 0 } },
  { "Olympus", "E-PM2", Flash, 7,		{ 2.523438, 1, 1.390625, 0 } },
  { "Olympus", "E-PM2", "2700K", 0,		{ 1.140625, 1, 3.554688, 0 } },
  { "Olympus", "E-PM2", "3000K", 0,		{ 1.296875, 1, 3.054688, 0 } },
  { "Olympus", "E-PM2", "3300K", 0,		{ 1.429688, 1, 2.640625, 0 } },
  { "Olympus", "E-PM2", "5000K", 0,		{ 1.882813, 1, 1.984375, 0 } },

  { "Olympus", "PEN-F", Daylight            , 0, { 1.820312, 1, 1.804688, 0 } },
  { "Olympus", "PEN-F", Shade               , 0, { 2.132812, 1, 1.414062, 0 } },
  { "Olympus", "PEN-F", Cloudy              , 0, { 1.960938, 1, 1.617188, 0 } },
  { "Olympus", "PEN-F", Tungsten            , 0, { 1.156250, 1, 3.265625, 0 } },
  { "Olympus", "PEN-F", CoolWhiteFluorescent, 0, { 1.765625, 1, 2.578125, 0 } },
  { "Olympus", "PEN-F", Underwater          , 0, { 1.789062, 1, 1.890625, 0 } },
  { "Olympus", "PEN-F", Flash               , 0, { 2.070312, 1, 1.539062, 0 } },

  { "Olympus", "SP500UZ", Daylight, -7,		{ 1.136719, 1, 2.359375, 0 } },
  { "Olympus", "SP500UZ", Daylight, 0,		{ 1.960937, 1, 1.585937, 0 } },
  { "Olympus", "SP500UZ", Daylight, 7,		{ 3.927660, 1.089362, 1, 0 } },
  { "Olympus", "SP500UZ", Cloudy, -7,		{ 1.191406, 1, 2.210937, 0 } },
  { "Olympus", "SP500UZ", Cloudy, 0,		{ 2.058594, 1, 1.484375, 0 } },
  { "Olympus", "SP500UZ", Cloudy, 7,		{ 4.404545, 1.163636, 1, 0 } },
  { "Olympus", "SP500UZ", EveningSun, -7,	{ 1.199219, 1, 2.214844, 0 } },
  { "Olympus", "SP500UZ", EveningSun, 0,	{ 2.074219, 1, 1.488281, 0 } },
  { "Olympus", "SP500UZ", EveningSun, 7,	{ 4.440909, 1.163636, 1, 0 } },
  { "Olympus", "SP500UZ", Tungsten, -7,		{ 1, 1.590062, 6.490683, 0 } },
  { "Olympus", "SP500UZ", Tungsten, 0,		{ 1.085937, 1, 2.742188, 0 } },
  { "Olympus", "SP500UZ", Tungsten, 7,		{ 1.996094, 1, 1.589844, 0 } },
  { "Olympus", "SP500UZ", Fluorescent, -7,	{ 1.324219, 1, 2.214844, 0 } },
  { "Olympus", "SP500UZ", Fluorescent, 0,	{ 2.285156, 1, 1.488281, 0 } },
  { "Olympus", "SP500UZ", Fluorescent, 7,	{ 4.890909, 1.163636, 1, 0 } },

  { "Olympus", "SP510UZ", Daylight, 0,		{ 1.656250, 1, 1.621094, 0 } },
  { "Olympus", "SP510UZ", Cloudy, 0,		{ 1.789063, 1, 1.546875, 0 } },
  { "Olympus", "SP510UZ", Incandescent, 0,	{ 1, 1.066667, 2.891667, 0 } },
  { "Olympus", "SP510UZ", WhiteFluorescent, 0,	{ 1.929688, 1, 1.562500, 0 } },
  { "Olympus", "SP510UZ", NeutralFluorescent, 0, { 1.644531, 1, 1.843750, 0 } },
  { "Olympus", "SP510UZ", DaylightFluorescent, 0, { 1.628906, 1, 2.210938, 0 } },

  /* Firmware version 1.002 */
  { "Olympus", "TG-4", Daylight, 0,             { 1.703125, 1, 1.695313, 0 } },
  { "Olympus", "TG-4", Cloudy, 0,               { 1.8125, 1, 1.5625, 0 } },
  { "Olympus", "TG-4", Incandescent, 0,         { 1.070313, 1, 2.765625, 0 } },
  { "Olympus", "TG-4", Fluorescent, 0,          { 1.601563, 1, 2.15625, 0 } },
  { "Olympus", "TG-4", Underwater, 0,           { 1.3125, 1, 2.257813, 0 } },

  { "Olympus", "TG-5", Daylight            , -7, { 1.4453125, 1, 2.0625, 0 } },
  { "Olympus", "TG-5", Daylight            , -6, { 1.484375, 1, 2.0234375, 0 } },
  { "Olympus", "TG-5", Daylight            , -5, { 1.515625, 1, 1.9921875, 0 } },
  { "Olympus", "TG-5", Daylight            , -4, { 1.546875, 1, 1.953125, 0 } },
  { "Olympus", "TG-5", Daylight            , -3, { 1.5859375, 1, 1.9140625, 0 } },
  { "Olympus", "TG-5", Daylight            , -2, { 1.6171875, 1, 1.8828125, 0 } },
  { "Olympus", "TG-5", Daylight            , -1, { 1.6484375, 1, 1.84375, 0 } },
  { "Olympus", "TG-5", Daylight            , 0, { 1.6875, 1, 1.8125, 0 } },
  { "Olympus", "TG-5", Daylight            , 1, { 1.71875, 1, 1.7734375, 0 } },
  { "Olympus", "TG-5", Daylight            , 2, { 1.75, 1, 1.734375, 0 } },
  { "Olympus", "TG-5", Daylight            , 3, { 1.78125, 1, 1.703125, 0 } },
  { "Olympus", "TG-5", Daylight            , 4, { 1.8203125, 1, 1.6640625, 0 } },
  { "Olympus", "TG-5", Daylight            , 5, { 1.8515625, 1, 1.625, 0 } },
  { "Olympus", "TG-5", Daylight            , 6, { 1.8828125, 1, 1.59375, 0 } },
  { "Olympus", "TG-5", Daylight            , 7, { 1.921875, 1, 1.5546875, 0 } },
  { "Olympus", "TG-5", Shade               , -7, { 1.6875, 1, 1.5625, 0 } },
  { "Olympus", "TG-5", Shade               , -6, { 1.7265625, 1, 1.5390625, 0 } },
  { "Olympus", "TG-5", Shade               , -5, { 1.765625, 1, 1.5078125, 0 } },
  { "Olympus", "TG-5", Shade               , -4, { 1.8046875, 1, 1.484375, 0 } },
  { "Olympus", "TG-5", Shade               , -3, { 1.84375, 1, 1.453125, 0 } },
  { "Olympus", "TG-5", Shade               , -2, { 1.8828125, 1, 1.4296875, 0 } },
  { "Olympus", "TG-5", Shade               , -1, { 1.921875, 1, 1.3984375, 0 } },
  { "Olympus", "TG-5", Shade               , 0, { 1.96875, 1, 1.375, 0 } },
  { "Olympus", "TG-5", Shade               , 1, { 2.0078125, 1, 1.34375, 0 } },
  { "Olympus", "TG-5", Shade               , 2, { 2.046875, 1, 1.3125, 0 } },
  { "Olympus", "TG-5", Shade               , 3, { 2.0859375, 1, 1.2890625, 0 } },
  { "Olympus", "TG-5", Shade               , 4, { 2.125, 1, 1.2578125, 0 } },
  { "Olympus", "TG-5", Shade               , 5, { 2.1640625, 1, 1.234375, 0 } },
  { "Olympus", "TG-5", Shade               , 6, { 2.203125, 1, 1.203125, 0 } },
  { "Olympus", "TG-5", Shade               , 7, { 2.2421875, 1, 1.1796875, 0 } },
  { "Olympus", "TG-5", Cloudy              , -7, { 1.53125, 1, 1.8984375, 0 } },
  { "Olympus", "TG-5", Cloudy              , -6, { 1.5625, 1, 1.8671875, 0 } },
  { "Olympus", "TG-5", Cloudy              , -5, { 1.6015625, 1, 1.8359375, 0 } },
  { "Olympus", "TG-5", Cloudy              , -4, { 1.6328125, 1, 1.8046875, 0 } },
  { "Olympus", "TG-5", Cloudy              , -3, { 1.671875, 1, 1.765625, 0 } },
  { "Olympus", "TG-5", Cloudy              , -2, { 1.703125, 1, 1.734375, 0 } },
  { "Olympus", "TG-5", Cloudy              , -1, { 1.7421875, 1, 1.703125, 0 } },
  { "Olympus", "TG-5", Cloudy              , 0, { 1.78125, 1, 1.671875, 0 } },
  { "Olympus", "TG-5", Cloudy              , 1, { 1.8125, 1, 1.6328125, 0 } },
  { "Olympus", "TG-5", Cloudy              , 2, { 1.8515625, 1, 1.6015625, 0 } },
  { "Olympus", "TG-5", Cloudy              , 3, { 1.8828125, 1, 1.5703125, 0 } },
  { "Olympus", "TG-5", Cloudy              , 4, { 1.921875, 1, 1.53125, 0 } },
  { "Olympus", "TG-5", Cloudy              , 5, { 1.953125, 1, 1.5, 0 } },
  { "Olympus", "TG-5", Cloudy              , 6, { 1.9921875, 1, 1.46875, 0 } },
  { "Olympus", "TG-5", Cloudy              , 7, { 2.0234375, 1, 1.4375, 0 } },
  { "Olympus", "TG-5", Tungsten            , -7, { 0.90625, 1, 3.5390625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -6, { 0.9296875, 1, 3.4765625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -5, { 0.953125, 1, 3.4140625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -4, { 0.9765625, 1, 3.3515625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -3, { 0.9921875, 1, 3.2890625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -2, { 1.015625, 1, 3.2265625, 0 } },
  { "Olympus", "TG-5", Tungsten            , -1, { 1.0390625, 1, 3.1640625, 0 } },
  { "Olympus", "TG-5", Tungsten            , 0, { 1.0625, 1, 3.109375, 0 } },
  { "Olympus", "TG-5", Tungsten            , 1, { 1.078125, 1, 3.046875, 0 } },
  { "Olympus", "TG-5", Tungsten            , 2, { 1.1015625, 1, 2.984375, 0 } },
  { "Olympus", "TG-5", Tungsten            , 3, { 1.125, 1, 2.921875, 0 } },
  { "Olympus", "TG-5", Tungsten            , 4, { 1.140625, 1, 2.859375, 0 } },
  { "Olympus", "TG-5", Tungsten            , 5, { 1.1640625, 1, 2.796875, 0 } },
  { "Olympus", "TG-5", Tungsten            , 6, { 1.1875, 1, 2.734375, 0 } },
  { "Olympus", "TG-5", Tungsten            , 7, { 1.2109375, 1, 2.671875, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -7, { 1.421875, 1, 2.828125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -6, { 1.453125, 1, 2.78125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -5, { 1.484375, 1, 2.7265625, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -4, { 1.5234375, 1, 2.6796875, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -3, { 1.5546875, 1, 2.6328125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -2, { 1.5859375, 1, 2.578125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, -1, { 1.6171875, 1, 2.53125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 0, { 1.65625, 1, 2.484375, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 1, { 1.6875, 1, 2.4296875, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 2, { 1.71875, 1, 2.3828125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 3, { 1.75, 1, 2.328125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 4, { 1.78125, 1, 2.28125, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 5, { 1.8203125, 1, 2.234375, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 6, { 1.8515625, 1, 2.1796875, 0 } },
  { "Olympus", "TG-5", CoolWhiteFluorescent, 7, { 1.8828125, 1, 2.1328125, 0 } },
  { "Olympus", "TG-5", Underwater          , -7, { 1.6640625, 1, 2.125, 0 } },
  { "Olympus", "TG-5", Underwater          , -6, { 1.71875, 1, 2.0703125, 0 } },
  { "Olympus", "TG-5", Underwater          , -5, { 1.7421875, 1, 2.046875, 0 } },
  { "Olympus", "TG-5", Underwater          , -4, { 1.78125, 1, 1.9921875, 0 } },
  { "Olympus", "TG-5", Underwater          , -3, { 1.8203125, 1, 1.9609375, 0 } },
  { "Olympus", "TG-5", Underwater          , -2, { 1.875, 1, 1.921875, 0 } },
  { "Olympus", "TG-5", Underwater          , -1, { 1.9140625, 1, 1.8828125, 0 } },
  { "Olympus", "TG-5", Underwater          , 0, { 1.9375, 1, 1.875, 0 } },
  { "Olympus", "TG-5", Underwater          , 1, { 1.953125, 1, 1.8359375, 0 } },
  { "Olympus", "TG-5", Underwater          , 2, { 1.9921875, 1, 1.828125, 0 } },
  { "Olympus", "TG-5", Underwater          , 3, { 2.046875, 1, 1.75, 0 } },
  { "Olympus", "TG-5", Underwater          , 4, { 2.0859375, 1, 1.6875, 0 } },
  { "Olympus", "TG-5", Underwater          , 5, { 2.125, 1, 1.6796875, 0 } },
  { "Olympus", "TG-5", Underwater          , 6, { 2.1875, 1, 1.640625, 0 } },
  { "Olympus", "TG-5", Underwater          , 7, { 2.203125, 1, 1.6015625, 0 } },
  { "Olympus", "TG-5", Flash               , -7, { 1.6953125, 1, 1.8203125, 0 } },
  { "Olympus", "TG-5", Flash               , -6, { 1.734375, 1, 1.7890625, 0 } },
  { "Olympus", "TG-5", Flash               , -5, { 1.7734375, 1, 1.7578125, 0 } },
  { "Olympus", "TG-5", Flash               , -4, { 1.8125, 1, 1.7265625, 0 } },
  { "Olympus", "TG-5", Flash               , -3, { 1.8515625, 1, 1.6953125, 0 } },
  { "Olympus", "TG-5", Flash               , -2, { 1.890625, 1, 1.6640625, 0 } },
  { "Olympus", "TG-5", Flash               , -1, { 1.9296875, 1, 1.6328125, 0 } },
  { "Olympus", "TG-5", Flash               , 0, { 1.9765625, 1, 1.6015625, 0 } },
  { "Olympus", "TG-5", Flash               , 1, { 2.015625, 1, 1.5625, 0 } },
  { "Olympus", "TG-5", Flash               , 2, { 2.0546875, 1, 1.53125, 0 } },
  { "Olympus", "TG-5", Flash               , 3, { 2.09375, 1, 1.5, 0 } },
  { "Olympus", "TG-5", Flash               , 4, { 2.1328125, 1, 1.46875, 0 } },
  { "Olympus", "TG-5", Flash               , 5, { 2.171875, 1, 1.4375, 0 } },
  { "Olympus", "TG-5", Flash               , 6, { 2.2109375, 1, 1.40625, 0 } },
  { "Olympus", "TG-5", Flash               , 7, { 2.25, 1, 1.375, 0 } },
  { "Olympus", "TG-5", "3000K"             , 0, { 1.0625, 1, 3.109375, 0 } },
  { "Olympus", "TG-5", "3300K"             , 0, { 1.265625, 1, 2.484375, 0 } },
  { "Olympus", "TG-5", "3600K"             , 0, { 1.3359375, 1, 2.28125, 0 } },
  { "Olympus", "TG-5", "3900K"             , 0, { 1.4140625, 1, 2.1171875, 0 } },
  { "Olympus", "TG-5", "4000K"             , 0, { 1.65625, 1, 2.484375, 0 } },
  { "Olympus", "TG-5", "4300K"             , 0, { 1.484375, 1, 2.140625, 0 } },
  { "Olympus", "TG-5", "4500K"             , 0, { 1.6015625, 1, 2.0859375, 0 } },
  { "Olympus", "TG-5", "4800K"             , 0, { 1.59375, 1, 1.953125, 0 } },
  { "Olympus", "TG-5", "5300K"             , 0, { 1.6875, 1, 1.8125, 0 } },
  { "Olympus", "TG-5", "6000K"             , 0, { 1.78125, 1, 1.671875, 0 } },
  { "Olympus", "TG-5", "6600K"             , 0, { 1.8125, 1, 1.6796875, 0 } },
  { "Olympus", "TG-5", "7500K"             , 0, { 1.96875, 1, 1.375, 0 } },

  /* Firmware version 1.5 */
  /* -7/+7 fine tuning is -7/+7 in amber-blue and zero in green-magenta */
  { "Olympus", "XZ-1", Daylight, -7,		{ 1.687500, 1, 2.054688, 0 } },
  { "Olympus", "XZ-1", Daylight, 0,		{ 1.968750, 1, 1.804687, 0 } },
  { "Olympus", "XZ-1", Daylight, 7,		{ 2.242187, 1, 1.546875, 0 } },
  { "Olympus", "XZ-1", Shade, -7,		{ 2.000000, 1, 1.718750, 0 } },
  { "Olympus", "XZ-1", Shade, 0,		{ 2.328125, 1, 1.507813, 0 } },
  { "Olympus", "XZ-1", Shade, 7,		{ 2.648438, 1, 1.289062, 0 } },
  { "Olympus", "XZ-1", Cloudy, -7,		{ 1.812500, 1, 1.898438, 0 } },
  { "Olympus", "XZ-1", Cloudy, 0,		{ 2.109375, 1, 1.671875, 0 } },
  { "Olympus", "XZ-1", Cloudy, 7,		{ 2.398438, 1, 1.437500, 0 } },
  { "Olympus", "XZ-1", Incandescent, -7,	{ 1.031250, 1, 3.500000, 0 } },
  { "Olympus", "XZ-1", Incandescent, 0,		{ 1.203125, 1, 3.070312, 0 } },
  { "Olympus", "XZ-1", Incandescent, 7,		{ 1.367187, 1, 2.632813, 0 } },
  { "Olympus", "XZ-1", Fluorescent, -7,		{ 1.640625, 1, 2.843750, 0 } },
  { "Olympus", "XZ-1", Fluorescent, 0,		{ 1.914062, 1, 2.500000, 0 } },
  { "Olympus", "XZ-1", Fluorescent, 7,		{ 2.179688, 1, 2.148437, 0 } },
  { "Olympus", "XZ-1", Underwater, -7,		{ 1.468750, 1, 2.296875, 0 } },
  { "Olympus", "XZ-1", Underwater, 0,		{ 1.710937, 1, 2.015625, 0 } },
  { "Olympus", "XZ-1", Underwater, 7,		{ 1.937500, 1, 1.726563, 0 } },

  // Firmware 1.001
  { "Olympus", "XZ-10", Daylight, 0,		{ 1.910156, 1, 1.679688, 0 } },
  { "Olympus", "XZ-10", Shade, 0,		{ 2.132813, 1, 1.343750, 0 } },
  { "Olympus", "XZ-10", Cloudy, 0,		{ 1.980469, 1, 1.644531, 0 } },
  { "Olympus", "XZ-10", Incandescent, 0,	{ 1.289063, 1, 2.878906, 0 } },
  { "Olympus", "XZ-10", Fluorescent, 0,		{ 1.781250, 1, 2.414063, 0 } },
  { "Olympus", "XZ-10", Flash, 0,		{ 2.289063, 1, 1.394531, 0 } },

  /* Firmware Version 1.1 */
  { "Panasonic", "DC-GX9", Daylight, 0,     { 2.23046875, 1, 1.8984375,  0 } },
  { "Panasonic", "DC-GX9", Cloudy, 0,       { 2.3671875,  1, 1.76953125, 0 } },
  { "Panasonic", "DC-GX9", Shade, 0,        { 2.51171875, 1, 1.6640625,  0 } },
  { "Panasonic", "DC-GX9", Incandescent, 0, { 1.609375,   1, 2.90234375, 0 } },
  { "Panasonic", "DC-GX9", Flash, 0,        { 2.3203125,  1, 1.8046875,  0 } },

  { "Panasonic", "DC-G9", Daylight           , -9, { 2.515625, 1, 1.6796875, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -8, { 2.48046875, 1, 1.69921875, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -7, { 2.453125, 1, 1.72265625, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -6, { 2.42578125, 1, 1.74609375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -5, { 2.390625, 1, 1.765625, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -4, { 2.3671875, 1, 1.7890625, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -3, { 2.32421875, 1, 1.8203125, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -2, { 2.27734375, 1, 1.85546875, 0 } },
  { "Panasonic", "DC-G9", Daylight           , -1, { 2.25390625, 1, 1.87890625, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 0, { 2.2265625, 1, 1.90234375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 1, { 2.19140625, 1, 1.9375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 2, { 2.1640625, 1, 1.984375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 3, { 2.12890625, 1, 2.02734375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 4, { 2.10546875, 1, 2.078125, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 5, { 2.078125, 1, 2.1171875, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 6, { 2.05078125, 1, 2.15234375, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 7, { 2.01953125, 1, 2.20703125, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 8, { 1.97265625, 1, 2.25, 0 } },
  { "Panasonic", "DC-G9", Daylight           , 9, { 1.94921875, 1, 2.3046875, 0 } },
  { "Panasonic", "DC-G9", D55                , 0, { 2.12109375, 1, 1.9375, 0 } },
  { "Panasonic", "DC-G9", Shade              , -9, { 2.8125, 1, 1.4765625, 0 } },
  { "Panasonic", "DC-G9", Shade              , -8, { 2.7734375, 1, 1.49609375, 0 } },
  { "Panasonic", "DC-G9", Shade              , -7, { 2.7421875, 1, 1.515625, 0 } },
  { "Panasonic", "DC-G9", Shade              , -6, { 2.71484375, 1, 1.53515625, 0 } },
  { "Panasonic", "DC-G9", Shade              , -5, { 2.67578125, 1, 1.5546875, 0 } },
  { "Panasonic", "DC-G9", Shade              , -4, { 2.64453125, 1, 1.57421875, 0 } },
  { "Panasonic", "DC-G9", Shade              , -3, { 2.59765625, 1, 1.6015625, 0 } },
  { "Panasonic", "DC-G9", Shade              , -2, { 2.546875, 1, 1.6328125, 0 } },
  { "Panasonic", "DC-G9", Shade              , -1, { 2.51953125, 1, 1.65234375, 0 } },
  { "Panasonic", "DC-G9", Shade              , 0, { 2.4921875, 1, 1.671875, 0 } },
  { "Panasonic", "DC-G9", Shade              , 1, { 2.453125, 1, 1.70703125, 0 } },
  { "Panasonic", "DC-G9", Shade              , 2, { 2.421875, 1, 1.74609375, 0 } },
  { "Panasonic", "DC-G9", Shade              , 3, { 2.3828125, 1, 1.78515625, 0 } },
  { "Panasonic", "DC-G9", Shade              , 4, { 2.35546875, 1, 1.828125, 0 } },
  { "Panasonic", "DC-G9", Shade              , 5, { 2.32421875, 1, 1.86328125, 0 } },
  { "Panasonic", "DC-G9", Shade              , 6, { 2.296875, 1, 1.89453125, 0 } },
  { "Panasonic", "DC-G9", Shade              , 7, { 2.2578125, 1, 1.94140625, 0 } },
  { "Panasonic", "DC-G9", Shade              , 8, { 2.20703125, 1, 1.98046875, 0 } },
  { "Panasonic", "DC-G9", Shade              , 9, { 2.1796875, 1, 2.02734375, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -9, { 2.6640625, 1, 1.5625, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -8, { 2.62890625, 1, 1.58203125, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -7, { 2.6015625, 1, 1.60546875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -6, { 2.57421875, 1, 1.625, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -5, { 2.53515625, 1, 1.64453125, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -4, { 2.5078125, 1, 1.66796875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -3, { 2.4609375, 1, 1.6953125, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -2, { 2.41796875, 1, 1.73046875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , -1, { 2.390625, 1, 1.75, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 0, { 2.36328125, 1, 1.76953125, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 1, { 2.32421875, 1, 1.8046875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 2, { 2.296875, 1, 1.84765625, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 3, { 2.26171875, 1, 1.88671875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 4, { 2.23046875, 1, 1.9375, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 5, { 2.203125, 1, 1.97265625, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 6, { 2.17578125, 1, 2.0078125, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 7, { 2.140625, 1, 2.0546875, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 8, { 2.09375, 1, 2.09765625, 0 } },
  { "Panasonic", "DC-G9", Cloudy             , 9, { 2.06640625, 1, 2.14453125, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -9, { 1.83203125, 1, 2.5546875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -8, { 1.80859375, 1, 2.5859375, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -7, { 1.7890625, 1, 2.62109375, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -6, { 1.76953125, 1, 2.65625, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -5, { 1.7421875, 1, 2.6875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -4, { 1.7265625, 1, 2.72265625, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -3, { 1.69140625, 1, 2.76953125, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -2, { 1.66015625, 1, 2.82421875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , -1, { 1.640625, 1, 2.859375, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 0, { 1.625, 1, 2.89453125, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 1, { 1.59765625, 1, 2.94921875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 2, { 1.578125, 1, 3.015625, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 3, { 1.5546875, 1, 3.0859375, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 4, { 1.53515625, 1, 3.1640625, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 5, { 1.515625, 1, 3.21875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 6, { 1.49609375, 1, 3.27734375, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 7, { 1.47265625, 1, 3.35546875, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 8, { 1.4375, 1, 3.42578125, 0 } },
  { "Panasonic", "DC-G9", Incandescent       , 9, { 1.421875, 1, 3.50390625, 0 } },
  { "Panasonic", "DC-G9", Flash              , -9, { 2.5546875, 1, 1.6171875, 0 } },
  { "Panasonic", "DC-G9", Flash              , -8, { 2.51953125, 1, 1.63671875, 0 } },
  { "Panasonic", "DC-G9", Flash              , -7, { 2.4921875, 1, 1.66015625, 0 } },
  { "Panasonic", "DC-G9", Flash              , -6, { 2.46484375, 1, 1.6796875, 0 } },
  { "Panasonic", "DC-G9", Flash              , -5, { 2.4296875, 1, 1.703125, 0 } },
  { "Panasonic", "DC-G9", Flash              , -4, { 2.40234375, 1, 1.72265625, 0 } },
  { "Panasonic", "DC-G9", Flash              , -3, { 2.359375, 1, 1.75390625, 0 } },
  { "Panasonic", "DC-G9", Flash              , -2, { 2.31640625, 1, 1.7890625, 0 } },
  { "Panasonic", "DC-G9", Flash              , -1, { 2.2890625, 1, 1.80859375, 0 } },
  { "Panasonic", "DC-G9", Flash              , 0, { 2.26171875, 1, 1.83203125, 0 } },
  { "Panasonic", "DC-G9", Flash              , 1, { 2.2265625, 1, 1.8671875, 0 } },
  { "Panasonic", "DC-G9", Flash              , 2, { 2.19921875, 1, 1.91015625, 0 } },
  { "Panasonic", "DC-G9", Flash              , 3, { 2.1640625, 1, 1.953125, 0 } },
  { "Panasonic", "DC-G9", Flash              , 4, { 2.13671875, 1, 2.00390625, 0 } },
  { "Panasonic", "DC-G9", Flash              , 5, { 2.11328125, 1, 2.0390625, 0 } },
  { "Panasonic", "DC-G9", Flash              , 6, { 2.0859375, 1, 2.07421875, 0 } },
  { "Panasonic", "DC-G9", Flash              , 7, { 2.05078125, 1, 2.125, 0 } },
  { "Panasonic", "DC-G9", Flash              , 8, { 2.00390625, 1, 2.16796875, 0 } },
  { "Panasonic", "DC-G9", Flash              , 9, { 1.98046875, 1, 2.21875, 0 } },

  { "Panasonic", "DC-FZ10002", DirectSunlight, 0, { 2.51953125, 1, 1.98046875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -9, { 2.84375, 1, 1.75, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -8, { 2.80859375, 1, 1.77734375, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -7, { 2.7734375, 1, 1.80078125, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -6, { 2.734375, 1, 1.828125, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -5, { 2.69921875, 1, 1.85546875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -4, { 2.6640625, 1, 1.87890625, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -3, { 2.625, 1, 1.90625, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -2, { 2.58984375, 1, 1.9296875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , -1, { 2.5546875, 1, 1.95703125, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 0, { 2.51953125, 1, 1.98046875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 1, { 2.484375, 1, 2.02734375, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 2, { 2.44921875, 1, 2.07421875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 3, { 2.4140625, 1, 2.12109375, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 4, { 2.37890625, 1, 2.16796875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 5, { 2.34375, 1, 2.21484375, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 6, { 2.30859375, 1, 2.26171875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 7, { 2.2734375, 1, 2.30859375, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 8, { 2.23828125, 1, 2.35546875, 0 } },
  { "Panasonic", "DC-FZ10002", Daylight      , 9, { 2.203125, 1, 2.3984375, 0 } },
  { "Panasonic", "DC-FZ10002", D55           , 0, { 2.31640625, 1, 2.12109375, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -9, { 3.30859375, 1, 1.48046875, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -8, { 3.265625, 1, 1.50390625, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -7, { 3.22265625, 1, 1.5234375, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -6, { 3.1796875, 1, 1.546875, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -5, { 3.140625, 1, 1.5703125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -4, { 3.09765625, 1, 1.58984375, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -3, { 3.0546875, 1, 1.61328125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -2, { 3.01171875, 1, 1.6328125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , -1, { 2.97265625, 1, 1.65625, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 0, { 2.9296875, 1, 1.67578125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 1, { 2.890625, 1, 1.71484375, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 2, { 2.84765625, 1, 1.7578125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 3, { 2.8046875, 1, 1.796875, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 4, { 2.765625, 1, 1.8359375, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 5, { 2.7265625, 1, 1.875, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 6, { 2.6875, 1, 1.9140625, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 7, { 2.64453125, 1, 1.953125, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 8, { 2.6015625, 1, 1.9921875, 0 } },
  { "Panasonic", "DC-FZ10002", Shade         , 9, { 2.5625, 1, 2.03125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -9, { 3.07421875, 1, 1.609375, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -8, { 3.03515625, 1, 1.63671875, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -7, { 2.99609375, 1, 1.65625, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -6, { 2.95703125, 1, 1.6796875, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -5, { 2.91796875, 1, 1.70703125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -4, { 2.87890625, 1, 1.73046875, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -3, { 2.83984375, 1, 1.75390625, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -2, { 2.80078125, 1, 1.77734375, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , -1, { 2.76171875, 1, 1.80078125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 0, { 2.72265625, 1, 1.82421875, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 1, { 2.68359375, 1, 1.8671875, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 2, { 2.6484375, 1, 1.91015625, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 3, { 2.609375, 1, 1.953125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 4, { 2.5703125, 1, 1.99609375, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 5, { 2.53515625, 1, 2.0390625, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 6, { 2.49609375, 1, 2.08203125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 7, { 2.45703125, 1, 2.125, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 8, { 2.41796875, 1, 2.1640625, 0 } },
  { "Panasonic", "DC-FZ10002", Cloudy        , 9, { 2.3828125, 1, 2.20703125, 0 } },
  { "Panasonic", "DC-FZ10002", Tungsten      , 0, { 1.73828125, 1, 2.99609375, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -9, { 1.9609375, 1, 2.64453125, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -8, { 1.9375, 1, 2.6875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -7, { 1.9140625, 1, 2.72265625, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -6, { 1.88671875, 1, 2.76171875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -5, { 1.86328125, 1, 2.8046875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -4, { 1.83984375, 1, 2.83984375, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -3, { 1.8125, 1, 2.87890625, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -2, { 1.78515625, 1, 2.91796875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , -1, { 1.76171875, 1, 2.95703125, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 0, { 1.73828125, 1, 2.99609375, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 1, { 1.71484375, 1, 3.06640625, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 2, { 1.69140625, 1, 3.13671875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 3, { 1.6640625, 1, 3.20703125, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 4, { 1.640625, 1, 3.27734375, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 5, { 1.6171875, 1, 3.34765625, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 6, { 1.59375, 1, 3.41796875, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 7, { 1.5703125, 1, 3.48828125, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 8, { 1.54296875, 1, 3.55859375, 0 } },
  { "Panasonic", "DC-FZ10002", Incandescent  , 9, { 1.51953125, 1, 3.62890625, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -9, { 2.921875, 1, 1.6875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -8, { 2.8828125, 1, 1.71484375, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -7, { 2.84765625, 1, 1.73828125, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -6, { 2.80859375, 1, 1.76171875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -5, { 2.7734375, 1, 1.7890625, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -4, { 2.734375, 1, 1.8125, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -3, { 2.69921875, 1, 1.8359375, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -2, { 2.66015625, 1, 1.859375, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , -1, { 2.625, 1, 1.88671875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 0, { 2.5859375, 1, 1.91015625, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 1, { 2.55078125, 1, 1.95703125, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 2, { 2.515625, 1, 2.0, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 3, { 2.4765625, 1, 2.046875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 4, { 2.44140625, 1, 2.08984375, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 5, { 2.40625, 1, 2.13671875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 6, { 2.37109375, 1, 2.1796875, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 7, { 2.3359375, 1, 2.2265625, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 8, { 2.296875, 1, 2.26953125, 0 } },
  { "Panasonic", "DC-FZ10002", Flash         , 9, { 2.26171875, 1, 2.31640625, 0 } },

  /* Panasonic DC-TZ96 Firmware Version 1.0 */
  { "Panasonic", "DC-TZ96", Daylight    , 0, { 2.1328125,  1, 1.79296875, 0 } },
  { "Panasonic", "DC-TZ96", Cloudy      , 0, { 2.30859375, 1, 1.65625,    0 } },
  { "Panasonic", "DC-TZ96", Shade       , 0, { 2.4609375,  1, 1.54296875, 0 } },
  { "Panasonic", "DC-TZ96", Incandescent, 0, { 1.4765625,  1, 2.55859375, 0 } },
  { "Panasonic", "DC-TZ96", Flash       , 0, { 2.3203125,  1, 1.56640625, 0 } },
  { "Panasonic", "DC-TZ96", D55         , 0, { 1.9765625,  1, 1.90234375, 0 } },
  { "Panasonic", "DC-TZ96", Tungsten    , 0, { 1.4765625,  1, 2.55859375, 0 } },

  { "Panasonic", "DMC-FZ8", Daylight, 0,	{ 1.904943, 1, 1.596958, 0 } },
  { "Panasonic", "DMC-FZ8", Cloudy, 0,		{ 2.060836, 1, 1.498099, 0 } },
  { "Panasonic", "DMC-FZ8", Shade, 0,		{ 2.258555, 1, 1.391635, 0 } },
  { "Panasonic", "DMC-FZ8", Incandescent, 0,	{ 1.247148, 1, 2.288973, 0 } },
  { "Panasonic", "DMC-FZ8", Flash, 0,		{ 2.072243, 1, 1.456274, 0 } },

  { "Panasonic", "DMC-FZ18", Daylight, 0,	{ 1.783270, 1, 1.889734, 0 } },
  { "Panasonic", "DMC-FZ18", Cloudy, 0,		{ 1.946768, 1, 1.680608, 0 } },
  { "Panasonic", "DMC-FZ18", Shade, 0,		{ 2.117871, 1, 1.558935, 0 } },
  { "Panasonic", "DMC-FZ18", Incandescent, 0,	{ 1.140684, 1, 2.627376, 0 } },
  { "Panasonic", "DMC-FZ18", Flash, 0,		{ 1.882129, 1, 1.703422, 0 } },

  { "Panasonic", "DMC-FZ28", Daylight, 0,	{ 1.684411, 1, 1.802281, 0 } },
  { "Panasonic", "DMC-FZ28", Cloudy, 0,		{ 1.825095, 1, 1.676806, 0 } },
  { "Panasonic", "DMC-FZ28", Shade, 0,		{ 1.996198, 1, 1.566540, 0 } },
  { "Panasonic", "DMC-FZ28", Incandescent, 0,	{ 1.117871, 1, 2.558935, 0 } },
  { "Panasonic", "DMC-FZ28", Flash, 0,		{ 1.939164, 1, 1.596958, 0 } },
  { "Panasonic", "DMC-FZ28", "3000K", 0,	{ 1.015209, 1, 2.771863, 0 } },
  { "Panasonic", "DMC-FZ28", "5000K", 0,	{ 1.585551, 1, 1.889734, 0 } },

  { "Panasonic", "DMC-FZ30", Daylight, 0,	{ 1.757576, 1, 1.446970, 0 } },
  { "Panasonic", "DMC-FZ30", Cloudy, 0,		{ 1.943182, 1, 1.276515, 0 } },
  { "Panasonic", "DMC-FZ30", Incandescent, 0,	{ 1.098485, 1, 2.106061, 0 } },
  { "Panasonic", "DMC-FZ30", Flash, 0,		{ 1.965909, 1, 1.303030, 0 } },

  { "Panasonic", "DMC-FZ50", Daylight, 0,	{ 2.095057, 1, 1.642586, 0 } },
  { "Panasonic", "DMC-FZ50", Cloudy, 0,		{ 2.319392, 1, 1.482890, 0 } },
  { "Panasonic", "DMC-FZ50", Shade, 0,		{ 2.463878, 1, 1.414449, 0 } },
  { "Panasonic", "DMC-FZ50", Incandescent, 0,	{ 1.365019, 1, 2.311787, 0 } },
  { "Panasonic", "DMC-FZ50", Flash, 0,		{ 2.338403, 1, 1.338403, 0 } },

  /* Firmware version 1.0 */
  /* -9/+9 fine tuning is -9/+9 in blue-amber and zero in magenta-green */
  { "Panasonic", "DMC-FZ200", Daylight, -9,	{ 1.811321, 1, 2.049057, 0 } },
  { "Panasonic", "DMC-FZ200", Daylight, 0,	{ 2.015094, 1, 1.758491, 0 } },
  { "Panasonic", "DMC-FZ200", Daylight, 9,	{ 2.264151, 1, 1.550943, 0 } },
  { "Panasonic", "DMC-FZ200", Cloudy, -9,	{ 1.935849, 1, 1.890566, 0 } },
  { "Panasonic", "DMC-FZ200", Cloudy, 0,	{ 2.154717, 1, 1.622642, 0 } },
  { "Panasonic", "DMC-FZ200", Cloudy, 9,	{ 2.426415, 1, 1.433962, 0 } },
  { "Panasonic", "DMC-FZ200", Shade, -9,	{ 2.052830, 1, 1.773585, 0 } },
  { "Panasonic", "DMC-FZ200", Shade, 0,		{ 2.286792, 1, 1.524528, 0 } },
  { "Panasonic", "DMC-FZ200", Shade, 9,		{ 2.569811, 1, 1.343396, 0 } },
  { "Panasonic", "DMC-FZ200", Incandescent, -9,	{ 1.230189, 1, 3.143396, 0 } },
  { "Panasonic", "DMC-FZ200", Incandescent, 0,	{ 1.369811, 1, 2.701887, 0 } },
  { "Panasonic", "DMC-FZ200", Incandescent, 9,	{ 1.539623, 1, 2.384906, 0 } },
  { "Panasonic", "DMC-FZ200", Flash, -9,	{ 1.962264, 1, 1.932076, 0 } },
  { "Panasonic", "DMC-FZ200", Flash, 0,		{ 2.184906, 1, 1.660377, 0 } },
  { "Panasonic", "DMC-FZ200", Flash, 9,		{ 2.460377, 1, 1.464151, 0 } },
  { "Panasonic", "DMC-FZ200", "2700K", 0,	{ 1.226415, 1, 3.018868, 0 } },
  { "Panasonic", "DMC-FZ200", "3000K", 0,	{ 1.332075, 1, 2.784906, 0 } },
  { "Panasonic", "DMC-FZ200", "3300K", 0,	{ 1.422642, 1, 2.573585, 0 } },
  { "Panasonic", "DMC-FZ200", "5000K", 0,	{ 1.928302, 1, 1.841509, 0 } },
  { "Panasonic", "DMC-FZ200", "5500K", 0,	{ 2.015094, 1, 1.758491, 0 } },
  { "Panasonic", "DMC-FZ200", "6500K", 0,	{ 2.203774, 1, 1.577358, 0 } },

  /* Firmware version 2.2 */
  /* Daylight is called "Fine Weather" in the EXIF tags and "Clear Sky" in the manual. */
  { "Panasonic", "DMC-FZ300", Daylight, 0,  { 2.144531, 1, 1.859375, 0 } }, /*5500K*/
  { "Panasonic", "DMC-FZ300", Cloudy, 0,  { 2.304688, 1, 1.707031, 0 } },
  { "Panasonic", "DMC-FZ300", Shade, 0,	  { 2.429688, 1, 1.609375, 0 } },
  { "Panasonic", "DMC-FZ300", Incandescent, 0,  { 1.5, 1, 2.722656, 0 } },
  { "Panasonic", "DMC-FZ300", Flash, 0,   { 2.304688, 1, 1.761719, 0 } },
  { "Panasonic", "DMC-FZ300", "2500K", 0, { 1.300781, 1, 3.214844, 0 } },
  { "Panasonic", "DMC-FZ300", "3000K", 0, { 1.480469, 1, 2.808594, 0 } },
  { "Panasonic", "DMC-FZ300", "5000K", 0, { 1.996094, 1, 2.0, 0 } },
  { "Panasonic", "DMC-FZ300", "6500K", 0, { 2.332031, 1, 1.6875, 0 } },
  { "Panasonic", "DMC-FZ300", "8500K", 0, { 2.574219, 1, 1.496094, 0 } },
  { "Panasonic", "DMC-FZ300", "10000K", 0, { 2.703125, 1, 1.390625, 0 } },

  // Panasonic DMC-FZ1000 Firmware 2.0.0
  { "Panasonic", "DMC-FZ1000", Daylight, 0,	{ 2.425781, 1, 1.886719, 0 } },
  { "Panasonic", "DMC-FZ1000", Cloudy, 0,	{ 2.621094, 1, 1.714844, 0 } },
  { "Panasonic", "DMC-FZ1000", Shade, 0,	{ 2.765625, 1, 1.625000, 0 } },
  { "Panasonic", "DMC-FZ1000", Incandescent, 0,	{ 1.718750, 1, 2.750000, 0 } },
  { "Panasonic", "DMC-FZ1000", Flash, 0,	{ 2.585938, 1, 1.796875, 0 } },

  { "Panasonic", "DMC-TZ60", Daylight, 0,	{ 2.060377, 1, 1.656604, 0 } },
  { "Panasonic", "DMC-TZ60", Cloudy, 0,		{ 2.192453, 1, 1.520755, 0 } },
  { "Panasonic", "DMC-TZ60", Shade, 0,		{ 2.335849, 1, 1.415094, 0 } },
  { "Panasonic", "DMC-TZ60", Incandescent, 0,	{ 1.384906, 1, 2.577358, 0 } },

  { "Panasonic", "DMC-TZ71", Daylight, 0,	{ 2.090909, 1, 1.818182, 0 } },
  { "Panasonic", "DMC-TZ71", Cloudy, 0,		{ 2.241509, 1, 1.660377, 0 } },
  { "Panasonic", "DMC-TZ71", Shade, 0,		{ 2.378788, 1, 1.590909, 0 } },
  { "Panasonic", "DMC-TZ71", Incandescent, 0,	{ 1.420455, 1, 2.727273, 0 } },

  { "Panasonic", "DMC-TZ100", Daylight, 0,	{ 2.406250, 1, 1.871093, 0 } },
  { "Panasonic", "DMC-TZ100", Cloudy, 0,	{ 2.550781, 1, 1.726562, 0 } },
  { "Panasonic", "DMC-TZ100", Shade, 0,		{ 2.753906, 1, 1.566406, 0 } },
  { "Panasonic", "DMC-TZ100", Incandescent, 0,	{ 1.691406, 1, 2.800781, 0 } },
  { "Panasonic", "DMC-TZ100", Flash, 0,		{ 2.609375, 1, 1.718750, 0 } },

  { "Panasonic", "DMC-G1", Daylight, 0,		{ 1.942966, 1, 1.448669, 0 } },
  { "Panasonic", "DMC-G1", Cloudy, 0,		{ 2.106464, 1, 1.326996, 0 } },
  { "Panasonic", "DMC-G1", Shade, 0,		{ 2.323194, 1, 1.224335, 0 } },
  { "Panasonic", "DMC-G1", Incandescent, 0,	{ 1.319392, 1, 2.148289, 0 } },
  { "Panasonic", "DMC-G1", Flash, 0,		{ 1.528517, 1, 1.277567, 0 } },

  { "Panasonic", "DMC-G2", DirectSunlight     , 0, { 1.89453125, 1, 1.4453125, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -9, { 2.193916349809886, 1, 1.2661596958174905, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -8, { 2.159695817490494, 1, 1.285171102661597, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -7, { 2.129277566539924, 1, 1.3003802281368821, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -6, { 2.091254752851711, 1, 1.3231939163498099, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -5, { 2.068441064638783, 1, 1.3422053231939164, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -4, { 2.038022813688213, 1, 1.3574144486692015, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -3, { 2.0038022813688214, 1, 1.3802281368821292, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -2, { 1.9657794676806084, 1, 1.403041825095057, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , -1, { 1.9277566539923954, 1, 1.4258555133079849, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 0, { 1.897338403041825, 1, 1.4486692015209126, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 1, { 1.8745247148288973, 1, 1.4828897338403042, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 2, { 1.8517110266159695, 1, 1.5171102661596958, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 3, { 1.8250950570342206, 1, 1.5551330798479088, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 4, { 1.7946768060836502, 1, 1.5893536121673004, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 5, { 1.779467680608365, 1, 1.623574144486692, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 6, { 1.7566539923954372, 1, 1.6577946768060836, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 7, { 1.726235741444867, 1, 1.6996197718631179, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 8, { 1.703422053231939, 1, 1.7338403041825095, 0 } },
  { "Panasonic", "DMC-G2", Daylight           , 9, { 1.6844106463878328, 1, 1.7756653992395437, 0 } },
  { "Panasonic", "DMC-G2", D55                , 0, { 1.78515625, 1, 1.43359375, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -9, { 2.5665399239543727, 1, 1.0836501901140685, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -8, { 2.5247148288973382, 1, 1.0988593155893536, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -7, { 2.4866920152091256, 1, 1.1140684410646389, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -6, { 2.444866920152091, 1, 1.1330798479087452, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -5, { 2.418250950570342, 1, 1.144486692015209, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -4, { 2.3840304182509504, 1, 1.1596958174904943, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -3, { 2.3422053231939164, 1, 1.1787072243346008, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -2, { 2.296577946768061, 1, 1.1977186311787071, 0 } },
  { "Panasonic", "DMC-G2", Shade              , -1, { 2.2547528517110265, 1, 1.220532319391635, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 0, { 2.220532319391635, 1, 1.2395437262357414, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 1, { 2.193916349809886, 1, 1.2661596958174905, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 2, { 2.167300380228137, 1, 1.296577946768061, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 3, { 2.1330798479087454, 1, 1.3307984790874525, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 4, { 2.098859315589354, 1, 1.3574144486692015, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 5, { 2.079847908745247, 1, 1.3878326996197718, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 6, { 2.053231939163498, 1, 1.4182509505703422, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 7, { 2.0190114068441063, 1, 1.4524714828897338, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 8, { 1.9923954372623573, 1, 1.479087452471483, 0 } },
  { "Panasonic", "DMC-G2", Shade              , 9, { 1.9657794676806084, 1, 1.520912547528517, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -9, { 2.346007604562738, 1, 1.1596958174904943, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -8, { 2.307984790874525, 1, 1.1749049429657794, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -7, { 2.273764258555133, 1, 1.1901140684410647, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -6, { 2.23574144486692, 1, 1.2129277566539924, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -5, { 2.2129277566539924, 1, 1.2281368821292775, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -4, { 2.178707224334601, 1, 1.2433460076045628, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -3, { 2.140684410646388, 1, 1.2623574144486691, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -2, { 2.098859315589354, 1, 1.285171102661597, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , -1, { 2.0608365019011408, 1, 1.3041825095057034, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 0, { 2.0234375, 1, 1.32421875, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 0, { 2.03041825095057, 1, 1.326996197718631, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 1, { 2.0038022813688214, 1, 1.3574144486692015, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 2, { 1.9809885931558935, 1, 1.3878326996197718, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 3, { 1.950570342205323, 1, 1.4258555133079849, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 4, { 1.9163498098859315, 1, 1.456273764258555, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 5, { 1.9011406844106464, 1, 1.4866920152091254, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 6, { 1.8783269961977187, 1, 1.5171102661596958, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 7, { 1.8479087452471483, 1, 1.5551330798479088, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 8, { 1.8212927756653992, 1, 1.585551330798479, 0 } },
  { "Panasonic", "DMC-G2", Cloudy             , 9, { 1.7984790874524714, 1, 1.6273764258555132, 0 } },
  { "Panasonic", "DMC-G2", Tungsten           , 0, { 1.33203125, 1, 2.11328125, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -9, { 1.543726235741445, 1, 1.8517110266159695, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -8, { 1.5171102661596958, 1, 1.8745247148288973, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -7, { 1.4980988593155893, 1, 1.9011406844106464, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -6, { 1.4714828897338403, 1, 1.935361216730038, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -5, { 1.456273764258555, 1, 1.9581749049429658, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -4, { 1.4372623574144487, 1, 1.984790874524715, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -3, { 1.4106463878326996, 1, 2.0152091254752853, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -2, { 1.3840304182509506, 1, 2.049429657794677, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , -1, { 1.3574144486692015, 1, 2.0836501901140685, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 0, { 1.338403041825095, 1, 2.11787072243346, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 1, { 1.3193916349809887, 1, 2.167300380228137, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 2, { 1.3041825095057034, 1, 2.216730038022814, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 3, { 1.285171102661597, 1, 2.273764258555133, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 4, { 1.2623574144486691, 1, 2.32319391634981, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 5, { 1.2509505703422052, 1, 2.3726235741444865, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 6, { 1.2357414448669202, 1, 2.4220532319391634, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 7, { 1.2167300380228137, 1, 2.4790874524714828, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 8, { 1.2015209125475286, 1, 2.5285171102661597, 0 } },
  { "Panasonic", "DMC-G2", Incandescent       , 9, { 1.1863117870722433, 1, 2.596958174904943, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -9, { 2.5893536121673004, 1, 1.0722433460076046, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -8, { 2.543726235741445, 1, 1.0874524714828897, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -7, { 2.5095057034220534, 1, 1.0988593155893536, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -6, { 2.463878326996198, 1, 1.11787072243346, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -5, { 2.4372623574144487, 1, 1.1330798479087452, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -4, { 2.403041825095057, 1, 1.1482889733840305, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -3, { 2.3612167300380227, 1, 1.167300380228137, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -2, { 2.3155893536121672, 1, 1.1863117870722433, 0 } },
  { "Panasonic", "DMC-G2", Flash              , -1, { 2.273764258555133, 1, 1.2053231939163498, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 0, { 2.2395437262357416, 1, 1.2243346007604563, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 1, { 2.2129277566539924, 1, 1.2547528517110267, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 2, { 2.1863117870722433, 1, 1.2813688212927756, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 3, { 2.1482889733840302, 1, 1.3155893536121672, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 4, { 2.1140684410646386, 1, 1.3460076045627376, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 5, { 2.098859315589354, 1, 1.3726235741444868, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 6, { 2.0722433460076046, 1, 1.403041825095057, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 7, { 2.038022813688213, 1, 1.4372623574144487, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 8, { 2.011406844106464, 1, 1.4638783269961977, 0 } },
  { "Panasonic", "DMC-G2", Flash              , 9, { 1.984790874524715, 1, 1.5019011406844107, 0 } },


  { "Panasonic", "DMC-G3", Daylight, 0,		{ 2.7925, 1, 1.5472, 0 } },
  { "Panasonic", "DMC-G3", Cloudy, 0,		{ 2.9660, 1, 1.4528, 0 } },
  { "Panasonic", "DMC-G3", Shade, 0,		{ 3.2642, 1, 1.3698, 0 } },
  { "Panasonic", "DMC-G3", Incandescent, 0,	{ 1.8491, 1, 2.2566, 0 } },
  { "Panasonic", "DMC-G3", Flash, 0,		{ 3.2868, 1, 1.3547, 0 } },
  { "Panasonic", "DMC-G3", "2700K", 0,		{ 1.5283, 1, 2.6717, 0 } },
  { "Panasonic", "DMC-G3", "3000K", 0,		{ 1.7094, 1, 2.4189, 0 } },
  { "Panasonic", "DMC-G3", "3300K", 0,		{ 1.8792, 1, 2.1849, 0 } },
  { "Panasonic", "DMC-G3", "5000K", 0,		{ 2.6679, 1, 1.6264, 0 } },
  { "Panasonic", "DMC-G3", "5500K", 0,		{ 2.7925, 1, 1.5472, 0 } },
  { "Panasonic", "DMC-G3", "6500K", 0,		{ 3.0679, 1, 1.4189, 0 } },

  { "Panasonic", "DMC-G5", Daylight           , -9, { 2.7433962264150944, 1, 1.430188679245283, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -8, { 2.70188679245283, 1, 1.4490566037735848, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -7, { 2.6452830188679246, 1, 1.471698113207547, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -6, { 2.5886792452830187, 1, 1.490566037735849, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -5, { 2.5547169811320756, 1, 1.5169811320754718, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -4, { 2.516981132075472, 1, 1.5358490566037737, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -3, { 2.460377358490566, 1, 1.5622641509433963, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -2, { 2.4075471698113207, 1, 1.588679245283019, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , -1, { 2.3811320754716983, 1, 1.6, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 0, { 2.3433962264150945, 1, 1.618867924528302, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 1, { 2.3056603773584907, 1, 1.649056603773585, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 2, { 2.267924528301887, 1, 1.6867924528301887, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 3, { 2.241509433962264, 1, 1.7207547169811321, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 4, { 2.2037735849056603, 1, 1.758490566037736, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 5, { 2.177358490566038, 1, 1.788679245283019, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 6, { 2.150943396226415, 1, 1.8226415094339623, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 7, { 2.1245283018867926, 1, 1.8452830188679246, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 8, { 2.1056603773584905, 1, 1.879245283018868, 0 } },
  { "Panasonic", "DMC-G5", Daylight           , 9, { 2.0754716981132075, 1, 1.909433962264151, 0 } },
  { "Panasonic", "DMC-G5", D55                , 0, { 2.1796875, 1, 1.6171875, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -9, { 3.0943396226415096, 1, 1.2754716981132075, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -8, { 3.0415094339622644, 1, 1.2943396226415094, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -7, { 2.981132075471698, 1, 1.3169811320754716, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -6, { 2.9169811320754717, 1, 1.3320754716981131, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -5, { 2.8754716981132074, 1, 1.3547169811320754, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -4, { 2.8339622641509434, 1, 1.3735849056603773, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -3, { 2.7735849056603774, 1, 1.3962264150943395, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -2, { 2.709433962264151, 1, 1.4188679245283018, 0 } },
  { "Panasonic", "DMC-G5", Shade              , -1, { 2.6792452830188678, 1, 1.430188679245283, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 0, { 2.637735849056604, 1, 1.4452830188679244, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 1, { 2.5962264150943395, 1, 1.4754716981132074, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 2, { 2.5547169811320756, 1, 1.509433962264151, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 3, { 2.5245283018867926, 1, 1.5358490566037737, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 4, { 2.483018867924528, 1, 1.569811320754717, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 5, { 2.452830188679245, 1, 1.6, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 6, { 2.422641509433962, 1, 1.6264150943396227, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 7, { 2.392452830188679, 1, 1.649056603773585, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 8, { 2.369811320754717, 1, 1.679245283018868, 0 } },
  { "Panasonic", "DMC-G5", Shade              , 9, { 2.339622641509434, 1, 1.7056603773584906, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -9, { 2.871698113207547, 1, 1.3547169811320754, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -8, { 2.8264150943396227, 1, 1.369811320754717, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -7, { 2.7660377358490567, 1, 1.3962264150943395, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -6, { 2.709433962264151, 1, 1.411320754716981, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -5, { 2.671698113207547, 1, 1.4377358490566037, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -4, { 2.6339622641509433, 1, 1.4566037735849056, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -3, { 2.577358490566038, 1, 1.4792452830188678, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -2, { 2.516981132075472, 1, 1.5018867924528303, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , -1, { 2.490566037735849, 1, 1.5132075471698114, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 0, { 2.452830188679245, 1, 1.5320754716981133, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 1, { 2.4113207547169813, 1, 1.5622641509433963, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 2, { 2.3735849056603775, 1, 1.6, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 3, { 2.3471698113207546, 1, 1.630188679245283, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 4, { 2.309433962264151, 1, 1.6641509433962265, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 5, { 2.279245283018868, 1, 1.6943396226415095, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 6, { 2.249056603773585, 1, 1.7245283018867925, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 7, { 2.2226415094339624, 1, 1.7471698113207548, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 8, { 2.2037735849056603, 1, 1.7773584905660378, 0 } },
  { "Panasonic", "DMC-G5", Cloudy             , 9, { 2.1735849056603773, 1, 1.8075471698113208, 0 } },
  { "Panasonic", "DMC-G5", Tungsten           , 0, { 1.609375, 1, 2.3359375, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -9, { 1.890566037735849, 1, 2.0641509433962266, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -8, { 1.860377358490566, 1, 2.0943396226415096, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -7, { 1.818867924528302, 1, 2.128301886792453, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -6, { 1.7849056603773585, 1, 2.1547169811320757, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -5, { 1.758490566037736, 1, 2.1924528301886794, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -4, { 1.7320754716981133, 1, 2.2226415094339624, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -3, { 1.6943396226415095, 1, 2.2566037735849056, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -2, { 1.6566037735849057, 1, 2.2943396226415094, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , -1, { 1.6377358490566039, 1, 2.3132075471698115, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 0, { 1.6150943396226416, 1, 2.339622641509434, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 1, { 1.588679245283019, 1, 2.3849056603773584, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 2, { 1.5622641509433963, 1, 2.4415094339622643, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 3, { 1.5433962264150944, 1, 2.486792452830189, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 4, { 1.5169811320754718, 1, 2.539622641509434, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 5, { 1.4981132075471697, 1, 2.5849056603773586, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 6, { 1.4792452830188678, 1, 2.6339622641509433, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 7, { 1.460377358490566, 1, 2.667924528301887, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 8, { 1.4490566037735848, 1, 2.7132075471698114, 0 } },
  { "Panasonic", "DMC-G5", Incandescent       , 9, { 1.430188679245283, 1, 2.762264150943396, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -9, { 2.852830188679245, 1, 1.3018867924528301, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -8, { 2.8037735849056604, 1, 1.3169811320754716, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -7, { 2.7471698113207546, 1, 1.3433962264150943, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -6, { 2.690566037735849, 1, 1.3584905660377358, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -5, { 2.6528301886792454, 1, 1.381132075471698, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -4, { 2.6150943396226416, 1, 1.4, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -3, { 2.5584905660377357, 1, 1.4226415094339622, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -2, { 2.5018867924528303, 1, 1.4452830188679244, 0 } },
  { "Panasonic", "DMC-G5", Flash              , -1, { 2.4716981132075473, 1, 1.4566037735849056, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 0, { 2.4339622641509435, 1, 1.4754716981132074, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 1, { 2.3962264150943398, 1, 1.5018867924528303, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 2, { 2.358490566037736, 1, 1.539622641509434, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 3, { 2.328301886792453, 1, 1.5660377358490567, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 4, { 2.290566037735849, 1, 1.6, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 5, { 2.260377358490566, 1, 1.630188679245283, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 6, { 2.2339622641509433, 1, 1.6603773584905661, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 7, { 2.2037735849056603, 1, 1.6830188679245284, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 8, { 2.1849056603773587, 1, 1.709433962264151, 0 } },
  { "Panasonic", "DMC-G5", Flash              , 9, { 2.158490566037736, 1, 1.739622641509434, 0 } },
  { "Panasonic", "DMC-G5", "2700K", 0,		{ 1.369811, 1, 2.743396, 0 } },
  { "Panasonic", "DMC-G5", "3000K", 0,		{ 1.516981, 1, 2.490566, 0 } },
  { "Panasonic", "DMC-G5", "5000K", 0,		{ 2.230189, 1, 1.713208, 0 } },
  { "Panasonic", "DMC-G5", "5500K", 0,		{ 2.343396, 1, 1.618868, 0 } },
  { "Panasonic", "DMC-G5", "6500K", 0,		{ 2.501887, 1, 1.505660, 0 } },

  /* Firmware version 1.1 */
  { "Panasonic", "DMC-G6", Daylight, 0,		{ 2.265625, 1, 1.699219, 0 } },
  { "Panasonic", "DMC-G6", Cloudy, 0,		{ 2.406250, 1, 1.593750, 0 } },
  { "Panasonic", "DMC-G6", Shade, 0,		{ 2.617188, 1, 1.511719, 0 } },
  { "Panasonic", "DMC-G6", Incandescent, 0,	{ 1.535156, 1, 2.375000, 0 } },
  { "Panasonic", "DMC-G6", Flash, 0,		{ 2.332031, 1, 1.566406, 0 } },

  { "Panasonic", "DMC-G7", Daylight, 0,		{ 2.351563, 1, 1.664063, 0 } },
  { "Panasonic", "DMC-G7", Cloudy, 0,		{ 2.519531, 1, 1.542969, 0 } },
  { "Panasonic", "DMC-G7", Shade, 0,		{ 2.691406, 1, 1.464844, 0 } },
  { "Panasonic", "DMC-G7", Incandescent, 0,	{ 1.625, 1, 2.425781, 0 } },
  { "Panasonic", "DMC-G7", Flash, 0,		{ 2.371094, 1, 1.589844, 0 } },

  { "Panasonic", "DMC-G8", "Daylight", -9,     { 2.710938, 1, 1.410156, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -8,     { 2.664063, 1, 1.425781, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -7,     { 2.621094, 1, 1.453125, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -6,     { 2.566406, 1, 1.472656, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -5,     { 2.53125, 1, 1.488281, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -4,     { 2.492188, 1, 1.515625, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -3,     { 2.449219, 1, 1.539063, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -2,     { 2.394531, 1, 1.566406, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", -1,     { 2.367188, 1, 1.578125, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 0,      { 2.328125, 1, 1.597656, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 1,      { 2.292969, 1, 1.632813, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 2,      { 2.257813, 1, 1.664063, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 3,      { 2.21875, 1, 1.703125, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 4,      { 2.191406, 1, 1.738281, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 5,      { 2.15625, 1, 1.769531, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 6,      { 2.121094, 1, 1.808594, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 7,      { 2.101563, 1, 1.839844, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 8,      { 2.082031, 1, 1.871094, 0 } },
  { "Panasonic", "DMC-G8", "Daylight", 9,      { 2.046875, 1, 1.90625, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -9,       { 2.910156, 1, 1.316406, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -8,       { 2.859375, 1, 1.335938, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -7,       { 2.8125, 1, 1.359375, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -6,       { 2.753906, 1, 1.375, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -5,       { 2.714844, 1, 1.394531, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -4,       { 2.675781, 1, 1.417969, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -3,       { 2.625, 1, 1.441406, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -2,       { 2.566406, 1, 1.464844, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", -1,       { 2.539063, 1, 1.476563, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 0,        { 2.5, 1, 1.492188, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 1,        { 2.460938, 1, 1.527344, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 2,        { 2.421875, 1, 1.554688, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 3,        { 2.382813, 1, 1.589844, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 4,        { 2.351563, 1, 1.625, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 5,        { 2.3125, 1, 1.65625, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 6,        { 2.273438, 1, 1.691406, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 7,        { 2.253906, 1, 1.71875, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 8,        { 2.234375, 1, 1.75, 0 } },
  { "Panasonic", "DMC-G8", "Cloudy", 9,        { 2.195313, 1, 1.785156, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -9,        { 3.109375, 1, 1.257813, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -8,        { 3.054688, 1, 1.273438, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -7,        { 3.003906, 1, 1.296875, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -6,        { 2.941406, 1, 1.3125, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -5,        { 2.898438, 1, 1.328125, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -4,        { 2.859375, 1, 1.351563, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -3,        { 2.804688, 1, 1.375, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -2,        { 2.742188, 1, 1.394531, 0 } },
  { "Panasonic", "DMC-G8", "Shade", -1,        { 2.710938, 1, 1.40625, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 0,         { 2.671875, 1, 1.425781, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 1,         { 2.628906, 1, 1.457031, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 2,         { 2.585938, 1, 1.484375, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 3,         { 2.542969, 1, 1.519531, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 4,         { 2.511719, 1, 1.550781, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 5,         { 2.472656, 1, 1.578125, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 6,         { 2.429688, 1, 1.613281, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 7,         { 2.410156, 1, 1.640625, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 8,         { 2.386719, 1, 1.667969, 0 } },
  { "Panasonic", "DMC-G8", "Shade", 9,         { 2.347656, 1, 1.703125, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -9, { 1.875, 1, 2, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -8, { 1.84375, 1, 2.027344, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -7, { 1.8125, 1, 2.0625, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -6, { 1.773438, 1, 2.085938, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -5, { 1.746094, 1, 2.113281, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -4, { 1.722656, 1, 2.148438, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -3, { 1.691406, 1, 2.183594, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -2, { 1.652344, 1, 2.21875, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", -1, { 1.632813, 1, 2.238281, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 0,  { 1.609375, 1, 2.265625, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 1,  { 1.585938, 1, 2.316406, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 2,  { 1.558594, 1, 2.363281, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 3,  { 1.535156, 1, 2.414063, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 4,  { 1.515625, 1, 2.46875, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 5,  { 1.488281, 1, 2.511719, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 6,  { 1.464844, 1, 2.566406, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 7,  { 1.453125, 1, 2.609375, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 8,  { 1.4375, 1, 2.652344, 0 } },
  { "Panasonic", "DMC-G8", "Incandescent", 9,  { 1.414063, 1, 2.707031, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -9,        { 3.425781, 1, 1.292969, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -8,        { 3.367188, 1, 1.3125, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -7,        { 3.3125, 1, 1.332031, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -6,        { 3.242188, 1, 1.351563, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -5,        { 3.195313, 1, 1.367188, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -4,        { 3.152344, 1, 1.390625, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -3,        { 3.09375, 1, 1.414063, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -2,        { 3.023438, 1, 1.4375, 0 } },
  { "Panasonic", "DMC-G8", "Flash", -1,        { 2.988281, 1, 1.449219, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 0,         { 2.945313, 1, 1.464844, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 1,         { 2.898438, 1, 1.5, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 2,         { 2.851563, 1, 1.527344, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 3,         { 2.804688, 1, 1.5625, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 4,         { 2.769531, 1, 1.597656, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 5,         { 2.726563, 1, 1.625, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 6,         { 2.679688, 1, 1.660156, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 7,         { 2.65625, 1, 1.6875, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 8,         { 2.632813, 1, 1.71875, 0 } },
  { "Panasonic", "DMC-G8", "Flash", 9,         { 2.585938, 1, 1.753906, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -9,        { 2.554688, 1, 1.480469, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -8,        { 2.511719, 1, 1.5, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -7,        { 2.46875, 1, 1.527344, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -6,        { 2.417969, 1, 1.546875, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -5,        { 2.382813, 1, 1.566406, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -4,        { 2.347656, 1, 1.59375, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -3,        { 2.304688, 1, 1.621094, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -2,        { 2.253906, 1, 1.644531, 0 } },
  { "Panasonic", "DMC-G8", "5000K", -1,        { 2.230469, 1, 1.660156, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 0,         { 2.195313, 1, 1.679688, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 1,         { 2.160156, 1, 1.71875, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 2,         { 2.125, 1, 1.75, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 3,         { 2.089844, 1, 1.789063, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 4,         { 2.066406, 1, 1.828125, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 5,         { 2.03125, 1, 1.863281, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 6,         { 1.996094, 1, 1.902344, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 7,         { 1.980469, 1, 1.933594, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 8,         { 1.960938, 1, 1.96875, 0 } },
  { "Panasonic", "DMC-G8", "5000K", 9,         { 1.929688, 1, 2.007813, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -9,        { 2.710938, 1, 1.410156, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -8,        { 2.664063, 1, 1.425781, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -7,        { 2.621094, 1, 1.453125, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -6,        { 2.566406, 1, 1.472656, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -5,        { 2.53125, 1, 1.488281, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -4,        { 2.492188, 1, 1.515625, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -3,        { 2.449219, 1, 1.539063, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -2,        { 2.394531, 1, 1.566406, 0 } },
  { "Panasonic", "DMC-G8", "5500K", -1,        { 2.367188, 1, 1.578125, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 0,         { 2.328125, 1, 1.597656, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 1,         { 2.292969, 1, 1.632813, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 2,         { 2.257813, 1, 1.664063, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 3,         { 2.21875, 1, 1.703125, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 4,         { 2.191406, 1, 1.738281, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 5,         { 2.15625, 1, 1.769531, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 6,         { 2.121094, 1, 1.808594, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 7,         { 2.101563, 1, 1.839844, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 8,         { 2.082031, 1, 1.871094, 0 } },
  { "Panasonic", "DMC-G8", "5500K", 9,         { 2.046875, 1, 1.90625, 0 } },

  /* Firmware version 1.2 */
  /* -9/+9 fine tuning is -9/+9 in blue-amber and zero in magenta-green */
  { "Panasonic", "DMC-GF1", Daylight, -9,	{ 1.817490, 1, 1.760456, 0 } },
  { "Panasonic", "DMC-GF1", Daylight, 0,	{ 2.049430, 1, 1.437262, 0 } },
  { "Panasonic", "DMC-GF1", Daylight, 9,	{ 2.368821, 1, 1.254753, 0 } },
  { "Panasonic", "DMC-GF1", Cloudy, -9,		{ 1.969582, 1, 1.608365, 0 } },
  { "Panasonic", "DMC-GF1", Cloudy, 0,		{ 2.220532, 1, 1.311787, 0 } },
  { "Panasonic", "DMC-GF1", Cloudy, 9,		{ 2.566540, 1, 1.148289, 0 } },
  { "Panasonic", "DMC-GF1", Shade, -9,		{ 2.171103, 1, 1.482890, 0 } },
  { "Panasonic", "DMC-GF1", Shade, 0,		{ 2.448669, 1, 1.209125, 0 } },
  { "Panasonic", "DMC-GF1", Shade, 9,		{ 2.832700, 1, 1.057034, 0 } },
  { "Panasonic", "DMC-GF1", Incandescent, -9,	{ 1.231939, 1, 2.623574, 0 } },
  { "Panasonic", "DMC-GF1", Incandescent, 0,	{ 1.391635, 1, 2.136882, 0 } },
  { "Panasonic", "DMC-GF1", Incandescent, 9,	{ 1.608365, 1, 1.870722, 0 } },
  { "Panasonic", "DMC-GF1", Flash, -9,		{ 2.125475, 1, 1.528517, 0 } },
  { "Panasonic", "DMC-GF1", Flash, 0,		{ 2.399239, 1, 1.247148, 0 } },
  { "Panasonic", "DMC-GF1", Flash, 9,		{ 2.775665, 1, 1.091255, 0 } },
  { "Panasonic", "DMC-GF1", "2700K", 0,		{ 1.155894, 1, 2.589354, 0 } },
  { "Panasonic", "DMC-GF1", "3000K", 0,		{ 1.285171, 1, 2.319392, 0 } },
  { "Panasonic", "DMC-GF1", "3300K", 0,		{ 1.391635, 1, 2.079848, 0 } },
  { "Panasonic", "DMC-GF1", "5000K", 0,		{ 1.969582, 1, 1.501901, 0 } },
  { "Panasonic", "DMC-GF1", "5500K", 0,		{ 2.049430, 1, 1.437262, 0 } },
  { "Panasonic", "DMC-GF1", "6500K", 0,		{ 2.288973, 1, 1.281369, 0 } },

  { "Panasonic", "DMC-GF3", Daylight, 0,	{ 1.9396, 1, 1.4377, 0 } },
  { "Panasonic", "DMC-GF3", Cloudy, 0,		{ 2.0679, 1, 1.3396, 0 } },
  { "Panasonic", "DMC-GF3", Shade, 0,		{ 2.2566, 1, 1.2717, 0 } },
  { "Panasonic", "DMC-GF3", Incandescent, 0,	{ 1.3019, 1, 2.0868, 0 } },
  { "Panasonic", "DMC-GF3", Flash, 0,		{ 2.1962, 1, 1.2717, 0 } },

  { "Panasonic", "DMC-GF5", Daylight, 0,	{ 2.354717, 1, 1.637736, 0 } },
  { "Panasonic", "DMC-GF5", Cloudy, 0,		{ 2.498113, 1, 1.547170, 0 } },
  { "Panasonic", "DMC-GF5", Shade, 0,		{ 2.720755, 1, 1.460377, 0 } },
  { "Panasonic", "DMC-GF5", Incandescent, 0,	{ 1.615094, 1, 2.335849, 0 } },
  { "Panasonic", "DMC-GF5", Flash, 0,		{ 2.626415, 1, 1.445283, 0 } },

  { "Panasonic", "DMC-GH2", Daylight, 0,        { 2.399240, 1, 1.634981, 0 } },
  { "Panasonic", "DMC-GH2", Cloudy, 0,		{ 2.520913, 1, 1.547529, 0 } },
  { "Panasonic", "DMC-GH2", Shade, 0,           { 2.718631, 1, 1.460076, 0 } },
  { "Panasonic", "DMC-GH2", Incandescent, 0,    { 1.634981, 1, 2.307985, 0 } },
  { "Panasonic", "DMC-GH2", Flash, 0,		{ 2.745247, 1, 1.444867, 0 } },

  // Firmware version 1.2
  { "Panasonic", "DMC-GH3", Daylight           , -9, { 2.690566037735849, 1, 1.6716981132075472, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -8, { 2.6641509433962263, 1, 1.6943396226415095, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -7, { 2.6188679245283017, 1, 1.7207547169811321, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -6, { 2.5735849056603772, 1, 1.7509433962264151, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -5, { 2.543396226415094, 1, 1.7660377358490567, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -4, { 2.516981132075472, 1, 1.788679245283019, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -3, { 2.4716981132075473, 1, 1.8264150943396227, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -2, { 2.4264150943396228, 1, 1.8641509433962264, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , -1, { 2.388679245283019, 1, 1.8867924528301887, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 0, { 2.362264150943396, 1, 1.909433962264151, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 1, { 2.332075471698113, 1, 1.9547169811320755, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 2, { 2.2943396226415094, 1, 1.9924528301886792, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 3, { 2.2754716981132077, 1, 2.0377358490566038, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 4, { 2.249056603773585, 1, 2.079245283018868, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 5, { 2.2226415094339624, 1, 2.116981132075472, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 6, { 2.1924528301886794, 1, 2.1547169811320757, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 7, { 2.158490566037736, 1, 2.207547169811321, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 8, { 2.120754716981132, 1, 2.2528301886792454, 0 } },
  { "Panasonic", "DMC-GH3", Daylight           , 9, { 2.0943396226415096, 1, 2.3056603773584907, 0 } },
  { "Panasonic", "DMC-GH3", D55                , 0, { 2.20703125, 1, 1.91796875, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -9, { 3.049056603773585, 1, 1.4641509433962263, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -8, { 3.018867924528302, 1, 1.4830188679245282, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -7, { 2.9660377358490564, 1, 1.509433962264151, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -6, { 2.913207547169811, 1, 1.5358490566037737, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -5, { 2.883018867924528, 1, 1.5509433962264152, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -4, { 2.852830188679245, 1, 1.569811320754717, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -3, { 2.8, 1, 1.6037735849056605, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -2, { 2.7471698113207546, 1, 1.6339622641509435, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , -1, { 2.7056603773584906, 1, 1.6528301886792454, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 0, { 2.6754716981132076, 1, 1.6754716981132076, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 1, { 2.641509433962264, 1, 1.7132075471698114, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 2, { 2.6, 1, 1.7471698113207548, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 3, { 2.581132075471698, 1, 1.7849056603773585, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 4, { 2.547169811320755, 1, 1.8264150943396227, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 5, { 2.516981132075472, 1, 1.8566037735849057, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 6, { 2.486792452830189, 1, 1.890566037735849, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 7, { 2.4452830188679244, 1, 1.9358490566037736, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 8, { 2.4, 1, 1.9735849056603774, 0 } },
  { "Panasonic", "DMC-GH3", Shade              , 9, { 2.369811320754717, 1, 2.0226415094339623, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -9, { 2.8264150943396227, 1, 1.558490566037736, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -8, { 2.8, 1, 1.5811320754716982, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -7, { 2.750943396226415, 1, 1.6075471698113208, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -6, { 2.70188679245283, 1, 1.6377358490566039, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -5, { 2.671698113207547, 1, 1.649056603773585, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -4, { 2.641509433962264, 1, 1.6716981132075472, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -3, { 2.5924528301886793, 1, 1.7056603773584906, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -2, { 2.547169811320755, 1, 1.739622641509434, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , -1, { 2.5056603773584905, 1, 1.7622641509433963, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 0, { 2.479245283018868, 1, 1.7811320754716982, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 1, { 2.449056603773585, 1, 1.8226415094339623, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 2, { 2.4113207547169813, 1, 1.860377358490566, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 3, { 2.392452830188679, 1, 1.9018867924528302, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 4, { 2.362264150943396, 1, 1.9433962264150944, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 5, { 2.332075471698113, 1, 1.9773584905660377, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 6, { 2.30188679245283, 1, 2.0113207547169814, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 7, { 2.2641509433962264, 1, 2.060377358490566, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 8, { 2.2264150943396226, 1, 2.1018867924528304, 0 } },
  { "Panasonic", "DMC-GH3", Cloudy             , 9, { 2.1962264150943396, 1, 2.150943396226415, 0 } },
  { "Panasonic", "DMC-GH3", Tungsten           , 0, { 1.6484375, 1, 2.82421875, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -9, { 1.8830188679245283, 1, 2.4716981132075473, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -8, { 1.860377358490566, 1, 2.5056603773584905, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -7, { 1.830188679245283, 1, 2.550943396226415, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -6, { 1.7962264150943397, 1, 2.5924528301886793, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -5, { 1.7773584905660378, 1, 2.6150943396226416, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -4, { 1.758490566037736, 1, 2.6490566037735848, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -3, { 1.728301886792453, 1, 2.7056603773584906, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -2, { 1.6943396226415095, 1, 2.758490566037736, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , -1, { 1.6679245283018869, 1, 2.792452830188679, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 0, { 1.649056603773585, 1, 2.8264150943396227, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 1, { 1.630188679245283, 1, 2.890566037735849, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 2, { 1.6037735849056605, 1, 2.9471698113207547, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 3, { 1.5924528301886793, 1, 3.0150943396226415, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 4, { 1.569811320754717, 1, 3.079245283018868, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 5, { 1.5509433962264152, 1, 3.1358490566037736, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 6, { 1.5320754716981133, 1, 3.1924528301886794, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 7, { 1.5056603773584907, 1, 3.267924528301887, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 8, { 1.4830188679245282, 1, 3.3358490566037737, 0 } },
  { "Panasonic", "DMC-GH3", Incandescent       , 9, { 1.460377358490566, 1, 3.4113207547169813, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -9, { 2.762264150943396, 1, 1.5358490566037737, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -8, { 2.732075471698113, 1, 1.5547169811320756, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -7, { 2.6867924528301885, 1, 1.5849056603773586, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -6, { 2.637735849056604, 1, 1.6113207547169812, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -5, { 2.611320754716981, 1, 1.6226415094339623, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -4, { 2.581132075471698, 1, 1.6452830188679246, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -3, { 2.5358490566037735, 1, 1.679245283018868, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -2, { 2.486792452830189, 1, 1.7132075471698114, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , -1, { 2.449056603773585, 1, 1.7320754716981133, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 0, { 2.422641509433962, 1, 1.7547169811320755, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 1, { 2.392452830188679, 1, 1.7962264150943397, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 2, { 2.3547169811320754, 1, 1.830188679245283, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 3, { 2.3358490566037737, 1, 1.8716981132075472, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 4, { 2.309433962264151, 1, 1.9132075471698113, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 5, { 2.279245283018868, 1, 1.9471698113207547, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 6, { 2.249056603773585, 1, 1.9811320754716981, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 7, { 2.211320754716981, 1, 2.030188679245283, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 8, { 2.177358490566038, 1, 2.0716981132075474, 0 } },
  { "Panasonic", "DMC-GH3", Flash              , 9, { 2.147169811320755, 1, 2.120754716981132, 0 } },


  /* Firmware version 2.7 */
  { "Panasonic", "DMC-GH4", Daylight           , -9, { 2.7734375, 1, 1.64453125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -8, { 2.74609375, 1, 1.66796875, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -7, { 2.6953125, 1, 1.69921875, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -6, { 2.6484375, 1, 1.7265625, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -5, { 2.62109375, 1, 1.7421875, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -4, { 2.59375, 1, 1.765625, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -3, { 2.54296875, 1, 1.80078125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -2, { 2.49609375, 1, 1.8359375, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , -1, { 2.4609375, 1, 1.859375, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 0, { 2.4296875, 1, 1.8828125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 1, { 2.40234375, 1, 1.92578125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 2, { 2.36328125, 1, 1.9609375, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 3, { 2.34375, 1, 2.0078125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 4, { 2.31640625, 1, 2.05078125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 5, { 2.2890625, 1, 2.0859375, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 6, { 2.2578125, 1, 2.125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 7, { 2.22265625, 1, 2.17578125, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 8, { 2.18359375, 1, 2.21875, 0 } },
  { "Panasonic", "DMC-GH4", Daylight           , 9, { 2.15625, 1, 2.2734375, 0 } },
  { "Panasonic", "DMC-GH4", D55                , 0, { 2.328125, 1, 1.8984375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -9, { 3.12109375, 1, 1.42578125, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -8, { 3.08984375, 1, 1.44140625, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -7, { 3.03515625, 1, 1.46875, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -6, { 2.98046875, 1, 1.49609375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -5, { 2.94921875, 1, 1.5078125, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -4, { 2.91796875, 1, 1.52734375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -3, { 2.86328125, 1, 1.55859375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -2, { 2.80859375, 1, 1.58984375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , -1, { 2.765625, 1, 1.609375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 0, { 2.734375, 1, 1.62890625, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 1, { 2.703125, 1, 1.66796875, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 2, { 2.66015625, 1, 1.69921875, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 3, { 2.640625, 1, 1.734375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 4, { 2.60546875, 1, 1.7734375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 5, { 2.57421875, 1, 1.8046875, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 6, { 2.54296875, 1, 1.83984375, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 7, { 2.5, 1, 1.8828125, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 8, { 2.45703125, 1, 1.921875, 0 } },
  { "Panasonic", "DMC-GH4", Shade              , 9, { 2.42578125, 1, 1.96484375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -9, { 2.93359375, 1, 1.52734375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -8, { 2.90234375, 1, 1.546875, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -7, { 2.8515625, 1, 1.57421875, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -6, { 2.80078125, 1, 1.6015625, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -5, { 2.7734375, 1, 1.6171875, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -4, { 2.7421875, 1, 1.63671875, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -3, { 2.69140625, 1, 1.671875, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -2, { 2.640625, 1, 1.70703125, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , -1, { 2.6015625, 1, 1.7265625, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 0, { 2.5703125, 1, 1.74609375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 1, { 2.54296875, 1, 1.78515625, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 2, { 2.5, 1, 1.8203125, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 3, { 2.48046875, 1, 1.86328125, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 4, { 2.44921875, 1, 1.90234375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 5, { 2.421875, 1, 1.9375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 6, { 2.390625, 1, 1.97265625, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 7, { 2.3515625, 1, 2.01953125, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 8, { 2.30859375, 1, 2.05859375, 0 } },
  { "Panasonic", "DMC-GH4", Cloudy             , 9, { 2.28125, 1, 2.109375, 0 } },
  { "Panasonic", "DMC-GH4", Tungsten           , 0, { 1.765625, 1, 2.80859375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -9, { 2.01171875, 1, 2.45703125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -8, { 1.9921875, 1, 2.48828125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -7, { 1.95703125, 1, 2.53125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -6, { 1.921875, 1, 2.578125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -5, { 1.90234375, 1, 2.59765625, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -4, { 1.8828125, 1, 2.6328125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -3, { 1.84765625, 1, 2.6875, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -2, { 1.8125, 1, 2.7421875, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , -1, { 1.78515625, 1, 2.7734375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 0, { 1.765625, 1, 2.80859375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 1, { 1.7421875, 1, 2.87109375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 2, { 1.71484375, 1, 2.92578125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 3, { 1.703125, 1, 2.9921875, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 4, { 1.68359375, 1, 3.05859375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 5, { 1.66015625, 1, 3.11328125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 6, { 1.640625, 1, 3.16796875, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 7, { 1.61328125, 1, 3.24609375, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 8, { 1.5859375, 1, 3.3125, 0 } },
  { "Panasonic", "DMC-GH4", Incandescent       , 9, { 1.56640625, 1, 3.38671875, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -9, { 2.8359375, 1, 1.57421875, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -8, { 2.80859375, 1, 1.59375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -7, { 2.7578125, 1, 1.625, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -6, { 2.7109375, 1, 1.65234375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -5, { 2.6796875, 1, 1.66796875, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -4, { 2.65234375, 1, 1.6875, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -3, { 2.6015625, 1, 1.72265625, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -2, { 2.5546875, 1, 1.7578125, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , -1, { 2.515625, 1, 1.77734375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 0, { 2.48828125, 1, 1.80078125, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 1, { 2.45703125, 1, 1.84375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 2, { 2.41796875, 1, 1.87890625, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 3, { 2.3984375, 1, 1.91796875, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 4, { 2.37109375, 1, 1.9609375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 5, { 2.33984375, 1, 1.99609375, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 6, { 2.3125, 1, 2.03125, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 7, { 2.2734375, 1, 2.08203125, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 8, { 2.234375, 1, 2.125, 0 } },
  { "Panasonic", "DMC-GH4", Flash              , 9, { 2.203125, 1, 2.171875, 0 } },


  /* Firmware version 2.6 */
  { "Panasonic", "DC-GH5", Daylight, 0, { 2.2421875, 1, 1.953125, 0 } },
  { "Panasonic", "DC-GH5", Cloudy, 0, { 2.38671875, 1, 1.83203125, 0 } },
  { "Panasonic", "DC-GH5", Shade, 0, { 2.52734375, 1, 1.7421875, 0 } },
  { "Panasonic", "DC-GH5", Incandescent, 0, { 1.58203125, 1, 2.87109375, 0 } },
  { "Panasonic", "DC-GH5", Flash, 0, { 2.27734375, 1, 1.890625, 0 } },
  { "Panasonic", "DC-GH5", D55, 0, { 2.125, 1, 1.98828125, 0 } },
  { "Panasonic", "DC-GH5", Tungsten, 0, { 1.58203125, 1, 2.87109375, 0 } },

  /* Firmware version 1.1 */
  /* -9/+9 fine tuning is -9/+9 in blue-amber and zero in magenta-green */
  { "Panasonic", "DMC-GX1", Daylight, -9,	{ 2.256604, 1, 1.916981, 0 } },
  { "Panasonic", "DMC-GX1", Daylight, 0,	{ 2.611321, 1, 1.615094, 0 } },
  { "Panasonic", "DMC-GX1", Daylight, 9,	{ 3.060377, 1, 1.411321, 0 } },
  { "Panasonic", "DMC-GX1", Cloudy, -9,		{ 2.407547, 1, 1.796226, 0 } },
  { "Panasonic", "DMC-GX1", Cloudy, 0,		{ 2.792453, 1, 1.513208, 0 } },
  { "Panasonic", "DMC-GX1", Cloudy, 9,		{ 3.271698, 1, 1.324528, 0 } },
  { "Panasonic", "DMC-GX1", Shade, -9,		{ 2.626415, 1, 1.698113, 0 } },
  { "Panasonic", "DMC-GX1", Shade, 0,		{ 3.045283, 1, 1.430189, 0 } },
  { "Panasonic", "DMC-GX1", Shade, 9,		{ 3.569811, 1, 1.249057, 0 } },
  { "Panasonic", "DMC-GX1", Incandescent, -9,	{ 1.464151, 1, 2.864151, 0 } },
  { "Panasonic", "DMC-GX1", Incandescent, 0,	{ 1.698113, 1, 2.415094, 0 } },
  { "Panasonic", "DMC-GX1", Incandescent, 9,	{ 1.988679, 1, 2.109434, 0 } },
  { "Panasonic", "DMC-GX1", Flash, -9,		{ 2.683019, 1, 1.667925, 0 } },
  { "Panasonic", "DMC-GX1", Flash, 0,		{ 3.109434, 1, 1.407547, 0 } },
  { "Panasonic", "DMC-GX1", Flash, 9,		{ 3.645283, 1, 1.230189, 0 } },
  { "Panasonic", "DMC-GX1", "2700K", 0,		{ 1.392453, 1, 2.875472, 0 } },
  { "Panasonic", "DMC-GX1", "3000K", 0,		{ 1.566038, 1, 2.596226, 0 } },
  { "Panasonic", "DMC-GX1", "3300K", 0,		{ 1.728302, 1, 2.335849, 0 } },
  { "Panasonic", "DMC-GX1", "5000K", 0,		{ 2.464151, 1, 1.698113, 0 } },
  { "Panasonic", "DMC-GX1", "5500K", 0,		{ 2.611321, 1, 1.615094, 0 } },
  { "Panasonic", "DMC-GX1", "6500K", 0,		{ 2.860377, 1, 1.490566, 0 } },

  // Firmware version 1.3
  { "Panasonic", "DMC-GX7", Daylight           , -9, { 2.878787878787879, 1, 1.4318181818181819, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -8, { 2.8333333333333335, 1, 1.4507575757575757, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -7, { 2.784090909090909, 1, 1.4734848484848484, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -6, { 2.727272727272727, 1, 1.4924242424242424, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -5, { 2.6856060606060606, 1, 1.5113636363636365, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -4, { 2.647727272727273, 1, 1.5378787878787878, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -3, { 2.5984848484848486, 1, 1.5643939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -2, { 2.5416666666666665, 1, 1.5909090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , -1, { 2.5113636363636362, 1, 1.6022727272727273, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 0, { 2.4734848484848486, 1, 1.621212121212121, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 1, { 2.4356060606060606, 1, 1.6590909090909092, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 2, { 2.397727272727273, 1, 1.6893939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 3, { 2.359848484848485, 1, 1.7272727272727273, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 4, { 2.3295454545454546, 1, 1.7651515151515151, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 5, { 2.2916666666666665, 1, 1.7992424242424243, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 6, { 2.25, 1, 1.8371212121212122, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 7, { 2.231060606060606, 1, 1.8674242424242424, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 8, { 2.212121212121212, 1, 1.9015151515151516, 0 } },
  { "Panasonic", "DMC-GX7", Daylight           , 9, { 2.1742424242424243, 1, 1.9393939393939394, 0 } },
  { "Panasonic", "DMC-GX7", D55                , 0, { 2.3125, 1, 1.61328125, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -9, { 3.265151515151515, 1, 1.2651515151515151, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -8, { 3.2083333333333335, 1, 1.2840909090909092, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -7, { 3.1553030303030303, 1, 1.303030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -6, { 3.087121212121212, 1, 1.321969696969697, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -5, { 3.0454545454545454, 1, 1.3371212121212122, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -4, { 3.0, 1, 1.3598484848484849, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -3, { 2.946969696969697, 1, 1.3825757575757576, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -2, { 2.878787878787879, 1, 1.4053030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , -1, { 2.8484848484848486, 1, 1.4166666666666667, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 0, { 2.803030303030303, 1, 1.4356060606060606, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 1, { 2.7613636363636362, 1, 1.4659090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 2, { 2.715909090909091, 1, 1.496212121212121, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 3, { 2.6704545454545454, 1, 1.5303030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 4, { 2.640151515151515, 1, 1.5606060606060606, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 5, { 2.5946969696969697, 1, 1.5909090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 6, { 2.553030303030303, 1, 1.625, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 7, { 2.5303030303030303, 1, 1.6515151515151516, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 8, { 2.507575757575758, 1, 1.6818181818181819, 0 } },
  { "Panasonic", "DMC-GX7", Shade              , 9, { 2.462121212121212, 1, 1.7121212121212122, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -9, { 3.0265151515151514, 1, 1.3409090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -8, { 2.977272727272727, 1, 1.3598484848484849, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -7, { 2.9242424242424243, 1, 1.3825757575757576, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -6, { 2.8636363636363638, 1, 1.4015151515151516, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -5, { 2.821969696969697, 1, 1.4204545454545454, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -4, { 2.784090909090909, 1, 1.4431818181818181, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -3, { 2.731060606060606, 1, 1.4659090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -2, { 2.6704545454545454, 1, 1.4886363636363635, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , -1, { 2.640151515151515, 1, 1.503787878787879, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 0, { 2.602272727272727, 1, 1.518939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 1, { 2.5606060606060606, 1, 1.5568181818181819, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 2, { 2.518939393939394, 1, 1.5833333333333333, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 3, { 2.477272727272727, 1, 1.621212121212121, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 4, { 2.446969696969697, 1, 1.6553030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 5, { 2.4053030303030303, 1, 1.6856060606060606, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 6, { 2.367424242424242, 1, 1.7234848484848484, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 7, { 2.3446969696969697, 1, 1.753787878787879, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 8, { 2.3257575757575757, 1, 1.7803030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Cloudy             , 9, { 2.284090909090909, 1, 1.8181818181818181, 0 } },
  { "Panasonic", "DMC-GX7", Tungsten           , 0, { 1.71484375, 1, 2.33203125, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -9, { 1.996212121212121, 1, 2.0606060606060606, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -8, { 1.9621212121212122, 1, 2.087121212121212, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -7, { 1.928030303030303, 1, 2.125, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -6, { 1.8863636363636365, 1, 2.1515151515151514, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -5, { 1.8598484848484849, 1, 2.178030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -4, { 1.8333333333333333, 1, 2.215909090909091, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -3, { 1.803030303030303, 1, 2.25, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -2, { 1.7613636363636365, 1, 2.287878787878788, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , -1, { 1.7424242424242424, 1, 2.3068181818181817, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 0, { 1.7159090909090908, 1, 2.3333333333333335, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 1, { 1.6856060606060606, 1, 2.390151515151515, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 2, { 1.6590909090909092, 1, 2.4318181818181817, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 3, { 1.6325757575757576, 1, 2.4886363636363638, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 4, { 1.6136363636363635, 1, 2.5416666666666665, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 5, { 1.5871212121212122, 1, 2.587121212121212, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 6, { 1.5606060606060606, 1, 2.643939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 7, { 1.5454545454545454, 1, 2.6893939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 8, { 1.5340909090909092, 1, 2.734848484848485, 0 } },
  { "Panasonic", "DMC-GX7", Incandescent       , 9, { 1.5075757575757576, 1, 2.787878787878788, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -9, { 2.9545454545454546, 1, 1.3143939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -8, { 2.9053030303030303, 1, 1.3333333333333333, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -7, { 2.856060606060606, 1, 1.356060606060606, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -6, { 2.7992424242424243, 1, 1.375, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -5, { 2.757575757575758, 1, 1.3901515151515151, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -4, { 2.7196969696969697, 1, 1.4166666666666667, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -3, { 2.6704545454545454, 1, 1.4393939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -2, { 2.609848484848485, 1, 1.4621212121212122, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , -1, { 2.5795454545454546, 1, 1.4734848484848484, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 0, { 2.5416666666666665, 1, 1.4924242424242424, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 1, { 2.5, 1, 1.5265151515151516, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 2, { 2.4583333333333335, 1, 1.553030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 3, { 2.4204545454545454, 1, 1.5909090909090908, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 4, { 2.390151515151515, 1, 1.625, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 5, { 2.352272727272727, 1, 1.6553030303030303, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 6, { 2.3106060606060606, 1, 1.6893939393939394, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 7, { 2.2916666666666665, 1, 1.7196969696969697, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 8, { 2.272727272727273, 1, 1.746212121212121, 0 } },
  { "Panasonic", "DMC-GX7", Flash              , 9, { 2.231060606060606, 1, 1.7840909090909092, 0 } },


  { "Panasonic", "DMC-GX8", Daylight           , -9, { 2.5859375, 1, 1.68359375, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -8, { 2.55078125, 1, 1.69921875, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -7, { 2.50390625, 1, 1.73046875, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -6, { 2.46875, 1, 1.7578125, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -5, { 2.43359375, 1, 1.78125, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -4, { 2.40625, 1, 1.8125, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -3, { 2.37109375, 1, 1.83984375, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -2, { 2.328125, 1, 1.87109375, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , -1, { 2.30078125, 1, 1.89453125, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 0, { 2.2734375, 1, 1.91796875, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 1, { 2.24609375, 1, 1.9609375, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 2, { 2.2109375, 1, 2.0, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 3, { 2.1953125, 1, 2.05078125, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 4, { 2.16796875, 1, 2.09765625, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 5, { 2.1328125, 1, 2.140625, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 6, { 2.09765625, 1, 2.1796875, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 7, { 2.0859375, 1, 2.21484375, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 8, { 2.0703125, 1, 2.26171875, 0 } },
  { "Panasonic", "DMC-GX8", Daylight           , 9, { 2.04296875, 1, 2.3046875, 0 } },
  { "Panasonic", "DMC-GX8", D55                , 0, { 2.15625, 1, 1.92578125, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -9, { 2.9375, 1, 1.453125, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -8, { 2.8984375, 1, 1.46484375, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -7, { 2.84765625, 1, 1.4921875, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -6, { 2.80859375, 1, 1.51953125, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -5, { 2.765625, 1, 1.5390625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -4, { 2.73828125, 1, 1.5625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -3, { 2.6953125, 1, 1.58984375, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -2, { 2.64453125, 1, 1.6171875, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , -1, { 2.6171875, 1, 1.63671875, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 0, { 2.5859375, 1, 1.65625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 1, { 2.5546875, 1, 1.69140625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 2, { 2.515625, 1, 1.7265625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 3, { 2.4921875, 1, 1.76953125, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 4, { 2.46484375, 1, 1.80859375, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 5, { 2.421875, 1, 1.84765625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 6, { 2.3828125, 1, 1.87890625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 7, { 2.37109375, 1, 1.9140625, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 8, { 2.3515625, 1, 1.953125, 0 } },
  { "Panasonic", "DMC-GX8", Shade              , 9, { 2.3203125, 1, 1.9921875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -9, { 2.75, 1, 1.55078125, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -8, { 2.7109375, 1, 1.5625, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -7, { 2.6640625, 1, 1.58984375, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -6, { 2.625, 1, 1.6171875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -5, { 2.58984375, 1, 1.63671875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -4, { 2.55859375, 1, 1.66796875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -3, { 2.5234375, 1, 1.6953125, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -2, { 2.4765625, 1, 1.72265625, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , -1, { 2.4453125, 1, 1.7421875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 0, { 2.41796875, 1, 1.76171875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 1, { 2.390625, 1, 1.8046875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 2, { 2.3515625, 1, 1.83984375, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 3, { 2.33203125, 1, 1.88671875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 4, { 2.3046875, 1, 1.9296875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 5, { 2.265625, 1, 1.96875, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 6, { 2.23046875, 1, 2.00390625, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 7, { 2.21875, 1, 2.0390625, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 8, { 2.19921875, 1, 2.078125, 0 } },
  { "Panasonic", "DMC-GX8", Cloudy             , 9, { 2.171875, 1, 2.12109375, 0 } },
  { "Panasonic", "DMC-GX8", Tungsten           , 0, { 1.640625, 1, 2.828125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -9, { 1.86328125, 1, 2.484375, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -8, { 1.83984375, 1, 2.5078125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -7, { 1.8046875, 1, 2.55078125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -6, { 1.78125, 1, 2.59765625, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -5, { 1.75390625, 1, 2.62890625, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -4, { 1.734375, 1, 2.671875, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -3, { 1.7109375, 1, 2.71875, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -2, { 1.6796875, 1, 2.76171875, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , -1, { 1.66015625, 1, 2.796875, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 0, { 1.640625, 1, 2.828125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 1, { 1.62109375, 1, 2.89453125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 2, { 1.59375, 1, 2.94921875, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 3, { 1.58203125, 1, 3.02734375, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 4, { 1.5625, 1, 3.09375, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 5, { 1.53515625, 1, 3.16015625, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 6, { 1.51171875, 1, 3.21484375, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 7, { 1.50390625, 1, 3.26953125, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 8, { 1.4921875, 1, 3.3359375, 0 } },
  { "Panasonic", "DMC-GX8", Incandescent       , 9, { 1.47265625, 1, 3.40234375, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -9, { 2.61328125, 1, 1.60546875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -8, { 2.578125, 1, 1.6171875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -7, { 2.53125, 1, 1.6484375, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -6, { 2.49609375, 1, 1.67578125, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -5, { 2.4609375, 1, 1.69921875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -4, { 2.43359375, 1, 1.7265625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -3, { 2.3984375, 1, 1.75390625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -2, { 2.3515625, 1, 1.78515625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , -1, { 2.32421875, 1, 1.8046875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 0, { 2.30078125, 1, 1.828125, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 1, { 2.2734375, 1, 1.8671875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 2, { 2.234375, 1, 1.90625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 3, { 2.21875, 1, 1.953125, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 4, { 2.19140625, 1, 1.99609375, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 5, { 2.15625, 1, 2.0390625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 6, { 2.1171875, 1, 2.07421875, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 7, { 2.109375, 1, 2.11328125, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 8, { 2.09375, 1, 2.15625, 0 } },
  { "Panasonic", "DMC-GX8", Flash              , 9, { 2.06640625, 1, 2.19921875, 0 } },


  { "Panasonic", "DMC-GX85", Daylight, 0,     { 2.4296875, 1, 1.62109375, 0 } },
  { "Panasonic", "DMC-GX85", Cloudy, 0,       { 2.6015625, 1, 1.50390625, 0 } },
  { "Panasonic", "DMC-GX85", Shade, 0,        { 2.77734375, 1, 1.4296875, 0 } },
  { "Panasonic", "DMC-GX85", Incandescent, 0, { 1.6875, 1, 2.37109375, 0 } },
  { "Panasonic", "DMC-GX85", Flash, 0,        { 2.51953125, 1, 1.51171875, 0 } },

  // Firmware version 1.1
  { "Panasonic", "DMC-GM1", Daylight           , -9, { 2.875, 1, 1.4621212121212122, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -8, { 2.8257575757575757, 1, 1.481060606060606, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -7, { 2.7803030303030303, 1, 1.5075757575757576, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -6, { 2.7196969696969697, 1, 1.5265151515151516, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -5, { 2.6818181818181817, 1, 1.5454545454545454, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -4, { 2.643939393939394, 1, 1.571969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -3, { 2.5946969696969697, 1, 1.5984848484848484, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -2, { 2.537878787878788, 1, 1.625, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , -1, { 2.507575757575758, 1, 1.6363636363636365, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 0, { 2.4696969696969697, 1, 1.6590909090909092, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 1, { 2.4318181818181817, 1, 1.696969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 2, { 2.390151515151515, 1, 1.7272727272727273, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 3, { 2.352272727272727, 1, 1.7651515151515151, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 4, { 2.3257575757575757, 1, 1.8068181818181819, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 5, { 2.284090909090909, 1, 1.8371212121212122, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 6, { 2.246212121212121, 1, 1.878787878787879, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 7, { 2.227272727272727, 1, 1.9090909090909092, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 8, { 2.2083333333333335, 1, 1.9431818181818181, 0 } },
  { "Panasonic", "DMC-GM1", Daylight           , 9, { 2.1704545454545454, 1, 1.981060606060606, 0 } },
  { "Panasonic", "DMC-GM1", D55                , 0, { 2.28515625, 1, 1.6484375, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -9, { 3.303030303030303, 1, 1.2916666666666667, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -8, { 3.25, 1, 1.3106060606060606, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -7, { 3.1931818181818183, 1, 1.3333333333333333, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -6, { 3.128787878787879, 1, 1.3522727272727273, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -5, { 3.0833333333333335, 1, 1.3674242424242424, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -4, { 3.037878787878788, 1, 1.3901515151515151, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -3, { 2.981060606060606, 1, 1.4128787878787878, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -2, { 2.9166666666666665, 1, 1.4356060606060606, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , -1, { 2.882575757575758, 1, 1.446969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 0, { 2.840909090909091, 1, 1.4659090909090908, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 1, { 2.7954545454545454, 1, 1.5, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 2, { 2.75, 1, 1.5303030303030303, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 3, { 2.7045454545454546, 1, 1.5643939393939394, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 4, { 2.6704545454545454, 1, 1.5984848484848484, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 5, { 2.628787878787879, 1, 1.625, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 6, { 2.5833333333333335, 1, 1.6590909090909092, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 7, { 2.5606060606060606, 1, 1.6893939393939394, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 8, { 2.537878787878788, 1, 1.7196969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Shade              , 9, { 2.492424242424242, 1, 1.753787878787879, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -9, { 3.037878787878788, 1, 1.371212121212121, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -8, { 2.9886363636363638, 1, 1.3901515151515151, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -7, { 2.9356060606060606, 1, 1.4128787878787878, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -6, { 2.875, 1, 1.4318181818181819, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -5, { 2.8333333333333335, 1, 1.4507575757575757, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -4, { 2.7916666666666665, 1, 1.4772727272727273, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -3, { 2.742424242424242, 1, 1.5, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -2, { 2.6818181818181817, 1, 1.5227272727272727, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , -1, { 2.6515151515151514, 1, 1.5378787878787878, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 0, { 2.609848484848485, 1, 1.553030303030303, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 1, { 2.5681818181818183, 1, 1.5909090909090908, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 2, { 2.5265151515151514, 1, 1.621212121212121, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 3, { 2.4886363636363638, 1, 1.6590909090909092, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 4, { 2.4583333333333335, 1, 1.6931818181818181, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 5, { 2.4166666666666665, 1, 1.7234848484848484, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 6, { 2.375, 1, 1.7613636363636365, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 7, { 2.356060606060606, 1, 1.7916666666666667, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 8, { 2.3333333333333335, 1, 1.821969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Cloudy             , 9, { 2.2954545454545454, 1, 1.8598484848484849, 0 } },
  { "Panasonic", "DMC-GM1", Tungsten           , 0, { 1.61328125, 1, 2.3828125, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -9, { 1.878787878787879, 1, 2.102272727272727, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -8, { 1.8484848484848484, 1, 2.132575757575758, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -7, { 1.8143939393939394, 1, 2.1704545454545454, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -6, { 1.7765151515151516, 1, 2.196969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -5, { 1.753787878787879, 1, 2.2234848484848486, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -4, { 1.7272727272727273, 1, 2.2613636363636362, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -3, { 1.696969696969697, 1, 2.2992424242424243, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -2, { 1.6590909090909092, 1, 2.337121212121212, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , -1, { 1.6401515151515151, 1, 2.356060606060606, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 0, { 1.6136363636363635, 1, 2.3863636363636362, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 1, { 1.5909090909090908, 1, 2.4393939393939394, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 2, { 1.5643939393939394, 1, 2.484848484848485, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 3, { 1.5378787878787878, 1, 2.5416666666666665, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 4, { 1.518939393939394, 1, 2.5984848484848486, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 5, { 1.4924242424242424, 1, 2.643939393939394, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 6, { 1.4696969696969697, 1, 2.7007575757575757, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 7, { 1.4545454545454546, 1, 2.75, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 8, { 1.4431818181818181, 1, 2.7954545454545454, 0 } },
  { "Panasonic", "DMC-GM1", Incandescent       , 9, { 1.4166666666666667, 1, 2.8484848484848486, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -9, { 2.946969696969697, 1, 1.321969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -8, { 2.893939393939394, 1, 1.3371212121212122, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -7, { 2.8446969696969697, 1, 1.3598484848484849, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -6, { 2.787878787878788, 1, 1.378787878787879, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -5, { 2.746212121212121, 1, 1.3977272727272727, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -4, { 2.7083333333333335, 1, 1.4204545454545454, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -3, { 2.659090909090909, 1, 1.4431818181818181, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -2, { 2.5984848484848486, 1, 1.4659090909090908, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , -1, { 2.5681818181818183, 1, 1.4772727272727273, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 0, { 2.5303030303030303, 1, 1.496212121212121, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 1, { 2.4886363636363638, 1, 1.5303030303030303, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 2, { 2.4507575757575757, 1, 1.5606060606060606, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 3, { 2.412878787878788, 1, 1.5946969696969697, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 4, { 2.382575757575758, 1, 1.628787878787879, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 5, { 2.340909090909091, 1, 1.6590909090909092, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 6, { 2.303030303030303, 1, 1.6931818181818181, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 7, { 2.2803030303030303, 1, 1.7234848484848484, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 8, { 2.2613636363636362, 1, 1.753787878787879, 0 } },
  { "Panasonic", "DMC-GM1", Flash              , 9, { 2.2234848484848486, 1, 1.7878787878787878, 0 } },


  /* Firmware version 1.1 */
  { "Panasonic", "DMC-GM5", Daylight, 0,        { 2.394531, 1, 1.554687, 0 } },
  { "Panasonic", "DMC-GM5", Cloudy, 0,          { 2.621093, 1, 1.437500, 0 } },
  { "Panasonic", "DMC-GM5", Shade, 0,           { 2.781250, 1, 1.371093, 0 } },
  { "Panasonic", "DMC-GM5", Incandescent, 0,    { 1.640625, 1, 2.253906, 0 } },
  { "Panasonic", "DMC-GM5", Flash, 0,           { 2.492187, 1, 1.402343, 0 } },

  { "Panasonic", "DMC-L1", Daylight, 0,		{ 1.980989, 1, 1.444867, 0 } },
  { "Panasonic", "DMC-L1", Cloudy, 0,		{ 2.129278, 1, 1.300380, 0 } },
  { "Panasonic", "DMC-L1", Shade, 0,		{ 2.361217, 1, 1.167300, 0 } },
  { "Panasonic", "DMC-L1", Incandescent, 0,	{ 1.368821, 1, 2.091255, 0 } },
  /* Flash multipliers are variable */
  { "Panasonic", "DMC-L1", Flash, 0,		{ 2.319392, 1, 1.053232, 0 } },

  /* DMC-L1 Kelvin presets */
  { "Panasonic", "DMC-L1", "2700K", 0,		{ 1.285171, 1, 2.520913, 0 } },
  { "Panasonic", "DMC-L1", "3000K", 0,		{ 1.406844, 1, 2.209126, 0 } },
  { "Panasonic", "DMC-L1", "3300K", 0,		{ 1.524715, 1, 2.163498, 0 } },
  { "Panasonic", "DMC-L1", "5000K", 0,		{ 1.942966, 1, 1.528517, 0 } },
  { "Panasonic", "DMC-L1", "5500K", 0,		{ 2.019011, 1, 1.403042, 0 } },
  { "Panasonic", "DMC-L1", "6500K", 0,		{ 2.216730, 1, 1.243346, 0 } },

  { "Panasonic", "DMC-LX1", Daylight, 0,	{ 1.837121, 1, 1.484848, 0 } },
  { "Panasonic", "DMC-LX1", Cloudy, 0,		{ 2.003788, 1, 1.310606, 0 } },
  { "Panasonic", "DMC-LX1", Incandescent, 0,	{ 1.098485, 1, 2.272727, 0 } },

  { "Panasonic", "DMC-LX2", Daylight, -3,	{ 2.456274, 1, 1.806084, 0 } },
  { "Panasonic", "DMC-LX2", Daylight, 0,	{ 2.114068, 1, 1.726236, 0 } },
  { "Panasonic", "DMC-LX2", Daylight, 3,	{ 1.916350, 1, 1.585551, 0 } },
  { "Panasonic", "DMC-LX2", Cloudy, -3,		{ 2.714829, 1, 1.650190, 0 } },
  { "Panasonic", "DMC-LX2", Cloudy, 0,		{ 2.338403, 1, 1.577947, 0 } },
  { "Panasonic", "DMC-LX2", Cloudy, 3,		{ 2.121673, 1, 1.448669, 0 } },
  { "Panasonic", "DMC-LX2", Shade, -3,		{ 2.939163, 1, 1.577947, 0 } },
  { "Panasonic", "DMC-LX2", Shade, 0,		{ 2.532319, 1, 1.509506, 0 } },
  { "Panasonic", "DMC-LX2", Shade, 3,		{ 2.292776, 1, 1.384030, 0 } },
  { "Panasonic", "DMC-LX2", Incandescent, -3,	{ 1.581749, 1, 2.524715, 0 } },
  { "Panasonic", "DMC-LX2", Incandescent, 0,	{ 1.365019, 1, 2.410646, 0 } },
  { "Panasonic", "DMC-LX2", Incandescent, 3,	{ 1.235741, 1, 2.212928, 0 } },

  { "Panasonic", "DMC-LX3", Daylight, 0,	{ 2.022814, 1, 1.623574, 0 } },
  { "Panasonic", "DMC-LX3", Cloudy, 0,		{ 2.224335, 1, 1.520913, 0 } },
  { "Panasonic", "DMC-LX3", Shade, 0,		{ 2.475285, 1, 1.399240, 0 } },
  { "Panasonic", "DMC-LX3", Flash, 0,		{ 2.296578, 1, 1.482890, 0 } },
  { "Panasonic", "DMC-LX3", Incandescent, 0,	{ 1.346008, 1, 2.269962, 0 } },

  { "Panasonic", "DMC-LX5", Daylight, 0,	{ 1.9202, 1, 1.7567, 0 } },
  { "Panasonic", "DMC-LX5", Cloudy, 0,		{ 2.0760, 1, 1.6388, 0 } },
  { "Panasonic", "DMC-LX5", Shade, 0,		{ 2.1635, 1, 1.5817, 0 } },
  { "Panasonic", "DMC-LX5", Flash, 0,		{ 2.0760, 1, 1.6388, 0 } },
  { "Panasonic", "DMC-LX5", Incandescent, 0,	{ 1.2243, 1, 2.5741, 0 } },

  { "Panasonic", "DMC-LX7", Daylight, -9,	{ 2.1358, 1, 1.5547, 0 } },
  { "Panasonic", "DMC-LX7", Daylight, 0,	{ 1.8906, 1, 1.7698, 0 } },
  { "Panasonic", "DMC-LX7", Daylight, 9,	{ 1.6755, 1, 2.0868, 0 } },
  { "Panasonic", "DMC-LX7", Cloudy, -9,		{ 2.2906, 1, 1.4226, 0 } },
  { "Panasonic", "DMC-LX7", Cloudy, 0,		{ 2.0341, 1, 1.6250, 0 } },
  { "Panasonic", "DMC-LX7", Cloudy, 9,		{ 1.8000, 1, 1.9132, 0 } },
  { "Panasonic", "DMC-LX7", Shade, -9,		{ 2.4415, 1, 1.3283, 0 } },
  { "Panasonic", "DMC-LX7", Shade, 0,		{ 2.1623, 1, 1.5132, 0 } },
  { "Panasonic", "DMC-LX7", Shade, 9,		{ 1.9170, 1, 1.7849, 0 } },
  { "Panasonic", "DMC-LX7", Flash, -9,		{ 2.3660, 1, 1.4679, 0 } },
  { "Panasonic", "DMC-LX7", Flash, 0,		{ 2.0943, 1, 1.6679, 0 } },
  { "Panasonic", "DMC-LX7", Flash, 9,		{ 1.8566, 1, 1.9698, 0 } },
  { "Panasonic", "DMC-LX7", Incandescent, -9,	{ 1.4566, 1, 2.4377, 0 } },
  { "Panasonic", "DMC-LX7", Incandescent, 0,	{ 1.2917, 1, 2.7841, 0 } },
  { "Panasonic", "DMC-LX7", Incandescent, 9,	{ 1.1434, 1, 3.2755, 0 } },

  { "Panasonic", "DMC-LX100", Daylight, 0,	{ 2.316406, 1, 1.601563, 0 } },
  { "Panasonic", "DMC-LX100", Cloudy, 0,	{ 2.468750, 1, 1.484375, 0 } },
  { "Panasonic", "DMC-LX100", Shade, 0,		{ 2.640625, 1, 1.410156, 0 } },
  { "Panasonic", "DMC-LX100", Flash, 0,		{ 2.410156, 1, 1.578125, 0 } },
  { "Panasonic", "DMC-LX100", Incandescent, 0,	{ 1.585938, 1, 2.382813, 0 } },

  { "Panasonic", "DC-LX100M2", Daylight, -9,     { 2.56640625, 1, 1.921875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -8,     { 2.5390625, 1, 1.953125, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -7,     { 2.50390625, 1, 1.98046875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -6,     { 2.46875, 1, 2.0078125, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -5,     { 2.4375, 1, 2.0390625, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -4,     { 2.40625, 1, 2.06640625, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -3,     { 2.37109375, 1, 2.09375, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -2,     { 2.33984375, 1, 2.12109375, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, -1,     { 2.30859375, 1, 2.15234375, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 0,      { 2.2734375, 1, 2.1796875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 1,      { 2.2421875, 1, 2.23046875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 2,      { 2.2109375, 1, 2.28125, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 3,      { 2.17578125, 1, 2.33203125, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 4,      { 2.14453125, 1, 2.38671875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 5,      { 2.11328125, 1, 2.4375, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 6,      { 2.078125, 1, 2.48828125, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 7,      { 2.046875, 1, 2.54296875, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 8,      { 2.01171875, 1, 2.59375, 0 } },
  { "Panasonic", "DC-LX100M2", Daylight, 9,      { 1.98046875, 1, 2.6484375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -9,        { 2.84375, 1, 1.69921875, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -8,        { 2.80859375, 1, 1.72265625, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -7,        { 2.7734375, 1, 1.75, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -6,        { 2.734375, 1, 1.7734375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -5,        { 2.69921875, 1, 1.80078125, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -4,        { 2.6640625, 1, 1.82421875, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -3,        { 2.625, 1, 1.84765625, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -2,        { 2.58984375, 1, 1.87109375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, -1,        { 2.5546875, 1, 1.8984375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 0,         { 2.51953125, 1, 1.921875, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 1,         { 2.48046875, 1, 1.96875, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 2,         { 2.4453125, 1, 2.015625, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 3,         { 2.41015625, 1, 2.0625, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 4,         { 2.37109375, 1, 2.109375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 5,         { 2.33984375, 1, 2.15234375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 6,         { 2.30078125, 1, 2.19921875, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 7,         { 2.265625, 1, 2.24609375, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 8,         { 2.2265625, 1, 2.2890625, 0 } },
  { "Panasonic", "DC-LX100M2", Shade, 9,         { 2.19140625, 1, 2.3359375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -9,       { 2.71875, 1, 1.80078125, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -8,       { 2.6875, 1, 1.828125, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -7,       { 2.65234375, 1, 1.85546875, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -6,       { 2.6171875, 1, 1.87890625, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -5,       { 2.58203125, 1, 1.90625, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -4,       { 2.546875, 1, 1.93359375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -3,       { 2.51171875, 1, 1.9609375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -2,       { 2.4765625, 1, 1.984375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, -1,       { 2.4453125, 1, 2.01171875, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 0,        { 2.41015625, 1, 2.0390625, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 1,        { 2.375, 1, 2.0859375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 2,        { 2.33984375, 1, 2.13671875, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 3,        { 2.3046875, 1, 2.18359375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 4,        { 2.26953125, 1, 2.234375, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 5,        { 2.23828125, 1, 2.28125, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 6,        { 2.203125, 1, 2.33203125, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 7,        { 2.16796875, 1, 2.37890625, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 8,        { 2.1328125, 1, 2.4296875, 0 } },
  { "Panasonic", "DC-LX100M2", Cloudy, 9,        { 2.09765625, 1, 2.4765625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -9, { 1.9375, 1, 2.76953125, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -8, { 1.9140625, 1, 2.8125, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -7, { 1.88671875, 1, 2.8515625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -6, { 1.86328125, 1, 2.89453125, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -5, { 1.83984375, 1, 2.9375, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -4, { 1.8125, 1, 2.9765625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -3, { 1.7890625, 1, 3.015625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -2, { 1.765625, 1, 3.0546875, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, -1, { 1.7421875, 1, 3.09765625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 0,  { 1.71484375, 1, 3.13671875, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 1,  { 1.69140625, 1, 3.2109375, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 2,  { 1.66796875, 1, 3.2890625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 3,  { 1.640625, 1, 3.36328125, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 4,  { 1.6171875, 1, 3.4375, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 5,  { 1.59375, 1, 3.51171875, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 6,  { 1.56640625, 1, 3.5859375, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 7,  { 1.54296875, 1, 3.6640625, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 8,  { 1.515625, 1, 3.734375, 0 } },
  { "Panasonic", "DC-LX100M2", Incandescent, 9,  { 1.4921875, 1, 3.8125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -9,        { 2.6640625, 1, 1.75, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -8,        { 2.6328125, 1, 1.77734375, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -7,        { 2.59765625, 1, 1.80078125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -6,        { 2.5625, 1, 1.828125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -5,        { 2.53125, 1, 1.85546875, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -4,        { 2.49609375, 1, 1.87890625, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -3,        { 2.4609375, 1, 1.90625, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -2,        { 2.4296875, 1, 1.9296875, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, -1,        { 2.39453125, 1, 1.95703125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 0,         { 2.359375, 1, 1.984375, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 1,         { 2.328125, 1, 2.03125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 2,         { 2.29296875, 1, 2.078125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 3,         { 2.2578125, 1, 2.125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 4,         { 2.2265625, 1, 2.171875, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 5,         { 2.19140625, 1, 2.21875, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 6,         { 2.15625, 1, 2.265625, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 7,         { 2.125, 1, 2.3125, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 8,         { 2.08984375, 1, 2.359375, 0 } },
  { "Panasonic", "DC-LX100M2", Flash, 9,         { 2.0546875, 1, 2.41015625, 0 } },
  { "Panasonic", "DC-LX100M2", "2700K", 0,       { 1.51171875, 1, 3.68359375, 0 } },
  { "Panasonic", "DC-LX100M2", "3000K", 0,       { 1.61328125, 1, 3.34765625, 0 } },
  { "Panasonic", "DC-LX100M2", "3300K", 0,       { 1.71484375, 1, 3.02734375, 0 } },
  { "Panasonic", "DC-LX100M2", "5000K", 0,       { 2.16796875, 1, 2.2890625, 0 } },
  { "Panasonic", "DC-LX100M2", "5500K", 0,       { 2.2734375, 1, 2.1796875, 0 } },
  { "Panasonic", "DC-LX100M2", "6500K", 0,       { 2.4375, 1, 2.0078125, 0 } },

  {"Panasonic", "DMC-LF1", Daylight, 0,     { 1.912879, 1, 1.75, 0 } },
  {"Panasonic", "DMC-LF1", Cloudy, 0,       { 2.037879, 1, 1.609848, 0 } },
  {"Panasonic", "DMC-LF1", Shade, 0,        { 2.159091, 1, 1.511364, 0 } },
  {"Panasonic", "DMC-LF1", Incandescent, 0, { 1.268939, 1, 2.799242, 0 } },

  { "Leica", "D-LUX 6", Daylight, 0,		{ 1.992453, 1, 1.750943, 0 } },
  { "Leica", "D-LUX 6", Cloudy, 0,		{ 2.143396, 1, 1.603774, 0 } },
  { "Leica", "D-LUX 6", Shade, 0,		{ 2.286792, 1, 1.498113, 0 } },
  { "Leica", "D-LUX 6", Flash, 0,		{ 2.219697, 1, 1.659091, 0 } },
  { "Leica", "D-LUX 6", Incandescent, 0,	{ 1.335849, 1, 2.762264, 0 } },

  /* It seems that the *ist D WB settings are not really presets. */
  { "Pentax", "*ist D", Daylight, 0,		{ 1.460938, 1, 1.019531, 0 } },
  { "Pentax", "*ist D", Shade, 0,		{ 1.734375, 1, 1.000000, 0 } },
  { "Pentax", "*ist D", Cloudy, 0,		{ 1.634921, 1.015873, 1, 0 } },
  { "Pentax", "*ist D", DaylightFluorescent, 0,	{ 1.657025, 1.057851, 1, 0 } },
  { "Pentax", "*ist D", NeutralFluorescent, 0,	{ 1.425781, 1, 1.117188, 0 } },
  { "Pentax", "*ist D", WhiteFluorescent, 0,	{ 1.328125, 1, 1.210938, 0 } },
  { "Pentax", "*ist D", Tungsten, 0,		{ 1.000000, 1, 2.226563, 0 } },
  { "Pentax", "*ist D", Flash, 0,		{ 1.750000, 1, 1.000000, 0 } },

  /* It seems that the *ist DL WB settings are not really presets. */
  { "Pentax", "*ist DL", Daylight, 0,		{ 1.546875, 1, 1.007812, 0 } },
  { "Pentax", "*ist DL", Shade, 0,		{ 1.933594, 1, 1.027344, 0 } },
  { "Pentax", "*ist DL", Cloudy, 0,		{ 1.703125, 1, 1.003906, 0 } },
  { "Pentax", "*ist DL", DaylightFluorescent, 0, { 2.593909, 1.299492, 1, 0 } },
  { "Pentax", "*ist DL", NeutralFluorescent, 0,	{ 1.539062, 1, 1.003906, 0 } },
  { "Pentax", "*ist DL", WhiteFluorescent, 0,	{ 1.390625, 1, 1.117188, 0 } },
  { "Pentax", "*ist DL", Tungsten, 0,		{ 1.000000, 1, 2.074219, 0 } },
  { "Pentax", "*ist DL", Flash, 0,		{ 1.621094, 1, 1.027344, 0 } },

  { "Pentax", "*ist DL2", Daylight, 0, { 1.4913330078125, 1, 1.0, 0 } },
  { "Pentax", "*ist DL2", Shade, 0, { 1.7974853515625, 1, 0.76318359375, 0 } },
  { "Pentax", "*ist DL2", Cloudy, 0, { 1.6123046875, 1, 0.896728515625, 0 } },
  { "Pentax", "*ist DL2", Tungsten, 0, { 0.846923828125, 1, 2.078125, 0 } },
  { "Pentax", "*ist DL2", DaylightFluorescent, 0, { 1.7481689453125, 1, 1.01513671875, 0 } },
  { "Pentax", "*ist DL2", NeutralFluorescent, 0, { 1.4468994140625, 1, 1.133544921875, 0 } },
  { "Pentax", "*ist DL2", WhiteFluorescent, 0, { 1.286376953125, 1, 1.3778076171875, 0 } },
  { "Pentax", "*ist DL2", Flash, 0, { 1.5531005859375, 1, 0.93701171875, 0 } },

  /* It seems that the *ist DS WB settings are not really presets. */
  { "Pentax", "*ist DS", Daylight, 0,		{ 1.632812, 1, 1.000000, 0 } },
  { "Pentax", "*ist DS", Shade, 0,		{ 1.964844, 1, 1.000000, 0 } },
  { "Pentax", "*ist DS", Cloudy, 0,		{ 1.761719, 1, 1.000000, 0 } },
  { "Pentax", "*ist DS", DaylightFluorescent, 0, { 1.910156, 1, 1.000000, 0 } },
  { "Pentax", "*ist DS", NeutralFluorescent, 0,	{ 1.521569, 1.003922, 1, 0 } },
  { "Pentax", "*ist DS", WhiteFluorescent, 0,	{ 1.496094, 1, 1.023438, 0 } },
  { "Pentax", "*ist DS", Tungsten, 0,		{ 1.000000, 1, 2.027344, 0 } },
  { "Pentax", "*ist DS", Flash, 0,		{ 1.695312, 1, 1.000000, 0 } },

  { "Pentax", "K10D", Daylight, 0,		{ 1.660156, 1, 1.066406, 0 } },
  { "Pentax", "K10D", Shade, 0,			{ 2.434783, 1.236715, 1, 0 } },
  { "Pentax", "K10D", Cloudy, 0,		{ 1.872428, 1.053498, 1, 0 } },
  { "Pentax", "K10D", DaylightFluorescent, 0,	{ 2.121094, 1, 1.078125, 0 } },
  { "Pentax", "K10D", NeutralFluorescent, 0,	{ 1.773438, 1, 1.226562, 0 } },
  { "Pentax", "K10D", WhiteFluorescent, 0,	{ 1.597656, 1, 1.488281, 0 } },
  { "Pentax", "K10D", Tungsten, 0,		{ 1.000000, 1, 2.558594, 0 } },
  { "Pentax", "K10D", Flash, 0,			{ 1.664062, 1, 1.046875, 0 } },

  /* DSP Firmware version 1.04.00.11 */
  { "Pentax", "K20D", Daylight, 0,		{ 1.5787353515625, 1.0, 1.263427734375, 0.0 } },
  { "Pentax", "K20D", Shade, 0,			{ 1.80029296875, 1.0, 0.9613037109375, 0.0 } },
  { "Pentax", "K20D", Cloudy, 0,		{ 1.6737060546875, 1.0, 1.116943359375, 0.0 } },
  { "Pentax", "K20D", DaylightFluorescent, 0,	{ 2.089111328125, 1.0, 1.190185546875, 0.0 } },
  { "Pentax", "K20D", NeutralFluorescent, 0,	{ 1.7725830078125, 1.0, 1.35498046875, 0.0 } },
  { "Pentax", "K20D", WhiteFluorescent, 0,	{ 1.6539306640625, 1.0, 1.6845703125, 0.0 } },
  { "Pentax", "K20D", Tungsten, 0,		{ 1.012939453125, 1.0, 2.34375, 0.0 } },
  { "Pentax", "K20D", Flash, 0,			{ 1.6737060546875, 1.0, 1.190185546875, 0.0 } },

  { "Pentax", "K100D", Daylight, 0,		{ 1.468750, 1, 1.023438, 0 } },
  { "Pentax", "K100D", Shade, 0,		{ 1.769531, 1, 1.000000, 0 } },
  { "Pentax", "K100D", Cloudy, 0,		{ 1.589844, 1, 1.000000, 0 } },
  { "Pentax", "K100D", DaylightFluorescent, 0,	{ 1.722656, 1, 1.039063, 0 } },
  { "Pentax", "K100D", NeutralFluorescent, 0,	{ 1.425781, 1, 1.160156, 0 } },
  { "Pentax", "K100D", WhiteFluorescent, 0,	{ 1.265625, 1, 1.414063, 0 } },
  { "Pentax", "K100D", Tungsten, 0,		{ 1, 1.015873, 2.055556, 0 } },
  { "Pentax", "K100D", Flash, 0,		{ 1.527344, 1, 1.000000, 0 } },

  { "Pentax", "K100D Super", Daylight, 0,	{ 1.593750, 1, 1.011719, 0 } },
  { "Pentax", "K100D Super", Shade, 0,		{ 1.917969, 1, 1.000000, 0 } },
  { "Pentax", "K100D Super", Cloudy, 0,		{ 1.703125, 1, 1.015625, 0 } },
  { "Pentax", "K100D Super", DaylightFluorescent, 0, { 1.708502, 1.036437, 1, 0 } },
  { "Pentax", "K100D Super", NeutralFluorescent, 0, { 1.634538, 1.028112, 1, 0 } },
  { "Pentax", "K100D Super", WhiteFluorescent, 0, { 1.425781, 1, 1.136719, 0 } },
  { "Pentax", "K100D Super", Tungsten, 0,	{ 1.015625, 1, 2.046875, 0 } },
  { "Pentax", "K100D Super", Flash, 0,		{ 1.670588, 1.003922, 1, 0 } },

  { "Pentax", "K110D", Daylight, 0,		{ 1.468750, 1, 1.023438, 0 } },
  { "Pentax", "K110D", Shade, 0,		{ 1.769531, 1, 1.000000, 0 } },
  { "Pentax", "K110D", Cloudy, 0,		{ 1.589844, 1, 1.000000, 0 } },
  { "Pentax", "K110D", DaylightFluorescent, 0,	{ 1.722656, 1, 1.039063, 0 } },
  { "Pentax", "K110D", NeutralFluorescent, 0,	{ 1.425781, 1, 1.160156, 0 } },
  { "Pentax", "K110D", WhiteFluorescent, 0,	{ 1.265625, 1, 1.414063, 0 } },
  { "Pentax", "K110D", Tungsten, 0,		{ 1, 1.015873, 2.055556, 0 } },
  { "Pentax", "K110D", Flash, 0,		{ 1.527344, 1, 1.000000, 0 } },

  { "Pentax", "K200D", Daylight, 0,		{ 1.804688, 1, 1.304688, 0 } },
  { "Pentax", "K200D", Shade, 0,		{ 2.140625, 1, 1.085937, 0 } },
  { "Pentax", "K200D", Cloudy, 0,		{ 1.957031, 1, 1.179687, 0 } },
  { "Pentax", "K200D", DaylightFluorescent, 0,	{ 2.121094, 1, 1.195313, 0 } },
  { "Pentax", "K200D", NeutralFluorescent, 0,	{ 1.773438, 1, 1.359375, 0 } },
  { "Pentax", "K200D", WhiteFluorescent, 0,	{ 1.597656, 1, 1.648437, 0 } },
  { "Pentax", "K200D", Tungsten, 0,		{ 1.000000, 1, 2.835937, 0 } },
  { "Pentax", "K200D", Flash, 0,		{ 1.917969, 1, 1.214844, 0 } },

  /* Pentax 645D Firmware Version 645D Ver 1.00 */
  { "Pentax", "645D", Daylight, 0,            { 1.8056640625, 1, 1.327392578125, 0 } },
  { "Pentax", "645D", Shade, 0,               { 2.0926513671875, 1, 0.9832763671875, 0 } },
  { "Pentax", "645D", Cloudy, 0,              { 1.916015625, 1, 1.1060791015625, 0 } },
  { "Pentax", "645D", Tungsten, 0,            { 1.1302490234375, 1, 2.09765625, 0 } },
  { "Pentax", "645D", DaylightFluorescent, 0,        { 2.2073974609375, 1, 1.2945556640625, 0 } },
  { "Pentax", "645D", WhiteFluorescent, 0,        { 1.8895263671875, 1, 1.45849609375, 0 } },
  { "Pentax", "645D", CoolWhiteFluorescent, 0,        { 1.748291015625, 1, 1.737060546875, 0 } },
  { "Pentax", "645D", Flash, 0,               { 1.9866943359375, 1, 1.08154296875, 0 } },
  { "Pentax", "645D", WarmWhiteFluorescent, 0,        { 1.4921875, 1, 2.146728515625, 0 } },

  /* Pentax K-01 Firmware Version 1.00 */
  { "Pentax", "K-01", Daylight, 0,            { 2.197021484375, 1, 1.6136474609375, 0 } },
  { "Pentax", "K-01", Shade, 0,               { 2.6251220703125, 1, 1.204345703125, 0 } },
  { "Pentax", "K-01", Cloudy, 0,              { 2.3671875, 1, 1.3797607421875, 0 } },
  { "Pentax", "K-01", Tungsten, 0,            { 1.3203125, 1, 2.9935302734375, 0 } },
  { "Pentax", "K-01", DaylightFluorescent, 0,        { 2.7127685546875, 1, 1.55517578125, 0 } },
  { "Pentax", "K-01", WhiteFluorescent, 0,        { 2.325927734375, 1, 1.7305908203125, 0 } },
  { "Pentax", "K-01", CoolWhiteFluorescent, 0,        { 2.114501953125, 1, 2.139892578125, 0 } },
  { "Pentax", "K-01", Flash, 0,               { 2.490966796875, 1, 1.4849853515625, 0 } },
  { "Pentax", "K-01", WarmWhiteFluorescent, 0,        { 1.7689208984375, 1, 2.677734375, 0 } },

  /* Pentax K-1 Firmware Version 1.52 */
  { "Pentax", "K-1", Daylight            , -7, { 2.3671875, 1, 1.49609375, 0 } },
  { "Pentax", "K-1", Daylight            , -6, { 2.33203125, 1, 1.51953125, 0 } },
  { "Pentax", "K-1", Daylight            , -5, { 2.296875, 1, 1.546875, 0 } },
  { "Pentax", "K-1", Daylight            , -4, { 2.26171875, 1, 1.57421875, 0 } },
  { "Pentax", "K-1", Daylight            , -3, { 2.2265625, 1, 1.59765625, 0 } },
  { "Pentax", "K-1", Daylight            , -2, { 2.1953125, 1, 1.625, 0 } },
  { "Pentax", "K-1", Daylight            , -1, { 2.1640625, 1, 1.65234375, 0 } },
  { "Pentax", "K-1", Daylight            , 0, { 2.1328125, 1, 1.68359375, 0 } },
  { "Pentax", "K-1", Daylight            , 1, { 2.1015625, 1, 1.7109375, 0 } },
  { "Pentax", "K-1", Daylight            , 2, { 2.0703125, 1, 1.73828125, 0 } },
  { "Pentax", "K-1", Daylight            , 3, { 2.0390625, 1, 1.76953125, 0 } },
  { "Pentax", "K-1", Daylight            , 4, { 2.0078125, 1, 1.80078125, 0 } },
  { "Pentax", "K-1", Daylight            , 5, { 1.98046875, 1, 1.83203125, 0 } },
  { "Pentax", "K-1", Daylight            , 6, { 1.953125, 1, 1.86328125, 0 } },
  { "Pentax", "K-1", Daylight            , 7, { 1.921875, 1, 1.89453125, 0 } },
  { "Pentax", "K-1", Shade               , -7, { 2.9375, 1, 1.125, 0 } },
  { "Pentax", "K-1", Shade               , -6, { 2.89453125, 1, 1.14453125, 0 } },
  { "Pentax", "K-1", Shade               , -5, { 2.8515625, 1, 1.16015625, 0 } },
  { "Pentax", "K-1", Shade               , -4, { 2.80859375, 1, 1.1796875, 0 } },
  { "Pentax", "K-1", Shade               , -3, { 2.76953125, 1, 1.203125, 0 } },
  { "Pentax", "K-1", Shade               , -2, { 2.7265625, 1, 1.22265625, 0 } },
  { "Pentax", "K-1", Shade               , -1, { 2.6875, 1, 1.2421875, 0 } },
  { "Pentax", "K-1", Shade               , 0, { 2.6484375, 1, 1.265625, 0 } },
  { "Pentax", "K-1", Shade               , 1, { 2.609375, 1, 1.28515625, 0 } },
  { "Pentax", "K-1", Shade               , 2, { 2.5703125, 1, 1.30859375, 0 } },
  { "Pentax", "K-1", Shade               , 3, { 2.53125, 1, 1.328125, 0 } },
  { "Pentax", "K-1", Shade               , 4, { 2.49609375, 1, 1.3515625, 0 } },
  { "Pentax", "K-1", Shade               , 5, { 2.4609375, 1, 1.375, 0 } },
  { "Pentax", "K-1", Shade               , 6, { 2.421875, 1, 1.3984375, 0 } },
  { "Pentax", "K-1", Shade               , 7, { 2.38671875, 1, 1.421875, 0 } },
  { "Pentax", "K-1", Cloudy              , -7, { 2.625, 1, 1.29296875, 0 } },
  { "Pentax", "K-1", Cloudy              , -6, { 2.5859375, 1, 1.31640625, 0 } },
  { "Pentax", "K-1", Cloudy              , -5, { 2.546875, 1, 1.33984375, 0 } },
  { "Pentax", "K-1", Cloudy              , -4, { 2.51171875, 1, 1.359375, 0 } },
  { "Pentax", "K-1", Cloudy              , -3, { 2.47265625, 1, 1.3828125, 0 } },
  { "Pentax", "K-1", Cloudy              , -2, { 2.4375, 1, 1.40625, 0 } },
  { "Pentax", "K-1", Cloudy              , -1, { 2.40234375, 1, 1.4296875, 0 } },
  { "Pentax", "K-1", Cloudy              , 0, { 2.3671875, 1, 1.45703125, 0 } },
  { "Pentax", "K-1", Cloudy              , 1, { 2.33203125, 1, 1.48046875, 0 } },
  { "Pentax", "K-1", Cloudy              , 2, { 2.296875, 1, 1.50390625, 0 } },
  { "Pentax", "K-1", Cloudy              , 3, { 2.26171875, 1, 1.53125, 0 } },
  { "Pentax", "K-1", Cloudy              , 4, { 2.23046875, 1, 1.55859375, 0 } },
  { "Pentax", "K-1", Cloudy              , 5, { 2.19921875, 1, 1.58203125, 0 } },
  { "Pentax", "K-1", Cloudy              , 6, { 2.16796875, 1, 1.609375, 0 } },
  { "Pentax", "K-1", Cloudy              , 7, { 2.1328125, 1, 1.63671875, 0 } },
  { "Pentax", "K-1", Tungsten            , -7, { 1.46875, 1, 2.63671875, 0 } },
  { "Pentax", "K-1", Tungsten            , -6, { 1.4453125, 1, 2.68359375, 0 } },
  { "Pentax", "K-1", Tungsten            , -5, { 1.42578125, 1, 2.7265625, 0 } },
  { "Pentax", "K-1", Tungsten            , -4, { 1.40234375, 1, 2.7734375, 0 } },
  { "Pentax", "K-1", Tungsten            , -3, { 1.3828125, 1, 2.8203125, 0 } },
  { "Pentax", "K-1", Tungsten            , -2, { 1.36328125, 1, 2.8671875, 0 } },
  { "Pentax", "K-1", Tungsten            , -1, { 1.34375, 1, 2.91796875, 0 } },
  { "Pentax", "K-1", Tungsten            , 0, { 1.32421875, 1, 2.96875, 0 } },
  { "Pentax", "K-1", Tungsten            , 1, { 1.3046875, 1, 3.015625, 0 } },
  { "Pentax", "K-1", Tungsten            , 2, { 1.28515625, 1, 3.0703125, 0 } },
  { "Pentax", "K-1", Tungsten            , 3, { 1.265625, 1, 3.12109375, 0 } },
  { "Pentax", "K-1", Tungsten            , 4, { 1.24609375, 1, 3.17578125, 0 } },
  { "Pentax", "K-1", Tungsten            , 5, { 1.23046875, 1, 3.23046875, 0 } },
  { "Pentax", "K-1", Tungsten            , 6, { 1.2109375, 1, 3.28515625, 0 } },
  { "Pentax", "K-1", Tungsten            , 7, { 1.19140625, 1, 3.33984375, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -7, { 1.8828125, 1, 2.50390625, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -6, { 1.85546875, 1, 2.546875, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -5, { 1.828125, 1, 2.58984375, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -4, { 1.80078125, 1, 2.6328125, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -3, { 1.7734375, 1, 2.67578125, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -2, { 1.75, 1, 2.72265625, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, -1, { 1.72265625, 1, 2.76953125, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 0, { 1.69921875, 1, 2.81640625, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 1, { 1.671875, 1, 2.86328125, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 2, { 1.6484375, 1, 2.91015625, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 3, { 1.625, 1, 2.9609375, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 4, { 1.6015625, 1, 3.01171875, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 5, { 1.578125, 1, 3.0625, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 6, { 1.5546875, 1, 3.1171875, 0 } },
  { "Pentax", "K-1", WarmWhiteFluorescent, 7, { 1.53125, 1, 3.16796875, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -7, { 2.46875, 1, 1.640625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -6, { 2.43359375, 1, 1.66796875, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -5, { 2.3984375, 1, 1.69921875, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -4, { 2.359375, 1, 1.7265625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -3, { 2.328125, 1, 1.75390625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -2, { 2.29296875, 1, 1.78515625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , -1, { 2.2578125, 1, 1.81640625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 0, { 2.2265625, 1, 1.84765625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 1, { 2.19140625, 1, 1.87890625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 2, { 2.16015625, 1, 1.91015625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 3, { 2.12890625, 1, 1.94140625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 4, { 2.09765625, 1, 1.9765625, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 5, { 2.06640625, 1, 2.0078125, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 6, { 2.0390625, 1, 2.04296875, 0 } },
  { "Pentax", "K-1", DayWhiteFluorescent , 7, { 2.0078125, 1, 2.078125, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -7, { 2.89453125, 1, 1.4453125, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -6, { 2.8515625, 1, 1.47265625, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -5, { 2.80859375, 1, 1.49609375, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -4, { 2.76953125, 1, 1.51953125, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -3, { 2.7265625, 1, 1.546875, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -2, { 2.6875, 1, 1.57421875, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , -1, { 2.6484375, 1, 1.6015625, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 0, { 2.609375, 1, 1.62890625, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 1, { 2.5703125, 1, 1.65625, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 2, { 2.53125, 1, 1.68359375, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 3, { 2.49609375, 1, 1.7109375, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 4, { 2.4609375, 1, 1.7421875, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 5, { 2.421875, 1, 1.76953125, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 6, { 2.38671875, 1, 1.80078125, 0 } },
  { "Pentax", "K-1", DaylightFluorescent , 7, { 2.3515625, 1, 1.83203125, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -7, { 2.23046875, 1, 1.9765625, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -6, { 2.19921875, 1, 2.01171875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -5, { 2.1640625, 1, 2.046875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -4, { 2.1328125, 1, 2.08203125, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -3, { 2.1015625, 1, 2.1171875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -2, { 2.0703125, 1, 2.15234375, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , -1, { 2.0390625, 1, 2.1875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 0, { 2.01171875, 1, 2.2265625, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 1, { 1.98046875, 1, 2.26171875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 2, { 1.953125, 1, 2.30078125, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 3, { 1.921875, 1, 2.33984375, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 4, { 1.89453125, 1, 2.3828125, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 5, { 1.8671875, 1, 2.421875, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 6, { 1.83984375, 1, 2.46484375, 0 } },
  { "Pentax", "K-1", WhiteFluorescent    , 7, { 1.8125, 1, 2.50390625, 0 } },

  /* Pentax K-3 Firmware version 1.00 */
  { "Pentax", "K-3", Daylight, 0,			{ 2.31250, 1, 1.48047, 0 } },
  { "Pentax", "K-3", Shade, 0,			{ 2.78516, 1, 1.16797, 0 } },
  { "Pentax", "K-3", Cloudy, 0,			{ 2.49219, 1, 1.35156, 0 } },
  { "Pentax", "K-3", Tungsten, 0,			{ 1.35938, 1, 2.74609, 0 } },
  { "Pentax", "K-3", DaylightFluorescent, 0,	{ 2.71094, 1, 1.51953, 0 } },
  { "Pentax", "K-3", DayWhiteFluorescent, 0,	{ 2.30469, 1, 1.70312, 0 } },
  { "Pentax", "K-3", WhiteFluorescent, 0,		{ 2.07031, 1, 2.03516, 0 } },
  { "Pentax", "K-3", WarmWhiteFluorescent, 0,	{ 1.74609, 1, 2.53125, 0 } },
  { "Pentax", "K-3", Flash, 0,			{ 2.56641, 1, 1.31641, 0 } },

  /* RICOH IMAGING COMPANY, LTD. PENTAX K-3 II Firmware Version 1.10 */
  { "Pentax", "K-3 II", Daylight, 0,             { 2.1875, 1, 1.578125, 0 } },
  { "Pentax", "K-3 II", Shade, 0,                { 2.703125, 1, 1.16015625, 0 } },
  { "Pentax", "K-3 II", Cloudy, 0,               { 2.41796875, 1, 1.33984375, 0 } },
  { "Pentax", "K-3 II", DaylightFluorescent, 0,  { 2.62890625, 1, 1.51171875, 0 } },
  { "Pentax", "K-3 II", DayWhiteFluorescent, 0,  { 2.23828125, 1, 1.6953125, 0 } },
  { "Pentax", "K-3 II", WhiteFluorescent, 0,     { 2.01171875, 1, 2.0234375, 0 } },
  { "Pentax", "K-3 II", WarmWhiteFluorescent, 0, { 1.6953125, 1, 2.515625, 0 } },
  { "Pentax", "K-3 II", Tungsten, 0,             { 1.3203125, 1, 2.7265625, 0 } },
  { "Pentax", "K-3 II", Flash, 0,                { 2.48828125, 1, 1.30859375, 0 } },

  { "Pentax", "K-5", Daylight, 0,		{ 2.1211, 1, 1.5781, 0 } },
  { "Pentax", "K-5", Shade, 0,			{ 2.5312, 1, 1.1758, 0 } },
  { "Pentax", "K-5", Cloudy, 0,			{ 2.2852, 1, 1.3477, 0 } },
  { "Pentax", "K-5", DaylightFluorescent, 0,	{ 2.6172, 1, 1.5195, 0 } },
  { "Pentax", "K-5", WhiteFluorescent, 0,	{ 2.2422, 1, 1.6914, 0 } },
  { "Pentax", "K-5", CoolWhiteFluorescent, 0,	{ 2.0391, 1, 2.0898, 0 } },
  { "Pentax", "K-5", WarmWhiteFluorescent, 0,	{ 1.7070, 1, 2.6172, 0 } },
  { "Pentax", "K-5", Tungsten, 0,		{ 1.2734, 1, 2.9258, 0 } },
  { "Pentax", "K-5", Flash, 0,			{ 2.4023, 1, 1.4492, 0 } },
  { "Pentax", "K-5", "3000K", 0,		{ 1.3867, 1, 2.2578, 0 } },
  { "Pentax", "K-5", "5000K", 0,		{ 2.0508, 1, 1.4883, 0 } },

  /* Pentax K-50 Firmware Version 1.00 */
  { "Pentax", "K-50", Daylight, 0,            { 2.244384765625, 1, 1.560302734375, 0 } },
  { "Pentax", "K-50", Shade, 0,               { 2.681640625, 1, 1.16455078125, 0 } },
  { "Pentax", "K-50", Cloudy, 0,              { 2.418212890625, 1, 1.334228515625, 0 } },
  { "Pentax", "K-50", Tungsten, 0,            { 1.3487548828125, 1, 2.8946533203125, 0 } },
  { "Pentax", "K-50", DaylightFluorescent, 0,        { 2.771240234375, 1, 1.5037841796875, 0 } },
  { "Pentax", "K-50", WhiteFluorescent, 0,        { 2.3760986328125, 1, 1.6734619140625, 0 } },
  { "Pentax", "K-50", CoolWhiteFluorescent, 0,        { 2.1600341796875, 1, 2.0692138671875, 0 } },
  { "Pentax", "K-50", Flash, 0,               { 2.544677734375, 1, 1.4359130859375, 0 } },
  { "Pentax", "K-50", WarmWhiteFluorescent, 0,        { 1.8070068359375, 1, 2.58935546875, 0 } },

  /* Pentax K-70 Firmware Version 1.10 */
  { "Pentax", "K-70", Daylight                     , 0, { 2.14453125, 1, 1.70703125, 0 } },
  { "Pentax", "K-70", Shade                        , 0, { 2.65625, 1, 1.25, 0 } },
  { "Pentax", "K-70", Cloudy                       , 0, { 2.37890625, 1, 1.46484375, 0 } },
  { "Pentax", "K-70", Tungsten                     , 0, { 1.3125, 1, 2.9765625, 0 } },
  { "Pentax", "K-70", WarmWhiteFluorescent         , 0, { 1.69921875, 1, 2.91015625, 0 } },
  { "Pentax", "K-70", DayWhiteFluorescent          , 0, { 2.23046875, 1, 1.9140625, 0 } },
  { "Pentax", "K-70", DaylightFluorescent          , 0, { 2.62890625, 1, 1.671875, 0 } },
  { "Pentax", "K-70", WhiteFluorescent             , 0, { 2.01953125, 1, 2.3046875, 0 } },

    /* Pentax K-500 Firmware Version 1.00 */
  { "Pentax", "K-500", Daylight, 0,           { 2.2076416015625, 1, 1.6060791015625, 0 } },
  { "Pentax", "K-500", Shade, 0,              { 2.6376953125, 1, 1.19873046875, 0 } },
  { "Pentax", "K-500", Cloudy, 0,             { 2.3785400390625, 1, 1.3734130859375, 0 } },
  { "Pentax", "K-500", Tungsten, 0,           { 1.32666015625, 1, 2.9796142578125, 0 } },
  { "Pentax", "K-500", DaylightFluorescent, 0,       { 2.725830078125, 1, 1.5479736328125, 0 } },
  { "Pentax", "K-500", WhiteFluorescent, 0,       { 2.337158203125, 1, 1.7225341796875, 0 } },
  { "Pentax", "K-500", CoolWhiteFluorescent, 0,       { 2.1246337890625, 1, 2.1298828125, 0 } },
  { "Pentax", "K-500", Flash, 0,              { 2.5029296875, 1, 1.4781494140625, 0 } },
  { "Pentax", "K-500", WarmWhiteFluorescent, 0,       { 1.7774658203125, 1, 2.665283203125, 0 } },

  // K-30 (FW v1.05)
  { "Pentax", "K-30", Daylight, 0,		{ 2.21301269531, 1, 1.62915039063, 0 } },
  { "Pentax", "K-30", Shade, 0,			{ 2.64428710938, 1, 1.21594238281, 0 } },
  { "Pentax", "K-30", Cloudy, 0,		{ 2.38452148438, 1, 1.39294433594, 0 } },
  { "Pentax", "K-30", DaylightFluorescent, 0,	{ 2.73254394531, 1, 1.57006835938, 0 } },
  { "Pentax", "K-30", WhiteFluorescent, 0,	{ 2.34289550781, 1, 1.74719238281, 0 } },
  { "Pentax", "K-30", CoolWhiteFluorescent, 0,	{ 2.13000488281, 1, 2.16040039063, 0 } },
  { "Pentax", "K-30", WarmWhiteFluorescent, 0,	{ 1.78186035156, 1, 2.70336914063, 0 } },
  { "Pentax", "K-30", Tungsten, 0,		{ 1.32995605469, 1, 3.02221679688, 0 } },
  { "Pentax", "K-30", Flash, 0,			{ 2.50915527344, 1, 1.49926757813, 0 } },

  // copied from K-5 II s
  { "Pentax", "K-5 II", Daylight, 0,		{ 2.136719, 1, 1.628906, 0 } },
  { "Pentax", "K-5 II", Shade, 0,		{ 2.550781, 1, 1.214844, 0 } },
  { "Pentax", "K-5 II", Cloudy, 0,		{ 2.300781, 1, 1.390625, 0 } },
  { "Pentax", "K-5 II", DaylightFluorescent, 0,	{ 2.636719, 1, 1.566406, 0 } },
  { "Pentax", "K-5 II", WhiteFluorescent, 0,	{ 2.261719, 1, 1.746094, 0 } },
  { "Pentax", "K-5 II", CoolWhiteFluorescent, 0, { 2.054688, 1, 2.156250, 0 } },
  { "Pentax", "K-5 II", WarmWhiteFluorescent, 0, { 1.718750, 1, 2.699219, 0 } },
  { "Pentax", "K-5 II", Tungsten, 0,		{ 1.265625, 1, 2.816406, 0 } },
  { "Pentax", "K-5 II", Flash, 0,		{ 1.851562, 1, 1.792969, 0 } },

  // firmware K-5 II s Ver 1.02
  { "Pentax", "K-5 II s", Daylight, 0,		{ 2.136719, 1, 1.628906, 0 } },
  { "Pentax", "K-5 II s", Shade, 0,		{ 2.550781, 1, 1.214844, 0 } },
  { "Pentax", "K-5 II s", Cloudy, 0,		{ 2.300781, 1, 1.390625, 0 } },
  { "Pentax", "K-5 II s", DaylightFluorescent, 0, { 2.636719, 1, 1.566406, 0 } },
  { "Pentax", "K-5 II s", WhiteFluorescent, 0,	{ 2.261719, 1, 1.746094, 0 } },
  { "Pentax", "K-5 II s", CoolWhiteFluorescent, 0, { 2.054688, 1, 2.156250, 0 } },
  { "Pentax", "K-5 II s", WarmWhiteFluorescent, 0, { 1.718750, 1, 2.699219, 0 } },
  { "Pentax", "K-5 II s", Tungsten, 0,		{ 1.265625, 1, 2.816406, 0 } },
  { "Pentax", "K-5 II s", Flash, 0,		{ 1.851562, 1, 1.792969, 0 } },

  { "Pentax", "K-7", Daylight, 0,		{ 1.808594, 1, 1.285156, 0 } },
  { "Pentax", "K-7", Shade, 0,			{ 2.207171, 1.019920, 1, 0 } },
  { "Pentax", "K-7", Cloudy, 0,			{ 1.960937, 1, 1.136719, 0 } },
  { "Pentax", "K-7", DaylightFluorescent, 0,	{ 2.281250, 1, 1.191406, 0 } },
  { "Pentax", "K-7", NeutralFluorescent, 0,	{ 1.937500, 1, 1.355469, 0 } },
  { "Pentax", "K-7", CoolWhiteFluorescent, 0,	{ 1.808594, 1, 1.687500, 0 } },
  { "Pentax", "K-7", WarmWhiteFluorescent, 0,	{ 1.589844, 1, 2.164063, 0 } },
  { "Pentax", "K-7", Tungsten, 0,		{ 1.105469, 1, 2.347656, 0 } },
  { "Pentax", "K-7", Flash, 0,			{ 2.093750, 1, 1.082031, 0 } },

  { "Pentax", "K-m", Daylight, 0,		{ 1.738281, 1, 1.363281, 0 } },
  { "Pentax", "K-m", Shade, 0,			{ 2.027344, 1, 1.027344, 0 } },
  { "Pentax", "K-m", Cloudy, 0,			{ 1.832031, 1, 1.183594, 0 } },
  { "Pentax", "K-m", DaylightFluorescent, 0,	{ 2.183594, 1, 1.250000, 0 } },
  { "Pentax", "K-m", NeutralFluorescent, 0,	{ 1.824219, 1, 1.417969, 0 } },
  { "Pentax", "K-m", WhiteFluorescent, 0,	{ 1.644531, 1, 1.714844, 0 } },
  { "Pentax", "K-m", Tungsten, 0,		{ 1.429687, 1, 1.980469, 0 } },
  { "Pentax", "K-m", Flash, 0,			{ 1.738281, 1, 1.363281, 0 } },

  /* Firmware version 1.11 */
  { "Pentax", "K-r", Daylight, 0,		{ 1.8477, 1, 1.3906, 0 } },
  { "Pentax", "K-r", Shade, 0,			{ 2.1133, 1, 1.0586, 0 } },
  { "Pentax", "K-r", Cloudy, 0,			{ 1.9766, 1, 1.1445, 0 } },
  { "Pentax", "K-r", DaylightFluorescent, 0,	{ 2.2617, 1, 1.3203, 0 } },
  { "Pentax", "K-r", WhiteFluorescent, 0,	{ 1.9414, 1, 1.4688, 0 } },
  { "Pentax", "K-r", CoolWhiteFluorescent, 0,	{ 1.7656, 1, 1.8164, 0 } },
  { "Pentax", "K-r", WarmWhiteFluorescent, 0,	{ 1.4766, 1, 2.2734, 0 } },
  { "Pentax", "K-r", Tungsten, 0,		{ 1.1016, 1, 2.5391, 0 } },
  { "Pentax", "K-r", Flash, 0,			{ 2.0117, 1, 1.1172, 0 } },

  { "Pentax", "K-x", Daylight, 0,		{ 1.8803, 1, 1.4054, 0 } },
  { "Pentax", "K-x", Shade, 0,			{ 2.2278, 1, 1.0309, 0 } },
  { "Pentax", "K-x", Cloudy, 0,			{ 2.0077, 1, 1.1853, 0 } },
  { "Pentax", "K-x", DaylightFluorescent, 0,	{ 2.3012, 1, 1.3359, 0 } },
  { "Pentax", "K-x", WhiteFluorescent, 0,	{ 1.9730, 1, 1.4826, 0 } },
  { "Pentax", "K-x", CoolWhiteFluorescent, 0,	{ 1.7915, 1, 1.8378, 0 } },
  { "Pentax", "K-x", WarmWhiteFluorescent, 0,	{ 1.5019, 1, 2.2973, 0 } },
  { "Pentax", "K-x", Tungsten, 0,		{ 1.0463, 1, 3.4015, 0 } },
  { "Pentax", "K-x", Flash, 0,			{ 2.1120, 1, 1.2741, 0 } },

  { "Pentax", "K-S1", Daylight                     , 0, { 2.277344, 1, 1.656250, 0 } },
  { "Pentax", "K-S1", Shade                        , 0, { 2.847656, 1, 1.230469, 0 } },
  { "Pentax", "K-S1", Cloudy                       , 0, { 2.546875, 1, 1.429688, 0 } },
  { "Pentax", "K-S1", Tungsten                     , 0, { 1.343750, 1, 3.089844, 0 } },
  { "Pentax", "K-S1", WarmWhiteFluorescent         , 0, { 1.777344, 1, 2.781250, 0 } },
  { "Pentax", "K-S1", DayWhiteFluorescent          , 0, { 2.367188, 1, 1.824219, 0 } },
  { "Pentax", "K-S1", DaylightFluorescent          , 0, { 2.761719, 1, 1.617188, 0 } },
  { "Pentax", "K-S1", WhiteFluorescent             , 0, { 2.125000, 1, 2.210938, 0 } },
  { "Pentax", "K-S1", Flash                        , 0, { 2.613281, 1, 1.351562, 0 } },

  { "Pentax", "K-S2", Daylight            , -7, { 2.51171875, 1, 1.5078125, 0 } },
  { "Pentax", "K-S2", Daylight            , -6, { 2.4765625, 1, 1.53125, 0 } },
  { "Pentax", "K-S2", Daylight            , -5, { 2.4375, 1, 1.55859375, 0 } },
  { "Pentax", "K-S2", Daylight            , -4, { 2.40234375, 1, 1.58203125, 0 } },
  { "Pentax", "K-S2", Daylight            , -3, { 2.3671875, 1, 1.609375, 0 } },
  { "Pentax", "K-S2", Daylight            , -2, { 2.33203125, 1, 1.63671875, 0 } },
  { "Pentax", "K-S2", Daylight            , -1, { 2.296875, 1, 1.6640625, 0 } },
  { "Pentax", "K-S2", Daylight            , 0, { 2.265625, 1, 1.6953125, 0 } },
  { "Pentax", "K-S2", Daylight            , 1, { 2.23046875, 1, 1.72265625, 0 } },
  { "Pentax", "K-S2", Daylight            , 2, { 2.19921875, 1, 1.75, 0 } },
  { "Pentax", "K-S2", Daylight            , 3, { 2.16796875, 1, 1.78125, 0 } },
  { "Pentax", "K-S2", Daylight            , 4, { 2.13671875, 1, 1.8125, 0 } },
  { "Pentax", "K-S2", Daylight            , 5, { 2.10546875, 1, 1.84375, 0 } },
  { "Pentax", "K-S2", Daylight            , 6, { 2.07421875, 1, 1.875, 0 } },
  { "Pentax", "K-S2", Daylight            , 7, { 2.04296875, 1, 1.90625, 0 } },
  { "Pentax", "K-S2", Shade               , -7, { 3.203125, 1, 1.08984375, 0 } },
  { "Pentax", "K-S2", Shade               , -6, { 3.15625, 1, 1.10546875, 0 } },
  { "Pentax", "K-S2", Shade               , -5, { 3.109375, 1, 1.125, 0 } },
  { "Pentax", "K-S2", Shade               , -4, { 3.0625, 1, 1.14453125, 0 } },
  { "Pentax", "K-S2", Shade               , -3, { 3.015625, 1, 1.1640625, 0 } },
  { "Pentax", "K-S2", Shade               , -2, { 2.97265625, 1, 1.18359375, 0 } },
  { "Pentax", "K-S2", Shade               , -1, { 2.9296875, 1, 1.203125, 0 } },
  { "Pentax", "K-S2", Shade               , 0, { 2.88671875, 1, 1.2265625, 0 } },
  { "Pentax", "K-S2", Shade               , 1, { 2.84375, 1, 1.24609375, 0 } },
  { "Pentax", "K-S2", Shade               , 2, { 2.80078125, 1, 1.265625, 0 } },
  { "Pentax", "K-S2", Shade               , 3, { 2.76171875, 1, 1.2890625, 0 } },
  { "Pentax", "K-S2", Shade               , 4, { 2.71875, 1, 1.3125, 0 } },
  { "Pentax", "K-S2", Shade               , 5, { 2.6796875, 1, 1.33203125, 0 } },
  { "Pentax", "K-S2", Shade               , 6, { 2.640625, 1, 1.35546875, 0 } },
  { "Pentax", "K-S2", Shade               , 7, { 2.60546875, 1, 1.37890625, 0 } },
  { "Pentax", "K-S2", Cloudy              , -7, { 2.859375, 1, 1.26171875, 0 } },
  { "Pentax", "K-S2", Cloudy              , -6, { 2.81640625, 1, 1.28515625, 0 } },
  { "Pentax", "K-S2", Cloudy              , -5, { 2.77734375, 1, 1.3046875, 0 } },
  { "Pentax", "K-S2", Cloudy              , -4, { 2.734375, 1, 1.328125, 0 } },
  { "Pentax", "K-S2", Cloudy              , -3, { 2.6953125, 1, 1.3515625, 0 } },
  { "Pentax", "K-S2", Cloudy              , -2, { 2.65234375, 1, 1.375, 0 } },
  { "Pentax", "K-S2", Cloudy              , -1, { 2.61328125, 1, 1.39453125, 0 } },
  { "Pentax", "K-S2", Cloudy              , 0, { 2.578125, 1, 1.421875, 0 } },
  { "Pentax", "K-S2", Cloudy              , 1, { 2.5390625, 1, 1.4453125, 0 } },
  { "Pentax", "K-S2", Cloudy              , 2, { 2.5, 1, 1.46875, 0 } },
  { "Pentax", "K-S2", Cloudy              , 3, { 2.46484375, 1, 1.4921875, 0 } },
  { "Pentax", "K-S2", Cloudy              , 4, { 2.4296875, 1, 1.51953125, 0 } },
  { "Pentax", "K-S2", Cloudy              , 5, { 2.39453125, 1, 1.546875, 0 } },
  { "Pentax", "K-S2", Cloudy              , 6, { 2.359375, 1, 1.5703125, 0 } },
  { "Pentax", "K-S2", Cloudy              , 7, { 2.32421875, 1, 1.59765625, 0 } },
  { "Pentax", "K-S2", Tungsten            , -7, { 1.4765625, 1, 2.7578125, 0 } },
  { "Pentax", "K-S2", Tungsten            , -6, { 1.453125, 1, 2.8046875, 0 } },
  { "Pentax", "K-S2", Tungsten            , -5, { 1.43359375, 1, 2.8515625, 0 } },
  { "Pentax", "K-S2", Tungsten            , -4, { 1.41015625, 1, 2.8984375, 0 } },
  { "Pentax", "K-S2", Tungsten            , -3, { 1.390625, 1, 2.94921875, 0 } },
  { "Pentax", "K-S2", Tungsten            , -2, { 1.37109375, 1, 2.99609375, 0 } },
  { "Pentax", "K-S2", Tungsten            , -1, { 1.3515625, 1, 3.046875, 0 } },
  { "Pentax", "K-S2", Tungsten            , 0, { 1.33203125, 1, 3.1015625, 0 } },
  { "Pentax", "K-S2", Tungsten            , 1, { 1.3125, 1, 3.15234375, 0 } },
  { "Pentax", "K-S2", Tungsten            , 2, { 1.29296875, 1, 3.20703125, 0 } },
  { "Pentax", "K-S2", Tungsten            , 3, { 1.2734375, 1, 3.26171875, 0 } },
  { "Pentax", "K-S2", Tungsten            , 4, { 1.25390625, 1, 3.31640625, 0 } },
  { "Pentax", "K-S2", Tungsten            , 5, { 1.234375, 1, 3.375, 0 } },
  { "Pentax", "K-S2", Tungsten            , 6, { 1.21875, 1, 3.43359375, 0 } },
  { "Pentax", "K-S2", Tungsten            , 7, { 1.19921875, 1, 3.4921875, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -7, { 1.984375, 1, 2.4765625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -6, { 1.953125, 1, 2.515625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -5, { 1.92578125, 1, 2.55859375, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -4, { 1.8984375, 1, 2.6015625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -3, { 1.8671875, 1, 2.6484375, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -2, { 1.83984375, 1, 2.69140625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, -1, { 1.8125, 1, 2.73828125, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 0, { 1.7890625, 1, 2.78515625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 1, { 1.76171875, 1, 2.83203125, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 2, { 1.734375, 1, 2.87890625, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 3, { 1.7109375, 1, 2.9296875, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 4, { 1.68359375, 1, 2.98046875, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 5, { 1.66015625, 1, 3.03125, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 6, { 1.63671875, 1, 3.08203125, 0 } },
  { "Pentax", "K-S2", WarmWhiteFluorescent, 7, { 1.61328125, 1, 3.1328125, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -7, { 2.62890625, 1, 1.63671875, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -6, { 2.58984375, 1, 1.6640625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -5, { 2.55078125, 1, 1.6953125, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -4, { 2.515625, 1, 1.72265625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -3, { 2.4765625, 1, 1.75, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -2, { 2.44140625, 1, 1.78125, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , -1, { 2.40625, 1, 1.8125, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 0, { 2.37109375, 1, 1.84375, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 1, { 2.3359375, 1, 1.875, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 2, { 2.30078125, 1, 1.90625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 3, { 2.265625, 1, 1.9375, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 4, { 2.234375, 1, 1.97265625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 5, { 2.203125, 1, 2.00390625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 6, { 2.16796875, 1, 2.0390625, 0 } },
  { "Pentax", "K-S2", DayWhiteFluorescent , 7, { 2.13671875, 1, 2.07421875, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -7, { 3.1171875, 1, 1.44921875, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -6, { 3.0703125, 1, 1.4765625, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -5, { 3.0234375, 1, 1.5, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -4, { 2.98046875, 1, 1.5234375, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -3, { 2.93359375, 1, 1.55078125, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -2, { 2.890625, 1, 1.578125, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , -1, { 2.84765625, 1, 1.60546875, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 0, { 2.80859375, 1, 1.6328125, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 1, { 2.765625, 1, 1.66015625, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 2, { 2.7265625, 1, 1.6875, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 3, { 2.6875, 1, 1.71484375, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 4, { 2.6484375, 1, 1.74609375, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 5, { 2.609375, 1, 1.7734375, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 6, { 2.5703125, 1, 1.8046875, 0 } },
  { "Pentax", "K-S2", DaylightFluorescent , 7, { 2.53125, 1, 1.8359375, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -7, { 2.359375, 1, 1.96484375, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -6, { 2.32421875, 1, 1.99609375, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -5, { 2.29296875, 1, 2.03125, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -4, { 2.2578125, 1, 2.06640625, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -3, { 2.22265625, 1, 2.1015625, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -2, { 2.19140625, 1, 2.13671875, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , -1, { 2.16015625, 1, 2.171875, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 0, { 2.12890625, 1, 2.2109375, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 1, { 2.09765625, 1, 2.24609375, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 2, { 2.06640625, 1, 2.28515625, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 3, { 2.03515625, 1, 2.32421875, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 4, { 2.00390625, 1, 2.36328125, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 5, { 1.9765625, 1, 2.40625, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 6, { 1.94921875, 1, 2.4453125, 0 } },
  { "Pentax", "K-S2", WhiteFluorescent    , 7, { 1.91796875, 1, 2.48828125, 0 } },
  { "Pentax", "K-S2", Flash               , -7, { 3.05078125, 1, 1.21484375, 0 } },
  { "Pentax", "K-S2", Flash               , -6, { 3.00390625, 1, 1.234375, 0 } },
  { "Pentax", "K-S2", Flash               , -5, { 2.9609375, 1, 1.25390625, 0 } },
  { "Pentax", "K-S2", Flash               , -4, { 2.91796875, 1, 1.27734375, 0 } },
  { "Pentax", "K-S2", Flash               , -3, { 2.875, 1, 1.296875, 0 } },
  { "Pentax", "K-S2", Flash               , -2, { 2.83203125, 1, 1.3203125, 0 } },
  { "Pentax", "K-S2", Flash               , -1, { 2.7890625, 1, 1.34375, 0 } },
  { "Pentax", "K-S2", Flash               , 0, { 2.75, 1, 1.3671875, 0 } },
  { "Pentax", "K-S2", Flash               , 1, { 2.70703125, 1, 1.38671875, 0 } },
  { "Pentax", "K-S2", Flash               , 2, { 2.66796875, 1, 1.4140625, 0 } },
  { "Pentax", "K-S2", Flash               , 3, { 2.62890625, 1, 1.4375, 0 } },
  { "Pentax", "K-S2", Flash               , 4, { 2.58984375, 1, 1.4609375, 0 } },
  { "Pentax", "K-S2", Flash               , 5, { 2.5546875, 1, 1.484375, 0 } },
  { "Pentax", "K-S2", Flash               , 6, { 2.515625, 1, 1.51171875, 0 } },
  { "Pentax", "K-S2", Flash               , 7, { 2.48046875, 1, 1.5390625, 0 } },


  /* Pentax Q Firmware Version Pentax Q Ver 1.00 */
  { "Pentax", "Q", Daylight, 0,               { 1.51513671875, 1, 1.85888970408413, 0 } },
  { "Pentax", "Q", Shade, 0,                  { 1.8182373046875, 1, 1.4179505991685, 0 } },
  { "Pentax", "Q", Cloudy, 0,                 { 1.70458984375, 1, 1.60858400586941, 0 } },
  { "Pentax", "Q", Tungsten, 0,               { 0.9697265625, 1, 3.05050134507214, 0 } },
  { "Pentax", "Q", DaylightFluorescent, 0,           { 1.9697265625, 1, 1.78735632183908, 0 } },
  { "Pentax", "Q", WhiteFluorescent, 0,           { 1.742431640625, 1, 2.02567865003668, 0 } },
  { "Pentax", "Q", CoolWhiteFluorescent, 0,           { 1.57958984375, 1, 2.50232330643189, 0 } },
  { "Pentax", "Q", Flash, 0,                  { 1.82958984375, 1, 1.5133284421619, 0 } },
  { "Pentax", "Q", WarmWhiteFluorescent, 0,           { 1.32568359375, 1, 2.90743458058205, 0 } },

  /* Pentax Q7 Firmware Version Pentax Q7     Ver 1.00 */
  { "Pentax", "Q7", Daylight, 0,              { 1.6688232421875, 1, 1.79434353285383, 0 } },
  { "Pentax", "Q7", Shade, 0,                 { 2.05029296875, 1, 1.40107277825186, 0 } },
  { "Pentax", "Q7", Cloudy, 0,                { 1.923095703125, 1, 1.59770815555285, 0 } },
  { "Pentax", "Q7", Tungsten, 0,              { 1.0172119140625, 1, 3.14640985005486, 0 } },
  { "Pentax", "Q7", DaylightFluorescent, 0,          { 2.2210693359375, 1, 1.76984030232842, 0 } },
  { "Pentax", "Q7", WhiteFluorescent, 0,          { 1.9072265625, 1, 2.0032914787273, 0 } },
  { "Pentax", "Q7", CoolWhiteFluorescent, 0,          { 1.780029296875, 1, 2.48262830671705, 0 } },
  { "Pentax", "Q7", Flash, 0,                 { 1.9866943359375, 1, 1.49945142021212, 0 } },
  { "Pentax", "Q7", WarmWhiteFluorescent, 0,          { 1.4940185546875, 1, 2.8882116298915, 0 } },

  /* Pentax Q10 Firmware Version 1.00 */
  { "Pentax", "Q10", Daylight, 0,             { 1.5614013671875, 1, 1.65324357405141, 0 } },
  { "Pentax", "Q10", Shade, 0,                { 1.965087890625, 1, 1.32717258261934, 0 } },
  { "Pentax", "Q10", Cloudy, 0,               { 1.84326171875, 1, 1.51346389228886, 0 } },
  { "Pentax", "Q10", Tungsten, 0,             { 0.9749755859375, 1, 2.98053855569155, 0 } },
  { "Pentax", "Q10", DaylightFluorescent, 0,         { 2.12890625, 1, 1.6764993880049, 0 } },
  { "Pentax", "Q10", WhiteFluorescent, 0,         { 1.8280029296875, 1, 1.89767441860465, 0 } },
  { "Pentax", "Q10", CoolWhiteFluorescent, 0,         { 1.7061767578125, 1, 2.35177478580171, 0 } },
  { "Pentax", "Q10", Flash, 0,                { 1.9041748046875, 1, 1.4203182374541, 0 } },
  { "Pentax", "Q10", WarmWhiteFluorescent, 0,         { 1.431884765625, 1, 2.7359853121175, 0 } },

  { "Pentax", "Caplio GX100", Daylight, 0,	{ 1.910001, 1, 1.820002, 0 } },
  { "Pentax", "Caplio GX100", Cloudy, 0,		{ 2.240003, 1, 1.530002, 0 } },
  { "Pentax", "Caplio GX100", Incandescent, 0,	{ 1.520002, 1, 2.520003, 0 } },
  { "Pentax", "Caplio GX100", Fluorescent, 0,	{ 1.840001, 1, 1.970001, 0 } },

  /* Ricoh GR Firmware Version 4.00 */
  { "Ricoh", "GR", Daylight, 0,			{ 2.980000, 1, 1.580000, 0 } },
  { "Ricoh", "GR", Shade, 0,			{ 3.450000, 1, 1.360000, 0 } },
  { "Ricoh", "GR", Cloudy, 0,			{ 3.170000, 1, 1.500000, 0 } },
  { "Ricoh", "GR", Incandescent, 0,		{ 1.920000, 1, 2.530000, 0 } },
  { "Ricoh", "GR", IncandescentWarm, 0,		{ 2.090000, 1, 2.030000, 0 } },
  { "Ricoh", "GR", DaylightFluorescent, 0,	{ 3.220000, 1, 1.510000, 0 } },
  { "Ricoh", "GR", WhiteFluorescent, 0,		{ 2.860000, 1, 1.700000, 0 } },
  { "Ricoh", "GR", CoolWhiteFluorescent, 0,	{ 2.610000, 1, 2.080000, 0 } },
  { "Ricoh", "GR", WarmWhiteFluorescent, 0,	{ 2.240000, 1, 2.160000, 0 } },
  { "Ricoh", "GR", "5000K", 0,			{ 2.890000, 1, 1.520000, 0 } },

  { "Ricoh", "GR II", Daylight, 0,		{ 3.062000, 1, 1.545000, 0 } },
  { "Ricoh", "GR II", Shade, 0,			{ 3.319000, 1, 1.385000, 0 } },
  { "Ricoh", "GR II", Cloudy, 0,			{ 3.058000, 1, 1.538000, 0 } },
  { "Ricoh", "GR II", Incandescent, 0,		{ 1.944000, 1, 2.384000, 0 } },
  { "Ricoh", "GR II", IncandescentWarm, 0,	{ 2.276000, 1, 2.052000, 0 } },
  { "Ricoh", "GR II", DaylightFluorescent, 0,	{ 3.062000, 1, 1.545000, 0 } },
  { "Ricoh", "GR II", WhiteFluorescent, 0,	{ 2.837000, 1, 1.698000, 0 } },
  { "Ricoh", "GR II", CoolWhiteFluorescent, 0,	{ 2.642000, 1, 2.125000, 0 } },
  { "Ricoh", "GR II", WarmWhiteFluorescent, 0,	{ 2.354000, 1, 2.027000, 0 } },
  { "Ricoh", "GR II", "5000K", 0,			{ 2.867000, 1, 1.662000, 0 } },

  { "Ricoh", "GR III", Daylight            , -14, { 2.9315068493150687, 1, 1.5420743639921721, 0 } },
  { "Ricoh", "GR III", Daylight            , 0, { 2.6418786692759295, 1, 1.7377690802348336, 0 } },
  { "Ricoh", "GR III", Daylight            , 14, { 2.3835616438356166, 1, 1.9530332681017613, 0 } },
  { "Ricoh", "GR III", Shade               , -14, { 3.5185909980430528, 1, 1.1585127201565557, 0 } },
  { "Ricoh", "GR III", Shade               , 0, { 3.1702544031311155, 1, 1.3033268101761253, 0 } },
  { "Ricoh", "GR III", Shade               , 14, { 2.8610567514677103, 1, 1.4637964774951076, 0 } },
  { "Ricoh", "GR III", Cloudy              , -14, { 3.2054794520547945, 1, 1.3385518590998042, 0 } },
  { "Ricoh", "GR III", Cloudy              , 0, { 2.888454011741683, 1, 1.5068493150684932, 0 } },
  { "Ricoh", "GR III", Cloudy              , 14, { 2.6066536203522506, 1, 1.6947162426614482, 0 } },
  { "Ricoh", "GR III", Tungsten            , -14, { 1.8395303326810175, 1, 2.684931506849315, 0 } },
  { "Ricoh", "GR III", Tungsten            , 0, { 1.659491193737769, 1, 3.0215264187866926, 0 } },
  { "Ricoh", "GR III", Tungsten            , 14, { 1.4951076320939334, 1, 3.401174168297456, 0 } },
  { "Ricoh", "GR III", WarmWhiteFluorescent, -14, { 2.148727984344423, 1, 2.6418786692759295, 0 } },
  { "Ricoh", "GR III", WarmWhiteFluorescent, 0, { 1.9373776908023483, 1, 2.974559686888454, 0 } },
  { "Ricoh", "GR III", WarmWhiteFluorescent, 14, { 1.74559686888454, 1, 3.3463796477495107, 0 } },
  { "Ricoh", "GR III", DayWhiteFluorescent , -14, { 2.8258317025440314, 1, 1.741682974559687, 0 } },
  { "Ricoh", "GR III", DayWhiteFluorescent , 0, { 2.547945205479452, 1, 1.9608610567514677, 0 } },
  { "Ricoh", "GR III", DayWhiteFluorescent , 14, { 2.2974559686888454, 1, 2.207436399217221, 0 } },
  { "Ricoh", "GR III", DaylightFluorescent , -14, { 3.3228962818003915, 1, 1.538160469667319, 0 } },
  { "Ricoh", "GR III", DaylightFluorescent , 0, { 2.9941291585127203, 1, 1.729941291585127, 0 } },
  { "Ricoh", "GR III", DaylightFluorescent , 14, { 2.700587084148728, 1, 1.9452054794520548, 0 } },
  { "Ricoh", "GR III", WhiteFluorescent    , -14, { 2.540117416829746, 1, 2.1291585127201564, 0 } },
  { "Ricoh", "GR III", WhiteFluorescent    , 0, { 2.289628180039139, 1, 2.395303326810176, 0 } },
  { "Ricoh", "GR III", WhiteFluorescent    , 14, { 2.0665362035225048, 1, 2.696673189823875, 0 } },

  { "Samsung", "EX1", Daylight, 0,		{ 1.8711, 1, 2.0039, 0 } },
  { "Samsung", "EX1", Cloudy, 0,		{ 2.3672, 1, 1.6797, 0 } },
  { "Samsung", "EX1", DaylightFluorescent, 0,	{ 1.9492, 1, 2.0586, 0 } },
  { "Samsung", "EX1", WhiteFluorescent, 0,	{ 1.4844, 1, 2.7852, 0 } },
  { "Samsung", "EX1", Tungsten, 0,		{ 1.2500, 1, 3.6834, 0 } },

  { "Samsung", "GX-1S", Daylight, 0,		{ 1.574219, 1, 1.109375, 0 } },
  { "Samsung", "GX-1S", Shade, 0,		{ 1.855469, 1, 1.000000, 0 } },
  { "Samsung", "GX-1S", Cloudy, 0,		{ 1.664062, 1, 1.000000, 0 } },
  { "Samsung", "GX-1S", DaylightFluorescent, 0,	{ 1.854251, 1.036437, 1, 0 } },
  { "Samsung", "GX-1S", NeutralFluorescent, 0,	{ 1.574219, 1, 1.171875, 0 } },
  { "Samsung", "GX-1S", WhiteFluorescent, 0,	{ 1.363281, 1, 1.335938, 0 } },
  { "Samsung", "GX-1S", Tungsten, 0,		{ 1.000000, 1, 2.226562, 0 } },
  { "Samsung", "GX-1S", Flash, 0,		{ 1.609375, 1, 1.031250, 0 } },

  { "Samsung", "GX10", Daylight, 0,		{ 1.660156, 1, 1.066406, 0 } },
  { "Samsung", "GX10", Shade, 0,		{ 2.434783, 1.236715, 1, 0 } },
  { "Samsung", "GX10", Cloudy, 0,		{ 1.872428, 1.053498, 1, 0 } },
  { "Samsung", "GX10", DaylightFluorescent, 0,	{ 2.121094, 1, 1.078125, 0 } },
  { "Samsung", "GX10", NeutralFluorescent, 0,	{ 1.773438, 1, 1.226562, 0 } },
  { "Samsung", "GX10", WhiteFluorescent, 0,	{ 1.597656, 1, 1.488281, 0 } },
  { "Samsung", "GX10", Tungsten, 0,		{ 1.000000, 1, 2.558594, 0 } },
  { "Samsung", "GX10", Flash, 0,		{ 1.664062, 1, 1.046875, 0 } },

  /* Copied from Pentax K20D */
  { "Samsung", "GX20", Daylight, 0,		{ 1.5787353515625, 1.0, 1.263427734375, 0.0 } },
  { "Samsung", "GX20", Shade, 0,		{ 1.80029296875, 1.0, 0.9613037109375, 0.0 } },
  { "Samsung", "GX20", Cloudy, 0,		{ 1.6737060546875, 1.0, 1.116943359375, 0.0 } },
  { "Samsung", "GX20", DaylightFluorescent, 0,	{ 2.089111328125, 1.0, 1.190185546875, 0.0 } },
  { "Samsung", "GX20", NeutralFluorescent, 0,	{ 1.7725830078125, 1.0, 1.35498046875, 0.0 } },
  { "Samsung", "GX20", WhiteFluorescent, 0,	{ 1.6539306640625, 1.0, 1.6845703125, 0.0 } },
  { "Samsung", "GX20", Tungsten, 0,		{ 1.012939453125, 1.0, 2.34375, 0.0 } },
  { "Samsung", "GX20", Flash, 0,		{ 1.6737060546875, 1.0, 1.190185546875, 0.0 } },

  // Samsung NX1 Firmware 1.41
  { "Samsung", "NX1", Daylight           , -7, { 1.6129032258064515, 1, 2.181451612903226, 0 } },
  { "Samsung", "NX1", Daylight           , -6, { 1.6411290322580645, 1, 2.1048387096774195, 0 } },
  { "Samsung", "NX1", Daylight           , -5, { 1.6693548387096775, 1, 2.032258064516129, 0 } },
  { "Samsung", "NX1", Daylight           , -4, { 1.7016129032258065, 1, 1.9556451612903225, 0 } },
  { "Samsung", "NX1", Daylight           , -3, { 1.7298387096774193, 1, 1.8790322580645162, 0 } },
  { "Samsung", "NX1", Daylight           , -2, { 1.7580645161290323, 1, 1.8064516129032258, 0 } },
  { "Samsung", "NX1", Daylight           , -1, { 1.7862903225806452, 1, 1.7298387096774193, 0 } },
  { "Samsung", "NX1", Daylight           ,  0, { 1.814516129032258, 1, 1.657258064516129, 0 } },
  { "Samsung", "NX1", Daylight           ,  1, { 1.846774193548387, 1, 1.6169354838709677, 0 } },
  { "Samsung", "NX1", Daylight           ,  2, { 1.8790322580645162, 1, 1.5725806451612903, 0 } },
  { "Samsung", "NX1", Daylight           ,  3, { 1.9112903225806452, 1, 1.532258064516129, 0 } },
  { "Samsung", "NX1", Daylight           ,  4, { 1.9435483870967742, 1, 1.4919354838709677, 0 } },
  { "Samsung", "NX1", Daylight           ,  5, { 1.9758064516129032, 1, 1.4516129032258065, 0 } },
  { "Samsung", "NX1", Daylight           ,  6, { 2.0080645161290325, 1, 1.4112903225806452, 0 } },
  { "Samsung", "NX1", Daylight           ,  7, { 2.036290322580645, 1, 1.3709677419354838, 0 } },
  { "Samsung", "NX1", Cloudy             , -7, { 1.685483870967742, 1, 1.9314516129032258, 0 } },
  { "Samsung", "NX1", Cloudy             , -6, { 1.717741935483871, 1, 1.8629032258064515, 0 } },
  { "Samsung", "NX1", Cloudy             , -5, { 1.7459677419354838, 1, 1.7983870967741935, 0 } },
  { "Samsung", "NX1", Cloudy             , -4, { 1.778225806451613, 1, 1.7298387096774193, 0 } },
  { "Samsung", "NX1", Cloudy             , -3, { 1.8064516129032258, 1, 1.6653225806451613, 0 } },
  { "Samsung", "NX1", Cloudy             , -2, { 1.8387096774193548, 1, 1.596774193548387, 0 } },
  { "Samsung", "NX1", Cloudy             , -1, { 1.8709677419354838, 1, 1.532258064516129, 0 } },
  { "Samsung", "NX1", Cloudy             ,  0, { 1.8991935483870968, 1, 1.4637096774193548, 0 } },
  { "Samsung", "NX1", Cloudy             ,  1, { 1.9314516129032258, 1, 1.4274193548387097, 0 } },
  { "Samsung", "NX1", Cloudy             ,  2, { 1.9637096774193548, 1, 1.3911290322580645, 0 } },
  { "Samsung", "NX1", Cloudy             ,  3, { 2.0, 1, 1.3548387096774193, 0 } },
  { "Samsung", "NX1", Cloudy             ,  4, { 2.032258064516129, 1, 1.3185483870967742, 0 } },
  { "Samsung", "NX1", Cloudy             ,  5, { 2.064516129032258, 1, 1.282258064516129, 0 } },
  { "Samsung", "NX1", Cloudy             ,  6, { 2.096774193548387, 1, 1.2459677419354838, 0 } },
  { "Samsung", "NX1", Cloudy             ,  7, { 2.1330645161290325, 1, 1.2096774193548387, 0 } },
  { "Samsung", "NX1", Tungsten           , -7, { 1.153225806451613, 1, 3.600806451612903, 0 } },
  { "Samsung", "NX1", Tungsten           , -6, { 1.1733870967741935, 1, 3.475806451612903, 0 } },
  { "Samsung", "NX1", Tungsten           , -5, { 1.1975806451612903, 1, 3.3548387096774195, 0 } },
  { "Samsung", "NX1", Tungsten           , -4, { 1.217741935483871, 1, 3.2298387096774195, 0 } },
  { "Samsung", "NX1", Tungsten           , -3, { 1.2379032258064515, 1, 3.1088709677419355, 0 } },
  { "Samsung", "NX1", Tungsten           , -2, { 1.2580645161290323, 1, 2.9838709677419355, 0 } },
  { "Samsung", "NX1", Tungsten           , -1, { 1.278225806451613, 1, 2.8629032258064515, 0 } },
  { "Samsung", "NX1", Tungsten           ,  0, { 1.3024193548387097, 1, 2.7379032258064515, 0 } },
  { "Samsung", "NX1", Tungsten           ,  1, { 1.3225806451612903, 1, 2.6733870967741935, 0 } },
  { "Samsung", "NX1", Tungsten           ,  2, { 1.346774193548387, 1, 2.6048387096774195, 0 } },
  { "Samsung", "NX1", Tungsten           ,  3, { 1.3709677419354838, 1, 2.536290322580645, 0 } },
  { "Samsung", "NX1", Tungsten           ,  4, { 1.3911290322580645, 1, 2.471774193548387, 0 } },
  { "Samsung", "NX1", Tungsten           ,  5, { 1.4153225806451613, 1, 2.403225806451613, 0 } },
  { "Samsung", "NX1", Tungsten           ,  6, { 1.439516129032258, 1, 2.3346774193548385, 0 } },
  { "Samsung", "NX1", Tungsten           ,  7, { 1.4596774193548387, 1, 2.2661290322580645, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -7, { 2.024193548387097, 1, 2.2338709677419355, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -6, { 2.060483870967742, 1, 2.157258064516129, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -5, { 2.096774193548387, 1, 2.0806451612903225, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -4, { 2.1330645161290325, 1, 2.004032258064516, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -3, { 2.1733870967741935, 1, 1.9274193548387097, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -2, { 2.2096774193548385, 1, 1.8508064516129032, 0 } },
  { "Samsung", "NX1", DaylightFluorescent, -1, { 2.245967741935484, 1, 1.7741935483870968, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  0, { 2.282258064516129, 1, 1.6975806451612903, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  1, { 2.318548387096774, 1, 1.657258064516129, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  2, { 2.3588709677419355, 1, 1.6129032258064515, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  3, { 2.399193548387097, 1, 1.5725806451612903, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  4, { 2.439516129032258, 1, 1.528225806451613, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  5, { 2.4798387096774195, 1, 1.4879032258064515, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  6, { 2.5201612903225805, 1, 1.4475806451612903, 0 } },
  { "Samsung", "NX1", DaylightFluorescent,  7, { 2.556451612903226, 1, 1.403225806451613, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -7, { 1.7620967741935485, 1, 2.7016129032258065, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -6, { 1.7943548387096775, 1, 2.6088709677419355, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -5, { 1.8225806451612903, 1, 2.5161290322580645, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -4, { 1.8548387096774193, 1, 2.4233870967741935, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -3, { 1.8870967741935485, 1, 2.3306451612903225, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -2, { 1.9193548387096775, 1, 2.2379032258064515, 0 } },
  { "Samsung", "NX1", NeutralFluorescent , -1, { 1.9516129032258065, 1, 2.1451612903225805, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  0, { 1.9838709677419355, 1, 2.0524193548387095, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  1, { 2.0161290322580645, 1, 2.004032258064516, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  2, { 2.0524193548387095, 1, 1.9516129032258065, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  3, { 2.0846774193548385, 1, 1.903225806451613, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  4, { 2.120967741935484, 1, 1.8508064516129032, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  5, { 2.157258064516129, 1, 1.7983870967741935, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  6, { 2.189516129032258, 1, 1.75, 0 } },
  { "Samsung", "NX1", NeutralFluorescent ,  7, { 2.225806451612903, 1, 1.6975806451612903, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -7, { 1.6088709677419355, 1, 3.467741935483871, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -6, { 1.6370967741935485, 1, 3.350806451612903, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -5, { 1.6653225806451613, 1, 3.2298387096774195, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -4, { 1.6935483870967742, 1, 3.1129032258064515, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -3, { 1.721774193548387, 1, 2.995967741935484, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -2, { 1.7540322580645162, 1, 2.875, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   , -1, { 1.782258064516129, 1, 2.7580645161290325, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  0, { 1.810483870967742, 1, 2.6370967741935485, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  1, { 1.842741935483871, 1, 2.5725806451612905, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  2, { 1.875, 1, 2.5080645161290325, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  3, { 1.903225806451613, 1, 2.443548387096774, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  4, { 1.935483870967742, 1, 2.379032258064516, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  5, { 1.967741935483871, 1, 2.314516129032258, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  6, { 2.0, 1, 2.25, 0 } },
  { "Samsung", "NX1", WhiteFluorescent   ,  7, { 2.032258064516129, 1, 2.185483870967742, 0 } },
  { "Samsung", "NX1", Flash              , -7, { 1.7459677419354838, 1, 1.7379032258064515, 0 } },
  { "Samsung", "NX1", Flash              , -6, { 1.778225806451613, 1, 1.6774193548387097, 0 } },
  { "Samsung", "NX1", Flash              , -5, { 1.810483870967742, 1, 1.6169354838709677, 0 } },
  { "Samsung", "NX1", Flash              , -4, { 1.8387096774193548, 1, 1.5564516129032258, 0 } },
  { "Samsung", "NX1", Flash              , -3, { 1.8709677419354838, 1, 1.4959677419354838, 0 } },
  { "Samsung", "NX1", Flash              , -2, { 1.903225806451613, 1, 1.435483870967742, 0 } },
  { "Samsung", "NX1", Flash              , -1, { 1.935483870967742, 1, 1.3790322580645162, 0 } },
  { "Samsung", "NX1", Flash              ,  0, { 1.9637096774193548, 1, 1.3185483870967742, 0 } },
  { "Samsung", "NX1", Flash              ,  1, { 2.0, 1, 1.2862903225806452, 0 } },
  { "Samsung", "NX1", Flash              ,  2, { 2.032258064516129, 1, 1.25, 0 } },
  { "Samsung", "NX1", Flash              ,  3, { 2.068548387096774, 1, 1.217741935483871, 0 } },
  { "Samsung", "NX1", Flash              ,  4, { 2.100806451612903, 1, 1.185483870967742, 0 } },
  { "Samsung", "NX1", Flash              ,  5, { 2.1370967741935485, 1, 1.153225806451613, 0 } },
  { "Samsung", "NX1", Flash              ,  6, { 2.1733870967741935, 1, 1.1209677419354838, 0 } },
  { "Samsung", "NX1", Flash              ,  7, { 2.2056451612903225, 1, 1.0887096774193548, 0 } },

  // Samsung NX100 (firmware 1.12) white balance presets with finetuning steps
  { "Samsung", "NX100", Daylight, -7,		{ 1.566333, 1, 1.526753, 0 } },
  { "Samsung", "NX100", Daylight, -6,		{ 1.593941, 1, 1.475202, 0 } },
  { "Samsung", "NX100", Daylight, -5,		{ 1.621793, 1, 1.423650, 0 } },
  { "Samsung", "NX100", Daylight, -4,		{ 1.649401, 1, 1.371854, 0 } },
  { "Samsung", "NX100", Daylight, -3,		{ 1.677010, 1, 1.320303, 0 } },
  { "Samsung", "NX100", Daylight, -2,		{ 1.704618, 1, 1.268507, 0 } },
  { "Samsung", "NX100", Daylight, -1,		{ 1.732470, 1, 1.216956, 0 } },
  { "Samsung", "NX100", Daylight, 0,		{ 1.760078, 1, 1.165404, 0 } },
  { "Samsung", "NX100", Daylight, 1,		{ 1.790129, 1, 1.137063, 0 } },
  { "Samsung", "NX100", Daylight, 2,		{ 1.820425, 1, 1.108722, 0 } },
  { "Samsung", "NX100", Daylight, 3,		{ 1.850721, 1, 1.080381, 0 } },
  { "Samsung", "NX100", Daylight, 4,		{ 1.880772, 1, 1.052040, 0 } },
  { "Samsung", "NX100", Daylight, 5,		{ 1.911068, 1, 1.023699, 0 } },
  { "Samsung", "NX100", Daylight, 6,		{ 1.941377, 1, 0.995622, 0 } },
  { "Samsung", "NX100", Daylight, 7,		{ 1.971874, 1, 0.967384, 0 } },
  { "Samsung", "NX100", Cloudy, -7,		{ 1.638896, 1, 1.446372, 0 } },
  { "Samsung", "NX100", Cloudy, -6,		{ 1.667725, 1, 1.397508, 0 } },
  { "Samsung", "NX100", Cloudy, -5,		{ 1.696555, 1, 1.348644, 0 } },
  { "Samsung", "NX100", Cloudy, -4,		{ 1.725629, 1, 1.299780, 0 } },
  { "Samsung", "NX100", Cloudy, -3,		{ 1.754459, 1, 1.250672, 0 } },
  { "Samsung", "NX100", Cloudy, -2,		{ 1.783533, 1, 1.201808, 0 } },
  { "Samsung", "NX100", Cloudy, -1,		{ 1.812607, 1, 1.152944, 0 } },
  { "Samsung", "NX100", Cloudy, 0,		{ 1.841437, 1, 1.103836, 0 } },
  { "Samsung", "NX100", Cloudy, 1,		{ 1.872954, 1, 1.077205, 0 } },
  { "Samsung", "NX100", Cloudy, 2,		{ 1.904471, 1, 1.050330, 0 } },
  { "Samsung", "NX100", Cloudy, 3,		{ 1.936233, 1, 1.023455, 0 } },
  { "Samsung", "NX100", Cloudy, 4,		{ 1.967852, 1, 0.996834, 0 } },
  { "Samsung", "NX100", Cloudy, 5,		{ 1.999289, 1, 0.969905, 0 } },
  { "Samsung", "NX100", Cloudy, 6,		{ 2.030876, 1, 0.943088, 0 } },
  { "Samsung", "NX100", Cloudy, 7,		{ 2.062458, 1, 0.916275, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -7,	{ 1.526020, 1, 2.316638, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -6,	{ 1.553140, 1, 2.238212, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -5,	{ 1.580015, 1, 2.159785, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -4,	{ 1.606890, 1, 2.081603, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -3,	{ 1.634009, 1, 2.003176, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -2,	{ 1.660884, 1, 1.924750, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, -1,	{ 1.687760, 1, 1.846567, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 0,	{ 1.714879, 1, 1.768141, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 1,	{ 1.744197, 1, 1.725140, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 2,	{ 1.773516, 1, 1.682140, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 3,	{ 1.803078, 1, 1.639384, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 4,	{ 1.832397, 1, 1.596384, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 5,	{ 1.861959, 1, 1.553384, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 6,	{ 1.891522, 1, 1.510628, 0 } },
  { "Samsung", "NX100", WhiteFluorescent, 7,	{ 1.920596, 1, 1.467628, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -7,	{ 1.691180, 1, 1.884437, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -6,	{ 1.720987, 1, 1.821158, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -5,	{ 1.751038, 1, 1.757146, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -4,	{ 1.780845, 1, 1.693379, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -3,	{ 1.810897, 1, 1.629612, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -2,	{ 1.840704, 1, 1.565844, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, -1,	{ 1.870755, 1, 1.502077, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 0,	{ 1.900318, 1, 1.438309, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 1,	{ 1.933056, 1, 1.403616, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 2,	{ 1.965795, 1, 1.368678, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 3,	{ 1.998045, 1, 1.333741, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 4,	{ 2.030784, 1, 1.298803, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 5,	{ 2.063279, 1, 1.263621, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 6,	{ 2.095773, 1, 1.228927, 0 } },
  { "Samsung", "NX100", NeutralFluorescent, 7,	{ 2.128756, 1, 1.193990, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -7, { 1.995358, 1, 1.613731, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -6, { 2.030784, 1, 1.559492, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -5, { 2.065722, 1, 1.504764, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -4, { 2.101393, 1, 1.450037, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -3, { 2.136330, 1, 1.395553, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -2, { 2.171757, 1, 1.341070, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, -1, { 2.207183, 1, 1.286343, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 0,	{ 2.242365, 1, 1.231859, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 1,	{ 2.280723, 1, 1.202052, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 2,	{ 2.319326, 1, 1.172001, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 3,	{ 2.357684, 1, 1.142194, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 4,	{ 2.396042, 1, 1.112143, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 5,	{ 2.434645, 1, 1.082336, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 6,	{ 2.473003, 1, 1.052284, 0 } },
  { "Samsung", "NX100", DaylightFluorescent, 7,	{ 2.511361, 1, 1.022233, 0 } },
  { "Samsung", "NX100", Tungsten, -7,		{ 0.951650, 1, 3.086007, 0 } },
  { "Samsung", "NX100", Tungsten, -6,		{ 0.968307, 1, 2.981315, 0 } },
  { "Samsung", "NX100", Tungsten, -5,		{ 0.985078, 1, 2.877497, 0 } },
  { "Samsung", "NX100", Tungsten, -4,		{ 1.001710, 1, 2.772538, 0 } },
  { "Samsung", "NX100", Tungsten, -3,		{ 1.018568, 1, 2.668214, 0 } },
  { "Samsung", "NX100", Tungsten, -2,		{ 1.035426, 1, 2.563645, 0 } },
  { "Samsung", "NX100", Tungsten, -1,		{ 1.052284, 1, 2.459565, 0 } },
  { "Samsung", "NX100", Tungsten, 0,		{ 1.069142, 1, 2.354996, 0 } },
  { "Samsung", "NX100", Tungsten, 1,		{ 1.087466, 1, 2.298070, 0 } },
  { "Samsung", "NX100", Tungsten, 2,		{ 1.105790, 1, 2.240655, 0 } },
  { "Samsung", "NX100", Tungsten, 3,		{ 1.123870, 1, 2.183484, 0 } },
  { "Samsung", "NX100", Tungsten, 4,		{ 1.142438, 1, 2.126313, 0 } },
  { "Samsung", "NX100", Tungsten, 5,		{ 1.160762, 1, 2.069142, 0 } },
  { "Samsung", "NX100", Tungsten, 6,		{ 1.179086, 1, 2.011972, 0 } },
  { "Samsung", "NX100", Tungsten, 7,		{ 1.197410, 1, 1.954801, 0 } },
  { "Samsung", "NX100", Flash, -7,		{ 1.843635, 1, 1.282189, 0 } },
  { "Samsung", "NX100", Flash, -6,		{ 1.876130, 1, 1.238700, 0 } },
  { "Samsung", "NX100", Flash, -5,		{ 1.908624, 1, 1.195456, 0 } },
  { "Samsung", "NX100", Flash, -4,		{ 1.941363, 1, 1.151967, 0 } },
  { "Samsung", "NX100", Flash, -3,		{ 1.973858, 1, 1.108722, 0 } },
  { "Samsung", "NX100", Flash, -2,		{ 2.006108, 1, 1.064989, 0 } },
  { "Samsung", "NX100", Flash, -1,		{ 2.038847, 1, 1.021989, 0 } },
  { "Samsung", "NX100", Flash, 0,		{ 2.071719, 1, 0.978723, 0 } },
  { "Samsung", "NX100", Flash, 1,		{ 2.107068, 1, 0.954980, 0 } },
  { "Samsung", "NX100", Flash, 2,		{ 2.142857, 1, 0.931301, 0 } },
  { "Samsung", "NX100", Flash, 3,		{ 2.178191, 1, 0.907358, 0 } },
  { "Samsung", "NX100", Flash, 4,		{ 2.213684, 1, 0.883661, 0 } },
  { "Samsung", "NX100", Flash, 5,		{ 2.249317, 1, 0.859903, 0 } },
  { "Samsung", "NX100", Flash, 6,		{ 2.284664, 1, 0.836022, 0 } },
  { "Samsung", "NX100", Flash, 7,		{ 2.320238, 1, 0.812302, 0 } },
  { "Samsung", "NX100", "5000K", 0,		{ 1.684339, 1, 1.094063, 0 } },
  { "Samsung", "NX100", "5500K", 0,		{ 1.692022, 1, 0.843950, 0 } },
  { "Samsung", "NX100", "6500K", 0,		{ 2.110371, 1, 0.879107, 0 } },

  // Copied from Samsung NX100
  { "Samsung", "NX5", Daylight, -7,		{ 1.566333, 1, 1.526753, 0 } },
  { "Samsung", "NX5", Daylight, -6,		{ 1.593941, 1, 1.475202, 0 } },
  { "Samsung", "NX5", Daylight, -5,		{ 1.621793, 1, 1.423650, 0 } },
  { "Samsung", "NX5", Daylight, -4,		{ 1.649401, 1, 1.371854, 0 } },
  { "Samsung", "NX5", Daylight, -3,		{ 1.677010, 1, 1.320303, 0 } },
  { "Samsung", "NX5", Daylight, -2,		{ 1.704618, 1, 1.268507, 0 } },
  { "Samsung", "NX5", Daylight, -1,		{ 1.732470, 1, 1.216956, 0 } },
  { "Samsung", "NX5", Daylight, 0,		{ 1.760078, 1, 1.165404, 0 } },
  { "Samsung", "NX5", Daylight, 1,		{ 1.790129, 1, 1.137063, 0 } },
  { "Samsung", "NX5", Daylight, 2,		{ 1.820425, 1, 1.108722, 0 } },
  { "Samsung", "NX5", Daylight, 3,		{ 1.850721, 1, 1.080381, 0 } },
  { "Samsung", "NX5", Daylight, 4,		{ 1.880772, 1, 1.052040, 0 } },
  { "Samsung", "NX5", Daylight, 5,		{ 1.911068, 1, 1.023699, 0 } },
  { "Samsung", "NX5", Daylight, 6,		{ 1.941377, 1, 0.995622, 0 } },
  { "Samsung", "NX5", Daylight, 7,		{ 1.971874, 1, 0.967384, 0 } },
  { "Samsung", "NX5", Cloudy, -7,		{ 1.638896, 1, 1.446372, 0 } },
  { "Samsung", "NX5", Cloudy, -6,		{ 1.667725, 1, 1.397508, 0 } },
  { "Samsung", "NX5", Cloudy, -5,		{ 1.696555, 1, 1.348644, 0 } },
  { "Samsung", "NX5", Cloudy, -4,		{ 1.725629, 1, 1.299780, 0 } },
  { "Samsung", "NX5", Cloudy, -3,		{ 1.754459, 1, 1.250672, 0 } },
  { "Samsung", "NX5", Cloudy, -2,		{ 1.783533, 1, 1.201808, 0 } },
  { "Samsung", "NX5", Cloudy, -1,		{ 1.812607, 1, 1.152944, 0 } },
  { "Samsung", "NX5", Cloudy, 0,		{ 1.841437, 1, 1.103836, 0 } },
  { "Samsung", "NX5", Cloudy, 1,		{ 1.872954, 1, 1.077205, 0 } },
  { "Samsung", "NX5", Cloudy, 2,		{ 1.904471, 1, 1.050330, 0 } },
  { "Samsung", "NX5", Cloudy, 3,		{ 1.936233, 1, 1.023455, 0 } },
  { "Samsung", "NX5", Cloudy, 4,		{ 1.967852, 1, 0.996834, 0 } },
  { "Samsung", "NX5", Cloudy, 5,		{ 1.999289, 1, 0.969905, 0 } },
  { "Samsung", "NX5", Cloudy, 6,		{ 2.030876, 1, 0.943088, 0 } },
  { "Samsung", "NX5", Cloudy, 7,		{ 2.062458, 1, 0.916275, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -7,	{ 1.526020, 1, 2.316638, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -6,	{ 1.553140, 1, 2.238212, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -5,	{ 1.580015, 1, 2.159785, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -4,	{ 1.606890, 1, 2.081603, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -3,	{ 1.634009, 1, 2.003176, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -2,	{ 1.660884, 1, 1.924750, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, -1,	{ 1.687760, 1, 1.846567, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 0,	{ 1.714879, 1, 1.768141, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 1,	{ 1.744197, 1, 1.725140, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 2,	{ 1.773516, 1, 1.682140, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 3,	{ 1.803078, 1, 1.639384, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 4,	{ 1.832397, 1, 1.596384, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 5,	{ 1.861959, 1, 1.553384, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 6,	{ 1.891522, 1, 1.510628, 0 } },
  { "Samsung", "NX5", WhiteFluorescent, 7,	{ 1.920596, 1, 1.467628, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -7,	{ 1.691180, 1, 1.884437, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -6,	{ 1.720987, 1, 1.821158, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -5,	{ 1.751038, 1, 1.757146, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -4,	{ 1.780845, 1, 1.693379, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -3,	{ 1.810897, 1, 1.629612, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -2,	{ 1.840704, 1, 1.565844, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, -1,	{ 1.870755, 1, 1.502077, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 0,	{ 1.900318, 1, 1.438309, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 1,	{ 1.933056, 1, 1.403616, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 2,	{ 1.965795, 1, 1.368678, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 3,	{ 1.998045, 1, 1.333741, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 4,	{ 2.030784, 1, 1.298803, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 5,	{ 2.063279, 1, 1.263621, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 6,	{ 2.095773, 1, 1.228927, 0 } },
  { "Samsung", "NX5", NeutralFluorescent, 7,	{ 2.128756, 1, 1.193990, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -7,	{ 1.995358, 1, 1.613731, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -6,	{ 2.030784, 1, 1.559492, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -5,	{ 2.065722, 1, 1.504764, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -4,	{ 2.101393, 1, 1.450037, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -3,	{ 2.136330, 1, 1.395553, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -2,	{ 2.171757, 1, 1.341070, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, -1,	{ 2.207183, 1, 1.286343, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 0,	{ 2.242365, 1, 1.231859, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 1,	{ 2.280723, 1, 1.202052, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 2,	{ 2.319326, 1, 1.172001, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 3,	{ 2.357684, 1, 1.142194, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 4,	{ 2.396042, 1, 1.112143, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 5,	{ 2.434645, 1, 1.082336, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 6,	{ 2.473003, 1, 1.052284, 0 } },
  { "Samsung", "NX5", DaylightFluorescent, 7,	{ 2.511361, 1, 1.022233, 0 } },
  { "Samsung", "NX5", Tungsten, -7,		{ 0.951650, 1, 3.086007, 0 } },
  { "Samsung", "NX5", Tungsten, -6,		{ 0.968307, 1, 2.981315, 0 } },
  { "Samsung", "NX5", Tungsten, -5,		{ 0.985078, 1, 2.877497, 0 } },
  { "Samsung", "NX5", Tungsten, -4,		{ 1.001710, 1, 2.772538, 0 } },
  { "Samsung", "NX5", Tungsten, -3,		{ 1.018568, 1, 2.668214, 0 } },
  { "Samsung", "NX5", Tungsten, -2,		{ 1.035426, 1, 2.563645, 0 } },
  { "Samsung", "NX5", Tungsten, -1,		{ 1.052284, 1, 2.459565, 0 } },
  { "Samsung", "NX5", Tungsten, 0,		{ 1.069142, 1, 2.354996, 0 } },
  { "Samsung", "NX5", Tungsten, 1,		{ 1.087466, 1, 2.298070, 0 } },
  { "Samsung", "NX5", Tungsten, 2,		{ 1.105790, 1, 2.240655, 0 } },
  { "Samsung", "NX5", Tungsten, 3,		{ 1.123870, 1, 2.183484, 0 } },
  { "Samsung", "NX5", Tungsten, 4,		{ 1.142438, 1, 2.126313, 0 } },
  { "Samsung", "NX5", Tungsten, 5,		{ 1.160762, 1, 2.069142, 0 } },
  { "Samsung", "NX5", Tungsten, 6,		{ 1.179086, 1, 2.011972, 0 } },
  { "Samsung", "NX5", Tungsten, 7,		{ 1.197410, 1, 1.954801, 0 } },
  { "Samsung", "NX5", Flash, -7,		{ 1.843635, 1, 1.282189, 0 } },
  { "Samsung", "NX5", Flash, -6,		{ 1.876130, 1, 1.238700, 0 } },
  { "Samsung", "NX5", Flash, -5,		{ 1.908624, 1, 1.195456, 0 } },
  { "Samsung", "NX5", Flash, -4,		{ 1.941363, 1, 1.151967, 0 } },
  { "Samsung", "NX5", Flash, -3,		{ 1.973858, 1, 1.108722, 0 } },
  { "Samsung", "NX5", Flash, -2,		{ 2.006108, 1, 1.064989, 0 } },
  { "Samsung", "NX5", Flash, -1,		{ 2.038847, 1, 1.021989, 0 } },
  { "Samsung", "NX5", Flash, 0,			{ 2.071719, 1, 0.978723, 0 } },
  { "Samsung", "NX5", Flash, 1,			{ 2.107068, 1, 0.954980, 0 } },
  { "Samsung", "NX5", Flash, 2,			{ 2.142857, 1, 0.931301, 0 } },
  { "Samsung", "NX5", Flash, 3,			{ 2.178191, 1, 0.907358, 0 } },
  { "Samsung", "NX5", Flash, 4,			{ 2.213684, 1, 0.883661, 0 } },
  { "Samsung", "NX5", Flash, 5,			{ 2.249317, 1, 0.859903, 0 } },
  { "Samsung", "NX5", Flash, 6,			{ 2.284664, 1, 0.836022, 0 } },
  { "Samsung", "NX5", Flash, 7,			{ 2.320238, 1, 0.812302, 0 } },
  { "Samsung", "NX5", "5000K", 0,		{ 1.684339, 1, 1.094063, 0 } },
  { "Samsung", "NX5", "5500K", 0,		{ 1.692022, 1, 0.843950, 0 } },
  { "Samsung", "NX5", "6500K", 0,		{ 2.110371, 1, 0.879107, 0 } },

  // Copied from Samsung NX100
  { "Samsung", "NX10", Daylight, -7,		{ 1.566333, 1, 1.526753, 0 } },
  { "Samsung", "NX10", Daylight, -6,		{ 1.593941, 1, 1.475202, 0 } },
  { "Samsung", "NX10", Daylight, -5,		{ 1.621793, 1, 1.423650, 0 } },
  { "Samsung", "NX10", Daylight, -4,		{ 1.649401, 1, 1.371854, 0 } },
  { "Samsung", "NX10", Daylight, -3,		{ 1.677010, 1, 1.320303, 0 } },
  { "Samsung", "NX10", Daylight, -2,		{ 1.704618, 1, 1.268507, 0 } },
  { "Samsung", "NX10", Daylight, -1,		{ 1.732470, 1, 1.216956, 0 } },
  { "Samsung", "NX10", Daylight, 0,		{ 1.760078, 1, 1.165404, 0 } },
  { "Samsung", "NX10", Daylight, 1,		{ 1.790129, 1, 1.137063, 0 } },
  { "Samsung", "NX10", Daylight, 2,		{ 1.820425, 1, 1.108722, 0 } },
  { "Samsung", "NX10", Daylight, 3,		{ 1.850721, 1, 1.080381, 0 } },
  { "Samsung", "NX10", Daylight, 4,		{ 1.880772, 1, 1.052040, 0 } },
  { "Samsung", "NX10", Daylight, 5,		{ 1.911068, 1, 1.023699, 0 } },
  { "Samsung", "NX10", Daylight, 6,		{ 1.941377, 1, 0.995622, 0 } },
  { "Samsung", "NX10", Daylight, 7,		{ 1.971874, 1, 0.967384, 0 } },
  { "Samsung", "NX10", Cloudy, -7,		{ 1.638896, 1, 1.446372, 0 } },
  { "Samsung", "NX10", Cloudy, -6,		{ 1.667725, 1, 1.397508, 0 } },
  { "Samsung", "NX10", Cloudy, -5,		{ 1.696555, 1, 1.348644, 0 } },
  { "Samsung", "NX10", Cloudy, -4,		{ 1.725629, 1, 1.299780, 0 } },
  { "Samsung", "NX10", Cloudy, -3,		{ 1.754459, 1, 1.250672, 0 } },
  { "Samsung", "NX10", Cloudy, -2,		{ 1.783533, 1, 1.201808, 0 } },
  { "Samsung", "NX10", Cloudy, -1,		{ 1.812607, 1, 1.152944, 0 } },
  { "Samsung", "NX10", Cloudy, 0,		{ 1.841437, 1, 1.103836, 0 } },
  { "Samsung", "NX10", Cloudy, 1,		{ 1.872954, 1, 1.077205, 0 } },
  { "Samsung", "NX10", Cloudy, 2,		{ 1.904471, 1, 1.050330, 0 } },
  { "Samsung", "NX10", Cloudy, 3,		{ 1.936233, 1, 1.023455, 0 } },
  { "Samsung", "NX10", Cloudy, 4,		{ 1.967852, 1, 0.996834, 0 } },
  { "Samsung", "NX10", Cloudy, 5,		{ 1.999289, 1, 0.969905, 0 } },
  { "Samsung", "NX10", Cloudy, 6,		{ 2.030876, 1, 0.943088, 0 } },
  { "Samsung", "NX10", Cloudy, 7,		{ 2.062458, 1, 0.916275, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -7,	{ 1.526020, 1, 2.316638, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -6,	{ 1.553140, 1, 2.238212, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -5,	{ 1.580015, 1, 2.159785, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -4,	{ 1.606890, 1, 2.081603, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -3,	{ 1.634009, 1, 2.003176, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -2,	{ 1.660884, 1, 1.924750, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, -1,	{ 1.687760, 1, 1.846567, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 0,	{ 1.714879, 1, 1.768141, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 1,	{ 1.744197, 1, 1.725140, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 2,	{ 1.773516, 1, 1.682140, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 3,	{ 1.803078, 1, 1.639384, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 4,	{ 1.832397, 1, 1.596384, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 5,	{ 1.861959, 1, 1.553384, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 6,	{ 1.891522, 1, 1.510628, 0 } },
  { "Samsung", "NX10", WhiteFluorescent, 7,	{ 1.920596, 1, 1.467628, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -7,	{ 1.691180, 1, 1.884437, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -6,	{ 1.720987, 1, 1.821158, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -5,	{ 1.751038, 1, 1.757146, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -4,	{ 1.780845, 1, 1.693379, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -3,	{ 1.810897, 1, 1.629612, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -2,	{ 1.840704, 1, 1.565844, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, -1,	{ 1.870755, 1, 1.502077, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 0,	{ 1.900318, 1, 1.438309, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 1,	{ 1.933056, 1, 1.403616, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 2,	{ 1.965795, 1, 1.368678, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 3,	{ 1.998045, 1, 1.333741, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 4,	{ 2.030784, 1, 1.298803, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 5,	{ 2.063279, 1, 1.263621, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 6,	{ 2.095773, 1, 1.228927, 0 } },
  { "Samsung", "NX10", NeutralFluorescent, 7,	{ 2.128756, 1, 1.193990, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -7,	{ 1.995358, 1, 1.613731, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -6,	{ 2.030784, 1, 1.559492, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -5,	{ 2.065722, 1, 1.504764, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -4,	{ 2.101393, 1, 1.450037, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -3,	{ 2.136330, 1, 1.395553, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -2,	{ 2.171757, 1, 1.341070, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, -1,	{ 2.207183, 1, 1.286343, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 0,	{ 2.242365, 1, 1.231859, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 1,	{ 2.280723, 1, 1.202052, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 2,	{ 2.319326, 1, 1.172001, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 3,	{ 2.357684, 1, 1.142194, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 4,	{ 2.396042, 1, 1.112143, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 5,	{ 2.434645, 1, 1.082336, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 6,	{ 2.473003, 1, 1.052284, 0 } },
  { "Samsung", "NX10", DaylightFluorescent, 7,	{ 2.511361, 1, 1.022233, 0 } },
  { "Samsung", "NX10", Tungsten, -7,		{ 0.951650, 1, 3.086007, 0 } },
  { "Samsung", "NX10", Tungsten, -6,		{ 0.968307, 1, 2.981315, 0 } },
  { "Samsung", "NX10", Tungsten, -5,		{ 0.985078, 1, 2.877497, 0 } },
  { "Samsung", "NX10", Tungsten, -4,		{ 1.001710, 1, 2.772538, 0 } },
  { "Samsung", "NX10", Tungsten, -3,		{ 1.018568, 1, 2.668214, 0 } },
  { "Samsung", "NX10", Tungsten, -2,		{ 1.035426, 1, 2.563645, 0 } },
  { "Samsung", "NX10", Tungsten, -1,		{ 1.052284, 1, 2.459565, 0 } },
  { "Samsung", "NX10", Tungsten, 0,		{ 1.069142, 1, 2.354996, 0 } },
  { "Samsung", "NX10", Tungsten, 1,		{ 1.087466, 1, 2.298070, 0 } },
  { "Samsung", "NX10", Tungsten, 2,		{ 1.105790, 1, 2.240655, 0 } },
  { "Samsung", "NX10", Tungsten, 3,		{ 1.123870, 1, 2.183484, 0 } },
  { "Samsung", "NX10", Tungsten, 4,		{ 1.142438, 1, 2.126313, 0 } },
  { "Samsung", "NX10", Tungsten, 5,		{ 1.160762, 1, 2.069142, 0 } },
  { "Samsung", "NX10", Tungsten, 6,		{ 1.179086, 1, 2.011972, 0 } },
  { "Samsung", "NX10", Tungsten, 7,		{ 1.197410, 1, 1.954801, 0 } },
  { "Samsung", "NX10", Flash, -7,		{ 1.843635, 1, 1.282189, 0 } },
  { "Samsung", "NX10", Flash, -6,		{ 1.876130, 1, 1.238700, 0 } },
  { "Samsung", "NX10", Flash, -5,		{ 1.908624, 1, 1.195456, 0 } },
  { "Samsung", "NX10", Flash, -4,		{ 1.941363, 1, 1.151967, 0 } },
  { "Samsung", "NX10", Flash, -3,		{ 1.973858, 1, 1.108722, 0 } },
  { "Samsung", "NX10", Flash, -2,		{ 2.006108, 1, 1.064989, 0 } },
  { "Samsung", "NX10", Flash, -1,		{ 2.038847, 1, 1.021989, 0 } },
  { "Samsung", "NX10", Flash, 0,		{ 2.071719, 1, 0.978723, 0 } },
  { "Samsung", "NX10", Flash, 1,		{ 2.107068, 1, 0.954980, 0 } },
  { "Samsung", "NX10", Flash, 2,		{ 2.142857, 1, 0.931301, 0 } },
  { "Samsung", "NX10", Flash, 3,		{ 2.178191, 1, 0.907358, 0 } },
  { "Samsung", "NX10", Flash, 4,		{ 2.213684, 1, 0.883661, 0 } },
  { "Samsung", "NX10", Flash, 5,		{ 2.249317, 1, 0.859903, 0 } },
  { "Samsung", "NX10", Flash, 6,		{ 2.284664, 1, 0.836022, 0 } },
  { "Samsung", "NX10", Flash, 7,		{ 2.320238, 1, 0.812302, 0 } },
  { "Samsung", "NX10", "5000K", 0,		{ 1.684339, 1, 1.094063, 0 } },
  { "Samsung", "NX10", "5500K", 0,		{ 1.692022, 1, 0.843950, 0 } },
  { "Samsung", "NX10", "6500K", 0,		{ 2.110371, 1, 0.879107, 0 } },

  // Copied from Samsung NX100
  { "Samsung", "NX11", Daylight, -7,		{ 1.566333, 1, 1.526753, 0 } },
  { "Samsung", "NX11", Daylight, -6,		{ 1.593941, 1, 1.475202, 0 } },
  { "Samsung", "NX11", Daylight, -5,		{ 1.621793, 1, 1.423650, 0 } },
  { "Samsung", "NX11", Daylight, -4,		{ 1.649401, 1, 1.371854, 0 } },
  { "Samsung", "NX11", Daylight, -3,		{ 1.677010, 1, 1.320303, 0 } },
  { "Samsung", "NX11", Daylight, -2,		{ 1.704618, 1, 1.268507, 0 } },
  { "Samsung", "NX11", Daylight, -1,		{ 1.732470, 1, 1.216956, 0 } },
  { "Samsung", "NX11", Daylight, 0,		{ 1.760078, 1, 1.165404, 0 } },
  { "Samsung", "NX11", Daylight, 1,		{ 1.790129, 1, 1.137063, 0 } },
  { "Samsung", "NX11", Daylight, 2,		{ 1.820425, 1, 1.108722, 0 } },
  { "Samsung", "NX11", Daylight, 3,		{ 1.850721, 1, 1.080381, 0 } },
  { "Samsung", "NX11", Daylight, 4,		{ 1.880772, 1, 1.052040, 0 } },
  { "Samsung", "NX11", Daylight, 5,		{ 1.911068, 1, 1.023699, 0 } },
  { "Samsung", "NX11", Daylight, 6,		{ 1.941377, 1, 0.995622, 0 } },
  { "Samsung", "NX11", Daylight, 7,		{ 1.971874, 1, 0.967384, 0 } },
  { "Samsung", "NX11", Cloudy, -7,		{ 1.638896, 1, 1.446372, 0 } },
  { "Samsung", "NX11", Cloudy, -6,		{ 1.667725, 1, 1.397508, 0 } },
  { "Samsung", "NX11", Cloudy, -5,		{ 1.696555, 1, 1.348644, 0 } },
  { "Samsung", "NX11", Cloudy, -4,		{ 1.725629, 1, 1.299780, 0 } },
  { "Samsung", "NX11", Cloudy, -3,		{ 1.754459, 1, 1.250672, 0 } },
  { "Samsung", "NX11", Cloudy, -2,		{ 1.783533, 1, 1.201808, 0 } },
  { "Samsung", "NX11", Cloudy, -1,		{ 1.812607, 1, 1.152944, 0 } },
  { "Samsung", "NX11", Cloudy, 0,		{ 1.841437, 1, 1.103836, 0 } },
  { "Samsung", "NX11", Cloudy, 1,		{ 1.872954, 1, 1.077205, 0 } },
  { "Samsung", "NX11", Cloudy, 2,		{ 1.904471, 1, 1.050330, 0 } },
  { "Samsung", "NX11", Cloudy, 3,		{ 1.936233, 1, 1.023455, 0 } },
  { "Samsung", "NX11", Cloudy, 4,		{ 1.967852, 1, 0.996834, 0 } },
  { "Samsung", "NX11", Cloudy, 5,		{ 1.999289, 1, 0.969905, 0 } },
  { "Samsung", "NX11", Cloudy, 6,		{ 2.030876, 1, 0.943088, 0 } },
  { "Samsung", "NX11", Cloudy, 7,		{ 2.062458, 1, 0.916275, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -7,	{ 1.526020, 1, 2.316638, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -6,	{ 1.553140, 1, 2.238212, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -5,	{ 1.580015, 1, 2.159785, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -4,	{ 1.606890, 1, 2.081603, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -3,	{ 1.634009, 1, 2.003176, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -2,	{ 1.660884, 1, 1.924750, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, -1,	{ 1.687760, 1, 1.846567, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 0,	{ 1.714879, 1, 1.768141, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 1,	{ 1.744197, 1, 1.725140, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 2,	{ 1.773516, 1, 1.682140, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 3,	{ 1.803078, 1, 1.639384, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 4,	{ 1.832397, 1, 1.596384, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 5,	{ 1.861959, 1, 1.553384, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 6,	{ 1.891522, 1, 1.510628, 0 } },
  { "Samsung", "NX11", WhiteFluorescent, 7,	{ 1.920596, 1, 1.467628, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -7,	{ 1.691180, 1, 1.884437, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -6,	{ 1.720987, 1, 1.821158, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -5,	{ 1.751038, 1, 1.757146, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -4,	{ 1.780845, 1, 1.693379, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -3,	{ 1.810897, 1, 1.629612, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -2,	{ 1.840704, 1, 1.565844, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, -1,	{ 1.870755, 1, 1.502077, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 0,	{ 1.900318, 1, 1.438309, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 1,	{ 1.933056, 1, 1.403616, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 2,	{ 1.965795, 1, 1.368678, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 3,	{ 1.998045, 1, 1.333741, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 4,	{ 2.030784, 1, 1.298803, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 5,	{ 2.063279, 1, 1.263621, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 6,	{ 2.095773, 1, 1.228927, 0 } },
  { "Samsung", "NX11", NeutralFluorescent, 7,	{ 2.128756, 1, 1.193990, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -7,	{ 1.995358, 1, 1.613731, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -6,	{ 2.030784, 1, 1.559492, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -5,	{ 2.065722, 1, 1.504764, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -4,	{ 2.101393, 1, 1.450037, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -3,	{ 2.136330, 1, 1.395553, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -2,	{ 2.171757, 1, 1.341070, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, -1,	{ 2.207183, 1, 1.286343, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 0,	{ 2.242365, 1, 1.231859, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 1,	{ 2.280723, 1, 1.202052, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 2,	{ 2.319326, 1, 1.172001, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 3,	{ 2.357684, 1, 1.142194, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 4,	{ 2.396042, 1, 1.112143, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 5,	{ 2.434645, 1, 1.082336, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 6,	{ 2.473003, 1, 1.052284, 0 } },
  { "Samsung", "NX11", DaylightFluorescent, 7,	{ 2.511361, 1, 1.022233, 0 } },
  { "Samsung", "NX11", Tungsten, -7,		{ 0.951650, 1, 3.086007, 0 } },
  { "Samsung", "NX11", Tungsten, -6,		{ 0.968307, 1, 2.981315, 0 } },
  { "Samsung", "NX11", Tungsten, -5,		{ 0.985078, 1, 2.877497, 0 } },
  { "Samsung", "NX11", Tungsten, -4,		{ 1.001710, 1, 2.772538, 0 } },
  { "Samsung", "NX11", Tungsten, -3,		{ 1.018568, 1, 2.668214, 0 } },
  { "Samsung", "NX11", Tungsten, -2,		{ 1.035426, 1, 2.563645, 0 } },
  { "Samsung", "NX11", Tungsten, -1,		{ 1.052284, 1, 2.459565, 0 } },
  { "Samsung", "NX11", Tungsten, 0,		{ 1.069142, 1, 2.354996, 0 } },
  { "Samsung", "NX11", Tungsten, 1,		{ 1.087466, 1, 2.298070, 0 } },
  { "Samsung", "NX11", Tungsten, 2,		{ 1.105790, 1, 2.240655, 0 } },
  { "Samsung", "NX11", Tungsten, 3,		{ 1.123870, 1, 2.183484, 0 } },
  { "Samsung", "NX11", Tungsten, 4,		{ 1.142438, 1, 2.126313, 0 } },
  { "Samsung", "NX11", Tungsten, 5,		{ 1.160762, 1, 2.069142, 0 } },
  { "Samsung", "NX11", Tungsten, 6,		{ 1.179086, 1, 2.011972, 0 } },
  { "Samsung", "NX11", Tungsten, 7,		{ 1.197410, 1, 1.954801, 0 } },
  { "Samsung", "NX11", Flash, -7,		{ 1.843635, 1, 1.282189, 0 } },
  { "Samsung", "NX11", Flash, -6,		{ 1.876130, 1, 1.238700, 0 } },
  { "Samsung", "NX11", Flash, -5,		{ 1.908624, 1, 1.195456, 0 } },
  { "Samsung", "NX11", Flash, -4,		{ 1.941363, 1, 1.151967, 0 } },
  { "Samsung", "NX11", Flash, -3,		{ 1.973858, 1, 1.108722, 0 } },
  { "Samsung", "NX11", Flash, -2,		{ 2.006108, 1, 1.064989, 0 } },
  { "Samsung", "NX11", Flash, -1,		{ 2.038847, 1, 1.021989, 0 } },
  { "Samsung", "NX11", Flash, 0,		{ 2.071719, 1, 0.978723, 0 } },
  { "Samsung", "NX11", Flash, 1,		{ 2.107068, 1, 0.954980, 0 } },
  { "Samsung", "NX11", Flash, 2,		{ 2.142857, 1, 0.931301, 0 } },
  { "Samsung", "NX11", Flash, 3,		{ 2.178191, 1, 0.907358, 0 } },
  { "Samsung", "NX11", Flash, 4,		{ 2.213684, 1, 0.883661, 0 } },
  { "Samsung", "NX11", Flash, 5,		{ 2.249317, 1, 0.859903, 0 } },
  { "Samsung", "NX11", Flash, 6,		{ 2.284664, 1, 0.836022, 0 } },
  { "Samsung", "NX11", Flash, 7,		{ 2.320238, 1, 0.812302, 0 } },
  { "Samsung", "NX11", "5000K", 0,		{ 1.684339, 1, 1.094063, 0 } },
  { "Samsung", "NX11", "5500K", 0,		{ 1.692022, 1, 0.843950, 0 } },
  { "Samsung", "NX11", "6500K", 0,		{ 2.110371, 1, 0.879107, 0 } },

  // Samsung NX200 Firmware Version 1.04
  { "Samsung", "NX200", Daylight, 0,		{ 2.773438, 1, 1.625000, 0 } },
  { "Samsung", "NX200", Cloudy, 0,		{ 2.902344, 1, 1.546875, 0 } },
  { "Samsung", "NX200", WhiteFluorescent, 0,	{ 2.445313, 1, 2.316406, 0 } },
  { "Samsung", "NX200", NeutralFluorescent, 0,	{ 2.746094, 1, 1.937500, 0 } },
  { "Samsung", "NX200", DaylightFluorescent, 0,	{ 3.214844, 1, 1.679688, 0 } },
  { "Samsung", "NX200", Tungsten, 0,		{ 1.511719, 1, 2.941406, 0 } },
  { "Samsung", "NX200", Flash, 0,		{ 2.914063, 1, 1.191406, 0 } },

  // Copied from Samsung NX200
  { "Samsung", "NX210", Daylight, 0,		{ 2.773438, 1, 1.625000, 0 } },
  { "Samsung", "NX210", Cloudy, 0,		{ 2.902344, 1, 1.546875, 0 } },
  { "Samsung", "NX210", WhiteFluorescent, 0,	{ 2.445313, 1, 2.316406, 0 } },
  { "Samsung", "NX210", NeutralFluorescent, 0,	{ 2.746094, 1, 1.937500, 0 } },
  { "Samsung", "NX210", DaylightFluorescent, 0,	{ 3.214844, 1, 1.679688, 0 } },
  { "Samsung", "NX210", Tungsten, 0,		{ 1.511719, 1, 2.941406, 0 } },
  { "Samsung", "NX210", Flash, 0,		{ 2.914063, 1, 1.191406, 0 } },

  // Copied from Samsung NX200
  { "Samsung", "NX20", Daylight, 0,		{ 2.773438, 1, 1.625000, 0 } },
  { "Samsung", "NX20", Cloudy, 0,		{ 2.902344, 1, 1.546875, 0 } },
  { "Samsung", "NX20", WhiteFluorescent, 0,	{ 2.445313, 1, 2.316406, 0 } },
  { "Samsung", "NX20", NeutralFluorescent, 0,	{ 2.746094, 1, 1.937500, 0 } },
  { "Samsung", "NX20", DaylightFluorescent, 0,	{ 3.214844, 1, 1.679688, 0 } },
  { "Samsung", "NX20", Tungsten, 0,		{ 1.511719, 1, 2.941406, 0 } },
  { "Samsung", "NX20", Flash, 0,		{ 2.914063, 1, 1.191406, 0 } },

  // Copied from Samsung NX200
  { "Samsung", "NX1000", Daylight, 0,		{ 2.773438, 1, 1.625000, 0 } },
  { "Samsung", "NX1000", Cloudy, 0,		{ 2.902344, 1, 1.546875, 0 } },
  { "Samsung", "NX1000", WhiteFluorescent, 0,	{ 2.445313, 1, 2.316406, 0 } },
  { "Samsung", "NX1000", NeutralFluorescent, 0,	{ 2.746094, 1, 1.937500, 0 } },
  { "Samsung", "NX1000", DaylightFluorescent, 0,{ 3.214844, 1, 1.679688, 0 } },
  { "Samsung", "NX1000", Tungsten, 0,		{ 1.511719, 1, 2.941406, 0 } },
  { "Samsung", "NX1000", Flash, 0,		{ 2.914063, 1, 1.191406, 0 } },

  // Copied from Samsung NX200
  { "Samsung", "NX1100", Daylight, 0,		{ 2.773438, 1, 1.625000, 0 } },
  { "Samsung", "NX1100", Cloudy, 0,		{ 2.902344, 1, 1.546875, 0 } },
  { "Samsung", "NX1100", WhiteFluorescent, 0,	{ 2.445313, 1, 2.316406, 0 } },
  { "Samsung", "NX1100", NeutralFluorescent, 0,	{ 2.746094, 1, 1.937500, 0 } },
  { "Samsung", "NX1100", DaylightFluorescent, 0,{ 3.214844, 1, 1.679688, 0 } },
  { "Samsung", "NX1100", Tungsten, 0,		{ 1.511719, 1, 2.941406, 0 } },
  { "Samsung", "NX1100", Flash, 0,		{ 2.914063, 1, 1.191406, 0 } },

  // Samsung NX300 Firmware Version 1.40
  { "Samsung", "NX300", Daylight, -7,			{ 2.308594, 1, 2.097656, 0 } },
  { "Samsung", "NX300", Daylight, -6,			{ 2.347656, 1, 2.027344, 0 } },
  { "Samsung", "NX300", Daylight, -5,			{ 2.390625, 1, 1.953125, 0 } },
  { "Samsung", "NX300", Daylight, -4,			{ 2.429688, 1, 1.882813, 0 } },
  { "Samsung", "NX300", Daylight, -3,			{ 2.472656, 1, 1.812500, 0 } },
  { "Samsung", "NX300", Daylight, -2,			{ 2.511719, 1, 1.742188, 0 } },
  { "Samsung", "NX300", Daylight, -1,			{ 2.554688, 1, 1.671875, 0 } },
  { "Samsung", "NX300", Daylight, 0,			{ 2.593750, 1, 1.601563, 0 } },
  { "Samsung", "NX300", Daylight, 1,			{ 2.636719, 1, 1.562500, 0 } },
  { "Samsung", "NX300", Daylight, 2,			{ 2.683594, 1, 1.523438, 0 } },
  { "Samsung", "NX300", Daylight, 3,			{ 2.726563, 1, 1.484375, 0 } },
  { "Samsung", "NX300", Daylight, 4,			{ 2.773438, 1, 1.445313, 0 } },
  { "Samsung", "NX300", Daylight, 5,			{ 2.816406, 1, 1.406250, 0 } },
  { "Samsung", "NX300", Daylight, 6,			{ 2.859375, 1, 1.367188, 0 } },
  { "Samsung", "NX300", Daylight, 7,			{ 2.906250, 1, 1.328125, 0 } },
  { "Samsung", "NX300", Cloudy, -7,			{ 2.394531, 1, 2.011719, 0 } },
  { "Samsung", "NX300", Cloudy, -6,			{ 2.433594, 1, 1.945313, 0 } },
  { "Samsung", "NX300", Cloudy, -5,			{ 2.476563, 1, 1.878906, 0 } },
  { "Samsung", "NX300", Cloudy, -4,			{ 2.519531, 1, 1.808594, 0 } },
  { "Samsung", "NX300", Cloudy, -3,			{ 2.562500, 1, 1.742188, 0 } },
  { "Samsung", "NX300", Cloudy, -2,			{ 2.605469, 1, 1.671875, 0 } },
  { "Samsung", "NX300", Cloudy, -1,			{ 2.644531, 1, 1.605469, 0 } },
  { "Samsung", "NX300", Cloudy, 0,			{ 2.687500, 1, 1.535156, 0 } },
  { "Samsung", "NX300", Cloudy, 1,			{ 2.734375, 1, 1.500000, 0 } },
  { "Samsung", "NX300", Cloudy, 2,			{ 2.781250, 1, 1.460938, 0 } },
  { "Samsung", "NX300", Cloudy, 3,			{ 2.828125, 1, 1.425781, 0 } },
  { "Samsung", "NX300", Cloudy, 4,			{ 2.875000, 1, 1.386719, 0 } },
  { "Samsung", "NX300", Cloudy, 5,			{ 2.917969, 1, 1.351563, 0 } },
  { "Samsung", "NX300", Cloudy, 6,			{ 2.964844, 1, 1.312500, 0 } },
  { "Samsung", "NX300", Cloudy, 7,			{ 3.011719, 1, 1.273438, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -7,	{ 2.000000, 1, 2.933594, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -6,	{ 2.035156, 1, 2.835938, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -5,	{ 2.070313, 1, 2.734375, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -4,	{ 2.105469, 1, 2.636719, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -3,	{ 2.140625, 1, 2.539063, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -2,	{ 2.175781, 1, 2.437500, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, -1,	{ 2.214844, 1, 2.339844, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 0,	{ 2.250000, 1, 2.238281, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 1,	{ 2.285156, 1, 2.183594, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 2,	{ 2.324219, 1, 2.128906, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 3,	{ 2.363281, 1, 2.078125, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 4,	{ 2.402344, 1, 2.023438, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 5,	{ 2.441406, 1, 1.968750, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 6,	{ 2.480469, 1, 1.914063, 0 } },
  { "Samsung", "NX300", CoolWhiteFluorescent, 7,	{ 2.519531, 1, 1.859375, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -7,	{ 2.253906, 1, 2.445313, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -6,	{ 2.292969, 1, 2.363281, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -5,	{ 2.332031, 1, 2.281250, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -4,	{ 2.371094, 1, 2.199219, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -3,	{ 2.410156, 1, 2.117188, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -2,	{ 2.453125, 1, 2.031250, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, -1,	{ 2.492188, 1, 1.949219, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 0,		{ 2.531250, 1, 1.867188, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 1,		{ 2.574219, 1, 1.820313, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 2,		{ 2.617188, 1, 1.777344, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 3,		{ 2.660156, 1, 1.730469, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 4,		{ 2.703125, 1, 1.687500, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 5,		{ 2.750000, 1, 1.640625, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 6,		{ 2.792969, 1, 1.593750, 0 } },
  { "Samsung", "NX300", DayWhiteFluorescent, 7,		{ 2.835938, 1, 1.550781, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -7,	{ 2.699219, 1, 2.128906, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -6,	{ 2.746094, 1, 2.054688, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -5,	{ 2.792969, 1, 1.984375, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -4,	{ 2.839844, 1, 1.910156, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -3,	{ 2.890625, 1, 1.839844, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -2,	{ 2.937500, 1, 1.765625, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, -1,	{ 2.984375, 1, 1.695313, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 0,		{ 3.031250, 1, 1.625000, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 1,		{ 3.082031, 1, 1.582031, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 2,		{ 3.136719, 1, 1.542969, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 3,		{ 3.187500, 1, 1.503906, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 4,		{ 3.238281, 1, 1.464844, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 5,		{ 3.292969, 1, 1.425781, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 6,		{ 3.343750, 1, 1.386719, 0 } },
  { "Samsung", "NX300", DaylightFluorescent, 7,		{ 3.394531, 1, 1.347656, 0 } },
  { "Samsung", "NX300", Tungsten, -7,			{ 1.390625, 1, 3.695313, 0 } },
  { "Samsung", "NX300", Tungsten, -6,			{ 1.414063, 1, 3.570313, 0 } },
  { "Samsung", "NX300", Tungsten, -5,			{ 1.437500, 1, 3.445313, 0 } },
  { "Samsung", "NX300", Tungsten, -4,			{ 1.460938, 1, 3.320313, 0 } },
  { "Samsung", "NX300", Tungsten, -3,			{ 1.488281, 1, 3.195313, 0 } },
  { "Samsung", "NX300", Tungsten, -2,			{ 1.511719, 1, 3.070313, 0 } },
  { "Samsung", "NX300", Tungsten, -1,			{ 1.535156, 1, 2.945313, 0 } },
  { "Samsung", "NX300", Tungsten, 0,			{ 1.562500, 1, 2.820313, 0 } },
  { "Samsung", "NX300", Tungsten, 1,			{ 1.585938, 1, 2.753906, 0 } },
  { "Samsung", "NX300", Tungsten, 2,			{ 1.613281, 1, 2.683594, 0 } },
  { "Samsung", "NX300", Tungsten, 3,			{ 1.640625, 1, 2.617188, 0 } },
  { "Samsung", "NX300", Tungsten, 4,			{ 1.667969, 1, 2.546875, 0 } },
  { "Samsung", "NX300", Tungsten, 5,			{ 1.695313, 1, 2.480469, 0 } },
  { "Samsung", "NX300", Tungsten, 6,			{ 1.722656, 1, 2.410156, 0 } },
  { "Samsung", "NX300", Tungsten, 7,			{ 1.750000, 1, 2.339844, 0 } },
  { "Samsung", "NX300", Flash, -7,			{ 2.613281, 1, 1.957031, 0 } },
  { "Samsung", "NX300", Flash, -6,			{ 2.660156, 1, 1.890625, 0 } },
  { "Samsung", "NX300", Flash, -5,			{ 2.707031, 1, 1.824219, 0 } },
  { "Samsung", "NX300", Flash, -4,			{ 2.753906, 1, 1.757813, 0 } },
  { "Samsung", "NX300", Flash, -3,			{ 2.800781, 1, 1.691406, 0 } },
  { "Samsung", "NX300", Flash, -2,			{ 2.843750, 1, 1.625000, 0 } },
  { "Samsung", "NX300", Flash, -1,			{ 2.890625, 1, 1.558594, 0 } },
  { "Samsung", "NX300", Flash, 0,			{ 2.937500, 1, 1.492188, 0 } },
  { "Samsung", "NX300", Flash, 1,			{ 2.988281, 1, 1.457031, 0 } },
  { "Samsung", "NX300", Flash, 2,			{ 3.039063, 1, 1.421875, 0 } },
  { "Samsung", "NX300", Flash, 3,			{ 3.089844, 1, 1.386719, 0 } },
  { "Samsung", "NX300", Flash, 4,			{ 3.140625, 1, 1.347656, 0 } },
  { "Samsung", "NX300", Flash, 5,			{ 3.191406, 1, 1.312500, 0 } },
  { "Samsung", "NX300", Flash, 6,			{ 3.238281, 1, 1.277344, 0 } },
  { "Samsung", "NX300", Flash, 7,			{ 3.289063, 1, 1.238281, 0 } },
  { "Samsung", "NX300", "5000K", 0,			{ 2.402344, 1, 1.945313, 0 } },
  { "Samsung", "NX300", "5500K", 0,			{ 2.636719, 1, 1.792969, 0 } },
  { "Samsung", "NX300", "6500K", 0,			{ 3.003906, 1, 1.644531, 0 } },

  // Samsung NX500 Firmware Version 1.12 (Final)
  { "Samsung", "NX500", Daylight,                   -7, { 1.665323, 1, 1.866935, 0 } },
  { "Samsung", "NX500", Daylight,                   -6, { 1.693548, 1, 1.802419, 0 } },
  { "Samsung", "NX500", Daylight,                   -5, { 1.725806, 1, 1.737903, 0 } },
  { "Samsung", "NX500", Daylight,                   -4, { 1.754032, 1, 1.673387, 0 } },
  { "Samsung", "NX500", Daylight,                   -3, { 1.786290, 1, 1.608871, 0 } },
  { "Samsung", "NX500", Daylight,                   -2, { 1.814516, 1, 1.544355, 0 } },
  { "Samsung", "NX500", Daylight,                   -1, { 1.846774, 1, 1.479839, 0 } },
  { "Samsung", "NX500", Daylight,                    0, { 1.875000, 1, 1.415323, 0 } },
  { "Samsung", "NX500", Daylight,                    1, { 1.907258, 1, 1.383065, 0 } },
  { "Samsung", "NX500", Daylight,                    2, { 1.939516, 1, 1.346774, 0 } },
  { "Samsung", "NX500", Daylight,                    3, { 1.971774, 1, 1.310484, 0 } },
  { "Samsung", "NX500", Daylight,                    4, { 2.008065, 1, 1.274194, 0 } },
  { "Samsung", "NX500", Daylight,                    5, { 2.040323, 1, 1.241935, 0 } },
  { "Samsung", "NX500", Daylight,                    6, { 2.072581, 1, 1.205645, 0 } },
  { "Samsung", "NX500", Daylight,                    7, { 2.104839, 1, 1.169355, 0 } },
  { "Samsung", "NX500", Cloudy,                     -7, { 1.741935, 1, 1.653226, 0 } },
  { "Samsung", "NX500", Cloudy,                     -6, { 1.774194, 1, 1.592742, 0 } },
  { "Samsung", "NX500", Cloudy,                     -5, { 1.806452, 1, 1.536290, 0 } },
  { "Samsung", "NX500", Cloudy,                     -4, { 1.834677, 1, 1.479839, 0 } },
  { "Samsung", "NX500", Cloudy,                     -3, { 1.866935, 1, 1.423387, 0 } },
  { "Samsung", "NX500", Cloudy,                     -2, { 1.899194, 1, 1.366935, 0 } },
  { "Samsung", "NX500", Cloudy,                     -1, { 1.931452, 1, 1.310484, 0 } },
  { "Samsung", "NX500", Cloudy,                      0, { 1.959677, 1, 1.254032, 0 } },
  { "Samsung", "NX500", Cloudy,                      1, { 1.995968, 1, 1.221774, 0 } },
  { "Samsung", "NX500", Cloudy,                      2, { 2.028226, 1, 1.189516, 0 } },
  { "Samsung", "NX500", Cloudy,                      3, { 2.064516, 1, 1.157258, 0 } },
  { "Samsung", "NX500", Cloudy,                      4, { 2.096774, 1, 1.129032, 0 } },
  { "Samsung", "NX500", Cloudy,                      5, { 2.133065, 1, 1.096774, 0 } },
  { "Samsung", "NX500", Cloudy,                      6, { 2.165323, 1, 1.064516, 0 } },
  { "Samsung", "NX500", Cloudy,                      7, { 2.201613, 1, 1.032258, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -7, { 1.661290, 1, 2.971774, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -6, { 1.689516, 1, 2.870968, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -5, { 1.721774, 1, 2.770161, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -4, { 1.750000, 1, 2.665323, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -3, { 1.778226, 1, 2.564516, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -2, { 1.810484, 1, 2.463710, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,       -1, { 1.838710, 1, 2.362903, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        0, { 1.870968, 1, 2.262097, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        1, { 1.903226, 1, 2.205645, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        2, { 1.935484, 1, 2.149194, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        3, { 1.967742, 1, 2.092742, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        4, { 2.000000, 1, 2.036290, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        5, { 2.032258, 1, 1.979839, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        6, { 2.064516, 1, 1.927419, 0 } },
  { "Samsung", "NX500", CoolWhiteFluorescent,        7, { 2.096774, 1, 1.870968, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -7, { 1.818548, 1, 2.314516, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -6, { 1.850806, 1, 2.233871, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -5, { 1.883065, 1, 2.153226, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -4, { 1.915323, 1, 2.076613, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -3, { 1.947581, 1, 1.995968, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -2, { 1.983871, 1, 1.915323, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,        -1, { 2.016129, 1, 1.838710, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         0, { 2.048387, 1, 1.758065, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         1, { 2.084677, 1, 1.713710, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         2, { 2.120968, 1, 1.669355, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         3, { 2.153226, 1, 1.629032, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         4, { 2.189516, 1, 1.584677, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         5, { 2.225806, 1, 1.540323, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         6, { 2.262097, 1, 1.495968, 0 } },
  { "Samsung", "NX500", DayWhiteFluorescent,         7, { 2.298387, 1, 1.451613, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -7, { 2.092742, 1, 1.915323, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -6, { 2.129032, 1, 1.846774, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -5, { 2.165323, 1, 1.782258, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -4, { 2.205645, 1, 1.717742, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -3, { 2.241935, 1, 1.649194, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -2, { 2.278226, 1, 1.584677, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,        -1, { 2.318548, 1, 1.520161, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         0, { 2.354839, 1, 1.451613, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         1, { 2.395161, 1, 1.415323, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         2, { 2.435484, 1, 1.379032, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         3, { 2.475806, 1, 1.342742, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         4, { 2.520161, 1, 1.306452, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         5, { 2.560484, 1, 1.274194, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         6, { 2.600806, 1, 1.237903, 0 } },
  { "Samsung", "NX500", DaylightFluorescent,         7, { 2.641129, 1, 1.201613, 0 } },
  { "Samsung", "NX500", Tungsten,                   -7, { 1.193548, 1, 3.084677, 0 } },
  { "Samsung", "NX500", Tungsten,                   -6, { 1.213710, 1, 2.979839, 0 } },
  { "Samsung", "NX500", Tungsten,                   -5, { 1.233871, 1, 2.875000, 0 } },
  { "Samsung", "NX500", Tungsten,                   -4, { 1.258065, 1, 2.770161, 0 } },
  { "Samsung", "NX500", Tungsten,                   -3, { 1.278226, 1, 2.661290, 0 } },
  { "Samsung", "NX500", Tungsten,                   -2, { 1.298387, 1, 2.556452, 0 } },
  { "Samsung", "NX500", Tungsten,                   -1, { 1.322581, 1, 2.451613, 0 } },
  { "Samsung", "NX500", Tungsten,                    0, { 1.342742, 1, 2.346774, 0 } },
  { "Samsung", "NX500", Tungsten,                    1, { 1.366935, 1, 2.290323, 0 } },
  { "Samsung", "NX500", Tungsten,                    2, { 1.391129, 1, 2.229839, 0 } },
  { "Samsung", "NX500", Tungsten,                    3, { 1.415323, 1, 2.173387, 0 } },
  { "Samsung", "NX500", Tungsten,                    4, { 1.439516, 1, 2.116935, 0 } },
  { "Samsung", "NX500", Tungsten,                    5, { 1.459677, 1, 2.056452, 0 } },
  { "Samsung", "NX500", Tungsten,                    6, { 1.483871, 1, 2.000000, 0 } },
  { "Samsung", "NX500", Tungsten,                    7, { 1.508065, 1, 1.943548, 0 } },
  { "Samsung", "NX500", Flash,                      -7, { 1.802419, 1, 1.483871, 0 } },
  { "Samsung", "NX500", Flash,                      -6, { 1.834677, 1, 1.435484, 0 } },
  { "Samsung", "NX500", Flash,                      -5, { 1.866935, 1, 1.383065, 0 } },
  { "Samsung", "NX500", Flash,                      -4, { 1.899194, 1, 1.330645, 0 } },
  { "Samsung", "NX500", Flash,                      -3, { 1.931452, 1, 1.282258, 0 } },
  { "Samsung", "NX500", Flash,                      -2, { 1.963710, 1, 1.229839, 0 } },
  { "Samsung", "NX500", Flash,                      -1, { 1.995968, 1, 1.177419, 0 } },
  { "Samsung", "NX500", Flash,                       0, { 2.028226, 1, 1.125000, 0 } },
  { "Samsung", "NX500", Flash,                       1, { 2.064516, 1, 1.096774, 0 } },
  { "Samsung", "NX500", Flash,                       2, { 2.100806, 1, 1.068548, 0 } },
  { "Samsung", "NX500", Flash,                       3, { 2.137097, 1, 1.040323, 0 } },
  { "Samsung", "NX500", Flash,                       4, { 2.173387, 1, 1.012097, 0 } },
  { "Samsung", "NX500", Flash,                       5, { 2.205645, 1, 0.983871, 0 } },
  { "Samsung", "NX500", Flash,                       6, { 2.241935, 1, 0.955645, 0 } },
  { "Samsung", "NX500", Flash,                       7, { 2.278226, 1, 0.927419, 0 } },
  { "Samsung", "NX500", "2700K",                     0, { 1.282258, 1, 3.173387, 0 } },
  { "Samsung", "NX500", "3000K",                     0, { 1.431452, 1, 2.842742, 0 } },
  { "Samsung", "NX500", "4000K",                     0, { 1.778226, 1, 2.064516, 0 } },
  { "Samsung", "NX500", "5000K",                     0, { 2.028226, 1, 1.673387, 0 } },
  { "Samsung", "NX500", "5500K",                     0, { 2.120968, 1, 1.560484, 0 } },
  { "Samsung", "NX500", "6500K",                     0, { 2.286290, 1, 1.350806, 0 } },

  // Firmware NXF1_01.01
  { "Samsung", "NX3000", Daylight, 0,			{ 2.214844, 1, 1.816406, 0 } },
  { "Samsung", "NX3000", Cloudy, 0,			{ 2.347656, 1, 1.656250, 0 } },
  { "Samsung", "NX3000", CoolWhiteFluorescent, 0,	{ 2.023438, 1, 2.460938, 0 } },
  { "Samsung", "NX3000", DayWhiteFluorescent, 0,	{ 2.175781, 1, 2.152344, 0 } },
  { "Samsung", "NX3000", DaylightFluorescent, 0,	{ 2.582031, 1, 1.835938, 0 } },
  { "Samsung", "NX3000", Tungsten, 0,			{ 1.539063, 1, 2.816406, 0 } },
  { "Samsung", "NX3000", Flash, 0,			{ 2.597656, 1, 1.566406, 0 } },

  { "Sony", "DSC-RX0", Daylight            , 0, { 2.05078125, 1, 1.828125, 0 } },
  { "Sony", "DSC-RX0", Shade               , 0, { 2.4375, 1, 1.515625, 0 } },
  { "Sony", "DSC-RX0", Cloudy              , 0, { 2.2109375, 1, 1.6796875, 0 } },
  { "Sony", "DSC-RX0", Tungsten            , 0, { 1.265625, 1, 3.22265625, 0 } },
  { "Sony", "DSC-RX0", WarmWhiteFluorescent, 0, { 1.53125, 1, 3.1953125, 0 } },
  { "Sony", "DSC-RX0", CoolWhiteFluorescent, 0, { 2.02734375, 1, 2.62890625, 0 } },
  { "Sony", "DSC-RX0", DayWhiteFluorescent , 0, { 1.94921875, 1, 2.04296875, 0 } },
  { "Sony", "DSC-RX0", DaylightFluorescent , 0, { 2.2265625, 1, 1.7421875, 0 } },
  { "Sony", "DSC-RX0", Flash               , 0, { 1.8359375, 1, 2.02734375, 0 } },
  { "Sony", "DSC-RX0", "2500K"             , 0, { 1.1015625, 1, 3.84765625, 0 } },
  { "Sony", "DSC-RX0", "3200K"             , 0, { 1.4140625, 1, 2.80859375, 0 } },
  { "Sony", "DSC-RX0", "4500K"             , 0, { 1.84765625, 1, 2.05078125, 0 } },
  { "Sony", "DSC-RX0", "6000K"             , 0, { 2.1953125, 1, 1.6953125, 0 } },
  { "Sony", "DSC-RX0", "8500K"             , 0, { 2.5625, 1, 1.43359375, 0 } },

  { "Sony", "DSC-RX1", Daylight, 0, { 2.38671875, 1, 1.55078125, 0 } },
  { "Sony", "DSC-RX1", Shade, 0, { 2.84375, 1, 1.2734375, 0 } },
  { "Sony", "DSC-RX1", Cloudy, 0, { 2.578125, 1, 1.41796875, 0 } },
  { "Sony", "DSC-RX1", Tungsten, 0, { 1.47265625, 1, 2.87109375, 0 } },
  { "Sony", "DSC-RX1", WarmWhiteFluorescent, 0, { 1.703125, 1, 2.8515625, 0 } },
  { "Sony", "DSC-RX1", CoolWhiteFluorescent, 0, { 2.1484375, 1, 2.33984375, 0 } },
  { "Sony", "DSC-RX1", DayWh