/*
 * Copyright (c) 2015-2020 Red Hat, Inc.
 *
 * All rights reserved.
 *
 * Author: Jan Friesse (jfriesse@redhat.com)
 *
 * This software licensed under BSD license, the text of which follows:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * - Neither the name of the Red Hat, Inc. nor the names of its
 *   contributors may be used to endorse or promote products derived from this
 *   software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <errno.h>
#include <grp.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "utils.h"

int
main(void)
{
	long long int ll;
	long long int lli;
	double dbl;
	double dbli;
	char buf[32];
	int set_umask;
	mode_t umask;
	gid_t gid;
	struct group *grp;
	char *s;

	assert(utils_strtonum("0", 0, 100, &ll) == 0);
	assert(ll == 0);

	assert(utils_strtonum("100", 0, 100, &ll) == 0);
	assert(ll == 100);

	assert(utils_strtonum("101", 0, 100, &ll) != 0);
	assert(utils_strtonum("0", 1, 100, &ll) != 0);

	errno = ERANGE;
	assert(utils_strtonum("10", 0, 100, &ll) == 0);
	assert(ll == 10);

	assert(utils_strtonum("-1", -1, 0, &ll) == 0);
	assert(ll == -1);

	assert(utils_strtonum("-10", -20, -10, &ll) == 0);
	assert(ll == -10);

	assert(utils_strtonum("0", 1, 0, &ll) == -1);

	for (lli = -100; lli <= 100; lli++) {
		assert(snprintf(buf, sizeof(buf), "%lld", lli) > 0);

		assert(utils_strtonum(buf, -100, 100, &ll) == 0);
		assert(ll == lli);
	}

	assert(utils_strtonum("test", -1000, 1000, &ll) == -1);
	assert(utils_strtonum("12a", -1000, 1000, &ll) == -1);

	assert(utils_strtonum_base("0", 0, 0100, 8, &ll) == 0);
	assert(ll == 0);

	assert(utils_strtonum_base("100", 0, 0100, 8, &ll) == 0);
	assert(ll == 0100);

	assert(utils_strtonum_base("0100", 0, 0100, 8, &ll) == 0);
	assert(ll == 0100);

	assert(utils_strtonum_base("101", 0, 0100, 8, &ll) != 0);
	assert(utils_strtonum_base("0", 01, 0100, 8, &ll) != 0);
	assert(utils_strtonum_base("8", 0, 0100, 8, &ll) != 0);

	for (lli = 0; lli <= 0777; lli++) {
		assert(snprintf(buf, sizeof(buf), "%llo", lli) > 0);

		assert(utils_strtonum_base(buf, 0, 0777, 8, &ll) == 0);
		assert(ll == lli);
	}

	assert(utils_strtonum_base("0", 0, 0100, 16, &ll) == 0);
	assert(ll == 0);

	assert(utils_strtonum_base("100", 0, 0x100, 16, &ll) == 0);
	assert(ll == 0x100);

	assert(utils_strtonum_base("0x100", 0, 0x100, 16, &ll) == 0);
	assert(ll == 0x100);

	assert(utils_strtonum_base("101", 0, 0x0100, 16, &ll) != 0);
	assert(utils_strtonum_base("0", 0x1, 0x0100, 16, &ll) != 0);

	for (lli = 0; lli <= 0x100; lli++) {
		assert(snprintf(buf, sizeof(buf), "%llx", lli) > 0);

		assert(utils_strtonum_base(buf, 0, 0x100, 16, &ll) == 0);
		assert(ll == lli);
	}

	assert(utils_parse_bool_str("on") == 1);
	assert(utils_parse_bool_str("yes") == 1);
	assert(utils_parse_bool_str("1") == 1);
	assert(utils_parse_bool_str("ON") == 1);
	assert(utils_parse_bool_str("YeS") == 1);

	assert(utils_parse_bool_str("off") == 0);
	assert(utils_parse_bool_str("no") == 0);
	assert(utils_parse_bool_str("0") == 0);
	assert(utils_parse_bool_str("oFf") == 0);

	assert(utils_parse_bool_str("of") == -1);
	assert(utils_parse_bool_str("noo") == -1);
	assert(utils_parse_bool_str("01") == -1);

	assert(utils_strtod("0", 0, 100, &dbl) == 0);
	assert(dbl == 0);

	assert(utils_strtod("0.0", 0, 100, &dbl) == 0);
	assert(dbl == 0);

	assert(utils_strtod("100", 0, 100, &dbl) == 0);
	assert(dbl == 100);

	assert(utils_strtod("100.0", 0, 100, &dbl) == 0);
	assert(dbl == 100);

	assert(utils_strtod("99.9", 0, 100, &dbl) == 0);
	assert(dbl == 99.9);

	assert(utils_strtod("101", 0, 100, &dbl) != 0);
	assert(utils_strtod("0", 1, 100, &dbl) != 0);

	errno = ERANGE;
	assert(utils_strtod("10", 0, 100, &dbl) == 0);
	assert(dbl == 10);

	assert(utils_strtod("-1", -1, 0, &dbl) == 0);
	assert(dbl == -1);

	assert(utils_strtod("-10", -20, -10, &dbl) == 0);
	assert(dbl == -10);

	assert(utils_strtod("0", 1, 0, &dbl) == -1);

	for (dbli = -100; dbli <= 100; dbli += 0.01) {
		assert(snprintf(buf, sizeof(buf), "%f", dbli) > 0);

		assert(utils_strtod(buf, -100, 101, &dbl) == 0);
		assert(fabs(dbl - dbli) < 0.0001);
	}

	assert(utils_strtod("test", -1000, 1000, &dbl) == -1);
	assert(utils_strtod("12a", -1000, 1000, &dbl) == -1);
	assert(utils_strtod("inf", -1000, 1000, &dbl) == -1);
	assert(utils_strtod("nan", -1000, 1000, &dbl) == -1);

	assert(utils_parse_umask("0", &set_umask, &umask) == 0);
	assert(set_umask == 1 && umask == 0);

	assert(utils_parse_umask("02", &set_umask, &umask) == 0);
	assert(set_umask == 1 && umask == 02);

	assert(utils_parse_umask("777", &set_umask, &umask) == 0);
	assert(set_umask == 1 && umask == 0777);

	assert(utils_parse_umask("", &set_umask, &umask) == 0);
	assert(set_umask == 0);

	assert(utils_parse_umask("02", &set_umask, &umask) == 0);
	assert(set_umask == 1 && umask == 02);

	assert(utils_parse_umask("string", &set_umask, &umask) == -1);
	assert(utils_parse_umask("888", &set_umask, &umask) == -1);
	assert(utils_parse_umask("8", &set_umask, &umask) == -1);
	assert(utils_parse_umask("unset", &set_umask, &umask) == -1);

	assert(utils_get_group_gid("0", &gid) == 0);
	assert(gid == 0);

	assert(utils_get_group_gid("", &gid) == 0);
	assert(gid == -1);

	assert(utils_get_group_gid("1000", &gid) == 0);
	assert(gid == 1000);

	assert(utils_get_group_gid("-1", &gid) == 0);
	assert(gid == -1);

	assert((grp = getgrgid(0)) != NULL);
	assert((s = strdup(grp->gr_name)) != NULL);

	assert(utils_get_group_gid(s, &gid) == 0);
	assert(gid == 0);

	free(s);

	assert(utils_get_group_gid("qdevicetestnonexistinggroup", &gid) == -1);

	return (0);
}
