#!/usr/bin/python3
# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright (C) 2010-2012 Bryce Harrington <bryce@canonical.com>
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


from __future__ import absolute_import, print_function, unicode_literals

import re

class PciDevice(object):
    def __init__(self, codename, regex, drivers=None, supported=True, title=None, opengl=None, gen=None):
        # Short codename, suitable for prepending to bug reports
        self.name = codename

        # Long Marketing name
        self.title = title

        # Pattern to seek in lspci or other text output
        self.regex = regex

        # Set of open (DDX) and/or proprietary drivers that should work for this hardware
        # If none are listed, then -vesa/-fbdev should be used
        self.drivers = drivers

        # Whether Ubuntu-X provides development support for the hardware (with
        # any video driver).  If unsupported, we redirect bug reporters upstream
        # instead of filing bugs at Launchpad, and we do not backport patches or
        # customize any settings.
        self.supported = supported

        # OpenGL support provided by the hardware.
        # Some drivers might support a lower level than this.
        self.opengl = opengl

        self.generation = gen
        # TODO: Indication of unity support?

        self.__rc = None

    @property
    def rc(self):
        if self.__rc is None:
            self.__rc = re.compile(self.regex, re.IGNORECASE)
        return self.__rc

INTEL   = 'xserver-xorg-video-intel'
POULSBO = 'xserver-xorg-video-psb'
RADEON  = 'xserver-xorg-video-ati'
FGLRX   = 'fglrx-installer'
NOUVEAU = 'xserver-xorg-video-nouveau'
NVIDIA  = 'nvidia-graphics-drivers'
NV71    = 'nvidia-graphics-drivers-71'
NV96    = 'nvidia-graphics-drivers-96'
NV173   = 'nvidia-graphics-drivers-173'

# Database of chip ids, current as of Jul, 2011
pci_devices = [
    # Intel GPUs
    #
    # References:
    # http://en.wikipedia.org/wiki/Intel_GMA
    # Driver source code
    PciDevice('i740',       '(8086:7800)',            [],      gen='i740',      opengl='1.1', supported=False ),
    PciDevice('i810',       '(8086:7121)',            [],      gen='i810',      opengl='1.2', supported=False ),
    PciDevice('i810dc100',  '(8086:7123)',            [],      gen='i810',      opengl='1.2', supported=False ),
    PciDevice('i810e',      '(8086:7125)',            [],      gen='i810',      opengl='1.2', supported=False ),
    PciDevice('i815',       '(8086:1132|82815)',      [],      gen='i810',      opengl='1.2', supported=False ),
    PciDevice('i830m',      '(8086:3577|82830)',      [INTEL], gen='i830',      opengl='1.3', supported=False ),
    PciDevice('i845g',      '(8086:2562|82845G)',     [INTEL], gen='i830',      opengl='1.3', supported=False ),
    PciDevice('i854',       '(8086:358e)',            [INTEL], gen='i855',      opengl='1.3', supported=False ),
    PciDevice('i855g',      '(8086:2572)',            [INTEL], gen='i855',      opengl='1.3', supported=False ),
    PciDevice('i855g',      '(8086:3582|855GM)',      [INTEL], gen='i855',      opengl='1.3', supported=False ),
    PciDevice('i865g',      '(8086:2572|82865G)',     [INTEL], gen='i855',      opengl='1.3', supported=False ),
    PciDevice('i915g',      '(8086:2582)',            [INTEL], gen='i915',      opengl='1.4' ),
    PciDevice('i915g',      '(8086:2782)',            [INTEL], gen='i915',      opengl='1.4' ),
    PciDevice('i915gm',     '(8086:2592|915GM)',      [INTEL], gen='i915',      opengl='1.4' ),
    PciDevice('i915gm',     '(8086:2792|915GM)',      [INTEL], gen='i915',      opengl='1.4' ),
    PciDevice('e7221',      '(8086:258a)',            [INTEL], opengl=None, supported=False ),
    PciDevice('i945g',      '(8086:2772|945G[ \/])',  [INTEL], gen='i945',      opengl='1.4' ),
    PciDevice('i945g',      '(8086:2776)',            [INTEL], gen='i945',      opengl='1.4' ),
    PciDevice('i945gm',     '(8086:27a2|945GM[ \/])', [INTEL], gen='i945',      opengl='1.4' ),
    PciDevice('i945gme',    '(8086:27a6)',            [INTEL], gen='i945',      opengl='1.4' ),
    PciDevice('i945gme',    '(8086:27ae|945GME)',     [INTEL], gen='i945',      opengl='1.4' ),
    PciDevice('pineview_g', '(8086:a001)',            [INTEL], gen='pineview',  opengl='2.0' ),
    PciDevice('pineview_g', '(8086:a002)',            [INTEL], gen='pineview',  opengl='2.0' ),
    PciDevice('pineview_m', '(8086:a011)',            [INTEL], gen='pineview',  opengl='2.0' ),
    PciDevice('pineview_m', '(8086:a012)',            [INTEL], gen='pineview',  opengl='2.0' ),
    PciDevice('i946gz',     '(8086:2972|82946GZ)',    [INTEL], opengl='1.4' ),
    PciDevice('g35_g',      '(8086:2982|82G35)',      [INTEL], gen='i965',      opengl='2.0' ),
    PciDevice('g35_g',      '(8086:2983)',            [INTEL], gen='i965',      opengl='2.0' ),
    PciDevice('i965q',      '(8086:2992|Q965)',       [INTEL], gen='i965',      opengl='1.4' ),
    PciDevice('i965g',      '(8086:29a2|G965)',       [INTEL], gen='i965',      opengl='2.0' ),
    PciDevice('i965g',      '(8086:29a3|G965)',       [INTEL], gen='i965',      opengl='2.0' ),
    PciDevice('q35',        '(8086:29b2)',            [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('q35',        '(8086:29b3)',            [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('g33',        '(8086:29c2|82G33)',      [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('g33',        '(8086:29c3)',            [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('q33',        '(8086:29d2)',            [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('q33',        '(8086:29d3)',            [INTEL], gen='bearlake',  opengl='1.5' ),
    PciDevice('i965gm',     '(8086:2a02|GM965)',      [INTEL], gen='i965',      opengl='2.1' ),
    PciDevice('i965gm',     '(8086:2a03)',            [INTEL], gen='i965',      opengl='2.1' ),
    PciDevice('i965gme',    '(8086:2a12)',            [INTEL], gen='i965',      opengl='2.1' ),
    PciDevice('i965gme',    '(8086:2a13)',            [INTEL], gen='i965',      opengl='2.1' ),
    PciDevice('gm45_gm',    '(8086:2a42)',            [INTEL], gen='cantiga',   opengl='2.1' ),
    PciDevice('gm45_gm',    '(8086:2a43)',            [INTEL], gen='cantiga',   opengl='2.1' ),
    PciDevice('g45_e_g',    '(8086:2e02)',            [INTEL], opengl='2.1' ),
    PciDevice('q43',        '(8086:2e12)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('q45',        '(8086:2e13)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('g45',        '(8086:2e22)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('g45',        '(8086:2e23)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('g41',        '(8086:2e32)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('g41',        '(8086:2e33)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('b43',        '(8086:2e42)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('b43',        '(8086:2e43)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('b43g1',      '(8086:2e92)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('b43',        '(8086:2e93)',            [INTEL], gen='eaglelake', opengl='2.1' ),
    PciDevice('ironlake DG','(8086:0042)',            [INTEL], gen='clarkdale',  opengl='2.1' ),
    PciDevice('ironlake MG','(8086:0046)',            [INTEL], gen='arrandale',  opengl='2.1' ),
    PciDevice('snb-gt1',    '(8086:0102)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-m-gt1',  '(8086:0106)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-s',      '(8086:010a)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-gt2',    '(8086:0112)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-m-gt2',  '(8086:0116)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-gt2+',   '(8086:0122)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('snb-m-gt2+', '(8086:0126)',            [INTEL], gen='sandybridge', opengl='3.0' ),
    PciDevice('ivb',           '(8086:0080)',         [INTEL],   gen='ivybridge', opengl='3.1' ),
    PciDevice('ivb-m-gt1',     '(8086:0156)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('ivb-m-gt2',     '(8086:0166)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('ivb-d-gt1',     '(8086:0152)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('ivb-d-gt2',     '(8086:0162)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('ivb-s-gt1',     '(8086:015a)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('ivb-s-gt1',     '(8086:016a)',         [INTEL],   gen='ivybridge', opengl='4.0' ),
    PciDevice('hsw_d_gt1',      '(8086:0402)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_d_gt2',      '(8086:0412)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_d_gt2+',     '(8086:0422)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_m_gt1',      '(8086:0406)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_m_gt2',      '(8086:0416)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_m_gt2+',     '(8086:0426)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_s_gt1',      '(8086:040a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_s_gt2',      '(8086:041a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_s_gt2+',     '(8086:042a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_d_gt1',  '(8086:0c02)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_d_gt2',  '(8086:0c12)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_d_gt2+', '(8086:0c22)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_m_gt1',  '(8086:0c06)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_m_gt2',  '(8086:0c16)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_m_gt2+', '(8086:0c26)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_s_gt1',  '(8086:0c0a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_s_gt2',  '(8086:0c1a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_sdv_s_gt2+', '(8086:0c2a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_d_gt1',  '(8086:0a02)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_d_gt2',  '(8086:0a12)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_d_gt2+', '(8086:0a22)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_m_gt1',  '(8086:0a06)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_m_gt2',  '(8086:0a16)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_m_gt2+', '(8086:0a26)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_s_gt1',  '(8086:0a0a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_s_gt2',  '(8086:0a1a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_ult_s_gt2+', '(8086:0a2a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_d_gt1',  '(8086:0d12)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_d_gt2',  '(8086:0d22)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_d_gt2+', '(8086:0d32)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_m_gt1',  '(8086:0d16)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_m_gt2',  '(8086:0d26)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_m_gt2+', '(8086:0d36)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_s_gt1',  '(8086:0d1a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_s_gt2',  '(8086:0d2a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('hsw_crw_s_gt2+', '(8086:0d3a)',        [INTEL],   gen='haswell',   opengl='3.2' ),
    PciDevice('vlv_po',         '(8086:0f30)',        [INTEL],   gen='', opengl='' ),

    # Poulsbo
    #
    # References:
    #    http://en.wikipedia.org/wiki/Comparison_of_Intel_graphics_processing_units
    PciDevice('psb',        '(8086:8108|GMA 500)',    [POULSBO], gen='poulsbo',    opengl='2.0', supported=False ),
    PciDevice('psb',        '(8086:8109)',            [POULSBO], gen='poulsbo',    opengl='2.0', supported=False ),
    PciDevice('psb_600',    '(8086:4102|GMA 600)',    [POULSBO], gen='lincroft',   opengl='2.0', supported=False ),
    PciDevice('psb_3600',   '(8086:0BE0|GMA 3600)',   [POULSBO], gen='cedarview',  opengl='3.0', supported=False ),
    PciDevice('psb_3600',   '(8086:0BE1|GMA 3600)',   [POULSBO], gen='cedarview',  opengl='3.0', supported=False ),
    PciDevice('psb_3650',   '(8086:0BE2|GMA 3650)',   [POULSBO], gen='cedartrail', opengl='3.0', supported=False ),
    PciDevice('psb_3650',   '(8086:0BE3|GMA 3650)',   [POULSBO], gen='cedartrail', opengl='3.0', supported=False ),

    # ATI GPUs
    #
    # References:
    #    http://xorg.freedesktop.org/wiki/RadeonFeature
    #    http://en.wikipedia.org/wiki/Radeon
    #    -ati driver source src/ati_pciids_gen.h
    #    -fglrx driver source lib/modules/fglrx/build_mod/fglrxko_pci_ids.h
    PciDevice('rv380',     '(1002:3150)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3151)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3152)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3154)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3155)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3e50)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:3e54)', [RADEON], opengl='2.1' ),
    PciDevice('rs100',     '(1002:4136)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rs200',     '(1002:4137)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r300',      '(1002:4144)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4145)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4146)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4147)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4148)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4149)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:414a)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:414b)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4150)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4151)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4152)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4153)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4154)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4155)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4156)', [RADEON], opengl='2.1' ),
    PciDevice('rs200',     '(1002:4237)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r200',      '(1002:4242)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rs100',     '(1002:4336)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rs200',     '(1002:4337)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rs200',     '(1002:4437)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv250',     '(1002:4966)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv250',     '(1002:4967)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r420',      '(1002:4a48)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a49)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4a)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4b)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4c)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4d)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4e)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a4f)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a50)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4a54)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4b48)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4b49)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4b4a)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4b4b)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:4b4c)', [RADEON], opengl='2.1' ),
    PciDevice('rv200',     '(1002:4c57)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv200',     '(1002:4c58)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:4c59)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:4c5a)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv250',     '(1002:4c64)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv250',     '(1002:4c66)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv250',     '(1002:4c67)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv280',     '(1002:4c6e)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r300',      '(1002:4e44)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4e45)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4e46)', [RADEON], opengl='2.1' ),
    PciDevice('r300',      '(1002:4e47)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4e48)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4e49)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4e4a)', [RADEON], opengl='2.1' ),
    PciDevice('r350',      '(1002:4e4b)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e50)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e51)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e52)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e53)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e54)', [RADEON], opengl='2.1' ),
    PciDevice('rv350',     '(1002:4e56)', [RADEON], opengl='2.1' ),
    PciDevice('r200',      '(1002:5148)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r200',      '(1002:514c)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('r200',      '(1002:514d)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv200',     '(1002:5157)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv200',     '(1002:5158)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:5159)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:515a)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:515e)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('RV370',     '(1002:5160)', [RADEON], opengl='2.1' ),
    PciDevice('RV370',     '(1002:5162)', [RADEON], opengl='2.1' ),
    PciDevice('RV370',     '(1002:5164)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5460)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5462)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5464)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5548)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5549)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554a)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554b)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554c)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554d)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554e)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:554f)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5550)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5551)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5552)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5554)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:564a)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:564b)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:564f)', [RADEON], opengl='3.3' ),
    PciDevice('rv410',     '(1002:5652)', [RADEON], opengl='3.3' ),
    PciDevice('rv410',     '(1002:5653)', [RADEON], opengl='3.3' ),
    PciDevice('rv410',     '(1002:5657)', [RADEON], opengl='3.3' ),
    PciDevice('rs300',     '(1002:5834)', [RADEON], opengl='3.3' ),
    PciDevice('rs300',     '(1002:5835)', [RADEON], opengl='3.3' ),
    PciDevice('rs480',     '(1002:5954)', [RADEON], opengl='3.3' ),
    PciDevice('rs480',     '(1002:5955)', [RADEON], opengl='3.3' ),
    PciDevice('rv280',     '(1002:5960)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv280',     '(1002:5961)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv280',     '(1002:5962)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv280',     '(1002:5964)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv280',     '(1002:5965)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rv100',     '(1002:5969)', [RADEON], opengl='1.3', supported=False ),
    PciDevice('rs480',     '(1002:5974)', [RADEON], opengl='2.1' ),
    PciDevice('rs480',     '(1002:5975)', [RADEON], opengl='2.1' ),
    PciDevice('rs400',     '(1002:5a41)', [RADEON], opengl='2.1' ),
    PciDevice('rs400',     '(1002:5a42)', [RADEON], opengl='2.1' ),
    PciDevice('rs400',     '(1002:5a61)', [RADEON], opengl='2.1' ),
    PciDevice('rs400',     '(1002:5a62)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5b60)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5b62)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5b63)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5b64)', [RADEON], opengl='2.1' ),
    PciDevice('rv380',     '(1002:5b65)', [RADEON], opengl='2.1' ),
    PciDevice('rv280',     '(1002:5c61)', [RADEON], opengl='2.1' ),
    PciDevice('rv280',     '(1002:5c63)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d48)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d49)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d4a)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d4c)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d4d)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d4e)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d4f)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d50)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d52)', [RADEON], opengl='2.1' ),
    PciDevice('r420',      '(1002:5d57)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e48)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e4a)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e4b)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e4c)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e4d)', [RADEON], opengl='2.1' ),
    PciDevice('rv410',     '(1002:5e4f)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7100)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7101)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7102)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7103)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7104)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7105)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7106)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7108)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:7109)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:710a)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:710b)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:710c)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:710e)', [RADEON], opengl='2.1' ),
    PciDevice('r520',      '(1002:710f)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7140)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7141)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7142)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7143)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7144)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7145)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7146)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7147)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7149)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714a)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714b)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714c)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714d)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714e)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:714f)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7151)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7152)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7153)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:715e)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:715f)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7180)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7181)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7183)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7186)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7187)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7188)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:718a)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:718b)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:718c)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:718d)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:718f)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7193)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7196)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:719b)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:719f)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c0)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c1)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c2)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c3)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c4)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c5)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c6)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71c7)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71cd)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71ce)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71d2)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71d4)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71d5)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71d6)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71da)', [RADEON], opengl='2.1' ),
    PciDevice('rv530',     '(1002:71de)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7200)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7210)', [RADEON], opengl='2.1' ),
    PciDevice('rv515',     '(1002:7211)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7240)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7243)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7244)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7245)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7246)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7247)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7248)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7249)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724a)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724b)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724c)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724d)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724e)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:724f)', [RADEON], opengl='2.1' ),
    PciDevice('rv570',     '(1002:7280)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7281)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7283)', [RADEON], opengl='2.1' ),
    PciDevice('r580',      '(1002:7284)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7287)', [RADEON], opengl='2.1' ),
    PciDevice('rv570',     '(1002:7288)', [RADEON], opengl='2.1' ),
    PciDevice('rv570',     '(1002:7289)', [RADEON], opengl='2.1' ),
    PciDevice('rv570',     '(1002:728b)', [RADEON], opengl='2.1' ),
    PciDevice('rv570',     '(1002:728c)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7290)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7291)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7293)', [RADEON], opengl='2.1' ),
    PciDevice('rv560',     '(1002:7297)', [RADEON], opengl='2.1' ),
    PciDevice('rs300',     '(1002:7834)', [RADEON], opengl='2.1' ),
    PciDevice('rs300',     '(1002:7835)', [RADEON], opengl='2.1' ),
    PciDevice('rs690',     '(1002:791e)', [RADEON], opengl='2.1' ),
    PciDevice('rs690',     '(1002:791f)', [RADEON], opengl='2.1' ),
    PciDevice('rs600',     '(1002:793f)', [RADEON], opengl='3.3' ),
    PciDevice('rs600',     '(1002:7941)', [RADEON], opengl='3.3' ),
    PciDevice('rs600',     '(1002:7942)', [RADEON], opengl='3.3' ),
    PciDevice('rs740',     '(1002:796c)', [RADEON], opengl='3.3' ),
    PciDevice('rs740',     '(1002:796d)', [RADEON], opengl='3.3' ),
    PciDevice('rs740',     '(1002:796e)', [RADEON], opengl='3.3' ),
    PciDevice('rs740',     '(1002:796f)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:9400)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:9401)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:9402)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:9403)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:9405)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:940a)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:940b)', [RADEON], opengl='3.3' ),
    PciDevice('r600',      '(1002:940f)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9440)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9441)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9442)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9443)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9444)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9446)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:944a)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:944b)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:944c)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:944e)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9450)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9452)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9456)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:945a)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:945b)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:945e)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9460)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:9462)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:946a)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:946b)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:947a)', [RADEON], opengl='3.3' ),
    PciDevice('rv770',     '(1002:947b)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9480)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9487)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9488)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9489)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:948a)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:948f)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9490)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9491)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9495)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:9498)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:949c)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:949e)', [RADEON], opengl='3.3' ),
    PciDevice('rv730',     '(1002:949f)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94a0)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94a1)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94a3)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94b1)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94b3)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94b4)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94b5)', [RADEON], opengl='3.3' ),
    PciDevice('rv740',     '(1002:94b9)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c0)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c1)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c3)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c4)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c5)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c6)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c7)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c8)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94c9)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94cb)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94cc)', [RADEON], opengl='3.3' ),
    PciDevice('rv610',     '(1002:94cd)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9500)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9501)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9504)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9505)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9506)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9507)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9508)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9509)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:950f)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9511)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9515)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9517)', [RADEON], opengl='3.3' ),
    PciDevice('rv670',     '(1002:9519)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9540)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9541)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9542)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:954e)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:954f)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9552)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9553)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9555)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:9557)', [RADEON], opengl='3.3' ),
    PciDevice('rv710',     '(1002:955f)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9580)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9581)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9583)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9586)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9587)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9588)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:9589)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958a)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958b)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958c)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958d)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958e)', [RADEON], opengl='3.3' ),
    PciDevice('rv630',     '(1002:958f)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c0)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c2)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c4)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c5)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c6)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c7)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95c9)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95cc)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95cd)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95ce)', [RADEON], opengl='3.3' ),
    PciDevice('rv620',     '(1002:95cf)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9590)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9596)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9597)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9598)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9599)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9591)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9593)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:9595)', [RADEON], opengl='3.3' ),
    PciDevice('rv635',     '(1002:959b)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9610)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9611)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9612)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9613)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9614)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9615)', [RADEON], opengl='3.3' ),
    PciDevice('rs780',     '(1002:9616)', [RADEON], opengl='3.3' ),
    PciDevice('sumo',      '(1002:9640)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:9641)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo2',     '(1002:9642)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo2',     '(1002:9643)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo2',     '(1002:9644)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo2',     '(1002:9645)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:9647)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:9648)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:9649)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:964a)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:964b)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:964c)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:964e)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('sumo',      '(1002:964f)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9710)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9711)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9712)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9713)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9714)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('rs880',     '(1002:9715)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9802)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9803)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9804)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9805)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9806)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9807)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9808)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:9809)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('palm',      '(1002:980a)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:6880)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:6888)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:6889)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:688a)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:688c)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:688d)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:6898)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:6899)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:689b)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cypress',   '(1002:689e)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('hemlock',   '(1002:689c)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('hemlock',   '(1002:689d)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68a0)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68a1)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68a8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68a9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68b0)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68b8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68b9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68ba)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68be)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('juniper',   '(1002:68bf)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68c0)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68c1)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68c7)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68c8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68c9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68d8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68d9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68da)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('redwood',   '(1002:68de)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e0)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e1)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e4)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e5)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68e9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68f1)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68f2)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68f8)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68f9)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68fa)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cedar',     '(1002:68fe)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6700)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6701)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6702)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6703)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6704)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6705)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6706)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6707)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6708)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6709)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6718)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:6719)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:671c)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:671d)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('cayman',    '(1002:671f)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6720)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6721)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6722)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6723)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6724)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6725)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6726)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6727)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6728)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6729)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6738)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:6739)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('barts',     '(1002:673e)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6740)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6741)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6742)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6743)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6744)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6745)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6746)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6747)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6748)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6749)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:674a)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6750)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6751)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6758)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6759)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:675b)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:675d)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:675f)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6840)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6841)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6842)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6843)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6849)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6850)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6858)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('turks',     '(1002:6859)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6760)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6761)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6762)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6763)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6764)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6765)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6766)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6767)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6768)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6770)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6771)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6772)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6778)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:6779)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('caicos',    '(1002:677b)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9900)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9901)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9903)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9904)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9905)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9906)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9907)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9908)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9909)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:990a)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:990f)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9910)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9913)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9917)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9918)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9919)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9990)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9991)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9992)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9993)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:9994)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:99a0)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:99a2)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('aruba',     '(1002:99a4)', [RADEON,FGLRX], opengl='4.1' ),
    PciDevice('tahiti',    '(1002:6780)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:6784)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:6788)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:678a)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:6790)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:6798)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:6799)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:679a)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:679e)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('tahiti',    '(1002:679f)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6800)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6801)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6802)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6806)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6808)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6809)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6810)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6816)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6817)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6818)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:6819)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('pitcairn',  '(1002:684c)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6820)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6821)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6823)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6824)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6825)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6826)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6827)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6828)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6829)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:682b)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:682d)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:682f)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6830)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6831)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6837)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6838)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:6839)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:683b)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:683d)', [RADEON,FGLRX], opengl='4.2' ),
    PciDevice('verde',     '(1002:683f)', [RADEON,FGLRX], opengl='4.2' ),

    # Nvidia
    #
    # References:
    #     http://en.wikipedia.org/wiki/Comparison_of_Nvidia_graphics_processing_units
    #     http://pci-ids.ucw.cz/read/PC/10de
    #     http://nouveau.freedesktop.org/wiki/CodeNames
    #     nvidia-current source: README.txt
    #
    # TODO:
    #   - NVC* not indicated
    #   - To update the list, chet source nvidia-current, then debuild
    PciDevice('NV40',                 '(10de:0040)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 Ultra',),
    PciDevice('NV40',                 '(10de:0041)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800',),
    PciDevice('NV40.2',               '(10de:0042)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 LE',),
    PciDevice('NV40.3',               '(10de:0043)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 XE',),
    PciDevice('NV40',                 '(10de:0044)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 XT',),
    PciDevice('NV40',                 '(10de:0045)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 GT',),
    PciDevice('NV45',                 '(10de:0046)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 GT',),
    PciDevice('NV40',                 '(10de:0047)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 GS',),
    PciDevice('NV40',                 '(10de:0048)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 XT',),
    PciDevice('G70',                  '(10de:0090)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 GTX',),
    PciDevice('G70',                  '(10de:0091)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 GTX',),
    PciDevice('G70',                  '(10de:0092)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 GT',),
    PciDevice('G70',                  '(10de:0093)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 GS',),
    PciDevice('G70',                  '(10de:0095)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 SLI',),
    PciDevice('G70',                  '(10de:0098)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7800',),
    PciDevice('G70',                  '(10de:0099)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7800 GTX',),
    PciDevice('NV41',                 '(10de:00c0)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 GS',),
    PciDevice('NV41.1',               '(10de:00c1)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800',),
    PciDevice('NV41.2',               '(10de:00c2)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 LE',),
    PciDevice('NV42',                 '(10de:00c3)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 XT',),
    PciDevice('NV41.8',               '(10de:00c8)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6800',),
    PciDevice('NV41.9',               '(10de:00c9)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6800 Ultra',),
    PciDevice('NV43',                 '(10de:00f1)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600 GT',),
    PciDevice('NV43',                 '(10de:00f2)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600',),
    PciDevice('NV43',                 '(10de:00f3)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200',),
    PciDevice('NV43',                 '(10de:00f4)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600 LE',),
    PciDevice('G70',                  '(10de:00f5)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7800 GS',),
    PciDevice('NV43',                 '(10de:00f6)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 GS',),
    PciDevice('NV45',                 '(10de:00f9)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6800 Ultra',),
    PciDevice('NV43',                 '(10de:0140)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600 GT',),
    PciDevice('NV43',                 '(10de:0141)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600',),
    PciDevice('NV43',                 '(10de:0142)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600 LE',),
    PciDevice('NV43',                 '(10de:0143)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6600 VE',),
    PciDevice('NV43',                 '(10de:0144)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6600',),
    PciDevice('NV43',                 '(10de:0145)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6610 XL',),
    PciDevice('NV43',                 '(10de:0146)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6600 TE/6200 TE',),
    PciDevice('NV43',                 '(10de:0147)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6700 XL',),
    PciDevice('NV43',                 '(10de:0148)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6600',),
    PciDevice('NV43',                 '(10de:0149)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6600 GT',),
    PciDevice('NV43',                 '(10de:014f)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200',),
    PciDevice('NV44',                 '(10de:0160)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6500',),
    PciDevice('NV44',                 '(10de:0161)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200 TurboCache(TM)',),
    PciDevice('NV44',                 '(10de:0162)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200SE TurboCache(TM)',),
    PciDevice('NV44',                 '(10de:0163)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200 LE',),
    PciDevice('NV44',                 '(10de:0164)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6200',),
    PciDevice('NV44',                 '(10de:0166)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6400',),
    PciDevice('NV44',                 '(10de:0167)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6200',),
    PciDevice('NV44',                 '(10de:0168)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6400',),
    PciDevice('NV44',                 '(10de:0169)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6250',),
    PciDevice('NV44',                 '(10de:016a)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7100 GS',),
    PciDevice('G80',                  '(10de:0191)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GTX',),
    PciDevice('G80',                  '(10de:0193)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GTS',),
    PciDevice('G80',                  '(10de:0194)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 Ultra',),
    PciDevice('G80',                  '(10de:0197)', [NOUVEAU,NVIDIA], opengl='3.3', title='Tesla C870',),
    PciDevice('G72',                  '(10de:01d0)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7350 LE',),
    PciDevice('G72',                  '(10de:01d1)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 LE',),
    PciDevice('G72',                  '(10de:01d2)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7550 LE',),
    PciDevice('G72',                  '(10de:01d3)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 SE/7200 GS',),
    PciDevice('G72M',                 '(10de:01d6)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7200',),
    PciDevice('G72M',                 '(10de:01d7)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7300',),
    PciDevice('G72M',                 '(10de:01d8)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7400',),
    PciDevice('G72',                  '(10de:01dd)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7500 LE',),
    PciDevice('G71',                  '(10de:01df)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 GS',),
    PciDevice('NV44A',                '(10de:0221)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200',),
    PciDevice('NV44',                 '(10de:0222)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6200 A-LE',),
    PciDevice('C51PV',                '(10de:0240)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6150',),
    PciDevice('C51',                  '(10de:0241)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6150 LE',),
    PciDevice('C51G',                 '(10de:0242)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6100',),
    PciDevice('C51',                  '(10de:0244)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6150',),
    PciDevice('C51',                  '(10de:0247)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 6100',),
    PciDevice('G71',                  '(10de:0290)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7900 GTX',),
    PciDevice('G71',                  '(10de:0291)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7900 GT/GTO',),
    PciDevice('G71',                  '(10de:0292)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7900 GS',),
    PciDevice('G71',                  '(10de:0293)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7950 GX2',),
    PciDevice('G71',                  '(10de:0294)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7950 GX2',),
    PciDevice('G71',                  '(10de:0295)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7950 GT',),
    PciDevice('G71',                  '(10de:0297)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7950 GTX',),
    PciDevice('G71',                  '(10de:0298)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7900 GS',),
    PciDevice('G73',                  '(10de:02e0)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7600 GT',),
    PciDevice('G73',                  '(10de:02e1)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7600 GS',),
    PciDevice('G73',                  '(10de:02e2)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 GT',),
    PciDevice('G71',                  '(10de:02e3)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7900 GS',),
    PciDevice('G71',                  '(10de:02e4)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7950 GT',),
    PciDevice('G73',                  '(10de:038b)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7650 GS',),
    PciDevice('G73',                  '(10de:0390)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7650 GS',),
    PciDevice('G73',                  '(10de:0391)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7600 GT',),
    PciDevice('G73',                  '(10de:0392)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7600 GS',),
    PciDevice('G73',                  '(10de:0393)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 GT',),
    PciDevice('G73',                  '(10de:0394)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7600 LE',),
    PciDevice('G73',                  '(10de:0395)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7300 GT',),
    PciDevice('G73',                  '(10de:0397)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7700',),
    PciDevice('G73',                  '(10de:0398)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7600',),
    PciDevice('G73',                  '(10de:0399)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce Go 7600 GT',),
    PciDevice('C61',                  '(10de:03d0)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6150SE nForce 430',),
    PciDevice('C61',                  '(10de:03d1)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6100 nForce 405',),
    PciDevice('C61',                  '(10de:03d2)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6100 nForce 400',),
    PciDevice('C61',                  '(10de:03d5)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 6100 nForce 420',),
    PciDevice('C61',                  '(10de:03d6)', [NOUVEAU,NVIDIA], opengl='2.1', title='GeForce 7025 / nForce 630a',),
    PciDevice('G84',                  '(10de:0400)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600 GTS',),
    PciDevice('G84',                  '(10de:0401)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600 GT',),
    PciDevice('G84',                  '(10de:0402)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600 GT',),
    PciDevice('G84',                  '(10de:0403)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600 GS',),
    PciDevice('G84',                  '(10de:0404)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 GS',),
    PciDevice('G84',                  '(10de:0405)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500M GS',),
    PciDevice('G84',                  '(10de:0406)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8300 GS',),
    PciDevice('G84',                  '(10de:0407)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600M GT',),
    PciDevice('G84',                  '(10de:0408)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9650M GS',),
    PciDevice('G84',                  '(10de:0409)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8700M GT',),
    PciDevice('G92',                  '(10de:0410)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 330',),
    PciDevice('G86',                  '(10de:0420)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 SE',),
    PciDevice('G86',                  '(10de:0421)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8500 GT',),
    PciDevice('G86',                  '(10de:0422)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 GS',),
    PciDevice('G86',                  '(10de:0423)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8300 GS',),
    PciDevice('G86',                  '(10de:0424)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 GS',),
    PciDevice('G86',                  '(10de:0425)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8600M GS',),
    PciDevice('G86',                  '(10de:0426)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400M GT',),
    PciDevice('G86',                  '(10de:0427)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400M GS',),
    PciDevice('G86',                  '(10de:0428)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400M G',),
    PciDevice('G86',                  '(10de:042c)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400 GT',),
    PciDevice('G86',                  '(10de:042e)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300M G',),
    PciDevice('C67',                  '(10de:0531)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7150M / nForce 630M',),
    PciDevice('C67',                  '(10de:0533)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7000M / nForce 610M',),
    PciDevice('C68',                  '(10de:053a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7050 PV / nForce 630a',),
    PciDevice('C68',                  '(10de:053b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7050 PV / nForce 630a',),
    PciDevice('C68',                  '(10de:053e)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7025 / nForce 630a',),
    PciDevice('GT200b',               '(10de:05e0)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 295',),
    PciDevice('GT200',                '(10de:05e1)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 280',),
    PciDevice('GT200',                '(10de:05e2)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 260',),
    PciDevice('GT200b',               '(10de:05e3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 285',),
    PciDevice('GT200b',               '(10de:05e6)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 275',),
    PciDevice('GT200',                '(10de:05e7)', [NOUVEAU,NVIDIA], opengl='3.3', title='Tesla C1060',),
    PciDevice('Tesla',                '(10de:0595)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla T10 Processor',),
    PciDevice('Tesla',                '(10de:068f)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla T10 Processor',),
    PciDevice('Tesla',                '(10de:0697)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla M1060',),
    PciDevice('Tesla',                '(10de:0743)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla M1060',),
    PciDevice('GT200',                '(10de:05ea)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 260',),
    PciDevice('GT200',                '(10de:05eb)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 295',),
    PciDevice('G92',                  '(10de:0600)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GTS 512',),
    PciDevice('G92',                  '(10de:0601)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GT',),
    PciDevice('G92',                  '(10de:0602)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GT',),
    PciDevice('G92',                  '(10de:0603)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 230',),
    PciDevice('G92',                  '(10de:0604)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GX2',),
    PciDevice('G92',                  '(10de:0605)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GT',),
    PciDevice('G92',                  '(10de:0606)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GS',),
    PciDevice('G92',                  '(10de:0607)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 240',),
    PciDevice('G92',                  '(10de:0608)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GTX',),
    PciDevice('G92',                  '(10de:0609)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800M GTS',),
    PciDevice('GT200',                '(10de:060a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 280M',),
    PciDevice('G92',                  '(10de:060b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GT',),
    PciDevice('G92',                  '(10de:060c)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800M GTX',),
    PciDevice('G92',                  '(10de:060d)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GS',),
    PciDevice('G92',                  '(10de:060f)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 285M',),
    PciDevice('G92',                  '(10de:0610)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GSO',),
    PciDevice('G92',                  '(10de:0611)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8800 GT',),
    PciDevice('G92',                  '(10de:0612)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GTX/9800 GTX+',),
    PciDevice('G92',                  '(10de:0613)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GTX+',),
    PciDevice('G92',                  '(10de:0614)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800 GT',),
    PciDevice('G92',                  '(10de:0615)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 250',),
    PciDevice('G92',                  '(10de:0617)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GTX',),
    PciDevice('G92',                  '(10de:0618)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTX 260M',),
    PciDevice('G94b',                 '(10de:0621)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 230',),
    PciDevice('G94',                  '(10de:0622)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GT',),
    PciDevice('G94',                  '(10de:0623)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GS',),
    PciDevice('G94',                  '(10de:0625)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GSO 512',),
    PciDevice('G94',                  '(10de:0626)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 130',),
    PciDevice('G94',                  '(10de:0627)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 140',),
    PciDevice('G94',                  '(10de:0628)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GTS',),
    PciDevice('G94',                  '(10de:062a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9700M GTS',),
    PciDevice('G94',                  '(10de:062b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GS',),
    PciDevice('G94',                  '(10de:062c)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9800M GTS',),
    PciDevice('G94',                  '(10de:062d)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GT',),
    PciDevice('G94',                  '(10de:062e)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GT',),
    PciDevice('',                     '(10de:0630)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 9700 S',),
    PciDevice('G94M',                 '(10de:0631)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 160M',),
    PciDevice('G94M',                 '(10de:0632)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 150M',),
    PciDevice('G94',                  '(10de:0635)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GSO',),
    PciDevice('G94',                  '(10de:0637)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600 GT',),
    PciDevice('G96',                  '(10de:0640)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500 GT',),
    PciDevice('G96',                  '(10de:0641)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400 GT',),
    PciDevice('G96',                  '(10de:0643)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500 GT',),
    PciDevice('G96',                  '(10de:0644)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500 GS',),
    PciDevice('G96',                  '(10de:0645)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500 GS',),
    PciDevice('G96',                  '(10de:0646)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 120',),
    PciDevice('G96',                  '(10de:0647)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600M GT',),
    PciDevice('G96',                  '(10de:0648)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600M GS',),
    PciDevice('G96',                  '(10de:0649)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9600M GT',),
    PciDevice('G96',                  '(10de:064a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9700M GT',),
    PciDevice('G96',                  '(10de:064b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9500M G',),
    PciDevice('G96',                  '(10de:064c)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9650M GT',),
    PciDevice('G96',                  '(10de:0651)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G 110M',),
    PciDevice('G96',                  '(10de:0652)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 130M',),
    PciDevice('G96M',                 '(10de:0653)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 120M',),
    PciDevice('G96',                  '(10de:0654)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 220M',),
    PciDevice('G96b',                 '(10de:0633)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 120',),
    PciDevice('G96',                  '(10de:0656)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9650 S',),
    PciDevice('G96',                  '(10de:065b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400 GT',),
    PciDevice('GF100',                '(10de:06c0)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 480',),
    PciDevice('GF100',                '(10de:06c4)', [NOUVEAU,NVIDIA], opengl='4.3', title='GeForce GTX 465',),
    PciDevice('GF100',                '(10de:06ca)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 480M',),
    PciDevice('GF100',                '(10de:06cd)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 470',),
    PciDevice('GF100',                '(10de:06d1)', [NOUVEAU,NVIDIA], opengl='4.1', title='Tesla C2050 / C2070',),
    PciDevice('',                     '(10de:0772)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla C2070                           0x06D1',),
    PciDevice('GF100',                '(10de:06d2)', [NOUVEAU,NVIDIA], opengl='4.1', title='Tesla M2070',),
    PciDevice('GF100',                '(10de:06de)', [NOUVEAU,NVIDIA], opengl='4.1', title='Tesla T20 Processor',),
    PciDevice('',                     '(10de:082f)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla M2050                           0x06DE',),
    PciDevice('C77',                  '(10de:0846)', [NOUVEAU,NVIDIA], opengl='3.3', title='Tesla M2050                           0x06DE',),
    PciDevice('GF100',                '(10de:06df)', [NOUVEAU,NVIDIA], opengl='4.1', title='Tesla M2070-Q',),
    PciDevice('G98',                  '(10de:06e0)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300 GE',),
    PciDevice('G98',                  '(10de:06e1)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300 GS',),
    PciDevice('G98',                  '(10de:06e2)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400',),
    PciDevice('G98',                  '(10de:06e3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 SE',),
    PciDevice('G98',                  '(10de:06e4)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400 GS',),
    PciDevice('G98',                  '(10de:06e5)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300M GS',),
    PciDevice('G98',                  '(10de:06e6)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G100',),
    PciDevice('G98',                  '(10de:06e7)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300 SE',),
    PciDevice('G98',                  '(10de:06e8)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9200M GS',),
    PciDevice('G98',                  '(10de:06e9)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300M GS',),
    PciDevice('G98M',                 '(10de:06ec)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G 105M',),
    PciDevice('G98M',                 '(10de:06ef)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G 103M',),
    PciDevice('G98',                  '(10de:06f1)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G105M',),
    PciDevice('C73',                  '(10de:07e0)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7150 / nForce 630i',),
    PciDevice('C73',                  '(10de:07e1)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7100 / nForce 630i',),
    PciDevice('C73',                  '(10de:07e2)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7050 / nForce 630i',),
    PciDevice('C73',                  '(10de:07e3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7050 / nForce 610i',),
    PciDevice('C73',                  '(10de:07e5)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 7050 / nForce 620i',),
    PciDevice('',                     '(10de:0840)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 8200M',),
    PciDevice('C77',                  '(10de:0844)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9100M G',),
    PciDevice('C77',                  '(10de:0845)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8200M G',),
    PciDevice('C77',                  '(10de:0846)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9200',),
    PciDevice('C78',                  '(10de:0847)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9100',),
    PciDevice('C77',                  '(10de:0848)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8300',),
    PciDevice('C77',                  '(10de:0849)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8200',),
    PciDevice('C77',                  '(10de:084a)', [NOUVEAU,NVIDIA], opengl='3.3', title='nForce 730a',),
    PciDevice('C77',                  '(10de:084b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9200',),
    PciDevice('C77',                  '(10de:084c)', [NOUVEAU,NVIDIA], opengl='3.3', title='nForce 980a/780a SLI',),
    PciDevice('C77',                  '(10de:084d)', [NOUVEAU,NVIDIA], opengl='3.3', title='nForce 750a SLI',),
    PciDevice('C77',                  '(10de:084f)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8100 / nForce 720a',),
    PciDevice('C79',                  '(10de:0860)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400',),
    PciDevice('C79',                  '(10de:0861)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400',),
    PciDevice('C79',                  '(10de:0862)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400M G',),
    PciDevice('C79',                  '(10de:0863)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400M',),
    PciDevice('C79',                  '(10de:0864)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300',),
    PciDevice('C79',                  '(10de:0865)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION',),
    PciDevice('C79',                  '(10de:0866)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400M G',),
    PciDevice('C79',                  '(10de:0867)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400',),
    PciDevice('C79',                  '(10de:0868)', [NOUVEAU,NVIDIA], opengl='3.3', title='nForce 760i SLI',),
    PciDevice('',                     '(10de:0869)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 9400',),
    PciDevice('C79',                  '(10de:086a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400',),
    PciDevice('C79',                  '(10de:086c)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9300 / nForce 730i',),
    PciDevice('C79',                  '(10de:086d)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9200',),
    PciDevice('C79',                  '(10de:086e)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9100M G',),
    PciDevice('C79',                  '(10de:086f)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8200M G',),
    PciDevice('C79',                  '(10de:0870)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400M',),
    PciDevice('C79',                  '(10de:0871)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9200',),
    PciDevice('C79',                  '(10de:0872)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G102M',),
    PciDevice('C79',                  '(10de:0873)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G102M',),
    PciDevice('C79',                  '(10de:0874)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION',),
    PciDevice('ION VGA',              '(10de:0876)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION',),
    PciDevice('C79',                  '(10de:087a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 9400',),
    PciDevice('ION VGA',              '(10de:087d)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION',),
    PciDevice('ION LE VGA',           '(10de:087e)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION LE',),
    PciDevice('ION LE VGA',           '(10de:087f)', [NOUVEAU,NVIDIA], opengl='3.3', title='ION LE',),
    PciDevice('',                     '(10de:08a0)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 320M',),
    PciDevice('GT216',                '(10de:08a2)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 320M',),
    PciDevice('GT216',                '(10de:08a3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 320M',),
    PciDevice('GT216',                '(10de:08a4)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 320M',),
    PciDevice('GT216',                '(10de:08a5)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 320M',),
    PciDevice('GT216',                '(10de:0a20)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 220',),
    PciDevice('',                     '(10de:0a22)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 315',),
    PciDevice('GT218',                '(10de:0a23)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce 210',),
    PciDevice('',                     '(10de:0a26)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 405',),
    PciDevice('',                     '(10de:0a27)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 405',),
    PciDevice('GT216',                '(10de:0a28)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 230M',),
    PciDevice('GT216',                '(10de:0a29)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 330M',),
    PciDevice('GT216',                '(10de:0a2a)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 230M',),
    PciDevice('GT216',                '(10de:0a2b)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 330M',),
    PciDevice('GT216',                '(10de:0a2d)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 320M',),
    PciDevice('',                     '(10de:0a32)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 415',),
    PciDevice('GT216',                '(10de:0a34)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 240M',),
    PciDevice('GT216',                '(10de:0a35)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 325M',),
    PciDevice('GT218',                '(10de:0a60)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce G210',),
    PciDevice('GT218',                '(10de:0a62)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 205',),
    PciDevice('GT218',                '(10de:0a63)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 310',),
    PciDevice('GT218',                '(10de:0a64)', [NOUVEAU,NVIDIA], opengl='4.1', title='Second Generation ION',),
    PciDevice('GT218',                '(10de:0a65)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce 210',),
    PciDevice('GT218',                '(10de:0a66)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 310',),
    PciDevice('',                     '(10de:0a67)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 315',),
    PciDevice('G98M',                 '(10de:0a68)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G105M',),
    PciDevice('G98M',                 '(10de:0a69)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G105M',),
    PciDevice('GT218',                '(10de:0a6e)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 305M',),
    PciDevice('GT218',                '(10de:0a6f)', [NOUVEAU,NVIDIA], opengl='4.1', title='Second Generation ION',),
    PciDevice('GT218',                '(10de:0a70)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 310M',),
    PciDevice('GT218',                '(10de:0a71)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 305M',),
    PciDevice('GT218',                '(10de:0a72)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 310M',),
    PciDevice('GT218',                '(10de:0a73)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 305M',),
    PciDevice('GT218',                '(10de:0a74)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce G210M',),
    PciDevice('GT218',                '(10de:0a75)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 310M',),
    PciDevice('',                     '(10de:0a76)', [NOUVEAU,NVIDIA], opengl='None', title='Second Generation ION',),
    PciDevice('',                     '(10de:0a7a)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 315M',),
    PciDevice('GT215',                '(10de:0ca0)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 330',),
    PciDevice('GT215',                '(10de:0ca2)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 320',),
    PciDevice('GT215',                '(10de:0ca3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 240',),
    PciDevice('GT215',                '(10de:0ca4)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 340',),
    PciDevice('',                     '(10de:0ca5)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 220',),
    PciDevice('GT215',                '(10de:0ca7)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 330',),
    PciDevice('GT215',                '(10de:0ca8)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 260M',),
    PciDevice('GT215',                '(10de:0ca9)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 250M',),
    PciDevice('',                     '(10de:0cac)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 220',),
    PciDevice('GT215',                '(10de:0caf)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GT 335M',),
    PciDevice('GT215',                '(10de:0cb0)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 350M',),
    PciDevice('GT215',                '(10de:0cb1)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce GTS 360M',),
    PciDevice('',                     '(10de:0dc0)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 440',),
    PciDevice('GF106',                '(10de:0dc4)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTS 450',),
    PciDevice('',                     '(10de:0dc5)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTS 450',),
    PciDevice('',                     '(10de:0dc6)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTS 450',),
    PciDevice('GF106',                '(10de:0dcd)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('GF106',                '(10de:0dce)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('GF106',                '(10de:0dd1)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 460M',),
    PciDevice('GF106',                '(10de:0dd2)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GT 445M',),
    PciDevice('',                     '(10de:0dd3)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 435M',),
    PciDevice('',                     '(10de:0dd6)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 550M',),
    PciDevice('GF108',                '(10de:0de0)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 440',),
    PciDevice('GF108',                '(10de:0de1)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GT 430',),
    PciDevice('GF108',                '(10de:0de2)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 420',),
    PciDevice('',                     '(10de:0de4)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 520',),
    PciDevice('',                     '(10de:0de5)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 530',),
    PciDevice('',                     '(10de:0de8)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 620M',),
    PciDevice('',                     '(10de:0de9)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 630M',),
    PciDevice('',                     '(10de:0dea)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 610M',),
    PciDevice('GF108',                '(10de:0deb)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('',                     '(10de:0dec)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 525M',),
    PciDevice('',                     '(10de:0ded)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 520M',),
    PciDevice('GF108',                '(10de:0dee)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GT 415M',),
    PciDevice('GF108',                '(10de:0df0)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 425M',),
    PciDevice('GF108',                '(10de:0df1)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 420M',),
    PciDevice('GF108',                '(10de:0df2)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 435M',),
    PciDevice('GF108',                '(10de:0df3)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 420M',),
    PciDevice('GF106',                '(10de:0df4)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GT 540M',),
    PciDevice('GF106',                '(10de:0df5)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 525M',),
    PciDevice('GF108',                '(10de:0df6)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 550M',),
    PciDevice('GF106',                '(10de:0df7)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 520M',),
    PciDevice('GF104',                '(10de:0e22)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 460',),
    PciDevice('',                     '(10de:0e23)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 460 SE',),
    PciDevice('',                     '(10de:0e24)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 460',),
    PciDevice('GF104',                '(10de:0e30)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 470M',),
    PciDevice('GF104',                '(10de:0e31)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 485M',),
    PciDevice('',                     '(10de:0f00)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 630',),
    PciDevice('',                     '(10de:0f01)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 620',),
    PciDevice('',                     '(10de:0fc0)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640',),
    PciDevice('',                     '(10de:0fc1)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640',),
    PciDevice('',                     '(10de:0fc2)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 630',),
    PciDevice('',                     '(10de:0fce)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640M LE',),
    PciDevice('',                     '(10de:0fd1)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 650M',),
    PciDevice('GK107',                '(10de:0fd2)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640M',),
    PciDevice('',                     '(10de:0fd3)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640M LE',),
    PciDevice('',                     '(10de:0fd4)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 660M',),
    PciDevice('',                     '(10de:0fd5)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 650M',),
    PciDevice('',                     '(10de:0fd8)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 640M',),
    PciDevice('',                     '(10de:0fe0)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 660M',),
    PciDevice('GF119',                '(10de:1040)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 520',),
    PciDevice('',                     '(10de:1042)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 510',),
    PciDevice('',                     '(10de:1048)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 605',),
    PciDevice('',                     '(10de:1049)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 620',),
    PciDevice('',                     '(10de:104a)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 610',),
    PciDevice('GF119',                '(10de:1050)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 520M',),
    PciDevice('GF119',                '(10de:1051)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 520MX',),
    PciDevice('',                     '(10de:1052)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 520M',),
    PciDevice('',                     '(10de:1054)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 410M',),
    PciDevice('',                     '(10de:1055)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 410M',),
    PciDevice('',                     '(10de:1058)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 610M',),
    PciDevice('',                     '(10de:1059)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 610M',),
    PciDevice('GF119',                '(10de:105a)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce 610M',),
    PciDevice('GF110',                '(10de:1080)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 580',),
    PciDevice('GF110',                '(10de:1081)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 570',),
    PciDevice('GF110',                '(10de:1082)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 560 Ti',),
    PciDevice('',                     '(10de:1084)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 560',),
    PciDevice('GF110',                '(10de:1086)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 570',),
    PciDevice('GF110',                '(10de:1087)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 560 Ti',),
    PciDevice('',                     '(10de:1088)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 590',),
    PciDevice('',                     '(10de:1089)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 580',),
    PciDevice('',                     '(10de:108b)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 580',),
    PciDevice('Tesla',                '(10de:1091)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla M2090',),
    PciDevice('Tesla',                '(10de:0974)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla X2090',),
    PciDevice('Tesla',                '(10de:1094)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla M2075',),
    PciDevice('Tesla',                '(10de:1096)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla C2075',),
    PciDevice('',                     '(10de:10c0)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce 9300 GS',),
    PciDevice('GT218',                '(10de:10c3)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 8400GS',),
    PciDevice('GT218',                '(10de:10c5)', [NOUVEAU,NVIDIA], opengl='3.3', title='GeForce 405',),
    PciDevice('',                     '(10de:0565)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 630M',),
    PciDevice('',                     '(10de:0568)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 630M',),
    PciDevice('',                     '(10de:067a)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 620M',),
    PciDevice('',                     '(10de:0680)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 620M',),
    PciDevice('',                     '(10de:20dd)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 620M',),
    PciDevice('GK104',                '(10de:1180)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 680',),
    PciDevice('',                     '(10de:1188)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 690',),
    PciDevice('',                     '(10de:1189)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 670',),
    PciDevice('',                     '(10de:118f)', [NOUVEAU,NVIDIA], opengl='None', title='Tesla K10',),
    PciDevice('',                     '(10de:11a0)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 680M',),
    PciDevice('GF114',                '(10de:1200)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 560 Ti',),
    PciDevice('GF114',                '(10de:1201)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 560',),
    PciDevice('',                     '(10de:1203)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 460 SE v2',),
    PciDevice('GF114',                '(10de:1205)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 460 v2',),
    PciDevice('',                     '(10de:1206)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 555',),
    PciDevice('',                     '(10de:1207)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 645',),
    PciDevice('',                     '(10de:1208)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTX 560 SE',),
    PciDevice('',                     '(10de:1210)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 570M',),
    PciDevice('',                     '(10de:1211)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 580M',),
    PciDevice('',                     '(10de:1212)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 675M',),
    PciDevice('',                     '(10de:1213)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 670M',),
    PciDevice('',                     '(10de:1241)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 545',),
    PciDevice('',                     '(10de:1243)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 545',),
    PciDevice('GF116',                '(10de:1244)', [NOUVEAU,NVIDIA], opengl='4.1', title='GeForce GTX 550 Ti',),
    PciDevice('GF116',                '(10de:1245)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTS 450',),
    PciDevice('',                     '(10de:1246)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 550M',),
    PciDevice('GF116',                '(10de:1247)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('',                     '(10de:212a)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 635M',),
    PciDevice('',                     '(10de:212b)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 635M',),
    PciDevice('',                     '(10de:212c)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 635M',),
    PciDevice('',                     '(10de:1248)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('',                     '(10de:1249)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GTS 450',),
    PciDevice('',                     '(10de:124b)', [NOUVEAU,NVIDIA], opengl='None', title='GeForce GT 640',),
    PciDevice('',                     '(10de:124d)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 555M',),
    PciDevice('',                     '(10de:10cc)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GT 635M',),
    PciDevice('',                     '(10de:1251)', [NOUVEAU,NVIDIA], opengl='4.2', title='GeForce GTX 560M',),
    PciDevice('NV40GL',               '(10de:004e)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 4000',),
    PciDevice('G70GL',                '(10de:009d)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 4500',),
    PciDevice('NV41',                 '(10de:00cc)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX Go1400',),
    PciDevice('NV41',                 '(10de:00cd)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 3450/4000 SDI',),
    PciDevice('NV41GL',               '(10de:00ce)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 1400',),
    PciDevice('NV45GL',               '(10de:00f8)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 3400/Quadro FX 4000',),
    PciDevice('NV43',                 '(10de:014a)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 440',),
    PciDevice('NV43',                 '(10de:014c)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 540M',),
    PciDevice('NV43GL',               '(10de:014d)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 550',),
    PciDevice('NV43GL',               '(10de:014e)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 540',),
    PciDevice('NV44',                 '(10de:0165)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 285',),
    PciDevice('G80',                  '(10de:019d)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 5600',),
    PciDevice('G80',                  '(10de:019e)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 4600',),
    PciDevice('G72M',                 '(10de:01da)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 110M',),
    PciDevice('G72M',                 '(10de:01db)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 120M',),
    PciDevice('G72GL',                '(10de:01dc)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 350M',),
    PciDevice('G72GL',                '(10de:01de)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 350',),
    PciDevice('C51',                  '(10de:0245)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 210S / GeForce 6150LE',),
    PciDevice('G71',                  '(10de:0299)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro NVS 510M',),
    PciDevice('G71',                  '(10de:029a)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 2500M',),
    PciDevice('G71',                  '(10de:029b)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 1500M',),
    PciDevice('G71',                  '(10de:029c)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 5500',),
    PciDevice('G71GL',                '(10de:029d)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 3500',),
    PciDevice('G71',                  '(10de:029e)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 1500',),
    PciDevice('G70',                  '(10de:029f)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 4500 X2',),
    PciDevice('G73',                  '(10de:039c)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 560M',),
    PciDevice('G73GL',                '(10de:039e)', [NOUVEAU,NVIDIA], opengl='2.1', title='Quadro FX 560',),
    PciDevice('G84',                  '(10de:040a)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 370',),
    PciDevice('G84M',                 '(10de:040b)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 320M',),
    PciDevice('G84M',                 '(10de:040c)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 570M',),
    PciDevice('G84',                  '(10de:040d)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 1600M',),
    PciDevice('G84',                  '(10de:040e)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 570',),
    PciDevice('G84',                  '(10de:040f)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 1700',),
    PciDevice('G84M',                 '(10de:0429)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 140M',),
    PciDevice('G86M',                 '(10de:042a)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 130M',),
    PciDevice('G86M',                 '(10de:042b)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 135M',),
    PciDevice('G86M',                 '(10de:042d)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 360M',),
    PciDevice('G86',                  '(10de:042f)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 290',),
    PciDevice('GT200GL',              '(10de:05ed)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadroplex 2200 D2',),
    PciDevice('GT200GL',              '(10de:05f8)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadroplex 2200 S4',),
    PciDevice('GT200GL',              '(10de:05f9)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro CX',),
    PciDevice('GT200GL',              '(10de:05fd)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 5800',),
    PciDevice('GT200GL',              '(10de:05fe)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 4800',),
    PciDevice('GT200GL',              '(10de:05ff)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 3800',),
    PciDevice('G92GL',                '(10de:0619)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 4700 X2',),
    PciDevice('G92',                  '(10de:061a)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 3700',),
    PciDevice('G92GL',                '(10de:061b)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro VX 200',),
    PciDevice('G92M',                 '(10de:061c)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 3600M',),
    PciDevice('G92',                  '(10de:061d)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 2800M',),
    PciDevice('G92',                  '(10de:061e)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 3700M',),
    PciDevice('G92',                  '(10de:061f)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 3800M',),
    PciDevice('G94',                  '(10de:0638)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 1800',),
    PciDevice('G94M',                 '(10de:063a)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 2700M',),
    PciDevice('G96',                  '(10de:0658)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 380',),
    PciDevice('G96',                  '(10de:0659)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 580',),
    PciDevice('G96',                  '(10de:065a)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 1700M',),
    PciDevice('G96M',                 '(10de:065c)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 770M',),
    PciDevice('GF100',                '(10de:06d8)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 6000',),
    PciDevice('GF100',                '(10de:06d9)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 5000',),
    PciDevice('',                     '(10de:06da)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 5000M',),
    PciDevice('',                     '(10de:06dc)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 6000',),
    PciDevice('GF100',                '(10de:06dd)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 4000',),
    PciDevice('G86M',                 '(10de:06ea)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 150M',),
    PciDevice('G98M',                 '(10de:06eb)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 160M',),
    PciDevice('G98',                  '(10de:06f8)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 420',),
    PciDevice('G98',                  '(10de:06f9)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 370 LP',),
    PciDevice('G98',                  '(10de:06fa)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 450',),
    PciDevice('G98',                  '(10de:06fb)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 370M',),
    PciDevice('G98',                  '(10de:06fd)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro NVS 295',),
    PciDevice('G98',                  '(10de:06ff)', [NOUVEAU,NVIDIA], opengl='3.3', title='HICx16 + Graphics',),
    PciDevice('GT216',                '(10de:0a2c)', [NOUVEAU,NVIDIA], opengl='3.3', title='NVS 5100M',),
    PciDevice('GT216GL',              '(10de:0a38)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro 400',),
    PciDevice('GT216',                '(10de:0a3c)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 880M',),
    PciDevice('GT218',                '(10de:0a6a)', [NOUVEAU,NVIDIA], opengl='4.1', title='NVS 2100M',),
    PciDevice('GT218',                '(10de:0a6c)', [NOUVEAU,NVIDIA], opengl='4.1', title='NVS 3100M',),
    PciDevice('GT218GL',              '(10de:0a78)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 380 LP',),
    PciDevice('GT218',                '(10de:0a7c)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 380M',),
    PciDevice('GT215',                '(10de:0cbc)', [NOUVEAU,NVIDIA], opengl='3.3', title='Quadro FX 1800M',),
    PciDevice('GF106GL',              '(10de:0dd8)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 2000',),
    PciDevice('',                     '(10de:0914)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 2000D                          0x0DD8',),
    PciDevice('GF106',                '(10de:0dda)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 2000M',),
    PciDevice('',                     '(10de:0def)', [NOUVEAU,NVIDIA], opengl='None', title='NVS 5400M',),
    PciDevice('GF108',                '(10de:0df8)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 600',),
    PciDevice('',                     '(10de:0df9)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 500M',),
    PciDevice('GF108',                '(10de:0dfa)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 1000M',),
    PciDevice('',                     '(10de:0dfc)', [NOUVEAU,NVIDIA], opengl='None', title='NVS 5200M',),
    PciDevice('GF104',                '(10de:0e3a)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 3000M',),
    PciDevice('GF104',                '(10de:0e3b)', [NOUVEAU,NVIDIA], opengl='4.1', title='Quadro 4000M',),
    PciDevice('',                     '(10de:0ffb)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro K2000M',),
    PciDevice('',                     '(10de:0ffc)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro K1000M',),
    PciDevice('',                     '(10de:0fff)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 410',),
    PciDevice('GF108',                '(10de:1056)', [NOUVEAU,NVIDIA], opengl='4.1', title='NVS 4200M',),
    PciDevice('',                     '(10de:1057)', [NOUVEAU,NVIDIA], opengl='None', title='NVS 4200M',),
    PciDevice('',                     '(10de:107d)', [NOUVEAU,NVIDIA], opengl='None', title='NVS 310',),
    PciDevice('',                     '(10de:109a)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 5010M',),
    PciDevice('',                     '(10de:109b)', [NOUVEAU,NVIDIA], opengl='None', title='Quadro 7000',),
    PciDevice('GT218',                '(10de:10d8)', [NOUVEAU,NVIDIA], opengl='4.1', title='NVS 300',),
    PciDevice('NV36',                 '(10de:00fa)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce PCX 5750',),
    PciDevice('NV35',                 '(10de:00fb)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce PCX 5900',),
    PciDevice('NV37GL',               '(10de:00fc)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 330/GeForce PCX 5300',),
    PciDevice('NV37GL',               '(10de:00fd)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 330/Quadro NVS 280 PCI-E',),
    PciDevice('NV38GL',               '(10de:00fe)', [NOUVEAU,NV173], opengl='2.1', title='Quadro FX 1300',),
    PciDevice('NV30',                 '(10de:0301)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5800 Ultra',),
    PciDevice('NV30',                 '(10de:0302)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5800',),
    PciDevice('NV30GL',               '(10de:0308)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 2000',),
    PciDevice('NV30GL',               '(10de:0309)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 1000',),
    PciDevice('NV31',                 '(10de:0311)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5600 Ultra',),
    PciDevice('NV31',                 '(10de:0312)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5600',),
    PciDevice('NV31',                 '(10de:0314)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5600XT',),
    PciDevice('NV31M',                '(10de:031a)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go5600',),
    PciDevice('NV31M',                '(10de:031b)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go5650',),
    PciDevice('NV31',                 '(10de:031c)', [NOUVEAU,NV173], opengl='2.1**', title='Quadro FX Go700',),
    PciDevice('NV34',                 '(10de:0320)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5200',),
    PciDevice('NV34',                 '(10de:0321)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5200 Ultra',),
    PciDevice('NV34',                 '(10de:0322)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5200',),
    PciDevice('NV34',                 '(10de:0323)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5200LE',),
    PciDevice('NV34M',                '(10de:0324)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go5200',),
    PciDevice('NV34M',                '(10de:0325)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go5250',),
    PciDevice('NV34',                 '(10de:0326)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5500',),
    PciDevice('NV34',                 '(10de:0327)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5100',),
    PciDevice('NV34M',                '(10de:0328)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go5200 32M/64M',),
    PciDevice('NV34GL',               '(10de:032a)', [NOUVEAU,NV173], opengl='1.5', title='Quadro NVS 55/280 PCI',),
    PciDevice('NV34GL',               '(10de:032b)', [NOUVEAU,NV173], opengl='1.5', title='Quadro FX 500/FX 600',),
    PciDevice('NV34GLM',              '(10de:032c)', [NOUVEAU,NV173], opengl='1.5', title='GeForce FX Go53xx',),
    PciDevice('NV34',                 '(10de:032d)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX Go5100',),
    PciDevice('NV35',                 '(10de:0330)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5900 Ultra',),
    PciDevice('NV35',                 '(10de:0331)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5900',),
    PciDevice('NV35',                 '(10de:0332)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5900XT',),
    PciDevice('NV38',                 '(10de:0333)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5950 Ultra',),
    PciDevice('NV35',                 '(10de:0334)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5900ZT',),
    PciDevice('NV35GL',               '(10de:0338)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 3000',),
    PciDevice('NV35GL',               '(10de:033f)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 700',),
    PciDevice('NV36.1',               '(10de:0341)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5700 Ultra',),
    PciDevice('NV36.2',               '(10de:0342)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5700',),
    PciDevice('NV36',                 '(10de:0343)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5700LE',),
    PciDevice('NV36.4',               '(10de:0344)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX 5700VE',),
    PciDevice('NV36',                 '(10de:0347)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX Go5700',),
    PciDevice('NV36',                 '(10de:0348)', [NOUVEAU,NV173], opengl='2.1**', title='GeForce FX Go5700',),
    PciDevice('NV36',                 '(10de:034c)', [NOUVEAU,NV173], opengl='2.1**', title='Quadro FX Go1000',),
    PciDevice('NV36GL',               '(10de:034e)', [NOUVEAU,NV173], opengl='2.0', title='Quadro FX 1100',),
    PciDevice('NV11',                 '(10de:0110)', [NOUVEAU,NV96], opengl='1.2', title='GeForce2 MX/MX 400',),
    PciDevice('NV11DDR',              '(10de:0111)', [NOUVEAU,NV96], opengl='1.2', title='GeForce2 MX 100/200',),
    PciDevice('NV11',                 '(10de:0112)', [NOUVEAU,NV96], opengl='1.2', title='GeForce2 Go',),
    PciDevice('NV11GL',               '(10de:0113)', [NOUVEAU,NV96], opengl='1.2', title='Quadro2 MXR/EX/Go',),
    PciDevice('NV17',                 '(10de:0170)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 460',),
    PciDevice('NV17',                 '(10de:0171)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 440',),
    PciDevice('NV17',                 '(10de:0172)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 420',),
    PciDevice('NV17',                 '(10de:0173)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 440-SE',),
    PciDevice('NV17',                 '(10de:0174)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 440 Go',),
    PciDevice('NV17',                 '(10de:0175)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 420 Go',),
    PciDevice('NV17',                 '(10de:0176)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 420 Go 32M',),
    PciDevice('NV17',                 '(10de:0177)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 460 Go',),
    PciDevice('NV17GL',               '(10de:0178)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 550 XGL',),
    PciDevice('NV17',                 '(10de:0179)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 440 Go 64M',),
    PciDevice('NV17GL',               '(10de:017a)', [NOUVEAU,NV96], opengl='1.4', title='Quadro NVS 400',),
    PciDevice('NV17GL',               '(10de:017c)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 500 GoGL',),
    PciDevice('NV17',                 '(10de:017d)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 410 Go 16M',),
    PciDevice('NV18',                 '(10de:0181)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 440 with AGP8X',),
    PciDevice('NV18',                 '(10de:0182)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 440SE with AGP8X',),
    PciDevice('NV18',                 '(10de:0183)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 420 with AGP8X',),
    PciDevice('NV18',                 '(10de:0185)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX 4000',),
    PciDevice('NV18GL',               '(10de:0188)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 580 XGL',),
    PciDevice('NV18GL',               '(10de:018a)', [NOUVEAU,NV96], opengl='1.4', title='Quadro NVS 280 SD',),
    PciDevice('NV18GL',               '(10de:018b)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 380 XGL',),
    PciDevice('NV18GL',               '(10de:018c)', [NOUVEAU,NV96], opengl='1.4', title='Quadro NVS 50 PCI',),
    PciDevice('NVCrush11',            '(10de:01a0)', [NOUVEAU,NV96], opengl='3.3', title='GeForce2 Integrated GPU',),
    PciDevice('NV18',                 '(10de:01f0)', [NOUVEAU,NV96], opengl='1.2', title='GeForce4 MX Integrated GPU',),
    PciDevice('NV20',                 '(10de:0200)', [NOUVEAU,NV96], opengl='1.3', title='GeForce3',),
    PciDevice('NV20',                 '(10de:0201)', [NOUVEAU,NV96], opengl='1.3', title='GeForce3 Ti 200',),
    PciDevice('NV20',                 '(10de:0202)', [NOUVEAU,NV96], opengl='1.3', title='GeForce3 Ti 500',),
    PciDevice('NV20DCC',              '(10de:0203)', [NOUVEAU,NV96], opengl='1.4', title='Quadro DCC',),
    PciDevice('NV25',                 '(10de:0250)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4600',),
    PciDevice('NV25',                 '(10de:0251)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4400',),
    PciDevice('NV25',                 '(10de:0253)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4200',),
    PciDevice('NV25GL',               '(10de:0258)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 900 XGL',),
    PciDevice('NV25GL',               '(10de:0259)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 750 XGL',),
    PciDevice('NV25GL',               '(10de:025b)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 700 XGL',),
    PciDevice('NV28',                 '(10de:0280)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4800',),
    PciDevice('NV28',                 '(10de:0281)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4200 with AGP8X',),
    PciDevice('NV28',                 '(10de:0282)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 Ti 4800 SE',),
    PciDevice('NV28',                 '(10de:0286)', [NOUVEAU,NV96], opengl='1.4', title='GeForce4 4200 Go',),
    PciDevice('NV28GL',               '(10de:0288)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 980 XGL',),
    PciDevice('NV28GL',               '(10de:0289)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 780 XGL',),
    PciDevice('NV28GLM',              '(10de:028c)', [NOUVEAU,NV96], opengl='1.4', title='Quadro4 700 GoGL',),
    PciDevice('NV4',                  '(10de:0020)', [NOUVEAU,NV71], opengl='1.1', title='RIVA TNT',),
    PciDevice('NV5',                  '(10de:0028)', [NOUVEAU,NV71], opengl='1.1', title='RIVA TNT2/TNT2 Pro',),
    PciDevice('NV5',                  '(10de:0029)', [NOUVEAU,NV71], opengl='1.1', title='RIVA TNT2 Ultra',),
    PciDevice('NV6',                  '(10de:002c)', [NOUVEAU,NV71], opengl='1.1', title='Vanta/Vanta LT',),
    PciDevice('NV5M64',               '(10de:002d)', [NOUVEAU,NV71], opengl='1.1', title='RIVA TNT2 Model 64/Model 64 Pro',),
    PciDevice('NV5',                  '(10de:00a0)', [NOUVEAU,NV71], opengl='1.1', title='Aladdin TNT2',),
    PciDevice('NV10',                 '(10de:0100)', [NOUVEAU,NV71], opengl='1.2', title='GeForce 256',),
    PciDevice('NV10DDR',              '(10de:0101)', [NOUVEAU,NV71], opengl='1.2', title='GeForce DDR',),
    PciDevice('NV10GL',               '(10de:0103)', [NOUVEAU,NV71], opengl='1.2', title='Quadro',),
    PciDevice('NV15',                 '(10de:0150)', [NOUVEAU,NV71], opengl='1.2', title='GeForce2 GTS/GeForce2 Pro',),
    PciDevice('NV15DDR',              '(10de:0151)', [NOUVEAU,NV71], opengl='1.2', title='GeForce2 Ti',),
    PciDevice('NV15BR',               '(10de:0152)', [NOUVEAU,NV71], opengl='1.2', title='GeForce2 Ultra',),
    PciDevice('NV15GL',               '(10de:0153)', [NOUVEAU,NV71], opengl='1.2', title='Quadro2 Pro',),
    ]

def get_pci_devices(text):
    devices = []
    if type(text) is list:
        items = list
    else:
        text = text.decode('utf8')
        if '\n' in text:
            regex_vga = re.compile('VGA compatible controller (.*)', re.IGNORECASE)
            items = regex_vga.findall(text)
        else:
            items = text.split(' ')
    if len(items) <= 0:
        return None
    for item in items:
        item = item.strip()
        if len(item) <= 0:
            continue
        for device in pci_devices:
            if device.rc.search(item):
                devices.append(device)
    return devices


# For debugging
if __name__ == '__main__':
    from subprocess import Popen, PIPE

    pci_text = """
    00:02.0 VGA compatible controller [0300]: Intel Corporation Core Processor Integrated Graphics Controller [8086:0046] (rev 02) (prog-if 00 [VGA controller])
        Subsystem: Lenovo Device [17aa:215a]
        Control: I/O+ Mem+ BusMaster+ SpecCycle- MemWINV- VGASnoop- ParErr- Stepping- SERR- FastB2B- DisINTx+
        Status: Cap+ 66MHz- UDF- FastB2B+ ParErr- DEVSEL=fast >TAbort- <TAbort- <MAbort- >SERR- <PERR- INTx-
        Capabilities: [90] MSI: Enable+ Count=1/1 Maskable- 64bit-
                Address: fee0f00c  Data: 4171
        Kernel driver in use: i915
        Kernel modules: i915

    00:03.0 VGA compatible controller [0300]: Intel Corporation Core Processor Integrated Graphics Controller [8086:0042] (rev 02) (prog-if 00 [VGA controller])

    00:04.0 VGA compatible controller [0300]: Intel Corporation Core Processor Integrated Graphics Controller [8086:3582] (rev 02) (prog-if 00 [VGA controller])

    00:1c.2 PCI bridge [0604]: Intel Corporation 5 Series/3400 Series Chipset PCI Express Root Port 3 [8086:3b46] (rev 06) (prog-if 00 [Normal decode])
        Control: I/O+ Mem+ BusMaster+ SpecCycle- MemWINV- VGASnoop- ParErr- Stepping- SERR+ FastB2B- DisINTx-
        Status: Cap+ 66MHz- UDF- FastB2B- ParErr- DEVSEL=fast >TAbort- <TAbort- <MAbort- >SERR- <PERR- INTx-

        ff:02.3 Host bridge [0600]: Intel Corporation Core Processor Reserved [8086:2d13] (rev 02)
        Subsystem: Lenovo Device [17aa:2196]
        Control: I/O- Mem+ BusMaster+ SpecCycle- MemWINV- VGASnoop- ParErr- Stepping- SERR- FastB2B- DisINTx-
        Status: Cap- 66MHz- UDF- FastB2B- ParErr- DEVSEL=fast >TAbort- <TAbort- <MAbort- >SERR- <PERR- INTx-
        Latency: 0

        """

    for device in get_pci_devices(pci_text):
        print(device.name, device.supported, ','.join(device.drivers))

    pci_text = Popen('lspci -vvnn', stdout=PIPE, shell=True).stdout.read()
    for device in get_pci_devices(pci_text):
        print(device.name, device.supported, ','.join(device.drivers))
