//===-- Core/PerfSupport.h - Perf measurement helpers -----------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
///
/// \file
/// \brief This file provides helper functionality for measuring performance and
/// recording data to file.
///
//===----------------------------------------------------------------------===//

#ifndef CLANG_MODERNIZE_PERFSUPPORT_H
#define CLANG_MODERNIZE_PERFSUPPORT_H

#include "Transform.h"
#include "llvm/ADT/StringRef.h"

#include <map>
#include <vector>

/// \brief A single piece of performance data: a duration in milliseconds and a
/// label for that duration.
struct PerfItem {
  PerfItem(const llvm::StringRef Label, float Duration)
      : Label(Label), Duration(Duration) {}

  /// Label for this performance measurement.
  std::string Label;

  /// Duration in milliseconds.
  float Duration;
};

/// Maps source file names to a vector of durations/labels.
typedef std::map<std::string, std::vector<PerfItem> > SourcePerfData;

/// Extracts durations collected by a Transform for all sources and adds them
/// to a SourcePerfData map where data is organized by source file.
extern void collectSourcePerfData(const Transform &T, SourcePerfData &Data);

/// Write timing results to a JSON formatted file.
///
/// File is placed in the directory given by \p DirectoryName. File is named in
/// a unique way with time and process ID to avoid naming collisions with
/// existing files or files being generated by other migrator processes.
void writePerfDataJSON(
    const llvm::StringRef DirectoryName,
    const SourcePerfData &TimingResults);

/// Dump a SourcePerfData map to llvm::errs().
extern void dumpPerfData(const SourcePerfData &Data);

#endif // CLANG_MODERNIZE_PERFSUPPORT_H
