package Net::Domain::TLD;
use strict;
use base qw( Exporter );
use 5.006;
our @EXPORT_OK = qw( tlds tld_exists %tld_profile );
our $VERSION = '1.75';

use warnings;
use Carp;
use Storable qw ( dclone );

use constant TLD_TYPES => qw ( new_open new_restricted gtld_open gtld_restricted gtld_new cc ccidn );

=head1 NAME

  Net::Domain::TLD - Work with TLD names 

=head1 SYNOPSIS

  use Net::Domain::TLD qw(tlds tld_exists);
  my @ccTLDs = tlds('cc');
  print "TLD ok\n" if tld_exists('ac','cc');

=head1 DESCRIPTION

  The purpose of this module is to provide user with current list of 
  available top level domain names including new ICANN additions and ccTLDs
  Currently TLD definitions have been acquired from the following sources:

  http://www.icann.org/tlds/
  http://www.dnso.org/constituency/gtld/gtld.html
  http://www.iana.org/cctld/cctld-whois.htm
  https://www.iana.org/domains/root/db

=cut

my %tld_profile = (
  reserved => {
    test => q{DNS testing names},
    example => q{Documentation names},
    invalid => q{Invalid names},
    localhost => q{Loopback names}
  },
  new_open => {
    info => q{Unrestricted use},
    xxx => q{sponsored top-level domain}
  },
  new_restricted => { 
    aero => q{Air-transport industry},
    asia => q{Companies, organisations and individuals in the Asia-Pacific region},
    arpa => q{Address and Routing Parameter Area},
    biz => q{Businesses},
    cat => q{Catalan linguistic and cultural community},
    coop => q{Cooperatives},
    jobs => q{Human Resource Management},
    mobi => q{Mobile},
    museum => q{Museums},
    name => q{For registration by individuals},
    post => q{Universal Postal Union},
    pro => q{Accountants, lawyers, and physicians},
    travel => q{Travel industry},
    tel => q{For businesses and individuals to publish contact data}
  },
  gtld_open => {
    com => q{Commercial organization},
    net => q{Network connection services provider},
    org => q{Non-profit organizations and industry standard groups}
  },
  gtld_restricted => {
    gov => q{United States Government},
    mil => q{United States Military},
    edu => q{Educational institution},
    int => q{International treaties/databases},
  },
  cc => {
    ac => q{Ascension Island},
    ad => q{Andorra},
    ae => q{United Arab Emirates},
    af => q{Afghanistan},
    ag => q{Antigua and Barbuda},
    ai => q{Anguilla},
    al => q{Albania},
    am => q{Armenia},
    an => q{Netherlands Antilles},
    ao => q{Angola},
    aq => q{Antartica},
    ar => q{Argentina},
    as => q{American Samoa},
    at => q{Austria},
    au => q{Australia},
    aw => q{Aruba},
    ax => q(Aland Islands),
    az => q{Azerbaijan},
    ba => q{Bosnia and Herzegovina},
    bb => q{Barbados},
    bd => q{Bangladesh},
    be => q{Belgium},
    bf => q{Burkina Faso},
    bg => q{Bulgaria},
    bh => q{Bahrain},
    bi => q{Burundi},
    bj => q{Benin},
    bl => q(Saint Barthelemy),
    bm => q{Bermuda},
    bn => q{Brunei Darussalam},
    bo => q{Bolivia},
    bq => q{Not assigned},
    br => q{Brazil},
    bs => q{Bahamas},
    bt => q{Bhutan},
    bv => q{Bouvet Island},
    bw => q{Botswana},
    by => q{Belarus},
    bz => q{Belize},
    ca => q{Canada},
    cc => q{Cocos (Keeling) Islands},
    cd => q{Congo, Democratic Republic of the},
    cf => q{Central African Republic},
    cg => q{Congo, Republic of},
    ch => q{Switzerland},
    ci => q{Cote d'Ivoire},
    ck => q{Cook Islands},
    cl => q{Chile},
    cm => q{Cameroon},
    cn => q{China},
    co => q{Colombia},
    cr => q{Costa Rica},
    cu => q{Cuba},
    cv => q{Cap Verde},
    cw => q{University of the Netherlands Antilles},
    cx => q{Christmas Island},
    cy => q{Cyprus},
    cz => q{Czech Republic},
    de => q{Germany},
    dj => q{Djibouti},
    dk => q{Denmark},
    dm => q{Dominica},
    do => q{Dominican Republic},
    dz => q{Algeria},
    ec => q{Ecuador},
    ee => q{Estonia},
    eg => q{Egypt},
    eh => q{Western Sahara},
    er => q{Eritrea},
    es => q{Spain},
    et => q{Ethiopia},
    eu => q{European Union},
    fi => q{Finland},
    fj => q{Fiji},
    fk => q{Falkland Islands (Malvina)},
    fm => q{Micronesia, Federal State of},
    fo => q{Faroe Islands},
    fr => q{France},
    ga => q{Gabon},
    gb => q{United Kingdom},
    gd => q{Grenada},
    ge => q{Georgia},
    gf => q{French Guiana},
    gg => q{Guernsey},
    gh => q{Ghana},
    gi => q{Gibraltar},
    gl => q{Greenland},
    gm => q{Gambia},
    gn => q{Guinea},
    gp => q{Guadeloupe},
    gq => q{Equatorial Guinea},
    gr => q{Greece},
    gs => q{South Georgia and the South Sandwich Islands},
    gt => q{Guatemala},
    gu => q{Guam},
    gw => q{Guinea-Bissau},
    gy => q{Guyana},
    hk => q{Hong Kong},
    hm => q{Heard and McDonald Islands},
    hn => q{Honduras},
    hr => q{Croatia/Hrvatska},
    ht => q{Haiti},
    hu => q{Hungary},
    id => q{Indonesia},
    ie => q{Ireland},
    il => q{Israel},
    im => q{Isle of Man},
    in => q{India},
    io => q{British Indian Ocean Territory},
    iq => q{Iraq},
    ir => q{Iran (Islamic Republic of)},
    is => q{Iceland},
    it => q{Italy},
    je => q{Jersey},
    jm => q{Jamaica},
    jo => q{Jordan},
    jp => q{Japan},
    ke => q{Kenya},
    kg => q{Kyrgyzstan},
    kh => q{Cambodia},
    ki => q{Kiribati},
    km => q{Comoros},
    kn => q{Saint Kitts and Nevis},
    kp => q{Korea, Democratic People's Republic},
    kr => q{Korea, Republic of},
    kw => q{Kuwait},
    ky => q{Cayman Islands},
    kz => q{Kazakhstan},
    la => q{Lao People's Democratic Republic},
    lb => q{Lebanon},
    lc => q{Saint Lucia},
    li => q{Liechtenstein},
    lk => q{Sri Lanka},
    lr => q{Liberia},
    ls => q{Lesotho},
    lt => q{Lithuania},
    lu => q{Luxembourg},
    lv => q{Latvia},
    ly => q{Libyan Arab Jamahiriya},
    ma => q{Morocco},
    mc => q{Monaco},
    md => q{Moldova, Republic of},
    me => q(Montenegro),
    mf => q{Saint Martin (French part)},
    mg => q{Madagascar},
    mh => q{Marshall Islands},
    mk => q{Macedonia, Former Yugoslav Republic},
    ml => q{Mali},
    mm => q{Myanmar},
    mn => q{Mongolia},
    mo => q{Macau},
    mp => q{Northern Mariana Islands},
    mq => q{Martinique},
    mr => q{Mauritania},
    ms => q{Montserrat},
    mt => q{Malta},
    mu => q{Mauritius},
    mv => q{Maldives},
    mw => q{Malawi},
    mx => q{Mexico},
    my => q{Malaysia},
    mz => q{Mozambique},
    na => q{Namibia},
    nc => q{New Caledonia},
    ne => q{Niger},
    nf => q{Norfolk Island},
    ng => q{Nigeria},
    ni => q{Nicaragua},
    nl => q{Netherlands},
    no => q{Norway},
    np => q{Nepal},
    nr => q{Nauru},
    nu => q{Niue},
    nz => q{New Zealand},
    om => q{Oman},
    pa => q{Panama},
    pe => q{Peru},
    pf => q{French Polynesia},
    pg => q{Papua New Guinea},
    ph => q{Philippines},
    pk => q{Pakistan},
    pl => q{Poland},
    pm => q{St. Pierre and Miquelon},
    pn => q{Pitcairn Island},
    pr => q{Puerto Rico},
    ps => q{Palestinian Territories},
    pt => q{Portugal},
    pw => q{Palau},
    py => q{Paraguay},
    qa => q{Qatar},
    re => q{Reunion Island},
    ro => q{Romania},
    rs => q(Serbia),
    ru => q{Russian Federation},
    rw => q{Rwanda},
    sa => q{Saudi Arabia},
    sb => q{Solomon Islands},
    sc => q{Seychelles},
    sd => q{Sudan},
    se => q{Sweden},
    sg => q{Singapore},
    sh => q{St. Helena},
    si => q{Slovenia},
    sj => q{Svalbard and Jan Mayen Islands},
    sk => q{Slovak Republic},
    sl => q{Sierra Leone},
    sm => q{San Marino},
    sn => q{Senegal},
    so => q{Somalia},
    sr => q{Suriname},
    ss => q{Not assigned},
    st => q{Sao Tome and Principe},
    su => q{Soviet Union},
    sv => q{El Salvador},
    sx => q{SX Registry SA B.V.},
    sy => q{Syrian Arab Republic},
    sz => q{Swaziland},
    tc => q{Turks and Caicos Islands},
    td => q{Chad},
    tf => q{French Southern Territories},
    tg => q{Togo},
    th => q{Thailand},
    tj => q{Tajikistan},
    tk => q{Tokelau},
    tl => q{Timor-Leste},
    tm => q{Turkmenistan},
    tn => q{Tunisia},
    to => q{Tonga},
    tp => q{East Timor},
    tr => q{Turkey},
    tt => q{Trinidad and Tobago},
    tv => q{Tuvalu},
    tw => q{Taiwan},
    tz => q{Tanzania},
    ua => q{Ukraine},
    ug => q{Uganda},
    uk => q{United Kingdom},
    um => q{US Minor Outlying Islands},
    us => q{United States},
    uy => q{Uruguay},
    uz => q{Uzbekistan},
    va => q{Holy See (City Vatican State)},
    vc => q{Saint Vincent and the Grenadines},
    ve => q{Venezuela},
    vg => q{Virgin Islands (British)},
    vi => q{Virgin Islands (USA)},
    vn => q{Vietnam},
    vu => q{Vanuatu},
    wf => q{Wallis and Futuna Islands},
    ws => q{Western Samoa},
    ye => q{Yemen},
    yt => q{Mayotte},
    yu => q{Yugoslavia},
    za => q{South Africa},
    zm => q{Zambia},
    zw => q{Zimbabwe}
  },
  ccidn => {
    'xn--0zwm56d' => q{Internet Assigned Numbers Authority},
    'xn--11b5bs3a9aj6g' => q{Internet Assigned Numbers Authority},
    'xn--1qqw23a' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    'xn--3bst00m' => q{Eagle Horizon Limited},
    'xn--3ds443g' => q{TLD REGISTRY LIMITED},
    'xn--3e0b707e' => q{KISA (Korea Internet & Security Agency)},
    'xn--45brj9c' => q{National Internet Exchange of India},
    'xn--45q11c' => q{Zodiac Scorpio Limited},
    'xn--4gbrim' => q{Suhub Electronic Establishment},
    'xn--54b7fta0cc' => q{Not assigned},
    'xn--55qw42g' => q{China Organizational Name Administration Center},
    'xn--55qx5d' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    'xn--6frz82g' => q{Afilias Limited},
    'xn--6qq986b3xl' => q{Tycoon Treasure Limited},
    'xn--80adxhks' => q{Foundation for Assistance for Internet Technologies and Infrastructure Development (FAITID)},
    'xn--80akhbyknj4f' => q{Internet Assigned Numbers Authority},
    'xn--80ao21a' => q{Association of IT Companies of Kazakhstan},
    'xn--80asehdb' => q{CORE Association},
    'xn--80aswg' => q{CORE Association},
    'xn--90a3ac' => q{Serbian National Internet Domain Registry (RNIDS)},
    'xn--90ais' => q{Not assigned},
    'xn--9t4b11yi5a' => q{Internet Assigned Numbers Authority},
    'xn--b4w605ferd' => q{Temasek Holdings (Private) Limited},
    'xn--c1avg' => q{Public Interest Registry},
    'xn--cg4bki' => q{SAMSUNG SDS CO., LTD},
    'xn--clchc0ea0b2g2a9gcd' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'xn--czr694b' => q{HU YI GLOBAL INFORMATION RESOURCES(HOLDING) COMPANY.HONGKONG LIMITED},
    'xn--czrs0t' => q{Wild Island, LLC},
    'xn--czru2d' => q{Zodiac Aquarius Limited},
    'xn--d1acj3b' => q{The Foundation for Network Initiatives “The Smart Internet”},
    'xn--d1alf' => q{Macedonian Academic Research Network Skopje},
    'xn--deba0ad' => q{Internet Assigned Numbers Authority},
    'xn--fiq228c5hs' => q{TLD REGISTRY LIMITED},
    'xn--fiq64b' => q{CITIC Group Corporation},
    'xn--fiqs8s' => q{China Internet Network Information Center},
    'xn--fiqz9s' => q{China Internet Network Information Center},
    'xn--flw351e' => q{Charleston Road Registry Inc.},
    'xn--fpcrj9c3d' => q{National Internet Exchange of India},
    'xn--fzc2c9e2c' => q{LK Domain Registry},
    'xn--g6w251d' => q{Internet Assigned Numbers Authority},
    'xn--gecrj9c' => q{National Internet Exchange of India},
    'xn--h2brj9c' => q{National Internet Exchange of India},
    'xn--hgbk6aj7f53bba' => q{Internet Assigned Numbers Authority},
    'xn--hlcj6aya9esc7a' => q{Internet Assigned Numbers Authority},
    'xn--hxt814e' => q{Zodiac Libra Limited},
    'xn--i1b6b1a6a2e' => q{Public Interest Registry},
    'xn--io0a7i' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    'xn--j1amh' => q{Ukrainian Network Information Centre (UANIC), Inc.},
    'xn--j6w193g' => q{Hong Kong Internet Registration Corporation Ltd.},
    'xn--jxalpdlp' => q{Internet Assigned Numbers Authority},
    'xn--kgbechtv' => q{Internet Assigned Numbers Authority},
    'xn--kprw13d' => q{Taiwan Network Information Center (TWNIC)},
    'xn--kpry57d' => q{Taiwan Network Information Center (TWNIC)},
    'xn--kput3i' => q{Beijing RITT-Net Technology Development Co., Ltd},
    'xn--l1acc' => q{Datacom Co.,Ltd},
    'xn--lgbbat1ad8j' => q{CERIST},
    'xn--mgb9awbf' => q{Telecommunications Regulatory Authority (TRA)},
    'xn--mgba3a4f16a' => q{Institute for Research in Fundamental Sciences (IPM)},
    'xn--mgbaam7a8h' => q{Telecommunications Regulatory Authority (TRA)},
    'xn--mgbab2bd' => q{CORE Association},
    'xn--mgbai9azgqp6j' => q{Not assigned},
    'xn--mgbayh7gpa' => q{National Information Technology Center (NITC)},
    'xn--mgbbh1a71e' => q{National Internet Exchange of India},
    'xn--mgbc0a9azcg' => q{Agence Nationale de Rglementation des Tlcommunications (ANRT)},
    'xn--mgberp4a5d4ar' => q{Communications and Information Technology Commission},
    'xn--mgbpl2fh' => q{Not assigned},
    'xn--mgbtx2b' => q{Not assigned},
    'xn--mgbx4cd0ab' => q{MYNIC Berhad},
    'xn--ngbc5azd' => q{International Domain Registry Pty. Ltd.},
    'xn--node' => q{Information Technologies Development Center (ITDC)},
    'xn--nqv7f' => q{Public Interest Registry},
    'xn--nqv7fs00ema' => q{Public Interest Registry},
    'xn--o3cw4h' => q{Thai Network Information Center Foundation},
    'xn--ogbpf8fl' => q{National Agency for Network Services (NANS)},
    'xn--p1acf' => q{Rusnames Limited},
    'xn--p1ai' => q{Coordination Center for TLD RU},
    'xn--pgbs0dh' => q{Agence Tunisienne d'Internet},
    'xn--q9jyb4c' => q{Charleston Road Registry Inc.},
    'xn--qcka1pmc' => q{Charleston Road Registry Inc.},
    'xn--rhqv96g' => q{Stable Tone Limited},
    'xn--s9brj9c' => q{National Internet Exchange of India},
    'xn--ses554g' => q{KNET Co., Ltd},
    'xn--unup4y' => q{Spring Fields, LLC},
    'xn--vermgensberater-ctb' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'xn--vermgensberatung-pwb' => q{Deutsche Vermgensberatung Aktiengesellschaft DVAG},
    'xn--vhquv' => q{Dash McCook, LLC},
    'xn--wgbh1c' => q{National Telecommunication Regulatory Authority - NTRA},
    'xn--wgbl6a' => q{Communications Regulatory Authority},
    'xn--xhq521b' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    'xn--xkc2al3hye2a' => q{LK Domain Registry},
    'xn--xkc2dl3a5ee0h' => q{National Internet Exchange of India},
    'xn--y9a3aq' => q{Not assigned},
    'xn--yfro4i67o' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'xn--ygbi2ammx' => q{Ministry of Telecom & Information Technology (MTIT)},
    'xn--zckzah' => q{Internet Assigned Numbers Authority},
    'xn--zfr164b' => q{China Organizational Name Administration Center},
  },
  gtld_new => {
    '&#x200f;טעסט&#x200e;' => q{Internet Assigned Numbers Authority},
    '&#x200f;קום&#x200e;' => q{VeriSign Sarl},
    '&#x200f;آزمایشی&#x200e;' => q{Internet Assigned Numbers Authority},
    '&#x200f;إختبار&#x200e;' => q{Internet Assigned Numbers Authority},
    '&#x200f;ابوظبي&#x200e;' => q{Abu Dhabi Systems and Information Centre},
    '&#x200f;ارامكو&#x200e;' => q{Aramco Services Company},
    '&#x200f;الاردن&#x200e;' => q{National Information Technology Center (NITC)},
    '&#x200f;الجزائر&#x200e;' => q{CERIST},
    '&#x200f;السعودية&#x200e;' => q{Communications and Information Technology Commission},
    '&#x200f;العليان&#x200e;' => q{Crescent Holding GmbH},
    '&#x200f;المغرب&#x200e;' => q{Agence Nationale de Réglementation des Télécommunications (ANRT)},
    '&#x200f;امارات&#x200e;' => q{Telecommunications Regulatory Authority (TRA)},
    '&#x200f;ایران&#x200e;' => q{Institute for Research in Fundamental Sciences (IPM)},
    '&#x200f;بارت&#x200e;' => q{Not assigned},
    '&#x200f;بازار&#x200e;' => q{CORE Association},
    '&#x200f;بيتك&#x200e;' => q{Kuwait Finance House},
    '&#x200f;بھارت&#x200e;' => q{National Internet Exchange of India},
    '&#x200f;تونس&#x200e;' => q{Agence Tunisienne d&#39;Internet},
    '&#x200f;سودان&#x200e;' => q{Sudan Internet Society},
    '&#x200f;سورية&#x200e;' => q{National Agency for Network Services (NANS)},
    '&#x200f;شبكة&#x200e;' => q{International Domain Registry Pty. Ltd.},
    '&#x200f;عراق&#x200e;' => q{Communications and Media Commission (CMC)},
    '&#x200f;عمان&#x200e;' => q{Telecommunications Regulatory Authority (TRA)},
    '&#x200f;فلسطين&#x200e;' => q{Ministry of Telecom &amp; Information Technology (MTIT)},
    '&#x200f;قطر&#x200e;' => q{Communications Regulatory Authority},
    '&#x200f;كوم&#x200e;' => q{VeriSign Sarl},
    '&#x200f;مصر&#x200e;' => q{National Telecommunication Regulatory Authority - NTRA},
    '&#x200f;مليسيا&#x200e;' => q{MYNIC Berhad},
    '&#x200f;موبايلي&#x200e;' => q{GreenTech Consultancy Company W.L.L.},
    '&#x200f;موقع&#x200e;' => q{Suhub Electronic Establishment},
    '&#x200f;همراه&#x200e;' => q{Asia Green IT System Bilgisayar San. ve Tic. Ltd. Sti.},
    '&#x200f;پاکستان&#x200e;' => q{Not assigned},
    '&#x200f;ڀارت&#x200e;' => q{Not assigned},
    'aaa' => q{American Automobile Association, Inc.},
    'aarp' => q{AARP},
    'abarth' => q{Fiat Chrysler Automobiles N.V.},
    'abb' => q{ABB Ltd},
    'abbott' => q{Abbott Laboratories, Inc.},
    'abbvie' => q{AbbVie Inc.},
    'abc' => q{Disney Enterprises, Inc.},
    'able' => q{Able Inc.},
    'abogado' => q{Top Level Domain Holdings Limited},
    'abudhabi' => q{Abu Dhabi Systems and Information Centre},
    'ac' => q{Network Information Center (AC Domain Registry) - c/o Cable and Wireless (Ascension Island)},
    'academy' => q{Half Oaks, LLC},
    'accenture' => q{Accenture plc},
    'accountant' => q{dot Accountant Limited},
    'accountants' => q{Knob Town, LLC},
    'aco' => q{ACO Severin Ahlmann GmbH &amp; Co. KG},
    'active' => q{The Active Network, Inc},
    'actor' => q{United TLD Holdco Ltd.},
    'ad' => q{Andorra Telecom},
    'adac' => q{Allgemeiner Deutscher Automobil-Club e.V. (ADAC)},
    'ads' => q{Charleston Road Registry Inc.},
    'adult' => q{ICM Registry AD LLC},
    'ae' => q{Telecommunication Regulatory Authority (TRA)},
    'aeg' => q{Aktiebolaget Electrolux},
    'aero' => q{Societe Internationale de Telecommunications Aeronautique (SITA INC USA)},
    'aetna' => q{Aetna Life Insurance Company},
    'af' => q{Ministry of Communications and IT},
    'afamilycompany' => q{Johnson Shareholdings, Inc.},
    'afl' => q{Australian Football League},
    'ag' => q{UHSA School of Medicine},
    'agakhan' => q{Fondation Aga Khan (Aga Khan Foundation)},
    'agency' => q{Steel Falls, LLC},
    'ai' => q{Government of Anguilla},
    'aig' => q{American International Group, Inc.},
    'aigo' => q{aigo Digital Technology Co,Ltd.},
    'airbus' => q{Airbus S.A.S.},
    'airforce' => q{United TLD Holdco Ltd.},
    'airtel' => q{Bharti Airtel Limited},
    'akdn' => q{Fondation Aga Khan (Aga Khan Foundation)},
    'al' => q{Electronic and Postal Communications Authority - AKEP},
    'alfaromeo' => q{Fiat Chrysler Automobiles N.V.},
    'alibaba' => q{Alibaba Group Holding Limited},
    'alipay' => q{Alibaba Group Holding Limited},
    'allfinanz' => q{Allfinanz Deutsche Vermögensberatung Aktiengesellschaft},
    'allstate' => q{Allstate Fire and Casualty Insurance Company},
    'ally' => q{Ally Financial Inc.},
    'alsace' => q{REGION D ALSACE},
    'alstom' => q{ALSTOM},
    'am' => q{&quot;Internet Society&quot; Non-governmental Organization},
    'americanexpress' => q{American Express Travel Related Services Company, Inc.},
    'americanfamily' => q{AmFam, Inc.},
    'amex' => q{American Express Travel Related Services Company, Inc.},
    'amfam' => q{AmFam, Inc.},
    'amica' => q{Amica Mutual Insurance Company},
    'amsterdam' => q{Gemeente Amsterdam},
    'an' => q{Retired},
    'analytics' => q{Campus IP LLC},
    'android' => q{Charleston Road Registry Inc.},
    'anquan' => q{QIHOO 360 TECHNOLOGY CO. LTD.},
    'anz' => q{Australia and New Zealand Banking Group Limited},
    'ao' => q{Faculdade de Engenharia da Universidade Agostinho Neto},
    'apartments' => q{June Maple, LLC},
    'app' => q{Charleston Road Registry Inc.},
    'apple' => q{Apple Inc.},
    'aq' => q{Antarctica Network Information Centre Limited},
    'aquarelle' => q{Aquarelle.com},
    'ar' => q{Presidencia de la Nación – Secretaría Legal y Técnica},
    'aramco' => q{Aramco Services Company},
    'archi' => q{STARTING DOT LIMITED},
    'army' => q{United TLD Holdco Ltd.},
    'arpa' => q{Internet Architecture Board (IAB)},
    'art' => q{UK Creative Ideas Limited},
    'arte' => q{Association Relative à la Télévision Européenne G.E.I.E.},
    'as' => q{AS Domain Registry},
    'asda' => q{Wal-Mart Stores, Inc.},
    'asia' => q{DotAsia Organisation Ltd.},
    'associates' => q{Baxter Hill, LLC},
    'at' => q{nic.at GmbH},
    'athleta' => q{The Gap, Inc.},
    'attorney' => q{United TLD Holdco, Ltd},
    'au' => q{.au Domain Administration (auDA)},
    'auction' => q{United TLD HoldCo, Ltd.},
    'audi' => q{AUDI Aktiengesellschaft},
    'audible' => q{Amazon Registry Services, Inc.},
    'audio' => q{Uniregistry, Corp.},
    'auspost' => q{Australian Postal Corporation},
    'author' => q{Amazon Registry Services, Inc.},
    'auto' => q{Cars Registry Limited},
    'autos' => q{DERAutos, LLC},
    'avianca' => q{Aerovias del Continente Americano S.A. Avianca},
    'aw' => q{SETAR},
    'aws' => q{Amazon Registry Services, Inc.},
    'ax' => q{Ålands landskapsregering},
    'axa' => q{AXA SA},
    'az' => q{IntraNS},
    'azure' => q{Microsoft Corporation},
    'ba' => q{Universtiy Telinformatic Centre (UTIC)},
    'baby' => q{Johnson &amp; Johnson Services, Inc.},
    'baidu' => q{Baidu, Inc.},
    'banamex' => q{Citigroup Inc.},
    'bananarepublic' => q{The Gap, Inc.},
    'band' => q{United TLD Holdco, Ltd},
    'bank' => q{fTLD Registry Services, LLC},
    'bar' => q{Punto 2012 Sociedad Anonima Promotora de Inversion de Capital Variable},
    'barcelona' => q{Municipi de Barcelona},
    'barclaycard' => q{Barclays Bank PLC},
    'barclays' => q{Barclays Bank PLC},
    'barefoot' => q{Gallo Vineyards, Inc.},
    'bargains' => q{Half Hallow, LLC},
    'bauhaus' => q{Werkhaus GmbH},
    'bayern' => q{Bayern Connect GmbH},
    'bb' => q{Government of Barbados - Ministry of Economic Affairs and Development
    Telecommunications Unit},
    'bbc' => q{British Broadcasting Corporation},
    'bbt' => q{BB&amp;T Corporation},
    'bbva' => q{BANCO BILBAO VIZCAYA ARGENTARIA, S.A.},
    'bcg' => q{The Boston Consulting Group, Inc.},
    'bcn' => q{Municipi de Barcelona},
    'bd' => q{Ministry of Post &amp; Telecommunications - Bangladesh Secretariat},
    'be' => q{DNS Belgium vzw/asbl},
    'beats' => q{Beats Electronics, LLC},
    'beauty' => q{L&#39;Oréal},
    'beer' => q{Top Level Domain Holdings Limited},
    'bentley' => q{Bentley Motors Limited},
    'berlin' => q{dotBERLIN GmbH &amp; Co. KG},
    'best' => q{BestTLD Pty Ltd},
    'bestbuy' => q{BBY Solutions, Inc.},
    'bet' => q{Afilias plc},
    'bf' => q{ARCE-AutoritÈ de RÈgulation des Communications Electroniques},
    'bg' => q{Register.BG},
    'bh' => q{Telecommunications Regulatory Authority (TRA)},
    'bharti' => q{Bharti Enterprises (Holding) Private Limited},
    'bi' => q{Centre National de l&#39;Informatique},
    'bible' => q{American Bible Society},
    'bid' => q{dot Bid Limited},
    'bike' => q{Grand Hollow, LLC},
    'bing' => q{Microsoft Corporation},
    'bingo' => q{Sand Cedar, LLC},
    'bio' => q{STARTING DOT LIMITED},
    'biz' => q{Neustar, Inc.},
    'bj' => q{Benin Telecoms S.A.},
    'bl' => q{Not assigned},
    'black' => q{Afilias Limited},
    'blackfriday' => q{Uniregistry, Corp.},
    'blanco' => q{BLANCO GmbH + Co KG},
    'blockbuster' => q{Dish DBS Corporation},
    'blog' => q{Knock Knock WHOIS There, LLC},
    'bloomberg' => q{Bloomberg IP Holdings LLC},
    'blue' => q{Afilias Limited},
    'bm' => q{Registry General Department, Ministry of Home Affairs},
    'bms' => q{Bristol-Myers Squibb Company},
    'bmw' => q{Bayerische Motoren Werke Aktiengesellschaft},
    'bn' => q{Brunei Darussalam Network Information Centre Sdn Bhd (BNNIC)},
    'bnl' => q{Banca Nazionale del Lavoro},
    'bnpparibas' => q{BNP Paribas},
    'bo' => q{Agencia para el Desarrollo de la Información de la Sociedad en Bolivia},
    'boats' => q{DERBoats, LLC},
    'boehringer' => q{Boehringer Ingelheim International GmbH},
    'bofa' => q{NMS Services, Inc.},
    'bom' => q{Núcleo de Informação e Coordenação do Ponto BR - NIC.br},
    'bond' => q{Bond University Limited},
    'boo' => q{Charleston Road Registry Inc.},
    'book' => q{Amazon Registry Services, Inc.},
    'booking' => q{Booking.com B.V.},
    'boots' => q{THE BOOTS COMPANY PLC},
    'bosch' => q{Robert Bosch GMBH},
    'bostik' => q{Bostik SA},
    'bot' => q{Amazon Registry Services, Inc.},
    'boutique' => q{Over Galley, LLC},
    'bq' => q{Not assigned},
    'br' => q{Comite Gestor da Internet no Brasil},
    'bradesco' => q{Banco Bradesco S.A.},
    'bridgestone' => q{Bridgestone Corporation},
    'broadway' => q{Celebrate Broadway, Inc.},
    'broker' => q{DOTBROKER REGISTRY LTD},
    'brother' => q{Brother Industries, Ltd.},
    'brussels' => q{DNS.be vzw},
    'bs' => q{The College of the Bahamas},
    'bt' => q{Ministry of Information and Communications},
    'budapest' => q{Top Level Domain Holdings Limited},
    'bugatti' => q{Bugatti International SA},
    'build' => q{Plan Bee LLC},
    'builders' => q{Atomic Madison, LLC},
    'business' => q{Spring Cross, LLC},
    'buy' => q{Amazon Registry Services, INC},
    'buzz' => q{DOTSTRATEGY CO.},
    'bv' => q{UNINETT Norid A/S},
    'bw' => q{Botswana Communications Regulatory Authority (BOCRA)},
    'by' => q{Reliable Software, Ltd.},
    'bz' => q{University of Belize},
    'bzh' => q{Association www.bzh},
    'ca' => q{Canadian Internet Registration Authority (CIRA) Autorité Canadienne pour les enregistrements Internet (ACEI)},
    'cab' => q{Half Sunset, LLC},
    'cafe' => q{Pioneer Canyon, LLC},
    'cal' => q{Charleston Road Registry Inc.},
    'call' => q{Amazon Registry Services, Inc.},
    'calvinklein' => q{PVH gTLD Holdings LLC},
    'cam' => q{AC Webconnecting Holding B.V.},
    'camera' => q{Atomic Maple, LLC},
    'camp' => q{Delta Dynamite, LLC},
    'cancerresearch' => q{Australian Cancer Research Foundation},
    'canon' => q{Canon Inc.},
    'capetown' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'capital' => q{Delta Mill, LLC},
    'capitalone' => q{Capital One Financial Corporation},
    'car' => q{Cars Registry Limited},
    'caravan' => q{Caravan International, Inc.},
    'cards' => q{Foggy Hollow, LLC},
    'care' => q{Goose Cross, LLC},
    'career' => q{dotCareer LLC},
    'careers' => q{Wild Corner, LLC},
    'cars' => q{Cars Registry Limited},
    'cartier' => q{Richemont DNS Inc.},
    'casa' => q{Top Level Domain Holdings Limited},
    'cash' => q{Delta Lake, LLC},
    'casino' => q{Binky Sky, LLC},
    'cat' => q{Fundacio puntCAT},
    'catering' => q{New Falls. LLC},
    'cba' => q{COMMONWEALTH BANK OF AUSTRALIA},
    'cbn' => q{The Christian Broadcasting Network, Inc.},
    'cbre' => q{CBRE, Inc.},
    'cbs' => q{CBS Domains Inc.},
    'cc' => q{eNIC Cocos (Keeling) Islands Pty. - Ltd. d/b/a Island Internet Services},
    'cd' => q{Office Congolais des Postes et Télécommunications - OCPT},
    'ceb' => q{The Corporate Executive Board Company},
    'center' => q{Tin Mill, LLC},
    'ceo' => q{CEOTLD Pty Ltd},
    'cern' => q{European Organization for Nuclear Research (&quot;CERN&quot;)},
    'cf' => q{Societe Centrafricaine de Telecommunications (SOCATEL)},
    'cfa' => q{CFA Institute},
    'cfd' => q{DOTCFD REGISTRY LTD},
    'cg' => q{ONPT Congo and Interpoint Switzerland},
    'ch' => q{SWITCH The Swiss Education &amp; Research Network},
    'chanel' => q{Chanel International B.V.},
    'channel' => q{Charleston Road Registry Inc.},
    'chase' => q{JPMorgan Chase Bank, National Association},
    'chat' => q{Sand Fields, LLC},
    'cheap' => q{Sand Cover, LLC},
    'chintai' => q{CHINTAI Corporation},
    'chloe' => q{Richemont DNS Inc.},
    'christmas' => q{Uniregistry, Corp.},
    'chrome' => q{Charleston Road Registry Inc.},
    'chrysler' => q{FCA US LLC.},
    'church' => q{Holly Fileds, LLC},
    'ci' => q{INP-HB Institut National Polytechnique Felix Houphouet Boigny},
    'cipriani' => q{Hotel Cipriani Srl},
    'circle' => q{Amazon Registry Services, Inc.},
    'cisco' => q{Cisco Technology, Inc.},
    'citadel' => q{Citadel Domain LLC},
    'citi' => q{Citigroup Inc.},
    'citic' => q{CITIC Group Corporation},
    'city' => q{Snow Sky, LLC},
    'cityeats' => q{Lifestyle Domain Holdings, Inc.},
    'ck' => q{Telecom Cook Islands Ltd.},
    'cl' => q{NIC Chile (University of Chile)},
    'claims' => q{Black Corner, LLC},
    'cleaning' => q{Fox Shadow, LLC},
    'click' => q{Uniregistry, Corp.},
    'clinic' => q{Goose Park, LLC},
    'clinique' => q{The Estée Lauder Companies Inc.},
    'clothing' => q{Steel Lake, LLC},
    'cloud' => q{ARUBA PEC S.p.A.},
    'club' => q{.CLUB DOMAINS, LLC},
    'clubmed' => q{Club Méditerranée S.A.},
    'cm' => q{Cameroon Telecommunications (CAMTEL)},
    'cn' => q{China Internet Network Information Center (CNNIC)},
    'co' => q{.CO Internet S.A.S.},
    'coach' => q{Koko Island, LLC},
    'codes' => q{Puff Willow, LLC},
    'coffee' => q{Trixy Cover, LLC},
    'college' => q{XYZ.COM LLC},
    'cologne' => q{NetCologne Gesellschaft für Telekommunikation mbH},
    'com' => q{VeriSign Global Registry Services},
    'comcast' => q{Comcast IP Holdings I, LLC},
    'commbank' => q{COMMONWEALTH BANK OF AUSTRALIA},
    'community' => q{Fox Orchard, LLC},
    'company' => q{Silver Avenue, LLC},
    'compare' => q{iSelect Ltd},
    'computer' => q{Pine Mill, LLC},
    'comsec' => q{VeriSign, Inc.},
    'condos' => q{Pine House, LLC},
    'construction' => q{Fox Dynamite, LLC},
    'consulting' => q{United TLD Holdco, LTD.},
    'contact' => q{Top Level Spectrum, Inc.},
    'contractors' => q{Magic Woods, LLC},
    'cooking' => q{Top Level Domain Holdings Limited},
    'cookingchannel' => q{Lifestyle Domain Holdings, Inc.},
    'cool' => q{Koko Lake, LLC},
    'coop' => q{DotCooperation LLC},
    'corsica' => q{Collectivité Territoriale de Corse},
    'country' => q{Top Level Domain Holdings Limited},
    'coupon' => q{Amazon Registry Services, Inc.},
    'coupons' => q{Black Island, LLC},
    'courses' => q{OPEN UNIVERSITIES AUSTRALIA PTY LTD},
    'cr' => q{National Academy of Sciences - Academia Nacional de Ciencias},
    'credit' => q{Snow Shadow, LLC},
    'creditcard' => q{Binky Frostbite, LLC},
    'creditunion' => q{CUNA Performance Resources, LLC},
    'cricket' => q{dot Cricket Limited},
    'crown' => q{Crown Equipment Corporation},
    'crs' => q{Federated Co-operatives Limited},
    'cruises' => q{Spring Way, LLC},
    'csc' => q{Alliance-One Services, Inc.},
    'cu' => q{CENIAInternet - Industria y San Jose
    Capitolio Nacional},
    'cuisinella' => q{SALM S.A.S.},
    'cv' => q{Agência Nacional das Comunicações (ANAC)},
    'cw' => q{University of Curacao},
    'cx' => q{Christmas Island Domain Administration Limited},
    'cy' => q{University of Cyprus},
    'cymru' => q{Nominet UK},
    'cyou' => q{Beijing Gamease Age Digital Technology Co., Ltd.},
    'cz' => q{CZ.NIC, z.s.p.o},
    'dabur' => q{Dabur India Limited},
    'dad' => q{Charleston Road Registry Inc.},
    'dance' => q{United TLD Holdco Ltd.},
    'date' => q{dot Date Limited},
    'dating' => q{Pine Fest, LLC},
    'datsun' => q{NISSAN MOTOR CO., LTD.},
    'day' => q{Charleston Road Registry Inc.},
    'dclk' => q{Charleston Road Registry Inc.},
    'dds' => q{Minds + Machines Group Limited},
    'de' => q{DENIC eG},
    'deal' => q{Amazon Registry Services, Inc.},
    'dealer' => q{Dealer Dot Com, Inc.},
    'deals' => q{Sand Sunset, LLC},
    'degree' => q{United TLD Holdco, Ltd},
    'delivery' => q{Steel Station, LLC},
    'dell' => q{Dell Inc.},
    'deloitte' => q{Deloitte Touche Tohmatsu},
    'delta' => q{Delta Air Lines, Inc.},
    'democrat' => q{United TLD Holdco Ltd.},
    'dental' => q{Tin Birch, LLC},
    'dentist' => q{United TLD Holdco, Ltd},
    'desi' => q{Desi Networks LLC},
    'design' => q{Top Level Design, LLC},
    'dev' => q{Charleston Road Registry Inc.},
    'dhl' => q{Deutsche Post AG},
    'diamonds' => q{John Edge, LLC},
    'diet' => q{Uniregistry, Corp.},
    'digital' => q{Dash Park, LLC},
    'direct' => q{Half Trail, LLC},
    'directory' => q{Extra Madison, LLC},
    'discount' => q{Holly Hill, LLC},
    'discover' => q{Discover Financial Services},
    'dish' => q{Dish DBS Corporation},
    'diy' => q{Lifestyle Domain Holdings, Inc.},
    'dj' => q{Djibouti Telecom S.A},
    'dk' => q{Dansk Internet Forum},
    'dm' => q{DotDM Corporation},
    'dnp' => q{Dai Nippon Printing Co., Ltd.},
    'do' => q{Pontificia Universidad Catolica Madre y Maestra - Recinto Santo Tomas de Aquino},
    'docs' => q{Charleston Road Registry Inc.},
    'doctor' => q{Brice Trail, LLC},
    'dodge' => q{FCA US LLC.},
    'dog' => q{Koko Mill, LLC},
    'doha' => q{Communications Regulatory Authority (CRA)},
    'domains' => q{Sugar Cross, LLC},
    'doosan' => q{Retired},
    'dot' => q{Dish DBS Corporation},
    'download' => q{dot Support Limited},
    'drive' => q{Charleston Road Registry Inc.},
    'dtv' => q{Dish DBS Corporation},
    'dubai' => q{Dubai Smart Government Department},
    'duck' => q{Johnson Shareholdings, Inc.},
    'dunlop' => q{The Goodyear Tire &amp; Rubber Company},
    'duns' => q{The Dun &amp; Bradstreet Corporation},
    'dupont' => q{E. I. du Pont de Nemours and Company},
    'durban' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'dvag' => q{Deutsche Vermögensberatung Aktiengesellschaft DVAG},
    'dz' => q{CERIST},
    'earth' => q{Interlink Co., Ltd.},
    'eat' => q{Charleston Road Registry Inc.},
    'ec' => q{NIC.EC (NICEC) S.A.},
    'eco' => q{Big Room Inc.},
    'edeka' => q{EDEKA Verband kaufmännischer Genossenschaften e.V.},
    'edu' => q{EDUCAUSE},
    'education' => q{Brice Way, LLC},
    'ee' => q{Eesti Interneti Sihtasutus (EIS)},
    'eg' => q{Egyptian Universities Network (EUN) - Supreme Council of Universities},
    'eh' => q{Not assigned},
    'email' => q{Spring Madison, LLC},
    'emerck' => q{Merck KGaA},
    'energy' => q{Binky Birch, LLC},
    'engineer' => q{United TLD Holdco Ltd.},
    'engineering' => q{Romeo Canyon},
    'enterprises' => q{Snow Oaks, LLC},
    'epost' => q{Deutsche Post AG},
    'epson' => q{Seiko Epson Corporation},
    'equipment' => q{Corn Station, LLC},
    'er' => q{Eritrea Telecommunication Services Corporation (EriTel)},
    'ericsson' => q{Telefonaktiebolaget L M Ericsson},
    'erni' => q{ERNI Group Holding AG},
    'es' => q{Red.es},
    'esq' => q{Charleston Road Registry Inc.},
    'estate' => q{Trixy Park, LLC},
    'esurance' => q{Esurance Insurance Company},
    'et' => q{Ethio telecom},
    'eu' => q{EURid vzw/asbl},
    'eurovision' => q{European Broadcasting Union (EBU)},
    'eus' => q{Puntueus Fundazioa},
    'events' => q{Pioneer Maple, LLC},
    'everbank' => q{EverBank},
    'exchange' => q{Spring Falls, LLC},
    'expert' => q{Magic Pass, LLC},
    'exposed' => q{Victor Beach, LLC},
    'express' => q{Sea Sunset, LLC},
    'extraspace' => q{Extra Space Storage LLC},
    'fage' => q{Fage International S.A.},
    'fail' => q{Atomic Pipe, LLC},
    'fairwinds' => q{FairWinds Partners, LLC},
    'faith' => q{dot Faith Limited},
    'family' => q{United TLD Holdco Ltd.},
    'fan' => q{Asiamix Digital Ltd},
    'fans' => q{Asiamix Digital Limited},
    'farm' => q{Just Maple, LLC},
    'farmers' => q{Farmers Insurance Exchange},
    'fashion' => q{Top Level Domain Holdings Limited},
    'fast' => q{Amazon Registry Services, Inc.},
    'fedex' => q{Federal Express Corporation},
    'feedback' => q{Top Level Spectrum, Inc.},
    'ferrari' => q{Fiat Chrysler Automobiles N.V.},
    'ferrero' => q{Ferrero Trading Lux S.A.},
    'fi' => q{Finnish Communications Regulatory Authority},
    'fiat' => q{Fiat Chrysler Automobiles N.V.},
    'fidelity' => q{Fidelity Brokerage Services LLC},
    'fido' => q{Rogers Communications Partnership},
    'film' => q{Motion Picture Domain Registry Pty Ltd},
    'final' => q{Núcleo de Informação e Coordenação do Ponto BR - NIC.br},
    'finance' => q{Cotton Cypress, LLC},
    'financial' => q{Just Cover, LLC},
    'fire' => q{Amazon Registry Services, Inc.},
    'firestone' => q{Bridgestone Licensing Services, Inc.},
    'firmdale' => q{Firmdale Holdings Limited},
    'fish' => q{Fox Woods, LLC},
    'fishing' => q{Top Level Domain Holdings Limited},
    'fit' => q{Minds + Machines Group Limited},
    'fitness' => q{Brice Orchard, LLC},
    'fj' => q{The University of the South Pacific - IT Services},
    'fk' => q{Falkland Islands Government},
    'flickr' => q{Yahoo! Domain Services Inc.},
    'flights' => q{Fox Station, LLC},
    'flir' => q{FLIR Systems, Inc.},
    'florist' => q{Half Cypress, LLC},
    'flowers' => q{Uniregistry, Corp.},
    'flsmidth' => q{Retired},
    'fly' => q{Charleston Road Registry Inc.},
    'fm' => q{FSM Telecommunications Corporation},
    'fo' => q{FO Council},
    'foo' => q{Charleston Road Registry Inc.},
    'foodnetwork' => q{Lifestyle Domain Holdings, Inc.},
    'football' => q{Foggy Farms, LLC},
    'ford' => q{Ford Motor Company},
    'forex' => q{DOTFOREX REGISTRY LTD},
    'forsale' => q{United TLD Holdco, LLC},
    'forum' => q{Fegistry, LLC},
    'foundation' => q{John Dale, LLC},
    'fox' => q{FOX Registry, LLC},
    'fr' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'fresenius' => q{Fresenius Immobilien-Verwaltungs-GmbH},
    'frl' => q{FRLregistry B.V.},
    'frogans' => q{OP3FT},
    'frontdoor' => q{Lifestyle Domain Holdings, Inc.},
    'frontier' => q{Frontier Communications Corporation},
    'ftr' => q{Frontier Communications Corporation},
    'fujitsu' => q{Fujitsu Limited},
    'fujixerox' => q{Xerox DNHC LLC},
    'fund' => q{John Castle, LLC},
    'furniture' => q{Lone Fields, LLC},
    'futbol' => q{United TLD Holdco, Ltd.},
    'fyi' => q{Silver Tigers, LLC},
    'ga' => q{Agence Nationale des Infrastructures Numériques et des Fréquences (ANINF)},
    'gal' => q{Asociación puntoGAL},
    'gallery' => q{Sugar House, LLC},
    'gallo' => q{Gallo Vineyards, Inc.},
    'gallup' => q{Gallup, Inc.},
    'game' => q{Uniregistry, Corp.},
    'games' => q{United TLD Holdco Ltd.},
    'gap' => q{The Gap, Inc.},
    'garden' => q{Top Level Domain Holdings Limited},
    'gb' => q{Reserved Domain - IANA},
    'gbiz' => q{Charleston Road Registry Inc.},
    'gd' => q{The National Telecommunications Regulatory Commission (NTRC)},
    'gdn' => q{Joint Stock Company &quot;Navigation-information systems&quot;},
    'ge' => q{Caucasus Online},
    'gea' => q{GEA Group Aktiengesellschaft},
    'gent' => q{Combell nv},
    'genting' => q{Resorts World Inc. Pte. Ltd.},
    'george' => q{Wal-Mart Stores, Inc.},
    'gf' => q{Net Plus},
    'gg' => q{Island Networks Ltd.},
    'ggee' => q{GMO Internet, Inc.},
    'gh' => q{Network Computer Systems Limited},
    'gi' => q{Sapphire Networks},
    'gift' => q{Uniregistry, Corp.},
    'gifts' => q{Goose Sky, LLC},
    'gives' => q{United TLD Holdco Ltd.},
    'giving' => q{Giving Limited},
    'gl' => q{TELE Greenland A/S},
    'glade' => q{Johnson Shareholdings, Inc.},
    'glass' => q{Black Cover, LLC},
    'gle' => q{Charleston Road Registry Inc.},
    'global' => q{Dot Global Domain Registry Limited},
    'globo' => q{Globo Comunicação e Participações S.A},
    'gm' => q{GM-NIC},
    'gmail' => q{Charleston Road Registry Inc.},
    'gmbh' => q{Extra Dynamite, LLC},
    'gmo' => q{GMO Internet, Inc.},
    'gmx' => q{1&amp;1 Mail &amp; Media GmbH},
    'gn' => q{Centre National des Sciences Halieutiques de Boussoura},
    'godaddy' => q{Go Daddy East, LLC},
    'gold' => q{June Edge, LLC},
    'goldpoint' => q{YODOBASHI CAMERA CO.,LTD.},
    'golf' => q{Lone Falls, LLC},
    'goo' => q{NTT Resonant Inc.},
    'goodhands' => q{Allstate Fire and Casualty Insurance Company},
    'goodyear' => q{The Goodyear Tire &amp; Rubber Company},
    'goog' => q{Charleston Road Registry Inc.},
    'google' => q{Charleston Road Registry Inc.},
    'gop' => q{Republican State Leadership Committee, Inc.},
    'got' => q{Amazon Registry Services, Inc.},
    'gov' => q{General Services Administration - Attn: QTDC, 2E08 (.gov Domain Registration)},
    'gp' => q{Networking Technologies Group},
    'gq' => q{GETESA},
    'gr' => q{ICS-FORTH GR},
    'grainger' => q{Grainger Registry Services, LLC},
    'graphics' => q{Over Madison, LLC},
    'gratis' => q{Pioneer Tigers, LLC},
    'green' => q{Afilias Limited},
    'gripe' => q{Corn Sunset, LLC},
    'group' => q{Romeo Town, LLC},
    'gs' => q{Government of South Georgia and South Sandwich Islands (GSGSSI)},
    'gt' => q{Universidad del Valle de Guatemala},
    'gu' => q{University of Guam - Computer Center},
    'guardian' => q{The Guardian Life Insurance Company of America},
    'gucci' => q{Guccio Gucci S.p.a.},
    'guge' => q{Charleston Road Registry Inc.},
    'guide' => q{Snow Moon, LLC},
    'guitars' => q{Uniregistry, Corp.},
    'guru' => q{Pioneer Cypress, LLC},
    'gw' => q{Autoridade Reguladora Nacional - Tecnologias de Informação e Comunicação da Guiné-Bissau},
    'gy' => q{University of Guyana},
    'hamburg' => q{Hamburg Top-Level-Domain GmbH},
    'hangout' => q{Charleston Road Registry Inc.},
    'haus' => q{United TLD Holdco, LTD.},
    'hbo' => q{HBO Registry Services, Inc.},
    'hdfc' => q{HOUSING DEVELOPMENT FINANCE CORPORATION LIMITED},
    'hdfcbank' => q{HDFC Bank Limited},
    'health' => q{DotHealth, LLC},
    'healthcare' => q{Silver Glen, LLC},
    'help' => q{Uniregistry, Corp.},
    'helsinki' => q{City of Helsinki},
    'here' => q{Charleston Road Registry Inc.},
    'hermes' => q{Hermes International},
    'hgtv' => q{Lifestyle Domain Holdings, Inc.},
    'hiphop' => q{Uniregistry, Corp.},
    'hisamitsu' => q{Hisamitsu Pharmaceutical Co.,Inc.},
    'hitachi' => q{Hitachi, Ltd.},
    'hiv' => q{Uniregistry, Corp.},
    'hk' => q{Hong Kong Internet Registration Corporation Ltd.},
    'hkt' => q{PCCW-HKT DataCom Services Limited},
    'hm' => q{HM Domain Registry},
    'hn' => q{Red de Desarrollo Sostenible Honduras},
    'hockey' => q{Half Willow, LLC},
    'holdings' => q{John Madison, LLC},
    'holiday' => q{Goose Woods, LLC},
    'homedepot' => q{Homer TLC, Inc.},
    'homegoods' => q{The TJX Companies, Inc.},
    'homes' => q{DERHomes, LLC},
    'homesense' => q{The TJX Companies, Inc.},
    'honda' => q{Honda Motor Co., Ltd.},
    'honeywell' => q{Honeywell GTLD LLC},
    'horse' => q{Top Level Domain Holdings Limited},
    'host' => q{DotHost Inc.},
    'hosting' => q{Uniregistry, Corp.},
    'hot' => q{Amazon Registry Services, Inc.},
    'hoteles' => q{Travel Reservations SRL},
    'hotmail' => q{Microsoft Corporation},
    'house' => q{Sugar Park, LLC},
    'how' => q{Charleston Road Registry Inc.},
    'hr' => q{CARNet - Croatian Academic and Research Network},
    'hsbc' => q{HSBC Holdings PLC},
    'ht' => q{Consortium FDS/RDDH},
    'htc' => q{HTC corporation},
    'hu' => q{Council of Hungarian Internet Providers (CHIP)},
    'hughes' => q{Hughes Satellite Systems Corporation},
    'hyatt' => q{Hyatt GTLD, L.L.C.},
    'hyundai' => q{Hyundai Motor Company},
    'ibm' => q{International Business Machines Corporation},
    'icbc' => q{Industrial and Commercial Bank of China Limited},
    'ice' => q{IntercontinentalExchange, Inc.},
    'icu' => q{One.com A/S},
    'id' => q{Perkumpulan Pengelola Nama Domain Internet Indonesia (PANDI)},
    'ie' => q{University College Dublin - Computing Services
    Computer Centre},
    'ieee' => q{IEEE Global LLC},
    'ifm' => q{ifm electronic gmbh},
    'iinet' => q{Connect West Pty. Ltd.},
    'ikano' => q{Ikano S.A.},
    'il' => q{Internet Society of Israel},
    'im' => q{Isle of Man Government},
    'imamat' => q{Fondation Aga Khan (Aga Khan Foundation)},
    'imdb' => q{Amazon Registry Services, Inc.},
    'immo' => q{Auburn Bloom, LLC},
    'immobilien' => q{United TLD Holdco Ltd.},
    'in' => q{National Internet Exchange of India},
    'industries' => q{Outer House, LLC},
    'infiniti' => q{NISSAN MOTOR CO., LTD.},
    'info' => q{Afilias Limited},
    'ing' => q{Charleston Road Registry Inc.},
    'ink' => q{Top Level Design, LLC},
    'institute' => q{Outer Maple, LLC},
    'insurance' => q{fTLD Registry Services LLC},
    'insure' => q{Pioneer Willow, LLC},
    'int' => q{Internet Assigned Numbers Authority},
    'intel' => q{Intel Corporation},
    'international' => q{Wild Way, LLC},
    'intuit' => q{Intuit Administrative Services, Inc.},
    'investments' => q{Holly Glen, LLC},
    'io' => q{IO Top Level Domain Registry - Cable and Wireless},
    'ipiranga' => q{Ipiranga Produtos de Petroleo S.A.},
    'iq' => q{Communications and Media Commission (CMC)},
    'ir' => q{Institute for Research in Fundamental Sciences},
    'irish' => q{Dot-Irish LLC},
    'is' => q{ISNIC - Internet Iceland ltd.},
    'iselect' => q{iSelect Ltd},
    'ismaili' => q{Fondation Aga Khan (Aga Khan Foundation)},
    'ist' => q{Istanbul Metropolitan Municipality},
    'istanbul' => q{Istanbul Metropolitan Municipality},
    'it' => q{IIT - CNR},
    'itau' => q{Itau Unibanco Holding S.A.},
    'itv' => q{ITV Services Limited},
    'iwc' => q{Richemont DNS Inc.},
    'jaguar' => q{Jaguar Land Rover Ltd},
    'java' => q{Oracle Corporation},
    'jcb' => q{JCB Co., Ltd.},
    'jcp' => q{JCP Media, Inc.},
    'je' => q{Island Networks (Jersey) Ltd.},
    'jeep' => q{FCA US LLC.},
    'jetzt' => q{Wild Frostbite, LLC},
    'jewelry' => q{Wild Bloom, LLC},
    'jlc' => q{Richemont DNS Inc.},
    'jll' => q{Jones Lang LaSalle Incorporated},
    'jm' => q{University of West Indies},
    'jmp' => q{Matrix IP LLC},
    'jnj' => q{Johnson &amp; Johnson Services, Inc.},
    'jo' => q{National Information Technology Center (NITC)},
    'jobs' => q{Employ Media LLC},
    'joburg' => q{ZA Central Registry NPC trading as ZA Central Registry},
    'jot' => q{Amazon Registry Services, Inc.},
    'joy' => q{Amazon Registry Services, Inc.},
    'jp' => q{Japan Registry Services Co., Ltd.},
    'jpmorgan' => q{JPMorgan Chase Bank, National Association},
    'jprs' => q{Japan Registry Services Co., Ltd.},
    'juegos' => q{Uniregistry, Corp.},
    'juniper' => q{JUNIPER NETWORKS, INC.},
    'kaufen' => q{United TLD Holdco Ltd.},
    'kddi' => q{KDDI CORPORATION},
    'ke' => q{Kenya Network Information Center (KeNIC)},
    'kerryhotels' => q{Kerry Trading Co. Limited},
    'kerrylogistics' => q{Kerry Trading Co. Limited},
    'kerryproperties' => q{Kerry Trading Co. Limited},
    'kfh' => q{Kuwait Finance House},
    'kg' => q{AsiaInfo Telecommunication Enterprise},
    'kh' => q{Telecommunication Regulator of Cambodia (TRC)},
    'ki' => q{Ministry of Communications, Transport, and Tourism Development},
    'kia' => q{KIA MOTORS CORPORATION},
    'kim' => q{Afilias Limited},
    'kinder' => q{Ferrero Trading Lux S.A.},
    'kindle' => q{Amazon Registry Services, Inc.},
    'kitchen' => q{Just Goodbye, LLC},
    'kiwi' => q{DOT KIWI LIMITED},
    'km' => q{Comores Telecom},
    'kn' => q{Ministry of Finance, Sustainable Development Information &amp; Technology},
    'koeln' => q{NetCologne Gesellschaft für Telekommunikation mbH},
    'komatsu' => q{Komatsu Ltd.},
    'kosher' => q{Kosher Marketing Assets LLC},
    'kp' => q{Star Joint Venture Company},
    'kpmg' => q{KPMG International Cooperative (KPMG International Genossenschaft)},
    'kpn' => q{Koninklijke KPN N.V.},
    'kr' => q{Korea Internet &amp; Security Agency (KISA)},
    'krd' => q{KRG Department of Information Technology},
    'kred' => q{KredTLD Pty Ltd},
    'kuokgroup' => q{Kerry Trading Co. Limited},
    'kw' => q{Ministry of Communications},
    'ky' => q{The Information and Communications Technology Authority},
    'kyoto' => q{Academic Institution: Kyoto Jyoho Gakuen},
    'kz' => q{Association of IT Companies of Kazakhstan},
    'la' => q{Lao National Internet Committee (LANIC), Ministry of Posts and Telecommunications},
    'lacaixa' => q{CAIXA D&#39;ESTALVIS I PENSIONS DE BARCELONA},
    'ladbrokes' => q{LADBROKES INTERNATIONAL PLC},
    'lamborghini' => q{Automobili Lamborghini S.p.A.},
    'lamer' => q{The Estée Lauder Companies Inc.},
    'lancaster' => q{LANCASTER},
    'lancia' => q{Fiat Chrysler Automobiles N.V.},
    'lancome' => q{L&#39;Oréal},
    'land' => q{Pine Moon, LLC},
    'landrover' => q{Jaguar Land Rover Ltd},
    'lanxess' => q{LANXESS Corporation},
    'lasalle' => q{Jones Lang LaSalle Incorporated},
    'lat' => q{ECOM-LAC Federación de Latinoamérica y el Caribe para Internet y el Comercio Electrónico},
    'latino' => q{Dish DBS Corporation},
    'latrobe' => q{La Trobe University},
    'law' => q{Minds + Machines Group Limited},
    'lawyer' => q{United TLD Holdco, Ltd},
    'lb' => q{American University of Beirut - Computing and Networking Services},
    'lc' => q{University of Puerto Rico},
    'lds' => q{IRI Domain Management, LLC},
    'lease' => q{Victor Trail, LLC},
    'leclerc' => q{A.C.D. LEC Association des Centres Distributeurs Edouard Leclerc},
    'lefrak' => q{LeFrak Organization, Inc.},
    'legal' => q{Blue Falls, LLC},
    'lego' => q{LEGO Juris A/S},
    'lexus' => q{TOYOTA MOTOR CORPORATION},
    'lgbt' => q{Afilias Limited},
    'li' => q{Universitaet Liechtenstein},
    'liaison' => q{Liaison Technologies, Incorporated},
    'lidl' => q{Schwarz Domains und Services GmbH &amp; Co. KG},
    'life' => q{Trixy Oaks, LLC},
    'lifeinsurance' => q{American Council of Life Insurers},
    'lifestyle' => q{Lifestyle Domain Holdings, Inc.},
    'lighting' => q{John McCook, LLC},
    'like' => q{Amazon Registry Services, Inc.},
    'lilly' => q{Eli Lilly and Company},
    'limited' => q{Big Fest, LLC},
    'limo' => q{Hidden Frostbite, LLC},
    'lincoln' => q{Ford Motor Company},
    'linde' => q{Linde Aktiengesellschaft},
    'link' => q{Uniregistry, Corp.},
    'lipsy' => q{Lipsy Ltd},
    'live' => q{United TLD Holdco Ltd.},
    'living' => q{Lifestyle Domain Holdings, Inc.},
    'lixil' => q{LIXIL Group Corporation},
    'lk' => q{Council for Information Technology - LK Domain Registrar},
    'loan' => q{dot Loan Limited},
    'loans' => q{June Woods, LLC},
    'locker' => q{Dish DBS Corporation},
    'locus' => q{Locus Analytics LLC},
    'loft' => q{Annco, Inc.},
    'lol' => q{Uniregistry, Corp.},
    'london' => q{Dot London Domains Limited},
    'lotte' => q{Lotte Holdings Co., Ltd.},
    'lotto' => q{Afilias Limited},
    'love' => q{Merchant Law Group LLP},
    'lpl' => q{LPL Holdings, Inc.},
    'lplfinancial' => q{LPL Holdings, Inc.},
    'lr' => q{Data Technology Solutions, Inc.},
    'ls' => q{National University of Lesotho},
    'lt' => q{Kaunas University of Technology},
    'ltd' => q{Over Corner, LLC},
    'ltda' => q{InterNetX Corp.},
    'lu' => q{RESTENA},
    'lundbeck' => q{H. Lundbeck A/S},
    'lupin' => q{LUPIN LIMITED},
    'luxe' => q{Top Level Domain Holdings Limited},
    'luxury' => q{Luxury Partners LLC},
    'lv' => q{University of Latvia - Institute of Mathematics and Computer Science
    Department of Network Solutions (DNS)},
    'ly' => q{General Post and Telecommunication Company},
    'ma' => q{Agence Nationale de Réglementation des Télécommunications (ANRT)},
    'macys' => q{Macys, Inc.},
    'madrid' => q{Comunidad de Madrid},
    'maif' => q{Mutuelle Assurance Instituteur France (MAIF)},
    'maison' => q{Victor Frostbite, LLC},
    'makeup' => q{L&#39;Oréal},
    'man' => q{MAN SE},
    'management' => q{John Goodbye, LLC},
    'mango' => q{PUNTO FA S.L.},
    'market' => q{United TLD Holdco, Ltd},
    'marketing' => q{Fern Pass, LLC},
    'markets' => q{DOTMARKETS REGISTRY LTD},
    'marriott' => q{Marriott Worldwide Corporation},
    'marshalls' => q{The TJX Companies, Inc.},
    'maserati' => q{Fiat Chrysler Automobiles N.V.},
    'mattel' => q{Mattel Sites, Inc.},
    'mba' => q{Lone Hollow, LLC},
    'mc' => q{Gouvernement de Monaco - Direction des Communications Electroniques},
    'mcd' => q{McDonald’s Corporation},
    'mcdonalds' => q{McDonald’s Corporation},
    'mckinsey' => q{McKinsey Holdings, Inc.},
    'md' => q{MoldData S.E.},
    'me' => q{Government of Montenegro},
    'med' => q{Medistry LLC},
    'media' => q{Grand Glen, LLC},
    'meet' => q{Charleston Road Registry Inc.},
    'melbourne' => q{The Crown in right of the State of Victoria, represented by its Department of State Development, Business and Innovation},
    'meme' => q{Charleston Road Registry Inc.},
    'memorial' => q{Dog Beach, LLC},
    'men' => q{Exclusive Registry Limited},
    'menu' => q{Wedding TLD2, LLC},
    'meo' => q{PT Comunicacoes S.A.},
    'metlife' => q{MetLife Services and Solutions, LLC},
    'mf' => q{Not assigned},
    'mg' => q{NIC-MG (Network Information Center Madagascar)},
    'mh' => q{Office of the Cabinet},
    'miami' => q{Top Level Domain Holdings Limited},
    'microsoft' => q{Microsoft Corporation},
    'mil' => q{DoD Network Information Center},
    'mini' => q{Bayerische Motoren Werke Aktiengesellschaft},
    'mint' => q{Intuit Administrative Services, Inc.},
    'mit' => q{Massachusetts Institute of Technology},
    'mitsubishi' => q{Mitsubishi Corporation},
    'mk' => q{Macedonian Academic Research Network Skopje},
    'ml' => q{Agence des Technologies de l’Information et de la Communication},
    'mlb' => q{MLB Advanced Media DH, LLC},
    'mls' => q{The Canadian Real Estate Association},
    'mm' => q{Ministry of Communications, Posts &amp; Telegraphs},
    'mma' => q{MMA IARD},
    'mn' => q{Datacom Co., Ltd.},
    'mo' => q{Bureau of Telecommunications Regulation (DSRT)},
    'mobi' => q{Afilias Technologies Limited dba dotMobi},
    'mobily' => q{GreenTech Consultancy Company W.L.L.},
    'moda' => q{United TLD Holdco Ltd.},
    'moe' => q{Interlink Co., Ltd.},
    'moi' => q{Amazon Registry Services, Inc.},
    'mom' => q{Uniregistry, Corp.},
    'monash' => q{Monash University},
    'money' => q{Outer McCook, LLC},
    'monster' => q{Monster Worldwide, Inc.},
    'montblanc' => q{Richemont DNS Inc.},
    'mopar' => q{FCA US LLC.},
    'mormon' => q{IRI Domain Management, LLC (&quot;Applicant&quot;)},
    'mortgage' => q{United TLD Holdco, Ltd},
    'moscow' => q{Foundation for Assistance for Internet Technologies and Infrastructure Development (FAITID)},
    'motorcycles' => q{DERMotorcycles, LLC},
    'mov' => q{Charleston Road Registry Inc.},
    'movie' => q{New Frostbite, LLC},
    'movistar' => q{Telefónica S.A.},
    'mp' => q{Saipan Datacom, Inc.},
    'mq' => q{MEDIASERV},
    'mr' => q{Université des Sciences, de Technologie et de Médecine},
    'ms' => q{MNI Networks Ltd.},
    'msd' => q{MSD Registry Holdings, Inc.},
    'mt' => q{NIC (Malta)},
    'mtn' => q{MTN Dubai Limited},
    'mtpc' => q{Mitsubishi Tanabe Pharma Corporation},
    'mtr' => q{MTR Corporation Limited},
    'mu' => q{Internet Direct Ltd},
    'museum' => q{Museum Domain Management Association},
    'mutual' => q{Northwestern Mutual MU TLD Registry, LLC},
    'mutuelle' => q{Fédération Nationale de la Mutualité Française},
    'mv' => q{Dhiraagu Pvt. Ltd. (DHIVEHINET)},
    'mw' => q{Malawi Sustainable Development Network Programme - (Malawi SDNP)},
    'mx' => q{NIC-Mexico - ITESM - Campus Monterrey},
    'my' => q{MYNIC Berhad},
    'mz' => q{Centro de Informatica de Universidade Eduardo Mondlane},
    'na' => q{Namibian Network Information Center},
    'nab' => q{National Australia Bank Limited},
    'nadex' => q{Nadex Domains, Inc},
    'nagoya' => q{GMO Registry, Inc.},
    'name' => q{VeriSign Information Services, Inc.},
    'nationwide' => q{Nationwide Mutual Insurance Company},
    'natura' => q{NATURA COSMÉTICOS S.A.},
    'navy' => q{United TLD Holdco Ltd.},
    'nba' => q{NBA REGISTRY, LLC},
    'nc' => q{Office des Postes et Telecommunications},
    'ne' => q{SONITEL},
    'nec' => q{NEC Corporation},
    'net' => q{VeriSign Global Registry Services},
    'netbank' => q{COMMONWEALTH BANK OF AUSTRALIA},
    'netflix' => q{Netflix, Inc.},
    'network' => q{Trixy Manor, LLC},
    'neustar' => q{NeuStar, Inc.},
    'new' => q{Charleston Road Registry Inc.},
    'news' => q{United TLD Holdco Ltd.},
    'next' => q{Next plc},
    'nextdirect' => q{Next plc},
    'nexus' => q{Charleston Road Registry Inc.},
    'nf' => q{Norfolk Island Data Services},
    'nfl' => q{NFL Reg Ops LLC},
    'ng' => q{Nigeria Internet Registration Association},
    'ngo' => q{Public Interest Registry},
    'nhk' => q{Japan Broadcasting Corporation (NHK)},
    'ni' => q{Universidad Nacional del Ingernieria - Centro de Computo},
    'nico' => q{DWANGO Co., Ltd.},
    'nike' => q{NIKE, Inc.},
    'nikon' => q{NIKON CORPORATION},
    'ninja' => q{United TLD Holdco Ltd.},
    'nissan' => q{NISSAN MOTOR CO., LTD.},
    'nissay' => q{Nippon Life Insurance Company},
    'nl' => q{SIDN (Stichting Internet  Domeinregistratie Nederland)},
    'no' => q{UNINETT Norid A/S},
    'nokia' => q{Nokia Corporation},
    'northwesternmutual' => q{Northwestern Mutual Registry, LLC},
    'norton' => q{Symantec Corporation},
    'now' => q{Amazon Registry Services, Inc.},
    'nowruz' => q{Asia Green IT System Bilgisayar San. ve Tic. Ltd. Sti.},
    'nowtv' => q{Starbucks (HK) Limited},
    'np' => q{Mercantile Communications Pvt. Ltd.},
    'nr' => q{CENPAC NET},
    'nra' => q{NRA Holdings Company, INC.},
    'nrw' => q{Minds + Machines GmbH},
    'ntt' => q{NIPPON TELEGRAPH AND TELEPHONE CORPORATION},
    'nu' => q{The IUSN Foundation},
    'nyc' => q{The City of New York by and through the New York City Department of Information Technology &amp; Telecommunications},
    'nz' => q{InternetNZ},
    'obi' => q{OBI Group Holding SE &amp; Co. KGaA},
    'off' => q{Johnson Shareholdings, Inc.},
    'office' => q{Microsoft Corporation},
    'okinawa' => q{BRregistry, Inc.},
    'olayan' => q{Crescent Holding GmbH},
    'olayangroup' => q{Crescent Holding GmbH},
    'oldnavy' => q{The Gap, Inc.},
    'ollo' => q{Dish DBS Corporation},
    'om' => q{Telecommunications Regulatory Authority (TRA)},
    'omega' => q{The Swatch Group Ltd},
    'one' => q{One.com A/S},
    'ong' => q{Public Interest Registry},
    'onl' => q{I-REGISTRY Ltd., Niederlassung Deutschland},
    'online' => q{DotOnline Inc.},
    'onyourside' => q{Nationwide Mutual Insurance Company},
    'ooo' => q{INFIBEAM INCORPORATION LIMITED},
    'open' => q{American Express Travel Related Services Company, Inc.},
    'oracle' => q{Oracle Corporation},
    'orange' => q{Orange Brand Services Limited},
    'org' => q{Public Interest Registry (PIR)},
    'organic' => q{Afilias Limited},
    'orientexpress' => q{Orient Express},
    'origins' => q{The Estée Lauder Companies Inc.},
    'osaka' => q{Interlink Co., Ltd.},
    'otsuka' => q{Otsuka Holdings Co., Ltd.},
    'ott' => q{Dish DBS Corporation},
    'ovh' => q{OVH SAS},
    'pa' => q{Universidad Tecnologica de Panama},
    'page' => q{Charleston Road Registry Inc.},
    'pamperedchef' => q{The Pampered Chef, Ltd.},
    'panasonic' => q{Panasonic Corporation},
    'panerai' => q{Richemont DNS Inc.},
    'paris' => q{City of Paris},
    'pars' => q{Asia Green IT System Bilgisayar San. ve Tic. Ltd. Sti.},
    'partners' => q{Magic Glen, LLC},
    'parts' => q{Sea Goodbye, LLC},
    'party' => q{Blue Sky Registry Limited},
    'passagens' => q{Travel Reservations SRL},
    'pay' => q{Amazon Registry Services, Inc.},
    'pccw' => q{PCCW Enterprises Limited},
    'pe' => q{Red Cientifica Peruana},
    'pet' => q{Afilias plc},
    'pf' => q{Gouvernement de la Polynésie française},
    'pfizer' => q{Pfizer Inc.},
    'pg' => q{PNG DNS Administration - Vice Chancellors Office
    The Papua New Guinea University of Technology},
    'ph' => q{PH Domain Foundation},
    'pharmacy' => q{National Association of Boards of Pharmacy},
    'philips' => q{Koninklijke Philips N.V.},
    'photo' => q{Uniregistry, Corp.},
    'photography' => q{Sugar Glen, LLC},
    'photos' => q{Sea Corner, LLC},
    'physio' => q{PhysBiz Pty Ltd},
    'piaget' => q{Richemont DNS Inc.},
    'pics' => q{Uniregistry, Corp.},
    'pictet' => q{Pictet Europe S.A.},
    'pictures' => q{Foggy Sky, LLC},
    'pid' => q{Top Level Spectrum, Inc.},
    'pin' => q{Amazon Registry Services, Inc.},
    'ping' => q{Ping Registry Provider, Inc.},
    'pink' => q{Afilias Limited},
    'pioneer' => q{Pioneer Corporation},
    'pizza' => q{Foggy Moon, LLC},
    'pk' => q{PKNIC},
    'pl' => q{Research and Academic Computer Network},
    'place' => q{Snow Galley, LLC},
    'play' => q{Charleston Road Registry Inc.},
    'playstation' => q{Sony Computer Entertainment Inc.},
    'plumbing' => q{Spring Tigers, LLC},
    'plus' => q{Sugar Mill, LLC},
    'pm' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'pn' => q{Pitcairn Island Administration},
    'pnc' => q{PNC Domain Co., LLC},
    'pohl' => q{Deutsche Vermögensberatung Aktiengesellschaft DVAG},
    'poker' => q{Afilias Domains No. 5 Limited},
    'politie' => q{Politie Nederland},
    'porn' => q{ICM Registry PN LLC},
    'post' => q{Universal Postal Union},
    'pr' => q{Gauss Research Laboratory Inc.},
    'pramerica' => q{Prudential Financial, Inc.},
    'praxi' => q{Praxi S.p.A.},
    'press' => q{DotPress Inc.},
    'prime' => q{Amazon Registry Services, Inc.},
    'pro' => q{Registry Services Corporation - dba RegistryPro},
    'prod' => q{Charleston Road Registry Inc.},
    'productions' => q{Magic Birch, LLC},
    'prof' => q{Charleston Road Registry Inc.},
    'progressive' => q{Progressive Casualty Insurance Company},
    'promo' => q{Afilias plc},
    'properties' => q{Big Pass, LLC},
    'property' => q{Uniregistry, Corp.},
    'protection' => q{XYZ.COM LLC},
    'pru' => q{Prudential Financial, Inc.},
    'prudential' => q{Prudential Financial, Inc.},
    'ps' => q{Ministry Of Telecommunications &amp; - Information Technology,
    Government Computer Center.},
    'pt' => q{Associação DNS.PT},
    'pub' => q{United TLD Holdco Ltd.},
    'pw' => q{Micronesia Investment and Development Corporation},
    'pwc' => q{PricewaterhouseCoopers LLP},
    'py' => q{NIC-PY},
    'qa' => q{Communications Regulatory Authority},
    'qpon' => q{dotCOOL, Inc.},
    'quebec' => q{PointQuébec Inc},
    'quest' => q{Quest ION Limited},
    'qvc' => q{QVC, Inc.},
    'racing' => q{Premier Registry Limited},
    'raid' => q{Johnson Shareholdings, Inc.},
    're' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'read' => q{Amazon Registry Services, Inc.},
    'realestate' => q{dotRealEstate LLC},
    'realtor' => q{Real Estate Domains LLC},
    'realty' => q{Fegistry, LLC},
    'recipes' => q{Grand Island, LLC},
    'red' => q{Afilias Limited},
    'redstone' => q{Redstone Haute Couture Co., Ltd.},
    'redumbrella' => q{Travelers TLD, LLC},
    'rehab' => q{United TLD Holdco Ltd.},
    'reise' => q{Foggy Way, LLC},
    'reisen' => q{New Cypress, LLC},
    'reit' => q{National Association of Real Estate Investment Trusts, Inc.},
    'ren' => q{Beijing Qianxiang Wangjing Technology Development Co., Ltd.},
    'rent' => q{XYZ.COM LLC},
    'rentals' => q{Big Hollow,LLC},
    'repair' => q{Lone Sunset, LLC},
    'report' => q{Binky Glen, LLC},
    'republican' => q{United TLD Holdco Ltd.},
    'rest' => q{Punto 2012 Sociedad Anonima Promotora de Inversion de Capital Variable},
    'restaurant' => q{Snow Avenue, LLC},
    'review' => q{dot Review Limited},
    'reviews' => q{United TLD Holdco, Ltd.},
    'rexroth' => q{Robert Bosch GMBH},
    'rich' => q{I-REGISTRY Ltd., Niederlassung Deutschland},
    'richardli' => q{Pacific Century Asset Management (HK) Limited},
    'ricoh' => q{Ricoh Company, Ltd.},
    'rightathome' => q{Johnson Shareholdings, Inc.},
    'rio' => q{Empresa Municipal de Informática SA - IPLANRIO},
    'rip' => q{United TLD Holdco Ltd.},
    'ro' => q{National Institute for R&amp;D in Informatics},
    'rocher' => q{Ferrero Trading Lux S.A.},
    'rocks' => q{United TLD Holdco, LTD.},
    'rodeo' => q{Top Level Domain Holdings Limited},
    'rogers' => q{Rogers Communications Partnership},
    'room' => q{Amazon Registry Services, Inc.},
    'rs' => q{Serbian National Internet Domain Registry (RNIDS)},
    'rsvp' => q{Charleston Road Registry Inc.},
    'ru' => q{Coordination Center for TLD RU},
    'ruhr' => q{regiodot GmbH &amp; Co. KG},
    'run' => q{Snow Park, LLC},
    'rw' => q{Rwanda Information Communication and Technology Association (RICTA)},
    'rwe' => q{RWE AG},
    'ryukyu' => q{BRregistry, Inc.},
    'sa' => q{Communications and Information Technology Commission},
    'saarland' => q{dotSaarland GmbH},
    'safe' => q{Amazon Registry Services, Inc.},
    'safety' => q{Safety Registry Services, LLC.},
    'sakura' => q{SAKURA Internet Inc.},
    'sale' => q{United TLD Holdco, Ltd},
    'salon' => q{Outer Orchard, LLC},
    'samsclub' => q{Wal-Mart Stores, Inc.},
    'samsung' => q{SAMSUNG SDS CO., LTD},
    'sandvik' => q{Sandvik AB},
    'sandvikcoromant' => q{Sandvik AB},
    'sanofi' => q{Sanofi},
    'sap' => q{SAP AG},
    'sapo' => q{PT Comunicacoes S.A.},
    'sarl' => q{Delta Orchard, LLC},
    'sas' => q{Research IP LLC},
    'save' => q{Amazon Registry Services, Inc.},
    'saxo' => q{Saxo Bank A/S},
    'sb' => q{Solomon Telekom Company Limited},
    'sbi' => q{STATE BANK OF INDIA},
    'sbs' => q{SPECIAL BROADCASTING SERVICE CORPORATION},
    'sc' => q{VCS Pty Ltd},
    'sca' => q{SVENSKA CELLULOSA AKTIEBOLAGET SCA (publ)},
    'scb' => q{The Siam Commercial Bank Public Company Limited (&quot;SCB&quot;)},
    'schaeffler' => q{Schaeffler Technologies AG &amp; Co. KG},
    'schmidt' => q{SALM S.A.S.},
    'scholarships' => q{Scholarships.com, LLC},
    'school' => q{Little Galley, LLC},
    'schule' => q{Outer Moon, LLC},
    'schwarz' => q{Schwarz Domains und Services GmbH &amp; Co. KG},
    'science' => q{dot Science Limited},
    'scjohnson' => q{Johnson Shareholdings, Inc.},
    'scor' => q{SCOR SE},
    'scot' => q{Dot Scot Registry Limited},
    'sd' => q{Sudan Internet Society},
    'se' => q{The Internet Infrastructure Foundation},
    'seat' => q{SEAT, S.A. (Sociedad Unipersonal)},
    'secure' => q{Amazon Registry Services, Inc.},
    'security' => q{XYZ.COM LLC},
    'seek' => q{Seek Limited},
    'select' => q{iSelect Ltd},
    'sener' => q{Sener Ingeniería y Sistemas, S.A.},
    'services' => q{Fox Castle, LLC},
    'ses' => q{SES},
    'seven' => q{Seven West Media Ltd},
    'sew' => q{SEW-EURODRIVE GmbH &amp; Co KG},
    'sex' => q{ICM Registry SX LLC},
    'sexy' => q{Uniregistry, Corp.},
    'sfr' => q{Societe Francaise du Radiotelephone - SFR},
    'sg' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'sh' => q{Government of St. Helena},
    'shangrila' => q{Shangri‐La International Hotel Management Limited},
    'sharp' => q{Sharp Corporation},
    'shaw' => q{Shaw Cablesystems G.P.},
    'shell' => q{Shell Information Technology International Inc},
    'shia' => q{Asia Green IT System Bilgisayar San. ve Tic. Ltd. Sti.},
    'shiksha' => q{Afilias Limited},
    'shoes' => q{Binky Galley, LLC},
    'shop' => q{GMO Registry, Inc.},
    'shopping' => q{Over Keep, LLC},
    'shouji' => q{QIHOO 360 TECHNOLOGY CO. LTD.},
    'show' => q{Snow Beach, LLC},
    'showtime' => q{CBS Domains Inc.},
    'shriram' => q{Shriram Capital Ltd.},
    'si' => q{Academic and Research Network of Slovenia (ARNES)},
    'silk' => q{Amazon Registry Services, Inc.},
    'sina' => q{Sina Corporation},
    'singles' => q{Fern Madison, LLC},
    'site' => q{DotSite Inc.},
    'sj' => q{UNINETT Norid A/S},
    'sk' => q{SK-NIC, a.s.},
    'ski' => q{STARTING DOT LIMITED},
    'skin' => q{L&#39;Oréal},
    'sky' => q{Sky International AG},
    'skype' => q{Microsoft Corporation},
    'sl' => q{Sierratel},
    'sling' => q{Hughes Satellite Systems Corporation},
    'sm' => q{Telecom Italia San Marino S.p.A.},
    'smart' => q{Smart Communications, Inc. (SMART)},
    'smile' => q{Amazon Registry Services, Inc.},
    'sn' => q{Universite Cheikh Anta Diop - NIC Senegal},
    'sncf' => q{SNCF (Société Nationale des Chemins de fer Francais)},
    'so' => q{Ministry of Post and Telecommunications},
    'soccer' => q{Foggy Shadow, LLC},
    'social' => q{United TLD Holdco Ltd.},
    'softbank' => q{SoftBank Group Corp.},
    'software' => q{United TLD Holdco, Ltd},
    'sohu' => q{Sohu.com Limited},
    'solar' => q{Ruby Town, LLC},
    'solutions' => q{Silver Cover, LLC},
    'song' => q{Amazon Registry Services, Inc.},
    'sony' => q{Sony Corporation},
    'soy' => q{Charleston Road Registry Inc.},
    'space' => q{DotSpace Inc.},
    'spiegel' => q{SPIEGEL-Verlag Rudolf Augstein GmbH &amp; Co. KG},
    'spot' => q{Amazon Registry Services, Inc.},
    'spreadbetting' => q{DOTSPREADBETTING REGISTRY LTD},
    'sr' => q{Telesur},
    'srl' => q{InterNetX Corp.},
    'srt' => q{FCA US LLC.},
    'ss' => q{Not assigned},
    'st' => q{Tecnisys},
    'stada' => q{STADA Arzneimittel AG},
    'staples' => q{Staples, Inc.},
    'star' => q{Star India Private Limited},
    'starhub' => q{StarHub Limited},
    'statebank' => q{STATE BANK OF INDIA},
    'statefarm' => q{State Farm Mutual Automobile Insurance Company},
    'statoil' => q{Statoil ASA},
    'stc' => q{Saudi Telecom Company},
    'stcgroup' => q{Saudi Telecom Company},
    'stockholm' => q{Stockholms kommun},
    'storage' => q{Self Storage Company LLC},
    'store' => q{DotStore Inc.},
    'stream' => q{dot Stream Limited},
    'studio' => q{United TLD Holdco Ltd.},
    'study' => q{OPEN UNIVERSITIES AUSTRALIA PTY LTD},
    'style' => q{Binky Moon, LLC},
    'su' => q{Russian Institute for Development of Public Networks - (ROSNIIROS)},
    'sucks' => q{Vox Populi Registry Ltd.},
    'supplies' => q{Atomic Fields, LLC},
    'supply' => q{Half Falls, LLC},
    'support' => q{Grand Orchard, LLC},
    'surf' => q{Top Level Domain Holdings Limited},
    'surgery' => q{Tin Avenue, LLC},
    'suzuki' => q{SUZUKI MOTOR CORPORATION},
    'sv' => q{SVNet},
    'swatch' => q{The Swatch Group Ltd},
    'swiftcover' => q{Swiftcover Insurance Services Limited},
    'swiss' => q{Swiss Confederation},
    'sx' => q{SX Registry SA B.V.},
    'sy' => q{National Agency for Network Services (NANS)},
    'sydney' => q{State of New South Wales, Department of Premier and Cabinet},
    'symantec' => q{Symantec Corporation},
    'systems' => q{Dash Cypress, LLC},
    'sz' => q{University of Swaziland - Department of Computer Science},
    'tab' => q{Tabcorp Holdings Limited},
    'taipei' => q{Taipei City Government},
    'talk' => q{Amazon Registry Services, Inc.},
    'taobao' => q{Alibaba Group Holding Limited},
    'target' => q{Target Domain Holdings, LLC},
    'tatamotors' => q{Tata Motors Ltd},
    'tatar' => q{Limited Liability Company &quot;Coordination Center of Regional Domain of Tatarstan Republic&quot;},
    'tattoo' => q{Uniregistry, Corp.},
    'tax' => q{Storm Orchard, LLC},
    'taxi' => q{Pine Falls, LLC},
    'tc' => q{Melrex TC},
    'tci' => q{Asia Green IT System Bilgisayar San. ve Tic. Ltd. Sti.},
    'td' => q{Société des télécommunications du Tchad (SOTEL TCHAD)},
    'tdk' => q{TDK Corporation},
    'team' => q{Atomic Lake, LLC},
    'tech' => q{Dot Tech LLC},
    'technology' => q{Auburn Falls, LLC},
    'tel' => q{Telnic Ltd.},
    'telecity' => q{TelecityGroup International Limited},
    'telefonica' => q{Telefónica S.A.},
    'temasek' => q{Temasek Holdings (Private) Limited},
    'tennis' => q{Cotton Bloom, LLC},
    'teva' => q{Teva Pharmaceutical Industries Limited},
    'tf' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'tg' => q{Autorite de Reglementation des secteurs de Postes et de Telecommunications (ART&amp;P)},
    'th' => q{Thai Network Information Center Foundation},
    'thd' => q{Homer TLC, Inc.},
    'theater' => q{Blue Tigers, LLC},
    'theatre' => q{XYZ.COM LLC},
    'tiaa' => q{Teachers Insurance and Annuity Association of America},
    'tickets' => q{Accent Media Limited},
    'tienda' => q{Victor Manor, LLC},
    'tiffany' => q{Tiffany and Company},
    'tips' => q{Corn Willow, LLC},
    'tires' => q{Dog Edge, LLC},
    'tirol' => q{punkt Tirol GmbH},
    'tj' => q{Information Technology Center},
    'tjmaxx' => q{The TJX Companies, Inc.},
    'tjx' => q{The TJX Companies, Inc.},
    'tk' => q{Telecommunication Tokelau Corporation (Teletok)},
    'tkmaxx' => q{The TJX Companies, Inc.},
    'tl' => q{Ministry of Transport and  Communications; National Division of  Information and Technology},
    'tm' => q{TM Domain Registry Ltd},
    'tmall' => q{Alibaba Group Holding Limited},
    'tn' => q{Agence Tunisienne d&#39;Internet},
    'to' => q{Government of the Kingdom of Tonga - H.R.H. Crown Prince Tupouto&#39;a
    c/o Consulate of Tonga},
    'today' => q{Pearl Woods, LLC},
    'tokyo' => q{GMO Registry, Inc.},
    'tools' => q{Pioneer North, LLC},
    'top' => q{Jiangsu Bangning Science &amp; Technology Co.,Ltd.},
    'toray' => q{Toray Industries, Inc.},
    'toshiba' => q{TOSHIBA Corporation},
    'total' => q{Total SA},
    'tours' => q{Sugar Station, LLC},
    'town' => q{Koko Moon, LLC},
    'toyota' => q{TOYOTA MOTOR CORPORATION},
    'toys' => q{Pioneer Orchard, LLC},
    'tp' => q{Retired},
    'tr' => q{Middle East Technical University - Department of Computer Engineering},
    'trade' => q{Elite Registry Limited},
    'trading' => q{DOTTRADING REGISTRY LTD},
    'training' => q{Wild Willow, LLC},
    'travel' => q{Tralliance Registry Management Company, LLC.},
    'travelchannel' => q{Lifestyle Domain Holdings, Inc.},
    'travelers' => q{Travelers TLD, LLC},
    'travelersinsurance' => q{Travelers TLD, LLC},
    'trust' => q{Artemis Internet Inc},
    'trv' => q{Travelers TLD, LLC},
    'tt' => q{University of the West Indies - Faculty of Engineering},
    'tube' => q{Latin American Telecom LLC},
    'tui' => q{TUI AG},
    'tunes' => q{Amazon Registry Services, Inc.},
    'tushu' => q{Amazon Registry Services, Inc.},
    'tv' => q{Ministry of Finance and Tourism},
    'tvs' => q{T V SUNDRAM IYENGAR  &amp; SONS PRIVATE LIMITED},
    'tw' => q{Taiwan Network Information Center (TWNIC)},
    'tz' => q{Tanzania Network Information Centre (tzNIC)},
    'ua' => q{Hostmaster Ltd.},
    'ubank' => q{National Australia Bank Limited},
    'ubs' => q{UBS AG},
    'uconnect' => q{FCA US LLC.},
    'ug' => q{Uganda Online Ltd.},
    'uk' => q{Nominet UK},
    'um' => q{Not assigned},
    'unicom' => q{China United Network Communications Corporation Limited},
    'university' => q{Little Station, LLC},
    'uno' => q{Dot Latin LLC},
    'uol' => q{UBN INTERNET LTDA.},
    'ups' => q{UPS Market Driver, Inc.},
    'us' => q{NeuStar, Inc.},
    'uy' => q{SeCIU - Universidad de la Republica},
    'uz' => q{Computerization and Information Technologies Developing Center - UZINFOCOM},
    'va' => q{Holy See Secretariat of State Department of Telecommunications},
    'vacations' => q{Atomic Tigers, LLC},
    'vana' => q{Lifestyle Domain Holdings, Inc.},
    'vanguard' => q{The Vanguard Group, Inc.},
    'vc' => q{Ministry of Telecommunications, Science, Technology and Industry},
    've' => q{Comisión Nacional de Telecomunicaciones (CONATEL)},
    'vegas' => q{Dot Vegas, Inc.},
    'ventures' => q{Binky Lake, LLC},
    'verisign' => q{VeriSign, Inc.},
    'vermögensberater' => q{Deutsche Vermögensberatung Aktiengesellschaft DVAG},
    'vermögensberatung' => q{Deutsche Vermögensberatung Aktiengesellschaft DVAG},
    'versicherung' => q{dotversicherung-registry GmbH},
    'vet' => q{United TLD Holdco, Ltd},
    'vg' => q{Telecommunications Regulatory Commission of the Virgin Islands},
    'vi' => q{Virgin Islands Public Telcommunications System - c/o COBEX Internet Services},
    'viajes' => q{Black Madison, LLC},
    'video' => q{United TLD Holdco, Ltd},
    'vig' => q{VIENNA INSURANCE GROUP AG Wiener Versicherung Gruppe},
    'viking' => q{Viking River Cruises (Bermuda) Ltd.},
    'villas' => q{New Sky, LLC},
    'vin' => q{Holly Shadow, LLC},
    'vip' => q{Minds + Machines Group Limited},
    'virgin' => q{Virgin Enterprises Limited},
    'visa' => q{Visa Worldwide Pte. Limited},
    'vision' => q{Koko Station, LLC},
    'vista' => q{Vistaprint Limited},
    'vistaprint' => q{Vistaprint Limited},
    'viva' => q{Saudi Telecom Company},
    'vivo' => q{Telefonica Brasil S.A.},
    'vlaanderen' => q{DNS.be vzw},
    'vn' => q{Ministry of Information and Communications of Socialist Republic of Viet Nam},
    'vodka' => q{Top Level Domain Holdings Limited},
    'volkswagen' => q{Volkswagen Group of America Inc.},
    'vote' => q{Monolith Registry LLC},
    'voting' => q{Valuetainment Corp.},
    'voto' => q{Monolith Registry LLC},
    'voyage' => q{Ruby House, LLC},
    'vu' => q{Telecom Vanuatu Limited},
    'vuelos' => q{Travel Reservations SRL},
    'wales' => q{Nominet UK},
    'walmart' => q{Wal-Mart Stores, Inc.},
    'walter' => q{Sandvik AB},
    'wang' => q{Zodiac Registry Limited},
    'wanggou' => q{Amazon Registry Services, Inc.},
    'warman' => q{Weir Group IP Limited},
    'watch' => q{Sand Shadow, LLC},
    'watches' => q{Richemont DNS Inc.},
    'weather' => q{The Weather Channel, LLC},
    'weatherchannel' => q{The Weather Channel, LLC},
    'webcam' => q{dot Webcam Limited},
    'weber' => q{Saint-Gobain Weber SA},
    'website' => q{DotWebsite Inc.},
    'wed' => q{Atgron, Inc.},
    'wedding' => q{Top Level Domain Holdings Limited},
    'weibo' => q{Sina Corporation},
    'weir' => q{Weir Group IP Limited},
    'wf' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'whoswho' => q{Who&#39;s Who Registry},
    'wien' => q{punkt.wien GmbH},
    'wiki' => q{Top Level Design, LLC},
    'williamhill' => q{William Hill Organization Limited},
    'win' => q{First Registry Limited},
    'windows' => q{Microsoft Corporation},
    'wine' => q{June Station, LLC},
    'winners' => q{The TJX Companies, Inc.},
    'wme' => q{William Morris Endeavor Entertainment, LLC},
    'wolterskluwer' => q{Wolters Kluwer N.V.},
    'woodside' => q{Woodside Petroleum Limited},
    'work' => q{Top Level Domain Holdings Limited},
    'works' => q{Little Dynamite, LLC},
    'world' => q{Bitter Fields, LLC},
    'ws' => q{Government of Samoa Ministry of Foreign Affairs &amp; Trade},
    'wtc' => q{World Trade Centers Association, Inc.},
    'wtf' => q{Hidden Way, LLC},
    'xbox' => q{Microsoft Corporation},
    'xerox' => q{Xerox DNHC LLC},
    'xfinity' => q{Comcast IP Holdings I, LLC},
    'xihuan' => q{QIHOO 360 TECHNOLOGY CO. LTD.},
    'xin' => q{Elegant Leader Limited},
    'xperia' => q{Sony Mobile Communications AB},
    'xxx' => q{ICM Registry LLC},
    'xyz' => q{XYZ.COM LLC},
    'yachts' => q{DERYachts, LLC},
    'yahoo' => q{Yahoo! Domain Services Inc.},
    'yamaxun' => q{Amazon Registry Services, Inc.},
    'yandex' => q{YANDEX, LLC},
    'ye' => q{TeleYemen},
    'yodobashi' => q{YODOBASHI CAMERA CO.,LTD.},
    'yoga' => q{Top Level Domain Holdings Limited},
    'yokohama' => q{GMO Registry, Inc.},
    'you' => q{Amazon Registry Services, Inc.},
    'youtube' => q{Charleston Road Registry Inc.},
    'yt' => q{Association Française pour le Nommage Internet en Coopération (A.F.N.I.C.)},
    'yun' => q{QIHOO 360 TECHNOLOGY CO. LTD.},
    'za' => q{ZA Domain Name Authority},
    'zappos' => q{Amazon Registry Services, Inc.},
    'zara' => q{Industria de Diseño Textil, S.A. (INDITEX, S.A.)},
    'zero' => q{Amazon Registry Services, Inc.},
    'zip' => q{Charleston Road Registry Inc.},
    'zippo' => q{Zadco Company},
    'zm' => q{Zambia Information and Communications Technology Authority (ZICTA)},
    'zone' => q{Outer Falls, LLC},
    'zuerich' => q{Kanton Zürich (Canton of Zurich)},
    'zw' => q{Postal and Telecommunications Regulatory Authority of Zimbabwe (POTRAZ)},
    'δοκιμή' => q{Internet Assigned Numbers Authority},
    'ελ' => q{ICS-FORTH GR},
    'бг' => q{Imena.BG AD},
    'бел' => q{Reliable Software, Ltd.},
    'дети' => q{The Foundation for Network Initiatives “The Smart Internet”},
    'ею' => q{EURid vzw/asbl},
    'испытание' => q{Internet Assigned Numbers Authority},
    'ком' => q{VeriSign Sarl},
    'мкд' => q{Macedonian Academic Research Network Skopje},
    'мон' => q{Datacom Co.,Ltd},
    'москва' => q{Foundation for Assistance for Internet Technologies and Infrastructure Development (FAITID)},
    'онлайн' => q{CORE Association},
    'орг' => q{Public Interest Registry},
    'рус' => q{Rusnames Limited},
    'рф' => q{Coordination Center for TLD RU},
    'сайт' => q{CORE Association},
    'срб' => q{Serbian National Internet Domain Registry (RNIDS)},
    'укр' => q{Ukrainian Network Information Centre (UANIC), Inc.},
    'қаз' => q{Association of IT Companies of Kazakhstan},
    'հայ' => q{&quot;Internet Society&quot; Non-governmental Organization},
    'कॉम' => q{VeriSign Sarl},
    'नेट' => q{VeriSign Sarl},
    'परीक्षा' => q{Internet Assigned Numbers Authority},
    'भारत' => q{National Internet Exchange of India},
    'भारतम्' => q{Not assigned},
    'भारोत' => q{Not assigned},
    'संगठन' => q{Public Interest Registry},
    'বাংলা' => q{Not assigned},
    'ভারত' => q{National Internet Exchange of India},
    'ভাৰত' => q{Not assigned},
    'ਭਾਰਤ' => q{National Internet Exchange of India},
    'ભારત' => q{National Internet Exchange of India},
    'ଭାରତ' => q{Not assigned},
    'இந்தியா' => q{National Internet Exchange of India},
    'இலங்கை' => q{LK Domain Registry},
    'சிங்கப்பூர்' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    'பரிட்சை' => q{Internet Assigned Numbers Authority},
    'భారత్' => q{National Internet Exchange of India},
    'ಭಾರತ' => q{Not assigned},
    'ഭാരതം' => q{Not assigned},
    'ලංකා' => q{LK Domain Registry},
    'คอม' => q{VeriSign Sarl},
    'ไทย' => q{Thai Network Information Center Foundation},
    'გე' => q{Information Technologies Development Center (ITDC)},
    'みんな' => q{Charleston Road Registry Inc.},
    'クラウド' => q{Amazon Registry Services, Inc.},
    'グーグル' => q{Charleston Road Registry Inc.},
    'コム' => q{VeriSign Sarl},
    'ストア' => q{Amazon Registry Services, Inc.},
    'セール' => q{Amazon Registry Services, Inc.},
    'テスト' => q{Internet Assigned Numbers Authority},
    'ファッション' => q{Amazon Registry Services, Inc.},
    'ポイント' => q{Amazon Registry Services, Inc.},
    '世界' => q{Stable Tone Limited},
    '中信' => q{CITIC Group Corporation},
    '中国' => q{China Internet Network Information Center (CNNIC)},
    '中國' => q{China Internet Network Information Center (CNNIC)},
    '中文网' => q{TLD REGISTRY LIMITED},
    '企业' => q{Dash McCook, LLC},
    '佛山' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    '信息' => q{Beijing Tele-info Network Technology Co., Ltd.},
    '健康' => q{Stable Tone Limited},
    '八卦' => q{Zodiac Scorpio Limited},
    '公司' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    '公益' => q{China Organizational Name Administration Center},
    '台湾' => q{Taiwan Network Information Center (TWNIC)},
    '台灣' => q{Taiwan Network Information Center (TWNIC)},
    '商城' => q{Zodiac Aquarius Limited},
    '商店' => q{Wild Island, LLC},
    '商标' => q{HU YI GLOBAL INFORMATION RESOURCES(HOLDING) COMPANY.HONGKONG LIMITED},
    '嘉里' => q{Kerry Trading Co. Limited},
    '嘉里大酒店' => q{Kerry Trading Co. Limited},
    '在线' => q{TLD REGISTRY LIMITED},
    '大众汽车' => q{Volkswagen (China) Investment Co., Ltd.},
    '大拿' => q{VeriSign Sarl},
    '娱乐' => q{Will Bloom, LLC},
    '家電' => q{Amazon Registry Services, Inc.},
    '工行' => q{Industrial and Commercial Bank of China Limited},
    '广东' => q{Guangzhou YU Wei Information Technology Co., Ltd.},
    '微博' => q{Sina Corporation},
    '慈善' => q{Excellent First Limited},
    '我爱你' => q{Tycoon Treasure Limited},
    '手机' => q{Beijing RITT-Net Technology Development Co., Ltd},
    '手表' => q{Richemont DNS Inc.},
    '政务' => q{China Organizational Name Administration Center},
    '政府' => q{Net-Chinese Co., Ltd.},
    '新加坡' => q{Singapore Network Information Centre (SGNIC) Pte Ltd},
    '新闻' => q{Xinhua News Agency Guangdong Branch 新华通讯社广东分社},
    '时尚' => q{RISE VICTORY LIMITED},
    '書籍' => q{Amazon Registry Services, Inc.},
    '机构' => q{Public Interest Registry},
    '测试' => q{Internet Assigned Numbers Authority},
    '淡马锡' => q{Temasek Holdings (Private) Limited},
    '測試' => q{Internet Assigned Numbers Authority},
    '游戏' => q{Spring Fields, LLC},
    '澳門' => q{Bureau of Telecommunications Regulation (DSRT)},
    '点看' => q{VeriSign Sarl},
    '珠宝' => q{Richemont DNS Inc.},
    '移动' => q{Afilias Limited},
    '组织机构' => q{Public Interest Registry},
    '网址' => q{KNET Co., Ltd},
    '网店' => q{Zodiac Libra Limited},
    '网站' => q{Global Website TLD Asia Limited},
    '网络' => q{Computer Network Information Center of Chinese Academy of Sciences （China Internet Network Information Center）},
    '联通' => q{China United Network Communications Corporation Limited},
    '诺基亚' => q{Nokia Corporation},
    '谷歌' => q{Charleston Road Registry Inc.},
    '购物' => q{Minds + Machines Group Limited},
    '集团' => q{Eagle Horizon Limited},
    '電訊盈科' => q{PCCW Enterprises Limited},
    '飞利浦' => q{Koninklijke Philips N.V.},
    '食品' => q{Amazon Registry Services, Inc.},
    '餐厅' => q{HU YI GLOBAL INFORMATION RESOURCES (HOLDING) COMPANY. HONGKONG LIMITED},
    '香格里拉' => q{Shangri‐La International Hotel Management Limited},
    '香港' => q{Hong Kong Internet Registration Corporation Ltd.},
    '닷넷' => q{VeriSign Sarl},
    '닷컴' => q{VeriSign Sarl},
    '삼성' => q{SAMSUNG SDS CO., LTD},
    '테스트' => q{Internet Assigned Numbers Authority},
    '한국' => q{KISA (Korea Internet &amp; Security Agency)},
  }
);

my $flat_profile = flatten ( \%tld_profile );

sub flatten {
  my $hashref = shift;
  my %results;
  @results{ keys %{ $hashref->{$_} } } = values % { $hashref->{$_} }
    for ( keys %$hashref );
  return \%results;
}

sub check_type {
  my $type = shift;
  croak "unknown TLD type: $type" unless grep { $type eq $_ } TLD_TYPES;
  return 1;
}

=head1 PUBLIC METHODS

  Each public function/method is described here.
  These are how you should interact with this module.

=head3 C<< tlds >>

  This routine returns the tlds requested.

  my @all_tlds = tlds; #array of tlds
  my $all_tlds = tlds; #hashref of tlds and their descriptions

  my @cc_tlds = tlds('cc'); #array of just 'cc' type tlds
  my $cc_tlds = tlds('cc'); #hashref of just 'cc' type tlds and their descriptions

  Valid types are:
    cc                 - country code domains
    ccidn              - internationalized country code top-level domain 
    gtld_open          - generic domains that anyone can register
    gtld_restricted    - generic restricted registration domains
    gtld_new           - new gTLDs
    new_open           - recently added generic domains
    new_restricted     - new restricted registration domains
    reserved           - RFC2606 restricted names, not returned by tlds

=cut

sub tlds {
  my $type = shift;
  check_type ( $type ) if $type;
  my $results = $type ? 
    wantarray ? [ keys %{ $tld_profile{$type} } ] : 
      dclone ( $tld_profile{$type} ) :
	wantarray ? [ map { keys %$_ } values %tld_profile ] : 
	  $flat_profile;
  return wantarray ? @$results : $results;
}

=head3 C<< tld_exists >>

  This routine returns true if the given domain exists and false otherwise.

  die "no such domain" unless tld_exists($tld); #call without tld type 
  die "no such domain" unless tld_exists($tld, 'new_open'); #call with tld type

=cut

sub tld_exists {
  my ( $tld, $type )  = ( lc ( $_[0] ), $_[1] );
  check_type ( $type ) if $type;
  my $result = $type ? 
    $tld_profile{$type}{$tld} ? 1 : 0 :
    $flat_profile->{$tld} ? 1 : 0;
  return $result;
}

=head1 COPYRIGHT

  Copyright (c) 2003-2016 Alex Pavlovic, all rights reserved.  This program
  is free software; you can redistribute it and/or modify it under the same terms
  as Perl itself.

=head1 AUTHORS

  Alexander Pavlovic <alex.pavlovic@devradius.com>
  Ricardo SIGNES <rjbs@cpan.org>

=cut

1;