#ifndef _I386_PERF_H_
#define _I386_PERF_H_

/* Per process cycle counter is third counter. */
#define PERF_CYCLES                 0x1

/* Countable things: (For PPro) */
/* Data Cache Unit (DCU) */
#define PERF_DATA_MEM_REFS          0x43
#define PERF_DCU_LINES_IN           0x45
#define PERF_DCU_M_LINES_IN         0x46
#define PERF_DCU_M_LINES_OUT        0x47
#define PERF_DCU_MISS_STANDING      0x48

/* Instruction Fetch Unit (IFU) */
#define PERF_IFU_IFETCH             0x80
#define PERF_IFU_IFETCH_MISS        0x81
#define PERF_ITLB_MISS              0x85
#define PERF_IFU_MEM_STALL          0x86
#define PERF_ILD_STALL              0x87

/* L2 Cache */
#define PERF_L2_IFETCH              0x28  /* MESI */
#define PERF_L2_LD                  0x29  /* MESI */
#define PERF_L2_ST                  0x2A  /* MESI */
#define PERF_L2_LINES_IN            0x24
#define PERF_L2_LINES_OUT           0x26
#define PERF_L2_LINES_INM           0x25
#define PERF_L2_LINES_OUTM          0x27
#define PERF_L2_RQSTS               0x2E  /* MESI */
#define PERF_L2_ADS                 0x21
#define PERF_L2_DBUS_BUSY           0x22
#define PERF_L2_DBUS_BUSY_RD        0x23

/* External Bus Logic */
#define PERF_BUS_DRDY_CLOCKS        0x62
#define PERF_BUS_LOCK_CLOCKS        0x63
#define PERF_BUS_REQ_OUTSTANDING    0x60
#define PERF_BUS_TRAN_BRD           0x65
#define PERF_BUS_TRAN_RFO           0x66
#define PERF_BUS_TRANS_WB           0x67
#define PERF_BUS_TRAN_IFETCH        0x68
#define PERF_BUS_TRAN_INVAL         0x69
#define PERF_BUS_TRAN_PWR           0x6A
#define PERF_BUS_TRAN_P             0x6B
#define PERF_BUS_TRANS_IO           0x6C
#define PERF_BUS_TRAN_DEF           0x6D
#define PERF_BUS_TRAN_BURST         0x6E
#define PERF_BUS_TRAN_ANY           0x70
#define PERF_BUS_TRAN_MEM           0x6F
#define PERF_BUS_DATA_RCV           0x64
#define PERF_BUS_BNR_DRV            0x61
#define PERF_BUS_HIT_DRV            0x7A
#define PERF_BUS_HITM_DRV           0x7B
#define PERF_BUS_SNOOP_STALL        0x7E

/* Floating point unit */
#define PERF_FLOPS                  0xC1
#define PERF_FP_COMP_OPS_EXE        0x10
#define PERF_FP_ASSIST              0x11
#define PERF_MUL                    0x12
#define PERF_DIV                    0x13
#define PERF_CYCLES_DIV_BUSY        0x14

/* Memory Ordering */
#define PERF_LD_BLOCK               0x03
#define PERF_SB_DRAINS              0x04
#define PERF_MISALIGN_MEM_REF       0x05

/* Instruction Decoding and Retirement */ 
#define PERF_INST_RETIRED           0xC0
#define PERF_UOPS_RETIRED           0xC2
#define PERF_INST_DECODER           0xD0

/* Interrupts */
#define PERF_HW_INT_RX              0xC8
#define PERF_CYCLES_INST_MASKED     0xC6
#define PERF_CYCLES_INT_PENDING_AND_MASKED 0xC7

/* Branches */
#define PERF_BR_INST_RETIRED        0xC4
#define PERF_BR_MISS_PRED_RETIRED   0xC5
#define PERF_BR_TAKEN_RETIRED       0xC9
#define PERF_BR_MISS_PRED_TAKEN_RET 0xCA
#define PERF_BR_INST_DECODED        0xE0
#define PERF_BR_BTB_MISSES          0xE2
#define PERF_BR_BOGUS               0xE4
#define PERF_BACLEARS               0xE6

/* Stalls */
#define PERF_RESOURCE_STALLS        0xA2
#define PERF_PARTIAL_RAT_STALLS     0xD2

/* Segment Register Loads */
#define PERF_SEGMENT_REG_LOADS      0x06

/* Clocks */
#define PERF_CPU_CLK_UNHALTED       0x79

/* Unit mask flags: */
#define PERF_SELF                 0x0000
#define PERF_ANY                  0x2000
#define PERF_CACHE_M              0x0800
#define PERF_CACHE_E              0x0400
#define PERF_CACHE_S              0x0200
#define PERF_CACHE_I              0x0100
#define PERF_CACHE_ALL            0x0F00

/*--------------------------------------------------------*/
/* Bit masks for configuration fields */
#define PERF_CTR_MASK          0xFF000000
#define PERF_INV_CTR_MASK      0x00800000
#define PERF_ENABLE            0x00400000
#define PERF_INT_ENABLE        0x00100000
#define PERF_PIN_CONTROL       0x00080000
#define PERF_EDGE_DETECT       0x00040000
#define PERF_OS                0x00020000
#define PERF_USR               0x00010000
#define PERF_UNIT_MASK         0x0000FF00
#define PERF_EVNT_MASK         0x000000FF

/* System calls */

#define PERF_RESET          0
#define PERF_SET_CONFIG     1
#define PERF_GET_CONFIG     2
#define PERF_START          3
#define PERF_STOP           4
#define PERF_READ           5
#define PERF_WRITE          6
#define PERF_WAIT           7
#define PERF_DEBUG          8
#define PERF_SET_OPT        9
#define PERF_GET_OPT        10
#define PERF_RESET_COUNTERS 11
#define PERF_FASTWRITE      12
#define PERF_FASTREAD       13
#define PERF_FASTCONFIG     14
#define PERF_SYS_RESET      20
#define PERF_SYS_SET_CONFIG 21
#define PERF_SYS_GET_CONFIG 22
#define PERF_SYS_START      23
#define PERF_SYS_STOP       24
#define PERF_SYS_READ       25
#define PERF_SYS_WRITE      26
#define PERF_SYS_RESET_COUNTERS 27

/* Options */

#define PERF_DO_CHILDREN    1
#define PERF_SUM_CPUS       2

/* Number of performance counters */
/* Counter 2 is always the virtual cycle counter. */

#define PERF_COUNTERS 3

struct perf_wait_struct { 
  pid_t  pid;
  int   *status;
  int    options;
  struct rusage *rusage;
  unsigned long long *counts;
};

static inline unsigned long long perf_get_cycles (void)
{
	unsigned long long ret;
        __asm__ __volatile__("rdtsc"
			    : "=A" (ret)
			    : /* no inputs */);
        return ret;
}

#ifdef __KERNEL__

#define MSR_PERFCTR0 0x00C1
#define MSR_PERFCTR1 0x00C2
#define MSR_EVNTSEL0 0x0186
#define MSR_EVNTSEL1 0x0187

struct proc_perf_opt_t {
    unsigned int do_children:16;
    unsigned int sum_cpus:8;
    unsigned int is_sys_holder:8;
};

struct proc_perf_t {
    struct proc_perf_opt_t options;
    unsigned int       conf   [PERF_COUNTERS];
    unsigned long long counter[PERF_COUNTERS]; /* Last one is always the TSC for this process */
    unsigned long long shadow_tsc;
};

#define PROC_PERF_INIT {{0,},{0,},{0,},0}

extern volatile int perf_sys_flag;

static inline
unsigned long long perf_rdmsr(unsigned int msr) {
    unsigned long long ret;
    __asm__ __volatile__("rdmsr"
			 : "=A" (ret)
			 : "c" (msr));
    return ret;
}

static inline
void perf_wrmsr(unsigned int msr,unsigned long long val) {
    __asm__ __volatile__("wrmsr"
			 : /* no Outputs */
			 : "c" (msr), "A" (val));
}

static inline
void perf_restore(unsigned int *config, unsigned long long *shadow_tsc) {
    perf_wrmsr(MSR_EVNTSEL0, config[0]);
    perf_wrmsr(MSR_PERFCTR0, 0);
    perf_wrmsr(MSR_EVNTSEL0+1, config[1]);
    perf_wrmsr(MSR_PERFCTR0+1, 0);
    *shadow_tsc = perf_get_cycles();
}

static inline
void perf_save(unsigned int *config, unsigned long long *counter, unsigned long long *shadow_tsc) {
    counter[0] += (perf_rdmsr(MSR_PERFCTR0)&((1ULL<<40)-1));
    counter[1] += (perf_rdmsr(MSR_PERFCTR0+1)&((1ULL<<40)-1));
    if (config[2])
      counter[2] += perf_get_cycles() - *shadow_tsc;
}

void perf_sys_remote_sync(void);
void smp_perf_sys_update(void);
#endif
#endif
