/*
 * DivIO nw80x subdriver
 *
 * Copyright (C) 2011 Jean-François Moine (http://moinejf.free.fr)
 * Copyright (C) 2003 Sylvain Munaut <tnt@246tNt.com>
 *			Kjell Claesson <keyson@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */

#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#define MODULE_NAME "nw80x"

#include "gspca.h"

MODULE_AUTHOR("Jean-Francois Moine <http://moinejf.free.fr>");
MODULE_DESCRIPTION("NW80x USB Camera Driver");
MODULE_LICENSE("GPL");

static int webcam;

/* specific webcam descriptor */
struct sd {
	struct gspca_dev gspca_dev;	/* !! must be the first item */

	u32 ae_res;
	s8 ag_cnt;
#define AG_CNT_START 13
	u8 exp_too_low_cnt;
	u8 exp_too_high_cnt;

	u8 bridge;
	u8 webcam;
};

enum bridges {
	BRIDGE_NW800,	/* and et31x110 */
	BRIDGE_NW801,
	BRIDGE_NW802,
};
enum webcams {
	Generic800,
	SpaceCam,	/* Trust 120 SpaceCam */
	SpaceCam2,	/* other Trust 120 SpaceCam */
	Cvideopro,	/* Conceptronic Video Pro */
	Dlink350c,
	DS3303u,
	Kr651us,
	Kritter,
	Mustek300,
	Proscope,
	Twinkle,
	DvcV6,
	P35u,
	Generic802,
	NWEBCAMS	/* number of webcams */
};

static const u8 webcam_chip[NWEBCAMS] = {
	[Generic800]	= BRIDGE_NW800,	/* 06a5:0000
					 * Typhoon Webcam 100 USB */

	[SpaceCam]	= BRIDGE_NW800,	/* 06a5:d800
				* Trust SpaceCam120 or SpaceCam100 PORTABLE */

	[SpaceCam2]	= BRIDGE_NW800,	/* 06a5:d800 - pas106
			* other Trust SpaceCam120 or SpaceCam100 PORTABLE */

	[Cvideopro]	= BRIDGE_NW802,	/* 06a5:d001
			* Conceptronic Video Pro 'CVIDEOPRO USB Webcam CCD' */

	[Dlink350c]	= BRIDGE_NW802,	/* 06a5:d001
					 * D-Link NetQam Pro 250plus */

	[DS3303u]	= BRIDGE_NW801,	/* 06a5:d001
				* Plustek Opticam 500U or ProLink DS3303u */

	[Kr651us]	= BRIDGE_NW802,	/* 06a5:d001
					 * Panasonic GP-KR651US */

	[Kritter]	= BRIDGE_NW802,	/* 06a5:d001
					 * iRez Kritter cam */

	[Mustek300]	= BRIDGE_NW802,	/* 055f:d001
					 * Mustek Wcam 300 mini */

	[Proscope]	= BRIDGE_NW802,	/* 06a5:d001
					 * Scalar USB Microscope (ProScope) */

	[Twinkle]	= BRIDGE_NW800,	/* 06a5:d800 - hv7121b? (seems pas106)
					 * Divio Chicony TwinkleCam
					 * DSB-C110 */

	[DvcV6]		= BRIDGE_NW802,	/* 0502:d001
					 * DVC V6 */

	[P35u]		= BRIDGE_NW801,	/* 052b:d001, 06a5:d001 and 06be:d001
					 * EZCam Pro p35u */

	[Generic802]	= BRIDGE_NW802,
};
/*
 * other webcams:
 *	- nw801 046d:d001
 *		Logitech QuickCam Pro (dark focus ring)
 *	- nw801 0728:d001
 *		AVerMedia Camguard
 *	- nw??? 06a5:d001
 *		D-Link NetQam Pro 250plus
 *	- nw800 065a:d800
 *		Showcam NGS webcam
 *	- nw??? ????:????
 *		Sceptre svc300
 */

/*
 * registers
 *    nw800/et31x110	  nw801		  nw802
 *	0000..009e	0000..00a1	0000..009e
 *	0200..0211	   id		   id
 *	0300..0302	   id		   id
 *	0400..0406	  (inex)	0400..0406
 *	0500..0505	0500..0506	  (inex)
 *	0600..061a	0600..0601	0600..0601
 *	0800..0814	   id		   id
 *	1000..109c	1000..10a1	1000..109a
 */

/* resolutions
 *	nw800: 320x240, 352x288
 *	nw801/802: 320x240, 640x480
 */
static const struct v4l2_pix_format cif_mode[] = {
	{320, 240, V4L2_PIX_FMT_JPGL, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 4 / 8,
		.colorspace = V4L2_COLORSPACE_JPEG},
	{352, 288, V4L2_PIX_FMT_JPGL, V4L2_FIELD_NONE,
		.bytesperline = 352,
		.sizeimage = 352 * 288 * 4 / 8,
		.colorspace = V4L2_COLORSPACE_JPEG}
};
static const struct v4l2_pix_format vga_mode[] = {
	{320, 240, V4L2_PIX_FMT_JPGL, V4L2_FIELD_NONE,
		.bytesperline = 320,
		.sizeimage = 320 * 240 * 4 / 8,
		.colorspace = V4L2_COLORSPACE_JPEG},
	{640, 480, V4L2_PIX_FMT_JPGL, V4L2_FIELD_NONE,
		.bytesperline = 640,
		.sizeimage = 640 * 480 * 3 / 8,
		.colorspace = V4L2_COLORSPACE_JPEG},
};

/*
 * The sequences below contain:
 *	- 1st and 2nd bytes: either
 *		- register number (BE)
 *		- I2C0 + i2c address
 *	- 3rd byte: data length (=0 for end of sequence)
 *	- n bytes: data
 */
#define I2C0 0xff

static const u8 nw800_init[] = {
	0x04, 0x05, 0x01, 0x61,
	0x04, 0x04, 0x01, 0x01,
	0x04, 0x06, 0x01, 0x04,
	0x04, 0x04, 0x03, 0x00, 0x00, 0x00,
	0x05, 0x05, 0x01, 0x00,
	0, 0, 0
};
static const u8 nw800_start[] = {
	0x04, 0x06, 0x01, 0xc0,
	0x00, 0x00, 0x40, 0x10, 0x43, 0x00, 0xb4, 0x01, 0x10, 0x00, 0x4f,
			  0xef, 0x0e, 0x00, 0x74, 0x01, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x3e, 0x00, 0x24,
			  0x03, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xa0, 0x48, 0xc3, 0x02, 0x88, 0x0c, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x06, 0x00, 0x08,
			  0x00, 0x32, 0x01, 0x01, 0x00, 0x16, 0x00, 0x04,
			  0x00, 0x4b, 0x00, 0x76, 0x00, 0x86, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x61, 0xc0,
	0x05, 0x00, 0x06, 0xe8, 0x00, 0x00, 0x00, 0x20, 0x20,
	0x06, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x83, 0x02, 0x20, 0x00, 0x13, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1d, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x62,
			  0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01, 0x20,
			  0x01, 0x60, 0x01, 0x00, 0x00,

	0x04, 0x04, 0x01, 0xff,
	0x04, 0x06, 0x01, 0xc4,

	0x04, 0x06, 0x01, 0xc0,
	0x00, 0x00, 0x40, 0x10, 0x43, 0x00, 0xb4, 0x01, 0x10, 0x00, 0x4f,
			  0xef, 0x0e, 0x00, 0x74, 0x01, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x3e, 0x00, 0x24,
			  0x03, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xa0, 0x48, 0xc3, 0x02, 0x88, 0x0c, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x06, 0x00, 0x08,
			  0x00, 0x32, 0x01, 0x01, 0x00, 0x16, 0x00, 0x04,
			  0x00, 0x4b, 0x00, 0x76, 0x00, 0x86, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x61, 0xc0,
	0x05, 0x00, 0x06, 0xe8, 0x00, 0x00, 0x00, 0x20, 0x20,
	0x06, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x83, 0x02, 0x20, 0x00, 0x13, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1d, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x62,
			  0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01, 0x20,
			  0x01, 0x60, 0x01, 0x00, 0x00,

	0x02, 0x00, 0x11, 0x48, 0x58, 0x9e, 0x48, 0x58, 0x00, 0x00, 0x00,
			  0x00, 0x84, 0x36, 0x05, 0x01, 0xf2, 0x86, 0x65,
			  0x40,
	0x00, 0x80, 0x01, 0xa0,
	0x10, 0x1a, 0x01, 0x00,
	0x00, 0x91, 0x02, 0x6c, 0x01,
	0x00, 0x03, 0x02, 0xc8, 0x01,
	0x10, 0x1a, 0x01, 0x00,
	0x10, 0x00, 0x01, 0x83,
	0x10, 0x8f, 0x0c, 0x62, 0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01,
			  0x20, 0x01, 0x60, 0x01,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x10, 0x1b, 0x02, 0x69, 0x00,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x05, 0x02, 0x01, 0x02,
	0x06, 0x00, 0x02, 0x04, 0xd9,
	0x05, 0x05, 0x01, 0x20,
	0x05, 0x05, 0x01, 0x21,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x08, 0x21, 0x3d, 0x52, 0x63, 0x75, 0x83,
			  0x91, 0x9e, 0xaa, 0xb6, 0xc1, 0xcc, 0xd6, 0xe0,
			  0xea,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x13, 0x13,
	0x10, 0x03, 0x01, 0x14,
	0x10, 0x41, 0x11, 0x00, 0x08, 0x21, 0x3d, 0x52, 0x63, 0x75, 0x83,
			  0x91, 0x9e, 0xaa, 0xb6, 0xc1, 0xcc, 0xd6, 0xe0,
			  0xea,
	0x10, 0x0b, 0x01, 0x14,
	0x10, 0x0d, 0x01, 0x20,
	0x10, 0x0c, 0x01, 0x34,
	0x04, 0x06, 0x01, 0xc3,
	0x04, 0x04, 0x01, 0x00,
	0x05, 0x02, 0x01, 0x02,
	0x06, 0x00, 0x02, 0x00, 0x48,
	0x05, 0x05, 0x01, 0x20,
	0x05, 0x05, 0x01, 0x21,
	0, 0, 0
};

/* 06a5:d001 - nw801 - Panasonic
 *		P35u */
static const u8 nw801_start_1[] = {
	0x05, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x0e, 0x00, 0x00, 0xf9, 0x02, 0x11, 0x00, 0x0e,
			  0x01, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x22, 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x08, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x69, 0xa8, 0x1f, 0x00,
			  0x0d, 0x02, 0x07, 0x00, 0x01, 0x00, 0x19, 0x00,
			  0xf2, 0x00, 0x18, 0x06, 0x10, 0x06, 0x10, 0x00,
			  0x36, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x00, 0x00, 0x00,
	0x05, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x22, 0x02, 0x80, 0x00, 0x1e, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x0a, 0x15, 0x08, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x01, 0x35, 0xfd, 0x07, 0x3d, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x14, 0x02,
			  0x00, 0x01, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
			  0x40, 0x00, 0x00, 0x00, 0x40, 0x20, 0x10, 0x06,
			  0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06, 0xf7,
	0x10, 0x40, 0x40, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80, 0x80,
			  0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99, 0xa4,
			  0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc, 0xcf,
			  0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54, 0x64,
			  0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2, 0xe2,
			  0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
	0x10, 0x80, 0x22, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x82, 0x02,
			  0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40, 0x01,
			  0xf0, 0x00,
	0, 0, 0,
};
static const u8 nw801_start_qvga[] = {
	0x02, 0x00, 0x10, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x18, 0x0b, 0x06, 0xa2, 0x86, 0x78,
	0x02, 0x0f, 0x01, 0x6b,
	0x10, 0x1a, 0x01, 0x15,
	0x00, 0x00, 0x01, 0x1e,
	0x10, 0x00, 0x01, 0x2f,
	0x10, 0x8c, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x11, 0x08, 0x29, 0x00, 0x18, 0x01, 0x1f, 0x00, 0xd2, 0x00,
							/* AE window */
	0, 0, 0,
};
static const u8 nw801_start_vga[] = {
	0x02, 0x00, 0x10, 0x78, 0xa0, 0x97, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xf0,
	0x02, 0x0f, 0x01, 0xd5,
	0x10, 0x1a, 0x01, 0x15,
	0x00, 0x00, 0x01, 0x0e,
	0x10, 0x00, 0x01, 0x22,
	0x10, 0x8c, 0x08, 0x00, 0x00, 0x7f, 0x02, 0x00, 0x00, 0xdf, 0x01,
	0x10, 0x11, 0x08, 0x51, 0x00, 0x30, 0x02, 0x3d, 0x00, 0xa4, 0x01,
	0, 0, 0,
};
static const u8 nw801_start_2[] = {
	0x10, 0x04, 0x01, 0x1a,
	0x10, 0x19, 0x01, 0x09,				/* clock */
	0x10, 0x24, 0x06, 0xc0, 0x00, 0x3f, 0x02, 0x00, 0x01,
							 /* .. gain .. */
	0x00, 0x03, 0x02, 0x92, 0x03,
	0x00, 0x1d, 0x04, 0xf2, 0x00, 0x24, 0x07,
	0x00, 0x7b, 0x01, 0xcf,
	0x10, 0x94, 0x01, 0x07,
	0x05, 0x05, 0x01, 0x01,
	0x05, 0x04, 0x01, 0x01,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x48, 0x11, 0x00, 0x37, 0x55, 0x6b, 0x7d, 0x8d, 0x9b, 0xa8,
			  0xb4, 0xbf, 0xca, 0xd4, 0xdd, 0xe6, 0xef, 0xf0,
			  0xf0,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x0c, 0x0c,
	0x10, 0x03, 0x01, 0x08,
	0x10, 0x48, 0x11, 0x00, 0x37, 0x55, 0x6b, 0x7d, 0x8d, 0x9b, 0xa8,
			  0xb4, 0xbf, 0xca, 0xd4, 0xdd, 0xe6, 0xef, 0xf0,
			  0xf0,
	0x10, 0x0b, 0x01, 0x0b,
	0x10, 0x0d, 0x01, 0x0b,
	0x10, 0x0c, 0x01, 0x1f,
	0x05, 0x06, 0x01, 0x03,
	0, 0, 0
};

/* nw802 (sharp IR3Y38M?) */
static const u8 nw802_start[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0xf9, 0x02, 0x10, 0x00, 0x4d,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x01, 0x00, 0x16, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x08, 0x00, 0x18, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa1, 0x02, 0x80, 0x00, 0x1d, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0xff, 0x01, 0xc0, 0x00, 0x14,
			  0x02, 0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
			  0x40,
	0x10, 0x1a, 0x01, 0x00,
	0x10, 0x00, 0x01, 0xad,
	0x00, 0x00, 0x01, 0x08,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x51, 0x00, 0xf0, 0x00, 0x3d, 0x00, 0xb4, 0x00,
	0x10, 0x1d, 0x08, 0x00, 0xa0, 0x00, 0xa0, 0x00, 0xa0, 0x00, 0xa0,
	0x10, 0x0e, 0x01, 0x27,
	0x10, 0x41, 0x11, 0x00, 0x0e, 0x35, 0x4f, 0x62, 0x71, 0x7f, 0x8b,
			  0x96, 0xa0, 0xa9, 0xb2, 0xbb, 0xc3, 0xca, 0xd2,
			  0xd8,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x14, 0x14,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x41, 0x11, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54, 0x64, 0x74,
			  0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2, 0xe2, 0xf1,
			  0xff,
/*			  0x00, 0x0e, 0x35, 0x4f, 0x62, 0x71, 0x7f, 0x8b,
 *			  0x96, 0xa0, 0xa9, 0xb2, 0xbb, 0xc3, 0xca, 0xd2,
 *			  0xd8,	*/
	0x10, 0x0b, 0x01, 0x10,
	0x10, 0x0d, 0x01, 0x11,
	0x10, 0x0c, 0x01, 0x1c,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};
/* et31x110 - Trust 120 SpaceCam */
static const u8 spacecam_init[] = {
	0x04, 0x05, 0x01, 0x01,
	0x04, 0x04, 0x01, 0x01,
	0x04, 0x06, 0x01, 0x04,
	0x04, 0x04, 0x03, 0x00, 0x00, 0x00,
	0x05, 0x05, 0x01, 0x00,
	0, 0, 0
};
static const u8 spacecam_start[] = {
	0x04, 0x06, 0x01, 0x44,
	0x00, 0x00, 0x40, 0x10, 0x43, 0x00, 0xb4, 0x01, 0x10, 0x00, 0x4f,
			  0xef, 0x0e, 0x00, 0x74, 0x01, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x3e, 0x00, 0x24,
			  0x03, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xa0, 0x48, 0xc3, 0x02, 0x88, 0x0c, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x06, 0x00, 0x08,
			  0x00, 0x32, 0x01, 0x01, 0x00, 0x16, 0x00, 0x04,
			  0x00, 0x4b, 0x00, 0x7c, 0x00, 0x80, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x05, 0x00, 0x06, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x83, 0x02, 0x20, 0x00, 0x11, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1d, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x62,
			  0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01, 0x20,
			  0x01, 0x60, 0x01, 0x00, 0x00,
	0x04, 0x06, 0x01, 0xc0,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x02, 0x00, 0x11, 0x48, 0x58, 0x9e, 0x48, 0x58, 0x00, 0x00, 0x00,
			  0x00, 0x84, 0x36, 0x05, 0x01, 0xf2, 0x86, 0x65,
			  0x40,
	0x00, 0x80, 0x01, 0xa0,
	0x10, 0x1a, 0x01, 0x00,
	0x00, 0x91, 0x02, 0x32, 0x01,
	0x00, 0x03, 0x02, 0x08, 0x02,
	0x10, 0x00, 0x01, 0x83,
	0x10, 0x8f, 0x0c, 0x62, 0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01,
			  0x20, 0x01, 0x60, 0x01,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x64, 0x99, 0xc0, 0xe2, 0xf9, 0xf9, 0xf9,
			  0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9,
			  0xf9,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x13, 0x13,
	0x10, 0x03, 0x01, 0x06,
	0x10, 0x41, 0x11, 0x00, 0x64, 0x99, 0xc0, 0xe2, 0xf9, 0xf9, 0xf9,
			  0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9,
			  0xf9,
	0x10, 0x0b, 0x01, 0x08,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1f,
	0x04, 0x06, 0x01, 0xc3,
	0x04, 0x05, 0x01, 0x40,
	0x04, 0x04, 0x01, 0x40,
	0, 0, 0
};
/* et31x110 - pas106 - other Trust SpaceCam120 */
static const u8 spacecam2_start[] = {
	0x04, 0x06, 0x01, 0x44,
	0x04, 0x06, 0x01, 0x00,
	0x00, 0x00, 0x40, 0x14, 0x83, 0x00, 0xba, 0x01, 0x10, 0x00, 0x4f,
			  0xef, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x06, 0x00, 0xfc,
			  0x01, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb8, 0x48, 0x0f, 0x04, 0x88, 0x14, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x01, 0x00, 0x03,
			  0x00, 0x24, 0x01, 0x01, 0x00, 0x16, 0x00, 0x04,
			  0x00, 0x4b, 0x00, 0x76, 0x00, 0x86, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x61, 0x00,
	0x05, 0x00, 0x06, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x80, 0x02, 0x20, 0x00, 0x13, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1d, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x62,
			  0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01, 0x20,
			  0x01, 0x60, 0x01, 0x00, 0x00,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x04, 0x04, 0x01, 0x40,
	0x04, 0x04, 0x01, 0x00,
	I2C0, 0x40, 0x0c, 0x02, 0x0c, 0x12, 0x07, 0x00, 0x00, 0x00, 0x05,
			  0x00, 0x00, 0x05, 0x05,
	I2C0, 0x40, 0x02, 0x11, 0x06,
	I2C0, 0x40, 0x02, 0x14, 0x00,
	I2C0, 0x40, 0x02, 0x13, 0x01,		/* i2c end */
	0x02, 0x00, 0x11, 0x48, 0x58, 0x9e, 0x48, 0x58, 0x00, 0x00, 0x00,
			  0x00, 0x84, 0x36, 0x05, 0x01, 0xf2, 0x86, 0x65,
			  0x40,
	I2C0, 0x40, 0x02, 0x02, 0x0c,		/* pixel clock */
	I2C0, 0x40, 0x02, 0x0f, 0x00,
	I2C0, 0x40, 0x02, 0x13, 0x01,		/* i2c end */
	0x10, 0x00, 0x01, 0x01,
	0x10, 0x8f, 0x0c, 0x62, 0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01,
			  0x20, 0x01, 0x60, 0x01,
	I2C0, 0x40, 0x02, 0x05, 0x0f,		/* exposure */
	I2C0, 0x40, 0x02, 0x13, 0x01,		/* i2c end */
	I2C0, 0x40, 0x07, 0x09, 0x0b, 0x0f, 0x05, 0x05, 0x0f, 0x00,
						/* gains */
	I2C0, 0x40, 0x03, 0x12, 0x04, 0x01,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x17, 0x3f, 0x69, 0x7b, 0x8c, 0x9a, 0xa7,
			  0xb3, 0xbf, 0xc9, 0xd3, 0xdd, 0xe6, 0xef, 0xf7,
			  0xf9,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x13, 0x13,
	0x10, 0x03, 0x01, 0x06,
	0x10, 0x41, 0x11, 0x00, 0x17, 0x3f, 0x69, 0x7b, 0x8c, 0x9a, 0xa7,
			  0xb3, 0xbf, 0xc9, 0xd3, 0xdd, 0xe6, 0xef, 0xf7,
			  0xf9,
	0x10, 0x0b, 0x01, 0x11,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x14,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x05, 0x01, 0x61,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* nw802 - Conceptronic Video Pro */
static const u8 cvideopro_start[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x54, 0x96, 0x98, 0xf9, 0x02, 0x18, 0x00, 0x4c,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x0b, 0x00, 0x1b, 0x00, 0xc8, 0x00, 0xf4,
			  0x05, 0xb4, 0x00, 0xcc, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa2, 0x00, 0xc6, 0x00, 0x60, 0x00, 0xc6,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0xae, 0x00, 0xd2, 0x00, 0xae, 0x00, 0xd2,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa8, 0x00, 0xc0, 0x00, 0x66, 0x00, 0xc0,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x0a, 0x00, 0x54, 0x00, 0x0a, 0x00, 0x54,
			  0x00, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6,
			  0x00, 0x5d, 0x00, 0xc7, 0x00, 0x7e, 0x00, 0x30,
	0x00, 0x80, 0x1f, 0x98, 0x43, 0x3f, 0x0d, 0x88, 0x20, 0x80, 0x3f,
			  0x47, 0xaf, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x0c, 0x00, 0x1c, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x24, 0x00, 0x4a, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0xff, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa0, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xe0, 0x00, 0x0c,
			  0x00, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x8c, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x3f, 0x06, 0xf2, 0x8f, 0xf0,
			  0x40,
	0x10, 0x1a, 0x01, 0x03,
	0x10, 0x00, 0x01, 0xac,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x3b, 0x01,
	0x10, 0x11, 0x08, 0x61, 0x00, 0xe0, 0x00, 0x49, 0x00, 0xa8, 0x00,
	0x10, 0x1f, 0x06, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x1d, 0x02, 0x40, 0x06,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x46, 0x62, 0x76, 0x86, 0x94, 0xa0,
			  0xab, 0xb6, 0xbf, 0xc8, 0xcf, 0xd7, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x12, 0x12,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x46, 0x62, 0x76, 0x86, 0x94, 0xa0,
			  0xab, 0xb6, 0xbf, 0xc8, 0xcf, 0xd7, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x0b, 0x01, 0x09,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x2f,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* nw802 - D-link dru-350c cam */
static const u8 dlink_start[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0x92, 0x03, 0x10, 0x00, 0x4d,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x01, 0x00, 0x16, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x10, 0x00, 0x36, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa1, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xc0, 0x00, 0x14,
			  0x02, 0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x01, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
			  0x40,
	0x10, 0x1a, 0x01, 0x00,
	0x10, 0x00, 0x01, 0xad,
	0x00, 0x00, 0x01, 0x08,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x51, 0x00, 0xf0, 0x00, 0x3d, 0x00, 0xb4, 0x00,
	0x10, 0x1d, 0x08, 0x40, 0x06, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x0e, 0x01, 0x20,
	0x10, 0x41, 0x11, 0x00, 0x07, 0x1e, 0x38, 0x4d, 0x60, 0x70, 0x7f,
			  0x8e, 0x9b, 0xa8, 0xb4, 0xbf, 0xca, 0xd5, 0xdf,
			  0xea,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x11, 0x11,
	0x10, 0x03, 0x01, 0x10,
	0x10, 0x41, 0x11, 0x00, 0x07, 0x1e, 0x38, 0x4d, 0x60, 0x70, 0x7f,
			  0x8e, 0x9b, 0xa8, 0xb4, 0xbf, 0xca, 0xd5, 0xdf,
			  0xea,
	0x10, 0x0b, 0x01, 0x19,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1e,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* 06a5:d001 - nw801 - Sony
 *		Plustek Opticam 500U or ProLink DS3303u (Hitachi HD49322BF) */
/*fixme: 320x240 only*/
static const u8 ds3303_start[] = {
	0x05, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x16, 0x00, 0x00, 0xf9, 0x02, 0x11, 0x00, 0x0e,
			  0x01, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x22, 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x08, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa9, 0xa8, 0x1f, 0x00,
			  0x0d, 0x02, 0x07, 0x00, 0x01, 0x00, 0x19, 0x00,
			  0xf2, 0x00, 0x18, 0x06, 0x10, 0x06, 0x10, 0x00,
			  0x36, 0x00,
	0x02, 0x00, 0x12, 0x03, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0x50,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x05, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0xff, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x2f, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x1f, 0x10, 0x08, 0x0a,
			  0x0a, 0x51, 0x00, 0xf1, 0x00, 0x3c, 0x00, 0xb4,
			  0x00, 0x01, 0x15, 0xfd, 0x07, 0x3d, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x8c, 0x04, 0x01, 0x20,
			  0x02, 0x00, 0x03, 0x00, 0x20, 0x00, 0x00, 0x00,
			  0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08, 0x03,
			  0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06, 0xf7,
	0x10, 0x40, 0x40, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80, 0x80,
			  0x00, 0x2d, 0x46, 0x58, 0x67, 0x74, 0x7f, 0x88,
			  0x94, 0x9d, 0xa6, 0xae, 0xb5, 0xbd, 0xc4, 0xcb,
			  0xd1, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54, 0x64,
			  0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2, 0xe2,
			  0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
	0x10, 0x80, 0x22, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x3f, 0x01,
			  0x00, 0x00, 0xef, 0x00, 0x02, 0x0a, 0x82, 0x02,
			  0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40, 0x01,
			  0xf0, 0x00,

	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x3f, 0x00, 0xf2, 0x8f, 0x81,
			  0x40,
	0x10, 0x1a, 0x01, 0x15,
	0x10, 0x00, 0x01, 0x2f,
	0x10, 0x8c, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x61, 0x00, 0xe0, 0x00, 0x49, 0x00, 0xa8, 0x00,
	0x10, 0x26, 0x06, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x24, 0x02, 0x40, 0x06,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x48, 0x11, 0x00, 0x15, 0x40, 0x67, 0x84, 0x9d, 0xb2, 0xc6,
			  0xd6, 0xe7, 0xf6, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9,
			  0xf9,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x16, 0x16,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x48, 0x11, 0x00, 0x15, 0x40, 0x67, 0x84, 0x9d, 0xb2, 0xc6,
			  0xd6, 0xe7, 0xf6, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9,
			  0xf9,
	0x10, 0x0b, 0x01, 0x26,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1c,
	0x05, 0x06, 0x01, 0x03,
	0x05, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* 06a5:d001 - nw802 - Panasonic
 *		GP-KR651US (Philips TDA8786) */
static const u8 kr651_start_1[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x44, 0x96, 0x98, 0xf9, 0x02, 0x18, 0x00, 0x48,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x0b, 0x00, 0x1b, 0x00, 0xc8, 0x00, 0xf4,
			  0x05, 0xb4, 0x00, 0xcc, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa2, 0x00, 0xc6, 0x00, 0x60, 0x00, 0xc6,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0xae, 0x00, 0xd2, 0x00, 0xae, 0x00, 0xd2,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa8, 0x00, 0xc0, 0x00, 0x66, 0x00, 0xc0,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x0a, 0x00, 0x54, 0x00, 0x0a, 0x00, 0x54,
			  0x00, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6,
			  0x00, 0x5d, 0x00, 0xc7, 0x00, 0x7e, 0x00, 0x30,
	0x00, 0x80, 0x1f, 0x18, 0x43, 0x3f, 0x0d, 0x88, 0x20, 0x80, 0x3f,
			  0x47, 0xaf, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x0c, 0x00, 0x1c, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x24, 0x00, 0x4a, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x02, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa0, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xe0, 0x00, 0x0c,
			  0x00, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0, 0, 0
};
static const u8 kr651_start_qvga[] = {
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
			  0x40,
	0x10, 0x1a, 0x01, 0x03,
	0x10, 0x00, 0x01, 0xac,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x29, 0x00, 0x18, 0x01, 0x1f, 0x00, 0xd2, 0x00,
	0x10, 0x1d, 0x06, 0xe0, 0x00, 0x0c, 0x00, 0x52, 0x00,
	0x10, 0x1d, 0x02, 0x28, 0x01,
	0, 0, 0
};
static const u8 kr651_start_vga[] = {
	0x02, 0x00, 0x11, 0x78, 0xa0, 0x8c, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x30, 0x03, 0x01, 0x82, 0x82, 0x98,
			  0x80,
	0x10, 0x1a, 0x01, 0x03,
	0x10, 0x00, 0x01, 0xa0,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x7f, 0x02, 0x00, 0x00, 0xdf, 0x01,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x51, 0x00, 0x30, 0x02, 0x3d, 0x00, 0xa4, 0x01,
	0x10, 0x1d, 0x06, 0xe0, 0x00, 0x0c, 0x00, 0x52, 0x00,
	0x10, 0x1d, 0x02, 0x68, 0x00,
};
static const u8 kr651_start_2[] = {
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x11, 0x3c, 0x5c, 0x74, 0x88, 0x99, 0xa8,
			  0xb7, 0xc4, 0xd0, 0xdc, 0xdc, 0xdc, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x0c, 0x0c,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x41, 0x11, 0x00, 0x11, 0x3c, 0x5c, 0x74, 0x88, 0x99, 0xa8,
			  0xb7, 0xc4, 0xd0, 0xdc, 0xdc, 0xdc, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x0b, 0x01, 0x10,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x2d,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* nw802 - iRez Kritter cam */
static const u8 kritter_start[] = {
	0x04, 0x06, 0x01, 0x06,
	0x00, 0x00, 0x40, 0x44, 0x96, 0x98, 0x94, 0x03, 0x18, 0x00, 0x48,
			  0x0f, 0x1e, 0x00, 0x0c, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x0b, 0x00, 0x1b, 0x00, 0x0a, 0x01, 0x28,
			  0x07, 0xb4, 0x00, 0xcc, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa2, 0x00, 0xc6, 0x00, 0x60, 0x00, 0xc6,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0xae, 0x00, 0xd2, 0x00, 0xae, 0x00, 0xd2,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa8, 0x00, 0xc0, 0x00, 0x66, 0x00, 0xc0,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x0a, 0x00, 0x54, 0x00, 0x0a, 0x00, 0x54,
			  0x00, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6,
			  0x00, 0x5d, 0x00, 0x0e, 0x00, 0x7e, 0x00, 0x30,
	0x00, 0x80, 0x1f, 0x18, 0x43, 0x3f, 0x0d, 0x88, 0x20, 0x80, 0x3f,
			  0x47, 0xaf, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0b, 0x02, 0x0c, 0x00, 0x1c, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x24, 0x00, 0x4a, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x02, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0xff, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa0, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xe0, 0x00, 0x0c,
			  0x00, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x8c, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x3f, 0x06, 0xf2, 0x8f, 0xf0,
			  0x40,
	0x10, 0x1a, 0x01, 0x03,
	0x10, 0x00, 0x01, 0xaf,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x3b, 0x01,
	0x10, 0x11, 0x08, 0x61, 0x00, 0xe0, 0x00, 0x49, 0x00, 0xa8, 0x00,
	0x10, 0x1d, 0x06, 0xe0, 0x00, 0x0c, 0x00, 0x52, 0x00,
	0x10, 0x1d, 0x02, 0x00, 0x00,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x0d, 0x36, 0x4e, 0x60, 0x6f, 0x7b, 0x86,
			  0x90, 0x98, 0xa1, 0xa9, 0xb1, 0xb7, 0xbe, 0xc4,
			  0xcb,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x0d, 0x0d,
	0x10, 0x03, 0x01, 0x02,
	0x10, 0x41, 0x11, 0x00, 0x0d, 0x36, 0x4e, 0x60, 0x6f, 0x7b, 0x86,
			  0x90, 0x98, 0xa1, 0xa9, 0xb1, 0xb7, 0xbe, 0xc4,
			  0xcb,
	0x10, 0x0b, 0x01, 0x17,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1e,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* nw802 - Mustek Wcam 300 mini */
static const u8 mustek_start[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x10, 0x00, 0x00, 0x92, 0x03, 0x10, 0x00, 0x4d,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb4, 0x6f, 0x3f, 0x0f, 0x88, 0x20, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x01, 0x00, 0x16, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0xfc, 0x05, 0x0c, 0x06,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa1, 0x02, 0x80, 0x00, 0x13, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xc0, 0x00, 0x14,
			  0x02, 0x00, 0x01, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x01, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
			  0x40,
	0x10, 0x1a, 0x01, 0x00,
	0x10, 0x00, 0x01, 0xad,
	0x00, 0x00, 0x01, 0x08,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1d, 0x08, 0x00, 0x20, 0x00, 0x20, 0x00, 0x20, 0x00, 0x20,
	0x10, 0x0e, 0x01, 0x0f,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x29, 0x4a, 0x64, 0x7a, 0x8c, 0x9e,
			  0xad, 0xba, 0xc7, 0xd3, 0xde, 0xe8, 0xf1, 0xf9,
			  0xff,
	0x10, 0x0f, 0x02, 0x11, 0x11,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x29, 0x4a, 0x64, 0x7a, 0x8c, 0x9e,
			  0xad, 0xba, 0xc7, 0xd3, 0xde, 0xe8, 0xf1, 0xf9,
			  0xff,
	0x10, 0x0b, 0x01, 0x1c,
	0x10, 0x0d, 0x01, 0x1a,
	0x10, 0x0c, 0x01, 0x34,
	0x04, 0x05, 0x01, 0x61,
	0x04, 0x04, 0x01, 0x40,
	0x04, 0x06, 0x01, 0x03,
	0, 0, 0
};

/* nw802 - Scope USB Microscope M2 (ProScope) (Hitachi HD49322BF) */
static const u8 proscope_init[] = {
	0x04, 0x05, 0x01, 0x21,
	0x04, 0x04, 0x01, 0x01,
	0, 0, 0
};
static const u8 proscope_start_1[] = {
	0x04, 0x06, 0x01, 0x04,
	0x00, 0x00, 0x40, 0x10, 0x01, 0x00, 0xf9, 0x02, 0x10, 0x00, 0x04,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x08, 0x00, 0x17, 0x00, 0xce, 0x00, 0xf4,
			  0x05, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0xce, 0x00, 0xf8, 0x03, 0x3e, 0x00, 0x86,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0xb6,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xf6, 0x03, 0x34, 0x04, 0xf6, 0x03, 0x34,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xe8,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb4, 0x6f, 0x1f, 0x0f, 0x08, 0x20, 0xa8, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x01, 0x00, 0x19, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x10, 0x00, 0x36, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x21, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xad, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x1f, 0x10, 0x08, 0x0a,
			  0x0a, 0x51, 0x00, 0xf1, 0x00, 0x3c, 0x00, 0xb4,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x8c, 0x04, 0x01,
			  0x20, 0x02, 0x00, 0x03, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x2d, 0x46, 0x58, 0x67, 0x74, 0x7f,
			  0x88, 0x94, 0x9d, 0xa6, 0xae, 0xb5, 0xbd, 0xc4,
			  0xcb, 0xd1, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x3f,
			  0x01, 0x00, 0x00, 0xef, 0x00, 0x09, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0, 0, 0
};
static const u8 proscope_start_qvga[] = {
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x9e, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x10, 0x02, 0xf2, 0x8f, 0x78,
			  0x40,
	0x10, 0x1a, 0x01, 0x06,
	0x00, 0x03, 0x02, 0xf9, 0x02,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1d, 0x08, 0xc0, 0x0d, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x0e, 0x01, 0x10,
	0, 0, 0
};
static const u8 proscope_start_vga[] = {
	0x00, 0x03, 0x02, 0xf9, 0x02,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x7f, 0x02, 0x00, 0x00, 0xdf, 0x01,
	0x02, 0x00, 0x11, 0x78, 0xa0, 0x8c, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x16, 0x00, 0x00, 0x82, 0x84, 0x00,
			  0x80,
	0x10, 0x1a, 0x01, 0x06,
	0x10, 0x00, 0x01, 0xa1,
	0x10, 0x1b, 0x02, 0x00, 0x00,
	0x10, 0x1d, 0x08, 0xc0, 0x0d, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x7f, 0x02, 0x00, 0x00, 0xdf, 0x01,
	0x10, 0x0e, 0x01, 0x10,
	0x10, 0x41, 0x11, 0x00, 0x10, 0x51, 0x6e, 0x83, 0x93, 0xa1, 0xae,
			  0xb9, 0xc3, 0xcc, 0xd4, 0xdd, 0xe4, 0xeb, 0xf2,
			  0xf9,
	0x10, 0x03, 0x01, 0x00,
	0, 0, 0
};
static const u8 proscope_start_2[] = {
	0x10, 0x0f, 0x02, 0x0c, 0x0c,
	0x10, 0x03, 0x01, 0x0c,
	0x10, 0x41, 0x11, 0x00, 0x10, 0x51, 0x6e, 0x83, 0x93, 0xa1, 0xae,
			  0xb9, 0xc3, 0xcc, 0xd4, 0xdd, 0xe4, 0xeb, 0xf2,
			  0xf9,
	0x10, 0x0b, 0x01, 0x0b,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1b,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x05, 0x01, 0x21,
	0x04, 0x04, 0x01, 0x00,
	0, 0, 0
};

/* nw800 - hv7121b? (seems pas106) - Divio Chicony TwinkleCam */
static const u8 twinkle_start[] = {
	0x04, 0x06, 0x01, 0x44,
	0x04, 0x06, 0x01, 0x00,
	0x00, 0x00, 0x40, 0x14, 0x83, 0x00, 0xba, 0x01, 0x10, 0x00, 0x4f,
			  0xef, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x06, 0x00, 0xfc,
			  0x01, 0x3e, 0x00, 0x86, 0x00, 0x3e, 0x00, 0x86,
			  0x00, 0x3e, 0x00, 0x86, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x56, 0x00, 0x9e,
			  0x00, 0x56, 0x00, 0x9e, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x6e, 0x00, 0xb6, 0x00, 0x6e, 0x00, 0x78,
			  0x04, 0x6e, 0x00, 0xb6, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xca, 0x03, 0x46, 0x04, 0xca, 0x03, 0x46,
			  0x04, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0xf0,
			  0x00, 0x3e, 0x00, 0xaa, 0x00, 0x88, 0x00, 0x2e,
	0x00, 0x80, 0x1f, 0xb8, 0x48, 0x0f, 0x04, 0x88, 0x14, 0x68, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0xa8, 0x01, 0x00, 0x03,
			  0x00, 0x24, 0x01, 0x01, 0x00, 0x16, 0x00, 0x04,
			  0x00, 0x4b, 0x00, 0x76, 0x00, 0x86, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0x61, 0x00,
	0x05, 0x00, 0x06, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x06, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0x80, 0x02, 0x20, 0x00, 0x11, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x08,
			  0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x10, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x00, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1d, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x62,
			  0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01, 0x20,
			  0x01, 0x60, 0x01, 0x00, 0x00,

	0x10, 0x85, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	0x04, 0x04, 0x01, 0x10,
	0x04, 0x04, 0x01, 0x00,
	0x04, 0x05, 0x01, 0x61,
	0x04, 0x04, 0x01, 0x01,
	I2C0, 0x40, 0x0c, 0x02, 0x0c, 0x12, 0x07, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x0a,
	I2C0, 0x40, 0x02, 0x11, 0x06,
	I2C0, 0x40, 0x02, 0x14, 0x00,
	I2C0, 0x40, 0x02, 0x13, 0x01,		/* i2c end */
	I2C0, 0x40, 0x02, 0x07, 0x01,
	0x02, 0x00, 0x11, 0x48, 0x58, 0x9e, 0x48, 0x58, 0x00, 0x00, 0x00,
			  0x00, 0x84, 0x36, 0x05, 0x01, 0xf2, 0x86, 0x65,
			  0x40,
	I2C0, 0x40, 0x02, 0x02, 0x0c,
	I2C0, 0x40, 0x02, 0x13, 0x01,
	0x10, 0x00, 0x01, 0x01,
	0x10, 0x8f, 0x0c, 0x62, 0x01, 0x24, 0x01, 0x62, 0x01, 0x24, 0x01,
			  0x20, 0x01, 0x60, 0x01,
	I2C0, 0x40, 0x02, 0x05, 0x0f,
	I2C0, 0x40, 0x02, 0x13, 0x01,
	I2C0, 0x40, 0x08, 0x08, 0x04, 0x0b, 0x01, 0x01, 0x02, 0x00, 0x17,
	I2C0, 0x40, 0x03, 0x12, 0x00, 0x01,
	0x10, 0x11, 0x08, 0x00, 0x00, 0x5f, 0x01, 0x00, 0x00, 0x1f, 0x01,
	I2C0, 0x40, 0x02, 0x12, 0x00,
	I2C0, 0x40, 0x02, 0x0e, 0x00,
	I2C0, 0x40, 0x02, 0x11, 0x06,
	0x10, 0x41, 0x11, 0x00, 0x17, 0x3f, 0x69, 0x7b, 0x8c, 0x9a, 0xa7,
			  0xb3, 0xbf, 0xc9, 0xd3, 0xdd, 0xe6, 0xef, 0xf7,
			  0xf9,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x0c, 0x0c,
	0x10, 0x03, 0x01, 0x06,
	0x10, 0x41, 0x11, 0x00, 0x17, 0x3f, 0x69, 0x7b, 0x8c, 0x9a, 0xa7,
			  0xb3, 0xbf, 0xc9, 0xd3, 0xdd, 0xe6, 0xef, 0xf7,
			  0xf9,
	0x10, 0x0b, 0x01, 0x19,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x0d,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x05, 0x01, 0x61,
	0x04, 0x04, 0x01, 0x41,
	0, 0, 0
};

/* nw802 dvc-v6 */
static const u8 dvcv6_start[] = {
	0x04, 0x06, 0x01, 0x06,
	0x00, 0x00, 0x40, 0x54, 0x96, 0x98, 0xf9, 0x02, 0x18, 0x00, 0x4c,
			  0x0f, 0x1f, 0x00, 0x0d, 0x02, 0x01, 0x00, 0x19,
			  0x00, 0x01, 0x00, 0x19, 0x00, 0x01, 0x00, 0x19,
			  0x00, 0x0b, 0x00, 0x1b, 0x00, 0xc8, 0x00, 0xf4,
			  0x05, 0xb4, 0x00, 0xcc, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa2, 0x00, 0xc6, 0x00, 0x60, 0x00, 0xc6,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
	0x00, 0x40, 0x40, 0x00, 0xae, 0x00, 0xd2, 0x00, 0xae, 0x00, 0xd2,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0xa8, 0x00, 0xc0, 0x00, 0x66, 0x00, 0xc0,
			  0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01,
			  0x00, 0x0a, 0x00, 0x54, 0x00, 0x0a, 0x00, 0x54,
			  0x00, 0x10, 0x00, 0x36, 0x00, 0xd2, 0x00, 0xee,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf6,
			  0x00, 0x5d, 0x00, 0xc7, 0x00, 0x7e, 0x00, 0x30,
	0x00, 0x80, 0x1f, 0x98, 0x43, 0x3f, 0x0d, 0x88, 0x20, 0x80, 0x3f,
			  0x47, 0xaf, 0x00, 0x00, 0xa8, 0x08, 0x00, 0x11,
			  0x00, 0x0c, 0x02, 0x0c, 0x00, 0x1c, 0x00, 0x94,
			  0x00, 0x10, 0x06, 0x24, 0x00, 0x4a, 0x00,
	0x02, 0x00, 0x12, 0x78, 0xa0, 0x9e, 0x78, 0xa0, 0x00, 0x00, 0x00,
			  0x00, 0xf0, 0x18, 0x0b, 0x06, 0x62, 0x82, 0xa0,
			  0x40, 0x20,
	0x03, 0x00, 0x03, 0x03, 0x00, 0x00,
	0x04, 0x00, 0x07, 0x01, 0x10, 0x00, 0x00, 0x00, 0xff, 0x00,
	0x06, 0x00, 0x02, 0x09, 0x99,
	0x08, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x40, 0xa0, 0x02, 0x80, 0x00, 0x12, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x10, 0x10, 0x10, 0x08, 0x0a,
			  0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  0x00, 0x49, 0x13, 0x00, 0x00, 0xe0, 0x00, 0x0c,
			  0x00, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
			  0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x10, 0x08,
			  0x03, 0x00, 0x00, 0x00, 0x00, 0x20, 0x10, 0x06,
			  0xf7, 0xee, 0x1c, 0x1c, 0xe9, 0xfc, 0x10, 0x80,
	0x10, 0x40, 0x40, 0x80, 0x00, 0x05, 0x35, 0x5e, 0x78, 0x8b, 0x99,
			  0xa4, 0xae, 0xb5, 0xbc, 0xc1, 0xc6, 0xc9, 0xcc,
			  0xcf, 0xd0, 0x00, 0x11, 0x22, 0x32, 0x43, 0x54,
			  0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3, 0xd2,
			  0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32, 0x43,
			  0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3, 0xc3,
			  0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x11, 0x22, 0x32,
			  0x43, 0x54, 0x64, 0x74, 0x84, 0x94, 0xa4, 0xb3,
	0x10, 0x80, 0x1b, 0xc3, 0xd2, 0xe2, 0xf1, 0xff, 0x00, 0x00, 0x00,
			  0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x05, 0x82,
			  0x02, 0xe4, 0x01, 0x40, 0x01, 0xf0, 0x00, 0x40,
			  0x01, 0xf0, 0x00,
	0x00, 0x03, 0x02, 0x94, 0x03,
	0x00, 0x1d, 0x04, 0x0a, 0x01, 0x28, 0x07,
	0x00, 0x7b, 0x02, 0xe0, 0x00,
	0x10, 0x8d, 0x01, 0x00,
	0x00, 0x09, 0x04, 0x1e, 0x00, 0x0c, 0x02,
	0x00, 0x91, 0x02, 0x0b, 0x02,
	0x10, 0x00, 0x01, 0xaf,
	0x02, 0x00, 0x11, 0x3c, 0x50, 0x8f, 0x3c, 0x50, 0x00, 0x00, 0x00,
			  0x00, 0x78, 0x3f, 0x3f, 0x06, 0xf2, 0x8f, 0xf0,
			  0x40,
	0x10, 0x1a, 0x01, 0x02,
	0x10, 0x00, 0x01, 0xaf,
	0x10, 0x85, 0x08, 0x00, 0x00, 0x3f, 0x01, 0x00, 0x00, 0xef, 0x00,
	0x10, 0x1b, 0x02, 0x07, 0x01,
	0x10, 0x11, 0x08, 0x61, 0x00, 0xe0, 0x00, 0x49, 0x00, 0xa8, 0x00,
	0x10, 0x1f, 0x06, 0x01, 0x20, 0x02, 0xe8, 0x03, 0x00,
	0x10, 0x1d, 0x02, 0x40, 0x06,
	0x10, 0x0e, 0x01, 0x08,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x54, 0x6f, 0x82, 0x91, 0x9f, 0xaa,
			  0xb4, 0xbd, 0xc5, 0xcd, 0xd5, 0xdb, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x03, 0x01, 0x00,
	0x10, 0x0f, 0x02, 0x12, 0x12,
	0x10, 0x03, 0x01, 0x11,
	0x10, 0x41, 0x11, 0x00, 0x0f, 0x54, 0x6f, 0x82, 0x91, 0x9f, 0xaa,
			  0xb4, 0xbd, 0xc5, 0xcd, 0xd5, 0xdb, 0xdc, 0xdc,
			  0xdc,
	0x10, 0x0b, 0x01, 0x16,
	0x10, 0x0d, 0x01, 0x10,
	0x10, 0x0c, 0x01, 0x1a,
	0x04, 0x06, 0x01, 0x03,
	0x04, 0x04, 0x01, 0x00,
};

static const u8 *webcam_start[] = {
	[Generic800] = nw800_start,
	[SpaceCam] = spacecam_start,
	[SpaceCam2] = spacecam2_start,
	[Cvideopro] = cvideopro_start,
	[Dlink350c] = dlink_start,
	[DS3303u] = ds3303_start,
	[Kr651us] = kr651_start_1,
	[Kritter] = kritter_start,
	[Mustek300] = mustek_start,
	[Proscope] = proscope_start_1,
	[Twinkle] = twinkle_start,
	[DvcV6] = dvcv6_start,
	[P35u] = nw801_start_1,
	[Generic802] = nw802_start,
};

/* -- write a register -- */
static void reg_w(struct gspca_dev *gspca_dev,
			u16 index,
			const u8 *data,
			int len)
{
	struct usb_device *dev = gspca_dev->dev;
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	if (len == 1)
		gspca_dbg(gspca_dev, D_USBO, "SET 00 0000 %04x %02x\n",
			  index, *data);
	else
		gspca_dbg(gspca_dev, D_USBO, "SET 00 0000 %04x %02x %02x ...\n",
			  index, *data, data[1]);
	memcpy(gspca_dev->usb_buf, data, len);
	ret = usb_control_msg(dev, usb_sndctrlpipe(dev, 0),
			0x00,
			USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			0x00,		/* value */
			index,
			gspca_dev->usb_buf,
			len,
			500);
	if (ret < 0) {
		pr_err("reg_w err %d\n", ret);
		gspca_dev->usb_err = ret;
	}
}

/* -- read registers in usb_buf -- */
static void reg_r(struct gspca_dev *gspca_dev,
			u16 index,
			int len)
{
	struct usb_device *dev = gspca_dev->dev;
	int ret;

	if (gspca_dev->usb_err < 0)
		return;
	ret = usb_control_msg(dev, usb_rcvctrlpipe(dev, 0),
			0x00,
			USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_DEVICE,
			0x00, index,
			gspca_dev->usb_buf, len, 500);
	if (ret < 0) {
		pr_err("reg_r err %d\n", ret);
		gspca_dev->usb_err = ret;
		return;
	}
	if (len == 1)
		gspca_dbg(gspca_dev, D_USBI, "GET 00 0000 %04x %02x\n",
			  index, gspca_dev->usb_buf[0]);
	else
		gspca_dbg(gspca_dev, D_USBI, "GET 00 0000 %04x %02x %02x ..\n",
			  index, gspca_dev->usb_buf[0],
			  gspca_dev->usb_buf[1]);
}

static void i2c_w(struct gspca_dev *gspca_dev,
			u8 i2c_addr,
			const u8 *data,
			int len)
{
	u8 val[2];
	int i;

	reg_w(gspca_dev, 0x0600, data + 1, len - 1);
	reg_w(gspca_dev, 0x0600, data, len);
	val[0] = len;
	val[1] = i2c_addr;
	reg_w(gspca_dev, 0x0502, val, 2);
	val[0] = 0x01;
	reg_w(gspca_dev, 0x0501, val, 1);
	for (i = 5; --i >= 0; ) {
		msleep(4);
		reg_r(gspca_dev, 0x0505, 1);
		if (gspca_dev->usb_err < 0)
			return;
		if (gspca_dev->usb_buf[0] == 0)
			return;
	}
	gspca_dev->usb_err = -ETIME;
}

static void reg_w_buf(struct gspca_dev *gspca_dev,
			const u8 *cmd)
{
	u16 reg;
	int len;

	for (;;) {
		reg = *cmd++ << 8;
		reg += *cmd++;
		len = *cmd++;
		if (len == 0)
			break;
		if (cmd[-3] != I2C0)
			reg_w(gspca_dev, reg, cmd, len);
		else
			i2c_w(gspca_dev, reg, cmd, len);
		cmd += len;
	}
}

static int swap_bits(int v)
{
	int r, i;

	r = 0;
	for (i = 0; i < 8; i++) {
		r <<= 1;
		if (v & 1)
			r++;
		v >>= 1;
	}
	return r;
}

static void setgain(struct gspca_dev *gspca_dev, u8 val)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 v[2];

	switch (sd->webcam) {
	case P35u:
		reg_w(gspca_dev, 0x1026, &val, 1);
		break;
	case Kr651us:
		/* 0 - 253 */
		val = swap_bits(val);
		v[0] = val << 3;
		v[1] = val >> 5;
		reg_w(gspca_dev, 0x101d, v, 2);	/* SIF reg0/1 (AGC) */
		break;
	}
}

static void setexposure(struct gspca_dev *gspca_dev, s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 v[2];

	switch (sd->webcam) {
	case P35u:
		v[0] = ((9 - val) << 3) | 0x01;
		reg_w(gspca_dev, 0x1019, v, 1);
		break;
	case Cvideopro:
	case DvcV6:
	case Kritter:
	case Kr651us:
		v[0] = val;
		v[1] = val >> 8;
		reg_w(gspca_dev, 0x101b, v, 2);
		break;
	}
}

static void setautogain(struct gspca_dev *gspca_dev, s32 val)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int w, h;

	if (!val) {
		sd->ag_cnt = -1;
		return;
	}
	sd->ag_cnt = AG_CNT_START;

	reg_r(gspca_dev, 0x1004, 1);
	if (gspca_dev->usb_buf[0] & 0x04) {	/* if AE_FULL_FRM */
		sd->ae_res = gspca_dev->pixfmt.width * gspca_dev->pixfmt.height;
	} else {				/* get the AE window size */
		reg_r(gspca_dev, 0x1011, 8);
		w = (gspca_dev->usb_buf[1] << 8) + gspca_dev->usb_buf[0]
		  - (gspca_dev->usb_buf[3] << 8) - gspca_dev->usb_buf[2];
		h = (gspca_dev->usb_buf[5] << 8) + gspca_dev->usb_buf[4]
		  - (gspca_dev->usb_buf[7] << 8) - gspca_dev->usb_buf[6];
		sd->ae_res = h * w;
		if (sd->ae_res == 0)
			sd->ae_res = gspca_dev->pixfmt.width *
					gspca_dev->pixfmt.height;
	}
}

static int nw802_test_reg(struct gspca_dev *gspca_dev,
			u16 index,
			u8 value)
{
	/* write the value */
	reg_w(gspca_dev, index, &value, 1);

	/* read it */
	reg_r(gspca_dev, index, 1);

	return gspca_dev->usb_buf[0] == value;
}

/* this function is called at probe time */
static int sd_config(struct gspca_dev *gspca_dev,
			const struct usb_device_id *id)
{
	struct sd *sd = (struct sd *) gspca_dev;

	if ((unsigned) webcam >= NWEBCAMS)
		webcam = 0;
	sd->webcam = webcam;
	gspca_dev->cam.needs_full_bandwidth = 1;
	sd->ag_cnt = -1;

	/*
	 * Autodetect sequence inspired from some log.
	 * We try to detect what registers exist or not.
	 * If 0x0500 does not exist => NW802
	 * If it does, test 0x109b. If it doesn't exist,
	 * then it's a NW801. Else, a NW800
	 * If a et31x110 (nw800 and 06a5:d800)
	 *	get the sensor ID
	 */
	if (!nw802_test_reg(gspca_dev, 0x0500, 0x55)) {
		sd->bridge = BRIDGE_NW802;
		if (sd->webcam == Generic800)
			sd->webcam = Generic802;
	} else if (!nw802_test_reg(gspca_dev, 0x109b, 0xaa)) {
		sd->bridge = BRIDGE_NW801;
		if (sd->webcam == Generic800)
			sd->webcam = P35u;
	} else if (id->idVendor == 0x06a5 && id->idProduct == 0xd800) {
		reg_r(gspca_dev, 0x0403, 1);		/* GPIO */
		gspca_dbg(gspca_dev, D_PROBE, "et31x110 sensor type %02x\n",
			  gspca_dev->usb_buf[0]);
		switch (gspca_dev->usb_buf[0] >> 1) {
		case 0x00:				/* ?? */
			if (sd->webcam == Generic800)
				sd->webcam = SpaceCam;
			break;
		case 0x01:				/* Hynix? */
			if (sd->webcam == Generic800)
				sd->webcam = Twinkle;
			break;
		case 0x0a:				/* Pixart */
			if (sd->webcam == Generic800)
				sd->webcam = SpaceCam2;
			break;
		}
	}
	if (webcam_chip[sd->webcam] != sd->bridge) {
		pr_err("Bad webcam type %d for NW80%d\n",
		       sd->webcam, sd->bridge);
		gspca_dev->usb_err = -ENODEV;
		return gspca_dev->usb_err;
	}
	gspca_dbg(gspca_dev, D_PROBE, "Bridge nw80%d - type: %d\n",
		  sd->bridge, sd->webcam);

	if (sd->bridge == BRIDGE_NW800) {
		switch (sd->webcam) {
		case DS3303u:
			gspca_dev->cam.cam_mode = cif_mode;	/* qvga */
			break;
		default:
			gspca_dev->cam.cam_mode = &cif_mode[1];	/* cif */
			break;
		}
		gspca_dev->cam.nmodes = 1;
	} else {
		gspca_dev->cam.cam_mode = vga_mode;
		switch (sd->webcam) {
		case Kr651us:
		case Proscope:
		case P35u:
			gspca_dev->cam.nmodes = ARRAY_SIZE(vga_mode);
			break;
		default:
			gspca_dev->cam.nmodes = 1;	/* qvga only */
			break;
		}
	}

	return gspca_dev->usb_err;
}

/* this function is called at probe and resume time */
static int sd_init(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;

	switch (sd->bridge) {
	case BRIDGE_NW800:
		switch (sd->webcam) {
		case SpaceCam:
			reg_w_buf(gspca_dev, spacecam_init);
			break;
		default:
			reg_w_buf(gspca_dev, nw800_init);
			break;
		}
		break;
	default:
		switch (sd->webcam) {
		case Mustek300:
		case P35u:
		case Proscope:
			reg_w_buf(gspca_dev, proscope_init);
			break;
		}
		break;
	}
	return gspca_dev->usb_err;
}

/* -- start the camera -- */
static int sd_start(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	const u8 *cmd;

	cmd = webcam_start[sd->webcam];
	reg_w_buf(gspca_dev, cmd);
	switch (sd->webcam) {
	case P35u:
		if (gspca_dev->pixfmt.width == 320)
			reg_w_buf(gspca_dev, nw801_start_qvga);
		else
			reg_w_buf(gspca_dev, nw801_start_vga);
		reg_w_buf(gspca_dev, nw801_start_2);
		break;
	case Kr651us:
		if (gspca_dev->pixfmt.width == 320)
			reg_w_buf(gspca_dev, kr651_start_qvga);
		else
			reg_w_buf(gspca_dev, kr651_start_vga);
		reg_w_buf(gspca_dev, kr651_start_2);
		break;
	case Proscope:
		if (gspca_dev->pixfmt.width == 320)
			reg_w_buf(gspca_dev, proscope_start_qvga);
		else
			reg_w_buf(gspca_dev, proscope_start_vga);
		reg_w_buf(gspca_dev, proscope_start_2);
		break;
	}

	sd->exp_too_high_cnt = 0;
	sd->exp_too_low_cnt = 0;
	return gspca_dev->usb_err;
}

static void sd_stopN(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	u8 value;

	/* 'go' off */
	if (sd->bridge != BRIDGE_NW801) {
		value = 0x02;
		reg_w(gspca_dev, 0x0406, &value, 1);
	}

	/* LED off */
	switch (sd->webcam) {
	case Cvideopro:
	case Kr651us:
	case DvcV6:
	case Kritter:
		value = 0xff;
		break;
	case Dlink350c:
		value = 0x21;
		break;
	case SpaceCam:
	case SpaceCam2:
	case Proscope:
	case Twinkle:
		value = 0x01;
		break;
	default:
		return;
	}
	reg_w(gspca_dev, 0x0404, &value, 1);
}

static void sd_pkt_scan(struct gspca_dev *gspca_dev,
			u8 *data,			/* isoc packet */
			int len)			/* iso packet length */
{
	/*
	 * frame header = '00 00 hh ww ss xx ff ff'
	 * with:
	 *	- 'hh': height / 4
	 *	- 'ww': width / 4
	 *	- 'ss': frame sequence number c0..dd
	 */
	if (data[0] == 0x00 && data[1] == 0x00
	 && data[6] == 0xff && data[7] == 0xff) {
		gspca_frame_add(gspca_dev, LAST_PACKET, NULL, 0);
		gspca_frame_add(gspca_dev, FIRST_PACKET, data + 8, len - 8);
	} else {
		gspca_frame_add(gspca_dev, INTER_PACKET, data, len);
	}
}

static void do_autogain(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *) gspca_dev;
	int luma;

	if (sd->ag_cnt < 0)
		return;
	if (--sd->ag_cnt >= 0)
		return;
	sd->ag_cnt = AG_CNT_START;

	/* get the average luma */
	reg_r(gspca_dev, sd->bridge == BRIDGE_NW801 ? 0x080d : 0x080c, 4);
	luma = (gspca_dev->usb_buf[3] << 24) + (gspca_dev->usb_buf[2] << 16)
		+ (gspca_dev->usb_buf[1] << 8) + gspca_dev->usb_buf[0];
	luma /= sd->ae_res;

	switch (sd->webcam) {
	case P35u:
		gspca_coarse_grained_expo_autogain(gspca_dev, luma, 100, 5);
		break;
	default:
		gspca_expo_autogain(gspca_dev, luma, 100, 5, 230, 0);
		break;
	}
}


static int sd_s_ctrl(struct v4l2_ctrl *ctrl)
{
	struct gspca_dev *gspca_dev =
		container_of(ctrl->handler, struct gspca_dev, ctrl_handler);

	gspca_dev->usb_err = 0;

	if (!gspca_dev->streaming)
		return 0;

	switch (ctrl->id) {
	/* autogain/gain/exposure control cluster */
	case V4L2_CID_AUTOGAIN:
		if (ctrl->is_new)
			setautogain(gspca_dev, ctrl->val);
		if (!ctrl->val) {
			if (gspca_dev->gain->is_new)
				setgain(gspca_dev, gspca_dev->gain->val);
			if (gspca_dev->exposure->is_new)
				setexposure(gspca_dev,
					    gspca_dev->exposure->val);
		}
		break;
	/* Some webcams only have exposure, so handle that separately from the
	   autogain/gain/exposure cluster in the previous case. */
	case V4L2_CID_EXPOSURE:
		setexposure(gspca_dev, gspca_dev->exposure->val);
		break;
	}
	return gspca_dev->usb_err;
}

static const struct v4l2_ctrl_ops sd_ctrl_ops = {
	.s_ctrl = sd_s_ctrl,
};

static int sd_init_controls(struct gspca_dev *gspca_dev)
{
	struct sd *sd = (struct sd *)gspca_dev;
	struct v4l2_ctrl_handler *hdl = &gspca_dev->ctrl_handler;

	gspca_dev->vdev.ctrl_handler = hdl;
	v4l2_ctrl_handler_init(hdl, 3);
	switch (sd->webcam) {
	case P35u:
		gspca_dev->autogain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_AUTOGAIN, 0, 1, 1, 1);
		/* For P35u choose coarse expo auto gain function gain minimum,
		 * to avoid a large settings jump the first auto adjustment */
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_GAIN, 0, 127, 1, 127 / 5 * 2);
		gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_EXPOSURE, 0, 9, 1, 9);
		break;
	case Kr651us:
		gspca_dev->autogain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_AUTOGAIN, 0, 1, 1, 1);
		gspca_dev->gain = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_GAIN, 0, 253, 1, 128);
		/* fall through */
	case Cvideopro:
	case DvcV6:
	case Kritter:
		gspca_dev->exposure = v4l2_ctrl_new_std(hdl, &sd_ctrl_ops,
			V4L2_CID_EXPOSURE, 0, 315, 1, 150);
		break;
	default:
		break;
	}

	if (hdl->error) {
		pr_err("Could not initialize controls\n");
		return hdl->error;
	}
	if (gspca_dev->autogain)
		v4l2_ctrl_auto_cluster(3, &gspca_dev->autogain, 0, false);
	return 0;
}

/* sub-driver description */
static const struct sd_desc sd_desc = {
	.name = MODULE_NAME,
	.config = sd_config,
	.init = sd_init,
	.init_controls = sd_init_controls,
	.start = sd_start,
	.stopN = sd_stopN,
	.pkt_scan = sd_pkt_scan,
	.dq_callback = do_autogain,
};

/* -- module initialisation -- */
static const struct usb_device_id device_table[] = {
	{USB_DEVICE(0x046d, 0xd001)},
	{USB_DEVICE(0x0502, 0xd001)},
	{USB_DEVICE(0x052b, 0xd001)},
	{USB_DEVICE(0x055f, 0xd001)},
	{USB_DEVICE(0x06a5, 0x0000)},
	{USB_DEVICE(0x06a5, 0xd001)},
	{USB_DEVICE(0x06a5, 0xd800)},
	{USB_DEVICE(0x06be, 0xd001)},
	{USB_DEVICE(0x0728, 0xd001)},
	{}
};
MODULE_DEVICE_TABLE(usb, device_table);

/* -- device connect -- */
static int sd_probe(struct usb_interface *intf,
			const struct usb_device_id *id)
{
	return gspca_dev_probe(intf, id, &sd_desc, sizeof(struct sd),
				THIS_MODULE);
}

static struct usb_driver sd_driver = {
	.name = MODULE_NAME,
	.id_table = device_table,
	.probe = sd_probe,
	.disconnect = gspca_disconnect,
#ifdef CONFIG_PM
	.suspend = gspca_suspend,
	.resume = gspca_resume,
	.reset_resume = gspca_resume,
#endif
};

module_usb_driver(sd_driver);

module_param(webcam, int, 0644);
MODULE_PARM_DESC(webcam,
	"Webcam type\n"
	"0: generic\n"
	"1: Trust 120 SpaceCam\n"
	"2: other Trust 120 SpaceCam\n"
	"3: Conceptronic Video Pro\n"
	"4: D-link dru-350c\n"
	"5: Plustek Opticam 500U\n"
	"6: Panasonic GP-KR651US\n"
	"7: iRez Kritter\n"
	"8: Mustek Wcam 300 mini\n"
	"9: Scalar USB Microscope M2 (Proscope)\n"
	"10: Divio Chicony TwinkleCam\n"
	"11: DVC-V6\n");
