/************************************* */
/* Rule Set Based Access Control       */
/* Author and (c) 1999-2003:           */
/*   Amon Ott <ao@rsbac.org>           */
/* Syscall wrapper functions for all   */
/* parts                               */
/* Last modified: 21/Jul/2003          */
/************************************* */

#ifndef __RSBAC_SYSCALLS_H
#define __RSBAC_SYSCALLS_H

#include <linux/unistd.h>
#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/helpers.h>
#include <rsbac/error.h>
#ifndef __KERNEL__
#include <errno.h>
#endif

enum rsbac_syscall_t
  {
    RSYS_stats,
    RSYS_check,
    RSYS_get_attr,
    RSYS_get_attr_n,
    RSYS_set_attr,
    RSYS_set_attr_n,
    RSYS_remove_target,
    RSYS_remove_target_n,
    RSYS_net_list_all_netdev,
    RSYS_net_template,
    RSYS_net_list_all_template,
    RSYS_switch,
    RSYS_adf_log_switch,
    RSYS_get_adf_log,
    RSYS_write,
    RSYS_log,
    RSYS_mac_set_curr_level,
    RSYS_mac_get_curr_level,
    RSYS_mac_get_max_level,
    RSYS_mac_get_min_level,
    RSYS_stats_pm,
    RSYS_pm,
    RSYS_pm_change_current_task,
    RSYS_pm_create_file,
    RSYS_ms_get_copied,
    RSYS_rc_copy_role,
    RSYS_rc_get_item,
    RSYS_rc_set_item,
    RSYS_rc_change_role,
    RSYS_rc_get_eff_rights_n,
    RSYS_rc_get_list,
    RSYS_auth_add_p_cap,
    RSYS_auth_remove_p_cap,
    RSYS_auth_add_f_cap,
    RSYS_auth_remove_f_cap,
    RSYS_auth_get_f_caplist,
    RSYS_acl,
    RSYS_acl_n,
    RSYS_acl_get_rights,
    RSYS_acl_get_rights_n,
    RSYS_acl_get_tlist,
    RSYS_acl_get_tlist_n,
    RSYS_acl_get_mask,
    RSYS_acl_get_mask_n,
    RSYS_acl_group,
    RSYS_reg,
    RSYS_jail,
    RSYS_init,
    RSYS_none
  };


struct rsys_check_t
  {
    int correct;
    int check_inode;
  };

struct rsys_get_attr_t
  {
    enum  rsbac_switch_target_t module;
    enum  rsbac_target_t target;
    union rsbac_target_id_t * tid;
    enum  rsbac_attribute_t attr;
    union rsbac_attribute_value_t * value;
          int inherit;
  };

struct rsys_get_attr_n_t
  {
    enum  rsbac_switch_target_t module;
    enum  rsbac_target_t target;
          char * t_name;
    enum  rsbac_attribute_t attr;
    union rsbac_attribute_value_t * value;
          int inherit;
  };

struct rsys_set_attr_t
  {
    enum  rsbac_switch_target_t module;
    enum  rsbac_target_t target;
    union rsbac_target_id_t * tid;
    enum  rsbac_attribute_t attr;
    union rsbac_attribute_value_t * value;
  };

struct rsys_set_attr_n_t
  {
    enum  rsbac_switch_target_t module;
    enum  rsbac_target_t target;
          char * t_name;
    enum  rsbac_attribute_t attr;
    union rsbac_attribute_value_t * value;
  };

struct rsys_remove_target_t
  {
    enum  rsbac_target_t target;
    union rsbac_target_id_t * tid;
  };

struct rsys_remove_target_n_t
  {
    enum rsbac_target_t target;
         char * t_name;
  };

struct rsys_net_list_all_netdev_t
  {
    rsbac_netdev_id_t * id_p;
    u_long maxnum;
  };

struct rsys_net_template_t
  {
    enum  rsbac_net_temp_syscall_t call;
          rsbac_net_temp_id_t id;
    union rsbac_net_temp_syscall_data_t * data_p;
  };

struct rsys_net_list_all_template_t
  {
    rsbac_net_temp_id_t * id_p;
    u_long maxnum;
  };

struct rsys_switch_t
  {
    enum rsbac_switch_target_t target;
         int value;
  };

struct rsys_adf_log_switch_t
  {
    enum rsbac_adf_request_t request;
    enum rsbac_target_t      target;
         u_int               value;
  };

struct rsys_get_adf_log_t
  {
    enum rsbac_adf_request_t   request;
    enum rsbac_target_t        target;
         u_int               * value_p;
  };

struct rsys_log_t
  {
    int type;
    char * buf;
    int len;
  };

struct rsys_mac_set_curr_level_t
  {
    rsbac_security_level_t level;
    rsbac_mac_category_vector_t * categories_p;
  };

struct rsys_mac_get_curr_level_t
  {
    rsbac_security_level_t      * level_p;
    rsbac_mac_category_vector_t * categories_p;
  };

struct rsys_mac_get_max_level_t
  {
    rsbac_security_level_t      * level_p;
    rsbac_mac_category_vector_t * categories_p;
  };

struct rsys_mac_get_min_level_t
  {
    rsbac_security_level_t      * level_p;
    rsbac_mac_category_vector_t * categories_p;
  };

struct rsys_pm_t
  {
    enum  rsbac_pm_function_type_t function;
    union rsbac_pm_function_param_t * param_p;
          rsbac_pm_tkt_id_t ticket;
  };

struct rsys_pm_change_current_task_t
  {
    rsbac_pm_task_id_t task;
  };

struct rsys_pm_create_file_t
  {
    const char * filename;
    int mode;
    rsbac_pm_object_class_id_t object_class;
  };

struct rsys_ms_get_copied_t
  {
    int fd;
  };

struct rsys_rc_copy_role_t
  {
    rsbac_rc_role_id_t from_role;
    rsbac_rc_role_id_t to_role;
  };

struct rsys_rc_get_item_t
  {
    enum  rsbac_rc_target_t target;
    union rsbac_rc_target_id_t * tid_p;
    union rsbac_rc_target_id_t * subtid_p;
    enum  rsbac_rc_item_t item;
    union rsbac_rc_item_value_t * value_p;
          rsbac_time_t * ttl_p;
  };

struct rsys_rc_set_item_t
  {
    enum  rsbac_rc_target_t target;
    union rsbac_rc_target_id_t * tid_p;
    union rsbac_rc_target_id_t * subtid_p;
    enum  rsbac_rc_item_t item;
    union rsbac_rc_item_value_t * value_p;
          rsbac_time_t ttl;
  };

struct rsys_rc_get_list_t
  {
    enum  rsbac_rc_target_t target;
    union rsbac_rc_target_id_t * tid_p;
    enum  rsbac_rc_item_t item;
          u_int maxnum;
          __u32 * array_p;
          rsbac_time_t * ttl_array_p;
  };

struct rsys_rc_change_role_t
  {
    rsbac_rc_role_id_t role;
  };

struct rsys_rc_get_eff_rights_n_t
  {
    enum rsbac_target_t target;
         char * t_name;
         rsbac_rc_request_vector_t * request_vector_p;
         rsbac_time_t * ttl_p;
  };

struct rsys_auth_add_p_cap_t
  {
    rsbac_pid_t pid;
    enum rsbac_auth_cap_type_t cap_type;
    struct rsbac_auth_cap_range_t cap_range;
    rsbac_time_t ttl;
  };

struct rsys_auth_remove_p_cap_t
  {
    rsbac_pid_t pid;
    enum rsbac_auth_cap_type_t cap_type;
    struct rsbac_auth_cap_range_t cap_range;
  };

struct rsys_auth_add_f_cap_t
  {
    char * filename;
    enum rsbac_auth_cap_type_t cap_type;
    struct rsbac_auth_cap_range_t cap_range;
    rsbac_time_t ttl;
  };

struct rsys_auth_remove_f_cap_t
  {
    char * filename;
    enum rsbac_auth_cap_type_t cap_type;
    struct rsbac_auth_cap_range_t cap_range;
  };

struct rsys_auth_get_f_caplist_t
  {
    char * filename;
    enum rsbac_auth_cap_type_t cap_type;
    struct rsbac_auth_cap_range_t * caplist;
    rsbac_time_t * ttllist;
    u_int maxnum;
  };

struct rsys_acl_t
  {
    enum   rsbac_acl_syscall_type_t call;
    struct rsbac_acl_syscall_arg_t * arg;
  };

struct rsys_acl_n_t
  {
    enum   rsbac_acl_syscall_type_t call;
    struct rsbac_acl_syscall_n_arg_t * arg;
  };

struct rsys_acl_get_rights_t
  {
    struct rsbac_acl_syscall_arg_t * arg;
           rsbac_acl_rights_vector_t * rights_p;
           u_int effective;
  };

struct rsys_acl_get_rights_n_t
  {
    struct rsbac_acl_syscall_n_arg_t * arg;
           rsbac_acl_rights_vector_t * rights_p;
           u_int effective;
  };

struct rsys_acl_get_tlist_t
  {
    enum   rsbac_target_t target;
    union  rsbac_target_id_t * tid;
    struct rsbac_acl_entry_t * entry_array;
           rsbac_time_t * ttl_array;
           u_int maxnum;
  };

struct rsys_acl_get_tlist_n_t
  {
    enum   rsbac_target_t target;
           char * t_name;
    struct rsbac_acl_entry_t * entry_array;
           rsbac_time_t * ttl_array;
           u_int maxnum;
  };

struct rsys_acl_get_mask_t
  {
    enum   rsbac_target_t target;
    union  rsbac_target_id_t * tid;
           rsbac_acl_rights_vector_t * mask_p;
  };

struct rsys_acl_get_mask_n_t
  {
    enum   rsbac_target_t target;
           char * t_name;
           rsbac_acl_rights_vector_t * mask_p;
  };

struct rsys_acl_group_t
  {
    enum  rsbac_acl_group_syscall_type_t call;
    union rsbac_acl_group_syscall_arg_t * arg_p;
  };

struct rsys_reg_t
  {
    long handle;
    void * arg;
  };

struct rsys_jail_t
  {
    rsbac_version_t      version;
    char               * path;
    rsbac_jail_ip_t      ip;
    rsbac_jail_flags_t   flags;
  };

struct rsys_init_t
  {
    char * root_dev;
  };

union rsbac_syscall_arg_t
  {
    struct rsys_check_t check;
    struct rsys_get_attr_t get_attr;
    struct rsys_get_attr_n_t get_attr_n;
    struct rsys_set_attr_t set_attr;
    struct rsys_set_attr_n_t set_attr_n;
    struct rsys_remove_target_t remove_target;
    struct rsys_remove_target_n_t remove_target_n;
    struct rsys_net_list_all_netdev_t net_list_all_netdev;
    struct rsys_net_template_t net_template;
    struct rsys_net_list_all_template_t net_list_all_template;
    struct rsys_switch_t switch_module;
    struct rsys_adf_log_switch_t adf_log_switch;
    struct rsys_get_adf_log_t get_adf_log;
    struct rsys_log_t log;
    struct rsys_mac_set_curr_level_t mac_set_curr_level;
    struct rsys_mac_get_curr_level_t mac_get_curr_level;
    struct rsys_mac_get_max_level_t mac_get_max_level;
    struct rsys_mac_get_min_level_t mac_get_min_level;
    struct rsys_pm_t pm;
    struct rsys_pm_change_current_task_t pm_change_current_task;
    struct rsys_pm_create_file_t pm_create_file;
    struct rsys_ms_get_copied_t ms_get_copied;
    struct rsys_rc_copy_role_t rc_copy_role;
    struct rsys_rc_get_item_t rc_get_item;
    struct rsys_rc_set_item_t rc_set_item;
    struct rsys_rc_get_list_t rc_get_list;
    struct rsys_rc_change_role_t rc_change_role;
    struct rsys_rc_get_eff_rights_n_t rc_get_eff_rights_n;
    struct rsys_auth_add_p_cap_t auth_add_p_cap;
    struct rsys_auth_remove_p_cap_t auth_remove_p_cap;
    struct rsys_auth_add_f_cap_t auth_add_f_cap;
    struct rsys_auth_remove_f_cap_t auth_remove_f_cap;
    struct rsys_auth_get_f_caplist_t auth_get_f_caplist;
    struct rsys_acl_t acl;
    struct rsys_acl_n_t acl_n;
    struct rsys_acl_get_rights_t acl_get_rights;
    struct rsys_acl_get_rights_n_t acl_get_rights_n;
    struct rsys_acl_get_tlist_t acl_get_tlist;
    struct rsys_acl_get_tlist_n_t acl_get_tlist_n;
    struct rsys_acl_get_mask_t acl_get_mask;
    struct rsys_acl_get_mask_n_t acl_get_mask_n;
    struct rsys_acl_group_t acl_group;
    struct rsys_reg_t reg;
    struct rsys_jail_t jail;
    struct rsys_init_t init;
           int dummy;
  };

#ifndef __KERNEL__

int rsbac_stats(void);

int rsbac_check(int correct, int check_inode);

int rsbac_write(void);

int rsbac_get_attr(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit);

int rsbac_get_attr_n(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit);

int rsbac_set_attr(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value);


int rsbac_set_attr_n(
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value);

int rsbac_remove_target(
        enum rsbac_target_t target,
        union rsbac_target_id_t * tid);

int rsbac_remove_target_n(enum rsbac_target_t target,
                            char * t_name);

int rsbac_net_list_all_netdev(rsbac_netdev_id_t * id_p, u_long maxnum);

int rsbac_net_template(enum rsbac_net_temp_syscall_t call,
                           rsbac_net_temp_id_t id,
                           union rsbac_net_temp_syscall_data_t * data_p);

int rsbac_net_list_all_template(rsbac_net_temp_id_t * id_p, u_long maxnum);

int rsbac_switch(enum rsbac_switch_target_t target, int value);

/************** MAC ***************/

int rsbac_mac_set_curr_level(rsbac_security_level_t level,
                             rsbac_mac_category_vector_t * categories_p);

int rsbac_mac_get_curr_level(rsbac_security_level_t      * level_p,
                             rsbac_mac_category_vector_t * categories_p);

int rsbac_mac_get_max_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p);

int rsbac_mac_get_min_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p);

/************** PM ***************/

int rsbac_stats_pm(void);

int rsbac_pm(enum  rsbac_pm_function_type_t    function,
                 union rsbac_pm_function_param_t * param_p,
                       rsbac_pm_tkt_id_t           ticket);

int rsbac_pm_change_current_task(rsbac_pm_task_id_t task);

int rsbac_pm_create_file(const char * filename,
                             int mode,
                             rsbac_pm_object_class_id_t object_class);

/************** MS ***************/

int rsbac_ms_get_copied(int fd);

/************** RC ***************/

int rsbac_rc_copy_role (rsbac_rc_role_id_t from_role,
                            rsbac_rc_role_id_t to_role);

int rsbac_rc_get_item (enum  rsbac_rc_target_t       target,
                           union rsbac_rc_target_id_t  * tid_p,
                           union rsbac_rc_target_id_t  * subtid_p,
                           enum  rsbac_rc_item_t         item,
                           union rsbac_rc_item_value_t * value_p,
                           rsbac_time_t          * ttl_p);

/* Setting values */
int rsbac_rc_set_item (enum  rsbac_rc_target_t       target,
                           union rsbac_rc_target_id_t  * tid_p,
                           union rsbac_rc_target_id_t  * subtid_p,
                           enum  rsbac_rc_item_t         item,
                           union rsbac_rc_item_value_t * value_p,
                           rsbac_time_t            ttl);

int rsbac_rc_get_list(enum  rsbac_rc_target_t       target,
                          union rsbac_rc_target_id_t  * tid_p,
                          enum  rsbac_rc_item_t         item,
                          u_int maxnum,
                          __u32  * array_p,
                          rsbac_time_t * ttl_array_p);

int rsbac_rc_change_role (rsbac_rc_role_id_t role);

int rsbac_rc_get_eff_rights_n(enum  rsbac_target_t   target,
                                  char           * t_name,
                                  rsbac_rc_request_vector_t * request_vector_p,
                                  rsbac_time_t          * ttl_p);

/************** AUTH ***************/

/* Provide means for adding and removing of capabilities */
int rsbac_auth_add_p_cap(rsbac_pid_t pid,
                         enum rsbac_auth_cap_type_t cap_type,
                         struct rsbac_auth_cap_range_t cap_range,
                         rsbac_time_t ttl);

int rsbac_auth_remove_p_cap(rsbac_pid_t pid,
                            enum rsbac_auth_cap_type_t cap_type,
                            struct rsbac_auth_cap_range_t cap_range);

int rsbac_auth_add_f_cap(char * filename,
                         enum rsbac_auth_cap_type_t cap_type,
                         struct rsbac_auth_cap_range_t cap_range,
                         rsbac_time_t ttl);

int rsbac_auth_remove_f_cap(char * filename,
                            enum rsbac_auth_cap_type_t cap_type,
                            struct rsbac_auth_cap_range_t cap_range);

/* caplist must have space for maxnum cap_range entries - first and last each! */
int rsbac_auth_get_f_caplist(char * filename,
                             enum rsbac_auth_cap_type_t cap_type,
                             struct rsbac_auth_cap_range_t caplist[],
                             rsbac_time_t ttllist[],
                             u_int maxnum);

/**********************************/
/************** REG ***************/

int rsbac_reg(rsbac_reg_handle_t handle,
              void * arg);


/**********************************/
/************** ACL ***************/

int rsbac_acl(enum   rsbac_acl_syscall_type_t call,
                  struct rsbac_acl_syscall_arg_t * arg);

int rsbac_acl_n(enum   rsbac_acl_syscall_type_t call,
                    struct rsbac_acl_syscall_n_arg_t * arg);

int rsbac_acl_get_rights (struct rsbac_acl_syscall_arg_t   * arg,
                              rsbac_acl_rights_vector_t * rights_p,
                              u_int                     effective);


int rsbac_acl_get_rights_n(struct rsbac_acl_syscall_n_arg_t * arg,
                               rsbac_acl_rights_vector_t * rights_p,
                               u_int                     effective);

int rsbac_acl_get_tlist (enum   rsbac_target_t              target,
                             union  rsbac_target_id_t         * tid,
                             struct rsbac_acl_entry_t           entry_array[],
                                    rsbac_time_t                ttl_array[],
                                    u_int                       maxnum);

int rsbac_acl_get_tlist_n(enum   rsbac_target_t      target,
                              char                     * t_name,
                              struct rsbac_acl_entry_t   entry_array[],
                              rsbac_time_t               ttl_array[],
                              u_int               maxnum);

int rsbac_acl_get_mask (enum   rsbac_target_t              target,
                            union  rsbac_target_id_t         * tid,
                                   rsbac_acl_rights_vector_t * mask_p);

int rsbac_acl_get_mask_n(enum rsbac_target_t              target,
                             char                      * t_name,
                             rsbac_acl_rights_vector_t * mask_p);

/********  ACL groups *********/

int rsbac_acl_group(enum  rsbac_acl_group_syscall_type_t call,
                        union rsbac_acl_group_syscall_arg_t * arg_p);


/**********************************/
/************** JAIL **************/

int rsbac_jail(rsbac_version_t version,
               char * path,
               rsbac_jail_ip_t ip,
               rsbac_jail_flags_t flags);

/************************************************* */
/*             DEBUG/LOG functions                 */
/************************************************* */

int rsbac_adf_log_switch(enum rsbac_adf_request_t request,
                         enum rsbac_target_t      target,
                         u_int value);

int rsbac_get_adf_log(enum rsbac_adf_request_t   request,
                      enum rsbac_target_t        target,
                      u_int                    * value_p);

/*
 * Commands to rsbac_log:
 *
 * 	0 -- Close the log.  Currently a NOP.
 * 	1 -- Open the log. Currently a NOP.
 * 	2 -- Read from the log.
 * 	3 -- Read up to the last 4k of messages in the ring buffer.
 * 	4 -- Read and clear last 4k of messages in the ring buffer
 * 	5 -- Clear ring buffer.
 */
int rsbac_log(int type,
                  char * buf,
                  int len);

int rsbac_init(char * root_dev);

#endif /* ifndef __KERNEL__ */

#endif
