/*
 * Copyright (C) 2000 Jens Axboe <axboe@suse.de>
 *
 * May be copied or modified under the terms of the GNU General Public
 * License.  See linux/COPYING for more information.
 *
 * Packet writing layer for ATAPI and SCSI CD-R, CD-RW, DVD-R, and
 * DVD-RW devices (aka an exercise in block layer masturbation)
 *
 *
 * TODO: (circa order of when I will fix it)
 * - Only able to write on CD-RW media right now.
 * - check host application code on media and set it in write page
 * - Generic interface for UDF to submit large packets for variable length
 *   packet writing
 * - interface for UDF <-> packet to negotiate a new location when a write
 *   fails.
 * - handle OPC, especially for -RW media
 *
 * ------------------------------------------------------------------------
 *
 * Newer changes -- see ChangeLog
 *
 * 0.0.2d (26/10/2000)
 * - (scsi) use implicit segment recounting for all hba's
 * - fix speed setting, was consistenly off on most drives
 * - only print capacity when opening for write
 * - fix off-by-two error in getting/setting write+read speed (affected
 *   reporting as well as actual speed used)
 * - possible to enable write caching on drive
 * - do ioctl marshalling on sparc64 from Ben Collins <bcollins@debian.org>
 * - avoid unaligned access on flags, should have been unsigned long of course
 * - fixed missed wakeup in kpacketd
 * - b_dev error (two places)
 * - fix buffer head b_count bugs
 * - fix hole merge bug, where tail could be added twice
 * - fsync and invalidate buffers on close
 * - check hash table for buffers first before using our own
 * - add read-ahead
 * - fixed several list races
 * - fix proc reporting for more than one device
 * - change to O_CREAT for creating devices
 * - added media_change hook
 * - added free buffers config option
 * - pkt_lock_tray fails on failed open (and oopses), remove it. unlock
 *   is done explicitly in pkt_remove dev anyway.
 * - added proper elevator insertion (should probably be part of elevator.c)
 * - moved kernel thread info to private device, spawn one for each writer
 * - added separate buffer list for dirty packet buffers
 * - fixed nasty data corruption bug
 * - remember to account request even when we don't gather data for it
 * - add ioctl to force wakeup of kernel thread (for debug)
 * - fixed packet size setting bug on zero detected
 * - changed a lot of the proc reporting to be more readable to "humans"
 * - set full speed for read-only opens
 *
 * 0.0.2c (08/09/2000)
 * - inc usage count of buffer heads
 * - add internal buffer pool to avoid deadlock on oom
 * - gather data for as many buffers as we have, before initiating write. this
 *   allows the laser to stay on longer, giving better performance.
 * - fix always busy when tray can't be locked
 * - remove request duplication nastiness, inject directly into the target
 * - adapted to devfs and elevator changes
 * - added proc interface
 *
 * 0.0.2b (21/06/2000)
 * - fix io_request_lock typos (missing '&')
 * - grab pkt_sem before invoking pkt_handle_queue
 * - SCSI uses queuedata too, mirror that in pd->queuedata (hack)
 * - remove SCSI sr debug messages
 * - really activate empty block querying (requires cvs UDF, CDRW branch)
 * - make sure sync_buffers doesn't consider us, or we can deadlock
 * - make sure people don't swap on us (for now ;)
 *
 * 0.0.2a (19/06/2000)
 * - add kpacketd kernel thread to handle actual data gathering
 * - pd->pkt_dev is now real device, not just minor
 * - add support for super_operations block_empty fn, to query fs for
 *   unused blocks that don't need reading
 * - "cache" blocks that are contained in the UDF file/dir packet
 * - rewrite pkt_gather_data to a one-step solution
 * - add private pktcdvd elevator
 * - shutdown write access to device upon write failure
 * - fix off-by-one bug in capacity
 * - setup sourceforge project (packet-cd.sourceforge.net)
 * - add more blk ioctls to pkt_ioctl
 * - set inactive request queue head
 * - change panic calls to BUG, better with kdb
 * - have pkt_gather_data check correct block size and kill rq if wrong
 * - rework locking
 * - introduce per-pd queues, simplifies pkt_request
 * - store pd in queuedata
 *
 *************************************************************************/

#define VERSION_CODE	"v0.0.2p 03/03/2002 Jens Axboe (axboe@suse.de)"

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/locks.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/file.h>
#include <linux/blk.h>
#include <linux/blkpg.h>
#include <linux/cdrom.h>
#include <linux/ide.h>
#include <linux/smp_lock.h>
#include <linux/pktcdvd.h>
#include <linux/kernel_stat.h>
#include <linux/sysrq.h>

#include <asm/unaligned.h>
#include <asm/uaccess.h>

/*
 * remove for next version -- for now, disable the mention option in the
 * SCSI section
 */
#if defined(CONFIG_SCSI_DEBUG_QUEUES)
#error "Don't compile with 'Enable extra checks in new queueing code' enabled"
#endif

#define SCSI_IOCTL_SEND_COMMAND	1

/*
 * 32 buffers of 2048 bytes
 */
#define PACKET_MAX_SIZE		32

#define NEXT_BH(bh, nbh)	\
	 (((bh)->b_rsector + ((bh)->b_size >> 9)) == (nbh)->b_rsector)

#define BH_IN_ORDER(b1, b2)	\
	((b1)->b_rsector < (b2)->b_rsector)

#define CONTIG_BH(b1, b2)	\
	((b1)->b_data + (b1)->b_size == (b2)->b_data)

#define ZONE(sector, pd)	\
	(((sector) + ((pd)->offset)) - (((sector) + ((pd)->offset)) & (((pd)->settings.size - 1))))

static int *pkt_sizes;
static int *pkt_blksize;
static int *pkt_readahead;
static struct pktcdvd_device *pkt_devs;
static struct proc_dir_entry *pkt_proc;
static DECLARE_WAIT_QUEUE_HEAD(pd_bh_wait);

/*
 * a bit of a kludge, but we want to be able to pass both real and packet
 * dev and get the right one.
 */
static inline struct pktcdvd_device *pkt_find_dev(kdev_t dev)
{
	int i;

	for (i = 0; i < MAX_WRITERS; i++)
		if (pkt_devs[i].dev == dev || pkt_devs[i].pkt_dev == dev)
			return &pkt_devs[i];

	return NULL;
}

/*
 * The following functions are the plugins to the ll_rw_blk
 * layer and decides whether a given request / buffer head can be
 * merged. We differ in a couple of ways from "normal" block
 * devices:
 *
 * - don't merge when the buffer / request crosses a packet block
 *   boundary
 * - merge buffer head even though it can't be added directly to the
 *   front or back of the list. this gives us better performance, since
 *   what would otherwise require multiple requests can now be handled
 *   in one (hole merging)
 * - at this point its just writes, reads have already been remapped
 *
 * The device original merge_ functions are stored in the packet device
 * queue (pd->q)
 *
 */
static inline int pkt_do_merge(request_queue_t *q, struct request *rq,
			       struct buffer_head *bh, int max_segs,
			       merge_request_fn *merge_fn,
			       struct pktcdvd_device *pd)
{
	void *ptr = q->queuedata;
	int ret;

	if (rq->cmd != WRITE)
		BUG();

	if (ZONE(rq->sector, pd) != ZONE(bh->b_rsector, pd))
		return ELEVATOR_NO_MERGE;

	/*
	 * NOTE: this is done under the io_request_lock/queue_lock, hence
	 * it is safe
	 */
	q->queuedata = pd->cdrw.queuedata;
	ret = merge_fn(q, rq, bh, max_segs);
	q->queuedata = ptr;
	return ret;
}

static int pkt_front_merge_fn(request_queue_t *q, struct request *rq,
			      struct buffer_head *bh, int max_segs)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(bh->b_rdev)];

	return pkt_do_merge(q, rq, bh, max_segs, pd->cdrw.front_merge_fn, pd);
}

static int pkt_back_merge_fn(request_queue_t *q, struct request *rq,
			     struct buffer_head *bh, int max_segs)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(bh->b_rdev)];

	return pkt_do_merge(q, rq, bh, max_segs, pd->cdrw.back_merge_fn, pd);
}

/*
 * rules similar to above
 */
static int pkt_merge_requests_fn(request_queue_t *q, struct request *rq,
				 struct request *nxt, int max_segs)
{
	struct pktcdvd_device *pd = pkt_find_dev(rq->rq_dev);
	struct packet_cdrw *cdrw = &pd->cdrw;
	void *ptr = q->queuedata;
	int ret;

	if (ZONE(rq->sector, pd) != ZONE(nxt->sector + nxt->nr_sectors - 1, pd))
		return 0;

	q->queuedata = cdrw->queuedata;
	ret = cdrw->merge_requests_fn(q, rq, nxt, max_segs);
	q->queuedata = ptr;
	return ret;
}

static int pkt_grow_bhlist(struct pktcdvd_device *pd, int count)
{
	struct packet_cdrw *cdrw = &pd->cdrw;
	struct buffer_head *bh;
	int i = 0;

	VPRINTK("grow_bhlist: count=%d\n", count);

	while (i < count) {
		bh = kmem_cache_alloc(bh_cachep, SLAB_KERNEL);
		if (!bh)
			break;

		bh->b_data = kmalloc(CD_FRAMESIZE, GFP_KERNEL);
		if (!bh->b_data) {
			kmem_cache_free(bh_cachep, bh);
			break;
		}
		bh->b_page = virt_to_page(bh->b_data);

		spin_lock_irq(&pd->lock);
		bh->b_pprev = &cdrw->bhlist;
		bh->b_next = cdrw->bhlist;
		cdrw->bhlist = bh;
		spin_unlock_irq(&pd->lock);

		bh->b_size = CD_FRAMESIZE;
		bh->b_list = PKT_BUF_LIST;
		atomic_inc(&cdrw->free_bh);
		i++;
	}

	return i;
}

static int pkt_shrink_bhlist(struct pktcdvd_device *pd, int count)
{
	struct packet_cdrw *cdrw = &pd->cdrw;
	struct buffer_head *bh;
	int i = 0;

	VPRINTK("shrink_bhlist: count=%d\n", count);

	while ((i < count) && cdrw->bhlist) {
		spin_lock_irq(&pd->lock);
		bh = cdrw->bhlist;
		cdrw->bhlist = bh->b_next;
		spin_unlock_irq(&pd->lock);
		if (bh->b_list != PKT_BUF_LIST)
			BUG();
		kfree(bh->b_data);
		kmem_cache_free(bh_cachep, bh);
		atomic_dec(&cdrw->free_bh);
		i++;
	}

	return i;
}

/*
 * These functions manage a simple pool of buffer_heads.
 */
static struct buffer_head *pkt_get_stacked_bh(struct pktcdvd_device *pd)
{
	unsigned long flags;
	struct buffer_head *bh;

	spin_lock_irqsave(&pd->lock, flags);
	bh = pd->stacked_bhlist;
	if (bh) {
		pd->stacked_bhlist = bh->b_next;
		bh->b_next = NULL;
		pd->stacked_bhcnt--;
		BUG_ON(pd->stacked_bhcnt < 0);
	}
	spin_unlock_irqrestore(&pd->lock, flags);

	return bh;
}

static void pkt_put_stacked_bh(struct pktcdvd_device *pd, struct buffer_head *bh)
{
	unsigned long flags;

	spin_lock_irqsave(&pd->lock, flags);
	if (pd->stacked_bhcnt < STACKED_BH_POOL_SIZE) {
		bh->b_next = pd->stacked_bhlist;
		pd->stacked_bhlist = bh;
		pd->stacked_bhcnt++;
		bh = NULL;
	}
	spin_unlock_irqrestore(&pd->lock, flags);
	if (bh) {
		kmem_cache_free(bh_cachep, bh);
	}
}

static void pkt_shrink_stacked_bhlist(struct pktcdvd_device *pd)
{
	struct buffer_head *bh;

	while ((bh = pkt_get_stacked_bh(pd)) != NULL) {
		kmem_cache_free(bh_cachep, bh);
	}
}

static int pkt_grow_stacked_bhlist(struct pktcdvd_device *pd)
{
	struct buffer_head *bh;
	int i;

	for (i = 0; i < STACKED_BH_POOL_SIZE; i++) {
		bh = kmem_cache_alloc(bh_cachep, GFP_KERNEL);
		if (!bh) {
			pkt_shrink_stacked_bhlist(pd);
			return 0;
		}
		pkt_put_stacked_bh(pd, bh);
	}
	return 1;
}


static request_queue_t *pkt_get_queue(kdev_t dev)
{
	struct pktcdvd_device *pd = pkt_find_dev(dev);
	if (!pd)
		return NULL;
	return &pd->cdrw.r_queue;
}

static void pkt_put_buffer(struct buffer_head *bh)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(bh->b_dev)];
	unsigned long flags;

	if (bh->b_list != PKT_BUF_LIST)
		return;

	bh->b_state = 0;
	bh->b_reqnext = NULL;
	bh->b_end_io = NULL;

	spin_lock_irqsave(&pd->lock, flags);
	bh->b_next = pd->cdrw.bhlist;
	pd->cdrw.bhlist = bh;
	spin_unlock_irqrestore(&pd->lock, flags);
	atomic_inc(&pd->cdrw.free_bh);
}

static inline void __pkt_inject_request(request_queue_t *q, struct request *rq)
{
	struct list_head *head =  &q->queue_head;

	VPRINTK("__pkt_inject_request: list_empty == %d, size=%d, cmd=%d\n",
		list_empty(&q->queue_head), rq->bh->b_size >> 9, rq->cmd);

	if (list_empty(&q->queue_head))
		q->plug_device_fn(q, rq->rq_dev);

	list_add_tail(&rq->queue, head);
}

static void pkt_inject_request(request_queue_t *q, struct request *rq)
{
	spin_lock_irq(&io_request_lock);
	__pkt_inject_request(q, rq);
	spin_unlock_irq(&io_request_lock);
}

static inline void __pkt_end_request(struct pktcdvd_device *pd)
{
	pd->rq = NULL;
	clear_bit(PACKET_RQ, &pd->flags);
	clear_bit(PACKET_BUSY, &pd->flags);
}

/*
 * io_request_lock must be held and interrupts disabled
 */
static void pkt_end_request(struct pktcdvd_device *pd)
{
	unsigned long flags;

	spin_lock_irqsave(&pd->lock, flags);
	__pkt_end_request(pd);
	spin_unlock_irqrestore(&pd->lock, flags);
}


static inline void __pkt_kill_request(struct request *rq, int uptodate, char *name)
{
	struct buffer_head *bh = rq->bh, *nbh;

	while (bh) {
		nbh = bh->b_reqnext;
		bh->b_reqnext = NULL;

		if (bh->b_end_io) {
			bh->b_end_io(bh, uptodate);
		} else {
			mark_buffer_clean(bh);
			mark_buffer_uptodate(bh, uptodate);
			unlock_buffer(bh);
		}

		bh = nbh;
	}

	end_that_request_last(rq);
}


void pkt_kill_request(struct pktcdvd_device *pd, struct request *rq, int ok)
{
	printk("pktcdvd: killing request\n");
	spin_lock_irq(&io_request_lock);
	__pkt_kill_request(rq, ok, pd->name);
	spin_unlock_irq(&io_request_lock);
	pkt_end_request(pd);
}

static void pkt_end_io_read(struct buffer_head *bh, int uptodate)
{
	if (!uptodate) {
		/* Obviously not correct, but it avoids locking up the kernel */
		printk("Ignoring read error on sector:%ld\n", bh->b_rsector);
		uptodate = 1;
	}

	mark_buffer_uptodate(bh, uptodate);
	unlock_buffer(bh);
}

/*
 * if the buffer is already in the buffer cache, grab it if we can lock
 * it down
 */
static struct buffer_head *pkt_get_hash(kdev_t dev, unsigned long block, int size)
{
	struct buffer_head *bh = NULL;

	bh = get_hash_table(dev, block, size);
	if (bh) {
		if (!test_and_set_bit(BH_Lock, &bh->b_state)) {
			brelse(bh);
			if (atomic_set_buffer_clean(bh))
				refile_buffer(bh);
			SetPageReferenced(bh->b_page);
		} else {
			brelse(bh);
			bh = NULL;
		}
	}

	return bh;
}

static inline struct buffer_head *__pkt_get_buffer(struct pktcdvd_device *pd,
						   unsigned long sector)
{
	struct buffer_head *bh;

	if (!atomic_read(&pd->cdrw.free_bh))
		BUG();

	atomic_dec(&pd->cdrw.free_bh);

	spin_lock_irq(&pd->lock);
	bh = pd->cdrw.bhlist;
	pd->cdrw.bhlist = bh->b_next;
	bh->b_next = NULL;
	spin_unlock_irq(&pd->lock);

	bh->b_next_free = NULL;
	bh->b_prev_free = NULL;
	bh->b_this_page = NULL;
	bh->b_pprev = NULL;
	bh->b_reqnext = NULL;

	init_waitqueue_head(&bh->b_wait);
	atomic_set(&bh->b_count, 1);
	bh->b_list = PKT_BUF_LIST;
	bh->b_state = (1 << BH_Mapped) | (1 << BH_Lock) | (1 << BH_Req);
	bh->b_dev = pd->pkt_dev;

	return bh;
}

static void pkt_end_io_write(struct buffer_head *, int);

static struct buffer_head *pkt_get_buffer(struct pktcdvd_device *pd,
					  unsigned long sector, int size)
{
	unsigned long block = sector / (size >> 9);
	struct buffer_head *bh;

	VPRINTK("get_buffer: sector=%ld, size=%d\n", sector, size);

	bh = pkt_get_hash(pd->pkt_dev, block, size);
	if (bh)
		pd->stats.bh_cache_hits += (size >> 9);
	else
		bh = __pkt_get_buffer(pd, sector);

	blk_started_io(bh->b_size >> 9);
	bh->b_blocknr = block;
	bh->b_end_io = pkt_end_io_write;
	bh->b_rsector = sector;
	bh->b_rdev = pd->dev;
	return bh;
}

/*
 * this rq is done -- io_request_lock must be held and interrupts disabled
 */
static void pkt_rq_end_io(struct pktcdvd_device *pd)
{
	unsigned long flags;

	VPRINTK("pkt_rq_end_io: rq=%p, cmd=%d, q=%p\n", pd->rq, pd->rq->cmd, pd->rq->q);

	spin_lock_irqsave(&pd->lock, flags);

	/*
	 * debug checks
	 */
	if (!test_bit(PACKET_RQ, &pd->flags))
		printk("pktcdvd: rq_end_io: RQ not set\n");
	if (!test_bit(PACKET_BUSY, &pd->flags))
		printk("pktcdvd: rq_end_io: BUSY not set\n");

	__pkt_end_request(pd);
	wake_up(&pd->wqueue);
	spin_unlock_irqrestore(&pd->lock, flags);
}

static inline void pkt_mark_readonly(struct pktcdvd_device *pd, int on)
{
	if (on)
		set_bit(PACKET_READONLY, &pd->flags);
	else
		clear_bit(PACKET_READONLY, &pd->flags);
}

static inline void __pkt_end_io_write(struct pktcdvd_device *pd,
				      struct buffer_head *bh, int uptodate)
{
	VPRINTK("end_io_write: bh=%ld, uptodate=%d\n", bh->b_blocknr, uptodate);

	/*
	 * general Linux bug, noone should clear the BH_Uptodate flag for
	 * a failed write...
	 */
	if (uptodate)
		mark_buffer_uptodate(bh, uptodate);
	else {
		printk("pktcdvd: %s: WRITE error sector %lu\n", pd->name, bh->b_rsector);
#if 0
		set_bit(PACKET_RECOVERY, &pd->flags);
		wake_up(&pd->wqueue);
#endif
	}

	pd->stats.bh_e++;

	atomic_dec(&pd->wrqcnt);
	if (atomic_read(&pd->wrqcnt) == 0) {
		pkt_rq_end_io(pd);
	}

	unlock_buffer(bh);
}

/*
 * we use this as our default b_end_io handler, since we need to take
 * the entire request off the list if just one of the clusters fail.
 * later on this should also talk to UDF about relocating blocks -- for
 * now we just drop the rq entirely. when doing the relocating we must also
 * lock the bh down to ensure that we can easily reconstruct the write should
 * it fail.
 */
static void pkt_end_io_write(struct buffer_head *bh, int uptodate)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(bh->b_rdev)];

	__pkt_end_io_write(pd, bh, uptodate);
	pkt_put_buffer(bh);
}

static void pkt_end_io_write_stacked(struct buffer_head *bh, int uptodate)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(bh->b_rdev)];
	struct buffer_head *rbh = bh->b_private;

	__pkt_end_io_write(pd, bh, uptodate);
	rbh->b_end_io(rbh, uptodate);
	pkt_put_stacked_bh(pd, bh);
	wake_up(&pd_bh_wait);
}

static int pkt_init_rq(struct pktcdvd_device *pd, struct request *rq)
{
	struct buffer_head *bh;
	unsigned int cnt, nr_segments;

	cnt = 0;
	nr_segments = 1;
	bh = rq->bh;
	while (bh) {
		struct buffer_head *nbh = bh->b_reqnext;

		bh->b_rdev = pd->pkt_dev;

		/*
		 * the buffer better be mapped and locked!
		 */
		if (!buffer_locked(bh) || !buffer_mapped(bh)) {
			printk("%lu, state %lx\n", bh->b_rsector, bh->b_state);
			BUG();
		}

		if (nbh) {
			if (!CONTIG_BH(bh, nbh))
				nr_segments++;

			/*
			 * if this happens, do report
			 */
			if ((bh->b_rsector + (bh->b_size >> 9))!=nbh->b_rsector) {
				printk("%lu (%p)-> %lu (%p) (%lu in all)\n",
				bh->b_rsector, bh, nbh->b_rsector, nbh,
				rq->nr_sectors);
				return 1;
			}
		}

		cnt += bh->b_size >> 9;
		bh = nbh;
	}

	rq->nr_segments = rq->nr_hw_segments = nr_segments;

	if (cnt != rq->nr_sectors) {
		printk("botched request %u (%lu)\n", cnt, rq->nr_sectors);
		return 1;
	}

	return 0;
}

/*
 * really crude stats for now...
 */
static void pkt_account_rq(struct pktcdvd_device *pd, int read, int written,
			   int bs)
{
	pd->stats.bh_s += (written / bs);
	pd->stats.secs_w += written;
	pd->stats.secs_r += read;
}

/*
 * does request span two packets? 0 == yes, 1 == no
 */
static int pkt_one_zone(struct pktcdvd_device *pd, struct request *rq)
{
	if (!pd->settings.size)
		return 0;

	if (!(rq->cmd & WRITE))
		return 1;

	return ZONE(rq->sector, pd) == ZONE(rq->sector + rq->nr_sectors -1, pd);
}

#if defined(CONFIG_CDROM_PKTCDVD_BEMPTY)
static void pkt_init_buffer(struct buffer_head *bh)
{
	set_bit(BH_Uptodate, &bh->b_state);
	set_bit(BH_Dirty, &bh->b_state);
	memset(bh->b_data, 0, bh->b_size);
}

static int pkt_sb_empty(struct pktcdvd_device *pd, struct buffer_head *bh)
{
	struct super_block *sb;
	struct super_operations *sop;
	unsigned long packet;
	int ret;

	ret = 0;
	if ((sb = get_super(pd->pkt_dev)) == NULL)
		goto out;
	if ((sop = sb->s_op) == NULL)
		goto out;
	if (sop->block_empty == NULL)
		goto out;

	packet = 0;
	if (sop->block_empty(sb, bh->b_blocknr, &packet))  {
		pkt_init_buffer(pd, bh);
		ret = 1;
	}

out:
	return ret;
}

#else /* defined(CONFIG_CDROM_PKTCDVD_BEMPTY) */

static int pkt_sb_empty(struct pktcdvd_device *pd, struct buffer_head *bh)
{
	return 0;
}

#endif /* defined(CONFIG_CDROM_PKTCDVD_BEMPTY) */

static int pkt_flush_cache(struct pktcdvd_device *pd);

static void pkt_flush_writes(struct pktcdvd_device *pd)
{
	if (pd->unflushed_writes) {
		pd->unflushed_writes = 0;
		pkt_flush_cache(pd);
	}
}

/*
 * basically just does a ll_rw_block for the bhs given to use, but we
 * don't return until we have them.
 */
static void pkt_read_bh(struct pktcdvd_device *pd, struct buffer_head *bh)
{
	/*
	 * UDF says it's empty, woohoo
	 */
	if (pkt_sb_empty(pd, bh))
		return;

	down(&pd->cache_sync_mutex);
	pkt_flush_writes(pd);
	generic_make_request(READ, bh);
	up(&pd->cache_sync_mutex);
}

static int pkt_index_bhs(struct buffer_head **bhs, int frames)
{
	struct buffer_head *bh;
	int index;
	int error = 0;

	/*
	 * now finish pending reads and connect the chain of buffers
	 */
	index = 0;
	while (index < frames) {
		bh = bhs[index];

		/*
		 * pin down private buffers (ie, force I/O to complete)
		 */
		if (bh->b_end_io == pkt_end_io_read) {
			lock_buffer(bh);
			bh->b_end_io = pkt_end_io_write;
			if (!buffer_uptodate(bh)) {
				printk("pktcdvd: read failure (%s, sec %lu)\n",
				       kdevname(bh->b_rdev), bh->b_rsector);
				error = 1;
			}
		}

		if (!buffer_locked(bh))
			BUG();

		/*
		 * attach previous
		 */
		if (index) {
			struct buffer_head *pbh = bhs[index - 1];

			if ((pbh->b_rsector + (pbh->b_size >> 9)) != bh->b_rsector) {
				printk("%lu -> %lu\n", pbh->b_rsector, bh->b_rsector);
				index = 0;
				break;
			}
			pbh->b_reqnext = bh;
		}
		index++;
	}

	if (error)
		return 0;

	if (index) {
		index--;
		bhs[index]->b_reqnext = NULL;
	}

	return index;
}

/*
 * fill in the holes of a request
 *
 * Returns: 0, keep 'em coming -- 1, stop queueing
 */
static int pkt_gather_data(struct pktcdvd_device *pd, struct request *rq)
{
	unsigned long start_s, end_s, sector;
	struct buffer_head *bh;
	unsigned int sectors, index;
	struct buffer_head *bhs[PACKET_MAX_SIZE];
	int frames = pd->settings.size >> 2;

	memset(bhs, 0, sizeof(bhs));

	/*
	 * all calculations are done with 512 byte sectors
	 */
	sectors = pd->settings.size - rq->nr_sectors;
	start_s = rq->sector - (rq->sector & (pd->settings.size - 1));
	end_s = start_s + pd->settings.size;

	VPRINTK("pkt_gather_data: cmd=%d\n", rq->cmd);
	VPRINTK("need %d sectors for %s\n", sectors, kdevname(pd->dev));
	VPRINTK("from %lu to %lu ", start_s, end_s);
	VPRINTK("(%lu - %lu)\n", rq->bh->b_rsector, rq->bhtail->b_rsector +
				 rq->current_nr_sectors);

	/*
	 * first fill-out map of the buffers we have
	 */
	bh = rq->bh;
	while (bh) {
		index = (bh->b_rsector & (pd->settings.size - 1)) / (bh->b_size >> 9);

		bhs[index] = bh;
		bh = bh->b_reqnext;

		/*
		 * make sure to detach from list!
		 */
		bhs[index]->b_reqnext = NULL;
	}

	/*
	 * now get buffers for missing blocks, and schedule reads for them
	 */
	for (index = 0, sector = start_s; sector < end_s; index++) {
		if (bhs[index]) {
			bh = bhs[index];
			goto next;
		}

		bh = pkt_get_buffer(pd, sector, CD_FRAMESIZE);

		bhs[index] = bh;
		rq->nr_sectors += bh->b_size >> 9;
		rq->nr_segments++;

		if (!buffer_uptodate(bh)) {
			bh->b_end_io = pkt_end_io_read;
			pkt_read_bh(pd, bh);
		}

	next:
		sector += bh->b_size >> 9;
	}

	index = pkt_index_bhs(bhs, frames);
#if 0
	if (!index)
		goto kill_it;
#endif

	rq->bh = bhs[0];
	rq->bhtail = bhs[index];
	rq->buffer = rq->bh->b_data;
	rq->current_nr_sectors = rq->bh->b_size >> 9;
	rq->hard_nr_sectors = rq->nr_sectors;
	rq->sector = rq->hard_sector = start_s;

	VPRINTK("unlocked last %lu\n", rq->bhtail->b_rsector);
	if (pkt_init_rq(pd, rq)) {
		for (index = 0; index < frames; index++) {
			bh = bhs[index];
			printk("[%d] %lu %d (%p -> %p)\n", index, bh->b_rsector,
					bh->b_size, bh, bh->b_reqnext);
		}
		goto kill_it;
	}

	pkt_account_rq(pd, sectors, rq->nr_sectors, rq->current_nr_sectors);

	/*
	 * sanity check
	 */
	if (rq->nr_sectors != pd->settings.size) {
		printk("pktcdvd: request mismatch %lu (should be %u)\n",
					rq->nr_sectors, pd->settings.size);
		BUG();
	}

	return 0;

	/*
	 * for now, just kill entire request and hope for the best...
	 */
kill_it:
	for (index = 0; index < frames; index++) {
		bh = bhs[index];
		buffer_IO_error(bh);
		if (bh->b_list == PKT_BUF_LIST)
			pkt_put_buffer(bh);
	}
	end_that_request_last(pd->rq);
	return 1;
}

/*
 * Returns: 1, keep 'em coming -- 0, wait for wakeup
 */
static int pkt_do_request(struct pktcdvd_device *pd, struct request *rq)
{
	VPRINTK("do_request: bh=%ld, nr_sectors=%ld, size=%d, cmd=%d\n", rq->bh->b_blocknr, rq->nr_sectors, pd->settings.size, rq->cmd);

	/*
	 * perfect match. the merge_* functions have already made sure that
	 * a request doesn't cross a packet boundary, so if the sector
	 * count matches it's good.
	 */
	if (rq->nr_sectors == pd->settings.size) {
		if (pkt_init_rq(pd, rq)) {
			pkt_kill_request(pd, rq, 0);
			return 1;
		}

		pkt_account_rq(pd, 0, rq->nr_sectors, rq->current_nr_sectors);
		return 0;
	}

	/*
	 * paranoia...
	 */
	if (rq->nr_sectors > pd->settings.size) {
		printk("pktcdvd: request too big! BUG! %lu\n", rq->nr_sectors);
		BUG();
	}

	return pkt_gather_data(pd, rq);
}

/*
 * recover a failed write, query for relocation if possible
 */
static int pkt_start_recovery(struct pktcdvd_device *pd, struct request *rq)
{
#if 0
	struct super_block *sb = get_super(pd->pkt_dev);
	struct buffer_head *bhs[PACKET_MAX_SIZE], *bh, *obh;
	unsigned long old_block, new_block, sector;
	int i, sectors;

	if (!sb || !sb->s_op || !sb->s_op->relocate_blocks)
		goto fail;

	old_block = (rq->sector & ~(pd->settings.size - 1)) / (rq->bh->b_size >> 9);
	if (sb->s_op->relocate_blocks(sb, old_block, &new_block))
		goto fail;

	memset(bhs, 0, sizeof(bhs));
	bh = rq->bh;
	while (bh) {
		i = (bh->b_rsector & (pd->settings.size - 1)) / (bh->b_size >> 9);

		bhs[i] = bh;
		bh = bh->b_reqnext;
		bhs[i]->b_reqnext = NULL;
	}

	sectors = 0;
	sector = new_block * (rq->bh->b_size >> 9);
	for (i = 0; i < (pd->settings.size >> 2); i++) {
		bh = bhs[i];

		/*
		 * three cases -->
		 *	1) bh is not there at all
		 *	2) bh is there and not ours, get a new one and
		 *	   invalidate this block for the future
		 *	3) bh is there and ours, just change the sector
		 */
		if (!bh) {
			obh = pkt_get_hash(pd->pkt_dev, new_block,CD_FRAMESIZE);
			bh = __pkt_get_buffer(pd, sector);
			if (obh) {
				if (buffer_uptodate(obh)) {
					memcpy(bh->b_data, obh->b_data, obh->b_size);
					set_bit(BH_Uptodate, &bh->b_state);
				}
				unlock_buffer(obh);
				bforget(obh);
			}
			bhs[i] = bh;
		} else if (bh->b_list != PKT_BUF_LIST) {
			bhs[i] = pkt_get_buffer(pd, sector, CD_FRAMESIZE);
			memcpy(bhs[i]->b_data, bh->b_data, CD_FRAMESIZE);
			unlock_buffer(bh);
			bforget(bh);
			bh = bhs[i];
			set_bit(BH_Uptodate, &bh->b_state);
		} else {
			bh->b_rsector = sector;
			bh->b_blocknr = new_block;
		}

		sector += (bh->b_size >> 9);
		new_block++;
		sectors +=  (bh->b_size >> 9);
	}

	i = pkt_index_bhs(bhs);
	if (!i)
		goto fail;

	rq->bh = bhs[0];
	rq->bhtail = bhs[i];
	rq->buffer = rq->bh->b_data;
	rq->current_nr_sectors = rq->bh->b_size >> 9;
	rq->hard_nr_sectors = rq->nr_sectors = sectors;
	rq->sector = rq->hard_sector = rq->bh->b_rsector;
	rq->errors = 0;
	clear_bit(PACKET_RECOVERY, &pd->flags);
	clear_bit(PACKET_BUSY, &pd->flags);
	return 0;

fail:
#endif
	printk("pktcdvd: rq recovery not possible\n");
	pkt_kill_request(pd, rq, 0);
	clear_bit(PACKET_RECOVERY, &pd->flags);
	return 1;
}

/*
 * handle the requests that got queued for this writer
 *
 * returns 0 for busy (already doing something), or 1 for queue new one
 *
 */
static int pkt_handle_queue(struct pktcdvd_device *pd, request_queue_t *q)
{
	struct request *rq;
	int ret;

	VPRINTK("handle_queue\n");

	/*
	 * nothing for us to do
	 */
	if (!test_bit(PACKET_RQ, &pd->flags))
		return 1;

	spin_lock_irq(&pd->lock);
	rq = pd->rq;
	spin_unlock_irq(&pd->lock);

	if (test_bit(PACKET_RECOVERY, &pd->flags))
		if (pkt_start_recovery(pd, rq))
			return 1;

	/*
	 * already being processed
	 */
	if (test_and_set_bit(PACKET_BUSY, &pd->flags))
		return 0;

	/*
	 * nothing to do
	 */
	ret = 1;
	if (rq == NULL) {
		printk("handle_queue: pd BUSY+RQ, but no rq\n");
		clear_bit(PACKET_RQ, &pd->flags);
		goto out;
	}

	/*
	 * reads are shipped directly to cd-rom, so they should not
	 * pop up here
	 */
	if (rq->cmd == READ)
		BUG();

	if ((rq->current_nr_sectors << 9) != CD_FRAMESIZE) {
		pkt_kill_request(pd, rq, 0);
		goto out;
	}

	if (!pkt_do_request(pd, rq)) {
		atomic_add(pd->settings.size >> 2, &pd->wrqcnt);
		down(&pd->cache_sync_mutex);
		pkt_inject_request(q, rq);
		pd->unflushed_writes = 1;
		up(&pd->cache_sync_mutex);
		return 0;
	}

out:
	clear_bit(PACKET_BUSY, &pd->flags);
	return ret;
}

/*
 * kpacketd is woken up, when writes have been queued for one of our
 * registered devices
 */
static int kcdrwd(void *foobar)
{
	struct pktcdvd_device *pd = foobar;
	request_queue_t *q, *my_queue;

	/*
	 * exit_files, mm (move to lazy-tlb, so context switches are come
	 * extremely cheap) etc
	 */
	daemonize();

	current->policy = SCHED_OTHER;
	set_user_nice(current, -20);
	sprintf(current->comm, pd->name);

	spin_lock_irq(&current->sigmask_lock);
	siginitsetinv(&current->blocked, sigmask(SIGKILL));
	flush_signals(current);
	spin_unlock_irq(&current->sigmask_lock);

	q = blk_get_queue(pd->dev);
	my_queue = blk_get_queue(pd->pkt_dev);

	for (;;) {
		DECLARE_WAITQUEUE(wait, current);

		add_wait_queue(&pd->wqueue, &wait);

		/*
		 * if PACKET_BUSY is cleared, we can queue
		 * another request. otherwise we need to unplug the
		 * cd-rom queue and wait for buffers to be flushed
		 * (which will then wake us up again when done).
		 */
		do {
			pkt_handle_queue(pd, q);

			set_current_state(TASK_INTERRUPTIBLE);

			if (test_bit(PACKET_BUSY, &pd->flags))
				break;

			spin_lock_irq(&io_request_lock);
			if (list_empty(&my_queue->queue_head)) {
				spin_unlock_irq(&io_request_lock);
				break;
			}
			set_current_state(TASK_RUNNING);

			my_queue->request_fn(my_queue);
			spin_unlock_irq(&io_request_lock);
		} while (1);

		generic_unplug_device(q);

		schedule();
		remove_wait_queue(&pd->wqueue, &wait);

		if (signal_pending(current)) {
			flush_signals(current);
		}
		if (pd->cdrw.time_to_die)
			break;
	}

	complete_and_exit(&pd->cdrw.thr_compl, 0);
	return 0;
}

static void pkt_attempt_remerge(struct pktcdvd_device *pd, request_queue_t *q,
				struct request *rq)
{
	struct request *nxt;

	while (!list_empty(&q->queue_head)) {
		if (rq->nr_sectors == pd->settings.size)
			break;

		nxt = blkdev_entry_next_request(&q->queue_head);

		if (ZONE(rq->sector, pd) != ZONE(nxt->sector, pd))
			break;
		else if (rq->sector + rq->nr_sectors > nxt->sector)
			break;

		rq->nr_sectors = rq->hard_nr_sectors += nxt->nr_sectors;
		rq->bhtail->b_reqnext = nxt->bh;
		rq->bhtail = nxt->bhtail;
		list_del(&nxt->queue);
		blkdev_release_request(nxt);
	}
}

/*
 * our request function.
 *
 * - reads are just tossed directly to the device, we don't care.
 * - writes, regardless of size, are added as the current pd rq and
 *   kcdrwd is woken up to handle it. kcdrwd will also make sure to
 *   reinvoke this request handler, once the given request has been
 *   processed.
 *
 * Locks: io_request_lock held
 *
 * Notes: all writers have their own queue, so all requests are for the
 *	  the same device
 */
static void pkt_request(request_queue_t *q)
{
	struct pktcdvd_device *pd = (struct pktcdvd_device *) q->queuedata;
	unsigned long flags;

	if (list_empty(&q->queue_head))
		return;

	while (!list_empty(&q->queue_head)) {
		struct request *rq = blkdev_entry_next_request(&q->queue_head);

		VPRINTK("pkt_request: cmd=%d, rq=%p, rq->sector=%ld, rq->nr_sectors=%ld\n", rq->cmd, rq, rq->sector, rq->nr_sectors);

		blkdev_dequeue_request(rq);

		rq->rq_dev = pd->dev;

		if (rq->cmd == READ)
			BUG();

		if (test_bit(PACKET_RECOVERY, &pd->flags))
			break;

		/*
		 * paranoia, shouldn't trigger...
		 */
		if (!pkt_one_zone(pd, rq)) {
			printk("rq->cmd=%d, rq->sector=%ld, rq->nr_sectors=%ld\n",
				rq->cmd, rq->sector, rq->nr_sectors);
			BUG();
		}

		pkt_attempt_remerge(pd, q, rq);

		spin_lock_irqsave(&pd->lock, flags);

		/*
		 * already gathering data for another read. the
		 * rfn will be reinvoked once that is done
		 */
		if (test_and_set_bit(PACKET_RQ, &pd->flags)) {
			list_add(&rq->queue, &q->queue_head);
			spin_unlock_irqrestore(&pd->lock, flags);
			break;
		}

		if (pd->rq)
			BUG();

		pd->rq = rq;
		spin_unlock_irqrestore(&pd->lock, flags);
		break;
	}
	VPRINTK("wake up wait queue\n");
	wake_up(&pd->wqueue);
}

static void pkt_print_settings(struct pktcdvd_device *pd)
{
	printk("pktcdvd: %s packets, ", pd->settings.fp ? "Fixed" : "Variable");
	printk("%u blocks, ", pd->settings.size >> 2);
	printk("Mode-%c disc\n", pd->settings.block_mode == 8 ? '1' : '2');
}

/*
 * A generic sense dump / resolve mechanism should be implemented across
 * all ATAPI + SCSI devices.
 */
static void pkt_dump_sense(struct cdrom_generic_command *cgc)
{
	static char *info[9] = { "No sense", "Recovered error", "Not ready",
				 "Medium error", "Hardware error", "Illegal request",
				 "Unit attention", "Data protect", "Blank check" };
	int i;
	struct request_sense *sense = cgc->sense;

	printk("pktcdvd:");
	for (i = 0; i < CDROM_PACKET_SIZE; i++)
		printk(" %02x", cgc->cmd[i]);
	printk(" - ");

	if (sense == NULL) {
		printk("no sense\n");
		return;
	}

	printk("sense %02x.%02x.%02x", sense->sense_key, sense->asc, sense->ascq);

	if (sense->sense_key > 8) {
		printk(" (INVALID)\n");
		return;
	}

	printk(" (%s)\n", info[sense->sense_key]);
}

/*
 * write mode select package based on pd->settings
 */
static int pkt_set_write_settings(struct pktcdvd_device *pd)
{
	struct cdrom_device_info *cdi = pd->cdi;
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	write_param_page *wp;
	char buffer[128];
	int ret, size;

	/* doesn't apply to DVD+RW (?) */
	if (pd->mmc3_profile == 0x1a)
		return 0;
/*	if (pd->mmc3_profile == 0x13) */
/*		return 0; */

	memset(buffer, 0, sizeof(buffer));
	init_cdrom_command(&cgc, buffer, sizeof(*wp), CGC_DATA_READ);
	cgc.sense = &sense;
	if ((ret = cdrom_mode_sense(cdi, &cgc, GPMODE_WRITE_PARMS_PAGE, 0))) {
		pkt_dump_sense(&cgc);
		return ret;
	}

	size = 2 + ((buffer[0] << 8) | (buffer[1] & 0xff));
	pd->mode_offset = (buffer[6] << 8) | (buffer[7] & 0xff);
	if (size > sizeof(buffer))
		size = sizeof(buffer);

	/*
	 * now get it all
	 */
	init_cdrom_command(&cgc, buffer, size, CGC_DATA_READ);
	cgc.sense = &sense;
	if ((ret = cdrom_mode_sense(cdi, &cgc, GPMODE_WRITE_PARMS_PAGE, 0))) {
		pkt_dump_sense(&cgc);
		return ret;
	}

	/*
	 * write page is offset header + block descriptor length
	 */
	wp = (write_param_page *) &buffer[sizeof(struct mode_page_header) + pd->mode_offset];

	wp->fp = pd->settings.fp;
	wp->track_mode = pd->settings.track_mode;
	wp->write_type = pd->settings.write_type;
	wp->data_block_type = pd->settings.block_mode;

	wp->multi_session = 0;

#ifdef PACKET_USE_LS
	wp->link_size = 7;
	wp->ls_v = 1;
#endif

	if (wp->data_block_type == PACKET_BLOCK_MODE1) {
		wp->session_format = 0;
		wp->subhdr2 = 0x20;
	} else if (wp->data_block_type == PACKET_BLOCK_MODE2) {
		wp->session_format = 0x20;
		wp->subhdr2 = 8;
#if 0
		wp->mcn[0] = 0x80;
		memcpy(&wp->mcn[1], PACKET_MCN, sizeof(wp->mcn) - 1);
#endif
	} else {
		/*
		 * paranoia
		 */
		printk("pktcdvd: write mode wrong %d\n", wp->data_block_type);
		return 1;
	}
	wp->packet_size = cpu_to_be32(pd->settings.size >> 2);

	cgc.buflen = cgc.cmd[8] = size;
	if ((ret = cdrom_mode_select(cdi, &cgc))) {
		pkt_dump_sense(&cgc);
		return ret;
	}

	pkt_print_settings(pd);
	return 0;
}

/*
 * 0 -- we can write to this track, 1 -- we can't
 */
static int pkt_good_track(track_information *ti)
{
	/*
	 * only good for CD-RW at the moment, not DVD-RW
	 */

	/*
	 * FIXME: only for FP
	 */
	if (ti->fp == 0)
		return 0;

	/*
	 * "good" settings as per Mt Fuji.
	 */
	if (ti->rt == 0 && ti->blank == 0 && ti->packet == 1)
		return 0;

	if (ti->rt == 0 && ti->blank == 1 && ti->packet == 1)
		return 0;

	if (ti->rt == 1 && ti->blank == 0 && ti->packet == 1)
		return 0;

	printk("pktcdvd: bad state %d-%d-%d\n", ti->rt, ti->blank, ti->packet);
	return 1;
}

/*
 * 0 -- we can write to this disc, 1 -- we can't
 */
static int pkt_good_disc(struct pktcdvd_device *pd, disc_information *di)
{
	switch (pd->mmc3_profile) {
		case 0x0a: /* CD-RW */
		case 0xffff: /* MMC3 not supported ? */
			break;
		case 0x1a: /* DVD+RW */
			return 0;
		case 0x13: /* DVD-RW Experimental */
			return 0;
		default:
			printk("pktcdvd: Wrong disc profile (%x)\n", pd->mmc3_profile);
			return 1;
	}

	/*
	 * for disc type 0xff we should probably reserve a new track.
	 * but i'm not sure, should we leave this to user apps? probably.
	 */
	if (di->disc_type == 0xff) {
		printk("pktcdvd: Unknown disc. No track?\n");
		return 1;
	}

	if (di->disc_type != 0x20 && di->disc_type != 0) {
		printk("pktcdvd: Wrong disc type (%x)\n", di->disc_type);
		return 1;
	}

	if (di->erasable == 0) {
		printk("pktcdvd: Disc not erasable\n");
		return 1;
	}

	if (pd->track_status == PACKET_SESSION_RESERVED) {
		printk("pktcdvd: Can't write to last track (reserved)\n");
		return 1;
	}

	return 0;
}

static int pkt_probe_settings(struct pktcdvd_device *pd)
{
	struct cdrom_generic_command cgc;
	unsigned char buf[12];
	disc_information di;
	track_information ti;
	int ret, track;
	int frames_per_packet;

	init_cdrom_command(&cgc, buf, 8, CGC_DATA_READ);
	cgc.cmd[0] = GPCMD_GET_CONFIGURATION;
	cgc.cmd[8] = 8;
	ret = pd->cdi->ops->generic_packet(pd->cdi, &cgc);
	pd->mmc3_profile = ret ? 0xffff : buf[6] << 8 | buf[7];

	memset(&di, 0, sizeof(disc_information));
	memset(&ti, 0, sizeof(track_information));

	if ((ret = cdrom_get_disc_info(pd->dev, &di))) {
		printk("failed get_disc\n");
		return ret;
	}

	pd->disc_status = di.disc_status;
	pd->track_status = di.border_status;

	if (pkt_good_disc(pd, &di))
		return -ENXIO;

	printk("pktcdvd: inserted media is CD-R%s\n", di.erasable ? "W" : "");
	pd->type = di.erasable ? PACKET_CDRW : PACKET_CDR;

	track = 1; /* (di.last_track_msb << 8) | di.last_track_lsb; */
	if ((ret = cdrom_get_track_info(pd->dev, track, 1, &ti))) {
		printk("pktcdvd: failed get_track\n");
		return ret;
	}

	if (pkt_good_track(&ti)) {
		printk("pktcdvd: can't write to this track\n");
		return -ENXIO;
	}

	/*
	 * we keep packet size in 512 byte units, makes it easier to
	 * deal with request calculations.
	 */
	frames_per_packet = be32_to_cpu(ti.fixed_packet_size);
	if (frames_per_packet == 0) {
		printk("pktcdvd: detected zero packet size!\n");
		frames_per_packet = PACKET_MAX_SIZE;
	}
	if (frames_per_packet > PACKET_MAX_SIZE) {
		printk("pktcdvd: packet size too large\n");
		return -ENXIO;
	}
	pd->settings.size = frames_per_packet << 2;
	pd->settings.fp = ti.fp;
	pd->offset = (be32_to_cpu(ti.track_start) << 2) & (pd->settings.size - 1);

	if (ti.nwa_v) {
		pd->nwa = be32_to_cpu(ti.next_writable);
		set_bit(PACKET_NWA_VALID, &pd->flags);
	}

	/*
	 * in theory we could use lra on -RW media as well and just zero
	 * blocks that haven't been written yet, but in practice that
	 * is just a no-go. we'll use that for -R, naturally.
	 */
	if (ti.lra_v) {
		pd->lra = be32_to_cpu(ti.last_rec_address);
		set_bit(PACKET_LRA_VALID, &pd->flags);
	} else {
		pd->lra = 0xffffffff;
		set_bit(PACKET_LRA_VALID, &pd->flags);
	}

	/*
	 * fine for now
	 */
	pd->settings.link_loss = 7;
	pd->settings.write_type = 0;	/* packet */
	pd->settings.track_mode = ti.track_mode;

	/*
	 * mode1 or mode2 disc
	 */
	switch (ti.data_mode) {
		case PACKET_MODE1:
			pd->settings.block_mode = PACKET_BLOCK_MODE1;
			break;
		case PACKET_MODE2:
			pd->settings.block_mode = PACKET_BLOCK_MODE2;
			break;
		default:
			printk("pktcdvd: unknown data mode\n");
			return 1;
	}
	return 0;
}

/*
 * enable/disable write caching on drive
 */
static int pkt_write_caching(struct pktcdvd_device *pd, int set)
{
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	unsigned char buf[64];
	int ret;

	memset(buf, 0, sizeof(buf));
	init_cdrom_command(&cgc, buf, sizeof(buf), CGC_DATA_READ);
	cgc.sense = &sense;
	cgc.buflen = pd->mode_offset + 12;

	/*
	 * caching mode page might not be there, so quiet this command
	 */
	cgc.quiet = 1;

	if ((ret = cdrom_mode_sense(pd->cdi, &cgc, GPMODE_WCACHING_PAGE, 0)))
		return ret;

	buf[pd->mode_offset + 10] |= (!!set << 2);

	cgc.buflen = cgc.cmd[8] = 2 + ((buf[0] << 8) | (buf[1] & 0xff));
	ret = cdrom_mode_select(pd->cdi, &cgc);
	if (ret) {
		printk("pktcdvd: write caching control failed\n");
		pkt_dump_sense(&cgc);
	} else if (!ret && set)
		printk("pktcdvd: enabled write caching on %s\n", pd->name);
	return ret;
}

/*
 * flush the drive cache to media
 */
static int pkt_flush_cache(struct pktcdvd_device *pd)
{
	struct cdrom_generic_command cgc;

	init_cdrom_command(&cgc, NULL, 0, CGC_DATA_NONE);
	cgc.cmd[0] = GPCMD_FLUSH_CACHE;
	cgc.quiet = 1;
	cgc.timeout = 60*HZ;

	/*
	 * the IMMED bit -- we default to not setting it, although that
	 * would allow a much faster close, this is safer
	 */
#if 0
	cgc.cmd[1] = 1 << 1;
#endif
	return pd->cdi->ops->generic_packet(pd->cdi, &cgc);
}

/*
 * Returns drive maximum write speed
 */
static int pkt_get_max_speed(struct pktcdvd_device *pd, unsigned *write_speed)
{
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	unsigned char buf[256+18];
	unsigned char *cap_buf;
	int ret, offset;

	memset(buf, 0, sizeof(buf));
	cap_buf = &buf[sizeof(struct mode_page_header) + pd->mode_offset];
	init_cdrom_command(&cgc, buf, sizeof(buf), CGC_DATA_UNKNOWN);
	cgc.sense = &sense;

	ret = cdrom_mode_sense(pd->cdi, &cgc, GPMODE_CAPABILITIES_PAGE, 0);
	if (ret) {
		cgc.buflen = pd->mode_offset + cap_buf[1] + 2 +
			     sizeof(struct mode_page_header);
		ret = cdrom_mode_sense(pd->cdi, &cgc, GPMODE_CAPABILITIES_PAGE, 0);
		if (ret) {
			pkt_dump_sense(&cgc);
			return ret;
		}
	}

	offset = 20;			    /* Obsoleted field, used by older drives */
	if (cap_buf[1] >= 28)
		offset = 28;		    /* Current write speed selected */
	if (cap_buf[1] >= 30) {
		/* If the drive reports at least one "Logical Unit Write
		 * Speed Performance Descriptor Block", use the information
		 * in the first block. (contains the highest speed)
		 */
		int num_spdb = (cap_buf[30] << 8) + cap_buf[31];
		if (num_spdb > 0)
			offset = 34;
	}

	*write_speed = ((cap_buf[offset] << 8) | cap_buf[offset + 1]) / 0xb0;
	return 0;
}

/* These tables from cdrecord - I don't have orange book */
/* standard speed CD-RW (1-4x) */
static char clv_to_speed[16] = {
	/* 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 */
	   0, 2, 4, 6, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
/* high speed CD-RW (-10x) */
static char hs_clv_to_speed[16] = {
	/* 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 */
	   0, 2, 4, 6, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
/* ultra high speed CD-RW */
static char us_clv_to_speed[16] = {
	/* 0  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 */
	   0, 2, 4, 8, 0, 0,16, 0,24,32,40,48, 0, 0, 0, 0
};

/*
 * reads the maximum media speed from ATIP
 */
static int pkt_media_speed(struct pktcdvd_device *pd, unsigned *speed)
{
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	unsigned char buf[64];
	unsigned int size, st, sp;
	int ret;

	init_cdrom_command(&cgc, buf, 2, CGC_DATA_READ);
	cgc.sense = &sense;
	cgc.cmd[0] = GPCMD_READ_TOC_PMA_ATIP;
	cgc.cmd[1] = 2;
	cgc.cmd[2] = 4; /* READ ATIP */
	cgc.cmd[8] = 2;
	ret = pd->cdi->ops->generic_packet(pd->cdi, &cgc);
	if (ret) {
		pkt_dump_sense(&cgc);
		return ret;
	}
	size = ((unsigned int) buf[0]<<8) + buf[1] + 2;
	if (size > sizeof(buf))
		size = sizeof(buf);

	init_cdrom_command(&cgc, buf, size, CGC_DATA_READ);
	cgc.sense = &sense;
	cgc.cmd[0] = GPCMD_READ_TOC_PMA_ATIP;
	cgc.cmd[1] = 2;
	cgc.cmd[2] = 4;
	cgc.cmd[8] = size;
	ret = pd->cdi->ops->generic_packet(pd->cdi, &cgc);
	if (ret) {
		pkt_dump_sense(&cgc);
		return ret;
	}

	if (!buf[6] & 0x40) {
		printk("pktcdvd: Disc type is not CD-RW\n");
		return 1;
	}
	if (!buf[6] & 0x4) {
		printk("pktcdvd: A1 values on media are not valid, maybe not CDRW?\n");
		return 1;
	}

	st = (buf[6] >> 3) & 0x7; /* disc sub-type */

	sp = buf[16] & 0xf; /* max speed from ATIP A1 field */

	/* Info from cdrecord */
	switch (st) {
		case 0: /* standard speed */
			*speed = clv_to_speed[sp];
			break;
		case 1: /* high speed */
			*speed = hs_clv_to_speed[sp];
			break;
		case 2: /* ultra high speed */
			*speed = us_clv_to_speed[sp];
			break;
		default:
			printk("pktcdvd: Unknown disc sub-type %d\n",st);
			return 1;
	}
	if (*speed) {
		printk("pktcdvd: Max. media speed: %d\n",*speed);
		return 0;
	} else {
		printk("pktcdvd: Unknown speed %d for sub-type %d\n",sp,st);
		return 1;
	}
}

/*
 * speed is given as the normal factor, e.g. 4 for 4x
 */
static int pkt_set_speed(struct pktcdvd_device *pd, unsigned write_speed, unsigned read_speed)
{
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	int ret;

	/*
	 * we set read and write time so that read spindle speed is one and
	 * a half as fast as write. although a drive can typically read much
	 * faster than write, this minimizes the spin up/down when we write
	 * and gather data. maybe 1/1 factor is faster, needs a bit of testing.
	 */
	write_speed = write_speed * 177; /* should be 176.4, but CD-RWs rounds down */
	write_speed = min_t(unsigned, write_speed, 0xffff);
	read_speed = read_speed * 177;
	read_speed = min_t(unsigned, read_speed, 0xffff);

	init_cdrom_command(&cgc, NULL, 0, CGC_DATA_NONE);
	cgc.sense = &sense;
	cgc.cmd[0] = GPCMD_SET_SPEED;
	cgc.cmd[2] = (read_speed >> 8) & 0xff;
	cgc.cmd[3] = read_speed & 0xff;
	cgc.cmd[4] = (write_speed >> 8) & 0xff;
	cgc.cmd[5] = write_speed & 0xff;

	if ((ret = pd->cdi->ops->generic_packet(pd->cdi, &cgc)))
		pkt_dump_sense(&cgc);

	return ret;
}

static int pkt_perform_opc(struct pktcdvd_device *pd)
{
	struct cdrom_generic_command cgc;
	struct request_sense sense;
	int ret;

	VPRINTK("pktcdvd: Performing OPC\n");

	init_cdrom_command(&cgc, NULL, 0, CGC_DATA_NONE);
	cgc.sense = &sense;
	cgc.timeout = 60*HZ;
	cgc.cmd[0] = GPCMD_SEND_OPC;
	cgc.cmd[1] = 1;
	if ((ret = pd->cdi->ops->generic_packet(pd->cdi, &cgc)))
		pkt_dump_sense(&cgc);
	return ret;
}

static int pkt_open_write(struct pktcdvd_device *pd)
{
	int ret;
	unsigned int write_speed, media_write_speed, read_speed;

	if ((ret = pkt_probe_settings(pd))) {
		DPRINTK("pktcdvd: %s failed probe\n", pd->name);
		return -EIO;
	}

	if ((ret = pkt_set_write_settings(pd))) {
		DPRINTK("pktcdvd: %s failed saving write settings\n", pd->name);
		return -EIO;
	}

	(void) pkt_write_caching(pd, USE_WCACHING);

	if ((ret = pkt_get_max_speed(pd, &write_speed)))
		write_speed = 16;
	switch (pd->mmc3_profile) {
		case 0x13: /* DVD-RW Experimental */
		case 0x1a: /* DVD+RW */
			read_speed = write_speed;
			break;
		default:
			if ((ret = pkt_media_speed(pd, &media_write_speed)))
				media_write_speed = 16;
			write_speed = min_t(unsigned, write_speed, media_write_speed);
			read_speed = (write_speed * 3) / 2;
			break;
	}

	if ((ret = pkt_set_speed(pd, write_speed, read_speed))) {
		DPRINTK("pktcdvd: %s couldn't set write speed\n", pd->name);
		/*
		return -EIO;
		*/
	}
	DPRINTK("pktcdvd: speed (R/W) %u/%u\n", read_speed, write_speed);
	pd->write_speed = write_speed;

	if ((ret = pkt_perform_opc(pd))) {
		DPRINTK("pktcdvd: %s Optimum Power Calibration failed\n", pd->name);
	}

	return 0;
}

/*
 * called at open time.
 */
static int pkt_open_dev(struct pktcdvd_device *pd, int write)
{
	int ret;
	long lba;

	if (!pd->dev)
		return -ENXIO;

	pd->bdev = bdget(kdev_t_to_nr(pd->dev));
	if (!pd->bdev) {
		printk("pktcdvd: can't find cdrom block device\n");
		return -ENXIO;
	}

	if ((ret = blkdev_get(pd->bdev, FMODE_READ, 0, BDEV_FILE))) {
		pd->bdev = NULL;
		return ret;
	}

	if ((ret = cdrom_get_last_written(pd->dev, &lba))) {
		printk("pktcdvd: cdrom_get_last_written failed\n");
		return ret;
	}

	pkt_sizes[MINOR(pd->pkt_dev)] = lba << 1;

	if (write) {
		if ((ret = pkt_open_write(pd)))
			return ret;
		pkt_mark_readonly(pd, 0);
	} else {
		(void) pkt_set_speed(pd, 0xffff, 0xffff);
		pkt_mark_readonly(pd, 1);
	}

	if (write)
		printk("pktcdvd: %lukB available on disc\n", lba << 1);

	return 0;
}

/*
 * called when the device is closed. makes sure that the device flushes
 * the internal cache before we close.
 */
static void pkt_release_dev(struct pktcdvd_device *pd, int flush)
{
	atomic_dec(&pd->refcnt);
	if (atomic_read(&pd->refcnt) > 0)
		return;

	fsync_dev(pd->pkt_dev);

	if (flush && pkt_flush_cache(pd))
		DPRINTK("pktcdvd: %s not flushing cache\n", pd->name);

	pkt_set_speed(pd, 0xffff, 0xffff);

	if (pd->bdev) {
		blkdev_put(pd->bdev, BDEV_FILE);
		pd->bdev = NULL;
	}
}

static int pkt_open(struct inode *inode, struct file *file)
{
	struct pktcdvd_device *pd = NULL;
	int ret;

	VPRINTK("pktcdvd: entering open\n");

	if (MINOR(inode->i_rdev) >= MAX_WRITERS) {
		printk("pktcdvd: max %d writers supported\n", MAX_WRITERS);
		ret = -ENODEV;
		goto out;
	}

	/*
	 * either device is not configured, or pktsetup is old and doesn't
	 * use O_CREAT to create device
	 */
	pd = &pkt_devs[MINOR(inode->i_rdev)];
	if (!pd->dev && !(file->f_flags & O_CREAT)) {
		VPRINTK("pktcdvd: not configured and O_CREAT not set\n");
		ret = -ENXIO;
		goto out;
	}

	atomic_inc(&pd->refcnt);
	if (atomic_read(&pd->refcnt) > 1) {
		if (file->f_mode & FMODE_WRITE) {
			VPRINTK("pktcdvd: busy open for write\n");
			ret = -EBUSY;
			goto out_dec;
		}

		/*
		 * Not first open, everything is already set up
		 */
		return 0;
	}

	if (((file->f_flags & O_ACCMODE) != O_RDONLY) || !(file->f_flags & O_CREAT)) {
		if (pkt_open_dev(pd, file->f_mode & FMODE_WRITE)) {
			ret = -EIO;
			goto out_dec;
		}
	}

	/*
	 * needed here as well, since ext2 (among others) may change
	 * the blocksize at mount time
	 */
	set_blocksize(pd->pkt_dev, CD_FRAMESIZE);
	return 0;

out_dec:
	atomic_dec(&pd->refcnt);
	if (atomic_read(&pd->refcnt) == 0) {
		if (pd->bdev) {
			blkdev_put(pd->bdev, BDEV_FILE);
			pd->bdev = NULL;
		}
	}
out:
	VPRINTK("pktcdvd: failed open (%d)\n", ret);
	return ret;
}

static int pkt_close(struct inode *inode, struct file *file)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(inode->i_rdev)];
	int ret = 0;

	if (pd->dev) {
		int flush = !test_bit(PACKET_READONLY, &pd->flags);
		pkt_release_dev(pd, flush);
	}

	return ret;
}

/*
 * pktcdvd i/o elevator parts
 */
static inline int pkt_bh_rq_ordered(struct buffer_head *bh, struct request *rq,
				    struct list_head *head)
{
	struct list_head *next;
	struct request *next_rq;

	next = rq->queue.next;
	if (next == head)
		return 0;

	next_rq = blkdev_entry_to_request(next);
	if (next_rq->rq_dev != rq->rq_dev)
		return bh->b_rsector > rq->sector;

	if (bh->b_rsector < next_rq->sector && bh->b_rsector > rq->sector)
		return 1;

	if (next_rq->sector > rq->sector)
		return 0;

	if (bh->b_rsector > rq->sector || bh->b_rsector < next_rq->sector)
		return 1;

	return 0;
}

static int pkt_elevator_merge(request_queue_t *q, struct request **req,
			      struct list_head *head,
			      struct buffer_head *bh, int rw,
			      int max_sectors)
{
	struct list_head *entry = &q->queue_head;
	unsigned int count = bh->b_size >> 9, ret = ELEVATOR_NO_MERGE;

	if (bh->b_reqnext)
		BUG();

	VPRINTK("pkt_elevator_merge: rw=%d, ms=%d, bh=%lu, dev=%d\n", rw, max_sectors, bh->b_rsector, bh->b_rdev);

	while ((entry = entry->prev) != head) {
		struct request *__rq = blkdev_entry_to_request(entry);
		if (__rq->waiting)
			continue;
		if (__rq->rq_dev != bh->b_rdev)
			continue;
		if (!*req && pkt_bh_rq_ordered(bh, __rq, &q->queue_head))
			*req = __rq;
		if (__rq->cmd != rw)
			continue;
		if (__rq->nr_sectors + count > max_sectors)
			continue;
		if (__rq->sector + __rq->nr_sectors == bh->b_rsector) {
			ret = ELEVATOR_BACK_MERGE;
			*req = __rq;
			break;
		} else if (__rq->sector - count == bh->b_rsector) {
			ret = ELEVATOR_FRONT_MERGE;
			*req = __rq;
			break;
		}
#if 0 /* makes sense, chance of two matches probably slim */
		else if (*req)
			break;
#endif
	}
	VPRINTK("*req=%p, ret=%d\n", *req, ret);

	return ret;
}

static int pkt_make_request(request_queue_t *q, int rw, struct buffer_head *bh)
{
	struct pktcdvd_device *pd;
	struct buffer_head *new_bh;

	if (MINOR(bh->b_rdev) >= MAX_WRITERS) {
		printk("pktcdvd: %s out of range\n", kdevname(bh->b_rdev));
		goto end_io;
	}

	pd = &pkt_devs[MINOR(bh->b_rdev)];
	if (!pd->dev) {
		printk("pktcdvd: request received for non-active pd\n");
		goto end_io;
	}

	/*
	 * quick remap a READ
	 */
	if (rw == READ || rw == READA) {
		down(&pd->cache_sync_mutex);
		pkt_flush_writes(pd);
		bh->b_rdev = pd->dev;
		generic_make_request(rw, bh);
		up(&pd->cache_sync_mutex);
		return 0;
	}

	if (!(rw & WRITE))
		BUG();

	if (test_bit(PACKET_READONLY, &pd->flags)) {
		printk("pktcdvd: WRITE for ro device %s (%lu)\n",
			pd->name, bh->b_rsector);
		goto end_io;
	}

	VPRINTK("pkt_make_request: bh:%p block:%ld size:%d\n",
		bh, bh->b_blocknr, bh->b_size);

	if (bh->b_size != CD_FRAMESIZE) {
		printk("pktcdvd: wrong bh size\n");
		goto end_io;
	}

	/*
	 * This is deadlock safe, since pkt_get_stacked_bh can only
	 * fail if there are already buffers in flight for this
	 * packet device. When the in-flight buffers finish, we
	 * will be woken up and try again.
	 */
	new_bh = kmem_cache_alloc(bh_cachep, GFP_ATOMIC);
	while (!new_bh) {
		DECLARE_WAITQUEUE(wait, current);

		generic_unplug_device(q);

		add_wait_queue(&pd_bh_wait, &wait);
		set_current_state(TASK_UNINTERRUPTIBLE);

		new_bh = pkt_get_stacked_bh(pd);
		if (!new_bh)
			schedule();

		set_current_state(TASK_RUNNING);
		remove_wait_queue(&pd_bh_wait, &wait);
	}

	new_bh->b_size = bh->b_size;
	new_bh->b_list = PKT_BUF_LIST + 1;
	new_bh->b_dev = bh->b_dev;
	atomic_set(&new_bh->b_count, 1);
	new_bh->b_rdev = bh->b_rdev;
	new_bh->b_state = bh->b_state;
	new_bh->b_page = bh->b_page;
	new_bh->b_data = bh->b_data;
	new_bh->b_private = bh;
	new_bh->b_end_io = pkt_end_io_write_stacked;
	new_bh->b_rsector = bh->b_rsector;

	return pd->make_request_fn(q, rw, new_bh);

end_io:
	buffer_IO_error(bh);
	return 0;
}

static void show_requests(request_queue_t *q)
{
	struct list_head *entry;

	spin_lock_irq(&io_request_lock);

	list_for_each(entry, &q->queue_head) {
		struct request *rq = blkdev_entry_to_request(entry);
		int zone = rq->sector & ~127;
		int hole;

		hole = 0;
		if ((rq->sector + rq->nr_sectors - (rq->bhtail->b_size >> 9))
		    != rq->bhtail->b_rsector)
			hole = 1;

		printk("rq: cmd %d, sector %lu (-> %lu), zone %u, hole %d, nr_sectors %lu\n", rq->cmd, rq->sector, rq->sector + rq->nr_sectors - 1, zone, hole, rq->nr_sectors);
	}

	spin_unlock_irq(&io_request_lock);
}

static void sysrq_handle_show_requests(int key, struct pt_regs *pt_regs,
		struct kbd_struct *kbd, struct tty_struct *tty)
{
	/*
	 * quick hack to show pending requests in /dev/pktcdvd0 queue
	 */
	queue_proc *qp = blk_dev[PACKET_MAJOR].queue;
	if (qp) {
		request_queue_t *q = qp(MKDEV(PACKET_MAJOR, 0));
		if (q)
			show_requests(q);
	}
}
static struct sysrq_key_op sysrq_show_requests_op = {
	handler:	sysrq_handle_show_requests,
	help_msg:	"showreQuests",
	action_msg:	"Show requests",
};

static void pkt_init_queue(struct pktcdvd_device *pd)
{
	request_queue_t *q = &pd->cdrw.r_queue;

	blk_init_queue(q, pkt_request);
	elevator_init(&q->elevator, ELEVATOR_PKTCDVD);
	pd->make_request_fn = q->make_request_fn;
	blk_queue_make_request(q, pkt_make_request);
	blk_queue_headactive(q, 0);
	q->front_merge_fn = pkt_front_merge_fn;
	q->back_merge_fn = pkt_back_merge_fn;
	q->merge_requests_fn = pkt_merge_requests_fn;
	q->queuedata = pd;
}

static int pkt_proc_device(struct pktcdvd_device *pd, char *buf)
{
	char *b = buf, *msg;
	struct list_head *foo;
	int i;

	b += sprintf(b, "\nWriter %s (%s):\n", pd->name, kdevname(pd->dev));

	b += sprintf(b, "\nSettings:\n");
	b += sprintf(b, "\tpacket size:\t\t%dkB\n", pd->settings.size / 2);

	if (pd->settings.write_type == 0)
		msg = "Packet";
	else
		msg = "Unknown";
	b += sprintf(b, "\twrite type:\t\t%s\n", msg);

	b += sprintf(b, "\tpacket type:\t\t%s\n", pd->settings.fp ? "Fixed" : "Variable");
	b += sprintf(b, "\tlink loss:\t\t%d\n", pd->settings.link_loss);

	b += sprintf(b, "\ttrack mode:\t\t%d\n", pd->settings.track_mode);

	if (pd->settings.block_mode == PACKET_BLOCK_MODE1)
		msg = "Mode 1";
	else if (pd->settings.block_mode == PACKET_BLOCK_MODE2)
		msg = "Mode 2";
	else
		msg = "Unknown";
	b += sprintf(b, "\tblock mode:\t\t%s\n", msg);

	b += sprintf(b, "\nStatistics:\n");
	b += sprintf(b, "\tbuffers started:\t%lu\n", pd->stats.bh_s);
	b += sprintf(b, "\tbuffers ended:\t\t%lu\n", pd->stats.bh_e);
	b += sprintf(b, "\tsectors written:\t%lu\n", pd->stats.secs_w);
	b += sprintf(b, "\tsectors read:\t\t%lu\n", pd->stats.secs_r);
	b += sprintf(b, "\tbuffer cache hits:\t%lu\n", pd->stats.bh_cache_hits);
	b += sprintf(b, "\tpage cache hits:\t%lu\n", pd->stats.page_cache_hits);

	b += sprintf(b, "\nMisc:\n");
	b += sprintf(b, "\treference count:\t%d\n", atomic_read(&pd->refcnt));
	b += sprintf(b, "\tflags:\t\t\t0x%lx\n", pd->flags);
	b += sprintf(b, "\twrite speed:\t\t%ukB/s\n", pd->write_speed * 150);
	b += sprintf(b, "\tstart offset:\t\t%lu\n", pd->offset);
	b += sprintf(b, "\tmode page offset:\t%u\n", pd->mode_offset);

	b += sprintf(b, "\nQueue state:\n");
	b += sprintf(b, "\tfree buffers:\t\t%u\n", atomic_read(&pd->cdrw.free_bh));
	b += sprintf(b, "\trequest active:\t\t%s\n", pd->rq ? "yes" : "no");
	b += sprintf(b, "\twrite rq depth:\t\t%d\n", atomic_read(&pd->wrqcnt));

	spin_lock_irq(&io_request_lock);
	i = 0;
	list_for_each(foo, &pd->cdrw.r_queue.queue_head)
		i++;
	spin_unlock_irq(&io_request_lock);
	b += sprintf(b, "\tqueue requests:\t\t%u\n", i);

	return b - buf;
}

static int pkt_read_proc(char *page, char **start, off_t off, int count,
			 int *eof, void *data)
{
	struct pktcdvd_device *pd = data;
	char *buf = page;
	int len;

	len = pkt_proc_device(pd, buf);
	buf += len;

	if (len <= off + count)
		*eof = 1;

	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;

	return len;
}

static int pkt_new_dev(struct pktcdvd_device *pd, kdev_t dev)
{
	struct cdrom_device_info *cdi;
	request_queue_t *q;
	int i;

	for (i = 0; i < MAX_WRITERS; i++) {
		if (pkt_devs[i].dev == dev) {
			printk("pktcdvd: %s already setup\n", kdevname(dev));
			return -EBUSY;
		}
	}

	for (i = 0; i < MAX_WRITERS; i++)
		if (pd == &pkt_devs[i])
			break;
	BUG_ON(i == MAX_WRITERS);

	cdi = cdrom_find_device(dev);
	if (cdi == NULL) {
		printk("pktcdvd: %s is not a CD-ROM\n", kdevname(dev));
		return -ENXIO;
	}

	MOD_INC_USE_COUNT;

	memset(pd, 0, sizeof(struct pktcdvd_device));
	atomic_set(&pd->cdrw.free_bh, 0);

	spin_lock_init(&pd->lock);
	if (pkt_grow_bhlist(pd, PACKET_MAX_SIZE) < PACKET_MAX_SIZE) {
		MOD_DEC_USE_COUNT;
		printk("pktcdvd: not enough memory for buffers\n");
		return -ENOMEM;
	}

	pd->stacked_bhcnt = 0;
	if (!pkt_grow_stacked_bhlist(pd)) {
		MOD_DEC_USE_COUNT;
		printk("pktcdvd: not enough memory for buffer heads\n");
		return -ENOMEM;
	}

	set_blocksize(dev, CD_FRAMESIZE);
	pd->cdi = cdi;
	pd->dev = dev;
	pd->bdev = NULL;
	pd->pkt_dev = MKDEV(PACKET_MAJOR, i);
	sprintf(pd->name, "pktcdvd%d", i);
	atomic_set(&pd->refcnt, 0);
	atomic_set(&pd->wrqcnt, 0);
	init_MUTEX(&pd->cache_sync_mutex);
	pd->unflushed_writes = 0;
	init_waitqueue_head(&pd->wqueue);
	init_completion(&pd->cdrw.thr_compl);

	/*
	 * store device merge functions (SCSI uses their own to build
	 * scatter-gather tables)
	 */
	q = blk_get_queue(dev);
	pkt_init_queue(pd);
	pd->cdrw.front_merge_fn = q->front_merge_fn;
	pd->cdrw.back_merge_fn = q->back_merge_fn;
	pd->cdrw.merge_requests_fn = q->merge_requests_fn;
	pd->cdrw.queuedata = q->queuedata;

	pd->cdrw.time_to_die = 0;
	pd->cdrw.pid = kernel_thread(kcdrwd, pd, CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (pd->cdrw.pid < 0) {
		MOD_DEC_USE_COUNT;
		printk("pktcdvd: can't start kernel thread\n");
		blk_cleanup_queue(&pd->cdrw.r_queue);
		pkt_shrink_stacked_bhlist(pd);
		pkt_shrink_bhlist(pd, PACKET_MAX_SIZE);
		memset(pd, 0, sizeof(*pd));
		return -EBUSY;
	}

	create_proc_read_entry(pd->name, 0, pkt_proc, pkt_read_proc, pd);
	DPRINTK("pktcdvd: writer %s sucessfully registered\n", cdi->name);
	return 0;
}

/*
 * arg contains file descriptor of CD-ROM device.
 */
static int pkt_setup_dev(struct pktcdvd_device *pd, unsigned int arg)
{
	struct inode *inode;
	struct file *file;
	int ret;

	if ((file = fget(arg)) == NULL) {
		printk("pktcdvd: bad file descriptor passed\n");
		return -EBADF;
	}

	ret = -EINVAL;
	if ((inode = file->f_dentry->d_inode) == NULL) {
		printk("pktcdvd: huh? file descriptor contains no inode?\n");
		goto out;
	}
	ret = -ENOTBLK;
	if (!S_ISBLK(inode->i_mode)) {
		printk("pktcdvd: device is not a block device (duh)\n");
		goto out;
	}
	ret = -EROFS;
	if (IS_RDONLY(inode)) {
		printk("pktcdvd: Can't write to read-only dev\n");
		goto out;
	}
	if ((ret = pkt_new_dev(pd, inode->i_rdev))) {
		printk("pktcdvd: all booked up\n");
		goto out;
	}

	atomic_inc(&pd->refcnt);

out:
	fput(file);
	return ret;
}

static int pkt_remove_dev(struct pktcdvd_device *pd)
{
	int ret;

	if (pd->cdrw.pid >= 0) {
		pd->cdrw.time_to_die = 1;
		wmb();
		ret = kill_proc(pd->cdrw.pid, SIGKILL, 1);
		if (ret) {
			printk("pkt_exit: can't kill kernel thread\n");
			return ret;
		}
		wait_for_completion(&pd->cdrw.thr_compl);
	}

	/*
	 * will also invalidate buffers for CD-ROM
	 */
	invalidate_device(pd->pkt_dev, 1);

	pkt_shrink_stacked_bhlist(pd);
	if ((ret = pkt_shrink_bhlist(pd, PACKET_MAX_SIZE)) != PACKET_MAX_SIZE)
		printk("pktcdvd: leaked %d buffers\n", PACKET_MAX_SIZE - ret);

	blk_cleanup_queue(&pd->cdrw.r_queue);
	remove_proc_entry(pd->name, pkt_proc);
	DPRINTK("pktcdvd: writer %s unregistered\n", pd->cdi->name);
	memset(pd, 0, sizeof(struct pktcdvd_device));
	MOD_DEC_USE_COUNT;
	return 0;
}

static int pkt_media_change(kdev_t dev)
{
	struct pktcdvd_device *pd = pkt_find_dev(dev);
	if (!pd)
		return 0;
	return cdrom_media_changed(pd->dev);
}

static int pkt_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{
	struct pktcdvd_device *pd = &pkt_devs[MINOR(inode->i_rdev)];

	VPRINTK("pkt_ioctl: cmd %d, dev %x\n", cmd, inode->i_rdev);

	if ((cmd != PACKET_SETUP_DEV) && !pd->dev) {
		DPRINTK("pktcdvd: dev not setup\n");
		return -ENXIO;
	}

	switch (cmd) {
	case PACKET_GET_STATS:
		if (copy_to_user(&arg, &pd->stats, sizeof(struct packet_stats)))
			return -EFAULT;
		break;

	case PACKET_SETUP_DEV:
		if (!capable(CAP_SYS_ADMIN))
			return -EPERM;
		if (pd->dev) {
			printk("pktcdvd: dev already setup\n");
			return -EBUSY;
		}
		return pkt_setup_dev(pd, arg);

	case PACKET_TEARDOWN_DEV:
		if (!capable(CAP_SYS_ADMIN))
			return -EPERM;
		if (atomic_read(&pd->refcnt) != 1)
			return -EBUSY;
		return pkt_remove_dev(pd);

	case BLKGETSIZE:
		return put_user(blk_size[PACKET_MAJOR][MINOR(inode->i_rdev)] << 1, (unsigned long *)arg);

	case BLKGETSIZE64:
		return put_user((u64)blk_size[PACKET_MAJOR][MINOR(inode->i_rdev)] << 10,
				(u64 *)arg);

	case BLKROSET:
		if (capable(CAP_SYS_ADMIN))
			set_bit(PACKET_READONLY, &pd->flags);
	case BLKROGET:
	case BLKSSZGET:
	case BLKRASET:
	case BLKRAGET:
	case BLKFLSBUF:
		if (!pd->bdev)
			return -ENXIO;
		return blk_ioctl(inode->i_rdev, cmd, arg);

	/*
	 * forward selected CDROM ioctls to CD-ROM, for UDF
	 */
	case CDROMMULTISESSION:
	case CDROMREADTOCENTRY:
	case CDROM_LAST_WRITTEN:
	case CDROM_SEND_PACKET:
	case SCSI_IOCTL_SEND_COMMAND:
		if (!pd->bdev)
			return -ENXIO;
		return ioctl_by_bdev(pd->bdev, cmd, arg);

	default:
		printk("pktcdvd: Unknown ioctl for %s (%x)\n", pd->name, cmd);
		return -ENOTTY;
	}

	return 0;
}

static struct block_device_operations pktcdvd_ops = {
	owner:			THIS_MODULE,
	open:			pkt_open,
	release:		pkt_close,
	ioctl:			pkt_ioctl,
	check_media_change:	pkt_media_change,
};

int pkt_init(void)
{
	int i;

	devfs_register(NULL, "pktcdvd", DEVFS_FL_DEFAULT, PACKET_MAJOR, 0,
		       S_IFBLK | S_IRUSR | S_IWUSR, &pktcdvd_ops, NULL);
	if (devfs_register_blkdev(PACKET_MAJOR, "pktcdvd", &pktcdvd_ops)) {
		printk("unable to register pktcdvd device\n");
		return -EIO;
	}

	pkt_sizes = kmalloc(MAX_WRITERS * sizeof(int), GFP_KERNEL);
	if (pkt_sizes == NULL)
		goto err;

	pkt_blksize = kmalloc(MAX_WRITERS * sizeof(int), GFP_KERNEL);
	if (pkt_blksize == NULL)
		goto err;

	pkt_readahead = kmalloc(MAX_WRITERS * sizeof(int), GFP_KERNEL);
	if (pkt_readahead == NULL)
		goto err;

	pkt_devs = kmalloc(MAX_WRITERS * sizeof(struct pktcdvd_device), GFP_KERNEL);
	if (pkt_devs == NULL)
		goto err;

	memset(pkt_devs, 0, MAX_WRITERS * sizeof(struct pktcdvd_device));
	memset(pkt_sizes, 0, MAX_WRITERS * sizeof(int));
	memset(pkt_blksize, 0, MAX_WRITERS * sizeof(int));

	for (i = 0; i < MAX_WRITERS; i++)
		pkt_readahead[i] = vm_max_readahead;

	blk_size[PACKET_MAJOR] = pkt_sizes;
	blksize_size[PACKET_MAJOR] = pkt_blksize;
	max_readahead[PACKET_MAJOR] = pkt_readahead;
	read_ahead[PACKET_MAJOR] = 128;
	set_blocksize(MKDEV(PACKET_MAJOR, 0), CD_FRAMESIZE);

	blk_dev[PACKET_MAJOR].queue = pkt_get_queue;

	pkt_proc = proc_mkdir("pktcdvd", proc_root_driver);

	register_sysrq_key('q', &sysrq_show_requests_op);

	DPRINTK("pktcdvd: %s\n", VERSION_CODE);
	return 0;

err:
	printk("pktcdvd: out of memory\n");
	devfs_unregister(devfs_find_handle(NULL, "pktcdvd", 0, 0,
		 	 DEVFS_SPECIAL_BLK, 0));
	devfs_unregister_blkdev(PACKET_MAJOR, "pktcdvd");
	kfree(pkt_devs);
	kfree(pkt_sizes);
	kfree(pkt_blksize);
	kfree(pkt_readahead);
	return -ENOMEM;
}

void pkt_exit(void)
{
	unregister_sysrq_key('q', &sysrq_show_requests_op);

	devfs_unregister(devfs_find_handle(NULL, "pktcdvd", 0, 0,
		 	 DEVFS_SPECIAL_BLK, 0));
	devfs_unregister_blkdev(PACKET_MAJOR, "pktcdvd");
	blk_dev[PACKET_MAJOR].queue = NULL;

	remove_proc_entry("pktcdvd", proc_root_driver);
	kfree(pkt_sizes);
	kfree(pkt_blksize);
	kfree(pkt_devs);
	kfree(pkt_readahead);
}

MODULE_DESCRIPTION("Packet writing layer for CD/DVD drives");
MODULE_AUTHOR("Jens Axboe <axboe@suse.de>");
MODULE_LICENSE("GPL");

module_init(pkt_init);
module_exit(pkt_exit);
