/*
 *  file.c
 *
 *  Copyright (C) 1995, 1996, 1997 by Paal-Kr. Engstad and Volker Lendecke
 *  Copyright (C) 1997 by Volker Lendecke
 *
 *  Please add a note about your changes to davfs in the ChangeLog file.
 */

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/fcntl.h>
#include <linux/stat.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/pagemap.h>
#include <linux/smp_lock.h>

#include <asm/uaccess.h>
#include <asm/system.h>

#include "dav_debug.h"
#include "davfs.h"

static int
dav_fsync(struct file *file, struct dentry * dentry, int datasync)
{
	VERBOSE("sync file %s/%s\n", DENTRY_PATH(dentry));
	return 0;
}

/*
 * Read a page synchronously.
 */
static int
dav_readpage_sync(struct dentry *dentry, struct page *page)
{
	char *buffer = kmap(page);
	unsigned long offset = page->index << PAGE_CACHE_SHIFT;
	int rsize = dav_get_rsize(dav_server_from_dentry(dentry));
	int count = PAGE_SIZE;
	int result;

	VERBOSE("file %s/%s, count=%d@%ld, rsize=%d\n",
		DENTRY_PATH(dentry), count, offset, rsize);

	result = dav_open(dentry, DAV_O_RDONLY);
	if (result < 0) {
		PARANOIA("%s/%s open failed, error=%d\n",
			 DENTRY_PATH(dentry), result);
		goto io_error;
	}

	do {
		if (count < rsize)
			rsize = count;

		result = dav_proc_read(dentry->d_inode, offset, rsize, buffer);
		if (result < 0)
			goto io_error;

		count -= result;
		offset += result;
		buffer += result;
		dentry->d_inode->i_atime = CURRENT_TIME;
		if (result < rsize)
			break;
	} while (count);

	memset(buffer, 0, count);
	flush_dcache_page(page);
	SetPageUptodate(page);
	result = 0;

io_error:
	kunmap(page);
	UnlockPage(page);
	return result;
}

/*
 * We are called with the page locked and we unlock it when done.
 */
static int
dav_readpage(struct file *file, struct page *page)
{
	int		error;
	struct dentry  *dentry = file->f_dentry;

	TRACE();

	get_page(page);
	error = dav_readpage_sync(dentry, page);
	put_page(page);
	return error;
}

/*
 * Write a page synchronously.
 * Offset is the data offset within the page.
 */
static int
dav_writepage_sync(struct inode *inode, struct page *page,
		   unsigned long offset, unsigned int count)
{
    char *buffer = kmap(page) + offset;
    struct dav_i_info *dii = inode->u.generic_ip;
    
    int wsize = dav_get_wsize(dav_server_from_inode(inode));
    int result, written = 0;
    
    TRACE();
    
    offset += page->index << PAGE_CACHE_SHIFT;
    VERBOSE("file ino=%ld,  count=%d@%ld, wsize=%d\n",
	    inode->i_ino,  count, offset, wsize);
    
    do {
	if (count < wsize)
	    wsize = count;
	
	result = dav_proc_write(inode, offset, wsize, buffer);
	if (result < 0) {
	    PARANOIA("failed write, wsize=%d, result=%d\n",
		     wsize, result);
	    break;
	}
	/* N.B. what if result < wsize?? */
#ifdef davFS_PARANOIA
	if (result < wsize)
	    PARANOIA("short write, wsize=%d, result=%d\n",
		     wsize, result);
#endif
	buffer += wsize;
	offset += wsize;
	written += wsize;
	count -= wsize;
	/*
	 * Update the inode now rather than waiting for a refresh.
	 */
	inode->i_mtime = inode->i_atime = CURRENT_TIME;
	if (offset > inode->i_size)
	    inode->i_size = offset;
	dii->flags |= DAV_F_LOCALWRITE;
    } while (count);
    
    kunmap(page);
    return written ? written : result;
}

/*
 * Write a page to the server. This will be used for NFS swapping only
 * (for now), and we currently do this synchronously only.
 *
 * We are called with the page locked and we unlock it when done.
 */
static int
dav_writepage(struct page *page)
{
	struct address_space *mapping = page->mapping;
	struct inode *inode;
	unsigned long end_index;
	unsigned offset = PAGE_CACHE_SIZE;
	int err=0;

	TRACE();

	if (!mapping)
		BUG();
	inode = mapping->host;
	if (!inode)
		BUG();

	end_index = inode->i_size >> PAGE_CACHE_SHIFT;

	/* easy case */
	if (page->index < end_index)
		goto do_it;
	/* things got complicated... */
	offset = inode->i_size & (PAGE_CACHE_SIZE-1);
	/* OK, are we completely out? */
	if (page->index >= end_index+1 || !offset)
		return -EIO;
do_it:
	get_page(page);
	err = dav_writepage_sync(inode, page, 0, offset);
	SetPageUptodate(page);
	UnlockPage(page);
	put_page(page);
	return err;
}

static int
dav_updatepage(struct file *file, struct page *page, unsigned long offset,
	       unsigned int count)
{
	struct dentry *dentry = file->f_dentry;

	DEBUG1("(%s/%s %d@%ld)\n", DENTRY_PATH(dentry), 
	       count, (page->index << PAGE_CACHE_SHIFT)+offset);

	return dav_writepage_sync(dentry->d_inode, page, offset, count);
} 

static ssize_t
dav_file_read(struct file * file, char * buf, size_t count, loff_t *ppos)
{
	struct dentry * dentry = file->f_dentry;
	ssize_t	status;

	VERBOSE("file %s/%s, count=%lu@%lu\n", DENTRY_PATH(dentry),
		(unsigned long) count, (unsigned long) *ppos);

	status = dav_revalidate_inode(dentry);
	if (status)
	{
		PARANOIA("%s/%s validation failed, error=%Zd\n",
			 DENTRY_PATH(dentry), status);
		goto out;
	}

	VERBOSE("before read, size=%ld, flags=%x, atime=%ld\n",
		(long)dentry->d_inode->i_size,
		dentry->d_inode->i_flags, dentry->d_inode->i_atime);

	status = generic_file_read(file, buf, count, ppos);
out:
	return status;
}

static int
dav_file_mmap(struct file * file, struct vm_area_struct * vma)
{
	struct dentry * dentry = file->f_dentry;
	int	status;

	VERBOSE("file %s/%s, address %lu - %lu\n",
		DENTRY_PATH(dentry), vma->vm_start, vma->vm_end);

	status = dav_revalidate_inode(dentry);
	if (status)
	{
		PARANOIA("%s/%s validation failed, error=%d\n",
			 DENTRY_PATH(dentry), status);
		goto out;
	}
	status = generic_file_mmap(file, vma);
out:
	return status;
}

/*
 * This does the "real" work of the write. The generic routine has
 * allocated the page, locked it, done all the page alignment stuff
 * calculations etc. Now we should just copy the data from user
 * space and write it back to the real medium..
 *
 * If the writer ends up delaying the write, the writer needs to
 * increment the page use counts until he is done with the page.
 */
static int dav_prepare_write(struct file *file, struct page *page, 
			     unsigned offset, unsigned to)
{
	TRACE();
	kmap(page);
	return 0;
}

static int dav_commit_write(struct file *file, struct page *page,
			    unsigned offset, unsigned to)
{
	int status;

	TRACE();

	status = -EFAULT;
	lock_kernel();
	status = dav_updatepage(file, page, offset, to-offset);
	unlock_kernel();
	kunmap(page);
	return status;
}

struct address_space_operations dav_file_aops = {
	readpage: dav_readpage,
	writepage: dav_writepage,
	prepare_write: dav_prepare_write,
	commit_write: dav_commit_write
};

/* 
 * Write to a file (through the page cache).
 */
static ssize_t
dav_file_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	struct dentry * dentry = file->f_dentry;
	ssize_t	result;

	VERBOSE("file %s/%s, count=%lu@%lu\n",
		DENTRY_PATH(dentry),
		(unsigned long) count, (unsigned long) *ppos);

	result = dav_revalidate_inode(dentry);
	if (result)
	{
		PARANOIA("%s/%s validation failed, error=%Zd\n",
			 DENTRY_PATH(dentry), result);
		goto out;
	}

	result = dav_open(dentry, DAV_O_WRONLY);
	if (result)
		goto out;

	if (count > 0)
	{
		result = generic_file_write(file, buf, count, ppos);
		VERBOSE("pos=%ld, size=%ld, mtime=%ld, atime=%ld\n",
			(long) file->f_pos, (long) dentry->d_inode->i_size,
			dentry->d_inode->i_mtime, dentry->d_inode->i_atime);
	}
out:
	return result;
}

static int
dav_file_open(struct inode *inode, struct file * file)
{
	int result=0;
	struct dentry *dentry = file->f_dentry;
	struct dav_i_info *dii = inode->u.generic_ip;
	int dav_mode = (file->f_mode & O_ACCMODE) - 1;

	TRACE();

	lock_kernel();
	result = dav_open(dentry, dav_mode);
	if (result)
		goto out;
	dii->openers++;
out:
	unlock_kernel();
	return result;
}

static int
dav_file_release(struct inode *inode, struct file * file)
{
    struct dav_i_info *dii = inode->u.generic_ip;

    TRACE();
    lock_kernel();
    if (!--dii->openers)
	dav_close(inode);
    unlock_kernel();
    return 0;
}

/*
 * Check whether the required access is compatible with
 * an inode's permission. dav doesn't recognize superuser
 * privileges, so we need our own check for this.
 */
static int
dav_file_permission(struct inode *inode, int mask)
{
	int mode = inode->i_mode;
	int error = 0;

	VERBOSE("mode=%x, mask=%x\n", mode, mask);

	/* Look at user permissions */
	mode >>= 6;
	if ((mode & 7 & mask) != mask)
		error = -EACCES;
	return error;
}

struct file_operations dav_file_operations =
{
	read:		dav_file_read,
	write:		dav_file_write,
	ioctl:		dav_ioctl,
	mmap:		dav_file_mmap,
	open:		dav_file_open,
	release:	dav_file_release,
	fsync:		dav_fsync,
};

struct inode_operations dav_file_inode_operations =
{
	permission:	dav_file_permission,
	revalidate:	dav_revalidate_inode,
	setattr:	dav_notify_change,
};
