/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#ifndef _H_FCDIAG
#define _H_FCDIAG

#ifndef LP6000
/* Applications using this header file should typedef the following */
typedef unsigned int uint32;
typedef unsigned char uchar;
typedef unsigned short ushort;
typedef void* MAILBOX;
#endif

/* the brdinfo structure */
typedef struct BRDINFO {
   uint32    a_mem_hi;          /* memory identifier for adapter access */
   uint32    a_mem_low;         /* memory identifier for adapter access */
   uint32    a_flash_hi;        /* memory identifier for adapter access */
   uint32    a_flash_low;       /* memory identifier for adapter access */
   uint32    a_ctlreg_hi;       /* memory identifier for adapter access */
   uint32    a_ctlreg_low;      /* memory identifier for adapter access */
   uint32    a_intrlvl;         /* interrupt level for adapter */
   uint32    a_pci;             /* PCI identifier (device / vendor id) */
   uint32    a_busid;           /* identifier of PCI bus adapter is on */
   uint32    a_devid;           /* identifier of PCI device number */
   uchar     a_rsvd1;           /* reserved for future use */
   uchar     a_rsvd2;           /* reserved for future use */
   uchar     a_siglvl;          /* signal handler used by library */
   uchar     a_ddi;             /* identifier device driver instance number */
   uint32    a_onmask;          /* mask of ONDI primatives supported */
   uint32    a_offmask;         /* mask of OFFDI primatives supported */
   uchar     a_drvrid[16];      /* driver version */
   uchar     a_fwname[32];      /* firmware version */
} brdinfo;

/* bits in a_onmask */
#define ONDI_MBOX       0x1     /* allows non-destructive mailbox commands */
#define ONDI_IOINFO     0x2     /* supports retrieval of I/O info */
#define ONDI_LNKINFO    0x4     /* supports retrieval of link info */
#define ONDI_NODEINFO   0x8     /* supports retrieval of node info */
#define ONDI_TRACEINFO  0x10    /* supports retrieval of trace info */
#define ONDI_SETTRACE   0x20    /* supports configuration of trace info */
#define ONDI_SLI1       0x40    /* hardware supports SLI-1 interface */
#define ONDI_SLI2       0x80    /* hardware supports SLI-2 interface */
#define ONDI_BIG_ENDIAN 0x100   /* DDI interface is BIG Endian */
#define ONDI_LTL_ENDIAN 0x200   /* DDI interface is LITTLE Endian */
#define ONDI_RMEM       0x400   /* allows reading of adapter shared memory */
#define ONDI_RFLASH     0x800   /* allows reading of adapter flash */
#define ONDI_RPCI       0x1000  /* allows reading of adapter pci registers */
#define ONDI_RCTLREG    0x2000  /* allows reading of adapter cntrol registers */
#define ONDI_CFGPARAM   0x4000  /* supports get/set configuration parameters */
#define ONDI_CT         0x8000  /* supports passthru CT interface */
#define ONDI_HBAAPI     0x10000 /* supports HBA API interface */

/* bits in a_offmask */
#define OFFDI_MBOX      0x1     /* allows all mailbox commands */
#define OFFDI_RMEM      0x2     /* allows reading of adapter shared memory */
#define OFFDI_WMEM      0x4     /* allows writing of adapter shared memory */
#define OFFDI_RFLASH    0x8     /* allows reading of adapter flash */
#define OFFDI_WFLASH    0x10    /* allows writing of adapter flash */
#define OFFDI_RPCI      0x20    /* allows reading of adapter pci registers */
#define OFFDI_WPCI      0x40    /* allows writing of adapter pci registers */
#define OFFDI_RCTLREG   0x80    /* allows reading of adapter cntrol registers */
#define OFFDI_WCTLREG   0x100   /* allows writing of adapter cntrol registers */
#define OFFDI_OFFLINE   0x80000000 /* if set, adapter is in offline state */

/* values for flag in SetDiagEnv */
#define DDI_SHOW        0x0
#define DDI_ONDI        0x1
#define DDI_OFFDI       0x2

#define DDI_BRD_SHOW    0x10
#define DDI_BRD_ONDI    0x11
#define DDI_BRD_OFFDI   0x12

/* unused field */
#define DDI_UNUSED      0xFFFFFFFFL     /* indicate unused field of brdinfo */

/* the ioinfo structure */
typedef struct IOINFO {
   uint32    a_mbxCmd;          /* mailbox commands issued */
   uint32    a_mboxCmpl;        /* mailbox commands completed */
   uint32    a_mboxErr;         /* mailbox commands completed, error status */
   uint32    a_iocbCmd;         /* iocb command ring issued */
   uint32    a_iocbRsp;     /* iocb rsp ring recieved */
   uint32    a_adapterIntr;     /* adapter interrupt events */
   uint32    a_fcpCmd;          /* FCP commands issued */
   uint32    a_fcpCmpl;         /* FCP command completions recieved */
   uint32    a_fcpErr;          /* FCP command completions errors */
   uint32    a_seqXmit;         /* IP xmit sequences sent */
   uint32    a_seqRcv;          /* IP sequences recieved */
   uint32    a_bcastXmit;       /* cnt of successful xmit broadcast commands issued */
   uint32    a_bcastRcv;        /* cnt of receive broadcast commands received */
   uint32    a_elsXmit;         /* cnt of successful  ELS request commands issued */
   uint32    a_elsRcv;          /* cnt of ELS request commands received */
   uint32    a_RSCNRcv;         /* cnt of RSCN commands recieved */
   uint32    a_seqXmitErr;      /* cnt of unsuccessful xmit broadcast cmds issued */
   uint32    a_elsXmitErr;      /* cnt of unsuccessful  ELS request commands issued  */
   uint32    a_elsBufPost;      /* cnt of ELS buffers posted to adapter */
   uint32    a_ipBufPost;       /* cnt of IP buffers posted to adapter */
   uint32    a_cnt1;            /* generic counter */
   uint32    a_cnt2;            /* generic counter */
   uint32    a_cnt3;            /* generic counter */
   uint32    a_cnt4;            /* generic counter */

} IOinfo;

/* the linkinfo structure */
typedef struct LINKINFO {
   uint32   a_linkEventTag;
   uint32   a_linkUp;
   uint32   a_linkDown;
   uint32   a_linkMulti;
   uint32   a_DID;
   uchar    a_topology;
   uchar    a_linkState;
   uchar    a_alpa;
   uchar    a_alpaCnt;
   uchar    a_alpaMap[128];
   uchar    a_wwpName[8];
   uchar    a_wwnName[8];
} LinkInfo;

/* values for a_topology */
#define LNK_LOOP                0x1
#define LNK_PUBLIC_LOOP         0x2
#define LNK_FABRIC              0x3
#define LNK_PT2PT               0x4

/* values for a_linkState */
#define LNK_DOWN                0x1
#define LNK_UP                  0x2
#define LNK_FLOGI               0x3
#define LNK_DISCOVERY           0x4
#define LNK_REDISCOVERY         0x5
#define LNK_READY               0x6

/* the traceinfo structure */
typedef struct TRACEINFO {
   uchar    a_event;
   uchar    a_cmd;
   ushort   a_status;
   uint32   a_information;
} TraceInfo;

/* values for flag */
#define TRC_SHOW        0x0
#define TRC_MBOX        0x1
#define TRC_IOCB        0x2
#define TRC_INTR        0x4
#define TRC_EVENT       0x8

/* values for a_event */
#define TRC_MBOX_CMD    0x1
#define TRC_MBOX_CMPL   0x2
#define TRC_IOCB_CMD    0x3
#define TRC_IOCB_RSP    0x4
#define TRC_INTR_RCV    0x5
#define TRC_EVENT1      0x6
#define TRC_EVENT2      0x7
#define TRC_EVENT_MASK  0x7
#define TRC_RING0       0x0
#define TRC_RING1       0x40
#define TRC_RING2       0x80
#define TRC_RING3       0xC0
#define TRC_RING_MASK   0xC0

/* the cfgparam structure */
typedef struct CFGPARAM {
   uchar   a_string[32];
   uint32  a_low;
   uint32  a_hi;
   uint32  a_default;
   uint32  a_current;
   ushort  a_flag;
   ushort  a_changestate;
   uchar   a_help[80];
} CfgParam;

#define MAX_CFG_PARAM 64

/* values for a_flag */
#define CFG_EXPORT      0x1     /* Export this parameter to the end user */
#define CFG_IGNORE      0x2     /* Ignore this parameter */
#define CFG_DEFAULT     0x8000  /* Reestablishing Link */

/* values for a_changestate */
#define CFG_REBOOT      0x0     /* Changes effective after ystem reboot */
#define CFG_DYNAMIC     0x1     /* Changes effective immediately */
#define CFG_RESTART     0x2     /* Changes effective after driver restart */

/* the icfgparam structure - internal use only */
typedef struct ICFGPARAM {
   char   *a_string;
   uint32  a_low;
   uint32  a_hi;
   uint32  a_default;
   uint32  a_current;
   ushort  a_flag;
   ushort  a_changestate;
   char   *a_help;
} iCfgParam;


/* the nodeinfo structure */
typedef struct NODEINFO {
   ushort  a_flag;
   ushort  a_state;
   uint32  a_did;
   uchar   a_wwpn[8];
   uchar   a_wwnn[8];
   uint32  a_targetid;
} NodeInfo;

#define MAX_NODES 512

/* Defines for a_state */
#define NODE_UNUSED     0       /* unused NL_PORT entry */
#define NODE_LIMBO      0x1     /* entry needs to hang around for wwpn / sid */
#define NODE_LOGOUT     0x2     /* NL_PORT is not logged in - entry is cached */
#define NODE_PLOGI      0x3     /* PLOGI was sent to NL_PORT */
#define NODE_LOGIN      0x4     /* NL_PORT is logged in / login REG_LOGINed */
#define NODE_PRLI       0x5     /* PRLI was sent to NL_PORT */
#define NODE_ALLOC      0x6     /* NL_PORT is  ready to initiate adapter I/O */
#define NODE_SEED       0x7     /* seed scsi id bind in table */

/* Defines for a_flag */
#define NODE_RPI_XRI        0x1         /* creating xri for entry */
#define NODE_REQ_SND        0x2         /* sent ELS request for this entry */
#define NODE_ADDR_AUTH      0x4         /* Authenticating addr for this entry */
#define NODE_RM_ENTRY       0x8         /* Remove this entry */
#define NODE_FARP_SND       0x10        /* sent FARP request for this entry */
#define NODE_FABRIC         0x20        /* this entry represents the Fabric */
#define NODE_FCP_TARGET     0x40        /* this entry is an FCP target */
#define NODE_IP_NODE        0x80        /* this entry is an IP node */
#define NODE_DISC_START     0x100       /* start discovery on this entry */
#define NODE_SEED_WWPN      0x200       /* Entry scsi id is seeded for WWPN */
#define NODE_SEED_WWNN      0x400       /* Entry scsi id is seeded for WWNN */
#define NODE_SEED_DID       0x800       /* Entry scsi id is seeded for DID */
#define NODE_SEED_MASK      0xe00       /* mask for seeded flags */
#define NODE_AUTOMAP        0x1000      /* This entry was automap'ed */
#define NODE_NS_REMOVED     0x2000      /* This entry removed from NameServer */

/* Defines for RegisterForEvent mask */
#define FC_REG_LINK_EVENT       0x1     /* Register for link up / down events */
#define FC_REG_RSCN_EVENT       0x2     /* Register for RSCN events */
#define FC_REG_CT_EVENT         0x4     /* Register for CT request events */
#define FC_REG_EVENT_MASK       0x3f    /* event mask */
#define FC_REG_ALL_PORTS        0x80    /* Register for all ports */

#define MAX_FC_EVENTS 8       /* max events user process can wait for per HBA */
#define FC_FSTYPE_ALL 0xffff  /* match on all fsTypes */

/* Defines for error codes */
#define FC_ERROR_BUFFER_OVERFLOW          0xff
#define FC_ERROR_RESPONSE_TIMEOUT         0xfe
#define FC_ERROR_LINK_UNAVAILABLE         0xfd
#define FC_ERROR_INSUFFICIENT_RESOURCES   0xfc
#define FC_ERROR_EXISTING_REGISTRATION    0xfb
#define FC_ERROR_INVALID_TAG              0xfa

/* User Library level Event structure */
typedef struct reg_evt {
   uint32 e_mask;
   uint32 e_gstype;
   uint32 e_pid;
   uint32 e_outsz;
   void (*e_func)(uint32, ...);
   void * e_ctx;
   void * e_out;
} RegEvent;

/* Defines for portid for CT interface */
#define CT_FabricCntlServer ((uint32)0xfffffd)
#define CT_NameServer       ((uint32)0xfffffc)
#define CT_TimeServer       ((uint32)0xfffffb)
#define CT_MgmtServer       ((uint32)0xfffffa)


/* functions from diagnostic specification */
uint32 InitDiagEnv(brdinfo *bi);
uint32 FreeDiagEnv(void);
uint32 SetDiagEnv(uint32 flag);
uint32 SetBrdEnv(uint32 board, uint32 flag);
uint32 GetIOinfo(uint32 board, IOinfo *ioinfo);
uint32 GetLinkInfo(uint32 board, LinkInfo *linkinfo);
uint32 GetCfgParam(uint32 board, CfgParam *cfgparam);
uint32 SetCfgParam(uint32 board, uint32 index, uint32 value);
uint32 GetNodeInfo(uint32 board, NodeInfo *nodeinfo);
int    GetCTInfo(uint32 board, uint32 portid, uchar *inbuf, uint32 incnt,
       uchar *outbuf, uint32 outcnt);
uint32 GetTraceInfo(uint32 board, TraceInfo *traceinfo);
uint32 SetTraceInfo(uint32 board, uint32 flag, uint32 depth);
uint32 IssueMbox(uint32 board, MAILBOX *mb, uint32 insize, uint32 outsize);
uint32 ReadMem(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 WriteMem(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 ReadFlash(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 WriteFlash(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 ReadCtlReg(uint32 board, uint32 *buffer, uint32 offset);
uint32 WriteCtlReg(uint32 board, uint32 *buffer, uint32 offset);
uint32 ReadPciCfg(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 WritePciCfg(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 ReadFcodeFlash(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 WriteFcodeFlash(uint32 board, uchar *buffer, uint32 offset, uint32 count);
uint32 SendElsCmd(uint32 board, uint32 opcode, uint32 did);
uint32 SendScsiCmd(uint32 board, void *wwn, void *req, uint32 sz, void *rsp,
       uint32 *rsz, void *sns, uint32 *snssz);
uint32 SendScsiRead(uint32 board, void *PortWWN, uint64 l, uint32 s,
       void *rsp, uint32 *rspCount, void *sns, uint32 *snsCount);
uint32 SendScsiWrite(uint32 board, void *PortWWN, uint64 l, uint32 s,
       void *rsp, uint32 *rspCount, void *sns, uint32 *snsCount);
uint32 SendFcpCmd(uint32 board, void *wwn, void *req, uint32 sz, void *data,
       uint32 *datasz, void *fcpRsp, uint32 *fcpRspsz);
void * RegisterForCTEvents(uint32 board, ushort type, void (*func)(uint32, ...), void *ctx, uint32 *pstat);
uint32 unRegisterForCTEvent(uint32 board, void *eventid);
uint32 RegisterForEvent(uint32 board, uint32 mask, void *type, uint32 outsz, void (*func)(uint32, ...), void *ctx);
uint32 unRegisterForEvent(uint32 board, uint32 eventid);

#if   defined(_KERNEL) || defined(__KERNEL__)
struct dfc_info {
   brdinfo              fc_ba;
   char                 * fc_iomap_io;  /* starting address for registers */
   char                 * fc_iomap_mem; /* starting address for SLIM */
   uchar                * fc_hmap;      /* handle for mapping memory */
    uint32               fc_refcnt;
    uint32               fc_flag;
};

/* Define for fc_flag */
#define DFC_STOP_IOCTL   1  /* Stop processing dfc ioctls */
#define DFC_MBOX_ACTIVE  2  /* mailbox is active thru dfc */

#endif

#endif  /* _H_FCDIAG */
