/*
 * hangcheck-thread.c
 *
 * Test driver for a little io daemon in kernelspace
 *
 * Copyright (C) 2002 Oracle Corporation.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have recieved a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 021110-1307, USA.
 */


#include <linux/module.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/reboot.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <asm/uaccess.h>

#include "hangcheck-thread.h"

#define DEFAULT_FENCE_MARGIN 60	/* Default is 1 minute (60 seconds */
#define SHUTDOWN_SIGS (sigmask(SIGKILL) | sigmask(SIGHUP) | \
                       sigmask(SIGINT) | sigmask(SIGQUIT))

static int hangcheck_margin = DEFAULT_FENCE_MARGIN;

MODULE_PARM(hangcheck_margin,"i");
MODULE_PARM_DESC(hangcheck_margin, "If the hangcheck thread has not been run in over hangcheck_margin seconds, the machine will reboot.");
MODULE_LICENSE("GPL");


/* Task of hangcheck thread */
static struct task_struct *hangcheck_task = NULL;
/* Wait queue for the hangcheck thread */
static wait_queue_head_t hangcheck_task_wait;
/* Last time scheduled */
static unsigned long hangcheck_last_jiffies;
/* Time to die */
static int hangcheck_die = 0;
/*
 * Lock around hangcheck_last_jiffies and hangcheck_die.  Yes, lock_kernel()
 * is probably as good, but there is no reason to hold up other CPUs.
 * This process is going to be scheduled enough with a non-trigger case
 * to make it possibly worthwhile.
 */
static spinlock_t hangcheck_lock = SPIN_LOCK_UNLOCKED;


int hangcheck_thread(void *thrway)
{
	int done = 0;
	unsigned long last_jiffies, cur_jiffies;
        sigset_t tmpsig;

        lock_kernel();
	sprintf(current->comm, "hangcheck");
        exit_files(current);
        current->files = init_task.files;
        atomic_inc(&current->files->count);
	daemonize();
        unlock_kernel();

	/* Block all signals except KILL, STOP, HUP, and INT */
	spin_lock_irq(&current->sigmask_lock);
	tmpsig = current->blocked;
	siginitsetinv(&current->blocked, SHUTDOWN_SIGS);
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	/* Note that we are running */
	spin_lock(&hangcheck_lock);
	hangcheck_task = current;
	spin_unlock(&hangcheck_lock);
	wake_up(&hangcheck_task_wait);

	printk(KERN_CRIT "hangcheck thread started successfully.\n");

	while (1) {
		spin_lock(&hangcheck_lock);
		done = hangcheck_die;
		last_jiffies = hangcheck_last_jiffies;
		cur_jiffies = jiffies;
		hangcheck_last_jiffies = cur_jiffies;
		spin_unlock(&hangcheck_lock);

		if (done)
			break;

                /* Hmm, does this handle wraparound? */
		if ((last_jiffies+(hangcheck_margin*HZ)) < cur_jiffies) {
			printk(KERN_CRIT "hangcheck: Triggering restart.\n");
			machine_restart(NULL);
		}

		schedule();
	}

	printk(KERN_CRIT "hangcheck thread exited.\n");

	spin_lock(&hangcheck_lock);
	hangcheck_task = NULL;
	spin_unlock(&hangcheck_lock);
	wake_up(&hangcheck_task_wait);

	return 0;
}  /* hangcheck_thread() */


static int __init hangcheck_init(void)
{
	int rc;

	hangcheck_last_jiffies = jiffies;
	init_waitqueue_head(&hangcheck_task_wait);

        version_hash_print();
	printk("Starting hangcheck thread (margin is %d seconds)...\n",
	       hangcheck_margin);
	rc = kernel_thread(hangcheck_thread, NULL, 0);
	if (rc < 0) {
		printk("Unable to start hangcheck thread.\n");
		return rc;
	} 

	spin_lock(&hangcheck_lock);
	while (!hangcheck_task) {
		spin_unlock(&hangcheck_lock);
		sleep_on(&hangcheck_task_wait);
		spin_lock(&hangcheck_lock);
	}
	spin_unlock(&hangcheck_lock);

	return 0;
}  /* hangcheck_init() */


static void __exit hangcheck_exit(void)
{
	spin_lock(&hangcheck_lock);
	hangcheck_die = 1;
	spin_unlock(&hangcheck_lock);

	spin_lock(&hangcheck_lock);
	while (hangcheck_task) {
		spin_unlock(&hangcheck_lock);
		sleep_on(&hangcheck_task_wait);
		spin_lock(&hangcheck_lock);
	}
	spin_unlock(&hangcheck_lock);
}  /* hangcheck_exit() */

module_init(hangcheck_init);
module_exit(hangcheck_exit);
