/*
 *	linux/mm/mprotect.c
 *
 *  (C) Copyright 1994 Linus Torvalds
 */
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/shm.h>
#include <linux/mman.h>
#include <linux/highmem.h>
#include <linux/grsecurity.h>

#ifdef CONFIG_GRKERNSEC_PAX_MPROTECT
#include <linux/elf.h>
#include <linux/fs.h>
#endif

#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <asm/pgtable.h>

/* RSBAC */
#ifdef CONFIG_RSBAC
#include <rsbac/adf.h>
#endif

static inline void change_pte_range(pmd_t * pmd, unsigned long address,
	unsigned long size, pgprot_t newprot)
{
	pte_t *pte, *mapping;
	unsigned long end;

	if (pmd_none(*pmd))
		return;
	if (pmd_bad(*pmd)) {
		pmd_ERROR(*pmd);
		pmd_clear(pmd);
		return;
	}
	mapping = pte = pte_offset_map(pmd, address);
	address &= ~PMD_MASK;
	end = address + size;
	if (end > PMD_SIZE)
		end = PMD_SIZE;
	do {
		if (pte_present(*pte)) {
			pte_t entry;

			/* Avoid an SMP race with hardware updated dirty/clean
			 * bits by wiping the pte and then setting the new pte
			 * into place.
			 */
			entry = ptep_get_and_clear(pte);
			set_pte(pte, pte_modify(entry, newprot));
		}
		address += PAGE_SIZE;
		pte++;
	} while (address && (address < end));
	pte_unmap(mapping);
}

static inline void change_pmd_range(pgd_t * pgd, unsigned long address,
	unsigned long size, pgprot_t newprot)
{
	pmd_t * pmd;
	unsigned long end;

	if (pgd_none(*pgd))
		return;
	if (pgd_bad(*pgd)) {
		pgd_ERROR(*pgd);
		pgd_clear(pgd);
		return;
	}
	pmd = pmd_offset(pgd, address);
	address &= ~PGDIR_MASK;
	end = address + size;
	if (end > PGDIR_SIZE)
		end = PGDIR_SIZE;
	do {
		change_pte_range(pmd, address, end - address, newprot);
		address = (address + PMD_SIZE) & PMD_MASK;
		pmd++;
	} while (address && (address < end));
}

static void change_protection(unsigned long start, unsigned long end, pgprot_t newprot)
{
	pgd_t *dir;
	unsigned long beg = start;

	dir = pgd_offset(current->mm, start);
	flush_cache_range(current->mm, beg, end);
	if (start >= end)
		BUG();
	spin_lock(&current->mm->page_table_lock);
	do {
		change_pmd_range(dir, start, end - start, newprot);
		start = (start + PGDIR_SIZE) & PGDIR_MASK;
		dir++;
	} while (start && (start < end));
	spin_unlock(&current->mm->page_table_lock);
	flush_tlb_range(current->mm, beg, end);
	return;
}

static inline int mprotect_fixup_all(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	int newflags, pgprot_t prot)
{
	struct vm_area_struct * prev = *pprev;
	struct mm_struct * mm = vma->vm_mm;

	if (prev && prev->vm_end == vma->vm_start && can_vma_merge(prev, newflags) &&
	    !vma->vm_file && !(vma->vm_flags & VM_SHARED)) {
		spin_lock(&mm->page_table_lock);
		prev->vm_end = vma->vm_end;
		__vma_unlink(mm, vma, prev);
		spin_unlock(&mm->page_table_lock);

		kmem_cache_free(vm_area_cachep, vma);
		mm->map_count--;

		return 0;
	}

	spin_lock(&mm->page_table_lock);
	vma->vm_flags = newflags;
	vma->vm_page_prot = prot;
	spin_unlock(&mm->page_table_lock);

	*pprev = vma;

	return 0;
}

static inline int mprotect_fixup_start(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long end,
	int newflags, pgprot_t prot)
{
	struct vm_area_struct * n, * prev = *pprev;

	*pprev = vma;

	if (prev && prev->vm_end == vma->vm_start && can_vma_merge(prev, newflags) &&
	    !vma->vm_file && !(vma->vm_flags & VM_SHARED)) {
		spin_lock(&vma->vm_mm->page_table_lock);
		prev->vm_end = end;
		vma->vm_start = end;
		spin_unlock(&vma->vm_mm->page_table_lock);

		return 0;
	}
	n = kmem_cache_alloc(vm_area_cachep, SLAB_KERNEL);
	if (!n)
		return -ENOMEM;
	*n = *vma;
	n->vm_end = end;
	n->vm_flags = newflags;
	n->vm_raend = 0;
	n->vm_page_prot = prot;
	if (n->vm_file)
		get_file(n->vm_file);
	if (n->vm_ops && n->vm_ops->open)
		n->vm_ops->open(n);
	vma->vm_pgoff += (end - vma->vm_start) >> PAGE_SHIFT;
	lock_vma_mappings(vma);
	spin_lock(&vma->vm_mm->page_table_lock);
	vma->vm_start = end;
	__insert_vm_struct(current->mm, n);
	spin_unlock(&vma->vm_mm->page_table_lock);
	unlock_vma_mappings(vma);

	return 0;
}

static inline int mprotect_fixup_end(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start,
	int newflags, pgprot_t prot)
{
	struct vm_area_struct * n;

	n = kmem_cache_alloc(vm_area_cachep, GFP_KERNEL);
	if (!n)
		return -ENOMEM;
	*n = *vma;
	n->vm_start = start;
	n->vm_pgoff += (n->vm_start - vma->vm_start) >> PAGE_SHIFT;
	n->vm_flags = newflags;
	n->vm_raend = 0;
	n->vm_page_prot = prot;
	if (n->vm_file)
		get_file(n->vm_file);
	if (n->vm_ops && n->vm_ops->open)
		n->vm_ops->open(n);
	lock_vma_mappings(vma);
	spin_lock(&vma->vm_mm->page_table_lock);
	vma->vm_end = start;
	__insert_vm_struct(current->mm, n);
	spin_unlock(&vma->vm_mm->page_table_lock);
	unlock_vma_mappings(vma);

	*pprev = n;

	return 0;
}

static inline int mprotect_fixup_middle(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start, unsigned long end,
	int newflags, pgprot_t prot)
{
	struct vm_area_struct * left, * right;

	left = kmem_cache_alloc(vm_area_cachep, SLAB_KERNEL);
	if (!left)
		return -ENOMEM;
	right = kmem_cache_alloc(vm_area_cachep, SLAB_KERNEL);
	if (!right) {
		kmem_cache_free(vm_area_cachep, left);
		return -ENOMEM;
	}
	*left = *vma;
	*right = *vma;
	left->vm_end = start;
	right->vm_start = end;
	right->vm_pgoff += (right->vm_start - left->vm_start) >> PAGE_SHIFT;
	left->vm_raend = 0;
	right->vm_raend = 0;
	if (vma->vm_file)
		atomic_add(2,&vma->vm_file->f_count);
	if (vma->vm_ops && vma->vm_ops->open) {
		vma->vm_ops->open(left);
		vma->vm_ops->open(right);
	}
	vma->vm_pgoff += (start - vma->vm_start) >> PAGE_SHIFT;
	vma->vm_raend = 0;
	vma->vm_page_prot = prot;
	lock_vma_mappings(vma);
	spin_lock(&vma->vm_mm->page_table_lock);
	vma->vm_start = start;
	vma->vm_end = end;
	vma->vm_flags = newflags;
	__insert_vm_struct(current->mm, left);
	__insert_vm_struct(current->mm, right);
	spin_unlock(&vma->vm_mm->page_table_lock);
	unlock_vma_mappings(vma);

	*pprev = right;

	return 0;
}

#if defined(CONFIG_GRKERNSEC_PAX_SEGMEXEC) || defined(CONFIG_GRKERNSEC_PAX_RANDEXEC)
static int __mprotect_fixup(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start, unsigned long end, unsigned int newflags);

static int mprotect_fixup(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start, unsigned long end, unsigned int newflags)
{
	if (vma->vm_flags & VM_MIRROR) {
		struct vm_area_struct * vma_m, * prev_m;
		unsigned long start_m, end_m;
		int error;

		start_m = vma->vm_start + (unsigned long)vma->vm_private_data;
		vma_m = find_vma_prev(vma->vm_mm, start_m, &prev_m);
		if (vma_m && vma_m->vm_start == start_m && (vma_m->vm_flags & VM_MIRROR)) {
			start_m = start + (unsigned long)vma->vm_private_data;
			end_m = end + (unsigned long)vma->vm_private_data;
			if ((current->flags & PF_PAX_SEGMEXEC) && !(newflags & VM_EXEC))
				error = __mprotect_fixup(vma_m, &prev_m, start_m, end_m, vma_m->vm_flags & ~(PROT_READ | PROT_WRITE | PROT_EXEC));
			else
				error = __mprotect_fixup(vma_m, &prev_m, start_m, end_m, newflags);
			if (error)
				return error;
		} else {
			printk("PAX: VMMIRROR: mprotect bug in %s, %08lx\n", current->comm, vma->vm_start);
			return -ENOMEM;
		}
	}

	return __mprotect_fixup(vma, pprev, start, end, newflags);
}

static int __mprotect_fixup(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start, unsigned long end, unsigned int newflags)
{
	pgprot_t newprot;
	int error;
#else
static int mprotect_fixup(struct vm_area_struct * vma, struct vm_area_struct ** pprev,
	unsigned long start, unsigned long end, unsigned int newflags)
{
	pgprot_t newprot;
	int error;

	if (newflags == vma->vm_flags) {
		*pprev = vma;
		return 0;
	}
#endif

#ifdef CONFIG_GRKERNSEC_PAX_PAGEEXEC
	if (!(current->flags & PF_PAX_PAGEEXEC) && (newflags & (VM_READ|VM_WRITE)))
		newprot = protection_map[(newflags | VM_EXEC) & 0xf];
	else
#endif
	newprot = protection_map[newflags & 0xf];
	if (start == vma->vm_start) {
		if (end == vma->vm_end)
			error = mprotect_fixup_all(vma, pprev, newflags, newprot);
		else
			error = mprotect_fixup_start(vma, pprev, end, newflags, newprot);
	} else if (end == vma->vm_end)
		error = mprotect_fixup_end(vma, pprev, start, newflags, newprot);
	else
		error = mprotect_fixup_middle(vma, pprev, start, end, newflags, newprot);

	if (error)
		return error;

	change_protection(start, end, newprot);
	return 0;
}

#ifdef CONFIG_GRKERNSEC_PAX_MPROTECT
/* PaX: non-PIC ELF libraries need relocations on their executable segments
 * therefore we'll grant them VM_MAYWRITE once during their life.
 *
 * The checks favor ld-linux.so behaviour which operates on a per ELF segment
 * basis because we want to allow the common case and not the special ones.
 */
static inline void pax_handle_maywrite(struct vm_area_struct * vma, unsigned long start)
{
	struct elfhdr elf_h;
	struct elf_phdr elf_p, p_dyn;
	elf_dyn dyn;
	unsigned long i, j = 65536UL / sizeof(struct elf_phdr);

#ifndef CONFIG_GRKERNSEC_PAX_NOELFRELOCS
	if ((vma->vm_start != start) ||
	    !vma->vm_file ||
	    !(vma->vm_flags & VM_MAYEXEC) ||
	    (vma->vm_flags & VM_MAYNOTWRITE))
#endif

	return;

	if (0 > kernel_read(vma->vm_file, 0UL, (char*)&elf_h, sizeof(elf_h)) ||
	    memcmp(elf_h.e_ident, ELFMAG, SELFMAG) ||

#ifdef CONFIG_GRKERNSEC_PAX_ETEXECRELOCS
	    (elf_h.e_type != ET_DYN && elf_h.e_type != ET_EXEC) ||
#else
	    elf_h.e_type != ET_DYN ||
#endif

	    !elf_check_arch(&elf_h) ||
	    elf_h.e_phentsize != sizeof(struct elf_phdr) ||
	    elf_h.e_phnum > j)
		return;

	for (i = 0UL; i < elf_h.e_phnum; i++) {
		if (0 > kernel_read(vma->vm_file, elf_h.e_phoff + i*sizeof(elf_p), (char*)&elf_p, sizeof(elf_p)))
			return;
		if (elf_p.p_type == PT_DYNAMIC) {
			p_dyn = elf_p;
			j = i;
		}
	}
	if (elf_h.e_phnum <= j)
		return;

	i = 0UL;
	do {
		if (0 > kernel_read(vma->vm_file, p_dyn.p_offset + i*sizeof(dyn), (char*)&dyn, sizeof(dyn)))
			return;
		if (dyn.d_tag == DT_TEXTREL || (dyn.d_tag == DT_FLAGS && (dyn.d_un.d_val & DF_TEXTREL))) {
			vma->vm_flags |= VM_MAYWRITE | VM_MAYNOTWRITE;
			gr_log_textrel(vma);
			return;
		}
		i++;
	} while (dyn.d_tag != DT_NULL);
	return;
}
#endif

long do_mprotect(struct mm_struct *mm, unsigned long start, size_t len, 
		 unsigned long prot)
{
	unsigned long nstart, end, tmp;
	struct vm_area_struct * vma, * next, * prev;
	int error = -EINVAL;

        /* RSBAC */
        #ifdef CONFIG_RSBAC
        enum  rsbac_target_t          rsbac_target = T_NONE;
        union rsbac_target_id_t       rsbac_target_id;
        union rsbac_attribute_value_t rsbac_attribute_value;
        int need_notify = FALSE;
        #endif

	if (start & ~PAGE_MASK)
		return -EINVAL;
	len = PAGE_ALIGN(len);
	end = start + len;
	if (end < start)
		return -ENOMEM;

#ifdef CONFIG_GRKERNSEC_PAX_SEGMEXEC
	if (current->flags & PF_PAX_SEGMEXEC) {
		if (end > SEGMEXEC_TASK_SIZE)
			return -EINVAL;
	} else
#endif

	if (end > TASK_SIZE)
		return -EINVAL;

	if (prot & ~(PROT_READ | PROT_WRITE | PROT_EXEC | PROT_SEM))
		return -EINVAL;
	if (end == start)
		return 0;

	down_write(&mm->mmap_sem);

	vma = find_vma_prev(mm, start, &prev);
	error = -ENOMEM;
	if (!vma || vma->vm_start > start)
		goto out;

	if (!gr_acl_handle_mprotect(vma->vm_file, prot)) {
		error = -EACCES;
		goto out;
	}

#ifdef CONFIG_GRKERNSEC_PAX_MPROTECT
	if ((current->flags & PF_PAX_MPROTECT) && (prot & PROT_WRITE))
		pax_handle_maywrite(vma, start);
#endif

	for (nstart = start ; ; ) {
		unsigned int newflags;
		int last = 0;

		/* Here we know that  vma->vm_start <= nstart < vma->vm_end. */

		newflags = prot | (vma->vm_flags & ~(PROT_READ | PROT_WRITE | PROT_EXEC));
		if ((newflags & ~(newflags >> 4)) & 0xf) {
			error = -EACCES;
			goto out;
		}

		if (vma->vm_ops && vma->vm_ops->mprotect) {
			error = vma->vm_ops->mprotect(vma, newflags);
			if (error < 0)
				goto out;
		}

#ifdef CONFIG_GRKERNSEC_PAX_MPROTECT
		/* PaX: disallow write access after relocs are done, hopefully noone else needs it... */
		if ((current->flags & PF_PAX_MPROTECT) && (prot & PROT_WRITE) && (vma->vm_flags & VM_MAYNOTWRITE))
			newflags &= ~VM_MAYWRITE;
#endif

#ifdef CONFIG_RSBAC
		if(   (prot & PROT_EXEC)
		   && !(vma->vm_flags & PROT_EXEC)
		  )
		{
#ifdef CONFIG_RSBAC_DEBUG
			if(rsbac_debug_aef)
				printk(KERN_DEBUG "sys_mprotect: calling ADF\n");
#endif
			if(vma->vm_file) {
		                rsbac_target = T_FILE;
				rsbac_target_id.file.device = vma->vm_file->f_dentry->d_inode->i_dev;
				rsbac_target_id.file.inode = vma->vm_file->f_dentry->d_inode->i_ino;
				rsbac_target_id.file.dentry_p = vma->vm_file->f_dentry;
			}
			else {
				rsbac_target = T_NONE;
				rsbac_target_id.dummy = 0;
			}
			rsbac_attribute_value.prot_bits = prot;
			if(!rsbac_adf_request(R_MAP_EXEC,
						  current->pid,
						  rsbac_target,
						  rsbac_target_id,
						  A_prot_bits,
						  rsbac_attribute_value)) 
			{
#ifdef CONFIG_RSBAC_DEBUG
				if(rsbac_debug_aef)
					printk(KERN_DEBUG "sys_mprotect: request NOT_GRANTED\n");
#endif
				error = -EPERM;
				break;
			}
			else
			  need_notify = TRUE;
		}
#endif

		if (vma->vm_end > end) {
			error = mprotect_fixup(vma, &prev, nstart, end, newflags);
			goto out;
		}
		if (vma->vm_end == end)
			last = 1;

		tmp = vma->vm_end;
		next = vma->vm_next;
		error = mprotect_fixup(vma, &prev, nstart, tmp, newflags);
		if (error)
			goto out;
		if (last)
			break;
		nstart = tmp;
		vma = next;
		if (!vma || vma->vm_start != nstart) {
			error = -ENOMEM;
			goto out;
		}
	}
	if (next && prev->vm_end == next->vm_start && can_vma_merge(next, prev->vm_flags) &&
	    !prev->vm_file && !(prev->vm_flags & VM_SHARED)) {
		spin_lock(&prev->vm_mm->page_table_lock);
		prev->vm_end = next->vm_end;
		__vma_unlink(prev->vm_mm, next, prev);
		spin_unlock(&prev->vm_mm->page_table_lock);

		kmem_cache_free(vm_area_cachep, next);
		prev->vm_mm->map_count--;
	}
out:
	up_write(&mm->mmap_sem);

        /* RSBAC: notify ADF of mapped segment */
        #ifdef CONFIG_RSBAC
        if(need_notify && !error)
          {
            union rsbac_target_id_t rsbac_new_target_id;

#ifdef CONFIG_RSBAC_DEBUG
            if (rsbac_debug_aef)
              printk(KERN_DEBUG "sys_mprotect(): calling ADF_set_attr\n");
#endif
            rsbac_new_target_id.dummy = 0;
            if (rsbac_adf_set_attr(R_MAP_EXEC,
                                   current->pid,
                                   rsbac_target,
                                   rsbac_target_id,
                                   T_NONE,
                                   rsbac_new_target_id,
                                   A_none,
                                   rsbac_attribute_value))
              {
                printk(KERN_WARNING
                       "sys_mprotect: rsbac_adf_set_attr() returned error\n");
              }
          }
        #endif

	return error;
}

asmlinkage long sys_mprotect(unsigned long start, size_t len, unsigned long prot)
{
	return(do_mprotect(current->mm, start, len, prot));
}
