#include <linux/kernel.h>
#include <linux/module.h>

#include "shfs.h"
#include "shfs_proc.h"
#include "shfs_proto.h"

/* identify remote OS type, what ops we shoul use, etc. */

char *linux_osf1_identify = "\
if n=`LC_ALL=POSIX uname -s` &&\
   test \"$n\" = Linux -o \"$n\" = OSF1 &&\
   head -c 0 >/dev/null 2>&1; then\
  i=0; while [ $i -lt 10000 ]; do\
    if mkdir /tmp/._shfs_.$i 2>/dev/null; then\
      s_TMP=/tmp/._shfs_.$i/;\
      break;\
    fi;\
    i=`expr $i + 1`;\
  done;\
  if [ $i -lt 10000 ]; then\
    rmdir $s_TMP; 2>/dev/null\
    echo \"$n\";\
  else\
    echo \"unknown\";\
  fi \
else\
  echo \"unknown\"; \
fi 2>/dev/null\n";

char *sunos_identify = "\
if n=`LC_ALL=POSIX uname -s` &&\
   test \"$n\" = SunOS; then\
  i=0; while [ $i -lt 10000 ]; do\
    if mkdir /tmp/._shfs_.$i 2>/dev/null; then\
      s_TMP=/tmp/._shfs_.$i/;\
      break;\
    fi;\
    i=`expr $i + 1`;\
  done;\
  if [ $i -lt 10000 ]; then\
    rmdir $s_TMP; 2>/dev/null\
    echo \"$n\";\
  else\
    echo \"unknown\";\
  fi \
else\
  echo \"unknown\"; \
fi 2>/dev/null\n";

char *generic_identify = "\
i=0; while [ $i -lt 10000 ]; do\
  if mkdir /tmp/._shfs_.$i 2>/dev/null; then\
    s_TMP=/tmp/._shfs_.$i/;\
    break;\
  fi;\
  i=`expr $i + 1`; \
done; \
if [ $i -lt 10000 ]; then\
  rmdir $s_TMP; 2>/dev/null\
  echo \"generic rw\"; \
else\
  echo \"unknown\";\
fi\n";

char **shfs_system_identify[] = {
	&linux_osf1_identify,		/* Linux or OSF1 */
	&sunos_identify,		/* SunOS */
	&generic_identify,		/* generic */
	NULL,				/* not writable /tmp/ */
};

int
do_identify(struct shfs_sb_info *info, int sys)
{
	char **ops = shfs_system_identify[sys];

	if (!*ops)
		return 0;
	return shfs_pipe_printf(info, *ops);
}

/*
 Protocol NOTES:

 * All operations should redirect stderr to /dev/null.
 * You *MUST* quote strings with single quotes (expansion ' -> '\'' 
   is done in filenames).
 * Read & write must work correctly when count == 0 (return zero bytes)
*/

/* Generic file operations */

char *generic_init = "s_init () { \
s_ROOT=\"$1\"; \
s_PRELIM=\"### $2\"; s_COMPLETE=\"### $3\"; s_NOP=\"### $4\"; s_NOTEMPTY=\"### $5\"; \
s_CONTINUE=\"### $6\"; s_TRANSIENT=\"### $7\"; shift 7; \
s_ERROR=\"### $1\"; s_EPERM=\"### $2\"; s_ENOSPC=\"### $3\"; s_ENOENT=\"### $4\"; \
i=0; while [ $i -lt 10000 ]; do\
  if mkdir /tmp/._shfs_.$i 2>/dev/null; then\
    s_TMP=/tmp/._shfs_.$i/;\
    break;\
  fi;\
  i=`expr $i + 1`; \
done; \
if [ $i -lt 10000 ]; then\
  echo $s_COMPLETE; \
else\
  echo $s_EPERM; \
fi }\n";

char *generic_finish = "s_finish () { \
if test \"$s_TMP\"; then\
	rm -rf $s_TMP; \
fi; \
echo $s_COMPLETE; }\n";

char *generic_lsdir = "s_lsdir () { \
if TZ=GMT LC_ALL=POSIX ls -lani$2 \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE;\
else\
  if test -d \"$s_ROOT$1\"; then\
    if ls \"$s_ROOT$1\" >/dev/null 2>&1; then\
      echo $s_COMPLETE;\
    else\
      echo $s_EPERM;\
    fi\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

/* Some dd implementations do not like count=0 */

char *generic_open = "s_open () { \
ok=0; \
if test x$2 = xR -o x$2 = xRW; then\
  if test -r \"$s_ROOT$1\"; then\
    ok=1;\
  fi \
fi; \
if test x$2 = xW -o x$2 = xRW; then\
  if test -w \"$s_ROOT$1\"; then\
    ok=1;\
  else\
    ok=0;\
  fi \
fi; \
if test $ok = 1; then\
  if test -s \"$s_ROOT$1\"; then\
    echo $s_COMPLETE;\
  else\
    if test `dd if=\"$s_ROOT$1\" bs=1 count=1 2>/dev/null|wc -c` -eq 0; then\
      echo $s_COMPLETE;\
    else\
      if test -s \"$s_ROOT$1\"; then\
        echo $s_COMPLETE;\
      else\
        echo $s_NOTEMPTY;\
      fi\
    fi\
  fi; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_read = "s_read () { \
if test \"$3\" = 0; then\
  if test -r \"$s_ROOT$1\"; then\
    echo $s_PRELIM; echo $s_COMPLETE;\
  else\
    echo $s_EPERM; \
  fi \
else\
  if x=`LC_ALL=POSIX dd if=\"$s_ROOT$1\" bs=$4 skip=$5 count=1 2>&1 >/dev/null`; then\
    if echo $s_PRELIM; [ `echo $x|cut -c1-3` = 0+0 ]; then\
      echo $s_COMPLETE;\
    elif dd if=\"$s_ROOT$1\" bs=$4 skip=$5 count=$6 conv=sync 2>/dev/null; then\
      echo $s_COMPLETE;\
    else\
      echo $s_ERROR;\
    fi; \
  else\
    if test -f \"$s_ROOT$1\"; then\
      echo $s_EPERM;\
    else\
      echo $s_ENOENT;\
    fi\
  fi \
fi; }\n";

char *generic_slow_read = "s_sread () { \
if test \"$3\" = 0; then\
  if test -r \"$s_ROOT$1\"; then\
    echo $s_PRELIM; echo $s_COMPLETE;\
  else\
    echo $s_EPERM; \
  fi \
else\
  if x=`LC_ALL=POSIX dd if=\"$s_ROOT$1\" bs=$4 skip=$5 count=1 2>&1 >/dev/null`; then\
    if echo $s_PRELIM; [ `echo $x|cut -c1-3` = 0+0 ]; then\
      echo 0;\
      echo $s_COMPLETE;\
    else\
      if touch \"$s_TMP._shfs_$$_$7\" 2>/dev/null &&\
         w=`dd if=\"$s_ROOT$1\" bs=$4 skip=$5 count=$6 2>/dev/null|tee \"$s_TMP._shfs_$$_$7\"|wc -c`; then\
        echo $w;\
        dd if=\"$s_TMP._shfs_$$_$7\" bs=$4 count=$6 2>/dev/null;\
        rm -f \"$s_TMP._shfs_$$_$7\";\
        echo $s_COMPLETE;\
      else\
        echo $s_EPERM;\
      fi\
    fi;\
  else\
    if test -f \"$s_ROOT$1\"; then\
      echo $s_EPERM;\
    else\
      echo $s_ENOENT;\
    fi\
  fi \
fi; }\n";

char *generic_write = "s_write () { \
if test -w \"$s_ROOT$1\"; then\
  echo $s_PRELIM;\
  if test \"$3\" = 0; then\
    echo $s_COMPLETE;\
  else\
   if dd of=\"$s_ROOT$1\" bs=1 seek=$2 count=$3 conv=notrunc 2>/dev/null; then\
      echo $s_COMPLETE;\
    else\
      echo $s_ENOSPC; dd of=/dev/null bs=1 count=$3 2>/dev/null;\
    fi\
  fi \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM; \
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_mkdir = "s_mkdir () { \
if mkdir \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  echo $s_EPERM; \
fi; }\n";

char *generic_rmdir = "s_rmdir () { \
if test -d \"$s_ROOT$1\"; then\
  if rmdir \"$s_ROOT$1\" 2>/dev/null; then\
    echo $s_COMPLETE;\
  else\
    echo $s_EPERM;\
  fi \
else\
  echo $_ENOENT; \
fi; }\n";

char *generic_mv = "s_mv () { \
if mv -f \"$s_ROOT$1\" \"$s_ROOT$2\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_rm = "s_rm () { \
if rm -f \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_creat = "s_creat () { \
if ( >\"$s_ROOT$1\"; chmod $2 \"$s_ROOT$1\" ) 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  echo $s_EPERM; \
fi; }\n";

char *generic_ln = "s_ln () { \
if ln -f \"$s_ROOT$1\" \"$s_ROOT$2\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  echo $s_EPERM; \
fi; }\n";

char *generic_sln = "s_sln () { \
if ln -s -f \"$1\" \"$s_ROOT$2\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  echo $s_EPERM; \
fi; }\n";

char *generic_chmod = "s_chmod () { \
if chmod $2 \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_chown = "s_chown () { \
if chown $2 \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

char *generic_chgrp = "s_chgrp () { \
if chgrp $2 \"$s_ROOT$1\" 2>/dev/null; then\
  echo $s_COMPLETE; \
else\
  if test -f \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

/* some dd implementations do not like count=0 */

char *generic_trunc = "s_trunc () { \
if test \"$3\" = 0; then\
  if ( >\"$s_ROOT$1\" ) 2>/dev/null; then\
    echo $s_COMPLETE; \
  else\
    if test -f \"$s_ROOT$1\"; then\
      echo $s_EPERM;\
    else\
      echo $s_ENOENT;\
    fi\
  fi \
else\
  if dd if=/dev/zero of=\"$s_ROOT$1\" bs=$2 seek=$3 count=0 2>/dev/null; then\
    echo $s_COMPLETE; \
  else\
    if test -f \"$s_ROOT$1\"; then\
      echo $s_EPERM;\
    else\
      echo $s_ENOENT;\
    fi\
  fi \
fi; }\n";

char *generic_settime = "s_settime () { \
if test -e \"$s_ROOT$1\"; then\
  if TZ=UTC touch -$2 -t $3 \"$s_ROOT$1\" 2>/dev/null; then\
    echo $s_COMPLETE;\
  else\
    echo $s_EPERM;\
  fi \
else\
  echo $s_ENOENT; \
fi; }\n";

/* returns "total avail" 1024 blocks */

char *generic_statfs = "s_statfs () { \
if test -z \"$s_ROOT\"; then\
  r=\"/\"; \
else\
  r=\"$s_ROOT\"; \
fi; \
LC_ALL=POSIX df -k -P \"$r\" 2>/dev/null|( xa=0; xb=0; xc=0; while read x a b c z; do \
  xa=$a; xb=$b; xc=$c; \
done; \
echo $xa $xb $xc );\
echo $s_COMPLETE; }\n";
	
/* Linux specific operations */

char *linux_read = "s_read () { \
if x=`LC_ALL=POSIX dd if=\"$s_ROOT$1\" bs=1 skip=$2 count=1 2>&1 >/dev/null`; then\
  if echo $s_PRELIM; test `echo $x|cut -c1-3` = 0+0; then\
    echo $s_COMPLETE;\
  elif dd if=\"$s_ROOT$1\" bs=$4 skip=$5 count=$6 conv=sync 2>/dev/null; then\
    echo $s_COMPLETE;\
  else\
    echo $s_ERROR;\
  fi; \
else\
  if test -e \"$s_ROOT$1\"; then\
    echo $s_EPERM;\
  else\
    echo $s_ENOENT;\
  fi \
fi; }\n";

/* We should have head command (OSF1 has one too) */

char *linux_write = "s_write () { \
if printf '' >\"$s_TMP._shfs_$$_$6\" 2>/dev/null; then\
  echo $s_PRELIM;\
  if head -c $3 >\"$s_TMP._shfs_$$_$6\" 2>/dev/null; then\
    if dd if=\"$s_TMP._shfs_$$_$6\" of=\"$s_ROOT$1\" bs=$4 seek=$5 conv=notrunc 2>/dev/null; then\
      echo $s_COMPLETE;\
    else\
      if test -w \"$s_ROOT$1\"; then\
        echo $s_ENOSPC;\
      else\
        echo $s_EPERM;\
      fi\
    fi;\
  else\
    echo $s_ENOSPC; dd of=/dev/null bs=1 count=$3 2>/dev/null;\
  fi;\
  rm -f \"$s_TMP._shfs_$$_$6\" 2>/dev/null; \
else\
  echo $s_EPERM; \
fi; }\n";

/* SunOS specific operations */

char *sunos_statfs = "s_statfs () { \
if test -z \"$s_ROOT\"; then\
  r=\"/\"; \
else\
  r=\"$s_ROOT\"; \
fi; \
LC_ALL=POSIX df -k \"$r\" 2>/dev/null|( xa=0; xb=0; xc=0; while read x a b c z; do \
  xa=$a; xb=$b; xc=$c; \
done; \
echo $xa $xb $xc );\
echo $s_COMPLETE; }\n";

/* Read only specific operations */

char *ro_slow_read = "s_sread () { \
  echo $s_EPERM; }\n";

char *ro_init = "s_init () { \
s_ROOT=\"$1\"; \
s_PRELIM=\"### $2\"; s_COMPLETE=\"### $3\"; s_NOP=\"### $4\"; s_NOTEMPTY=\"### $5\"; \
s_CONTINUE=\"### $6\"; s_TRANSIENT=\"### $7\"; shift 7; \
s_ERROR=\"### $1\"; s_EPERM=\"### $2\"; s_ENOSPC=\"### $3\"; s_ENOENT=\"### $4\"; \
echo $s_COMPLETE; }\n";

char *ro_finish = "s_finish () { \
echo $s_COMPLETE; }\n";


char **shfs_system_ops[][SHFS_OPS] = {
	{
		&generic_init,			/* Linux or OSF1 */
		&generic_finish,
		&generic_lsdir,
		&generic_open,
		&linux_read,
		&generic_slow_read,
		&linux_write,
		&generic_mkdir,
		&generic_rmdir,
		&generic_mv,
		&generic_rm,
		&generic_creat,
		&generic_ln,
		&generic_sln,
		&generic_chmod,
		&generic_chown,
		&generic_chgrp,
		&generic_trunc,
		&generic_settime,
		&generic_statfs,
	}, {
		&generic_init,			/* SunOS */
		&generic_finish,
		&generic_lsdir,
		&generic_open,
		&generic_read,
		&generic_slow_read,
		&generic_write,
		&generic_mkdir,
		&generic_rmdir,
		&generic_mv,
		&generic_rm,
		&generic_creat,
		&generic_ln,
		&generic_sln,
		&generic_chmod,
		&generic_chown,
		&generic_chgrp,
		&generic_trunc,
		&generic_settime,
		&sunos_statfs,
	}, {
		&generic_init,			/* generic */
		&generic_finish,
		&generic_lsdir,
		&generic_open,
		&generic_read,
		&generic_slow_read,
		&generic_write,
		&generic_mkdir,
		&generic_rmdir,
		&generic_mv,
		&generic_rm,
		&generic_creat,
		&generic_ln,
		&generic_sln,
		&generic_chmod,
		&generic_chown,
		&generic_chgrp,
		&generic_trunc,
		&generic_settime,
		&generic_statfs,
	}, {
		&ro_init,			/* read only */
		&ro_finish,
		&generic_lsdir,
		&generic_open,
		&generic_read,
		&ro_slow_read,
		&generic_write,
		&generic_mkdir,
		&generic_rmdir,
		&generic_mv,
		&generic_rm,
		&generic_creat,
		&generic_ln,
		&generic_sln,
		&generic_chmod,
		&generic_chown,
		&generic_chgrp,
		&generic_trunc,
		&generic_settime,
		&generic_statfs,
	},
};

/* Machine independent operations */

int
do_init(struct shfs_sb_info *info, int sys)
{
	int res = 0, r;
	int i;
	char ***ops = shfs_system_ops[sys];

	for (i = 0; i < SHFS_OPS; i++) {
		if (ops[i]) {
			r = shfs_pipe_printf(info, *ops[i]);
			if (r < 0)
				return r;
			res += r;
		}
	}
	r = shfs_pipe_printf(info, "s_init \"%s\" %d %d %d %d %d %d %d %d %d %d\n",
		info->mnt.root,\
		REP_PRELIM, REP_COMPLETE, REP_NOP, REP_NOTEMPTY,\
		REP_CONTINUE, REP_TRANSIENT,\
		REP_ERROR, REP_EPERM, REP_ENOSPC, REP_ENOENT);
	if (r < 0)
		return r;
	res += r;
	return res;
}

int
do_ping(struct shfs_sb_info *info, unsigned long seq)
{
	return shfs_pipe_printf(info, "\n\necho; echo '### %d'; echo %lu; echo '### %d'\n", REP_PRELIM, seq, REP_NOP);
}

