#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/smp_lock.h>

#include "shfs.h"
#include "shfs_proc.h"

static int shfs_d_revalidate(struct dentry*, int);
static int shfs_d_delete(struct dentry*);

static struct dentry_operations shfs_dentry_operations = {
	d_revalidate:	shfs_d_revalidate,
	d_delete:	shfs_d_delete,
};

static int shfs_readdir(struct file*, void*, filldir_t);
static int shfs_dir_open(struct inode*, struct file*);

struct file_operations shfs_dir_operations = {
	read:		generic_read_dir,
	readdir:	shfs_readdir,
	open:		shfs_dir_open,
};

static struct dentry *shfs_lookup(struct inode*, struct dentry*);
static int shfs_mkdir(struct inode*, struct dentry*, int);
static int shfs_create(struct inode*, struct dentry*, int);
static int shfs_rmdir(struct inode*, struct dentry*);
static int shfs_rename(struct inode*, struct dentry*, struct inode*, struct dentry*);
static int shfs_unlink(struct inode*, struct dentry*);
static int shfs_link(struct dentry*, struct inode*, struct dentry*);
static int shfs_symlink(struct inode*, struct dentry*, const char*);

struct inode_operations shfs_dir_inode_operations = {
	create:		shfs_create,
	lookup:		shfs_lookup,
	link:		shfs_link,
	unlink:		shfs_unlink,
	symlink:	shfs_symlink,
	mkdir:		shfs_mkdir,
	rmdir:		shfs_rmdir,
	rename:		shfs_rename,
	setattr:	shfs_notify_change,
	revalidate:	shfs_revalidate_inode,
};

static int
shfs_dir_open(struct inode *inode, struct file *f)
{
	struct super_block *sb = inode->i_sb;
	struct shfs_sb_info *info = (struct shfs_sb_info*)sb->u.generic_sbp;
	int error = 0;

	DEBUG("\n");
	if (!shfs_lock(info))
		return -EINTR;
	error = shfs_dcache_get_dir(f->f_dentry, NULL);
	shfs_unlock(info);
	if (error < 0)
		VERBOSE("shfs_dcache_get_dir failed! (%d)\n", error);
	return shfs_remove_sigpipe(error);
}

static int
shfs_readdir(struct file *f, void *dirent, filldir_t filldir)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct inode *inode = dentry->d_inode;
	struct shfs_dir_entry *entry;
	int pos, res;
	
	DEBUG(" reading %s, f_pos=%d\n", dentry->d_name.name, (unsigned int)f->f_pos);

	switch ((unsigned int) f->f_pos) {
	case 0:
		DEBUG("f_pos=0\n");
		if ((res = filldir(dirent, ".", 1, 0, inode->i_ino, DT_DIR)) < 0)
			goto out;
		f->f_pos = 1;
	case 1:
		DEBUG("f_pos=1\n");
		if ((res = filldir(dirent, "..", 2, 1, dentry->d_parent->d_inode->i_ino, DT_DIR)) < 0)
			goto out;
		f->f_pos = 2;
	default:
		DEBUG(" \n");
		if (!shfs_lock(info))
			return -EINTR;
		res = shfs_dcache_get_dir(dentry, &entry);
		if (res < 0) {
			shfs_unlock(info);
			VERBOSE(" shfs_dcache_get failed! (%d)\n", res);
			goto out;
		}
		pos = 2;
		while (entry) {
			if ((pos == f->f_pos)) {
				struct qstr qname;
				unsigned long ino;
				qname.name = entry->name;
				qname.len = strlen(qname.name);
				ino = find_inode_number(dentry, &qname);
				if (!ino)
					ino = entry->ino;
				if (filldir(dirent, qname.name, qname.len, f->f_pos, ino, DT_UNKNOWN) >= 0)
					f->f_pos++;
			}
			pos++;
			entry = entry->next;
		}
		shfs_unlock(info);
	}
	return 0;
out:
	return shfs_remove_sigpipe(res);
}

static struct dentry*
shfs_lookup(struct inode *dir, struct dentry *dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct shfs_fattr fattr;
	struct inode *inode;
	int error;
	
	DEBUG(" dname:%s\n", dentry->d_name.name);
	if (!shfs_lock(info))
		return ERR_PTR(-EINTR);
	error = shfs_get_attr(dentry, &fattr);
	shfs_unlock(info);
	if (error < 0) {
		DEBUG(" file not found!\n");
		inode = NULL;
		dentry->d_op = &shfs_dentry_operations;
		d_add(dentry, inode);
		error = shfs_remove_sigpipe(error);
		if (error == -EINTR)
			return ERR_PTR(error);
		return NULL; 
	}

	inode = shfs_iget(dir->i_sb, &fattr);
	if (inode) {
		dentry->d_op = &shfs_dentry_operations;
		d_add(dentry, inode);
	}
	
	return NULL; 
}

static int
shfs_instantiate(struct dentry *dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct shfs_fattr fattr;
	struct inode *inode;
	int error;
	
	DEBUG("\n");
	if (!shfs_lock(info))
		return -EINTR;
	error = shfs_get_attr(dentry, &fattr);
	shfs_unlock(info);
	if (error < 0) {
		VERBOSE(" shfs_get_attr failed! (%d)\n", error);
		return shfs_remove_sigpipe(error);
	}
	
	inode = shfs_iget(dentry->d_sb, &fattr);
	if (!inode)
		return -EACCES;
		
	d_instantiate(dentry, inode);
	return 0;	
}

static int
shfs_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char buf[SHFS_PATH_MAX];
	int res;
	
	DEBUG("\n");

	if (info->mnt.readonly)
		return -EROFS;
	
	if (shfs_get_name(dentry, buf) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}

	if ((res = shfs_proc_mkdir(info, buf)) < 0) {
		VERBOSE("mkdir failed!\n");
		return shfs_remove_sigpipe(res);
	}
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	return shfs_instantiate(dentry);
}

static int
shfs_create(struct inode* dir, struct dentry *dentry, int mode)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char buf[SHFS_PATH_MAX];
	int res;
	
	DEBUG("\n");
	if (info->mnt.readonly)
		return -EROFS;
	
	if (shfs_get_name(dentry, buf) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}

	if ((res = shfs_proc_create(info, buf, mode)) < 0) {
		VERBOSE(" Create failed!\n");
		return shfs_remove_sigpipe(res);
	}
	
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	return shfs_instantiate(dentry);
}

static int
shfs_rmdir(struct inode* dir, struct dentry *dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char buf[SHFS_PATH_MAX];

	DEBUG(" \n");
	if (info->mnt.readonly)
		return -EROFS;
	
	if (!d_unhashed(dentry))
		return -EBUSY;
	
	if (shfs_get_name(dentry, buf) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}

	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	return shfs_remove_sigpipe(shfs_proc_rmdir(info, buf));
}

static int
shfs_rename(struct inode *old_dir, struct dentry *old_dentry, struct inode *new_dir, struct dentry *new_dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)old_dentry->d_sb->u.generic_sbp;
	char buf1[SHFS_PATH_MAX], buf2[SHFS_PATH_MAX];
	int res;
	
	DEBUG(" \n");

	if (info->mnt.readonly)
		return -EROFS;

	if (shfs_get_name(old_dentry, buf1) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}
	if (shfs_get_name(new_dentry, buf2) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}
	
	if (new_dentry->d_inode)
		d_delete(new_dentry);
	
	if ((res = shfs_proc_rename(info, buf1, buf2)) < 0)
		return shfs_remove_sigpipe(res);
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(old_dir);
	shfs_dcache_invalidate_dir(new_dir);
	shfs_unlock(info);
	
	return shfs_remove_sigpipe(res);
}

static int
shfs_unlink(struct inode *dir, struct dentry *dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char buf[SHFS_PATH_MAX];

	DEBUG(" \n");
	if (info->mnt.readonly)
		return -EROFS;
	
	if (shfs_get_name(dentry, buf) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}

	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	return shfs_remove_sigpipe(shfs_proc_unlink(info, buf));
}

static int
shfs_link(struct dentry *old_dentry, struct inode *dir, struct dentry *new_dentry)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)old_dentry->d_sb->u.generic_sbp;
	char old[SHFS_PATH_MAX], new[SHFS_PATH_MAX];
	int res;

	DEBUG(" \n");	
	if (info->mnt.readonly)
		return -EROFS;

	if (shfs_get_name(old_dentry, old) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}
	if (shfs_get_name(new_dentry, new) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}
	
	if ((res = shfs_proc_link(info, old, new)) < 0)
		return shfs_remove_sigpipe(res);
	
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	
	return shfs_instantiate(new_dentry);
}

static int
shfs_symlink(struct inode *dir, struct dentry *dentry, const char *oldname)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char new[SHFS_PATH_MAX];
	int res;

	DEBUG(" \n");
	if (info->mnt.readonly)
		return -EROFS;

	if (shfs_get_name(dentry, new) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}
	
	if ((res = shfs_proc_symlink(info, oldname, new)) < 0)
		return shfs_remove_sigpipe(res);

	DEBUG(" \n");	
	if (!shfs_lock(info))
		return -EINTR;
	shfs_dcache_invalidate_dir(dir);
	shfs_unlock(info);
	return shfs_instantiate(dentry);
}

static int
shfs_d_revalidate(struct dentry *dentry, int flags)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	int error = 0;

	DEBUG("%s\n", dentry->d_name.name);
	if (!shfs_lock(info))
		return -EINTR;
	error = shfs_dcache_revalidate_entry(dentry);
	shfs_unlock(info);
	return shfs_remove_sigpipe(error);
}

static int
shfs_d_delete(struct dentry *dentry)
{
	if (dentry->d_inode) {
		if (is_bad_inode(dentry->d_inode)) {
			DEBUG("bad inode, unhashing\n");
			return 1;
		}
	} else {
		/* ??? */
	}
	return 0;
}

