/*
 *  linux/arch/i386/mm/fault.c
 *
 *  Copyright (C) 1995  Linus Torvalds
 */

#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/mm.h>
#include <linux/smp.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/tty.h>
#include <linux/vt_kern.h>		/* For unblank_screen() */
#include <linux/unistd.h>
#include <linux/compiler.h>

#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <asm/hardirq.h>

extern void die(const char *,struct pt_regs *,long);

spinlock_t oops_lock = SPIN_LOCK_UNLOCKED;

/*
 * Ugly, ugly, but the goto's result in better assembly..
 */
int __verify_write(const void * addr, unsigned long size)
{
	struct vm_area_struct * vma;
	unsigned long start = (unsigned long) addr;

	if (!size)
		return 1;

	vma = find_vma(current->mm, start);
	if (!vma)
		goto bad_area;
	if (vma->vm_start > start)
		goto check_stack;

good_area:
	if (!(vma->vm_flags & VM_WRITE))
		goto bad_area;
	size--;
	size += start & ~PAGE_MASK;
	size >>= PAGE_SHIFT;
	start &= PAGE_MASK;

	for (;;) {
	survive:
		{
			int fault = handle_mm_fault(current->mm, vma, start, 1);
			if (!fault)
				goto bad_area;
			if (fault < 0)
				goto out_of_memory;
		}
		if (!size)
			break;
		size--;
		start += PAGE_SIZE;
		if (start < vma->vm_end)
			continue;
		vma = vma->vm_next;
		if (!vma || vma->vm_start != start)
			goto bad_area;
		if (!(vma->vm_flags & VM_WRITE))
			goto bad_area;;
	}
	return 1;

check_stack:
	if (!(vma->vm_flags & VM_GROWSDOWN))
		goto bad_area;
	if (expand_stack(vma, start) == 0)
		goto good_area;

bad_area:
	return 0;

out_of_memory:
	if (current->pid == 1) {
		yield();
		goto survive;
	}
	goto bad_area;
}

/* Sometimes the CPU reports invalid exceptions on prefetch.
   Check that here and ignore.
   Opcode checker based on code by Richard Brunner */
static int is_prefetch(struct pt_regs *regs, unsigned long addr)
{ 
	unsigned char *instr = (unsigned char *)(regs->eip);
	int scan_more = 1;
	int prefetch = 0; 
	unsigned char *max_instr = instr + 15;

	/* Avoid recursive faults. This is just an optimization,
	   they must be handled correctly too */
	if (regs->eip == addr)
		return 0; 

	if (((regs->xcs & 0xffff) != __KERNEL_CS) && ((regs->xcs & 0xffff) != __USER_CS))
		return 0;

	while (scan_more && instr < max_instr) { 
		unsigned char opcode;
		unsigned char instr_hi;
		unsigned char instr_lo;

		if ((regs->xcs & 3) ? get_user(opcode,instr) : __get_user(opcode, instr))
			break; 

		instr_hi = opcode & 0xf0; 
		instr_lo = opcode & 0x0f; 
		instr++;

		switch (instr_hi) { 
		case 0x20:
		case 0x30:
			/* Values 0x26,0x2E,0x36,0x3E are valid x86
			   prefixes.  In long mode, the CPU will signal
			   invalid opcode if some of these prefixes are
			   present so we will never get here anyway */
			scan_more = ((instr_lo & 7) == 0x6);
			break;
			
		case 0x40:
			/* May be valid in long mode (REX prefixes) */
			break; 
			
		case 0x60:
			/* 0x64 thru 0x67 are valid prefixes in all modes. */
			scan_more = (instr_lo & 0xC) == 0x4;
			break;		
		case 0xF0:
			/* 0xF0, 0xF2, and 0xF3 are valid prefixes in all modes. */
			scan_more = !instr_lo || (instr_lo>>1) == 1;
			break;			
		case 0x00:
			/* Prefetch instruction is 0x0F0D or 0x0F18 */
			scan_more = 0;
			if ((regs->xcs & 3) ? get_user(opcode,instr) : __get_user(opcode, instr)) 
				break;
			prefetch = (instr_lo == 0xF) &&
				(opcode == 0x0D || opcode == 0x18);
			break;			
		default:
			scan_more = 0;
			break;
		} 
	}

#if 0
	if (prefetch)
		printk("%s: prefetch caused page fault at %lx/%lx\n", current->comm,
		       regs->eip, addr);
#endif
	return prefetch;
}


/*
 * Unlock any spinlocks which will prevent us from getting the
 * message out
 */
void bust_spinlocks(int yes)
{
	if (yes) {
		oops_in_progress = 1;
#ifdef CONFIG_SMP
		global_irq_lock = 0;	/* Many serial drivers do __global_cli() */
#endif
	} else {
		int loglevel_save = console_loglevel;
#ifdef CONFIG_VT
		unblank_screen();
#endif
		oops_in_progress = 0;
		/*
		 * OK, the message is on the console.  Now we call printk()
		 * without oops_in_progress set so that printk will give klogd
		 * a poke.  Hold onto your hats...
		 */
		console_loglevel = 15;		/* NMI oopser may have shut the console up */
		printk(" ");
		console_loglevel = loglevel_save;
	}
}

asmlinkage void do_invalid_op(struct pt_regs *, unsigned long);
extern unsigned long idt;

#if defined(CONFIG_GRKERNSEC_PAX_PAGEEXEC) || defined(CONFIG_GRKERNSEC_PAX_SEGMEXEC)
static int pax_handle_fetch_fault(struct pt_regs *regs);
#endif

/*
 * This routine handles page faults.  It determines the address,
 * and the problem, and then passes it off to one of the appropriate
 * routines.
 *
 * error_code:
 *	bit 0 == 0 means no page found, 1 means protection fault
 *	bit 1 == 0 means read, 1 means write
 *	bit 2 == 0 means kernel, 1 means user-mode
 */
#ifdef CONFIG_GRKERNSEC_PAX_PAGEEXEC
static int do_page_fault(struct pt_regs *regs, unsigned long error_code, unsigned long address)
#else
asmlinkage int do_page_fault(struct pt_regs *regs, unsigned long error_code)
#endif
{
	struct task_struct *tsk;
	struct mm_struct *mm;
	struct vm_area_struct * vma;
#ifndef CONFIG_GRKERNSEC_PAX_PAGEEXEC
	unsigned long address;
#endif
	unsigned long page;
	unsigned long fixup;
	int write;
	siginfo_t info;

#ifndef CONFIG_GRKERNSEC_PAX_PAGEEXEC
	/* get the address */
	__asm__("movl %%cr2,%0":"=r" (address));

	/* It's safe to allow irq's after cr2 has been saved */
	if (regs->eflags & X86_EFLAGS_IF)
		local_irq_enable();
#endif

	tsk = current;

	/*
	 * We fault-in kernel-space virtual memory on-demand. The
	 * 'reference' page table is init_mm.pgd.
	 *
	 * NOTE! We MUST NOT take any locks for this case. We may
	 * be in an interrupt or a critical region, and should
	 * only copy the information from the master page table,
	 * nothing more.
	 *
	 * This verifies that the fault happens in kernel space
	 * (error_code & 4) == 0, and that the fault was not a
	 * protection error (error_code & 1) == 0.
	 */
	if (address >= TASK_SIZE && !(error_code & 5))
		goto vmalloc_fault;

	mm = tsk->mm;
	info.si_code = SEGV_MAPERR;

	/*
	 * If we're in an interrupt or have no user
	 * context, we must not take the fault..
	 */
	if (in_interrupt() || !mm)
		goto bad_area_nosemaphore;

	down_read(&mm->mmap_sem);

	vma = find_vma(mm, address);
	if (!vma)
		goto bad_area;
	if (vma->vm_start <= address)
		goto good_area;
	if (!(vma->vm_flags & VM_GROWSDOWN))
		goto bad_area;
	if (error_code & 4) {
		/*
		 * accessing the stack below %esp is always a bug.
		 * The "+ 32" is there due to some instructions (like
		 * pusha) doing post-decrement on the stack and that
		 * doesn't show up until later..
		 */
		if (address + 32 < regs->esp)
			goto bad_area;
	}
	if (expand_stack(vma, address))
		goto bad_area;
/*
 * Ok, we have a good vm_area for this memory access, so
 * we can handle it..
 */
good_area:
	info.si_code = SEGV_ACCERR;
	write = 0;
	switch (error_code & 3) {
		default:	/* 3: write, present */
#ifdef TEST_VERIFY_AREA
			if (regs->cs == KERNEL_CS)
				printk("WP fault at %08lx\n", regs->eip);
#endif
			/* fall through */
		case 2:		/* write, not present */
			if (!(vma->vm_flags & VM_WRITE))
				goto bad_area;
			write++;
			break;
		case 1:		/* read, present */
			goto bad_area;
		case 0:		/* read, not present */
			if (!(vma->vm_flags & (VM_READ | VM_EXEC)))
				goto bad_area;
	}

 survive:
	/*
	 * If for any reason at all we couldn't handle the fault,
	 * make sure we exit gracefully rather than endlessly redo
	 * the fault.
	 */
	switch (handle_mm_fault(mm, vma, address, write)) {
	case 1:
		tsk->min_flt++;
		break;
	case 2:
		tsk->maj_flt++;
		break;
	case 0:
		goto do_sigbus;
	default:
		goto out_of_memory;
	}

	/*
	 * Did it hit the DOS screen memory VA from vm86 mode?
	 */
	if (regs->eflags & VM_MASK) {
		unsigned long bit = (address - 0xA0000) >> PAGE_SHIFT;
		if (bit < 32)
			tsk->thread.screen_bitmap |= 1 << bit;
	}
	up_read(&mm->mmap_sem);
	return 0;

/*
 * Something tried to access memory that isn't in our memory map..
 * Fix it, but check if it's kernel or user first..
 */
bad_area:
	up_read(&mm->mmap_sem);

bad_area_nosemaphore:
	/* User mode accesses just cause a SIGSEGV */
	if (error_code & 4) {
#ifdef CONFIG_GRKERNSEC_PAX_SEGMEXEC
		if (current->flags & PF_PAX_SEGMEXEC) {

#if defined(CONFIG_GRKERNSEC_PAX_EMUTRAMP) || defined(CONFIG_GRKERNSEC_PAX_RANDEXEC)
		if ((error_code == 4) && (regs->eip + SEGMEXEC_TASK_SIZE == address)) {
			switch (pax_handle_fetch_fault(regs)) {

#ifdef CONFIG_GRKERNSEC_PAX_RANDEXEC
			case 5:
				return 0;
#endif

#ifdef CONFIG_GRKERNSEC_PAX_EMUTRAMP
			case 4:
				return 0;
			case 3:
			case 2:
				return 1;
#endif

			case 1:
			default:
			}
		}
#endif

			if (address >= SEGMEXEC_TASK_SIZE) {
				pax_report_fault(regs, (void*)regs->eip, (void*)regs->esp);
				do_exit(SIGKILL);
			}
		}
#endif
		if (is_prefetch(regs, address))
			return;

		tsk->thread.cr2 = address;
		tsk->thread.error_code = error_code;
		tsk->thread.trap_no = 14;
		info.si_signo = SIGSEGV;
		info.si_errno = 0;
		/* info.si_code has been set above */
		info.si_addr = (void *)address;
		force_sig_info(SIGSEGV, &info, tsk);
		return 0;
	}

	/*
	 * Pentium F0 0F C7 C8 bug workaround.
	 */
	if (boot_cpu_data.f00f_bug) {
		unsigned long nr;
		
		nr = (address - idt) >> 3;

		if (nr == 6) {
			do_invalid_op(regs, 0);
			return 0;
		}
	}

no_context:
	/* Are we prepared to handle this kernel fault?  */
	if ((fixup = search_exception_table(regs->eip)) != 0) {
		regs->eip = fixup;
		return 0;
	}

	if (is_prefetch(regs, address))
		return;

/*
 * Oops. The kernel tried to access some bad page. We'll have to
 * terminate things with extreme prejudice.
 */
	spin_lock(&oops_lock);
	bust_spinlocks(1);

	if (address < PAGE_SIZE)
		printk(KERN_ALERT "Unable to handle kernel NULL pointer dereference");

#ifdef CONFIG_GRKERNSEC_PAX_KERNEXEC
	else if (init_mm.start_code + __KERNEL_TEXT_OFFSET <= address && address < init_mm.end_code + __KERNEL_TEXT_OFFSET) {
		if (tsk->curr_ip)
			printk(KERN_ERR "PAX: From %u.%u.%u.%u: %s:%d, uid/euid: %u/%u, attempted to modify kernel code",
					 NIPQUAD(tsk->curr_ip), tsk->comm, tsk->pid, tsk->uid, tsk->euid);
		else
			printk(KERN_ERR "PAX: %s:%d, uid/euid: %u/%u, attempted to modify kernel code",
					 tsk->comm, tsk->pid, tsk->uid, tsk->euid);
	}
#endif

	else
		printk(KERN_ALERT "Unable to handle kernel paging request");
	printk(" at virtual address %08lx\n",address);
	printk(" printing eip:\n");
	printk("%08lx\n", regs->eip);
	asm("movl %%cr3,%0":"=r" (page));
	page = ((unsigned long *) __va(page))[address >> 22];
	printk(KERN_ALERT "*pde = %08lx\n", page);
	if (page & 1) {
		page &= PAGE_MASK;
		address &= 0x003ff000;
		page = ((unsigned long *) __va(page))[address >> PAGE_SHIFT];
		printk(KERN_ALERT "*pte = %08lx\n", page);
	}
	die("Oops", regs, error_code);
	bust_spinlocks(0);
	spin_unlock(&oops_lock);
	do_exit(SIGKILL);

/*
 * We ran out of memory, or some other thing happened to us that made
 * us unable to handle the page fault gracefully.
 */
out_of_memory:
	if (tsk->pid == 1) {
		yield();
		goto survive;
	}
	up_read(&mm->mmap_sem);
	printk("VM: killing process %s\n", tsk->comm);
	if (error_code & 4)
		do_exit(SIGKILL);
	goto no_context;

do_sigbus:
	up_read(&mm->mmap_sem);

	/* Kernel mode? Handle exceptions or die */
	if (!(error_code & 4))
		goto no_context;

	if (is_prefetch(regs, address))
		return;

	tsk->thread.cr2 = address;
	tsk->thread.error_code = error_code;
	tsk->thread.trap_no = 14;
	info.si_signo = SIGBUS;
	info.si_errno = 0;
	info.si_code = BUS_ADRERR;
	info.si_addr = (void *)address;
	force_sig_info(SIGBUS, &info, tsk);

	return 0;

vmalloc_fault:
	{
		/*
		 * Synchronize this task's top level page-table
		 * with the 'reference' page table.
		 *
		 * Do _not_ use "tsk" here. We might be inside
		 * an interrupt in the middle of a task switch..
		 */
		int offset = __pgd_offset(address);
		pgd_t *pgd, *pgd_k;
		pmd_t *pmd, *pmd_k;
		pte_t *pte_k;

		asm("movl %%cr3,%0":"=r" (pgd));
		pgd = offset + (pgd_t *)__va(pgd);
		pgd_k = init_mm.pgd + offset;

		if (!pgd_present(*pgd_k))
			goto no_context;
		set_pgd(pgd, *pgd_k);
		
		pmd = pmd_offset(pgd, address);
		pmd_k = pmd_offset(pgd_k, address);
		if (!pmd_present(*pmd_k))
			goto no_context;
		set_pmd(pmd, *pmd_k);

		pte_k = pte_offset_kernel(pmd_k, address);
		if (!pte_present(*pte_k))
			goto no_context;
		return 0;
	}
}
#ifdef CONFIG_GRKERNSEC_PAX_PAGEEXEC
/* PaX: called with the page_table_lock spinlock held */
static inline pte_t * pax_get_pte(struct mm_struct *mm, unsigned long address)
{
	pgd_t *pgd;
	pmd_t *pmd;

	pgd = pgd_offset(mm, address);
	if (!pgd || !pgd_present(*pgd))
		return 0;
	pmd = pmd_offset(pgd, address);
	if (!pmd || !pmd_present(*pmd))
		return 0;
	return pte_offset_kernel(pmd, address);
}
#endif

/*
 * PaX: decide what to do with offenders (regs->eip = fault address)
 *
 * returns 1 when task should be killed
 *         2 when sigreturn trampoline was detected
 *         3 when rt_sigreturn trampoline was detected
 *         4 when gcc trampoline was detected
 *	   5 when legitimate ET_EXEC was detected
 */
#if defined(CONFIG_GRKERNSEC_PAX_PAGEEXEC) || defined(CONFIG_GRKERNSEC_PAX_SEGMEXEC)
static int pax_handle_fetch_fault(struct pt_regs *regs)
{
#ifdef CONFIG_GRKERNSEC_PAX_EMUTRAMP
	static const unsigned char trans[8] = {6, 1, 2, 0, 13, 5, 3, 4};
#endif
	int err;
	
#ifdef CONFIG_GRKERNSEC_PAX_RANDEXEC
	if (current->flags & PF_PAX_RANDEXEC) {
		unsigned long esp_4;
		if (regs->eip >= current->mm->start_code &&
		    regs->eip < current->mm->end_code)
		{
			err = get_user(esp_4, (unsigned long*)(regs->esp-4UL));
			if (err || esp_4 == regs->eip)
				return 1;
			regs->eip += current->mm->delta_exec;
			return 5;
		}
	}
#endif

#ifdef CONFIG_GRKERNSEC_PAX_EMUTRAMP

#ifndef CONFIG_GRKERNSEC_PAX_EMUSIGRT
	if (!(current->flags & PF_PAX_EMUTRAMP))
		return 1;
#endif

	do { /* PaX: sigreturn emulation */
		unsigned char pop, mov;
		unsigned short sys;
		unsigned long nr;

		err = get_user(pop, (unsigned char *)(regs->eip));
		err |= get_user(mov, (unsigned char *)(regs->eip + 1));
		err |= get_user(nr, (unsigned long *)(regs->eip + 2));
		err |= get_user(sys, (unsigned short *)(regs->eip + 6));

		if (err)
			break;

		if (pop == 0x58 &&
		    mov == 0xb8 &&
		    nr == __NR_sigreturn &&
		    sys == 0x80cd)
		{

#ifdef CONFIG_GRKERNSEC_PAX_EMUSIGRT
			int sig;
			struct k_sigaction *ka;
			__sighandler_t handler;

			if (get_user(sig, (int *)regs->esp))
				return 1;
			if (sig < 1 || sig > _NSIG || sig == SIGKILL || sig == SIGSTOP)
				return 1;
			ka = &current->sig->action[sig-1];
			handler = ka->sa.sa_handler;
			if (handler == SIG_DFL || handler == SIG_IGN) {
				if (!(current->flags & PF_PAX_EMUTRAMP))
					return 1;
			} else if (ka->sa.sa_flags & SA_SIGINFO)
				return 1;
#endif

			regs->esp += 4;
			regs->eax = nr;
			regs->eip += 8;
			return 2;
		}
	} while (0);

	do { /* PaX: rt_sigreturn emulation */
		unsigned char mov;
		unsigned short sys;
		unsigned long nr;

		err = get_user(mov, (unsigned char *)(regs->eip));
		err |= get_user(nr, (unsigned long *)(regs->eip + 1));
		err |= get_user(sys, (unsigned short *)(regs->eip + 5));

		if (err)
			break;

		if (mov == 0xb8 &&
		    nr == __NR_rt_sigreturn &&
		    sys == 0x80cd)
		{

#ifdef CONFIG_GRKERNSEC_PAX_EMUSIGRT
			int sig;
			struct k_sigaction *ka;
			__sighandler_t handler;

			if (get_user(sig, (int *)regs->esp))
				return 1;
			if (sig < 1 || sig > _NSIG || sig == SIGKILL || sig == SIGSTOP)
				return 1;
			ka = &current->sig->action[sig-1];
			handler = ka->sa.sa_handler;
			if (handler == SIG_DFL || handler == SIG_IGN) {
				if (!(current->flags & PF_PAX_EMUTRAMP))
					return 1;
			} else if (!(ka->sa.sa_flags & SA_SIGINFO))
				return 1;
#endif

			regs->eax = nr;
			regs->eip += 7;
			return 3;
		}
	} while (0);

#ifdef CONFIG_GRKERNSEC_PAX_EMUSIGRT
	if (!(current->flags & PF_PAX_EMUTRAMP))
		return 1;
#endif

	do { /* PaX: gcc trampoline emulation #1 */
		unsigned char mov1, mov2;
		unsigned short jmp;
		unsigned long addr1, addr2, ret;
		unsigned short call;

		err = get_user(mov1, (unsigned char *)regs->eip);
		err |= get_user(addr1, (unsigned long *)(regs->eip + 1));
		err |= get_user(mov2, (unsigned char *)(regs->eip + 5));
		err |= get_user(addr2, (unsigned long *)(regs->eip + 6));
		err |= get_user(jmp, (unsigned short *)(regs->eip + 10));
		err |= get_user(ret, (unsigned long *)regs->esp);

		if (err)
			break;

		err = get_user(call, (unsigned short *)(ret-2));
		if (err)
			break;

		if ((mov1 & 0xF8) == 0xB8 &&
		    (mov2 & 0xF8) == 0xB8 &&
		    (mov1 & 0x07) != (mov2 & 0x07) &&
		    (jmp & 0xF8FF) == 0xE0FF &&
		    (mov2 & 0x07) == ((jmp>>8) & 0x07) &&
		    (call & 0xF8FF) == 0xD0FF &&
		    regs->eip == ((unsigned long*)regs)[trans[(call>>8) & 0x07]])
		{
			((unsigned long *)regs)[trans[mov1 & 0x07]] = addr1;
			((unsigned long *)regs)[trans[mov2 & 0x07]] = addr2;
			regs->eip = addr2;
			return 4;
		}
	} while (0);

	do { /* PaX: gcc trampoline emulation #2 */
		unsigned char mov, jmp;
		unsigned long addr1, addr2, ret;
		unsigned short call;

		err = get_user(mov, (unsigned char *)regs->eip);
		err |= get_user(addr1, (unsigned long *)(regs->eip + 1));
		err |= get_user(jmp, (unsigned char *)(regs->eip + 5));
		err |= get_user(addr2, (unsigned long *)(regs->eip + 6));
		err |= get_user(ret, (unsigned long *)regs->esp);

		if (err)
			break;

		err = get_user(call, (unsigned short *)(ret-2));
		if (err)
			break;

		if ((mov & 0xF8) == 0xB8 &&
		    jmp == 0xE9 &&
		    (call & 0xF8FF) == 0xD0FF &&
		    regs->eip == ((unsigned long*)regs)[trans[(call>>8) & 0x07]])
		{
			((unsigned long *)regs)[trans[mov & 0x07]] = addr1;
			regs->eip += addr2 + 10;
			return 4;
		}
	} while (0);

	do { /* PaX: gcc trampoline emulation #3 */
		unsigned char mov, jmp;
		char offset;
		unsigned long addr1, addr2, ret;
		unsigned short call;

		err = get_user(mov, (unsigned char *)regs->eip);
		err |= get_user(addr1, (unsigned long *)(regs->eip + 1));
		err |= get_user(jmp, (unsigned char *)(regs->eip + 5));
		err |= get_user(addr2, (unsigned long *)(regs->eip + 6));
		err |= get_user(ret, (unsigned long *)regs->esp);

		if (err)
			break;

		err = get_user(call, (unsigned short *)(ret-3));
		err |= get_user(offset, (char *)(ret-1));
		if (err)
			break;

		if ((mov & 0xF8) == 0xB8 &&
		    jmp == 0xE9 &&
		    call == 0x55FF)
		{
			unsigned long addr;

			err = get_user(addr, (unsigned long*)(regs->ebp + (unsigned long)(long)offset));
			if (err || regs->eip != addr)
				break;

			((unsigned long *)regs)[trans[mov & 0x07]] = addr1;
			regs->eip += addr2 + 10;
			return 4;
		}
	} while (0);

	do { /* PaX: gcc trampoline emulation #4 */
		unsigned char mov, jmp, sib;
		char offset;
		unsigned long addr1, addr2, ret;
		unsigned short call;

		err = get_user(mov, (unsigned char *)regs->eip);
		err |= get_user(addr1, (unsigned long *)(regs->eip + 1));
		err |= get_user(jmp, (unsigned char *)(regs->eip + 5));
		err |= get_user(addr2, (unsigned long *)(regs->eip + 6));
		err |= get_user(ret, (unsigned long *)regs->esp);

		if (err)
			break;

		err = get_user(call, (unsigned short *)(ret-4));
		err |= get_user(sib, (unsigned char *)(ret-2));
		err |= get_user(offset, (char *)(ret-1));
		if (err)
			break;

		if ((mov & 0xF8) == 0xB8 &&
		    jmp == 0xE9 &&
		    call == 0x54FF &&
		    sib == 0x24)
		{
			unsigned long addr;

			err = get_user(addr, (unsigned long*)(regs->esp + 4 + (unsigned long)(long)offset));
			if (err || regs->eip != addr)
				break;

			((unsigned long *)regs)[trans[mov & 0x07]] = addr1;
			regs->eip += addr2 + 10;
			return 4;
		}
	} while (0);

	do { /* PaX: gcc trampoline emulation #5 */
		unsigned char mov, jmp, sib;
		unsigned long addr1, addr2, ret, offset;
		unsigned short call;

		err = get_user(mov, (unsigned char *)regs->eip);
		err |= get_user(addr1, (unsigned long *)(regs->eip + 1));
		err |= get_user(jmp, (unsigned char *)(regs->eip + 5));
		err |= get_user(addr2, (unsigned long *)(regs->eip + 6));
		err |= get_user(ret, (unsigned long *)regs->esp);

		if (err)
			break;

		err = get_user(call, (unsigned short *)(ret-7));
		err |= get_user(sib, (unsigned char *)(ret-5));
		err |= get_user(offset, (unsigned long *)(ret-4));
		if (err)
			break;

		if ((mov & 0xF8) == 0xB8 &&
		    jmp == 0xE9 &&
		    call == 0x94FF &&
		    sib == 0x24)
		{
			unsigned long addr;

			err = get_user(addr, (unsigned long*)(regs->esp + 4 + offset));
			if (err || regs->eip != addr)
				break;

			((unsigned long *)regs)[trans[mov & 0x07]] = addr1;
			regs->eip += addr2 + 10;
			return 4;
		}
	} while (0);
#endif

	return 1; /* PaX in action */
}

void pax_report_insns(void *pc)
{
	unsigned long i;

	printk(KERN_ERR "PAX: bytes at PC: ");
	for (i = 0; i < 20; i++) {
		unsigned char c;
		if (get_user(c, (unsigned char*)pc+i)) {
			printk("<invalid address>.");
			break;
		}
		printk("%02x ", c);
	}
	printk("\n");
}
#endif

#ifdef CONFIG_GRKERNSEC_PAX_PAGEEXEC
/*
 * PaX: handle the extra page faults or pass it down to the original handler
 *
 * returns 0 when nothing special was detected
 *         1 when sigreturn trampoline (syscall) has to be emulated
 */
asmlinkage int pax_do_page_fault(struct pt_regs *regs, unsigned long error_code)
{
	struct mm_struct *mm = current->mm;
	unsigned long address;
	pte_t *pte;
	unsigned char pte_mask;
	int ret;

	__asm__("movl %%cr2,%0":"=r" (address));

	/* It's safe to allow irq's after cr2 has been saved */
	if (likely(regs->eflags & X86_EFLAGS_IF))
		local_irq_enable();

	if (unlikely((error_code & 5) != 5 ||
		     address >= TASK_SIZE ||
		     !(current->flags & PF_PAX_PAGEEXEC)))
		return do_page_fault(regs, error_code, address);

	/* PaX: it's our fault, let's handle it if we can */

	/* PaX: take a look at read faults before acquiring any locks */
	if (unlikely((error_code == 5) && (regs->eip == address))) { 
		/* instruction fetch attempt from a protected page in user mode */
		ret = pax_handle_fetch_fault(regs);
		switch (ret) {
#ifdef CONFIG_GRKERNSEC_PAX_RANDEXEC
		case 5:
			return 0;
#endif

#ifdef CONFIG_GRKERNSEC_PAX_EMUTRAMP
		case 4:
			return 0;
		case 3:
		case 2:
			return 1;
#endif
		case 1:
		default:
			pax_report_fault(regs, (void*)regs->eip, (void*)regs->esp);
			do_exit(SIGKILL);
		}
	}

	pte_mask = _PAGE_ACCESSED | _PAGE_USER | ((error_code & 2) << (_PAGE_BIT_DIRTY-1));

	spin_lock(&mm->page_table_lock);
	pte = pax_get_pte(mm, address);
	if (unlikely(!pte || !(pte_val(*pte) & _PAGE_PRESENT) || pte_exec(*pte))) {
		spin_unlock(&mm->page_table_lock);
		do_page_fault(regs, error_code, address);
		return 0;
	}

	if (unlikely((error_code == 7) && !pte_write(*pte))) {
		/* write attempt to a protected page in user mode */
		spin_unlock(&mm->page_table_lock);
		do_page_fault(regs, error_code, address);
		return 0;
	}

	/*
	 * PaX: fill DTLB with user rights and retry
	 */
	__asm__ __volatile__ (
		"orb %2,%1\n"
#if defined(CONFIG_M586) || defined(CONFIG_M586TSC)
/*   
 * PaX: let this uncommented 'invlpg' remind us on the behaviour of Intel's   
 * (and AMD's) TLBs. namely, they do not cache PTEs that would raise *any*
 * page fault when examined during a TLB load attempt. this is true not only
 * for PTEs holding a non-present entry but also present entries that will
 * raise a page fault (such as those set up by PaX, or the copy-on-write
 * mechanism). in effect it means that we do *not* need to flush the TLBs
 * for our target pages since their PTEs are simply not in the TLBs at all.
 * the best thing in omitting it is that we gain around 15-20% speed in the
 * fast path of the page fault handler and can get rid of tracing since we
 * can no longer flush unintended entries.
 */

		"invlpg %0\n"
#endif

		"testb $0,%0\n"
		"xorb %3,%1\n"
		:
		: "m" (*(char*)address), "m" (*(char*)pte) , "q" (pte_mask) , "i" (_PAGE_USER)
		: "memory", "cc");
	spin_unlock(&mm->page_table_lock);
	return 0;
}
#endif
