#!/bin/bash
## -----------------------------------------------------------------------
##   
##   Copyright 2000-2001 H. Peter Anvin - All Rights Reserved
##
##   This program is free software; you can redistribute it and/or modify
##   it under the terms of the GNU General Public License as published by
##   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
##   USA; either version 2 of the License, or (at your option) any later
##   version; incorporated herein by reference.
##
## -----------------------------------------------------------------------
## $Id: make.sh,v 1.37 2001/05/01 18:38:08 hpa Exp $

##
## Main make script for SuperRescue CD
##

if [ -z "$1" ]; then
    echo "Usage: $0 version" 1>&2
    exit 1
fi

BUILDER=`cat Builder 2>/dev/null`
if [ -z "$BUILDER" ]; then
    host=`hostname --fqdn 2>/dev/null`
    echo "Please create the Builder file first, for example:" 1>&2
    echo "  echo 'Your Name Here <$LOGNAME@$host>' > Builder" 1>&2
    exit 1
fi

# Missing trees?
if [ ! -d usr -o ! -d root ]; then
    echo "Missing usr and/or root tree!" 1>&2
    exit 1
fi

set -xe

DATE=`date`
BUILDDIRS='rootpatch irtree isolinux'
READMES='README README-BUILD CHANGES'
MAKES='make.sh makeinit.sh'

umask 022

# Initial cleanup
rm -rf isotree roottree buildtree gzdir
rm -f  superrescue-$1.iso superrescue-$1.iso.gz buildtree-$1.tar.gz
mkdir -p isotree isotree/boot isotree/isolinux roottree buildtree

# Make initrd
./makeinit.sh

# Set up boot files
cd isotree/isolinux
  ln ../../usr/boot/vmlinuz* .
  ln ../../usr/lib/syslinux/isolinux.bin .
  cp ../../initrd.gz .
  cp -a ../../isolinux/. .
  # Turn symlinks into hard links
  for file in *; do
    if [ -L "$file" ]; then
      target=`ls -l --color=no "$file" | cut -d\> -f2- | cut -c2-`
      rm -f "$file"
      ln "$target" "$file"
    fi
  done
cd ../..

# Generate root tree.
cp -a root/. roottree/. 
# Cleanup of root which is hard to do at tree-creation time..
cp -af rootpatch/. roottree/.
find roottree -name lost+found | xargs -0rt rm -rf
find roottree/var/lock roottree/var/run -type f -print0 | xargs -0rt rm -f
rm -rf roottree/work
# This one is useful across all trees!
find roottree usr isotree isolinux \
    \( -name \*~ -or -name \#\* \) -print0 | xargs -0rt rm -f
rm -f roottree/.bash_history roottree/root/.bash_history roottree/home/*/.bash_history
rm -f roottree/.history roottree/root/.history roottree/home/*/.history
rm -rf roottree/.ssh roottree/root/.ssh roottree/home/*/.ssh
rm -rf roottree/.netscape roottree/root/.netscape roottree/home/*/.netscape
rm -f roottree/.rpmmacros roottree/root/.rpmmacros roottree/home/*/.rpmmacros
rm -f roottree/.TWM-errors roottree/root/.TWM-errors roottree/home/*/.TWM-errors
# We definitely don't want inetd running by default
rm -f roottree/etc/rc.d/rc?.d/S??inet
# tunnelv doesn't work right.  Kill it.
rm -f roottree/etc/rc.d/rc?.d/S??tunnelv
# NFS locking has possible remote root exploit
rm -f roottree/etc/rc.d/rc?.d/S??nfslock
rm -rf roottree/tmp roottree/var/tmp
mkdir -p roottree/tmp roottree/var/tmp
chown root.root roottree/tmp roottree/var/tmp
chmod 1777 roottree/tmp roottree/var/tmp

# Some things don't really need to be in the root, now when the initrd
# can mount /usr for us, thanks to pivot_root
movehier='bin sbin lib etc/codepages etc/termcap'
for ent in $movehier; do
    graft=`dirname $ent`
    tail=`basename $ent`
    if [ ! -L roottree/$ent ]; then
	if [ -d roottree/$ent -o -f roottree/$ent ]; then
	    mkdir -p isotree/root/$graft
	    mv roottree/$ent isotree/root/$ent
	    ( cd roottree/$graft && ln -s /usr/root/$ent . )
	fi
    fi
done

# Create root tarball
( cd roottree && tar cfp - . | gzip -9 > ../isotree/root.tar.gz )

# Generate distribution copy of build tree
# (Be careful when to use "cp" versus "cp -a" here!)
mkdir -p buildtree/root buildtree/usr
cp -a $BUILDDIRS buildtree
cp $MAKES $READMES buildtree

# Create build tree tarball
( cd buildtree && tar cfp ../isotree/buildtree-$1.tar . )
( cd isotree && gzip -9 buildtree-$1.tar && \
  ln -s buildtree-$1.tar.gz buildtree.tar.gz )

# Copy buildtree tarball to the root directory; for separate upload.
cp isotree/buildtree-$1.tar.gz .

# Create version number and buildinfo files
echo "$1" > isotree/hpa-mega-rescue-disk-1
cat > isotree/buildinfo.txt <<EOF
SuperRescue $1
Build by $BUILDER on $DATE
EOF

# Install README files
cp $READMES isotree

# Things that must not be in the usr tree
rm -f usr/root.tar.gz usr/buildtree.tar.gz
rm -f usr/isolinux usr/boot/isoboot.img usr/boot/isoboot.cat
rm -f usr/hpa-mega-rescue-disk-1 usr/buildinfo.txt
find usr \( -name rr_moved -o -name .rr_moved -o -name lost+found \) -print0 \
	| xargs -0rt rm -rf
( cd usr && rm -f $READMES )

# Generate ISO filesystem.
# Note: normally we would add -J for Joliet long filenames here,
# but that's purely a cosmetic issue (-R is required, however), and
# mkisofs seems to have problems with this fileset.
rm -f mkisofs.log
mkisofs -v -v -o superrescue-$1.iso \
    -b isolinux/isolinux.bin -c isolinux/isoboot.cat \
    -no-emul-boot -boot-load-size 4 -boot-info-table \
    -l -J -R -hide-rr-moved \
    -A SuperRescue -P "$BUILDER" -V "SuperRescue $1" isotree usr \
    > mkisofs.log 2>&1

# Generate compressed version
mkdir -p gzdir
ln -f superrescue-$1.iso gzdir/superrescue-$1.iso
( cd gzdir && gzip -9f superrescue-$1.iso )
mv gzdir/superrescue-$1.iso.gz .
rm -rf gzdir
# gzip seems to fsck up permissions somehow...
chmod 444 superrescue-$1.iso superrescue-$1.iso.gz
