/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { useState } from "preact/hooks";

// FIX default import https://github.com/microsoft/TypeScript/issues/49189
import {
  OfficerAccount,
  opFixedSuccess,
  TalerExchangeResultByMethod2,
  TalerHttpError,
} from "@gnu-taler/taler-util";
import {
  buildPaginatedResult,
  useExchangeApiContext,
} from "@gnu-taler/web-util/browser";
import _useSWR, { mutate, SWRHook } from "swr";
import { useOfficer } from "./officer.js";
const useSWR = _useSWR as unknown as SWRHook;

export const PAGINATED_LIST_SIZE = 10;
// when doing paginated request, ask for one more
// and use it to know if there are more to request
export const PAGINATED_LIST_REQUEST = PAGINATED_LIST_SIZE + 1;

export function revalidateCurrentLegitimizations() {
  return mutate(
    (key) =>
      Array.isArray(key) && key[key.length - 1] === "getAmlLegitimizations",
    undefined,
    { revalidate: true },
  );
}

/**
 * @param account
 * @param args
 * @returns
 */
export function useCurrentLegitimizations(accoutnStr: string) {
  const officer = useOfficer();
  const session = officer.state === "ready" ? officer.account : undefined;
  const {
    lib: { exchange: api },
  } = useExchangeApiContext();

  const [offset, setOffset] = useState<string>();

  async function fetcher([officer, account, offset]: [
    OfficerAccount,
    string | undefined,
    string | undefined,
    boolean | undefined,
  ]) {
    return await api.getAmlLegitimizations(officer, {
      order: "dec",
      offset,
      active: true,
      account,
      limit: PAGINATED_LIST_REQUEST,
    });
  }

  const { data, error } = useSWR<
    TalerExchangeResultByMethod2<"getAmlLegitimizations">,
    TalerHttpError
  >(!session ? undefined : [session, accoutnStr, offset, "getAmlLegitimizations"], fetcher);

  if (error) return error;
  if (data === undefined) return undefined;
  // if (data.type !== "ok") return data;

  return buildPaginatedResult(
    data.body.measures,
    offset,
    setOffset,
    (d) => String(d.rowid),
    PAGINATED_LIST_REQUEST,
  );
}
