static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * This program is free software: you can redistribute it and/or modify       \n"
" * it under the terms of the GNU General Public License as published by       \n"
" * the Free Software Foundation; either version 3 of the License, or          \n"
" * (at your option) any later version.                                        \n"
" *                                                                            \n"
" * This program is distributed in the hope that it will be useful,            \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              \n"
" * GNU General Public License for more details.                               \n"
" *                                                                            \n"
" * You should have received a copy of the GNU General Public License          \n"
" * along with this program.  If not, see <http://www.gnu.org/licenses/>.      \n"
" *                                                                            \n"
" * This operation generates solid noise textures based on the                 \n"
" * `Noise' and `Turbulence' functions described in the paper                  \n"
" *                                                                            \n"
" *    Perlin, K, and Hoffert, E. M., \"Hypertexture\",                        \n"
" *    Computer Graphics 23, 3 (August 1989)                                   \n"
" *                                                                            \n"
" * The algorithm implemented here also makes possible the                     \n"
" * creation of seamless tiles.                                                \n"
" *                                                                            \n"
" * Copyright (C) 1997, 1998 Marcelo de Gomensoro Malheiros                    \n"
" *                                                                            \n"
" * GEGL port: Thomas Manni <thomas.manni@free.fr>                             \n"
" *                                                                            \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (x_size, _(\"X Size\"), 4.0)                                  \n"
"    description (_(\"Horizontal texture size\"))                              \n"
"    value_range (0.1, 16.0)                                                   \n"
"    ui_range    (0.1, 16.0)                                                   \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"x\")                                             \n"
"                                                                              \n"
"property_double (y_size, _(\"Y Size\"), 4.0)                                  \n"
"    description (_(\"Vertical texture size\"))                                \n"
"    value_range (0.1, 16.0)                                                   \n"
"    ui_range    (0.1, 16.0)                                                   \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"y\")                                             \n"
"                                                                              \n"
"property_int    (detail, _(\"Detail\"), 1)                                    \n"
"    description (_(\"Detail level\"))                                         \n"
"    ui_range    (0, 15)                                                       \n"
"    value_range (0, 15)                                                       \n"
"                                                                              \n"
"property_boolean (tileable, _(\"Tileable\"), FALSE)                           \n"
"    description  (_(\"Create a tileable output\"))                            \n"
"                                                                              \n"
"property_boolean (turbulent, _(\"Turbulent\"), FALSE)                         \n"
"    description  (_(\"Make a turbulent noise\"))                              \n"
"                                                                              \n"
"property_seed (seed, _(\"Random seed\"), rand)                                \n"
"                                                                              \n"
"property_int    (width, _(\"Width\"), 1024)                                   \n"
"    description (_(\"Width of the generated buffer\"))                        \n"
"    value_range (0, G_MAXINT)                                                 \n"
"    ui_range    (0, 4096)                                                     \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"x\")                                             \n"
"    ui_meta     (\"role\", \"output-extent\")                                 \n"
"                                                                              \n"
"property_int (height, _(\"Height\"), 768)                                     \n"
"    description(_(\"Height of the generated buffer\"))                        \n"
"    value_range (0, G_MAXINT)                                                 \n"
"    ui_range    (0, 4096)                                                     \n"
"    ui_meta     (\"unit\", \"pixel-distance\")                                \n"
"    ui_meta     (\"axis\", \"y\")                                             \n"
"    ui_meta     (\"role\", \"output-extent\")                                 \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_POINT_RENDER                                                  \n"
"#define GEGL_OP_C_SOURCE noise-solid.c                                        \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"                                                                              \n"
"#define TABLE_SIZE  64                                                        \n"
"#define WEIGHT(T)   ((2.0 * fabs (T) - 3.0) * (T) * (T) + 1.0)                \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble x;                                                                  \n"
"  gdouble y;                                                                  \n"
"} Vector2;                                                                    \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gint         xclip;                                                         \n"
"  gint         yclip;                                                         \n"
"  gdouble      offset;                                                        \n"
"  gdouble      factor;                                                        \n"
"  gdouble      xsize;                                                         \n"
"  gdouble      ysize;                                                         \n"
"  gint         perm_tab[TABLE_SIZE];                                          \n"
"  Vector2      grad_tab[TABLE_SIZE];                                          \n"
"} NsParamsType;                                                               \n"
"                                                                              \n"
"static void                                                                   \n"
"solid_noise_init (GeglProperties *o)                                          \n"
"{                                                                             \n"
"  gint          i, j, k, t;                                                   \n"
"  gdouble       m;                                                            \n"
"  GRand        *gr;                                                           \n"
"  NsParamsType *params;                                                       \n"
"                                                                              \n"
"  params = (NsParamsType *) o->user_data;                                     \n"
"                                                                              \n"
"  g_assert (params != NULL);                                                  \n"
"                                                                              \n"
"  gr = g_rand_new_with_seed (o->seed);                                        \n"
"                                                                              \n"
"  /*  Set scaling factors  */                                                 \n"
"  if (o->tileable)                                                            \n"
"    {                                                                         \n"
"      params->xsize = ceil (o->x_size);                                       \n"
"      params->ysize = ceil (o->y_size);                                       \n"
"      params->xclip = (gint) params->xsize;                                   \n"
"      params->yclip = (gint) params->ysize;                                   \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      params->xsize = o->x_size;                                              \n"
"      params->ysize = o->y_size;                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  Set totally empiric normalization values  */                            \n"
"  if (o->turbulent)                                                           \n"
"    {                                                                         \n"
"      params->offset = 0.0;                                                   \n"
"      params->factor = 1.0;                                                   \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      params->offset = 0.94;                                                  \n"
"      params->factor = 0.526;                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  Initialize the permutation table  */                                    \n"
"  for (i = 0; i < TABLE_SIZE; i++)                                            \n"
"    params->perm_tab[i] = i;                                                  \n"
"                                                                              \n"
"  for (i = 0; i < (TABLE_SIZE >> 1); i++)                                     \n"
"    {                                                                         \n"
"      j = g_rand_int_range (gr, 0, TABLE_SIZE);                               \n"
"      k = g_rand_int_range (gr, 0, TABLE_SIZE);                               \n"
"      t = params->perm_tab[j];                                                \n"
"      params->perm_tab[j] = params->perm_tab[k];                              \n"
"      params->perm_tab[k] = t;                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /*  Initialize the gradient table  */                                       \n"
"  for (i = 0; i < TABLE_SIZE; i++)                                            \n"
"    {                                                                         \n"
"      do                                                                      \n"
"        {                                                                     \n"
"          params->grad_tab[i].x = g_rand_double_range (gr, -1, 1);            \n"
"          params->grad_tab[i].y = g_rand_double_range (gr, -1, 1);            \n"
"          m = params->grad_tab[i].x * params->grad_tab[i].x +                 \n"
"                   params->grad_tab[i].y * params->grad_tab[i].y;             \n"
"        }                                                                     \n"
"      while (m == 0.0 || m > 1.0);                                            \n"
"                                                                              \n"
"      m = 1.0 / sqrt(m);                                                      \n"
"      params->grad_tab[i].x *= m;                                             \n"
"      params->grad_tab[i].y *= m;                                             \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_rand_free (gr);                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"static gdouble                                                                \n"
"plain_noise (gdouble         x,                                               \n"
"             gdouble         y,                                               \n"
"             guint           s,                                               \n"
"             GeglProperties *o)                                               \n"
"{                                                                             \n"
"  Vector2       v;                                                            \n"
"  gint          a, b, i, j, n;                                                \n"
"  gdouble       sum;                                                          \n"
"  NsParamsType *p;                                                            \n"
"                                                                              \n"
"  p = (NsParamsType *) o->user_data;                                          \n"
"                                                                              \n"
"  sum = 0.0;                                                                  \n"
"  x *= s;                                                                     \n"
"  y *= s;                                                                     \n"
"  a = (gint) floor (x);                                                       \n"
"  b = (gint) floor (y);                                                       \n"
"                                                                              \n"
"  for (i = 0; i < 2; i++)                                                     \n"
"    {                                                                         \n"
"      for (j = 0; j < 2; j++)                                                 \n"
"        {                                                                     \n"
"          if (o->tileable)                                                    \n"
"            n = p->perm_tab[(((a + i) % (p->xclip * s)) + p->perm_tab[((b + j) % (p->yclip * s)) % TABLE_SIZE]) % TABLE_SIZE];\n"
"          else                                                                \n"
"            n = p->perm_tab[(a + i + p->perm_tab[(b + j) % TABLE_SIZE]) % TABLE_SIZE];\n"
"          v.x = x - a - i;                                                    \n"
"          v.y = y - b - j;                                                    \n"
"          sum += WEIGHT(v.x) * WEIGHT(v.y) * (p->grad_tab[n].x * v.x + p->grad_tab[n].y * v.y);\n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return sum / s;                                                             \n"
"}                                                                             \n"
"                                                                              \n"
"static gdouble                                                                \n"
"noise (gdouble         x,                                                     \n"
"       gdouble         y,                                                     \n"
"       GeglProperties *o)                                                     \n"
"{                                                                             \n"
"  gint          i;                                                            \n"
"  guint         s;                                                            \n"
"  gdouble       sum;                                                          \n"
"  NsParamsType *p;                                                            \n"
"                                                                              \n"
"  p = (NsParamsType *) o->user_data;                                          \n"
"                                                                              \n"
"  s = 1;                                                                      \n"
"  sum = 0.0;                                                                  \n"
"  x *= p->xsize;                                                              \n"
"  y *= p->ysize;                                                              \n"
"                                                                              \n"
"  for (i = 0; i <= o->detail; i++)                                            \n"
"    {                                                                         \n"
"      if (o->turbulent)                                                       \n"
"        sum += fabs (plain_noise (x, y, s, o));                               \n"
"      else                                                                    \n"
"        sum += plain_noise (x, y, s, o);                                      \n"
"      s <<= 1;                                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  return (sum + p->offset) * p->factor;                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"  const Babl     *format = babl_format (\"Y' float\");                        \n"
"                                                                              \n"
"  if (o->user_data == NULL)                                                   \n"
"    o->user_data = g_slice_new0 (NsParamsType);                               \n"
"                                                                              \n"
"  solid_noise_init (o);                                                       \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize (GObject *object)                                                    \n"
"{                                                                             \n"
"  GeglOperation *op = (void*) object;                                         \n"
"  GeglProperties *o = GEGL_PROPERTIES (op);                                   \n"
"                                                                              \n"
"  if (o->user_data)                                                           \n"
"    {                                                                         \n"
"      g_slice_free (NsParamsType, o->user_data);                              \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS (gegl_op_parent_class)->finalize (object);                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"c_process (GeglOperation       *operation,                                    \n"
"           void                *out_buf,                                      \n"
"           glong                n_pixels,                                     \n"
"           const GeglRectangle *roi,                                          \n"
"           gint                 level)                                        \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"  gint      x, y;                                                             \n"
"  gfloat    val;                                                              \n"
"  gfloat   *output = out_buf;                                                 \n"
"                                                                              \n"
"  for (y = roi->y; y < (roi->y + roi->height); y++)                           \n"
"    {                                                                         \n"
"      for (x = roi->x; x < (roi->x + roi->width); x++)                        \n"
"        {                                                                     \n"
"          val = noise (((gdouble) x / o->width), ((gdouble) y / o->height), o);\n"
"          *output = val;                                                      \n"
"          output++;                                                           \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *out_buf,                                        \n"
"         const GeglRectangle *roi,                                            \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"  const Babl         *format = gegl_operation_get_format (operation,          \n"
"                                                          \"output\");        \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (out_buf, roi, level, format,               \n"
"                                   GEGL_ACCESS_WRITE, GEGL_ABYSS_NONE);       \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"    c_process (operation, iter->data[0], iter->length, &iter->roi[0], level); \n"
"                                                                              \n"
"  return  TRUE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  return gegl_rectangle_infinite_plane ();                                    \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GObjectClass             *object_class;                                     \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationSourceClass *source_class;                                     \n"
"                                                                              \n"
"  object_class    = G_OBJECT_CLASS (klass);                                   \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  source_class = GEGL_OPERATION_SOURCE_CLASS (klass);                         \n"
"                                                                              \n"
"  object_class->finalize = finalize;                                          \n"
"                                                                              \n"
"  source_class->process = process;                                            \n"
"  operation_class->get_bounding_box = get_bounding_box;                       \n"
"  operation_class->prepare = prepare;                                         \n"
"  operation_class->opencl_support = FALSE;                                    \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",               \"gegl:noise-solid\",                             \n"
"    \"title\",              _(\"Solid Noise\"),                               \n"
"    \"categories\",         \"render\",                                       \n"
"    \"position-dependent\", \"true\",                                         \n"
"    \"license\",            \"GPL3+\",                                        \n"
"    \"description\", _(\"Create a random cloud-like texture\"),               \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
