#! /usr/bin/env perl

# Copyright (C) 2021-2025 Guido Flohr <guido.flohr@cantanea.com>,
# all rights reserved.

# This program is free software. It comes without any warranty, to
# the extent permitted by applicable law. You can redistribute it
# and/or modify it under the terms of the Do What the Fuck You Want
# to Public License, Version 2, as published by Sam Hocevar. See
# http://www.wtfpl.net/ for more details.

use strict;
use integer;

use Test::More;
use Data::Dumper;

use Chess::Plisco qw(:all);
use Chess::Plisco::Engine::Position;

my ($pos, @moves, @expect);

my @tests = (
	{
		name => 'Lone white king moves',
		fen => '7k/8/8/8/8/8/8/K7 w - - 0 1',
		san => 'Kb1',
	},
	{
		name => 'Lone black king moves',
		fen => '7k/8/8/8/8/8/8/K7 b - - 0 1',
		san => 'Kg8',
	},
	{
		name => 'Start position 1. e4',
		san => 'e4',
	},
	{
		name => 'Remove ep shift after 1. e4',
		fen => 'rnbqkbnr/pppppppp/8/8/4P3/8/PPPP1PPP/RNBQKBNR b KQkq - 0 1',
		san => 'Nf6',
	},
	{
		name => 'White queen-side castling',
		fen => '7k/8/8/8/8/8/8/R3K3 w Q - 0 1',
		san => 'O-O-O',
	},
	{
		name => 'Black king captures rook',
		fen => '7k/6R1/8/8/8/8/8/4K3 b - - 0 1',
		san => 'Kxg7',
	},
	{
		name => 'White promotes to queen',
		fen => '7k/4P3/8/8/8/8/8/4K3 w - - 0 1',
		san => 'e8=Q',
	},
	{
		name => 'White promotes to queen and captures',
		fen => '5q1k/4P3/8/8/8/8/8/4K3 w - - 0 1',
		san => 'exf8=Q',
	},
	{
		name => 'Simple white pawn capture',
		fen => '7k/8/8/8/8/1p6/P7/K7 w - - 0 1',
		san => 'axb3',
	},
	{
		name => 'Simple black pawn capture',
		fen => '7k/p7/1P6/8/8/8/8/K7 b - - 0 1',
		san => 'axb6',
	},
	{
		name => 'Flohr - Poisl, Kautsky mem4, 1927',
		fen => 'r2q1rk1/pbpnbpp1/1p1p3p/4p3/2PPP1nP/P1NB1N2/1P1BQPP1/R3K2R b KQ - 1 12',
		san => 'exd4',
	},
);

plan tests => 5 * @tests;

foreach my $test (@tests) {
	my $pos = Chess::Plisco::Engine::Position->new($test->{fen});
	my $zk_before = $pos->signature;
	ok defined $zk_before, "$test->{name}: zk defined after move";
	my $move = $pos->parseMove($test->{san});
	ok $move, "$test->{name}: parse $test->{san}";
	ok $pos->doMove($move), "$test->{name}: do $test->{san}";
	my $zk_updated = $pos->signature;
	ok defined $zk_updated, "$test->{name}: zk after move";
	# Internally, we do not check the legality of en-passant captures becuase
	# the check is relatively expensive, and signatures have to be updated
	# very often.
	my $fen = $pos->toFEN(force_en_passant_square => 1);
	$pos = Chess::Plisco::Engine::Position->new($fen);
	is $zk_updated, $pos->signature, "$test->{name}: zk updated";
}