"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRoutes = defineRoutes;
var _configSchema = require("@osd/config-schema");
var _stream = require("stream");
var _ml_agent_router = require("./ml_routes/ml_agent_router");
var _router_registry = require("./ml_routes/router_registry");
var _prompts = require("../prompts");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Forward request to external AG-UI server
 */
async function forwardToAgUI(agUiUrl, request, response, dataSourceId, logger) {
  // Prepare request body - include dataSourceId if provided
  const requestBody = dataSourceId ? {
    ...(request.body || {}),
    dataSourceId
  } : request.body;
  logger === null || logger === void 0 || logger.debug('Forwarding to external AG-UI', {
    agUiUrl,
    dataSourceId
  });

  // Forward the request to AG-UI server using native fetch (Node 18+)
  const agUiResponse = await fetch(agUiUrl, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      Accept: 'text/event-stream'
    },
    body: JSON.stringify(requestBody)
  });
  if (!agUiResponse.ok) {
    return response.customError({
      statusCode: agUiResponse.status,
      body: {
        message: `AG-UI server error: ${agUiResponse.statusText}`
      }
    });
  }

  // Convert Web ReadableStream to Node.js Readable stream
  const reader = agUiResponse.body.getReader();
  const stream = new _stream.Readable({
    async read() {
      try {
        const {
          done,
          value
        } = await reader.read();
        if (done) {
          this.push(null); // Signal end of stream
        } else {
          this.push(Buffer.from(value)); // Push as Buffer for binary mode
        }
      } catch (error) {
        this.destroy(error);
      }
    }
  });
  return response.ok({
    headers: {
      'Content-Type': 'text/event-stream',
      'Content-Encoding': 'identity',
      // Prevents compression buffering
      'Cache-Control': 'no-cache',
      Connection: 'keep-alive',
      'Transfer-Encoding': 'chunked',
      // Enables HTTP chunked transfer
      'X-Accel-Buffering': 'no' // Disables nginx buffering
    },

    body: stream
  });
}
function defineRoutes(router, logger, agUiUrl, getCapabilitiesResolver, mlCommonsAgentId, observabilityAgentId) {
  // Proxy route for AG-UI requests
  router.post({
    path: '/api/chat/proxy',
    validate: {
      body: _configSchema.schema.object({
        threadId: _configSchema.schema.string(),
        runId: _configSchema.schema.string(),
        messages: _configSchema.schema.arrayOf(_configSchema.schema.any()),
        tools: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
        context: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
        state: _configSchema.schema.maybe(_configSchema.schema.any()),
        forwardedProps: _configSchema.schema.maybe(_configSchema.schema.any())
      }),
      query: _configSchema.schema.maybe(_configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      }))
    }
  }, async (context, request, response) => {
    var _request$query;
    const dataSourceId = (_request$query = request.query) === null || _request$query === void 0 ? void 0 : _request$query.dataSourceId;
    try {
      var _request$body$forward;
      // Inject server-side system prompt if present
      (0, _prompts.injectSystemPrompt)(request.body.messages, (_request$body$forward = request.body.forwardedProps) === null || _request$body$forward === void 0 ? void 0 : _request$body$forward.queryAssistLanguage);

      // Check if ML Commons agentic features are enabled via capabilities
      const capabilitiesResolver = getCapabilitiesResolver === null || getCapabilitiesResolver === void 0 ? void 0 : getCapabilitiesResolver();
      const capabilities = capabilitiesResolver ? await capabilitiesResolver(request) : undefined;

      // Initialize ML agent routers based on current capabilities or configured agent IDs
      // This ensures routers are registered based on actual runtime capabilities
      _router_registry.MLAgentRouterRegistry.initialize(capabilities, observabilityAgentId);

      // Get the registered ML agent router (if any)
      const mlRouter = _ml_agent_router.MLAgentRouterFactory.getRouter();
      if (mlRouter) {
        logger.info(`Routing to ML Commons agent via ${mlRouter.getRouterName()}`);
        return await mlRouter.forward(context, request, response, logger, mlCommonsAgentId, dataSourceId, observabilityAgentId);
      }
      if (!agUiUrl) {
        return response.customError({
          statusCode: 503,
          body: {
            message: 'No AI agent available: ML Commons agent not enabled and AG-UI URL not configured'
          }
        });
      }

      // Forward to AG-UI capable endpoint. This is the default router.
      return await forwardToAgUI(agUiUrl, request, response, dataSourceId, logger);
    } catch (error) {
      logger.error(`AI agent routing error: ${error}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: error instanceof Error ? error.message : 'Unknown error occurred'
        }
      });
    }
  });
}