# frozen_string_literal: true

require 'gitlab'

module GitlabQuality
  module TestTooling
    module Report
      module GroupIssues
        class IncidentChecker
          GITLAB_PRODUCTION_PROJECT_ID = '7444821' # gitlab-com/gl-infra/production

          def self.get_active_incidents(token: nil, gitlab_url: 'https://gitlab.com')
            return [] unless token

            begin
              client = GitlabClient::IssuesClient.new(token: token, endpoint: "#{gitlab_url}/api/v4", project: GITLAB_PRODUCTION_PROJECT_ID)

              issues = client.find_issues(options: {
                labels: 'Incident::Active',
                state: 'opened',
                per_page: 10,
                order_by: 'created_at',
                sort: 'desc'
              })

              issues.map do |issue|
                {
                  title: issue.title,
                  url: issue.web_url
                }
              end

            rescue Gitlab::Error::Error => e
              Runtime::Logger.error "GitLab API error fetching incidents: #{e.message}"
              []
            rescue StandardError => e
              Runtime::Logger.error "Warning: Could not fetch active incidents: #{e.message}"
              []
            end
          end

          def self.format_incidents_for_issue(incidents)
            return "" if incidents.empty?

            incident_list = incidents.map { |inc| "- [#{inc[:title]}](#{inc[:url]})" }.join("\n")

            <<~MARKDOWN

        ### Related GitLab Incidents
        The following active incidents may be related to these test failures:

        #{incident_list}

        Check the [GitLab Production Issues](https://gitlab.com/gitlab-com/gl-infra/production/-/issues/?label_name%5B%5D=Incident%3A%3AActive) for updates.
            MARKDOWN
          end
        end
      end
    end
  end
end
