#!/bin/env bash

# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

set -euo pipefail

set -x

cd "$(dirname "$0")"

rm -rf build lockfiles
mkdir build
mkdir lockfiles

function lock() {
    mkdir -p build/$1
    cat >build/$1/pyproject.toml <<EOF
# AUTOGENERATED DO NOT EDIT

[tool.poetry]
name = "apache-tvm-bootstrap"
authors = []
version = "0.0.1"
description = ""

[tool.poetry.dependencies]
python = "^$1"
pip = "*"
poetry = "1.8.1"
setuptools = "*"
EOF

    # Install poetry so that the env can be locked.
    python3 -mvenv build/$1/_venv
    pwd
    . build/$1/_venv/bin/activate
    (mkdir -p build/$1/downloaded && cd build/$1/downloaded && pip3 download pip setuptools && pip3 install *.whl)
    pip3 install poetry poetry-plugin-export
    (cd build/$1 && poetry lock)

    # Now export requirements.txt and constraints.txt for
    # requirements.txt has to be generated by scanning pyproject.toml and translating the [tool.poetry.dependencies] section.
    (cd build/$1 && python3 <<EOF )
found_deps = False
requirements = []
for line in open("pyproject.toml"):
    if line.startswith("[tool.poetry.dependencies]"):
        found_deps = True
        continue
    if found_deps and "=" in line:
        package = line.split("=", 1)[0].strip()
        if package != "python":
            requirements.append(package)

with open("requirements.txt", "w") as f:
    for r in sorted(requirements):
        f.write(f"{r}\n")
EOF

    # For
    (cd build/$1 && poetry export -f constraints.txt -o constraints.txt)


    (cd build/$1 && python3 <<EOF )
import os
import pkginfo
import subprocess

with open("constraints.txt", "a") as constraints_f:
    for f in sorted(os.scandir("downloaded"), key=lambda x: x.name):
        if not f.is_file():
            continue
        p = pkginfo.get_metadata("downloaded/" + f.name)
        if p.name == "my-test-package":
            continue
        constraints_f.write(
            f"{p.name}=={p.version} {subprocess.check_output(['pip3', 'hash', '-a', 'sha256', p.filename], encoding='utf-8').split()[1]}\n")
EOF

    # Assemble the directory passed to docker
    cp build/$1/requirements.txt lockfiles/requirements-$1.txt
    cp build/$1/constraints.txt lockfiles/constraints-$1.txt
    deactivate
}

lock 3.7
lock 3.8
lock 3.9
