/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import com.google.common.annotations.VisibleForTesting;
import io.netty.channel.EventLoop;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import lombok.Generated;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.namespace.NamespaceService;
import org.apache.pulsar.broker.resources.TopicListener;
import org.apache.pulsar.broker.resources.TopicResources;
import org.apache.pulsar.broker.service.BrokerServiceException;
import org.apache.pulsar.broker.service.ServerCnx;
import org.apache.pulsar.broker.topiclistlimit.TopicListMemoryLimiter;
import org.apache.pulsar.broker.topiclistlimit.TopicListSizeResultCache;
import org.apache.pulsar.common.api.proto.CommandGetTopicsOfNamespace;
import org.apache.pulsar.common.api.proto.CommandWatchTopicListClose;
import org.apache.pulsar.common.api.proto.ServerError;
import org.apache.pulsar.common.naming.NamespaceName;
import org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.common.semaphore.AsyncDualMemoryLimiter;
import org.apache.pulsar.common.semaphore.AsyncDualMemoryLimiterImpl;
import org.apache.pulsar.common.semaphore.AsyncSemaphore;
import org.apache.pulsar.common.topics.TopicList;
import org.apache.pulsar.common.topics.TopicsPattern;
import org.apache.pulsar.common.topics.TopicsPatternFactory;
import org.apache.pulsar.common.util.Backoff;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.common.util.collections.ConcurrentLongHashMap;
import org.apache.pulsar.metadata.api.NotificationType;
import org.apache.pulsar.metadata.api.extended.SessionEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TopicListService {
    private static final Logger log = LoggerFactory.getLogger(TopicListService.class);
    private final NamespaceService namespaceService;
    private final TopicResources topicResources;
    private final PulsarService pulsar;
    private final ServerCnx connection;
    private final boolean enableSubscriptionPatternEvaluation;
    private final int maxSubscriptionPatternLength;
    private final int topicListUpdateMaxQueueSize;
    private final ConcurrentLongHashMap<CompletableFuture<TopicListWatcher>> watchers;
    private final Backoff retryBackoff;

    public TopicListService(PulsarService pulsar, ServerCnx connection, boolean enableSubscriptionPatternEvaluation, int maxSubscriptionPatternLength) {
        this(pulsar, connection, enableSubscriptionPatternEvaluation, maxSubscriptionPatternLength, 10000);
    }

    @VisibleForTesting
    public TopicListService(PulsarService pulsar, ServerCnx connection, boolean enableSubscriptionPatternEvaluation, int maxSubscriptionPatternLength, int topicListUpdateMaxQueueSize) {
        this.namespaceService = pulsar.getNamespaceService();
        this.pulsar = pulsar;
        this.connection = connection;
        this.enableSubscriptionPatternEvaluation = enableSubscriptionPatternEvaluation;
        this.maxSubscriptionPatternLength = maxSubscriptionPatternLength;
        this.topicListUpdateMaxQueueSize = topicListUpdateMaxQueueSize;
        this.watchers = ConcurrentLongHashMap.newBuilder().expectedItems(8).concurrencyLevel(1).build();
        this.topicResources = pulsar.getPulsarResources().getTopicResources();
        this.retryBackoff = new Backoff(100L, TimeUnit.MILLISECONDS, 25L, TimeUnit.SECONDS, 0L, TimeUnit.MILLISECONDS);
    }

    public void inactivate() {
        for (Long watcherId : new HashSet(this.watchers.keys())) {
            this.deleteTopicListWatcher(watcherId);
        }
    }

    public void handleWatchTopicList(NamespaceName namespaceName, long watcherId, long requestId, String topicsPatternString, TopicsPattern.RegexImplementation topicsPatternRegexImplementation, String topicsHash, Semaphore lookupSemaphore) {
        TopicsPattern topicsPattern;
        topicsPatternString = TopicList.removeTopicDomainScheme((String)topicsPatternString);
        if (!this.enableSubscriptionPatternEvaluation || topicsPatternString.length() > this.maxSubscriptionPatternLength) {
            Object msg = "Unable to create topic list watcher: ";
            msg = !this.enableSubscriptionPatternEvaluation ? (String)msg + "Evaluating subscription patterns is disabled." : (String)msg + "Pattern longer than maximum: " + this.maxSubscriptionPatternLength;
            log.warn("[{}] {} on namespace {}", new Object[]{this.connection.toString(), msg, namespaceName});
            this.connection.getCommandSender().sendErrorResponse(requestId, ServerError.NotAllowedError, (String)msg);
            lookupSemaphore.release();
            return;
        }
        try {
            topicsPattern = TopicsPatternFactory.create((String)topicsPatternString, (TopicsPattern.RegexImplementation)topicsPatternRegexImplementation);
        }
        catch (Exception e) {
            log.warn("[{}] Unable to create topic list watcher: Invalid pattern: {} on namespace {}", new Object[]{this.connection.toString(), topicsPatternString, namespaceName});
            this.connection.getCommandSender().sendErrorResponse(requestId, ServerError.InvalidTopicName, "Invalid topics pattern: " + e.getMessage());
            lookupSemaphore.release();
            return;
        }
        CompletionStage<TopicListWatcher> watcherFuture = new CompletableFuture<TopicListWatcher>();
        CompletableFuture existingWatcherFuture = (CompletableFuture)this.watchers.putIfAbsent(watcherId, watcherFuture);
        if (existingWatcherFuture != null) {
            if (log.isDebugEnabled()) {
                log.debug("[{}] Watcher with the same watcherId={} is already created. Refreshing.", (Object)this.connection, (Object)watcherId);
            }
            watcherFuture = existingWatcherFuture.thenCompose(watcher -> {
                CompletableFuture future = new CompletableFuture();
                Runnable callback = () -> future.complete(watcher);
                if (watcher.prepareUpdateTopics(callback)) {
                    this.updateTopicListWatcher((TopicListWatcher)watcher).exceptionally(ex -> {
                        callback.run();
                        return null;
                    });
                }
                return future;
            });
        } else {
            this.initializeTopicsListWatcher((CompletableFuture<TopicListWatcher>)watcherFuture, namespaceName, watcherId, topicsPattern);
        }
        CompletableFuture<TopicListWatcher> finalWatcherFuture = watcherFuture;
        ((CompletableFuture)finalWatcherFuture.thenAccept(watcher -> {
            Collection<String> topicList = watcher.getMatchingTopics();
            String hash = TopicList.calculateHash(topicList);
            if (hash.equals(topicsHash)) {
                topicList = Collections.emptyList();
            }
            if (log.isDebugEnabled()) {
                log.debug("[{}] Received WatchTopicList for namespace [//{}] by {}", new Object[]{this.connection.toString(), namespaceName, requestId});
            }
            this.sendTopicListSuccessWithPermitAcquiringRetries(watcherId, requestId, topicList, hash, watcher::sendingCompleted, watcher::close);
            lookupSemaphore.release();
        })).exceptionally(ex -> {
            log.warn("[{}] Error WatchTopicList for namespace [//{}] by {}: {}", new Object[]{this.connection.toString(), namespaceName, requestId, ex.getMessage()});
            this.connection.getCommandSender().sendErrorResponse(requestId, BrokerServiceException.getClientErrorCode(new BrokerServiceException.ServerMetadataException((Throwable)ex)), ex.getMessage());
            this.watchers.remove(watcherId, (Object)finalWatcherFuture);
            lookupSemaphore.release();
            return null;
        });
    }

    private void sendTopicListSuccessWithPermitAcquiringRetries(long watcherId, long requestId, Collection<String> topicList, String hash, Runnable successfulCompletionCallback, Runnable failedCompletionCallback) {
        this.performOperationWithPermitAcquiringRetries(watcherId, "topic list success", permitAcquireErrorHandler -> () -> this.connection.getCommandSender().sendWatchTopicListSuccess(requestId, watcherId, hash, topicList, (Function<Throwable, CompletableFuture<Void>>)permitAcquireErrorHandler).whenComplete((__, t) -> {
            if (t != null) {
                log.warn("[{}] Failed to send topic list success for watcherId={}. Watcher is not active.", new Object[]{this.connection, watcherId, t});
                failedCompletionCallback.run();
            } else {
                successfulCompletionCallback.run();
            }
        }));
    }

    public void initializeTopicsListWatcher(CompletableFuture<TopicListWatcher> watcherFuture, NamespaceName namespace, long watcherId, TopicsPattern topicsPattern) {
        AtomicReference watcherRef = new AtomicReference();
        Consumer<List<String>> afterListing = topics -> {
            TopicListWatcher watcher = new TopicListWatcher(this, watcherId, namespace, topicsPattern, (List<String>)topics, (Executor)this.connection.ctx().executor(), this.topicListUpdateMaxQueueSize);
            watcherRef.set(watcher);
            this.topicResources.registerPersistentTopicListener((TopicListener)watcher);
        };
        this.getTopics(namespace, watcherId, afterListing).whenComplete((topics, exception) -> {
            TopicListWatcher w = (TopicListWatcher)watcherRef.get();
            if (exception != null) {
                if (w != null) {
                    w.close();
                    this.topicResources.deregisterPersistentTopicListener((TopicListener)w);
                }
                Throwable unwrappedException = FutureUtil.unwrapCompletionException((Throwable)exception);
                if (this.connection.isActive() && (unwrappedException instanceof AsyncSemaphore.PermitAcquireTimeoutException || unwrappedException instanceof AsyncSemaphore.PermitAcquireQueueFullException)) {
                    long retryAfterMillis = this.retryBackoff.next();
                    log.info("[{}] {} when initializing topic list watcher watcherId={} for namespace {}. Retrying in {} ms.", new Object[]{this.connection, unwrappedException.getMessage(), watcherId, namespace, retryAfterMillis});
                    this.connection.ctx().executor().schedule(() -> this.initializeTopicsListWatcher(watcherFuture, namespace, watcherId, topicsPattern), retryAfterMillis, TimeUnit.MILLISECONDS);
                } else {
                    log.warn("[{}] Failed to initialize topic list watcher watcherId={} for namespace {}.", new Object[]{this.connection, watcherId, namespace, unwrappedException});
                    watcherFuture.completeExceptionally(unwrappedException);
                }
            } else if (!watcherFuture.complete(w)) {
                log.warn("[{}] Watcher future was already completed. Deregistering watcherId={}.", (Object)this.connection, (Object)watcherId);
                w.close();
                this.topicResources.deregisterPersistentTopicListener((TopicListener)w);
                this.watchers.remove(watcherId, (Object)watcherFuture);
            }
        });
    }

    private CompletableFuture<List<String>> getTopics(NamespaceName namespace, long watcherId) {
        return this.getTopics(namespace, watcherId, null);
    }

    private CompletableFuture<List<String>> getTopics(NamespaceName namespace, long watcherId, Consumer<List<String>> afterListing) {
        BooleanSupplier isPermitRequestCancelled = () -> !this.connection.isActive() || !this.watchers.containsKey(watcherId);
        if (isPermitRequestCancelled.getAsBoolean()) {
            return CompletableFuture.failedFuture((Throwable)new AsyncSemaphore.PermitAcquireCancelledException("Permit acquisition was cancelled"));
        }
        return this.getTopics(namespace, afterListing, isPermitRequestCancelled);
    }

    private CompletableFuture<List<String>> getTopics(NamespaceName namespace, Consumer<List<String>> afterListing, BooleanSupplier isPermitRequestCancelled) {
        TopicListSizeResultCache.ResultHolder listSizeHolder = this.pulsar.getBrokerService().getTopicListSizeResultCache().getTopicListSize(namespace.toString(), CommandGetTopicsOfNamespace.Mode.PERSISTENT);
        AsyncDualMemoryLimiterImpl maxTopicListInFlightLimiter = this.pulsar.getBrokerService().getMaxTopicListInFlightLimiter();
        return listSizeHolder.getSizeAsync().thenCompose(arg_0 -> this.lambda$getTopics$15((AsyncDualMemoryLimiter)maxTopicListInFlightLimiter, isPermitRequestCancelled, namespace, listSizeHolder, afterListing, arg_0));
    }

    CompletableFuture<Void> updateTopicListWatcher(TopicListWatcher watcher) {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        try {
            this.internalUpdateTopicListWatcher(watcher, future);
        }
        catch (Exception e) {
            future.completeExceptionally(e);
        }
        return future;
    }

    void internalUpdateTopicListWatcher(TopicListWatcher watcher, CompletableFuture<Void> future) {
        NamespaceName namespace = watcher.namespace;
        long watcherId = watcher.id;
        this.getTopics(namespace, watcherId).whenComplete((topics, exception) -> {
            if (exception != null) {
                Throwable unwrappedException = FutureUtil.unwrapCompletionException((Throwable)exception);
                if (this.connection.isActive() && !watcher.closed && (unwrappedException instanceof AsyncSemaphore.PermitAcquireTimeoutException || unwrappedException instanceof AsyncSemaphore.PermitAcquireQueueFullException)) {
                    long retryAfterMillis = this.retryBackoff.next();
                    log.info("[{}] {} when updating topic list watcher watcherId={} for namespace {}. Retrying in {} ms.", new Object[]{this.connection, unwrappedException.getMessage(), watcherId, namespace, retryAfterMillis});
                    this.connection.ctx().executor().schedule(() -> this.internalUpdateTopicListWatcher(watcher, future), retryAfterMillis, TimeUnit.MILLISECONDS);
                } else {
                    log.warn("[{}] Failed to update topic list watcher watcherId={} for namespace {}.", new Object[]{this.connection, watcherId, namespace, unwrappedException});
                    future.completeExceptionally(unwrappedException);
                }
            } else {
                watcher.updateTopics((List<String>)topics);
                future.complete(null);
            }
        });
    }

    public void handleWatchTopicListClose(CommandWatchTopicListClose commandWatchTopicListClose) {
        long requestId = commandWatchTopicListClose.getRequestId();
        long watcherId = commandWatchTopicListClose.getWatcherId();
        this.deleteTopicListWatcher(watcherId);
        this.connection.getCommandSender().sendSuccessResponse(requestId);
    }

    public void deleteTopicListWatcher(Long watcherId) {
        CompletableFuture watcherFuture = (CompletableFuture)this.watchers.remove(watcherId.longValue());
        if (watcherFuture == null) {
            log.info("[{}] TopicListWatcher was not registered on the connection: {}", (Object)watcherId, (Object)this.connection.toString());
            return;
        }
        if (!watcherFuture.isDone() && watcherFuture.completeExceptionally(new IllegalStateException("Closed watcher before creation was complete"))) {
            log.info("[{}] Closed watcher before its creation was completed. watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
            return;
        }
        watcherFuture.whenComplete((watcher, t) -> {
            if (watcher != null) {
                this.topicResources.deregisterPersistentTopicListener((TopicListener)watcher);
                watcher.close();
                log.info("[{}] Closed watcher, watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
            } else if (t != null) {
                log.info("[{}] Closed watcher that failed to be created. watcherId={}", (Object)this.connection.toString(), (Object)watcherId);
            }
        });
    }

    public void sendTopicListUpdate(long watcherId, String topicsHash, List<String> deletedTopics, List<String> newTopics, Runnable completionCallback) {
        this.performOperationWithPermitAcquiringRetries(watcherId, "topic list update", permitAcquireErrorHandler -> () -> this.connection.getCommandSender().sendWatchTopicListUpdate(watcherId, newTopics, deletedTopics, topicsHash, (Function<Throwable, CompletableFuture<Void>>)permitAcquireErrorHandler).whenComplete((__, t) -> {
            if (t != null) {
                log.warn("[{}] Failed to send topic list update for watcherId={}. Watcher will be in inconsistent state.", new Object[]{this.connection, watcherId, t});
            }
            completionCallback.run();
        }));
    }

    private void performOperationWithPermitAcquiringRetries(long watcherId, String operationName, Function<Function<Throwable, CompletableFuture<Void>>, Supplier<CompletableFuture<Void>>> asyncOperationFactory) {
        AtomicReference<Supplier<CompletableFuture>> operationRef = new AtomicReference<Supplier<CompletableFuture>>();
        Function<Throwable, CompletableFuture<Void>> permitAcquireErrorHandler = this.createPermitAcquireErrorHandler(watcherId, operationName, () -> (CompletableFuture)((Supplier)operationRef.get()).get());
        Supplier<CompletableFuture<Void>> asyncOperation = asyncOperationFactory.apply(permitAcquireErrorHandler);
        operationRef.set(() -> {
            if (!this.connection.isActive() || !this.watchers.containsKey(watcherId)) {
                return CompletableFuture.completedFuture(null);
            }
            return ((CompletableFuture)asyncOperation.get()).thenRun(() -> this.retryBackoff.reset());
        });
        ((Supplier)operationRef.get()).get();
    }

    private Function<Throwable, CompletableFuture<Void>> createPermitAcquireErrorHandler(long watcherId, String operationName, Supplier<CompletableFuture<Void>> operationRef) {
        EventLoop scheduledExecutor = this.connection.ctx().channel().eventLoop();
        AtomicInteger retryCount = new AtomicInteger(0);
        return arg_0 -> this.lambda$createPermitAcquireErrorHandler$26(watcherId, retryCount, operationName, (ScheduledExecutorService)scheduledExecutor, operationRef, arg_0);
    }

    @VisibleForTesting
    CompletableFuture<TopicListWatcher> getWatcherFuture(long watcherId) {
        return (CompletableFuture)this.watchers.get(watcherId);
    }

    private /* synthetic */ CompletableFuture lambda$createPermitAcquireErrorHandler$26(long watcherId, AtomicInteger retryCount, String operationName, ScheduledExecutorService scheduledExecutor, Supplier operationRef, Throwable t) {
        Throwable unwrappedException = FutureUtil.unwrapCompletionException((Throwable)t);
        if (unwrappedException instanceof AsyncSemaphore.PermitAcquireCancelledException || unwrappedException instanceof AsyncSemaphore.PermitAcquireAlreadyClosedException || !this.connection.isActive() || !this.watchers.containsKey(watcherId)) {
            return CompletableFuture.completedFuture(null);
        }
        long retryDelay = this.retryBackoff.next();
        retryCount.incrementAndGet();
        log.info("[{}] Cannot acquire direct memory tokens for sending {}. Retry {} in {} ms. {}", new Object[]{this.connection, operationName, retryCount.get(), retryDelay, t.getMessage()});
        CompletableFuture future = new CompletableFuture();
        scheduledExecutor.schedule(() -> TopicListService.lambda$createPermitAcquireErrorHandler$25(future, (Supplier)operationRef), retryDelay, TimeUnit.MILLISECONDS);
        return future;
    }

    private static /* synthetic */ void lambda$createPermitAcquireErrorHandler$25(CompletableFuture future, Supplier operationRef) {
        FutureUtil.completeAfter((CompletableFuture)future, (CompletableFuture)((CompletableFuture)operationRef.get()));
    }

    private /* synthetic */ CompletionStage lambda$getTopics$15(AsyncDualMemoryLimiter maxTopicListInFlightLimiter, BooleanSupplier isPermitRequestCancelled, NamespaceName namespace, TopicListSizeResultCache.ResultHolder listSizeHolder, Consumer afterListing, Long initialSize) {
        return maxTopicListInFlightLimiter.withAcquiredPermits(initialSize.longValue(), AsyncDualMemoryLimiter.LimitType.HEAP_MEMORY, isPermitRequestCancelled, initialPermits -> this.namespaceService.getListOfUserTopics(namespace, CommandGetTopicsOfNamespace.Mode.PERSISTENT).thenComposeAsync(topics -> {
            long actualSize = TopicListMemoryLimiter.estimateTopicListSize((List)topics);
            listSizeHolder.updateSize(actualSize);
            if (afterListing != null) {
                afterListing.accept(topics);
            }
            if (initialSize != actualSize) {
                return maxTopicListInFlightLimiter.withUpdatedPermits(initialPermits, actualSize, isPermitRequestCancelled, updatedPermits -> {
                    this.retryBackoff.reset();
                    return CompletableFuture.completedFuture(topics);
                }, CompletableFuture::failedFuture);
            }
            this.retryBackoff.reset();
            return CompletableFuture.completedFuture(topics);
        }, (Executor)this.connection.ctx().executor()), CompletableFuture::failedFuture).thenApplyAsync(Function.identity(), (Executor)this.connection.ctx().executor());
    }

    public static class TopicListWatcher
    implements TopicListener {
        private static final int DEFAULT_TOPIC_LIST_UPDATE_MAX_QUEUE_SIZE = 10000;
        private final Set<String> matchingTopics;
        private final TopicListService topicListService;
        private final long id;
        private final NamespaceName namespace;
        private final TopicsPattern topicsPattern;
        private final Executor executor;
        private volatile boolean closed = false;
        private boolean sendingInProgress;
        private final BlockingDeque<Runnable> sendTopicListUpdateTasks;
        private boolean updatingTopics;
        private List<String> matchingTopicsBeforeDisconnected;
        private boolean disconnected;
        private List<Runnable> updateCallbacks = new LinkedList<Runnable>();
        private boolean updatingWhileDisconnected;

        public TopicListWatcher(TopicListService topicListService, long id, NamespaceName namespace, TopicsPattern topicsPattern, List<String> topics, Executor executor, int topicListUpdateMaxQueueSize) {
            this.topicListService = topicListService;
            this.id = id;
            this.namespace = namespace;
            this.topicsPattern = topicsPattern;
            this.executor = executor;
            this.matchingTopics = (Set)TopicList.filterTopics(topics, (TopicsPattern)topicsPattern, Collectors.toCollection(LinkedHashSet::new));
            this.sendingInProgress = true;
            this.sendTopicListUpdateTasks = new LinkedBlockingDeque<Runnable>(topicListUpdateMaxQueueSize);
        }

        public synchronized Collection<String> getMatchingTopics() {
            return new ArrayList<String>(this.matchingTopics);
        }

        public synchronized void onTopicEvent(String topicName, NotificationType notificationType) {
            if (this.closed) {
                return;
            }
            String partitionedTopicName = TopicName.get((String)topicName).getPartitionedTopicName();
            String domainLessTopicName = TopicList.removeTopicDomainScheme((String)partitionedTopicName);
            if (this.topicsPattern.matches(domainLessTopicName)) {
                List<String> newTopics = Collections.emptyList();
                List<String> deletedTopics = Collections.emptyList();
                if (notificationType == NotificationType.Deleted) {
                    if (this.matchingTopics.remove(topicName)) {
                        deletedTopics = Collections.singletonList(topicName);
                    }
                } else if (notificationType == NotificationType.Created && this.matchingTopics.add(topicName)) {
                    newTopics = Collections.singletonList(topicName);
                }
                if (!newTopics.isEmpty() || !deletedTopics.isEmpty()) {
                    String hash = TopicList.calculateHash(this.matchingTopics);
                    this.sendTopicListUpdate(hash, deletedTopics, newTopics);
                }
            }
        }

        private synchronized void sendTopicListUpdate(String hash, List<String> deletedTopics, List<String> newTopics) {
            if (this.closed) {
                return;
            }
            Runnable task = () -> this.topicListService.sendTopicListUpdate(this.id, hash, deletedTopics, newTopics, this::sendingCompleted);
            if (!this.sendingInProgress) {
                this.sendingInProgress = true;
                this.executor.execute(task);
            } else if (!this.sendTopicListUpdateTasks.offer(task) && this.prepareUpdateTopics(null)) {
                log.warn("Update queue was full for watcher id {} matching {}. Performing full refresh.", (Object)this.id, (Object)this.topicsPattern.inputPattern());
                this.executor.execute(() -> this.topicListService.updateTopicListWatcher(this));
            }
        }

        @VisibleForTesting
        synchronized void sendingCompleted() {
            if (this.closed) {
                this.sendTopicListUpdateTasks.clear();
                return;
            }
            Runnable task = this.sendTopicListUpdateTasks.poll();
            if (task != null) {
                this.executor.execute(task);
            } else {
                this.sendingInProgress = false;
            }
        }

        public synchronized void close() {
            this.closed = true;
            this.sendTopicListUpdateTasks.clear();
            this.updateCallbacks.clear();
        }

        synchronized boolean prepareUpdateTopics(Runnable afterUpdateCompletionCallback) {
            if (!this.updatingTopics) {
                this.updatingTopics = true;
                this.sendingInProgress = true;
                this.sendTopicListUpdateTasks.clear();
                this.matchingTopics.clear();
                if (afterUpdateCompletionCallback != null) {
                    this.updateCallbacks.add(afterUpdateCompletionCallback);
                }
                return true;
            }
            if (afterUpdateCompletionCallback != null) {
                this.updateCallbacks.add(afterUpdateCompletionCallback);
            }
            return false;
        }

        synchronized void updateTopics(List<String> topics) {
            if (this.closed) {
                return;
            }
            this.matchingTopics.clear();
            TopicList.filterTopicsToStream(topics, (TopicsPattern)this.topicsPattern).forEach(this.matchingTopics::add);
            this.updatingTopics = false;
            if (this.disconnected) {
                this.handleNewAndDeletedTopicsWhileDisconnected();
                this.matchingTopicsBeforeDisconnected = null;
                this.disconnected = false;
            }
            for (Runnable callback : this.updateCallbacks) {
                try {
                    callback.run();
                }
                catch (Exception e) {
                    log.warn("Error executing topic list update callback: {}", (Object)callback, (Object)e);
                }
            }
            this.updateCallbacks.clear();
            this.sendingCompleted();
        }

        private synchronized void handleNewAndDeletedTopicsWhileDisconnected() {
            if (this.matchingTopicsBeforeDisconnected == null) {
                return;
            }
            ArrayList<String> newTopics = new ArrayList<String>();
            ArrayList<String> deletedTopics = new ArrayList<String>();
            HashSet<String> remainingTopics = new HashSet<String>(this.matchingTopics);
            for (String topic : this.matchingTopicsBeforeDisconnected) {
                if (remainingTopics.remove(topic)) continue;
                deletedTopics.add(topic);
            }
            newTopics.addAll(remainingTopics);
            if (!newTopics.isEmpty() || !deletedTopics.isEmpty()) {
                String hash = TopicList.calculateHash(this.matchingTopics);
                this.sendTopicListUpdate(hash, deletedTopics, newTopics);
            }
        }

        public NamespaceName getNamespaceName() {
            return this.namespace;
        }

        public synchronized void onSessionEvent(SessionEvent event) {
            switch (event) {
                case SessionReestablished: 
                case Reconnected: {
                    this.executor.execute(() -> {
                        TopicListWatcher topicListWatcher = this;
                        synchronized (topicListWatcher) {
                            if (!this.updatingWhileDisconnected) {
                                this.updatingWhileDisconnected = true;
                                CompletableFuture<Void> future = this.topicListService.updateTopicListWatcher(this);
                                future.whenComplete((__, ___) -> {
                                    TopicListWatcher topicListWatcher = this;
                                    synchronized (topicListWatcher) {
                                        this.updatingWhileDisconnected = false;
                                    }
                                });
                            }
                        }
                    });
                    break;
                }
                case SessionLost: 
                case ConnectionLost: {
                    if (this.disconnected) break;
                    this.disconnected = true;
                    this.matchingTopicsBeforeDisconnected = new ArrayList<String>(this.matchingTopics);
                    this.prepareUpdateTopics(null);
                }
            }
        }

        @VisibleForTesting
        @Generated
        public boolean isUpdatingTopics() {
            return this.updatingTopics;
        }
    }
}

