/*
 * Decompiled with CFR 0.152.
 */
package org.apache.orc.tools;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.orc.ColumnStatistics;
import org.apache.orc.CompressionKind;
import org.apache.orc.OrcFile;
import org.apache.orc.OrcProto;
import org.apache.orc.Reader;
import org.apache.orc.StripeInformation;
import org.apache.orc.StripeStatistics;
import org.apache.orc.TypeDescription;
import org.apache.orc.Writer;
import org.apache.orc.impl.ColumnStatisticsImpl;
import org.apache.orc.impl.OrcAcidUtils;
import org.apache.orc.impl.OrcIndex;
import org.apache.orc.impl.RecordReaderImpl;
import org.apache.orc.tools.JsonFileDump;
import org.apache.orc.tools.PrintData;
import org.apache.orc.util.BloomFilter;
import org.apache.orc.util.BloomFilterIO;

public final class FileDump {
    public static final String UNKNOWN = "UNKNOWN";
    public static final String SEPARATOR = StringUtils.repeat("_", 120) + "\n";
    public static final int DEFAULT_BLOCK_SIZE = 0x10000000;
    public static final String DEFAULT_BACKUP_PATH = System.getProperty("java.io.tmpdir");
    public static final PathFilter HIDDEN_AND_SIDE_FILE_FILTER = new PathFilter(){

        public boolean accept(Path p) {
            String name = p.getName();
            return !name.startsWith("_") && !name.startsWith(".") && !name.endsWith("_flush_length");
        }
    };

    private FileDump() {
    }

    public static void main(Configuration conf, String[] args) throws Exception {
        ArrayList<Integer> rowIndexCols = new ArrayList<Integer>(0);
        Options opts = FileDump.createOptions();
        CommandLine cli = new GnuParser().parse(opts, args);
        if (cli.hasOption('h')) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("orcfiledump", opts);
            return;
        }
        boolean dumpData = cli.hasOption('d');
        boolean recover = cli.hasOption("recover");
        boolean skipDump = cli.hasOption("skip-dump");
        String backupPath = DEFAULT_BACKUP_PATH;
        if (cli.hasOption("backup-path")) {
            backupPath = cli.getOptionValue("backup-path");
        }
        if (cli.hasOption("r")) {
            String val = cli.getOptionValue("r");
            if (val != null && val.trim().equals("*")) {
                rowIndexCols = null;
            } else {
                String[] colStrs = cli.getOptionValue("r").split(",");
                rowIndexCols = new ArrayList(colStrs.length);
                for (String colStr : colStrs) {
                    rowIndexCols.add(Integer.parseInt(colStr));
                }
            }
        }
        boolean printTimeZone = cli.hasOption('t');
        boolean jsonFormat = cli.hasOption('j');
        String[] files = cli.getArgs();
        if (files.length == 0) {
            System.err.println("Error : ORC files are not specified");
            return;
        }
        ArrayList<String> filesInPath = new ArrayList<String>();
        for (String filename : files) {
            Path path = new Path(filename);
            filesInPath.addAll(FileDump.getAllFilesInPath(path, conf));
        }
        if (dumpData) {
            PrintData.main(conf, filesInPath.toArray(new String[filesInPath.size()]));
        } else if (recover && skipDump) {
            FileDump.recoverFiles(filesInPath, conf, backupPath);
        } else if (jsonFormat) {
            boolean prettyPrint = cli.hasOption('p');
            JsonFileDump.printJsonMetaData(filesInPath, conf, rowIndexCols, prettyPrint, printTimeZone);
        } else {
            FileDump.printMetaData(filesInPath, conf, rowIndexCols, printTimeZone, recover, backupPath);
        }
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        FileDump.main(conf, args);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    static Reader getReader(Path path, Configuration conf, List<String> corruptFiles) throws IOException {
        FileSystem fs = path.getFileSystem(conf);
        long dataFileLen = fs.getFileStatus(path).getLen();
        System.err.println("Processing data file " + path + " [length: " + dataFileLen + "]");
        Path sideFile = OrcAcidUtils.getSideFile(path);
        boolean sideFileExists = fs.exists(sideFile);
        boolean openDataFile = false;
        boolean openSideFile = false;
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            openDataFile = !dfs.isFileClosed(path);
            boolean bl = openSideFile = sideFileExists && !dfs.isFileClosed(sideFile);
        }
        if (openDataFile || openSideFile) {
            if (openDataFile && openSideFile) {
                System.err.println("Unable to perform file dump as " + path + " and " + sideFile + " are still open for writes.");
                return null;
            } else if (openSideFile) {
                System.err.println("Unable to perform file dump as " + sideFile + " is still open for writes.");
                return null;
            } else {
                System.err.println("Unable to perform file dump as " + path + " is still open for writes.");
            }
            return null;
        }
        Reader reader = null;
        if (!sideFileExists) return OrcFile.createReader(path, OrcFile.readerOptions(conf));
        long maxLen = OrcAcidUtils.getLastFlushLength(fs, path);
        long sideFileLen = fs.getFileStatus(sideFile).getLen();
        System.err.println("Found flush length file " + sideFile + " [length: " + sideFileLen + ", maxFooterOffset: " + maxLen + "]");
        if (maxLen == -1L) {
            if (dataFileLen <= maxLen) return null;
            System.err.println("Data file has more data than max footer offset:" + maxLen + ". Adding data file to recovery list.");
            if (corruptFiles == null) return null;
            corruptFiles.add(path.toUri().toString());
            return null;
        }
        try {
            reader = OrcFile.createReader(path, OrcFile.readerOptions(conf).maxLength(maxLen));
            if (dataFileLen <= maxLen) return reader;
            System.err.println("Data file has more data than max footer offset:" + maxLen + ". Adding data file to recovery list.");
            if (corruptFiles == null) return reader;
            corruptFiles.add(path.toUri().toString());
            return reader;
        }
        catch (Exception e) {
            if (corruptFiles != null) {
                corruptFiles.add(path.toUri().toString());
            }
            System.err.println("Unable to read data from max footer offset. Adding data file to recovery list.");
            return null;
        }
    }

    public static Collection<String> getAllFilesInPath(Path path, Configuration conf) throws IOException {
        ArrayList<String> filesInPath = new ArrayList<String>();
        FileSystem fs = path.getFileSystem(conf);
        FileStatus fileStatus = fs.getFileStatus(path);
        if (fileStatus.isDir()) {
            FileStatus[] fileStatuses;
            for (FileStatus fileInPath : fileStatuses = fs.listStatus(path, HIDDEN_AND_SIDE_FILE_FILTER)) {
                if (fileInPath.isDir()) {
                    filesInPath.addAll(FileDump.getAllFilesInPath(fileInPath.getPath(), conf));
                    continue;
                }
                filesInPath.add(fileInPath.getPath().toString());
            }
        } else {
            filesInPath.add(path.toString());
        }
        return filesInPath;
    }

    private static void printMetaData(List<String> files, Configuration conf, List<Integer> rowIndexCols, boolean printTimeZone, boolean recover, String backupPath) throws IOException {
        ArrayList<String> corruptFiles = new ArrayList<String>();
        for (String filename : files) {
            FileDump.printMetaDataImpl(filename, conf, rowIndexCols, printTimeZone, corruptFiles);
            System.out.println(SEPARATOR);
        }
        if (!corruptFiles.isEmpty()) {
            if (recover) {
                FileDump.recoverFiles(corruptFiles, conf, backupPath);
            } else {
                System.err.println(corruptFiles.size() + " file(s) are corrupted. Run the following command to recover corrupted files.\n");
                StringBuilder buffer = new StringBuilder();
                buffer.append("hive --orcfiledump --recover --skip-dump");
                for (String file : corruptFiles) {
                    buffer.append(' ');
                    buffer.append(file);
                }
                System.err.println(buffer.toString());
                System.out.println(SEPARATOR);
            }
        }
    }

    private static void printMetaDataImpl(String filename, Configuration conf, List<Integer> rowIndexCols, boolean printTimeZone, List<String> corruptFiles) throws IOException {
        int i;
        Path file = new Path(filename);
        Reader reader = FileDump.getReader(file, conf, corruptFiles);
        if (reader == null) {
            return;
        }
        TypeDescription schema = reader.getSchema();
        System.out.println("Structure for " + filename);
        System.out.println("File Version: " + reader.getFileVersion().getName() + " with " + (Object)((Object)reader.getWriterVersion()));
        RecordReaderImpl rows = (RecordReaderImpl)reader.rows();
        System.out.println("Rows: " + reader.getNumberOfRows());
        System.out.println("Compression: " + (Object)((Object)reader.getCompressionKind()));
        if (reader.getCompressionKind() != CompressionKind.NONE) {
            System.out.println("Compression size: " + reader.getCompressionSize());
        }
        System.out.println("Type: " + reader.getSchema().toString());
        System.out.println("\nStripe Statistics:");
        List<StripeStatistics> stripeStats = reader.getStripeStatistics();
        for (int n = 0; n < stripeStats.size(); ++n) {
            System.out.println("  Stripe " + (n + 1) + ":");
            StripeStatistics ss = stripeStats.get(n);
            for (i = 0; i < ss.getColumnStatistics().length; ++i) {
                System.out.println("    Column " + i + ": " + ss.getColumnStatistics()[i].toString());
            }
        }
        ColumnStatistics[] stats = reader.getStatistics();
        int colCount = stats.length;
        if (rowIndexCols == null) {
            rowIndexCols = new ArrayList<Integer>(colCount);
            for (i = 0; i < colCount; ++i) {
                rowIndexCols.add(i);
            }
        }
        System.out.println("\nFile Statistics:");
        for (i = 0; i < stats.length; ++i) {
            System.out.println("  Column " + i + ": " + stats[i].toString());
        }
        System.out.println("\nStripes:");
        int stripeIx = -1;
        for (StripeInformation stripe : reader.getStripes()) {
            Object encoding;
            ++stripeIx;
            long stripeStart = stripe.getOffset();
            OrcProto.StripeFooter footer = rows.readStripeFooter(stripe);
            if (printTimeZone) {
                String tz = footer.getWriterTimezone();
                if (tz == null || tz.isEmpty()) {
                    tz = UNKNOWN;
                }
                System.out.println("  Stripe: " + stripe.toString() + " timezone: " + tz);
            } else {
                System.out.println("  Stripe: " + stripe.toString());
            }
            long sectionStart = stripeStart;
            for (OrcProto.Stream section : footer.getStreamsList()) {
                String kind = section.hasKind() ? section.getKind().name() : UNKNOWN;
                System.out.println("    Stream: column " + section.getColumn() + " section " + kind + " start: " + sectionStart + " length " + section.getLength());
                sectionStart += section.getLength();
            }
            for (int i2 = 0; i2 < footer.getColumnsCount(); ++i2) {
                encoding = footer.getColumns(i2);
                StringBuilder buf = new StringBuilder();
                buf.append("    Encoding column ");
                buf.append(i2);
                buf.append(": ");
                buf.append(((OrcProto.ColumnEncoding)encoding).getKind());
                if (((OrcProto.ColumnEncoding)encoding).getKind() == OrcProto.ColumnEncoding.Kind.DICTIONARY || ((OrcProto.ColumnEncoding)encoding).getKind() == OrcProto.ColumnEncoding.Kind.DICTIONARY_V2) {
                    buf.append("[");
                    buf.append(((OrcProto.ColumnEncoding)encoding).getDictionarySize());
                    buf.append("]");
                }
                System.out.println(buf);
            }
            if (rowIndexCols == null || rowIndexCols.isEmpty()) continue;
            boolean[] sargColumns = new boolean[colCount];
            encoding = rowIndexCols.iterator();
            while (encoding.hasNext()) {
                int colIdx = (Integer)encoding.next();
                sargColumns[colIdx] = true;
            }
            OrcIndex indices = rows.readRowIndex(stripeIx, null, null, null, sargColumns);
            for (int col : rowIndexCols) {
                StringBuilder buf = new StringBuilder();
                String rowIdxString = FileDump.getFormattedRowIndices(col, indices.getRowGroupIndex(), schema);
                buf.append(rowIdxString);
                String bloomFilString = FileDump.getFormattedBloomFilters(col, indices, reader.getWriterVersion(), reader.getSchema().findSubtype(col).getCategory(), footer.getColumns(col));
                buf.append(bloomFilString);
                System.out.println(buf);
            }
        }
        FileSystem fs = file.getFileSystem(conf);
        long fileLen = fs.getFileStatus(file).getLen();
        long paddedBytes = FileDump.getTotalPaddingSize(reader);
        double percentPadding = (double)paddedBytes / (double)fileLen * 100.0;
        DecimalFormat format = new DecimalFormat("##.##");
        System.out.println("\nFile length: " + fileLen + " bytes");
        System.out.println("Padding length: " + paddedBytes + " bytes");
        System.out.println("Padding ratio: " + format.format(percentPadding) + "%");
        List<String> keys = reader.getMetadataKeys();
        for (int i3 = 0; i3 < keys.size(); ++i3) {
            if (i3 == 0) {
                System.out.println("\nUser Metadata:");
            }
            ByteBuffer byteBuffer = reader.getMetadataValue(keys.get(i3));
            System.out.println("  " + keys.get(i3) + "=" + StandardCharsets.UTF_8.decode(byteBuffer));
        }
        rows.close();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void recoverFiles(List<String> corruptFiles, Configuration conf, String backup) throws IOException {
        for (String corruptFile : corruptFiles) {
            System.err.println("Recovering file " + corruptFile);
            Path corruptPath = new Path(corruptFile);
            FileSystem fs = corruptPath.getFileSystem(conf);
            FSDataInputStream fdis = fs.open(corruptPath);
            try {
                long corruptFileLen;
                int toRead;
                ArrayList<Long> footerOffsets = new ArrayList<Long>();
                for (long remaining = corruptFileLen = fs.getFileStatus(corruptPath).getLen(); remaining > 0L; remaining -= (long)toRead) {
                    toRead = (int)Math.min(0x10000000L, remaining);
                    byte[] data = new byte[toRead];
                    long startPos = corruptFileLen - remaining;
                    fdis.readFully(startPos, data, 0, toRead);
                    int index = 0;
                    byte[] magicBytes = "ORC".getBytes(StandardCharsets.UTF_8);
                    while (index != -1) {
                        long nextFooterOffset;
                        if ((index = FileDump.indexOf(data, magicBytes, index + 1)) == -1 || !FileDump.isReadable(corruptPath, conf, nextFooterOffset = startPos + (long)index + (long)magicBytes.length + 1L)) continue;
                        footerOffsets.add(nextFooterOffset);
                    }
                    System.err.println("Scanning for valid footers - startPos: " + startPos + " toRead: " + toRead + " remaining: " + remaining);
                }
                System.err.println("Readable footerOffsets: " + footerOffsets);
                FileDump.recoverFile(corruptPath, fs, conf, footerOffsets, backup);
            }
            catch (Exception e) {
                Path recoveryFile = FileDump.getRecoveryFile(corruptPath);
                if (fs.exists(recoveryFile)) {
                    fs.delete(recoveryFile, false);
                }
                System.err.println("Unable to recover file " + corruptFile);
                e.printStackTrace();
                System.err.println(SEPARATOR);
                continue;
            }
            finally {
                fdis.close();
                continue;
            }
            System.err.println(corruptFile + " recovered successfully!");
            System.err.println(SEPARATOR);
        }
    }

    private static void recoverFile(Path corruptPath, FileSystem fs, Configuration conf, List<Long> footerOffsets, String backup) throws IOException {
        Path recoveredPath = FileDump.getRecoveryFile(corruptPath);
        if (fs.exists(recoveredPath)) {
            fs.delete(recoveredPath, false);
        }
        if (footerOffsets == null || footerOffsets.isEmpty()) {
            System.err.println("No readable footers found. Creating empty orc file.");
            TypeDescription schema = TypeDescription.createStruct();
            Writer writer = OrcFile.createWriter(recoveredPath, OrcFile.writerOptions(conf).setSchema(schema));
            writer.close();
        } else {
            FSDataInputStream fdis = fs.open(corruptPath);
            FileStatus fileStatus = fs.getFileStatus(corruptPath);
            FSDataOutputStream fdos = fs.create(recoveredPath, true, conf.getInt("io.file.buffer.size", 4096), fileStatus.getReplication(), fileStatus.getBlockSize());
            try {
                long fileLen;
                int toRead;
                for (long remaining = fileLen = footerOffsets.get(footerOffsets.size() - 1).longValue(); remaining > 0L; remaining -= (long)toRead) {
                    toRead = (int)Math.min(0x10000000L, remaining);
                    byte[] data = new byte[toRead];
                    long startPos = fileLen - remaining;
                    fdis.readFully(startPos, data, 0, toRead);
                    fdos.write(data);
                    System.err.println("Copying data to recovery file - startPos: " + startPos + " toRead: " + toRead + " remaining: " + remaining);
                }
            }
            catch (Exception e) {
                fs.delete(recoveredPath, false);
                throw new IOException(e);
            }
            finally {
                fdis.close();
                fdos.close();
            }
        }
        if (FileDump.isReadable(recoveredPath, conf, Long.MAX_VALUE)) {
            String scheme = corruptPath.toUri().getScheme();
            String authority = corruptPath.toUri().getAuthority();
            String filePath = corruptPath.toUri().getPath();
            Path backupDataPath = backup.equals(DEFAULT_BACKUP_PATH) ? new Path(scheme, authority, DEFAULT_BACKUP_PATH + filePath) : Path.mergePaths((Path)new Path(backup), (Path)corruptPath);
            FileDump.moveFiles(fs, corruptPath, backupDataPath);
            Path sideFilePath = OrcAcidUtils.getSideFile(corruptPath);
            Path backupSideFilePath = new Path(backupDataPath.getParent(), sideFilePath.getName());
            FileDump.moveFiles(fs, sideFilePath, backupSideFilePath);
            FileDump.moveFiles(fs, recoveredPath, corruptPath);
            System.err.println("Validation of recovered file successful!");
        }
    }

    private static void moveFiles(FileSystem fs, Path src, Path dest) throws IOException {
        try {
            if (!fs.exists(dest.getParent())) {
                fs.mkdirs(dest.getParent());
            }
            fs.delete(dest, false);
            if (!fs.rename(src, dest)) {
                throw new IOException("Unable to move " + src + " to " + dest);
            }
            System.err.println("Moved " + src + " to " + dest);
        }
        catch (Exception e) {
            throw new IOException("Unable to move " + src + " to " + dest, e);
        }
    }

    private static Path getRecoveryFile(Path corruptPath) {
        return new Path(corruptPath.getParent(), corruptPath.getName() + ".recovered");
    }

    private static boolean isReadable(Path corruptPath, Configuration conf, long maxLen) {
        try {
            OrcFile.createReader(corruptPath, OrcFile.readerOptions(conf).maxLength(maxLen));
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    private static int indexOf(byte[] data, byte[] pattern, int index) {
        if (data == null || data.length == 0 || pattern == null || pattern.length == 0 || index > data.length || index < 0) {
            return -1;
        }
        int j = 0;
        for (int i = index; i < data.length; ++i) {
            j = pattern[j] == data[i] ? ++j : 0;
            if (j != pattern.length) continue;
            return i - pattern.length + 1;
        }
        return -1;
    }

    private static String getFormattedBloomFilters(int col, OrcIndex index, OrcFile.WriterVersion version, TypeDescription.Category type, OrcProto.ColumnEncoding encoding) {
        OrcProto.BloomFilterIndex[] bloomFilterIndex = index.getBloomFilterIndex();
        StringBuilder buf = new StringBuilder();
        BloomFilter stripeLevelBF = null;
        if (bloomFilterIndex != null && bloomFilterIndex[col] != null) {
            int idx = 0;
            buf.append("\n    Bloom filters for column ").append(col).append(":");
            for (OrcProto.BloomFilter bf : bloomFilterIndex[col].getBloomFilterList()) {
                BloomFilter toMerge = BloomFilterIO.deserialize(index.getBloomFilterKinds()[col], encoding, version, type, bf);
                buf.append("\n      Entry ").append(idx++).append(":").append(FileDump.getBloomFilterStats(toMerge));
                if (stripeLevelBF == null) {
                    stripeLevelBF = toMerge;
                    continue;
                }
                stripeLevelBF.merge(toMerge);
            }
            String bloomFilterStats = FileDump.getBloomFilterStats(stripeLevelBF);
            buf.append("\n      Stripe level merge:").append(bloomFilterStats);
        }
        return buf.toString();
    }

    private static String getBloomFilterStats(BloomFilter bf) {
        StringBuilder sb = new StringBuilder();
        int bitCount = bf.getBitSize();
        int popCount = 0;
        for (long l : bf.getBitSet()) {
            popCount += Long.bitCount(l);
        }
        int k = bf.getNumHashFunctions();
        float loadFactor = (float)popCount / (float)bitCount;
        float expectedFpp = (float)Math.pow(loadFactor, k);
        DecimalFormat df = new DecimalFormat("###.####");
        sb.append(" numHashFunctions: ").append(k);
        sb.append(" bitCount: ").append(bitCount);
        sb.append(" popCount: ").append(popCount);
        sb.append(" loadFactor: ").append(df.format(loadFactor));
        sb.append(" expectedFpp: ").append(expectedFpp);
        return sb.toString();
    }

    private static String getFormattedRowIndices(int col, OrcProto.RowIndex[] rowGroupIndex, TypeDescription schema) {
        OrcProto.RowIndex index;
        StringBuilder buf = new StringBuilder();
        buf.append("    Row group indices for column ").append(col).append(":");
        if (rowGroupIndex == null || col >= rowGroupIndex.length || (index = rowGroupIndex[col]) == null) {
            buf.append(" not found\n");
            return buf.toString();
        }
        TypeDescription colSchema = schema.findSubtype(col);
        for (int entryIx = 0; entryIx < index.getEntryCount(); ++entryIx) {
            buf.append("\n      Entry ").append(entryIx).append(": ");
            OrcProto.RowIndexEntry entry = index.getEntry(entryIx);
            if (entry == null) {
                buf.append("unknown\n");
                continue;
            }
            OrcProto.ColumnStatistics colStats = entry.getStatistics();
            if (colStats == null) {
                buf.append("no stats at ");
            } else {
                ColumnStatisticsImpl cs = ColumnStatisticsImpl.deserialize(colSchema, colStats);
                buf.append(((Object)cs).toString());
            }
            buf.append(" positions: ");
            for (int posIx = 0; posIx < entry.getPositionsCount(); ++posIx) {
                if (posIx != 0) {
                    buf.append(",");
                }
                buf.append(entry.getPositions(posIx));
            }
        }
        return buf.toString();
    }

    public static long getTotalPaddingSize(Reader reader) throws IOException {
        long paddedBytes = 0L;
        List<StripeInformation> stripes = reader.getStripes();
        for (int i = 1; i < stripes.size(); ++i) {
            long prevStripeOffset = stripes.get(i - 1).getOffset();
            long prevStripeLen = stripes.get(i - 1).getLength();
            paddedBytes += stripes.get(i).getOffset() - (prevStripeOffset + prevStripeLen);
        }
        return paddedBytes;
    }

    static Options createOptions() {
        Options result = new Options();
        OptionBuilder.withLongOpt((String)"data");
        OptionBuilder.withDescription((String)"Should the data be printed");
        result.addOption(OptionBuilder.create((char)'d'));
        OptionBuilder.withLongOpt((String)"timezone");
        OptionBuilder.withDescription((String)"Print writer's time zone");
        result.addOption(OptionBuilder.create((char)'t'));
        OptionBuilder.withLongOpt((String)"help");
        OptionBuilder.withDescription((String)"print help message");
        result.addOption(OptionBuilder.create((char)'h'));
        OptionBuilder.withLongOpt((String)"rowindex");
        OptionBuilder.withArgName((String)"comma separated list of column ids for which row index should be printed");
        OptionBuilder.withDescription((String)"Dump stats for column number(s)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create((char)'r'));
        OptionBuilder.withLongOpt((String)"json");
        OptionBuilder.withDescription((String)"Print metadata in JSON format");
        result.addOption(OptionBuilder.create((char)'j'));
        OptionBuilder.withLongOpt((String)"pretty");
        OptionBuilder.withDescription((String)"Pretty print json metadata output");
        result.addOption(OptionBuilder.create((char)'p'));
        OptionBuilder.withLongOpt((String)"recover");
        OptionBuilder.withDescription((String)"recover corrupted orc files generated by streaming");
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"skip-dump");
        OptionBuilder.withDescription((String)"used along with --recover to directly recover files without dumping");
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"backup-path");
        OptionBuilder.withDescription((String)"specify a backup path to store the corrupted files (default: /tmp)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create());
        return result;
    }
}

