/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cdc.avro;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.avro.LogicalTypes;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericFixed;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.util.Utf8;
import org.apache.cassandra.cdc.avro.RecordReader;
import org.apache.cassandra.spark.utils.Preconditions;
import org.jetbrains.annotations.NotNull;

public interface TypeConversion<T> {
    public TypeMapping typeMapping();

    public T convert(Schema var1, @NotNull Object var2);

    public static void ensureInputValueType(String conversionName, Object input, Class<?> expectedType) {
        Class<?> inputType = input.getClass();
        if (!expectedType.isAssignableFrom(inputType)) {
            throw new IllegalArgumentException(String.format("%s expects %s input, but has %s", conversionName, expectedType.getSimpleName(), inputType.getSimpleName()));
        }
    }

    public static byte[] getArray(ByteBuffer buffer) {
        return TypeConversion.getArray(buffer, buffer.position(), buffer.remaining());
    }

    public static byte[] getArray(ByteBuffer buffer, int position, int length) {
        if (buffer.hasArray()) {
            int boff = buffer.arrayOffset() + position;
            return Arrays.copyOfRange(buffer.array(), boff, boff + length);
        }
        byte[] bytes = new byte[length];
        ByteBuffer dup = buffer.duplicate();
        dup.position(position).limit(position + length);
        dup.get(bytes);
        return bytes;
    }

    public static class ListConversion
    implements TypeConversion<List<?>> {
        public static final TypeMapping LIST_IDENTITY_MAPPING = TypeMapping.of("array", "array");

        @Override
        public TypeMapping typeMapping() {
            return LIST_IDENTITY_MAPPING;
        }

        @Override
        public List<?> convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, List.class);
            return ((List)fieldValue).stream().map(value -> RecordReader.get().convert(fieldSchema.getElementType(), value)).collect(Collectors.toList());
        }
    }

    public static class UdtConversion
    implements TypeConversion<Map<?, ?>> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("record", "record_udt");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public Map<?, ?> convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, GenericRecord.class);
            Preconditions.checkArgument((fieldSchema.getType() == Schema.Type.RECORD && fieldValue instanceof GenericRecord ? 1 : 0) != 0, (String)"UdtConversion expects a GenericRecord, but has %s", (Object[])new Object[]{fieldSchema});
            GenericRecord record = (GenericRecord)fieldValue;
            return fieldSchema.getFields().stream().collect(Collectors.toMap(Schema.Field::name, field -> this.readValue(record, field.name())));
        }

        private Object readValue(GenericRecord kv, String name) {
            return RecordReader.get().read(kv, name);
        }
    }

    public static class MapConversion
    implements TypeConversion<Map<?, ?>> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("array", "array_map");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public Map<?, ?> convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, List.class);
            Schema subschema = fieldSchema.getElementType();
            Preconditions.checkArgument((subschema.getType() == Schema.Type.RECORD && subschema.getField("key") != null && subschema.getField("value") != null ? 1 : 0) != 0, (String)"MapConversion expects List to contain key value pairs, but has %s", (Object[])new Object[]{subschema});
            List input = (List)fieldValue;
            return input.stream().collect(Collectors.toMap(this::readKey, this::readValue));
        }

        private Object readKey(GenericRecord kv) {
            return this.readKvRec(kv, "key");
        }

        private Object readValue(GenericRecord kv) {
            return this.readKvRec(kv, "value");
        }

        private Object readKvRec(GenericRecord kv, String name) {
            return RecordReader.get().read(kv, name);
        }
    }

    public static class SetConversion
    implements TypeConversion<Set<?>> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("array", "array_set");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public Set<?> convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, List.class);
            return ((List)fieldValue).stream().map(value -> RecordReader.get().convert(fieldSchema.getElementType(), value)).collect(Collectors.toSet());
        }
    }

    public static class TimestampConversion
    implements TypeConversion<Date> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("long", "timestamp-micros", "timestamp");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public Date convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, Long.class);
            Long epochTimeInMicros = (Long)fieldValue;
            return new Date(TimeUnit.MICROSECONDS.toMillis(epochTimeInMicros));
        }
    }

    public static class DecimalConversion
    implements TypeConversion<BigDecimal> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("fixed", "decimal", "decimal");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public BigDecimal convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, GenericFixed.class);
            GenericFixed fixed = (GenericFixed)fieldValue;
            int scale = ((LogicalTypes.Decimal)fieldSchema.getLogicalType()).getScale();
            return new BigDecimal(new BigInteger(fixed.bytes()), scale);
        }
    }

    public static class VarIntConversion
    implements TypeConversion<BigInteger> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("fixed", "decimal", "varint");
        private final DecimalConversion decimalConversion = new DecimalConversion();

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public BigInteger convert(Schema fieldSchema, @NotNull Object fieldValue) {
            int scale = ((LogicalTypes.Decimal)fieldSchema.getLogicalType()).getScale();
            Preconditions.checkState((scale == 0 ? 1 : 0) != 0, (String)"Not a valid varint type", (Object[])new Object[0]);
            return this.decimalConversion.convert(fieldSchema, fieldValue).toBigInteger();
        }
    }

    public static class InetAddressConversion
    implements TypeConversion<InetAddress> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("bytes", "inet", "inet");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public InetAddress convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, ByteBuffer.class);
            ByteBuffer bb = (ByteBuffer)fieldValue;
            try {
                return InetAddress.getByAddress(TypeConversion.getArray(bb));
            }
            catch (UnknownHostException e) {
                throw new AssertionError((Object)e);
            }
        }
    }

    public static class DateConversion
    implements TypeConversion<LocalDate> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("int", "date", "date");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public LocalDate convert(Schema fieldSchema, @NotNull Object fieldValue) {
            TypeConversion.ensureInputValueType(this.getClass().getSimpleName(), fieldValue, Integer.class);
            Integer dateValue = (Integer)fieldValue;
            return LocalDate.ofEpochDay(dateValue.intValue());
        }
    }

    public static class UUIDConversion
    implements TypeConversion<UUID> {
        private static final TypeMapping TYPE_MAPPING = TypeMapping.of("string", "uuid");

        @Override
        public TypeMapping typeMapping() {
            return TYPE_MAPPING;
        }

        @Override
        public UUID convert(Schema fieldSchema, @NotNull Object fieldValue) {
            String uuidStr;
            if (fieldValue instanceof Utf8) {
                uuidStr = ((Utf8)fieldValue).toString();
            } else if (fieldValue instanceof String) {
                uuidStr = (String)fieldValue;
            } else {
                throw new IllegalArgumentException("UUIDConversion expects String input, but has " + fieldValue.getClass().getSimpleName());
            }
            return UUID.fromString(uuidStr);
        }
    }

    public static class TypeMapping {
        public final String sourceTypeName;
        public final String targetTypeName;
        public final String cqlTypeName;

        public static TypeMapping of(String sourceTypeName, String targetTypeName, String cqlTypeName) {
            return new TypeMapping(sourceTypeName, targetTypeName, cqlTypeName);
        }

        public static TypeMapping of(String sourceTypeName, String targetTypeName) {
            return TypeMapping.of(sourceTypeName, targetTypeName, targetTypeName);
        }

        private TypeMapping(String sourceTypeName, String targetTypeName, String cqlTypeName) {
            Preconditions.checkNotNull((Object)sourceTypeName);
            Preconditions.checkNotNull((Object)targetTypeName);
            Preconditions.checkNotNull((Object)cqlTypeName);
            this.sourceTypeName = sourceTypeName;
            this.targetTypeName = targetTypeName;
            this.cqlTypeName = cqlTypeName;
        }

        public int hashCode() {
            return Objects.hash(this.sourceTypeName, this.targetTypeName, this.cqlTypeName);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof TypeMapping)) {
                return false;
            }
            TypeMapping that = (TypeMapping)obj;
            return Objects.equals(this.sourceTypeName, that.sourceTypeName) && Objects.equals(this.targetTypeName, that.targetTypeName) && Objects.equals(this.cqlTypeName, that.cqlTypeName);
        }
    }

    public static interface Registry {
        public void register(TypeConversion<?> var1);

        public TypeConversion<?> lookup(Schema var1);
    }
}

