/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.reader;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.ThreadLocalRandom;
import java.util.zip.Checksum;
import org.apache.cassandra.analytics.reader.common.RawInputStream;
import org.apache.cassandra.analytics.stats.Stats;
import org.apache.cassandra.spark.data.SSTable;
import org.apache.cassandra.spark.reader.CompressionMetadata;
import org.apache.cassandra.spark.reader.common.AbstractCompressionMetadata;
import org.apache.cassandra.spark.reader.common.ChunkCorruptException;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.ChecksumType;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CompressedRawInputStream
extends RawInputStream {
    private static final Logger LOGGER = LoggerFactory.getLogger(CompressedRawInputStream.class);
    @Nullable
    private final SSTable ssTable;
    private final CompressionMetadata metadata;
    private byte[] compressed;
    private long currentCompressed = 0L;
    private final Checksum checksum;
    private final byte[] checksumBytes = new byte[4];

    private CompressedRawInputStream(@Nullable SSTable ssTable, DataInputStream source, CompressionMetadata metadata, Stats stats) {
        super(source, new byte[metadata.chunkLength()], stats);
        this.ssTable = ssTable;
        this.metadata = metadata;
        this.checksum = ChecksumType.CRC32.newInstance();
        this.compressed = new byte[metadata.compressor().initialCompressedBufferLength(metadata.chunkLength())];
    }

    @VisibleForTesting
    static CompressedRawInputStream fromInputStream(InputStream in, InputStream compressionInfoInputStream, boolean hasCompressedLength, double crcCheckChance) throws IOException {
        return CompressedRawInputStream.fromInputStream(null, new DataInputStream(in), compressionInfoInputStream, hasCompressedLength, crcCheckChance, (Stats)Stats.DoNothingStats.INSTANCE);
    }

    static CompressedRawInputStream fromInputStream(@Nullable SSTable ssTable, DataInputStream dataInputStream, InputStream compressionInfoInputStream, boolean hasCompressedLength, double crcCheckChance, Stats stats) throws IOException {
        return CompressedRawInputStream.from(ssTable, dataInputStream, CompressionMetadata.fromInputStream(compressionInfoInputStream, hasCompressedLength, crcCheckChance), stats);
    }

    public static CompressedRawInputStream from(@Nullable SSTable ssTable, DataInputStream dataInputStream, CompressionMetadata compressionMetadata, Stats stats) {
        return new CompressedRawInputStream(ssTable, dataInputStream, compressionMetadata, stats);
    }

    public boolean isEOF() {
        return this.current >= this.metadata.getDataLength();
    }

    private void assertChunkPos(AbstractCompressionMetadata.Chunk chunk) {
        assert (this.currentCompressed <= chunk.offset) : String.format("Requested chunk at input offset %d is less than current compressed position at %d", chunk.offset, this.currentCompressed);
    }

    private void decompressChunk(AbstractCompressionMetadata.Chunk chunk, double crcChance) throws IOException {
        int checkSumFromChunk;
        this.assertChunkPos(chunk);
        this.source.skipBytes((int)(chunk.offset - this.currentCompressed));
        this.currentCompressed = chunk.offset;
        if (this.compressed.length < chunk.length) {
            this.compressed = new byte[chunk.length];
        }
        if (chunk.length > 0) {
            try {
                this.source.readFully(this.compressed, 0, chunk.length);
            }
            catch (EOFException exception) {
                throw new IOException(String.format("Failed to read %d bytes from offset %d.", chunk.length, chunk.offset), exception);
            }
            checkSumFromChunk = this.source.readInt();
            this.stats.readBytes(chunk.length + this.checksumBytes.length);
        } else {
            int lastBytesLength = 0;
            while (true) {
                int readLength;
                if (lastBytesLength >= this.compressed.length) {
                    byte[] buffer = new byte[lastBytesLength * 2];
                    System.arraycopy(this.compressed, 0, buffer, 0, lastBytesLength);
                    this.compressed = buffer;
                }
                if ((readLength = this.source.read(this.compressed, lastBytesLength, this.compressed.length - lastBytesLength)) < 0) break;
                this.stats.readBytes(readLength);
                lastBytesLength += readLength;
            }
            chunk.setLength(lastBytesLength - 4);
            checkSumFromChunk = ByteBufferUtil.toInt((ByteBuffer)ByteBuffer.wrap(this.compressed, lastBytesLength - 4, 4));
        }
        this.validBufferBytes = this.metadata.compressor().uncompress(this.compressed, 0, chunk.length, this.buffer, 0);
        this.stats.decompressedBytes(chunk.length, this.validBufferBytes);
        if (crcChance > ThreadLocalRandom.current().nextDouble()) {
            this.checksum.update(this.compressed, 0, chunk.length);
            if (checkSumFromChunk != (int)this.checksum.getValue()) {
                throw new ChunkCorruptException("bad chunk " + String.valueOf(chunk));
            }
            this.checksum.reset();
        }
        this.currentCompressed += (long)(chunk.length + this.checksumBytes.length);
        this.alignBufferOffset();
    }

    private void alignBufferOffset() {
        this.bufferOffset = this.current & (long)(-this.buffer.length);
    }

    protected void reBuffer() throws IOException {
        try {
            this.decompressChunk(this.metadata.chunkAtPosition(this.current), this.metadata.crcCheckChance());
        }
        catch (IOException exception) {
            if (this.ssTable != null) {
                LOGGER.warn("IOException decompressing SSTable position={} sstable='{}'", new Object[]{this.current, this.ssTable, exception});
                this.stats.decompressionException(this.ssTable, (Throwable)exception);
            }
            throw exception;
        }
    }

    public long skip(long count) throws IOException {
        long precheck = this.maybeStandardSkip(count);
        if (precheck >= 0L) {
            return precheck;
        }
        long remaining = count - this.skipBuffered();
        long totalCompressedBytes = 0L;
        long totalDecompressedBytes = 0L;
        long startCurrent = this.current;
        long startCompressed = this.currentCompressed;
        long startBufferOffset = this.bufferOffset;
        while (totalDecompressedBytes + (long)this.buffer.length < remaining) {
            AbstractCompressionMetadata.Chunk chunk = this.metadata.chunkAtPosition(this.current);
            if (chunk.length < 0) {
                this.current = startCurrent;
                this.currentCompressed = startCompressed;
                this.bufferOffset = startBufferOffset;
                return this.standardSkip(remaining);
            }
            this.assertChunkPos(chunk);
            int chunkLength = chunk.length + this.checksumBytes.length;
            totalCompressedBytes += (long)((int)(chunk.offset - this.currentCompressed));
            this.currentCompressed = chunk.offset;
            totalCompressedBytes += (long)chunkLength;
            this.currentCompressed += (long)chunkLength;
            totalDecompressedBytes += (long)this.buffer.length;
            this.alignBufferOffset();
            this.current += (long)this.buffer.length;
        }
        long skipped = this.source.skip(totalCompressedBytes);
        assert (skipped == totalCompressedBytes) : "Bytes skipped should equal compressed length";
        remaining -= totalDecompressedBytes;
        remaining -= this.standardSkip(remaining);
        long total = count - remaining;
        this.stats.skippedBytes(total);
        return total;
    }
}

