/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.cassandra.bridge.TokenRange;
import org.apache.cassandra.cdc.api.CommitLog;
import org.apache.cassandra.cdc.api.CommitLogMarkers;
import org.apache.cassandra.cdc.api.CommitLogReader;
import org.apache.cassandra.cdc.api.Marker;
import org.apache.cassandra.cdc.stats.ICdcStats;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogReadHandler;
import org.apache.cassandra.db.commitlog.CommitLogSegmentReader;
import org.apache.cassandra.db.commitlog.FourZeroPartitionUpdateWrapper;
import org.apache.cassandra.db.commitlog.PartitionUpdateWrapper;
import org.apache.cassandra.db.commitlog.SeekableCommitLogSegmentReader;
import org.apache.cassandra.db.partitions.AbstractBTreePartition;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.DeserializationHelper;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.UnknownTableException;
import org.apache.cassandra.io.util.CdcRandomAccessReader;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.spark.exceptions.TransportFailureException;
import org.apache.cassandra.spark.utils.AsyncExecutor;
import org.apache.cassandra.spark.utils.LoggerHelper;
import org.apache.cassandra.spark.utils.Pair;
import org.apache.cassandra.spark.utils.ThrowableUtils;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.TokenUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class BufferingCommitLogReader
implements CommitLogReadHandler,
AutoCloseable,
Comparable<BufferingCommitLogReader>,
CommitLogReader {
    private static final int LEGACY_END_OF_SEGMENT_MARKER = 0;
    @VisibleForTesting
    public static final int ALL_MUTATIONS = -1;
    private final CommitLog log;
    private final CRC32 checksum;
    @Nullable
    private final TokenRange tokenRange;
    private final ReadStatusTracker statusTracker;
    private int position = 0;
    private final ICdcStats stats;
    private final long segmentId;
    private final int messagingVersion;
    @NotNull
    private final Marker startMarker;
    private final LoggerHelper logger;
    @Nullable
    private final AsyncExecutor executor;
    @Nullable
    private final Consumer<Marker> listener;
    @NotNull
    private final CommitLogMarkers markers;
    @Nullable
    private final Long startTimestampMicros;
    private byte[] buffer;
    List<PartitionUpdateWrapper> updates = new ArrayList<PartitionUpdateWrapper>(1024);
    private RandomAccessReader reader;
    private CommitLogDescriptor desc = null;
    private boolean skipped = false;

    @VisibleForTesting
    public BufferingCommitLogReader(@NotNull CommitLog log, @Nullable Marker startMarker, @NotNull ICdcStats stats, @Nullable Consumer<Marker> listener) {
        this(log, null, (CommitLogMarkers)CommitLogMarkers.of((Marker)startMarker), 0, stats, null, listener, null, false);
    }

    public BufferingCommitLogReader(@NotNull CommitLog log, @Nullable TokenRange tokenRange, @NotNull CommitLogMarkers markers, int partitionId, @NotNull ICdcStats stats, @Nullable AsyncExecutor executor, @Nullable Consumer<Marker> listener, @Nullable Long startTimestampMicros, boolean readHeader) {
        this.log = log;
        this.tokenRange = tokenRange;
        this.statusTracker = new ReadStatusTracker(-1, false);
        this.checksum = new CRC32();
        this.buffer = new byte[4096];
        this.reader = BufferingCommitLogReader.reader(log);
        this.markers = markers;
        this.listener = listener;
        this.startTimestampMicros = startTimestampMicros;
        Pair pair = (Pair)CommitLog.extractVersionAndSegmentId((CommitLog)log).orElseThrow(() -> new IllegalStateException("Could not extract segmentId from CommitLog filename"));
        this.messagingVersion = (Integer)pair.getLeft();
        this.segmentId = (Long)pair.getRight();
        this.logger = new LoggerHelper(LoggerFactory.getLogger(BufferingCommitLogReader.class), new Object[]{"instance", log.instance().nodeName(), "dc", log.instance().dataCenter(), "log", log.name(), "size", log.maxOffset(), "segmentId", this.segmentId, "partitionId", partitionId});
        Marker startMarker = markers.startMarker(log);
        this.startMarker = startMarker.segmentId() == this.segmentId ? startMarker : log.zeroMarker();
        this.stats = stats;
        this.executor = executor;
        this.logger.trace("Opening BufferingCommitLogReader", new Supplier[0]);
        try {
            if (readHeader || this.startMarker.position() == 0) {
                this.readHeader();
                if (this.skip(this.startMarker)) {
                    this.logger.trace("Skipping commit log after reading header", new Supplier[0]);
                    stats.skippedCommitLogsCount(1L);
                    this.skipped = true;
                    return;
                }
            } else if (this.shouldSkipSegmentId(this.startMarker)) {
                this.logger.trace("Skipping log", new Supplier[0]);
                stats.skippedCommitLogsCount(1L);
                this.skipped = true;
                return;
            }
            this.read();
        }
        catch (Throwable t) {
            this.skipped = true;
            if (this.isNotFoundError(t)) {
                return;
            }
            this.logger.warn("Exception reading CommitLog", t, new Object[0]);
            throw new RuntimeException(t);
        }
    }

    public static RandomAccessReader reader(CommitLog log) {
        return new CdcRandomAccessReader(log);
    }

    private void readHeader() throws IOException {
        this.logger.trace("Reading header", new Supplier[0]);
        long startTimeNanos = System.nanoTime();
        try {
            this.desc = CommitLogDescriptor.readHeader((DataInput)this.reader, (EncryptionContext)DatabaseDescriptor.getEncryptionContext());
        }
        catch (IOException e) {
            this.logger.warn("IOException reading CommitLog header", (Throwable)e, new Object[0]);
            this.stats.commitLogHeaderReadFailureCount(1L);
        }
        if (this.desc == null) {
            this.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Could not read commit log descriptor in file %s", this.log.name()), CommitLogReadHandler.CommitLogReadErrorReason.UNRECOVERABLE_DESCRIPTOR_ERROR, false));
        } else {
            long timeTakenToReadHeader = System.nanoTime() - startTimeNanos;
            this.logger.debug("Read log header", new Object[]{"segmentId", this.desc.id, "compression", this.desc.compression, "version", this.desc.version, "messagingVersion", this.desc.getMessagingVersion(), "timeNanos", timeTakenToReadHeader});
            this.stats.commitLogHeaderReadTime(timeTakenToReadHeader);
        }
    }

    private void read() {
        try {
            this.readCommitLogSegment();
        }
        catch (Throwable t) {
            if (this.isNotFoundError(t)) {
                return;
            }
            Throwable cause = ThrowableUtils.rootCause((Throwable)t);
            this.logger.warn("Exception reading CommitLog", cause, new Object[0]);
            throw new RuntimeException(cause);
        }
    }

    private boolean isNotFoundError(Throwable t) {
        TransportFailureException transportEx = (TransportFailureException)ThrowableUtils.rootCause((Throwable)t, TransportFailureException.class);
        if (transportEx != null && transportEx.isNotFound()) {
            this.logger.warn("CommitLog not found, assuming removed by underlying storage", (Throwable)transportEx, new Object[0]);
            return true;
        }
        return false;
    }

    private void readCommitLogSegment() throws IOException {
        SeekableCommitLogSegmentReader segmentReader;
        long startTimeNanos = System.nanoTime();
        try {
            segmentReader = new SeekableCommitLogSegmentReader(this.segmentId, this, this.desc, this.reader, this.logger, false);
        }
        catch (Exception e) {
            this.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Unable to create segment reader for commit log file: %s", e), CommitLogReadHandler.CommitLogReadErrorReason.UNRECOVERABLE_UNKNOWN_ERROR, false));
            return;
        }
        try {
            if (this.reader.getFilePointer() < (long)this.startMarker.position()) {
                this.stats.commitLogBytesSkippedOnRead((long)this.startMarker.position() - this.reader.getFilePointer());
                segmentReader.seek(this.startMarker.position());
            }
            for (CommitLogSegmentReader.SyncSegment syncSegment : segmentReader) {
                this.statusTracker.errorContext = String.format("Next section at %d in %s", syncSegment.fileStartPosition, this.log.name());
                this.readSection(syncSegment.input, syncSegment.endPosition);
                this.position = (int)this.reader.getFilePointer();
                if (this.listener != null) {
                    this.listener.accept(this.log.markerAt(this.segmentId, this.position));
                }
                if (this.statusTracker.shouldContinue()) continue;
                break;
            }
        }
        catch (RuntimeException re) {
            if (re.getCause() instanceof IOException) {
                throw (IOException)re.getCause();
            }
            throw re;
        }
        this.logger.debug("Finished reading commit log", new Object[]{"updates", this.updates.size(), "timeNanos", System.nanoTime() - startTimeNanos});
    }

    public boolean skip(@Nullable Marker highWaterMark) throws IOException {
        if (BufferingCommitLogReader.shouldSkip(this.reader)) {
            this.logger.debug("Skipping playback of empty log", new Supplier[0]);
            return true;
        }
        long segmentIdFromFilename = CommitLogDescriptor.fromFileName((String)this.log.name()).id;
        if (segmentIdFromFilename != this.desc.id) {
            CommitLogReadHandler.CommitLogReadException readException = new CommitLogReadHandler.CommitLogReadException(String.format("Segment id mismatch (filename %d, descriptor %d) in file %s", segmentIdFromFilename, this.desc.id, this.log.name()), CommitLogReadHandler.CommitLogReadErrorReason.RECOVERABLE_DESCRIPTOR_ERROR, false);
            return this.shouldSkipSegmentOnError(readException);
        }
        return this.shouldSkipSegmentId(highWaterMark);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean shouldSkip(RandomAccessReader reader) throws IOException {
        try {
            reader.mark();
            int end = reader.readInt();
            long filecrc = (long)reader.readInt() & 0xFFFFFFFFL;
            boolean bl = end == 0 && filecrc == 0L;
            return bl;
        }
        catch (EOFException e) {
            boolean bl = true;
            return bl;
        }
        finally {
            reader.reset();
        }
    }

    private boolean shouldSkipSegmentId(@Nullable Marker highWaterMark) {
        this.logger.debug("Reading commit log", new Object[]{"version", this.messagingVersion, "compression", this.desc != null ? this.desc.compression : "disabled"});
        if (highWaterMark != null && highWaterMark.segmentId() > this.segmentId) {
            this.logger.info("Skipping read of fully-flushed log", new Object[]{"segmentId", this.segmentId, "minSegmentId", highWaterMark.segmentId()});
            return true;
        }
        return false;
    }

    private void readSection(FileDataInput reader, int end) throws IOException {
        long startTimeNanos = System.nanoTime();
        this.logger.trace("Reading log section", new Object[]{"end", end});
        while (this.statusTracker.shouldContinue() && reader.getFilePointer() < (long)end && !reader.isEOF()) {
            long claimedCRC32;
            int serializedSize;
            int mutationStart = (int)reader.getFilePointer();
            this.logger.trace("Reading mutation at", new Object[]{"position", mutationStart});
            try {
                if ((long)end - reader.getFilePointer() < 4L) {
                    this.logger.trace("Not enough bytes left for another mutation in this CommitLog section, continuing", new Supplier[0]);
                    this.statusTracker.requestTermination();
                    return;
                }
                serializedSize = reader.readInt();
                if (serializedSize == 0) {
                    this.logger.trace("Encountered end of segment marker at", new Object[]{"position", reader.getFilePointer()});
                    this.statusTracker.requestTermination();
                    return;
                }
                if (serializedSize < 10) {
                    if (this.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Invalid mutation size %d at %d in %s", serializedSize, mutationStart, this.statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, this.statusTracker.tolerateErrorsInSection))) {
                        this.statusTracker.requestTermination();
                    }
                    this.stats.commitLogInvalidSizeMutationCount(1L);
                    return;
                }
                long claimedSizeChecksum = CommitLogFormat.calculateClaimedChecksum(reader);
                this.checksum.reset();
                CommitLogFormat.updateChecksum(this.checksum, serializedSize);
                if (this.checksum.getValue() != claimedSizeChecksum) {
                    if (this.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Mutation size checksum failure at %d in %s", mutationStart, this.statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, this.statusTracker.tolerateErrorsInSection))) {
                        this.statusTracker.requestTermination();
                    }
                    this.stats.mutationsChecksumMismatchCount(1L);
                    return;
                }
                if (serializedSize > this.buffer.length) {
                    this.buffer = new byte[(int)(1.2 * (double)serializedSize)];
                }
                reader.readFully(this.buffer, 0, serializedSize);
                claimedCRC32 = CommitLogFormat.calculateClaimedCRC32(reader);
            }
            catch (EOFException eof) {
                if (this.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Unexpected end of segment at %d in %s", mutationStart, this.statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.EOF, this.statusTracker.tolerateErrorsInSection))) {
                    this.statusTracker.requestTermination();
                }
                this.stats.commitLogSegmentUnexpectedEndErrorCount(1L);
                return;
            }
            this.checksum.update(this.buffer, 0, serializedSize);
            if (claimedCRC32 != this.checksum.getValue()) {
                if (this.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Mutation checksum failure at %d in %s", mutationStart, this.statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, this.statusTracker.tolerateErrorsInSection))) {
                    this.statusTracker.requestTermination();
                }
                this.stats.mutationsChecksumMismatchCount(1L);
                continue;
            }
            int mutationPosition = (int)reader.getFilePointer();
            this.readMutationInternal(this.buffer, serializedSize, mutationPosition);
            this.statusTracker.addProcessedMutation();
        }
        this.stats.commitLogSegmentReadTime(System.nanoTime() - startTimeNanos);
    }

    @VisibleForTesting
    private void readMutationInternal(byte[] inputBuffer, int size, int mutationPosition) throws IOException {
        Mutation mutation;
        try (DataInputBuffer bufIn = new DataInputBuffer(inputBuffer, 0, size);){
            mutation = Mutation.serializer.deserialize((DataInputPlus)bufIn, this.messagingVersion, DeserializationHelper.Flag.LOCAL);
        }
        catch (UnknownTableException ex) {
            if (ex.id == null) {
                return;
            }
            this.logger.trace("Invalid mutation", new Object[]{ex});
            this.stats.mutationsIgnoredUnknownTableCount(1L);
            return;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable((Throwable)t);
            Path p = Files.createTempFile("mutation", "dat", new FileAttribute[0]);
            try (DataOutputStream out = new DataOutputStream(Files.newOutputStream(p, new OpenOption[0]));){
                out.write(inputBuffer, 0, size);
            }
            this.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Unexpected error deserializing mutation; saved to %s.  This may be caused by replaying a mutation against a table with the same name but incompatible schema.  Exception follows: %s", p, t), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, false));
            this.stats.mutationsDeserializeFailedCount(1L);
            return;
        }
        Supplier[] supplierArray = new Supplier[6];
        supplierArray[0] = () -> "keyspace";
        supplierArray[1] = () -> ((Mutation)mutation).getKeyspaceName();
        supplierArray[2] = () -> "key";
        supplierArray[3] = () -> ((Mutation)mutation).key();
        supplierArray[4] = () -> "mutation";
        supplierArray[5] = () -> "{" + mutation.getPartitionUpdates().stream().map(AbstractBTreePartition::toString).collect(Collectors.joining(", ")) + "}";
        this.logger.trace("Read mutation for", supplierArray);
        this.stats.mutationsReadCount(1L);
        this.stats.mutationsReadBytes((long)size);
        this.handleMutation(mutation, size, mutationPosition, this.desc);
    }

    @Override
    public void close() {
        if (this.updates == null) {
            return;
        }
        try {
            this.logger.trace("Closing log", new Supplier[0]);
            this.reader.close();
            this.reader = null;
            this.updates = null;
        }
        catch (Throwable t) {
            this.logger.error("Unexpected exception closing reader", t, new Object[0]);
        }
    }

    @Override
    public int compareTo(@NotNull BufferingCommitLogReader o) {
        return Long.compare(this.segmentId, o.segmentId);
    }

    public List<PartitionUpdateWrapper> updates() {
        return this.updates;
    }

    public CommitLog log() {
        return this.log;
    }

    public long segmentId() {
        return this.segmentId;
    }

    public int position() {
        return this.position;
    }

    public boolean skipped() {
        return this.skipped;
    }

    public CommitLogReader.Result result() {
        return new CommitLogReader.Result((CommitLogReader)this);
    }

    public boolean shouldSkipSegmentOnError(CommitLogReadHandler.CommitLogReadException e) {
        this.logger.warn("CommitLog error on shouldSkipSegment", (Throwable)e, new Object[0]);
        return false;
    }

    public void handleUnrecoverableError(CommitLogReadHandler.CommitLogReadException e) throws IOException {
        this.logger.error("CommitLog unrecoverable error", (Throwable)e, new Object[0]);
        this.statusTracker.requestTermination();
        this.stats.corruptCommitLog(this.log);
        throw e;
    }

    public void handleMutation(Mutation mutation, int size, int mutationPosition, @Nullable CommitLogDescriptor desc) {
        if (!mutation.trackedByCDC()) {
            this.logger.debug("Ignore mutation not tracked by CDC", new Supplier[0]);
            return;
        }
        mutation.getPartitionUpdates().stream().filter(update -> this.filter(mutationPosition, (PartitionUpdate)update)).map(update -> Pair.of((Object)update, (Object)this.maxTimestamp((PartitionUpdate)update))).filter(this::withinTimeWindow).peek(pair -> ((PartitionUpdate)pair.getLeft()).validate()).map(this::toCdcUpdate).forEach(this.updates::add);
    }

    private long maxTimestamp(PartitionUpdate update) {
        if (update.rowCount() == 1 && !update.lastRow().deletion().isLive()) {
            return update.lastRow().deletion().time().markedForDeleteAt();
        }
        return update.maxTimestamp();
    }

    private PartitionUpdateWrapper toCdcUpdate(Pair<PartitionUpdate, Long> update) {
        return this.toCdcUpdate((PartitionUpdate)update.getLeft(), (Long)update.getRight());
    }

    private PartitionUpdateWrapper toCdcUpdate(PartitionUpdate update, long maxTimestampMicros) {
        return new FourZeroPartitionUpdateWrapper(update, maxTimestampMicros, this.executor);
    }

    private boolean filter(int position, PartitionUpdate update) {
        return this.isCdcEnabled(update) && this.withinRange(position, update);
    }

    private boolean isCdcEnabled(PartitionUpdate update) {
        if (update.metadata().params.cdc) {
            return true;
        }
        String keyspace = BufferingCommitLogReader.getKeyspace(update);
        String table = BufferingCommitLogReader.getTable(update);
        this.logger.debug("Ignore partition update from table not tracked by CDC", new Object[]{"keyspace", keyspace, "table", table});
        this.stats.untrackedChangesIgnored(keyspace, table, 1L);
        return false;
    }

    private boolean withinTimeWindow(Pair<PartitionUpdate, Long> update) {
        PartitionUpdate pu = (PartitionUpdate)update.getLeft();
        long mutationTimestamp = (Long)update.getRight();
        boolean shouldInclude = this.withinTimeWindow(mutationTimestamp);
        if (!shouldInclude) {
            String keyspace = BufferingCommitLogReader.getKeyspace(pu);
            String table = BufferingCommitLogReader.getTable(pu);
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Exclude the update due to out of the allowed time window.", new Object[]{"update", "'" + String.valueOf(pu) + "'", "keyspace", keyspace, "table", table, "timestampMicros", mutationTimestamp, "maxAgeMicros", this.startTimestampMicros == null ? "null" : this.startTimestampMicros});
            } else {
                this.logger.warn("Exclude the update due to out of the allowed time window.", null, new Object[]{"keyspace", keyspace, "table", table, "timestampMicros", mutationTimestamp, "maxAgeMicros", this.startTimestampMicros == null ? "null" : this.startTimestampMicros});
            }
            this.stats.droppedOldMutation(BufferingCommitLogReader.getKeyspace(pu), BufferingCommitLogReader.getTable(pu), mutationTimestamp);
            return false;
        }
        return true;
    }

    private boolean withinTimeWindow(long mutationTimestamp) {
        if (this.startTimestampMicros == null) {
            return true;
        }
        return mutationTimestamp >= this.startTimestampMicros;
    }

    private boolean withinRange(int position, PartitionUpdate update) {
        if (this.tokenRange == null) {
            return true;
        }
        BigInteger token = TokenUtils.tokenToBigInteger((Token)update.partitionKey().getToken());
        if (this.tokenRange.contains(token)) {
            Marker marker = this.log.markerAt(this.segmentId, position);
            if (this.markers.canIgnore(marker, token)) {
                this.logger.debug("Ignoring mutation before start marker", new Object[]{"position", position, "marker", marker.position(), "markerSegmentId", marker.segmentId()});
                return false;
            }
            return true;
        }
        String keyspace = BufferingCommitLogReader.getKeyspace(update);
        String table = BufferingCommitLogReader.getTable(update);
        this.logger.debug("Ignore out of range partition update.", new Object[]{"keyspace", keyspace, "table", table});
        this.stats.outOfTokenRangeChangesIgnored(keyspace, table, 1L);
        return false;
    }

    private static String getKeyspace(PartitionUpdate partitionUpdate) {
        return partitionUpdate.metadata().keyspace;
    }

    private static String getTable(PartitionUpdate partitionUpdate) {
        return partitionUpdate.metadata().name;
    }

    private static class ReadStatusTracker {
        private int mutationsLeft;
        public String errorContext = "";
        public boolean tolerateErrorsInSection;
        private boolean error = false;

        private ReadStatusTracker(int mutationLimit, boolean tolerateErrorsInSection) {
            this.mutationsLeft = mutationLimit;
            this.tolerateErrorsInSection = tolerateErrorsInSection;
        }

        public void addProcessedMutation() {
            if (this.mutationsLeft == -1) {
                return;
            }
            --this.mutationsLeft;
        }

        public boolean shouldContinue() {
            return !this.error && this.mutationsLeft != 0;
        }

        public void requestTermination() {
            this.error = true;
        }
    }

    private static class CommitLogFormat {
        private CommitLogFormat() {
        }

        public static long calculateClaimedChecksum(FileDataInput input) throws IOException {
            return (long)input.readInt() & 0xFFFFFFFFL;
        }

        public static void updateChecksum(CRC32 checksum, int serializedSize) {
            FBUtilities.updateChecksumInt((Checksum)checksum, (int)serializedSize);
        }

        public static long calculateClaimedCRC32(FileDataInput input) throws IOException {
            return (long)input.readInt() & 0xFFFFFFFFL;
        }
    }
}

